/***************************************************************************
                          knapster.cpp  -  description
                             -------------------
    begin                : Mon Dec  6 23:50:46 CST 1999
    copyright            : (C) 1999,2000 by John Donoghue
    email                : donoghue@chariot.net.au
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "knapster.h"
#include "support_funcs.h"

#include "propertiesdlg.h"
#include "napsterdownload.h"
#include "napsterupload.h"

#include <kmenubar.h>
#include <qpopupmenu.h>
#include <kapp.h>
#include <ktabctl.h>
#include <qcolor.h>
#include <qpalette.h>
#include <qframe.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qcombo.h>
#include <qlineedit.h>
#include <qlistview.h>
#ifdef KDE2
#include <kmessagebox.h>
#else
#include <kmsgbox.h>
#endif /* KDE2 */
#include <ksock.h>
#include <kconfig.h>
#include <qtimer.h>
#include <qfile.h>
#ifdef KDE2
#include <kconfig.h>
#include <kglobal.h>
#include <klocale.h>
#endif /* KDE2 */

#include "searchpanel.h"
#include "consolepanel.h"
#include "downloadpanel.h"
#include "uploadpanel.h"
#include "librarypanel.h"

#include <iostream.h>

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/time.h>

Knapster::Knapster(const char *name)
 : KTMainWindow(name)
{
#ifdef KDE2
  KConfig *config = KGlobal::config();
#endif  

#ifdef KDE2
#else
  // allow $ in the config file to read
  kapp->getConfig()->setDollarExpansion(false);
#endif

  Preferences p=PropertiesDlg::readPrefs();
  setCaption(name);

  // size the window if we need to
  if(p.winw>10 && p.winh>10) resize(p.winw,p.winh);

  listener=NULL; // no listener on the connection yet

  dloadval=0;    // id counter for downloads

  // set it to auto delete items from trhe download list as they are removed
  downloadlist.setAutoDelete(true);
  uploadlist.setAutoDelete(true);

  pinglist.setAutoDelete(true);

  // setup the statusbar
  enableStatusBar();
  statusBar()->setInsertOrder( KStatusBar::LeftToRight );

  statusBar()->insertItem("Not connected                                      ",0);

  statusBar()->insertItem("00000000000 songs in 000000000000 Libraries. Total 00000000000 Gig.",1);
  statusBar()->setAlignment(AlignRight,1); // too bad it doesn't work!!
  statusBar()->message("Not connected");

  // menubar
  setupMenus();

  // the tabs
  setupTabs();

  // create the timer for dl updates
  QTimer *timer=new QTimer(this);
  if(timer) {
     connect(timer,SIGNAL(timeout()),this,SLOT(timeout()));

     timer->start(2000); // update ever 2 second
  }

  writeConsole("KNapster " VERSION " - John Donoghue, 2000\ndonoghue@chariot.net.au\n");

}

Knapster::~Knapster()
{
  closeDown();
  exit(0);     // FIX to stop unknown seg11
}

void Knapster::closeDown()
{
  // connection will close as destroyed
  // have to do it here as if we do it in deconstructor, kapp may no longer be there
  // ( if quit() was called )

#ifdef KDE2
  KConfig *config = KGlobal::config();
#endif

  if(kapp) {
    if(player.isRunning()) {
      player.kill();
      return;
    }
#ifdef KDE2
    config->writeEntry("winw",width());
    config->writeEntry("winh",height());

    // force save
    config->sync();
#else
    kapp->getConfig()->writeEntry("winw",width());
    kapp->getConfig()->writeEntry("winh",height());

    // force save
    kapp->getConfig()->sync();
#endif

    kapp->quit();
  }
}

// create the tabs and the items in it
void Knapster::setupTabs()
{
  Preferences p=PropertiesDlg::readPrefs();

  KTabCtl *tab=new KTabCtl(this);

  // tab1 - console -------------------
  console=new ConsolePanel(tab);
  connect(console,SIGNAL(command(const char *)),this,SLOT(startConsoleCommand(const char *)));
  tab->addTab(console,i18n("Console"));

  // tab2 - search --------------------
  searchpanel=new SearchPanel(tab);
  connect(searchpanel,SIGNAL(search(int,const char *)),this,SLOT(startSearch(int,const char *)));
  connect(searchpanel,SIGNAL(download(QListViewItem *)),this,SLOT(searchDownload(QListViewItem *)));
  connect(searchpanel,SIGNAL(ping(const char *)),this,SLOT(addPingItem(const char *)));
  connect(searchpanel,SIGNAL(whois(const char *)),this,SLOT(sendWhois(const char *)));
  // set the showfullpathname
  searchpanel->setShowSearchPath(p.showpath);
  // turn multi select on/off
  searchpanel->setMultiSelect(p.multiselect);

  tab->addTab(searchpanel,i18n("Search"));

  // tab3 - downloads =-------------------------
  downloadpanel=new DownloadPanel(tab);
  connect(downloadpanel,SIGNAL(ping(const char *)),this,SLOT(addPingItem(const char *)));
  connect(downloadpanel,SIGNAL(whois(const char *)),this,SLOT(sendWhois(const char *)));
  connect(downloadpanel,SIGNAL(terminate(QListViewItem *)),this,SLOT(downloadTerminate(QListViewItem *)));
  connect(downloadpanel,SIGNAL(play(const char *)),this,SLOT(playsong_dl(const char *)));
  tab->addTab(downloadpanel,i18n("Downloads"));

  // tab4 - uploads  ---------------------------
  //QWidget *frame=new QWidget(tab);
  uploadpanel=new UploadPanel(tab);
  connect(uploadpanel,SIGNAL(ping(const char *)),this,SLOT(addPingItem(const char *)));
  connect(uploadpanel,SIGNAL(whois(const char *)),this,SLOT(sendWhois(const char *)));
  tab->addTab(uploadpanel,i18n("Uploads"));

  // tab5 - local mp3 --------------------------
  librarypanel=new LibraryPanel(tab,"library");
  connect(librarypanel,SIGNAL(play(const char *)),this,SLOT(playsong(const char *)));
  connect(librarypanel,SIGNAL(rescan()),this,SLOT(loadMP3LibraryPaths()));
  tab->addTab(librarypanel,i18n("MP3 Library"));

  // set tab as the main widget
  setView(tab);
}

// create the mainmenu bar and menus
void Knapster::setupMenus()
{
  // who wants floating menus - not me
#ifdef KDE2
  // cannot find matching functions... 
#else
  menuBar()->enableFloating(false);
  menuBar()->enableMoving(false);
#endif /* KDE2 */

  // Project
  QPopupMenu *m=new QPopupMenu();
  m->insertItem(i18n("&LogOn/Off"),this,SLOT(menuLogon()));
  m->insertItem(i18n("&Relogin"),this,SLOT(menuRelogon()));
    m->insertSeparator();
  m->insertItem(i18n("E&xit"),this,SLOT(closeDown()));

  menuBar()->insertItem(i18n("&Connect"),m);

  // view
  m=new QPopupMenu();
  m->insertItem(i18n("&Preferences"),this,SLOT(menuProperties()));
    m->insertSeparator();
  m->insertItem(i18n("&Clear console"),this,SLOT(menuConsoleClear()));

  menuBar()->insertItem(i18n("&Options"),m);

  // help
#ifdef KDE2
  QPopupMenu *q=helpMenu("KNapster " VERSION "\n"
                         "A linux KDE Napster client\n"
                         "\nwritten by\n"
                         "John Donoghue, 2000.\n"
                         "donoghue@chariot.net.au\n");                     
#else
  QPopupMenu *q=kapp->getHelpMenu(true,
        "KNapster " VERSION "\n"
        "A linux KDE Napster client\n"
        "\nwritten by\n"
        "John Donoghue, 2000.\n"
        "donoghue@chariot.net.au\n"
       );
#endif
  menuBar()->insertItem(i18n("&Help"),q);
}

// write a string to the console
void Knapster::writeConsole(char *s)
{
  console->writeString(s);
}

// write to the console - the QString version
void Knapster::writeConsole(QString s)
{
  // write text to the console window
  console->writeString(s);
}

// slot for the search textedit
void Knapster::startSearch(int type,const char *text)
{
  if(type==0) {
    QString tmp=text;

    // skip FILNAME CONTAINS
    extractString(tmp);
    extractString(tmp);
    writeConsole(QString(i18n("Searching for ")) + extractString(tmp));

    Preferences p=PropertiesDlg::readPrefs();

    if(p.clearsearch) { // do we want to clear the searchlist
       searchpanel->clearSearch();
    }
    statusBar()->changeItem("Searching...",0);
    connection.search(text);
  }
  else { // search by user
    writeConsole(QString(i18n("Searching for user ")) + QString(text));

    Preferences p=PropertiesDlg::readPrefs();

    if(p.clearsearch) { // do we want to clear the searchlist
       searchpanel->clearSearch();
    }
    statusBar()->changeItem(i18n("Searching..."),0);

    connection.search_user(text);
  }
}

// menu selection slot for clearing the console
void Knapster::menuConsoleClear()
{
  console->clear();
}

// menu selection slot for opening prefs dlg
void Knapster::menuProperties()
{
  Preferences oldprefs=PropertiesDlg::readPrefs();
  PropertiesDlg p(this);
  p.show();
  Preferences newprefs=PropertiesDlg::readPrefs();

  searchpanel->setShowSearchPath(newprefs.showpath);
  searchpanel->setMultiSelect(newprefs.multiselect);

  return;
}
// menu slection slot menu logon/off
void Knapster::menuLogon()
{

  // are we logged on??
  if(connection.isConnected()) { // YES - then log off
     if(listener) {
        delete listener;
        listener=NULL;
     }

     connection.Disconnect();
     writeConsole(i18n("Disconnected"));
     statusBar()->message(i18n("Not Connected"));

     return;
  }

  // else login

  menuConsoleClear();

  searchpanel->clearSearch();

  // write what's happening
  writeConsole("KNapster " VERSION " - John Donoghue, 2000\ndonoghue@chariot.net.au\n");


  // get connect password,username etc
  Preferences p=PropertiesDlg::readPrefs();

  // check that prefs have been set
  if(p.username.length()==0 || p.password.length()==0) {

#ifdef KDE2
     KMessageBox::warningYesNo(this, "Knapster " VERSION ,
           QString(i18n("To logon, the username and password must be given in the preferences.")));
#else
     KMsgBox::yesNo(this,"Knapster " VERSION ,
           QString(i18n("To logon, the username and password must be given in the preferences.")),
           KMsgBox::EXCLAMATION,
           i18n("Ok"));
#endif /* KDE2 */
     return;
  }

  writeConsole(i18n("Looking up server..."));
  statusBar()->message(i18n("Trying to connect..."));

  // do a little process all pending msg's here!!!!!!!
  kapp->processEvents(1000);

  if(p.serveraddr.length()==0) {
#ifdef KDE2
   KMessageBox::warningYesNo(this, "Knapster " VERSION ,
           QString(i18n("No napster server has been specified in the preferences.")));
#else
     KMsgBox::yesNo(this,"Knapster " VERSION ,
           QString(i18n("No napster server has been specified in the preferences.")),
           KMsgBox::EXCLAMATION,
           i18n("Ok"));
#endif /* KDE2 */
     return;
  }

  // get best host
  if(connection.getBestHost(p.serveraddr)) {
    writeConsole(i18n("connecting to best host..."));
    kapp->processEvents(1000);
    // connect to host
    if(connection.Connect(p.port)) {
      writeConsole(i18n("logging on..."));
      kapp->processEvents(1000);
      // login
      if(!connection.Login(p.username,p.password,p.email,p.speed,p.newuser))
         writeConsole(QString(i18n("Could not logon: ")) + connection.getLastErrorStr());
    }
    else writeConsole(QString(i18n("Could not connect to best host: ")) + connection.getLastErrorStr());

  }
  else writeConsole(QString(i18n("Could not connect to server: ")) + connection.getLastErrorStr() );

  // did we get through all that ok?
  if(connection.isConnected()) {  // YES - we're connected
     statusBar()->clear();
     statusBar()->changeItem(i18n("Connected"),0);
     writeConsole(i18n("Connected to ") + QString(connection.getIPAddressString().c_str()) + "\n");

     if(connection.getListener()==-1) {
       writeConsole(i18n("***Warning: ") + QString(i18n("no listener could be created - some downloads will not work\n")));
     }

     // add the klistener
     listener = new KSocket(connection.getSocket());
     listener->enableRead(true);
     // attach to the socket listener slots
     connect(listener,SIGNAL(readEvent(KSocket *)),this,SLOT(napsterRead(KSocket *)));
     connect(listener,SIGNAL(closeEvent(KSocket *)),this,SLOT(napsterClose(KSocket *)));

     if(p.newuser) { // we were a new user - but not now!!
#ifdef KDE2
        KGlobal::config()->writeEntry("newuser",false);
#else
        kapp->getConfig()->writeEntry("newuser",false);
#endif /* KDE2 */
     }
  }
  else {
     // didn't connect - reason should already have been printed
     statusBar()->message(i18n("Not connected"));
  }


  // TESTIT
  songpos=0;
  songnames=librarypanel->getAllSongNames();
}

// menu selection slot for menu RElogin
void Knapster::menuRelogon()
{
  // can't relogin if already connected
  if(connection.isConnected()) {
     writeConsole(i18n("Still connected?? - logoff first"));
     return;
  }
  // have we got the host ??
  if(!connection.haveHost()) {
     // we haven't a host - have to log on as normal
     menuLogon();
     return;
  }
  // else contine relogin

  // clear console but keep the search items since they should still be valid
  menuConsoleClear();

  // print whats going on
  writeConsole("KNapster " VERSION " - John Donoghue, 2000\ndonoghue@chariot.net.au\n");
  statusBar()->message(i18n("Trying to relogin..."));

  // do a little process all pending msg's here!!!!!!!
  kapp->processEvents(1000);

  // get connect password etc
  Preferences p=PropertiesDlg::readPrefs();

  if(connection.Connect(p.port)) { // try to connect
      writeConsole(i18n("logging on..."));
      kapp->processEvents(1000);
      if(!connection.Login(p.username,p.password,p.email,p.speed,p.newuser))
         writeConsole(QString(i18n("Could not logon: ")) + connection.getLastErrorStr());
  }
  else writeConsole(QString(i18n("Could not connect to best host")) + connection.getLastErrorStr());

  // did we connect
  if(connection.isConnected()) { // YES - tell the world.
     statusBar()->clear();
     statusBar()->changeItem(i18n("Connected"),0);
     writeConsole(i18n("Connected to ") + QString(connection.getIPAddressString().c_str()) + "\n");
     if(connection.getListener()==-1) {
       writeConsole(i18n("** Warning: ") + QString(i18n("no listener could be created - some downloads will not work\n")));
     }

     listener = new KSocket(connection.getSocket());
     listener->enableRead(true);
     connect(listener,SIGNAL(readEvent(KSocket *)),this,SLOT(napsterRead(KSocket *)));
     connect(listener,SIGNAL(closeEvent(KSocket *)),this,SLOT(napsterClose(KSocket *)));

     if(p.newuser) { // we were a new user - but not now!!
#ifdef KDE2
        KGlobal::config()->writeEntry("newuser",false);
#else
        kapp->getConfig()->writeEntry("newuser",false);
#endif
     }
  }
  else { // no connection made
     statusBar()->message(i18n("Not connected"));
  }
}

// read socket slot for the napster connection
void Knapster::napsterRead(KSocket *s)
{
  NAPBLOCKPTR blk;
  SongInfo *tmpinfo;

  // read pending block
  blk=connection.readBlock();

#ifdef DEBUG_4
  if(blk->type!=NAP_ERROR) writeConsole( QString("** read [")+ QString().setNum(blk->type) +
       QString("] size: ") + QString().setNum(blk->size)+ "=" + blk->data );
#endif

  // send the data wherever it has to go
  switch(blk->type) {
    case NAP_LOGIN_RESULT:
       break;
    case NAP_SERVER_STATS:
       showServerStats(blk);
       break;
    case NAP_UPLOAD_REQ:
       // add the item to the uploadlist
#ifdef DEBUG_1
       cerr<<"someone has requested a song!! "<<blk->data<<endl<<flush;
#endif
       processUploadRequest(blk->data);
       break;
    case NAP_PING_REQ:
#ifdef DEBUG_1
       cerr<<"someone pinged us!!\n"<<flush;
#endif
       connection.writeBlock(NAP_PING_RESULT,blk->data,blk->size);
       break;
    case NAP_PING_RESULT:
         processPingResult(blk->data);
       break;
    case NAP_USER_MSG:  // got an incomming message
         processUserMessage(blk->data);
       break;
    case NAP_WHOIS_RESULT:
       processWhoisResult(blk->data);
       break;
    case NAP_WHOIS_NOTLOGGEDON:
       processWhoisOffline(blk->data);
       break;
    case NAP_CHANLIST_RESULT:
       writeConsole(QString(i18n("channel: ")) + blk->data );
       break;
    case NAP_CHANLIST_REQ:
       writeConsole(i18n("End of channel list"));
       break;
    case NAP_CHAN_USERINFO:
    case NAP_CHANUSERS_RESULT:
       writeConsole(QString(i18n("user: ")) + blk->data );
       break;
    case NAP_CHAN_USERJOINED:
         processChanUserJoined(blk->data);
       break;
    case NAP_CHAN_USERLEFT:
         processChanUserLeft(blk->data);
       break;
    case NAP_CHAN_TOPIC:
       writeConsole(QString(i18n("topic for ")) + blk->data );
       break;
    case NAP_CHAN_MSG:
         processChanMessage(blk->data);
       break;
    case NAP_JOINCHAN_RESULT: // joined OK??
       writeConsole(QString(i18n("Joined channel ")) + blk->data);
       break;
    case NAP_SEARCH_RESULT:
       if(blk->size>0) {
         tmpinfo=parseSearchBuffer(blk->data);
         if(tmpinfo) searchpanel->addSearchItem(tmpinfo);
       }
       break;
    case NAP_BROWSE_RESULT:
       if(blk->size>0) {
         tmpinfo=parseBrowseBuffer(blk->data);
         if(tmpinfo) searchpanel->addSearchItem(tmpinfo);
       }
       break;
    case NAP_SEARCH_COMPLETE:
    case NAP_BROWSE_COMPLETE:
       writeConsole(i18n("Finished Search"));
       statusBar()->changeItem(i18n("Search finished"),0);
       break;
    case NAP_DOWNLOAD_RESULT:
       if(blk->size>0) {
         tmpinfo=parseDownloadBuffer(blk->data);
         if(tmpinfo) addDownloadItem(tmpinfo);
       }
       break;
    case NAP_REMOTEQUEUEFULL:
       writeConsole(QString(i18n("User remote queue full: ")) + blk->data);
       if(blk->size>0) {
         tmpinfo=parseRemoteQueueBuffer(blk->data);
         if(tmpinfo) delDownloadItem(tmpinfo);
       }
       break;
    case NAP_DOWNLOAD_ERROR:
// REMOVE from download list???????????
       writeConsole(QString(i18n("Napster server error downloading ")) + blk->data);
       break;
    case NAP_SYSTEM_MSG:
       writeConsole(blk->data);
       break;
    case NAP_BROWSE_ERROR:
       writeConsole(QString(i18n("Browse error: ")) + blk->data );
       break;
    case NAP_ERROR_MSG:
       writeConsole(QString("** ") + blk->data );
       break;

    default:
#ifdef DEBUG_3
         if(blk->type!=NAP_ERROR) writeConsole( QString(i18n("Unknown blk ["))+ QString().setNum(blk->type) +
               QString("] size: ") + QString().setNum(blk->size)+ "=" + blk->data );
#endif
      break;
  }
}

// close socket slot for the napster connection
void Knapster::napsterClose(KSocket *s)
{
  writeConsole(i18n("Napster connection has been closed"));

  // make sure it was disconnected properly
  connection.Disconnect();
  // remove listener
  if(listener) {
    delete listener;
    listener=NULL;
  }

  statusBar()->message(i18n("Not connected"));
}

// function called by napster read on getting a server stats block
// -> get the counts and display in status bar
void Knapster::showServerStats(NAPBLOCKPTR n)
{
  int libs,songs,gigs;
  sscanf(n->data,"%d %d %d",&libs,&songs,&gigs);

  statusBar()->changeItem( QString().setNum(songs) + QString(i18n(" Songs in "))
    + QString().setNum(libs) + QString(i18n(" Libraries. Total ")) + QString().setNum(gigs) + QString(i18n(" Gig.")) ,1 );
}

// slot for downlaod terminate item signal
void Knapster::downloadTerminate(QListViewItem *it)
{
  if(it) {
#ifdef DEBUG_1
    cerr<<"trying to kill "<<it->text(DLV_FILE)<<endl<<flush;
#endif
    SongInfo *sg;
    // find it
    for(sg=downloadlist.first();sg!=NULL;sg=downloadlist.next()) {
      if(sg->md5.findRev(it->text(DLV_XTRA))==0) {
         // found match
         if(sg->dload) {
             sg->dload->cancel();
         }
         break;
      }
    }
  }
}

// slot for downlaod popupmenu terminate item

void Knapster::playsong(const char *name)
{
  Preferences p=PropertiesDlg::readPrefs();

  if(name) {
    // no player -> set default
    if(p.mp3player.length()<=0) p.mp3player="mpg123";
#ifdef DEBUG_1
    cerr<<"trying to play "<<name<<endl<<flush;
#endif
    QString cmdarg;
    QString tmp = p.mp3player + " \"" + name + "\"";

    if(player.isRunning()) {
      player.kill();
      return;
    }

    player.clearArguments();

    cmdarg = extractString(tmp);
    while(cmdarg.length()>0) {
      player << (const char *)cmdarg;
      cmdarg = extractString(tmp);
    }
    cerr<<"start play!!\n"<<flush;
    if(!player.start( /*KProcess::DontCare*/ /*,KProcess::AllOutput*/))
         cout<<"error running mp3player!!!\n"<<flush;
  }
  return;
}

void Knapster::playsong_dl(const char *name)
{
  Preferences p=PropertiesDlg::readPrefs();

  if(name) {

#ifdef DEBUG_1
    cerr<<"looking for song to play "<<name<<endl<<flush;
#endif

    // find the matching song in list and get the local filename for it
    SongInfo *sg;

    QString cmpname = getFileName(name);
    for(sg=downloadlist.first();sg!=NULL;sg=downloadlist.next()) {
      if(getFileName(sg->filename)==cmpname && sg->dload) {
          break;
      }
    }
    // no song found
    if(sg==NULL) return;

    // no player -> set default
    if(p.mp3player.length()<=0) p.mp3player="mpg123";
#ifdef DEBUG_1
    cerr<<"trying to play "<<sg->dload->getLocalName()<<endl<<flush;
#endif
    QString cmdarg;
    QString tmp = p.mp3player + " \"" + sg->dload->getLocalName() + "\"";

    if(player.isRunning()) {
      player.kill();
      return;
    }

    player.clearArguments();

    cmdarg = extractString(tmp);
    while(cmdarg.length()>0) {
      player << (const char *)cmdarg;
      cmdarg = extractString(tmp);
    }
    cerr<<"start play!!\n"<<flush;
    if(!player.start( /*KProcess::DontCare*/ /*,KProcess::AllOutput*/))
         cout<<"error running mp3player!!!\n"<<flush;
  }
  return;
}

// slot for user selction of a item to download from the searchpanel
void Knapster::searchDownload(QListViewItem *it)
{ // HELPME
  if(it) {
     connection.download(it->text(6),it->text(5)); // filename, user
     writeConsole(QString(i18n("Requesting song ")) + it->text(0));
#ifdef DEBUG_1
   cout<<"In searchDownload()------------------\n"<<endl<<flush;
#endif

  }
  return;
}

// func called to delete a downlaod item on recieval of REMOTE QUEUE full
void Knapster::delDownloadItem(SongInfo *tmpinfo)
{
#ifdef DEBUG_1
  cout<<"In download 620 error------------------------\n"<<endl<<flush;
#endif
  // find in the list and in the listview and delete it

//FIND
  SongInfo *sg;
  for(sg=downloadlist.first();sg!=NULL;sg=downloadlist.next()) {
    if(sg->filename==tmpinfo->filename && sg->user==tmpinfo->user) {

#ifdef DEBUG_1
        cerr<<"$$$$$$$$ removing "<<tmpinfo->filename
             <<" because of 620 msg ***************\n"<<flush<<endl;
#endif
        QString tmpstr;
//JADE        tmpstr.sprintf(i18n("** Couldn't download %s from %s because their download queue is full\n"), sg->filename, sg->user);
        tmpstr = i18n("** Couldn't download ") + sg->filename + i18n(" from ") + sg->user + i18n(" because their download queue is full");

        writeConsole(tmpstr.data());

        downloadpanel->removeDownloadItem(sg);
        if(sg->dload) delete sg->dload;
        downloadlist.remove(sg);

//        // tell server we are finished downloading a file
//        connection.download_finished_signal();
      return;
    }
  }
  return;
}


// function called on recieval of a download result block
void Knapster::addDownloadItem(SongInfo *tmpinfo)
{
#ifdef DEBUG_1
  cout<<"In add down load item------------------------\n"<<endl<<flush;
#endif

  Preferences p=PropertiesDlg::readPrefs();

  if(tmpinfo->port==0) { // download 2 type of download
    if(connection.getListener()==-1) {
       QString tmpstr;
//JADE       tmpstr.sprintf(i18n("** Can't download %s from %s as there is no listener connection."), tmpinfo->filename, tmpinfo->user);
       tmpstr =i18n("** Can't download ") + tmpinfo->filename + i18n("from") + tmpinfo->user + i18n(" as there is no listener connection.");

       writeConsole(tmpstr.data());
       return;
     }

  }

  // if the file is there give a bit of a message and possible options
  if(getFileSize(makeFullFilePath(p.downloaddir,getFileName(tmpinfo->filename)))!=0)
  {
    // file is already here
#ifdef KDE2
      int reply = KMessageBox::warningYesNoCancel(this,"Knapster " VERSION ,
         QString(i18n("You are about download file \n")) + getFileName(tmpinfo->filename) +
                i18n("\n over an existing file.\nThis file can be overwriiten or a resume can be attempted.\n")+
                i18n(" - note that the bitrate/frequency should match for the song to resume properly."),
         i18n("Resume"),i18n("Overwrite"));
#else
      int reply = KMsgBox::yesNoCancel(this,"Knapster " VERSION ,
         QString(i18n("You are about download file \n")) + getFileName(tmpinfo->filename) +
                i18n("\n over an existing file.\nThis file can be overwriiten or a resume can be attempted.\n")+
                i18n(" - note that the bitrate/frequency should match for the song to resume properly."),
             KMsgBox::EXCLAMATION,
         i18n("Resume"),i18n("Overwrite"),i18n("Cancel"));
#endif /* KDE2 */

      if(reply==3) return; // aborted
      if(reply==2) {
           // overwrite - kill the old file
           QFile::remove(makeFullFilePath(p.downloaddir,getFileName(tmpinfo->filename)));
           // TODO: test if it did ************
      }
  }

//  }

  // create & add the download to the queue
  NapsterDownload *dload=connection.startDownload(tmpinfo->ip,tmpinfo->port,tmpinfo->filename,
      p.username,p.speed,makeFullFilePath(p.downloaddir,getFileName(tmpinfo->filename)));

  if(dload) {

    QString tmpstr;
//JADE    tmpstr.sprintf(i18n("Queueing download of %s from %s"), tmpinfo->filename, tmpinfo->user);
// doesn't like the imbedded %s in the string
    tmpstr =  i18n("Queueing download of ") + tmpinfo->filename + " from " + tmpinfo->user;
    writeConsole(tmpstr.data());

    if(downloadpanel->numItems()==0) dloadval=0;

    dloadval++;  // get next id

    SongInfo *song=new SongInfo(*tmpinfo);

    if(!song) return; // out of memory ???????????????????

    song->md5 = tmpinfo->md5 + ":" + QString().setNum(dloadval); // save unique value
    song->dload=dload;  // save ptr to the download

    downloadpanel->addDownloadItem(song);

    downloadlist.append(song);

   }
   else {
     QString tmpstr;
//JADE     tmpstr.sprintf(i18n("Couldn't connect to %s to download %s"), tmpinfo->user, tmpinfo->filename);
     writeConsole(tmpstr.data());
   }
   return;
}

// update the download list - also checks for pending downloads
// called via timer
void Knapster::timeout()
{
  // send pending shared songs
  sendSharedSongs();

  // check for pending downloads/uploads
  checkForConnections();

  // update the lists
  updateDownloadList();

  updateUploadList();

  return;
}

void Knapster::updateDownloadList()
{
  int activeconnections=0;

  // update dloadlist
  SongInfo *sg;
  for(sg=downloadlist.first();sg!=NULL;sg=downloadlist.next()) {
    if(sg->dload && !sg->ignore) {
      sg->dload->getInfo(&sg->status,&sg->read,&sg->size,&sg->bps);

      // update the shown list
      downloadpanel->updateDownloadItem(sg);

      if(sg->status==NapsterDownload::DL_FINISHED) {
        if((PropertiesDlg::readPrefs()).beep) kapp->beep();
        sg->ignore=true;
        writeConsole(i18n("Finished download of ") + sg->filename );

        // delete
        downloadpanel->removeDownloadItem(sg);
        delete sg->dload;
        downloadlist.remove(sg);

        // tell server we are finished downloading a file
        connection.download_finished_signal();

      }
      else if(sg->status==NapsterDownload::DL_KILLED) {
        if((PropertiesDlg::readPrefs()).beep) kapp->beep();
        sg->ignore=true;
        writeConsole(i18n("User terminated download of ") + sg->filename );

        // delete
        downloadpanel->removeDownloadItem(sg);
        delete sg->dload;
        downloadlist.remove(sg);

        // tell server we are finished downloading a file
        connection.download_finished_signal();

      }
      else if(sg->status==NapsterDownload::DL_NOCONNECT) {
        if((PropertiesDlg::readPrefs()).beep) kapp->beep();
        sg->ignore=true;
        QString tmpstr;
//JADE        tmpstr.sprintf(i18n("Couldn't connect to %s to start download of %s"), sg->user, sg->filename);
        tmpstr = i18n("Couldn't connect to ") + sg->user + i18n(" to start download of ") + sg->filename;

        writeConsole(tmpstr.data());

        // delete
        downloadpanel->removeDownloadItem(sg);
        delete sg->dload;
        downloadlist.remove(sg);

        // tell server we are finished downloading a file
        connection.download_finished_signal();

      }
      else if(sg->status==NapsterDownload::DL_NOLOCALFILE) {
        if((PropertiesDlg::readPrefs()).beep) kapp->beep();
        sg->ignore=true;
        QString tmpstr;
//JADE        tmpstr.sprintf(i18n("Couldn't open local file to save %s from %s"), sg->filename, sg->user);
        tmpstr = i18n("Couldn't open local file to save ") + sg->filename + i18n(" from ") + sg->user;

        writeConsole(tmpstr.data());

        // delete
        downloadpanel->removeDownloadItem(sg);
        delete sg->dload;
        downloadlist.remove(sg);

        // tell server we are finished downloading a file
        connection.download_finished_signal();

      }
      else if(sg->status<0) { // one of the other error conditions
        if((PropertiesDlg::readPrefs()).beep) kapp->beep();
        sg->ignore=true;
        QString tmpstr;
// JADE        tmpstr.sprintf(i18n("Error occured during download of %s from %s"), sg->filename, sg->user);
        tmpstr = i18n("Error occured during download of ") + sg->filename + i18n(" from ") + sg->user;

        writeConsole(tmpstr.data());
//JADE        tmpstr.sprintf(i18n("** read %d of %d bytes."), sg->read, sg->size);
        tmpstr = i18n("** read ") + QString().setNum(sg->read) +
                QString(" / ") + QString().setNum(sg->size) + i18n(" bytes");
        writeConsole(tmpstr.data());

        downloadpanel->removeDownloadItem(sg);
        delete sg->dload;
        downloadlist.remove(sg);

        // tell server we are finished downloading a file
        connection.download_finished_signal();

      }
      else if(sg->status!=NapsterDownload::DL_QUEUED) {
         // an INIT or DL_DOWNLOAD
         activeconnections++;
      }
    }
  }

  // can we start up another??
  int maxdl=(PropertiesDlg::readPrefs()).maxdownloads;

  if((maxdl<=0) || (maxdl>0 && maxdl>activeconnections)) {
    // startup a queued connection
    for(sg=downloadlist.first();sg!=NULL;sg=downloadlist.next()) {
      if(sg->dload && sg->status==NapsterDownload::DL_QUEUED) {
         startDownload(sg);
         activeconnections++;
         if(maxdl>0 && maxdl<=activeconnections) break;
      }
    }
  }
  return;
}
void Knapster::updateUploadList()
{

  // update uploadlist
  SongInfo *sg;
  for(sg=uploadlist.first();sg!=NULL;sg=uploadlist.next()) {
    if(sg->dload && !sg->ignore) {
      sg->dload->getInfo(&sg->status,&sg->read,&sg->size,&sg->bps);

      // update the shown list
      uploadpanel->updateUploadItem(sg);

      if(sg->status==NapsterUpload::DL_FINISHED) {
        //if((PropertiesDlg::readPrefs()).beep) kapp->beep();
        sg->ignore=true;
        writeConsole(i18n("Finished upload of ") + sg->filename );

        // delete
        uploadpanel->removeUploadItem(sg);
        delete sg->dload;
        uploadlist.remove(sg);

        // tell server we are finished uploading a file
        connection.upload_finished_signal();

      }
      else if(sg->status==NapsterUpload::DL_KILLED) {
        //if((PropertiesDlg::readPrefs()).beep) kapp->beep();
        sg->ignore=true;
        writeConsole(i18n("User terminated upload of ") + sg->filename );

        // delete
        uploadpanel->removeUploadItem(sg);
        delete sg->dload;
        uploadlist.remove(sg);

        // tell server we are finished uploading a file
        connection.upload_finished_signal();

      }
      else if(sg->status<0) { // one of the other error conditions
//        if((PropertiesDlg::readPrefs()).beep) kapp->beep();
        sg->ignore=true;
        QString tmpstr;
//JADE        tmpstr.sprintf(i18n("Error occured during upload of %s from %s"),sg->filename, sg->user);
        tmpstr = i18n("Error occured during upload ") + sg->filename + i18n(" from ") + sg->user;
        writeConsole(tmpstr.data());

        uploadpanel->removeUploadItem(sg);
        delete sg->dload;
        uploadlist.remove(sg);

        // tell server we are finished downloading a file
        connection.upload_finished_signal();

      }
    }
  }

  return;
}

bool Knapster::startDownload(SongInfo *song)
{
  if(song->port!=0) {  // dload 1
    QString tmpstr;
//JADE    tmpstr.sprintf(i18n("Starting download of %s from %s") ,song->filename, song->user);
    tmpstr = i18n("Starting download of ") + song->filename + i18n(" from ") + song->user;
    writeConsole(tmpstr.data());

    song->dload->start();
    // tell server we are downloading a file
    connection.download_started_signal();
  }
  else {
    QString s = QString(song->user)+ QString(" \"") + song->filename + QString("\"");

    connection.writeBlock(NAP_DOWNLOAD_REQ_NEW, s,s.length());  // ask for new type dload
#ifdef DEBUG_1
    cerr<<"XXXX sending new commands of "<<s<<endl<<flush;
#endif
    QString tmpstr;
    tmpstr = i18n("Waiting for ") + song->filename + i18n(" to connect...");

    // tell the world that we have started initialising it
    song->dload->setInit();

    // tell server we are downloading a file
    // should do it here really, since the file isn't being downloaded yet,
    // but doing it here makes it easier to ensure that the finish signal is
    // send for each matching start
    // ie: if we waited for the real start and someone terminated the dload
    // before we may not be sure if a start signal was sent or not
    connection.download_started_signal();
  }
  return true;
}

void Knapster::loadMP3LibraryPaths()
{
  // clear the libview in case thier was something there
  librarypanel->clear();

  Preferences p = PropertiesDlg::readPrefs();
  writeConsole(i18n("Loading local mp3's from preferences..."));

  char *str;
  QString path;
  bool share;
  int depth;

  for(str=p.pathlist.first();str!=NULL;str=p.pathlist.next()) {
     // get out the parts
     if(parsePathItem(str,path,&share,&depth)) {
        librarypanel->addDir(path,share,depth);
     }
  }

  writeConsole(i18n("done"));
}

// get out the path,shared,depth from a string in the form
// path|shared|depth
bool Knapster::parsePathItem(const QString &str,QString &path,bool *share,int *depth)
{
  QString temp=str;
  QString t1;
  int pos;
  pos=temp.find('|');
  if(pos==-1) return false;
  path=temp.left(pos);
  temp=temp.right(temp.length()-(pos+1));

  pos=temp.find('|');
  if(pos==-1) return false;
  t1=temp.left(pos);
  if(t1=="Yes") *share=true;
  else *share=false;

  temp=temp.right(temp.length()-(pos+1));
  *depth=temp.toInt();

  return true;
}
