/*************************************************/
/* methods for class MainWidget                  */
/*                                               */
/* main window of this application               */
/*   widget management                           */
/*                                               */
/* Andreas Rostin                                */
/* 27.10.97                                      */
/*************************************************/
#include <kapp.h>
#include <kfiledialog.h>
#include <kmainwindow.h>
#include <kstddirs.h>
#include <kmenubar.h>
#include <kpopupmenu.h>

#include <qwidgetfactory.h>
#include <qpainter.h>
#include <qkeycode.h>
#include <qprinter.h>
#include <qprintdialog.h>
#include <qfileinfo.h>
#include <qmessagebox.h>
#include <qstringlist.h>
#include <qslider.h>
#include <qcombobox.h>

#include "xwire.h"
#include "mainw.h"
#include "netw.h"
#include "klogic.h"
#include "klogicIO.h"
#include "version.h"
#include "xdevice.h"
#include "clientw.h"
#include "simw.h"
#include "propGlob.h"
#include "dlgLib.h"
#include "devLib.h"
#include "deviceTypes.h"

#include "mainw.moc"

/***************************************************/
/* static methods of class MainWidget              */
/***************************************************/
QString MainWidget::PATH;
QString MainWidget::TBPATH;
int MainWidget::mode = MAIN_MODE_DRAW;
DevLibrary * MainWidget::activeLib = (DevLibrary *)NULL;

// private static
void MainWidget::setPATH(KApplication *app)
{	QPixmap pix;

	QString klogic_path(getenv("KLOGIC_PATH"));

	if (!klogic_path.isNull()) {
		// path comes from environment
		warning("using local directories: [remove KLOGIC_PATH] try \"make install\"");
		PATH = klogic_path + "/pics/";
		TBPATH = klogic_path + "/toolbar/";
		if (FALSE == pix.load(PATH + "AND.xpm"))
			fatal("***** aborting ***** : you should install klogic with 'make install` - or set $KLOGIC_PATH");
	} else {
		// path comes from kde
		bool found = false;
		KStandardDirs *stdDir = app->dirs();
		QStringList sl = stdDir->findDirs("data", "klogic");
		QStringList::ConstIterator it;
		for( it = sl.begin(); !found && it != sl.end(); ++it ) {
			PATH = (*it) + "pics/";
			TBPATH = (*it) + "toolbar/";
			if (TRUE == pix.load(PATH + "AND.xpm")) found = true;
		}
		if (!found)
			fatal("***** aborting ***** : cannot find my icons ... (KDE path settings ...!?!?)");
	}
}

/***************************************************/
/* methods of class MainWidget                     */
/***************************************************/
MainWidget::MainWidget(KApplication *_app, const char *file)
	:KMainWindow(NULL)
{
	app = _app;
	setPATH(app);

	setBackgroundMode( PaletteBase );    // set widgets background
	resize(Global::Screen::SCREEN_SIZE_X, Global::Screen::SCREEN_SIZE_Y);                  // set default size in pixels
	setPlainCaption(Global::Klogic::Name + " - " + Global::Klogic::emptyFileName);

	// read config
	readProperties(app->config());

	// load device pixmaps
	XDevice::loadPix();

	// initial net
	ClientW *client_widget = new ClientW(this, file);
	client_widgets.Append(client_widget);
	if (!(main_netwidget = client_widget->net())) fatal("unable to create inital circuit..\n");

	if (file) {
		Global::CurrentCircuit::filename = file;
		QFileInfo fi(Global::CurrentCircuit::filename);
		setPlainCaption(Global::Klogic::Name + " - " + fi.fileName());
	}

	client_widget->show();
	connect(main_netwidget, SIGNAL(netChanged(int)), this, SLOT(netChange(int)));
	connect(main_netwidget, SIGNAL(createNewWidget(Circuit *)), this, SLOT(newClient(Circuit *)));
	connect(main_netwidget, SIGNAL(netDeleted(Circuit *)), this, SLOT(netDelete(Circuit *)));
	connect(main_netwidget, SIGNAL(netChanged(int)), client_widget, SLOT(netChange(int)));

	initToolbar();
	setToolbarPixmap();

	initMenubar();
	setCentralWidget(client_widget);

	scrollSimW = new ScrollableSimw(Global::Klogic::Name + " - " + i18n("oscillograph"));
	simW = new SimW(scrollSimW->getClientParent(), scrollSimW, main_netwidget->getActive());
	scrollSimW->setClient(simW);
	connect(scrollSimW, SIGNAL(simWidgetGoesHidden()), this, SLOT(simWidgetHidden()));

	connect(main_netwidget, SIGNAL(graphChanged()), simW, SLOT(graphChange()));
	connect(main_netwidget, SIGNAL(simStepped()), simW, SLOT(simStep()));
	connect(main_netwidget, SIGNAL(showGraph(Circuit *)), simW, SLOT(changeNet(Circuit *)));
	connect(main_netwidget, SIGNAL(addToLib(XDevice *)), this, SLOT(addToLib(XDevice *)));
	simW->setSimMode(main_netwidget->getSimMode());
	scrollSimW->hide();
	simw = 0;

	setMode(ID_DRAW);
}

MainWidget::~MainWidget()
{
}

void MainWidget::paintEvent(QPaintEvent *)
{
	ClientW *client_widget;
	KlogicList<ClientW> *lc = client_widgets.First();
	while(lc) {
		client_widget = lc->Get();
		if (client_widget->isActiveWindow())
			client_widget->repaint(FALSE);
		lc = lc->Next();
	}
}

bool MainWidget::queryClose()
{
	if (main_netwidget->hasChanged()) {
		int ret;
		ret = QMessageBox::warning(this, i18n("circuit has changed.."),
			i18n("Save current changes?"),
			QMessageBox::Yes,
			QMessageBox::No,
			QMessageBox::Cancel);
		if (QMessageBox::Yes == ret) {
			menuCallback(ID_SAVE_AS);
			return TRUE;
		}
		if (QMessageBox::No == ret) {
			return TRUE;
		}
		return FALSE;
	}
	return TRUE;
}

void MainWidget::initToolbar()
{	QPixmap px;

	gate_toolbar = new KToolBar(this);
	px.load(TBPATH + "tbDraw.xpm");
	gate_toolbar->insertButton(px, ID_DRAW, SIGNAL(clicked()),
			this, SLOT(modeDraw()), TRUE, i18n("draw wires"), -1);
	gate_toolbar->setToggle(ID_DRAW, TRUE);
	gate_toolbar->setButton(ID_DRAW, TRUE);

	px.load(TBPATH + "tbBus.xpm");
	gate_toolbar->insertButton(px, DeviceType::fBUS, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, i18n("Bus Connector"), -1);
	gate_toolbar->setToggle(DeviceType::fBUS, TRUE);

	px.load(TBPATH + "tbSel.xpm");
	gate_toolbar->insertButton(px, ID_SEL, SIGNAL(clicked()),
			this, SLOT(modeSelect()), TRUE, i18n("select groups"), -1);
	gate_toolbar->setToggle(ID_SEL, TRUE);

	gate_toolbar->insertSeparator();
	gate_toolbar->insertSeparator();
	gate_toolbar->insertSeparator();
	gate_toolbar->insertSeparator();

	px.load(TBPATH + "tbNET.xpm");
	gate_toolbar->insertButton(px, DeviceType::fNET, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, i18n("new sub-circuit"), -1);
	gate_toolbar->setToggle(DeviceType::fNET, TRUE);

	px.load(TBPATH + "tbFx.xpm");
	gate_toolbar->insertButton(px, DeviceType::fEQU, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, i18n("equation device"), -1);
	gate_toolbar->setToggle(DeviceType::fEQU, TRUE);

	px.load(TBPATH + "tbAND.xpm");
	gate_toolbar->insertButton(px, DeviceType::fAND, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, i18n("AND gate"), -1);
	gate_toolbar->setToggle(DeviceType::fAND, TRUE);

	px.load(TBPATH + "tbOR.xpm");
	gate_toolbar->insertButton(px, DeviceType::fOR, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, i18n("OR gate"), -1);
	gate_toolbar->setToggle(DeviceType::fOR, TRUE);

	px.load(TBPATH + "tbINV.xpm");
	gate_toolbar->insertButton(px, DeviceType::fONE, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, i18n("inverter"), -1);
	gate_toolbar->setToggle(DeviceType::fONE, TRUE);

	px.load(TBPATH + "tbXOR.xpm");
	gate_toolbar->insertButton(px, DeviceType::fXOR, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, i18n("XOR gate"), -1);
	gate_toolbar->setToggle(DeviceType::fXOR, TRUE);

	px.load(TBPATH + "tbRS.xpm");
	gate_toolbar->insertButton(px, DeviceType::fRS, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, i18n("RS flipflop"), -1);
	gate_toolbar->setToggle(DeviceType::fRS, TRUE);

	px.load(TBPATH + "tbD.xpm");
	gate_toolbar->insertButton(px, DeviceType::fDFF, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, i18n("D flipflop"), -1);
	gate_toolbar->setToggle(DeviceType::fDFF, TRUE);

	px.load(TBPATH + "tbJK.xpm");
	gate_toolbar->insertButton(px, DeviceType::fJK, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, i18n("JK flipflop"), -1);
	gate_toolbar->setToggle(DeviceType::fJK, TRUE);

	px.load(TBPATH + "tbRAM.xpm");
	gate_toolbar->insertButton(px, DeviceType::fRAM, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, i18n("RAM 32x8"), -1);
	gate_toolbar->setToggle(DeviceType::fRAM, TRUE);

	px.load(TBPATH + "tbTRI.xpm");
	gate_toolbar->insertButton(px, DeviceType::fTRI, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, i18n("Tristate output"), -1);
	gate_toolbar->setToggle(DeviceType::fTRI, TRUE);

	px.load(TBPATH + "tbOSC.xpm");
	gate_toolbar->insertButton(px, DeviceType::fOSC, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, i18n("Oscillator"), -1);
	gate_toolbar->setToggle(DeviceType::fOSC, TRUE);

	px.load(TBPATH + "tbSWI.xpm");
	gate_toolbar->insertButton(px, DeviceType::fSWI, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, i18n("Switch"), -1);
	gate_toolbar->setToggle(DeviceType::fSWI, TRUE);

	px.load(TBPATH + "tbSS.xpm");
	gate_toolbar->insertButton(px, DeviceType::fSS, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, i18n("7 Segment"), -1);
	gate_toolbar->setToggle(DeviceType::fSS, TRUE);

	px.load(TBPATH + "tbLED.xpm");
	gate_toolbar->insertButton(px, DeviceType::fLEDgreen, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, i18n("LED"), -1);
	gate_toolbar->setToggle(DeviceType::fLEDgreen, TRUE);

/*
	px.load(TBPATH + "tbMatrix.xpm");
	gate_toolbar->insertButton(px, DeviceType::fMATRIX, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, i18n("Matrix Display"), -1);
	gate_toolbar->setToggle(DeviceType::fMATRIX, TRUE);
*/

	px.load(TBPATH + "tbPWR.xpm");
	gate_toolbar->insertButton(px, DeviceType::fPWR, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, i18n("Static"), -1);
	gate_toolbar->setToggle(DeviceType::fPWR, TRUE);

	px.load(TBPATH + "tbTXT.xpm");
	gate_toolbar->insertButton(px, DeviceType::fTXT, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, i18n("Text"), -1);
	gate_toolbar->setToggle(DeviceType::fTXT, TRUE);

	px.load(TBPATH + "tbIN.xpm");
	gate_toolbar->insertButton(px, DeviceType::fIN, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, i18n("Input"), -1);
	gate_toolbar->setToggle(DeviceType::fIN, TRUE);
	gate_toolbar->hideItem(DeviceType::fIN);

	px.load(TBPATH + "tbOUT.xpm");
	gate_toolbar->insertButton(px, DeviceType::fOUT, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, i18n("Output"), -1);
	gate_toolbar->setToggle(DeviceType::fOUT, TRUE);
	gate_toolbar->hideItem(DeviceType::fOUT);

	gate_toolbar->setBarPos(KToolBar::Top);
	gate_toolbar->enable(KToolBar::Show);

	addToolBar(gate_toolbar);
	gate_toolbar->show();

	// -----------------------------------------------------------------------------------
	// -----------------------------------------------------------------------------------
	// toolbar: library chooser
	lib_toolbar = new KToolBar(this);

	QLabel *lLibrary = new QLabel(lib_toolbar, "ID_LIB_CHOOSER");
	lLibrary->setText("   " + i18n("Library") + ": ");
	lib_toolbar->insertWidget(0, 40, lLibrary);

	cbLibrary = new QComboBox(lib_toolbar);
	cbLibrary->setFixedWidth(140);
	connect(cbLibrary, SIGNAL(activated(int)), SLOT(libraryChanged(int)));
	lib_toolbar->insertWidget(ID_LIB_CHOOSER, 140, (QWidget *)cbLibrary);

	QLabel *lDevice = new QLabel(lib_toolbar);
	lDevice->setText("    " + i18n("Device") + ": ");
	lib_toolbar->insertWidget(0, 40, lDevice);

	// toolbar: library device chooser
	cbLibDev = new QComboBox(lib_toolbar, "ID_LIBDEV_CHOOSER");
	cbLibDev->setFixedWidth(140);
	connect(cbLibDev, SIGNAL(activated(int)), SLOT(libraryDeviceChanged(int)));
	lib_toolbar->insertWidget(ID_LIBDEV_CHOOSER, 140, (QWidget *)cbLibDev);

	addToolBar(lib_toolbar);
	if (cbLibrary->count()) lib_toolbar->show();
	else lib_toolbar->hide();

	// -----------------------------------------------------------------------------------
	// -----------------------------------------------------------------------------------
	sim_toolbar = new KToolBar(this);

	px.load(TBPATH + "tbShow.xpm");
	sim_toolbar->insertButton(px, ID_SIM_SHOW, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, i18n("simulation graph"), -1);
	sim_toolbar->setToggle(ID_SIM_SHOW, TRUE);

	sim_toolbar->insertSeparator();
	sim_toolbar->insertSeparator();

	px.load(TBPATH + "tbStep.xpm");
	sim_toolbar->insertButton(px, NetWidget::MODE_SIM_STEP, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, i18n("single step simulation"), -1);
	sim_toolbar->setToggle(NetWidget::MODE_SIM_STEP, TRUE);

	px.load(TBPATH + "tbSim.xpm");
	sim_toolbar->insertButton(px, NetWidget::MODE_SIM_MULT, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, i18n("simulate"), -1);
	sim_toolbar->setToggle(NetWidget::MODE_SIM_MULT, TRUE);
	sim_toolbar->setButton(NetWidget::MODE_SIM_MULT, TRUE);

	sim_toolbar->insertSeparator();
	sim_toolbar->insertSeparator();

	QLabel *speed_text = new QLabel(sim_toolbar);
	speed_text->setFixedSize(50, 20);
	speed_text->setText(i18n("speed"));

	speed_slider = new QSlider(QSlider::Horizontal, sim_toolbar);
	speed_slider->setFixedSize(100, 20);
	speed_slider->setRange(NetWidget::MIN_SIMSTEPTIME, NetWidget::MAX_SIMSTEPTIME);
	speed_slider->setValue(NetWidget::MAX_SIMSTEPTIME - NetWidget::simTime());
	connect(speed_slider, SIGNAL(sliderMoved(int)), this, SLOT(simTimeSliderChanged(int)));

	sim_toolbar->setBarPos(KToolBar::Top);
	sim_toolbar->enable(KToolBar::Show);

	addToolBar(sim_toolbar);
	sim_toolbar->show();
	tool_visible = 1;

}

// exchange toolbar pixmaps depending on the used device symbolset
void MainWidget::setToolbarPixmap()
{	QPixmap px;
	int symbolset = XDevice::getSymbolSet();

	if (symbolset == XDevice::SYMBOLSET_DIN40900) {
		px.load(TBPATH + "tbAND.xpm");
		gate_toolbar->setButtonPixmap(DeviceType::fAND, px);
		px.load(TBPATH + "tbOR.xpm");
		gate_toolbar->setButtonPixmap(DeviceType::fOR, px);
		px.load(TBPATH + "tbXOR.xpm");
		gate_toolbar->setButtonPixmap(DeviceType::fXOR, px);
		px.load(TBPATH + "tbINV.xpm");
		gate_toolbar->setButtonPixmap(DeviceType::fONE, px);
	} else {
		px.load(TBPATH + "tbIECAND.xpm");
		gate_toolbar->setButtonPixmap(DeviceType::fAND, px);
		px.load(TBPATH + "tbIECOR.xpm");
		gate_toolbar->setButtonPixmap(DeviceType::fOR, px);
		px.load(TBPATH + "tbIECXOR.xpm");
		gate_toolbar->setButtonPixmap(DeviceType::fXOR, px);
		px.load(TBPATH + "tbIECINV.xpm");
		gate_toolbar->setButtonPixmap(DeviceType::fONE, px);
	}
}

void MainWidget::initMenubar()
{
	menubar = menuBar();
	CHECK_PTR(menubar);

	pm_file = new QPopupMenu;
	pm_file->insertItem(i18n("&New..."), ID_NEW);
	pm_file->insertItem(i18n("&Open..."), ID_OPEN);
	pm_file->setAccel(Key_F4, ID_OPEN);
	pm_file->insertSeparator();
	pm_file->insertItem(i18n("&Save"), ID_SAVE);
	pm_file->setAccel(Key_F5, ID_SAVE);
	pm_file->insertItem(i18n("Save &As..."), ID_SAVE_AS);
	pm_file->setAccel(Key_F6, ID_SAVE_AS);
	pm_file->insertSeparator();
	pm_file->insertItem(i18n("&Library Maintenance .."), ID_LIBRARY_MAIN);
	pm_file->setAccel(Key_F7, ID_LIBRARY_MAIN);
	pm_file->insertSeparator();
	pm_file->insertItem(i18n("&Print..."), ID_PRINT);
	pm_file->setAccel(Key_F8, ID_PRINT);
	pm_file->insertSeparator();
	pm_file->insertItem(i18n("&Exit"), ID_EXIT);

	pm_edit = new QPopupMenu;
	pm_edit->insertItem(i18n("&Copy"), ID_COPY);
	pm_edit->insertItem(i18n("&Paste"), ID_PASTE);
	pm_edit->insertItem(i18n("C&ut"), ID_CUT);

	pm_logic = new QPopupMenu;
	pm_logic->insertItem(i18n("&Connection Mode"), ID_DRAW);
	pm_logic->setAccel(Key_C + CTRL, ID_DRAW);
	pm_logic->insertItem(i18n("&Selection Mode"), ID_SEL);
	pm_logic->setAccel(Key_G + CTRL, ID_SEL);
	pm_logic->insertSeparator();
	pm_logic->insertItem(i18n("&Equation"), DeviceType::fEQU);
	pm_logic->setAccel(Key_E + CTRL, DeviceType::fEQU);
	pm_logic->insertItem(i18n("&AND"), DeviceType::fAND);
	pm_logic->setAccel(Key_A + CTRL, DeviceType::fAND);
	pm_logic->insertItem(i18n("&OR"), DeviceType::fOR);
	pm_logic->setAccel(Key_O + CTRL, DeviceType::fOR);
	pm_logic->insertItem(i18n("&XOR"), DeviceType::fXOR);
	pm_logic->setAccel(Key_X + CTRL, DeviceType::fXOR);
	pm_logic->insertItem(i18n("&Inverter"), DeviceType::fONE);
	pm_logic->setAccel(Key_I+ CTRL, DeviceType::fONE);
	pm_logic->insertItem(i18n("&RS flipflop"), DeviceType::fRS);
	pm_logic->setAccel(Key_R+ CTRL, DeviceType::fRS);
	pm_logic->insertItem(i18n("&D flipflop"), DeviceType::fDFF);
	pm_logic->setAccel(Key_D+ CTRL, DeviceType::fDFF);
	pm_logic->insertItem(i18n("&JK flipflop"), DeviceType::fJK);
	pm_logic->setAccel(Key_J+ CTRL, DeviceType::fJK);
	pm_logic->insertItem(i18n("RA&M 32x8"), DeviceType::fRAM);
	pm_logic->setAccel(Key_M+ CTRL, DeviceType::fRAM);
	pm_logic->insertItem(i18n("Tristate output"), DeviceType::fTRI);
	pm_logic->setAccel(Key_3+ CTRL, DeviceType::fTRI);
	pm_logic->insertItem(i18n("Bus Connector"), DeviceType::fBUS);
	pm_logic->setAccel(Key_Y+ CTRL, DeviceType::fBUS);
	pm_logic->insertItem(i18n("Os&cillator"), DeviceType::fOSC);
	pm_logic->setAccel(Key_Z+ CTRL, DeviceType::fOSC);
	pm_logic->insertItem(i18n("&Switch"), DeviceType::fSWI);
	pm_logic->setAccel(Key_S+ CTRL, DeviceType::fSWI);
	pm_logic->insertItem(i18n("&7-Segment"), DeviceType::fSS);
	pm_logic->setAccel(Key_7+ CTRL, DeviceType::fSS);
	pm_logic->insertItem(i18n("&LED"), DeviceType::fLEDgreen);
	pm_logic->setAccel(Key_L+ CTRL, DeviceType::fLEDgreen);
/*
	pm_logic->insertItem(i18n("Matrix"), DeviceType::fMATRIX);
*/
	pm_logic->insertItem(i18n("&Static"), DeviceType::fPWR);
	pm_logic->setAccel(Key_P+ CTRL, DeviceType::fPWR);
	pm_logic->insertItem(i18n("&Text"), DeviceType::fTXT);
	pm_logic->setAccel(Key_T+ CTRL, DeviceType::fTXT);
	pm_logic->insertSeparator();
	pm_logic->insertItem(i18n("&new sub-circuit"), DeviceType::fNET);
	pm_logic->insertItem(i18n("&Input"), DeviceType::fIN);
	pm_logic->insertItem(i18n("&Output"), DeviceType::fOUT);
	pm_logic->setItemEnabled(DeviceType::fIN, FALSE);
	pm_logic->setItemEnabled(DeviceType::fOUT, FALSE);
	pm_logic->setCheckable(TRUE);
	pm_logic->setItemChecked(ID_DRAW, TRUE);

	pm_sim = new QPopupMenu;
	pm_sim->setCheckable(TRUE);
	pm_sim->insertItem(i18n("Si&mulate"), NetWidget::MODE_SIM_MULT);
	pm_sim->setAccel(Key_M, NetWidget::MODE_SIM_MULT);
	pm_sim->setItemChecked(NetWidget::MODE_SIM_MULT, TRUE);
	pm_sim->insertItem(i18n("&Step through"), NetWidget::MODE_SIM_STEP);
	pm_sim->setAccel(Key_S, NetWidget::MODE_SIM_STEP);
	pm_sim->insertSeparator();
	pm_sim->insertItem(i18n("Show &Graph"), ID_SIM_SHOW);
	pm_sim->setAccel(Key_G, ID_SIM_SHOW);
	pm_sim->insertSeparator();
	pm_sim->insertItem(i18n("Show &Equations"), ID_SIM_EQU);
	pm_sim->setAccel(Key_E, ID_SIM_EQU);

	pm_options = new QPopupMenu;
	pm_options->insertItem(i18n("&Global Settings "), ID_GLOBAL);
	pm_options->insertItem(i18n("&Hide Toolbar"), ID_TOOL_VISIBLE);
	pm_options->setCheckable(FALSE);

	pm_help=helpMenu();	// empty args: it uses global KAboutData
	CHECK_PTR(pm_help);
	pm_help->insertItem(i18n("maintenance"), ID_MAINTENANCE);

	menubar->insertItem(i18n("&File"), pm_file );
	menubar->insertSeparator();
	menubar->insertItem(i18n("&Edit"), pm_edit );
	menubar->insertSeparator();
	menubar->insertItem(i18n("&Logic"), pm_logic );
	menubar->insertSeparator();
	menubar->insertItem(i18n("&Simulation"), pm_sim);
	menubar->insertSeparator();
	menubar->insertItem(i18n("&Options"), pm_options);
	menubar->insertSeparator();
	menubar->insertItem(i18n("&Help"), pm_help);		
	menubar->show();

	connect (pm_file, SIGNAL (activated (int)), SLOT (menuCallback(int)));
	connect (pm_edit, SIGNAL (activated (int)), SLOT (menuCallback(int)));
	connect (pm_logic, SIGNAL (activated (int)), SLOT (menuCallback(int)));
	connect (pm_sim, SIGNAL (activated (int)), SLOT (menuCallback(int)));
	connect (pm_options, SIGNAL (activated (int)), SLOT (menuCallback(int)));
	connect (pm_help, SIGNAL (activated (int)), SLOT (menuCallback(int)));
}

//************* slots **************

void MainWidget::menuCallback(int id)
{	QPrinter prt;
	QPainter p;
	QFileInfo fi;
	QString new_filename;
	int i;
	bool save_as_sub;

	switch(id) {
		case DeviceType::fEQU:
		case DeviceType::fAND:
		case DeviceType::fOR:
		case DeviceType::fXOR:
		case DeviceType::fONE:
		case DeviceType::fRS:
		case DeviceType::fDFF:
		case DeviceType::fJK:
		case DeviceType::fRAM:
		case DeviceType::fTRI:
		case DeviceType::fBUS:
		case DeviceType::fSS:
		case DeviceType::fOSC:
		case DeviceType::fSWI:
		case DeviceType::fLEDgreen:
		case DeviceType::fMATRIX:
		case DeviceType::fIN:
		case DeviceType::fOUT:
		case DeviceType::fNET:
		case DeviceType::fPWR:
		case DeviceType::fTXT:
			devChange(id);
			break;
		case ID_OPEN:
			fi.setFile(Global::CurrentCircuit::filename);
			new_filename = KFileDialog::getOpenFileName(fi.dirPath(), "*.circuit");
			if (!new_filename.isEmpty()) {
				closeAllClients();
				i = main_netwidget->openNet(new_filename, circuit_changed);
				if (i > 0) {
					Global::CurrentCircuit::filename = new_filename;
					if (i == 1) {	// not a subcircuit
						fi.setFile(Global::CurrentCircuit::filename);
						setPlainCaption(Global::Klogic::Name + " - " + fi.fileName());
					}
				} else {
					Global::CurrentCircuit::filename = new_filename;
					fi.setFile(Global::CurrentCircuit::filename);
					setPlainCaption(Global::Klogic::Name + " - " + fi.fileName());
				}
			}
			break;
		case ID_SAVE:
			// save-command: always save toplevel circuit!
			fi.setFile(Global::CurrentCircuit::filename);
			if (fi.exists()) {
				main_netwidget->saveNet(Global::CurrentCircuit::filename, false);
				setPlainCaption(Global::Klogic::Name + " - " + fi.fileName());
			} else menuCallback(ID_SAVE_AS);
			break;
		case ID_SAVE_AS:
			// save-as-command: ask for saving sub circuit
			save_as_sub = main_netwidget->saveAsSubNet();
			fi.setFile(Global::CurrentCircuit::filename);
			new_filename = KFileDialog::getSaveFileName(fi.dirPath(), "*.circuit");
			if (!new_filename.isEmpty()) {
				main_netwidget->saveNet(new_filename, save_as_sub);
				Global::CurrentCircuit::filename = new_filename;
				fi.setFile(Global::CurrentCircuit::filename);
				setPlainCaption(Global::Klogic::Name + " - " + fi.fileName());
			}
			break;
		case ID_PRINT:
			if (TRUE == QPrintDialog::getPrinterSetup(&prt)) {
				p.begin(&prt);
				main_netwidget->printNet(&p);
				p.end();
			}
			break;
		case ID_NEW:
			closeAllClients();
			main_netwidget->newNet();
			fi.setFile(Global::CurrentCircuit::filename);
			Global::CurrentCircuit::filename = fi.dirPath() + "/" + Global::Klogic::emptyFileName;
			setPlainCaption(Global::Klogic::Name + " - " + Global::Klogic::emptyFileName);
			break;
		case ID_EXIT:
			if (queryClose())
				exit(0);
			break;
		case ID_COPY:
			getActive()->rmenuSelCallback(RPOPSEL_COPY);
			break;
		case ID_PASTE:
			getActive()->rmenuSelCallback(RPOPSEL_PASTE);
			break;
		case ID_CUT:
			getActive()->rmenuSelCallback(RPOPSEL_CUT);
			break;
		case ID_GLOBAL:
			dialogGlobal();
			break;
		case ID_LIBRARY_MAIN:
			dialogLibrary();
			break;
		case ID_TOOL_VISIBLE:
			setToolbar();
			break;
		case ID_DRAW:
			setMode(ID_DRAW);
			break;
		case ID_SEL:
			setMode(ID_SEL);
			break;
		case NetWidget::MODE_SIM_STEP:
			if (NetWidget::MODE_SIM_STEP != main_netwidget->getSimMode()) {
				simMode(id);
				speed_slider->setEnabled(false);
			} else {
				KlogicList<ClientW> *lc = client_widgets.First();
				while(lc) {
					lc->Get()->net()->simStep();
					lc = lc->Next();
				}
			}
			break;
		case NetWidget::MODE_SIM_MULT:
			if (NetWidget::MODE_SIM_MULT != main_netwidget->getSimMode()) {
				simMode(id);
				speed_slider->setEnabled(true);
			}
			break;
		case ID_SIM_SHOW:
			if (simw) {
				scrollSimW->hide();	// calls simWidgetHidden() via signal
			} else {
				pm_sim->changeItem(i18n("Hide &Graph"),ID_SIM_SHOW);
				sim_toolbar->setButton(ID_SIM_SHOW, TRUE);
				// display the circuit shown in the main widget
				simW->changeNet(main_netwidget->getActive());
				scrollSimW->show();
				simw = 1;
			}
			break;
		case ID_SIM_EQU:
			main_netwidget->showCircuitEquations();
			break;
		case ID_MAINTENANCE:
			fprintf(stdout, "              INSTANCES\n");
			fprintf(stdout, "-----------------------------------------\n");
			fprintf(stdout, "Circuit:\t%d\n", Circuit::instance);
			fprintf(stdout, "-----------------------------------------\n");
			fprintf(stdout, "XDevice:\t%d\n", XDevice::instance);
			fprintf(stdout, "Device:\t\t%d\n", Device::instance);
			fprintf(stdout, "Calculator:\t%d\n", Calculator::instance);
			fprintf(stdout, "Fifo:\t\t%d\n", Fifo::instance);
			fprintf(stdout, "XWire:\t\t%d\n", XWire::instance);
			fprintf(stdout, "-----------------------------------------\n");
			break;
		default:
			break;
	}
}

void MainWidget::turnAllCheckButton(int id)
{
	// set all buttons off
	gate_toolbar->setButton(ID_DRAW, FALSE);
	menubar->setItemChecked(ID_DRAW, FALSE);
	gate_toolbar->setButton(ID_SEL, FALSE);
	menubar->setItemChecked(ID_SEL, FALSE);
	gate_toolbar->setButton(DeviceType::fEQU, FALSE);
	menubar->setItemChecked(DeviceType::fEQU, FALSE);
	gate_toolbar->setButton(DeviceType::fAND, FALSE);
	menubar->setItemChecked(DeviceType::fAND, FALSE);
	gate_toolbar->setButton(DeviceType::fOR, FALSE);
	menubar->setItemChecked(DeviceType::fOR, FALSE);
	gate_toolbar->setButton(DeviceType::fXOR, FALSE);
	menubar->setItemChecked(DeviceType::fXOR, FALSE);
	gate_toolbar->setButton(DeviceType::fONE, FALSE);
	menubar->setItemChecked(DeviceType::fONE, FALSE);
	gate_toolbar->setButton(DeviceType::fSS, FALSE);
	menubar->setItemChecked(DeviceType::fSS, FALSE);
	gate_toolbar->setButton(DeviceType::fRS, FALSE);
	menubar->setItemChecked(DeviceType::fDFF, FALSE);
	gate_toolbar->setButton(DeviceType::fDFF, FALSE);
	menubar->setItemChecked(DeviceType::fRS, FALSE);
	gate_toolbar->setButton(DeviceType::fJK, FALSE);
	menubar->setItemChecked(DeviceType::fJK, FALSE);
	gate_toolbar->setButton(DeviceType::fRAM, FALSE);
	menubar->setItemChecked(DeviceType::fRAM, FALSE);
	gate_toolbar->setButton(DeviceType::fTRI, FALSE);
	menubar->setItemChecked(DeviceType::fTRI, FALSE);
	gate_toolbar->setButton(DeviceType::fBUS, FALSE);
	menubar->setItemChecked(DeviceType::fBUS, FALSE);
	gate_toolbar->setButton(DeviceType::fOSC, FALSE);
	menubar->setItemChecked(DeviceType::fOSC, FALSE);
	gate_toolbar->setButton(DeviceType::fSWI, FALSE);
	menubar->setItemChecked(DeviceType::fSWI, FALSE);
	gate_toolbar->setButton(DeviceType::fLEDgreen, FALSE);
	menubar->setItemChecked(DeviceType::fLEDgreen, FALSE);
	gate_toolbar->setButton(DeviceType::fMATRIX, FALSE);
	menubar->setItemChecked(DeviceType::fMATRIX, FALSE);
	gate_toolbar->setButton(DeviceType::fIN, FALSE);
	menubar->setItemChecked(DeviceType::fIN, FALSE);
	gate_toolbar->setButton(DeviceType::fOUT, FALSE);
	menubar->setItemChecked(DeviceType::fOUT, FALSE);
	gate_toolbar->setButton(DeviceType::fNET, FALSE);
	menubar->setItemChecked(DeviceType::fNET, FALSE);
	gate_toolbar->setButton(DeviceType::fPWR, FALSE);
	menubar->setItemChecked(DeviceType::fPWR, FALSE);
	gate_toolbar->setButton(DeviceType::fTXT, FALSE);
	menubar->setItemChecked(DeviceType::fTXT, FALSE);

	gate_toolbar->setButton(id, TRUE);
	menubar->setItemChecked(id, TRUE);
}

void MainWidget::dialogGlobal()
{
/*
	QDialog *oGlobalPropertiesDlg = (QDialog *)
            QWidgetFactory::create( "global_settings.ui" );

	oGlobalPropertiesDlg->exec();
	delete oGlobalPropertiesDlg;
}
*/
	int ret;

	PropGlob *dlg = new PropGlob(this, i18n("global settings"), main_netwidget);
	dlg->move(pos().x() + 50, pos().y() + 50);
	dlg->exec();
	ret = dlg->result();
	if (ret == QDialog::Accepted)
		saveProperties(app->config());
	speed_slider->setValue(NetWidget::MAX_SIMSTEPTIME - NetWidget::simTime());
	delete dlg;
}

// if called with a device pointer,
// add a device to a library
void MainWidget::dialogLibrary(XDevice *new_dev)
{

	DlgLib *dlg = new DlgLib(this, i18n("library maintenance"), &libList, activeLib, new_dev);

	dlg->move(pos().x() + 50, pos().y() + 50);
	dlg->exec();
	delete dlg;

	// library list changed?
	cbLibrary->clear();
	KlogicList<DevLibrary> *ll = libList.First();
	while(ll) {
		cbLibrary->insertItem(ll->getText());
		ll = ll->Next();
	}

	// list empty?
	if (!cbLibrary->count()) {
		libraryChanged(-1);
		lib_toolbar->hide();
		return;
	}
	lib_toolbar->show();
	cbLibrary->setEnabled(true);

	// active library changed by dialog?
	activeLib = dlg->getActive();

	// actualize lib device list
	if (activeLib) {
		// set active lib in combobox
		for(int i = 0; i < cbLibrary->count(); i++) {
			if (cbLibrary->text(i).compare(activeLib->getFileName()) == 0) {
				cbLibrary->setCurrentItem(i);
				libraryChanged(i);
				return;
			}
		}
	} else {
		cbLibrary->setCurrentItem(0);
		libraryChanged(0);
	}

}

void MainWidget::readProperties(KConfig *config)
{	int ret;

	config->setGroup("global");
	ret = config->readNumEntry("virt_resolution_x", Global::Screen::VIRT_SCREEN_SIZE_X);
	Global::Screen::VIRT_SCREEN_SIZE_X = ret;
	ret = config->readNumEntry("virt_resolution_y", Global::Screen::VIRT_SCREEN_SIZE_Y);
	Global::Screen::VIRT_SCREEN_SIZE_Y = ret;

	config->setGroup("device");
	ret = config->readNumEntry("size", XDevice::defSize());
	XDevice::setDefSize(ret);
	ret = config->readNumEntry("width", XDevice::defWidth());
	XDevice::setDefWidth(ret);
	ret = config->readNumEntry("delay", Device::defDelay());
	Device::setDefDelay(ret);
	ret = config->readNumEntry("undefined", Device::defUndefined());
	Device::setDefUndefined(ret);
	ret = config->readNumEntry("invert", XDevice::isDefInverted());
	XDevice::setDefInverted(ret);
	ret = config->readNumEntry("nameDisplayed", XDevice::nameDisplayedGlobal());
	XDevice::displayNameGlobal(ret);
	QString fnt = config->readEntry("device_font", XDevice::getFont());
	XDevice::setFont(fnt);
	ret = config->readNumEntry("device_font_size", XDevice::getFontSize());
	XDevice::setFontSize(ret);
	ret = config->readNumEntry("device_symbolset", XDevice::getSymbolSet());
	XDevice::setSymbolSet(ret);

	config->setGroup("simulation");
	ret = config->readNumEntry("time", NetWidget::simTime());
	NetWidget::setSimTime(ret);
	ret = config->readNumEntry("burst", NetWidget::simBurst());
	NetWidget::setSimBurst(ret);

	config->setGroup("pulsed");
	ret = config->readNumEntry("clock", Device::defClock());
	Device::setDefClock(ret);
	ret = config->readNumEntry("reverse_trigger", Device::triggerInverted());
	Device::invertTrigger(ret);
	ret = config->readNumEntry("file_format", klogicIO::getSaveFileFormat());
	klogicIO::setSaveFileFormat(ret);
}

void MainWidget::saveProperties(KConfig *config)
{
	config->setGroup("device");
	config->writeEntry("size", XDevice::defSize());
	config->writeEntry("width", XDevice::defWidth());
	config->writeEntry("delay", Device::defDelay());
	config->writeEntry("undefined", Device::defUndefined());
	config->writeEntry("invert", XDevice::isDefInverted());
	config->writeEntry("nameDisplayed", XDevice::nameDisplayedGlobal());
	config->writeEntry("device_font", XDevice::getFont());
	config->writeEntry("device_font_size", XDevice::getFontSize());
	config->writeEntry("device_symbolset", XDevice::getSymbolSet());

	config->setGroup("simulation");
	config->writeEntry("time", NetWidget::simTime());
	config->writeEntry("burst", NetWidget::simBurst());

	config->setGroup("pulsed");
	config->writeEntry("clock", Device::defClock());
	config->writeEntry("reverse_trigger", Device::triggerInverted());
	config->writeEntry("file_format", klogicIO::getSaveFileFormat());
	config->sync();
}


void MainWidget::setToolbar()
{
	if (tool_visible == 1) {
		gate_toolbar->hide();
		lib_toolbar->hide();
		sim_toolbar->hide();
		menubar->changeItem(i18n("Show &Toolbar"),ID_TOOL_VISIBLE);
		tool_visible = 0;
	} else {
		gate_toolbar->show();
		if (cbLibrary->count()) lib_toolbar->show();
		sim_toolbar->show();
		menubar->changeItem(i18n("Hide &Toolbar"),ID_TOOL_VISIBLE);
		tool_visible = 1;
	}
	//repaint(FALSE);
}

// change mode of drawing (drop devices OR draw wires / move objects)
void MainWidget::devChange(int i)
{
	setMode(i);
	NetWidget::setDevice(i);
}

// slot
void MainWidget::modeDraw()
{
	setMode(ID_DRAW);
}

// slot
void MainWidget::modeSelect()
{
	setMode(ID_SEL);
}

// change mode of drawing (drop objects/wire, move objects or make selections)
void MainWidget::setMode(int new_mode)
{
	if (getMode() == MAIN_MODE_SELECT && new_mode != ID_SEL) {
		KlogicList<ClientW> *lc = client_widgets.First();
		while(lc) {
			lc->Get()->net()->removeSelection();
			lc = lc->Next();
		}
	}

	switch(new_mode) {
		case ID_SEL:
			// choosen selection mode
			turnAllCheckButton(ID_SEL);
			MainWidget::mode = MAIN_MODE_SELECT;
			break;
		case ID_DRAW:
			// choosen draw wires
			turnAllCheckButton(ID_DRAW);
			MainWidget::mode = MAIN_MODE_DRAW;
			break;
		default:
			// standard device choosen
			turnAllCheckButton(new_mode);
			MainWidget::mode = MAIN_MODE_DROP;
			break;
	}

}

// static
int MainWidget::getMode()
{
	return MainWidget::mode;
}

// slot
// set toolbar buttons/menu entries in dependence of the new net
void MainWidget::netChange(int hasparent)
{
	// check if there are open sub circuits left
	KlogicList<ClientW> *lc = client_widgets.First();
	while(!hasparent && lc) {
		if (lc->Get()->net()->getActive()->parent())
			hasparent = 1;
		lc = lc->Next();
	}

	// if there are no sub-circuits anymore, lock the devices input and output
	if (hasparent) {
		pm_logic->setItemEnabled(DeviceType::fIN, TRUE);
		pm_logic->setItemEnabled(DeviceType::fOUT, TRUE);
		gate_toolbar->showItem(DeviceType::fIN);
		gate_toolbar->showItem(DeviceType::fOUT);
	} else {
		pm_logic->setItemEnabled(DeviceType::fIN, FALSE);
		pm_logic->setItemEnabled(DeviceType::fOUT, FALSE);
		gate_toolbar->hideItem(DeviceType::fIN);
		gate_toolbar->hideItem(DeviceType::fOUT);
		if (mode == MAIN_MODE_DROP) {
			setMode(ID_DRAW);
		}
	}
}

// slot
// change the simulation step time via the toolbar slider
void MainWidget::simTimeSliderChanged(int value)
{
	NetWidget::setSimTime(NetWidget::MAX_SIMSTEPTIME - value);
}

// slot coming from toolbar
void MainWidget::libraryChanged(int libidx)
{
	if (libidx < 0) {
		cbLibrary->setEnabled(false);
		cbLibDev->clear();
		cbLibDev->setEnabled(false);
		activeLib = (DevLibrary *)NULL;
		devChange(DeviceType::fAND);
		return;
	}
	QString libname = cbLibrary->text(libidx);
	KlogicList<DevLibrary> *ll = libList.With(libname);
	if (ll) {
		activeLib = ll->Get();
	} else {
		// some error occured: disable everything(?)
		cbLibrary->clear();
		cbLibrary->setEnabled(false);
		cbLibDev->clear();
		cbLibDev->setEnabled(false);
		activeLib = (DevLibrary *)NULL;
		devChange(DeviceType::fAND);
		return;
	}
	QStringList names;
	activeLib->deviceList(names);
	cbLibDev->clear();
	cbLibDev->insertStringList(names);
	if (names.count()) {
		cbLibDev->setEnabled(true);
		cbLibDev->setCurrentItem(0);
		libraryDeviceChanged(0);
	} else {
		cbLibDev->setEnabled(false);
		devChange(DeviceType::fAND);
	}
}

// slot coming from toolbar
// change the currently active library device
void MainWidget::libraryDeviceChanged(int devidx)
{	int device_id;

	if (!activeLib) return;
	QString devname = cbLibDev->text(devidx);
	device_id = activeLib->getDevID(devname);
	devChange(device_id);
}

// slot coming from NetWidget
void MainWidget::addToLib(XDevice *new_dev)
{
	dialogLibrary(new_dev);
}

// slot: signal netContentChanged coming from NetWidget
// find active window, redraw it and all client widgets with the same content
void MainWidget::repaintClients()
{	NetWidget *activeNetWidget = getActive();

	// find inactive widgets containing the same net
	KlogicList<ClientW> *lc = client_widgets.First();
	while(lc) {
		NetWidget *netw = lc->Get()->net();
		if (activeNetWidget != netw)
			netw->repaint(TRUE);
		lc = lc->Next();
	}
	circuit_changed = true;
}

// slot: signal that all changes are saved
void MainWidget::circuitUnchanged()
{	NetWidget *activeNetWidget = getActive();
	Circuit *activeDeviceNet;
	NetWidget *netw;
	KlogicList<ClientW> *lc;

	if (activeNetWidget) {
		activeDeviceNet = activeNetWidget->getActive();

		// find inactive widgets containing the same net
		lc = client_widgets.First();
		while(lc) {
			netw = lc->Get()->net();
			//if (activeNetWidget != netw && activeDeviceNet == netw->getActive())
			if (activeNetWidget != netw)
				netw->repaint(FALSE);
			lc = lc->Next();
		}
	}

	lc = client_widgets.First();
	while(lc) {
		lc->Get()->net()->resetChanges();
		lc = lc->Next();
	}

	circuit_changed = false;
}

// find the currently active netwidget
NetWidget * MainWidget::getActive()
{
	KlogicList<ClientW> *lc = client_widgets.First();
	while(lc) {
		if (lc->Get()->isActiveWindow())
			return lc->Get()->net();
		lc = lc->Next();
	}
	return (NetWidget *)NULL;
}

// slot
// sub circuit deleted
void MainWidget::netDelete(Circuit *delnet)
{
	NetWidget *oldnet;
	ClientW *old_client_widget;
	KlogicList<ClientW> *lc = client_widgets.First()->Next();

	// close all client widgets containing the deleted circuit
	while(lc) {
		oldnet = lc->Get()->net();
		old_client_widget = lc->Get();
		if (oldnet->contained(delnet)) {
			lc = lc->Next();		// will become invalid on next call!!!
			old_client_widget->closeEvent((QCloseEvent *)NULL);
		} else {
			lc = lc->Next();
		}
	}

	// check if deleted circuit is shown in the sim.graph
	if (simW->currentNet() == delnet) simW->changeNet(main_netwidget->getActive());
	netChange(0);
}

// slot
// display circuit in a new window
void MainWidget::newClient(Circuit *devnet)
{	NetWidget *newnet;

	ClientW *client_widget = new ClientW(this, devnet);
	client_widgets.Append(client_widget);

	newnet = client_widget->net();
	connect(newnet, SIGNAL(createNewWidget(Circuit *)), this, SLOT(newClient(Circuit *)));
	connect(newnet, SIGNAL(netDeleted(Circuit *)), this, SLOT(netDelete(Circuit *)));
	// NetWidget reports to clientw and this
	connect(newnet, SIGNAL(netChanged(int)), client_widget, SLOT(netChange(int)));
	connect(newnet, SIGNAL(netChanged(int)), this, SLOT(netChange(int)));

	// device deleted which was shown in the simulation graph
	connect(newnet, SIGNAL(graphChanged()), simW, SLOT(graphChange()));
	// NetWidget shown in simulation graph has changed
	connect(newnet, SIGNAL(showGraph(Circuit *)), simW, SLOT(changeNet(Circuit *)));
	// NetWidget add library device
	connect(newnet, SIGNAL(addToLib(XDevice *)), this, SLOT(addToLib(XDevice *)));
	// perform a simulation stepin the simulation graph
	connect(newnet, SIGNAL(simStepped()), simW, SLOT(simStep()));

	client_widget->show();

	if (newnet->getActive()->parent()) netChange(1);
	else netChange(0);
}

// remove all but the first client windows!
void MainWidget::closeAllClients()
{
	KlogicList<ClientW> *lc = client_widgets.First()->Next();
	KlogicList<ClientW> *lc_next;
	while(lc) {
		lc_next = lc;
		lc = lc->First();
		lc_next->Get()->close();
		lc = lc->Next();
	}
	netChange(0);
}

// close client widget
void MainWidget::closeClient(ClientW *client_widget)
{
	NetWidget *oldnet;
	KlogicList<ClientW> *lc = client_widgets.With(client_widget);
	if (lc && lc->Get()) {
		lc->Destroy(client_widget);
		oldnet = client_widget->net();
		disconnect(oldnet, SIGNAL(createNewWidget(Circuit *)), this, SLOT(newClient(Circuit *)));
		disconnect(oldnet, SIGNAL(netDeleted(Circuit *)), this, SLOT(netDelete(Circuit *)));
		disconnect(oldnet, SIGNAL(netChanged(int)), client_widget, SLOT(netChange(int)));
		disconnect(oldnet, SIGNAL(netChanged(int)), this, SLOT(netChange(int)));
		// signals for the widget showing simulation graphs
		disconnect(oldnet, SIGNAL(graphChanged()), simW, SLOT(graphChange()));
		disconnect(oldnet, SIGNAL(showGraph(Circuit *)), simW, SLOT(changeNet(Circuit *)));
		disconnect(oldnet, SIGNAL(addToLib(XDevice *)), this, SLOT(addToLib(XDevice *)));
		disconnect(oldnet, SIGNAL(simStepped()), simW, SLOT(simStep()));
		netChange(0);
	} else fatal("no such widget?");
}

// slot
// simulation widget goes hidden
void MainWidget::simWidgetHidden()
{
	simw = 0;
	sim_toolbar->setButton(ID_SIM_SHOW, FALSE);
	pm_sim->changeItem(i18n("Show &Graph"),ID_SIM_SHOW);
}

// change mode of simulation
void MainWidget::simMode(int i)
{
	NetWidget::setSimMode(i);
	simW->setSimMode(i);

	sim_toolbar->setButton(NetWidget::MODE_SIM_STEP, FALSE);
	pm_sim->setItemChecked(NetWidget::MODE_SIM_STEP, FALSE);
	sim_toolbar->setButton(NetWidget::MODE_SIM_MULT, FALSE);
	pm_sim->setItemChecked(NetWidget::MODE_SIM_MULT, FALSE);
	sim_toolbar->setButton(i, TRUE);
	pm_sim->setItemChecked(i, TRUE);
}

