#ifndef __DEVICE_RAM__
#define __DEVICE_RAM__

#include <qpainter.h>
#include "xdevice.h"
#include "busOutputArray.h"
#include "grid.h"

class RAMDevice : public XDevice
{
public:
	RAMDevice(int x, int y, int size = -1, int del = -1, int undef = -1, int clock = -1)
		: XDevice(DeviceType::fRAM, x, y, size, del, undef, clock)
	{
	}

	virtual ~RAMDevice()
	{
		delete[] ram;
		delete bus;
	}

	virtual void init()
	{
		ramsize = 32;
		ram = new char[ramsize];

		// named output
		bus = new BusOutputArray(this, true, false);

		// named input
		write_enable_id = addInputName(INPUT_RAM_WRITE, 0);
		addInputName(IO_BUS_ADDRESS[0], 9);
		addInputName(IO_BUS_ADDRESS[1], 10);
		addInputName(IO_BUS_ADDRESS[2], 11);
		addInputName(IO_BUS_ADDRESS[3], 12);
		addInputName(IO_BUS_ADDRESS[4], 13);

		// internal variables
		address_id = addInternalName(INTERNAL_RAM_ADDRESS);
		data_id = addInternalName(INTERNAL_RAM_ADDRESS);

		setSize(neededSize());

		XDevice::init();
	}

	virtual void * getInstance()
	{
		return (void *)this;
	}

	virtual int RAMSize()
	{
		return ramsize;
	}

	virtual void setRAMSize(int new_size)
	{
		if (new_size > 0 && new_size <= 32) {
			for (int i = ramsize - 1; i < new_size; i++)
				ram[i] = 0;
			ramsize = new_size;
		}
	}

	virtual void setRAMStorage(char *newram)
	{
		if (sizeof(newram) != sizeof(ram)) {
			fatal("RAMDevice::setRAMStorage: invalid RAM size");
			return;
		}
		memcpy(ram, newram, ramsize * sizeof(char));
	}

	virtual char * getRAMStorageCopy()
	{
		char *newram = new char[ramsize];
		memcpy(newram, ram, ramsize * sizeof(char));
		return newram;
	}

	virtual void setEquation()
	{
		Device::setEquation("D0  D1  D2  D3  D4  D5  D6  D7", data_id);
		Device::setEquation("A0  A1  A2  A3  A4", address_id);
	}

	virtual bool hasClock()
	{
		return false;
	}

	virtual bool hasNamedInput()
	{
		return true;
	}

	virtual bool hasNamedOutput()
	{
		return true;
	}

	virtual bool sizeChangeable()
	{
		return false;
	}

	virtual void setImage()
	{	QPixmap *pic = getImage();
		QPainter p;
		QPoint pt(0,0);

		if (width() == 3) {
			int i =  PTOPH->height() + (PMIDH->height() * deviceSize) + PBOTH->height();
			pic->resize(PTOPH->width(), i);

			p.begin(pic);
			// top of the device
			p.drawPixmap(pt, *PTOPH);
			pt.setY(pt.y() + PTOPH->height());

			// middle-part of device
			for(i=0;i<deviceSize;i++) {
				p.drawPixmap(pt, *PMIDH);
				pt.setY(pt.y() + PMIDH->height());
			}

			// bottom of device
			p.drawPixmap(pt, *PBOTH);
		} else {
			int i =  PTOPL->height() + (PMIDL->height() * deviceSize) + PBOTL->height();
			pic->resize(PTOPL->width(), i);

			p.begin(pic);
			// top of the device
			p.drawPixmap(pt, *PTOPL);
			pt.setY(pt.y() + PTOPL->height());

			// middle-part of device
			for(i=0;i<deviceSize;i++) {
				p.drawPixmap(pt, *PMIDL);
				pt.setY(pt.y() + PMIDL->height());
			}

			// bottom of device
			p.drawPixmap(pt, *PBOTL);
		}

		// draw input- and output names
		p.setFont(STATfont);
		KlogicList<Value> *lv = named_input.First();
		int xdistance = IOREG + 3;
		int pos;
		while(lv) {
			if (!lv->Get()->isInternal()) {
				pos = lv->Get()->getPosition() * Grid::GRID;
				if (lv->Get()->getPosition() == 0)
					p.drawText(xdistance, pos + 7, lv->getText());
				else if (pos == (deviceSize * Grid::GRID))
					p.drawText(xdistance, pos + 5, lv->getText());
				else
					p.drawText(xdistance, pos + 6, lv->getText());
			}
			lv = lv->Next();
		}
		KlogicList<Calculator> *lop = named_output.First()->Next();
		xdistance = pic->width() / 2;
		int xwidth = xdistance - 6;
		while(lop) {
			if (!lop->Get()->isInternal()) {
				pos = lop->Get()->getPosition() * Grid::GRID;
				if (pos == 0)
					p.drawText(xdistance, pos - 2, xwidth, 10,
						Qt::AlignRight, lop->getText());
				else if (pos == (deviceSize * Grid::GRID))
					p.drawText(xdistance, pos - 4 , xwidth, 10,
						Qt::AlignRight, lop->getText());
				else
					p.drawText(xdistance, pos - 3, xwidth, 10,
						Qt::AlignRight, lop->getText());
			}
			lop = lop->Next();
		}
		drawConnectionLines(&p);
		XObject::setImage();
		p.end();

		if (m_oTextDev) m_oTextDev->setImage();
	}

	virtual void printImage(QPainter *, const QRect &)
	{
	}

	virtual bool outputIsActive()
	{
		if (tristate_active) return false;
		return true;
	}

	// return device's output value
	virtual int output(int output_id)
	{
		// bus device output: handle internal output without any delay
		int weight = bus->getOutputWeight(output_id);
		if (weight == -1) fatal("RAMDevice::output(): no valid bus index");

		KlogicList<Calculator> *l_data = named_output.With(data_id);
		if (!l_data) fatal("RAMDevice::output: bus without data!?");

		int data = l_data->Get()->getResult();
 
		int wr = undef();
		KlogicList<Device> *l_wr = connection_list.With(write_enable_id);
		if (l_wr) wr = l_wr->Get()->output(l_wr->getID2());

		KlogicList<Calculator> *l_addr = named_output.With(address_id);
		int address = l_addr->Get()->getResult();
		if (address < 0 || address > ramsize) fatal("Device::output: ram address");

		if (wr) ram[address] = data;
		data = ram[address];
		forceOutputChange();
 
		if (tristate_active) return TRISTATE;
		return 1 && (data & weight);
	}

private:
	static const QString INPUT_RAM_WRITE;
	static const QString INTERNAL_RAM_ADDRESS;
	static const QString INTERNAL_RAM_DATA;

	// ram
	int address_id;         // ram device: internal address
	int data_id;            // ram device: internal data
	int write_enable_id;    // ram write enable

	char *ram;
	int ramsize;

	BusOutputArray *bus;
};

#endif
