#ifndef __Device_CLASS__
#define __Device_CLASS__

#include <map>
using namespace std;

class QString;

#include <calculator.h>
#include <value.h>
#include <fifo.h>
#include <klogicList.h>
#include <deviceTypes.h>
class Circuit;
class BusOutputArray;

#define ADDNAMED_OK	0
#define ADDNAMED_FULL	-1
#define ADDNAMED_OP	-2
#define ADDNAMED_UNIQUE	-3

#define NOSPACE			1

// logical tristate level
#define TRISTATE		-1

// a logic device
class Device
{
public:
	static int instance;

	Device(int func, int delay = -1, int undef = -1, int clock = -1);
	virtual ~Device();

	virtual void * getInstance();

	//***** type and status methods
	int type();
	void changeLEDType(int);
	int delay();
	void setDelay(int);
	int getID();
	void setID(int);
	QString getName(bool bNoSpace = false) const;
	virtual void setName(const QString&);
	int undef();
	virtual void setUndef(int);
	virtual bool sizeChangeable() { return false; }
	virtual bool setSize(int) { return false; }
	virtual int neededSize() { return 0; }
	virtual bool hasBooleanOutput(int) { return true; }

	static int defUndefined();
	static void setDefUndefined(int);
	static int defDelay();
	static void setDefDelay(int);
	static int defClock();
	static void setDefClock(int);
	static void invertTrigger(int);
	static int triggerInverted();
	static void resetCounter();

	//***** methods to manage named IO
	virtual int addInputName(const QString&, int pos = -1, int input_id = 0);
	virtual int changeInputName(int, const QString&);
	virtual void removeInputName(int);
	virtual int addOutputName(const QString&, int pos = -1, int output_id = 0);
	virtual int changeOutputName(int, const QString&);
	virtual void removeOutputName(int);

	int getInputID(QString&);
	int getOutputID(QString&);
	virtual bool hasNamedInput();
	virtual bool hasNamedOutput();
	bool inputNameIsUnique(QString&);

	void setOutputNameType(int, int);
	virtual bool outputIsActive(int);	// returns wether output is active or tristate
	void outputSetTristate(int, bool);	// set output able to be tristate
	bool outputIsTristate(int);		// returns if output could be tristate
	void setTristate(bool);			// set device able to be tristate
	bool isTristate();			// returns if device could be tristate
	bool isTristateControlInput(int);	// returns if input is the tristate control input

	int addInternalName(const QString&);
	void removeInternalName(int);

	KlogicList<Value> *getNamedIRef();
	KlogicList<Calculator> *getNamedORef();

	//***** flipflop properties
	int hasMaster();
	void setMaster(int);

	//***** clock behaviour
	int clock();
	virtual void setClock(int);
	virtual bool hasClock();

	//***** calculation methods
	virtual void Calculate(int);		// calculate device
	virtual void Propagate(int);		// propagate output value
	void flush(int);			// set all outputs to a new value

	//***** get values
	virtual int input(int input_id = 0);
	virtual int input(int input_id, bool& bIsBoolean);
	virtual int output(int output_id = 0);
	virtual int outputChanged();		// output changed since last access?
	virtual void forceOutputChange();	// force output changed since last access

	void setStaticOutput(int, int);		// set a named output to a new value (used for net devices only)

	//***** equation related methods
	const KlogicList<Calculator> * getOutput(const QString&);
	const KlogicList<Value> * getInput(const QString&);
	QString getEquation(int);		// return the equation for a specific output
	virtual void setEquation();		// set equation string (for predefined devices)
	void setEquation(char);			// set equation string (for unnamed outputs)
	void setEquation(const char *, int output_id = 0);
	QueueInfo parseEquation();		// setup operation stacks from the equation strings
	// returns the correct name for an input
	QString qualifiedOutputName(Device *, const char *, const char *qual_pref = (char *)NULL);
	// append all equations of this device to the given list
	virtual void getAllEquations(map<QString, OutputInfo>&, const QString&, bool);
	// actialize device's equations with the given list
	void setAllEquations(KlogicList<OutputInfo> *, bool change_size = true);

	//***** connection methods
	// connect two devices (1/0)
	int connectDevices(Device *, int, int);
	// disconnect two devices (1/0)
	int disconnectDevices(Device *, int, int);
	// exchange names of connections

	// constants
	static const int MINDELAY = 1;
	static const int MAXDELAY = 99;

	static const int INPUT = 1;
	static const int TMP_OUTPUT = 2;
	static const int INTERNAL_OUTPUT = 3;
	static const int FINAL_OUTPUT = 4;

	static const bool NO_QUALS = true;

	static const int DEFAULT_OSC_CNT = 10;
	static const int DEFAULT_UNDEF = 0;
	static const int DEFAULT_DELAY = 1;

	static bool IMPORT_IGNORE_GLOBAL;

	// clock types
	static const int CLK_RISING_2EDGE = 0;
	static const int CLK_FALLING_2EDGE = 1;
	static const int CLK_HIGH_VALUE	 = 2;
	static const int CLK_LOW_VALUE = 3;
	static const int CLK_NONE = 4;
	static const int CLK_RISING_1EDGE = 5;
	static const int CLK_FALLING_1EDGE = 6;

protected:
	const char * getOutputName(int);
	int suggestedOutputType(int);

	// named output names/calculation stacks
	// id1: the device id of the interface device
	// id2: the number of the output within this device (maybe?)
	KlogicList<Calculator> named_output;

	// named input names/values
	// id1: the device id of the interface device
	// id2: the number of the inputs within this device
	KlogicList<Value> named_input;

	// id of probably used internal outputs within the device
	// needed to internally set equations in device.cpp
	int mq_id;		// internal Q (master ff)
	int m_c1_1_id;		// internal C1 (the previous C1)
	int tristate_enable_id;	// tristate enable

	BusOutputArray *output_bus;

	// tristate ouputs present
	int tristate;

	// pulsed devices
	int clock_type;

	// static input value (devices without inputs, e.g. the switch)
	int static_input_value;

	// tristate is active?
	bool tristate_active;

	// input-connection of device
	KlogicList<Device> connection_list;

	int master;		// master flipflop enable/disable

private:
	// for naming the devices uniquely
	static int STATcnt[DeviceType::TYPE_CNT];
	static QString STATname[DeviceType::TYPE_CNT];

	static int STATdef_undefined;
	static int STATdef_delay;
	static int STATdef_clock;
	static int STATreverse_trigger;

	// type/function of the device
	int function;

	// sum over all output values (needed for drawing)
	int force_output_changed;
	int old_result;
	int new_result;

	int id;			// unique device id

	QString m_sName;	// device name

public:
	// old file support
	QString device2string();
	bool string2device(QString);
};

#endif

