#include <stdlib.h>

#include <qstring.h>
#include <qxml.h>

#include "klogic.h"
#include "xnet.h"
#include "xdevice.h"
#include "deviceTypes.h"
#include "xwire.h"
#include "xmlImport.h"
#include "xmlExport.h"

/*******************************************/
/* very restriced XML line parser          */
/* (used only for libraries)               */
/*                                         */
/* Andreas Rostin 25.09.2001               */
/*******************************************/
void XmlLineParser::setLine(const QString& line)
{	QString __tag;

	// start tag
	int tagstart1 = line.find("<");
	if (tagstart1 == -1) {
		sTag.truncate(0);
		attributes.truncate(0);
		content = line;
		endTag = false;
		return;
	}
	int tagstart2 = line.find("</");

	int tagend1 = line.find(">", 0);
	if (tagend1 == -1) {
		sTag.truncate(0);
		attributes.truncate(0);
		content = line;
		endTag = false;
		return;
	}

	// end tag only
	if (tagstart1 == tagstart2) {
		sTag = line.mid(tagstart2 + 2, line.length() - tagstart2 - 3);
		iTag = XmlObject::getID(sTag);
		endTag = true;
		content = line.left(tagstart2);
		return;
	}

	// start tag only
	if (tagstart2 == -1) {
		__tag = line.mid(tagstart1 + 1, tagend1 - tagstart1 - 1);

		// check for attributes
		int pos = __tag.find(" ");
		if (pos != -1) {
			sTag = __tag.left(pos);
			attributes = __tag.right(__tag.length() - pos - 1);
		} else {
			sTag = __tag;
			attributes.truncate(0);
		}

		// check for combined start/end tag
		if (sTag[sTag.length() - 1] == '/') {
			sTag = sTag.left(sTag.length() - 1);
			endTag = true;
		} else endTag = false;

		iTag = XmlObject::getID(sTag);
		content = line.right(line.length() - tagend1 - 1);
		return;
	}

	// start and end tag
	__tag = line.mid(tagstart1 + 1, tagend1 - tagstart1 - 1);
	int pos = __tag.find(" ");
	if (pos != -1) {
		sTag = __tag.left(pos);
		attributes = __tag.right(__tag.length() - pos - 1);
	} else {
		sTag = __tag;
		attributes.truncate(0);
	}
	iTag = XmlObject::getID(sTag);
	endTag = true;
	content = line.mid(tagend1 + 1, tagstart2 - tagend1 - 1);
}

QString XmlLineParser::getTag()
{
	return sTag;
}

int XmlLineParser::tag()
{
	return iTag;
}

bool XmlLineParser::isEndTag()
{
	return endTag;
}

QString XmlLineParser::getUnquotedContent()
{
	return XmlField::unquote(content);
}

QString XmlLineParser::getContent()
{
	return content;
}

QString XmlLineParser::getAttributes()
{
	return attributes;
}

/*******************************************/
/* XML SAX2 Import Interface               */
/*                                         */
/* Andreas Rostin 29.08.2001               */
/*******************************************/
int XMLImportHandler::error_cnt = 0;
QString XMLImportHandler::error_msg = "";

// static
// import a xml file either into the given _net (sub circuit) or using the _net as main circuit
bool XMLImportHandler::import(XDeviceNet *_net, bool selected, int dx, int dy, QTextStream & stream)
{
	Device::IMPORT_IGNORE_GLOBAL = true;

	XMLImportHandler import_handler(_net, selected, dx, dy);
	QXmlSimpleReader reader;
	QXmlInputSource source(stream);

	reader.setContentHandler(&import_handler);
	reader.parse(source);

	Device::IMPORT_IGNORE_GLOBAL = false;

	if (error_cnt) {
		fprintf(stderr, "XMLImportHandler::import: errors occured:\n----------------\n");
		fprintf(stderr, "\n%s\n---------------------\n", (const char *)error_msg);
		return false;
	}
	return true;
}

// static
// create a net device from a xml file
XDevice * XMLImportHandler::import(XDeviceNet *_net, int dx, int dy, QTextStream & stream)
{
	Device::IMPORT_IGNORE_GLOBAL = true;

	XMLImportHandler import_handler(_net, dx, dy);
	QXmlSimpleReader reader;
	QXmlInputSource source(stream);
 
	reader.setContentHandler(&import_handler);
	reader.parse(source);

	Device::IMPORT_IGNORE_GLOBAL = false;

	if (error_cnt) {
		fprintf(stderr, "XMLImportHandler::import: errors occured:\n----------------\n");
		fprintf(stderr, "\n%s\n---------------------\n", (const char *)error_msg);
	}

	return import_handler.getDevice();
}

// static
// return no of errors occured during import
int XMLImportHandler::errorsOccured()
{
	return error_cnt;
}

// static
// return error messages  
QString& XMLImportHandler::getErrors()
{
	return error_msg;
}

XMLImportHandler::XMLImportHandler(XDeviceNet *_net, bool selected, int dx, int dy)
{
	net = _net;
	imp_selected = selected;
	imp_dx = dx;
	imp_dy = dy;
	importDev = false;
	localImport = false;
	init();
}

XMLImportHandler::XMLImportHandler(XDeviceNet *_net, int dx, int dy)
{
	net = _net;
	imp_selected = false;
	imp_dx = dx;
	imp_dy = dy;
	importDev = true;
	localImport = true;
	init();
}

XMLImportHandler::~XMLImportHandler()
{
}

XDevice * XMLImportHandler::getDevice()
{
	return importedDev;
}

void XMLImportHandler::init()
{
	error_msg = "";
	error_cnt = 0;

	importedDev = (XDevice *)NULL;
	activeTagIdx = 0;
	activeNet[activeTagIdx] = net;
	activeNetIdx = 0;
	activeDev = (XDevice *)NULL;
	activeID = 0;
	activeWire = (XWire *)NULL;
	activeWireConn = (XWire *)NULL;
	activeDevConn = (XWire *)NULL;
	importSub = false;

	// debugging
	characters_call_cnt = 0;
}

bool XMLImportHandler::startDocument()
{
	init();
	return true;
}

// currently, this method assues that everything is ok with the file!
// to be optimized by error handling!
bool XMLImportHandler::startElement(const QString&, const QString&, const QString& qName, const QXmlAttributes& atts)
{
	int id = XmlObject::getID(qName);
	if (!id) {
		error_msg += i18n("unknown tag");
		error_msg += "<";
		error_msg += qName;
		error_msg += ">\n";
		error_cnt++;
	}
	activeTag[++activeTagIdx] = id;

	// -----------------------------------------
	// main tags
	// -----------------------------------------
	if (id == XmlObject::KLOGIC_MAIN) {
		if (XmlObject::symbolEquals(atts.qName(0), XmlObject::FILE_TYPE)) {
			if (atts.value(0) == XmlObject::ATT_CIRCUIT)
				importSub = false;
			else
				importSub = true;
		}
		return true;
	}

	if (id == XmlObject::CIRCUIT) {
		if (atts.length() != 5) {
			error_msg += i18n("circuit: too few attributes\n");
			error_cnt++;
			return false;
		}
		int circuitID = atts.value(0).toInt();
		int type = atts.value(1).toInt();
		int size = atts.value(2).toInt();

		int x = imp_dx;
		int y = imp_dy;
		if (!importDev) {
			x += atts.value(3).toInt();
			y += atts.value(4).toInt();
		}

		activeNetIdx++;
		if (activeNetIdx == 2) {
			// only important in top level circuit
			imp_dx_hidden = imp_dx;
			imp_dx = 0;
			imp_dy_hidden = imp_dy;
			imp_dy = 0;
			imp_selected_hidden = imp_selected;
			imp_selected = false;
		}

		if (!importSub && !importDev) {
			// the very first circuit to be the existing root circuit
			importSub = true;
			activeNet[activeNetIdx] = net;
		} else {
			// create sub circuits
			activeDev = activeNet[activeNetIdx - 1]->newDevice(type, x, y, size);
			if (!activeDev) {
				error_msg += i18n("unable to create circuit device\n");
				error_cnt++;
				return false;
			}
			activeNet[activeNetIdx] = activeDev->devIsNet();
			if (!activeNet[activeNetIdx]) {
				error_msg += i18n("unable to create circuit\n");
				error_cnt++;
				return false;
			}
			activeDev = (XDevice *)NULL;

		}
		wiremap[activeNetIdx].init();

		// the id-mapping of a subcircuit must be known in the parent circuit
		devmap[activeNetIdx - 1].fileID(circuitID, activeNet[activeNetIdx]->getID());
		devmap[activeNetIdx].init();

		// but the new sub-circuit does not need to know its own id-mapping(?)
		//devmap[activeNetIdx].fileID(circuitID, activeNet[activeNetIdx]->getID());

		// only a device should be imported?
		if (importDev) {
			imp_dx = 0;
			imp_dy = 0;
			importSub = true;
			importDev = false;
			importedDev = activeNet[activeNetIdx];
		}

		return true;
	}

	if (id == XmlObject::DEVICE) {
		// create devices
		if (!activeNet[activeNetIdx]) {
			error_msg += i18n("no parent net for device\n");
			error_cnt++;
			return false;
		}
		if (atts.length() != 5) {
			error_msg += i18n("device: too few attributes\n");
			error_cnt++;
			return false;
		}
		activeID = atts.value(0).toInt();
		int type = atts.value(1).toInt();
		int size = atts.value(2).toInt();
		int x = imp_dx;
		int y = imp_dy;
		if (!importDev) {
			x += atts.value(3).toInt();
			y += atts.value(4).toInt();
		}
		activeDev = activeNet[activeNetIdx]->newDevice(type, x, y, size);
		if (!activeDev) {
			error_msg += i18n("unable to create device (circuit depth ");
			error_msg += QString::number(activeNetIdx);
			error_msg += ")\n";
			error_cnt++;
			return false;
		}
		devmap[activeNetIdx].fileID(activeID, activeDev->getID());

		// only a device should be imported?
		if (importDev) {
			imp_dx = 0;
			imp_dy = 0;
			importedDev = activeDev;
			importDev = false;
		}

		return true;
	}

	if (id == XmlObject::WIRE) {
		// create wires
		if (!activeNet) {
			error_msg += i18n("no parent net for device\n");
			error_cnt++;
			return false;
		}
		if (atts.length() != 1) {
			error_msg += i18n("wire: too few attributes\n");
			error_cnt++;
			return false;
		}
		activeID = atts.value(0).toInt();
		activeWire = activeNet[activeNetIdx]->newWire();
		activeWire->setExportDelta(imp_dx, imp_dy);
		if (!activeWire) {
			error_msg += i18n("unable to create wire (circuit depth ");
			error_msg += QString::number(activeNetIdx);
			error_msg += ")\n";
			error_cnt++;
			return false;
		}

		// id mapping
		wiremap[activeNetIdx].fileID(activeID, activeWire->getID());
		return true;
	}

	if (id == XmlObject::WIRE_CONN) {
		// create wire <--> wire connections
		if (!activeNet) {
			error_msg += i18n("no parent net for wire <--> wire connections\n");
			error_cnt++;
			return false;
		}
		if (atts.length() != 1) {
			error_msg += i18n("wire<-->wire connection: too few attributes\n");
			error_cnt++;
			return false;
		}
		int fileID = atts.value(0).toInt();
		int wireID = wiremap[activeNetIdx].mapID(fileID);
		activeWireConn = activeNet[activeNetIdx]->getWire(wireID);
		activeID = 0;	// it stores the second id!
		if (!activeWireConn) {
			error_msg += i18n("unable to retrieve wire for id %d ");
			error_msg += QString::number(fileID);
			error_msg += i18n("(circuit depth ");
			error_msg += QString::number(activeNetIdx);
			error_msg += ")\n";
			error_cnt++;
			return false;
		}
		return true;
	}

	if (id == XmlObject::DEV_CONN) {
		// create dev <--> wire connections
		if (!activeNet) {
			error_msg += i18n("no parent net for wire <--> wire connections\n");
			error_cnt++;
			return false;
		}
		if (atts.length() != 1) {
			error_msg += i18n("wire<-->device connection: too few attributes\n");
			error_cnt++;
			return false;
		}
		int fileID = atts.value(0).toInt();
		int wireID = wiremap[activeNetIdx].mapID(fileID);
		activeDevConn = activeNet[activeNetIdx]->getWire(wireID);
		activeID = 0;	// it stores the second id!
		if (!activeDevConn) {
			error_msg += i18n("unable to retrieve wire for id ");
			error_msg += QString::number(fileID);
			error_msg += i18n("(circuit depth ");
			error_msg += QString::number(activeNetIdx);
			error_msg += ")\n";
			error_cnt++;
			return false;
		}
		return true;
	}

	// -----------------------------------------
	// sub tags with attributes
	// remember the attributes
	// -----------------------------------------
	// connections
	if (id == XmlObject::POS) {
		if (atts.length() > 2) {
			error_msg += i18n("position: too few attributes\n");
			error_cnt++;
			return false;
		}
		// a pos tag has id and inverter attribute
		activeID = atts.value(0).toInt();
		if (atts.length() == 2)
			activeInv = atts.value(1).toInt();
		return true;
	}

	// equation devices
	if (id == XmlObject::OUT_EQU) {
		if (atts.length() != 3) {
			error_msg += i18n("equ-device output equations: too few attributes\n");
			error_cnt++;
			return false;
		}
		activePos = atts.value(0).toInt();
		activeIntern = atts.value(1).toInt();
		activeName = atts.value(2);
		return true;
	}

	// equation devices
	if (id == XmlObject::IN_NAME) {
		if (atts.length() != 2) {
			error_msg += i18n("equ-device input name: too few attributes\n");
			error_cnt++;
			return false;
		}
		activePos = atts.value(0).toInt();
		activeIntern = atts.value(1).toInt();
		return true;
	}

	// sub circuits
	if (id == XmlObject::OUT_ID ||
	    id == XmlObject::IN_ID) {
		if (atts.length() != 1) {
			error_msg += i18n("sub circuit input/output: too few attributes\n");
			error_cnt++;
			return false;
		}
		activePos = atts.value(0).toInt();
		return true;
	}

	// --------------------------------------------------------------------
	// sub tags with no content (characters() will not be called or ignored)
	// process the tags
	// --------------------------------------------------------------------
	// device: display text
	if (id == XmlObject::TEXT_DISP) {
		if (activeDev) activeDev->displayText(true);
		else activeNet[activeNetIdx]->displayText(true);
	}

	// push button
	if (id == XmlObject::PUSH) {
		if (activeDev) activeDev->setSwitchType(false);
	}

	// output inverted by default
	if (id == XmlObject::INV) {
		if (activeDev) activeDev->setInverted(true);
	}

	return true;
}

// tag content arrives
// the current tag id is stored in activeTag array
bool XMLImportHandler::characters(const QString& ch)
{
	// debugging
	characters_call_cnt++;

	if (ch.stripWhiteSpace().isEmpty()) return true;

	int id = activeTag[activeTagIdx];
	if (!id) return true;

	// childs of activeNet
	if (activeWire) {
		if (id == XmlObject::WIRE) {
			bool ret = *activeWire << ch;
			if (!ret) {
				error_msg += QString::number(characters_call_cnt);
				error_msg += ": ";
				error_msg += i18n("unable to import wire id");
				error_msg += " <";
				error_msg += QString::number(activeID);
				error_msg += ">\n";
				error_cnt++;
				return true;
			}
		}
		return true;
	}

	if (activeDev) {
		return setDeviceTag(activeDev, ch);
	}

	if (activeWireConn) {
		int fileID = activeID;
		activeID = wiremap[activeNetIdx].mapID(activeID);
		XWire *wire = activeNet[activeNetIdx]->getWire(activeID);
		if (!wire) {
			error_msg += QString::number(characters_call_cnt);
			error_msg += ": ";
			error_msg += i18n("unable to retrieve second wire for wire<-->wire connection with");
			error_msg += " ID2=";
			error_msg += QString::number(fileID);
			error_msg += "\n";
			error_cnt++;
			return true;
		}
		int delim = ch.find(':');
		if (delim < 0) {
			error_msg += QString::number(characters_call_cnt);
			error_msg += ": ";
			error_msg += i18n("invalid position format for wire<-->wire connection with");
			error_msg += " ID2=";
			error_msg += QString::number(fileID);
			error_msg += "\n";
			error_cnt++;
			return true;
		}
		int x = ch.left(delim).toInt() + imp_dx;
		int y = ch.right(ch.length() - delim - 1).toInt() + imp_dy;
		QPoint pos(x,y);
		if (Wire::NODE_NOLOCK == activeWireConn->lockNode(pos)) {
			if (Wire::NODE_NOLOCK == wire->lockNode(pos)) {
				error_msg += QString::number(characters_call_cnt);
				error_msg += ": ";
				error_msg += i18n("unable to establish wire<-->wire connection with");
				error_msg += " ID2=";
				error_msg += QString::number(fileID);
				error_msg += i18n(" in position ");
				error_msg += QString::number(x);
				error_msg += ":";
				error_msg += QString::number(y);
				error_msg += " for activeNetIdx <";
				error_msg += QString::number(activeNetIdx);
				error_msg += "\n";
				error_cnt++;
				return true;
			} else activeWireConn->checkConnection(wire);
		} else wire->checkConnection(activeWireConn);

		return true;
	}

	if (activeDevConn) {
		// activeDevConn contains the wire
		// the id of the device is in activeID
		int fileID = activeID;
		activeID = devmap[activeNetIdx].mapID(activeID);
		int delim = ch.find(':');
		if (delim < 0) {
			error_msg += QString::number(characters_call_cnt);
			error_msg += ": ";
			error_msg += i18n("invalid position format for wire<-->dev connection with");
			error_msg += " ID2=";
			error_msg += QString::number(fileID);
			error_msg += ")\n";
			error_cnt++;
			return true;
		}
		int x = ch.left(delim).toInt() + imp_dx;
		int y = ch.right(ch.length() - delim - 1).toInt() + imp_dy;
		QPoint pos(x,y);
		if (!activeDevConn->contains(pos)) {
			error_msg += QString::number(characters_call_cnt);
			error_msg += ": ";
			error_msg += i18n("no such wire node ");
			error_msg += QString::number(x);
			error_msg += ":";
			error_msg += QString::number(y);
			error_msg += "\n";
			error_cnt++;
			return true;
		}
		if (DCONN != activeNet[activeNetIdx]->checkConnection(activeID, activeInv, activeDevConn)) {
			error_msg += QString::number(characters_call_cnt);
			error_msg += ": ";
			error_msg += i18n("unable to establish wire<-->dev connection with device id");
			error_msg += QString::number(fileID);
			error_msg += i18n(" in position ");
			error_msg += QString::number(x);
			error_msg += ":";
			error_msg += QString::number(y);
			error_msg += " ";
			error_msg += i18n("for");
			error_msg += " activeNetIdx <";
			error_msg += QString::number(activeNetIdx);
			error_msg += ">\n";
			error_cnt++;
			return true;
		}
		return true;
	}

	if (activeNetIdx > 0) {
		return setDeviceTag(activeNet[activeNetIdx], ch);
	}

	// these should be handled the very first, so they almost never be checked
	if (id == XmlObject::PROGRAM_VERSION) {
		// informational only
		// always the current version is stored
		return true;
	}

	if (id == XmlObject::CREATION_DATE) {
		if (!localImport) CurrentCircuit::creation = ch;
		return true;
	}

	if (id == XmlObject::UPDATE_DATE) {
		// informational only
		return true;
	}

	if (id == XmlObject::AUTHOR) {
		if (!localImport) CurrentCircuit::author = ch;
		return true;
	}

	if (id == XmlObject::COMMENT) {
		if (!localImport) CurrentCircuit::comment = ch;
		return true;
	}

	return true;
}

// check import content of a device tag
// set the more often used tags to the top!
// use "dev" instead of "activeDev" here!
bool XMLImportHandler::setDeviceTag(XDevice *dev, const QString& ch)
{
	int id = activeTag[activeTagIdx];

	if (id == XmlObject::TEXT) {
		activeNet[activeNetIdx]->setText(dev->getID(), ch);

		// actualize named_input and named_output list
		if (dev->type() == DeviceType::fIN || dev->type() == DeviceType::fOUT) {
			activeNet[activeNetIdx]->correctIO(dev->getID(), activeID);
		}

		return true;
	}

	if (id == XmlObject::DELAY) {
		dev->setDelay(ch.toInt());
		return true;
	}

	if (id == XmlObject::UNDEF) {
		dev->setUndef(ch.toInt());
		return true;
	}

	if (id == XmlObject::STAT) {
		dev->setStaticInput(ch.toInt());
		return true;
	}

	if (id == XmlObject::CLK_TYPE) {
		dev->setClock(ch.toInt());
		return true;
	}

	if (id == XmlObject::MASTER) {
		dev->setMaster(ch.toInt());
		return true;
	}

	if (id == XmlObject::OSC_ON) {
		dev->setOszOn(ch.toInt());
		return true;
	}

	if (id == XmlObject::OSC_OFF) {
		dev->setOszOff(ch.toInt());
		return true;
	}

	if (id == XmlObject::OSC_STEP) {
		dev->setCurrOszStep(ch.toInt());
		return true;
	}

	if (id == XmlObject::RAM) {
		int delim_pos;
		int curr_pos = 0;
		if (-1 == (delim_pos = ch.find(';'))) {
			error_msg += i18n("ram error: no length\n");
			error_cnt++;
			return true;
		}
		int cnt = ch.mid(curr_pos, delim_pos - curr_pos).toInt();
		int str_len = ch.length();
		dev->setRAMSize(cnt);
		char *newram = dev->getRAMStorageCopy();
		for(int i = 0; i < cnt; i++) {
			curr_pos = delim_pos + 1;
			if (curr_pos > str_len) {
				error_msg += i18n("ram error: invalid length\n");
				error_cnt++;
				return true;
			}
			delim_pos = ch.find(';', curr_pos);
			if (delim_pos == -1) delim_pos = str_len;
			newram[i] = ch.mid(curr_pos, delim_pos - curr_pos).toInt();
		}
		dev->setRAMStorage(newram);
		delete [] newram;
		return true;
	}

	if (id == XmlObject::TRI) {
		dev->setTristateControlPos(ch.toInt());
		return true;
	}

	// equation device output
	if (id == XmlObject::OUT_EQU) {
		int dev_id;
		if (!activeIntern)
			dev_id = dev->addOutputName(activeName, activePos, 0);
		else
			dev_id = dev->addInternalName(activeName);
		dev->Device::setEquation(ch, dev_id);
		dev->setSize(dev->neededSize());
		return true;
	}

	// equation device input
	if (id == XmlObject::IN_NAME) {
		int input_id = ch.toInt();
		if (!activeIntern) dev->addInputName(ch, activePos, input_id);
		else dev->addInternalName(ch);
		dev->setSize(dev->neededSize());
		return true;
	}

	// sub circuit output: it's a reference to a device inside
	// the device name could come in later sections!
	// the mapping will be done while doing device import for this circuit device
	if (id == XmlObject::OUT_ID) {
		int output_id_old = ch.toInt();
		dev->addOutputName(ch, activePos, output_id_old);
		return true;
	}

	// sub circuit input: it's a reference to a device inside
	// the device name could come in later sections!
	// the mapping will be done while doing device import for this circuit device
	if (id == XmlObject::IN_ID) {
		int input_id_old = ch.toInt();
		dev->addInputName(ch, activePos, input_id_old);
		return true;
	}

	// device width: net and equ only
	if (id == XmlObject::WIDTH) {
		int width = ch.toInt();
		dev->setWidth(width);
		return true;
	}

	// device width: net and equ only
	if (id == XmlObject::BUSTYPE) {
		bool bMux = ch.toInt();
		dev->setMux(bMux);
		return true;
	}

	if (id == XmlObject::CREATION_DATE ||
		id == XmlObject::UPDATE_DATE ||
		id == XmlObject::AUTHOR ||
		id == XmlObject::COMMENT) return true;

	error_msg += "XMLImportHandler::setDeviceTag ";
	error_msg += i18n("error: tag ");
	error_msg += QString::number(id);
	error_msg += i18n(" ignored!\n");
	error_cnt++;
	return true;
}

bool XMLImportHandler::endElement(const QString&, const QString&, const QString& qName)
{
	int id = XmlObject::getID(qName);

	if (id == XmlObject::DEV_CONN)
		activeDevConn = (XWire *)NULL;

	if (id == XmlObject::WIRE_CONN)
		activeWireConn = (XWire *)NULL;

	if (id == XmlObject::DEVICE) {
		activeDev->parseEquation();

		if (activeDev->isTristate())
			activeDev->setTristateInputImport();

		activeDev->setImage();

		activeDev->setTextDevPos();

		activeDev->select(imp_selected);

		activeDev = (XDevice *)NULL;
	}

	if (id == XmlObject::WIRE)
		activeWire->select(imp_selected);
		activeWire = (XWire *)NULL;

	if (id == XmlObject::CIRCUIT) {
		// only important in top level circuit
		if (activeNetIdx == 2) {
			imp_dx = imp_dx_hidden;
			imp_dy = imp_dy_hidden;
			imp_selected = imp_selected_hidden;
		}

		activeNet[activeNetIdx]->select(imp_selected);

		// the image could have changed by io devices!
		if (activeNet[activeNetIdx]->type() == DeviceType::fNET)
			activeNet[activeNetIdx]->setImage();

		// parse the equations
		if (activeNet[activeNetIdx]->type() == DeviceType::fEQU)
			activeNet[activeNetIdx]->parseEquation();

		activeNet[activeNetIdx]->setTextDevPos();

		// close the circuit
		activeNetIdx--;
		if (activeNetIdx < 0) {
			error_msg += i18n("unexpected circuit close\n");
			error_cnt++;
			return false;
		}
	}

	if (id != activeTag[activeTagIdx--]) {
		error_msg += i18n("unexpected tag close <");
		error_msg += qName;
		error_msg += ">\n";
		error_cnt++;
		return false;
	}
	if (activeTagIdx < 0) {
		error_msg += i18n("negative tag stack .. while closing <");
		error_msg += qName;
		error_msg += ">\n";
		error_cnt++;
		return false;
	}

	return true;
}

bool XMLImportHandler::endDocument()
{
	return true;
}


