/*******************************************************/
/* methods for a net-widget                            */
/*                                                     */
/* this contains the current active circuit widget,    */
/* manages various modes, status, drawing and io       */
/*                                                     */
/* Andreas Rostin                                      */
/* 27.10.97                                            */
/*******************************************************/
#include <qpainter.h>
#include <qlistbox.h>
#include <qstring.h>
#include <qdir.h>
#include <qmessagebox.h>
#include <qpushbutton.h>
#include <qpopupmenu.h>
#include <qtimer.h>
#include <qwidget.h>
#include <qframe.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qdialog.h>
#include <qbuttongroup.h>
#include <qradiobutton.h>
#include <qpixmap.h>
#include <qlabel.h>
#include <qcombobox.h>

#include "mainw.h"
#include "netw.h"
#include "klogic.h"
#include "klogicIO.h"
#include "xnet.h"
#include "xdevice.h"
#include "deviceTypes.h" 
#include "selection.h"
#include "propOsz.h"
#include "propDev.h"
#include "propName.h"
#include "propPwr.h"
#include "propText.h"
#include "propSwitch.h"
#include "dlgEqu.h"
#include "xmlImport.h"

#include "netw.moc"
//********************************************
//* static methods of NetWidget              *
//********************************************

int NetWidget::STATsimStepTime = 40;
int NetWidget::STATsimBurst = 1;
QTimer NetWidget::simTimer;
QTimer NetWidget::shortcutWarningTimer;
int NetWidget::actualDevFunc = 0;
int NetWidget::simmode = 0;

// static method
int NetWidget::simTime()
{
	return STATsimStepTime;
}

// static method
void NetWidget::setSimTime(int newi)
{
	if ((newi >= MIN_SIMSTEPTIME) && (newi <=  MAX_SIMSTEPTIME)) {
		STATsimStepTime = newi;
		if (simTimer.isActive())
			simTimer.changeInterval(STATsimStepTime);
	}
}

// static method
int NetWidget::simBurst()
{
	return STATsimBurst;
}

// static method
void NetWidget::setSimBurst(int newi)
{
	if ((newi >= MIN_SIMBURST) && (newi <=  MAX_SIMBURST))
		STATsimBurst = newi;
}

// static method
// set current object for dropping
void NetWidget::setDevice(int newdev)
{
	actualDevFunc = newdev;
}

// static method
// change between permanent and stepping simulation mode
void NetWidget::setSimMode(int newmode)
{
	simmode = newmode;

	// permanent simulation mode
	if (simmode == MODE_SIM_MULT) {
		simTimer.start(STATsimStepTime);
	}
	else {
		simTimer.stop();
	}
}

// static method
// return current simulation mode
int NetWidget::getSimMode()
{
	return simmode;
}

//********************************************
//* methods of NetWidget                     *
//********************************************
NetWidget::NetWidget(QWidget *hparent, QWidget *parent) : QFrame(hparent)
{
	client = false;

	setBackgroundMode(PaletteBase);    // set widgets background
	setBackgroundColor(white);
	setFrameStyle( QFrame::Panel | QFrame::Raised );
	resize(VIRT_SCREEN_SIZE_X, VIRT_SCREEN_SIZE_Y);                  // set default size in pixels

	activeNet = new XDeviceNet();
	activeDev = (XDevice *)NULL;
	activeWire = (XWire *)NULL;
	activeSelection = new Selection();
	_parent = parent;

	// drawing mode
	setSimMode(NetWidget::MODE_SIM_MULT);
	// device to drop when in drop-mode
	actualDevFunc = 0;

	// popup for the right mouse key (on devices)
	rpop = new QPopupMenu();
	connect(rpop, SIGNAL (activated (int)), SLOT (rmenuCallback(int)));

	// popup for the right mouse key (on selections)
	rselpop = new QPopupMenu();
	rselpop->insertItem(i18n("Copy"), RPOPSEL_COPY);
	rselpop->insertItem(i18n("Paste"), RPOPSEL_PASTEAT);
	rselpop->insertItem(i18n("Cut"), RPOPSEL_CUT);
	connect(rselpop, SIGNAL (activated (int)), SLOT (rmenuSelCallback(int)));

	changed = false;
	movecursor = 0;
	rmenuActive = 0;

	runtimeShortcutWarning = true;

	connect(&simTimer, SIGNAL(timeout()), this, SLOT(simStep()));
	connect(&shortcutWarningTimer, SIGNAL(timeout()), this, SLOT(enableShortcutWarning()));
	connect(this, SIGNAL(netContentChanged()), parent, SLOT(repaintClients()));
	connect(this, SIGNAL(netContentUnchanged()), parent, SLOT(circuitUnchanged()));
}

NetWidget::NetWidget(QWidget *hparent, QWidget *parent, XDeviceNet *net)
	: QFrame(hparent)
{
	client = true;

	setBackgroundMode(PaletteBase);    // set widgets background
	setBackgroundColor(white);
	setFrameStyle( QFrame::Panel | QFrame::Raised );
	resize(VIRT_SCREEN_SIZE_X, VIRT_SCREEN_SIZE_Y);                // set default size in pixels

	activeNet = net;
	activeDev = (XDevice *)NULL;
	activeWire = (XWire *)NULL;
	activeSelection = new Selection();
	_parent = hparent;

	// device to drop when in drop-mode
	actualDevFunc = 0;

	// popup for the right mouse key (on devices)
	rpop = new QPopupMenu();
	connect(rpop, SIGNAL (activated (int)), SLOT (rmenuCallback(int)));

	// popup for the right mouse key (on selections)
	rselpop = new QPopupMenu();
	rselpop->insertItem(i18n("Copy"), RPOPSEL_COPY);
	rselpop->insertItem(i18n("Paste"), RPOPSEL_PASTEAT);
	rselpop->insertItem(i18n("Cut"), RPOPSEL_CUT);
	connect(rselpop, SIGNAL (activated (int)), SLOT (rmenuSelCallback(int)));

	changed = false;
	movecursor = 0;
	rmenuActive = 0;

	runtimeShortcutWarning = true;

	connect(&simTimer, SIGNAL(timeout()), this, SLOT(simStep()));
	connect(&shortcutWarningTimer, SIGNAL(timeout()), this, SLOT(enableShortcutWarning()));
	connect(this, SIGNAL(netContentChanged()), parent, SLOT(repaintClients()));
	connect(this, SIGNAL(netContentUnchanged()), parent, SLOT(circuitUnchanged()));
}

NetWidget::~NetWidget()
{
	if (client) return;

	delete activeNet;
	delete activeSelection;
}

void NetWidget::mousePressEvent( QMouseEvent *e )
{	QPainter p;

	if (e->button() == LeftButton) {
		if (rmenuActive) {
			rmenuActive = 0;
			return;
		}

		if (MainWidget::getMode() == MainWidget::MAIN_MODE_DRAW) {
			// device clicked? set active device
			// devices have priority - since we have bus devices
			activeDev = activeNet->containsDev(e->pos());
			if (activeDev) {
				activeDev->setOrigin();

				// if device is a push button device,  toggle its status
				if (activeDev->isInteractive() && !activeDev->isSwitchType()) {
					activeDev->toggleStaticInput();
					p.begin(this);
					activeNet->drawAll(this, &p);
					p.end();
				}

				return;
			}

			// wire clicked? set active wire
			activeWire = activeNet->containsWire(e->pos());
			if (activeWire) {
				return;
			}

			// nothing clicked? new wire
			p.begin(this);
			activeWire = activeNet->newWire(&p, e->pos());
			p.end();
			if (activeWire) {
				emit netContentChanged();
				return;
			}
		}
		if (MainWidget::getMode() == MainWidget::MAIN_MODE_DROP) {
			// move device
			activeDev = activeNet->containsDev(e->pos());
			if (activeDev) {
				activeDev->setOrigin();

				// if device is a push button device,  toggle its status
				if (activeDev->isInteractive() && !activeDev->isSwitchType()) {
					activeDev->toggleStaticInput();
					p.begin(this);
					activeNet->drawAll(this, &p);
					p.end();
				} else {
					p.begin(this);
					activeNet->drawDev(this, &p, activeDev);
					p.end();

					// redraw client widgets with the same content as this
					emit netContentChanged();
					changed = true;
				}
				return;
			}
			// create new device
			activeDev = activeNet->newDevice(actualDevFunc, e->pos().x(), e->pos().y());
			if (!activeDev && (actualDevFunc == DeviceType::fIN)) {
				QMessageBox::warning(_parent, i18n("device failure"), i18n("unable to add more named inputs\nresize this sub circuit"));
				emit netContentChanged();
				changed = true;
				return;
			}
			if (!activeDev && (actualDevFunc == DeviceType::fOUT)) {
				QMessageBox::warning(_parent, i18n("device failure"), i18n("unable to add more named outputs\nresize this sub circuit"));
				emit netContentChanged();
				changed = true;
				return;
			}
			if (!activeDev) {
				QMessageBox::warning(_parent, i18n("device failure"), i18n("panic: cannot create device!"));
				return;
			}
			if (activeNet->containsDev(activeDev)) {
				deleteDev(0);
				emit netContentChanged();
				changed = true;
				return;
			}
			activeDev->setOrigin();
			p.begin(this);
			activeNet->drawDev(this, &p, activeDev);
			p.end();
			emit netContentChanged();
			if (activeDev->drawGraph() && activeDev->graphEnabled())
				emit graphChanged();
			return;
		}
		if (MainWidget::getMode() == MainWidget::MAIN_MODE_SELECT) {
			if (activeSelection->getStatus() == SEL_EXISTS && activeSelection->onFrame(e->pos())) {
				// selection move mode
				activeSelection->beginMove(e->pos());
			} else if (activeSelection->getStatus() == SEL_EXISTS &&  NULL != (activeDev = activeNet->containsDev(e->pos()))) {
				if (activeSelection->contains(activeDev)) {
					// selection move mode
					activeSelection->beginMove(e->pos());
				} else {
					// select device, then selection move mode
					activeSelection->addTemp(activeDev);
					activeSelection->beginMove(e->pos());
					p.begin(this);
					activeSelection->erase(&p);
					activeNet->drawAll(this, &p);
					activeSelection->draw(&p);
					p.end();
				}
			} else {
				// new selection?
				activeSelection->setNet(activeNet);

				// select/deselect single device
				activeDev = activeNet->containsDev(e->pos());
				if (activeDev) {
					if (activeSelection->contains(activeDev)) activeSelection->remove(activeDev);
					else activeSelection->add(activeDev);
					activeDev = (XDevice *)NULL;
					activeWire = (XWire *)NULL;
					p.begin(this);
					activeSelection->erase(&p);
					activeNet->drawAll(this, &p);
					activeSelection->draw(&p);
					p.end();
					return;
				}

				// select/deselect single wire
				activeWire = activeNet->containsWire(e->pos());
				if (activeWire) {
					if (activeSelection->contains(activeWire)) activeSelection->remove(activeWire);
					else activeSelection->add(activeWire);
					activeDev = (XDevice *)NULL;
					activeWire = (XWire *)NULL;
					p.begin(this);
					activeSelection->erase(&p);
					activeNet->drawAll(this, &p);
					activeSelection->draw(&p);
					p.end();
					return;
				}

				// group selections
				p.begin(this);
				activeSelection->erase(&p);
				p.end();
				activeSelection->beginFrame(e->pos());
			}
			p.begin(this);
			activeSelection->erase(&p);
			activeNet->drawAll(this, &p);
			activeSelection->draw(&p);
			p.end();
			return;
		}
	}

	if (e->button() == RightButton) {
		// right clicked in selection mode? open popup to choose operation
		if (MainWidget::getMode() == MainWidget::MAIN_MODE_SELECT) {
			activeSelection->at(e->pos());
			selRightSelMenu(e);
			rmenuActive = 1;
			return;
		}

		// device right clicked? open popup to chooce operation
		if (NULL != (activeDev = activeNet->containsDev(e->pos()))) {
			selRightMenu(e);
			rmenuActive = 1;
			return;
		}

		// right clicked twice?
		if (rmenuActive) {
			rmenuActive = 0;
			return;
		}

		// wire right clicked? remove node 
		p.begin(this);
		if (activeNet->removeNode(&p, e->pos())) {
			activeNet->drawAll(this, &p);
			activeWire = (XWire *)NULL;
			emit netContentChanged();
			changed = true;
			p.end();
			return;
		}
		p.end();
	}
}

void NetWidget::mouseReleaseEvent( QMouseEvent *e)
{	int ret;
	QPainter p;

	if (e->button() == RightButton) return;

	if (MainWidget::getMode() == MainWidget::MAIN_MODE_SELECT) {
		if (activeSelection->getStatus() == SEL_START) {	// making frames
			activeSelection->endFrame();
			return;
		}
		if (activeSelection->getStatus() == SEL_MOVE) {	// moving frames or select devices
			activeSelection->endMove();

			if (! activeSelection->hasMoved()) {
				// select/deselect single device when group was not moved
				activeDev = activeNet->containsDev(e->pos());
				if (activeDev) {
					if (activeSelection->isTemp(activeDev)) activeSelection->fixTemp(activeDev);
					else activeSelection->remove(activeDev);
					activeDev = (XDevice *)NULL;
					activeWire = (XWire *)NULL;
					p.begin(this);
					activeSelection->erase(&p);
					activeNet->drawAll(this, &p);
					activeSelection->draw(&p);
					p.end();
					activeDev = (XDevice *)NULL;
					return;
				}
			}
			activeDev = activeNet->containsDev(e->pos());
			if (activeDev) {
				// look if some wires want to drop nodes..
				activeDev->garbageCollection();
				activeDev = (XDevice *)NULL;
			}
			return;
		}
		return;
	}

	// check for collision with another device
	if (activeDev) {
		//look for devices lying in the device 
		if (activeNet->containsDev(activeDev)) {
			activeDev->toOrigin();
			p.begin(this);
			activeNet->drawDev(this, &p, activeDev);
			p.end();
		}

		// if device was not moved and it is interactive, toggle its status
		if (activeDev->isInteractive() && activeDev->isOrigin(e->pos())) {
			activeDev->toggleStaticInput();
			p.begin(this);
			activeNet->drawAll(this, &p);
			p.end();
		}

		// look if some wires want to drop nodes..
		activeDev->garbageCollection();
		activeDev = (XDevice *)NULL;

		emit netContentChanged();
		changed = true;
	}
	if (activeWire) {
		// check for connection to device or another wire
		ret = activeNet->checkConnection(activeWire);
		switch (ret) {
		case NFAIL:
			QMessageBox::warning(_parent,
				i18n("connection failure"),
				i18n("only one output per wire allowed"));
			break;
		case DFAILMAXI:
			QMessageBox::warning(_parent,
				i18n("connection failure"),
				i18n("maximum number of inputs exeeded"));
			break;
		case DFAILMAXO:
			QMessageBox::warning(_parent,
				i18n("connection failure"),
				i18n("maximum number of outputs exeeded"));
			break;
		case WSHORT:
			QMessageBox::warning(_parent,
				i18n("connection failure"),
				i18n("shortcut\n"));
			break;
		default:
			break;
		}
		p.begin(this);
		activeWire->erase(&p);
		activeNet->releaseWire(&p, activeWire);
		activeWire = (XWire *)NULL;
		activeNet->drawAll(this, &p);
		p.end();
		emit netContentChanged();
		changed = true;
	}
}

void NetWidget::mouseMoveEvent( QMouseEvent *e )
{	QPainter p;

	if (MainWidget::getMode() == MainWidget::MAIN_MODE_SELECT) {
		if (FALSE == hasMouseTracking()) setMouseTracking(TRUE);
		if (activeSelection->getStatus() == SEL_EXISTS && activeSelection->onFrame(e->pos())) {
			if (!movecursor) {
				movecursor = 1;
				setCursor(sizeAllCursor);
				return;
			}
		} else {
			if (movecursor) {
				movecursor = 0;
				setCursor(arrowCursor);
				return;
			}
			if (activeSelection->getStatus() == SEL_START) {		// making frames
				activeSelection->changeFrame(e->pos());
				p.begin(this);
				activeSelection->erase(&p);
				activeNet->drawAll(this, &p);
				activeSelection->draw(&p);
				p.end();
				emit netContentChanged();
				return;
			}
			if (activeSelection->getStatus() == SEL_MOVE) {		// moving frames
				p.begin(this);
				activeSelection->moveFrame(e->pos(), &p);
				activeNet->drawAll(this, &p);
				p.end();
				emit netContentChanged();
				return;
			}
		}
		return;
	}
	if (TRUE == hasMouseTracking()) {
		activeDev = (XDevice *)NULL;
		activeWire = (XWire *)NULL;
		setMouseTracking(FALSE);
	}

	if (rmenuActive) return;

	// move wire/device
	if (activeDev) {
		if (activeDev->setPos(e->pos())) {
			p.begin(this);
			activeNet->drawDev(this, &p, activeDev);
			p.end();
			emit netContentChanged();
		}
	}
	if (activeWire) {
		p.begin(this);
		activeWire->erase(&p);
		activeWire->updateNode(e->pos());
		activeNet->drawAll(this, &p);
		p.end();
		emit netContentChanged();
	}
}

void NetWidget::mouseDoubleClickEvent( QMouseEvent *e )
{
	// device clicked?
	activeDev = activeNet->containsDev(e->pos());
	if (activeDev) {
		if (activeDev->type() == DeviceType::fSWI) {
			activeDev = (XDevice *)NULL;
			return;
		}

		// switch to sub circuit?
		if (toSubNet()) {
			emit netContentChanged();
			return;
		}

		// open device properties?
		if (openProp()) {
			emit netContentChanged();
			return;
		}
	}
}

// always call whith repaint=FALSE
void NetWidget::paintEvent(QPaintEvent *)
{	int x, y;
	int x1s, y1s, x2s, y2s;
	QPainter p;

	x1s = visi.x() / GRID * GRID - GRID;
	x2s = (visi.width() + visi.x()) / GRID * GRID + GRID;
	y1s = visi.y() / GRID * GRID - GRID;
	y2s = (visi.height() +  visi.y()) / GRID * GRID + GRID;

	if (!isActiveWindow()) erase();

	p.begin(this);
	p.setPen(black);
	for(x = x1s ; x < x2s ;x += GRID)
		for(y = y1s; y < y2s; y += GRID)
			p.drawPoint(x, y);
	activeNet->drawAll(this, &p);
	activeSelection->draw(&p);
	p.end();
}

int NetWidget::printNet(QPainter *p)
{
	activeNet->drawAll((QPaintDevice *)NULL, p);
	return 1;
}

// right mouse button popup entries
// menu depends on device type
void NetWidget::selRightMenu(QMouseEvent *e)
{
	if (!activeDev) return;

	rpop->clear();
	rpop->insertItem(i18n("properties"), RPOP_PROP);
	if (activeDev->type() == DeviceType::fNET) {
		rpop->insertItem(i18n("open"), RPOP_SUB);
		rpop->insertItem(i18n("open in window"), RPOP_SUBW);
		rpop->insertItem(i18n("display graph"), RPOP_SHOWGRAPH);
	}
	if (activeDev->type() == DeviceType::fEQU) {
		rpop->insertItem(i18n("equations"), RPOP_EQU);
	}
	if (activeDev->drawGraph()) {
		if (activeDev->graphEnabled()) {
			rpop->insertItem(i18n("hide graph"), RPOP_GRAPH);
		} else {
			rpop->insertItem(i18n("show graph"), RPOP_GRAPH);
		}
	}
	if (activeDev->type() == DeviceType::fEQU || activeDev->type() == DeviceType::fNET) {
		rpop->insertItem(i18n("add to lib"), RPOP_ADD_LIB);
	}

	rpop->insertSeparator();
	rpop->insertItem(i18n("remove"), RPOP_DEL);

	rpop->popup(mapToGlobal(e->pos()));
}

// callback for right mouse button popup
void NetWidget::rmenuCallback(int val)
{
	switch(val) {
		case RPOP_DEL:
			deleteDev(1);
			break;
		case RPOP_PROP:
			openProp();
			break;
		case RPOP_EQU:
			showDeviceEquations();
			break;
		case RPOP_SUB:
			toSubNet();	// to sub circuit, same window
			break;
		case RPOP_SUBW:
			toSubNetW();	// to sub circuit, new window
			break;
		case RPOP_GRAPH:
			if (!activeDev) return;
			if (activeDev->graphEnabled()) activeDev->enableGraph(0);
			else activeDev->enableGraph(1);
			emit graphChanged();
			break;
		case RPOP_SHOWGRAPH:
			if (activeDev->devIsNet())
				emit showGraph(activeDev->devIsNet());
			break;
		case RPOP_ADD_LIB:
			// send signal to main widget
			if (activeDev)
				emit addToLib(activeDev);
			break;
		default:
			break;
	}
	rmenuActive = 0;
}

// right mouse button popup entries on selections
void NetWidget::selRightSelMenu(QMouseEvent *e)
{	//(entries are static)
	rselpop->popup(mapToGlobal(e->pos()));
}

//callback for right mouse button popup on selections
// also called vom outside (mainw.ccp/menu)
void NetWidget::rmenuSelCallback(int val)
{	QPainter p;

	switch(val) {
		case RPOPSEL_CUT:
			p.begin(this);
			activeSelection->cut(&p, activeNet);
			activeNet->drawAll(this, &p);
			p.end();
			emit netContentChanged();
			emit graphChanged();
			changed = true;
			break;
		case RPOPSEL_COPY:
			activeSelection->copy(activeNet);
			break;
		case RPOPSEL_PASTE:
			activeSelection->at(QPoint(40, 10));
		case RPOPSEL_PASTEAT:
			p.begin(this);
			activeSelection->paste(&p, activeNet);
			activeSelection->erase(&p);
			activeNet->drawStatus(this, &p);
			activeNet->drawAll(this, &p);
			activeSelection->draw(&p);
			p.end();
			emit netContentChanged();
			emit graphChanged();
			changed = true;
			break;
		default:
			break;
	}
	rmenuActive = 0;
}

// remove old selections
void NetWidget::removeSelection()
{	QPainter p;

	p.begin(this);
	activeSelection->remove(&p, 1);
	activeSelection->erase(&p);
	activeNet->drawAll(this, &p);
	activeSelection->draw(&p);
	p.end();
}

// remove device
void NetWidget::deleteDev(int ask)
{	QPainter p;
	int simw_rel = 1;

	if (activeDev->type() == DeviceType::fNET) {
		if (ask) {
			if (1 == QMessageBox::warning(_parent, i18n("remove device"),
					i18n("remove entire sub-circuit?"), i18n("OK"), i18n("Cancel")))
			{
				activeDev = (XDevice *)NULL;
				return;
			}
		}
		emit netContentChanged();
		emit netDeleted(activeDev->devIsNet());
	}
	if (!activeDev->drawGraph() ||  !activeDev->graphEnabled())
		simw_rel = 0;

	p.begin(this);
	activeNet->deleteDevice(&p, activeDev);
	activeNet->drawAll(this, &p);
	p.end();
	activeDev = (XDevice *)NULL;

	if (simw_rel) emit graphChanged();
}

// open device properties?
int NetWidget::openProp()
{	QPainter p;

	// open properties for oszillators
	if (activeDev->type() == DeviceType::fOSC) {
		PropOsz *dlg = new PropOsz(_parent, i18n("oscillator properties"), activeNet, activeDev);
		if (simmode == MODE_SIM_MULT) simTimer.stop();
		dlg->exec();
		if (simmode == MODE_SIM_MULT) simTimer.start(STATsimStepTime);
		delete dlg;
		activeDev = (XDevice *)NULL;
		emit netContentChanged();
		changed = true;
		emit graphChanged();
		return 1;
	}

	// open properties for text devices
	if (activeDev->type() == DeviceType::fTXT) {
		PropText *dlg = new PropText(_parent, i18n("text label"), activeNet, activeDev);
		dlg->exec();
		delete dlg;
		activeDev = (XDevice *)NULL;
		emit netContentChanged();
		changed = true;
		return 1;
	}

	// open properties for power sources
	if (activeDev->type() == DeviceType::fPWR) {
		PropPwr *dlg = new PropPwr(_parent, i18n("power device properties"), activeNet, activeDev);
		dlg->exec();
		delete dlg;
		p.begin(this);
		activeNet->drawStatus(this, &p);
		p.end();
		activeDev = (XDevice *)NULL;
		emit netContentChanged();
		changed = true;
		emit graphChanged();
		return 1;
	}

	// open properties for switch devices
	if (activeDev->type() == DeviceType::fSWI) {
		PropSwitch *dlg = new PropSwitch(_parent, i18n("switch device properties"), activeNet, activeDev);
		if (simmode == MODE_SIM_MULT) simTimer.stop();
		dlg->exec();
		if (simmode == MODE_SIM_MULT) simTimer.start(STATsimStepTime);
		
		delete dlg;
		activeDev = (XDevice *)NULL;
		emit netContentChanged();
		changed = true;
		emit graphChanged();
		return 1;
	}

	// open properties for inputs and ouputs
	if ((activeDev->type() == DeviceType::fIN) ||
	    (activeDev->type() == DeviceType::fOUT)) {
		PropName *dlg = new PropName(_parent, i18n("interface device name"), this, activeNet, activeDev);
		if (simmode == MODE_SIM_MULT) simTimer.stop();
		dlg->exec();
		if (simmode == MODE_SIM_MULT) simTimer.start(STATsimStepTime);
		
		delete dlg;
		activeDev = (XDevice *)NULL;
		emit netContentChanged();
		changed = true;
		emit graphChanged();
		return 1;
	}

	// open dialog with common device properties
	PropDev *dlg = new PropDev(_parent, i18n("device properties"),this ,activeNet, activeDev);
	if (simmode == MODE_SIM_MULT) simTimer.stop();
	//dlg->show();
	dlg->exec();
	activeDev->forceOutputChange();
	if (simmode == MODE_SIM_MULT) simTimer.start(STATsimStepTime);

	delete dlg;
	activeDev = (XDevice *)NULL;
	emit netContentChanged();
	changed = true;
	emit graphChanged();
	return 1;
}

// switch to sub circuit in a new window?
int NetWidget::toSubNetW()
{	XDeviceNet *new_activeNet;

	if (!activeDev) return 0;

	new_activeNet = activeDev->devIsNet();
	if (new_activeNet) {
		// remove the current selection before switching to a new circuit
		removeSelection();
		new_activeNet->forceOutputChange();
		emit createNewWidget(new_activeNet);	// signal goes to main widget (mainw.cpp)
		return 1;
	}
	return 0;
}

// switch to sub circuit?
int NetWidget::toSubNet()
{	XDeviceNet *new_activeNet;

	new_activeNet = activeDev->devIsNet();
	if (new_activeNet) {
		// remove the current selection before switching to a new circuit
		removeSelection();

		activeNet = new_activeNet;
		activeDev = (XDevice *)NULL;
		activeWire = (XWire *)NULL;
		// signal mainw info about parent or not (changes toolbar)
		if (activeNet->parent())
			emit netChanged(1);
		else
			emit netChanged(0);
		activeNet->forceOutputChange();
		repaint(TRUE);
		return 1;
	}
	return 0;
}

// switch to parent of active net
void NetWidget::toParentNet()
{	XDeviceNet *new_activeNet;

	// remove the current selection before switching to a new circuit
	removeSelection();

	new_activeNet = activeNet->parent();
	if (!new_activeNet) {
		warning("no parent net!?");
		return;
	}
	activeNet = new_activeNet;
	activeWire = (XWire *)NULL;
	activeDev = (XDevice *)NULL;
	repaint(TRUE);
	// signal mainw info about having a parent (changes toolbar)
	if (activeNet->parent())
		emit netChanged(1);
	else
		emit netChanged(0);
	activeNet->forceOutputChange();
}

bool NetWidget::saveAsSubNet()
{	XDeviceNet *_net = activeNet->rootParent();

	_net = activeNet->rootParent();
	if (_net != activeNet) {	
		int ret = QMessageBox::warning(_parent,
				i18n("save circuit"),
				i18n("the current circuit is a sub circuit.\nsave it as a.."),
				i18n("main circuit"),
				i18n("sub circuit"));
		if (ret) return true;
	}
	return false;
}

int NetWidget::saveNet(QString filename, bool as_sub_net)
{	XDeviceNet *net;
	int ret;

	if (as_sub_net)
		net = activeNet;
	else
		net = activeNet->rootParent();

	emit netContentUnchanged();
	changed = false;

	klogicIO io(filename, net);
	if (as_sub_net) io.setSubFilename();

	ret = io.write(as_sub_net, false, 0, 0);

	if (ret <= 0) {
		QMessageBox::warning(_parent,
			i18n("save error"),
			i18n("unable to write file"));
		return 0;
	}

	return 1;
}

// load file
// init: refresh client widget
// return values:
// 0: error occured
// 1: ok, normal circuit
// 2: ok, subcircuit
int NetWidget::openNet(QString filename, bool main_circuit_changed)
{	XDeviceNet *_net;
	klogicIO *_io;
	QPainter p;
	bool create_sub = false;

	// -----------------------------------------
	// check file format
	// -----------------------------------------
	_io = new klogicIO(filename);
	int file_type = _io->checkFileFormat();

	if (file_type == klogicIO::XML_SUB || file_type == klogicIO::KLOGIC_SUB) {
		//fprintf(stderr, "SUB import\n");
		// -----------------------------------------
		// is a sub circuit
		// -----------------------------------------
		create_sub = true;
		_net = activeNet;
	} else if (file_type == klogicIO::XML_MAIN || file_type == klogicIO::KLOGIC_MAIN) {
		//fprintf(stderr, "MAIN import\n");
		// -----------------------------------------
		// replace main circuit: ask!
		// -----------------------------------------
		if (main_circuit_changed && !activeNet->rootParent()->empty()) {
			if (QMessageBox::Yes != QMessageBox::warning(_parent,
				i18n("new circuit"), 
				i18n("delete current circuit?"), 
				QMessageBox::Yes, 
				QMessageBox::No))
				return 1;
		}
		_net = activeNet->rootParent();
		activeNet = _net;
		activeDev = (XDevice *)NULL;
		activeWire = (XWire *)NULL;

		p.begin(this);
		activeNet->deleteNet(&p);
		p.end();
		emit netContentUnchanged();
		uniqueID::reset();
		changed = false;
	} else {
		QMessageBox::warning(_parent,
			i18n("read error"),
			i18n("unknown file format"));
		return 0;
	}

	if (simmode == MODE_SIM_MULT) simTimer.stop();
	_io->setNet(_net);

	// -----------------------------------------
	// read the file and create the circuit now
	// -----------------------------------------
	bool result = false;
	switch (file_type) {
		case klogicIO::XML_SUB:
		case klogicIO::XML_MAIN:
			result = _io->readXML();
			break;
		case klogicIO::KLOGIC_SUB:
		case klogicIO::KLOGIC_MAIN:
			result = _io->readNet(create_sub);
			break;
	}

	if (simmode == MODE_SIM_MULT) simTimer.start(STATsimStepTime);

	// -----------------------------------------
	// analyze the result
	// -----------------------------------------
	if (!result) {
		if (XMLImportHandler::errorsOccured()) {
			QMessageBox::warning(_parent,
				i18n("read error"),
				(const char *)XMLImportHandler::getErrors());
		} else {
			QMessageBox::warning(_parent,
			i18n("read error"),
			i18n("unable to read file"));
		}

		p.begin(this);
/*
		// remove useless nodes/wires
		activeNet->garbageCollection(&p);
*/
		// redraw everything
		activeNet->drawStatus(this, &p);
		p.end();

		return 0;
	}

	// -----------------------------------------
	// final steps
	// -----------------------------------------
	repaint(TRUE);

	// signal mainw info about having a parent or not (changes toolbar)
	if (activeNet->parent())
		emit netChanged(1);
	else
		emit netChanged(0);
	delete _io;
	emit graphChanged();
	emit netContentUnchanged();

	p.begin(this);
	// remove useless nodes/wires
	activeNet->garbageCollection(&p);
	// redraw everything
	activeNet->drawStatus(this, &p);
	p.end();

	if (create_sub) return 2;
	return 1;
}

int NetWidget::newNet()
{	QPainter p;

	if ((changed && !activeNet->empty() &&
			QMessageBox::Yes == (QMessageBox::warning(_parent,
			i18n("new circuit"), 
			i18n("delete current circuit?"), 
			QMessageBox::Yes,
			QMessageBox::No))) || (!changed)) {
		p.begin(this);
		activeNet->deleteNet(&p);
		p.end();
		repaint(TRUE);
		emit netContentUnchanged();
		changed = false;
		emit graphChanged();
		uniqueID::reset();
		return 1;
	}
	return 0;
}

// display dialog with the equations of the current circuit
void NetWidget::showCircuitEquations()
{
	if (!activeNet->checkCircuitNames()) {
		if (QMessageBox::No == (QMessageBox::warning(this,
				i18n("Parse circuit equations"),
				i18n("There are some non unique device names \nwithin your circuit ... \n\nrename them automaticly?"),
				QMessageBox::Yes,
				QMessageBox::No)))
			return;
		activeNet->unifyCircuitNames();
	}
	if (activeNet->hasImpossibleEquations()) {
		QMessageBox::information(this,
			i18n("Parse circuit equations"),
			i18n("Unable to display equations for circuits containing \nRAM or Tristate devices!"));
		return;
	} 
	DlgEqu *equation_dialog = new DlgEqu(this, i18n("Circuit Equation Editor"), activeNet);
	equation_dialog->exec();
	delete equation_dialog;
}

// display dialog with the equations of the active device
void NetWidget::showDeviceEquations()
{
	if (!activeDev) return;
	DlgEqu * equation_dialog = new DlgEqu(this, i18n("Device Equation Editor"), activeDev);
	equation_dialog->exec();
	delete equation_dialog;
}

// retrieve the active XDeviceNet
XDeviceNet *NetWidget::getActive()
{
	return activeNet;
}

// check if given net is this or a parent of this
int NetWidget::contained(XDeviceNet *rootnet)
{	XDeviceNet *_rootnet = activeNet;

	if (_rootnet == rootnet) return 1;
	while(NULL != (_rootnet = _rootnet->parent())) {
		if (_rootnet == rootnet) return 1;
	}
	return 0;
}

// private slot
// perform a simulation step
void NetWidget::simStep()
{	QPainter p;
	int ret;

	if (!client) {
		// process a simulation step
		activeNet->rootParent()->Burst(STATsimBurst);
		emit simStepped();
	}
	p.begin(this);
	// draw wires and devices with changed image
	ret = activeNet->drawStatus(this, &p);
	p.end();

	// runtime shortcut occured?
	if (ret == WSHORT && runtimeShortcutWarning) {
		if (simmode == MODE_SIM_MULT) simTimer.stop();
		QMessageBox::warning(_parent,
			i18n("runtime error"),
			i18n("shortcut\n"));
		if (simmode == MODE_SIM_MULT) simTimer.start(STATsimStepTime);
		runtimeShortcutWarning = false;
		shortcutWarningTimer.start(10000, true);	// single shot after 20 sec!
	}
}

// private slot
// enable runtime shortcut warning again, 10sec ather the last one
void NetWidget::enableShortcutWarning()
{
	runtimeShortcutWarning = true;
}

// public slot
// update visible rect, emitted by MainWidget
void NetWidget::visible(QRect newvisi)
{
	visi = newvisi;
}

// apply device defaults to all existing devices
void NetWidget::applyDefaults()
{	XDeviceNet *toplevel = activeNet->rootParent();

	toplevel->applyDefaults();
	repaint(TRUE);
}

// returns wether net has changed or not
bool NetWidget::hasChanged()
{
	return changed;
}

// reset change flag, mainwidget is the caller
void NetWidget::resetChanges()
{
	changed = false;
}

