// ------------------------------------------------
// bus device can be a multiplexer or demultiplexer
//
// multiplexer has one input and many outputs
// demultiplexer has many inputs and one output
//
// Andreas Rostin
// 07.03.2001
// ------------------------------------------------

#ifndef __DEVICE_BUS__
#define __DEVICE_BUS__

#include <qpainter.h>
#include "xdevice.h"

class BusDevice : public XDevice
{
public:
	BusDevice(int x, int y, int size = -1, int = -1, int undef = -1, int clock = -1)
		: XDevice(DeviceType::fBUS, x, y, size, 0, undef, clock)
	{
		init();
	}

	virtual ~BusDevice()
	{
	}

	// not in use .. (constructor)
	virtual void init()
	{
		setMux(false);	// fan in
		XDevice::setSize(8);

		setEquation();
		parseEquation();
		setImage();
	}

	virtual void setMux(bool flag)
	{
		mux_mode = flag;
		setImage();
		setEquation();
		parseEquation();
	}

	virtual bool isMux()
	{
		return mux_mode;
	}

	virtual bool setSize(int new_size)
	{
		bool retval = XDevice::setSize(new_size);
		setEquation();
		parseEquation();
		return retval;
	}

	virtual void * getInstance()
	{
		return (void *)this;
		
	}

	virtual void setEquation()
	{
		KlogicList<OutputInfo> output_list;
		KlogicList<OutputInfo> *ls;

		// create list with needed equations
		if (mux_mode) {
			fprintf(stdout, "mux mode\n");
			// -----------------------------------------------
			// multiplexing mode (fan out): single input, many output
			// -----------------------------------------------
			for(int i = 0; i < size(); i++) {
				OutputInfo *oi = new OutputInfo;

				oi->output_name.sprintf("D%d", i);
				oi->output_id = getOutputID(oi->output_name);

				oi->equation = "DI % ";	// bit operator (OP_BIT)
				oi->equation += QString::number(i);

				oi->output_type = Device::FINAL_OUTPUT;

				ls = output_list.Append(oi);
				ls->setText(oi->output_name);
				ls->setID2(oi->output_type);
			}
		} else {
			fprintf(stdout, "demux mode\n");
			// -----------------------------------------------
			// demultiplexing mode (fan in): many inputs, single output
			// -----------------------------------------------
			OutputInfo *oi = new OutputInfo;
			oi->output_name.sprintf("DO");
			oi->output_id = getOutputID(oi->output_name);
			oi->output_type = Device::FINAL_OUTPUT;

			QString input;
			for(int i = 0; i < size(); i++) {
				input.sprintf("D%d", i);
				if (oi->equation.isEmpty()) {
					oi->equation = input;
				} else {
					oi->equation += "  ";	// bin operator (OP_BIN)
					oi->equation += input;
				}
			}
			ls = output_list.Append(oi);
			ls->setText(oi->output_name);
			ls->setID2(oi->output_type);
		}

		// change device, but not the size
		setAllEquations(&output_list, false);

		// destroy list
		ls = output_list.First();
		while(ls) {
			delete ls->Get();
			ls = ls->Next();
		}
	}

	virtual bool hasClock()
	{
		return false;
	}

	virtual bool hasNamedInput()
	{
		return true;
	}

	virtual bool hasNamedOutput()
	{
		return true;
	}

	virtual bool sizeChangeable()
	{
		return true;
	}

	virtual void setImage()
	{	QPixmap *pic = getImage();
		QPainter p;
		QPoint pt(0,0);
		int width = 21;

		// resize image first, erase everything
		pic->resize(width, 6 + ((deviceSize - 1) * GRID));
		pic->fill();

		p.begin(pic);

		// top of the device
		pt.setY(pt.y() + 3);

		// body
		p.setPen(Qt::black);
		p.setBrush(Qt::black);
		p.drawRect(8, 0, width - 17, deviceSize * GRID);

		// input/output lines
		if (mux_mode) {
			// one input, many output
			for(int i = 0; i < deviceSize; i++) {

				// input line
				if (!i) p.drawLine(3, pt.y(), 8, pt.y());
				// output line
				p.drawLine(width - 9, pt.y(), width - 3, pt.y());

				pt.setY(pt.y() + GRID);
			}
		} else {
			// many input, one output
			for(int i = 0; i < deviceSize; i++) {
				// output line
				if (!i) p.drawLine(width - 9, pt.y(), width - 3, pt.y());
				// input line
				p.drawLine(3, pt.y(), 8, pt.y());

				pt.setY(pt.y() + GRID);
			}
		}

		drawConnectionLines(&p);
		XObject::setImage();

		p.end();
	}

	virtual void printImage(QPainter *, const QRect &)
	{
	}

	virtual bool outputIsActive()
	{
		return true;
	}

private:
	// type of bus: multiplexer or demultiplexer
	bool mux_mode;
};

#endif
