/***************************************************************************
                          			kinkattaviewlist.cpp
                             -------------------
    begin                : Fri Jan 15 2001
    copyright            : (C) 2000 by Benjamin Meyer
    email                : benjamin@csh.rit.edu
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "kinkattaviewlist.h"
#include <qapplication.h>
#include <qpixmap.h>
#include "buddylistwindow.h"

DynamicTip::DynamicTip( QWidget * parent )
    : QToolTip( parent )
{
    // no explicit initialization needed
}


void DynamicTip::maybeTip( const QPoint &pos )
{
  toolTipData r( ((KinkattaListView*)(parentWidget()->parentWidget()))->tip(pos) );
  if ( !(r.cords).isValid() )
	  return;
  tip( r.cords, r.tip );
}

/***************************************************************************
 * Constructor
 ***************************************************************************/
KinkattaListView::KinkattaListView(QWidget *parent, const char *name ) : KListView(parent,name) {
  currentSelectedItem = NULL;
	t = new DynamicTip( this->viewport() );
	connect(this, SIGNAL(pressed ( QListViewItem *)), this, SLOT(itemPressed(QListViewItem *)));
	internalCursor.setShape(0);
	this->setFocusPolicy(QWidget::ClickFocus);
	this->setCursor(internalCursor);

	buddyEdit= new EditBox(this->viewport(),"EditBox");
	buddyEdit->hide();
}

/***************************************************************************
 * Deconstructor
 ***************************************************************************/
KinkattaListView::~KinkattaListView(){
  //delete buddyEdit;
}

/***************************************************************************
 * The tooltip thinks it is time to show a tooltip, check with the main
 * window to see if that is so.
 ***************************************************************************/
toolTipData KinkattaListView::tip( const QPoint & p ){
  return (((BuddyListWindow*)(this->parentWidget()))->AmIATip(p));
}

/***************************************************************************
 * Item has been pressed, lets watch it.
 ***************************************************************************/
void KinkattaListView::itemPressed(QListViewItem *item){
  if (item != NULL){
    if (checkIfFolder(item) == false){
      currentSelectedItem = item;
      internalCursor.setShape(13);
      qApp->setOverrideCursor(internalCursor);
    }
  }
}

/***************************************************************************
 * A key has been pressed.
 ***************************************************************************/
void KinkattaListView::keyPressEvent(QKeyEvent *event){
	QListViewItemIterator it( this );
	QChar c = event->ascii();
	int key = event->key();
	bool found = FALSE;

	//from this point on, we need a selected item
	if( ! this->currentItem() )
		return;

	if( (key == Key_Backspace) || (key == Key_Delete) ){
		if( checkIfFolder( this->currentItem() ) )
			emit( removeFolder( ((BuddyListWindowItem*)this->currentItem())->name ) );
		else
			emit( removeBuddy( ((BuddyListWindowItem*)this->currentItem())->name ) );
		return;
	}

	//if we got here, we better damn well have a letter!
	//let the default action take care of everything else (PgUp, Home, etc)
	if( !c.isLetterOrNumber() ){
		KListView::keyPressEvent(event);
		return;
	}

	//skip to the current selection
	while( it.current() && ( it.current() != this->currentItem() ) )
		it++;

	it++; //don't check the current selection, start right below it

	for( ; it.current(); ++it){
		if( it.current()->text(0).lower().startsWith( c )  ){
			this->setCurrentItem( it.current() );
			found = TRUE;
			break;
		}
	}

	//if the item was not found, then we need to loop around
	if( !found ){
		QListViewItemIterator it2( this );
		for( ; it2.current(); ++it2){
			if( it2.current()->text(0).lower().startsWith( event->text().lower() )  ){
				this->setCurrentItem( it2.current() );
				found = TRUE;
				break;
			}
		}
	}

	if(found)
		ensureItemVisible( this->currentItem() );

}

/***************************************************************************
 * The mouse has been released.
 ***************************************************************************/
void KinkattaListView::contentsMouseReleaseEvent(QMouseEvent* event){
  QListViewItem *newFolder = this->currentItem();
  // Make sure they are both real.
  if (currentSelectedItem != NULL  && newFolder != NULL){
    // Make sure that they are not the same.
    if (this->isSelected(currentSelectedItem) == false){
      // Ok we have two valid items.
      // Is the new one a folder or buddy?
      if (checkIfFolder(newFolder) != true){
				// get the buddies parent
				newFolder = newFolder->parent();
			}
			// Just in case the parent was null
	if (newFolder != NULL){
	// If the new folder and buddies current parent are the same don't do anything.
        if (newFolder != 	currentSelectedItem->parent())
          emit moveItem(currentSelectedItem, newFolder);
      }
    }
  }
  currentSelectedItem = NULL;
  KListView::contentsMouseReleaseEvent(event);
  qApp->restoreOverrideCursor();
}

/***************************************************************************
 * Checks if the item that is selected is a folder or not.
 * @Return bool if "selected" is a folder.
 ***************************************************************************/
bool KinkattaListView::checkIfFolder(QListViewItem* selected){
	/*	if have parent: must be a buddy.
		*	if no parent: must be a folder.
		*/
	return (!selected->parent());
}

/***************************************************************************
 * Returns the current item (in BuddyListWindow*) form.
 ***************************************************************************/
BuddyListWindowItem* KinkattaListView::currentItem(){
  QListViewItem *cItem = KListView::currentItem();
	if (cItem == NULL)
		return NULL;
	else
		return (BuddyListWindowItem*)cItem;
}

/***************************************************************************
 * Returns the current name (helper function, really)
 ***************************************************************************/
QString KinkattaListView::currentName(){
  BuddyListWindowItem *cItem = currentItem();
	if (cItem == NULL)
		return QString::null;
	else
		return cItem->name;
}

/***************************************************************************
 * Overrides the default action of QEditLine  Hides this window.
 ***************************************************************************/
void KinkattaListView::focusOutEvent( QFocusEvent * ){
	currentSelectedItem = NULL;
	qApp->restoreOverrideCursor();
}

/**
	* BuddyListWindowItem.  Derived from QListViewItem so we can get some more
	* functionality out of it.  Dimi Shahbaz
	*/

BuddyListWindowItem::BuddyListWindowItem(KListView *parent, QString label1)
	: QListViewItem (parent, label1) {
  mDecoration = Online;
	fontSettings = setup::settings()->buddylistFontSettings;
}

BuddyListWindowItem::BuddyListWindowItem(QListViewItem *parent, QString label1)
	: QListViewItem (parent, label1) {
  mDecoration = Online;
	fontSettings = setup::settings()->buddylistFontSettings;
}

int BuddyListWindowItem::width( const QFontMetrics& ,
			  const KListView* lv, int c ) const{
	QFont f;

	if( mDecoration == Offline ){
		f = fontSettings->buddylist_OfflineFont;
	} else if( mDecoration == SignOn ) {
		f = fontSettings->buddylist_SignOnFont;
	} else { //( mDecoration == Online )
		f = fontSettings->buddylist_OnlineFont;
	}

	QFontMetrics newFm(f);

	return QListViewItem::width( newFm, lv, c );

}

void BuddyListWindowItem::updateFontSettings(){
	fontSettings = setup::settings()->buddylistFontSettings;
}

BuddyListWindowItem::~BuddyListWindowItem(){

}

void BuddyListWindowItem::setDecoration(int d){
	mDecoration = d;
}

int BuddyListWindowItem::decoration(){
	return mDecoration;
}

void BuddyListWindowItem::paintCell(QPainter *p, const QColorGroup &cg, int column, int width, int align){
	if(column != 0) //we only want to deal with the 1st column (index = 0)
		return QListViewItem::paintCell(p, cg, column, width, align);

	if(mDecoration == Online) //no need to bother with the default view
		return QListViewItem::paintCell(p, cg, column, width, align);

	if(!p)
		return;

	const QPixmap *icon = pixmap(column);
	int iconWidth = icon ? icon->width() : 0;
	int stringWidthPx;

	//QFont f = p->font(); //Much better than: QFont::defaultFont();
	QFont f;
	if(mDecoration == Offline){
		f = fontSettings->buddylist_OfflineFont;
	} else if(mDecoration == SignOn) {
		f = fontSettings->buddylist_SignOnFont;
	} else {
		return;
	}

	QFontMetrics fm(f);
	stringWidthPx = fm.width(text(column));

	QString name = text(column);
	QString tmpStr = name;
	QString ellipsis = "...";
	int i = 1;
	while ( fm.width( tmpStr ) + iconWidth > width){
		tmpStr = name;
		tmpStr.truncate( tmpStr.length() - i);
		if(tmpStr.length() == 1){ //always leave the last character there, like Qt does
			tmpStr += ellipsis;
			break;
		}
		tmpStr += ellipsis;
		i++;
	}
	name = tmpStr;

	int r = (icon ? iconWidth : 1) + listView()->itemMargin();
	if(!this->isSelected()){ //draw dark text on light background
		p->fillRect( 1, 0, width-1, height(), cg.brush( QColorGroup::Base ));

		if(!name.isEmpty()){
			if(mDecoration == Offline){
				int r, g, b;
				cg.text().rgb(&r, &g, &b);
				p->setPen( QColor( r+128 < 255 ? r+128 : 255,
														g+128 < 255 ? r+128 : 255,
														b+128 < 255 ? r+128 : 255));			//lighter than text() color
			}
			else {//mDecoration == SignOn
				p->setPen( cg.text() );			//normal text color
			}

			p->setFont( f );
			p->drawText( r, 0, width - r, height(), Qt::AlignLeft|Qt::AlignVCenter, name );
		}
	} else{ //this->isSelected() //draw light text on dark background
		p->fillRect( 1, 0, width-1, height(), cg.brush( QColorGroup::Highlight ));

		if(!name.isEmpty()){
			p->setPen( cg.highlightedText() );
			p->setFont( f );
			p->drawText( r, 0, width - r, height(), Qt::AlignLeft|Qt::AlignVCenter, name );
		}
	}

	if(icon){
		p->drawPixmap( 1, 1 , *icon);
		//r += icon->width() + listView()->itemMargin();
	}

}



/***************************************************************************
 * Constructor
 ***************************************************************************/
EditBox::EditBox(QWidget *parent, const char *name ) : QLineEdit(parent,name) {
  focus = 0;
}

/***************************************************************************
 * Deconstructor
 ***************************************************************************/
EditBox::~EditBox(){
}

/***************************************************************************
 * Overrides the default action of QEditLine  Hides this window.
 ***************************************************************************/
void EditBox::focusOutEvent( QFocusEvent * ){
  emit returnPressed();
  if (focus == 1){
    this->hide();
    focus = 0;
  }
  else
    focus++;
}


// KinkattaListView.cpp


