/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2008  g新部 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implwebkit_gtkd warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include <glib/gi18n.h>

#include "kz-webkit-gtk.h"

#include "kz-embed.h"

typedef struct _KzWebKitGtkPrivate	KzWebKitGtkPrivate;
struct _KzWebKitGtkPrivate
{
	gint progress;
	gchar *location;
	gchar *title;
	gchar *link_message;
	gboolean lock;
	gboolean is_loading;
};
#define KZ_WEBKIT_GTK_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), KZ_TYPE_WEBKIT_GTK, KzWebKitGtkPrivate))

static GType           kz_type_webkit_gtk = 0;
static WebKitWebViewClass *kz_webkit_gtk_parent_class;

GtkWidget  *kz_webkit_gtk_new          (void);
static void kz_webkit_gtk_class_init   (KzWebKitGtkClass *klass);
static void kz_webkit_gtk_iface_init   (KzEmbedIFace     *iface);
static void kz_webkit_gtk_init         (KzWebKitGtk      *webkit_gtk);
static void kz_webkit_gtk_dispose      (GObject          *object);

static void	    load_url               (KzEmbed      *kzembed,
					    const gchar  *url);
static void	    stop_load              (KzEmbed      *kzembed);
static void	    reload		   (KzEmbed      *kzembed,
					    KzEmbedReloadFlag flags);
static void	    go_back                (KzEmbed      *kzembed);
static void	    go_forward             (KzEmbed      *kzembed);
static gboolean	    can_go_back            (KzEmbed      *kzembed);
static gboolean	    can_go_forward         (KzEmbed      *kzembed);
static gboolean     can_cut_selection      (KzEmbed      *kzembed);
static gboolean     can_copy_selection     (KzEmbed      *kzembed);
static gboolean     can_paste              (KzEmbed      *kzembed);
static void	    cut_selection          (KzEmbed      *kzembed);
static void	    copy_selection         (KzEmbed      *kzembed);
static void	    paste                  (KzEmbed      *kzembed);
static void	    select_all             (KzEmbed      *kzembed);
static gdouble	    get_progress	   (KzEmbed      *kzembed);
static const gchar *get_title              (KzEmbed      *kzembed);
static const gchar *get_location           (KzEmbed      *kzembed);
static gchar       *get_link_message       (KzEmbed      *kzembed);
static gchar	   *ensure_title           (KzEmbed      *kzembed);
static gboolean     selection_is_collapsed (KzEmbed      *kzembed);
static gboolean     is_loading             (KzEmbed      *kzembed);

static void	cb_title_changed   (WebKitWebView* web_view,
				    WebKitWebFrame* web_frame,
				    const gchar* title,
				    gpointer data);
static void	cb_progress_changed(WebKitWebView* web_view,
				    gint progress,
				    gpointer data);
static void	cb_load_committed  (WebKitWebView* web_view,
				    WebKitWebFrame* frame,
				    gpointer data);
static void	cb_load_started    (WebKitWebView* web_view,
				    WebKitWebFrame* frame,
				    gpointer data);
static void	cb_load_finished   (WebKitWebView* web_view,
				    WebKitWebFrame* frame,
				    gpointer data);
static void	cb_hover_link      (WebKitWebView* web_view,
				    const gchar *title,
				    const gchar *location,
				    gpointer data);

static void set_default_preferences (KzWebKitGtk *webkit);

void
kz_webkit_gtk_register_type (GTypeModule *module)
{
	static const GTypeInfo kz_webkit_gtk_info =
	{
		sizeof (KzWebKitGtkClass),
		NULL,		/* base_init */
		NULL,		/* base_finalize */
		(GClassInitFunc) kz_webkit_gtk_class_init,
		NULL,		/* class_finalize */
		NULL,		/* class_data */
		sizeof (KzWebKitGtk),
		0,		/* n_preallocs */
		(GInstanceInitFunc) kz_webkit_gtk_init,
	};

	const GInterfaceInfo kz_embed_info =
	{
		(GInterfaceInitFunc) kz_webkit_gtk_iface_init,
		NULL,
		NULL
	};

	kz_type_webkit_gtk = g_type_module_register_type(module,
						 WEBKIT_TYPE_WEB_VIEW,
						 "KzWebKitGtk",
						 &kz_webkit_gtk_info,
						 (GTypeFlags)0);

	g_type_module_add_interface(module,
				    KZ_TYPE_WEBKIT_GTK,
				    KZ_TYPE_EMBED,
				    &kz_embed_info);
}

GType
kz_webkit_gtk_get_type (void)
{
       return kz_type_webkit_gtk;
}

static void
kz_webkit_gtk_class_init (KzWebKitGtkClass *klass)
{
	GObjectClass *object_class;
	GtkWidgetClass *widget_class;

	kz_webkit_gtk_parent_class = WEBKIT_WEB_VIEW_CLASS(g_type_class_peek_parent(klass));
	object_class = G_OBJECT_CLASS(klass);
	widget_class = GTK_WIDGET_CLASS(klass);

	object_class->dispose = kz_webkit_gtk_dispose;

	g_type_class_add_private(object_class, sizeof(KzWebKitGtkPrivate));
}


static void
kz_webkit_gtk_iface_init (KzEmbedIFace *iface)
{
	iface->load_url               = load_url;
	iface->view_source            = NULL;
	iface->is_loading             = is_loading;
	iface->get_title              = get_title;
	iface->get_location           = get_location;
	iface->ensure_title           = ensure_title;
	iface->get_link_message       = get_link_message;
	iface->get_progress           = get_progress;
	iface->can_cut_selection      = can_cut_selection;
	iface->can_copy_selection     = can_copy_selection;
	iface->can_paste              = can_paste;
	iface->cut_selection          = cut_selection;
	iface->copy_selection         = copy_selection;
	iface->paste                  = paste;
	iface->select_all             = select_all;
	iface->get_selection_string   = NULL;
	iface->find                   = NULL;
	iface->incremental_search     = NULL;
	iface->selection_is_collapsed = selection_is_collapsed;
	iface->get_links              = NULL;
	iface->copy_page              = NULL;
	iface->shistory_copy          = NULL;
	iface->shistory_get_pos       = NULL;
	iface->shistory_get_nth       = NULL;
	iface->reload                 = reload;
	iface->stop_load              = stop_load;
	iface->go_back                = go_back;
	iface->go_forward             = go_forward;
	iface->can_go_back            = can_go_back;
	iface->can_go_forward         = can_go_forward;
	iface->can_go_nav_link        = NULL;
	iface->go_nav_link            = NULL;
	iface->append_nav_link        = NULL;
	iface->set_nav_link           = NULL;
	iface->set_nth_nav_link       = NULL;
	iface->get_nav_link           = NULL;
	iface->get_nth_nav_link       = NULL;
	iface->get_nav_links          = NULL;
	iface->go_history_index       = NULL;
	iface->do_command             = NULL;
	iface->can_do_command         = NULL;
	iface->get_lock               = NULL;
	iface->set_lock               = NULL;
	iface->get_body_text          = NULL;
#if 0
	iface->get_selection_source   = NULL;
#endif
	iface->set_encoding           = NULL;
	iface->get_encoding           = NULL;
	iface->print                  = NULL;
	iface->print_preview          = NULL;
	iface->get_printer_list       = NULL;
	iface->create_thumbnail       = NULL;
	iface->save_with_content      = NULL;
	iface->set_text_into_textarea = NULL;
	iface->get_text_from_textarea = NULL;
	iface->zoom_set               = NULL;
	iface->zoom_get               = NULL;
	iface->set_text_size          = NULL;
	iface->get_text_size          = NULL;
	iface->get_html_with_contents = NULL;
	iface->set_history            = NULL;
	iface->get_history            = NULL;
	iface->get_last_modified      = NULL;
	iface->fine_scroll            = NULL;
	iface->page_up                = NULL;
	iface->page_down              = NULL; 
	iface->get_allow_javascript   = NULL;
	iface->set_allow_javascript   = NULL; 
	iface->get_allow_images       = NULL;
	iface->set_allow_images       = NULL;
#if 0
	iface->set_edit_mode          = NULL;
	iface->set_vwebkit_gtkw_mode          = NULL;
#endif

	iface->link_message           = NULL;
	iface->js_status              = NULL;
	iface->location               = NULL;
	iface->title                  = NULL;
	iface->progress               = NULL;
	iface->net_start              = NULL;
	iface->net_stop               = NULL;
	iface->new_window             = NULL;
	iface->open_uri               = NULL;
	iface->size_to                = NULL;
	iface->dom_key_down           = NULL;
	iface->dom_key_press          = NULL;
	iface->dom_key_up             = NULL;
	iface->dom_mouse_down         = NULL;
	iface->dom_mouse_up           = NULL;
	iface->dom_mouse_click        = NULL;
	iface->dom_mouse_dbl_click    = NULL;
	iface->dom_mouse_over         = NULL;
	iface->dom_mouse_out          = NULL;
	iface->security_change        = NULL;
	iface->status_change          = NULL;
}

static void
kz_webkit_gtk_init (KzWebKitGtk *webkit)
{
	KzWebKitGtkPrivate *priv = KZ_WEBKIT_GTK_GET_PRIVATE(webkit);

	priv->location = NULL;
	priv->title = NULL;
	priv->link_message = NULL;

	priv->progress = 0;

	priv->lock = FALSE;
	priv->is_loading = FALSE;

	g_signal_connect(webkit, "title-changed",
			 G_CALLBACK(cb_title_changed), NULL);
	g_signal_connect(webkit, "load-progress-changed",
			 G_CALLBACK(cb_progress_changed), NULL);
	g_signal_connect(webkit, "load-committed",
			 G_CALLBACK(cb_load_committed), NULL);
	g_signal_connect(webkit, "load-started",
			 G_CALLBACK(cb_load_started), NULL);
	g_signal_connect(webkit, "load-finished",
			 G_CALLBACK(cb_load_finished), NULL);
	g_signal_connect(webkit, "hovering-over-link",
			 G_CALLBACK(cb_hover_link), NULL);
	set_default_preferences(webkit);
}

static void
kz_webkit_gtk_dispose (GObject *object)
{
	KzWebKitGtkPrivate *priv = KZ_WEBKIT_GTK_GET_PRIVATE(object);
	if (priv->location)
		g_free(priv->location);
	if (priv->title)
		g_free(priv->title);
	if (priv->link_message)
		g_free(priv->link_message);
	priv->location = NULL;
	priv->title = NULL;
	priv->link_message = NULL;

	if (G_OBJECT_CLASS(kz_webkit_gtk_parent_class)->dispose)
		G_OBJECT_CLASS(kz_webkit_gtk_parent_class)->dispose(object);
}

GtkWidget *
kz_webkit_gtk_new (void)
{
	return GTK_WIDGET(g_object_new(KZ_TYPE_WEBKIT_GTK,
				       NULL));
}

static void
load_url (KzEmbed *kzembed, const gchar  *url)
{
	g_return_if_fail(KZ_IS_WEBKIT_GTK(kzembed));

	webkit_web_view_open(WEBKIT_WEB_VIEW(kzembed), url);
}

static void
reload (KzEmbed *kzembed, KzEmbedReloadFlag flags)
{
	g_return_if_fail(KZ_IS_WEBKIT_GTK(kzembed));

	webkit_web_view_reload(WEBKIT_WEB_VIEW(kzembed));
}

static void
stop_load (KzEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_WEBKIT_GTK(kzembed));

	webkit_web_view_stop_loading(WEBKIT_WEB_VIEW(kzembed));
}

static void
go_back (KzEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_WEBKIT_GTK(kzembed));

	webkit_web_view_go_back(WEBKIT_WEB_VIEW(kzembed));
}

static void
go_forward (KzEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_WEBKIT_GTK(kzembed));

	webkit_web_view_go_forward(WEBKIT_WEB_VIEW(kzembed));
}

static gboolean
can_go_back (KzEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_WEBKIT_GTK(kzembed), FALSE);

	return webkit_web_view_can_go_back(WEBKIT_WEB_VIEW(kzembed));
}

static gboolean
can_go_forward (KzEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_WEBKIT_GTK(kzembed), FALSE);

	return webkit_web_view_can_go_forward(WEBKIT_WEB_VIEW(kzembed));
}

static gboolean
can_cut_selection (KzEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_WEBKIT_GTK(kzembed), FALSE);

	return webkit_web_view_can_cut_clipboard(WEBKIT_WEB_VIEW(kzembed));
}

static gboolean
can_copy_selection (KzEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_WEBKIT_GTK(kzembed), FALSE);

	return webkit_web_view_can_copy_clipboard(WEBKIT_WEB_VIEW(kzembed));
}

static gboolean
can_paste (KzEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_WEBKIT_GTK(kzembed), FALSE);

	return webkit_web_view_can_paste_clipboard(WEBKIT_WEB_VIEW(kzembed));
}

static void
cut_selection (KzEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_WEBKIT_GTK(kzembed));

	webkit_web_view_cut_clipboard(WEBKIT_WEB_VIEW(kzembed));
}

static void
copy_selection (KzEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_WEBKIT_GTK(kzembed));

	webkit_web_view_copy_clipboard(WEBKIT_WEB_VIEW(kzembed));
}

static void
paste (KzEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_WEBKIT_GTK(kzembed));

	webkit_web_view_paste_clipboard(WEBKIT_WEB_VIEW(kzembed));
}

static void
select_all (KzEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_WEBKIT_GTK(kzembed));

	webkit_web_view_select_all(WEBKIT_WEB_VIEW(kzembed));
}

static const gchar *
get_title (KzEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_WEBKIT_GTK(kzembed), NULL);

	return KZ_WEBKIT_GTK_GET_PRIVATE(kzembed)->title;
}

static const gchar *
get_location (KzEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_WEBKIT_GTK(kzembed), NULL);

	return KZ_WEBKIT_GTK_GET_PRIVATE(kzembed)->location;
}

static gchar *
get_link_message (KzEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_WEBKIT_GTK(kzembed), NULL);

	return g_strdup(KZ_WEBKIT_GTK_GET_PRIVATE(kzembed)->link_message);
}

static gchar *
ensure_title (KzEmbed *kzembed)
{
	const gchar *title;

	title = get_title(kzembed);
	if (title)
		return g_strdup(title);

	title = get_location(kzembed);
	if (title)
		return g_strdup(title);

	return g_strdup(_("No title"));
}

static gdouble
get_progress (KzEmbed *kzembed)
{
	return (gdouble) KZ_WEBKIT_GTK_GET_PRIVATE(kzembed)->progress / 100;
}

static gboolean
selection_is_collapsed (KzEmbed *kzembed)
{
	return webkit_web_view_has_selection(WEBKIT_WEB_VIEW(kzembed));
}

static gboolean
is_loading (KzEmbed *kzembed)
{
	return KZ_WEBKIT_GTK_GET_PRIVATE(kzembed)->is_loading;
}

static void
cb_title_changed (WebKitWebView* web_view,
		  WebKitWebFrame* web_frame,
		  const gchar* title,
		  gpointer data)
{
	KzWebKitGtkPrivate *priv = KZ_WEBKIT_GTK_GET_PRIVATE(web_view);
	if (priv->title)
		g_free(priv->title);
	priv->title = g_strdup(title);

	g_signal_emit_by_name(web_view, "kz-title");
}

static void
cb_progress_changed (WebKitWebView* web_view,
		     gint progress,
		     gpointer data)
{
	KzWebKitGtkPrivate *priv = KZ_WEBKIT_GTK_GET_PRIVATE(web_view);
	priv->progress = progress;

	g_signal_emit_by_name(web_view, "kz-progress");
}

static void
cb_load_committed (WebKitWebView* web_view,
	           WebKitWebFrame* frame,
		   gpointer data)
{
	KzWebKitGtkPrivate *priv = KZ_WEBKIT_GTK_GET_PRIVATE(web_view);
	const gchar* location = webkit_web_frame_get_uri(frame);
	if (priv->location)
		g_free(priv->location);
	priv->location = g_strdup(location);
	g_signal_emit_by_name(web_view, "kz-location");
}

static void
cb_load_started (WebKitWebView* web_view,
	          WebKitWebFrame* frame,
		  gpointer data)
{
	KZ_WEBKIT_GTK_GET_PRIVATE(web_view)->is_loading = TRUE;
	g_signal_emit_by_name(web_view, "kz-net-start");
}

static void
cb_load_finished (WebKitWebView* web_view,
	          WebKitWebFrame* frame,
		  gpointer data)
{
	KZ_WEBKIT_GTK_GET_PRIVATE(web_view)->is_loading = FALSE;
	g_signal_emit_by_name(web_view, "kz-net-stop");
}

static void
cb_hover_link (WebKitWebView* web_view,
	       const gchar *title,
	       const gchar *location,
	       gpointer data)
{
	KzWebKitGtkPrivate *priv = KZ_WEBKIT_GTK_GET_PRIVATE(web_view);

	if (priv->link_message)
		g_free(priv->link_message);
	priv->link_message = g_strdup(location);

	g_signal_emit_by_name(web_view, "kz-link-message");
}

static void
set_font_preferences (KzProfile *profile, WebKitWebSettings *settings)
{
	/* font settings */
	GList *list, *node;
	gint size_variable = -1, size_fixed = -1;
	gint min_size_variable = -1, min_size_fixed = -1;
	gchar langs[1024], *lang = NULL;
	gchar *serif_font, *sans_font, *mono_font;
	double dpi_pixel_ratio = 72.0 / 96.0;

	/* workaround */
	if (kz_profile_get_value(profile, "Language", "accept_languages", 
				&langs, G_N_ELEMENTS(langs),
				KZ_PROFILE_VALUE_TYPE_STRING))
	{
		gchar **split_str;
		split_str = g_strsplit(langs, ",", 1);

		if (split_str[0])
		{
			lang = g_strdup(split_str[0]);
			g_strfreev(split_str);
		}
	}
	if (!lang)
		lang = g_strdup("");

	serif_font = g_strdup_printf("name_serif_%s", lang);
	sans_font = g_strdup_printf("name_sans-serif_%s", lang);
	mono_font = g_strdup_printf("name_monospace_%s", lang);

	list = kz_profile_enum_key(profile, "Font", TRUE);
	for (node = list; node; node = g_list_next(node))
	{
		const gchar *key = (const gchar*)node->data;

		if (!key || !*key) continue;
		if (g_str_has_prefix(key, "size_variable_"))
		{
			gint value;
			kz_profile_get_value(profile, "Font", key,
					&value, sizeof(value),
					KZ_PROFILE_VALUE_TYPE_INT);
			if (value > size_variable)
				size_variable = value;
		}
		else if (g_str_has_prefix(key, "size_fixed_"))
		{
			gint value;
			kz_profile_get_value(profile, "Font", key,
					&value, sizeof(value),
					KZ_PROFILE_VALUE_TYPE_INT);
			if (value > size_fixed)
				size_fixed = value;
		}
		else if (g_str_has_prefix(key, "min-size_variable_"))
		{
			gint value;
			kz_profile_get_value(profile, "Font", key,
					&value, sizeof(value),
					KZ_PROFILE_VALUE_TYPE_INT);
			if (value > min_size_variable)
				min_size_variable = value;
		}
		else if (g_str_has_prefix(key, "min-size_fixed_"))
		{
			gint value;
			kz_profile_get_value(profile, "Font", key,
					&value, sizeof(value),
					KZ_PROFILE_VALUE_TYPE_INT);
			if (value > min_size_fixed)
				min_size_fixed = value;
		}
		else if (!strcmp(key, serif_font))
		{
			gchar value[1024];
			kz_profile_get_value(profile, "Font", key,
					&value, strlen(value)+1,
					KZ_PROFILE_VALUE_TYPE_STRING);
			g_object_set(settings,
				     "serif-font-family", value,
				     "default-font-family", value,
				     NULL);
		}
		else if (!strcmp(key, sans_font))
		{
			gchar value[1024];
			kz_profile_get_value(profile, "Font", key,
					&value, strlen(value)+1,
					KZ_PROFILE_VALUE_TYPE_STRING);
			g_object_set(settings,
				     "sans-serif-font-family", value,
				     NULL);
		}
		else if (!strcmp(key, mono_font))
		{
			gchar value[1024];
			kz_profile_get_value(profile, "Font", key,
					&value, strlen(value)+1,
					KZ_PROFILE_VALUE_TYPE_STRING);
			g_object_set(settings,
				     "monospace-serif-font-family", value,
				     NULL);
		}
	}
	g_list_free(list);
	g_free(serif_font);
	g_free(sans_font);
	g_free(mono_font);

	if (size_variable >= 0)
		g_object_set(settings, "default-font-size", (gint)(size_variable * dpi_pixel_ratio), NULL);
	if (size_fixed >= 0)
		g_object_set(settings, "default-monospace-font-size", (gint)(size_fixed * dpi_pixel_ratio), NULL);
	if (min_size_variable >= 0)
		g_object_set(settings, "minimum-font-size", (gint)(min_size_variable * dpi_pixel_ratio), NULL);
	if (min_size_fixed >= 0)
		g_object_set(settings, "minimum-logical-font-size", (gint)(min_size_fixed * dpi_pixel_ratio), NULL);
}

static void
set_default_preferences (KzWebKitGtk *webkit)
{
	KzProfile *profile = KZ_GET_GLOBAL_PROFILE;
	gchar value[1024];

	WebKitWebSettings *settings;
	settings = webkit_web_view_get_settings(WEBKIT_WEB_VIEW(webkit));
	if (!settings)
		return;

	if (kz_profile_get_value(profile, "Language", "charset_default", 
				&value, G_N_ELEMENTS(value),
				KZ_PROFILE_VALUE_TYPE_STRING))
	{
		g_object_set(settings, "default-encoding", value, NULL);
	}

	set_font_preferences (profile, settings);
	
	g_object_set(settings,
		     "enable-plugins", TRUE,
		     "enable-scripts", TRUE,
		     "auto-load-images", TRUE,
		     NULL);
}

