/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2004 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *  $Id: kz-gtk-webcore.cpp 3282 2007-10-15 10:51:03Z ikezoe $
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif /* HAVE_CONFIG_H */

#include <glib/gi18n.h>
#include <math.h>

#include "kazehakase.h"
#include "kz-gtk-webcore.h"
#include "kz-embed-prefs.h"
#include "kz-embed.h"
#include "kz-prompt-dialog.h"
#include "kz-proxy-item.h"
#include "gtk-utils.h"
#include "utils.h"

typedef struct _KzGtkWebcorePrivate	KzGtkWebcorePrivate;
struct _KzGtkWebcorePrivate
{
	gint cur_requests;
	gint total_requests;
	gchar *link_message;
	guint32 button_press_time;

	gboolean is_loading;
	gboolean lock;
	WebiSettings *settings;
};
#define KZ_GTK_WEBCORE_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), KZ_TYPE_GTK_WEBCORE, KzGtkWebcorePrivate))

static GType kz_type_gtk_webcore = 0;
static WebiClass *kz_gtk_webcore_parent_class;

GtkWidget  *kz_gtk_webcore_new          (void);
static void kz_gtk_webcore_class_init   (KzGtkWebcoreClass *klass);
static void kz_gtk_webcore_iface_init   (KzEmbedIFace *iface);
static void kz_gtk_webcore_init         (KzGtkWebcore *gtk_webcore);
static void kz_gtk_webcore_dispose      (GObject      *object);

/* gtk_webcore class */
static void         kz_gtk_webcore_status           (Webi     *webi,
						     WebiLoadStatus *status);
static void         kz_gtk_webcore_load_start       (Webi     *webi);
static void         kz_gtk_webcore_load_stop        (Webi     *webi);
static void         kz_gtk_webcore_location         (Webi     *webi);
static void         kz_gtk_webcore_title            (Webi     *webi);
static void         kz_gtk_webcore_req_js_prompt    (Webi     *webi,
						     WebiPromptArgs* args);
static void         kz_gtk_webcore_req_auth_prompt  (Webi     *webi,
						     WebiAuthArgs* args);
static Webi        *kz_gtk_webcore_req_new_window   (Webi     *webi,
						     const gchar *url);
static void         kz_gtk_webcore_close_window     (Webi     *webi);

static gboolean     kz_gtk_webcore_set_cookie       (Webi       *webi,
						     const gchar *url,
						     WebiCookie *cookie);
/* embed iface */
static void         kz_gtk_webcore_load_url         (KzEmbed      *kzembed,
						     const gchar  *url);
static gboolean     kz_gtk_webcore_is_loading       (KzEmbed      *kzembed);
static const gchar *kz_gtk_webcore_get_title        (KzEmbed      *kzembed);
static const gchar *kz_gtk_webcore_get_location     (KzEmbed      *kzembed);
static gchar       *kz_gtk_webcore_ensure_title     (KzEmbed      *kzembed);
static gchar       *kz_gtk_webcore_get_link_message (KzEmbed      *kzembed);

static void         kz_gtk_webcore_reload           (KzEmbed      *kzembed,
						     KzEmbedReloadFlag flags);
static void         kz_gtk_webcore_stop_load        (KzEmbed      *kzembed);
static void         kz_gtk_webcore_go_back          (KzEmbed      *kzembed);
static void         kz_gtk_webcore_go_forward       (KzEmbed      *kzembed);
static gboolean     kz_gtk_webcore_can_go_back      (KzEmbed      *kzembed);
static gboolean     kz_gtk_webcore_can_go_forward   (KzEmbed      *kzembed);
static gboolean     kz_gtk_webcore_can_go_nav_link  (KzEmbed      *kzembed,
						     KzEmbedNavLink link);
static void         kz_gtk_webcore_go_nav_link      (KzEmbed      *kzembed,
						     KzEmbedNavLink link);
static void         kz_gtk_webcore_set_nth_nav_link (KzEmbed      *kzembed,
						     KzEmbedNavLink link,
						     KzNavi       *navi,
						     guint         n);
static KzNavi      *kz_gtk_webcore_get_nth_nav_link (KzEmbed      *kzembed,
						     KzEmbedNavLink link,
						     guint         n);
static GList       *kz_gtk_webcore_get_nav_links    (KzEmbed      *kzembed,
						     KzEmbedNavLink link);

static gboolean     kz_gtk_webcore_get_lock         (KzEmbed      *kzembed);
static void         kz_gtk_webcore_set_lock         (KzEmbed      *kzembed,
						     gboolean      lock);

static void         kz_gtk_webcore_shistory_get_nth (KzEmbed      *kzembed, 
						     int           nth,
						     gboolean      is_relative,
						     char        **aUrl,
						     char        **aTitle);

static gboolean     kz_gtk_webcore_can_cut_selection  (KzEmbed      *kzembed);
static gboolean     kz_gtk_webcore_can_copy_selection (KzEmbed      *kzembed);
static gboolean     kz_gtk_webcore_can_paste          (KzEmbed      *kzembed);
static void         kz_gtk_webcore_cut_selection      (KzEmbed      *kzembed);
static void         kz_gtk_webcore_copy_selection     (KzEmbed      *kzembed);
static void         kz_gtk_webcore_paste              (KzEmbed      *kzembed);
static void         kz_gtk_webcore_select_all         (KzEmbed      *kzembed);

static gdouble      kz_gtk_webcore_get_progress     (KzEmbed      *kzembed);

static gint         kz_gtk_webcore_get_text_size    (KzEmbed      *kzembed);
static void         kz_gtk_webcore_set_text_size    (KzEmbed      *kzembed,
						     gint         zoom,
						     gboolean     reflow);

static gboolean     kz_gtk_webcore_find             (KzEmbed      *kzembed,
						     const char   *keyword,
						     gboolean      backward);

static gboolean     kz_gtk_webcore_selection_is_collapsed (KzEmbed      *kzembed);

static gboolean	    kz_gtk_webcore_get_allow_javascript  (KzEmbed      *kzembed);
static void	    kz_gtk_webcore_set_allow_javascript  (KzEmbed      *kzembed,
							  gboolean      allow);
static gboolean	    kz_gtk_webcore_get_allow_images      (KzEmbed      *kzembed);
static void	    kz_gtk_webcore_set_allow_images      (KzEmbed      *kzembed,
							  gboolean      allow);
/* KzEmbedPrefs interfaces */
static void          kz_gtk_webcore_prefs_iface_init      (KzEmbedPrefsIFace *iface);
#if 0
static gboolean      kz_gtk_webcore_prefs_init	    (KzEmbedPrefs *embed_prefs);
static gboolean      kz_gtk_webcore_prefs_get_font_list   (KzEmbedPrefs *embed_prefs,
						     const gchar  *lang_group,
						     const gchar  *font_type,
						     GList      **font_list,
						     GList      **all_font_list,
						     gchar       **default_font);
static gboolean      kz_gtk_webcore_prefs_get_passwords   (KzEmbedPrefs *embed_prefs,
						     GList **passwords);
static gboolean      kz_gtk_webcore_prefs_remove_passwords(KzEmbedPrefs *embed_prefs,
						     GList *passwords);
#endif
void
kz_gtk_webcore_register_type (GTypeModule *module)
{
	static const GTypeInfo kz_gtk_webcore_info =
	{
		sizeof (KzGtkWebcoreClass),
		NULL,		/* base_init */
		NULL,		/* base_finalize */
		(GClassInitFunc) kz_gtk_webcore_class_init,
		NULL,		/* class_finalize */
		NULL,		/* class_data */
		sizeof (KzGtkWebcore),
		0,		/* n_preallocs */
		(GInstanceInitFunc) kz_gtk_webcore_init,
	};

	const GInterfaceInfo kz_embed_info =
	{
		(GInterfaceInitFunc) kz_gtk_webcore_iface_init,
		NULL,
		NULL
	};

	const GInterfaceInfo kz_embed_prefs_info =
	{
		(GInterfaceInitFunc) kz_gtk_webcore_prefs_iface_init,
		NULL,
		NULL
	};

	kz_type_gtk_webcore = g_type_module_register_type(module,
						    WEBI_TYPE_WEBI,
						    "KzGtkWebcore",
						    &kz_gtk_webcore_info,
						    (GTypeFlags)0);

	g_type_module_add_interface(module,
				    KZ_TYPE_GTK_WEBCORE,
				    KZ_TYPE_EMBED,
				    &kz_embed_info);

	g_type_module_add_interface(module,
				    KZ_TYPE_GTK_WEBCORE,
				    KZ_TYPE_EMBED_PREFS,
				    &kz_embed_prefs_info);
}

GType
kz_gtk_webcore_get_type (void)
{
       return kz_type_gtk_webcore;
}

static void
kz_gtk_webcore_class_init (KzGtkWebcoreClass *klass)
{
	GObjectClass *object_class;
	GtkWidgetClass *widget_class;
	WebiClass *gtk_webcore_class;

	kz_gtk_webcore_parent_class = (WebiClass *)g_type_class_peek_parent (klass);
	object_class = (GObjectClass *) klass;
	widget_class = (GtkWidgetClass *) klass;
	gtk_webcore_class  = (WebiClass *) klass;

	object_class->dispose   = kz_gtk_webcore_dispose;
	gtk_webcore_class->status     = kz_gtk_webcore_status;
	gtk_webcore_class->load_start = kz_gtk_webcore_load_start;
	gtk_webcore_class->load_stop  = kz_gtk_webcore_load_stop;
	gtk_webcore_class->location   = kz_gtk_webcore_location;
	gtk_webcore_class->title      = kz_gtk_webcore_title;
#if 0
	gtk_webcore_class->progress   = kz_gtk_webcore_progress;
#endif
	gtk_webcore_class->req_js_prompt   = kz_gtk_webcore_req_js_prompt;
	gtk_webcore_class->req_auth_prompt = kz_gtk_webcore_req_auth_prompt;
	gtk_webcore_class->req_new_window  = kz_gtk_webcore_req_new_window;
	gtk_webcore_class->close_window    = kz_gtk_webcore_close_window;
	gtk_webcore_class->set_cookie      = kz_gtk_webcore_set_cookie;

	g_type_class_add_private (object_class, sizeof(KzGtkWebcorePrivate));
}


static void
kz_gtk_webcore_iface_init (KzEmbedIFace *iface)
{
	iface->load_url               = kz_gtk_webcore_load_url;
	iface->view_source            = NULL;
	iface->is_loading             = kz_gtk_webcore_is_loading;
	iface->get_title              = kz_gtk_webcore_get_title;
	iface->get_location           = kz_gtk_webcore_get_location;
	iface->ensure_title           = kz_gtk_webcore_ensure_title;
	iface->get_link_message       = kz_gtk_webcore_get_link_message;
	iface->get_progress           = kz_gtk_webcore_get_progress;
	iface->can_cut_selection      = kz_gtk_webcore_can_cut_selection;
	iface->can_copy_selection     = kz_gtk_webcore_can_copy_selection;
	iface->can_paste              = kz_gtk_webcore_can_paste;
	iface->cut_selection          = kz_gtk_webcore_cut_selection;
	iface->copy_selection         = kz_gtk_webcore_copy_selection;
	iface->paste                  = kz_gtk_webcore_paste;
	iface->select_all             = kz_gtk_webcore_select_all;
	iface->get_selection_string   = NULL;
	iface->find                   = kz_gtk_webcore_find;
	iface->incremental_search     = kz_gtk_webcore_find; /* tentavie */
	iface->selection_is_collapsed = kz_gtk_webcore_selection_is_collapsed;
	iface->get_links              = NULL;
	iface->copy_page              = NULL;
	iface->shistory_copy          = NULL;
	iface->shistory_get_pos       = NULL;
	iface->shistory_get_nth       = kz_gtk_webcore_shistory_get_nth;
	iface->reload                 = kz_gtk_webcore_reload;
	iface->stop_load              = kz_gtk_webcore_stop_load;
	iface->go_back                = kz_gtk_webcore_go_back;
	iface->go_forward             = kz_gtk_webcore_go_forward;
	iface->can_go_back            = kz_gtk_webcore_can_go_back;
	iface->can_go_forward         = kz_gtk_webcore_can_go_forward;
	iface->can_go_nav_link        = kz_gtk_webcore_can_go_nav_link;
	iface->go_nav_link            = kz_gtk_webcore_go_nav_link;
	iface->append_nav_link        = NULL;
	iface->set_nav_link           = NULL;
	iface->set_nth_nav_link       = kz_gtk_webcore_set_nth_nav_link;
	iface->get_nav_link           = NULL;
	iface->get_nth_nav_link       = kz_gtk_webcore_get_nth_nav_link;
	iface->get_nav_links          = kz_gtk_webcore_get_nav_links;
	iface->go_history_index       = NULL;
	iface->do_command             = NULL;
	iface->can_do_command         = NULL;
	iface->get_lock               = kz_gtk_webcore_get_lock;
	iface->set_lock               = kz_gtk_webcore_set_lock;
	iface->get_body_text          = NULL;
#if 0
	iface->get_selection_source   = NULL;
#endif
	iface->set_encoding           = NULL;
	iface->get_encoding           = NULL;
	iface->print                  = NULL;
	iface->print_preview          = NULL;
	iface->get_printer_list       = NULL;
	iface->create_thumbnail       = NULL;
	iface->save_with_content      = NULL;
	iface->set_text_into_textarea = NULL;
	iface->get_text_from_textarea = NULL;
	iface->zoom_set               = kz_gtk_webcore_set_text_size; /* tentative */
	iface->zoom_get               = kz_gtk_webcore_get_text_size; /* tentative */
	iface->set_text_size          = kz_gtk_webcore_set_text_size;
	iface->get_text_size          = kz_gtk_webcore_get_text_size;
	iface->get_html_with_contents = NULL;
	iface->set_history            = NULL;
	iface->get_history            = NULL;
	iface->get_last_modified      = NULL;
	iface->fine_scroll            = NULL;
	iface->page_up                = NULL;
	iface->page_down              = NULL; 
	iface->get_allow_javascript   = kz_gtk_webcore_get_allow_javascript; 
	iface->set_allow_javascript   = kz_gtk_webcore_set_allow_javascript; 
	iface->get_allow_images       = kz_gtk_webcore_get_allow_images;
	iface->set_allow_images       = kz_gtk_webcore_set_allow_images;
#if 0
	iface->set_edit_mode          = NULL;
	iface->set_view_mode          = NULL;
#endif

	iface->link_message           = NULL;
	iface->js_status              = NULL;
	iface->location               = NULL;
	iface->title                  = NULL;
	iface->progress               = NULL;
	iface->net_start              = NULL;
	iface->net_stop               = NULL;
	iface->new_window             = NULL;
	iface->open_uri               = NULL;
	iface->size_to                = NULL;
	iface->dom_key_down           = NULL;
	iface->dom_key_press          = NULL;
	iface->dom_key_up             = NULL;
	iface->dom_mouse_down         = NULL;
	iface->dom_mouse_up           = NULL;
	iface->dom_mouse_click        = NULL;
	iface->dom_mouse_dbl_click    = NULL;
	iface->dom_mouse_over         = NULL;
	iface->dom_mouse_out          = NULL;
	iface->security_change        = NULL;
	iface->status_change          = NULL;
}


static void
kz_gtk_webcore_prefs_iface_init (KzEmbedPrefsIFace *iface)
{
	iface->init             = NULL; /* kz_gtk_webcore_prefs_init;*/
	iface->get_font_list    = NULL; /* kz_gtk_webcore_prefs_get_font_list; */
	iface->get_passwords    = NULL; /* kz_gtk_webcore_prefs_get_passwords; */
	iface->remove_passwords = NULL; /* kz_gtk_webcore_prefs_remove_passwords; */
}

static void
set_font_preferences (KzProfile *profile, KzGtkWebcore *gtk_webcore)
{
	WebiSettings *settings = KZ_GTK_WEBCORE_GET_PRIVATE(gtk_webcore)->settings;
	/* font settings */
	GList *list, *node;
	gint i = 0;
	gint size_variable = -1, size_fixed = -1;
	gint min_size_variable = -1, min_size_fixed = -1;

	/* workaround */
	gchar langs[1024], *lang = NULL;
	if (kz_profile_get_value(profile, "Language", "accept_languages", 
				&langs, G_N_ELEMENTS(langs),
				KZ_PROFILE_VALUE_TYPE_STRING))
	{
		gchar **split_str;
		split_str = g_strsplit(langs, ",", 1);

		if (split_str[0])
		{
			lang = g_strdup(split_str[0]);
			g_strfreev(split_str);
		}
	}
	if (!lang)
		lang = g_strdup("");

	gchar *serif_font, *sans_font, *mono_font;
	serif_font = g_strdup_printf("name_serif_%s", lang);
	sans_font = g_strdup_printf("name_sans-serif_%s", lang);
	mono_font = g_strdup_printf("name_monospace_%s", lang);

	list = kz_profile_enum_key(profile, "Font", TRUE);
	for (node = list; node; node = g_list_next(node))
	{
		const gchar *key = (const gchar*)node->data;

		if (!key || !*key) continue;
		if (g_str_has_prefix(key, "size_variable_"))
		{
			gint value;
			kz_profile_get_value(profile, "Font", key,
					&value, sizeof(value),
					KZ_PROFILE_VALUE_TYPE_INT);
			if (value > size_variable)
				size_variable = value;
		}
		else if (g_str_has_prefix(key, "size_fixed_"))
		{
			gint value;
			kz_profile_get_value(profile, "Font", key,
					&value, sizeof(value),
					KZ_PROFILE_VALUE_TYPE_INT);
			if (value > size_fixed)
				size_fixed = value;
		}
		else if (g_str_has_prefix(key, "min-size_variable_"))
		{
			gint value;
			kz_profile_get_value(profile, "Font", key,
					&value, sizeof(value),
					KZ_PROFILE_VALUE_TYPE_INT);
			if (value > min_size_variable)
				min_size_variable = value;
		}
		else if (g_str_has_prefix(key, "min-size_fixed_"))
		{
			gint value;
			kz_profile_get_value(profile, "Font", key,
					&value, sizeof(value),
					KZ_PROFILE_VALUE_TYPE_INT);
			if (value > min_size_fixed)
				min_size_fixed = value;
		}
		else if (!strcmp(key, serif_font))
		{
			gchar value[1024];
			kz_profile_get_value(profile, "Font", key,
					&value, strlen(value)+1,
					KZ_PROFILE_VALUE_TYPE_STRING);
			settings->serif_font_family = g_strdup(value);
			settings->standard_font_family = g_strdup(value);
		}
		else if (!strcmp(key, sans_font))
		{
			gchar value[1024];
			kz_profile_get_value(profile, "Font", key,
					&value, strlen(value)+1,
					KZ_PROFILE_VALUE_TYPE_STRING);
			settings->sans_serif_font_family = g_strdup(value);
		}
		else if (!strcmp(key, mono_font))
		{
			gchar value[1024];
			kz_profile_get_value(profile, "Font", key,
					&value, strlen(value)+1,
					KZ_PROFILE_VALUE_TYPE_STRING);
			settings->fixed_font_family = g_strdup(value);
		}
	}
	g_list_free(list);
	g_free(serif_font);
	g_free(sans_font);
	g_free(mono_font);

	double dpi_pixel_ratio = 72.0 / 96.0;
	if (size_variable >= 0)
		settings->default_font_size = (gfloat)(size_variable * dpi_pixel_ratio);
	if (size_fixed >= 0)
		settings->default_fixed_font_size = (gfloat)(size_fixed * dpi_pixel_ratio);
	if (min_size_variable >= 0)
		settings->minimum_font_size = (gfloat)(min_size_variable * dpi_pixel_ratio);
	if (min_size_fixed >= 0)
		settings->minimum_font_size = (gfloat)(min_size_fixed * dpi_pixel_ratio);
}

static void
build_user_agent (gchar *user_agent, gint size)
{
	gchar *system;

	system = kz_utils_get_system_name();

	g_snprintf(user_agent, size,
		   "Mozilla/5.0 (X11; %s; U;) AppleWebKit/146.1 (KHTML, like Gecko) Kazehakase" VERSION,
		   system);
	g_free (system);
}

static void
set_user_agent (KzProfile *profile, KzGtkWebcore *gtk_webcore)
{
	WebiSettings *settings = KZ_GTK_WEBCORE_GET_PRIVATE(gtk_webcore)->settings;
	gchar tmp_string[1024];
	gboolean value = FALSE;
	kz_profile_get_value(profile, "Global", "override_user_agent", 
			     &value, sizeof(gboolean),
			     KZ_PROFILE_VALUE_TYPE_BOOL);
	if (value)
	{
		value = kz_profile_get_value(profile, "Global", "user_agent", 
				&tmp_string, G_N_ELEMENTS(tmp_string),
				KZ_PROFILE_VALUE_TYPE_STRING);
	}
	else /* set default */
	{
		build_user_agent(tmp_string, G_N_ELEMENTS(tmp_string));
	}

	settings->user_agent_string = g_strdup(tmp_string);
}

static void
set_proxy (KzProfile *profile, KzGtkWebcore *gtk_webcore)
{
	WebiSettings *settings = KZ_GTK_WEBCORE_GET_PRIVATE(gtk_webcore)->settings;
	gchar tmp_string[1024];
	gchar *http_host, *https_host, *ftp_host;
	guint http_port, https_port, ftp_port;
	gboolean value = FALSE, use_same_proxy;
	KzProxyItem *item = NULL;

	kz_profile_get_value(profile, "Global", "use_proxy", 
			     &value, sizeof(gboolean),
			     KZ_PROFILE_VALUE_TYPE_BOOL);
	if (!value)
		return;

	value = kz_profile_get_value(profile, "Global", "proxy_name", 
				&tmp_string, G_N_ELEMENTS(tmp_string),
				KZ_PROFILE_VALUE_TYPE_STRING);
	if (!value)
		return;
	
	item = kz_proxy_find(tmp_string);
	if (!item)
		return;

	g_object_get(G_OBJECT(item),
		     "use_same_proxy", &use_same_proxy,
		     NULL);
	if (use_same_proxy)
	{
		g_object_get(G_OBJECT(item),
			     "http_host",  &http_host,
			     "http_port",  &http_port,
			     NULL);
		if (!http_host)
			return;
	}
	else
	{
		g_object_get(G_OBJECT(item),
			     "http_host",  &http_host,
			     "http_port",  &http_port,
			     "https_host", &https_host,
			     "https_port", &https_port,
			     "ftp_host",   &ftp_host,
			     "ftp_port",   &ftp_port,
			     NULL);
	}

	/* proxy for HTTP protocol */
	if (http_host)
		settings->http_proxy = g_strdup(http_host);
#if 0
	/* proxy for HTTPS protocol */
	if (https_host)
		settings->https_proxy = g_strdup(https_host);
	
	/* proxy for FTP protocol */
	if (ftp_host)
		settings->ftp_proxy = g_strdup(ftp_host);
#endif	
	if (http_host)
		g_free(http_host);
	if (https_host)
		g_free(https_host);
	if (ftp_host)
		g_free(ftp_host);
}

static void
set_default_preferences (KzGtkWebcore *gtk_webcore)
{
	WebiSettings *settings = KZ_GTK_WEBCORE_GET_PRIVATE(gtk_webcore)->settings;
	KzProfile *profile = KZ_GET_GLOBAL_PROFILE;

	gchar value[1024];
	if (kz_profile_get_value(profile, "Language", "charset_default", 
				&value, G_N_ELEMENTS(value),
				KZ_PROFILE_VALUE_TYPE_STRING))
	{
		settings->default_text_encoding = g_strdup(value);
	}

	set_user_agent(profile, gtk_webcore);
	set_font_preferences (profile, gtk_webcore);
	set_proxy(profile, gtk_webcore);
	
	settings->plugins_enabled = TRUE;
	settings->javascript_enabled = TRUE;
	settings->java_enabled = TRUE;
	settings->autoload_images = TRUE;

	webi_set_settings(WEBI(gtk_webcore), settings);
#if 0
	OSB::Features *features = get_osb_features(WEBI(gtk_webcore));
	features->setToolbarsVisible(FALSE);
	features->setStatusBarVisible(FALSE);
#endif
}

static void
kz_gtk_webcore_init (KzGtkWebcore *gtk_webcore)
{
	KzGtkWebcorePrivate *priv = KZ_GTK_WEBCORE_GET_PRIVATE (gtk_webcore);

	priv->is_loading = FALSE;
	priv->cur_requests   = 0;
	priv->total_requests = 0;
	priv->link_message   = NULL;

	priv->button_press_time = 0;

	priv->settings = g_new0(WebiSettings, 1);

	set_default_preferences(gtk_webcore);
	webi_set_emit_internal_status(WEBI(gtk_webcore), TRUE);

	gtk_widget_show(GTK_WIDGET(gtk_webcore));
}

static void
free_webi_settings (WebiSettings *settings)
{
	if (settings->default_text_encoding)
		g_free((gchar*)settings->default_text_encoding);
	if (settings->serif_font_family)
		g_free((gchar*)settings->serif_font_family);
	if (settings->sans_serif_font_family)
		g_free((gchar*)settings->sans_serif_font_family);
	if (settings->fixed_font_family)
		g_free((gchar*)settings->fixed_font_family);
	if (settings->standard_font_family)
		g_free((gchar*)settings->standard_font_family);
	if (settings->cursive_font_family)
		g_free((gchar*)settings->cursive_font_family);
	if (settings->fantasy_font_family)
		g_free((gchar*)settings->fantasy_font_family);
	if (settings->user_agent_string)
		g_free((gchar*)settings->user_agent_string);
	if (settings->http_proxy)
		g_free((gchar*)settings->http_proxy);
}

static void
kz_gtk_webcore_dispose (GObject *object)
{
	KzGtkWebcorePrivate *priv = KZ_GTK_WEBCORE_GET_PRIVATE(object);

	if (priv->link_message)
		g_free(priv->link_message);
	priv->link_message = NULL;

	if (priv->settings)
	{
		free_webi_settings(priv->settings);
		g_free(priv->settings);
	}
	priv->settings = NULL;

	if (G_OBJECT_CLASS(kz_gtk_webcore_parent_class)->dispose)
		G_OBJECT_CLASS(kz_gtk_webcore_parent_class)->dispose(object);
}

GtkWidget *
kz_gtk_webcore_new (void)
{
	return GTK_WIDGET(g_object_new(KZ_TYPE_GTK_WEBCORE, NULL));
}

static void
kz_gtk_webcore_status (Webi *webi, WebiLoadStatus *status)
{
	KzGtkWebcorePrivate *priv;
	g_return_if_fail(KZ_IS_GTK_WEBCORE(webi));
	g_return_if_fail(status);

	priv = KZ_GTK_WEBCORE_GET_PRIVATE (webi);

	switch(status->status) {
	case WEBI_LOADING_START:
		priv->is_loading = TRUE;
		priv->cur_requests = 0;
		priv->total_requests = 0;
		g_signal_emit_by_name(webi, "kz-net-start");
		break;
	case WEBI_LOADING:
		priv->cur_requests = status->totalReceived;
		priv->total_requests = status->totalSize;
		g_signal_emit_by_name(webi, "kz-progress");
		break;
	case WEBI_LOADING_COMPLETE:
	case WEBI_LOADING_ERROR:
		priv->is_loading = FALSE;
		priv->cur_requests = status->totalReceived;
		priv->total_requests = status->totalSize;
		g_signal_emit_by_name(webi, "kz-net-stop");
		break;
	default:
		break;
	}

	/* call parent method */
}


static void
kz_gtk_webcore_load_start (Webi *webi)
{
	KzGtkWebcorePrivate *priv;
	g_return_if_fail(KZ_IS_GTK_WEBCORE(webi));

	priv = KZ_GTK_WEBCORE_GET_PRIVATE (webi);
	priv->is_loading = TRUE;

	g_signal_emit_by_name(webi, "kz-net-start");

	/* call parent method */
}


static void
kz_gtk_webcore_load_stop (Webi *webi)
{
	KzGtkWebcorePrivate *priv;
	g_return_if_fail(KZ_IS_GTK_WEBCORE(webi));

	priv = KZ_GTK_WEBCORE_GET_PRIVATE (webi);
	priv->is_loading = FALSE;

	g_signal_emit_by_name(webi, "kz-net-stop");

	/* call parent method */
}


static void
kz_gtk_webcore_location (Webi *webi)
{
	g_return_if_fail(KZ_IS_GTK_WEBCORE(webi));

	g_signal_emit_by_name(webi, "kz-location");

	/* call parent method */
}


static void
kz_gtk_webcore_title (Webi *webi)
{
	g_return_if_fail(KZ_IS_GTK_WEBCORE(webi));

	g_signal_emit_by_name(webi, "kz-title");

	/* call parent method */
}


static void
kz_gtk_webcore_req_js_prompt (Webi *webi, WebiPromptArgs* args)
{
	g_return_if_fail(KZ_IS_GTK_WEBCORE(webi));
}


static void
kz_gtk_webcore_req_auth_prompt (Webi *webi, WebiAuthArgs* args)
{
	KzPromptDialog *prompt;
	gchar *message;

	g_return_if_fail(KZ_IS_GTK_WEBCORE(webi));

	prompt= KZ_PROMPT_DIALOG(kz_prompt_dialog_new(TYPE_PROMPT_USER_PASS));

	kz_prompt_dialog_set_title(prompt, _("Authentication"));
	message = g_strdup_printf (_("Enter username and password for %s at %s"),
				   args->realm, args->url);
	kz_prompt_dialog_set_message_text(prompt, message);
	g_free(message);
	kz_prompt_dialog_run(prompt);

	args->out_ok_pressed = kz_prompt_dialog_get_confirm_value(prompt);
	args->out_username = g_strdup(kz_prompt_dialog_get_user(prompt));
	args->out_password = g_strdup(kz_prompt_dialog_get_password(prompt));

	gtk_widget_destroy(GTK_WIDGET(prompt));
}


static Webi *
kz_gtk_webcore_req_new_window (Webi *webi, const gchar *url)
{
	Webi *new_gtk_webcore = NULL;
	g_return_val_if_fail(KZ_IS_GTK_WEBCORE(webi), NULL);
	
	g_signal_emit_by_name(webi, "kz-new-window", &new_gtk_webcore);
	webi_load_url(new_gtk_webcore, url);

	return new_gtk_webcore;
}

static void
kz_gtk_webcore_close_window (Webi *webi)
{
	g_return_if_fail(KZ_IS_GTK_WEBCORE(webi));
	
	gtk_widget_destroy(GTK_WIDGET(webi));
}

static gboolean
kz_gtk_webcore_set_cookie (Webi *webi, const gchar *url, WebiCookie *cookie)
{
	g_return_val_if_fail(KZ_IS_GTK_WEBCORE(webi), FALSE);

	return TRUE;
}

static gboolean
kz_gtk_webcore_is_loading (KzEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_GTK_WEBCORE(kzembed), FALSE);

	return KZ_GTK_WEBCORE_GET_PRIVATE(kzembed)->is_loading;
}


static void
kz_gtk_webcore_load_url (KzEmbed *kzembed, const gchar *url)
{
	g_return_if_fail(KZ_IS_GTK_WEBCORE(kzembed));

	webi_load_url(WEBI(kzembed), url);
}


static const gchar *
kz_gtk_webcore_get_title (KzEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_GTK_WEBCORE(kzembed), NULL);

	return webi_get_title(WEBI(kzembed));
}


static const gchar *
kz_gtk_webcore_get_location (KzEmbed *kzembed)
{
	const gchar *location;
	g_return_val_if_fail(KZ_IS_GTK_WEBCORE(kzembed), NULL);

	location = webi_get_location(WEBI(kzembed));
	return location ? location : "";
}

static gchar *
kz_gtk_webcore_get_link_message (KzEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_GTK_WEBCORE(kzembed), NULL);

	return g_strdup(KZ_GTK_WEBCORE_GET_PRIVATE(kzembed)->link_message);
}

static gchar *
kz_gtk_webcore_ensure_title (KzEmbed *kzembed)
{
	const gchar *title;

	title = kz_gtk_webcore_get_title(kzembed);
	if (title)
		return g_strdup(title);

	title = kz_gtk_webcore_get_location(kzembed);
	if (title)
		return g_strdup(title);

	return g_strdup(_("No title"));
}

static void
kz_gtk_webcore_shistory_get_nth (KzEmbed *kzembed, 
			         int nth,
				 gboolean is_relative,
				 char **aUrl,
				 char **aTitle)
{
	g_return_if_fail(KZ_IS_GTK_WEBCORE(kzembed));
}

static void
kz_gtk_webcore_reload (KzEmbed *kzembed, KzEmbedReloadFlag flags)
{
	g_return_if_fail(KZ_IS_GTK_WEBCORE(kzembed));

	webi_refresh(WEBI(kzembed));
}


static void
kz_gtk_webcore_stop_load (KzEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_GTK_WEBCORE(kzembed));

	webi_stop_load(WEBI(kzembed));
}


static void
kz_gtk_webcore_go_back (KzEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_GTK_WEBCORE(kzembed));

	return webi_go_back(WEBI(kzembed));
}


static void
kz_gtk_webcore_go_forward (KzEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_GTK_WEBCORE(kzembed));

	return webi_go_forward(WEBI(kzembed));
}


static gboolean
kz_gtk_webcore_can_go_back (KzEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_GTK_WEBCORE(kzembed), FALSE);

	return webi_can_go_back(WEBI(kzembed));
}


static gboolean
kz_gtk_webcore_can_go_forward (KzEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_GTK_WEBCORE(kzembed), FALSE);

	return webi_can_go_forward(WEBI(kzembed));
}


static gboolean
kz_gtk_webcore_can_go_nav_link (KzEmbed *kzembed, KzEmbedNavLink link)
{
	g_return_val_if_fail(KZ_IS_GTK_WEBCORE(kzembed), FALSE);

	return FALSE;
}

static void
kz_gtk_webcore_go_nav_link (KzEmbed *kzembed, KzEmbedNavLink link)
{
	g_return_if_fail(KZ_IS_GTK_WEBCORE(kzembed));

	return;
}

static void
kz_gtk_webcore_set_nth_nav_link (KzEmbed *kzembed,
			   KzEmbedNavLink link,
			   KzNavi *navi,
			   guint n)
{
	g_return_if_fail(KZ_IS_GTK_WEBCORE(kzembed));

	return;
}

static KzNavi *
kz_gtk_webcore_get_nth_nav_link (KzEmbed *kzembed,
			   KzEmbedNavLink link,
			   guint n)
{
	g_return_val_if_fail(KZ_IS_GTK_WEBCORE(kzembed), NULL);

	return NULL;
}

static GList *
kz_gtk_webcore_get_nav_links (KzEmbed *kzembed,
			KzEmbedNavLink link)
{
	g_return_val_if_fail(KZ_IS_GTK_WEBCORE(kzembed), NULL);

	return NULL;
}

static gdouble
kz_gtk_webcore_get_progress (KzEmbed *kzembed)
{
	gdouble progress;
	KzGtkWebcorePrivate *priv;

	g_return_val_if_fail(KZ_IS_GTK_WEBCORE(kzembed), 0.0);

	priv = KZ_GTK_WEBCORE_GET_PRIVATE (kzembed);

	if (priv->total_requests <= 0 ||
	    priv->cur_requests <= 0)
	{
		return 0.0;
	}

	progress = (gdouble) priv->cur_requests
		/ (gdouble) priv->total_requests;

	return (progress < 1.0) ? progress : 1.0;
}


static gint
kz_gtk_webcore_get_text_size (KzEmbed *kzembed)
{
	gfloat multiplier;

	g_return_val_if_fail(KZ_IS_GTK_WEBCORE(kzembed), 100);

	multiplier = webi_get_text_multiplier(WEBI(kzembed));
	
	if (multiplier == -1.0) return 100;

	return (gint) rint(multiplier * 100);
}


static void
kz_gtk_webcore_set_text_size (KzEmbed *kzembed, gint zoom, gboolean reflow)
{
	gfloat multiplier;

	g_return_if_fail(KZ_IS_GTK_WEBCORE(kzembed));

	multiplier = (gfloat)(zoom) / 100;

	webi_set_text_multiplier(WEBI(kzembed), multiplier);
}


static gboolean
kz_gtk_webcore_find (KzEmbed *kzembed, const char *keyword, gboolean backward)
{
	g_return_val_if_fail(KZ_IS_GTK_WEBCORE(kzembed), FALSE);

	return webi_find(WEBI(kzembed), keyword, FALSE, backward); 
}

static gboolean
kz_gtk_webcore_can_cut_selection (KzEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_GTK_WEBCORE(kzembed), FALSE);

	/* not implemented yet */
	return FALSE;
}

static gboolean
kz_gtk_webcore_can_copy_selection (KzEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_GTK_WEBCORE(kzembed), FALSE);

	/* alwasy return TRUE */
	return TRUE;
}

static gboolean
kz_gtk_webcore_can_paste (KzEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_GTK_WEBCORE(kzembed), FALSE);
	/* not implemented yet */
	return FALSE;
}

static void
kz_gtk_webcore_cut_selection (KzEmbed *kzembed)
{
	/* not implemented yet */
}

static void
kz_gtk_webcore_copy_selection (KzEmbed *kzembed)
{
	gchar *selected;
	selected = webi_get_current_selection_as_text(WEBI(kzembed));

	gtkutil_copy_text(selected);

	if (selected)
		g_free (selected);
}

static void
kz_gtk_webcore_paste (KzEmbed *kzembed)
{
	/* not implemented yet */
}

static void
kz_gtk_webcore_select_all (KzEmbed *kzembed)
{
	/* not implemented yet */
}

static gboolean
kz_gtk_webcore_selection_is_collapsed (KzEmbed *kzembed)
{
	/* not implemented yet */
	return FALSE;
}

static gboolean
kz_gtk_webcore_get_allow_javascript (KzEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_GTK_WEBCORE(kzembed), FALSE);

	WebiSettings *settings = KZ_GTK_WEBCORE_GET_PRIVATE(kzembed)->settings;

	return settings->javascript_enabled ? TRUE : FALSE;
}

static void
kz_gtk_webcore_set_allow_javascript (KzEmbed *kzembed, gboolean allow)
{
	g_return_if_fail(KZ_IS_GTK_WEBCORE(kzembed));

	WebiSettings *settings = KZ_GTK_WEBCORE_GET_PRIVATE(kzembed)->settings;

	settings->javascript_enabled = allow;
	webi_set_settings(WEBI(kzembed), settings);
}

static gboolean
kz_gtk_webcore_get_allow_images (KzEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_GTK_WEBCORE(kzembed), FALSE);

	WebiSettings *settings = KZ_GTK_WEBCORE_GET_PRIVATE(kzembed)->settings;

	return settings->autoload_images;
}

static void
kz_gtk_webcore_set_allow_images (KzEmbed *kzembed, gboolean allow)
{
	g_return_if_fail(KZ_IS_GTK_WEBCORE(kzembed));

	WebiSettings *settings = KZ_GTK_WEBCORE_GET_PRIVATE(kzembed)->settings;

	settings->autoload_images = allow;
	webi_set_settings(WEBI(kzembed), settings);
}

static gboolean
kz_gtk_webcore_get_lock (KzEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_GTK_WEBCORE(kzembed), FALSE);
	return KZ_GTK_WEBCORE_GET_PRIVATE(kzembed)->lock;
}

static void
kz_gtk_webcore_set_lock (KzEmbed *kzembed, gboolean lock)
{
	g_return_if_fail(KZ_IS_GTK_WEBCORE(kzembed));
	KZ_GTK_WEBCORE_GET_PRIVATE(kzembed)->lock = lock;
}

