/*
 * debugFile.c
 * Routines for generating an assembly file with debugging information
 *
 * Copyright (c) 2000 University of Utah and the Flux Group.
 * All rights reserved.
 *
 * This file is licensed under the terms of the GNU Public License.
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * Contributed by the Flux Research Group, Department of Computer Science,
 * University of Utah, http://www.cs.utah.edu/flux/
 */

#if defined(KAFFE_XDEBUGGING) || defined(KAFFE_XPROFILER)

#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <stab.h>
#include <ctype.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "jmalloc.h"
#include "classMethod.h"
#include "code.h"

#include "mangle.h"
#include "debugFile.h"
#include "xprofiler.h"

struct debug_file *machine_debug_file;
char *machine_debug_filename = 0;

static char *debug_header = "This file was automatically generated by Kaffe";

struct debug_file *createDebugFile(char *filename)
{
	struct debug_file *retval = 0;
	
	/* Allocate space for the debug_file struct and filename */
	if( (retval = (struct debug_file *)
	     KMALLOC(sizeof(struct debug_file) + strlen(filename) + 1)) )
	{
		retval->df_filename = (char *)(retval + 1);
		strcpy(retval->df_filename, filename);
		if( (retval->df_file = fopen(retval->df_filename, "w")) )
		{
			addDebugInfo(retval,
				     DIA_Comment, debug_header,
				     DIA_DONE);
		}
		else
		{
			KFREE(retval);
			retval = 0;
		}
	}
	return( retval );
}

void deleteDebugFile(struct debug_file *df)
{
	if( df )
	{
		/* If there was an error in writing the file remove it */
		if( ferror(df->df_file) )
			remove(df->df_filename);
		fclose(df->df_file);
		KFREE(df);
	}
}

int addDebugInfo(struct debug_file *df, int tag, ...)
{
	int retval = 1;
	va_list args;

#if defined(KAFFE_XPROFILER)
	xProfilingOff();
#endif
	va_start(args, tag);
	if( df )
	{
		/* Walk over the arguments until we hit the terminator */
		while( tag != DIA_DONE )
		{
			struct mangled_method *mm;
			char *str, *name, *addr;
			int line, size;
			Method *meth;

			switch( tag )
			{
			case DIA_FunctionSymbolS:
				name = va_arg(args, char *);
				addr = va_arg(args, char *);
				size = va_arg(args, int);
				fprintf(df->df_file,
					".weak %s\n"
					"%s = %p\n",
					name,
					name,
					addr);
				if( size > 0 )
				{
					fprintf(df->df_file,
						".weak %s_end\n"
						"%s_end = %p\n",
						name,
						name,
						addr + size);
				}
				break;
			case DIA_FunctionSymbol:
				mm = va_arg(args, struct mangled_method *);
				addr = va_arg(args, char *);
				size = va_arg(args, int);
				fprintf(df->df_file, ".weak ");
				printMangledMethod(mm, df->df_file);
				fprintf(df->df_file, "\n");
				printMangledMethod(mm, df->df_file);
				fprintf(df->df_file, " = %p\n", addr);
				if( size > 0 )
				{
					fprintf(df->df_file, ".weak ");
					printMangledMethod(mm, df->df_file);
					fprintf(df->df_file, "_end\n");
					printMangledMethod(mm, df->df_file);
					fprintf(df->df_file,
						"_end = %p\n",
						addr + size);
				}
				break;
			case DIA_Function:
				meth = va_arg(args, Method *);
				mm = va_arg(args, struct mangled_method *);
				line = va_arg(args, int);
				addr = va_arg(args, char *);
				size = va_arg(args, int);
				/* Add the stabs info to the file */
				fprintf(df->df_file,
					"  /* START %s/%s%s */\n"
					".stabs \"",
					meth->name->data,
					CLASS_CNAME(meth->class),
					METHOD_SIGD(meth));
				printMangledMethod(mm, df->df_file);
				fprintf(df->df_file, ":F\",%d,0,%d,%p\n",
					N_FUN,
					line,
					addr);
				/* Add the symbols to the file */
				fprintf(df->df_file,
					"  /* Symbol: %s/%s%s */\n"
					".weak ",
					meth->name->data,
					CLASS_CNAME(meth->class),
					METHOD_SIGD(meth));
				printMangledMethod(mm, df->df_file);
				fprintf(df->df_file, "\n");
				printMangledMethod(mm, df->df_file);
				fprintf(df->df_file,
					" = %p\n"
					".weak ",
					addr);
				printMangledMethod(mm, df->df_file);
				fprintf(df->df_file, "_end\n");
				printMangledMethod(mm, df->df_file);
				fprintf(df->df_file,
					"_end = %p\n",
					addr + size);
				break;
			case DIA_Symbol:
				name = va_arg(args, char *);
				addr = va_arg(args, char *);
				fprintf(df->df_file, "%s = %p\n", name, addr);
				break;
			case DIA_EndFunction:
				addr = va_arg(args, char *);
				/*
				 * Add an empty filename name symbol so that
				 * the debugger will know where the real
				 * end of the information is.  Otherwise, it
				 * uses the next symbol which may not be
				 * contiguous.
				 */
				fprintf(df->df_file,
					".stabs \"\",%d,0,0,%p\n",
					N_SO,
					addr);
				break;
			case DIA_SourceLine:
				line = va_arg(args, int);
				addr = va_arg(args, char *);
				fprintf(df->df_file,
					".stabn %d,0,%d,%p\n",
					N_SLINE,
					line,
					addr);
				break;
			case DIA_SourceFile:
				name = va_arg(args, char *);
				addr = va_arg(args, char *);
				fprintf(df->df_file,
					"\n\n.stabs \"%s\",%d,0,0,%p\n",
					name,
					N_SO,
					addr);
				break;
			case DIA_Comment:
				str = va_arg(args, char *);
				fprintf(df->df_file,
					"/* %s */\n",
					str);
				break;
			}
			/* Get the next tag to process */
			tag = va_arg(args, int);
		}
		fflush(df->df_file);
		/* Check for I/O error */
		if( ferror(df->df_file) )
			retval = 0;
	}
	va_end(args);
#if defined(KAFFE_XPROFILER)
	xProfilingOn();
#endif
	return( retval );
}

#endif /* KAFFE_XDEBUGGING */
