/* vi:ts=4:sw=4
 *
 * JGREP	- Japanized GREP
 *
 * Code Contributions By:	Atsushi Nakamura		ann@mrit.mei.co.jp
 *
 */

#define EXTERN
#include "vim.h"
#ifdef JP
#include "jp.h"
#endif
#include "globals.h"
#include "proto.h"

#undef free

#define MAXSCRIPT 4096
#define KCHARR "NJESX.,"
#define KCHARW "NJESX"

#ifdef JP

static void usage();

	void
emsg(msg)
	char *msg;
{
	fprintf(stderr, "%s\n", msg);
}

static int o_conly = FALSE;		/* -c */
static int o_icase = FALSE;		/* -i */
static int o_nonly = FALSE;		/* -l */
static int o_noerr = FALSE;		/* -s */
static int o_nomat = FALSE;		/* -v */
static int o_wsrch = FALSE;		/* -w */
static int o_scode = FALSE;		/* -C */

	static void
usage()
{
	fprintf(stderr, "Jgrep/%s by ann@mrit.mei.co.jp\n", JpVersion);
	fprintf(stderr, "Usage: jgrep [ -cilsvwCNJESX., ] pattern [filename ...]\n");
	fprintf(stderr, "	-c     Print only a count of the lines.\n");
	fprintf(stderr, "	-i     Ignore case.\n");
	fprintf(stderr, "	-l     Print only file name.\n");
	fprintf(stderr, "	-s     Suppress error message.\n");
	fprintf(stderr, "	-v     Print lines not match with the pattern.\n");
	fprintf(stderr, "	-w     Search \\<pattern\\>.\n");
	fprintf(stderr, "	-C     Print kanji code.\n");
	fprintf(stderr, "	-N -J -E -S -X	\n");
	fprintf(stderr, "	       kanji code for output.\n");
	fprintf(stderr, "	-. -,  kanji code for input.\n");
	fprintf(stderr, "\n");
	fprintf(stderr, "Environment JMASK specifies default Kanji code.\n");
	exit(1);
}


char jread;
char jdisp;

	void
main(argc, argv)
	int				argc;
	char		  **argv;
{
	char *pattern;
	regexp *prog;

#ifdef JP
	static char jmask[4] = JP;
#endif
	int	 i;
	char *cp;

	++argv;
	/*
	 * Process the command line arguments
	 */

#ifdef JP
	if ((cp = (char *)getenv("JMASK")) != NULL)
		strncpy(jmask, cp, 4);

	if (jmask[0] && !strchr(KCHARR, jmask[0]))
	{
		fprintf(stderr, "Unknown Kanji code %c for reading.\n", jmask[0]);
		return;
	}

	if (jmask[1] && !strchr(KCHARW, jmask[1]))
	{
		fprintf(stderr, "Unknown Kanji code %c for writing.\n", jmask[1]);
		return;
	}

	jread = jmask[0];
	jdisp = jmask[1];
#else
	jread = jdisp = JP_NONE;
#endif

	if (argc > 1 && argv[0][0] == '-')
	{
		char *cp;
		for(cp = argv[0] + 1; *cp; cp++)
		{
			switch(*cp)
			{
			case 'c':
				o_conly = TRUE;
				break;

			case 'i':
				o_icase = TRUE;
				break;

			case 'l':
				o_nonly = TRUE;
				break;

			case 's':
				o_noerr = TRUE;
				break;

			case 'v':
				o_nomat = TRUE;
				break;

			case 'w':
				o_wsrch = TRUE;
				break;

			case 'C':
				o_scode = TRUE;
				break;

			case 'N':
			case 'J':
			case 'E':
			case 'S':
			case 'X':
				jdisp = *cp;
				break;

            case 'x':
                if (* ++cp == NUL)
                {
                    cp --;
                    fprintf(stderr, "One of NJESX., after '%c'\n", *cp);
                    usage();
                    break;
                }

			case ',':
			case '.':
				jread = *cp;
				break;

			default:
				fprintf(stderr, "Unknown option '%c'\n", *cp);
				usage();
				break;
			}
		}
		++ argv;
		-- argc;
	}

	if (argc > 1)
	{
		pattern = argv[0];
		++ argv;
		-- argc;
	}
	else
	{
		usage();
		return;
	}

	/*
	 * Allocate space for the generic buffer
	 */
#ifdef JP
	if ((IObuff = malloc(IOSIZE * 2)) == NULL)
#else
	if ((IObuff = malloc(IOSIZE)) == NULL)
#endif
		exit(0);

#if !defined(UNIX) && !defined(MSDOS)
	ExpandWildCards(argc - 1, argv, &numfiles, &files, TRUE, TRUE);
	if (numfiles != 0)
		files_exp = TRUE;
#else
	files = argv;
	numfiles = argc - 1;
#endif

/*
 * compile search pattern with converting Japanese characters
 */
 	{
		int len;

		len = strlen(pattern);
#ifdef JP
		len = kanjiconvsfrom(pattern, len, o_wsrch ? IObuff + 2 : IObuff,
										   o_wsrch ? IOSIZE - 2 : IOSIZE,
										   NULL,
										   jread, NULL);
#else
		strncpy(o_wsrch ? IObuff + 2 : IObuff, pattern);
#endif

		if (o_wsrch)
		{
			len += 2;
			IObuff[0] = '\\';
			IObuff[1] = '<';
			IObuff[len ++] = '\\';
			IObuff[len ++] = '>';
		}
		IObuff[len] = NUL;
		prog = regcomp(IObuff);
	}

/*
 * execute grep for each file
 */
 	{	void do_grep();

		if (numfiles == 0)
			do_grep(stdin, prog, NULL);
		else
			for(i = 0; i < numfiles; i++)
			{
				FILE *fp;

#ifdef AMIGA
				fname_case(files[i]);		/* set correct case for file name */
#endif
				fp = fopen(files[i], "r");
				if (fp)
				{
					do_grep(fp, prog, numfiles == 1 ? NULL: files[i]);
					fclose(fp);
				}
				else if (!o_noerr)
					fprintf(stderr, "jgrep: %s cannot open\n", files[i]);
			}
	}
/*
 * free compiled search pattern
 */
 	free(prog);
}

	void
do_grep(fp, prog, fname)
	FILE *fp;
	regexp *prog;
	char *fname;
{
	int count;

	reg_ic = o_icase;
	count = 0;

	reset_jcount();

	while(!feof(fp))
	{
		int len;
		static char Line[IOSIZE];

		if (fgets(Line, IOSIZE, fp) == NULL)
			break;

		len = strlen(Line);
		len = kanjiconvsfrom(Line, len, IObuff, IOSIZE * 2, NULL,
															jread, NULL);
		IObuff[len] = NUL;
		
		if (regexec(prog, IObuff, TRUE) ^ o_nomat)
		{
			count ++;
			if (!o_conly)
			{
				if (fname)
					printf("%s:", fname);
				if (o_scode)
					printf("(%c)", judge_jcode(jread));
				if (o_nonly)
					printf("\n");
			}

			if (o_nonly)
				return;

			if (!o_conly)
			{
				char *prline;

				prline = kanjiconvsto(IObuff, jdisp);
				printf("%s", prline);

				if (prline != IObuff)
					free(prline);
			}
		}
	}
	if (o_conly)
	{
		if (fname)
			printf("%s:%d", fname, count);
		else
			printf("%d", count);

		if (o_scode)
			printf(" %c", judge_jcode(jread));
		printf("\n");
	}
}

	void
getout(r)
	int	r;
{
	fprintf(stderr, "\n");
	exit(r);
}

#else
main(){}	/* dummy */
#endif
