/* 
 * Copyright (C) 1999-2001 Joachim Wieland <joe@mcknight.de>
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA 02111, USA.
 */

#include "jftpgw.h"
#ifdef HAVE_LINUX_NETFILTER_IPV4_H
#include <linux/netfilter_ipv4.h>
#endif

const int tcp_proto = 6;
extern unsigned int actvportsamount;
extern unsigned int pasvportsamount;

int openportiaddr(unsigned long addr, unsigned int port,
		  unsigned int localport, struct clientinfo* clntinfo) {
	struct sockaddr_in sin;
	memset((void*)&sin, 0, sizeof(sin));
	sin.sin_family = AF_INET;
	sin.sin_addr.s_addr = addr;
	sin.sin_port = htons(port);
	return openport(sin, localport);
}

int openportaddr(char* addr, unsigned int port,
		 unsigned int localport, struct clientinfo* clntinfo) {
	int cs = clntinfo->clientsocket;
	struct sockaddr_in sin;
	unsigned long int iaddr = inet_addr(addr);

	if (iaddr == (unsigned long int) UINT_MAX
			&& strcmp(addr, BROADCAST)) {
		say(cs, "500 Invalid address to open");
		log(3, "The address %s is invalid", addr);
		return -1;
	}
	memset((void*)&sin, 0, sizeof(sin));
	sin.sin_family = AF_INET;
	sin.sin_addr.s_addr = iaddr;
	sin.sin_port = htons(port);
	return openport(sin, localport);
}

int openportname(char* hostname, unsigned int port,
		 unsigned int localport, struct clientinfo* clntinfo) {
	struct hostent *host;
	char sendbuf[MAX_LINE_SIZE];
	int cs = clntinfo->clientsocket;
	struct sockaddr_in sin;
	memset((void*)&sin, 0, sizeof(sin));
	host = gethostbyname(hostname);
	if (!host) {
		log(3, "Could not look up %s", hostname);
		snprintf(sendbuf, sizeof(sendbuf),
				"500 Could not look up %s\r\n", hostname);
		say(cs, sendbuf);
		return -1;
	}
	sin.sin_family = AF_INET;
	memcpy((void*)&sin.sin_addr.s_addr, (void*) host->h_addr,
			host->h_length);
	sin.sin_port = htons(port);
	return openport(sin, localport);
}

/* openport() connects to HOST on port PORT */

int openport(struct sockaddr_in sin,
	     unsigned int localport /*, struct clientinfo *clntinfo */) {
	int handle;
	int one = 1;
	struct sockaddr_in dp;

	if (localport && localport <= IPPORT_RESERVED && getuid() == 0) {
		log(8, "Changing ID to root (bind to priv port)");
		changeid("root", UID);
	}

	handle = socket(AF_INET, SOCK_STREAM, tcp_proto);
	if (handle < 0) {
		log(2, "Error opening the socket: %s", strerror(errno));
		return -1;
	}
	
	if (localport) {
		setsockopt(handle, SOL_SOCKET, SO_REUSEADDR,
				(void*)&one, sizeof(one));
		memset((void*)&dp, 0, sizeof(dp));
		dp.sin_family = AF_INET;
		dp.sin_addr.s_addr = INADDR_ANY;
		dp.sin_port = htons(localport);

		if (bind(handle, (struct sockaddr*)&dp, sizeof(dp)) < 0) {
			int err = errno;
			if (getuid() == 0) {
				log(5, "Could not bind to the data port %d as root",
					localport);
				log(5, "Reason: %s", strerror(err));
				log(8, "Changing back (bind to priv port)");
				changeid(get_option("runasuser"), EUID);
			}
			/* try to bind to any other port */
			memset((void*)&dp, 0, sizeof(dp));
			dp.sin_family = AF_INET;
			dp.sin_addr.s_addr = INADDR_ANY;
			dp.sin_port = 0;
			if (bind(handle, (struct sockaddr*)&dp, sizeof(dp)) < 0) {
				int err = errno;
				log(5, "Could not bind to a free port");
				log(5, "Reason: %s", strerror(err));
				return -1;
			} else {
				log(8, "Successfully bound to another port (%d)",
					ntohs(dp.sin_port));
			}
		} else {
			if (getuid() == 0) {
				log(8, "Changing back (bind to priv port)");
				changeid(get_option("runasuser"), EUID);
			}
		}
	}

	if (connect(handle, (struct sockaddr*) &sin, sizeof(sin)) < 0) {
		log(1, "Error connecting: %s", strerror(errno));
	/*	say(cs, "500 Could not connect to the peer\r\n"); */
		return -1;
	}

	return handle;
}

int getanylocalport(int sd, struct sockaddr_in* sin) {
	/* default jftpgw behavior */
	sin->sin_port = INPORT_ANY;
	if (bind(sd, (struct sockaddr*) sin,
				sizeof(struct sockaddr)) < 0) {
		log(2, "Could not bind to a free port: %s",
				strerror(errno));
		return -1;
	} else {
		return sd;
	}
}

int getportinrange(int sd, struct sockaddr_in* sin,
		struct portrangestruct* prs, unsigned int portsamount) {

	struct portrangestruct *prscur;
	/* the following part is mostly from the proftpd patch by TJ
	 * Saunders <tj@digisle.net> - 10/14/00 */

	unsigned int found_pasv_port = 0;
	int pasv_range_len, pasv_port_index;
	int *pasv_range, *pasv_ports;
	int attempt, random_index;
	int tries = 0;

	/* hack up wu-ftpd's implementation of this to feature work here.  What
	 * can I say?  I'm a plagiarist and a hack to the Nth degree, and feel
	 * little shame about it, as long as it satisfies proftpd users'
	 * needs. Credits go to kinch, I think...that's the name in the wu-ftpd
	 * source code. -- TJ
	 */

	if (!prs) {
		return getanylocalport(sd, sin);
	}

	pasv_range_len = portsamount;

	pasv_range = (int *) malloc(pasv_range_len * sizeof(int));
	pasv_ports = (int *) malloc((pasv_range_len + 1) * sizeof(int));

	/* populate the array with all the port numbers in the configured
	 * range.
	 */
	pasv_port_index = pasv_range_len;
	prscur = prs;
	do {
		unsigned int inner_index;
		if (!prscur) {
			log(2, "prscur was NIL in %s, %d", __FILE__, __LINE__);
			break;
		}
		inner_index = prscur->endport + 1;
		do {
			/* the first port that is registered is endport, the
			 * last one is startport */
			inner_index--;
			pasv_port_index--;
			pasv_range[pasv_port_index] = inner_index;
		} while (inner_index > prscur->startport);
		prscur = prscur->next;
	} while (pasv_port_index > 0);

	/* seed the random number generator
	 */
	srand(time(NULL));

	/* randomly choose a port from within the range, and call
	 * inet_create_connection().  If that call fails, try a different
	 * port, until all in the range have been tried.
	 */
	for (attempt = 3; attempt > 0 && (!found_pasv_port); attempt--) {
		for (pasv_port_index = pasv_range_len; pasv_port_index > 0 &&
			(!found_pasv_port); pasv_port_index--) {

			/* if this is the first attempt through the passive
			 * ports range, randomize the order of the port
			 * numbers used (eg no linear probing), and store
			 * this random order into the pasv_ports array, to
			 * be attempted again on the next two runs. -- TJ
			 */
			if (attempt == 3) {

				/* obtain a random index into the port range
				 * array
				 */
				random_index = (int) ((1.0 * pasv_port_index
					* rand()) / (RAND_MAX + 1.0));

				/* copy the port at that index into the
				 * array from which port numbers will be
				 * selected for passing into
				 * inet_create_connections()
				 */
				pasv_ports[pasv_port_index] =
					pasv_range[random_index];

				/* now, alter the order of the port numbers
				 * in the pasv_range array by moving the
				 * non-selected numbers down, so that the
				 * next randomly chosen port number will be
				 * from the range of as-yet unchosen ports.
				 * -- TJ
				 */
				while (++random_index < pasv_port_index) {
					pasv_range[random_index - 1] =
						pasv_range[random_index];
				}
			}

			sin->sin_port = htons(pasv_ports[pasv_port_index]);
			if (bind(sd, (struct sockaddr*) sin,
						sizeof(struct sockaddr)) < 0) {
				log(9, "Tried port %d in vain",
						pasv_ports[pasv_port_index]);
				tries++;
			} else {
				found_pasv_port = 1;
				log(8, "Found free port %d after %d tries",
						pasv_ports[pasv_port_index],
						tries);
			}
		}
	}

	free(pasv_range);
	free(pasv_ports);

	if (!found_pasv_port) {
		/* if not able to find an open port in the given range,
		 * default to normal proftpd behavior (using INPORT_ANY),
		 * and log the failure -- symptom of a too-small port range
		 * configuration.  -- TJ
		 */

		log(4, "unable to find a free port in the port range"
		"; defaulting to INPORT_ANY");
		return getanylocalport(sd, sin);
	}

	return sd;
}



/* openlocalport() binds to a free port on the own machine */

int openlocalport(struct sockaddr_in *sin, struct clientinfo* clntinfo,
		  struct portrangestruct* prs, int whichaddr) {
#ifdef HAVE_SOCKLEN_T
	socklen_t slen;
#else
	int slen;
#endif
	int cs = clntinfo->clientsocket;
	int sd;
	unsigned int portsamount;

	if ((sd = socket(AF_INET, SOCK_STREAM, tcp_proto)) < 0) {
		log(1, "Could not create socket to bind to a free port: %s",
				strerror(errno));
		say(cs, "500 Could not create socket to bind to a free port\r\n");
		return -1;
	}
	memset((void*)sin, 0, sizeof(*sin));

	switch(whichaddr) {
		case SERVERADDR:
			/* we bind to the interface through which we talk to
			 * the server, so we are a client, so we are in
			 * active mode */
			sin->sin_addr.s_addr = clntinfo->serveraddr;
			portsamount = actvportsamount;
			break;
		case CLIENTADDR:
			/* passive mode */
			sin->sin_addr.s_addr = clntinfo->clientaddr;
			portsamount = pasvportsamount;
			break;
		default:
			/* dummy  -  suppress compiler warnings */
			portsamount = 0;
	}
	sin->sin_family = AF_INET;

	if(getportinrange(sd, sin, prs, portsamount) < 0) {
		say(cs, "500 Could not bind to a free port\r\n");
		return -1;
	}
	if (listen(sd, 1) < 0) {
		log(2, "Could not listen on a free port: %s", strerror(errno));
		say(cs, "500 Could not listen on a free port\r\n");
		return -1;
	}

	/* re-read the socket data to get the actual port */

	slen = sizeof(struct sockaddr);
	if (getsockname(sd, (struct sockaddr*) sin, &slen) < 0) {
		log(2, "getsockname failed after binding to a free port: %s",
				strerror(errno));
		say(cs, "500 getsockname() failed after binding to a free port\r\n");
		return -1;
	}

	return sd;
}


unsigned long int get_our_addr_to(unsigned long int to_addr,
				  unsigned int port) {
	int i;
	int sd;
	struct sockaddr_in sin;

	if ((sd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
		log(3, "Cannot create socket to send UDP packet to the transparent proxy client");
		return -1;
	}
	sin.sin_family = AF_INET;
	sin.sin_port = port;
	sin.sin_addr.s_addr = to_addr;
	if (connect(sd, (struct sockaddr*) &sin, sizeof(sin)) < 0) {
		log(5, "Cannot connect to the transparent proxy client");
		return -1;
	}
	if (send(sd, "Hi", 2, 0) < 0) {
		log(5, "Cannot send sample text to the transparent proxy client");
		return -1;
	}
	i = sizeof(sin);
	if (getsockname(sd, (struct sockaddr*) &sin, &i) < 0) {
		log(2, "getsockname() failed to determine our IP (transparent proxy -> udp)");
		return -1;
	}
	close(sd);
	return sin.sin_addr.s_addr;
}


/* nf_getsockname() - netfilter SO_ORIGINAL_DST variant of getsockopt()
 *
 * Within the new Linux netfilter framework, NAT functionality is cleanly
 * separated from the TCP/IP core processing. In old days, you could easily
 * retrieve the original destination (IP address and port) of a transparently
 * proxied connection by calling the normal getsockname() syscall.
 * With netfilter, getsockname() returns the real local IP address and port.
 * However, the netfilter code gives all TCP sockets a new socket option,
 * SO_ORIGINAL_DST, for retrieval of the original IP/port combination.
 *
 * This file implements a function nf_getsockname(), with the same calling
 * convention as getsockname() itself; it uses SO_ORIGINAL_DST, and if that
 * fails, falls back to using getsockname() itself.
 *
 * Public domain by Patrick Schaaf <bof@bof.de>
 */

int nf_getsockname(int fd, struct sockaddr *sa, int *salen)
{
	if (*salen != sizeof(struct sockaddr_in)) {
		errno = EINVAL;
		return -1;
	}
#ifdef SO_ORIGINAL_DST
	if (0 == getsockopt(fd, SOL_IP, SO_ORIGINAL_DST, sa, salen)) {
		return 0;
	}
#endif
	return getsockname(fd, sa, salen);
}


unsigned long int get_showaddr(int shandle, int type) {
	struct sockaddr_in sin;
	int i;
#ifdef HAVE_SOCKLEN_T
	socklen_t slen;
#else
	int slen;
#endif
	slen = sizeof (sin);
	if (type == SHOW_ORIGINAL_IP) {
		i = nf_getsockname(shandle, (struct sockaddr*) &sin, &slen);
	} else {
		i = getsockname(shandle, (struct sockaddr*) &sin, &slen);
	}
	if (i != 0) {
		log(2, "getsockname failed. Can't get the IP of the interface");
		return -1;
	}
	return sin.sin_addr.s_addr;
}

