/* 
 * Copyright (C) 1999-2001 Joachim Wieland <joe@mcknight.de>
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA 02111, USA.
 */

#include <ctype.h>
#include <syslog.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <unistd.h>
#ifdef HAVE_GETOPT_LONG
#include <getopt.h>
#else
#include "support/getopt.h"
#include "support/getopt.c"
#include "support/getopt1.c"
#endif

#include "jftpgw.h"


int init_logfiles(const char*);
void reset_confst(struct configstruct*);
void print_version(void);
void print_help(void);

struct configstruct config;
struct clientinfo clntinfo;
extern struct loginfo li;
int multithread;
int timeout;
int main_server_pid = 0;
int chrooted = 0;
extern int should_read_config;
char *conffilename;
char *binaryname;


static struct option const long_option_arr[] =
{
	{ "single", no_argument, NULL, 's' },
	{ "encrypt", no_argument, NULL, 'e' },
	{ "version", no_argument, NULL, 'v' },
	{ "version", no_argument, NULL, 'V' },
	{ "help", no_argument, NULL, 'h' },
	{ "configfile", required_argument, NULL, 'f' },
	{ NULL, 0, NULL, 0 }
};

int main(int argc, char** argv) {

	/* We bind to 0.0.0.0, but our real IP is the one returned by the
	 * PASV command */

	struct sigaction sa, cf;
	int ret;
	const char* option;
	const char* bindaddress;

	/* default: Multithread, i.e. fork for each connection */
	multithread = 1;

	/* set the name of the config file to an initial value, it may be
	 * overwritten later
	 * */
	conffilename = strdup(DEFAULTCONFFILE);
	enough_mem(conffilename);

	if (strrchr(argv[0], '/')) {
		binaryname = strdup(strrchr(argv[0], '/') + 1);
	} else {
		binaryname = strdup(argv[0]);
	}
	enough_mem(binaryname);

	/* parse the command line */
	while ((ret=getopt_long(argc, argv, "sevVhf:", long_option_arr, NULL)) != EOF) {
		switch (ret) {
			case 0: break;
			case 's':
				multithread = 0;
				break;
			case 'e':
				encrypt_password();
				exit(0);
			case 'v':
			case 'V':
				print_version();
				exit(0);
			case 'h':
				print_help();
				exit(0);
			case 'f':
				set_conffilename(optarg);
				break;
			default:
				break;
		}
	}

	/* Read the configuration */
	config.debuglevel = 9;
	config.logf_name = (char*) 0;
	config.logf = (FILE*) 0;

	ret = read_config(conffilename);
	if (ret) {
		return 1;
	}

	/* Initialise the log files  -  not yet chroot()ed */
	if (get_option("atstartchangerootdir")) {
		ret = init_logfiles(get_option("atstartchangerootdir"));
	} else if (get_option("changerootdir")) {
		ret = init_logfiles(get_option("changerootdir"));
	} else {
		ret = init_logfiles((char*) 0);
	}

	option = get_option("atstartchangerootdir");
	if (option) {
		if (geteuid() == 0) {
			chdir(option);
			ret = chroot(option);
			if (!ret) {
				log(7, "Changed root directory at start up"
					"to %s", option);
				chrooted = 1;
			} else {
				log(7, "Error changeing root "
				"directory on start up to %s: %s",
				option, strerror(errno));
				chrooted = 0;
			}
			chdir("/");
		}
	}
	if (getuid() == 0) {
		/* set the effective UID to the one of "runuser". We can't
		 * drop the real user id since we never know if the user
		 * wants to bind to a priviledged port (he could send us a
		 * HUP signal and we have to reread the configuration file.
		 * So we can't say, if port > IPPORT_RESERVED change real uid)
		 */
		log(8, "Changing ID to the runasuser-ID (initial)");
		changeid(get_option("runasuser"), EUID);
	}


	/* Install the signal handlers */

	/* for SIGCHLD install just the reap function
	 *
	 * the register/unregister thing is installed after we've bound
	 * successfully */

	sa.sa_handler = reap_chld_info;
	sigemptyset (&sa.sa_mask);
#ifndef WINDOWS
	sa.sa_flags = SA_RESTART;
#endif
	sigaction (SIGCHLD, &sa, 0);

	cf.sa_handler = read_default_conf;
	should_read_config = 0;
	sigemptyset(&cf.sa_mask);
#ifndef WINDOWS
	cf.sa_flags = SA_RESTART;
#endif
	sigaction (SIGHUP, &cf, 0);

	cf.sa_handler = terminate;
	sigemptyset(&cf.sa_mask);
	sigaction (SIGTERM, &cf, 0);
	sigaction (SIGQUIT, &cf, 0);
	sigaction (SIGABRT, &cf, 0);

	clntinfo.user = clntinfo.pass = clntinfo.destination = (char*) 0;
	clntinfo.throughput = 0;
	clntinfo.boundsocket_list = (int*) 0;

	atexit(closedescriptors);

	bindaddress = getlisten();

	if (!ret) {
		ret = handlecmds(bindaddress, 
				 &clntinfo);
		log(8, "Exited from handlecmds");
	}

	log(8, "Exiting");
	if (ret) {
		return 1;
	} else {
		return 0;
	}
}


int daemonize()
{
	int childpid;

	if( (childpid = fork()) < 0) return(-1);
	else if(childpid > 0) exit(0);

	errno = 0;
	chdir("/");
	setsid();
	return(0);
}


void read_default_conf(int signo) {
	should_read_config = 1;
}

int reread_config() {
	int ret;
	const char* tmp;
	sigset_t sigset, oldset;
	log(6, "SIGHUP received - Rereading config file. My pid: %d", getpid());
	/* block other SIGHUP signals */

	sigemptyset(&sigset);
	sigemptyset(&oldset);
	sigaddset(&sigset, SIGCHLD);
	while ((ret = sigprocmask(SIG_BLOCK, &sigset,
		&oldset)) < 0 && errno == EINTR)  {}
	if (ret < 0) {
		log(2, "Error blocking signals: %s", strerror(errno));
	}
	clean_allowedlist();
	conf_delete_config();
	ret = read_config(conffilename);
	if (ret) {
		/* the following line is a BADF if we had a SIGHUP */
		log(1, "Error rereading config file. Exiting");
		exit(2);
	}
	/* Are we chroot()ed ? */
	if (chrooted) {
		tmp = get_option("atstartchangerootdir");
		if (!tmp) {
			tmp = get_option("changerootdir");
		}
	} else {
		tmp = (char*) 0;
	}
	reset_confst(&config);
	init_logfiles(tmp);
	should_read_config = 0;
	while ((ret = sigprocmask(SIG_UNBLOCK, &sigset,
		&oldset)) < 0 && errno == EINTR) {}
	if (ret < 0) {
		log(2, "Error unblocking signal mask: %s", strerror(errno));
		return -1;
	}
	return 0;
}

void terminate (int signo) {
	/* exit is not POSIX-reentrant but in SVR4 SVID */
	exit(0);
}

int init_logfiles(const char* dir_prefix) {
	int ret =0, i;
	char buf[1024];
	const char* option;
	const char* specs;
	const char* style;
	char* cpy;
	const int MIDLENGTH = 64;
	struct cmdlogent_t* cmdlogfilescur = &config.cmdlogfiles;
	struct cmdlogent_t* cmdlogdirscur = &config.cmdlogdirs;
	int logf_name_size;

	if (!dir_prefix) {
		dir_prefix = "";
	}

	if (get_option("logstyle") 
		&& strcasecmp(get_option("logstyle"), "syslog") == 0) {
		config.syslog = 1;
	} else {
		config.syslog = 0;
	}
	/* read the logfile name from the configuration file */
	option = get_option("logfile");
	if (!option) {
		if (strlen(dir_prefix)) {
			logf_name_size = strlen(dir_prefix)
					+ strlen(DEFAULTLOGFILE) + 1 + 1;
			config.logf_name = (char*) malloc(logf_name_size);
			enough_mem(config.logf_name);
			snprintf(config.logf_name, logf_name_size,
					"%s/%s", dir_prefix, DEFAULTLOGFILE);
		} else {
			config.logf_name = strdup(DEFAULTLOGFILE);
			enough_mem(config.logf_name);
		}
		char_squeeze(config.logf_name, '/');
		log(4, "No logfile specified in configuration file");
	} else {
		if (strlen(dir_prefix)) {
			logf_name_size = strlen(dir_prefix)
						+ strlen(option) + 1 + 1;
			config.logf_name = (char*) malloc(logf_name_size);
			enough_mem(config.logf_name);
			snprintf(config.logf_name, logf_name_size,
					"%s/%s", dir_prefix, option);
		} else {
			config.logf_name = strdup(option);
			enough_mem(config.logf_name);
		}
		char_squeeze(config.logf_name, '/');
	}
	if (!get_option("debuglevel")) {
		log(4, "No debug level specified. Using level 7");
		config.debuglevel = 7;
	} else {
		const char* tmp = get_option("debuglevel");
		errno = 0;
		config.debuglevel = strtol(tmp, (char**) 0, 10);
		if (errno || config.debuglevel < 0 || config.debuglevel > 9) {
			log(4, "Invalid debuglevel specified: \"%s\". Using level 7", tmp);
			config.debuglevel = 7;
		}
	}


	/* Open the log file if not using syslog */
	if (!config.syslog) {
		if (!(config.logf = open_logfile(config.logf_name))) {
			return -1;
		}
	} else {
		if (multithread) {
			openlog(binaryname, LOG_PID, LOG_DAEMON);
		} else {
			openlog(binaryname, LOG_PID, LOG_USER);
		}
		syslog(LOG_INFO, "Opened syslog");
	}

	/* Open the cmdlogfilex logfile(s) */

	for (i =1; i < 9999; i++) {
		snprintf(buf, sizeof(buf), "cmdlogfile%d", i);
		option = get_option(buf);
		if (!option) {
			break;
		}
		else {
			int cpysize;
			snprintf(buf, sizeof(buf), "cmdlogfile%d-specs", i);
			specs = get_option(buf);
			if (strlen(dir_prefix)) {
				cpysize = strlen(option)
						+ strlen(dir_prefix) + 2;
				cpy = (char*) malloc(cpysize);
				enough_mem(cpy);
				snprintf(cpy, cpysize, "%s/%s", dir_prefix,
						option);
			} else {
				cpy = strdup(option);
				enough_mem(cpy);
			}
			char_squeeze(cpy, '/');

			if (i != 1) {
				cmdlogfilescur->next = malloc(sizeof
					(struct cmdlogent_t));
				enough_mem(cmdlogfilescur->next);
				cmdlogfilescur = cmdlogfilescur->next;
			}
			cmdlogfilescur->logf_name = cpy;
			cpy = (char*) 0;
			cpysize = strlen(specs) + 2;
			cmdlogfilescur->specs = (char*) malloc(cpysize);
			enough_mem(cmdlogfilescur->specs);
			snprintf(cmdlogfilescur->specs, cpysize, "%s ", specs);

			snprintf(buf, sizeof(buf), "cmdlogfile%d-style", i);
			style = get_option(buf);
			if (!style) {
				/* use commonlog as default */
				cmdlogfilescur->style = LS_COMMONLOG;
			} else if (strcasecmp(style, "xferlog") == 0) {
				cmdlogfilescur->style = LS_XFERLOG;
			} else if (strcasecmp(style, "commonlog") == 0) {
				cmdlogfilescur->style = LS_COMMONLOG;
			} else {
				/* default */
				log(4, "Invalid logfile style %s for %s",
						style, buf);
				cmdlogfilescur->style = LS_COMMONLOG;
			}

			cmdlogfilescur->next = 0;
			if ((cmdlogfilescur->logf = 
				open_logfile(cmdlogfilescur->logf_name))
					!= NULL) {
				ret = 0;
			} else {
				/* no free()s here, the process will
				 * terminate anyway */
				return -1;
			}
		}
	}

	/* Do the same with directories */

	for (i = 1; i < 9999; i++) {
		snprintf(buf, sizeof(buf), "connectionlogdir%d", i);
		option = get_option(buf);
		if (!option) {
			break;
		} else {
			const char* prefix =0, *suffix =0;
			int size;
			snprintf(buf, sizeof(buf),
					"connectionlogdir%d-specs", i);
			specs = get_option(buf);
			if (!specs) {
				/* will never log */
				log(4, "Warning: no specs specified (%s)", buf);
				specs = "";
			}
			snprintf(buf, sizeof(buf),
					"connectionlogdir%d-fileprefix", i);
			prefix = get_option(buf);
			if (!prefix) {
				prefix = "";
			}
			snprintf(buf, sizeof(buf),
					"connectionlogdir%d-filesuffix", i);
			suffix = get_option(buf);
			if (!suffix) {
				suffix = "";
			}
			size = strlen(option) + MIDLENGTH +
				strlen(prefix) + strlen(suffix) + 1;
			cpy = (char*) malloc(size);
			enough_mem(cpy);

			snprintf(cpy, size,
				"%s/%s%%Y-%%m-%%d--%%H:%%M:%%S-%%%%d%s",
				option, prefix, suffix);

/*			log(9, "logging to (raw) file: %s", cpy); */

			if (i != 1) {
				cmdlogdirscur->next = malloc(sizeof
					(struct cmdlogent_t));
				enough_mem(cmdlogdirscur->next);
				cmdlogdirscur = cmdlogdirscur->next;
			}
			cmdlogdirscur->logf_name = cpy;
			cpy = (char*) 0;
			cmdlogdirscur->logf_size = size;
			cmdlogdirscur->logf = NULL;

			size = strlen(specs) + 2;
			cmdlogdirscur->specs = (char*) malloc(size);
			enough_mem(cmdlogdirscur->specs);

			snprintf(cmdlogdirscur->specs, size, "%s ", specs);

			snprintf(buf, sizeof(buf),
					"connectionlogdir%d-style", i);
			style = get_option(buf);
			if (!style) {
				/* use commonlog as default */
				cmdlogdirscur->style = LS_COMMONLOG;
			} else if (strcasecmp(style, "xferlog") == 0) {
				cmdlogdirscur->style = LS_XFERLOG;
			} else if (strcasecmp(style, "commonlog") == 0) {
				cmdlogdirscur->style = LS_COMMONLOG;
			} else {
				/* default */
				log(4, "Invalid logfile style %s for %s",
						style, buf);
				cmdlogdirscur->style = LS_COMMONLOG;
			}

			cmdlogdirscur->next = 0;
		}
	}
	return ret;
}

int init_logdirs(const char* prefix) {
	struct cmdlogent_t* clf = &config.cmdlogdirs;
	time_t nowtime;
	char* tmp =0;
	int ret = 0;

	if (config.cmdlogdirs.logf_name == NULL) {
		return 0;
	}

	nowtime = time(NULL);

	if (!prefix) {
		prefix = "";
	}
	tmp = (char*) malloc(clf->logf_size);
	enough_mem(tmp);

	while (clf && !ret) {
		strftime(tmp, clf->logf_size, clf->logf_name,
				localtime(&nowtime));
		snprintf(clf->logf_name, clf->logf_size, tmp, getpid());
		free(tmp);
		if (prefix) {
			int tmpsize = strlen(clf->logf_name)
					+ strlen(prefix)
					+ 1
					+ 1;
			tmp = (char*) malloc(tmpsize);
			enough_mem(tmp);

			snprintf(tmp, tmpsize, "%s/%s",
					prefix, clf->logf_name);

			char_squeeze(tmp, '/');
			free(clf->logf_name);
			clf->logf_name = tmp;
			tmp = (char*) 0;
		}
		if ((clf->logf = open_logfile(clf->logf_name))) {
			ret = 0;
		} else {
			ret = -1;
		}
		clf = clf->next;
	}
	return ret;
}

void print_version(void) {
	printf(PACKAGE" v"JFTPGW_VERSION);
#ifdef SFTP_SUPPORT
	printf("  -  sftp support enabled");
#else
	printf("  -  without sftp support");
#endif
	printf("\n");
}

void print_help(void) {
	print_version();
	printf("usage: jftpgw [OPTION]\n\n");
	printf("Valid options:\n");
	printf("  -h, --help                Display this help text\n");
	printf("  -e, --encrypt             Use jftpgw to obtain an encrypted password\n");
	printf("  -f, --configfile file     Load file instead of default config file\n");
	printf("  -s, --single              Run jftpgw single threaded (do not fork)\n");
	printf("  -V, -v, --version         Display the version\n");
	printf("\nReport bugs to Joachim Wieland <joe@mcknight.de>\n");
}

void removepidfile(void) {
	const char* option;
	if (getpid() != main_server_pid) {
		/* the program has not become a daemon */
		return;
	}

	option = get_option("pidfile");
	if (option) {
		int i;
		if (getuid() == 0 && geteuid() != 0) {
			log(8, "Changing ID to root (create pidfile)");
			changeid("root", UID);
		}
		i = unlink(option);
		if (i < 0) {
			log(3, "Could not unlink the pidfile %s", option);
		}
		if (getuid() == 0) {
			log(8, "Changing id back (deleting pidfile)");
			changeid(get_option("runasuser"), EUID);
		}
	}
}

void sayterminating(void) {
	log(6, "jftpgw terminating");
}

void closedescriptors(void) {
	int i;
	log(9, "In closedescriptors()");

	/* free the log info structure. Free the members, the structure for
	 * itself is on the stack */
	/* li.cmd must not be freed, it's   li.cmd = buffer;  */
	/* the same for li.name */
	if (li.host) {
		free(li.host);
	}
	if (li.user) {
		free(li.user);
	}
	if (conffilename) {
		free(conffilename);
	}

	/* close the logfiles and delete the structures */
	reset_confst(&config);

	/* maybe these variables are not yet allocated */
	if (clntinfo.user) {
		free(clntinfo.user);
	}
	if (clntinfo.pass) {
		free(clntinfo.pass);
	}
	if (clntinfo.destination) {
		free(clntinfo.destination);
	}
	clean_allowedlist();
	free_errstr();

	if (clntinfo.boundsocket_list) {
		for (i = 0; i < clntinfo.boundsocket_niface; i++) {
	        	close(clntinfo.boundsocket_list[i]);
		}
		free(clntinfo.boundsocket_list);
	}
	close(clntinfo.clientsocket);
	close(clntinfo.serversocket);
}

