#include <fcntl.h>
#include <ctype.h>
#include <sys/time.h>
#include <arpa/telnet.h> /* for IAC, IP */
#include "jftpgw.h"


/* include the different command sets */

#include "cmds.h"
#include "std_cmds.h"

#ifdef SFTP_SUPPORT
#include "secsftp_cmds.h"
#endif

extern int timeout;
extern int serverport;
extern struct loginstruct lsforward;
char* portcmd;

struct loginfo li = { NULL, NULL, NULL, 0, 0 };
struct conn_info_st conn_info = { 0, 0, 0, 0 };
struct cmdhandlerstruct *cmdhandler;

int checkforabort(struct clientinfo*);

int handlecmds(const char* hostnames, struct clientinfo *clntinfo) {
	char *buffer = 0;
	int ss, cs, ret;
	int i;
	unsigned long int iaddr;
	struct hostent *host;

	conn_info.lsforward = &lsforward;
	conn_info.li = &li;
	conn_info.clntinfo = clntinfo;

	ss = clntinfo->serversocket = -1;
	clntinfo->dataclientsock = clntinfo->dataserversock = -1;
	conn_info.usergiven = 0;
	portcmd = 0;
	if ((ret = waitclient(hostnames, clntinfo)) < 0) {
		/* An error occured */
		return -1;
	}
	cs = clntinfo->clientsocket;

	/* use li.host as a temporary pointer */
	li.host = getpeer_ip(clntinfo->clientsocket);
	iaddr = inet_addr(li.host);
	free(li.host);
	host = gethostbyaddr((char*) &iaddr, sizeof(iaddr), AF_INET);
	if (host) {
		li.host = strdup(host->h_name);
	} else {
		struct in_addr in;
		in.s_addr = clntinfo->clientaddr;
		li.host = strdup(inet_ntoa(in));
	}
	enough_mem(li.host);

	li.user = 0;

	while (1) {
contin:
#ifdef SFTP_SUPPORT
		if (clntinfo->servermode == SECURE_SFTP) {
			cmdhandler = &secsftp_cmdhandler[0];
		} else {
#endif
			cmdhandler = &std_cmdhandler[0];
#ifdef SFTP_SUPPORT
		}
#endif
		ss = clntinfo->serversocket;
		if (timeout) {
			log(2, "Timeout in %s line %d\n", __FILE__ ,__LINE__);
			return -1;
		}
		if (buffer) {
			/* log the command */
			log_cmd(&li);
			free(buffer);
			buffer = 0;
			li.respcode = 0;
			li.transferred = 0;
		}
		buffer = ftp_readline(cs);
		if (!buffer && timeout) {
			log(2, "Timeout in %s line %d\n", __FILE__ ,__LINE__);
			err_time_readline(cs);
			return -1;
		}
		if (!buffer) {
			err_readline(cs);
			return -1;
		}
	
		if (buffer[0] == '\0') {
			/* empty line. Prevent logging of the command */
			free(buffer);
			buffer = 0;
			goto contin;
		}
		
		li.cmd = buffer;

		/* Retrieve a file */
		clntinfo->mode = RETR;
		i = 0;

		/* If we are not yet connected, the only valid commands are
		 * USER and PASS */
		if (ss == -1 && !checkbegin(buffer, "USER ")
			     && !checkbegin(buffer, "PASS ")
			     && !checkbegin(buffer, "QUIT")) {
			say(cs, "530 Please login with USER and PASS.\r\n");
			li.respcode = 530;
			goto contin;
		}

		while (cmdhandler[i].cmd) {
			if (checkbegin(buffer, cmdhandler[i].cmd)) {
				int ret = (cmdhandler[i].func)
						(buffer, &conn_info);
				switch (ret) {
					case CMD_PASS:
						if (passcmd(buffer, clntinfo)
								< 0) {
							free(buffer);
							return -1;
						}
						break;
					case CMD_HANDLED:
						break;
					case CMD_QUIT:
						free(buffer);
						return 0;
				}
				/* found and called proper function */
				goto contin;
			}
			i++;
		}
		/* we didn't find a handler function for the command */
		/* pass it, if we are using the standard command set */

		if (cmdhandler == &std_cmdhandler[0]) {
			if (passcmd(buffer, clntinfo) < 0) {
				free(buffer);
				return -1;
			}
		} else {
			say(clntinfo->clientsocket, 
			"500 Command not understood or not implemented.\r\n");
		}
	}
	/* li.host and li.user  are freed at the termination of the programm */
}



/* getuserdest() parses the "USER xyz" login string */

int getuserdest(char* buffer, struct clientinfo *clntinfo) {
	char* dest, *user, *end, *orig_login;
	char* buffnew = 0;
	const char* port, *opt;
	int i, f, cs;
	i = 5;
	f = 0;
	cs = clntinfo->clientsocket;
	clntinfo->servermode = getservermode();
	while (buffer[i] == ' ')
		i++;
	if(i > strlen(buffer)) {
		say(cs, "500 'USER': command requires a parameter.\r\n");
		li.respcode = 500;
		return 1;
	}

	user = clntinfo->user = (char*) malloc(strlen(buffer) + 1);
	enough_mem(user);
	dest = clntinfo->destination = (char*) malloc(strlen(buffer)+1);
	enough_mem(dest);
	clntinfo->destinationport = 0;

	opt = get_option("passallauth");
	if (opt) {
		/* passallauth specified */
		/* transform buffer into "buffer"@targethost,21 */

		orig_login = strdup(&buffer[i]);
		enough_mem(orig_login);
		
		buffer = (char*) realloc(buffer,
				  strlen("USER ")
				+ 1                         /* " */
				+ strlen(orig_login)        /* orig_login */
				+ 1                         /* " */
				+ 1                         /* @ */
				+ strlen(opt)               /* opt */
				+ 1);                       /* Termination */
		sprintf(buffer, "USER \"%s\"@%s", orig_login, opt);
		free(orig_login);
		orig_login = 0;
	}
	
	if (get_forward(&buffer[i], &lsforward) == 0) {
		/* fill at least fwlogin in the structure */
		lsforward.fwlogin = extract_userhost(&buffer[i]);
	} else {
		/* parse username from buffer[i] */
		lsforward.fwuser = extract_username(&buffer[i]);
		buffer = lsforward.login;
		user = clntinfo->user = 
			(char*) realloc(user, strlen(buffer) + 1);
		enough_mem(user);
		dest = clntinfo->destination =
			(char*) realloc(dest, strlen(buffer) + 1);
		enough_mem(dest);

		i = 0;
	}

	if (buffer[i] == '\"') {
		/* everything between two "" is a username
		 * useful when logging in to a proxy */
		end = strrchr(&buffer[i+1], '\"');
		
		if (end) {
			/* quotation marks balanced */
		
			/* f specifies the length of the username */
			f = end - &buffer[i+1];
			strncpy(user, &buffer[i+1], f);
			i += f + 2 /* quotation marks */;

			/* The next character after the closing
			 * quotation mark should be a @ or , so the
			 * while loop is not executed and user gets
			 * null-terminated. If it is not, the
			 * characters are appended to user[] */
		}
	}
	while(buffer[i] != '@' && buffer[i] != ',' 
			&& i < strlen(buffer)) {
		user[f] = buffer[i];
		i++; f++;
	}
	user[f] = '\0';
	i++;  /* shift over @ or , */
	if (buffer[i-1] == '\0') {
		const char* tmp;
		const char* addtarget = 0;
		char ipbuf[16];
		/* first, lets terminate user. Since it is the last
		 * value in the line, it is followed by an \r\n */
		while(user[--f] < 32 && f >= 0) {
			user[f] = '\0';
		}
		tmp = get_option("defaultforward");
		if (!tmp) {
			struct sockaddr_in sin;
			const char* fwhost = 0;
#ifdef HAVE_SOCKLEN_T
		        socklen_t namelen;
#else
 			int namelen;
#endif
        		namelen = sizeof(sin);

			log(6, "No defaultroute specified, acting as transparent proxy");
			sin.sin_addr.s_addr = get_showaddr(
					clntinfo->clientsocket);
			strcpy(ipbuf, inet_ntoa(*(struct in_addr *)
						&sin.sin_addr));
			tmp = ipbuf;
			
			/* Determine the IP the client sees of us */
		        if (getpeername(clntinfo->clientsocket,
				(struct sockaddr *) &sin, &namelen) != 0) {
        			log(3, "Could not get peername for TP");
			}
			clntinfo->clientaddr = get_our_addr_to(
					sin.sin_addr.s_addr, sin.sin_port);
			sin.sin_addr.s_addr = clntinfo->clientaddr;
			log(8, "Our IP to the client is %s", inet_ntoa(sin.sin_addr));

 			if ((fwhost = get_option("transparent-forward"))) {
 				/* user => user@targethost */
 				/* tmp => dest => forwardhost */
 
 				user = realloc(user, strlen(user) 
 						+ 1 + strlen(ipbuf) + 1);
 				enough_mem(user);
 				clntinfo->user = user;
 				strcat(user, "@");
 				strcat(user, ipbuf);
 
 				addtarget = fwhost;
 				log(8, "Using transparent proxy. Connecting to %s. User: %s", addtarget, user);
 			} else {
 				log(8, "Connecting to %s as transparent proxy",
 					tmp);
 				/* resize dest */
 				dest = (char*) realloc(dest, strlen(tmp)+1);
 				clntinfo->destination = dest;
 				strcpy(dest, tmp);
 				/* we don't have to free tmp */
 				addtarget = 0;
 			}
		} else {
			addtarget = tmp;
		}
		log(9, "Addtarget: %s", addtarget);
		if (addtarget) {
			/* copy the option of defaultforward into buffer */
			buffnew = (char*) malloc(strlen(buffer) 
 							+ strlen(addtarget)
 							+ 1 + 1);
			enough_mem(buffnew);
			dest = (char*) realloc(dest, strlen(addtarget) + 1);
 			clntinfo->destination = dest;
			strcpy(buffnew, buffer);
			strcat(buffnew, "@");
 			strcat(buffnew, addtarget);
			buffer = buffnew;
 			/* we don't have to free addtarget */
			log(9, "Appended defaultforward option. New login string: %s. buffer[i]: %s", buffer, buffer+i);
		}
	}
	if (buffer[i-1] != '\0') {
		f =0;
		while(buffer[i] && i < strlen(buffer)
				&& buffer[i] != ','
				&& buffer[i] != ':') {
			dest[f] = buffer[i];
			i++; f++;
		}
		dest[f] = '\0';
		while (f > 0 && dest[--f] < 32) {
			dest[f] = '\0';
		}
		port = strchr(buffer + i, ',');
		if (!port) {
			port = strchr(buffer + i, ':');
		}
		if (port) {
			/* There is either a port specification
			 * (decimal number) or server/client mode
			 * specification */
		
			/* shift over the : or , */
			port++;
			if (0 == strchr("aps", *port))
				/* *port != 'a' && *port != 'p') */ {
				log(9, "analysing port: %s", port);
				clntinfo->destinationport = atoi(port);
				port = strchr(buffer + i + 1, ',');
				if (!port) {
					port = strchr(buffer + i + 1,
							':');
				}
				if (port) {
					port++;
				}
			}
#ifdef SFTP_SUPPORT
			if (port && (strchr("aps", *port))) {
#else
			if (port && (strchr("ap", *port))) {
#endif
				log(9, "mode specified: %s", port);
				switch (*port) {
					case 'a': 
					clntinfo->servermode = ACTIVE;
					log(9, "p-s: active");
					break;
					case 'p':
					clntinfo->servermode = PASSIVE;
					log(9, "p-s: passive");
					break;
#ifdef SFTP_SUPPORT
					case 's':
					clntinfo->servermode = SECURE_SFTP;
					log(9, "p-s: secure-sftp");
					break;
#endif
				}
			}
		}
	}
	if (buffnew) {
		free(buffnew);
		buffnew = 0;
	}
	if (!lsforward.fwuser) {
		/* lsforward.fwuser = (char*) malloc(strlen(user) + 1);
		strcpy(lsforward.fwuser, user); */
		lsforward.fwuser = extract_username(user);
	}
	li.user = strdup(lsforward.fwuser);
	enough_mem(li.user);
	if (!clntinfo->destinationport) {
		clntinfo->destinationport = getserverport();
	}

	log(7, "Client logged in: User: %s, Dest: %s", user, dest);
	switch (clntinfo->servermode) {
		case SECURE_SFTP:
			log(8, "Secure sftp mode");
			break;
		default:
			log(8, "Standard FTP mode, port: %d",
				clntinfo->destinationport);
	}

	/* Get the Password */
	return 0;
}


int getpasswd(char* buffer, struct clientinfo *clntinfo) {
	int i, f, cs;
	char* pass;
	pass = clntinfo->pass = (char*) malloc(strlen(buffer) + 1);
	cs = clntinfo->clientsocket;
	
	f = strlen("PASS ");
	i = 0;
	while (buffer[f] == ' ')
		f++;
	while (buffer[f]) {
		pass[i] = buffer[f];
		i++; f++;
	}
	memset(buffer + f, 0, strlen(buffer) - f);
	pass[i] = '\0';
	while (i > 0 && pass[i] < 32) {
		pass[i] = '\0';
		i--;
	}
	return 0;
}


int passcmd(char* buffer, struct clientinfo *clntinfo) {
	int cs = clntinfo->clientsocket;
	int ss = clntinfo->serversocket;
	char* sendbuf =0;
	char *last = 0;
	int ret;

	sendbuf = (char*) malloc(strlen(buffer) + 3);
	strcpy(sendbuf, buffer);
	strcat(sendbuf, "\r\n");
	say(ss, sendbuf);
	free(sendbuf);
	last = passall(ss, cs);
	if (!last) {
		if (timeout) {
			log(2, "Timeout in %s line %d\n", __FILE__ ,__LINE__);
			err_time_readline(cs);
		} else {
			err_readline(cs);
		}
		return -1;
	}
	if (checkdigits(last, 150) || checkdigits(last, 125)) {

		if (portcmd) {
			ret = activeclient(portcmd, clntinfo);
			free(portcmd);
			portcmd = 0;
			if (ret) {
				log(2, "Error in activeclient()");
			}
		}

		if (initiate_transfer(clntinfo) < 0) {
			free(last);
			return -1;
		}
		
		ret = transmitfile(clntinfo);

		if (!ret) {
			char* t = passall(ss, cs);
			li.respcode = respcode(t);
			free(t);
		}
		else {
			if (ret == 1) { /* generate error message */
				if (timeout) {
			log(2, "Timeout in %s line %d\n", __FILE__ ,__LINE__);
					err_time_readline(cs);
				} else {
					err_readline(cs);
				}
			}
			li.respcode = respcode(last);
			free(last);
			return -1;
		}
	}
	free(/*at*/ last);  /*  ;-)  */
	last = 0;
	return 0;
}


int initiate_transfer(struct clientinfo *clntinfo) {
	int cs = clntinfo->clientsocket;
	int ret;
#ifdef HAVE_SOCKLEN_T
	socklen_t count;
#else
	int count;
#endif
	fd_set acc_set;
	struct timeval tmo = { 300, 0 };
	struct sockaddr_in sin;

	count = sizeof(sin);
	
	if (clntinfo->servermode == ACTIVE 
	    && clntinfo->clientmode == PASSIVE) {
		/* accept connections from both sides */
		int servacc = 0, clntacc = 0;
		while (servacc == 0 || clntacc == 0) {
			FD_ZERO(&acc_set);
			FD_SET(clntinfo->dataserversock, &acc_set);
			FD_SET(clntinfo->dataclientsock, &acc_set);
			ret = select(MAX_VAL(clntinfo->dataserversock,
					 clntinfo->dataclientsock) + 1,
					&acc_set, NULL, NULL, &tmo);
			if (ret < 0) {
				log(2, "Select() error");
				return -1;
			}
			if (ret == 0) {
				log(2, "Connection timed out");
				li.respcode = 500;
				say(cs, "500 Connection timed out\r\n");
				return -1;
			}
			if (FD_ISSET(clntinfo->dataserversock, &acc_set)) {
				ret = accept(clntinfo->dataserversock,
					(struct sockaddr*) &sin, &count);
				if (ret < 0) {
					log(2, "Error in accept() #1: %s",
							strerror(errno));
					return -1;
				}
				close(clntinfo->dataserversock);
				clntinfo->dataserversock = ret;
				servacc = 1;
			}
			if (FD_ISSET(clntinfo->dataclientsock, &acc_set)) {
				ret = accept(clntinfo->dataclientsock,
					(struct sockaddr*) &sin, &count);
				if (ret < 0) {
					log(2, "Error in accept() #2: %s",
							strerror(errno));
					return -1;
				}
				close(clntinfo->dataclientsock);
				clntinfo->dataclientsock = ret;
				clntinfo->waitforconnect = 0;
				clntacc = 1;
			}
		}
	}
	if (clntinfo->waitforconnect) {
		FD_SET(*clntinfo->waitforconnect, &acc_set);
		ret = select(*clntinfo->waitforconnect + 1, &acc_set,
				NULL, NULL, &tmo);
		if (ret < 0) {
			log(2, "Select() error");
			return -1;
		}
		if (ret == 0) {
			log(2, "Connection timed out");
			li.respcode = 500;
			say(cs, "500 Connection timed out\r\n");
			return -1;
		}

		/* accept connection from one side */
		ret = accept(*clntinfo->waitforconnect,
				(struct sockaddr*) &sin, &count);
		if (ret < 0) {
			log(2, "Error in accept() #3: %s",
					strerror(errno));
			return -1;
		}
		close(*clntinfo->waitforconnect);
		*clntinfo->waitforconnect = ret;
	}
	
	if (clntinfo->mode == STOR) {
		/* swap descriptors */
		ret = clntinfo->dataserversock;
		clntinfo->dataserversock = clntinfo->dataclientsock;
		clntinfo->dataclientsock = ret;
	}
	return 0;
}

/* #define TRANSMITBUFSIZE  (10*1024) */
#define TRANSMITBUFSIZE  PIPE_BUF
int transmitfile(struct clientinfo *clntinfo)  {
	char* buffer = (char*) malloc(TRANSMITBUFSIZE);
	char* pbuf = 0;
	int count = 0, nwritten, totwritten, sret = 0, scret = 0;
	int cs = clntinfo->clientsocket;
	/* error == 0 : success
	 * error == 1 : fail, generate error message
	 * error == 2 : fail, error message already generated
	 */
	int n, ret, error = 0;
	int fdflags;
	fd_set readset, writeset, exceptset;
	struct timeval readtime = { 300, 0 };
	struct timeval writetime = { 300, 0 }; 
	time_t start, done, delay;
	sigset_t sigset, oldset;

	enough_mem(buffer);
	FD_ZERO(&readset);
	FD_ZERO(&writeset);
	FD_ZERO(&exceptset);
	FD_SET(clntinfo->dataserversock, &readset);
	FD_SET(cs, &exceptset);
	FD_SET(cs, &readset);
	FD_SET(clntinfo->dataclientsock, &writeset);

	log(7, "Throughputrate is %3.3f", clntinfo->throughput);

	/* set the descriptor nonblocking */
	if ((fdflags = fcntl(clntinfo->dataclientsock, F_GETFL)) < 0) {
		log(2, "Error getting fcntl() flags");
		fdflags = 0;
	}
	ret = fcntl(clntinfo->dataclientsock, F_SETFL, fdflags | O_NONBLOCK);
	if (ret < 0) {
		log(2, "Error setting fd to nonblocking");
	}

	totwritten = 0;
	start = time(NULL);

	sigemptyset(&sigset);
	sigemptyset(&oldset);
	sigaddset(&sigset, SIGCHLD);
	ret = sigprocmask(SIG_BLOCK, &sigset, &oldset);
	if (ret < 0) {
		log(3, "sigprocmask() error: %s", strerror(errno));
	}
	
	while((sret = select(MAX_VAL(clntinfo->dataserversock, cs) + 1,
				&readset, NULL, &exceptset,
				&readtime) > 0)) {
		/* save the errno value from sigprocmask() */
		n = errno;
		ret = sigprocmask(SIG_UNBLOCK, &sigset, &oldset);
		if (ret < 0) {
			log(3, "sigprocmask() error releasing the blocked"
				" signals: %s", strerror(errno));
		}
		errno = n;

		if (FD_ISSET(clntinfo->dataserversock, &readset)) {
			count = read(clntinfo->dataserversock,
				buffer, TRANSMITBUFSIZE);
			if (count == 0) {
				break;
			}
			if (count < 0) {
				log(3, "read error: %s", strerror(errno));
				sret = 1;
				error = 1;
				break;
			}
			if (clntinfo->havetoconvert != CONV_NOTCONVERT) {
				char* tmp;
				if (clntinfo->havetoconvert == CONV_TOASCII) {
					tmp = to_ascii(buffer, &count);
					if (count > TRANSMITBUFSIZE) {
						buffer = realloc(buffer, count);
					}
					memcpy((void*) buffer,
						(void*) tmp, count);
					free(tmp);
				}
				if (clntinfo->havetoconvert == CONV_FRMASCII) {
					tmp = from_ascii(buffer, &count);
					buffer = tmp;
				}
			}

			pbuf = buffer;
			do {
				FD_ZERO(&writeset);
				FD_SET(clntinfo->dataclientsock, &writeset);

				sigemptyset(&sigset);
				sigemptyset(&oldset);
				sigaddset(&sigset, SIGCHLD);
				ret = sigprocmask(SIG_BLOCK, &sigset, &oldset);
				if (ret < 0) {
					log(3, "sigprocmask() error: %s",
						strerror(errno));
				}

				scret = select(clntinfo->dataclientsock + 1,
					NULL, &writeset, NULL, &writetime);
				/* save the errno value from sigprocmask() */
				n = errno;
				ret = sigprocmask(SIG_UNBLOCK, &sigset,
						&oldset);
				if (ret < 0) {
					log(3, "sigprocmask() error releasing "
						"the blocked signals: %s",
						strerror(errno));
				}
				errno = n;

				if (scret <= 0) {
					break;
				}
				/* otherwise the descriptor must be ready */
				nwritten = write(clntinfo->dataclientsock,
						buffer, count);
				if(nwritten < 0) break;
				totwritten += nwritten;
				done = time(NULL);
				delay = (totwritten/1024.00) /
					clntinfo->throughput;
				delay += start;
				delay -= done;
				if (delay > 0 && clntinfo->throughput != -1) {
					sleep(delay);
				}
				if (nwritten != count) {
					pbuf += nwritten;
					count -= nwritten;
				} else {
					pbuf = 0;
				}
			} while (pbuf);
		}
		if(FD_ISSET(cs, &exceptset) || FD_ISSET(cs, &readset)) {
			if (checkforabort(clntinfo)) {
				error = 2;
				break;
			}
		}
		/* restore the sets */
		FD_ZERO(&readset);
		FD_ZERO(&writeset);
		FD_ZERO(&exceptset);
	        FD_SET(clntinfo->dataserversock, &readset);
	        FD_SET(clntinfo->dataclientsock, &writeset);
	        FD_SET(cs, &exceptset);
	        FD_SET(cs, &readset);
		/* Linux modifies the timeout struct */
		readtime.tv_sec = 300;
		readtime.tv_usec = 0;
		writetime.tv_sec = 300;
		writetime.tv_usec = 0;
	}

	/* Restore flags */
	ret = fcntl(clntinfo->dataclientsock, F_SETFL, fdflags);
	if (ret < 0) {
		log(2, "Error resetting fd flags");
	}

	if (sret == -1 || scret == -1) {
		log(2, "Error in select() occured: %s", strerror(errno));
		li.respcode = 500;
		say(cs, "500 Error in select() occured\r\n");
		error = 2;  /* do not generate an error message */
	}
	/* count == 0 means correct end of transfer */
	if ((sret == 0 || scret == 0) && count != 0) {
		log(2, "Connection timed out in transmitfile(): %d, %d",
				readtime.tv_sec, writetime.tv_sec);
		log(9, "Sockets: dataserver: %d, dataclient: %d, client: %d",
				clntinfo->dataserversock,
				clntinfo->dataclientsock,
				cs);
		if (scret == 0) {
			log(9, "scret == 0 (write)");
		}
		if (sret == 0) {
			log(9, "sret == 0 (read)");
		}
		error = 1;
	}

	close(clntinfo->dataclientsock);
	close(clntinfo->dataserversock);
	clntinfo->dataclientsock = -1;
	clntinfo->dataserversock = -1;
	/*
	clntinfo->servermode = ASCLIENT;
	clntinfo->clientmode = ASCLIENT;
	*/

	li.transferred = totwritten;
	log(7, "Transferred %d bytes", totwritten);
	
	free(buffer);
	clntinfo->havetoconvert = CONV_NOTCONVERT;
	return error;
}

int checkforabort(struct clientinfo* clntinfo) {
	int clientfd = clntinfo->clientsocket;
	int serverfd = clntinfo->serversocket;
	char buffer[1024];
	char* buf =0;
	char rabseq[3] = { IAC, IP, 0 };
	char sabseq[4] = { IAC, IP, IAC, 0 };
	char dmseq[1] = { DM };
	int ret;

	struct timeval tm = { 300, 0 };
	fd_set set, exset;
	FD_ZERO(&set);
	FD_SET(clientfd, &set);
	
	/* this read won't block */
	ret = read(clientfd, buffer, sizeof(buffer) - 1);
	if (ret < 0) {
		return 0;
	}
	buffer[ret] = '\0';

	if (strncmp(buffer, rabseq, 2) != 0) {
		return 0;
	}

	FD_SET(clientfd, &set);
	ret = select(clientfd + 1, &set, NULL, &set, &tm);

	if (ret <= 0) {
		return 0;
	}

	ret = read(clientfd, buffer, sizeof(buffer) - 1);
	buffer[ret] = '\0';

	if(buffer[0] != (char) DM ||
		strncasecmp(buffer + 1,"ABOR",4) != 0) {
		return 0;
	}
		
	/* abort */

	if (clntinfo->dataclientsock == -1) {
		/* there is no open connection to the client */
		say(clientfd, "225 Abort successful\r\n");
	} else {
		/* Close the connection to the client */
		close(clntinfo->dataclientsock);
		clntinfo->dataclientsock = -1;
		say(clientfd, "426 Transfer aborted. "
				"Closed data connection.\r\n");
		say(clientfd, "226 Abort successful\r\n");
	}
	
	if (clntinfo->dataserversock == -1) {
		/* if we don't have a connection to the server we can
		 * quit here
		 */
		return 1;
	}
#ifdef SFTP_SUPPORT
	if (clntinfo->servermode == SECURE_SFTP) {
		/* just close the connection */
		cancelled = 1;
		do_cancel();
		close(clntinfo->dataserversock);
		return 1;
	}
#endif
	
	if (send(serverfd, sabseq, 3, MSG_OOB) != 3
		|| send(serverfd, dmseq, 1, 0) != 1) {
		log(3, "send() error: %s", strerror(errno));
	}
	say(serverfd, "ABOR\r\n");

	
	FD_ZERO(&set);
	FD_ZERO(&exset);
	FD_SET(serverfd, &set);
	FD_SET(serverfd, &exset);
	tm.tv_sec = 2;  /* grant the server 2 seconds to shut down */
	tm.tv_usec = 0;

	ret = select(serverfd + 1, &set, NULL, &exset, &tm);
	if (ret <= 0) {
		close(clntinfo->dataserversock);
		clntinfo->dataserversock = -1;
	}
	ret = ftp_getrc(serverfd, &buf);
	switch (ret) {
		case 425:
		case 426:
			ret = ftp_getrc(serverfd, 0);
			li.respcode = ret;
			/* fallthrough */
		case 226:
			log(6, "Client requested ABORT - Aborted successfully");
			break;
		default:
			log(6, "Answer of server after ABORT unknown: (%s).",
					buffer);
	}
	free(buf);
	return 1;
}

