#include "jftpgw.h"

extern int timeout;
static void saypasv(int, char*, unsigned short int);


/* pasvserver() handles nearly the whole passive connection, i.e., it
 * requests a port on the server, binds to one on the proxy host and sends
 * the address & port to the client
 * 
 * Parameters: clntinfo: connection information
 *
 * Return value: -1 on error,
 *                0 on success
 *
 * Called by: handlecmds (if the client issued a PORT or PASV command)
 *
 * */

int pasvserver(struct clientinfo* clntinfo) {
	int ss, cs, ret, pssock;
	char* brk;
	char* buffer;
	struct sockaddr_in pasvserv_sin;
	
	ss = clntinfo->serversocket;
	cs = clntinfo->clientsocket;
	
	say(ss, "PASV\r\n");
	buffer = ftp_readline(ss);
	if (!buffer) {
		if (timeout) {
			log(2, "Timeout in %s line %d\n", __FILE__ ,__LINE__);
			err_time_readline(cs);
		} else {
			err_readline(cs);
			log(2, "Error reading response to PASV from server: %s",
				strerror(errno));
		}
		return -1;
	}

	/* 
	 * An example of the answer would be:
	 * "227 Entering Passive Mode (127,0,0,1,7,123)"
	 */
	/* Look for the beginning of the IP */
	brk = (char*) strchr(buffer, '(');

	if (!checkdigits(buffer, 227) || !brk) {
		/* invalid return string */
		say(cs, buffer);
		log(2, "Strange answer after sending PASV: %s", buffer);
		free(buffer);
		return -1;
	}
	
	/* skip over the bracket sign */
	brk++;

	/* call parsesock to parse the answer string and get the values into
	 * PASVSERVERSOCK */
	ret = parsesock(brk, &pasvserv_sin, PASSIVE);
	if (ret) {
		say(cs, "500 Could not parse a valid address from the PASV response\r\n");
		log(2, "Could not parse a valid socket from the PASV "
		    " response: %s", buffer);
		free(buffer);
		return -1;
	}

	/* open the port on the foreign machine specified by PASVSERVERSOCK
	 * Use clntinfo->dataport as a port on our side. This is necessary
	 * if there is a packet filtering firewall between the proxy machine
	 * and the one we it connects to */
	pssock = openportiaddr(pasvserv_sin.sin_addr.s_addr,
			ntohs(pasvserv_sin.sin_port), clntinfo->dataport,
			clntinfo);
	if (pssock < 0) {
		say(cs, "500 Could not connect to the specified PASV host\r\n");
		log(3, "Could not connect to the specified PASV host: (%s)", buffer);
		free(buffer);
		return -1;
	}
	clntinfo->dataserversock = pssock;

	free(buffer);
	return 0;
}


/* pasvclient() opens a port on the proxy machine and tells the addr and
 * port to the client
 *
 * Parameters: clntinfo: connection information
 *
 * Return values: -1 on error
 *                 0 on success
 *
 * Called by: handlecmds (if the client issued a PORT or PASV command)
 */


int pasvclient(struct clientinfo* clntinfo) {
	int pcsock;
	int cs = clntinfo->clientsocket;
	struct sockaddr_in pasvclientsin;
	struct in_addr in;
	
	clntinfo->clientmode = PASSIVE;
	/* open a port on the interface the client is connected to and store
	 * the values in PASVCLIENTSIN */
	pcsock = openownport(&pasvclientsin, clntinfo, CLIENTADDR);
	if (pcsock < 0) {
		/*say(cs, "500 Error binding to a port\r\n");*/
		log(2, "Could not bind (%s)", strerror(errno));
		return -1;
	}
	clntinfo->dataclientsock = pcsock;
	clntinfo->waitforconnect = &clntinfo->dataclientsock;

	/* write the values to the client socket CS */
	in.s_addr = clntinfo->clientaddr;
	saypasv(cs, inet_ntoa(in), htons(pasvclientsin.sin_port));
	
	return 0;
}


/* saypasv() writes the "227 Entering Passive Mode ..." to the client
 * descriptor CS
 * 
 * Parameters: cs: client descriptor
 *             addr: Our address of the interface the client connected
 *                   through
 *             port: The corresponding port on our side
 *
 * Return value: void
 *
 * Called by: pasvclient()
 *
 * */ 

static void saypasv(int cs, char* addr, unsigned short int port) {
	unsigned short int lo, hi;
	char buffer[MAX_LINE_SIZE], *dot;
	
	hi = port % 256;
	lo = (port - hi) / 256;
	
	while ((dot = strchr(addr, '.'))) {
		*dot = ',';
	}
	
	sprintf(buffer, "227 Entering Passive Mode (%s,%d,%d)\r\n",
			addr, lo, hi);
	say(cs, buffer);
}


