#include <sys/time.h>
#include "jftpgw.h"

extern int timeout;
extern struct loginstruct lsforward;
extern struct loginfo li;
static int changecode(char *const, const char*);
#ifdef SFTP_SUPPORT
static int sftp_login(struct clientinfo*);
#endif
static int std_login(struct clientinfo*);

static void free_ls();


/* login handles the login to the server (the data is passed via CLNTINFO)
 *
 * Parameters: clntinfo: connection information
 * 
 * Return values: -1 for a system call failure (so that it can be
 *                   written via strerror to the client
 *                -2 for other errors
 *                 0 on success
 * Called by: handlecmds() after that version has read the username and
 *            maybe the password
 *
 *
 * Login response codes: 
 * 	Success:
 *		- forwards server response
 * 	Error:
 * 		- 530: Login incorrect
 * 		- 531: You are not allowed to connect to that host
 * 		- 532: Could not look up the destination host
 *		- 533: Error connecting to the server via ssh
 */



int login(struct clientinfo *clntinfo) {

	int cs = clntinfo->clientsocket;
	int allowed = 0;
	int ret;
	unsigned long int target;
	char* peer_ip_cln, *peer_ip_svr =0;
	struct hostent* host = 0;

	allowed = checkfwpass(clntinfo);
	if (!allowed) {
		say(cs, "530 Login incorrect\r\n");
		return -1;
	}	

	allowed = 0;

	/* connections to 255.255.255.255 are nonsense, so it must be an
	 * error */
	target = inet_addr(clntinfo->destination);
	peer_ip_cln = getpeer_ip(clntinfo->clientsocket);

	if (target == (unsigned long int) UINT_MAX) {
		/* clntinfo->destination may be a hostname */
		host = gethostbyname(clntinfo->destination);
		if (host) {
			char* destuser;
			if (strcmp(lsforward.fwuser, clntinfo->user) == 0) {
				/* this was not a forward */
				/* set the argument `destuser' for
				 * is_ip_target() to 0, so that user `john'
				 * is not able to log in, if only `>john' is
				 * specified (the > allows forwards only */
				destuser = 0;
			} else {
				destuser = clntinfo->user;
			}

			peer_ip_svr = strdup(gethostentip(host->h_addr));
			allowed = is_ip_target(peer_ip_cln, peer_ip_svr,
					lsforward.fwuser, destuser);
		} else {
			allowed = 0;
		}
	} else {
		char* destuser;
		peer_ip_svr = strdup(clntinfo->destination);
		if (strcmp(lsforward.fwuser, clntinfo->user) == 0) {
			/* this was not a forward */
			/* set the argument `destuser' for
			 * is_ip_target() to 0, so that user `john'
			 * is not able to log in, if only `>john' is
			 * specified (the > allows forwards only */
			destuser = 0;
		} else {
			destuser = clntinfo->user;
		}
		allowed = is_ip_target(peer_ip_cln,
				clntinfo->destination,
				lsforward.fwuser,
				destuser);
	}

	if (!allowed) {
		say(cs, "531 You are not allowed to connect to that host.\r\n");
		li.respcode = 531;
		/* free the login struct lsforward */
		free_ls();

		free(peer_ip_cln);
		free(peer_ip_svr);
		return -2;
	}

	switch(clntinfo->servermode) {
#ifdef SFTP_SUPPORT
		case SECURE_SFTP:
			ret = sftp_login(clntinfo);
			break;
#endif
		default:
			ret = std_login(clntinfo);
	}

	if (ret) {
		/* free the login struct lsforward */
		free_ls();
		
		free(peer_ip_cln);
		free(peer_ip_svr);
		return ret;
	}
	
	log(6, "Logged in to %s as %s!", clntinfo->destination, clntinfo->user);
	
		
	/* get the throughput rate */

	clntinfo->throughput = get_rate(lsforward.fwlogin, lsforward.login,
				inet_addr(peer_ip_cln),
				(char*) 0,  /* don't have a srcname */
				inet_addr(peer_ip_svr),
				clntinfo->destination);

	free(peer_ip_cln);
	free(peer_ip_svr);
	/* free the login struct lsforward */
	free_ls();
	return 0;
}

static
int std_login(struct clientinfo* clntinfo) {

	int port = clntinfo->destinationport;
	int cs = clntinfo->clientsocket;
	int ss = -1;
	int ret;
	char* peer_ip =0;
	char* buffer =0, *sendbuf =0;
	struct message welcomemsg = { 0, 0 };
	struct message authresp = { 0, 0 };
	char codebuf[5];


	ss = openportname(clntinfo->destination, port, 0, clntinfo);
	if (ss < 0) {
		int err = errno;
		sendbuf = (char*) malloc(
				strlen("Error connecting to %s: %s\r\n")
				+ strlen(clntinfo->destination)
				+ strlen(strerror(err))
				+ 1);
		enough_mem(sendbuf);
		sprintf(sendbuf, "Error connecting to %s: %s\r\n",
				clntinfo->destination, strerror(err));
		say(cs, sendbuf);
		free(sendbuf);
		return -2;
	}

	peer_ip = getpeer_ip(ss);
	clntinfo->serveraddr = get_showaddr(ss);
	free(peer_ip);
	clntinfo->serversocket = ss;

	/* Read the welcome line */
	welcomemsg = readall(ss);
	if (!welcomemsg.fullmsg) {
		if (timeout) {
			log(2, "Timeout in %s line %d\n", __FILE__ ,__LINE__);
			err_time_readline(cs);
		} else {
			err_readline(cs);
		}
		return -1;
	}

	log(9, "Connected to %s, got \"%s\" as welcome line", clntinfo->destination, welcomemsg.fullmsg);
	
	if (!checkbegin(welcomemsg.lastmsg, "220 ")) {
		log(1, "Not a valid FTP server response (%s)", welcomemsg.fullmsg);
		say(cs, welcomemsg.fullmsg);
		exit(1);
	}
	
	say(ss, "SYST\r\n");
	buffer = ftp_readline(ss);
	if (!buffer && timeout) {
		log(1, "Timeout in %s line %d\n", __FILE__ ,__LINE__);
		err_time_readline(cs);
		return -1;
	}
	if (!buffer) {
		err_readline(cs);
		return -1;
	}
	if (!strlen(buffer)) {
		log(1, "The server did not respond to the initial SYST command");
		free(buffer);
		return -1;
	}
	free(buffer);
	buffer = 0;
	
	sendbuf = (char*) malloc(strlen("USER \r\n")
			+ strlen(clntinfo->user) + 1);
	enough_mem(sendbuf);

	sprintf(sendbuf, "USER %s\r\n", clntinfo->user);
	ret = say(ss, sendbuf);
	free(sendbuf);
	sendbuf = 0;
	if (ret < 0) {
		log(2, "Error writing the user name to the server: %s",
			strerror(errno));
		return -1;
	}

	authresp = readall(ss);
	buffer = authresp.lastmsg;
	if (!buffer && timeout) {
			log(2, "Timeout in %s line %d\n", __FILE__ ,__LINE__);
		err_time_readline(cs);
		return -2;
	}
	if (!buffer) {
		err_readline(cs);
		return -1;
	}
	if (!checkdigits(buffer, 331) && !checkdigits(buffer, 230)) {
		say(cs, buffer);
		log(7, "Got \"%s\" after sending the username.", buffer);
		free(buffer);
		return -2;
	}
	
	if (!checkdigits(buffer, 230)) {
		sendbuf = (char*) malloc(strlen("PASS \r\n") 
					  + strlen(clntinfo->pass) + 1);
		enough_mem(sendbuf);
		sprintf(sendbuf, "PASS %s\r\n", clntinfo->pass);
		ret = say(ss, sendbuf);
		free(sendbuf);
		sendbuf = 0;
		if (ret < 0) {
			log(2, "Error writing the password to the server: %s",
				strerror(errno));
			return -1;
		}

		authresp = readall(ss);
		if (!authresp.fullmsg) {
			free(welcomemsg.fullmsg);
			if (timeout) {
				log(2, "Timeout in %s line %d\n", __FILE__ ,
						__LINE__); 
				err_time_readline(cs);
			}
			else {
				err_readline(cs);
			}
			return -1;
		}
	}
	free(buffer);

	buffer = (char*) malloc(strlen(welcomemsg.fullmsg)
			+ strlen(authresp.fullmsg) + 1);
	enough_mem(buffer);
	
	/* change the status code of the welcomemsg */
	
	/* save the digit code of the server response from the
	 * authentication in codebuf */
	strncpy(codebuf, authresp.fullmsg, 4);
	codebuf[3] = '-';
	codebuf[4] = '\0';
	
	/* Change the digit code of the welcome message to the one of the
	 * server response from the authentication */
	changecode(welcomemsg.fullmsg, codebuf);
	strcpy(buffer, welcomemsg.fullmsg);
	strcat(buffer, authresp.fullmsg);
	
	li.respcode = getcode(authresp.lastmsg);
	
	if (!checkdigits(authresp.lastmsg, 230)) {
		say(cs, authresp.lastmsg);
		log(8, "Got \"%s\" after sending the password", authresp.fullmsg);
		return -2;	
	}

	/* successful login */

	/* Now the client receives the welcome message as well as the
	 * response of the server from the authentication with the same code
	 * at the beginning */
	say(cs, buffer);

	free(welcomemsg.fullmsg);
	free(authresp.fullmsg);
	welcomemsg.fullmsg = welcomemsg.lastmsg 
		= authresp.fullmsg = authresp.lastmsg =0;
	free(buffer);
	buffer =0;


	return 0;
}


#ifdef SFTP_SUPPORT

static
int sftp_login(struct clientinfo* clntinfo) {

	int ret;
	char* args[6] = { "progname", "-l", clntinfo->user, "", "", "" };
	const char* crypt_pw;
	const char* ssh_options;
	int pw_match = 0;
	int nargs;
	struct in_addr in;
	
	if (!gethostaddr(clntinfo->destination,
				&in)) {
		say(clntinfo->clientsocket, "532 Could not look up the "
				"destination host\r\n");
		li.respcode = 532;
		return -2;
	}
	crypt_pw = get_sshpasswd(in.s_addr,
			clntinfo->user);
	if ( ! crypt_pw) {
		pw_match = 0;
		log(3, "No password specified for user %s",
				clntinfo->user);
	} else {
		pw_match = (cryptcmp(crypt_pw,
			clntinfo->pass) == 0);
	}
	if (!pw_match) {
		say(clntinfo->clientsocket, "530 Login incorrect.\r\n");
		li.respcode = 530;
		free(clntinfo->user);
		clntinfo->user =0;
		return -2;
	}

	ssh_options = get_sshoptions(in.s_addr, clntinfo->user);
	if (ssh_options) {
		args[3] = "-s";
		args[4] = strdup(ssh_options);
		args[5] = clntinfo->destination;
		enough_mem(args[5]);
		nargs = 6;
	} else {
		args[3] = clntinfo->destination;
		nargs = 4;
	}

	ret = sftp_open(args, nargs);
	if (nargs == 6) {
		free(args[4]);
	}
	if (ret) {
		log(2, "Error connecting to the sftp server");
		say(clntinfo->clientsocket, "533 Error connecting to the server via ssh\r\n"); 
		li.respcode = 533;
		return -2;
	}
	return 0;
}

#endif /* SFTP_SUPPORT */



/* changecode() changes the response code at the beginning of each line to a
 * new value. In fact, NEW is a _char*_, so you can put any data at the
 * beginning of each line, not just a number
 * 
 * Parameters: msg: the origianl message
 *             new: the portion that should be prepended.
 *
 * Return values: 0 on success
 *
 * Called by login() in order to display the original 220 welcome message
 *           after the login with a changed code.
 * 
 * */

static int changecode(char *const msg, const char* new) {
	char *nextline = msg;
	while (nextline) {
		if (nextline[0] >= '0' && nextline[0] <= '9') {
			/* we subtract 2 because of "\r\n". If nextline is
			 * "foo\r\n" we may at most substitute "bar" or
			 * another 3 character long string, since 
			 *      3       ==        5         - 2  */
			if (strlen(new) <= strlen(nextline) - 2) {
				strncpy(nextline, new, strlen(new));
			}
		}
		nextline = strstr(nextline, "\r\n");
		nextline += 2;
		if (!*nextline) { /* end */
			nextline = 0;
		}
	}
	return 0;
}

void free_ls(void) {
	if (lsforward.fwlogin) {
		free(lsforward.fwlogin);
	}
	if (lsforward.fwpass) {
		free(lsforward.fwpass);
	}
	/* fwuser is always used */
	free(lsforward.fwuser);
	if (lsforward.login) {
		free(lsforward.login);
	}
	if (lsforward.pass) {
		free(lsforward.pass);
	}
	lsforward.fwlogin = lsforward.fwpass = lsforward.fwuser
		= lsforward.login = lsforward.pass = 0;
}
