#include "jftpgw.h"
#include <signal.h>

/* reads a single line from fd and returns a pointer to a malloc()ed 
 * char array.
 *
 * Parameters: fd: The file descriptor to read from
 *
 * Return values: (char*) 0 on error, a pointer to a malloc()ed char array
 *                that contains the read data on success
 *
 * Called by: various functions
 */

extern int timeout;

char *ftp_readline(int fd) {
	const int MAXSIZE = 3;
	int n, ret, length = 0;
	int data_read = 0;
	char *temp, *buffer;
	fd_set readset;
	struct timeval rtimeout = { 300, 0 };
	sigset_t sigset, oldset;
	
	FD_ZERO(&readset);
	FD_SET(fd, &readset);

	timeout = 0;
	
	sigemptyset(&sigset);
	sigemptyset(&oldset);
	sigaddset(&sigset, SIGCHLD);
	ret = sigprocmask(SIG_BLOCK, &sigset, &oldset);
	if (ret < 0) {
		log(3, "sigprocmask() error: %s", strerror(errno));
	}
	if ((ret = select(fd + 1, &readset, NULL, NULL, &rtimeout)) <= 0) {
		if (ret == 0) {
			timeout = 1;
		}
		/* save the errno value from sigprocmask() */
		n = errno;
		ret = sigprocmask(SIG_UNBLOCK, &sigset, &oldset);
		if (ret < 0) {
			log(3, "sigprocmask() error releasing the blocked"
				" signals: %s", strerror(errno));
		}
		errno = n;
		return 0;
	}
	ret = sigprocmask(SIG_UNBLOCK, &sigset, &oldset);
	if (ret < 0) {
		log(3, "sigprocmask() error releasing the blocked"
			" signals: %s", strerror(errno));
	}
	
	if (fd < 0) {
		log(1, "ftp_readline: Not connected");
		return 0;
	}

	buffer = (char *)malloc(MAXSIZE);
	enough_mem(buffer);

	temp = buffer;

	while ((n = read(fd, temp, 1)) > 0) {
		data_read = 1;
		if (*temp == '\r') continue;
		if (*temp == '\n') break;
		if (*temp == '\0') break;
		length++;
		if ((length+1) % MAXSIZE == 0) {
			buffer = (char*) realloc(buffer, length + 1 + MAXSIZE);
			enough_mem(buffer);
			temp = buffer + length - 1;
		}
		temp++;
	}
	if (n < 0 || (n == 0 && data_read == 0)) {
		free(buffer);
		return 0;
	}
	buffer[length] = '\0';
	return buffer;
}


/* reads from fd until it has a "xxx data" response and returns the
 * response code xxx.
 * If **data != 0, *data is set to the address of the malloc()ed array that
 * contains the read data.
 *
 * Parameters: fd: The file descriptor to read from
 *             data: A pointer to a pointer that should be set to the address
 *                   of the read data.
 *
 * Return value: 0 on error
 *               the response code on success
 * Called by: checkforabort()
 *
 */

int ftp_getrc(int fd, char **data) {
	char *line;
	char *tmp =0;
	int response;

	while ((line = ftp_readline(fd))) {
		if (!*line) {
			log(1, "Big bad error.. empty string returned by ftp_readline");
			break;
		}
		if (line[0] < '0' || line[0] > '9') {
			free(line);
			continue;
		}
		if (line[3] == ' ') {
			tmp = (char*) malloc(strlen(line) + 1);
			enough_mem(tmp);
			strcpy(tmp, line);
			break;
		}
		/* free the line if we continue in the loop */
		free(line);
	}
	if (!line || !*line) return 0;
	sscanf(line, "%d ", &response);
	if (data) {
		*data = tmp;
	} else {
		free(tmp);
	}
	/* free the line that was not freed within the whlie loop because of
	 * a break statement */
	free(line);
	tmp =0;
	if (response >= 100)
		return response;
	return 0;
}

/* readall() reads from an fd and returnes the whole data in a structure
 * message.
 * 
 * Parameters: sourcefd: The file descriptor to read from
 * 
 * Return values: A struct message with
 *                  - both char* arrays in it set to NULL on error
 *                  - one char* array pointing to the whole message and the
 *                    other one pointing to the last line of it, starting with
 *                    the response code.
 * 
 * Called by: login() to get the whole welcome message and to analyze
 *            the authentication message
 *
 */

struct message readall(int sourcefd) {
	char* line =0, *buf =0, *tmp =0, *last =0, *linestart =0;
	struct message ret;
	
	while ((line = ftp_readline(sourcefd))) {
		if (!*line) {
			log(1, "Big bad error.. empty string returned by ftp_readline");
			free(line);
			break;
		}
		if (buf) {
			tmp = (char*) malloc(strlen(buf) + strlen(line) + 3);
			enough_mem(tmp);
			strcpy(tmp, buf);
			last = tmp + strlen(buf);
			free(buf);
			buf = 0;
		} else {
			tmp = (char*) malloc(strlen(line) + 3);
			enough_mem(tmp);
			tmp[0] = '\0';
			last = tmp;
		}
		linestart = tmp + strlen(tmp);
		strcat(tmp, line);
		strcat(tmp, "\r\n");
		buf = tmp;
		tmp =0;
		if (line[0] < '0' || line[0] > '9') {
			free(line);
			continue;
		}
		if (line[3] == ' ') {
			break;
		}
		free(line);
		line = (char*) 0;
	}
	if (!line) {
		free(buf);
		ret.fullmsg = (char*) 0;
		ret.lastmsg = (char*) 0;
		return ret;
	} else {
		free(line);
	}
	ret.fullmsg = buf;
	ret.lastmsg = last;
	return ret;
}

/* passall() reads from sourcefd until there is no data left to read and writes
 * everything to targetfd
 *
 * Parameters: sourcefd: The file descriptor to read from
 *             targetfd: The file descriptor to write to
 *
 * Return value: 0 on error
 *               a pointer that contains the malloc()ed char array with the
 *               passed data on success
 *
 * Called by: handlecmds() to pass the message sent after a QUIT
 *            passcmd() to pass the control connection when passing a command
 *            passcmd() to pass the control connection after having transmit
 *                      the file 
 */

char* passall(int sourcefd, int targetfd) {
	char* line =0, *sendbuf =0;
	
	while ((line = ftp_readline(sourcefd))) {
		if (!*line) {
			log(1, "Big bad error.. empty string returned by ftp_readline");
			break;
		}
		sendbuf = (char*) malloc(strlen(line) + 3);
		enough_mem(sendbuf);
		strcpy(sendbuf, line);
		free(line);
		strcat(sendbuf, "\r\n");
		say(targetfd, sendbuf);
		/*
		if (line[0] < '0' || line[0] > '9') {
			free(line);
			free(sendbuf);
			continue;
		}*/
		if (strlen(sendbuf) > 4 && sendbuf[3] == ' ') {
			/* the loop is left here */
			break;
		} else {
			free(sendbuf);
			sendbuf = 0;
		}
	}
	return sendbuf;
}

