#include <ctype.h>
#include <sys/stat.h>
#include "jftpgw.h"

extern int multithread;
extern int main_server_pid;
extern int chrooted;
extern struct limitstruct* limitbase;
extern const int tcp_proto;

int sig_chld_occurred;
int should_read_config;

/* bindport binds to the specified PORT on HOSTNAME (which may also be a
 * dot-notation IP and returns the socket descriptor
 *
 * Parameters: hostname & port: Where to bind
 *
 * Return value: The socket descriptor of the bound socket
 *
 * Called by: waitclient
 *
 * */

int bindport(const char *hostname, int port) {
	struct hostent *host = 0;
	unsigned long inetaddr =1;
	int shandle;
	int one = 1;
	struct sockaddr_in sin;

	memset((void*)&sin, 0, sizeof(sin));
	sin.sin_family = AF_INET;
	inetaddr = inet_addr(hostname);
	if (inetaddr == (unsigned long int) UINT_MAX) {
		/* HOSTNAME was probably a name since inet_addr returned
		 * an error.
		 * Look up the name to get the IP 
		 */
		host = gethostbyname(hostname);
		if (!host) {
			log(1, "Could not resolve %s: %s",
				hostname, strerror(errno));
			perror("Could not resolve the hostname");
			return -1;
		}
		memcpy((void*)&sin.sin_addr.s_addr, (void*)host->h_addr,
				host->h_length);
	}
	else {
		/* okay, HOSTNAME was a valid dot-notation IP */
		sin.sin_addr.s_addr = inetaddr;
	}
	sin.sin_port = htons(port);

	if (getuid() == 0 && geteuid() != 0) {
		/* become root again - use our function instead of plain
		 * setuid() for the logging */
		log(8, "Changing ID to root (socket(), bind(), pidfile)");
		changeid("root", UID);
	}
	shandle = socket(AF_INET, SOCK_STREAM, tcp_proto);
	if (shandle < 0) {
		log(1, "Error creating socket to bind: %s", strerror(errno));
		perror("Error creating socket to bind to");
		return -1;
	}
	if (setsockopt(shandle, SOL_SOCKET, SO_REUSEADDR,
				(void*) &one, sizeof(one)) < 0) {
		log(3, "Error setting socket to SO_REUSEADDR");
	}
	if (bind(shandle, (struct sockaddr *) &sin, sizeof(sin)) < 0) {
		log(1, "Error binding: %s", strerror(errno));
		perror("Error binding");
		return -1;
	}
	if (listen(shandle, 5) < 0) {
		log(1, "Error listening on the socket: %s", strerror(errno));
		perror("Error listening on a bound socket");
		return -1;
	}

	log(6, "Listening on %s, port %d", hostname, port);

	if (getuid() == 0) {
		log(8, "Changing id back (socket(), bind(), pidfile)");
		changeid(get_option("runasuser"), EUID);
	}

	return shandle;
}


/* waitclient() waits for a client to connect. It binds to the ports and
 * listens to them. If a connection comes it, jftpgw forks. The parent process
 * keeps on listening whereas the child process handles the connection
 *
 * Parameters: hostnames:      Where to bind to
 *             clntinfo:       connection information
 *
 * Return value: -1 on error (if error message has been sent to the client)
 *               -2 on error (if error message should be created by
 *                            strerror()
 *               socket handle: on success (by the child process
 *
 *               Note: The parent process never returns from this function,
 *                     it is terminated by a signal
 */


int waitclient(const char* hostnames, struct clientinfo* clntinfo) {
	char* peer_ip =0, *hostnames2;
	char* part =0, *portdel =0;
	int chldpid =0;
	const char* option = 0;
	int nfd =0;
	int ahandle, shandle, ret, offset, port, i, maxfd, found;
	struct sigaction sa;
	struct sockaddr_in sin;
	struct limitstruct* ls = 0;
	fd_set bindset, backupset;
#ifdef HAVE_SOCKLEN_T
	socklen_t size;
#else
	int size;
#endif
	size = sizeof(sin);
	if (multithread) {
		daemonize();
	}

	option = get_option("pidfile");
	if (option) {
		FILE* pidf = fopen(option, "w");
		if (pidf) {
			fprintf(pidf, "%ld\n", (long) getpid());
			fclose(pidf);
			/* if successful register function to remove the
			 * pidfile */
			atexit(removepidfile);
		} else {
			log(2, "Error creating pidfile %s", option);
		}
	}
	/* this has to be done for the daemonization. We do it now after
	 * the pidfile has been created */
	umask(0);

	FD_ZERO(&bindset);
	maxfd = 0;
	clntinfo->boundsocket_niface = 0;
	/* split up hostname */
	offset = 0;
	/* we have to make the string suitable for quotstrtok by appending a
	 * WHITESPACE character */
	hostnames2 = (char*) malloc(strlen(hostnames) + 2);
	enough_mem(hostnames2);
	strcpy(hostnames2, hostnames);
	strcat(hostnames2, " ");
	while ((part = quotstrtok(hostnames2, WHITESPACES, &offset))) {
		portdel = strchr(part, ':');
		*portdel = 0;
		errno = 0;
		port = strtol(portdel+1, (char**) 0, 10);
		if (errno) {
			log(4, "Invalid port specification: %s. "
			   "Using default value %d", portdel, DEFAULTBINDPORT);
			port = 2370;
		}
		portdel =0;

		log(9, "binding to %s, port %d", part, port);

		shandle = bindport(part, port);
		free(part);
		part =0;
		if (shandle < 0) {
			free(hostnames2);
			return -1;
		}
		FD_SET(shandle, &bindset);
		maxfd = MAX_VAL(shandle, maxfd);
		clntinfo->boundsocket_niface++;
	}
	free(hostnames2);
	hostnames2 = 0;
	clntinfo->boundsocket_list =
		(int*) malloc(sizeof(int) * clntinfo->boundsocket_niface);
	enough_mem(clntinfo->boundsocket_list);

	/* we have successfully bound */

	sa.sa_handler = childterm;
	sig_chld_occurred = 0;
	sigemptyset (&sa.sa_mask);
	sa.sa_flags = SA_RESTART;
	sigaction (SIGCHLD, &sa, 0);

	found = 0;
	i = 0;
	do {
		if (FD_ISSET(i, &bindset)) {
			clntinfo->boundsocket_list[found] = i;
			found++;
		}
		i++;
	} while (found < clntinfo->boundsocket_niface);

	if (clntinfo->boundsocket_niface == 0) {
		log(2, "No interfaces found to bind to");
		return -1;
	}

	/* Close stdin,stdout,stderr */
	for(i = 0; i <= 2; i++) {
		close(i);
	}
	main_server_pid = getpid();
	atexit(sayterminating);

	memcpy(&backupset, &bindset, sizeof(fd_set));
	while(1) {
		unsigned long tg;
		struct in_addr t_in;
		if (nfd == 0) {
			memcpy(&bindset, &backupset, sizeof(fd_set));
			while (1) {
				/* eternal select() */
				nfd = select(maxfd + 1, &bindset, 0, 0, 0);
				if (nfd > 0) {
					break;
				}
				if (errno == EINTR) {
					memcpy(&bindset, &backupset,
						sizeof(fd_set));
					if (sig_chld_occurred) {
						get_chld_pid();
					}
					if (should_read_config) {
						log(9, "should read config. Calling function");
						reread_config();
					}
					continue;
				}
				log(1, "select() failed: %s", strerror(errno));
				return -1;
			}
		}
		/* a descriptor is ready */
		i = 0;
		while (!FD_ISSET(i, &bindset)) {
			i++;
		}
		nfd--;
		shandle = i;
		while(1) {
			ahandle = accept(shandle, (struct sockaddr *) &sin,
					&size);
			if (ahandle < 0) {
				if (errno == EINTR) {
					continue;
				}
				log(1, "accept() failed: %s", strerror(errno));
				return -1;
			}
			break;
		}
		peer_ip = getpeer_ip(ahandle);
		log(7, "Connection from %s", peer_ip);
		tg = get_showaddr(ahandle);
		t_in.s_addr = tg;
		log(7, "Tried to connect to %s", inet_ntoa(t_in));
		ret = is_below_limit(inet_addr(peer_ip), &ls);
		if (!ret) {
			say(ahandle, "500 Too many connections from your IP, "
				"sorry\r\n");
			free(peer_ip);
			close(ahandle);
			continue;
		}
		if (multithread) {
			if ((chldpid = fork()) < 0) {
				log(1, "Error forking: %s", strerror(errno));
				free(peer_ip);
				close(ahandle);
				return -1;
			}
			if (chldpid > 0) {
				/* parent process */
				/* register the PID */
				register_pid(ls, chldpid);
				free(peer_ip);
				close(ahandle);
			}
		}
		if (!multithread || chldpid == 0) {
			const char* tmp = 0;

			/* The clients ignore the SIGHUP signal. Thus
			 * the user can issue a killall -HUP jftpgw
			 * and the master jftpgw process rereads its
			 * configuration file without affecting the
			 * child servers */

			sa.sa_handler = SIG_IGN;
			sigemptyset(&sa.sa_mask);
			sa.sa_flags = 0;
			sigaction(SIGHUP, &sa, 0);

			/* And they just reap the status of exited children
			 */
			sa.sa_handler = reap_chld_info;
			sigemptyset(&sa.sa_mask);
			sa.sa_flags = SA_RESTART;
			sigaction(SIGCHLD, &sa, 0);

			for (i = 0; i < clntinfo->boundsocket_niface; i++) {
				close(clntinfo->boundsocket_list[i]);
			}
			free(clntinfo->boundsocket_list);
			clntinfo->boundsocket_list = (int*) 0;
			free_limitst(limitbase);
			limitbase = 0;

			log(8, "forked to pid %d", getpid());
			if (!is_ip_allowed(peer_ip, 0)) {
				say(ahandle, "500 You are not allowed to "
					"connect. Goodbye.\r\n");
				log(5, "%s was not allowed to connect.",
					peer_ip);
				close(ahandle);
				free(peer_ip);
				return -2;
			}
			else {
				log(5, "%s is allowed to connect.", peer_ip);
			}

			clntinfo->clientsocket = ahandle;
			clntinfo->dataport = get_usesport(ahandle) - 1;
			clntinfo->clientaddr = get_showaddr(ahandle);
			if (getuid() == 0 && geteuid() != 0) {
				/* become root again to open the logfile(s) in
				 * the logdir(s) */
				log(8, "Changing ID to root (init_logdirs(), "
					"chroot())");
				changeid("root", UID);
			}

			option = get_option("changerootdir");
			if (option) {
				if (geteuid() == 0) {
					chdir(option);
					i = chroot(option);
					if (!i) {
						log(7, "Changed root directory "
							"to %s", option);
						chrooted = 1;
					} else {
						log(7, "Error changeing root "
						"directory to %s: %s",
						option, strerror(errno));
						chrooted = 0;
					}
					chdir("/");
				}
			}
			/* chroot()ed environment ? */
			if (!chrooted) {
				tmp = get_option("atstartchangerootdir");
				if (!tmp) {
					tmp = get_option("changerootdir");
				}
			} else {
				tmp = (char*) 0;
			}
			if (init_logdirs(tmp)) {
				say(ahandle, "500 Error opening a log file "
					"(see jftpgw log)\r\n");
				free(peer_ip);
				close(ahandle);
				return -1;
			}

			/* change the effective user id */
			if (getuid() == 0) {
				log(8, "Changing ID back (init_logdirs(), "
					"chroot())");
				if (clntinfo->dataport > 1024) {
					/* If we don't need root privileges
					 * any more, change the UID */
					changeid(get_option("runasuser"), UID);
				} else {
					changeid(get_option("runasuser"), EUID);
				}
			}
			if (get_option("welcomeline")) {
				const char* welcome = get_option("welcomeline");
				int i;
				char* welcmsg = (char*)
					malloc(strlen(welcome)+4+2+1);

				enough_mem(welcmsg);
				strcpy(welcmsg, "220 ");
				strcat(welcmsg, welcome);

				/* remove all special characters */
				for (i = 0; i < strlen(welcmsg); i++) {
					if (welcmsg[i] < 32) {
						/* a special character */
						welcmsg[i] = '_';
					}
				}
				log(9, "Saying this text as welcomeline: %s",
						welcmsg);
				strcat(welcmsg, "\r\n");
				say(ahandle, welcmsg);
				free(welcmsg);
			} else {
				say(ahandle, "220 Joe FTP Proxy Server/Gateway (v"JFTPGW_VERSION") ready\r\n");
			}
			free(peer_ip);
			peer_ip =0;
			return ahandle;
		}
	}
}
