/*
 * @(#)Statement.java	1.16 98/09/29
 * 
 * Copyright 1996-1998 by Sun Microsystems, Inc.,
 * 901 San Antonio Road, Palo Alto, California, 94303, U.S.A.
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of Sun Microsystems, Inc. (<code>Confidential Information<code>).  You
 * shall not disclose such Confidential Information and shall use
 * it only in accordance with the terms of the license agreement
 * you entered into with Sun.
 */

package java.sql;

/**
 * <P>The object used for executing a static SQL statement
 * and obtaining the results produced by it. 
 *
 * <P>Only one ResultSet per Statement can be open at any point in
 * time. Therefore, if the reading of one ResultSet is interleaved
 * with the reading of another, each must have been generated by
 * different Statements. All statement <code>execute</code> methods implicitly
 * close a statment's current ResultSet if an open one exists.
 *
 * @see Connection#createStatement
 * @see ResultSet 
 */
public interface Statement {

    /**
     * Executes a SQL statement that returns a single ResultSet.
     *
     * @param sql typically this is a static SQL SELECT statement
     * @return a ResultSet that contains the data produced by the
     * query; never null 
     * @exception SQLException if a database access error occurs
     */
    ResultSet executeQuery(String sql) throws SQLException;

    /**
     * Executes an SQL INSERT, UPDATE or DELETE statement. In addition,
     * SQL statements that return nothing, such as SQL DDL statements,
     * can be executed.
     *
     * @param sql a SQL INSERT, UPDATE or DELETE statement or a SQL
     * statement that returns nothing
     * @return either the row count for INSERT, UPDATE or DELETE or 0
     * for SQL statements that return nothing
     * @exception SQLException if a database access error occurs
     */
    int executeUpdate(String sql) throws SQLException;

    /**
	 * Releases this <code>Statement</code> object's database 
	 * and JDBC resources immediately instead of waiting for
	 * this to happen when it is automatically closed.
     * It is generally good practice to release resources as soon as
	 * you are finished with them to avoid tying up database
	 * resources.
     * <P><B>Note:</B> A Statement is automatically closed when it is
     * garbage collected. When a Statement is closed, its current
     * ResultSet, if one exists, is also closed.  
     *
     * @exception SQLException if a database access error occurs
     */
    void close() throws SQLException;

    //----------------------------------------------------------------------

    /**
     * Returns the maximum number of bytes allowed
     * for any column value. 
	 * This limit is the maximum number of bytes that can be
	 * returned for any column value.
	 * The limit applies only to BINARY,
     * VARBINARY, LONGVARBINARY, CHAR, VARCHAR, and LONGVARCHAR
     * columns.  If the limit is exceeded, the excess data is silently
     * discarded.
     *
     * @return the current max column size limit; zero means unlimited 
     * @exception SQLException if a database access error occurs
     */
    int getMaxFieldSize() throws SQLException;
    
    /**
	 * Sets the limit for the maximum number of bytes in a column to
	 * the given number of bytes.  This is the maximum number of bytes 
	 * that can be returned for any column value.  This limit applies
	 * only to BINARY, VARBINARY, LONGVARBINARY, CHAR, VARCHAR, and
     * LONGVARCHAR fields.  If the limit is exceeded, the excess data
     * is silently discarded. For maximum portability, use values
     * greater than 256.
     *
     * @param max the new max column size limit; zero means unlimited 
     * @exception SQLException if a database access error occurs
     */
    void setMaxFieldSize(int max) throws SQLException;

    /**
     * Retrieves the maximum number of rows that a
     * ResultSet can contain.  If the limit is exceeded, the excess
     * rows are silently dropped.
     *
     * @return the current max row limit; zero means unlimited
     * @exception SQLException if a database access error occurs
     */
    int getMaxRows() throws SQLException;

    /**
     * Sets the limit for the maximum number of rows that any
     * ResultSet can contain to the given number.
	 * If the limit is exceeded, the excess
     * rows are silently dropped.
     *
     * @param max the new max rows limit; zero means unlimited 
     * @exception SQLException if a database access error occurs
     */
    void setMaxRows(int max) throws SQLException;

    /**
	 * Sets escape processing on or off.
     * If escape scanning is on (the default), the driver will do
     * escape substitution before sending the SQL to the database.
     *
     * Note: Since prepared statements have usually been parsed prior
     * to making this call, disabling escape processing for prepared
     * statements will have no effect.
     *
     * @param enable true to enable; false to disable
     * @exception SQLException if a database access error occurs
     */
    void setEscapeProcessing(boolean enable) throws SQLException;

    /**
	 * Retrieves the number of seconds the driver will
     * wait for a Statement to execute. If the limit is exceeded, a
     * SQLException is thrown.
     *
     * @return the current query timeout limit in seconds; zero means unlimited 
     * @exception SQLException if a database access error occurs
     */
    int getQueryTimeout() throws SQLException;

    /**
     * Sets the number of seconds the driver will
     * wait for a Statement to execute to the given number of seconds.
	 * If the limit is exceeded, a SQLException is thrown.
     *
     * @param seconds the new query timeout limit in seconds; zero means 
     * unlimited 
     * @exception SQLException if a database access error occurs
     */
    void setQueryTimeout(int seconds) throws SQLException;

    /**
	 * Cancels this <code>Statement</code> object if both the DBMS and
	 * driver support aborting an SQL statement.
     * This method can be used by one thread to cancel a statement that
     * is being executed by another thread.
     *
     * @exception SQLException if a database access error occurs
     */
    void cancel() throws SQLException;

    /**
     * Retrieves the first warning reported by calls on this Statement.
     * Subsequent Statement warnings will be chained to this
     * SQLWarning.
     *
     * <p>The warning chain is automatically cleared each time
     * a statement is (re)executed.
     *
     * <P><B>Note:</B> If you are processing a ResultSet, any
     * warnings associated with ResultSet reads will be chained on the
     * ResultSet object.
     *
     * @return the first SQLWarning or null 
     * @exception SQLException if a database access error occurs
     */
    SQLWarning getWarnings() throws SQLException;

    /**
	 * Clears all the warnings reported on this <code>Statement</code>
	 * object. After a call to this method,
	 * the method <code>getWarnings</code> will return 
	 * null until a new warning is reported for this Statement.  
     *
     * @exception SQLException if a database access error occurs
     */
    void clearWarnings() throws SQLException;

    /**
     * Defines the SQL cursor name that will be used by
     * subsequent Statement <code>execute</code> methods. This name can then be
     * used in SQL positioned update/delete statements to identify the
     * current row in the ResultSet generated by this statement.  If
     * the database doesn't support positioned update/delete, this
     * method is a noop.  To insure that a cursor has the proper isolation
     * level to support updates, the cursor's SELECT statement should be
     * of the form 'select for update ...'. If the 'for update' phrase is 
     * omitted, positioned updates may fail.
     *
     * <P><B>Note:</B> By definition, positioned update/delete
     * execution must be done by a different Statement than the one
     * which generated the ResultSet being used for positioning. Also,
     * cursor names must be unique within a connection.
     *
     * @param name the new cursor name, which must be unique within
	 *             a connection
     * @exception SQLException if a database access error occurs
     */
    void setCursorName(String name) throws SQLException;
	
    //----------------------- Multiple Results --------------------------

    /**
     * Executes a SQL statement that may return multiple results.
     * Under some (uncommon) situations a single SQL statement may return
     * multiple result sets and/or update counts.  Normally you can ignore
     * this unless you are (1) executing a stored procedure that you know may
     * return multiple results or (2) you are dynamically executing an
     * unknown SQL string.  The  methods <code>execute</code>,
	 * <code>getMoreResults</code>, <code>getResultSet</code>,
     * and <code>getUpdateCount</code> let you navigate through multiple results.
     *
     * The <code>execute</code> method executes a SQL statement and indicates the
     * form of the first result.  You can then use getResultSet or
     * getUpdateCount to retrieve the result, and getMoreResults to
     * move to any subsequent result(s).
     *
     * @param sql any SQL statement
     * @return true if the next result is a ResultSet; false if it is
     * an update count or there are no more results
     * @exception SQLException if a database access error occurs
     * @see #getResultSet
     * @see #getUpdateCount
     * @see #getMoreResults 
     */
    boolean execute(String sql) throws SQLException;
	
    /**
     *  Returns the current result as a <code>ResultSet</code> object. 
     *  This method should be called only once per result.
     *
     * @return the current result as a ResultSet; null if the result
     * is an update count or there are no more results
     * @exception SQLException if a database access error occurs
     * @see #execute 
     */
    ResultSet getResultSet() throws SQLException; 

    /**
     *  Returns the current result as an update count;
     *  if the result is a ResultSet or there are no more results, -1
     *  is returned. 
     *  This method should be called only once per result.
     * 
     * @return the current result as an update count; -1 if it is a
     * ResultSet or there are no more results
     * @exception SQLException if a database access error occurs
     * @see #execute 
     */
    int getUpdateCount() throws SQLException; 

    /**
     * Moves to a Statement's next result.  It returns true if 
     * this result is a ResultSet.  This method also implicitly
     * closes any current ResultSet obtained with getResultSet.
     *
     * There are no more results when (!getMoreResults() &&
     * (getUpdateCount() == -1)
     *
     * @return true if the next result is a ResultSet; false if it is
     * an update count or there are no more results
     * @exception SQLException if a database access error occurs
     * @see #execute 
     */
    boolean getMoreResults() throws SQLException; 


    //--------------------------JDBC 2.0-----------------------------


    /**
     * JDBC 2.0
     *
     * Gives the driver a hint as to the direction in which
	 * the rows in a result set
     * will be processed. The hint applies only to result sets created 
     * using this Statement object.  The default value is 
     * ResultSet.FETCH_FORWARD.
     * <p>Note that this method sets the default fetch direction for 
	 * result sets generated by this <code>Statement</code> object.
	 * Each result set has its own methods for getting and setting
	 * its own fetch direction.
     * @param direction the initial direction for processing rows
     * @exception SQLException if a database access error occurs
	 * or the given direction
     * is not one of ResultSet.FETCH_FORWARD, ResultSet.FETCH_REVERSE, or
     * ResultSet.FETCH_UNKNOWN
     */
    void setFetchDirection(int direction) throws SQLException;

    /**
     * JDBC 2.0
     *
     * Retrieves the direction for fetching rows from
	 * database tables that is the default for result sets
	 * generated from this <code>Statement</code> object.
	 * If this <code>Statement</code> object has not set
	 * a fetch direction by calling the method <code>setFetchDirection</code>,
	 * the return value is implementation-specific.
     *
     * @return the default fetch direction for result sets generated
	 *          from this <code>Statement</code> object
     * @exception SQLException if a database access error occurs
     */
    int getFetchDirection() throws SQLException;

    /**
     * JDBC 2.0
     *
     * Gives the JDBC driver a hint as to the number of rows that should 
     * be fetched from the database when more rows are needed.  The number 
     * of rows specified affects only result sets created using this 
     * statement. If the value specified is zero, then the hint is ignored.
     * The default value is zero.
     *
     * @param rows the number of rows to fetch
     * @exception SQLException if a database access error occurs, or the
     * condition 0 <= rows <= this.getMaxRows() is not satisfied.
     */
    void setFetchSize(int rows) throws SQLException;
  
    /**
     * JDBC 2.0
     *
     * Retrieves the number of result set rows that is the default 
	 * fetch size for result sets
	 * generated from this <code>Statement</code> object.
	 * If this <code>Statement</code> object has not set
	 * a fetch size by calling the method <code>setFetchSize</code>,
	 * the return value is implementation-specific.
     * @return the default fetch size for result sets generated
	 *          from this <code>Statement</code> object
     * @exception SQLException if a database access error occurs
     */
    int getFetchSize() throws SQLException;

    /**
     * JDBC 2.0
     *
     * Retrieves the result set concurrency.
     */
    int getResultSetConcurrency() throws SQLException;

    /**
     * JDBC 2.0
     *
     * Determine the result set type.
     */
    int getResultSetType()  throws SQLException;

    /**
     * JDBC 2.0
     *
     * Adds a SQL command to the current batch of commmands for the statement.
     * This method is optional.
     *
     * @param sql typically this is a static SQL INSERT or UPDATE statement
     * @exception SQLException if a database access error occurs, or the
     * driver does not support batch statements
     */
    void addBatch( String sql ) throws SQLException;

    /**
     * JDBC 2.0
     *
     * Makes the set of commands in the current batch empty.
     * This method is optional.
     *
     * @exception SQLException if a database access error occurs or the
     * driver does not support batch statements
     */
    void clearBatch() throws SQLException;

    /**
     * JDBC 2.0
     * 
     * Submits a batch of commands to the database for execution.
     * This method is optional.
     *
     * @return an array of update counts containing one element for each
     * command in the batch.  The array is ordered according 
     * to the order in which commands were inserted into the batch.
     * @exception SQLException if a database access error occurs or the
     * driver does not support batch statements
     */
    int[] executeBatch() throws SQLException;

    /**
     * JDBC 2.0
     * 
     * Returns the <code>Connection</code> object
	 * that produced this <code>Statement</code> object.
	 * @return the connection that produced this statement
     * @exception SQLException if a database access error occurs
     */
    Connection getConnection()  throws SQLException;
}	
