/*
 * @(#)FileDialog.java	1.37 98/10/02
 *
 * Copyright 1995-1998 by Sun Microsystems, Inc.,
 * 901 San Antonio Road, Palo Alto, California, 94303, U.S.A.
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of Sun Microsystems, Inc. ("Confidential Information").  You
 * shall not disclose such Confidential Information and shall use
 * it only in accordance with the terms of the license agreement
 * you entered into with Sun.
 */
package java.awt;

import java.awt.peer.FileDialogPeer;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.ObjectInputStream;

/**
 * The <code>FileDialog</code> class displays a dialog window
 * from which the user can select a file.
 * <p>
 * Since it is a modal dialog, when the application calls
 * its <code>show</code> method to display the dialog,
 * it blocks the rest of the application until the user has
 * chosen a file.
 *
 * @see Window#show
 *
 * @version 	1.37, 10/02/98
 * @author 	Sami Shaio
 * @author 	Arthur van Hoff
 * @since       JDK1.0
 */
public class FileDialog extends Dialog {

    /**
     * This constant value indicates that the purpose of the file
     * dialog window is to locate a file from which to read.
     */
    public static final int LOAD = 0;

    /**
     * This constant value indicates that the purpose of the file
     * dialog window is to locate a file to which to write.
     */
    public static final int SAVE = 1;

    /*
     * There are 2 FileDialog Modes : <code>LOAD<code> and
     * <code>SAVE<code>.
     * This integer will represent one or the other.
     * If the mode is not specified it will default to LOAD.
     *
     * @serial
     * @see getMode()
     * @see setMode()
     * @see java.awt.FileDialog#LOAD
     * @see java.awt.FileDialog#SAVE
     */
    int mode;
    /*
     * The string specifying the directory to display
     * in the file dialog.
     * This variable may be null.
     *
     * @serial
     * @see getDirectory()
     * @see setDirectory()
     */
    String dir;
    /*
     * The string specifying the initial value of the
     * filename text field in the file dialog.
     * This variable may be null.
     *
     * @serial
     * @see getFile()
     * @see setFile()
     */
    String file;
    /*
     * The filter used as the file dialog's filename filter.
     * The file dialog will only be displaying files whose
     * names are accepted by this filter.
     * This variable may be null.
     *
     * @serial
     * @see getFilenameFIlter()
     * @see setFilenameFilter()
     * @see FileNameFilter
     */
    FilenameFilter filter;

    private static final String base = "filedlg";
    private static int nameCounter = 0;

    /*
     * JDK 1.1 serialVersionUID
     */
     private static final long serialVersionUID = 5035145889651310422L;


    static {
        /* ensure that the necessary native libraries are loaded */
	Toolkit.loadLibraries();
	initIDs();
    }

    /**
     * Initialize JNI field and method IDs for fields that may be
       accessed from C.
     */
    private static native void initIDs();

    /**
     * Creates a file dialog for loading a file.  The title of the
     * file dialog is initially empty.
     * @param parent the owner of the dialog
     * @since JDK1.1
     */
    public FileDialog(Frame parent) {
	this(parent, "", LOAD);
    }

    /**
     * Creates a file dialog window with the specified title for loading
     * a file. The files shown are those in the current directory.
     * @param     parent   the owner of the dialog.
     * @param     title    the title of the dialog.
     */
    public FileDialog(Frame parent, String title) {
	this(parent, title, LOAD);
    }

    /**
     * Creates a file dialog window with the specified title for loading
     * or saving a file.
     * <p>
     * If the value of <code>mode</code> is <code>LOAD</code>, then the
     * file dialog is finding a file to read. If the value of
     * <code>mode</code> is <code>SAVE</code>, the file dialog is finding
     * a place to write a file.
     * @param     parent   the owner of the dialog.
     * @param     title   the title of the dialog.
     * @param     mode   the mode of the dialog.
     * @see       java.awt.FileDialog#LOAD
     * @see       java.awt.FileDialog#SAVE
     */
    public FileDialog(Frame parent, String title, int mode) {
	super(parent, title, true);
	this.mode = mode;
	setLayout(null);
    }

    /**
     * Construct a name for this component.  Called by getName() when the
     * name is null.
     */
    String constructComponentName() {
        synchronized (getClass()) {
	    return base + nameCounter++;
	}
    }

    /**
     * Creates the file dialog's peer.  The peer allows us to change the look
     * of the file dialog without changing its functionality.
     */
    public void addNotify() {
        synchronized(getTreeLock()) {
	    if (peer == null)
	        peer = getToolkit().createFileDialog(this);
	    super.addNotify();
	}
    }

    /**
     * Indicates whether this file dialog box is for loading from a file
     * or for saving to a file.
     * @return   the mode of this file dialog window, either
     *               <code>FileDialog.LOAD</code> or
     *               <code>FileDialog.SAVE</code>.
     * @see      java.awt.FileDialog#LOAD
     * @see      java.awt.FileDialog#SAVE
     * @see      java.awt.FileDialog#setMode
     */
    public int getMode() {
	return mode;
    }

    /**
     * Sets the mode of the file dialog.
     * @param      mode  the mode for this file dialog, either
     *                 <code>FileDialog.LOAD</code> or
     *                 <code>FileDialog.SAVE</code>.
     * @see        java.awt.FileDialog#LOAD
     * @see        java.awt.FileDialog#SAVE
     * @see        java.awt.FileDialog#getMode
     * @exception  IllegalArgumentException if an illegal file
     *                 dialog mode is used.
     * @since      JDK1.1
     */
    public void setMode(int mode) {
	switch (mode) {
	  case LOAD:
	  case SAVE:
	    this.mode = mode;
	    break;
	  default:
	    throw new IllegalArgumentException("illegal file dialog mode");
	}
    }

    /**
     * Gets the directory of this file dialog.
     * @return    the (potentially null or invalid) directory of this
     *            FileDialog.
     * @see       java.awt.FileDialog#setDirectory
     */
    public String getDirectory() {
	return dir;
    }

    /**
     * Sets the directory of this file dialog window to be the
     * specified directory. Specifying a <code>null</code> or an
     * invalid directory implies an implementation-defined default.
     * This default will not be realized, however, until the user
     * has selected a file. Until this point, <code>getDirectory()</code>
     * will return the value passed into this method.<p>
     * Specifying "" as the directory is exactly equivalent to
     * specifying <code>null</code> as the directory.
     * @param     dir   the specific directory.
     * @see       java.awt.FileDialog#getDirectory
     */
    public void setDirectory(String dir) {
        this.dir = (dir != null && dir.equals("")) ? null : dir;
	FileDialogPeer peer = (FileDialogPeer)this.peer;
	if (peer != null) {
	    peer.setDirectory(this.dir);
	}
    }

    // NOTE: This method may be called by privileged threads.
    //       This functionality is implemented in a package-private method 
    //       to insure that it cannot be overridden by client subclasses. 
    //       DO NOT INVOKE CLIENT CODE ON THIS THREAD!
    final void setDirectory_NoClientCode(String dir) {
        this.dir = (dir != null && dir.equals("")) ? null : dir;
    }

    /**
     * Gets the selected file of this file dialog.
     * @return    the currently selected file of this file dialog window,
     *                or <code>null</code> if none is selected.
     * @see       java.awt.FileDialog#setFile
     */
    public String getFile() {
	return file;
    }

    /**
     * Sets the selected file for this file dialog window to be the
     * specified file. This file becomes the default file if it is set
     * before the file dialog window is first shown.<p> Specifying "" as
     * the file is exactly equivalent to specifying <code>null</code>
     * as the file.
     * @param    file   the file being set.
     * @see      java.awt.FileDialog#getFile
     */
    public void setFile(String file) {
        this.file = (file != null && file.equals("")) ? null : file;
	FileDialogPeer peer = (FileDialogPeer)this.peer;
	if (peer != null) {
	    peer.setFile(this.file);
	}
    }

    // NOTE: This method may be called by privileged threads.
    //       This functionality is implemented in a package-private method 
    //       to insure that it cannot be overridden by client subclasses. 
    //       DO NOT INVOKE CLIENT CODE ON THIS THREAD!
    final void setFile_NoClientCode(String file) {
        this.file = (file != null && file.equals("")) ? null : file;
    }

    /**
     * Determines this file dialog's filename filter. A filename filter
     * allows the user to specify which files appear in the file dialog
     * window.
     * @return    this file dialog's filename filter.
     * @see       java.io.FilenameFilter
     * @see       java.awt.FileDialog#setFilenameFilter
     */
    public FilenameFilter getFilenameFilter() {
	return filter;
    }

    /**
     * Sets the filename filter for this file dialog window to the
     * specified filter.
     * @param   filter   the specified filter.
     * @see     java.io.FilenameFilter
     * @see     java.awt.FileDialog#getFilenameFilter
     */
    public synchronized void setFilenameFilter(FilenameFilter filter) {
	this.filter = filter;
	FileDialogPeer peer = (FileDialogPeer)this.peer;
	if (peer != null) {
	    peer.setFilenameFilter(filter);
	}
    }

    private void readObject(ObjectInputStream s)
        throws ClassNotFoundException, IOException
    {
        s.defaultReadObject();

	// 1.1 Compatibility: "" is not converted to null in 1.1
	if (dir != null && dir.equals("")) {
	    dir = null;
	}
	if (file != null && file.equals("")) {
	    file = null;
	}
    }

    /**
     * Returns the parameter string representing the state of this file
     * dialog window. This string is useful for debugging.
     * @return  the parameter string of this file dialog window.
     */
    protected String paramString() {
	String str = super.paramString();
	str += ",dir= " + dir;
	str += ",file= " + file;
	return str + ((mode == LOAD) ? ",load" : ",save");
    }

    boolean postsOldMouseEvents() {
        return false;
    }
}
