#!/usr/bin/env perl
#	$Id: JdayTest.pl,v 1.3 2005/10/06 18:32:03 hiram Exp $
#
#	JdayTest.pl - test the Jday.pl and J2D.pl functions
#

use warnings;
use strict;
use CalDate;	# subroutines: JulDate, CalDate, FormatCalDate

######################################################################
#
#	TestJ2D - test a conversion from julian date to calendar date
#	Input: Arg0 is the julian decimal date: nnnnn.nnnnnn
#		Arg1 is the expected result: "YYYY/MM/DD HH:MM:SS"
#
sub TestJ2D() {
	my ($jd, $caldate) = @_;
	my %lcd = ( J_DATE => $$jd );
	&CalDate(\%lcd);
	my $D=&FormatCalDate(\%lcd);
	printf("TestJ2D: %.6f =? %s ", $$jd, $$caldate);
	if ( $D eq $$caldate ) {
		print "\t- OK\n";
		return 1;
	} else {
		print "not = $D - ERROR\n";
		return 0;
	}
}
######################################################################
#
#	TestJD - test a conversion from calendar to julian date
#	Input: Arg0 is the string: "YYYY MM DD HH MM SS"
#		Arg1 is the expected answer: "d*.dddddd"
#
sub TestJD() {
	my ($caldate, $jd) = @_;
	print "TestJD:  ";
	my @caldata = split(" ", $$caldate);
	print "@caldata", " =? ", sprintf("%.6f", $$jd);
	my %lcd = ( J_DATE => 0.0 );
	@lcd{'YEAR', 'MONTH', 'DAY', 'HOUR', 'MINUTE', 'SECOND'} = @caldata;
	my $tjd = sprintf("%.6f", &JulDate(\%lcd) );
	if ( $tjd = $$jd ) {
		print "\t- OK\n";
		return 1;
	} else {
		print " not = $tjd - ERROR\n";
		return 0;
	}
}

my $Success = 0;
my $TestCount = 0;

my $CD="-4712 1 1 12 0 0";
my $JD="0.000000";
$Success += &TestJD( \$CD, \$JD );
$TestCount++;

$JD="0.000000";
$CD="-4712-01-01 12:00:00";
$Success += &TestJ2D( \$JD, \$CD );
$TestCount++;

$CD="-1 12 31 12 0 0";
$JD="1721057.000000";
$Success += &TestJD( \$CD, \$JD );
$TestCount++;

$JD="1721057.000000";
$CD="-1-12-31 12:00:00";
$Success += &TestJ2D( \$JD, \$CD );
$TestCount++;

$CD="0 1 1 12 0 0";
$JD="1721058.000000";
$Success += &TestJD( \$CD, \$JD );
$TestCount++;

$JD="1721058.000000";
$CD="0-01-01 12:00:00";
$Success += &TestJ2D( \$JD, \$CD );
$TestCount++;

$CD="0 2 29 12 0 0";
$JD="1721117.000000";
$Success += &TestJD( \$CD, \$JD );
$TestCount++;

$JD="1721117.000000";
$CD="0-02-29 12:00:00";
$Success += &TestJ2D( \$JD, \$CD );
$TestCount++;

$CD="0 3 1 12 0 0";
$JD="1721118.000000";
$Success += &TestJD( \$CD, \$JD );
$TestCount++;

$JD="1721118.000000";
$CD="0-03-01 12:00:00";
$Success += &TestJ2D( \$JD, \$CD );
$TestCount++;

$CD="0 12 31 12 0 0";
$JD="1721423.000000";
$Success += &TestJD( \$CD, \$JD );
$TestCount++;

$JD="1721423.000000";
$CD="0-12-31 12:00:00";
$Success += &TestJ2D( \$JD, \$CD );
$TestCount++;

$CD="1 1 1 12 0 0";
$JD="1721424.000000";
$Success += &TestJD( \$CD, \$JD );
$TestCount++;

$JD="1721424.000000";
$CD="1-01-01 12:00:00";
$Success += &TestJ2D( \$JD, \$CD );
$TestCount++;

$CD="1970 1 1 0 0 0";
$JD="2440587.500000";
$Success += &TestJD( \$CD, \$JD );
$TestCount++;

$JD="2440587.500000";
$CD="1970-01-01 00:00:00";
$Success += &TestJ2D( \$JD, \$CD );
$TestCount++;

$CD="2000 8 18 05 25 27";
$JD="2451774.726007";
$Success += &TestJD( \$CD, \$JD );
$TestCount++;

$JD="2451774.726007";
$CD="2000-08-18 05:25:27";
$Success += &TestJ2D( \$JD, \$CD );
$TestCount++;

$CD="1582 10 4 23 59 59";
$JD="2299160.499988";
$Success += &TestJD( \$CD, \$JD );
$TestCount++;

$CD="1582 10 15 0 0 0";
$JD="2299160.500000";
$Success += &TestJD( \$CD, \$JD );
$TestCount++;

$JD = "2299160.499988";
$CD = "1582-10-04 23:59:59";
$Success += &TestJ2D( \$JD, \$CD );
$TestCount++;

$JD = "2299160.500000";
$CD = "1582-10-15 00:00:00";
$Success += &TestJ2D( \$JD, \$CD );
$TestCount++;

printf("TEST COMPLETE: Test count: %d, Test success: %d, Test fail: %d\n", $TestCount, $Success, $TestCount - $Success);

if ( 0 != ($TestCount - $Success)) {
    exit 255;
} else {
    exit 0;
}
