/*-
 * Copyright (c) 2001, 2002 Lev Walkin <vlm@lionet.info>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: rsh.c,v 1.17 2004/03/23 17:43:42 vlm Exp $
 */

#include "ipcad.h"
#include "storage.h"
#include "cfgvar.h"
#include "servers.h"
#include "opt.h"

/*
 * RSH server body
 */
extern int process_rsh_request(int sock);

int rsh_rw_timeout;	/* RSH read-write timeout */

static void *prr_handler();
static int fork_rsh_request(int client);
static int set_rsh_client_socket_options(int client);

void *
rsh_server(void *srvp) {
	struct sockaddr_in sin;
	struct sockaddr_in srem;
	server *srv = (server *)srvp;
	size_t	addrlen;
	int lcmd = 0;	/* Listening socket at "shell" (514) */
	int client;
	socklen_t on = 1;
	sigset_t set, oset;

	if((lcmd = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP)) == -1) {
		fprintf(stderr, "Can't create socket for %s.\n",
			srv->name);
		return NULL;
	}

	setsockopt(lcmd, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on));

	memset(&sin, 0, sizeof(sin));

	sin = srv->addr;

	if( bind(lcmd, (struct sockaddr *)&sin, sizeof(sin)) == -1) {
		fprintf(stderr, "Can't bind socket for %s.\n",
			srv->name);
		return NULL;
	}

	if( listen(lcmd, 5) == -1 ) {
		fprintf(stderr, "Can't set %s socket to listen state.\n",
			srv->name);
		return NULL;
	}

	/* Flag for parent thread */
	srv->started_ok = 1;

	sigemptyset(&set);
	sigaddset(&set, SIGALRM);

	while(1) {

		if(signoff_now)
			break;

		/* Unblock SIGALRM */
		sigprocmask(SIG_UNBLOCK, &set, &oset);

		addrlen = sizeof(srem);
		client = accept(lcmd, (struct sockaddr *)&srem, &addrlen);

		/* Block SIGARLM */
		sigprocmask(SIG_SETMASK, &oset, NULL);

		if(signoff_now)
			break;

		if(client == -1)
			continue;

		/*
		 * Do a couple of sanity checks.
		 */
		if(addrlen != sizeof(srem)) {
			close(client);
			continue;
		}

		if( ntohs(srem.sin_port) > 1023 ) {
			write(client, "Permission denied.\n",
				sizeof("Permission denied.\n"));
			close(client);
			continue;
		}

		/*
		 * Establish proper socket options for this client.
		 */
		if(set_rsh_client_socket_options(client)) {
			close(client);
			continue;
		}

		fork_rsh_request(client);

	}

	/* Impossible thing */
	return NULL;
}

int
fork_rsh_request(int client) {
	pthread_t thid;
	void *argp;

	if(max_clients <= 0) {
		write(client, "Too many clients.\n",
			sizeof("Too many clients.\n"));
		close(client);
	}

	argp = malloc(sizeof(client));
	if(!argp) {
		close(client);
		return -1;
	}
	memcpy(argp, &client, sizeof(client));

	if( pthread_create(&thid, &thread_attr_detach, prr_handler, argp) ) {
		free(argp);
		close(client);
		return -1;
	}

	return 0;
}

static void *
prr_handler(void *argp) {
	int client = *(int *)argp;

	free(argp);

	pthread_mutex_lock(&max_clients_lock);
	max_clients--;
	pthread_mutex_unlock(&max_clients_lock);

	process_rsh_request(client);

	pthread_mutex_lock(&max_clients_lock);
	max_clients++;
	pthread_mutex_unlock(&max_clients_lock);

	return NULL;
}


static int
set_rsh_client_socket_options(int client) {
	struct linger lng = { 1, 5 }; 
	struct timeval tv = { rsh_rw_timeout / 1000, 0 };
	long l;

	/* Set lingering on close */
	if(setsockopt(client, SOL_SOCKET, SO_LINGER, &lng, sizeof(lng)))
		return -1;

	/* Helps cleaning sockets in TIME_WAIT state on some systems */
	l = 1;
	(void)setsockopt(client, SOL_SOCKET, SO_REUSEADDR, &l, sizeof(l));

	/* Set receive and send timeouts */
	/* Unsupported by the Linux on which I've tested it, so ignore errors. */
#ifdef	SO_RCVTIMEO
	setsockopt(client, SOL_SOCKET, SO_RCVTIMEO, &tv, sizeof(tv));
#endif
#ifdef	SO_SNDTIMEO
	setsockopt(client, SOL_SOCKET, SO_SNDTIMEO, &tv, sizeof(tv));
#endif

#ifdef	SO_KEEPALIVE
	/* Set keepalive against broken connections */
	l = 1;
	setsockopt(client, SOL_SOCKET, SO_KEEPALIVE, &l, sizeof(l));
#endif	/* SO_KEEPALIVE */

#ifdef	IPPROTO_IP

#ifndef	__OpenBSD__
#ifdef	TCP_NODELAY
	/* Disable Nagle algorithm for efficiency */
	l = 1;
	setsockopt(client, IPPROTO_IP, TCP_NODELAY, &l, sizeof(l));
#endif	/* TCP_NODELAY */
#endif	/* __OpenBSD__ */

#ifdef	IP_TTL
	/* Set TTL reasonably low to avoid remote attacks */
	l = conf->rsh_ttl;
	setsockopt(client, IPPROTO_IP, IP_TTL, &l, sizeof(l));
#endif	/* IP_TTL */

#endif	/* IPPROTO_IP */

	return 0;
}

