/*-
 * Copyright (c) 2001, 2002 Lev Walkin <vlm@lionet.info>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: dump.c,v 1.12 2004/03/06 05:29:15 vlm Exp $
 */

#include "ipcad.h"
#include "cfgvar.h"
#include "opt.h"
#include "service.h"
#include "storage.h"
#include "disp.h"

/*
 * This function attempts to dump the main storage contents into the file.
 * It could be done using traditional save-to-temp-AND-rename approach,
 * but done otherwise. This allows a possibility of saving to a dedicated
 * file which appropriate permissions, even if its parent directory does not
 * have such.
 */
int
make_dump(char *dump_file, FILE *msgstream) {
	int df;
	struct stat sb;
	FILE *f;

	if(!dump_file
	  || (!conf->chroot_to && *dump_file != '/')
		/*
		 * Naive protection against unsafe paths.
		 */
	  || strstr(dump_file, "/..")
	  || !strncmp(dump_file, "/etc/", 5)
	  || !strncmp(dump_file, "/dev/", 5)
	) {
		fprintf(msgstream, "Incomplete (incorrect) pathname\n");
		return -1;
	}

	df = open(dump_file, O_CREAT | O_WRONLY, 0600);
	if(df == -1) {
		fprintf(msgstream, "Can't open dump file %s.\n", dump_file);
		return -1;
	}

	/* If file isn't locked by open(2) call, lock it manually. */
	if( try_lock(df, 1) ) {
		fprintf(msgstream, "Can't lock dump file %s.\n", dump_file);
		close(df);
		return -1;
	}

	if(fstat(df, &sb) != 0) {
		fprintf(msgstream, "Can't fstat(%s).\n", dump_file);
		close(df);
		return -1;
	}

	if((sb.st_mode & S_IFMT) != S_IFREG) {
		fprintf(msgstream, "%s: Not a regular file.\n", dump_file);
		close(df);
		return -1;
	}

	if(sb.st_uid != getuid()) {
		fprintf(msgstream, "Invalid dump file owner.\n");
		close(df);
		return -1;
	}

	if((sb.st_mode & 0077)) {
		fprintf(msgstream, "Insecure dump file permissions.\n");
		close(df);
		return -1;
	}

	if( ftruncate(df, 0) != 0 ) {
		fprintf(msgstream, "Can't truncate file.\n");
		close(df);
		return -1;
	}

	if(!(f = fdopen(df, "w"))) {
		fprintf(msgstream, "Can't fdopen().\n");
		close(df);
		return -1;
	}

	if(display(f, 0)) {
		fprintf(msgstream, "Empty table saved to %s.\n",
			dump_file);
	} else {
		fprintf(msgstream, "Main storage table saved to %s\n",
			dump_file);
	}

	fflush(f);
	fclose(f);

	return 0;
}


