/*-
 * Copyright (c) 2001, 2002, 2003 Lev Walkin <vlm@lionet.info>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: cfgvar.c,v 1.29 2004/03/19 08:12:30 vlm Exp $
 */

#include "ipcad.h"
#include "sf_lite.h"
#include "cfgvar.h"
#include "disp.h"


config_t static_config;
config_t *conf = &static_config;

/*
 * Run-time global variables
 */

int
cfg_add_iface(char *iface, int iflags, char *filter) {
	packet_source_t *ps;
	int ret;

	fprintf(stderr, "Opening %s... ", iface);

	if(conf->capture_ports)
		iflags |= IFLAG_RSH_EXTRA | IFLAG_LARGE_CAP;
	if(!(iflags & IFLAG_NF_DISABLE))
		iflags |= IFLAG_LARGE_CAP;

	if((iflags & IFLAG_LARGE_CAP)) fprintf(stderr, "[LCap] ");
	if((iflags & IFLAG_RSH_EXTRA)) fprintf(stderr, "[ERSH] ");
	if((iflags & IFLAG_NF_DISABLE)) fprintf(stderr, "[!NF] ");

	/*
	 * Create a structure describing the packet source.
	 */
	ps = create_packet_source(iface, iflags, filter);
	if(ps == NULL) {
		perror("Can't initialize");
		exit(EX_NOINPUT);
	}

	/*
	 * Initialize the packet source.
	 */
	ret = init_packet_source(ps);
	if(ret == -1) {
		switch(errno) {
		case ENETDOWN:
			/*
			 * Use this device even if it is DOWN now.
			 */
			fprintf(stderr, "[DOWN, yet available] ");
			break;
		case ENODEV:
		case ENXIO:
			/*
			 * Use this device even if it is not configured now.
			 */
			fprintf(stderr, "[NODEV, yet configured] ");
			break;
		default:
			perror("Can't initialize");
			exit(EX_NOINPUT);
		}
		assert(ps->state == PST_EMBRYONIC);
	}

	/*
	 * Add the packet source into the list.
	 */
	ps->next = conf->packet_sources_head;
	conf->packet_sources_head = ps;

	fprintf(stderr, "Initialized as %d\n", ps->ifIndex);

	return 0;
}


struct rsh_entries {
	char *username;
	unsigned int addr;
	int privlevel;
	struct rsh_entries *next;
};

static struct rsh_entries *rshes = NULL;

int
cfg_add_rsh_host(char *ru, char *rh, int privlevel) {
	unsigned int ia;
	struct rsh_entries *res;
	
	if(!ru || !rh) {
		fprintf(stderr, "Invalid [user@]host_addr specification: %s\n", rh);
		exit(EX_DATAERR);
	}

	if( sf_iaton(rh, &ia) == -1 ) {
		perror("host_addr:");
		fprintf(stderr, "Invalid [user@]host_addr specification: %s", rh);
		exit(EX_DATAERR);
	}

	if((res = rshes)) {
		while(res->next)
			res = res->next;
		res = res->next = (struct rsh_entries *)malloc(sizeof(*res));
	} else {
		res = rshes = (struct rsh_entries *)malloc(sizeof(*res));
	}

	if(!res) {
		fprintf(stderr, "Memory allocation failed.\n");
		exit(EX_OSERR);
	}

	res->username = strdup(ru);
	res->addr = ia;
	res->privlevel = privlevel;
	res->next = NULL;

	if(!res->username) {
		fprintf(stderr, "Memory allocation failed.\n");
		exit(EX_OSERR);
	}

	return 0;
}

/* 0: None, 1: view only, 2: default, 3: admin  */
int
cfg_check_rsh(char *ru, struct in_addr *ia) {
	struct rsh_entries *res;

	if(!rshes)	/* Allow world readable mode if no rules defined. */
		return 1;

	for(res = rshes; res; res=res->next) {
		if(res->addr != ia->s_addr)
			continue;
		if(!*res->username)
			return res->privlevel;
		if(strcmp(res->username, ru) == 0)
			return res->privlevel;
	}

	return 0;
}

int
cfg_add_aggregate_ports_table(int from, int to, int into) {
	struct ap_table *apt;

	if((apt = conf->ap_table) == NULL) {
		apt = conf->ap_table = malloc(sizeof *apt);
	} else {
		while(apt->next)
			apt = apt->next;
		apt = apt->next = malloc(sizeof *apt);
	}

	if(apt == NULL) {
		fprintf(stderr, "Memory allocation failed.\n");
		exit(EX_OSERR);
	}


	if(to < from) {
		int tmp = from;
		from = to;
		to = tmp;
	}

	apt->port_start = from;
	apt->port_end = to;
	apt->port_into = into;
	apt->next = NULL;

	printf("Aggregate ports %d-%d into %d\n",
		from, to, into);

	return 0;
}


int
cfg_add_atable2(unsigned int ip, unsigned int mask, unsigned int strip) {
	struct atable *at;

	if(!(at = conf->atable)) {
		at = conf->atable = malloc(sizeof *at);
	} else {
		while(at->next)
			at = at->next;
		at = at->next = malloc(sizeof *at);
	}

	if(!at) {
		fprintf(stderr, "Memory allocation failed.\n");
		exit(EX_OSERR);
	}

	at->ip.s_addr = ip;
	at->mask.s_addr = mask;
	at->strip.s_addr = strip;
	at->next = NULL;

	if(strip == (unsigned int)-1) {
		at->strip_bits = 32;
	} else {
		u_int32_t __n = strip;
		__n = (__n & 0x55555555) + ((__n & 0xaaaaaaaa) >> 1);
		__n = (__n & 0x33333333) + ((__n & 0xcccccccc) >> 2);
		__n = (__n & 0x0f0f0f0f) + ((__n & 0xf0f0f0f0) >> 4);
		__n = (__n & 0x00ff00ff) + ((__n & 0xff00ff00) >> 8);
		__n = (__n & 0x0000ffff) + ((__n & 0xffff0000) >> 16);
		at->strip_bits = __n;
	}

	printf("Aggregate network ");
	print_ip(stdout, at->ip);
	printf("/");
	print_ip(stdout, at->mask);
	printf(" -> ");
	print_ip(stdout, at->strip);
	printf("\n");

	return 0;
}

int
cfg_add_atable(char *sip, char *smask, char *sstrip) {
	unsigned int ip;
	unsigned int mask;
	unsigned int strip;
	unsigned long t;

	t = strchr(sstrip, '.')?1:0;
	if(t && !sf_iaton(sstrip, &strip)) return -1;
	if(!t) {
		errno=0;
		t = strtoul(sstrip, NULL, 10);
		if((t > 32) || errno) return -1;
		if(!t) strip = 0;
		else strip = htonl(0xffffffff << (32 - t));
	}

	if(!sf_iaton(sip, &ip))
		return -1;

	t = strchr(smask, '.')?1:0;
	if(t && !sf_iaton(smask, &mask)) return -1;
	if(!t) {
		errno=0;
		t = strtoul(smask, NULL, 10);
		if((t > 32) || errno) return -1;
		if(!t) mask = 0;
		else mask = htonl(0xffffffff << (32 - t));
	}

	return cfg_add_atable2(ip, mask, strip);
}


int
init_packet_source(packet_source_t *ps) {

	if(strncasecmp(ps->ifName, "ulog", 5) == 0) {
		return init_packet_source_ulog(ps, 1);
	}

#ifdef	PSRC_bpf
	return init_packet_source_bpf(ps);
#elif	PSRC_pcap
	return init_packet_source_pcap(ps);
#else
#error	unknown packet source
#endif

}


