/*-
 * Copyright (c) 2001 Lev Walkin <vlm@spelio.net.ru>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: ppd.c,v 1.15 2001/08/28 08:46:20 vlm Exp $
 */

#include "ipcad.h"
#include "rw.h"
#include "opt.h"
#include "got.h"

struct ip *
find_ipv4_level(packet_source *ps, char *packet, int caplen) {
	register struct ip *ipv4h = NULL;

	if(caplen < sizeof(struct ip))
		return NULL;

	switch(ps->dlt) {
	case DLT_EN10MB: {

		/* Unknown protocol level */
		if(ntohs(((struct ether_header *)packet)->ether_type) != ETHERTYPE_IP)
			return NULL;

		ipv4h = (struct ip *)(packet + ETHER_HDR_LEN);

		break;
	  };
	case DLT_PPP:
		ipv4h = (struct ip *)(packet + 4);

		/* We don't know whether it is IPv4 */
		break;
	case DLT_RAW:	/* Some PPP implementations */
		ipv4h = (struct ip *)packet;
		/* We don't know whether it is IPv4 */
		break;
	case DLT_NULL:
		ipv4h = (struct ip *)(packet + 4);
		/* We don't know whether it is IPv4 */
		break;
#ifdef	DLT_LINUX_SLL
	case DLT_LINUX_SLL:	/* Linux cooked socket */
		ipv4h = (struct ip *)(packet + 16);
		/* We don't know whether it is IPv4 */
		break;
#endif
	default:
		return NULL;
	};

	if(ipv4h->ip_v != IPVERSION)
		return NULL;

	if(caplen - (int)((char *)ipv4h - (char *)packet)
		< sizeof(struct ip)) {
		/* It is not complete IP packet */
		return NULL;
	};

	return ipv4h;
};

void
process_packet_data(
	unsigned char *psp,
#ifdef	PSRC_bpf
	const struct bpf_hdr *bh,
#else
	const struct pcap_pkthdr *ph,
#endif	/* PSRC_bpf */
	const unsigned char *packet
) {
#ifdef	PSRC_bpf
	int caplen = bh->bh_caplen;
#else
	int caplen = ph->caplen;
#endif	/* PSRC_bpf */

	struct ip *ipv4h;
	int iplen;

#define	PSP	((packet_source *)psp)

	ipv4h = find_ipv4_level(PSP, (char *)packet, caplen);
	if(!ipv4h)
		return;

	iplen = ntohs(ipv4h->ip_len);

	/*
	 * Enter critical section
	 */

#ifdef	HAVE_PTHREAD_CANCEL
	pthread_setcancelstate(PTHREAD_CANCEL_DISABLE, NULL);
#endif

	/* Lock chain */
	pthread_mutex_lock(&ipstream_mutex);

	got_ipv4_packet(&ipv4h->ip_src, &ipv4h->ip_dst, iplen);

	/* Unlock chain */
	pthread_mutex_unlock(&ipstream_mutex);

#ifdef	HAVE_PTHREAD_CANCEL
	pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, NULL);
#endif

	/*
	 * Enf of critical section
	 */

	/* Perform statistics counting */
	PSP->bytes_cur += iplen;
	PSP->packets_cur += 1;

}

