/*-
 * Copyright (c) 2001 Lev Walkin <vlm@spelio.net.ru>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: ips-bpf.c,v 1.10 2002/02/06 20:35:45 vlm Exp $
 */

#include "ipcad.h"
#include "opt.h"

int apply_ip_filter(int bpf, int dlt);

packet_source *
init_packet_source(char *ifname, int iflags) {
	int bpf;
	char buf[16];
	int c;
	unsigned int dlt;
	unsigned int bufsize;
	struct ifreq ifr;
	packet_source *ps;


	/* Open spare BPF device */
	c = 0;
	do {
		sprintf(buf, "/dev/bpf%d", c++);
		bpf = open(buf, O_RDWR
#ifdef	O_NONBLOCK
		| O_NONBLOCK
#endif
		);
	} while( bpf == -1 && errno == EBUSY );

	if(bpf == -1)
		return NULL;

	strncpy( ifr.ifr_name, ifname, sizeof(ifr.ifr_name) );

	/* Bind to a specified interface */
	if( ioctl(bpf, BIOCSETIF, &ifr) == -1 ) {
		close(bpf);
		return NULL;
	};

	/* Enable promiscuous mode */
	if(iflags & 2) {
		if( ioctl(bpf, BIOCPROMISC) == -1 ) {
			fprintf(stderr, "[Can't put into promiscuous mode!] ");
			close(bpf);
			return NULL;
		};
	}

	/* Getting type of underlying link */
	if( ioctl(bpf, BIOCGDLT, &dlt) == -1 ) {
		close(bpf);
		return NULL;
	};

	/* Initialize packet filter for optimized throughput */
	if ( apply_ip_filter(bpf, dlt) ) {
		fprintf(stderr, "[Warning: Can't initialize filter!] ");
		close(bpf);
		errno = ENODEV;
		return NULL;
	};

	if(iflags & 1) {
#ifdef	BIOCSSEESENT
		long l = 0;
		if( ioctl(bpf, BIOCSSEESENT, &l) == -1 ) {
			close(bpf);
			return NULL;
		};
#else
#warning "Ignore sent" interface feature is not supported by this BPF implementation. Minor warning.

		printf("\"Ignore sent\" interface feature is not supported by kernel.\n");
#ifndef	EOSERR
#define	EOSERR	EINVAL
#endif
		errno = EOSERR;
		close(bpf);
		return NULL;
#endif
	};

	/* Get buffer length */
	if ( ioctl( bpf, BIOCGBLEN, &bufsize ) == -1 ) {
		close(bpf);
		return NULL;
	};

	/* Don't do buffering in interactive mode */;
	if(daemon_mode == 0) {
		unsigned int on = 1;
		if(ioctl(bpf, BIOCIMMEDIATE, &on) == 0) {
			fprintf(stderr, "[Interactive mode: Buffering disabled] ");
		};
	};

	/* Create packet source descriptor */
	ps = init_packet_source_common(ifname, iflags);
	if(!ps) {
		close(bpf);
		return NULL;
	};

	ps->fd = bpf;
	ps->dlt = dlt;
	ps->bufsize = bufsize;
	ps->buf = (char *)malloc(bufsize);
	if(!ps->buf) {
		free(ps);
		return NULL;
	};

	return ps;
};

#ifndef	RAW_HDR_LEN
#define	RAW_HDR_LEN	0
#endif

#ifndef	NULL_HDR_LEN
#define	NULL_HDR_LEN	4
#endif

#ifndef	PPP_HDR_LEN
#define	PPP_HDR_LEN	4
#endif

#ifndef	PPP_IP
#define	PPP_IP	0x0021
#endif

int
apply_RAW_filter(bpf) {
	struct bpf_insn insns[] = {
		BPF_STMT(BPF_RET+BPF_K, RAW_HDR_LEN + sizeof(struct ip)),
	};
	struct bpf_program bp = {
		sizeof(insns) / sizeof(struct bpf_insn),
		insns
	};

	if ( ioctl( bpf, BIOCSETF, &bp ) == -1 ) {
		return -1;
	};

	return 0;
};

int
apply_NULL_filter(bpf) {
	struct bpf_insn insns[] = {
		BPF_STMT(BPF_LD+BPF_W+BPF_ABS, 0),
		BPF_JUMP(BPF_JMP+BPF_JEQ+BPF_K, 0x02000000, 0, 1),
		BPF_STMT(BPF_RET+BPF_K, NULL_HDR_LEN + sizeof(struct ip)),
		BPF_STMT(BPF_RET+BPF_K, 0),
	};
	struct bpf_program bp = {
		sizeof(insns) / sizeof(struct bpf_insn),
		insns
	};

	if ( ioctl( bpf, BIOCSETF, &bp ) == -1 ) {
		return -1;
	};

	return 0;
};

int
apply_PPP_filter(bpf) {
	struct bpf_insn insns[] = {
		BPF_STMT(BPF_LD + BPF_ABS + BPF_H, 0x0002),
		BPF_JUMP(BPF_JMP + BPF_JEQ + BPF_K, PPP_IP, 0, 1),
		BPF_STMT(BPF_RET+BPF_K, PPP_HDR_LEN + sizeof(struct ip)),
		BPF_STMT(BPF_RET+BPF_K, 0),
	};
	struct bpf_program bp = {
		sizeof(insns) / sizeof(struct bpf_insn),
		insns
	};

	if ( ioctl( bpf, BIOCSETF, &bp ) == -1 ) {
		return -1;
	};

	return 0;
};

int
apply_EN10MB_filter(bpf) {
	struct bpf_insn insns[] = {
		BPF_STMT(BPF_LD+BPF_H+BPF_ABS, 12),
		BPF_JUMP(BPF_JMP+BPF_JEQ+BPF_K, ETHERTYPE_IP, 0, 1),
		BPF_STMT(BPF_RET+BPF_K, ETHER_HDR_LEN + sizeof(struct ip)),
		BPF_STMT(BPF_RET+BPF_K, 0),
	};

	struct bpf_program bp = {
		sizeof(insns) / sizeof(struct bpf_insn),
		insns
	};

	if ( ioctl( bpf, BIOCSETF, &bp ) == -1 ) {
		return -1;
	};

	return 0;
};

int
apply_ip_filter(int bpf, int dlt) {

	switch(dlt) {
		case DLT_RAW:	/* Some PPP implementations, etc. */
			return apply_RAW_filter(bpf);
		case DLT_NULL:	/* Loopback */
			return apply_NULL_filter(bpf);
		case DLT_EN10MB:	/* Ethernet */
			return apply_EN10MB_filter(bpf);
		case DLT_PPP:
			return apply_PPP_filter(bpf);
		default:
			return -1;
	};

	return -1;
};

