%{

#include "headers.h"
#include "cfgvar.h"
#include "servers.h"
#include "rsh.h"
#include "opt.h"

int yylex();
int yyerror();

extern int ipcacfglineno;

%}

/*
 * Token value definition
 */
%union {
	int tv_int;
	char *tv_char;
	struct in_addr tv_ip;
}

/*
 * Token types returned by scanner
 */
%token	<tv_char>	TOK_STRING

/*
 * Reserved words
 */
%token				ERROR
%token				EQ
%token				AT
%token				SEMICOLON
%token				SLASH
%token				IFACE
%token				INONLY
%token				PROMISC
%token				MEMSIZE
%token				RSH
%token				DUMP
%token				CHROOT
%token				AGGR
%token				STRIP
%token				DENY
%token				ALLOW
%token				TIMEOUT
%token				TTL

%token				ADMIN
%token				BACKUP
%token				DEFAULT
%token				VIEW_ONLY

%type	<tv_int>	privlevel
%type	<tv_int>	IFlags
%type	<tv_ip>		at_ip

%%
cfg_file:
	| sequence;

sequence: block
	| sequence block
	;

ps:
	| SEMICOLON;

privlevel:
	ADMIN	{ $$ = 5; }
	| BACKUP	{ $$ = 3; }
	| DEFAULT	{ $$ = 2; }
	| VIEW_ONLY { $$ = 1; }
	| DENY { $$ = 0; }
	;

IFlags:
	INONLY { $$ = 1; }
	| PROMISC { $$ = 2; }
	| IFlags INONLY { $$ |= 1; }
	| IFlags PROMISC { $$ |= 2; }

block:
	TTL EQ TOK_STRING ps {
		default_ttl = atoi($3);
		free($3);
	}
	| IFACE TOK_STRING ps {
		cfg_add_iface($2, 0);
		free($2);
	}
	| IFACE TOK_STRING IFlags ps {
		cfg_add_iface($2, $3);
		free($2);
	}

	| DUMP EQ TOK_STRING ps {
		if( cfg_dump_file )
			free(cfg_dump_file);
		cfg_dump_file = $3;
	};

	| RSH ALLOW at_ip ps {
		if(add_server(rsh_server, "RSH Server", &($3)))
			return yyerror("Failed to install RSH server");
		fprintf(stderr, "Configured RSH Server listening at %s\n",
			inet_ntoa($3));
	}
	| RSH DENY at_ip ps {
		fprintf(stderr, "Warning: Option at line %d has no effect\n",
			ipcacfglineno);
	}
	| RSH TIMEOUT EQ TOK_STRING ps {
		int to_ms;
		to_ms = atoi($4);
		free($4);
		if(to_ms < 0)
			to_ms = -1;	/* INFTIM */
		else
			to_ms = to_ms * 1000;
		rsh_rw_timeout = to_ms;
	}

	| RSH TOK_STRING ps {
		cfg_add_rsh_host("", $2, 2);
		free($2);
	}
	| RSH TOK_STRING privlevel ps {
		cfg_add_rsh_host("", $2, $3);
		free($2);
	}

	| RSH AT TOK_STRING ps {
		cfg_add_rsh_host("", $3, 2);
		free($3);
	}

	| RSH AT TOK_STRING privlevel ps {
		cfg_add_rsh_host("", $3, $4);
		free($3);
	}

	| RSH TOK_STRING AT TOK_STRING ps {
		cfg_add_rsh_host($2, $4, 2);
		free($2); free($4);
	}
	| RSH TOK_STRING AT TOK_STRING privlevel ps {
		cfg_add_rsh_host($2, $4, $5);
		free($2); free($4);
	}
	| CHROOT EQ TOK_STRING ps {
		if(chroot_to)
			free(chroot_to);
		chroot_to = $3;
	}

	| MEMSIZE EQ TOK_STRING ps {
		char *p = $3;

		memsize = atoi($3);
		if(memsize <= 0) {
			free($3);
			return yyerror("Invalid memory limit");
		};
		while(*++p) {
			if(*p >= '0' && *p <= '9')
				continue;
		switch(*p) {
			case ' ':
				break;
			case 'k':
				memsize *= 1024;
				break;
			case 'm':
				memsize *= 1024 * 1024;
				break;
			case 'g':
				memsize *= 1024 * 1024 * 1024;
				break;
			case 'e':
				memsize *= sizeof(struct ipstream);
				break;
			default:
				fprintf(stderr, "Invalid memory limit at line %d near '%c': %s\n",
					*p, ipcacfglineno, $3);
				return -1;
		};
		};
		free($3);
	}
	| AGGR TOK_STRING SLASH TOK_STRING STRIP TOK_STRING ps {
		if(cfg_add_atable($2, $4, $6) != 0) {
			free($2);
			free($4);
			return yyerror("Parse error");
		};
		free($2);
		free($4);
	}
	;

at_ip:
	{ $$.s_addr = INADDR_ANY; }
	| AT TOK_STRING {
		if(inet_aton($2, &($$)) != 1)
			return yyerror("IP address expected");
	}

%%

int
yyerror(char *s) {
	fprintf(stderr, "Config parse error near line %d: %s\n", ipcacfglineno, s);
	return -1;
};

