/* ipband.c
 *
 * Version 0.6 Oct 03, 2001
 *
 * ipband - network bandwidth watchdog
 * By Andrew Nevynniy <anevynni@russelmetals.com>
 *
 * Much of the code is based on ipaudit by Jon Rifkin <jon.rifkin@uconn.edu>
 *
 * Thanks to Nic Bellamy for promisc mode on/off and variable type correction
 * patch.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/*
------------------------------------------------------------------------
Include Files
------------------------------------------------------------------------
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <pcap.h>
#include <sys/types.h>
#include <sys/socket.h>			/* BSD AF_INET */
#include <signal.h>
#include <unistd.h>
#include <time.h>
#include <netinet/in.h>			
#include <netdb.h>	
#include "hash.h"

/*
------------------------------------------------------------------------
Defines
------------------------------------------------------------------------
*/

#define DUMP
#undef  DUMP

#define VERSION_STR "ipband 0.6" 

#define TRUE 1
#define FALSE 0

#define U_CHAR unsigned char

/*  Length of saved packets  */
#define PLEN 68 

/*  Length of packet headers */
#define POFF_ETH  14
#define POFF_PPP   4
#define POFF_RAW   0


/*
------------------------------------------------------------------------
Type Definitions
------------------------------------------------------------------------
*/

/*  Packet structure used by pcap library  */
typedef struct {
	U_CHAR src[6];
	U_CHAR dst[6];
	U_CHAR ptype[2];     /*  ==0x800 if ip  */
	} eth_struct_t;

typedef struct {
	U_CHAR version[1];
	U_CHAR service[1];
	U_CHAR length[2];
	U_CHAR id[2];
	U_CHAR flag[2];
	U_CHAR ttl[1];
	U_CHAR prot[1];
	U_CHAR chksum[2];
	U_CHAR srcip[4];
	U_CHAR dstip[4];
	U_CHAR srcpt[2];
	U_CHAR dstpt[2];
	} ip_struct_t;


/*  Subnet detail data */
typedef struct {
	long       nbyte;
	/* These 2 are keys for deleting detail data for a given subnet */
	int	   subnet_src;
	int        subnet_dst;
} data_t;


/*  Per subnet aggregate data  */
typedef struct {
	long       nbyte;
	/*  
	 *   Non-zero value in logtime means: a) we started detailed 
	 *   logging for this subnet; b) we keep logging on next cycle
	 *   and don't spin off another logging; c) we only zero byte
	 *   counters for this subnet and don't delete this subnet from
	 *   hash table; d) we check if bandwidth goes _below_ limit to
	 *   stop logging and create a report.
	 */
	time_t	   logtime;
	/*
	 *    Number of seconds threshold was exceeded since we started
	 *    detailed logging
	 */
	int	   exc_time;
	/*
	 *    For pre-loaded subnets we store their bandwidth
	 *    threshold value
	 */
	float	   band;
	/*
	 *    Accumulated threshold exceed time in seconds since
	 *    ipband started. Only makes sense for preloaded subnets
	 *    as otherwise subnet data is deleted when usage drops.
	 */
	unsigned int exc_accum;

} aggr_data_t;


/* Linked list for tcp and udp services cache */
typedef struct ll_srvc_s {
	struct ll_srvc_s *next;
	int port;
	char *sname;
	}
	ll_srvc_t;


/*
------------------------------------------------------------------------
Global variables
------------------------------------------------------------------------
*/

extern int errno;
extern char pcap_version[];

/*
------------------------------------------------------------------------
Module variables
------------------------------------------------------------------------
*/

int    isig_m        = 0;		/* Interupt flag for capture loop */
int    preload_m     = FALSE;		/* Subnets are preloaded flag */
int    debug_m       = 0;		/* Debug option */
char   *pcapdev_m    = NULL;		/* Device to listen to */
pcap_t *pcapfile_m   = NULL;	 	/* Pcap input file descriptor */
int    pcapoffset_m  = 0;		/* IP header offset */
char   *filtercmd_m  = NULL;		/* Pcap filter string */
char   *repfname_m   = NULL;		/* Subnet report output file */
int    mask_m	     = 24;		/* Network aggregation mask bits */
int    cycle_m	     = 60;		/* Number of sec to average data */
int    rcycle_m	     = 300;		/* How long in sec bandwidth
					   threshold may be exceeded */
float  thresh_m	     = 7.0;		/* Bandwidth threshold in kBps */
int    fork_m 	     = FALSE;		/* Fork flag */
int    top_m	     = 0;		/* No of top connections in report */
char   *config_m     = NULL;		/* Config file name */
char   *mailto_m     = NULL;		/* E-mail address for reporting */
char   *mailfoot_m   = NULL;		/* Footer file for e-mail report */
int    report_aggr_m = FALSE;		/* Flag to report aggr exceed time */
time_t started_m     = (time_t) 0;	/* Time when we started */
int    promisc_m     = TRUE;		/* Use promiscious mode? */
int    *iplist_m     = NULL;		/* List of local networks */
int    niplist_m     = 0;		/* Number of local networks */

ll_srvc_t *ll_tcp_cache = NULL;    	/* Resolved tcp services cache */
ll_srvc_t *ll_udp_cache = NULL;    	/* Resolved udp services cache */


/*
------------------------------------------------------------------------
Local Function Prototypes
------------------------------------------------------------------------
*/

void print_usage ();
void print_datalink ();
void set_defaults();
void dump_options();
int  read_config (char *);
void read_options (int argc, char *argv[]);
int  parse_subnets (char *, hlist_t **);
void parse_ip_range (char *, int **, int *);
int  in_iprange (int, int *, int);
void preload_subnets(char *, hlist_t **);
void open_interface (int); 
int  get_packetoffset (int);
void ihandler (int);
void storepkt (struct pcap_pkthdr *, ip_struct_t *, hlist_t **, hlist_t **);
void subnet_report (hlist_t **, U_CHAR *,float, int, unsigned int);
void va_report(char *,...);
void detail_cleanup(hlist_t **, U_CHAR *);
int  delete_next(hlist_t **, int, int);
void proc_aggr (hlist_t **, hlist_t **);
char *get_service(int, int); 
char *hex2dot (char *);
void get_two_tok(char *, char **, char **); 
int  is_space(char);
char *find_nonspace (char *);
char *find_space (char *); 
int  strcmpi (char *, char *);
int  is_true_str (char *);
int  compare_bytes (const void *, const void *);
void str2ip (char *, int *, int *);

/*
------------------------------------------------------------------------
Main Function
------------------------------------------------------------------------
*/

int main (int argc, char *argv[]) {

        struct pcap_pkthdr pkthdr;  
	U_CHAR *raw_pkt_save  = NULL;
        U_CHAR *raw_pkt       = NULL;  
        hlist_t **hconn       = NULL;	/* subnet aggreg connection table */
        hlist_t **hconn_d     = NULL;	/* subnet detail connection table */
	fd_set rdfs; 
	int fd;
	int retval;
	pid_t pid;
	eth_struct_t *eth_pkt = NULL;
	ip_struct_t  *ip_pkt  = NULL;
	time_t t0, t1;
	char *config_name_def = "/etc/ipband.conf";


	/*  Record application start time */
	started_m = time(NULL);

        /*  Initialize aggregate and detail hash tables  */
	hconn   = hash_init();
        hconn_d = hash_init();

	/*  Read default config file */
	config_m = strdup(config_name_def);
	read_config(config_m);

	/*  Read command line options (override config file) and interface */
	read_options (argc, argv); 

	/* Print all options for debug */
	if (debug_m) dump_options();

	/*  Check for interface  */
	if( 1 == (argc-optind) ) {
	pcapdev_m = strdup(argv[optind]);
	}
	if( !pcapdev_m) {
	print_usage();
	return(1);
	}


	/* Try to fork */
	if (fork_m) {
	   switch (pid = fork()) {
	     case 0:
		break;
	     case -1:
		printf("Couldn't run in background, exiting...\n");
		exit(1);
	     default:
		printf("Running in background...%d\n",(int) pid);
		exit(0);
	   }
	}

	/* Convert number of mask bits to mask in hex */
	if (mask_m == 32) { 		/* Ugly */
		mask_m = 0xffffffff;
	} else {
		mask_m = 0xffffffff >> mask_m;
		mask_m ^= 0xffffffff;
	}

	/*  If subnet option was specified in config file we will now
	    read config file one more time and add subnet data to the table.
	    We couldn't do this before since there was no guarantee that
	    subnet mask appeared before subnets list in the file */
	if (preload_m) parse_subnets(config_m,hconn); 

	/*  Open pcap file  */
        open_interface(promisc_m);
 
        /*  Allocate room for saved raw packet  */
        raw_pkt_save = (U_CHAR *) malloc (PLEN);
	
	/* Print datalink type as returned by pcap */
	if (debug_m) print_datalink();

        /*  Install interupt handler */
        signal (SIGINT,    ihandler);   /*  intercepts  ^C           */
        signal (SIGTERM,   ihandler);   /*  intercepts  ^kill <PID>  */

	/*  Initialize info for select(). Using select as might 
	    add multiple interfaces later */
	FD_ZERO (&rdfs);
        fd = pcap_fileno(pcapfile_m);
        FD_SET (fd, &rdfs);

	/*  Record cycle start time */
	t0 = time(NULL);

        /*  Read packets until interupt signal */
        while (isig_m==0) {

        /*  Wait for packet on one of the interfaces  */
        retval = select (fd+1, &rdfs, NULL, NULL, NULL);

	/*  If user interupt caught during select() call, retval will
	    be <0.  By continuing we re-test isig_m which should now
	    be set by the interupt handler 
	 */
	if (retval<0) continue;

	/*  Read packet */
	raw_pkt = (U_CHAR *) pcap_next (pcapfile_m, &pkthdr);
        if (raw_pkt==NULL) continue; 

	/*  Skip this packet if ethernet and not ip  */
	if (pcapoffset_m==POFF_ETH) {
	   eth_pkt = (eth_struct_t *) raw_pkt;
	   if (! (eth_pkt->ptype[0]==8 && eth_pkt->ptype[1]==0) ) continue;
	}

	/*  Find pointer to ip packet  */
	ip_pkt = (ip_struct_t *) (raw_pkt + pcapoffset_m);

	/*  Dump packet contents if debugging */
	if (3==debug_m) {
	unsigned int ibyte;
	int iwidth;
	printf ("Raw Packet Length %d\n", pkthdr.len);
	printf ("Captured   Length %d\n", pkthdr.caplen);
	printf ("Captured bytes ...\n");
	iwidth=0;
	for (ibyte=0;ibyte<pkthdr.caplen;ibyte++) {
		printf (" %03d", raw_pkt[ibyte]);
		if (++iwidth==16) {
			printf ("\n");
			iwidth=0;
			}
		}
		printf ("\n");
	}

	/*  Set ports to 0 if not UDP or TCP  */
	if ( ip_pkt->prot[0]!=0x11 && ip_pkt->prot[0]!=0x06 ) {
		if (ip_pkt->prot[0]==1) {
			memset (ip_pkt->dstpt, 0, 2);
		} else {
			memset (ip_pkt->srcpt, 0, 2);
			memset (ip_pkt->dstpt, 0, 2);
        	}
	}

	/*  Dump packet ip data if debugging */
	if (3==debug_m) {
	printf ("*%03d.%03d.%03d.%03d -> %03d.%03d.%03d.%03d  %3d %5d %5d\n",
	ip_pkt->srcip[0],ip_pkt->srcip[1],ip_pkt->srcip[2],ip_pkt->srcip[3],
	ip_pkt->dstip[0],ip_pkt->dstip[1],ip_pkt->dstip[2],ip_pkt->dstip[3],
	ip_pkt->prot[0],
	ip_pkt->srcpt[0]*256+ip_pkt->srcpt[1],
	ip_pkt->dstpt[0]*256+ip_pkt->dstpt[1]);
	}
 
	/*  Store packets in the hash tables */
	storepkt(&pkthdr, ip_pkt, hconn, hconn_d); 

	/* In the end of the loop check if it's time to check aggregate
	   table for bandwidth usage, reset the table and start
	   logging individual subnets */
	t1 = time(NULL);
	if ( (int) difftime(t1,t0) >= cycle_m) {
	   t0 = t1;
	   proc_aggr(hconn, hconn_d); 		/*  Process aggregate table  */
	 }


	} 	/* end of main loop (isig_m == 0) */


	/*  Close files  */    
	pcap_close(pcapfile_m);

	/*  Clear error if breaking during pcap call  */
	errno = 0;


exit(0);

}


/*
------------------------------------------------------------------------
Local Functions
------------------------------------------------------------------------
*/

void print_usage(void) {
   printf("\nUsage: ipband [OPTIONS] interface\n");
   printf("  -a            -  Averaging period in seconds. ");
   printf(		     "Default is 60.\n");
   printf("  -A            -  Include accumulated threshold exceeded time\n");
   printf("                   since ipband start in the report.\n");
   printf("  -b kBps       -  Default bandwidth threshold in kBytes.\n");
   printf("                   per sec. Default is 7 kBps i.e 56 kbps.\n");
   printf("  -c filename   -  Read configuration file. Default is ");
   printf(                   "/etc/ipaband.conf.\n");
   printf("  -C            -  Ignore configuration file\n");
   printf("  -d level      -  Debug level: 0 - no debuging; 1 - summary;\n");
   printf("                   2 - subnet stats; 3 - all packets captured.\n");
   printf("  -f filterstr  -  Use pcap filters (see tcpdump).\n");
   printf("  -F            -  Fork and run in background.\n");
   printf("  -h            -  Print this help.\n");
   printf("  -l filename   -  E-mail report footer file.\n");
   printf("  -L ip-range   -  Range of local ip addresses.\n");
   printf("  -m maskbuts   -  Set number of network mask bits (1-32)\n");
   printf("                   for subnet traffic aggregation.\n");
   printf("                   Default is 24 (255.255.255.0).\n");
   printf("  -M email addr -  Mail report to given addresses.\n");
   printf("  -o filename   -  Subnet report output file. Default is\n");
   printf("                   ipband.txt in current directory.\n");
   printf("  -P            -  Don't use promiscuous mode on network interface.\n");
   printf("  -r            -  Reporting period - number of seconds\n");
   printf("                   banwidth threshold may be exceeded before\n");
   printf("                   it should be reported. Default is 300.\n");
   printf("  -t number     -  Limit report to a given number of connections\n");
   printf("                   with highest byte count. Default is no limit.\n");
   printf("  -v            -  Print version and exit.\n");
   printf("\nExample:\n");
   printf("  ipband eth0 -f \"net 10.10.0.0/16\" -m 24 -a 300 -r 900\n");
   printf("\tWill capture packets from/to ip addresses matching\n");
   printf("\t10.10.0.0/255.255.0.0, tally traffic by the third octet,\n");
   printf("\tcalculate bandwidth utilization every 5 minutes and report\n");
   printf("\tper host traffic every 15 minutes.\n\n");
}


/*  Read options from command line  */
void read_options (int argc, char *argv[]) {

     char optchar;
     while(-1 != (optchar=getopt(argc,argv,"a:Ab:c:Cd:Ff:hl:L:m:M:o:Pr:t:v")))
     {
		switch (optchar) {
		case '?':
			exit(1);
		/*  Get averaging period in seconds */
		case 'a':
			cycle_m = atoi(optarg);
			if( cycle_m < 1 ) {
			   printf("ERROR: Averaging priod  must be at least 1 sec\n");
			   exit(1);
			   }
			break;
		/* Include total exceed time in the report */
		case 'A' :
			report_aggr_m = TRUE;
			break;
		/*  Bandwidth threshold in kBps  */
		case 'b':
			thresh_m = (float) atof(optarg);
			if( thresh_m < 0.0 ) {
			   printf("ERROR: Negative banwidth threshold (black hole?)\n");
			   exit(1);
			   }
			break;
		/*  Read config file */
		case 'c':
			set_defaults();
			config_m = strdup(optarg);
			read_config(config_m);
			break;
		/* Ignore config file */
		case 'C' :
			set_defaults();
			break;
		/*  Debugging option  */
		case 'd':
			debug_m = atoi(optarg);
			break;
		/*  Do we fork?  */
		case 'F':
			fork_m = TRUE;
			break;
		/*  Get pcap filter string  */
		case 'f':
			filtercmd_m = strdup (optarg);
			break;
		/*  Print help  */
		case 'h':
			print_usage();
			exit(1);
			break;
		/*  Get number of subnet mask bits  */
		case 'm':
			mask_m = atoi(optarg);
			if( (mask_m < 0) || (mask_m > 32)) {
			   printf("ERROR: invalid number of mask bits\n");
			   exit(1);
			   }
			break;
		/*  Get e-mail footer file name  */
		case 'l':
			mailfoot_m = strdup (optarg);
			break;
		/* Get range of local networks */
		case 'L':
			 parse_ip_range (optarg, &iplist_m, &niplist_m);
                         break;
		/*  Get address to mail reports to  */
		case 'M':
			mailto_m = strdup (optarg);
			break;
		/* Output file name */
		case 'o':
			repfname_m = strdup(optarg);
			break;
		/* Don't use promiscuous mode */
		case 'P':
			promisc_m = FALSE;
			break;
		/*  Get reporting period in seconds */
		case 'r':
			rcycle_m = atoi(optarg);
			if( rcycle_m < cycle_m ) {
			   printf("ERROR: reporting period cannot be less ");
			   printf("than averaging period\n");
			   exit(1);
			   }
			break;
		/* Top traffic */
		case 't':
			top_m = atoi(optarg);
			if( (top_m < 0) ) {
			   printf("ERROR: need a positive number for ");
			   printf("per-host connection report limit\n");
			   exit(1);
			   }
			break;
		/* Print version */
		case 'v':
			printf ("%s (compiled %s)\n", VERSION_STR, __DATE__);
			printf ("libpcap version %s\n", pcap_version);
			exit(0);
		default:
			exit(1);
		}
	}
}


void open_interface (int promisc) {

	struct bpf_program  fcode;
	char   ebuf[PCAP_ERRBUF_SIZE];

	pcapfile_m = pcap_open_live(pcapdev_m, PLEN, promisc, 1000, ebuf);

	if (pcapfile_m==NULL) {
	printf("ipband: Trouble opening <%s>, msg=\"%s\"\n", 
			pcapdev_m, ebuf);
	exit(1);
	}

	/*  Find IP header offset  */
	pcapoffset_m = get_packetoffset(pcap_datalink(pcapfile_m));
	
	/*  Apply user requested packet filter code */
	if (pcap_compile(pcapfile_m, &fcode, filtercmd_m, 0, 0) < 0)
		printf("compile: %s", pcap_geterr(pcapfile_m));
	if (pcap_setfilter(pcapfile_m, &fcode) < 0)
		printf("setfilter:  %s", pcap_geterr(pcapfile_m));

	/*  Problem with pcap_setfilter?  Sets error, unset here  */
	errno = 0;

}

/* Return IP header offset depending on the interface type */
int get_packetoffset (int DataLinkType) {

	int PacketOffset;
	
	switch (DataLinkType) {
		case DLT_EN10MB:
		case DLT_IEEE802:
			PacketOffset = POFF_ETH;
			break;
		case DLT_PPP:
			PacketOffset = POFF_PPP;
			break;
		case DLT_RAW:
			PacketOffset = POFF_RAW;
			break;
		/* For others we guess  */
		default:
			PacketOffset = 0;
		}

	return PacketOffset;
}

/* Prints datalink type */
void print_datalink() {

printf ("Interface (%s) ", pcapdev_m);
switch (pcap_datalink(pcapfile_m)) {
case DLT_EN10MB:      printf ("DataLinkType = %s\n", "DLT_EN10MB"); break;
case DLT_IEEE802:     printf ("DataLinkType = %s\n", "DLT_IEEE802"); break;
case DLT_SLIP:        printf ("DataLinkType = %s\n", "DLT_SLIP"); break;
case DLT_SLIP_BSDOS:  printf ("DataLinkType = %s\n", "DLT_SLIP_BSDOS"); break;
case DLT_PPP:         printf ("DataLinkType = %s\n", "DLT_PPP"); break;
case DLT_PPP_BSDOS:   printf ("DataLinkType = %s\n", "DLT_PPP_BSDOS"); break;
case DLT_FDDI:        printf ("DataLinkType = %s\n", "DLT_FDDI"); break;
case DLT_NULL:        printf ("DataLinkType = %s\n", "DLT_NULL"); break;
case DLT_RAW:         printf ("DataLinkType = %s\n", "DLT_RAW"); break;
case DLT_ATM_RFC1483: printf ("DataLinkType = %s\n", "DLT_ATM_RFC1483"); break;
default:              printf ("DataLinkType = %d\n", pcap_datalink(pcapfile_m));
}
printf("\n");
}


/* Print options in use for debug purposes */
void dump_options () {
	

printf("\n%s (compiled %s)\n", VERSION_STR, __DATE__);
printf("libpcap version %s\n", pcap_version);
printf("started %s",ctime(&started_m));
printf("\nOption values:\n");
printf("\tDebug level: %d\n",debug_m);
printf("\tPromiscuous mode: %s\n",promisc_m?"yes":"no");
printf("\tConfiguration file: %s\n",config_m);
printf("\tAveraging period (sec): %d\n",cycle_m);
printf("\tReporting peroid (sec): %d\n",rcycle_m);
printf("\tBandwidth threshold (kBps): %g\n",thresh_m);
printf("\tPcap filter string: %s\n",filtercmd_m);
printf("\tSubnet mask bits: %d\n",mask_m);
printf("\tReport output file: %s\n",repfname_m);
printf("\tReport mail to: %s\n",mailto_m);
printf("\tReport mail footer file: %s\n",mailfoot_m);
printf("\tReport top connections: %d\n",top_m);
printf("\n");

/*  Print local network ranges  */
if(iplist_m) {
	int i;
	static char buf1[20], buf2[20];
	printf ("Local IP range:\n");
	for (i=0;i<niplist_m;i++) {
	sprintf(buf1,"%08x",iplist_m[2*i  ]);
	sprintf(buf2,"%08x",iplist_m[2*i+1]);
		printf ("\t%-15s ", hex2dot(buf1));
		printf ("- %-15s\n",hex2dot(buf2));
		}
	printf ("\n");
	}

}

/* Interupt handler (called when program recieves operating system signal */
void ihandler (int cursig) {

	/*  Set flag to terminate main() polling loop 
	 *  when excution reaches bottom  */
	isig_m = 1;

	/*  FLUSH BUFFERS  */
	fflush (stdout);

	/*  RE-INSTALL SIGNAL HANDLER  */
	signal (cursig, SIG_DFL);

	if (debug_m) {
	time_t seconds;
	time (&seconds);
	fprintf (stderr, "ipband received signal number <%i> ", cursig);
	fprintf (stderr, "on %s",ctime(&seconds));
	}

}


/* Print detail report for given subnet */
void subnet_report (hlist_t **ha_d, U_CHAR *key, float thresh, 
		    int exc_time, unsigned int exc_aggr) 
{

	hlist_t **conn = NULL;
	int 	nconn, j, count;
	hlist_t *t;
	data_t  *data;
	char    ip1[16], ip2[16];
	int     pt1, pt2, prot;
	float	kbytes;
	time_t  now = time(NULL);
	int	i;
	int	ip_src = 0;
	int	ip_dst = 0;
	int 	ip_key = 0;
	int	max_conn;
	char	*srvcs;			/* Service name */
	char    *prots;			/* Protocol name */
	struct  protoent *prot_s;	/* IP protocol structure */
	struct  netent   *net_s;	/* Resolved network name sructure */
	int	ikey;			/* key as integer */


	/* Get array of pointers to data and number of connections */
	nconn = 0;
       	conn = hash_getlist(ha_d,&nconn);

	/* Sort array descending by byte count */
	qsort(conn,nconn,sizeof(hlist_t *),compare_bytes);

	/* Get network name if available */
	sscanf(key,"%08x",&ikey);
	net_s = getnetbyaddr(ikey,AF_INET);

	/* Print e-mail subject to include subnet info */
	if(net_s) va_report("Subject: Bandwidth report for %s <%s>\n\n",hex2dot(key),net_s->n_name);
	else va_report("Subject: Bandwidth report for %s\n\n",hex2dot(key));

	/* Print header */
	va_report("\nDate:   %s", ctime(&now));
	if(top_m) va_report("Showing top %d connections\n",top_m);
	va_report("Network: %s", hex2dot(key));
	if(net_s) va_report(" <%s>",net_s->n_name);
	va_report("\n");
	va_report("Bandwidth threshold: %.2f kBps, exceeded for: %.2f min\n",thresh,(float) exc_time/60.0);

	/* Total accumulated time and percentage would not apply
	 * if subnets are not pre-loaded as they would be deleted
	 * once bandwidth usage dropped below threshold */
	if(report_aggr_m && preload_m) {
	  char *exc_str;
	  size_t lastch;
	  float elapsed;

	  exc_str = ctime(&started_m);
	  lastch = strlen(exc_str) - 1;
	  if (exc_str[lastch] == '\n') exc_str[lastch] = '\0';

           /* How many secs passed since we started? */
          elapsed = (float) difftime(time(NULL),started_m);

	  va_report("Threshold exceeded time since %s: %.2f min (%4.2f%%)\n",exc_str, (float) exc_aggr/60.0, exc_aggr*100.0/elapsed);
	}

	va_report("===============================================================================\n");
	va_report("FROM            < PORT>     TO              < PORT>  PROT   KBYTES  SERVICE\n");
	va_report("-------------------------------------------------------------------------------\n");

	/* Walk hash table */
	max_conn = (top_m && top_m < nconn) ? top_m : nconn;
	for(j=0, count=0; j<nconn && count<max_conn; j++){

		t=conn[j];
		data = (data_t *) t->data;

		for(i=3; i>=0; i--) {
		ip_src += t->key[i] << (8*i);
		ip_dst += t->key[i+4] << (8*(i+4));
		}

		/* Skiping subnets other than our */
		ip_src = data->subnet_src;
		ip_dst = data->subnet_dst;
		sscanf(key,"%08x",&ip_key);
		if ( !(ip_src == ip_key || ip_dst == ip_key) ) continue;

		/* Inreasing loop counter only after other subnets are
		   skipped */
		count++;

		/*  Get ip addresses and ports  */
		sprintf (ip1, "%u.%u.%u.%u", 
			t->key[0], t->key[1], t->key[2], t->key[3]);
		sprintf (ip2, "%u.%u.%u.%u", 
			t->key[4], t->key[5], t->key[6], t->key[7]);
		pt1  = (int) t->key[ 8]*256 + t->key[ 9];
		pt2  = (int) t->key[10]*256 + t->key[11];
		prot = t->key[12];


		/* For tcp or udp we try to resolve service name */
		if (6 == prot) {                /* tcp */
		srvcs = get_service(pt1,prot);
		if (!strlen(srvcs)) srvcs = get_service(pt2,prot);
		prots = "tcp";

		} else if (17 == prot ) {       /* udp */
		srvcs = get_service(pt1,prot);
		if (!strlen(srvcs)) srvcs = get_service(pt2,prot);
		prots = "udp";

		} else {                        /* not tcp or udp */
		   if ( (prot_s = getprotobynumber(prot)) ) {
		   prots = prot_s->p_name;
		   } else {
		   sprintf(prots,"%u",prot);
	 	   }
		srvcs = "";
		}


		/*  Print key info  */
		va_report("%-15s <%5u> <-> %-15s <%5u> %4s",
			   ip1,  pt1,      ip2,   pt2, prots);

		/* Total bytes, time and service */
		kbytes = (data->nbyte)/1024.0;
		va_report(" %9.2f",kbytes);
		if (srvcs) va_report("  %.11s",srvcs);
		va_report("\n");

	}	/* End looping through connections */

	va_report("===============================================================================\n");

	/* Close report file handles */
	va_report(NULL);

	/* Free array of pointers */
	if( NULL != conn ) free(conn);

}


/*
Store packet info in aggregate hash table, keyed by subnet
Store packet info in detail table if subnet is being logged
*/

void storepkt (struct pcap_pkthdr *pkthdr, ip_struct_t *ip, 
	       hlist_t **ha, hlist_t **ha_d) {

	U_CHAR     	key_src[9];  /*  src subnet as hex string  */
	U_CHAR     	key_dst[9];  /*  dst subnet as hex string  */
        U_CHAR     	key[13];     /*  detail logging key */ 
	aggr_data_t     *data,    idata;
	data_t		*data_d,  idata_d;
	int        	ndata,    ndata_d;
	int        	datasize, datasize_d;
	int        	keysize,  keysize_d;
	int	   	ip_src, ip_dst;
	int        	length;
	int 	   	i;
	int		detail_flag = FALSE;

	/* Calculate data packet length */
	length = ip->length[1] + 256*(int) ip->length[0] + 14;

	/* Get ip src and dst ip address and convert to integers */
	ip_src = ip_dst = 0;
	for(i=0; i<4; i++){
	   ip_src = ip_src<<8;
	   ip_dst = ip_dst<<8;
	   ip_src |= (int) ip->srcip[i];
	   ip_dst |= (int) ip->dstip[i];
	}

	/* Apply mask and get our key - network number */
	ip_src &= mask_m;
	ip_dst &= mask_m;
	sprintf(key_src,"%08x",ip_src);
	sprintf(key_dst,"%08x",ip_dst);

	/* Store length of this packet */
	idata.nbyte = length;

	/* Set logtime to zero (when adding the key first time)
	   meaning that we don't start detailed logging of this subnet yet */
	idata.logtime = (time_t) 0;

	/* Initialize exceed time counters */
	idata.exc_time = 0;
	idata.exc_accum = 0;
	
	/* Set bandwidth threshold to zero if we don't have preloaded nets */
	idata.band = 0.0;

	/*  Set size of data structures  */
	datasize = sizeof(aggr_data_t);
	keysize  = sizeof(key_src);

		#ifdef DUMP
		printf("Hash table key: ");
		printf("src %s - dst %s len - %d\n",key_src,key_dst,length);
		#endif


   /*********************************************************************/
   /*    If preload_m is set, we only update and do not add new nodes   */
   /*********************************************************************/

   		/* Processing source network */
	
   /* If local ip range specified and src network is not in that range,
    * don't process this network */
   if (!iplist_m || (iplist_m && in_iprange(ip_src,iplist_m,niplist_m))){
	 
        /* Add first instance of source key to table */
	if (! hash_finddata(ha,(U_CHAR *)&key_src,keysize,
			       (U_CHAR **)&data,&ndata)){ 
	   if ( !preload_m ) {
	   datasize = sizeof(idata);
	   hash_addnode(ha,(U_CHAR *)&key_src,keysize,
			   (U_CHAR *)&idata,datasize);
	   }

        /* Key already present, update info */
        } else  {
           /* Update byte count */
           data->nbyte += idata.nbyte;
           /* Do we log packet to detail table? */
           if ( data->logtime != 0 ) detail_flag = TRUE;
        }
   }		/* End of processing source network */

   		/* Processing destination network */

   /* If local ip range specified and dst network is not in that range,
    * don't process this network */
   if (!iplist_m || (iplist_m && in_iprange(ip_dst,iplist_m,niplist_m))) {

        /* If src and dst on same subnet don't log connection twice */
        if ( ip_src != ip_dst ) {
   
           /* Add first instance of destination key to table */
	   if (! hash_finddata(ha,(U_CHAR *)&key_dst,keysize,
				  (U_CHAR **)&data,&ndata)){

		if ( !preload_m ) {
		   datasize = sizeof(idata);
		   hash_addnode(ha,(U_CHAR *)&key_dst,keysize,
				   (U_CHAR *)&idata,datasize);
		}

           /* Key already present, update info */
           } else  {
              /* Update byte count */
              data->nbyte += idata.nbyte;
              /* Do we log packet to detail table? */
              if ( data->logtime != 0 ) detail_flag = TRUE;
           }

        } 	/* End of if not on the same subnet */

   }		/* End of processing destination network */


   /*********************************************************************/
   /*      If this packet should be logged to subnet detail table       */
   /*********************************************************************/

   if( detail_flag ) { 

	/* Make key - order so smallest ip first store data */
	if (memcmp(ip->srcip, ip->dstip, 4) < 0) {
	   memcpy (key+ 0, ip->srcip, 4);
	   memcpy (key+ 4, ip->dstip, 4);
	   memcpy (key+ 8, ip->srcpt, 2);
	   memcpy (key+10, ip->dstpt, 2);
	} else {
	   memcpy (key+ 0, ip->dstip, 4);
	   memcpy (key+ 4, ip->srcip, 4);
	   memcpy (key+ 8, ip->dstpt, 2);
	   memcpy (key+10, ip->srcpt, 2);
	}

	   memcpy (key+12, ip->prot,  1);
	   idata_d.nbyte = length;
	   /* Fill in subnets this packet belongs to for easier deleting */
	   idata_d.subnet_src = ip_src;
	   idata_d.subnet_dst = ip_dst;

	/*  Set size of data structures  */
	datasize_d = sizeof(data_t);
	keysize_d  = sizeof(key);

       /* Add first instance of this key to table */
        if (! hash_finddata(ha_d,(U_CHAR *)&key,     keysize_d,
				 (U_CHAR **)&data_d, &ndata_d) )
	{
	datasize_d = sizeof(idata_d);
	hash_addnode(ha_d,(U_CHAR *)&key,     keysize_d,
			  (U_CHAR *)&idata_d, datasize_d);
				  
	/* Key already present, update info */
	} else  {
	data_d->nbyte += idata_d.nbyte;
	}

   }   /* End logging to subnet detail table */

}


/* Process per-subnet aggregate hash table */
void proc_aggr (hlist_t **ha, hlist_t **ha_d) {
	hlist_t 	*t;
	aggr_data_t  	*data;
	FILE    	*outfile_m = stdout;
	float		kbytes, kBps;
	float		thresh;
	int 		exc_time;
	hiter_t		ti;		/* table iterator */

	
	/* Walk hash table */
	for(t=hash_getfirst(ha,&ti); t; t = hash_getnext(ha,&ti)){

		data = (aggr_data_t *) t->data;

		/* What is bandwidth threshold for this subnet */
		thresh = (data->band) ? data->band : thresh_m;

		/* Total bytes and bandwidth */
		kbytes = (data->nbyte)/1024.0;
		kBps = (float) kbytes/cycle_m;

		/* If detailed logging for this subnet in progress */
		if ( (long) data->logtime != 0 ){

		   /* Usage still high */
		   if ( kBps >= thresh ) {

	            /* How long threshold has been exceeded this cycle? */
	            exc_time = (int) difftime(time(NULL),data->logtime);

		       /* Is it time to cry out loud? */
		       if ( exc_time >= rcycle_m ){

	                 /* How long threshold has been exceeded so far? */
	                 data->exc_time += exc_time;

			 /* Accumulated exceed time since app started */
			 if (report_aggr_m) data->exc_accum += exc_time;

		         subnet_report(ha_d,t->key,thresh,
				       data->exc_time,
				       data->exc_accum);
	          	 data->logtime = 0;

		       }
		   
		   /* If bandwidth dropped below limit we stop
		      detailed logging */
		   } else {

		       /* Delete subnet entries from detail log */
		       detail_cleanup(ha_d,t->key);
		
		       /* Unset detail logging flag for this subnet */
		       data->logtime = 0;

		       /* Clear exc_time for subnet */
		       data->exc_time = 0;
		   }

		} /* End if detailed logging in progress */


		/* if bandwidth threshold is exceeded for the first time
		   we start detailed logging for this subnet
		   setting logtime value in aggr_data_t structure to
		   current time */
		if ( kBps >= thresh && 0 == (long) data->logtime ) {
		data->logtime = time(NULL);
		}
		

		if (2==debug_m) {
		/*  Print subnet table  */
		   if ( kBps > thresh && (long) data->logtime != 0) 
			fprintf (outfile_m, "*");
		   else 
			fprintf (outfile_m, " ");
		fprintf (outfile_m, "%-15s", hex2dot(t->key));
		fprintf (outfile_m, " %7.2f kB ",kbytes);
		fprintf (outfile_m, " %7.2f/%6.2f kBps",kBps,thresh); 
		fprintf (outfile_m, "\n");
		}

		/* Clean-up */
		/* If subnet is being logged - zero counters */
		if ( (long) data->logtime != 0 ){
		data->nbyte = 0;
		} else {
		   /*
		   If not - delete it. But *only* if 
		   we are NOT working with preloaded subnets!
		    */
		   if (! preload_m ) {
			hash_delnode(ha, t->key, t->nkey);
		   /* For preloaded subnets - just clear counters */
		   } else {
		     data->nbyte = 0;
		   }
		}

	} /* End of walking table */
	
	if (2==debug_m) {
	fprintf(outfile_m,"************************************************\n");
	}

}

/*
Delete entries for specific subnet from detail hash table when bandwidth
usage for that subnet drops below limit.
This is a hash table function but not generic to be put in hash.c.
*/

void detail_cleanup (hlist_t **ha_d, U_CHAR *key) {
	
	int 	ip_key;
	int 	hash;
	int 	result;

	/* Get size of hash table */
	int slots = N_HASH_SLOTS; 	/* from hash.h */
	int ntable;
	int nhashbit = 1;
	slots--;
	while (slots>>=1) nhashbit++;
	ntable = 1 << nhashbit;

	/* Get subnet number in int */
	sscanf(key,"%08x",&ip_key);

	/* Walk table */
	for (hash = 0; hash < ntable; hash++) {

	   if (NULL == ha_d[hash]) continue;
	
	   result = TRUE;
	   while (result) result = delete_next(ha_d,hash,ip_key);

	} 	/* end of walk table loop */
}


/* Delete next node matching ip_key from given slot */
int delete_next(hlist_t **ha_d, int hash, int ip_key) {

	hlist_t	*t_prev;
	hlist_t *t;
	data_t  *data;
	int	 subnet_src, subnet_dst;

	for(t=ha_d[hash]; t; t_prev=t, t=t->next) {

		 data = (data_t *) t->data;
		 subnet_src = data->subnet_src;
		 subnet_dst = data->subnet_dst;
		 if((subnet_src == ip_key) || (subnet_dst == ip_key)) {
		 	free(t->data);
			free(t->key);
			if (t == ha_d[hash]) ha_d[hash] = t->next;
			else t_prev->next = t->next; 
			free((void *) t);
			return 1;
		 } 
	} 
	return 0;
}



/* Convert 8 byte hex string to dotted octet string */
char * hex2dot (char * str) {

	int i,ip;
	int p[4];
	static char buf[20];

	sscanf(str,"%08x",&ip);

	for (i=0;i<4;i++) {
		p[i] = ip & 0xff;
		ip >>= 8;
	}
	
	sprintf (buf, "%d.%d.%d.%d", p[3], p[2], p[1], p[0]);
	return buf;

}

/* Parse config file */
int read_config (char *filename) {
	FILE *fin = NULL;
	char buffer[512];
	char *str;
	char *key, *val;

	fin = fopen (filename, "r");
	if (NULL==fin)  return errno;

        while ( (str=fgets(buffer, 512, fin)) ) { 

	get_two_tok(str, &key, &val); 

	/*  Test for comment or empty line  */
	if (*key=='#' || *key=='\0') continue;

	/* Test for valid options */
	if (!strcmpi("debug",key)) {
		debug_m = atoi(val);

	} else if (!strcmpi("filter",key)) {
		filtercmd_m = strdup(val);	

	} else if (!strcmpi("fork",key)) {
		fork_m = is_true_str(val);	

	} else if (!strcmpi("outfile",key)) {
		repfname_m = strdup(val);

	} else if (!strcmpi("interface",key)) {
		pcapdev_m = strdup(val);

	} else if (!strcmpi("promisc",key)) {
		promisc_m = is_true_str(val);

	} else if (!strcmpi("average",key)) {
		cycle_m = atoi(val);
		if( cycle_m < 1 ) {
		   printf("ERROR: Cycle must be at least 1 sec\n");
		   exit(1);
		   }
	} else if (!strcmpi("bandwidth",key)) {
		thresh_m = (float) atof(val);
		if( thresh_m < 0.0 ) {
		   printf("ERROR: Unreasonable banwidth threshold\n");
		   exit(1);
		   }
	} else if (!strcmpi("accumulate",key)) {
		report_aggr_m = is_true_str(val);	

	} else if (!strcmpi("report",key)) {
		rcycle_m = atoi(val);
		if( rcycle_m < cycle_m ) {
		   printf("ERROR: reporting period cannot be less ");
		   printf("than averaging period\n");
		   exit(1);
		   }
	} else if (!strcmpi("localrange",key)) {
		parse_ip_range (val, &iplist_m, &niplist_m);

	} else if (!strcmpi("mailto",key)) {
		mailto_m = strdup(val);

	} else if (!strcmpi("mailfoot",key)) {
		mailfoot_m = strdup(val);

	} else if (!strcmpi("maskbits",key)) {
		mask_m = atoi(val);
		if( (mask_m < 0) || (mask_m > 32)) {
		   printf("ERROR: invalid number of mask bits\n");
		   exit(1);
		   }
	} else if (!strcmpi("top",key)) {
		top_m = atoi(val);
		if( (top_m < 0) ) {
		   printf("ERROR: need a positive number for ");
		   printf("top connection report limit\n");
		   exit(1);
		   }
	} else if (!strcmpi("subnet",key)) {

		/* Set preload flag - we are now limited to specified nets 
		   Will process option(s) later when subnet mask is known
		   for sure */
		preload_m = TRUE;

	} else {
		fprintf(stderr,"ipband: Error reading ipband config file. ");
		fprintf(stderr,"  Unrecognized option: \"%s\"", key);

	} /* End of test for options */

	} /* End of while () looping through config file */

	fclose(fin); 

	return 1;
}


/* Process subnet options in config file */
int parse_subnets (char *filename, hlist_t **ha) {
	FILE *fin = NULL;
	char *str;
	char *key, *val;
	char buff[512];

	fin = fopen (filename, "r");
	if (NULL==fin)  return errno;

        while ( (str=fgets(buff, 512, fin)) ) { 

	get_two_tok(str, &key, &val); 

	/* Find subnet options and load it into hash table */
	if (!strcmpi("subnet",key)) preload_subnets(val,ha);

	}

	fclose(fin); 

	return 1;

}


/* Build hash table for subnets to be monitored */
void preload_subnets(char *str, hlist_t **ha){

        U_CHAR 		key[9];     	/* Subnet key as hex string */ 
	aggr_data_t     *data,    idata;
	int        	datasize, keysize;
	int		ndata;
	float  		bwidth = 0.0;
	int    		p[4];
	int    		netip  = 0;
	char   		*buf   = (char *) malloc(strlen(str)+1); 
	int    		i;

	/*  Break subnet option string into ip/bandwidth */
	if (6 != sscanf(str,"%d.%d.%d.%d%s%f",&p[0],&p[1],&p[2],&p[3], buf,&bwidth) ) {
	  fprintf(stderr,"ipband: Error parsing subnet option in config file: %s\n",str);
	  free(buf);
	  return;
	}

	free(buf);

	/* Convert network address to integer */
	for(i=0; i<4; i++){
	   netip = netip<<8;
	   netip |= p[i];
	}

	/* Apply mask */
	netip &= mask_m;

	sprintf(key,"%08x",netip);

	/* Set bandwidth threshold for this net */
	idata.band = bwidth;

	/* Initialize all other fields */
	idata.nbyte = 0;
	idata.logtime = (time_t) 0;
	idata.exc_time = 0;
	idata.exc_accum = 0;

	/*  Set size of data structures  */
	datasize = sizeof(aggr_data_t);
	keysize  = sizeof(key);

	/* Add first instance to table */
	if(!hash_finddata(ha,(U_CHAR *)&key,keysize,(U_CHAR **)&data,&ndata)){
	datasize = sizeof(idata);
	hash_addnode(ha,(U_CHAR *)&key,keysize,(U_CHAR *)&idata,datasize);

	/* Key already present, update bandwidth */
	} else  {
	data->band = bwidth;
	}
}


/*  Set all options to default values */
#define FREE(P) if ((P)!=NULL) { free(P); (P)=NULL; } 
void set_defaults(void) {

	debug_m       = FALSE;  
	preload_m     = FALSE;
	mask_m	      = 24;
	cycle_m	      = 60;
	rcycle_m      = 300;
	thresh_m      = 7.0;
	fork_m	      = FALSE;
	top_m	      = 0;
	promisc_m     = TRUE;
	niplist_m     = 0;

	/* These were malloc'ed by strdup and can be freed */
	FREE(config_m);
	FREE(pcapdev_m);
	FREE(filtercmd_m); 
	FREE(repfname_m);
	FREE(mailto_m);
	FREE(mailfoot_m);
	FREE(iplist_m);

}

/*  Return pointers to first two space delimited tokens, 
    null terminating first token if necessary.
    If no first,second token then pointers point to '\0'
*/
void get_two_tok(char *str, char **tok1, char **tok2) {

	/*  Find start of first token  */
	str = find_nonspace(str);
	*tok1 = *tok2 = str;
	if (*str=='\0') return;

	/*  Find end of first token  */
	*tok2 = str = find_space (str);
	if (*str=='\0') return;

	/*  terminate first token  */
	*(str++) = '\0';

	/*  find second token   */
	*tok2 = find_nonspace(str);

	/*  Remove trailing space  */
	str = str + strlen(str) - 1;
	while (is_space(*str)) {
		str--;
	}
	*(++str) = '\0';
}
	
/*  Test for space *OR* equals sign 
 *  (to allow shell scripts lines like TERM=vt1000 to be used as config
 *  files
 *  */
int is_space(char c) {
	return c==' ' || c=='\t' || c=='\n' || c=='\r' || c=='=';
}

/*  Find first non-space char */
char *find_nonspace (char *str) {
	while (*str && is_space(*str)) str++;
	return str;
}

/*  Find first space char */
char *find_space (char *str) {
	while (*str && !is_space(*str)) str++;
	return str;
}

/*  Compare two strings ignoring case  */
int strcmpi (char *a, char *b) {
	int equal = 1;
	char c,d;
	while (equal && *a) {
		c = *a++;
		d = *b++;
		if ('a'<=c && c<='z') c += 'A' - 'a';
		if ('a'<=d && d<='z') d += 'A' - 'a';
		equal = (c==d);
	}
	if (equal) return 0;
	if (c<d)   return -1;
	return 1;
}

/*  Return true of string is yes, on, ok ignoring case  */
int is_true_str (char *str) {
	return
		(! strcmpi("yes",str)) ||
		(! strcmpi("true",str)) ||
		(! strcmpi("on",str)) ||
		(! strcmpi("ok",str));
}

/* Argument to qsort() for sorting subnet detail hash table */
int compare_bytes (const void *a, const void *b) {

	data_t *dataa, *datab;

	const hlist_t **ta = (const hlist_t **) a;
	const hlist_t **tb = (const hlist_t **) b;

	dataa = (data_t *) (*ta)->data;
	datab = (data_t *) (*tb)->data;

	return (datab->nbyte) - (dataa->nbyte);
}


/* Get service name by tcp or udp port number and store in cache */
char *get_service(int port, int prot) {

	ll_srvc_t *p;
	char *srvcs;
	char *prots;

	struct servent  *srvc_s;
	struct protoent *prot_s;
	int found = 0;
	srvcs = "";

	/* For tcp or udp we try to resolve service name */
	if (6 == prot) {		/* tcp */

	/* Check service name in cache */
	   p = ll_tcp_cache;
 	   while (p) {
	   	if (port == p->port) {
			found = 1;
			srvcs = p->sname;
		}
	   p = p->next;
	}

	/* Not in cache? Put there */
	if( !found) {

	   /* Resolve name */
	   if ((srvc_s = getservbyport(htons(port),"tcp")))
	        srvcs = srvc_s->s_name;

	   /* Insert name in front of tcp cache linked list */
	   if( (p = (ll_srvc_t *) malloc(sizeof(*p))) ){
	  	   p->port = port;
		   p->sname = strdup(srvcs);
		   p->next = ll_tcp_cache;
		   ll_tcp_cache = p;
	   }
	}


	} else if (17 == prot ) {	/* udp */

	  if ( (prot_s = getprotobynumber(prot)) ) {
		prots = prot_s->p_name;
	  }

	/* Check service name in cache */
	   p = ll_udp_cache;
 	   while (p) {
	   	if (port == p->port) {
			found = 1;
			srvcs = p->sname;
		}
	   p = p->next;
	}

	/* Not in cache? Put there */
	if( !found) {

	   /* Resolve name */
	   if ((srvc_s = getservbyport(htons(port),prots)))
	        srvcs = srvc_s->s_name;

	   /* Insert name in front of udp cache linked list */
	   if( (p = (ll_srvc_t *) malloc(sizeof(*p))) ){
	 	   p->port = port;
		   p->sname = strdup(srvcs);
		   p->next = ll_udp_cache;
		   ll_udp_cache = p;
	   }

	}

	} 

	return srvcs;
}


/* Use variable length arguments to output reports to multiple facilities */
void va_report(char *cp,...) {

	va_list va;
	static FILE 	*sendmail;	/* static to persist across calls */
	static FILE 	*repfile;	/* static to persist across calls */ 
	FILE 		*ffoot = NULL;
	char 		buffer[512];
	char 		*str;

	if (!cp){ 	/* Cleanup when called with NULL format string */

		if (repfile) {
			if (repfile != stdout) fclose(repfile);
			repfile = NULL;
			}

		if (sendmail && mailto_m) {
			/* Append mail footer if needed */
			if (mailfoot_m) {
			   if ( (ffoot = fopen (mailfoot_m, "r")) ){
		              while ( (str=fgets(buffer, 512, ffoot)) )  
			      fputs(str,sendmail);
			   }
			}
			/* Close handle */
			pclose(sendmail);
			sendmail = NULL;
			}

	} else {		/* Get handles and print */

	if (!repfile) {

		if (! repfname_m) repfname_m = "ipband.txt";
		if (strcmp("-",repfname_m)) repfile = fopen (repfname_m, "a");
		else			    repfile = stdout;

			if (NULL==repfile) {
			fprintf (stderr, "ERROR:  Cannot open output file <%s>\n", repfname_m);
			exit(1);
			}
	}

	if (mailto_m) {		/* If e-mail option is set */

	   /* Assume sendmail in /usr/sbin */
	   if(!sendmail) {
		   
		sendmail = popen("/usr/sbin/sendmail -t -ba","w");
		if (NULL==sendmail) {
		fprintf (stderr, "ERROR:  error opening /usr/sbin/sendmail\n");
		exit(1);
		}
		/* Sendmail headers */
		fprintf(sendmail,"To: %s\n",mailto_m);
		fprintf(sendmail,"From: IP bandwdth watchdog <>\n");
	   }

	}

	if (strncmp(cp,"Subject:",8)) {		/* Skip mail subject line */
	   va_start(va,cp);
	   if( repfile) vfprintf(repfile,cp,va);
	   va_end(va);
	}

	if (mailto_m) {
	   va_start(va,cp);
	   if( sendmail) vfprintf(sendmail,cp,va);
	   va_end(va);
	}
	
	}
}

/* Get list of local networks */
void parse_ip_range (char *arg_in, int **iplist, int *niplist) {
	char *arg_cpy = (char *) malloc (strlen(arg_in)+1);
	char *ipstr   = (char *) malloc (strlen(arg_in)+1);
	char *netstr  = (char *) malloc (strlen(arg_in)+1);
	char *range1  = NULL;
	char *range2  = NULL;
	int  mask;
	int  net;
	int  ip1, ip2;
	int  n;
	char *p;

	*iplist = NULL;

	/*  Count number of ranges (equals number of : + 1 )  */
	p = arg_in;
	n = 1;
	while (*p++) {
		if (*p==':') n++;
	}

	/*  allocate storage  */
	*iplist = (int *) malloc (2 * n * sizeof(int));
	if (*iplist==NULL) {
		*niplist = 0;
		return;
	}

	strcpy  (arg_cpy, arg_in);
	range2 = arg_cpy;

	/*  break string into separate ranges  */
	*niplist = 0;
	while (NULL!=range2) {

		/*  Break arg into (1st range):(remaining ranges)  */
		range1 = range2;
		range2 = strchr(range1, ':');
		if (NULL!=range2) *range2++ = '\0';


		/*  Look for range expressed as (lo ip)-(hi ip)  */
	 	if (2==sscanf (range1, "%[0-9.]-%[0-9.]", ipstr, netstr)) {
			str2ip(ipstr,  &ip1, &mask);
			str2ip(netstr, &ip2, &mask);

		/*  break range into (ip)/(net)  */
		} else if (2==sscanf (range1, "%[0-9.]/%[0-9]", ipstr, netstr)) {

			/*  read ip address  */
			str2ip (ipstr, &ip1, &mask);

			net = atoi(netstr);
			if (net<0) net=0;
			else if (net>32) net=32;
			mask = 0xffffffff >> net;
			if (mask==-1) mask = 0;
			ip2 = ip1 | mask;

		/*  Look for single ip address  */
		} else if (sscanf (range1, "%[0-9.].%[0-9].", ipstr, netstr)) {
			str2ip (ipstr, &ip1, &mask);
			ip2 = ip1 | mask;

		/*  Bad input format  */
		} else {
		fprintf (stderr, "ERROR:  Cannot read network range argument (-l option).\n");
		fprintf (stderr, "  Program continues with using default network range.\n");
		*niplist = 0;
		if (NULL!=*iplist) free (*iplist);
		return;
		}

		/* Store results  */
		(*iplist)[(*niplist)++] = ip1;
		(*iplist)[(*niplist)++] = ip2;
	} 

	free (netstr);
	free (ipstr);
	free (arg_cpy);

	/*  Correct double counting of niplist  */
	*niplist /= 2;

}
/*  Convert strings like "138.99.201.5" or "137.99.26" to int ip address  */
void str2ip (char *ipstr, int *ipout, int *mask) {
	int ip[4];
	int n = sscanf (ipstr, "%d.%d.%d.%d", ip, ip+1, ip+2, ip+3);
	int i;
	*ipout = 0;
	for (i=0;i<4;i++) {
		*ipout = *ipout<<8;
		if (i<n) *ipout |= (ip[i] & 0xff);
	}
	*mask = 0xffffffff >> (8*n);

	/* for reasons unknown 0xffffffff >> 32 -> -1, so set to 0  */
	if (*mask==-1)  *mask=0;
}

/*  Determine if ip addresse is within one of the ranges in iplist  */
int in_iprange (int ip, int *iplist, int niplist) {
	int i;
	for (i=0;i<2*niplist;i+=2)
	   if (ip>=iplist[i] && ip<=iplist[i+1])   return 1;
	return 0;
}
