/*
 * ion/event.c
 *
 * Copyright (c) Tuomo Valkonen 1999-2001. 
 * See the included file LICENSE for details.
 */

#include <stdlib.h>
#include <unistd.h>
#include <sys/time.h>

#include "common.h"
#include "event.h"
#include "clientwin.h"
#include "screen.h"
#include "frame.h"
#include "property.h"
#include "pointer.h"
#include "key.h"
#include "focus.h"
#include "cursor.h"
#include "signal.h"
#include "global.h"
#include "draw.h"
#include "input.h"
#include "selection.h"
#include "thing.h"
#include "sizehint.h"


/*{{{ Prototypes */


static void handle_expose(const XExposeEvent *ev);
static void handle_map_request(const XMapRequestEvent *ev);
static void handle_configure_request(XConfigureRequestEvent *ev);
static void handle_enter_window(XEvent *ev);
static void handle_unmap_notify(const XUnmapEvent *ev);
static void handle_destroy_notify(const XDestroyWindowEvent *ev);
static void handle_client_message(const XClientMessageEvent *ev);
static void handle_focus_in(const XFocusChangeEvent *ev);
static void handle_focus_out(const XFocusChangeEvent *ev);
static void handle_property(const XPropertyEvent *ev);
static void handle_colormap_notify(const XColormapEvent *ev);
static void pointer_handler(XEvent *ev);
static void keyboard_handler(XEvent *ev);


/*}}}*/


/*{{{ Event-reading, mainloop */


void get_event(XEvent *ev)
{
	fd_set rfds;
	
	while(1){
		check_signals();
	
		if(QLength(wglobal.dpy)>0){
			XNextEvent(wglobal.dpy, ev);
			return;
		}
		
		XFlush(wglobal.dpy);

		FD_ZERO(&rfds);
		FD_SET(wglobal.conn, &rfds);

		if(select(wglobal.conn+1, &rfds, NULL, NULL, NULL)>0){
			XNextEvent(wglobal.dpy, ev);
			return;
		}
	}
}


void get_event_mask(XEvent *ev, long mask)
{
	fd_set rfds;
	bool found=FALSE;
	
	while(1){
		check_signals();
		
		while(XCheckMaskEvent(wglobal.dpy, mask, ev)){
			if(ev->type!=MotionNotify)
				return;
			found=TRUE;
		}

		if(found)
			return;
		
		FD_ZERO(&rfds);
		FD_SET(wglobal.conn, &rfds);

		select(wglobal.conn+1, &rfds, NULL, NULL, NULL);
	}
}


#define CASE_EVENT(X) case X: 
/*	fprintf(stderr, "[%#lx] %s\n", ev->xany.window, #X);*/


#define SKIP_FOCUSENTER_EVENTS(EV) while(XCheckMaskEvent(wglobal.dpy, \
	EnterWindowMask|FocusChangeMask, EV)) /*nothing */;

/* 
 * Workspace switching and focusing with CF_WARP.
 * 
 * - switch_workspace sets focus (immediately) to previous active
 *   winobj on workspace.
 * - skip_focusenter should receive an enter event and focus to the
 *   window that is really wanted to have the focus.
 */

static void skip_focusenter()
{
	XEvent ev;
	XEvent tmp;
	
	tmp.type=None;
	
	if(wglobal.current_wswindow==NULL ||
	   !on_active_workspace((WThing*)wglobal.current_wswindow))
		return;

	XSync(wglobal.dpy, False);
	
	while(XCheckMaskEvent(wglobal.dpy,
						  EnterWindowMask|FocusChangeMask, &ev)){
	#ifdef CF_WARP
		if(ev.type==EnterNotify && wglobal.focus_next==NULL){
			protect_previous();
			handle_enter_window(&ev);
			unprotect_previous();
			XFlush(wglobal.dpy);
		}else
	#endif
		
		/* Have to handle last focus in or else we may not get
		 * correct colormap.
		 */
		if(ev.type==FocusIn && wglobal.focus_next==NULL)
			memcpy(&tmp, &ev, sizeof(tmp));
	}
	
	if(tmp.type!=None)
		handle_focus_in(&(tmp.xfocus));
}


void handle_event(XEvent *ev)
{
	switch(ev->type){
	CASE_EVENT(MapRequest)
		handle_map_request(&(ev->xmaprequest));
		break;
	CASE_EVENT(ConfigureRequest)
		handle_configure_request(&(ev->xconfigurerequest));
		break;
	CASE_EVENT(UnmapNotify)
		handle_unmap_notify(&(ev->xunmap));
		break;
	CASE_EVENT(DestroyNotify)
		handle_destroy_notify(&(ev->xdestroywindow));
		break;
	CASE_EVENT(ClientMessage)
		handle_client_message(&(ev->xclient));
		break;
	CASE_EVENT(PropertyNotify)
		handle_property(&(ev->xproperty));
		break;
	CASE_EVENT(FocusIn)
		handle_focus_in(&(ev->xfocus));
		break;
	CASE_EVENT(FocusOut)
		handle_focus_out(&(ev->xfocus));
		break;
	CASE_EVENT(EnterNotify)
		handle_enter_window(ev);
		break;
	CASE_EVENT(Expose)		
		handle_expose(&(ev->xexpose));
		break;
	CASE_EVENT(KeyPress)
		if(wglobal.input_mode!=INPUT_WAITRELEASE)
			keyboard_handler(ev);
		break;
	CASE_EVENT(KeyRelease)
		if(wglobal.input_mode==INPUT_WAITRELEASE){
			if(unmod(ev->xkey.state, ev->xkey.keycode)==0)
				ungrab_kb_ptr();
			skip_focusenter();
		}
		break;
	CASE_EVENT(ButtonPress)
		pointer_handler(ev);
		break;
	CASE_EVENT(ColormapNotify)
		handle_colormap_notify(&(ev->xcolormap));
		break;
	CASE_EVENT(MappingNotify)
		XRefreshKeyboardMapping(&(ev->xmapping));
		update_modmap();
		break;
	CASE_EVENT(SelectionClear)
		clear_selection();
		break;
	CASE_EVENT(SelectionNotify)
		receive_selection(&(ev->xselection));
		break;
	CASE_EVENT(SelectionRequest)
		send_selection(&(ev->xselectionrequest));
		break;
	}
}


void mainloop()
{
	XEvent ev;
	
	for(;;){
		get_event(&ev);
		handle_event(&ev);
		
		XSync(wglobal.dpy, False);

		if(wglobal.focus_next!=NULL){
			SKIP_FOCUSENTER_EVENTS(&ev);
			do_set_focus(wglobal.focus_next);
			wglobal.focus_next=NULL;
		}
	}
}


/*}}}*/


/*{{{ Map, unmap, destroy */


static void handle_map_request(const XMapRequestEvent *ev)
{
	WThing *thing;
	WScreen *scr;
	
	thing=FIND_WINDOW(ev->window);
	
	if(thing!=NULL)
		return;
	
	scr=FIND_WINDOW_T(ev->parent, WScreen);
	
	if(scr==NULL)
		return;
	
	manage_clientwin(scr, ev->window, 0);
}


static void handle_unmap_notify(const XUnmapEvent *ev)
{
	WClientWin *cwin;

	/* We are not interested in SubstructureNotify -unmaps. */
	if(ev->event!=ev->window && ev->send_event!=True)
		return;

	cwin=find_clientwin(ev->window);
	
	if(cwin==NULL)
		return;

	unmap_clientwin(cwin);
}


static void handle_destroy_notify(const XDestroyWindowEvent *ev)
{
	WClientWin *cwin;

	cwin=find_clientwin(ev->window);
	
	if(cwin==NULL)
		return;
	
	destroy_clientwin(cwin);
}


/*}}}*/


/*{{{ Client configure/property/message */


static void refit(WClientWin *cwin)
{
	WFrame *frame;
	
	frame=FIND_PARENT(cwin, WFrame);
	
	if(frame!=NULL)
		fit_clientwin_frame(cwin, frame);
}


static void handle_configure_request(XConfigureRequestEvent *ev)
{
	WClientWin *cwin;
	XWindowChanges wc;
	
	cwin=find_clientwin(ev->window);
	
	if(cwin==NULL){
		wc.border_width=ev->border_width;
		wc.sibling=None;
		wc.stack_mode=ev->detail;
		wc.x=ev->x;
		wc.y=ev->y;
		wc.width=ev->width;
		wc.height=ev->height;
		XConfigureWindow(wglobal.dpy, ev->window, ev->value_mask, &wc);
		return;
	}

	if((ev->value_mask&(CWWidth|CWHeight|CWBorderWidth))!=0){
		/* Transients will not get resized properly unless the
		 * wanted sizes are first set and later modified to fit
		 */
		if(ev->value_mask&CWWidth)
			cwin->geom.w=ev->width;
		if(ev->value_mask&CWHeight)
			cwin->geom.h=ev->height;
		if(ev->value_mask&CWBorderWidth)
			cwin->orig_bw=ev->border_width;
		refit(cwin);
	}else{
		sendconfig_clientwin(cwin);
	}
}


static void handle_client_message(const XClientMessageEvent *ev)
{
#if 0
	WClientWin *cwin;

	if(ev->message_type!=wglobal.atom_wm_change_state)
		return;
	
	cwin=find_clientwin(ev->window);

	if(cwin==NULL)
		return;
	
	if(ev->format==32 && ev->data.l[0]==IconicState){
		if(cwin->state==NormalState)
			iconify_clientwin(cwin);
	}
#endif
}


static void handle_property(const XPropertyEvent *ev)
{
	WClientWin *cwin;
	WClient *client;
	WScreen *scr;
	
	cwin=find_clientwin(ev->window);
	
	if(cwin==NULL)
		return;
	
	switch(ev->atom){
	case XA_WM_NORMAL_HINTS:
		get_clientwin_size_hints(cwin);
		/*refit(cwin);*/
		return;
	
	case XA_WM_NAME:
		/*if(cwin->name!=NULL)
			XFree((void*)cwin->name);
		cwin->name=get_string_property(cwin->win, XA_WM_NAME, NULL);*/
		set_clientwin_name(cwin, get_string_property(cwin->win, XA_WM_NAME,
													 NULL));
		break;
		
	/*case XA_WM_ICON_NAME:
		if(cwin->icon_name!=NULL)
			XFree((void*)cwin->icon_name);
		cwin->icon_name=get_string_property(cwin->win, XA_WM_ICON_NAME, NULL);
		break;*/

	case XA_WM_TRANSIENT_FOR:
		/*warn("Changes in WM_TRANSIENT_FOR property are unsupported.");*/
		scr=SCREEN_OF(cwin);
		unmap_clientwin(cwin);
		manage_clientwin(scr, ev->window, 0);
		
	default:
		if(ev->atom==wglobal.atom_wm_protocols)
			get_protocols(cwin);
		return;
	}
	
	/*client=FIND_PARENT(cwin, WClient);
	if(client!=NULL)
		client_update_label(client);*/
}


/*}}}*/


/*{{{ Colormap */


static void install_cmap(WScreen *scr, Colormap cmap)
{
	if(cmap==None)
		cmap=scr->default_cmap;
	
	XInstallColormap(wglobal.dpy, cmap);
}


static bool focused_clientwin(WClientWin *cwin)
{
	WClient *client=FIND_PARENT(cwin, WClient);
	WFrame *frame;
	
	if(client==NULL || LAST_THING(client, WClientWin)!=cwin)
		return FALSE;
	
	frame=FIND_PARENT(client, WFrame);
	
	return (frame!=NULL && frame->current_client==client &&
			IS_ACTIVE_FRAME(frame));
}


static void set_cmap(WClientWin *cwin, Colormap cmap)
{
	cwin->cmap=cmap;
	if(focused_clientwin(cwin))
		install_cmap(SCREEN_OF(cwin), cwin->cmap);
}


static void handle_colormap_notify(const XColormapEvent *ev)
{
	WClientWin *cwin;

	if(!ev->new)
		return;

	cwin=find_clientwin(ev->window);

	if(cwin!=NULL)
		set_cmap(cwin, ev->colormap);
}


/*}}}*/


/*{{{ Expose */

static void redraw_wwin(WWindow *wwin)
{
	if(WTHING_IS(wwin, WFrame))
		draw_frame((WFrame*)wwin, FALSE);
	else if(WTHING_IS(wwin, WInput))
		input_draw((WInput*)wwin, FALSE);
}


static void handle_expose(const XExposeEvent *ev)
{
	WWindow *wwin;
	WScreen *scr;
	XEvent tmp;
	
	while(XCheckWindowEvent(wglobal.dpy, ev->window, ExposureMask, &tmp))
		/* nothing */;

	wwin=FIND_WINDOW_T(ev->window, WWindow);

	if(wwin!=NULL){
		redraw_wwin(wwin);
		return;
	}
	
	if(wglobal.grab_holder==NULL || !WTHING_IS(wglobal.grab_holder, WClient))
		return;
	
	FOR_ALL_SCREENS(scr){
		if(scr->grdata.tabdrag_win==ev->window){
			draw_tabdrag((WClient*)wglobal.grab_holder);
			break;
		}
	}
}


/*}}}*/


/*{{{ Enter window, focus */


static void handle_enter_window(XEvent *ev)
{
	XEnterWindowEvent *eev=&(ev->xcrossing);
	WThing *thing=NULL;
	bool inf=TRUE;
	
	do{
		if(eev->detail!=NotifyInferior)
			inf=FALSE;
	}while(XCheckMaskEvent(wglobal.dpy, EnterWindowMask, ev));

	if(inf)
		return;

	if(eev->window==eev->root){
		/* Ignore root window enter */
		return;
	}
	
	thing=FIND_WINDOW_T(eev->window, WThing);
	
	if(thing==NULL)
		return;

	set_previous(thing);
	set_focus(thing);
}


static void activate(WWindow *wwin)
{
	set_current_wswindow(wwin);
	if(WTHING_IS(wwin, WFrame))
		activate_frame((WFrame*)wwin);
}


static void deactivate(WWindow *wwin)
{
	wglobal.current_wswindow=NULL;
	if(WTHING_IS(wwin, WFrame))
		deactivate_frame((WFrame*)wwin);
}


static void handle_focus_in(const XFocusChangeEvent *ev)
{
	WThing *thing;
	WWindow *wwin;
	WScreen *scr;
	Colormap cmap=None;

	thing=FIND_WINDOW_T(ev->window, WThing);
	
	if(thing==NULL)
		return;

	/* Set current screen */
	scr=SCREEN_OF(thing);
	wglobal.current_screen=scr;
	
	/* Handle colormap */
	if(WTHING_IS(thing, WClientWin))
		cmap=((WClientWin*)thing)->cmap;
	
	install_cmap(scr, cmap);

	/* Set active WWindow etc. */
	if(WTHING_IS(thing, WWindow)){
		wwin=(WWindow*)thing;
		if(wwin->xic!=NULL)
			XSetICFocus(wwin->xic);
	}
	
	wwin=window_of(thing);
	
	if(wglobal.current_wswindow==wwin)
		return;
	
	if(wglobal.current_wswindow!=NULL)
		deactivate(wglobal.current_wswindow);

	if(wwin!=NULL)
		activate(wwin);
}


static void handle_focus_out(const XFocusChangeEvent *ev)
{
	WWindow *wwin;

	/*if(ev->window==SCREEN->root.win){
		SCREEN->active=FALSE;
		wwin=wglobal.current_wswindow;
		if(wwin!=NULL)
			redraw_wwin(wwin);
		return;
	}*/

	wwin=FIND_WINDOW_T(ev->window, WWindow);
	
	if(wwin==NULL)
		return;
	
	if(wwin->xic!=NULL)
		XUnsetICFocus(wwin->xic);
}


/*}}}*/


/*{{{ Pointer, keyboard */


void do_grab_kb_ptr(Window win, WThing *thing)
{
	wglobal.grab_holder=thing;
	wglobal.input_mode=INPUT_GRAB;
	
	XSelectInput(wglobal.dpy, win, ROOT_MASK&~FocusChangeMask);
	XGrabPointer(wglobal.dpy, win, True, GRAB_POINTER_MASK,
				 GrabModeAsync, GrabModeAsync, win,
				 x_cursor(CURSOR_DEFAULT), CurrentTime);
	XGrabKeyboard(wglobal.dpy, win, False, GrabModeAsync,
				  GrabModeAsync, CurrentTime);
	XSelectInput(wglobal.dpy, win, ROOT_MASK);
}


void grab_kb_ptr(WThing *thing)
{
	do_grab_kb_ptr(ROOT_OF(thing), thing);
}


void ungrab_kb_ptr()
{
	XUngrabKeyboard(wglobal.dpy, CurrentTime);
	XUngrabPointer(wglobal.dpy, CurrentTime);
	
	wglobal.grab_holder=NULL;
	wglobal.input_mode=INPUT_NORMAL;
}


#define GRAB_EV_MASK (GRAB_POINTER_MASK|ExposureMask|  \
					  KeyPressMask|KeyReleaseMask|     \
					  EnterWindowMask|FocusChangeMask)
	
static void pointer_handler(XEvent *ev)
{
	do_grab_kb_ptr(ev->xbutton.root, NULL);
	
	handle_button_press(&(ev->xbutton));

	while(wglobal.input_mode!=INPUT_NORMAL){
		XFlush(wglobal.dpy);
		get_event_mask(ev, GRAB_EV_MASK);
		
		switch(ev->type){
		CASE_EVENT(ButtonRelease)
			if(handle_button_release(&(ev->xbutton)))
				ungrab_kb_ptr();
			break;
		CASE_EVENT(MotionNotify)
			handle_pointer_motion(&(ev->xmotion));
			break;
		CASE_EVENT(Expose)		
			handle_expose(&(ev->xexpose));
			break;
		}
	}
}


static void keyboard_handler(XEvent *ev)
{
	handle_keypress(&(ev->xkey));
	
	while(wglobal.input_mode!=INPUT_NORMAL &&
		  wglobal.input_mode!=INPUT_WAITRELEASE){
		XFlush(wglobal.dpy);
		get_event_mask(ev, GRAB_EV_MASK);
		
		switch(ev->type){
		CASE_EVENT(Expose)
			handle_expose(&(ev->xexpose));
			break;
		CASE_EVENT(KeyPress)
			handle_keypress(&(ev->xkey));
			break;
		}
	}
	
	skip_focusenter();
}


/*}}}*/


