-- | Formats Haskell source code as HTML with CSS.
module Language.Haskell.HsColour.CSS (hscolour) where

import Language.Haskell.HsColour.Anchors
import Language.Haskell.HsColour.Classify as Classify
import Language.Haskell.HsColour.HTML (renderAnchors, renderComment,
                                       renderNewLinesAnchors, escape)

-- | Formats Haskell source code as a complete HTML document with CSS.
hscolour :: Bool   -- ^ Whether to include anchors.
         -> Bool   -- ^ Whether output should be partial
                   --   (= no stylesheet link will be included.)
         -> String -- ^ Title for HTML page.
         -> String -- ^ Haskell source code.
         -> String -- ^ An HTML document containing the coloured 
                   --   Haskell source code.
hscolour anchor partial title =
  (if partial then id else top'n'tail title)
  . pre
  . (if anchor 
        then renderNewLinesAnchors
             . concatMap (renderAnchors renderToken)
             . insertAnchors
        else concatMap renderToken)
  . tokenise

top'n'tail :: String -> String -> String
top'n'tail title  = (cssPrefix title ++) . (++cssSuffix)

pre :: String -> String
pre = ("<pre>"++) . (++"</pre>")

renderToken :: (TokenType,String) -> String
renderToken (cls,text) =
        before ++ (if cls == Comment then renderComment text else escape text) ++ after
    where
        before = if null cls2 then "" else "<span class='" ++ cls2 ++ "'>"
        after  = if null cls2 then "" else "</span>"
        cls2 = cssClass cls


cssClass Keyword  = "keyword"
cssClass Keyglyph = "keyglyph"
cssClass Layout   = "layout"
cssClass Comment  = "comment"
cssClass Conid    = "conid"
cssClass Varid    = "varid"
cssClass Conop    = "conop"
cssClass Varop    = "varop"
cssClass String   = "str"
cssClass Char     = "chr"
cssClass Number   = "num"
cssClass Cpp      = "cpp"
cssClass Error    = "sel"
cssClass Definition = "definition"
cssClass _        = ""


cssPrefix title = unlines
    ["<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Strict//EN\" \"http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd\">"
    ,"<html>"
    ,"<head>"
    ,"<!-- Generated by HsColour, http://www.cs.york.ac.uk/fp/darcs/hscolour/ -->"
    ,"<title>"++title++"</title>"
    ,"<link type='text/css' rel='stylesheet' href='hscolour.css' />"
    ,"</head>"
    ,"<body>"
    ]
    
cssSuffix = unlines
    ["</body>"
    ,"</html>"
    ]
