<?php
// $Horde: horde/lib/Text.php,v 1.6 2001/10/06 18:28:31 chuck Exp $

/**
 * The Text:: class provides common methods for manipulating text.
 *
 * @author  Jon Parise <jon@horde.org>
 * @version $Revision: 1.6 $
 * @since   Horde 1.3
 * @package horde
 */
class Text {

    /**
     * Filter the given text based on the words found in $words.
     *
     * @param string $text         The text to filter.
     * @param string $words_file   Filename containing the words to replace.
     * @param string $replacement  The replacement string.
     *
     * @return string The filtered version of $text.
     */
    function filter($text, $words_file, $replacement)
    {
        if (@is_readable($words_file)) {
            
            /* Read the file and iterate through the lines. */
            $lines = file($words_file);
            foreach ($lines as $line) {
                
                /* Strip whitespace and comments. */
                $line = trim($line);
                $line = preg_replace('|#.*$|', '', $line);
                
                /* Filter the text. */
                if (!empty($line)) {
                    $text = preg_replace("/(\b(\w*)$line\b|\b$line(\w*)\b)/i",
                                         $replacement, $text);
                }
            }
        }
        
        return $text;
    }
    
    /**
     * Wraps the text of a message.
     *
     * @param string $text         String containing the text to wrap.
     * @param int    $length       Wrap $text at this number of characters.
     * @param string $break_char   Character to use when breaking lines.
     *
     * @return  string      String containing the wrapped text.
     */
    function wrap($text, $length = 80, $break_char = "\n")
    {
        $paragraphs = explode("\n", $text);
        for ($i = 0; $i < count($paragraphs); $i++) {
            $paragraphs[$i] = wordwrap($paragraphs[$i], $length, $break_char);
        }
        return implode($break_char, $paragraphs);
    }
    
    /**
     * Turns all URLs in the text into hyperlinks.
     *
     * @param string $text               The text to be transformed.
     * @param optional boolean $capital  Sometimes it's useful to generate <A and </A>
     *                                   so you can know which tags you just generated.
     *
     * @return string              The linked text.
     */
    function linkUrls($text, $capital = false)
    {
        if ($capital) {
            $a = 'A';
        } else {
            $a = 'a';
        }
        return preg_replace('|(\w+)://([^\s"<]*[\w+#?/&=])|', '<' . $a . ' href="\1://\2" target="_blank">\1://\2</' . $a . '>', $text);
    }
    
    /**
     * Replace occurences of %VAR% with VAR, if VAR exists in the
     * webserver's environment. Ignores all text after a # character
     * (shell-style comments).
     *
     * @param string $text	The text to expand.
     * @return string		The expanded text.
     */
    function expandEnvironment($text)
    {
        if (preg_match("|([^#]*)#.*|", $text, $regs)) {
            $text = $regs[1];
            
            if (strlen($text) > 0) {
                $text = $text . "\n";
            }
        }
        
        while (preg_match("|%([A-Za-z_]+)%|", $text, $regs)) {
            $text = preg_replace("|%([A-Za-z_]+)%|", getenv($regs[1]), $text);
        }
        return $text;
    }
    
    /**
     * Convert a line of text to display properly in HTML.
     *
     * @param string $text  The string of text to convert.
     *
     * @return string       The HTML-compliant converted text.
     */
    function htmlSpaces($text = '')
    {
        $text = htmlentities($text);
        $text = str_replace("\t", '&nbsp; &nbsp; &nbsp; &nbsp; ', $text);
        $text = str_replace('  ', '&nbsp; ', $text);
        $text = str_replace('  ', ' &nbsp;', $text);

        return $text;
    }

    /**
     * Same as htmlSpaces() but converts all spaces to &nbsp;
     * @see htmlSpaces()
     *
     * @param string $text  The string of text to convert.
     *
     * @return string       The HTML-compliant converted text.
     */
    function htmlAllSpaces($text = '')
    {
        $text = Text::htmlSpaces($text);
        $text = str_replace(' ', '&nbsp;', $text);

        return $text;
    }

}
?>
