<?php
/**
 * The Secret:: class provides an API for encrypting and decrypting
 * small pieces of data with the use of a shared key.
 *
 * The Secret:: functions use the PEAR Crypt/HCEMD5 module.
 *
 * @author  Chuck Hagenbuch <chuck@horde.org>
 * @version $Revision: 1.12.2.1 $
 * @since   Horde 1.3
 * @package horde
 */
class Secret {
    
    /**
     * Take a small piece of data and encrypt it with a key.
     * @access public
     *
     * @param string $key      The key to use for encryption.
     * @param string $message  The plaintext message.
     *
     * @return string  The ciphertext message.
     */
    function write($key, $message)
    {
        if (function_exists('mcrypt_is_broken')) {
            Secret::srand();
            $td = @mcrypt_module_open(MCRYPT_GOST, '', MCRYPT_MODE_ECB, '');
            if ($td) {
                $iv = mcrypt_create_iv(mcrypt_enc_get_iv_size($td), MCRYPT_RAND);
                @mcrypt_generic_init($td, $key, $iv);
                $encrypted_data = mcrypt_generic($td, $message);
                mcrypt_generic_end($td);
                
                return $encrypted_data;
            }
        }
        
        include_once 'Crypt/HCEMD5.php';
        
        $hcemd5 = new Crypt_HCEMD5($key);
        return $hcemd5->encodeMimeSelfRand($message);
    }
    
    /**
     * Decrypt a message encrypted with Secret::write().
     * @access public
     *
     * @param string $key      The key to use for decryption.
     * @param string $message  The ciphertext message.
     *
     * @return string  The plaintext message.
     */
    function read($key, $ciphertext)
    {
        if (function_exists('mcrypt_is_broken')) {
            Secret::srand();
            $td = @mcrypt_module_open(MCRYPT_GOST, '', MCRYPT_MODE_ECB, '');
            if ($td) {
                $iv = mcrypt_create_iv(mcrypt_enc_get_iv_size($td), MCRYPT_RAND);
                @mcrypt_generic_init($td, $key, $iv);
                $decrypted_data = mdecrypt_generic($td, $ciphertext);
                mcrypt_generic_end($td);
                
                return $decrypted_data;
            }
        }
        
        include_once 'Crypt/HCEMD5.php';
        
        $hcemd5 = new Crypt_HCEMD5($key);
        return $hcemd5->decodeMimeSelfRand($ciphertext);
    }
    
    /**
     * Generate a secret key (for encryption), either using a random
     * md5 string and storing it in a cookie if the user has cookies
     * enabled, or munging some known values if they don't.
     *
     * @access public
     * @param optional string $keyname  The name of the key to set. Defaults to 'generic'.
     * @return string The secret key that has been generated.
     */
    function setKey($keyname = 'generic')
    {
        if (isset($GLOBALS['HTTP_COOKIE_VARS']) && (count($GLOBALS['HTTP_COOKIE_VARS']) > 0)) {
            if (isset($GLOBALS['HTTP_COOKIE_VARS'][$keyname . '_key'])) {
                $key = $GLOBALS['HTTP_COOKIE_VARS'][$keyname . '_key'];
            } else {
                Secret::srand();
                $key = md5(uniqid(mt_rand()));
                $GLOBALS['HTTP_COOKIE_VARS'][$keyname . '_key'] = $key;
                setcookie($keyname . '_key', $key, null, $GLOBALS['registry']->getParam('cookie_path'), $GLOBALS['registry']->getParam('cookie_domain'));
            }
        } else {
            $key = md5(session_id() . $GLOBALS['registry']->getParam('server_name'));
        }
        
        return $key;
    }
    
    /**
     * Return a secret key, either from a cookie, or if the cookie
     * isn't there, assume we are using a munged version of a known
     * base value.
     *
     * @access public
     * @param optional string $keyname  The name of the key to get. Defaults to 'generic'.
     * @return string The secret key.
     */
    function getKey($keyname = 'generic')
    {
        if (isset($GLOBALS['HTTP_COOKIE_VARS'][$keyname . '_key'])) {
            return $GLOBALS['HTTP_COOKIE_VARS'][$keyname . '_key'];
        } else {
            return md5(session_id() . $GLOBALS['registry']->getParam('server_name'));
        }
    }
    
    /**
     * Ensure that the random number generator is initialized only once.
     * @access private
     */
    function srand()
    {
        static $initialized;
        
        if (empty($initialized)) {
            mt_srand((double)microtime() * 1000000);
            $initialized = true;
        }
    }
    
}
?>
