/*
 * Copyright (c) 2004 Niels Provos <provos@citi.umich.edu>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <sys/types.h>
#include <sys/param.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <sys/tree.h>

#include <stdio.h>
#include <stdlib.h>
#include <err.h>
#include <string.h>
#include <ctype.h>

#include <dnet.h>

#include "ethernet.h"

struct etherindex {
	SPLAY_ENTRY(etherindex) node;

	char *index_word;

	struct ethernetcode **list;
	size_t list_size;
	size_t list_mem;
};

struct ethernetcode {
	uint32_t prefix;
	char *vendor;
	int count;
};

static SPLAY_HEAD(ethertree, etherindex) etherroot;

static int
compare(struct etherindex *a, struct etherindex *b)
{
	return (strcmp(a->index_word, b->index_word));
}

SPLAY_PROTOTYPE(ethertree, etherindex, node, compare);

SPLAY_GENERATE(ethertree, etherindex, node, compare);

/*
 * These codes have been obtained from
 *
 *   http://www.cavebear.com/CaveBear/Ethernet/vendor.html
 *
 * and other random sources on the Internet.
 */

static struct ethernetcode codes[] = {
	{ 0x000000, "xerox corporation" },
	{ 0x000001, "superlan-2u" },
	{ 0x000002, "bbn (was internal usage only, no longer used)" },
	{ 0x000003, "xerox corporation" },
	{ 0x000004, "xerox corporation" },
	{ 0x000005, "xerox corporation" },
	{ 0x000006, "xerox corporation" },
	{ 0x000007, "xerox corporation" },
	{ 0x000008, "xerox corporation" },
	{ 0x000009, "powerpipes?" },
	{ 0x00000a, "omron tateisi electronics co." },
	{ 0x00000b, "matrix corporation" },
	{ 0x00000c, "cisco" },
	{ 0x00000d, "fibronics ltd." },
	{ 0x00000e, "fujitsu" },
	{ 0x00000f, "next" },
	{ 0x000010, "hughes" },
	{ 0x000011, "tektrnix" },
	{ 0x000012, "information technology limited" },
	{ 0x000013, "camex" },
	{ 0x000014, "netronix" },
	{ 0x000015, "datapoint corporation" },
	{ 0x000016, "du pont pixel systems" },
	{ 0x000017, "tekelec" },
	{ 0x000018, "webster" },
	{ 0x000019, "applied dynamics international" },
	{ 0x00001a, "amd" },
	{ 0x00001b, "novell" },
	{ 0x00001c, "jdr microdevices generic, ne2000 drivers" },
	{ 0x00001d, "ctron" },
	{ 0x00001e, "telsist industria electronica" },
	{ 0x00001f, "cryptall communications corp." },
	{ 0x000020, "diab" },
	{ 0x000021, "sc&c" },
	{ 0x000022, "visual technology" },
	{ 0x000023, "abb automation ab, dept. q" },
	{ 0x000024, "olicom" },
	{ 0x000025, "ramtek corp." },
	{ 0x000026, "sha-ken co., ltd." },
	{ 0x000027, "japan radio company" },
	{ 0x000028, "prodigy systems corporation" },
	{ 0x000029, "imc" },
	{ 0x00002a, "trw" },
	{ 0x00002b, "crisp automation, inc" },
	{ 0x00002c, "nrc - network resources corporation - multigate hub1+, hub2, etc" },
	{ 0x00002d, "chromatics inc" },
	{ 0x00002e, "societe evira" },
	{ 0x00002f, "timeplex inc." },
	{ 0x000030, "vg laboratory systems ltd" },
	{ 0x000031, "qpsx communications pty ltd" },
	{ 0x000032, "gpt limited (reassigned from gec computers ltd)" },
	{ 0x000033, "egan machinery company" },
	{ 0x000034, "network resources corporation" },
	{ 0x000035, "spectragraphics corporation" },
	{ 0x000036, "atari corporation" },
	{ 0x000037, "oxford metrics ltd" },
	{ 0x000038, "css labs" },
	{ 0x000039, "intel" },
	{ 0x00003a, "chyron corporation" },
	{ 0x00003b, "hyundai/axil sun clones" },
	{ 0x00003c, "auspex" },
	{ 0x00003d, "at&t" },
	{ 0x00003e, "simpact" },
	{ 0x00003f, "syntrex inc" },
	{ 0x000040, "applicon, inc." },
	{ 0x000041, "ice corporation" },
	{ 0x000042, "metier management systems ltd." },
	{ 0x000043, "micro technology" },
	{ 0x000044, "castell" },
	{ 0x000045, "ford aerospace & comm. corp." },
	{ 0x000046, "isc-br" },
	{ 0x000047, "nicolet instruments corp." },
	{ 0x000048, "epson" },
	{ 0x000049, "apricot" },
	{ 0x00004a, "adc codenoll technology corp." },
	{ 0x00004b, "apt" },
	{ 0x00004c, "nec" },
	{ 0x00004d, "dci corporation" },
	{ 0x00004e, "ampex corporation" },
	{ 0x00004f, "logicraft 386-ware p.c. emulator" },
	{ 0x000050, "radisys corporation" },
	{ 0x000051, "hob electronic gmbh & co. kg" },
	{ 0x000052, "optical data systems" },
	{ 0x000053, "compucorp" },
	{ 0x000054, "modicon, inc." },
	{ 0x000055, "at&t" },
	{ 0x000056, "dr. b. struck" },
	{ 0x000057, "scitex corporation ltd." },
	{ 0x000058, "racore computer products inc" },
	{ 0x000059, "hellige gmbh" },
	{ 0x00005a, "sk (schneider & koch in europe and syskonnect outside of europe)" },
	{ 0x00005b, "eltec" },
	{ 0x00005c, "telematics international inc." },
	{ 0x00005d, "rce" },
	{ 0x00005e, "u.s. department of defense (iana)" },
	{ 0x00005f, "sumitomo" },
	{ 0x000060, "kontron elektronik gmbh" },
	{ 0x000061, "gateway communications" },
	{ 0x000062, "honeywell" },
	{ 0x000063, "hp" },
	{ 0x000064, "yokogawa digital computer corp" },
	{ 0x000065, "network general" },
	{ 0x000066, "talaris" },
	{ 0x000067, "soft * rite, inc." },
	{ 0x000068, "rosemount controls" },
	{ 0x000069, "sgi" },
	{ 0x00006a, "computer consoles inc." },
	{ 0x00006b, "mips" },
	{ 0x00006d, "case" },
	{ 0x00006e, "artisoft, inc." },
	{ 0x00006f, "madge networks ltd. token-ring adapters" },
	{ 0x000070, "hcl limited" },
	{ 0x000071, "adra systems inc." },
	{ 0x000072, "miniware technology" },
	{ 0x000073, "dupont" },
	{ 0x000074, "ricoh company ltd." },
	{ 0x000075, "bell northern research (bnr)" },
	{ 0x000076, "abekas video system" },
	{ 0x000077, "interphase [used in other systems, e.g. mips, motorola]" },
	{ 0x000078, "labtam australia" },
	{ 0x000079, "networth incorporated [bought by compaq, used in netelligent series]" },
	{ 0x00007a, "ardent" },
	{ 0x00007b, "research machines" },
	{ 0x00007c, "ampere incorporated" },
	{ 0x00007d, "cray" },
	{ 0x00007e, "netframe multiprocessor network servers" },
	{ 0x00007f, "linotype-hell ag linotronic typesetters" },
	{ 0x000080, "cray communications (formerly dowty network services) [also shows as harris (3m) (new) and/or imagen(?) elsewhere]" },
	{ 0x000081, "synoptics" },
	{ 0x000082, "lectra systemes sa" },
	{ 0x000083, "tadpole technology  [had optical data systems which is wrong according to both]" },
	{ 0x000084, "aquila (?), adi systems inc.(?)" },
	{ 0x000085, "canon inc." },
	{ 0x000086, "gateway communications inc. (then megahertz & now 3com)" },
	{ 0x000087, "hitachi" },
	{ 0x000088, "computer network tech. corp." },
	{ 0x000089, "cayman" },
	{ 0x00008a, "datahouse information systems" },
	{ 0x00008b, "infotron" },
	{ 0x00008c, "alloy computer products, inc." },
	{ 0x00008d, "verdix corporation" },
	{ 0x00008e, "solbourne(?), jupiter(?) (i've had confirming mail on solbourne)" },
	{ 0x00008f, "raytheon company" },
	{ 0x000090, "microcom" },
	{ 0x000091, "anritsu corporation" },
	{ 0x000092, "unisys, cogent (both reported)" },
	{ 0x000093, "proteon" },
	{ 0x000094, "asante" },
	{ 0x000095, "sony" },
	{ 0x000096, "marconi electronics ltd." },
	{ 0x000097, "epoch" },
	{ 0x000098, "cross com" },
	{ 0x000099, "memorex telex corporations" },
	{ 0x00009a, "rc computer a/s" },
	{ 0x00009b, "information international, inc" },
	{ 0x00009c, "rolm mil-spec computers" },
	{ 0x00009d, "locus computing corporation" },
	{ 0x00009e, "marli s.a." },
	{ 0x00009f, "ameristar technology" },
	{ 0x0000a0, "sanyo" },
	{ 0x0000a1, "marquette electric co." },
	{ 0x0000a2, "wellfleet" },
	{ 0x0000a3, "nat" },
	{ 0x0000a4, "acorn" },
	{ 0x0000a5, "csc" },
	{ 0x0000a6, "network general (internal assignment, not for products)" },
	{ 0x0000a7, "ncd" },
	{ 0x0000a8, "stratus" },
	{ 0x0000a9, "network systems" },
	{ 0x0000aa, "xerox" },
	{ 0x0000ab, "logic modeling corporation" },
	{ 0x0000ac, "conware netzpartner [had apollo, claimed incorrect]" },
	{ 0x0000ad, "bruker instruments inc." },
	{ 0x0000ae, "dassault" },
	{ 0x0000af, "nuclear data acquisition interface modules (aim)" },
	{ 0x0000b0, "rnd" },
	{ 0x0000b1, "alpha microsystems inc." },
	{ 0x0000b2, "televideo systems, inc." },
	{ 0x0000b3, "cimlinc" },
	{ 0x0000b4, "edimax" },
	{ 0x0000b5, "datability terminal servers" },
	{ 0x0000b6, "micro-matic research" },
	{ 0x0000b7, "dove fastnet" },
	{ 0x0000b8, "seikosha co., ltd." },
	{ 0x0000b9, "mcdonnell douglas computer sys" },
	{ 0x0000ba, "siig, inc." },
	{ 0x0000bb, "tri-data systems inc. netway products, 3274 emulators" },
	{ 0x0000bc, "allen-bradley" },
	{ 0x0000bd, "mitsubishi cable company" },
	{ 0x0000be, "the nti group" },
	{ 0x0000bf, "symmetric computer systems" },
	{ 0x0000c0, "western digital now smc (std. microsystems corp.)" },
	{ 0x0000c1, "olicom a/s" },
	{ 0x0000c2, "information presentation tech." },
	{ 0x0000c3, "harris corp computer sys div" },
	{ 0x0000c4, "waters div. of millipore" },
	{ 0x0000c5, "farallon computing inc" },
	{ 0x0000c6, "hp intelligent networks operation (formerly eon systems)" },
	{ 0x0000c7, "arix corporation" },
	{ 0x0000c8, "altos" },
	{ 0x0000c9, "emulex terminal servers, print servers" },
	{ 0x0000ca, "lancity cable modems (now owned by baynetworks)" },
	{ 0x0000cb, "compu-shack electronic gmbh" },
	{ 0x0000cc, "densan co., ltd." },
	{ 0x0000cd, "industrial research limited" },
	{ 0x0000ce, "megadata corp." },
	{ 0x0000cf, "hayes microcomputer products" },
	{ 0x0000d0, "develcon electronics, ltd." },
	{ 0x0000d1, "adaptec, inc. nodem product" },
	{ 0x0000d2, "sbe inc" },
	{ 0x0000d3, "wang labs" },
	{ 0x0000d4, "puredata" },
	{ 0x0000d5, "micrognosis international" },
	{ 0x0000d6, "punch line holding" },
	{ 0x0000d7, "dartmouth college (ned router)" },
	{ 0x0000d8, "old novell ne1000's (before about 1987?) (also 3com)" },
	{ 0x0000d9, "nippon telegraph & telephone" },
	{ 0x0000da, "atex" },
	{ 0x0000db, "british telecommunications plc" },
	{ 0x0000dc, "hayes microcomputer products" },
	{ 0x0000dd, "gould" },
	{ 0x0000de, "unigraph" },
	{ 0x0000df, "bell & howell pub sys div" },
	{ 0x0000e0, "quadram corp." },
	{ 0x0000e1, "hitachi" },
	{ 0x0000e2, "acer counterpoint" },
	{ 0x0000e3, "integrated micro products ltd" },
	{ 0x0000e4, "mips?" },
	{ 0x0000e5, "sigmex ltd." },
	{ 0x0000e6, "aptor produits de comm indust" },
	{ 0x0000e7, "star gate technologies" },
	{ 0x0000e8, "accton technology corporation" },
	{ 0x0000e9, "isicad, inc." },
	{ 0x0000ea, "upnod ab" },
	{ 0x0000eb, "matsushita comm. ind. co. ltd." },
	{ 0x0000ec, "microprocess" },
	{ 0x0000ed, "april" },
	{ 0x0000ee, "network designers limited [also knx ltd, a former division]" },
	{ 0x0000ef, "alantec (now owned by foresystems)" },
	{ 0x0000f0, "samsung" },
	{ 0x0000f1, "magna computer corporation" },
	{ 0x0000f2, "spider communications (montreal, not spider systems)" },
	{ 0x0000f3, "gandalf data ltd. - canada" },
	{ 0x0000f4, "allied telesis, inc." },
	{ 0x0000f5, "diamond sales limited" },
	{ 0x0000f6, "madge" },
	{ 0x0000f7, "youth keep enterprise co ltd" },
	{ 0x0000f8, "dec" },
	{ 0x0000f9, "quotron systems inc." },
	{ 0x0000fa, "microsage computer systems inc" },
	{ 0x0000fb, "rechner zur kommunikation" },
	{ 0x0000fc, "meiko" },
	{ 0x0000fd, "high level hardware (orion, uk)" },
	{ 0x0000fe, "annapolis micro systems" },
	{ 0x0000ff, "camtec electronics (uk) ltd." },
	{ 0x000100, "equip'trans" },
	{ 0x000102, "bbn (bolt beranek and newman, inc.) internal usage (not registered)" },
	{ 0x000103, "3com corporation" },
	{ 0x000104, "dvico co., ltd." },
	{ 0x000105, "beckhoff gmbh" },
	{ 0x000106, "tews datentechnik gmbh" },
	{ 0x000107, "leiser gmbh" },
	{ 0x000108, "avlab technology, inc." },
	{ 0x000109, "nagano japan radio co., ltd." },
	{ 0x00010a, "cis technology inc." },
	{ 0x00010b, "space cyberlink, inc." },
	{ 0x00010c, "system talks inc." },
	{ 0x00010d, "coreco, inc." },
	{ 0x00010e, "bri-link technologies co., ltd" },
	{ 0x00010f, "nishan systems, inc." },
	{ 0x000110, "gotham networks" },
	{ 0x000111, "idigm inc." },
	{ 0x000112, "shark multimedia inc." },
	{ 0x000113, "olympus optical co., ltd." },
	{ 0x000114, "kanda tsushin kogyo co., ltd." },
	{ 0x000115, "extratech corporation" },
	{ 0x000116, "netspect technologies, inc." },
	{ 0x000117, "canal +" },
	{ 0x000118, "ez digital co., ltd." },
	{ 0x000119, "action controls pty. ltd." },
	{ 0x00011a, "eeh datalink gmbh" },
	{ 0x00011b, "unizone technologies, inc." },
	{ 0x00011c, "universal talkware corporation" },
	{ 0x00011d, "centillium communications" },
	{ 0x00011e, "precidia technologies, inc." },
	{ 0x00011f, "rc networks, inc." },
	{ 0x000120, "oscilloquartz s.a." },
	{ 0x000121, "rapidstream inc." },
	{ 0x000122, "trend communications, ltd." },
	{ 0x000123, "digital electronics corp." },
	{ 0x000124, "acer incorporated" },
	{ 0x000125, "yaesu musen co., ltd." },
	{ 0x000126, "pac labs" },
	{ 0x000127, "the open group limited" },
	{ 0x000128, "enjoyweb, inc." },
	{ 0x000129, "dfi inc." },
	{ 0x00012a, "telematica sistems inteligente" },
	{ 0x00012b, "telenet co., ltd." },
	{ 0x00012c, "aravox technologies, inc." },
	{ 0x00012d, "komodo technology" },
	{ 0x00012e, "pc partner ltd." },
	{ 0x00012f, "twinhead international corp" },
	{ 0x000130, "extreme networks" },
	{ 0x000131, "detection systems, inc." },
	{ 0x000132, "dranetz - bmi" },
	{ 0x000133, "kyowa electronic instruments c" },
	{ 0x000134, "sig positec systems ag" },
	{ 0x000135, "kdc corp." },
	{ 0x000136, "cybertan technology, inc." },
	{ 0x000137, "it farm corporation" },
	{ 0x000138, "xavi technologies corp." },
	{ 0x000139, "point multimedia systems" },
	{ 0x00013a, "shelcad communications, ltd." },
	{ 0x00013b, "bna systems" },
	{ 0x00013c, "tiw systems" },
	{ 0x00013d, "riscstation ltd." },
	{ 0x00013e, "ascom tateco ab" },
	{ 0x00013f, "neighbor world co., ltd." },
	{ 0x000140, "sendtek corporation" },
	{ 0x000141, "cable print" },
	{ 0x000142, "cisco systems, inc." },
	{ 0x000143, "ieee 802" },
	{ 0x000144, "cereva networks, inc." },
	{ 0x000145, "winsystems, inc." },
	{ 0x000146, "tesco controls, inc." },
	{ 0x000147, "zhone technologies" },
	{ 0x000148, "x-traweb inc." },
	{ 0x000149, "t.d.t. transfer data test gmbh" },
	{ 0x00014a, "sony computer science labs., i" },
	{ 0x00014b, "ennovate networks, inc." },
	{ 0x00014c, "berkeley process control" },
	{ 0x00014d, "shin kin enterprises co., ltd" },
	{ 0x00014e, "win enterprises, inc." },
	{ 0x00014f, "luminous networks, inc." },
	{ 0x000150, "megahertz (now 3com) modem" },
	{ 0x000151, "ensemble communications" },
	{ 0x000152, "chromatek inc." },
	{ 0x000153, "archtek telecom corporation" },
	{ 0x000154, "g3m corporation" },
	{ 0x000155, "promise technology, inc." },
	{ 0x000156, "firewiredirect.com, inc." },
	{ 0x000157, "syswave co., ltd" },
	{ 0x000158, "electro industries/gauge tech" },
	{ 0x000159, "s1 corporation" },
	{ 0x00015a, "digital video broadcasting" },
	{ 0x00015b, "italtel s.p.a/rf-up-i" },
	{ 0x00015c, "cadant inc." },
	{ 0x00015d, "pirus networks" },
	{ 0x00015e, "best technology co., ltd." },
	{ 0x00015f, "digital design gmbh" },
	{ 0x000160, "elmex co., ltd." },
	{ 0x000161, "meta machine technology" },
	{ 0x000162, "cygnet technologies, inc." },
	{ 0x000163, "ndc (national datacomm corporation)" },
	{ 0x000164, "cisco systems, inc." },
	{ 0x000165, "airswitch corporation" },
	{ 0x000166, "tc group a/s" },
	{ 0x000167, "hioki e.e. corporation" },
	{ 0x000168, "w&g (wandel & goltermann) [incorrect according to w&g]" },
	{ 0x000169, "celestix networks pte ltd." },
	{ 0x00016a, "alitec" },
	{ 0x00016b, "lightchip, inc." },
	{ 0x00016c, "foxconn" },
	{ 0x00016d, "triton network systems" },
	{ 0x00016e, "conklin corporation" },
	{ 0x00016f, "haitai electronics co., ltd." },
	{ 0x000170, "ese embedded system engineer'g" },
	{ 0x000171, "allied data technologies" },
	{ 0x000172, "technoland co., ltd." },
	{ 0x000173, "jni corporation" },
	{ 0x000174, "cyberoptics corporation" },
	{ 0x000175, "radiant communications corp." },
	{ 0x000176, "orient silver enterprises" },
	{ 0x000177, "edsl" },
	{ 0x000178, "margi systems, inc." },
	{ 0x000179, "wireless technology, inc." },
	{ 0x00017a, "chengdu maipu electric industrial co., ltd." },
	{ 0x00017b, "heidelberger druckmaschinen ag" },
	{ 0x00017c, "ag-e gmbh" },
	{ 0x00017d, "thermoquest" },
	{ 0x00017e, "adtek system science co., ltd." },
	{ 0x00017f, "experience music project" },
	{ 0x000180, "aopen, inc." },
	{ 0x000181, "nortel networks" },
	{ 0x000182, "dica technologies ag" },
	{ 0x000183, "anite telecoms" },
	{ 0x000184, "sieb & meyer ag" },
	{ 0x000185, "aloka co., ltd." },
	{ 0x000186, "disch gmbh" },
	{ 0x000187, "i2se gmbh" },
	{ 0x000188, "lxco technologies ag" },
	{ 0x000189, "refraction technology, inc." },
	{ 0x00018a, "roi computer ag" },
	{ 0x00018b, "netlinks co., ltd." },
	{ 0x00018c, "mega vision" },
	{ 0x00018d, "audesi technologies" },
	{ 0x00018e, "logitec corporation" },
	{ 0x00018f, "kenetec, inc." },
	{ 0x000190, "smk-m" },
	{ 0x000191, "syred data systems" },
	{ 0x000192, "texas digital systems" },
	{ 0x000193, "hanbyul telecom co., ltd." },
	{ 0x000194, "capital equipment corporation" },
	{ 0x000195, "sena technologies, inc." },
	{ 0x000196, "cisco systems, inc." },
	{ 0x000197, "cisco systems, inc." },
	{ 0x000198, "darim vision" },
	{ 0x000199, "heisei electronics" },
	{ 0x00019a, "leunig gmbh" },
	{ 0x00019b, "kyoto microcomputer co., ltd." },
	{ 0x00019c, "jds uniphase inc." },
	{ 0x00019d, "e-control systems, inc." },
	{ 0x00019e, "ess technology, inc." },
	{ 0x00019f, "phonex broadband" },
	{ 0x0001a0, "infinilink corporation" },
	{ 0x0001a1, "mag-tek, inc." },
	{ 0x0001a2, "logical co., ltd." },
	{ 0x0001a3, "genesys logic, inc." },
	{ 0x0001a4, "microlink corporation" },
	{ 0x0001a5, "nextcomm, inc." },
	{ 0x0001a6, "scientific-atlanta arcodan a/s" },
	{ 0x0001a7, "unex technology corporation" },
	{ 0x0001a8, "welltech computer co., ltd." },
	{ 0x0001a9, "bmw ag" },
	{ 0x0001aa, "airspan communications, ltd." },
	{ 0x0001ab, "main street networks" },
	{ 0x0001ac, "sitara networks, inc." },
	{ 0x0001ad, "coach master international  d.b.a. cmi worldwide, inc." },
	{ 0x0001ae, "trex enterprises" },
	{ 0x0001af, "motorola computer group" },
	{ 0x0001b0, "fulltek technology co., ltd." },
	{ 0x0001b1, "general bandwidth" },
	{ 0x0001b2, "digital processing systems, inc." },
	{ 0x0001b3, "precision electronic manufacturing" },
	{ 0x0001b4, "wayport, inc." },
	{ 0x0001b5, "turin networks, inc." },
	{ 0x0001b6, "saejin t&m co., ltd." },
	{ 0x0001b7, "centos, inc." },
	{ 0x0001b8, "netsensity, inc." },
	{ 0x0001b9, "skf condition monitoring" },
	{ 0x0001ba, "ic-net, inc." },
	{ 0x0001bb, "frequentis" },
	{ 0x0001bc, "brains corporation" },
	{ 0x0001bd, "peterson electro-musical products, inc." },
	{ 0x0001be, "gigalink co., ltd." },
	{ 0x0001bf, "teleforce co., ltd." },
	{ 0x0001c0, "compulab, ltd." },
	{ 0x0001c1, "exbit technology" },
	{ 0x0001c2, "ark research corp." },
	{ 0x0001c3, "acromag, inc." },
	{ 0x0001c4, "neowave, inc." },
	{ 0x0001c5, "simpler networks" },
	{ 0x0001c6, "quarry technologies" },
	{ 0x0001c7, "cisco systems, inc." },
	{ 0x0001c8, "thomas conrad corp." },
	{ 0x0001c9, "cisco systems, inc." },
	{ 0x0001ca, "geocast network systems, inc." },
	{ 0x0001cb, "netgame, ltd." },
	{ 0x0001cc, "japan total design communication co., ltd." },
	{ 0x0001cd, "artem" },
	{ 0x0001ce, "custom micro products, ltd." },
	{ 0x0001cf, "alpha data parallel systems, ltd." },
	{ 0x0001d0, "vitalpoint, inc." },
	{ 0x0001d1, "conet communications, inc." },
	{ 0x0001d2, "macpower peripherals, ltd." },
	{ 0x0001d3, "paxcomm, inc." },
	{ 0x0001d4, "leisure time, inc." },
	{ 0x0001d5, "haedong info & comm co., ltd" },
	{ 0x0001d6, "man roland druckmaschinen ag" },
	{ 0x0001d7, "f5 networks, inc." },
	{ 0x0001d8, "teltronics, inc." },
	{ 0x0001d9, "sigma, inc." },
	{ 0x0001da, "wincomm corporation" },
	{ 0x0001db, "freecom technologies gmbh" },
	{ 0x0001dc, "activetelco" },
	{ 0x0001dd, "avail networks" },
	{ 0x0001de, "trango systems, inc." },
	{ 0x0001df, "isdn communications, ltd." },
	{ 0x0001e0, "fast systems, inc." },
	{ 0x0001e1, "kinpo electronics, inc." },
	{ 0x0001e2, "ando electric corporation" },
	{ 0x0001e3, "siemens ag" },
	{ 0x0001e4, "sitera, inc." },
	{ 0x0001e5, "supernet, inc." },
	{ 0x0001e6, "hewlett-packard company" },
	{ 0x0001e7, "hewlett-packard company" },
	{ 0x0001e8, "force10 networks, inc." },
	{ 0x0001e9, "litton marine systems b.v." },
	{ 0x0001ea, "cirilium corp." },
	{ 0x0001eb, "c-com corporation" },
	{ 0x0001ec, "ericsson group" },
	{ 0x0001ed, "seta corp." },
	{ 0x0001ee, "comtrol europe, ltd." },
	{ 0x0001ef, "camtel technology corp." },
	{ 0x0001f0, "tridium, inc." },
	{ 0x0001f1, "innovative concepts, inc." },
	{ 0x0001f3, "qps, inc." },
	{ 0x0001f4, "enterasys networks" },
	{ 0x0001f5, "erim s.a." },
	{ 0x0001f6, "association of musical electronics industry" },
	{ 0x0001f7, "image display systems, inc." },
	{ 0x0001f8, "adherent systems, ltd." },
	{ 0x0001f9, "teraglobal communications corp." },
	{ 0x0001fa, "compaq" },
	{ 0x0001fb, "dotop technology, inc." },
	{ 0x0001fc, "keyence corporation" },
	{ 0x0001fd, "digital voice systems, inc." },
	{ 0x0001fe, "digital equipment corporation" },
	{ 0x0001ff, "data direct networks, inc." },
	{ 0x000200, "net & sys co., ltd." },
	{ 0x000201, "ifm electronic gmbh" },
	{ 0x000202, "amino communications, ltd." },
	{ 0x000203, "woonsang telecom, inc." },
	{ 0x000204, "novell" },
	{ 0x000205, "hamilton (sparc clones)" },
	{ 0x000206, "telital r&d denmark a/s" },
	{ 0x000208, "unify networks, inc." },
	{ 0x000209, "shenzhen sed information technology co., ltd." },
	{ 0x00020a, "gefran spa" },
	{ 0x00020b, "native networks, inc." },
	{ 0x00020c, "metro-optix" },
	{ 0x00020d, "micronpc.com" },
	{ 0x00020e, "laurel networks, inc." },
	{ 0x00020f, "aatr" },
	{ 0x000210, "fenecom" },
	{ 0x000211, "nature worldwide technology corp." },
	{ 0x000212, "sierracom" },
	{ 0x000213, "s.d.e.l." },
	{ 0x000214, "dtvro" },
	{ 0x000215, "cotas computer technology a/b" },
	{ 0x000216, "esi (extended systems, inc) print servers" },
	{ 0x000217, "cisco systems, inc." },
	{ 0x000218, "advanced scientific corp" },
	{ 0x000219, "paralon technologies" },
	{ 0x00021a, "zuma networks" },
	{ 0x00021b, "kollmorgen-servotronix" },
	{ 0x00021c, "network elements, inc." },
	{ 0x00021d, "data general communication ltd." },
	{ 0x00021e, "simtel s.r.l." },
	{ 0x00021f, "aculab plc" },
	{ 0x000220, "canon aptex, inc." },
	{ 0x000221, "dsp application, ltd." },
	{ 0x000222, "chromisys, inc." },
	{ 0x000223, "clicktv" },
	{ 0x000224, "lantern communications, inc." },
	{ 0x000225, "certus technology, inc." },
	{ 0x000226, "xesystems, inc." },
	{ 0x000227, "esd gmbh" },
	{ 0x000228, "necsom, ltd." },
	{ 0x000229, "adtec corporation" },
	{ 0x00022a, "asound electronic" },
	{ 0x00022b, "tamura electric works, ltd." },
	{ 0x00022c, "abb bomem, inc." },
	{ 0x00022d, "agere systems" },
	{ 0x00022e, "teac corp. r& d" },
	{ 0x00022f, "p-cube, ltd." },
	{ 0x000230, "intersoft electronics" },
	{ 0x000231, "axis" },
	{ 0x000232, "avision, inc." },
	{ 0x000233, "mantra communications, inc." },
	{ 0x000234, "imperial technology, inc." },
	{ 0x000235, "paragon networks international" },
	{ 0x000236, "init gmbh" },
	{ 0x000237, "cosmo research corp." },
	{ 0x000238, "serome technology, inc." },
	{ 0x000239, "visicom" },
	{ 0x00023a, "zsk stickmaschinen gmbh" },
	{ 0x00023b, "redback networks" },
	{ 0x00023c, "creative technology, ltd." },
	{ 0x00023d, "nuspeed, inc." },
	{ 0x00023e, "selta telematica s.p.a" },
	{ 0x00023f, "compal electronics, inc." },
	{ 0x000240, "seedek co., ltd." },
	{ 0x000241, "amer.com" },
	{ 0x000242, "videoframe systems" },
	{ 0x000243, "raysis co., ltd." },
	{ 0x000244, "surecom technology co." },
	{ 0x000245, "lampus co, ltd." },
	{ 0x000246, "all-win tech co., ltd." },
	{ 0x000247, "great dragon information technology (group) co., ltd." },
	{ 0x000248, "pila gmbh & co." },
	{ 0x000249, "aviv infocom co, ltd." },
	{ 0x00024a, "cisco systems, inc." },
	{ 0x00024b, "cisco systems, inc." },
	{ 0x00024c, "sibyte, inc." },
	{ 0x00024d, "mannesman dematic colby pty. ltd." },
	{ 0x00024e, "datacard group" },
	{ 0x00024f, "ipm datacom s.r.l." },
	{ 0x000250, "geyser networks, inc." },
	{ 0x000251, "soma networks" },
	{ 0x000252, "carrier corporation" },
	{ 0x000253, "televideo, inc." },
	{ 0x000254, "worldgate" },
	{ 0x000255, "ibm corporation" },
	{ 0x000256, "alpha processor, inc." },
	{ 0x000257, "microcom corp." },
	{ 0x000258, "flying packets communications" },
	{ 0x000259, "tsann kuen china (shanghai)enterprise co., ltd. it group" },
	{ 0x00025a, "catena networks" },
	{ 0x00025b, "cambridge silicon radio" },
	{ 0x00025c, "sci systems (kunshan) co., ltd." },
	{ 0x00025e, "high technology ltd" },
	{ 0x00025f, "nortel networks" },
	{ 0x000260, "accordion networks, inc." },
	{ 0x000261, "i3 micro technology ab" },
	{ 0x000262, "soyo group soyo com tech co., ltd" },
	{ 0x000263, "ups manufacturing srl" },
	{ 0x000264, "audioramp.com" },
	{ 0x000265, "virditech co. ltd." },
	{ 0x000266, "thermalogic corporation" },
	{ 0x000267, "node runner, inc." },
	{ 0x000268, "harris government communications" },
	{ 0x000269, "nadatel co., ltd" },
	{ 0x00026a, "cocess telecom co., ltd." },
	{ 0x00026b, "bcm computers co., ltd." },
	{ 0x00026c, "philips cft" },
	{ 0x00026d, "adept telecom" },
	{ 0x00026e, "negen access, inc." },
	{ 0x00026f, "senao international co., ltd." },
	{ 0x000270, "crewave co., ltd." },
	{ 0x000271, "vpacket communications" },
	{ 0x000272, "cc&c technologies, inc." },
	{ 0x000273, "coriolis networks" },
	{ 0x000274, "tommy technologies corp." },
	{ 0x000275, "smart technologies, inc." },
	{ 0x000276, "primax electronics ltd." },
	{ 0x000277, "cash systemes industrie" },
	{ 0x000278, "samsung electro-mechanics co., ltd." },
	{ 0x000279, "control applications, ltd." },
	{ 0x00027a, "ioi technology corporation" },
	{ 0x00027b, "amplify net, inc." },
	{ 0x00027c, "trilithic, inc." },
	{ 0x00027d, "cisco systems, inc." },
	{ 0x00027e, "cisco systems, inc." },
	{ 0x00027f, "ask-technologies.com" },
	{ 0x000280, "mu net, inc." },
	{ 0x000281, "red-m (communications) ltd." },
	{ 0x000282, "viaclix, inc." },
	{ 0x000283, "spectrum controls, inc." },
	{ 0x000284, "alstom t&d p&c" },
	{ 0x000285, "riverstone networks" },
	{ 0x000286, "occam networks" },
	{ 0x000287, "adapcom" },
	{ 0x000288, "global village (pccard in mac portable)" },
	{ 0x000289, "dne technologies" },
	{ 0x00028a, "ambit microsystems corporation" },
	{ 0x00028b, "vdsl systems oy" },
	{ 0x00028c, "micrel-synergy semiconductor" },
	{ 0x00028d, "movita technologies, inc." },
	{ 0x00028e, "rapid 5 networks, inc." },
	{ 0x00028f, "globetek, inc." },
	{ 0x000290, "woorigisool, inc." },
	{ 0x000291, "open network co., ltd." },
	{ 0x000292, "logic innovations, inc." },
	{ 0x000293, "solid data systems" },
	{ 0x000294, "tokyo sokushin co., ltd." },
	{ 0x000295, "ip.access limited" },
	{ 0x000296, "lectron co,. ltd." },
	{ 0x000297, "c-cor.net" },
	{ 0x000298, "broadframe corporation" },
	{ 0x000299, "apex, inc." },
	{ 0x00029a, "storage apps" },
	{ 0x00029b, "kreatel communications ab" },
	{ 0x00029d, "merix corp." },
	{ 0x00029e, "information equipment co., ltd." },
	{ 0x00029f, "l-3 communication aviation recorders" },
	{ 0x0002a0, "flatstack ltd." },
	{ 0x0002a1, "world wide packets" },
	{ 0x0002a2, "hilscher gmbh" },
	{ 0x0002a3, "abb power automation" },
	{ 0x0002a4, "addpac technology co., ltd." },
	{ 0x0002a5, "compaq computer corporation" },
	{ 0x0002a6, "effinet systems co., ltd." },
	{ 0x0002a7, "vivace networks" },
	{ 0x0002a8, "air link technology" },
	{ 0x0002a9, "racom, s.r.o." },
	{ 0x0002aa, "plcom co., ltd." },
	{ 0x0002ab, "ctc union technologies co., ltd." },
	{ 0x0002ac, "3par data" },
	{ 0x0002ad, "asahi optical co., ltd." },
	{ 0x0002ae, "scannex electronics ltd." },
	{ 0x0002af, "telecruz technology, inc." },
	{ 0x0002b0, "hokubu communication & industrial co., ltd." },
	{ 0x0002b1, "anritsu, ltd." },
	{ 0x0002b2, "cablevision" },
	{ 0x0002b3, "intel corporation" },
	{ 0x0002b4, "daphne" },
	{ 0x0002b5, "avnet, inc." },
	{ 0x0002b6, "acrosser technology co., ltd." },
	{ 0x0002b7, "watanabe electric industry co., ltd." },
	{ 0x0002b8, "whi konsult ab" },
	{ 0x0002b9, "cisco systems, inc." },
	{ 0x0002ba, "cisco systems, inc." },
	{ 0x0002bb, "continuous computing" },
	{ 0x0002bc, "lvl 7 systems, inc." },
	{ 0x0002bd, "bionet co., ltd." },
	{ 0x0002be, "totsu engineering, inc." },
	{ 0x0002bf, "dotrocket, inc." },
	{ 0x0002c0, "bencent tzeng industry co., ltd." },
	{ 0x0002c1, "innovative electronic designs, inc." },
	{ 0x0002c2, "net vision telecom" },
	{ 0x0002c3, "arelnet ltd." },
	{ 0x0002c4, "vector international buba" },
	{ 0x0002c5, "evertz microsystems ltd." },
	{ 0x0002c6, "data track technology plc" },
	{ 0x0002c7, "alps electric co., ltd." },
	{ 0x0002c8, "technocom communications technology (pte) ltd" },
	{ 0x0002c9, "mellanox technologies" },
	{ 0x0002ca, "endpoints, inc." },
	{ 0x0002cb, "tristate ltd." },
	{ 0x0002cc, "m.c.c.i" },
	{ 0x0002cd, "teledream, inc." },
	{ 0x0002ce, "foxjet, inc." },
	{ 0x0002cf, "zygate communications, inc." },
	{ 0x0002d0, "comdial corporation" },
	{ 0x0002d1, "vivotek, inc." },
	{ 0x0002d2, "workstation ag" },
	{ 0x0002d3, "netbotz, inc." },
	{ 0x0002d4, "pda peripherals, inc." },
	{ 0x0002d5, "acr" },
	{ 0x0002d6, "nice systems" },
	{ 0x0002d7, "empeg ltd" },
	{ 0x0002d8, "brecis communications corporation" },
	{ 0x0002d9, "reliable controls" },
	{ 0x0002da, "exio communications, inc." },
	{ 0x0002db, "netsec" },
	{ 0x0002dc, "fujitsu general limited" },
	{ 0x0002dd, "bromax communications, ltd." },
	{ 0x0002de, "astrodesign, inc." },
	{ 0x0002df, "net com systems, inc." },
	{ 0x0002e0, "etas gmbh" },
	{ 0x0002e1, "integrated network corporation" },
	{ 0x0002e2, "ndc infared engineering" },
	{ 0x0002e3, "lite-on communications, inc." },
	{ 0x0002e4, "jc hyun systems, inc." },
	{ 0x0002e5, "timeware ltd." },
	{ 0x0002e6, "gould instrument systems, inc." },
	{ 0x0002e7, "cab gmbh & co kg" },
	{ 0x0002e8, "e.d.&a." },
	{ 0x0002e9, "cs systemes de securite - c3s" },
	{ 0x0002ea, "videonics, inc." },
	{ 0x0002eb, "easent communications" },
	{ 0x0002ec, "maschoff design engineering" },
	{ 0x0002ed, "dxo telecom co., ltd." },
	{ 0x0002ee, "nokia danmark a/s" },
	{ 0x0002ef, "ccc network systems group ltd." },
	{ 0x0002f0, "ame optimedia technology co., ltd." },
	{ 0x0002f1, "pinetron co., ltd." },
	{ 0x0002f2, "edevice, inc." },
	{ 0x0002f3, "media serve co., ltd." },
	{ 0x0002f4, "pctel, inc." },
	{ 0x0002f5, "vive synergies, inc." },
	{ 0x0002f6, "equipe communications" },
	{ 0x0002f7, "arm" },
	{ 0x0002f8, "seakr engineering, inc." },
	{ 0x0002f9, "mimos semiconductor sdn bhd" },
	{ 0x0002fa, "dx antenna co., ltd." },
	{ 0x0002fb, "baumuller aulugen-systemtechnik gmbh" },
	{ 0x0002fc, "cisco systems, inc." },
	{ 0x0002fd, "cisco systems, inc." },
	{ 0x0002fe, "viditec, inc." },
	{ 0x0002ff, "handan broad infocom" },
	{ 0x000300, "netcontinuum, inc." },
	{ 0x000301, "avantas networks corporation" },
	{ 0x000302, "oasys telecom, inc." },
	{ 0x000303, "jama electronics co., ltd." },
	{ 0x000304, "pacific broadband communications" },
	{ 0x000305, "smart network devices gmbh" },
	{ 0x000306, "fusion in tech co., ltd." },
	{ 0x000307, "secure works, inc." },
	{ 0x000308, "am communications, inc." },
	{ 0x000309, "texcel technology plc" },
	{ 0x00030a, "argus technologies" },
	{ 0x00030b, "hunter technology, inc." },
	{ 0x00030c, "telesoft technologies ltd." },
	{ 0x00030d, "uniwill computer corp." },
	{ 0x00030e, "core communications co., ltd." },
	{ 0x00030f, "legend digital china ltd." },
	{ 0x000310, "link evolution corp." },
	{ 0x000311, "micro technology co., ltd." },
	{ 0x000312, "tr-systemtechnik gmbh" },
	{ 0x000313, "access media spa" },
	{ 0x000314, "teleware network systems" },
	{ 0x000315, "cidco incorporated" },
	{ 0x000316, "nobell communications, inc." },
	{ 0x000317, "merlin systems, inc." },
	{ 0x000318, "cyras systems, inc." },
	{ 0x000319, "infineon ag" },
	{ 0x00031a, "beijing broad telecom ltd., china" },
	{ 0x00031b, "cellvision systems, inc." },
	{ 0x00031c, "svenska hardvarufabriken ab" },
	{ 0x00031d, "taiwan commate computer, inc." },
	{ 0x00031e, "optranet, inc." },
	{ 0x00031f, "condev ltd." },
	{ 0x000320, "xpeed, inc." },
	{ 0x000321, "reco research co., ltd." },
	{ 0x000322, "idis co., ltd." },
	{ 0x000323, "cornet technology, inc." },
	{ 0x000324, "tottori sanyo electric co., ltd." },
	{ 0x000325, "arima computer corp." },
	{ 0x000326, "iwasaki information systems co., ltd." },
	{ 0x000327, "act'l" },
	{ 0x000328, "mace group, inc." },
	{ 0x000329, "f3, inc." },
	{ 0x00032a, "unidata communication systems, inc." },
	{ 0x00032b, "gai datenfunksysteme gmbh" },
	{ 0x00032c, "abb industrie ag" },
	{ 0x00032d, "ibase technology, inc." },
	{ 0x00032e, "scope information management, ltd." },
	{ 0x00032f, "global sun technology, inc." },
	{ 0x000330, "imagenics, co., ltd." },
	{ 0x000331, "cisco systems, inc." },
	{ 0x000332, "cisco systems, inc." },
	{ 0x000333, "digitel co., ltd." },
	{ 0x000334, "newport electronics" },
	{ 0x000335, "mirae technology" },
	{ 0x000336, "zetes technologies" },
	{ 0x000337, "vaone, inc." },
	{ 0x000338, "oak technology" },
	{ 0x000339, "eurologic systems, ltd." },
	{ 0x00033a, "silicon wave, inc." },
	{ 0x00033b, "tami tech co., ltd." },
	{ 0x00033c, "daiden co., ltd." },
	{ 0x00033d, "ilshin lab" },
	{ 0x00033e, "tateyama system laboratory co., ltd." },
	{ 0x00033f, "bigband networks, ltd." },
	{ 0x000340, "floware wireless systems, ltd." },
	{ 0x000341, "axon digital design" },
	{ 0x000342, "nortel networks" },
	{ 0x000343, "martin professional a/s" },
	{ 0x000344, "tietech.co., ltd." },
	{ 0x000345, "routrek networks corporation" },
	{ 0x000346, "hitachi kokusai electric, inc." },
	{ 0x000347, "intel corporation" },
	{ 0x000348, "norscan instruments, ltd." },
	{ 0x000349, "vidicode datacommunicatie b.v." },
	{ 0x00034a, "rias corporation" },
	{ 0x00034b, "nortel networks" },
	{ 0x00034c, "shanghai digivision technology co., ltd." },
	{ 0x00034d, "chiaro networks, ltd." },
	{ 0x00034e, "pos data company, ltd." },
	{ 0x00034f, "sur-gard security" },
	{ 0x000350, "bticino spa" },
	{ 0x000351, "diebold, inc." },
	{ 0x000352, "colubris networks" },
	{ 0x000353, "mitac, inc." },
	{ 0x000354, "fiber logic communications" },
	{ 0x000355, "terabeam internet systems" },
	{ 0x000356, "wincor nixdorf gmbh & co kg" },
	{ 0x000357, "intervoice-brite, inc." },
	{ 0x000358, "icable system co., ltd." },
	{ 0x000359, "digitalsis" },
	{ 0x00035a, "phototron limited" },
	{ 0x00035b, "bridge wave communications" },
	{ 0x00035c, "saint song corp." },
	{ 0x00035d, "bosung hi-net co., ltd." },
	{ 0x00035e, "metropolitan area networks, inc." },
	{ 0x00035f, "schuehle mess - und. kontrollsysteme" },
	{ 0x000360, "pac interactive technology, inc." },
	{ 0x000361, "widcomm, inc." },
	{ 0x000362, "vodtel communications, inc." },
	{ 0x000363, "miraesys co., ltd." },
	{ 0x000364, "scenix semiconductor, inc." },
	{ 0x000365, "kira information & communications, ltd." },
	{ 0x000366, "asm pacific technology" },
	{ 0x000367, "jasmine networks, inc." },
	{ 0x000368, "embedone co., ltd." },
	{ 0x000369, "nippon antenna co., ltd." },
	{ 0x00036a, "mainnet, ltd." },
	{ 0x00036b, "cisco systems, inc." },
	{ 0x00036c, "cisco systems, inc." },
	{ 0x00036d, "runtop, inc." },
	{ 0x00036e, "nicon systems (pty) limited" },
	{ 0x00036f, "telsey spa" },
	{ 0x000370, "nxtv, inc." },
	{ 0x000371, "acomz networks corp." },
	{ 0x000372, "ulan" },
	{ 0x000373, "aselsan a.s" },
	{ 0x000374, "hunter watertech" },
	{ 0x000375, "netmedia, inc." },
	{ 0x000376, "graphtec technology, inc." },
	{ 0x000377, "gigabit wireless" },
	{ 0x000378, "humax co., ltd." },
	{ 0x000379, "proscend communications, inc." },
	{ 0x00037a, "taiyo yuden co., ltd." },
	{ 0x00037b, "idec izumi corporation" },
	{ 0x00037c, "coax media" },
	{ 0x00037d, "stellcom" },
	{ 0x00037e, "portech communications, inc." },
	{ 0x00037f, "atheros communications, inc." },
	{ 0x000381, "ingenico international" },
	{ 0x000382, "a-one co., ltd." },
	{ 0x000383, "metera networks, inc." },
	{ 0x000384, "aeta" },
	{ 0x000385, "actelis networks, inc." },
	{ 0x000386, "ho net, inc." },
	{ 0x000387, "blaze network products" },
	{ 0x000388, "fastfame technology co., ltd." },
	{ 0x000389, "plantronics" },
	{ 0x00038a, "america online, inc." },
	{ 0x00038b, "plus-one i&t, inc." },
	{ 0x00038c, "total impact" },
	{ 0x00038d, "pcs revenue control systems, inc." },
	{ 0x00038e, "atoga systems, inc." },
	{ 0x00038f, "weinschel corporation" },
	{ 0x000390, "digital video communications, inc." },
	{ 0x000392, "hyundai teletek co., ltd." },
	{ 0x000393, "apple computer, inc." },
	{ 0x000394, "connect one" },
	{ 0x000395, "california amplifier" },
	{ 0x000396, "ez cast co., ltd." },
	{ 0x000397, "watchfront electronics" },
	{ 0x000398, "wisi" },
	{ 0x000399, "dongju informations & communications co., ltd." },
	{ 0x00039a, "nsine, ltd." },
	{ 0x00039b, "netchip technology, inc." },
	{ 0x00039c, "optimight communications, inc." },
	{ 0x00039d, "acer communications & multimedia, inc." },
	{ 0x00039e, "tera system co., ltd." },
	{ 0x00039f, "cisco systems, inc." },
	{ 0x0003a0, "cisco systems, inc." },
	{ 0x0003a1, "hiper information & communication, inc." },
	{ 0x0003a2, "catapult communications" },
	{ 0x0003a3, "mavix, ltd." },
	{ 0x0003a4, "data storage and information management" },
	{ 0x0003a5, "medea corporation" },
	{ 0x0003a7, "unixtar technology, inc." },
	{ 0x0003a8, "idot computers, inc." },
	{ 0x0003a9, "axcent media ag" },
	{ 0x0003aa, "watlow" },
	{ 0x0003ab, "bridge information systems" },
	{ 0x0003ac, "fronius schweissmaschinen" },
	{ 0x0003ad, "emerson energy systems ab" },
	{ 0x0003ae, "allied advanced manufacturing pte, ltd." },
	{ 0x0003af, "paragea communications" },
	{ 0x0003b0, "xsense technology corp." },
	{ 0x0003b1, "abbott laboratories hpd" },
	{ 0x0003b2, "radware" },
	{ 0x0003b3, "ia link systems co., ltd." },
	{ 0x0003b4, "macrotek international corp." },
	{ 0x0003b5, "entra technology co." },
	{ 0x0003b6, "qsi corporation" },
	{ 0x0003b7, "zaccess systems" },
	{ 0x0003b8, "netkit solutions, llc" },
	{ 0x0003b9, "hualong telecom co., ltd." },
	{ 0x0003ba, "sun microsystems" },
	{ 0x0003bb, "signal communications limited" },
	{ 0x0003bc, "cot gmbh" },
	{ 0x0003bd, "omnicluster technologies, inc." },
	{ 0x0003be, "netility" },
	{ 0x0003bf, "centerpoint broadband technologies, inc." },
	{ 0x0003c0, "rftnc co., ltd." },
	{ 0x0003c1, "packet dynamics ltd" },
	{ 0x0003c2, "solphone k.k." },
	{ 0x0003c3, "micronik multimedia" },
	{ 0x0003c4, "tomra systems asa" },
	{ 0x0003c5, "mobotix ag" },
	{ 0x0003c6, "morning star technologies inc" },
	{ 0x0003c7, "hopf elektronik gmbh" },
	{ 0x0003c8, "cml emergency services" },
	{ 0x0003c9, "tecom co., ltd." },
	{ 0x0003ca, "mts systems corp." },
	{ 0x0003cb, "nippon systems development co., ltd." },
	{ 0x0003cc, "momentum computer, inc." },
	{ 0x0003cd, "clovertech, inc." },
	{ 0x0003ce, "eten technologies, inc." },
	{ 0x0003cf, "muxcom, inc." },
	{ 0x0003d0, "koankeiso co., ltd." },
	{ 0x0003d1, "takaya corporation" },
	{ 0x0003d2, "crossbeam systems, inc." },
	{ 0x0003d3, "internet energy systems, inc." },
	{ 0x0003d4, "alloptic, inc." },
	{ 0x0003d5, "advanced communications co., ltd." },
	{ 0x0003d6, "radvision, ltd." },
	{ 0x0003d7, "nextnet wireless, inc." },
	{ 0x0003d8, "impath networks, inc." },
	{ 0x0003d9, "secheron sa" },
	{ 0x0003da, "takamisawa cybernetics co., ltd." },
	{ 0x0003db, "apogee electronics corp." },
	{ 0x0003dc, "lexar media, inc." },
	{ 0x0003dd, "comark corp." },
	{ 0x0003de, "otc wireless" },
	{ 0x0003df, "desana systems" },
	{ 0x0003e0, "radioframe networks, inc." },
	{ 0x0003e1, "winmate communication, inc." },
	{ 0x0003e2, "comspace corporation" },
	{ 0x0003e3, "cisco systems, inc." },
	{ 0x0003e4, "cisco systems, inc." },
	{ 0x0003e5, "hermstedt sg" },
	{ 0x0003e6, "entone technologies, inc." },
	{ 0x0003e7, "logostek co. ltd." },
	{ 0x0003e8, "wavelength digital limited" },
	{ 0x0003e9, "akara canada, inc." },
	{ 0x0003ea, "mega system technologies, inc." },
	{ 0x0003eb, "atrica" },
	{ 0x0003ec, "icg research, inc." },
	{ 0x0003ed, "shinkawa electric co., ltd." },
	{ 0x0003ee, "mknet corporation" },
	{ 0x0003ef, "oneline ag" },
	{ 0x0003f0, "redfern broadband networks" },
	{ 0x0003f1, "cicada semiconductor, inc." },
	{ 0x0003f2, "seneca networks" },
	{ 0x0003f3, "dazzle multimedia, inc." },
	{ 0x0003f4, "netburner" },
	{ 0x0003f5, "chip2chip" },
	{ 0x0003f6, "allegro networks, inc." },
	{ 0x0003f7, "plast-control gmbh" },
	{ 0x0003f8, "sancastle technologies, inc." },
	{ 0x0003f9, "pleiades communications, inc." },
	{ 0x0003fa, "timetra networks" },
	{ 0x0003fb, "toko seiki company, ltd." },
	{ 0x0003fc, "intertex data ab" },
	{ 0x0003fd, "cisco systems, inc." },
	{ 0x0003fe, "cisco systems, inc." },
	{ 0x0003ff, "connectix" },
	{ 0x000400, "lexmark" },
	{ 0x000401, "osaki electric co., ltd." },
	{ 0x000402, "nexsan technologies, ltd." },
	{ 0x000403, "nexsi corporation" },
	{ 0x000404, "makino milling machine co., ltd." },
	{ 0x000405, "acn technologies" },
	{ 0x000406, "fa. metabox ag" },
	{ 0x000407, "topcon positioning systems, inc." },
	{ 0x000408, "sanko electronics co., ltd." },
	{ 0x000409, "cratos networks" },
	{ 0x00040a, "sage systems" },
	{ 0x00040b, "3com europe ltd." },
	{ 0x00040c, "kanno work's ltd." },
	{ 0x00040d, "avaya, inc." },
	{ 0x00040e, "avm gmbh" },
	{ 0x00040f, "asus network technologies, inc." },
	{ 0x000410, "spinnaker networks, inc." },
	{ 0x000411, "inkra networks, inc." },
	{ 0x000412, "wavesmith networks, inc." },
	{ 0x000413, "snom technology ag" },
	{ 0x000414, "umezawa musen denki co., ltd." },
	{ 0x000415, "rasteme systems co., ltd." },
	{ 0x000416, "parks s/a comunicacoes digitais" },
	{ 0x000417, "elau ag" },
	{ 0x000418, "teltronic s.a.u." },
	{ 0x000419, "fibercycle networks, inc." },
	{ 0x00041a, "ines gmbh" },
	{ 0x00041b, "digital interfaces ltd." },
	{ 0x00041c, "ipdialog, inc." },
	{ 0x00041d, "corega of america" },
	{ 0x00041e, "shikoku instrumentation co., ltd." },
	{ 0x00041f, "sony computer entertainment, inc." },
	{ 0x000420, "slim devices, inc." },
	{ 0x000421, "ocular networks" },
	{ 0x000422, "gordon kapes, inc." },
	{ 0x000423, "intel corporation" },
	{ 0x000424, "tmc s.r.l." },
	{ 0x000425, "atmel corporation" },
	{ 0x000426, "autosys" },
	{ 0x000427, "cisco systems, inc." },
	{ 0x000428, "cisco systems, inc." },
	{ 0x000429, "pixord corporation" },
	{ 0x00042a, "wireless networks, inc." },
	{ 0x00042b, "it access co., ltd." },
	{ 0x00042c, "minet, inc." },
	{ 0x00042d, "sarian systems, ltd." },
	{ 0x00042e, "netous technologies, ltd." },
	{ 0x00042f, "international communications products, inc." },
	{ 0x000430, "netgem" },
	{ 0x000431, "play industries" },
	{ 0x000432, "voyetra turtle beach, inc." },
	{ 0x000433, "cyberboard a/s" },
	{ 0x000434, "accelent systems, inc." },
	{ 0x000435, "comptek international, inc." },
	{ 0x000436, "elansat technologies, inc." },
	{ 0x000437, "powin information technology, inc." },
	{ 0x000438, "nortel networks" },
	{ 0x000439, "rosco entertainment technology, inc." },
	{ 0x00043a, "intelligent telecommunications, inc." },
	{ 0x00043b, "lava computer mfg., inc." },
	{ 0x00043c, "sonos co., ltd." },
	{ 0x00043d, "indel ag" },
	{ 0x00043e, "telencomm" },
	{ 0x00043f, "electronic systems technology, inc." },
	{ 0x000440, "cyberpixie, inc." },
	{ 0x000441, "half dome systems, inc." },
	{ 0x000442, "nact" },
	{ 0x000443, "agilent technologies, inc." },
	{ 0x000444, "wireless home" },
	{ 0x000445, "lms skalar instruments gmbh" },
	{ 0x000446, "cyzentech co., ltd." },
	{ 0x000447, "acrowave systems co., ltd." },
	{ 0x000448, "polaroid professional imaging" },
	{ 0x000449, "mapletree networks" },
	{ 0x00044a, "ipolicy networks, inc." },
	{ 0x00044b, "nvidia" },
	{ 0x00044c, "jenoptik" },
	{ 0x00044d, "cisco systems, inc." },
	{ 0x00044e, "cisco systems, inc." },
	{ 0x00044f, "leukhardt systemelektronik gmbh" },
	{ 0x000450, "dmd computers srl" },
	{ 0x000451, "medrad, inc." },
	{ 0x000452, "rocketlogix, inc." },
	{ 0x000453, "yotta yotta, inc." },
	{ 0x000454, "quadriga uk" },
	{ 0x000455, "antara.net" },
	{ 0x000456, "pipinghot networks" },
	{ 0x000457, "universal access technology, inc." },
	{ 0x000458, "fusion x co., ltd." },
	{ 0x000459, "veristar corporation" },
	{ 0x00045a, "the linksys group, inc." },
	{ 0x00045b, "techsan electronics co., ltd." },
	{ 0x00045c, "mobiwave pte ltd" },
	{ 0x00045d, "beka elektronik" },
	{ 0x00045e, "poly trax information technology ag" },
	{ 0x00045f, "evalue technology, inc." },
	{ 0x000460, "knilink technology, inc." },
	{ 0x000461, "epox computer co., ltd." },
	{ 0x000462, "dakos data & communication co., ltd." },
	{ 0x000463, "philips communication security & imaging" },
	{ 0x000464, "fantasma networks, inc." },
	{ 0x000465, "ist isdn support tecknik gmbh" },
	{ 0x000466, "armitel co." },
	{ 0x000467, "wuhan research institute" },
	{ 0x000468, "vivity, inc." },
	{ 0x000469, "innocom, inc." },
	{ 0x00046a, "navini networks" },
	{ 0x00046b, "palm wireless, inc." },
	{ 0x00046c, "cyber technology co., ltd." },
	{ 0x00046d, "cisco systems, inc." },
	{ 0x00046e, "cisco systems, inc." },
	{ 0x00046f, "digitel s/a industria eletronica" },
	{ 0x000470, "ipunplugged ab" },
	{ 0x000471, "iprad" },
	{ 0x000472, "telelynx, inc." },
	{ 0x000473, "photonex corporation" },
	{ 0x000474, "legrand" },
	{ 0x000475, "3 com corporation" },
	{ 0x000476, "3 com corporation" },
	{ 0x000477, "e-appliance corp." },
	{ 0x000478, "g. star technology corporation" },
	{ 0x000479, "radius co., ltd." },
	{ 0x00047a, "axxessit asa" },
	{ 0x00047b, "schlumberger" },
	{ 0x00047c, "skidata ag" },
	{ 0x00047d, "pelco" },
	{ 0x00047e, "nkf electronics" },
	{ 0x00047f, "chr. mayr gmbh & co. kg" },
	{ 0x000480, "foundry networks, inc." },
	{ 0x000481, "econolite control products, inc." },
	{ 0x000482, "medialogic corp." },
	{ 0x000483, "deltron technology, inc." },
	{ 0x000484, "amann gmbh" },
	{ 0x000485, "picolight" },
	{ 0x000486, "ittc, university of kansas" },
	{ 0x000487, "cogency semiconductor, inc." },
	{ 0x000488, "eurotherm action incorporated." },
	{ 0x000489, "yafo networks, inc." },
	{ 0x00048a, "temia vertriebs gmbh" },
	{ 0x00048b, "poscon corporation" },
	{ 0x00048c, "nayna networks, inc." },
	{ 0x00048d, "tone commander systems, inc." },
	{ 0x00048e, "ohm tech labs, inc." },
	{ 0x00048f, "td systems corp." },
	{ 0x000490, "optical access" },
	{ 0x000491, "technovision, inc." },
	{ 0x000492, "hive internet, ltd." },
	{ 0x000493, "tsinghua unisplendour co., ltd." },
	{ 0x000494, "breezecom, ltd." },
	{ 0x000495, "tejas networks" },
	{ 0x000496, "extreme networks" },
	{ 0x000497, "macrosystem digital video ag" },
	{ 0x000499, "chino corporation" },
	{ 0x00049a, "cisco systems, inc." },
	{ 0x00049b, "cisco systems, inc." },
	{ 0x00049c, "surgient networks, inc." },
	{ 0x00049d, "ipanema technologies" },
	{ 0x00049e, "wirelink co., ltd." },
	{ 0x00049f, "metrowerks" },
	{ 0x0004a0, "verity instruments, inc." },
	{ 0x0004a1, "pathway connectivity" },
	{ 0x0004a2, "l.s.i. japan co., ltd." },
	{ 0x0004a3, "microchip technology, inc." },
	{ 0x0004a4, "netenabled, inc." },
	{ 0x0004a5, "barco projection systems nv" },
	{ 0x0004a6, "saf tehnika ltd." },
	{ 0x0004a7, "fabiatech corporation" },
	{ 0x0004a8, "broadmax technologies, inc." },
	{ 0x0004a9, "sandstream technologies, inc." },
	{ 0x0004aa, "jetstream communications" },
	{ 0x0004ab, "comverse network systems, inc." },
	{ 0x0004ac, "ibm" },
	{ 0x0004ad, "malibu networks" },
	{ 0x0004ae, "liquid metronics" },
	{ 0x0004af, "digital fountain, inc." },
	{ 0x0004b0, "elesign co., ltd." },
	{ 0x0004b1, "signal technology, inc." },
	{ 0x0004b2, "essegi srl" },
	{ 0x0004b3, "videotek, inc." },
	{ 0x0004b4, "ciac" },
	{ 0x0004b5, "equitrac corporation" },
	{ 0x0004b6, "tellumat (pty) ltd." },
	{ 0x0004b7, "amb i.t. holding" },
	{ 0x0004b8, "kumahira co., ltd." },
	{ 0x0004b9, "s.i. soubou, inc." },
	{ 0x0004ba, "kdd media will corporation" },
	{ 0x0004bb, "bardac corporation" },
	{ 0x0004bc, "giantec, inc." },
	{ 0x0004bd, "motorola bcs" },
	{ 0x0004be, "optxcon, inc." },
	{ 0x0004bf, "versa logic corp." },
	{ 0x0004c0, "cisco systems, inc." },
	{ 0x0004c1, "cisco systems, inc." },
	{ 0x0004c2, "magnipix, inc." },
	{ 0x0004c3, "castor informatique" },
	{ 0x0004c4, "allen & health" },
	{ 0x0004c5, "ase technologies, usa" },
	{ 0x0004c6, "yamaha motor co., ltd." },
	{ 0x0004c7, "netmount" },
	{ 0x0004c8, "liba maschinefabrik gmbh" },
	{ 0x0004c9, "micro electron co., ltd." },
	{ 0x0004ca, "freems corp." },
	{ 0x0004cb, "tdsoft communication, ltd." },
	{ 0x0004cc, "peek traffic bv." },
	{ 0x0004cd, "informedia research group" },
	{ 0x0004ce, "patria ailon" },
	{ 0x0004cf, "seagate technology" },
	{ 0x0004d0, "softlink s.r.o." },
	{ 0x0004d1, "drew technologies, inc." },
	{ 0x0004d2, "adcon telemetry ag" },
	{ 0x0004d3, "toyokeiki co., ltd." },
	{ 0x0004d4, "proview electronics co., ltd." },
	{ 0x0004d5, "hitachi communication systems, inc." },
	{ 0x0004d6, "takagi industrial co., ltd." },
	{ 0x0004d7, "omitec instrumentation ltd." },
	{ 0x0004d8, "ipwireless, inc." },
	{ 0x0004d9, "titan electronics, inc." },
	{ 0x0004da, "relax technology, inc." },
	{ 0x0004db, "tellus group corp." },
	{ 0x0004dc, "nortel networks" },
	{ 0x0004dd, "cisco systems, inc." },
	{ 0x0004de, "cisco systems, inc." },
	{ 0x0004df, "teracom telematica ltda." },
	{ 0x0004e0, "procket networks" },
	{ 0x0004e1, "infinior microsystems" },
	{ 0x0004e2, "smc networks, inc." },
	{ 0x0004e3, "accton technology corp." },
	{ 0x0004e4, "daeryung ind., inc." },
	{ 0x0004e5, "glonet systems, inc." },
	{ 0x0004e6, "banyan network private limited" },
	{ 0x0004e7, "lightpointe communications, inc" },
	{ 0x0004e8, "ier, inc." },
	{ 0x0004e9, "infiniswitch corporation" },
	{ 0x0004ea, "hewlett-packard company" },
	{ 0x0004eb, "paxonet communications, inc." },
	{ 0x0004ec, "memobox sa" },
	{ 0x0004ed, "billion electric co., ltd." },
	{ 0x0004ee, "lincoln electric company" },
	{ 0x0004ef, "polestar corp." },
	{ 0x0004f0, "international computers, ltd" },
	{ 0x0004f1, "wherenet" },
	{ 0x0004f2, "circa communications, ltd." },
	{ 0x0004f3, "fs forth-systeme gmbh" },
	{ 0x0004f4, "infinite electronics inc." },
	{ 0x0004f5, "snowshore networks, inc." },
	{ 0x0004f6, "amphus" },
	{ 0x0004f7, "omega band, inc." },
	{ 0x0004f8, "qualicable tv industria e com., ltda" },
	{ 0x0004f9, "xtera communications, inc." },
	{ 0x0004fa, "mist inc." },
	{ 0x0004fb, "commtech, inc." },
	{ 0x0004fc, "stratus computer (de), inc." },
	{ 0x0004fd, "japan control engineering co., ltd." },
	{ 0x0004fe, "pelago networks" },
	{ 0x0004ff, "acronet co., ltd." },
	{ 0x000500, "cisco systems, inc." },
	{ 0x000501, "cisco systems, inc." },
	{ 0x000502, "apple (pci bus macs)" },
	{ 0x000503, "iconag" },
	{ 0x000504, "naray information & communication enterprise" },
	{ 0x000505, "systems integration solutions, inc." },
	{ 0x000506, "reddo networks ab" },
	{ 0x000507, "fine appliance corp." },
	{ 0x000508, "inetcam, inc." },
	{ 0x000509, "avoc nishimura ltd." },
	{ 0x00050a, "ics spa" },
	{ 0x00050b, "sicom systems, inc." },
	{ 0x00050c, "network photonics, inc." },
	{ 0x00050d, "midstream technologies, inc." },
	{ 0x00050e, "3ware, inc." },
	{ 0x00050f, "tanaka s/s ltd." },
	{ 0x000510, "infinite shanghai communication terminals ltd." },
	{ 0x000511, "complementaty technologies ltd" },
	{ 0x000512, "meshnetworks, inc." },
	{ 0x000513, "vtlinx multimedia systems, inc." },
	{ 0x000514, "kdt systems co., ltd." },
	{ 0x000515, "nuark co., ltd." },
	{ 0x000516, "smart modular technologies" },
	{ 0x000517, "shellcomm, inc." },
	{ 0x000518, "jupiters technology" },
	{ 0x000519, "siemens building technologies ag," },
	{ 0x00051a, "3com europe ltd." },
	{ 0x00051b, "magic control technology corporation" },
	{ 0x00051c, "xnet technology corp." },
	{ 0x00051d, "airocon, inc." },
	{ 0x00051e, "rhapsody networks" },
	{ 0x00051f, "taijin media co., ltd." },
	{ 0x000520, "smartronix, inc." },
	{ 0x000521, "control microsystems" },
	{ 0x000522, "lea*d corporation, inc." },
	{ 0x000523, "avl list gmbh" },
	{ 0x000524, "btl system (hk) limited" },
	{ 0x000525, "puretek industrial co., ltd." },
	{ 0x000526, "ipas gmbh" },
	{ 0x000527, "sj tek co. ltd" },
	{ 0x000528, "new focus, inc." },
	{ 0x000529, "shanghai broadan communication technology co., ltd" },
	{ 0x00052a, "ikegami tsushinki co., ltd." },
	{ 0x00052b, "horiba, ltd." },
	{ 0x00052c, "supreme magic corporation" },
	{ 0x00052d, "zoltrix international limited" },
	{ 0x00052e, "cinta networks" },
	{ 0x00052f, "leviton voice and data" },
	{ 0x000530, "andiamo systems, inc." },
	{ 0x000531, "cisco systems, inc." },
	{ 0x000532, "cisco systems, inc." },
	{ 0x000533, "sanera systems, inc." },
	{ 0x000534, "northstar engineering ltd." },
	{ 0x000535, "chip pc ltd." },
	{ 0x000536, "danam communications, inc." },
	{ 0x000537, "nets technology co., ltd." },
	{ 0x000538, "merilus, inc." },
	{ 0x000539, "a brand new world in sweden ab" },
	{ 0x00053a, "willowglen services pte ltd" },
	{ 0x00053b, "harbour networks ltd., co. beijing" },
	{ 0x00053c, "xircom" },
	{ 0x00053d, "agere systems" },
	{ 0x00053e, "kid systeme gmbh" },
	{ 0x00053f, "visiontek, inc." },
	{ 0x000540, "fast corporation" },
	{ 0x000541, "advanced systems co., ltd." },
	{ 0x000542, "otari, inc." },
	{ 0x000543, "iq wireless gmbh" },
	{ 0x000544, "valley technologies, inc." },
	{ 0x000545, "internet photonics" },
	{ 0x000546, "kdd network systems co., ltd." },
	{ 0x000547, "starent networks" },
	{ 0x000548, "disco corporation" },
	{ 0x000549, "salira optical network systems" },
	{ 0x00054a, "ario data networks, inc." },
	{ 0x00054b, "micro innovation ag" },
	{ 0x00054c, "rf innovations pty ltd" },
	{ 0x00054d, "brans technologies, inc." },
	{ 0x00054e, "philips components" },
	{ 0x000550, "digi-tech communications limited" },
	{ 0x000551, "f & s elektronik systeme gmbh" },
	{ 0x000552, "xycotec computer gmbh" },
	{ 0x000553, "dvc company, inc." },
	{ 0x000554, "rangestar wireless" },
	{ 0x000555, "japan cash machine co., ltd." },
	{ 0x000556, "360 systems" },
	{ 0x000557, "agile tv corporation" },
	{ 0x000558, "synchronous, inc." },
	{ 0x000559, "intracom s.a." },
	{ 0x00055a, "power dsine ltd." },
	{ 0x00055b, "charles industries, ltd." },
	{ 0x00055c, "kowa company, ltd." },
	{ 0x00055d, "d-link systems, inc." },
	{ 0x00055e, "cisco systems, inc." },
	{ 0x00055f, "cisco systems, inc." },
	{ 0x000560, "leader comm.co., ltd" },
	{ 0x000561, "nac image technology, inc." },
	{ 0x000562, "digital view limited" },
	{ 0x000563, "j-works, inc." },
	{ 0x000564, "tsinghua bitway co., ltd." },
	{ 0x000565, "tailyn communication company ltd." },
	{ 0x000566, "secui.com corporation" },
	{ 0x000567, "etymonic design, inc." },
	{ 0x000568, "piltofish networks ab" },
	{ 0x000569, "vmware, inc." },
	{ 0x00056b, "c.p. technology co., ltd." },
	{ 0x00056c, "hung chang co., ltd." },
	{ 0x00056d, "pacific corporation" },
	{ 0x00056e, "national enhance technology, inc." },
	{ 0x00056f, "innomedia technologies pvt. ltd." },
	{ 0x000570, "baydel ltd." },
	{ 0x000571, "seiwa electronics co." },
	{ 0x000572, "deonet co., ltd." },
	{ 0x000573, "cisco systems, inc." },
	{ 0x000574, "cisco systems, inc." },
	{ 0x000575, "cds-electronics bv" },
	{ 0x000576, "nsm technology ltd." },
	{ 0x000577, "sm information & communication" },
	{ 0x000579, "universal control solution corp." },
	{ 0x00057a, "hatteras networks" },
	{ 0x00057b, "chung nam electronic co., ltd." },
	{ 0x00057c, "rco security ab" },
	{ 0x00057d, "sun communications, inc." },
	{ 0x00057e, "eckelmann steuerungstechnik gmbh" },
	{ 0x00057f, "acqis technology" },
	{ 0x000580, "fibrolan ltd." },
	{ 0x000581, "snell & wilcox ltd." },
	{ 0x000582, "clearcube technology" },
	{ 0x000583, "imagecom limited" },
	{ 0x000584, "absolutevalue systems, inc." },
	{ 0x000585, "juniper networks, inc." },
	{ 0x000586, "lucent technologies" },
	{ 0x000587, "locus, incorporated" },
	{ 0x000588, "sensoria corp." },
	{ 0x000589, "national datacomputer" },
	{ 0x00058a, "netcom co., ltd." },
	{ 0x00058b, "ipmental, inc." },
	{ 0x00058c, "opentech inc." },
	{ 0x00058d, "lynx photonic networks, inc." },
	{ 0x00058e, "ahead communications system gmbh" },
	{ 0x00058f, "clcsoft co." },
	{ 0x000590, "ascom business systems" },
	{ 0x000591, "active silicon ltd." },
	{ 0x000592, "pultex corp." },
	{ 0x000593, "grammar engine inc." },
	{ 0x000594, "ixxat automation gmbh" },
	{ 0x000595, "alesis corporation" },
	{ 0x000596, "genotech co., ltd." },
	{ 0x00059a, "powercomputing (mac clone)" },
	{ 0x0005a8, "powercomputing mac clones" },
	{ 0x00060d, "hp" },
	{ 0x000629, "ibm" },
	{ 0x00062b, "intraserver technology" },
	{ 0x00067c, "cisco" },
	{ 0x0006c1, "cisco" },
	{ 0x000701, "cisco" },
	{ 0x00070d, "cisco" },
	{ 0x000800, "multitech systems, inc." },
	{ 0x000852, "technically elite concepts" },
	{ 0x000855, "fermilab" },
	{ 0x0008c7, "compaq" },
	{ 0x000a27, "apple computer, inc." },
	{ 0x001000, "cable television" },
	{ 0x001001, "mck communications" },
	{ 0x001002, "actia" },
	{ 0x001003, "imatron, inc." },
	{ 0x001004, "the brantley coile company,inc" },
	{ 0x001005, "uec commercial" },
	{ 0x001006, "racal recorders ltd." },
	{ 0x001007, "cisco systems catalyst 1900" },
	{ 0x001008, "vienna systems corporation" },
	{ 0x001009, "horo quartz" },
	{ 0x00100a, "williams communications group" },
	{ 0x00100b, "cisco" },
	{ 0x00100c, "ito co., ltd." },
	{ 0x00100d, "cisco systems catalyst 2924-xl" },
	{ 0x00100e, "micro linear coporation" },
	{ 0x00100f, "industrial cpu systems" },
	{ 0x001010, "initio corporation" },
	{ 0x001011, "cisco" },
	{ 0x001012, "processor systems (i) pvt ltd" },
	{ 0x001013, "industrial computer source" },
	{ 0x001014, "cisco systems, inc." },
	{ 0x001015, "oomon inc." },
	{ 0x001016, "t.sqware" },
	{ 0x001017, "micos gmbh" },
	{ 0x001018, "broadcom corporation" },
	{ 0x001019, "sirona dental systems" },
	{ 0x00101a, "picturetel corp." },
	{ 0x00101b, "cornet technology, inc." },
	{ 0x00101c, "ohm technologies intl, llc" },
	{ 0x00101d, "winbond electronics corp." },
	{ 0x00101e, "matsushita electronic" },
	{ 0x00101f, "cisco" },
	{ 0x001020, "welch allyn, data collection" },
	{ 0x001021, "encanto networks, inc." },
	{ 0x001022, "satcom media corporation" },
	{ 0x001023, "flowwise networks, inc." },
	{ 0x001024, "nagoya electric works co., ltd" },
	{ 0x001025, "grayhill inc." },
	{ 0x001026, "accelerated networks, inc." },
	{ 0x001027, "l-3 communications east" },
	{ 0x001028, "computer technica, inc." },
	{ 0x001029, "cisco systems catalyst 5000" },
	{ 0x00102a, "zf microsystems, inc." },
	{ 0x00102b, "umax data systems, inc." },
	{ 0x00102c, "lasat networks a/s" },
	{ 0x00102d, "hitachi software engineering" },
	{ 0x00102e, "network systems & technologies" },
	{ 0x00102f, "cisco" },
	{ 0x001030, "wi-lan, inc." },
	{ 0x001031, "objective communications, inc." },
	{ 0x001032, "alta technology" },
	{ 0x001033, "accesslan communications, inc." },
	{ 0x001034, "gnp computers" },
	{ 0x001035, "elitegroup computer" },
	{ 0x001036, "inter-tel integrated systems" },
	{ 0x001037, "cyq've technology co., ltd." },
	{ 0x001038, "micro research institute, inc." },
	{ 0x001039, "vectron systems gmbh" },
	{ 0x00103a, "diamond network tech" },
	{ 0x00103b, "hippi networking forum" },
	{ 0x00103c, "ic ensemble, inc." },
	{ 0x00103d, "phasecom, ltd." },
	{ 0x00103e, "netschools corporation" },
	{ 0x00103f, "tollgrade communications, inc." },
	{ 0x001040, "intermec corporation" },
	{ 0x001041, "bristol babcock, inc." },
	{ 0x001042, "alacritech" },
	{ 0x001043, "a2 corporation" },
	{ 0x001044, "innolabs corporation" },
	{ 0x001045, "nortel networks" },
	{ 0x001046, "alcorn mcbride inc." },
	{ 0x001047, "echo eletric co. ltd." },
	{ 0x001048, "htrc automation, inc." },
	{ 0x001049, "shoreline teleworks, inc." },
	{ 0x00104a, "the parvuc corporation" },
	{ 0x00104b, "3com" },
	{ 0x00104c, "computer access technology" },
	{ 0x00104d, "surtec industries, inc." },
	{ 0x00104e, "ceologic" },
	{ 0x00104f, "storage technology corporation" },
	{ 0x001050, "rion co., ltd." },
	{ 0x001051, "cmicro corporation" },
	{ 0x001052, "mettler-toledo (albstadt) gmbh" },
	{ 0x001053, "computer technology corp." },
	{ 0x001054, "cisco systems, inc." },
	{ 0x001055, "fujitsu microelectronics, inc." },
	{ 0x001056, "sodick co., ltd." },
	{ 0x001057, "rebel.com, inc." },
	{ 0x001058, "arrowpoint communications,inc." },
	{ 0x001059, "diablo research co. llc" },
	{ 0x00105a, "3com fast etherlink xl in a gateway 2000" },
	{ 0x00105b, "net insight ab" },
	{ 0x00105c, "quantum designs (h.k.) ltd." },
	{ 0x00105d, "drager, business unit" },
	{ 0x00105e, "hekimian laboratories, inc." },
	{ 0x00105f, "in-snec" },
	{ 0x001060, "billington novell ne200 compatible" },
	{ 0x001061, "hostlink corp." },
	{ 0x001062, "nx server, ilnc." },
	{ 0x001063, "starguide digital networks" },
	{ 0x001064, "digital equipment corp." },
	{ 0x001065, "radyne corporation" },
	{ 0x001066, "advanced control systems, inc." },
	{ 0x001067, "redback" },
	{ 0x001068, "comos telecom" },
	{ 0x001069, "helioss communications, inc." },
	{ 0x00106a, "digital microwave corporation" },
	{ 0x00106b, "sonus networks, inc." },
	{ 0x00106c, "infratec plus gmbh" },
	{ 0x00106d, "integrity communications, inc." },
	{ 0x00106e, "tadiran com. ltd." },
	{ 0x00106f, "trenton technology inc." },
	{ 0x001070, "caradon trend ltd." },
	{ 0x001071, "advanet inc." },
	{ 0x001072, "gvn technologies, inc." },
	{ 0x001073, "technobox, inc." },
	{ 0x001074, "aten international co., ltd." },
	{ 0x001075, "creative design solutions,inc." },
	{ 0x001076, "eurem gmbh" },
	{ 0x001077, "saf drive systems, ltd." },
	{ 0x001078, "nuera communications, inc." },
	{ 0x001079, "cisco" },
	{ 0x00107a, "ambicom (was tandy?)" },
	{ 0x00107b, "cisco" },
	{ 0x00107c, "p-com, inc." },
	{ 0x00107d, "aurora communications, ltd." },
	{ 0x00107e, "bachmann electronic gmbh" },
	{ 0x00107f, "crestron electronics, inc." },
	{ 0x001080, "metawave communications" },
	{ 0x001081, "dps, inc." },
	{ 0x001082, "jna telecommunications limited" },
	{ 0x001083, "hp-ux e 9000/889" },
	{ 0x001084, "k-bot communications" },
	{ 0x001085, "polaris communications, inc." },
	{ 0x001086, "atto technology, inc." },
	{ 0x001087, "xstreamis plc" },
	{ 0x001088, "american networks inc." },
	{ 0x001089, "websonic" },
	{ 0x00108a, "teralogic, inc." },
	{ 0x00108b, "laseranimation sollinger gmbh" },
	{ 0x00108c, "fujitsu telecommunications" },
	{ 0x00108d, "johnson controls, inc." },
	{ 0x00108e, "hugh symons concept" },
	{ 0x00108f, "raptor systems" },
	{ 0x001090, "cimetrics, inc." },
	{ 0x001091, "no wires needed bv" },
	{ 0x001092, "netcore inc." },
	{ 0x001093, "cms computers, ltd." },
	{ 0x001094, "adtech, inc." },
	{ 0x001095, "thomson consumer electronics" },
	{ 0x001096, "tracewell systems, inc." },
	{ 0x001097, "winnet metropolitan" },
	{ 0x001098, "starnet technologies, inc." },
	{ 0x001099, "innomedia, inc." },
	{ 0x00109a, "netline" },
	{ 0x00109b, "vixel corporation" },
	{ 0x00109c, "m-system co., ltd." },
	{ 0x00109d, "clarinet systems, inc." },
	{ 0x00109e, "aware, inc." },
	{ 0x00109f, "pavo, inc." },
	{ 0x0010a0, "innovex technologies, inc." },
	{ 0x0010a1, "kendin semiconductor, inc." },
	{ 0x0010a2, "tns" },
	{ 0x0010a3, "omnitronix, inc." },
	{ 0x0010a4, "xircom" },
	{ 0x0010a5, "oxford instruments" },
	{ 0x0010a6, "cisco" },
	{ 0x0010a7, "unex technology corporation" },
	{ 0x0010a8, "reliance computer corp." },
	{ 0x0010a9, "adhoc technologies" },
	{ 0x0010aa, "media4, inc." },
	{ 0x0010ab, "koito industries, ltd." },
	{ 0x0010ac, "imci technologies" },
	{ 0x0010ad, "softronics usb, inc." },
	{ 0x0010ae, "shinko electric industries co." },
	{ 0x0010af, "tac systems, inc." },
	{ 0x0010b0, "meridian technology corp." },
	{ 0x0010b1, "for-a co., ltd." },
	{ 0x0010b2, "coactive aesthetics" },
	{ 0x0010b3, "nokia multimedia terminals" },
	{ 0x0010b4, "atmosphere networks" },
	{ 0x0010b5, "accton technology corporation" },
	{ 0x0010b6, "entrata communications corp." },
	{ 0x0010b7, "coyote technologies, llc" },
	{ 0x0010b8, "ishigaki computer system co." },
	{ 0x0010b9, "maxtor corp." },
	{ 0x0010ba, "martinho-davis systems, inc." },
	{ 0x0010bb, "data & information technology" },
	{ 0x0010bc, "nortel networks" },
	{ 0x0010bd, "the telecommunication" },
	{ 0x0010be, "telexis corp." },
	{ 0x0010bf, "inter air wireless" },
	{ 0x0010c0, "arma, inc." },
	{ 0x0010c1, "oi electric co., ltd." },
	{ 0x0010c2, "willnet, inc." },
	{ 0x0010c3, "csi-control systems" },
	{ 0x0010c4, "media links co., ltd." },
	{ 0x0010c5, "protocol technologies, inc." },
	{ 0x0010c6, "usi" },
	{ 0x0010c7, "data transmission network" },
	{ 0x0010c8, "communications electronics" },
	{ 0x0010c9, "mitsubishi electronics" },
	{ 0x0010ca, "integral access" },
	{ 0x0010cb, "facit k.k." },
	{ 0x0010cc, "clp computer logistik" },
	{ 0x0010cd, "interface concept" },
	{ 0x0010ce, "volamp, ltd." },
	{ 0x0010cf, "fiberlane communications" },
	{ 0x0010d0, "witcom, ltd." },
	{ 0x0010d1, "top layer networks, inc." },
	{ 0x0010d2, "nitto tsushinki co., ltd" },
	{ 0x0010d3, "grips electronic gmbh" },
	{ 0x0010d4, "storage computer corporation" },
	{ 0x0010d5, "imasde canarias, s.a." },
	{ 0x0010d6, "itt a/cd" },
	{ 0x0010d7, "argosy en 220 fast ethernet pcmcia" },
	{ 0x0010d8, "calista" },
	{ 0x0010d9, "ibm japan, fujisawa mt+d" },
	{ 0x0010da, "motion engineering, inc." },
	{ 0x0010db, "netscreen technologies, inc." },
	{ 0x0010dc, "micro-star international" },
	{ 0x0010dd, "enable semiconductor, inc." },
	{ 0x0010de, "international datacasting" },
	{ 0x0010df, "rise computer inc." },
	{ 0x0010e0, "cobalt microserver, inc." },
	{ 0x0010e1, "s.i. tech, inc." },
	{ 0x0010e2, "arraycomm, inc." },
	{ 0x0010e3, "compaq computer corporation" },
	{ 0x0010e4, "nsi corporation" },
	{ 0x0010e5, "solectron texas" },
	{ 0x0010e6, "applied intelligent" },
	{ 0x0010e7, "breezecom" },
	{ 0x0010e8, "telocity, incorporated" },
	{ 0x0010e9, "raidtec ltd." },
	{ 0x0010ea, "adept technology" },
	{ 0x0010eb, "selsius systems, ilnc." },
	{ 0x0010ec, "rpcg, llc" },
	{ 0x0010ed, "sundance technology, inc." },
	{ 0x0010ee, "cti products, inc." },
	{ 0x0010ef, "db networks, inc." },
	{ 0x0010f0, "rittal-werk rudolf loh" },
	{ 0x0010f1, "i-o corporation" },
	{ 0x0010f2, "antec" },
	{ 0x0010f3, "nexcom international co., ltd." },
	{ 0x0010f4, "vertical networks, inc." },
	{ 0x0010f5, "amherst systems, inc." },
	{ 0x0010f6, "cisco" },
	{ 0x0010f7, "iriichi technologies" },
	{ 0x0010f8, "kenwood tmi corporation" },
	{ 0x0010f9, "unique systems, inc." },
	{ 0x0010fa, "zayante, inc." },
	{ 0x0010fb, "zida technologies limited" },
	{ 0x0010fc, "broadband networks, inc." },
	{ 0x0010fd, "cocom a/s" },
	{ 0x0010fe, "digital equipment corporation" },
	{ 0x0010ff, "cisco systems, inc." },
	{ 0x001700, "kabel" },
	{ 0x001c7c, "perq systems corporation" },
	{ 0x002000, "lexmark" },
	{ 0x002001, "dsp solutions, inc." },
	{ 0x002002, "seritech enterprise co., ltd." },
	{ 0x002003, "pixel power ltd." },
	{ 0x002004, "yamatake-honeywell co., ltd." },
	{ 0x002005, "simpletech" },
	{ 0x002006, "garrett communications, inc." },
	{ 0x002007, "sfa, inc." },
	{ 0x002008, "cable & computer technology" },
	{ 0x002009, "packard bell elec., inc." },
	{ 0x00200a, "source-comm corp." },
	{ 0x00200b, "octagon systems corp." },
	{ 0x00200c, "adastra systems corp" },
	{ 0x00200d, "carl zeiss" },
	{ 0x00200e, "satellite technology mgmt, inc" },
	{ 0x00200f, "tanbac co., ltd." },
	{ 0x002010, "jeol system technology co. ltd" },
	{ 0x002011, "canopus co ltd" },
	{ 0x002012, "camtronics medical systems" },
	{ 0x002013, "diversified technology, inc." },
	{ 0x002014, "global view co., ltd." },
	{ 0x002015, "actis computer sa" },
	{ 0x002016, "showa electric wire & cable co" },
	{ 0x002017, "orbotech" },
	{ 0x002018, "realtek" },
	{ 0x002019, "ohler gmbh" },
	{ 0x00201a, "nbase" },
	{ 0x00201b, "northern telecom/network" },
	{ 0x00201c, "excel, inc." },
	{ 0x00201d, "katana products" },
	{ 0x00201e, "netquest corporation" },
	{ 0x00201f, "best power technology, inc." },
	{ 0x002020, "megatron computer industries" },
	{ 0x002021, "algorithms software pvt. ltd." },
	{ 0x002022, "teknique, inc." },
	{ 0x002023, "t.c. technologies pty. ltd" },
	{ 0x002024, "pacific communication sciences" },
	{ 0x002025, "control technology inc (industrial controls and network interfaces)" },
	{ 0x002026, "amkly systems, inc." },
	{ 0x002027, "ming fortune industry co., ltd" },
	{ 0x002028, "bloomberg" },
	{ 0x002029, "teleprocessing csu/dsu (now owned by adc/kentrox)" },
	{ 0x00202a, "n.v. dzine" },
	{ 0x00202b, "atml (advanced telecommunications modules, ltd.)" },
	{ 0x00202c, "welltronix co., ltd." },
	{ 0x00202d, "taiyo corporation" },
	{ 0x00202e, "daystar digital" },
	{ 0x00202f, "zeta communications, ltd." },
	{ 0x002030, "analog & digital systems" },
	{ 0x002031, "ertec gmbh" },
	{ 0x002032, "alcatel taisel" },
	{ 0x002033, "synapse technologies, inc." },
	{ 0x002034, "rotec industrieautomation gmbh" },
	{ 0x002035, "ibm" },
	{ 0x002036, "bmc software" },
	{ 0x002037, "seagate technology" },
	{ 0x002038, "vme microsystems international" },
	{ 0x002039, "scinets" },
	{ 0x00203a, "digital bi0metrics inc." },
	{ 0x00203b, "wisdm ltd." },
	{ 0x00203c, "eurotime ab" },
	{ 0x00203d, "novar electronics corporation" },
	{ 0x00203e, "logican technologies, inc." },
	{ 0x00203f, "juki corporation" },
	{ 0x002040, "motorola broadband communications sector" },
	{ 0x002041, "data net" },
	{ 0x002042, "datametrics corp" },
	{ 0x002043, "neuron company limited" },
	{ 0x002044, "genitech pty ltd" },
	{ 0x002045, "solcom systems limited" },
	{ 0x002046, "ciprico, inc." },
	{ 0x002047, "steinbrecher corp." },
	{ 0x002048, "fore systems inc" },
	{ 0x002049, "comtron, inc." },
	{ 0x00204a, "pronet gmbh" },
	{ 0x00204b, "autocomputer co ltd" },
	{ 0x00204c, "mitron computer pte ltd" },
	{ 0x00204d, "inovis gmbh" },
	{ 0x00204e, "network security systems, inc." },
	{ 0x00204f, "deutsche aerospace ag" },
	{ 0x002050, "korea computer inc." },
	{ 0x002051, "phoenix data commununications" },
	{ 0x002052, "ragula systems" },
	{ 0x002053, "huntsville microsystems, inc." },
	{ 0x002054, "eastern research, inc." },
	{ 0x002055, "altech co., ltd." },
	{ 0x002056, "neoproducts" },
	{ 0x002057, "titze datentechnik gmbh" },
	{ 0x002058, "allied signal inc." },
	{ 0x002059, "miro computer products ag" },
	{ 0x00205a, "computer identics" },
	{ 0x00205b, "skyline technology" },
	{ 0x00205c, "internet systems/ florida inc." },
	{ 0x00205d, "nanomatic oy" },
	{ 0x00205e, "castle rock, inc." },
	{ 0x00205f, "gammadata computer gmbh" },
	{ 0x002060, "alcatel italia s.p.a." },
	{ 0x002061, "dynatech communications inc" },
	{ 0x002062, "scorpion logic, ltd." },
	{ 0x002063, "wipro infotech ltd" },
	{ 0x002064, "protec microsystems, inc." },
	{ 0x002065, "supernet networking inc." },
	{ 0x002066, "general magic inc" },
	{ 0x002067, "node runner inc" },
	{ 0x002068, "isdyne" },
	{ 0x002069, "isdn systems corporation" },
	{ 0x00206a, "osaka computer corp." },
	{ 0x00206b, "minolta co., ltd network printers" },
	{ 0x00206c, "evergreen technology corp." },
	{ 0x00206d, "data race, inc." },
	{ 0x00206e, "xact, inc." },
	{ 0x00206f, "flowpoint corporation" },
	{ 0x002070, "hynet, ltd." },
	{ 0x002071, "ibr gmbh" },
	{ 0x002072, "worklink innovations" },
	{ 0x002073, "fusion systems corporation" },
	{ 0x002074, "sungwoon systems" },
	{ 0x002075, "motorola communication israel" },
	{ 0x002076, "reudo corporation" },
	{ 0x002077, "kardios systems corp." },
	{ 0x002078, "runtop inc" },
	{ 0x002079, "mikron gmbh" },
	{ 0x00207a, "wise communications, inc." },
	{ 0x00207b, "level one communications" },
	{ 0x00207c, "autec gmbh" },
	{ 0x00207d, "advanced computer applications" },
	{ 0x00207e, "finecom co., ltd." },
	{ 0x00207f, "kyoei sangyo co., ltd." },
	{ 0x002080, "synergy (uk) ltd." },
	{ 0x002081, "titan electronics" },
	{ 0x002082, "oneac corporation" },
	{ 0x002083, "presticom incorporated" },
	{ 0x002084, "oce printing systems, gmbh" },
	{ 0x002085, "3com" },
	{ 0x002086, "microtech electronics limited" },
	{ 0x002087, "memotec communications corp." },
	{ 0x002088, "global village communication" },
	{ 0x002089, "t3plus networking, inc." },
	{ 0x00208a, "sonix communications ltd" },
	{ 0x00208b, "focus enhancements" },
	{ 0x00208c, "galaxy networks inc" },
	{ 0x00208d, "cmd technology" },
	{ 0x00208e, "chevin software eng. ltd." },
	{ 0x00208f, "eci telecom ltd." },
	{ 0x002090, "advanced compression" },
	{ 0x002091, "j125, national security agency" },
	{ 0x002092, "chess engineering b.v." },
	{ 0x002093, "landings technology corp." },
	{ 0x002094, "cubix corporation" },
	{ 0x002095, "riva electronics" },
	{ 0x002096, "siebe environmental controls" },
	{ 0x002097, "applied signal technology" },
	{ 0x002098, "hectronic ab" },
	{ 0x002099, "bon electric co., ltd." },
	{ 0x00209a, "the 3do company" },
	{ 0x00209b, "ersat electronic gmbh" },
	{ 0x00209c, "primary access corp." },
	{ 0x00209d, "lippert automationstechnik" },
	{ 0x00209e, "brown's operating system" },
	{ 0x00209f, "mercury computer systems, inc." },
	{ 0x0020a0, "oa laboratory co., ltd." },
	{ 0x0020a1, "dovatron" },
	{ 0x0020a2, "galcom networking ltd." },
	{ 0x0020a3, "divicom inc." },
	{ 0x0020a4, "multipoint networks" },
	{ 0x0020a5, "newer technology" },
	{ 0x0020a6, "proxim inc" },
	{ 0x0020a7, "pairgain technologies, inc." },
	{ 0x0020a8, "sast technology corp." },
	{ 0x0020a9, "white horse industrial" },
	{ 0x0020aa, "digimedia vision ltd." },
	{ 0x0020ab, "micro industries corp." },
	{ 0x0020ac, "interflex datensysteme gmbh" },
	{ 0x0020ad, "linq systems" },
	{ 0x0020ae, "ornet data communication tech." },
	{ 0x0020af, "3com" },
	{ 0x0020b0, "gateway devices, inc." },
	{ 0x0020b1, "comtech research inc." },
	{ 0x0020b2, "csp (printline multiconnectivity converter)" },
	{ 0x0020b3, "scltec communications systems" },
	{ 0x0020b4, "terma elektronik as" },
	{ 0x0020b5, "yaskawa electric corporation" },
	{ 0x0020b6, "agile networks inc" },
	{ 0x0020b7, "namaqua computerware" },
	{ 0x0020b8, "prime option, inc." },
	{ 0x0020b9, "metricom, inc." },
	{ 0x0020ba, "center for high performance" },
	{ 0x0020bb, "zax corporation" },
	{ 0x0020bc, "jtec pty ltd." },
	{ 0x0020bd, "niobrara r & d corporation" },
	{ 0x0020be, "lan access corp." },
	{ 0x0020bf, "aehr test systems" },
	{ 0x0020c0, "pulse electronics, inc." },
	{ 0x0020c1, "taiko electric works, ltd." },
	{ 0x0020c2, "texas memory systems, inc." },
	{ 0x0020c3, "counter solutions ltd." },
	{ 0x0020c4, "inet,inc." },
	{ 0x0020c5, "eagle ne2000" },
	{ 0x0020c6, "nectec" },
	{ 0x0020c7, "akai professional m.i. corp." },
	{ 0x0020c8, "larscom incorporated" },
	{ 0x0020c9, "victron bv" },
	{ 0x0020ca, "digital ocean" },
	{ 0x0020cb, "pretec electronics corp." },
	{ 0x0020cc, "digital services, ltd." },
	{ 0x0020cd, "hybrid networks, inc." },
	{ 0x0020ce, "logical design group, inc." },
	{ 0x0020cf, "test & measurement systems inc" },
	{ 0x0020d0, "versalynx corp. the one port terminal server" },
	{ 0x0020d1, "microcomputer systems (m) sdn." },
	{ 0x0020d2, "rad data communications ltd" },
	{ 0x0020d3, "ost (ouet standard telematique)" },
	{ 0x0020d4, "cabletron - zeittnet inc." },
	{ 0x0020d5, "vipa gmbh" },
	{ 0x0020d6, "breezecom" },
	{ 0x0020d7, "japan minicomputer systems co." },
	{ 0x0020d8, "netwave" },
	{ 0x0020d9, "panasonic technologies, inc./" },
	{ 0x0020da, "xylan" },
	{ 0x0020db, "xnet technology, inc." },
	{ 0x0020dc, "densitron taiwan ltd" },
	{ 0x0020dd, "awa ltd." },
	{ 0x0020de, "japan digital laborat'y co.ltd" },
	{ 0x0020df, "kyosan electric mfg. co., ltd." },
	{ 0x0020e0, "premax pe-200 (pcmcia ne2000-clone card, sold by infoexpress)" },
	{ 0x0020e1, "alamar electronics" },
	{ 0x0020e2, "information resource" },
	{ 0x0020e3, "mcd kencom corporation" },
	{ 0x0020e4, "hsing tech enterprise co., ltd" },
	{ 0x0020e5, "apex data" },
	{ 0x0020e6, "lidkoping machine tools ab" },
	{ 0x0020e7, "b&w nuclear service company" },
	{ 0x0020e8, "datatrek corporation" },
	{ 0x0020e9, "dantel" },
	{ 0x0020ea, "efficient networks, inc." },
	{ 0x0020eb, "cincinnati microwave, inc." },
	{ 0x0020ec, "techware systems corp." },
	{ 0x0020ed, "giga-byte technology co., ltd." },
	{ 0x0020ee, "gtech corporation" },
	{ 0x0020ef, "usc corporation" },
	{ 0x0020f0, "universal microelectronics co." },
	{ 0x0020f1, "altos india limited" },
	{ 0x0020f2, "sun microsystems, inc." },
	{ 0x0020f3, "raynet corporation" },
	{ 0x0020f4, "spectrix corporation" },
	{ 0x0020f5, "pandatel ag" },
	{ 0x0020f6, "net tek & karlnet inc" },
	{ 0x0020f7, "cyberdata" },
	{ 0x0020f8, "carrera computers inc" },
	{ 0x0020f9, "paralink networks, inc." },
	{ 0x0020fa, "gde systems, inc." },
	{ 0x0020fb, "octel communications corp." },
	{ 0x0020fc, "matrox" },
	{ 0x0020fd, "itv technologies, inc." },
	{ 0x0020fe, "topware inc. / grand computer" },
	{ 0x0020ff, "symmetrical technologies" },
	{ 0x003000, "allwell technology corp." },
	{ 0x003001, "smp" },
	{ 0x003002, "expand networks" },
	{ 0x003003, "phasys ltd." },
	{ 0x003004, "leadtek research inc." },
	{ 0x003005, "fujitsu siemens computers" },
	{ 0x003006, "superpower computer" },
	{ 0x003007, "opti, inc." },
	{ 0x003008, "avio digital, inc." },
	{ 0x003009, "tachion networks, inc." },
	{ 0x00300a, "aztech systems ltd." },
	{ 0x00300b, "mphase technologies, inc." },
	{ 0x00300c, "congruency, ltd." },
	{ 0x00300d, "mmc technology, inc." },
	{ 0x00300e, "klotz digital ag" },
	{ 0x00300f, "imt - information management t" },
	{ 0x003010, "visionetics international" },
	{ 0x003011, "hms fieldbus systems ab" },
	{ 0x003012, "digital engineering ltd." },
	{ 0x003013, "nec corporation" },
	{ 0x003014, "divio, inc." },
	{ 0x003015, "cp clare corp." },
	{ 0x003016, "ishida co., ltd." },
	{ 0x003017, "terastack ltd." },
	{ 0x003018, "jetway information co., ltd." },
	{ 0x003019, "cisco systems, inc." },
	{ 0x00301a, "smartbridges pte. ltd." },
	{ 0x00301b, "shuttle, inc." },
	{ 0x00301c, "altvater airdata systems" },
	{ 0x00301d, "skystream, inc." },
	{ 0x00301e, "3com europe ltd." },
	{ 0x00301f, "optical networks, inc." },
	{ 0x003020, "tsi, inc.." },
	{ 0x003021, "hsing tech. enterprise co.,ltd" },
	{ 0x003022, "fong kai industrial co., ltd." },
	{ 0x003023, "cogent computer systems, inc." },
	{ 0x003024, "cisco systems, inc." },
	{ 0x003025, "checkout computer systems, ltd" },
	{ 0x003026, "heitel" },
	{ 0x003027, "kerbango, inc." },
	{ 0x003028, "fase saldatura srl" },
	{ 0x003029, "opicom" },
	{ 0x00302a, "southern information" },
	{ 0x00302b, "inalp networks, inc." },
	{ 0x00302c, "sylantro systems corporation" },
	{ 0x00302d, "quantum bridge communications" },
	{ 0x00302e, "hoft & wessel ag" },
	{ 0x00302f, "smiths industries" },
	{ 0x003030, "harmonix corporation" },
	{ 0x003031, "lightwave communications, inc." },
	{ 0x003032, "magicram, inc." },
	{ 0x003033, "orient telecom co., ltd." },
	{ 0x003036, "rmp elektroniksysteme gmbh" },
	{ 0x003037, "packard bell nec services" },
	{ 0x003038, "xcp, inc." },
	{ 0x003039, "softbook press" },
	{ 0x00303a, "maatel" },
	{ 0x00303b, "powercom technology" },
	{ 0x00303c, "onnto corp." },
	{ 0x00303d, "iva corporation" },
	{ 0x00303e, "radcom ltd." },
	{ 0x00303f, "turbocomm tech inc." },
	{ 0x003040, "cisco systems, inc." },
	{ 0x003041, "saejin t & m co., ltd." },
	{ 0x003042, "detewe-deutsche telephonwerke" },
	{ 0x003043, "idream technologies, pte. ltd." },
	{ 0x003044, "portsmith llc" },
	{ 0x003045, "village networks, inc. (vni)" },
	{ 0x003046, "controlled electronic manageme" },
	{ 0x003047, "nissei electric co., ltd." },
	{ 0x003048, "supermicro computer, inc." },
	{ 0x003049, "bryant technology, ltd." },
	{ 0x00304a, "fraunhofer institute ims" },
	{ 0x00304b, "orbacom systems, inc." },
	{ 0x00304c, "appian communications, inc." },
	{ 0x00304d, "esi" },
	{ 0x00304e, "bustec production ltd." },
	{ 0x00304f, "planet technology corporation" },
	{ 0x003050, "versa technology" },
	{ 0x003051, "orbit avionic & communication" },
	{ 0x003052, "elastic networks" },
	{ 0x003053, "basler ag" },
	{ 0x003054, "castlenet technology, inc." },
	{ 0x003055, "hitachi semiconductor america," },
	{ 0x003056, "beck ipc gmbh" },
	{ 0x003057, "e-tel corporation" },
	{ 0x003058, "api motion" },
	{ 0x003059, "digital-logic ag" },
	{ 0x00305a, "telgen corporation" },
	{ 0x00305b, "module department" },
	{ 0x00305c, "smar laboratories corp." },
	{ 0x00305d, "digitra systems, inc." },
	{ 0x00305e, "abelko innovation" },
	{ 0x00305f, "imacon aps" },
	{ 0x003060, "starmatix, inc." },
	{ 0x003061, "mobytel" },
	{ 0x003062, "path 1 network technol's inc." },
	{ 0x003063, "santera systems, inc." },
	{ 0x003064, "adlink technology, inc." },
	{ 0x003065, "apple computer, inc." },
	{ 0x003066, "digital wireless corporation" },
	{ 0x003067, "biostar microtech int'l corp." },
	{ 0x003068, "cybernetics tech. co., ltd." },
	{ 0x003069, "impacct technology corp." },
	{ 0x00306a, "penta media co., ltd." },
	{ 0x00306b, "cmos systems, inc." },
	{ 0x00306c, "hitex holding gmbh" },
	{ 0x00306d, "lucent technologies" },
	{ 0x00306e, "hewlett packard" },
	{ 0x00306f, "seyeon tech. co., ltd." },
	{ 0x003070, "1net corporation" },
	{ 0x003071, "cisco systems, inc." },
	{ 0x003072, "intellibyte inc." },
	{ 0x003073, "international microsystems, in" },
	{ 0x003074, "equiinet ltd." },
	{ 0x003075, "adtech" },
	{ 0x003076, "akamba corporation" },
	{ 0x003077, "onprem networks" },
	{ 0x003078, "cisco systems, inc." },
	{ 0x003079, "cqos, inc." },
	{ 0x00307a, "advanced technology & systems" },
	{ 0x00307b, "cisco systems, inc." },
	{ 0x00307c, "adid sa" },
	{ 0x00307d, "gre america, inc." },
	{ 0x00307e, "redflex communication systems" },
	{ 0x00307f, "irlan ltd." },
	{ 0x003080, "cisco systems, inc." },
	{ 0x003081, "altos c&c" },
	{ 0x003082, "taihan electric wire co., ltd." },
	{ 0x003083, "ivron systems" },
	{ 0x003084, "allied telesyn internaional" },
	{ 0x003085, "cisco systems, inc." },
	{ 0x003086, "transistor devices, inc." },
	{ 0x003087, "vega grieshaber kg" },
	{ 0x003088, "siara systems, inc." },
	{ 0x003089, "spectrapoint wireless, llc" },
	{ 0x00308a, "nicotra sistemi s.p.a" },
	{ 0x00308b, "brix networks" },
	{ 0x00308c, "advanced digital information" },
	{ 0x00308d, "pinnacle systems, inc." },
	{ 0x00308e, "cross match technologies, inc." },
	{ 0x00308f, "micrilor, inc." },
	{ 0x003090, "cyra technologies, inc." },
	{ 0x003091, "taiwan first line elec. corp." },
	{ 0x003092, "modunorm gmbh" },
	{ 0x003093, "sonnet technologies, inc." },
	{ 0x003094, "cisco systems, inc." },
	{ 0x003095, "procomp informatics, ltd." },
	{ 0x003096, "cisco systems, inc." },
	{ 0x003097, "exomatic ab" },
	{ 0x003098, "global converging technologies" },
	{ 0x003099, "boenig und kallenbach ohg" },
	{ 0x00309a, "astro terra corp." },
	{ 0x00309b, "smartware" },
	{ 0x00309c, "timing applications, inc." },
	{ 0x00309d, "nimble microsystems, inc." },
	{ 0x00309e, "workbit corporation." },
	{ 0x00309f, "amber networks" },
	{ 0x0030a0, "tyco submarine systems, ltd." },
	{ 0x0030a1, "opti tech co., ltd." },
	{ 0x0030a2, "lightner engineering" },
	{ 0x0030a3, "cisco systems, inc." },
	{ 0x0030a4, "woodwind communications system" },
	{ 0x0030a5, "active power" },
	{ 0x0030a6, "vianet technologies, ltd." },
	{ 0x0030a7, "schweitzer engineering" },
	{ 0x0030a8, "ol'e communications, inc." },
	{ 0x0030a9, "netiverse, inc." },
	{ 0x0030aa, "axus microsystems, inc." },
	{ 0x0030ab, "delta networks, inc." },
	{ 0x0030ac, "systeme lauer gmbh & co., ltd." },
	{ 0x0030ad, "shanghai communication" },
	{ 0x0030ae, "times n system, inc." },
	{ 0x0030af, "honeywell reqelsysteme gmbh" },
	{ 0x0030b0, "convergenet technologies" },
	{ 0x0030b1, "goc gesellschaft fur optische" },
	{ 0x0030b2, "wescam - healdsburg" },
	{ 0x0030b3, "san valley systems, inc." },
	{ 0x0030b4, "intersil corp." },
	{ 0x0030b5, "tadiran microwave networks" },
	{ 0x0030b6, "cisco systems, inc." },
	{ 0x0030b7, "teletrol systems, inc." },
	{ 0x0030b8, "riverdelta networks" },
	{ 0x0030b9, "ectel" },
	{ 0x0030ba, "ac&t system co., ltd." },
	{ 0x0030bb, "cacheflow, inc." },
	{ 0x0030bc, "optronic ag" },
	{ 0x0030bd, "belkin components" },
	{ 0x0030be, "city-net technology, inc." },
	{ 0x0030bf, "multidata gmbh" },
	{ 0x0030c0, "lara technology, inc." },
	{ 0x0030c1, "hewlett-packard" },
	{ 0x0030c2, "comone" },
	{ 0x0030c3, "flueckiger elektronik ag" },
	{ 0x0030c4, "niigata canotec co., inc." },
	{ 0x0030c5, "cadence design systems" },
	{ 0x0030c6, "control solutions, inc." },
	{ 0x0030c7, "macromate corp." },
	{ 0x0030c8, "gad line, ltd." },
	{ 0x0030c9, "luxn, n" },
	{ 0x0030ca, "discovery com" },
	{ 0x0030cb, "omni flow computers, inc." },
	{ 0x0030cc, "tenor networks, inc." },
	{ 0x0030cd, "conexant systems, inc." },
	{ 0x0030ce, "zaffire" },
	{ 0x0030cf, "two technologies, inc." },
	{ 0x0030d1, "inova corporation" },
	{ 0x0030d2, "win technologies, co., ltd." },
	{ 0x0030d3, "agilent technologies" },
	{ 0x0030d4, "comtier" },
	{ 0x0030d5, "dresearch gmbh" },
	{ 0x0030d6, "msc vertriebs gmbh" },
	{ 0x0030d7, "innovative systems, l.l.c." },
	{ 0x0030d8, "sitek" },
	{ 0x0030d9, "datacore software corp." },
	{ 0x0030da, "comtrend co." },
	{ 0x0030db, "mindready solutions, inc." },
	{ 0x0030dc, "rightech corporation" },
	{ 0x0030dd, "indigita corporation" },
	{ 0x0030de, "wago kontakttechnik gmbh" },
	{ 0x0030df, "kb/tel telecomunicaciones" },
	{ 0x0030e0, "oxford semiconductor ltd." },
	{ 0x0030e1, "acrotron systems, inc." },
	{ 0x0030e2, "garnet systems co., ltd." },
	{ 0x0030e3, "sedona networks corp." },
	{ 0x0030e4, "chiyoda system riken" },
	{ 0x0030e5, "amper datos s.a." },
	{ 0x0030e6, "siemens medical systems" },
	{ 0x0030e7, "cnf mobile solutions, inc." },
	{ 0x0030e8, "ensim corp." },
	{ 0x0030e9, "gma communication manufact'g" },
	{ 0x0030ea, "intelect communictions, inc." },
	{ 0x0030eb, "turbonet communications, inc." },
	{ 0x0030ec, "borgardt" },
	{ 0x0030ed, "expert magnetics corp." },
	{ 0x0030ee, "dsg technology, inc." },
	{ 0x0030ef, "neon technology, inc." },
	{ 0x0030f0, "uniform industrial corp." },
	{ 0x0030f1, "accton technology corp." },
	{ 0x0030f2, "cisco systems, inc." },
	{ 0x0030f3, "at work computers" },
	{ 0x0030f4, "stardot technologies" },
	{ 0x0030f5, "wild lab. ltd." },
	{ 0x0030f6, "securelogix corporation" },
	{ 0x0030f7, "ramix inc." },
	{ 0x0030f8, "dynapro systems, inc." },
	{ 0x0030f9, "sollae systems co., ltd." },
	{ 0x0030fa, "telica, inc." },
	{ 0x0030fb, "azs technology ag" },
	{ 0x0030fc, "terawave communications, inc." },
	{ 0x0030fd, "integrated systems design" },
	{ 0x0030fe, "dsa gmbh" },
	{ 0x0030ff, "datafab systems, inc." },
	{ 0x004000, "pci componentes da amzonia ltd" },
	{ 0x004001, "zero one technology co ltd (zyxel?)" },
	{ 0x004002, "perle systems limited" },
	{ 0x004003, "westinghouse process control" },
	{ 0x004004, "icm co. ltd." },
	{ 0x004005, "trendware international inc.; linksys; simple net; all three reported" },
	{ 0x004006, "sampo technology corporation" },
	{ 0x004007, "telmat informatique" },
	{ 0x004008, "a plus info corporation" },
	{ 0x004009, "tachibana tectron co ltd" },
	{ 0x00400a, "pivotal technologies, inc." },
	{ 0x00400b, "crescendo (now owned by cisco)" },
	{ 0x00400c, "general micro systems, inc." },
	{ 0x00400d, "lannet data communications" },
	{ 0x00400e, "memotec communications, inc." },
	{ 0x00400f, "datacom technologies" },
	{ 0x004010, "sonic mac ethernet interfaces" },
	{ 0x004011, "facilities andover environmental controllers" },
	{ 0x004012, "windata, inc." },
	{ 0x004013, "ntt data communication systems corp" },
	{ 0x004014, "comsoft gmbh" },
	{ 0x004015, "ascom" },
	{ 0x004016, "hadax electronics, inc." },
	{ 0x004017, "xcd xjet - hp printer server card" },
	{ 0x004018, "adobe systems, inc." },
	{ 0x004019, "aeon systems, inc." },
	{ 0x00401a, "fuji electric co., ltd." },
	{ 0x00401b, "printer systems corp." },
	{ 0x00401c, "ast pentium/90 pc (emulating amd eisa card)" },
	{ 0x00401d, "invisible software, inc." },
	{ 0x00401e, "icc" },
	{ 0x00401f, "colorgraph ltd" },
	{ 0x004020, "pilkington communication" },
	{ 0x004021, "raster graphics" },
	{ 0x004022, "klever computers, inc." },
	{ 0x004023, "logic corporation" },
	{ 0x004024, "compac inc." },
	{ 0x004025, "molecular dynamics" },
	{ 0x004026, "melco inc" },
	{ 0x004027, "smc massachusetts [hadsigma (?), maybe the s?]" },
	{ 0x004028, "netcomm" },
	{ 0x004029, "compex" },
	{ 0x00402a, "canoga-perkins" },
	{ 0x00402b, "trigem" },
	{ 0x00402c, "isis distributed systems, inc." },
	{ 0x00402d, "harris adacom corporation" },
	{ 0x00402e, "precision software, inc." },
	{ 0x00402f, "xlnt designs inc (xdi)" },
	{ 0x004030, "gk computer" },
	{ 0x004031, "kokusai electric co., ltd" },
	{ 0x004032, "digital communications" },
	{ 0x004033, "addtron technology co., ltd." },
	{ 0x004034, "bustek corporation" },
	{ 0x004035, "opcom" },
	{ 0x004036, "tribestar" },
	{ 0x004037, "sea-ilan, inc." },
	{ 0x004038, "talent electric incorporated" },
	{ 0x004039, "optec daiichi denko co ltd" },
	{ 0x00403a, "impact technologies" },
	{ 0x00403b, "synerjet international corp." },
	{ 0x00403c, "forks, inc." },
	{ 0x00403d, "teradata" },
	{ 0x00403e, "raster ops corporation" },
	{ 0x00403f, "ssangyong computer systems" },
	{ 0x004040, "ring access, inc." },
	{ 0x004041, "fujikura ltd." },
	{ 0x004042, "n.a.t. gmbh" },
	{ 0x004043, "nokia data communications" },
	{ 0x004044, "qnix computer co., ltd." },
	{ 0x004045, "twinhead corporation" },
	{ 0x004046, "udc research limited" },
	{ 0x004047, "wind river systems" },
	{ 0x004048, "smd informatica s.a." },
	{ 0x004049, "tegimenta ag" },
	{ 0x00404a, "west australian department" },
	{ 0x00404b, "maple computer systems" },
	{ 0x00404c, "hypertec pty ltd." },
	{ 0x00404d, "telecomm techniques" },
	{ 0x00404e, "fluent, inc." },
	{ 0x00404f, "space & naval warfare systems" },
	{ 0x004050, "ironics, incorporated" },
	{ 0x004051, "gracilis, inc." },
	{ 0x004052, "star technologies inc" },
	{ 0x004053, "datum [bancomm division] tymserve 2000" },
	{ 0x004054, "thinking machines corporation" },
	{ 0x004055, "metronix gmbh" },
	{ 0x004056, "mcm japan ltd." },
	{ 0x004057, "lockheed-sanders" },
	{ 0x004058, "kronos, inc." },
	{ 0x004059, "yoshida kogyo k.k." },
	{ 0x00405a, "goldstar information & comm." },
	{ 0x00405b, "funasset limited" },
	{ 0x00405c, "future systems, inc." },
	{ 0x00405d, "star-tek inc" },
	{ 0x00405e, "north hills israel" },
	{ 0x00405f, "afe computers ltd." },
	{ 0x004060, "comendec ltd" },
	{ 0x004061, "datatech enterprises co., ltd." },
	{ 0x004062, "e-systems, inc./garland div." },
	{ 0x004063, "via technologies, inc." },
	{ 0x004064, "kla instruments corporation" },
	{ 0x004065, "gte spacenet" },
	{ 0x004066, "hitachi cable, ltd." },
	{ 0x004067, "omnibyte corporation" },
	{ 0x004068, "extended systems" },
	{ 0x004069, "lemcom systems inc" },
	{ 0x00406a, "kentek information systems inc" },
	{ 0x00406b, "sysgen" },
	{ 0x00406c, "copernique" },
	{ 0x00406d, "lanco, inc." },
	{ 0x00406e, "corollary, inc." },
	{ 0x00406f, "sync research inc" },
	{ 0x004070, "interware co., ltd." },
	{ 0x004071, "atm computer gmbh" },
	{ 0x004072, "applied innovation" },
	{ 0x004073, "bass associates" },
	{ 0x004074, "cable and wireless" },
	{ 0x004075, "m-trade (uk) ltd" },
	{ 0x004076, "amp incorporated" },
	{ 0x004077, "maxton technology corporation" },
	{ 0x004078, "wearnes automation pte ltd" },
	{ 0x004079, "juko manufacture company, ltd." },
	{ 0x00407a, "societe d'exploitation du cnit" },
	{ 0x00407b, "scientific atlanta" },
	{ 0x00407c, "qume corporation" },
	{ 0x00407d, "extension technology corp." },
	{ 0x00407e, "evergreen systems, inc." },
	{ 0x00407f, "agema infrared systems ab" },
	{ 0x004080, "athenix corporation" },
	{ 0x004081, "mannesmann scangraphic gmbh" },
	{ 0x004082, "laboratory equipment corp" },
	{ 0x004083, "tda industria de produtos" },
	{ 0x004084, "honeywell inc." },
	{ 0x004085, "saab instruments ab" },
	{ 0x004086, "michels & kleberhoff computer" },
	{ 0x004087, "ubitrex corporation" },
	{ 0x004088, "mobuis nubus (mac) combination video/ethertalk" },
	{ 0x004089, "meidensha corporation" },
	{ 0x00408a, "tps teleprocessing sys. gmbh" },
	{ 0x00408b, "raylan corporation" },
	{ 0x00408c, "axis" },
	{ 0x00408d, "the goodyear tire & rubber co." },
	{ 0x00408e, "cxr/digilog" },
	{ 0x00408f, "wm-data minfo ab" },
	{ 0x004090, "ansel communications pc ne2000 compatible twisted-pair ethernet cards" },
	{ 0x004091, "procomp industria eletronica" },
	{ 0x004092, "asp computer products, inc." },
	{ 0x004093, "paxdata networks ltd." },
	{ 0x004094, "shographics inc" },
	{ 0x004095, "eagle technologies [umc also reported]" },
	{ 0x004096, "telesystems slw inc" },
	{ 0x004097, "datex division of" },
	{ 0x004098, "dressler gmbh & co." },
	{ 0x004099, "newgen systems corp." },
	{ 0x00409a, "network express inc" },
	{ 0x00409b, "hal computer systems inc." },
	{ 0x00409c, "transware" },
	{ 0x00409d, "digiboard ethernet-isdn bridges" },
	{ 0x00409e, "concurrent technologies ltd." },
	{ 0x00409f, "lancast/casat technology inc" },
	{ 0x0040a0, "goldstar co., ltd." },
	{ 0x0040a1, "ergo computing" },
	{ 0x0040a2, "kingstar technology inc." },
	{ 0x0040a3, "microunity systems engineering" },
	{ 0x0040a4, "rose electronics" },
	{ 0x0040a5, "clinicomp intl." },
	{ 0x0040a6, "cray" },
	{ 0x0040a7, "itautec philco s.a." },
	{ 0x0040a8, "imf international ltd." },
	{ 0x0040a9, "datacom inc." },
	{ 0x0040aa, "valmet automation inc" },
	{ 0x0040ab, "roland dg corporation" },
	{ 0x0040ac, "super workstation, inc." },
	{ 0x0040ad, "sma regelsysteme gmbh" },
	{ 0x0040ae, "delta controls, inc." },
	{ 0x0040af, "digital products, inc. (dpi)." },
	{ 0x0040b0, "bytex corporation, engineering" },
	{ 0x0040b1, "codonics inc." },
	{ 0x0040b2, "systemforschung" },
	{ 0x0040b3, "par microsystems corporation" },
	{ 0x0040b4, "3com k.k." },
	{ 0x0040b5, "video technology computers ltd" },
	{ 0x0040b6, "computerm corporation" },
	{ 0x0040b7, "stealth computer systems" },
	{ 0x0040b8, "idea associates" },
	{ 0x0040b9, "macq electronique sa" },
	{ 0x0040ba, "alliant computer systems corp." },
	{ 0x0040bb, "goldstar cable co., ltd." },
	{ 0x0040bc, "algorithmics ltd." },
	{ 0x0040bd, "starlight networks inc" },
	{ 0x0040be, "boeing defense & space" },
	{ 0x0040bf, "channel systems intern'l inc." },
	{ 0x0040c0, "vista controls corporation" },
	{ 0x0040c1, "bizerba-werke wilheim kraut" },
	{ 0x0040c2, "applied computing devices" },
	{ 0x0040c3, "fischer and porter co." },
	{ 0x0040c4, "kinkei system corporation" },
	{ 0x0040c5, "micom communications corp." },
	{ 0x0040c6, "fibernet research, inc." },
	{ 0x0040c7, "danpex corporation" },
	{ 0x0040c8, "milan" },
	{ 0x0040c9, "ncube" },
	{ 0x0040ca, "first internat'l computer, inc" },
	{ 0x0040cb, "lanwan technologies" },
	{ 0x0040cc, "silcom manufacturing technology inc" },
	{ 0x0040cd, "tera microsystems, inc." },
	{ 0x0040ce, "net-source, inc." },
	{ 0x0040cf, "strawberry tree inc" },
	{ 0x0040d0, "dec/compaq" },
	{ 0x0040d1, "fukuda denshi co., ltd." },
	{ 0x0040d2, "pagine corporation" },
	{ 0x0040d3, "kimpsion international corp." },
	{ 0x0040d4, "gage talker corp." },
	{ 0x0040d5, "sartorius ag" },
	{ 0x0040d6, "locamation b.v." },
	{ 0x0040d7, "studio gen inc" },
	{ 0x0040d8, "ocean office automation ltd" },
	{ 0x0040d9, "american megatrends inc." },
	{ 0x0040da, "telspec ltd" },
	{ 0x0040db, "advanced technical solutions" },
	{ 0x0040dc, "tritec electronic gmbh" },
	{ 0x0040dd, "hong technologies" },
	{ 0x0040de, "elettronica san giorgio" },
	{ 0x0040df, "digalog systems, inc." },
	{ 0x0040e0, "atomwide ltd." },
	{ 0x0040e1, "marner international inc" },
	{ 0x0040e2, "mesa ridge technologies inc" },
	{ 0x0040e3, "quin systems ltd" },
	{ 0x0040e4, "e-m technology, inc." },
	{ 0x0040e5, "sybus corporation" },
	{ 0x0040e6, "c.a.e.n." },
	{ 0x0040e7, "arnos instruments & computer" },
	{ 0x0040e8, "charles river data systems,inc" },
	{ 0x0040e9, "accord systems, inc." },
	{ 0x0040ea, "plaintree systems inc" },
	{ 0x0040eb, "martin marietta corporation" },
	{ 0x0040ec, "mikasa system engineering" },
	{ 0x0040ed, "network controls international inc" },
	{ 0x0040ee, "optimem" },
	{ 0x0040ef, "hypercom, inc." },
	{ 0x0040f0, "micro systems inc" },
	{ 0x0040f1, "chuo electronics co., ltd." },
	{ 0x0040f2, "janich & klass computertechnik" },
	{ 0x0040f3, "netcor" },
	{ 0x0040f4, "cameo communications, inc." },
	{ 0x0040f5, "oem engines" },
	{ 0x0040f6, "katron computers inc" },
	{ 0x0040f7, "polaroid medical imaging sys." },
	{ 0x0040f8, "systemhaus discom" },
	{ 0x0040f9, "combinet" },
	{ 0x0040fa, "microboards inc" },
	{ 0x0040fb, "cascade communications corp." },
	{ 0x0040fc, "ibr computer technik gmbh" },
	{ 0x0040fd, "lxe" },
	{ 0x0040fe, "symplex communications" },
	{ 0x0040ff, "telebit corporation personal netblazer" },
	{ 0x004854, "digital semiconductor 21143/2 based 10/100" },
	{ 0x004f49, "realtek" },
	{ 0x004f4b, "pine technology ltd." },
	{ 0x005000, "nexo communications, inc." },
	{ 0x005001, "yamashita systems corp." },
	{ 0x005002, "omnisec ag" },
	{ 0x005003, "gretag macbeth ag" },
	{ 0x005004, "3com 3c90x" },
	{ 0x005006, "tac ab" },
	{ 0x005007, "siemens telecommunication" },
	{ 0x005008, "tiva microcomputer corp. (tmc)" },
	{ 0x005009, "philips broadband networks" },
	{ 0x00500a, "iris technologies, inc." },
	{ 0x00500b, "cisco systems, inc." },
	{ 0x00500c, "etek labs, inc." },
	{ 0x00500d, "satori electoric co., ltd." },
	{ 0x00500e, "chromatis networks,inc." },
	{ 0x00500f, "cisco" },
	{ 0x005010, "novanet learning, inc." },
	{ 0x005012, "cbl - gmbh" },
	{ 0x005013, "chaparral technologies, inc." },
	{ 0x005014, "cisco systems, inc." },
	{ 0x005015, "bright star engineering" },
	{ 0x005016, "sst/woodhead industries" },
	{ 0x005017, "rsr s.r.l." },
	{ 0x005018, "advanced multimedia internet" },
	{ 0x005019, "spring tide networks, inc." },
	{ 0x00501a, "uisiqn" },
	{ 0x00501b, "abl canada, inc." },
	{ 0x00501c, "jatom systems, inc." },
	{ 0x00501e, "miranda technologies, inc." },
	{ 0x00501f, "mrg systems, ltd." },
	{ 0x005020, "mediastar co., ltd." },
	{ 0x005021, "eis international, inc." },
	{ 0x005022, "zonet technology, inc." },
	{ 0x005023, "pg design electronics, inc." },
	{ 0x005024, "navic systems, inc." },
	{ 0x005026, "cosystems, inc." },
	{ 0x005027, "genicom corporation" },
	{ 0x005028, "aval communications" },
	{ 0x005029, "1394 printer working group" },
	{ 0x00502a, "cisco systems, inc." },
	{ 0x00502b, "genrad ltd." },
	{ 0x00502c, "soyo computer, inc." },
	{ 0x00502d, "accel, inc." },
	{ 0x00502e, "cambex corporation" },
	{ 0x00502f, "tollbridge technologies, inc." },
	{ 0x005030, "future plus systems" },
	{ 0x005031, "aeroflex laboratories, inc." },
	{ 0x005032, "picazo communications, inc." },
	{ 0x005033, "mayan networks" },
	{ 0x005036, "netcam, ltd." },
	{ 0x005037, "koga electronics co." },
	{ 0x005038, "dain telecom co., ltd." },
	{ 0x005039, "mariner networks" },
	{ 0x00503a, "datong electronics ltd." },
	{ 0x00503b, "mediafire corporation" },
	{ 0x00503c, "tsinghua novel electronics" },
	{ 0x00503e, "cisco systems, inc." },
	{ 0x00503f, "anchor games" },
	{ 0x005040, "emware, inc." },
	{ 0x005041, "ctx opto electronic corp." },
	{ 0x005042, "sci manufacturing" },
	{ 0x005043, "marvell semiconductor, inc." },
	{ 0x005044, "asaca corporation" },
	{ 0x005045, "rioworks solutions, inc." },
	{ 0x005046, "menicx international co., ltd." },
	{ 0x005048, "infolibria" },
	{ 0x005049, "ellacoya networks, inc." },
	{ 0x00504a, "elteco a.s." },
	{ 0x00504b, "barco n.v. bcs" },
	{ 0x00504c, "galil motion control, inc." },
	{ 0x00504d, "repotec group" },
	{ 0x00504e, "umc um9008 ne2000-compatible isa card for pc" },
	{ 0x00504f, "olencom electronics" },
	{ 0x005050, "cisco" },
	{ 0x005051, "iwatsu electric co., ltd." },
	{ 0x005052, "tiara networks, inc." },
	{ 0x005053, "cisco systems, inc." },
	{ 0x005054, "cisco systems, inc." },
	{ 0x005055, "doms a/s" },
	{ 0x005056, "vmware" },
	{ 0x005057, "broadband access systems" },
	{ 0x005058, "vegastream limited" },
	{ 0x005059, "suite technology systems" },
	{ 0x00505a, "network alchemy, inc." },
	{ 0x00505b, "kawasaki lsi u.s.a., inc." },
	{ 0x00505c, "tundo corporation" },
	{ 0x00505e, "digitek micrologic s.a." },
	{ 0x00505f, "brand innovators" },
	{ 0x005060, "tandberg telecom as" },
	{ 0x005062, "kouwell electronics corp.  **" },
	{ 0x005063, "oy comsel system ab" },
	{ 0x005064, "cae electronics" },
	{ 0x005065, "densei-lambad co., ltd." },
	{ 0x005066, "atecom gmbh advanced" },
	{ 0x005067, "aerocomm, inc." },
	{ 0x005068, "electronic industries" },
	{ 0x005069, "pixstream incorporated" },
	{ 0x00506a, "edeva, inc." },
	{ 0x00506b, "spx-ateg" },
	{ 0x00506c, "g & l beijer electronics ab" },
	{ 0x00506d, "videojet systems" },
	{ 0x00506e, "corder engineering corporation" },
	{ 0x00506f, "g-connect" },
	{ 0x005070, "chaintech computer co., ltd." },
	{ 0x005071, "aiwa co., ltd." },
	{ 0x005072, "corvis corporation" },
	{ 0x005073, "cisco systems, inc." },
	{ 0x005074, "advanced hi-tech corp." },
	{ 0x005075, "kestrel solutions" },
	{ 0x005076, "ibm" },
	{ 0x005077, "prolific technology, inc." },
	{ 0x005078, "megaton house, ltd." },
	{ 0x00507a, "xpeed, inc." },
	{ 0x00507b, "merlot communications" },
	{ 0x00507c, "videocon ag" },
	{ 0x00507d, "ifp" },
	{ 0x00507e, "newer technology" },
	{ 0x00507f, "draytek corp." },
	{ 0x005080, "cisco systems, inc." },
	{ 0x005081, "murata machinery, ltd." },
	{ 0x005082, "foresson corporation" },
	{ 0x005083, "gilbarco, inc." },
	{ 0x005084, "atl products" },
	{ 0x005086, "telkom sa, ltd." },
	{ 0x005087, "terasaki electric co., ltd." },
	{ 0x005088, "amano corporation" },
	{ 0x005089, "safety management systems" },
	{ 0x00508b, "compaq computer corporation" },
	{ 0x00508c, "rsi systems" },
	{ 0x00508d, "abit computer corporation" },
	{ 0x00508e, "optimation, inc." },
	{ 0x00508f, "asita technologies int'l ltd." },
	{ 0x005090, "dctri" },
	{ 0x005091, "netaccess, inc." },
	{ 0x005092, "rigaku industrial corporation" },
	{ 0x005093, "boeing" },
	{ 0x005094, "pace micro technology plc" },
	{ 0x005095, "peracom networks" },
	{ 0x005096, "salix technologies, inc." },
	{ 0x005097, "mmc-embedded" },
	{ 0x005098, "globaloop, ltd." },
	{ 0x005099, "3com europe, ltd." },
	{ 0x00509a, "tag electronic systems" },
	{ 0x00509b, "switchcore ab" },
	{ 0x00509c, "beta research" },
	{ 0x00509d, "the industree b.v." },
	{ 0x00509e, "les technologies" },
	{ 0x00509f, "horizon computer" },
	{ 0x0050a0, "delta computer systems, inc." },
	{ 0x0050a1, "carlo gavazzi, inc." },
	{ 0x0050a2, "cisco" },
	{ 0x0050a3, "transmedia communications, inc" },
	{ 0x0050a4, "io tech, inc." },
	{ 0x0050a5, "capitol business systems, ltd." },
	{ 0x0050a6, "optronics" },
	{ 0x0050a7, "cisco systems, inc." },
	{ 0x0050a8, "opencon systems, inc." },
	{ 0x0050a9, "moldat wireless technolgies" },
	{ 0x0050aa, "konica corporation" },
	{ 0x0050ab, "naltec, inc." },
	{ 0x0050ac, "maple computer corporation" },
	{ 0x0050ad, "communique wireless corp." },
	{ 0x0050ae, "iwaki electronics co., ltd." },
	{ 0x0050af, "intergon, inc." },
	{ 0x0050b0, "technology atlanta corporation" },
	{ 0x0050b1, "giddings & lewis" },
	{ 0x0050b2, "brodel automation" },
	{ 0x0050b3, "voiceboard corporation" },
	{ 0x0050b4, "satchwell control systems, ltd" },
	{ 0x0050b5, "fichet-bauche" },
	{ 0x0050b6, "good way ind. co., ltd." },
	{ 0x0050b7, "boser technology co., ltd." },
	{ 0x0050b8, "inova computers gmbh & co. kg" },
	{ 0x0050b9, "xitron technologies, inc." },
	{ 0x0050ba, "d-link" },
	{ 0x0050bb, "cms technologies" },
	{ 0x0050bc, "hammer storage solutions" },
	{ 0x0050bd, "cisco" },
	{ 0x0050be, "fast multimedia ag" },
	{ 0x0050bf, "mototech inc." },
	{ 0x0050c0, "gatan, inc." },
	{ 0x0050c1, "gemflex networks, ltd." },
	{ 0x0050c2, "ieee registration authority" },
	{ 0x0050c4, "imd" },
	{ 0x0050c5, "ads technologies, inc." },
	{ 0x0050c6, "loop telecommunication" },
	{ 0x0050c8, "addonics communications, inc." },
	{ 0x0050c9, "maspro denkoh corp." },
	{ 0x0050ca, "net to net technologies" },
	{ 0x0050cb, "jetter" },
	{ 0x0050cc, "xyratex" },
	{ 0x0050cd, "digianswer a/s" },
	{ 0x0050ce, "lg international corp." },
	{ 0x0050cf, "vanlink communication" },
	{ 0x0050d0, "minerva systems" },
	{ 0x0050d1, "cisco systems, inc." },
	{ 0x0050d2, "bae systems canada, inc." },
	{ 0x0050d3, "digital audio" },
	{ 0x0050d4, "joohong information &" },
	{ 0x0050d5, "ad systems corp." },
	{ 0x0050d6, "atlas copco tools ab" },
	{ 0x0050d7, "telstrat" },
	{ 0x0050d8, "unicorn computer corp." },
	{ 0x0050d9, "engetron-engenharia eletronica" },
	{ 0x0050da, "3com corporation" },
	{ 0x0050db, "contemporary control" },
	{ 0x0050dc, "tas telefonbau a. schwabe" },
	{ 0x0050dd, "serra soldadura, s.a." },
	{ 0x0050de, "signum systems corp." },
	{ 0x0050df, "airfiber, inc." },
	{ 0x0050e1, "ns tech electronics sdn bhd" },
	{ 0x0050e2, "cisco" },
	{ 0x0050e3, "telegate" },
	{ 0x0050e4, "apple computer, inc." },
	{ 0x0050e6, "hakusan corporation" },
	{ 0x0050e7, "paradise innovations (asia)" },
	{ 0x0050e8, "nomadix inc." },
	{ 0x0050ea, "xel communictions, inc." },
	{ 0x0050eb, "alpha-top corporation" },
	{ 0x0050ec, "olicom a/s" },
	{ 0x0050ed, "anda networks" },
	{ 0x0050ee, "tek digitel corporation" },
	{ 0x0050ef, "spe systemhaus gmbh" },
	{ 0x0050f0, "cisco systems, inc." },
	{ 0x0050f1, "libit signal processing, ltd." },
	{ 0x0050f2, "microsoft corp." },
	{ 0x0050f3, "global net information co.,ltd" },
	{ 0x0050f4, "sigmatek gmbh & co. kg" },
	{ 0x0050f6, "pan-international" },
	{ 0x0050f7, "venture manufacturing" },
	{ 0x0050f8, "entrega technologies, inc." },
	{ 0x0050fa, "oxtel, ltd." },
	{ 0x0050fb, "vsk electronics" },
	{ 0x0050fc, "edimax technology co., ltd." },
	{ 0x0050fd, "isioncomm co., ltd." },
	{ 0x0050fe, "pctvnet asa" },
	{ 0x0050ff, "hakko electronics co., ltd." },
	{ 0x005500, "xerox" },
	{ 0x006000, "xycom inc." },
	{ 0x006001, "innosys, inc." },
	{ 0x006002, "screen subtitling systems, ltd" },
	{ 0x006003, "teraoka weigh system pte, ltd." },
	{ 0x006004, "computadores modulares sa" },
	{ 0x006005, "feedback data ltd." },
	{ 0x006006, "sotec co., ltd" },
	{ 0x006007, "acres gaming, inc." },
	{ 0x006008, "3com" },
	{ 0x006009, "cisco" },
	{ 0x00600a, "sord computer corporation" },
	{ 0x00600b, "logware gmbh" },
	{ 0x00600c, "applied data systems, inc." },
	{ 0x00600d, "microdesign gmbh" },
	{ 0x00600e, "wavenet international, inc." },
	{ 0x00600f, "westell, inc." },
	{ 0x006010, "network machines, inc." },
	{ 0x006011, "crystal semiconductor corp." },
	{ 0x006012, "power computing corporation" },
	{ 0x006013, "netstal maschinen ag" },
	{ 0x006014, "edec co., ltd." },
	{ 0x006015, "net2net corporation" },
	{ 0x006016, "clariion" },
	{ 0x006017, "tokimec inc." },
	{ 0x006018, "stellar one corporation" },
	{ 0x006019, "boehringer mannheim corp." },
	{ 0x00601a, "keithley instruments" },
	{ 0x00601b, "mesa electronics" },
	{ 0x00601c, "telxon corporation" },
	{ 0x00601d, "lucent" },
	{ 0x00601e, "softlab, inc." },
	{ 0x00601f, "stallion technologies" },
	{ 0x006020, "pivotal networking, inc." },
	{ 0x006021, "dsc corporation" },
	{ 0x006022, "vicom systems, inc." },
	{ 0x006023, "pericom semiconductor corp." },
	{ 0x006024, "gradient technologies, inc." },
	{ 0x006025, "active imaging inc." },
	{ 0x006026, "viking components, inc." },
	{ 0x006027, "superior modular products" },
	{ 0x006028, "macrovision corporation" },
	{ 0x006029, "cary peripherals inc." },
	{ 0x00602a, "symicron computer" },
	{ 0x00602b, "peak audio" },
	{ 0x00602c, "linx data terminals, inc." },
	{ 0x00602d, "alerton technologies, inc." },
	{ 0x00602e, "cyclades corporation" },
	{ 0x00602f, "cisco" },
	{ 0x006030, "villagetronic used on amiga" },
	{ 0x006031, "hrk systems" },
	{ 0x006032, "i-cube, inc." },
	{ 0x006033, "acuity imaging, inc." },
	{ 0x006034, "robert bosch gmbh" },
	{ 0x006035, "dallas semiconductor, inc." },
	{ 0x006036, "austrian research center" },
	{ 0x006037, "philips semiconductors" },
	{ 0x006038, "nortel networks" },
	{ 0x006039, "sancom technology, inc." },
	{ 0x00603a, "quick controls ltd." },
	{ 0x00603b, "amtec spa" },
	{ 0x00603c, "hagiwara sys-com co., ltd." },
	{ 0x00603d, "3cx" },
	{ 0x00603e, "cisco" },
	{ 0x00603f, "patapsco designs" },
	{ 0x006040, "netro corp." },
	{ 0x006041, "3a international, inc." },
	{ 0x006042, "tks (usa), inc." },
	{ 0x006043, "comsoft systems, inc." },
	{ 0x006044, "litton/poly-scientific" },
	{ 0x006045, "pathlight technologies" },
	{ 0x006046, "vmetro, inc." },
	{ 0x006047, "cisco" },
	{ 0x006048, "emc corporation" },
	{ 0x006049, "vina technologies" },
	{ 0x00604a, "saic ideas group" },
	{ 0x00604b, "biodata gmbh" },
	{ 0x00604c, "sat" },
	{ 0x00604d, "mmc networks, inc." },
	{ 0x00604e, "cycle computer (sun motherboard replacements)" },
	{ 0x00604f, "suzuki mfg. co., ltd." },
	{ 0x006050, "internix inc." },
	{ 0x006051, "quality semiconductor" },
	{ 0x006052, "realtek (rtl 8029 == pci ne2000)" },
	{ 0x006053, "toyoda machine works, ltd." },
	{ 0x006054, "controlware gmbh" },
	{ 0x006055, "cornell university" },
	{ 0x006056, "network tools, inc." },
	{ 0x006057, "murata manufacturing co., ltd." },
	{ 0x006058, "copper mountain" },
	{ 0x006059, "technical communications corp." },
	{ 0x00605a, "celcore, inc." },
	{ 0x00605b, "intraserver technology inc." },
	{ 0x00605c, "cisco" },
	{ 0x00605d, "scanivalve corp." },
	{ 0x00605e, "liberty technology networking" },
	{ 0x00605f, "nippon unisoft corporation" },
	{ 0x006060, "dawning technologies, inc." },
	{ 0x006061, "whistle communications corp." },
	{ 0x006062, "telesync, inc." },
	{ 0x006063, "psion dacom plc." },
	{ 0x006064, "netcomm limited" },
	{ 0x006065, "bernecker & rainer" },
	{ 0x006066, "lacroix technolgie" },
	{ 0x006067, "acer lan" },
	{ 0x006068, "eicon technology corporation" },
	{ 0x006069, "brocade communications systems" },
	{ 0x00606a, "mitsubishi wireless comm. inc." },
	{ 0x00606b, "aichi electronics co.,ltd." },
	{ 0x00606c, "arescom" },
	{ 0x00606d, "digital equipment corp." },
	{ 0x00606e, "davicom semiconductor, inc." },
	{ 0x00606f, "clarion corporation of america" },
	{ 0x006070, "cisco" },
	{ 0x006071, "midas lab, inc." },
	{ 0x006072, "vxl instruments, limited" },
	{ 0x006073, "redcreek communications, inc." },
	{ 0x006074, "qsc audio products" },
	{ 0x006075, "pentek, inc." },
	{ 0x006076, "schlumberger technologies" },
	{ 0x006077, "prisa networks" },
	{ 0x006078, "power measurement ltd." },
	{ 0x006079, "wavephore networks, inc." },
	{ 0x00607a, "dvs gmbh" },
	{ 0x00607b, "fore systems, inc." },
	{ 0x00607c, "waveaccess, ltd." },
	{ 0x00607d, "sentient networks inc." },
	{ 0x00607e, "gigalabs, inc." },
	{ 0x00607f, "aurora technologies, inc." },
	{ 0x006080, "microtronix datacom ltd." },
	{ 0x006081, "tv/com international" },
	{ 0x006082, "novalink technologies, inc." },
	{ 0x006083, "cisco" },
	{ 0x006084, "digital video" },
	{ 0x006085, "storage concepts" },
	{ 0x006086, "logic replacement tech. ltd." },
	{ 0x006087, "kansai electric co., ltd." },
	{ 0x006088, "white mountain dsp, inc." },
	{ 0x006089, "xata" },
	{ 0x00608a, "citadel computer" },
	{ 0x00608b, "confertech international" },
	{ 0x00608c, "3com" },
	{ 0x00608d, "unipulse corp." },
	{ 0x00608e, "he electronics, technologie &" },
	{ 0x00608f, "tekram technology co., ltd." },
	{ 0x006090, "able communications, inc." },
	{ 0x006091, "first pacific networks, inc." },
	{ 0x006092, "micro/sys, inc." },
	{ 0x006093, "varian" },
	{ 0x006094, "amd pcnet pci" },
	{ 0x006095, "accu-time systems, inc." },
	{ 0x006096, "t.s. microtech inc." },
	{ 0x006097, "3com" },
	{ 0x006098, "ht communications" },
	{ 0x006099, "lan media corporation" },
	{ 0x00609a, "njk techno co." },
	{ 0x00609b, "astro-med, inc." },
	{ 0x00609c, "perkin-elmer corporation" },
	{ 0x00609d, "pmi food equipment group" },
	{ 0x00609e, "x3 - information technology" },
	{ 0x00609f, "phast corporation" },
	{ 0x0060a0, "switched network" },
	{ 0x0060a1, "vpnet" },
	{ 0x0060a2, "nihon unisys limited co." },
	{ 0x0060a3, "continuum technology corp." },
	{ 0x0060a4, "grinaker system technologies" },
	{ 0x0060a5, "performance telecom corp." },
	{ 0x0060a6, "particle measuring systems" },
	{ 0x0060a7, "microsens gmbh & co. kg" },
	{ 0x0060a8, "tidomat ab" },
	{ 0x0060a9, "gesytec mbh" },
	{ 0x0060aa, "intelligent devices inc. (idi)" },
	{ 0x0060ab, "larscom incorporated" },
	{ 0x0060ac, "resilience corporation" },
	{ 0x0060ad, "megachips corporation" },
	{ 0x0060ae, "trio information systems ab" },
	{ 0x0060af, "pacific micro data, inc." },
	{ 0x0060b0, "hp" },
	{ 0x0060b1, "input/output, inc." },
	{ 0x0060b2, "process control corp." },
	{ 0x0060b3, "z-com, inc." },
	{ 0x0060b4, "glenayre r&d inc." },
	{ 0x0060b5, "keba gmbh" },
	{ 0x0060b6, "land computer co., ltd." },
	{ 0x0060b7, "channelmatic, inc." },
	{ 0x0060b8, "corelis inc." },
	{ 0x0060b9, "nitsuko corporation" },
	{ 0x0060ba, "sahara networks, inc." },
	{ 0x0060bb, "cabletron - netlink, inc." },
	{ 0x0060bc, "keunyoung electronics &" },
	{ 0x0060bd, "hubbell-pulsecom" },
	{ 0x0060be, "webtronics" },
	{ 0x0060bf, "macraigor systems, inc." },
	{ 0x0060c0, "nera as" },
	{ 0x0060c1, "wavespan corporation" },
	{ 0x0060c2, "mpl ag" },
	{ 0x0060c3, "netvision corporation" },
	{ 0x0060c4, "soliton systems k.k." },
	{ 0x0060c5, "ancot corp." },
	{ 0x0060c6, "dcs ag" },
	{ 0x0060c7, "amati communications corp." },
	{ 0x0060c8, "kuka welding systems & robots" },
	{ 0x0060c9, "controlnet, inc." },
	{ 0x0060ca, "harmonic systems incorporated" },
	{ 0x0060cb, "hitachi zosen corporation" },
	{ 0x0060cc, "emtrak, incorporated" },
	{ 0x0060cd, "videoserver, inc." },
	{ 0x0060ce, "acclaim communications" },
	{ 0x0060cf, "alteon networks, inc." },
	{ 0x0060d0, "snmp research incorporated" },
	{ 0x0060d1, "cascade communications" },
	{ 0x0060d2, "lucent technologies taiwan" },
	{ 0x0060d3, "at&t" },
	{ 0x0060d4, "eldat communication ltd." },
	{ 0x0060d5, "miyachi technos corp." },
	{ 0x0060d6, "novatel wireless technologies" },
	{ 0x0060d7, "ecole polytechnique federale" },
	{ 0x0060d8, "elmic systems, inc." },
	{ 0x0060d9, "transys networks inc." },
	{ 0x0060da, "jbm electronics co." },
	{ 0x0060db, "ntp elektronik a/s" },
	{ 0x0060dc, "toyo communication equipment" },
	{ 0x0060dd, "myricom, inc." },
	{ 0x0060de, "kayser-threde gmbh" },
	{ 0x0060df, "inrange technologies corp." },
	{ 0x0060e0, "axiom technology co., ltd." },
	{ 0x0060e1, "orckit communications ltd." },
	{ 0x0060e2, "quest engineering & dev." },
	{ 0x0060e3, "arbin instruments" },
	{ 0x0060e4, "compuserve, inc." },
	{ 0x0060e5, "fuji automation co., ltd." },
	{ 0x0060e6, "shomiti systems incorporated" },
	{ 0x0060e7, "randata" },
	{ 0x0060e8, "hitachi computer products" },
	{ 0x0060e9, "atop technologies, inc." },
	{ 0x0060ea, "streamlogic" },
	{ 0x0060eb, "fourthtrack systems" },
	{ 0x0060ec, "hermary opto electronics inc." },
	{ 0x0060ed, "ricardo test automation ltd." },
	{ 0x0060ee, "apollo" },
	{ 0x0060ef, "flytech technology co., ltd." },
	{ 0x0060f0, "johnson & johnson medical, inc" },
	{ 0x0060f1, "exp computer, inc." },
	{ 0x0060f2, "lasergraphics, inc." },
	{ 0x0060f3, "netcom systems, inc." },
	{ 0x0060f4, "advanced computer solutions," },
	{ 0x0060f5, "phobos fastethernet for unix ws" },
	{ 0x0060f6, "nextest communication" },
	{ 0x0060f7, "datafusion systems" },
	{ 0x0060f8, "loran international techn. inc" },
	{ 0x0060f9, "diamond lane communications" },
	{ 0x0060fa, "educational technology" },
	{ 0x0060fb, "packeteer, inc." },
	{ 0x0060fc, "conservation through" },
	{ 0x0060fd, "netics, inc." },
	{ 0x0060fe, "lynx system developers, inc." },
	{ 0x0060ff, "quvis, inc." },
	{ 0x0070b0, "m/a-com inc. companies" },
	{ 0x0070b3, "data recall ltd." },
	{ 0x008000, "multitech systems inc" },
	{ 0x008001, "periphonics corporation" },
	{ 0x008002, "satelcom (uk) ltd" },
	{ 0x008003, "hytec electronics ltd." },
	{ 0x008004, "antlow computers, ltd." },
	{ 0x008005, "cactus computer inc." },
	{ 0x008006, "compuadd corporation" },
	{ 0x008007, "dlog nc-systeme" },
	{ 0x008008, "dynatech computer systems" },
	{ 0x008009, "jupiter systems (older mx-600 series machines)" },
	{ 0x00800a, "japan computer corp." },
	{ 0x00800b, "csk corporation" },
	{ 0x00800c, "videcom limited" },
	{ 0x00800d, "vosswinkel fu" },
	{ 0x00800e, "atlantix corporation" },
	{ 0x00800f, "smc" },
	{ 0x008010, "commodore" },
	{ 0x008011, "digital systems int'l. inc." },
	{ 0x008012, "ims corp. ims failure analysis tester" },
	{ 0x008013, "thomas conrad corp." },
	{ 0x008014, "esprit systems" },
	{ 0x008015, "seiko systems inc" },
	{ 0x008016, "wandel & goltermann" },
	{ 0x008017, "pfu" },
	{ 0x008018, "kobe steel, ltd." },
	{ 0x008019, "dayna communications etherprint product" },
	{ 0x00801a, "bell atlantic" },
	{ 0x00801b, "kodiak technology" },
	{ 0x00801c, "cisco" },
	{ 0x00801d, "integrated inference machines" },
	{ 0x00801e, "xinetron, inc." },
	{ 0x00801f, "krupp atlas electronik gmbh" },
	{ 0x008020, "network products" },
	{ 0x008021, "newbridge networks corporation" },
	{ 0x008022, "scan-optics" },
	{ 0x008023, "integrated business networks" },
	{ 0x008024, "kalpana" },
	{ 0x008025, "stollmann gmbh" },
	{ 0x008026, "network products corporation" },
	{ 0x008027, "adaptive systems, inc." },
	{ 0x008028, "tradpost (hk) ltd" },
	{ 0x008029, "microdyne corporation" },
	{ 0x00802a, "test systems & simulations inc" },
	{ 0x00802b, "integrated marketing co" },
	{ 0x00802c, "the sage group plc" },
	{ 0x00802d, "xylogics, inc. annex terminal servers" },
	{ 0x00802e, "plexcom, inc." },
	{ 0x00802f, "national instruments corp." },
	{ 0x008030, "nexus electronics" },
	{ 0x008031, "basys, corp." },
	{ 0x008032, "access co., ltd." },
	{ 0x008033, "formation (?)" },
	{ 0x008034, "smt-goupil" },
	{ 0x008035, "technology works" },
	{ 0x008036, "reflex manufacturing systems" },
	{ 0x008037, "ericsson business comm." },
	{ 0x008038, "data research & applications" },
	{ 0x008039, "alcatel stc australia" },
	{ 0x00803a, "varityper, inc." },
	{ 0x00803b, "apt communications, inc." },
	{ 0x00803c, "tvs electronics ltd" },
	{ 0x00803d, "surigiken co ltd" },
	{ 0x00803e, "synernetics" },
	{ 0x00803f, "hyundai electronics" },
	{ 0x008040, "john fluke manufacturing co." },
	{ 0x008041, "veb kombinat robotron" },
	{ 0x008042, "force computers" },
	{ 0x008043, "networld inc" },
	{ 0x008044, "systech computer corp." },
	{ 0x008045, "matshta" },
	{ 0x008046, "university of toronto" },
	{ 0x008047, "in-net corp." },
	{ 0x008048, "compex, used by commodore and dec at least" },
	{ 0x008049, "nissin electric co ltd" },
	{ 0x00804a, "pro-log" },
	{ 0x00804b, "eagle technologies pty.ltd." },
	{ 0x00804c, "contec co., ltd." },
	{ 0x00804d, "cyclone microsystems, inc." },
	{ 0x00804e, "apex computer company" },
	{ 0x00804f, "daikin industries, ltd." },
	{ 0x008050, "ziatech" },
	{ 0x008051, "adc fibermux" },
	{ 0x008052, "network professor" },
	{ 0x008053, "intellicom, inc." },
	{ 0x008054, "frontier technologies corp." },
	{ 0x008055, "fermilab" },
	{ 0x008056, "sphinx elektronik gmbh" },
	{ 0x008057, "adsoft ltd" },
	{ 0x008058, "printer systems corporation" },
	{ 0x008059, "stanley electric co., ltd" },
	{ 0x00805a, "tulip" },
	{ 0x00805b, "condor systems, inc." },
	{ 0x00805c, "agilis(?)" },
	{ 0x00805d, "canstar" },
	{ 0x00805e, "lsi logic corporation" },
	{ 0x00805f, "compaq" },
	{ 0x008060, "network interface corporation" },
	{ 0x008061, "litton systems, inc." },
	{ 0x008062, "interface co." },
	{ 0x008063, "richard hirschmann gmbh & co" },
	{ 0x008064, "wyse" },
	{ 0x008065, "cybergraphic systems pty ltd." },
	{ 0x008066, "arcom control systems, ltd." },
	{ 0x008067, "square d company" },
	{ 0x008068, "yamatech scientific ltd." },
	{ 0x008069, "computone systems" },
	{ 0x00806a, "eri (empac research inc.)" },
	{ 0x00806b, "schmid telecommunication" },
	{ 0x00806c, "cegelec projects ltd" },
	{ 0x00806d, "century systems corp." },
	{ 0x00806e, "nippon steel corporation" },
	{ 0x00806f, "onelan ltd" },
	{ 0x008070, "computadoras micron" },
	{ 0x008071, "sai technology" },
	{ 0x008072, "microplex systems ltd" },
	{ 0x008073, "dwb associates" },
	{ 0x008074, "fisher controls" },
	{ 0x008075, "parsytec gmbh" },
	{ 0x008076, "mcnc" },
	{ 0x008077, "brother industries, ltd." },
	{ 0x008078, "practical peripherals, inc." },
	{ 0x008079, "microbus designs ltd" },
	{ 0x00807a, "aitech systems ltd." },
	{ 0x00807b, "artel communications corp." },
	{ 0x00807c, "fibercom" },
	{ 0x00807d, "equinox systems inc" },
	{ 0x00807e, "southern pacific ltd." },
	{ 0x00807f, "dy-4 incorporated" },
	{ 0x008080, "datamedia corporation" },
	{ 0x008081, "kendall square research corp." },
	{ 0x008082, "pep modular computers gmbh" },
	{ 0x008083, "amdahl" },
	{ 0x008084, "the cloud inc." },
	{ 0x008085, "h-three systems corporation" },
	{ 0x008086, "computer generation inc." },
	{ 0x008087, "okidata" },
	{ 0x008088, "victor company of japan, ltd." },
	{ 0x008089, "tecnetics (pty) ltd." },
	{ 0x00808a, "summit (?)" },
	{ 0x00808b, "dacoll limited" },
	{ 0x00808c, "netscout systems (formerly frontier software development)" },
	{ 0x00808d, "westcove technology bv" },
	{ 0x00808e, "radstone technology" },
	{ 0x00808f, "c. itoh electronics, inc." },
	{ 0x008090, "microtek" },
	{ 0x008091, "tokyo electric co.,ltd" },
	{ 0x008092, "japan computer industry, inc." },
	{ 0x008093, "xyron corporation" },
	{ 0x008094, "sattcontrol ab" },
	{ 0x008095, "basic merton handelsges.m.b.h." },
	{ 0x008096, "hds" },
	{ 0x008097, "centralp automatismes" },
	{ 0x008098, "tdk corporation" },
	{ 0x008099, "klockner moeller ipc" },
	{ 0x00809a, "novus networks ltd" },
	{ 0x00809b, "justsystem corporation" },
	{ 0x00809c, "luxcom, inc." },
	{ 0x00809d, "datacraft manufactur'g pty ltd" },
	{ 0x00809e, "datus gmbh" },
	{ 0x00809f, "alcatel" },
	{ 0x0080a0, "edisa hewlett packard s/a" },
	{ 0x0080a1, "microtest" },
	{ 0x0080a2, "creative electronic systems" },
	{ 0x0080a3, "lantronix (see also 0800a3)" },
	{ 0x0080a4, "liberty electronics" },
	{ 0x0080a5, "speed international" },
	{ 0x0080a6, "republic technology inc" },
	{ 0x0080a7, "measurex corp" },
	{ 0x0080a8, "vitacom corporation" },
	{ 0x0080a9, "clearpoint research" },
	{ 0x0080aa, "maxpeed" },
	{ 0x0080ab, "dukane network integration" },
	{ 0x0080ac, "imlogix, division of genesys" },
	{ 0x0080ad, "telebit" },
	{ 0x0080ae, "hughes network systems" },
	{ 0x0080af, "allumer co., ltd." },
	{ 0x0080b0, "advanced information" },
	{ 0x0080b1, "softcom a/s" },
	{ 0x0080b2, "net (network equipment technologies)" },
	{ 0x0080b3, "aval data corporation" },
	{ 0x0080b4, "sophia systems" },
	{ 0x0080b5, "united networks inc." },
	{ 0x0080b6, "themis corporation" },
	{ 0x0080b7, "stellar computer" },
	{ 0x0080b8, "bug, incorporated" },
	{ 0x0080b9, "arche technoligies inc." },
	{ 0x0080ba, "specialix (asia) pte ltd" },
	{ 0x0080bb, "hughes lan systems" },
	{ 0x0080bc, "hitachi engineering co., ltd" },
	{ 0x0080bd, "the furukawa electric co., ltd" },
	{ 0x0080be, "aries research" },
	{ 0x0080bf, "takaoka electric mfg. co. ltd." },
	{ 0x0080c0, "penril datability networks" },
	{ 0x0080c1, "lanex corporation" },
	{ 0x0080c2, "ieee 802.1 committee" },
	{ 0x0080c3, "bicc information systems & svc" },
	{ 0x0080c4, "document technologies, inc." },
	{ 0x0080c5, "novellco de mexico" },
	{ 0x0080c6, "soho" },
	{ 0x0080c7, "xircom" },
	{ 0x0080c8, "d-link (also solectek pocket adapters, and linksys pcmcia)" },
	{ 0x0080c9, "alberta microelectronic centre" },
	{ 0x0080ca, "netcom research incorporated" },
	{ 0x0080cb, "falco data products" },
	{ 0x0080cc, "microwave bypass systems" },
	{ 0x0080cd, "micronics computer, inc." },
	{ 0x0080ce, "broadcast television systems" },
	{ 0x0080cf, "embedded performance inc." },
	{ 0x0080d0, "computer products international" },
	{ 0x0080d1, "kimtron corporation" },
	{ 0x0080d2, "shinnihondenko co., ltd." },
	{ 0x0080d3, "shiva appletalk-ethernet interface" },
	{ 0x0080d4, "chase limited" },
	{ 0x0080d5, "cadre technologies" },
	{ 0x0080d6, "apple mac portable(?)" },
	{ 0x0080d7, "fantum electronics" },
	{ 0x0080d8, "network peripherals" },
	{ 0x0080d9, "emk elektronik" },
	{ 0x0080da, "bruel & kjaer" },
	{ 0x0080db, "graphon corporation" },
	{ 0x0080dc, "picker international" },
	{ 0x0080dd, "gmx inc/gimix" },
	{ 0x0080de, "gipsi s.a." },
	{ 0x0080df, "adc codenoll technology corp." },
	{ 0x0080e0, "xtp systems inc" },
	{ 0x0080e1, "stmicroelectronics" },
	{ 0x0080e2, "t.d.i. co., ltd." },
	{ 0x0080e3, "coral (?)" },
	{ 0x0080e4, "northwest digital systems, inc" },
	{ 0x0080e5, "mylex corporation" },
	{ 0x0080e6, "peer networks, inc." },
	{ 0x0080e7, "lynwood scientific dev ltd" },
	{ 0x0080e8, "cumulus corporatiion" },
	{ 0x0080e9, "madge networks" },
	{ 0x0080ea, "the fiber company" },
	{ 0x0080eb, "compcontrol b.v." },
	{ 0x0080ec, "supercomputing solutions, inc." },
	{ 0x0080ed, "iq technologies, inc." },
	{ 0x0080ee, "thomson csf" },
	{ 0x0080ef, "rational" },
	{ 0x0080f0, "kyushu matsushita electric co" },
	{ 0x0080f1, "opus" },
	{ 0x0080f2, "raycom systems inc" },
	{ 0x0080f3, "sun electronics corp" },
	{ 0x0080f4, "telemechanique electrique" },
	{ 0x0080f5, "quantel ltd" },
	{ 0x0080f6, "synergy microsystems" },
	{ 0x0080f7, "zenith communications products" },
	{ 0x0080f8, "mizar, inc." },
	{ 0x0080f9, "heurikon corporation" },
	{ 0x0080fa, "rwt gmbh" },
	{ 0x0080fb, "bvm limited" },
	{ 0x0080fc, "avatar corporation" },
	{ 0x0080fd, "exsceed corpration" },
	{ 0x0080fe, "azure technologies inc" },
	{ 0x0080ff, "soc. de teleinformatique rtc" },
	{ 0x009000, "diamond multimedia" },
	{ 0x009001, "nishimu elctronics industries" },
	{ 0x009002, "allgon ab" },
	{ 0x009003, "aplio" },
	{ 0x009004, "3com" },
	{ 0x009005, "protech systems co., ltd." },
	{ 0x009006, "hamamatsu photonics k.k." },
	{ 0x009007, "domex technology corp." },
	{ 0x009008, "han a systems, inc." },
	{ 0x009009, "i controls, inc." },
	{ 0x00900a, "proton electronic industrial" },
	{ 0x00900b, "lanner electronics, inc." },
	{ 0x00900c, "cisco systems, inc." },
	{ 0x00900d, "overland data inc." },
	{ 0x00900e, "handlink technologies, inc." },
	{ 0x00900f, "kawasaki heavy industries, ltd" },
	{ 0x009010, "simulation laboratories, inc." },
	{ 0x009011, "wavtrace, inc." },
	{ 0x009012, "globespan semiconductor, inc." },
	{ 0x009013, "samsan corp." },
	{ 0x009014, "rotork instruments, ltd." },
	{ 0x009015, "centigram communications corp." },
	{ 0x009016, "zac" },
	{ 0x009017, "zypcom, inc." },
	{ 0x009018, "ito electric industry co, ltd." },
	{ 0x009019, "hermes electronics co., ltd." },
	{ 0x00901a, "unisphere solutions" },
	{ 0x00901b, "digital controls" },
	{ 0x00901c, "mps software gmbh" },
	{ 0x00901d, "pec (nz) ltd." },
	{ 0x00901e, "selesta ingegne ria s.p.a." },
	{ 0x00901f, "adtec productions, inc." },
	{ 0x009020, "philips analytical x-ray b.v." },
	{ 0x009021, "cisco systems, inc." },
	{ 0x009022, "ivex" },
	{ 0x009023, "zilog inc." },
	{ 0x009024, "pipelinks, inc." },
	{ 0x009025, "vision systems ltd. pty" },
	{ 0x009026, "advanced switching" },
	{ 0x009027, "intel" },
	{ 0x009028, "nippon signal co., ltd." },
	{ 0x009029, "crypto ag" },
	{ 0x00902a, "communication devices, inc." },
	{ 0x00902b, "cisco ethernet switches and light streams" },
	{ 0x00902c, "data & control equipment ltd." },
	{ 0x00902d, "data electronics" },
	{ 0x00902e, "namco limited" },
	{ 0x00902f, "netcore systems, inc." },
	{ 0x009030, "honeywell-dating" },
	{ 0x009031, "mysticom, ltd." },
	{ 0x009032, "pelcombe group ltd." },
	{ 0x009033, "innovaphone gmbh" },
	{ 0x009034, "imagic, inc." },
	{ 0x009035, "alpha telecom, inc." },
	{ 0x009036, "ens, inc." },
	{ 0x009037, "acucomm, inc." },
	{ 0x009038, "fountain technologies, inc." },
	{ 0x009039, "shasta networks" },
	{ 0x00903a, "nihon media tool inc." },
	{ 0x00903b, "triems research lab, inc." },
	{ 0x00903c, "atlantic network systems" },
	{ 0x00903d, "biopac systems, inc." },
	{ 0x00903e, "n.v. philips industrial" },
	{ 0x00903f, "aztec radiomedia" },
	{ 0x009040, "castle networks, inc." },
	{ 0x009041, "applied digital access" },
	{ 0x009042, "eccs" },
	{ 0x009043, "nichibei denshi co., ltd." },
	{ 0x009044, "assured digital, inc." },
	{ 0x009045, "mariposa technology" },
	{ 0x009046, "dexdyne, ltd." },
	{ 0x009047, "giga fast e. ltd." },
	{ 0x009048, "zeal corporation" },
	{ 0x009049, "entridia corporation" },
	{ 0x00904a, "concur system technologies" },
	{ 0x00904b, "gemtek technology co., ltd." },
	{ 0x00904c, "epigram, inc." },
	{ 0x00904d, "spec s.a." },
	{ 0x00904e, "delem bv" },
	{ 0x00904f, "abb power t&d company, inc." },
	{ 0x009050, "teleste oy" },
	{ 0x009051, "ultimate technology corp." },
	{ 0x009052, "selcom elettronica s.r.l." },
	{ 0x009053, "daewoo electronics co., ltd." },
	{ 0x009054, "innovative semiconductors, inc" },
	{ 0x009055, "parker hannifin corporation" },
	{ 0x009056, "telestream, inc." },
	{ 0x009057, "aanetcom, inc." },
	{ 0x009058, "ultra electronics ltd." },
	{ 0x009059, "telecom device k.k." },
	{ 0x00905a, "dearborn group, inc." },
	{ 0x00905b, "raymond and lae engineering" },
	{ 0x00905c, "edmi" },
	{ 0x00905d, "netcom sicherheitstechnik gmbh" },
	{ 0x00905e, "rauland-borg corporation" },
	{ 0x00905f, "cisco systems, inc." },
	{ 0x009060, "system create corp." },
	{ 0x009061, "pacific research & engineering" },
	{ 0x009062, "icp vortex computersysteme" },
	{ 0x009063, "coherent communications" },
	{ 0x009064, "thomson broadcast systems" },
	{ 0x009065, "finisar corporation" },
	{ 0x009066, "troika design, inc." },
	{ 0x009067, "walkabout computers, inc." },
	{ 0x009068, "dvt corp." },
	{ 0x009069, "juniper networks, inc." },
	{ 0x00906a, "turnstone systems, inc." },
	{ 0x00906b, "applied resources, inc." },
	{ 0x00906c, "gwt global weighing" },
	{ 0x00906d, "cisco" },
	{ 0x00906e, "praxon, inc." },
	{ 0x00906f, "cisco systems, inc." },
	{ 0x009070, "neo networks, inc." },
	{ 0x009071, "badger technology, inc." },
	{ 0x009072, "simrad as" },
	{ 0x009073, "gaio technology" },
	{ 0x009074, "argon networks, inc." },
	{ 0x009075, "nec do brasil s.a." },
	{ 0x009076, "fmt aircraft gate support" },
	{ 0x009077, "advanced fibre communications" },
	{ 0x009078, "mer telemanagement" },
	{ 0x009079, "clearone inc." },
	{ 0x00907a, "spectralink corp." },
	{ 0x00907b, "e-tech, inc." },
	{ 0x00907c, "digitalcast, inc." },
	{ 0x00907d, "home wireless networks" },
	{ 0x00907e, "vetronix corp." },
	{ 0x00907f, "watchguard technologies, inc." },
	{ 0x009080, "not limited, inc." },
	{ 0x009081, "aloha networks, inc." },
	{ 0x009082, "force institute" },
	{ 0x009083, "turbo communication, inc." },
	{ 0x009084, "atech system" },
	{ 0x009085, "golden enterprises, inc." },
	{ 0x009086, "cisco" },
	{ 0x009087, "itis" },
	{ 0x009088, "baxall security ltd." },
	{ 0x009089, "softcom microsystems, inc." },
	{ 0x00908a, "bayly communications, inc." },
	{ 0x00908b, "cell computing, inc." },
	{ 0x00908c, "etrend electronics, inc." },
	{ 0x00908d, "vickers electronics systems" },
	{ 0x00908e, "nortel networks broadband access" },
	{ 0x00908f, "audiocodes ltd." },
	{ 0x009090, "i-bus" },
	{ 0x009091, "digitalscape, inc." },
	{ 0x009092, "cisco" },
	{ 0x009093, "nanao corporation" },
	{ 0x009094, "osprey technologies, inc." },
	{ 0x009095, "universal avionics" },
	{ 0x009096, "askey computer corp." },
	{ 0x009097, "sycamore networks" },
	{ 0x009098, "sbc designs, inc." },
	{ 0x009099, "allied telesis,k.k." },
	{ 0x00909a, "one world systems, inc." },
	{ 0x00909b, "markpoint ab" },
	{ 0x00909c, "combox, ltd." },
	{ 0x00909d, "gse systems, inc." },
	{ 0x00909e, "delphi engineering group" },
	{ 0x00909f, "digi-data corporation" },
	{ 0x0090a0, "8x8 inc." },
	{ 0x0090a1, "flying pig systems, ltd." },
	{ 0x0090a2, "cybertan technology, inc." },
	{ 0x0090a3, "medialincs co., ltd." },
	{ 0x0090a4, "altiga networks" },
	{ 0x0090a5, "spectra logic" },
	{ 0x0090a6, "cisco systems, inc." },
	{ 0x0090a7, "clientec corporation" },
	{ 0x0090a8, "ninetiles networks ltd." },
	{ 0x0090a9, "western digital" },
	{ 0x0090aa, "indigo active vision" },
	{ 0x0090ab, "cisco" },
	{ 0x0090ac, "optivision, inc." },
	{ 0x0090ad, "aspect electronics, inc." },
	{ 0x0090ae, "italtel spa" },
	{ 0x0090af, "j. morita mfg. corp." },
	{ 0x0090b0, "vadem" },
	{ 0x0090b1, "cisco" },
	{ 0x0090b2, "avici systems inc." },
	{ 0x0090b3, "agranat systems" },
	{ 0x0090b4, "willowbrook technologies" },
	{ 0x0090b5, "nikon corporation" },
	{ 0x0090b6, "fibex systems" },
	{ 0x0090b7, "digital lightwave, inc." },
	{ 0x0090b8, "rohde & schwarz gmbh & co. kg" },
	{ 0x0090b9, "beran instruments ltd." },
	{ 0x0090ba, "valid networks, inc." },
	{ 0x0090bb, "tainet communication system" },
	{ 0x0090bc, "telemann co., ltd." },
	{ 0x0090bd, "omnia communications, inc." },
	{ 0x0090be, "ibc/integrated business" },
	{ 0x0090bf, "cisco systems, inc." },
	{ 0x0090c0, "k.j. law engineers, inc." },
	{ 0x0090c1, "eda industries" },
	{ 0x0090c2, "jk microsystems, inc." },
	{ 0x0090c3, "topic semiconductor corp." },
	{ 0x0090c4, "javelin systems, inc." },
	{ 0x0090c5, "internet magic, inc." },
	{ 0x0090c6, "optim systems, inc." },
	{ 0x0090c7, "icom inc." },
	{ 0x0090c8, "waverider communications" },
	{ 0x0090c9, "productivity enhancement" },
	{ 0x0090ca, "accord video" },
	{ 0x0090cb, "wireless online, inc." },
	{ 0x0090cc, "planex communications, inc." },
	{ 0x0090cd, "ent-empresa nacional" },
	{ 0x0090ce, "tetra gmbh" },
	{ 0x0090cf, "nortel" },
	{ 0x0090d0, "alcatel bell" },
	{ 0x0090d1, "leichu enterprise co., ltd." },
	{ 0x0090d2, "artel video systems" },
	{ 0x0090d3, "giesecke & devrient gmbh" },
	{ 0x0090d4, "bindview development corp." },
	{ 0x0090d5, "euphonix, inc." },
	{ 0x0090d6, "crystal group" },
	{ 0x0090d7, "netboost corp." },
	{ 0x0090d8, "whitecross systems" },
	{ 0x0090d9, "cisco systems, inc." },
	{ 0x0090da, "dynarc, inc." },
	{ 0x0090db, "next level communications" },
	{ 0x0090dc, "teco information systems" },
	{ 0x0090dd, "the miharu communications" },
	{ 0x0090de, "cardkey systems, inc." },
	{ 0x0090df, "mitsubishi chemical" },
	{ 0x0090e0, "systran corp." },
	{ 0x0090e1, "telena s.p.a." },
	{ 0x0090e2, "distributed processing" },
	{ 0x0090e3, "avex electronics inc." },
	{ 0x0090e4, "nec america, inc." },
	{ 0x0090e5, "teknema, inc." },
	{ 0x0090e6, "acer laboratories, inc." },
	{ 0x0090e7, "horsch elektronik ag" },
	{ 0x0090e8, "moxa technologies corp., ltd." },
	{ 0x0090e9, "janz computer ag" },
	{ 0x0090ea, "alpha technologies, inc." },
	{ 0x0090eb, "sentry telecom systems" },
	{ 0x0090ec, "pyrescom" },
	{ 0x0090ed, "central system research" },
	{ 0x0090ee, "personal communications" },
	{ 0x0090ef, "integrix, inc." },
	{ 0x0090f0, "harmonic lightwaves, ltd." },
	{ 0x0090f1, "dot hill systems corporation" },
	{ 0x0090f2, "cisco ethernet switches and light streams" },
	{ 0x0090f3, "aspect communications" },
	{ 0x0090f4, "lightning instrumentation" },
	{ 0x0090f5, "clevo co." },
	{ 0x0090f6, "escalate networks, inc." },
	{ 0x0090f7, "nbase communications ltd." },
	{ 0x0090f8, "mediatrix telecom" },
	{ 0x0090f9, "leitch" },
	{ 0x0090fa, "giganet, inc." },
	{ 0x0090fb, "portwell, inc." },
	{ 0x0090fc, "network computing devices" },
	{ 0x0090fd, "coppercom, inc." },
	{ 0x0090fe, "elecom co., ltd.  (laneed div." },
	{ 0x0090ff, "tellus technology inc." },
	{ 0x009d8e, "cardiac recorders, inc." },
	{ 0x00a000, "bay networks ethernet switch" },
	{ 0x00a001, "watkins-johnson company" },
	{ 0x00a002, "leeds & northrup australia" },
	{ 0x00a003, "staefa control system" },
	{ 0x00a004, "netpower, inc." },
	{ 0x00a005, "daniel instruments, ltd." },
	{ 0x00a006, "image data processing" },
	{ 0x00a007, "apexx technology, inc." },
	{ 0x00a008, "netcorp" },
	{ 0x00a009, "whitetree network" },
	{ 0x00a00a, "r.d.c. communication" },
	{ 0x00a00b, "computex co., ltd." },
	{ 0x00a00c, "kingmax technology inc. pcmcia card" },
	{ 0x00a00d, "the panda project" },
	{ 0x00a00e, "visual networks, inc." },
	{ 0x00a00f, "broadband technologies" },
	{ 0x00a010, "syslogic datentechnik ag" },
	{ 0x00a011, "mutoh industries ltd." },
	{ 0x00a012, "b.a.t.m. advanced technologies" },
	{ 0x00a013, "teltrend ltd." },
	{ 0x00a014, "csir" },
	{ 0x00a015, "wyle" },
	{ 0x00a016, "micropolis corp." },
	{ 0x00a017, "j b m corporation" },
	{ 0x00a018, "creative controllers, inc." },
	{ 0x00a019, "nebula consultants, inc." },
	{ 0x00a01a, "binar elektronik ab" },
	{ 0x00a01b, "premisys communications, inc." },
	{ 0x00a01c, "nascent networks corporation" },
	{ 0x00a01d, "sixnet" },
	{ 0x00a01e, "est corporation" },
	{ 0x00a01f, "tricord systems, inc." },
	{ 0x00a020, "citicorp/tti" },
	{ 0x00a021, "general dynamics-" },
	{ 0x00a022, "centre for development of" },
	{ 0x00a023, "applied creative technology," },
	{ 0x00a024, "3com" },
	{ 0x00a025, "redcom labs inc." },
	{ 0x00a026, "teldat, s.a." },
	{ 0x00a027, "firepower systems, inc." },
	{ 0x00a028, "conner peripherals" },
	{ 0x00a029, "coulter corporation" },
	{ 0x00a02a, "trancell systems" },
	{ 0x00a02b, "transitions research corp." },
	{ 0x00a02c, "interwave communications" },
	{ 0x00a02d, "1394 trade association" },
	{ 0x00a02e, "brand communications, ltd." },
	{ 0x00a02f, "pirelli cavi" },
	{ 0x00a030, "captor nv/sa" },
	{ 0x00a031, "hazeltine corporation, ms 1-17" },
	{ 0x00a032, "ges singapore pte. ltd." },
	{ 0x00a033, "imc mess-systeme gmbh" },
	{ 0x00a034, "axel" },
	{ 0x00a035, "cylink corporation" },
	{ 0x00a036, "applied network technology" },
	{ 0x00a037, "datascope corporation" },
	{ 0x00a038, "email electronics" },
	{ 0x00a039, "ross technology, inc." },
	{ 0x00a03a, "kubotek corporation" },
	{ 0x00a03b, "toshin electric co., ltd." },
	{ 0x00a03c, "eg&g nuclear instruments" },
	{ 0x00a03d, "opto - 22" },
	{ 0x00a03e, "atm forum" },
	{ 0x00a03f, "computer society microproces'r" },
	{ 0x00a040, "apple (pci mac)" },
	{ 0x00a041, "leybold-inficon" },
	{ 0x00a042, "spur products corp." },
	{ 0x00a043, "american technology labs, inc." },
	{ 0x00a044, "ntt intelligent technology" },
	{ 0x00a045, "phoenix contact gmbh & co." },
	{ 0x00a046, "scitex corp. ltd." },
	{ 0x00a047, "integrated fitness corp." },
	{ 0x00a048, "questech, ltd." },
	{ 0x00a049, "digitech industries, inc." },
	{ 0x00a04a, "nisshin electric co., ltd." },
	{ 0x00a04b, "sonic systems inc. etherfe 10/100 pci for mac or pc" },
	{ 0x00a04c, "innovative systems & tech. inc" },
	{ 0x00a04d, "eda instruments, inc." },
	{ 0x00a04e, "voelker technologies, inc." },
	{ 0x00a04f, "ameritec corp." },
	{ 0x00a050, "cypress semiconductor" },
	{ 0x00a051, "angia communications. inc." },
	{ 0x00a052, "stanilite electronics pty. ltd" },
	{ 0x00a053, "compact devices, inc." },
	{ 0x00a055, "linktech, inc." },
	{ 0x00a056, "micropross" },
	{ 0x00a057, "elsa" },
	{ 0x00a058, "glory, ltd." },
	{ 0x00a059, "hamilton hallmark" },
	{ 0x00a05a, "kofax image products" },
	{ 0x00a05b, "marquip, inc." },
	{ 0x00a05c, "inventory conversion, inc./" },
	{ 0x00a05d, "cs computer systeme gmbh" },
	{ 0x00a05e, "myriad logic inc." },
	{ 0x00a05f, "btg engineering bv" },
	{ 0x00a060, "acer peripherals, inc." },
	{ 0x00a061, "puritan bennett" },
	{ 0x00a062, "aes prodata" },
	{ 0x00a063, "jrl systems, inc." },
	{ 0x00a064, "kvb/analect" },
	{ 0x00a065, "nexland, inc." },
	{ 0x00a066, "isa co., ltd." },
	{ 0x00a067, "network services group" },
	{ 0x00a068, "bhp limited" },
	{ 0x00a069, "truetime" },
	{ 0x00a06a, "verilink corp." },
	{ 0x00a06b, "dms dorsch mikrosystem gmbh" },
	{ 0x00a06c, "shindengen electric mfg." },
	{ 0x00a06d, "mannesmann tally corporation" },
	{ 0x00a06e, "austron, inc." },
	{ 0x00a06f, "the appcon group, inc." },
	{ 0x00a070, "coastcom" },
	{ 0x00a071, "video lottery technologies,inc" },
	{ 0x00a072, "ovation systems ltd." },
	{ 0x00a073, "com21" },
	{ 0x00a074, "perception technology" },
	{ 0x00a075, "micron technology, inc." },
	{ 0x00a076, "cardware lab, inc." },
	{ 0x00a077, "fujitsu nexion, inc." },
	{ 0x00a078, "marconi communications" },
	{ 0x00a079, "alps electric (usa), inc." },
	{ 0x00a07a, "advanced peripherals" },
	{ 0x00a07b, "dawn computer incorporation" },
	{ 0x00a07c, "tonyang nylon co., ltd." },
	{ 0x00a07d, "seeq technology, inc." },
	{ 0x00a07e, "avid technology, inc." },
	{ 0x00a07f, "gsm-syntel, ltd." },
	{ 0x00a080, "antares microsystems" },
	{ 0x00a081, "alcatel data networks" },
	{ 0x00a082, "nkt elektronik a/s" },
	{ 0x00a083, "intel" },
	{ 0x00a084, "dataplex pty. ltd." },
	{ 0x00a086, "amber wave systems, inc." },
	{ 0x00a087, "mitel semiconductor, ltd." },
	{ 0x00a088, "essential communications" },
	{ 0x00a089, "xpoint technologies, inc." },
	{ 0x00a08a, "brooktrout technology, inc." },
	{ 0x00a08b, "aston electronic designs ltd." },
	{ 0x00a08c, "multimedia lans, inc." },
	{ 0x00a08d, "jacomo corporation" },
	{ 0x00a08e, "nokia internet communications" },
	{ 0x00a08f, "desknet systems, inc." },
	{ 0x00a090, "timestep corporation" },
	{ 0x00a091, "applicom international" },
	{ 0x00a092, "intermate international [lan printer interfaces]" },
	{ 0x00a093, "b/e aerospace" },
	{ 0x00a094, "comsat corporation" },
	{ 0x00a095, "acacia networks, inc." },
	{ 0x00a096, "mitsumi electric co., ltd." },
	{ 0x00a097, "jc information systems" },
	{ 0x00a098, "network appliance corp." },
	{ 0x00a099, "k-net ltd." },
	{ 0x00a09a, "nihon kohden america" },
	{ 0x00a09b, "qpsx communications, ltd." },
	{ 0x00a09c, "xyplex, inc." },
	{ 0x00a09d, "johnathon freeman technologies" },
	{ 0x00a09e, "ictv" },
	{ 0x00a09f, "commvision corp." },
	{ 0x00a0a0, "compact data, ltd." },
	{ 0x00a0a1, "epic data inc." },
	{ 0x00a0a2, "digicom s.p.a." },
	{ 0x00a0a3, "reliable power meters" },
	{ 0x00a0a4, "micros systems, inc." },
	{ 0x00a0a5, "teknor microsysteme, inc." },
	{ 0x00a0a6, "m.i. systems, k.k." },
	{ 0x00a0a7, "vorax corporation" },
	{ 0x00a0a8, "renex corporation" },
	{ 0x00a0a9, "gn nettest (canada) inc." },
	{ 0x00a0aa, "spacelabs medical" },
	{ 0x00a0ab, "netcs informationstechnik gmbh" },
	{ 0x00a0ac, "gilat satellite networks, ltd." },
	{ 0x00a0ad, "marconi spa" },
	{ 0x00a0ae, "network peripherals, inc." },
	{ 0x00a0af, "wms industries" },
	{ 0x00a0b0, "i-o data device, inc." },
	{ 0x00a0b1, "first virtual corporation" },
	{ 0x00a0b2, "shima seiki" },
	{ 0x00a0b3, "zykronix" },
	{ 0x00a0b4, "texas microsystems, inc." },
	{ 0x00a0b5, "3h technology" },
	{ 0x00a0b6, "sanritz automation co., ltd." },
	{ 0x00a0b7, "cordant, inc." },
	{ 0x00a0b8, "symbios logic inc." },
	{ 0x00a0b9, "eagle technology, inc." },
	{ 0x00a0ba, "patton electronics co." },
	{ 0x00a0bb, "hilan gmbh" },
	{ 0x00a0bc, "viasat, incorporated" },
	{ 0x00a0bd, "i-tech corp." },
	{ 0x00a0be, "integrated circuit systems,inc" },
	{ 0x00a0bf, "wireless data group motorola" },
	{ 0x00a0c0, "digital link corp." },
	{ 0x00a0c1, "ortivus medical ab" },
	{ 0x00a0c2, "r.a. systems co., ltd." },
	{ 0x00a0c3, "unicomputer gmbh" },
	{ 0x00a0c4, "cristie electronics ltd." },
	{ 0x00a0c5, "zyxel communication" },
	{ 0x00a0c6, "qualcomm incorporated" },
	{ 0x00a0c7, "tadiran telecommunications" },
	{ 0x00a0c8, "adtran, inc." },
	{ 0x00a0c9, "intel (pro100b and pro100+) [used on cisco pix firewall among others]" },
	{ 0x00a0ca, "fujitsu denso ltd." },
	{ 0x00a0cb, "ark telecommunications, inc." },
	{ 0x00a0cc, "lite-on (used by macsense in adapter for mac, also seen in pcs)" },
	{ 0x00a0cd, "dr. johannes heidenhain gmbh" },
	{ 0x00a0ce, "astrocom corporation" },
	{ 0x00a0cf, "sotas, inc." },
	{ 0x00a0d0, "ten x technology, inc." },
	{ 0x00a0d1, "national semiconductor [compaq docking station]" },
	{ 0x00a0d2, "allied telesyn" },
	{ 0x00a0d3, "instem computer systems, ltd." },
	{ 0x00a0d4, "radiolan,  inc." },
	{ 0x00a0d5, "sierra wireless inc." },
	{ 0x00a0d6, "sbe, inc." },
	{ 0x00a0d7, "kasten chase applied research" },
	{ 0x00a0d8, "spectra - tek" },
	{ 0x00a0d9, "convex computer corporation" },
	{ 0x00a0da, "integrated systems" },
	{ 0x00a0db, "fisher & paykel production" },
	{ 0x00a0dc, "o.n. electronic co., ltd." },
	{ 0x00a0dd, "azonix corporation" },
	{ 0x00a0de, "yamaha corporation" },
	{ 0x00a0df, "sts technologies, inc." },
	{ 0x00a0e0, "tennyson technologies pty ltd" },
	{ 0x00a0e1, "westport research" },
	{ 0x00a0e2, "keisoku giken corp." },
	{ 0x00a0e3, "xkl systems corp." },
	{ 0x00a0e4, "optiquest" },
	{ 0x00a0e5, "nhc communications" },
	{ 0x00a0e6, "dialogic corporation" },
	{ 0x00a0e7, "central data corporation" },
	{ 0x00a0e8, "reuters holdings plc" },
	{ 0x00a0e9, "electronic retailing systems" },
	{ 0x00a0ea, "ethercom corp." },
	{ 0x00a0eb, "fastcomm communications corp." },
	{ 0x00a0ec, "transmitton ltd." },
	{ 0x00a0ed, "pri automation" },
	{ 0x00a0ee, "nashoba networks" },
	{ 0x00a0ef, "lucidata ltd." },
	{ 0x00a0f0, "toronto microelectronics inc." },
	{ 0x00a0f1, "mti" },
	{ 0x00a0f2, "infotek communications, inc." },
	{ 0x00a0f3, "staubli" },
	{ 0x00a0f4, "ge" },
	{ 0x00a0f5, "radguard ltd." },
	{ 0x00a0f6, "autogas systems, inc." },
	{ 0x00a0f7, "v.i computer corp." },
	{ 0x00a0f8, "symbol technologies, inc." },
	{ 0x00a0f9, "bintec communications ag" },
	{ 0x00a0fa, "marconi communication gmbh" },
	{ 0x00a0fb, "toray engineering co., ltd." },
	{ 0x00a0fc, "image sciences, inc." },
	{ 0x00a0fd, "scitex digital printing, inc." },
	{ 0x00a0fe, "boston technology, inc." },
	{ 0x00a0ff, "tellabs operations, inc." },
	{ 0x00aa00, "intel" },
	{ 0x00aa01, "intel corporation" },
	{ 0x00aa02, "intel corporation" },
	{ 0x00aa3c, "olivetti telecom spa (olteco)" },
	{ 0x00b009, "grass valley group" },
	{ 0x00b017, "infogear technology corp." },
	{ 0x00b019, "casi-rusco" },
	{ 0x00b01c, "westport technologies" },
	{ 0x00b01e, "rantic labs, inc." },
	{ 0x00b02a, "orsys gmbh" },
	{ 0x00b02d, "viagate technologies, inc." },
	{ 0x00b03b, "hiq networks" },
	{ 0x00b048, "marconi communications inc." },
	{ 0x00b04a, "cisco systems, inc." },
	{ 0x00b052, "intellon corporation" },
	{ 0x00b064, "cisco systems, inc." },
	{ 0x00b069, "honewell oy" },
	{ 0x00b06d, "jones futurex inc." },
	{ 0x00b080, "mannesmann ipulsys b.v." },
	{ 0x00b086, "locsoft limited" },
	{ 0x00b08e, "cisco systems, inc." },
	{ 0x00b091, "transmeta corp." },
	{ 0x00b094, "alaris, inc." },
	{ 0x00b09a, "morrow technologies corp." },
	{ 0x00b09d, "point grey research inc." },
	{ 0x00b0ac, "siae-microelettronica s.p.a." },
	{ 0x00b0ae, "symmetricom" },
	{ 0x00b0b3, "xstreamis plc" },
	{ 0x00b0c2, "cisco systems, inc." },
	{ 0x00b0c7, "tellabs operations, inc." },
	{ 0x00b0d0, "computer products international" },
	{ 0x00b0db, "nextcell, inc." },
	{ 0x00b0df, "reliable data technology, inc." },
	{ 0x00b0e7, "british federal ltd." },
	{ 0x00b0ec, "eacem" },
	{ 0x00b0ee, "ajile systems, inc." },
	{ 0x00b0f0, "caly networks" },
	{ 0x00b0f5, "networth technologies, inc." },
	{ 0x00bb01, "octothorpe corp." },
	{ 0x00bbf0, "ungermann-bass inc." },
	{ 0x00c000, "lanoptics ltd" },
	{ 0x00c001, "diatek patient managment" },
	{ 0x00c002, "sercomm corporation" },
	{ 0x00c003, "globalnet communications" },
	{ 0x00c004, "japan business computer co.ltd" },
	{ 0x00c005, "livingston enterprises inc portmaster (oemed by cayman)" },
	{ 0x00c006, "nippon avionics co ltd" },
	{ 0x00c007, "pinnacle data systems inc" },
	{ 0x00c008, "seco srl" },
	{ 0x00c009, "kt technology (s) pte inc" },
	{ 0x00c00a, "micro craft" },
	{ 0x00c00b, "norcontrol a.s." },
	{ 0x00c00c, "ark pc technology, inc." },
	{ 0x00c00d, "advanced logic research inc" },
	{ 0x00c00e, "psitech inc" },
	{ 0x00c00f, "qnx software systems ltd. [also quantum software systems ltd]" },
	{ 0x00c010, "hirakawa hewtech corp." },
	{ 0x00c011, "interactive computing devices" },
	{ 0x00c012, "netspan corp" },
	{ 0x00c013, "netrix" },
	{ 0x00c014, "telematics calabasas" },
	{ 0x00c015, "new media corp" },
	{ 0x00c016, "electronic theatre controls" },
	{ 0x00c017, "fluke" },
	{ 0x00c018, "lanart corp" },
	{ 0x00c019, "leap technology, inc." },
	{ 0x00c01a, "corometrics medical systems" },
	{ 0x00c01b, "socket communications" },
	{ 0x00c01c, "interlink communications ltd." },
	{ 0x00c01d, "grand junction networks, inc. (cisco catalyst also reported)" },
	{ 0x00c01e, "la francaise des jeux" },
	{ 0x00c01f, "s.e.r.c.e.l." },
	{ 0x00c020, "arco electronic, control ltd." },
	{ 0x00c021, "netexpress" },
	{ 0x00c022, "lasermaster technologies, inc." },
	{ 0x00c023, "tutankhamon electronics" },
	{ 0x00c024, "eden sistemas de computacao sa" },
	{ 0x00c025, "dataproducts corporation" },
	{ 0x00c026, "lans technology co., ltd." },
	{ 0x00c027, "cipher systems, inc." },
	{ 0x00c028, "jasco corporation" },
	{ 0x00c029, "kabel rheydt ag" },
	{ 0x00c02a, "ohkura electric co" },
	{ 0x00c02b, "gerloff gesellschaft fur" },
	{ 0x00c02c, "centrum communications, inc." },
	{ 0x00c02d, "fuji photo film co., ltd." },
	{ 0x00c02e, "netwiz" },
	{ 0x00c02f, "okuma corp" },
	{ 0x00c030, "integrated engineering b. v." },
	{ 0x00c031, "design research systems, inc." },
	{ 0x00c032, "i-cubed limited" },
	{ 0x00c033, "telebit corporation" },
	{ 0x00c034, "dale computer corporation" },
	{ 0x00c035, "quintar company" },
	{ 0x00c036, "raytech electronic corp" },
	{ 0x00c037, "dynatem" },
	{ 0x00c038, "raster image processing system" },
	{ 0x00c039, "silicon systems" },
	{ 0x00c03a, "men-mikro elektronik gmbh" },
	{ 0x00c03b, "multiaccess computing corp" },
	{ 0x00c03c, "tower tech s.r.l." },
	{ 0x00c03d, "wiesemann & theis gmbh" },
	{ 0x00c03e, "fa. gebr. heller gmbh" },
	{ 0x00c03f, "stores automated systems inc" },
	{ 0x00c040, "ecci" },
	{ 0x00c041, "digital transmission systems" },
	{ 0x00c042, "datalux corp." },
	{ 0x00c043, "stratacom" },
	{ 0x00c044, "emcom corporation" },
	{ 0x00c045, "isolation systems inc" },
	{ 0x00c046, "kemitron ltd" },
	{ 0x00c047, "unimicro systems inc" },
	{ 0x00c048, "bay technical associates" },
	{ 0x00c049, "us robotics total control (tm) netserver card" },
	{ 0x00c04a, "group 2000 ag" },
	{ 0x00c04b, "creative microsystems" },
	{ 0x00c04c, "department of foreign affairs" },
	{ 0x00c04d, "mitec ltd" },
	{ 0x00c04e, "comtrol corporation" },
	{ 0x00c04f, "dell" },
	{ 0x00c050, "toyo denki seizo k.k." },
	{ 0x00c051, "advanced integration research" },
	{ 0x00c052, "burr-brown" },
	{ 0x00c053, "davox corporation" },
	{ 0x00c054, "network peripherals, ltd." },
	{ 0x00c055, "modular computing technologies" },
	{ 0x00c056, "somelec" },
	{ 0x00c057, "myco electronics" },
	{ 0x00c058, "dataexpert corp" },
	{ 0x00c059, "nippondenso corp" },
	{ 0x00c05a, "semaphore communications corp." },
	{ 0x00c05b, "networks northwest inc" },
	{ 0x00c05c, "elonex plc" },
	{ 0x00c05d, "l&n technologies" },
	{ 0x00c05e, "vari-lite inc" },
	{ 0x00c05f, "fine-pal company limited" },
	{ 0x00c060, "id scandinavia a/s" },
	{ 0x00c061, "solectek corporation" },
	{ 0x00c062, "impulse technology" },
	{ 0x00c063, "morning star technologies inc may be miswrite of 0003c6" },
	{ 0x00c064, "general datacomm ind inc" },
	{ 0x00c065, "scope communications inc" },
	{ 0x00c066, "docupoint, inc." },
	{ 0x00c067, "united barcode industries" },
	{ 0x00c068, "philp drake electronics ltd" },
	{ 0x00c069, "california microwave inc" },
	{ 0x00c06a, "zahner-elektrik gmbh & co kg" },
	{ 0x00c06b, "osi plus corporation" },
	{ 0x00c06c, "svec computer corp" },
	{ 0x00c06d, "boca research, inc." },
	{ 0x00c06e, "haft technology, inc." },
	{ 0x00c06f, "komatsu ltd" },
	{ 0x00c070, "sectra secure-transmission ab" },
	{ 0x00c071, "areanex communications, inc." },
	{ 0x00c072, "knx ltd" },
	{ 0x00c073, "xedia corporation" },
	{ 0x00c074, "toyoda automatic loom works ltd" },
	{ 0x00c075, "xante corporation " },
	{ 0x00c076, "i-data international a-s" },
	{ 0x00c077, "daewoo telecom ltd" },
	{ 0x00c078, "computer systems engineering" },
	{ 0x00c079, "fonsys co ltd" },
	{ 0x00c07a, "priva bv" },
	{ 0x00c07b, "ascend" },
	{ 0x00c07c, "hightech information" },
	{ 0x00c07d, "risc developments ltd" },
	{ 0x00c07e, "kubota corporation electronic" },
	{ 0x00c07f, "nupon computing corp" },
	{ 0x00c080, "netstar inc" },
	{ 0x00c081, "metrodata ltd" },
	{ 0x00c082, "moore products co" },
	{ 0x00c083, "trace mountain products, inc." },
	{ 0x00c084, "data link corp ltd" },
	{ 0x00c085, "canon" },
	{ 0x00c086, "the lynk corporation" },
	{ 0x00c087, "uunet technologies inc" },
	{ 0x00c088, "ekf elektronik gmbh" },
	{ 0x00c089, "telindus distribution" },
	{ 0x00c08a, "lauterbach datentechnik gmbh" },
	{ 0x00c08b, "risq modular systems inc" },
	{ 0x00c08c, "performance technologies inc" },
	{ 0x00c08d, "tronix product development" },
	{ 0x00c08e, "network information technology" },
	{ 0x00c08f, "matsushita electric works, ltd." },
	{ 0x00c090, "praim s.r.l." },
	{ 0x00c091, "jabil circuit, inc." },
	{ 0x00c092, "mennen medical inc" },
	{ 0x00c093, "alta research corp." },
	{ 0x00c094, "vmx inc." },
	{ 0x00c095, "znyx # network appliance" },
	{ 0x00c096, "tamura corporation" },
	{ 0x00c097, "archipel sa" },
	{ 0x00c098, "chuntex electronic co., ltd." },
	{ 0x00c099, "yoshiki industrial co.,ltd." },
	{ 0x00c09a, "photonics corporation" },
	{ 0x00c09b, "reliance comm/tec, r-tec systems inc" },
	{ 0x00c09c, "toa electronic ltd" },
	{ 0x00c09d, "distributed systems int'l, inc." },
	{ 0x00c09e, "cache computers, inc." },
	{ 0x00c09f, "quanta computer inc" },
	{ 0x00c0a0, "advance micro research, inc." },
	{ 0x00c0a1, "tokyo denshi sekei co" },
	{ 0x00c0a2, "intermedium a/s" },
	{ 0x00c0a3, "dual enterprises corporation" },
	{ 0x00c0a4, "unigraf oy" },
	{ 0x00c0a5, "dickens data systems" },
	{ 0x00c0a6, "exicom australia pty. ltd" },
	{ 0x00c0a7, "seel ltd" },
	{ 0x00c0a8, "gvc corporation" },
	{ 0x00c0a9, "barron mccann ltd" },
	{ 0x00c0aa, "silicon valley computer" },
	{ 0x00c0ab, "jupiter technology inc" },
	{ 0x00c0ac, "gambit computer communications" },
	{ 0x00c0ad, "computer communication systems" },
	{ 0x00c0ae, "towercom co inc dba pc house" },
	{ 0x00c0af, "teklogix inc." },
	{ 0x00c0b0, "gcc technologies,inc." },
	{ 0x00c0b1, "genius net co." },
	{ 0x00c0b2, "norand corporation" },
	{ 0x00c0b3, "comstat datacomm corporation" },
	{ 0x00c0b4, "myson technology inc" },
	{ 0x00c0b5, "corporate network systems inc" },
	{ 0x00c0b6, "meridian data inc" },
	{ 0x00c0b7, "american power conversion corp" },
	{ 0x00c0b8, "fraser's hill ltd." },
	{ 0x00c0b9, "funk software inc" },
	{ 0x00c0ba, "netvantage" },
	{ 0x00c0bb, "forval creative inc" },
	{ 0x00c0bc, "telecom australia/cssc" },
	{ 0x00c0bd, "inex technologies, inc." },
	{ 0x00c0be, "alcatel - sel" },
	{ 0x00c0bf, "technology concepts ltd" },
	{ 0x00c0c0, "shore microsystems inc" },
	{ 0x00c0c1, "quad/graphics inc" },
	{ 0x00c0c2, "infinite networks ltd." },
	{ 0x00c0c3, "acuson computed sonography" },
	{ 0x00c0c4, "computer operational" },
	{ 0x00c0c5, "sid informatica" },
	{ 0x00c0c6, "personal media corp" },
	{ 0x00c0c7, "sparktrum microsystems, inc." },
	{ 0x00c0c8, "micro byte pty ltd" },
	{ 0x00c0c9, "bailey controls co" },
	{ 0x00c0ca, "alfa, inc." },
	{ 0x00c0cb, "control technology corporation" },
	{ 0x00c0cc, "telesciences co systems, inc." },
	{ 0x00c0cd, "comelta s.a." },
	{ 0x00c0ce, "cei systems & engineering pte" },
	{ 0x00c0cf, "imatran voima oy" },
	{ 0x00c0d0, "ratoc system inc" },
	{ 0x00c0d1, "comtree technology corporation (efa also reported)" },
	{ 0x00c0d2, "syntellect inc" },
	{ 0x00c0d3, "olympus image systems, inc." },
	{ 0x00c0d4, "axon networks inc" },
	{ 0x00c0d5, "quancom electronic gmbh" },
	{ 0x00c0d6, "j1 systems, inc." },
	{ 0x00c0d7, "taiwan trading center dba" },
	{ 0x00c0d8, "universal data systems" },
	{ 0x00c0d9, "quinte network confidentiality equipment inc" },
	{ 0x00c0da, "nice systems ltd." },
	{ 0x00c0db, "ipc corporation (pte) ltd" },
	{ 0x00c0dc, "eos technologies, inc." },
	{ 0x00c0dd, "qlogic corporation" },
	{ 0x00c0de, "zcomm inc" },
	{ 0x00c0df, "kye systems corp" },
	{ 0x00c0e0, "dsc communication corp." },
	{ 0x00c0e1, "sonic solutions" },
	{ 0x00c0e2, "calcomp, inc." },
	{ 0x00c0e3, "ositech communications inc" },
	{ 0x00c0e4, "landis & gyr powers inc" },
	{ 0x00c0e5, "gespac s.a." },
	{ 0x00c0e6, "txport" },
	{ 0x00c0e7, "fiberdata ab" },
	{ 0x00c0e8, "plexcom inc" },
	{ 0x00c0e9, "oak solutions ltd" },
	{ 0x00c0ea, "array technology ltd." },
	{ 0x00c0eb, "seh computertechnik gmbh" },
	{ 0x00c0ec, "dauphin technology" },
	{ 0x00c0ed, "us army electronic proving ground" },
	{ 0x00c0ee, "kyocera corporation" },
	{ 0x00c0ef, "abit corporation" },
	{ 0x00c0f0, "kingston technology corporation" },
	{ 0x00c0f1, "shinko electric co ltd" },
	{ 0x00c0f2, "transition engineering inc" },
	{ 0x00c0f3, "network communications corp" },
	{ 0x00c0f4, "interlink system co., ltd." },
	{ 0x00c0f5, "metacomp inc" },
	{ 0x00c0f6, "celan technology inc." },
	{ 0x00c0f7, "engage communication, inc." },
	{ 0x00c0f8, "about computing inc." },
	{ 0x00c0f9, "harris and jeffries, inc." },
	{ 0x00c0fa, "canary communications inc" },
	{ 0x00c0fb, "advanced technology labs" },
	{ 0x00c0fc, "asdg incorporated" },
	{ 0x00c0fd, "prosum" },
	{ 0x00c0fe, "aptec computer systems, inc." },
	{ 0x00c0ff, "box hill systems corporation" },
	{ 0x00cf1c, "communication machinery corp." },
	{ 0x00d000, "ferran scientific, inc." },
	{ 0x00d001, "vst technologies, inc." },
	{ 0x00d002, "ditech corporation" },
	{ 0x00d003, "comda enterprises corp." },
	{ 0x00d004, "pentacom ltd." },
	{ 0x00d005, "zhs zeitmanagementsysteme" },
	{ 0x00d006, "cisco systems, inc." },
	{ 0x00d007, "mic associates, inc." },
	{ 0x00d008, "mactell corporation" },
	{ 0x00d009, "hsing tech. enterprise co. ltd" },
	{ 0x00d00a, "lanaccess telecom s.a." },
	{ 0x00d00b, "rhk technology, inc." },
	{ 0x00d00c, "snijder micro systems" },
	{ 0x00d00d, "micromeritics instrument" },
	{ 0x00d00e, "pluris, inc." },
	{ 0x00d00f, "speech design gmbh" },
	{ 0x00d010, "convergent networks, inc." },
	{ 0x00d011, "prism video, inc." },
	{ 0x00d012, "gateworks corp." },
	{ 0x00d013, "primex aerospace company" },
	{ 0x00d014, "root, inc." },
	{ 0x00d015, "univex microtechnology corp." },
	{ 0x00d016, "scm microsystems, inc." },
	{ 0x00d017, "syntech information co., ltd." },
	{ 0x00d018, "qwes. com, inc." },
	{ 0x00d019, "dainippon screen corporate" },
	{ 0x00d01a, "urmet sud s.p.a." },
	{ 0x00d01b, "mimaki engineering co., ltd." },
	{ 0x00d01c, "sbs technologies," },
	{ 0x00d01d, "furuno electric co., ltd." },
	{ 0x00d01e, "pingtel corp." },
	{ 0x00d01f, "ctam pty. ltd." },
	{ 0x00d020, "aim system, inc." },
	{ 0x00d021, "regent electronics corp." },
	{ 0x00d022, "incredible technologies, inc." },
	{ 0x00d023, "infortrend technology, inc." },
	{ 0x00d024, "cognex corporation" },
	{ 0x00d025, "xrosstech, inc." },
	{ 0x00d026, "hirschmann austria gmbh" },
	{ 0x00d027, "applied automation, inc." },
	{ 0x00d028, "omneon video networks" },
	{ 0x00d029, "wakefern food corporation" },
	{ 0x00d02a, "flexion systems" },
	{ 0x00d02b, "jetcell, inc." },
	{ 0x00d02c, "campbell scientific, inc." },
	{ 0x00d02d, "ademco" },
	{ 0x00d02e, "communication automation corp." },
	{ 0x00d02f, "vlsi technology inc." },
	{ 0x00d030, "safetran systems corp." },
	{ 0x00d031, "industrial logic corporation" },
	{ 0x00d032, "yano electric co., ltd." },
	{ 0x00d033, "dalian daxian network" },
	{ 0x00d034, "ormec systems corp." },
	{ 0x00d035, "behavior tech. computer corp." },
	{ 0x00d036, "technology atlanta corp." },
	{ 0x00d037, "philips-dvs-lo bdr" },
	{ 0x00d038, "fivemere, ltd." },
	{ 0x00d039, "utilicom, inc." },
	{ 0x00d03a, "zoneworx, inc." },
	{ 0x00d03b, "vision products pty. ltd." },
	{ 0x00d03c, "vieo, inc." },
	{ 0x00d03e, "rocketchips, inc." },
	{ 0x00d03f, "american communication" },
	{ 0x00d040, "sysmate co., ltd." },
	{ 0x00d041, "amigo technology co., ltd." },
	{ 0x00d042, "mahlo gmbh & co. ug" },
	{ 0x00d043, "zonal retail data systems" },
	{ 0x00d044, "alidian networks, inc." },
	{ 0x00d045, "kvaser ab" },
	{ 0x00d046, "dolby laboratories, inc." },
	{ 0x00d047, "xn technologies" },
	{ 0x00d048, "ecton, inc." },
	{ 0x00d049, "impresstek co., ltd." },
	{ 0x00d04a, "presence technology gmbh" },
	{ 0x00d04b, "la cie group s.a." },
	{ 0x00d04c, "eurotel telecom ltd." },
	{ 0x00d04d, "div of research & statistics" },
	{ 0x00d04e, "logibag" },
	{ 0x00d04f, "bitronics, inc." },
	{ 0x00d050, "iskratel" },
	{ 0x00d051, "o2 micro, inc." },
	{ 0x00d052, "ascend communications, inc." },
	{ 0x00d053, "connected systems" },
	{ 0x00d054, "sas institute inc." },
	{ 0x00d055, "kathrein-werke kg" },
	{ 0x00d056, "somat corporation" },
	{ 0x00d057, "ultrak, inc." },
	{ 0x00d058, "cisco systems, inc." },
	{ 0x00d059, "ambit microsystems corp." },
	{ 0x00d05a, "symbionics, ltd." },
	{ 0x00d05b, "acroloop motion control" },
	{ 0x00d05c, "technotrend systemtechnik gmbh" },
	{ 0x00d05d, "intelliworxx, inc." },
	{ 0x00d05e, "stratabeam technology, inc." },
	{ 0x00d05f, "valcom, inc." },
	{ 0x00d060, "panasonic european" },
	{ 0x00d061, "tremon enterprises co., ltd." },
	{ 0x00d062, "digigram" },
	{ 0x00d063, "cisco systems, inc." },
	{ 0x00d064, "multitel" },
	{ 0x00d065, "toko electric" },
	{ 0x00d066, "wintriss engineering corp." },
	{ 0x00d067, "campio communications" },
	{ 0x00d068, "iwill corporation" },
	{ 0x00d069, "technologic systems" },
	{ 0x00d06a, "linkup systems corporation" },
	{ 0x00d06b, "sr telecom inc." },
	{ 0x00d06c, "sharewave, inc." },
	{ 0x00d06d, "acrison, inc." },
	{ 0x00d06e, "trendview recorders ltd." },
	{ 0x00d06f, "kmc controls" },
	{ 0x00d070, "long well electronics corp." },
	{ 0x00d071, "echelon corp." },
	{ 0x00d072, "broadlogic" },
	{ 0x00d073, "acn advanced communications" },
	{ 0x00d074, "taqua systems, inc." },
	{ 0x00d075, "alaris medical systems, inc." },
	{ 0x00d076, "merrill lynch & co., inc." },
	{ 0x00d077, "lucent technologies" },
	{ 0x00d078, "eltex of sweden ab" },
	{ 0x00d079, "cisco systems, inc." },
	{ 0x00d07a, "amaquest computer corp." },
	{ 0x00d07b, "comcam international ltd." },
	{ 0x00d07c, "koyo electronics inc. co.,ltd." },
	{ 0x00d07d, "cosine communications" },
	{ 0x00d07e, "keycorp ltd." },
	{ 0x00d07f, "strategy & technology, limited" },
	{ 0x00d080, "exabyte corporation" },
	{ 0x00d081, "real time devices usa, inc." },
	{ 0x00d082, "iowave inc." },
	{ 0x00d083, "invertex, inc." },
	{ 0x00d084, "nexcomm systems, inc." },
	{ 0x00d085, "otis elevator company" },
	{ 0x00d086, "foveon, inc." },
	{ 0x00d087, "microfirst inc." },
	{ 0x00d088, "mainsail networks, inc." },
	{ 0x00d089, "dynacolor, inc." },
	{ 0x00d08a, "photron usa" },
	{ 0x00d08b, "adva limited" },
	{ 0x00d08c, "genoa technology, inc." },
	{ 0x00d08d, "phoenix group, inc." },
	{ 0x00d08e, "nvision inc." },
	{ 0x00d08f, "ardent technologies, inc." },
	{ 0x00d090, "cisco systems, inc." },
	{ 0x00d091, "smartsan systems, inc." },
	{ 0x00d092, "glenayre western multiplex" },
	{ 0x00d093, "tq - components gmbh" },
	{ 0x00d094, "timeline vista, inc." },
	{ 0x00d095, "xylan corporation" },
	{ 0x00d096, "3com europe ltd." },
	{ 0x00d097, "cisco systems, inc." },
	{ 0x00d098, "ips automation" },
	{ 0x00d099, "elcard oy" },
	{ 0x00d09a, "filanet corporation" },
	{ 0x00d09b, "spectel ltd." },
	{ 0x00d09c, "kapadia communications" },
	{ 0x00d09d, "veris industries" },
	{ 0x00d09e, "2wire, inc." },
	{ 0x00d09f, "novtek test systems" },
	{ 0x00d0a0, "mips denmark" },
	{ 0x00d0a1, "oskar vierling gmbh + co. kg" },
	{ 0x00d0a2, "integrated device" },
	{ 0x00d0a3, "vocal data, inc." },
	{ 0x00d0a4, "alantro communications" },
	{ 0x00d0a5, "american arium" },
	{ 0x00d0a6, "lanbird technology co., ltd." },
	{ 0x00d0a7, "tokyo sokki kenkyujo co., ltd." },
	{ 0x00d0a8, "network engines, inc." },
	{ 0x00d0a9, "shinano kenshi co., ltd." },
	{ 0x00d0aa, "chase communications" },
	{ 0x00d0ab, "deltakabel telecom cv" },
	{ 0x00d0ac, "grayson wireless" },
	{ 0x00d0ad, "tl industries" },
	{ 0x00d0ae, "oresis communications, inc." },
	{ 0x00d0af, "cutler-hammer, inc." },
	{ 0x00d0b0, "bitswitch ltd." },
	{ 0x00d0b1, "omega electronics sa" },
	{ 0x00d0b2, "xiotech corporation" },
	{ 0x00d0b3, "drs flight safety and" },
	{ 0x00d0b4, "katsujima co., ltd." },
	{ 0x00d0b5, "dotcom" },
	{ 0x00d0b6, "crescent networks, inc." },
	{ 0x00d0b7, "intel corpotation" },
	{ 0x00d0b8, "iomega corp." },
	{ 0x00d0b9, "microtek international, inc." },
	{ 0x00d0ba, "cisco systems, inc." },
	{ 0x00d0bb, "cisco systems, inc." },
	{ 0x00d0bc, "cisco systems, inc." },
	{ 0x00d0bd, "sican gmbh" },
	{ 0x00d0be, "emutec inc." },
	{ 0x00d0bf, "pivotal technologies" },
	{ 0x00d0c0, "cisco systems, inc." },
	{ 0x00d0c1, "harmonic data systems, ltd." },
	{ 0x00d0c2, "balthazar technology ab" },
	{ 0x00d0c3, "vivid technology pte, ltd." },
	{ 0x00d0c4, "teratech corporation" },
	{ 0x00d0c5, "computational systems, inc." },
	{ 0x00d0c6, "thomas & betts corp." },
	{ 0x00d0c7, "pathway, inc." },
	{ 0x00d0c8, "i/o consulting a/s" },
	{ 0x00d0c9, "advantech co., ltd." },
	{ 0x00d0ca, "intrinsyc software inc." },
	{ 0x00d0cb, "dasan co., ltd." },
	{ 0x00d0cc, "technologies lyre inc." },
	{ 0x00d0cd, "atan technology inc." },
	{ 0x00d0ce, "asyst electronic" },
	{ 0x00d0cf, "moreton bay" },
	{ 0x00d0d0, "zhongxing telecom ltd." },
	{ 0x00d0d1, "sirocco systems, inc." },
	{ 0x00d0d2, "epilog corporation" },
	{ 0x00d0d3, "cisco systems, inc." },
	{ 0x00d0d4, "v-bits, inc." },
	{ 0x00d0d5, "grundig ag" },
	{ 0x00d0d6, "aethra telecomunicazioni" },
	{ 0x00d0d7, "b2c2, inc." },
	{ 0x00d0d8, "3com corporation" },
	{ 0x00d0d9, "dedicated microcomputers" },
	{ 0x00d0da, "taicom data systems co., ltd." },
	{ 0x00d0db, "mcquay international" },
	{ 0x00d0dc, "modular mining systems, inc." },
	{ 0x00d0dd, "sunrise telecom, inc." },
	{ 0x00d0de, "philips multimedia network" },
	{ 0x00d0df, "kuzumi electronics, inc." },
	{ 0x00d0e0, "dooin electronics co." },
	{ 0x00d0e1, "avionitek israel inc." },
	{ 0x00d0e2, "mrt micro, inc." },
	{ 0x00d0e3, "ele-chem engineering co., ltd." },
	{ 0x00d0e4, "cisco systems, inc." },
	{ 0x00d0e5, "solidum systems corp." },
	{ 0x00d0e6, "ibond inc." },
	{ 0x00d0e7, "vcon telecommunication ltd." },
	{ 0x00d0e8, "mac system co., ltd." },
	{ 0x00d0e9, "advantage century" },
	{ 0x00d0ea, "nextone communications, inc." },
	{ 0x00d0eb, "lightera networks, inc." },
	{ 0x00d0ec, "nakayo telecommunications, inc" },
	{ 0x00d0ed, "xiox" },
	{ 0x00d0ee, "dictaphone corporation" },
	{ 0x00d0ef, "igt" },
	{ 0x00d0f0, "convision technology gmbh" },
	{ 0x00d0f1, "sega enterprises, ltd." },
	{ 0x00d0f2, "monterey networks" },
	{ 0x00d0f3, "solari di udine spa" },
	{ 0x00d0f4, "carinthian tech institute" },
	{ 0x00d0f5, "orange micro, inc." },
	{ 0x00d0f6, "northchurch communications inc" },
	{ 0x00d0f7, "next nets corporation" },
	{ 0x00d0f8, "fujian star terminal" },
	{ 0x00d0f9, "acute communications corp." },
	{ 0x00d0fa, "racal guardata" },
	{ 0x00d0fb, "tek microsystems, incorporated" },
	{ 0x00d0fc, "granite microsystems" },
	{ 0x00d0fd, "optima tele.com, inc." },
	{ 0x00d0fe, "astral point" },
	{ 0x00d0ff, "cisco systems, inc." },
	{ 0x00dd00, "ungermann-bass ibm rt" },
	{ 0x00dd01, "ungermann-bass" },
	{ 0x00dd02, "ungermann-bass inc." },
	{ 0x00dd03, "ungermann-bass inc." },
	{ 0x00dd04, "ungermann-bass inc." },
	{ 0x00dd05, "ungermann-bass inc." },
	{ 0x00dd06, "ungermann-bass inc." },
	{ 0x00dd07, "ungermann-bass inc." },
	{ 0x00dd08, "ungermann-bass" },
	{ 0x00dd09, "ungermann-bass inc." },
	{ 0x00dd0a, "ungermann-bass inc." },
	{ 0x00dd0b, "ungermann-bass inc." },
	{ 0x00dd0c, "ungermann-bass inc." },
	{ 0x00dd0d, "ungermann-bass inc." },
	{ 0x00dd0e, "ungermann-bass inc." },
	{ 0x00dd0f, "ungermann-bass inc." },
	{ 0x00e000, "fujitsu, ltd" },
	{ 0x00e001, "strand lighting limited" },
	{ 0x00e002, "crossroads systems, inc." },
	{ 0x00e003, "nokia wireless business commun" },
	{ 0x00e004, "pmc-sierra, inc." },
	{ 0x00e005, "technical corp." },
	{ 0x00e006, "silicon integrated sys. corp." },
	{ 0x00e007, "network alchemy ltd." },
	{ 0x00e008, "amazing controls! inc." },
	{ 0x00e009, "marathon technologies corp." },
	{ 0x00e00a, "diba, inc." },
	{ 0x00e00b, "rooftop communications corp." },
	{ 0x00e00c, "motorola" },
	{ 0x00e00d, "radiant systems" },
	{ 0x00e00e, "avalon imaging systems, inc." },
	{ 0x00e00f, "shanghai baud data" },
	{ 0x00e010, "hess sb-automatenbau gmbh" },
	{ 0x00e011, "uniden corporation" },
	{ 0x00e012, "pluto technologies" },
	{ 0x00e013, "eastern electronic co., ltd." },
	{ 0x00e014, "cisco" },
	{ 0x00e015, "heiwa corporation" },
	{ 0x00e016, "rapid-city (now a part of bay networks)" },
	{ 0x00e017, "exxact gmbh" },
	{ 0x00e018, "asustek intel 82558-based integrated fast ethernet for wim" },
	{ 0x00e019, "ing. giordano elettronica" },
	{ 0x00e01a, "comtec systems. co., ltd." },
	{ 0x00e01b, "sphere communications, inc." },
	{ 0x00e01c, "mobility electronicsy" },
	{ 0x00e01d, "webtv networks, inc." },
	{ 0x00e01e, "cisco" },
	{ 0x00e01f, "avidia systems, inc." },
	{ 0x00e020, "tecnomen oy" },
	{ 0x00e021, "freegate corp." },
	{ 0x00e022, "medialight inc." },
	{ 0x00e023, "telrad" },
	{ 0x00e024, "gadzoox networks" },
	{ 0x00e025, "dit co., ltd." },
	{ 0x00e026, "eastman kodak co." },
	{ 0x00e027, "dux, inc." },
	{ 0x00e028, "aptix corporation" },
	{ 0x00e029, "smc etherpower ii 10/100" },
	{ 0x00e02a, "tandberg television as" },
	{ 0x00e02b, "extreme networks" },
	{ 0x00e02c, "ast - built into 5166m pc motherboard (win95 id's as intel)" },
	{ 0x00e02d, "innomedialogic, inc." },
	{ 0x00e02e, "spc electronics corporation" },
	{ 0x00e02f, "mcns holdings, l.p." },
	{ 0x00e030, "melita international corp." },
	{ 0x00e031, "hagiwara electric co., ltd." },
	{ 0x00e032, "misys financial systems, ltd." },
	{ 0x00e033, "e.e.p.d. gmbh" },
	{ 0x00e034, "cisco" },
	{ 0x00e035, "loughborough sound images, plc" },
	{ 0x00e036, "pioneer corporation" },
	{ 0x00e037, "century corporation" },
	{ 0x00e038, "proxima corporation" },
	{ 0x00e039, "paradyne 7112 t1 dsu/csu" },
	{ 0x00e03a, "cabletron systems, inc." },
	{ 0x00e03b, "prominet corporation" },
	{ 0x00e03c, "advansys" },
	{ 0x00e03d, "focon electronic systems a/s" },
	{ 0x00e03e, "alfatech, inc." },
	{ 0x00e03f, "jaton corporation" },
	{ 0x00e040, "deskstation technology, inc." },
	{ 0x00e041, "cspi" },
	{ 0x00e042, "pacom data ltd." },
	{ 0x00e043, "vitalcom" },
	{ 0x00e044, "lsics corporation" },
	{ 0x00e045, "touchwave, inc." },
	{ 0x00e046, "bently nevada corp." },
	{ 0x00e047, "infocus systems" },
	{ 0x00e048, "sdl communications, inc." },
	{ 0x00e049, "microwi electronic gmbh" },
	{ 0x00e04a, "enhanced messaging systems,inc" },
	{ 0x00e04b, "jump industrielle" },
	{ 0x00e04c, "realtek semiconductor corp." },
	{ 0x00e04d, "internet initiative japan, inc" },
	{ 0x00e04e, "sanyo denki co., ltd." },
	{ 0x00e04f, "cisco" },
	{ 0x00e050, "executone information" },
	{ 0x00e051, "talx corporation" },
	{ 0x00e052, "foundry networks, inc." },
	{ 0x00e053, "cellport labs, inc." },
	{ 0x00e054, "kodai hitec co., ltd." },
	{ 0x00e055, "ingenieria electronica" },
	{ 0x00e056, "holontech corporation" },
	{ 0x00e057, "han microtelecom. co., ltd." },
	{ 0x00e058, "phase one denmark a/s" },
	{ 0x00e059, "controlled environments, ltd." },
	{ 0x00e05a, "galea network security" },
	{ 0x00e05b, "west end systems corp." },
	{ 0x00e05c, "matsushita kotobuki" },
	{ 0x00e05d, "unitec co., ltd." },
	{ 0x00e05e, "japan aviation electronics" },
	{ 0x00e05f, "e-net, inc." },
	{ 0x00e060, "sherwood" },
	{ 0x00e061, "edgepoint networks, inc." },
	{ 0x00e062, "host engineering" },
	{ 0x00e063, "cabletron - yago systems, inc." },
	{ 0x00e064, "samsung electronics" },
	{ 0x00e065, "optical access international" },
	{ 0x00e066, "promax systems, inc." },
	{ 0x00e067, "eac automation-consulting gmbh" },
	{ 0x00e068, "merrimac systems inc." },
	{ 0x00e069, "jaycor networks, inc." },
	{ 0x00e06a, "kapsch ag" },
	{ 0x00e06b, "w&g special products" },
	{ 0x00e06c, "baltimore technologies, ltd." },
	{ 0x00e06d, "compuware corporation" },
	{ 0x00e06e, "far systems spa" },
	{ 0x00e06f, "terayon corp." },
	{ 0x00e070, "dh technology" },
	{ 0x00e071, "epis microcomputer" },
	{ 0x00e072, "lynk" },
	{ 0x00e073, "national amusement" },
	{ 0x00e074, "tiernan communications, inc." },
	{ 0x00e075, "atlas computer equipment, inc." },
	{ 0x00e076, "development concepts, inc." },
	{ 0x00e077, "webgear, inc." },
	{ 0x00e078, "berkeley networks" },
	{ 0x00e079, "a.t.n.r." },
	{ 0x00e07a, "mikrodidakt ab" },
	{ 0x00e07b, "bay networks" },
	{ 0x00e07c, "mettler-toledo, inc." },
	{ 0x00e07d, "encore (netronix?) 10/100 pci fast ethernet card" },
	{ 0x00e07e, "walt disney imagineering" },
	{ 0x00e07f, "logististem srl" },
	{ 0x00e080, "control resources corporation" },
	{ 0x00e081, "tyan computer corp. onboard intel 82558 10/100" },
	{ 0x00e082, "anerma" },
	{ 0x00e083, "jato technologies, inc." },
	{ 0x00e084, "compulite r&d" },
	{ 0x00e085, "global maintech, inc." },
	{ 0x00e086, "cybex computer products" },
	{ 0x00e087, "lecroy" },
	{ 0x00e088, "ltx corporation" },
	{ 0x00e089, "ion networks, inc." },
	{ 0x00e08a, "gec avery, ltd." },
	{ 0x00e08b, "qlogic corp." },
	{ 0x00e08c, "neoparadigm labs, inc." },
	{ 0x00e08d, "pressure systems, inc." },
	{ 0x00e08e, "utstarcom" },
	{ 0x00e08f, "cisco systems catalyst 2900 series" },
	{ 0x00e090, "beckman lab. automation div." },
	{ 0x00e091, "lg electronics, inc." },
	{ 0x00e092, "admtek incorporated" },
	{ 0x00e093, "ackfin networks" },
	{ 0x00e094, "osai srl" },
	{ 0x00e095, "advanced-vision technolgies" },
	{ 0x00e096, "shimadzu corporation" },
	{ 0x00e097, "carrier access corporation" },
	{ 0x00e098, "trend" },
	{ 0x00e099, "samson ag" },
	{ 0x00e09a, "positron industries, inc." },
	{ 0x00e09b, "engage networks, inc." },
	{ 0x00e09c, "mii" },
	{ 0x00e09d, "sarnoff corporation" },
	{ 0x00e09e, "quantum corporation" },
	{ 0x00e09f, "pixel vision" },
	{ 0x00e0a0, "wiltron co." },
	{ 0x00e0a1, "hima paul hildebrandt" },
	{ 0x00e0a2, "microslate inc." },
	{ 0x00e0a3, "cisco" },
	{ 0x00e0a4, "esaote s.p.a." },
	{ 0x00e0a5, "comcore semiconductor, inc." },
	{ 0x00e0a6, "telogy networks, inc." },
	{ 0x00e0a7, "ipc information systems, inc." },
	{ 0x00e0a8, "sat gmbh&co" },
	{ 0x00e0a9, "funai electric co., ltd." },
	{ 0x00e0aa, "electrosonic ltd." },
	{ 0x00e0ab, "dimat s.a." },
	{ 0x00e0ac, "midsco, inc." },
	{ 0x00e0ad, "ees technology, ltd." },
	{ 0x00e0ae, "xaqti corporation" },
	{ 0x00e0af, "general dynamics information" },
	{ 0x00e0b0, "cisco systems various systems reported" },
	{ 0x00e0b1, "packet engines, inc." },
	{ 0x00e0b2, "telmax communications corp." },
	{ 0x00e0b3, "etherwan systems, inc." },
	{ 0x00e0b4, "techno scope co., ltd." },
	{ 0x00e0b5, "ardent communications corp." },
	{ 0x00e0b6, "entrada networks" },
	{ 0x00e0b7, "pi group, ltd." },
	{ 0x00e0b8, "amd pcnet in a gateway 2000" },
	{ 0x00e0b9, "byas systems" },
	{ 0x00e0ba, "berghof automationstechnik" },
	{ 0x00e0bb, "nbx corporation" },
	{ 0x00e0bc, "symon communications, inc." },
	{ 0x00e0bd, "interface systems, inc." },
	{ 0x00e0be, "genroco international, inc." },
	{ 0x00e0bf, "torrent networking" },
	{ 0x00e0c0, "seiwa erectric mfg. co., ltd." },
	{ 0x00e0c1, "memorex telex japan, ltd." },
	{ 0x00e0c2, "necsy spa" },
	{ 0x00e0c3, "sakai system development corp." },
	{ 0x00e0c4, "horner electric, inc." },
	{ 0x00e0c5, "bcom electronics inc." },
	{ 0x00e0c6, "link2it, l.l.c." },
	{ 0x00e0c7, "eurotech srl" },
	{ 0x00e0c8, "virtual access, ltd." },
	{ 0x00e0c9, "automatedlogic corporation" },
	{ 0x00e0ca, "best data products" },
	{ 0x00e0cb, "reson, inc." },
	{ 0x00e0cc, "hero systems, ltd." },
	{ 0x00e0cd, "sensis corporation" },
	{ 0x00e0ce, "arn" },
	{ 0x00e0cf, "integrated device" },
	{ 0x00e0d0, "netspeed, inc." },
	{ 0x00e0d1, "telsis limited" },
	{ 0x00e0d2, "versanet communications, inc." },
	{ 0x00e0d3, "datentechnik gmbh" },
	{ 0x00e0d4, "excellent computer" },
	{ 0x00e0d5, "arcxel technologies, inc." },
	{ 0x00e0d6, "computer & communication" },
	{ 0x00e0d7, "sunshine electronics, inc." },
	{ 0x00e0d8, "lanbit computer, inc." },
	{ 0x00e0d9, "tazmo co., ltd." },
	{ 0x00e0da, "assured access" },
	{ 0x00e0db, "viavideo communications" },
	{ 0x00e0dc, "nexware corp." },
	{ 0x00e0dd, "zenith electronics corporation" },
	{ 0x00e0de, "datax nv" },
	{ 0x00e0df, "ke kommunikations-electronik" },
	{ 0x00e0e0, "si electronics, ltd." },
	{ 0x00e0e1, "g2 networks, ilnc." },
	{ 0x00e0e2, "innova corp." },
	{ 0x00e0e3, "sk-elektronik gmbh" },
	{ 0x00e0e4, "fanuc robotics north america," },
	{ 0x00e0e5, "cinco networks, inc." },
	{ 0x00e0e6, "incaa datacom b.v." },
	{ 0x00e0e7, "raytheon e-systems, inc." },
	{ 0x00e0e8, "gretacoder data systems ag" },
	{ 0x00e0e9, "data labs, inc." },
	{ 0x00e0ea, "innovat communications, inc." },
	{ 0x00e0eb, "digicom systems, incorporated" },
	{ 0x00e0ec, "celestica inc." },
	{ 0x00e0ed, "new link" },
	{ 0x00e0ee, "marel hf" },
	{ 0x00e0ef, "dionex" },
	{ 0x00e0f0, "abler technology, inc." },
	{ 0x00e0f1, "that corporation" },
	{ 0x00e0f2, "arlotto comnet, inc." },
	{ 0x00e0f3, "websprint communications, inc." },
	{ 0x00e0f4, "inside technology a/s" },
	{ 0x00e0f5, "teles ag" },
	{ 0x00e0f6, "decision europe" },
	{ 0x00e0f7, "cisco" },
	{ 0x00e0f8, "diana control ab" },
	{ 0x00e0f9, "cisco" },
	{ 0x00e0fa, "trl technology, ltd." },
	{ 0x00e0fb, "leightronix, inc." },
	{ 0x00e0fc, "huawei technologies co., ltd." },
	{ 0x00e0fd, "a-trend technology co., ltd." },
	{ 0x00e0fe, "cisco" },
	{ 0x00e0ff, "security dynamics technologies" },
	{ 0x00e6d3, "nixdorf computer corp." },
	{ 0x020406, "bbn internal usage (not registered)" },
	{ 0x020701, "interlan [now racal-interlan] dec (unibus or qbus), apollo, cisco" },
	{ 0x021c7c, "perq systems corporation" },
	{ 0x026060, "3com" },
	{ 0x026086, "satelcom megapac (uk)" },
	{ 0x02608c, "3com" },
	{ 0x027001, "racal-datacom" },
	{ 0x0270b0, "m/a-com inc. companies" },
	{ 0x0270b3, "data recall ltd" },
	{ 0x029d8e, "cardiac recorders inc." },
	{ 0x02a0c9, "intel" },
	{ 0x02aa3c, "olivetti" },
	{ 0x02bb01, "octothorpe corp." },
	{ 0x02c08c, "3com corporation" },
	{ 0x02cf1c, "communication machinery corp." },
	{ 0x02cf1f, "cmc" },
	{ 0x02e03b, "prominet corporation gigabit ethernet switch" },
	{ 0x02e6d3, "bti (bus-tech, inc.) ibm mainframes" },
	{ 0x040ae0, "xmit ag computer networks" },
	{ 0x048845, "bay networks token ring line card" },
	{ 0x04e0c4, "triumph-adler ag" },
	{ 0x080001, "computer vision" },
	{ 0x080002, "3com" },
	{ 0x080003, "acc" },
	{ 0x080004, "cromemco incorporated" },
	{ 0x080005, "symbolics symbolics lisp machines" },
	{ 0x080006, "siemens nixdorf pc clone" },
	{ 0x080007, "apple" },
	{ 0x080008, "bbn" },
	{ 0x080009, "hp" },
	{ 0x08000a, "nestar" },
	{ 0x08000b, "unisys" },
	{ 0x08000c, "miklyn development co." },
	{ 0x08000d, "icl (international computers, ltd.)" },
	{ 0x08000e, "ncr" },
	{ 0x08000f, "smc" },
	{ 0x080010, "at&t [misrepresentation of 800010?]" },
	{ 0x080011, "tektronix" },
	{ 0x080012, "bell atlantic integrated syst." },
	{ 0x080013, "exxon" },
	{ 0x080014, "excelan bbn butterfly, masscomp, silicon graphics" },
	{ 0x080015, "stc business systems" },
	{ 0x080016, "barrister info sys corp" },
	{ 0x080017, "national semiconductor corp. (used to have network system corp., wrong nsc)" },
	{ 0x080018, "pirelli focom networks" },
	{ 0x080019, "general electric corporation" },
	{ 0x08001a, "data general" },
	{ 0x08001b, "data general" },
	{ 0x08001c, "kdd-kokusai debnsin denwa co." },
	{ 0x08001d, "able communications inc." },
	{ 0x08001e, "apollo" },
	{ 0x08001f, "sharp" },
	{ 0x080020, "sun" },
	{ 0x080021, "3m company" },
	{ 0x080022, "nbi (nothing but initials)" },
	{ 0x080023, "matsushita denso" },
	{ 0x080024, "10net communications/dca" },
	{ 0x080025, "cdc" },
	{ 0x080026, "norsk data (nord)" },
	{ 0x080027, "pcs computer systems gmbh" },
	{ 0x080028, "ti explorer" },
	{ 0x080029, "megatek corporation" },
	{ 0x08002a, "mosaic technologies inc." },
	{ 0x08002b, "dec" },
	{ 0x08002c, "britton lee inc." },
	{ 0x08002d, "lan-tec inc." },
	{ 0x08002e, "metaphor" },
	{ 0x08002f, "prime computer prime 50-series lhc300" },
	{ 0x080030, "cern" },
	{ 0x080031, "little machines inc." },
	{ 0x080032, "tigan" },
	{ 0x080033, "bausch & lomb" },
	{ 0x080034, "filenet corporation" },
	{ 0x080035, "microfive corporation" },
	{ 0x080036, "intergraph cae stations" },
	{ 0x080037, "fuji xerox" },
	{ 0x080038, "bull" },
	{ 0x080039, "spider systems" },
	{ 0x08003a, "orcatech inc." },
	{ 0x08003b, "torus systems" },
	{ 0x08003c, "schlumberger well services" },
	{ 0x08003d, "cadnetix" },
	{ 0x08003e, "motorola" },
	{ 0x08003f, "fred koschara enterprises" },
	{ 0x080040, "ferranti computer sys. limited" },
	{ 0x080041, "dca (digital comm. assoc.)" },
	{ 0x080042, "japan macnics corp." },
	{ 0x080043, "pixel computer inc." },
	{ 0x080044, "dsi (david systems, inc.)" },
	{ 0x080045, "???? (maybe xylogics, but they claim not to know this number)" },
	{ 0x080046, "sony" },
	{ 0x080047, "sequent" },
	{ 0x080048, "eurotherm gauging systems" },
	{ 0x080049, "univation" },
	{ 0x08004a, "banyan systems inc." },
	{ 0x08004b, "planning research corp." },
	{ 0x08004c, "encore" },
	{ 0x08004d, "corvus systems inc." },
	{ 0x08004e, "bicc [3com bought bicc, so may appear on 3com equipment as well]" },
	{ 0x08004f, "cygnet systems" },
	{ 0x080050, "daisy systems corp." },
	{ 0x080051, "experdata" },
	{ 0x080052, "insystec" },
	{ 0x080053, "middle east tech. university" },
	{ 0x080055, "stanford telecomm. inc." },
	{ 0x080056, "stanford university" },
	{ 0x080057, "evans & sutherland (?)" },
	{ 0x080058, "??? decsystem-20" },
	{ 0x080059, "a/s mycron" },
	{ 0x08005a, "ibm" },
	{ 0x08005b, "vta technologies inc." },
	{ 0x08005c, "four phase systems" },
	{ 0x08005d, "gould inc." },
	{ 0x08005e, "counterpoint computer inc." },
	{ 0x08005f, "saber technology corp." },
	{ 0x080060, "industrial networking inc." },
	{ 0x080061, "jarogate ltd." },
	{ 0x080062, "general dynamics" },
	{ 0x080063, "plessey" },
	{ 0x080064, "autophon ag" },
	{ 0x080065, "genrad inc." },
	{ 0x080066, "agfa printers, phototypesetters etc." },
	{ 0x080067, "comdesign" },
	{ 0x080068, "ridge" },
	{ 0x080069, "sgi" },
	{ 0x08006a, "attst (?)" },
	{ 0x08006b, "accel technologies inc." },
	{ 0x08006c, "suntek technology int'l" },
	{ 0x08006d, "whitechapel computer works" },
	{ 0x08006e, "excelan" },
	{ 0x08006f, "philips apeldoorn b.v." },
	{ 0x080070, "mitsubishi" },
	{ 0x080071, "matra (dsie)" },
	{ 0x080072, "xerox corp univ grant program" },
	{ 0x080073, "tecmar inc." },
	{ 0x080074, "casio" },
	{ 0x080075, "dde (danish data elektronik a/s)" },
	{ 0x080076, "pc lan technologies" },
	{ 0x080077, "tsl (now retix)" },
	{ 0x080078, "accell corporation" },
	{ 0x080079, "sgi" },
	{ 0x08007a, "indata" },
	{ 0x08007b, "sanyo electric co. ltd." },
	{ 0x08007c, "vitalink translan iii" },
	{ 0x08007e, "amalgamated wireless(aus) ltd" },
	{ 0x08007f, "carnegie-mellon university" },
	{ 0x080080, "xios" },
	{ 0x080081, "crosfield electronics" },
	{ 0x080082, "veritas software" },
	{ 0x080083, "seiko denshi" },
	{ 0x080084, "tomen electronics corp." },
	{ 0x080085, "elxsi" },
	{ 0x080086, "imagen/qms" },
	{ 0x080087, "xyplex" },
	{ 0x080088, "mcdata corporation" },
	{ 0x080089, "kinetix" },
	{ 0x08008a, "performance technology" },
	{ 0x08008b, "pyramid" },
	{ 0x08008c, "network research corporation" },
	{ 0x08008d, "xyvision xyvision machines" },
	{ 0x08008e, "tandem" },
	{ 0x08008f, "chipcom corp." },
	{ 0x080090, "retix" },
	{ 0x08bbcc, "ak-nord edv vertriebsges. mbh" },
	{ 0x09006a, "at&t" },
	{ 0x10005a, "ibm" },
	{ 0x100090, "hp" },
	{ 0x1000d4, "dec" },
	{ 0x1000e0, "apple a/ux (modified addresses for licensing)" },
	{ 0x1000e8, "national semiconductor" },
	{ 0x2e2e2e, "laa (locally administered address) for meditech systems" },
	{ 0x3c0000, "3com" },
	{ 0x400003, "net ware (?)" },
	{ 0x444553, "micrsoft" },
	{ 0x444649, "dfi (diamond flower industries)" },
	{ 0x475443, "gtc (not registered!) (this number is a multicast!)" },
	{ 0x484453, "hds ???" },
	{ 0x484c00, "network solutions" },
	{ 0x4854e8, "winbond?" },
	{ 0x4c424c, "information modes software modified addresses (not registered?)" },
	{ 0x525400, "realtek (uptech? also reported)" },
	{ 0x52544c, "novell 2000" },
	{ 0x5254ab, "realtek (a realtek 8029 based pci card)" },
	{ 0x565857, "aculab plc audio bridges" },
	{ 0x800010, "at&t [misrepresented as 080010? one source claims this is correct]" },
	{ 0x80ad00, "cnet technology inc. (probably an error, see instead 0080ad)" },
	{ 0xaa0000, "dec obsolete" },
	{ 0xaa0001, "dec obsolete" },
	{ 0xaa0002, "dec obsolete" },
	{ 0xaa0003, "dec" },
	{ 0xaa0004, "dec" },
	{ 0xc00000, "western digital (may be reversed 00 00 c0?)" },
	{ 0xe20c0f, "kingston technologies" },
	{ 0xec1000, "enance source co., ltd. pc clones(?)" },
	{ 0, NULL }
};

static int
ethernetcode_index(struct ethertree *etherroot, struct ethernetcode *code)
{
	struct etherindex tmp, *entry;
	char line[1024], *p, *e;

	strlcpy(line, code->vendor, sizeof(line));
	e = line;

	/* Walk through every single word and index it */
	while ((p = strsep(&e, " ")) != NULL) {
		tmp.index_word = p;
		if ((entry = SPLAY_FIND(ethertree, etherroot, &tmp)) == NULL) {
			/* Generate a new entry for this word */
			entry = calloc(1, sizeof(struct etherindex));
			if (entry == NULL)
				err(1, "%s: calloc", __func__);

			if ((entry->index_word = strdup(p)) == NULL)
				err(1, "%s: strdup", __func__);

			entry->list_mem = 32;
			if ((entry->list = calloc(entry->list_mem,
				 sizeof(struct ethernetcode *))) == NULL)
				err(1, "%s: calloc");

			SPLAY_INSERT(ethertree, etherroot, entry);
		}

		if (entry->list_size >= entry->list_mem) {
			struct ethernetcode **tmp;

			/* We require more memory for this key word */
			entry->list_mem <<= 1;
			tmp = realloc(entry->list,
			    entry->list_mem * sizeof(struct ethernetcode *));
			if (tmp == NULL)
				err(1, "%s: realloc", __func__);
			entry->list = tmp;
		}

		entry->list[entry->list_size++] = code;
	}

	return (0);
}

#define TEST(x, y) do { \
	if (ethernetcode_find_prefix(x, 0) != (y)) \
		errx(1, "%s: %s does not match %.6x", __func__, x, y); \
} while (0)

void
ethernetcode_test(void)
{
	TEST("cisco", 0x00000c);
	TEST("cisco catalyst", 0x001007);
	TEST("juniper networks", 0x000585);
	TEST("3com", 0x00103);
	TEST("zzzzzzzz xxxxxxxx", 0x000000);
}

void
ethernetcode_init(void)
{
	struct ethernetcode *code = &codes[0];

	SPLAY_INIT(&etherroot);

	while (code->vendor != NULL) {
		ethernetcode_index(&etherroot, code);

		++code;
	}

	ethernetcode_test();
}

/*
 * Returns the code that matches the best, 0 on error.
 */

static uint32_t
ethernetcode_find_best(struct etherindex **results, int size, int random)
{
	extern rand_t *honeyd_rand;
	int i, j, max = 0, count = 0;
	struct ethernetcode *code = NULL;

	if (!size)
		return (0);

	/* Reset the counters */
	for (i = 0; i < size; i++) {
		struct etherindex *ei = results[i];
		for (j = 0; j < ei->list_size; j++)
			ei->list[j]->count = 0;
	}

	for (i = 0; i < size; i++) {
		struct etherindex *ei = results[i];
		for (j = 0; j < ei->list_size; j++) {
			ei->list[j]->count++;
			if (ei->list[j]->count > max) {
				max = ei->list[j]->count;
				code = ei->list[j];
				count = 1;
			} else if (ei->list[j]->count == max && random) {
				/* Randomly select one of the best matches */
				count++;
				if (rand_uint8(honeyd_rand) % count == 0)
					code = ei->list[j];
			}
		}
	}

	return (code->prefix);
}

uint32_t
ethernetcode_find_prefix(char *vendor, int random) {
	struct etherindex *results[20];
	struct etherindex tmp, *entry;
	char line[1024], *p, *e;
	int pos = 0;

	strlcpy(line, vendor, sizeof(line));
	e = line;

	/* Walk through every single word and find the codes for it */
	while ((p = strsep(&e, " ")) != NULL && pos < 20) {
		int i;
		
		/* Change the string to lower case for the match */
		for (i = 0; i < strlen(p); i++)
			p[i] = tolower(p[i]);

		tmp.index_word = p;
		if ((entry = SPLAY_FIND(ethertree, &etherroot, &tmp)) == NULL)
			continue;

		results[pos++] = entry;
	}

	return (ethernetcode_find_best(results, pos, random));
}

struct addr *
ethernetcode_make_address(char *vendor)
{
	extern rand_t *honeyd_rand;
	uint32_t prefix = 0;
	u_char address[ETH_ADDR_LEN], *p;
	struct addr *ea;
	int i;

	/* Check if it is a regular mac address: xx:xx:xx:xx:xx:xx */
	p = address;
	for (i = 0; i < strlen(vendor) && p < address + ETH_ADDR_LEN; i += 3) {
		char hex[3];

		if (!isxdigit(vendor[i]) || !isxdigit(vendor[i+1]))
			break;

		hex[0] = vendor[i];
		hex[1] = vendor[i+1];
		hex[2] = '\0';

		*p++ = strtoul(hex, NULL, 16);

		if (i + 2 < strlen(vendor) && vendor[i + 2] != ':')
			break;
	}

	/* We could not parse the hex digits, so search for a vendor instead */
	if (p < address + ETH_ADDR_LEN) {
		if ((prefix = ethernetcode_find_prefix(vendor, 1)) == 0)
			return (NULL);

		/* We have a 24-bit prefix that is vendor dependant */
		address[2] = prefix & 0xff; prefix >>= 8;
		address[1] = prefix & 0xff; prefix >>= 8;
		address[0] = prefix & 0xff; prefix >>= 8;

		if (prefix != 0)
			return (NULL);

		for (i = 3; i < ETH_ADDR_LEN; i++)
			address[i] = rand_uint8(honeyd_rand);
	}

	if ((ea = calloc(1, sizeof(struct addr))) == NULL)
		return (NULL);

	addr_pack(ea, ADDR_TYPE_ETH, ETH_ADDR_BITS, address, ETH_ADDR_LEN);

	return (ea);
}

struct addr *
ethernetcode_clone(struct addr *src)
{
	extern rand_t *honeyd_rand;
	struct addr *ea;
	int i;

	if ((ea = calloc(1, sizeof(struct addr))) == NULL)
		return (NULL);

	memcpy(ea, src, sizeof(struct addr));

	/* Very low-level hack, might break when dnet changes */
	for (i = 3; i < ETH_ADDR_LEN; i++)
		ea->addr_data8[i] = rand_uint8(honeyd_rand);

	return (ea);
}
