#include "config.h"
#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <signal.h>
#include <dirent.h>
#include <errno.h>
#include <pthread.h>
#include <syslog.h>
#include <grp.h>
#include <netdb.h>
#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif
#ifdef HAVE_NETINET_TCP_H
#include <netinet/tcp.h>
#endif
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/socket.h>
#include <sys/time.h>
#include "alternative.h"
#include "mimetype.h"
#include "serverconfig.h"
#include "libfs.h"
#include "liblist.h"
#include "libstr.h"
#include "cgi.h"
#include "userconfig.h"
#include "session.h"
#include "httpauth.h"
#include "send.h"
#include "client.h"
#include "target.h"
#include "log.h"
#ifndef HAVE_SETENV
#include "envir.h"
#endif
#ifdef HAVE_COMMAND
#include "command.h"
#endif
#ifdef HAVE_SSL
#include "libssl.h"
#endif
#ifdef HAVE_CACHE
#include "cache.h"
#endif

#define REQUEST_BUFFER_CHUNK  4096
#define MAIN_LOOP_TIMER          1

#define fe_MAX_REQUESTSIZE      -2
#define fe_TIMEOUT              -3
#define fe_CLIENT_DISCONNECTED  -4
#define fe_READ_ERROR           -5
#define fe_FORCE_QUIT           -6

#define rs_NONE                  0
#define rs_QUIT_SERVER           1
#define rs_UNBAN_CLIENTS         2
#define rs_TOUCH_LOGFILES        3

#ifdef HAVE_SSL
#define RANDOM_BUFFER_SIZE     512
#endif

typedef struct {
	char *config_dir;
	bool daemon;
	bool config_check;
} t_settings;

volatile int received_signal = rs_NONE;
char *hs_conlen      = "Content-Length: ";
char *fb_filesystem  = "access denied via filesystem";
char *fb_symlink     = "symlink not allowed";
char *fb_accesslist  = "access denied via accesslist";
char *fb_alterlist   = "access denied via alterlist";
char *version_string = "Hiawatha v"VERSION
#ifdef HAVE_SSL
			", SSL"
#endif
#ifdef HAVE_CACHE
			", cache"
#endif
#ifdef HAVE_COMMAND
			", CommandChannel"
#endif
#ifdef DEBUG
			", debug"
#endif
;

#ifdef DEBUG

/* timer functions for debugging
 */

double t1, t2;
struct timeval tp;

void start_clock(char *mesg) {
	fprintf(stderr, "(%5d) START: %s\n", (int)getpid(), mesg);
	gettimeofday(&tp, NULL);
	t1 = (double)tp.tv_sec + (1.e-6)*tp.tv_usec;
}

void stop_clock(void) {
	gettimeofday(&tp, NULL);
	t2 = (double)tp.tv_sec + (1.e-6)*tp.tv_usec;
	fprintf(stderr, "(%5d)  time: %f\n", (int)getpid(), t2 - t1);
}

#endif

/* Create all logfiles with the right ownership and accessrights
 */
void touch_logfiles(t_config *config, char *dir) {
	t_host *host;

	touch_logfile(dir, config->system_logfile, 0640, config->server_uid, config->server_gid);
	if (config->garbage_logfile != NULL) {
		touch_logfile(dir, config->garbage_logfile, 0640, config->server_uid, config->server_gid);
	}

	host = config->first_host;
	while (host != NULL) {
		if (host->access_fileptr != NULL) {
			fflush(host->access_fileptr);
		}
		touch_logfile(dir, host->access_logfile, 0640, config->server_uid, config->server_gid);
		touch_logfile(dir, host->error_logfile, 0640, config->server_uid, config->server_gid);
		host = host->next;
	}
}

/* Check if the requested file is a CGI program.
 */
void check_target_is_cgi(t_session *session) {
	t_cgi_handler *cgi;

	if ((session->fcgi_server = fcgi_server_match(session->config->fcgi_server, &(session->host->fast_cgi), session->file_on_disk)) != NULL) {
		session->cgi_type = fastcgi;
	} else if (in_charlist(file_extension(session->file_on_disk), &(session->config->cgi_extension))) {
		session->cgi_type = program;
	} else {
		cgi = session->config->cgi_handler;
		while (cgi != NULL) {
			if (in_charlist(file_extension(session->file_on_disk), &(cgi->extension))) {
				session->cgi_handler = cgi->handler;
				session->cgi_type = script;
				break;
			}
			cgi = cgi->next;
		}
	}
}

/* Handle an HTTP error.
 */
int handle_error(t_session *session) {
	int retval = 200, length;
	
	session->handling_error = true;
	session->mimetype = NULL;
	if (session->file_on_disk != NULL) {
		free(session->file_on_disk);
	}
	length = strlen(session->host->website_root);
	if ((session->file_on_disk = (char*)malloc(length + strlen(session->host->error_handler) + 4)) == NULL) { // + 3 for .gz (gzip encoding)
		return 500;
	}
	strcpy(session->file_on_disk, session->host->website_root);
	strcpy(session->file_on_disk + length, session->host->error_handler);

	check_target_is_cgi(session);

	if (session->cgi_type != none) {
		if (session->host->return_code != -1) {
			session->return_code = session->host->return_code;
		}
		retval = execute_cgi(session);
	} else {
		retval = send_file(session);
	}

	return retval;
}

/* Handle the request from a client.
 */
int handle_request(t_session *session) {
	int retval = 200;

	check_target_is_cgi(session);

	if ((strcmp(session->method, "GET") == 0) || session->head_request) {
		if (session->cgi_type != none) {
			session->body = NULL;
			retval = execute_cgi(session);
		} else {
			retval = send_file(session);
		}
		if (retval == 404) {
			if (session->host->show_index && (session->uri[strlen(session->uri) - 1] == '/')) {
				retval = show_index(session);
			}
		}
	} else if (strcmp(session->method, "POST") == 0) {
		if (session->cgi_type != none) {
			retval = execute_cgi(session);
		} else {
			retval = 405;
		}
	} else if (strcmp(session->method, "OPTIONS") == 0) {
		retval = handle_options_request(session);
	} else if (strcmp(session->method, "TRACE") == 0) {
		if (session->binding->enable_trace) {
			retval = handle_trace_request(session);
		} else {
			retval = 501;
		}
	} else if (strcmp(session->method, "PUT") == 0) {
		if (session->binding->enable_alter) {
			retval = handle_put_request(session);
		} else {
			retval = 501;
		}
	} else if (strcmp(session->method, "DELETE") == 0) {
		if (session->binding->enable_alter) {
			retval = handle_delete_request(session);
		} else {
			retval = 501;
		}
	} else if (strcmp(session->method, "CONNECT") == 0) {
		retval = 501;
	} else if (strcmp(session->method, "PROPFIND") == 0) {
		retval = 501;
	} else {
		retval = 400;
	}

	return retval;
}

/* Read the request from a client socket.
 */
int fetch_request(t_session *session) {
	char *new_reqbuf, *strstart, *strend;
	long bytes_read, header_length = -1, content_length = -1;
	int result = 200, timer;
	fd_set read_fds;
	struct timeval select_timeout;
	bool keep_reading = true;

	select_timeout.tv_sec = 1;
	select_timeout.tv_usec = 0;
	if (session->kept_alive == 0) {
		timer = session->binding->time_for_1st_request;
	} else {
		timer = session->binding->time_for_request;
	}

	do {
		/* Check if requestbuffer contains a complete request.
		 */
		if (session->request != NULL) {
			if (header_length == -1) {
				if ((strstart = strstr(session->request, "\r\n\r\n")) != NULL) {
					header_length = strstart + 4 - session->request;
					session->header_length = header_length;
				}
			}
			if (header_length != -1) {
				if (content_length == -1) {
					if ((strstart = strcasestr(session->request, hs_conlen)) != NULL) {
						strstart += 16;
						if ((strend = strstr(strstart, "\r\n")) != NULL) {
							*strend = '\0';
							if ((content_length = str2int(strstart)) == -1) {
								result = 400;
								break;
							}
							session->content_length = content_length;
							*strend = '\r';

							if (header_length + content_length > session->binding->max_request_size) {
								keep_reading = false;
								result = fe_MAX_REQUESTSIZE;
								break;
							}

							if (header_length + content_length + 1 > session->buffer_size) {
								session->buffer_size = header_length + content_length + 1;
								if ((new_reqbuf = (char*)realloc(session->request, session->buffer_size)) != NULL) {
									session->request = new_reqbuf;
								} else {
									result = fe_READ_ERROR;
									keep_reading = false;
									break;
								}
							}
						}
					} else {
						break;
					}
				}
				if (content_length > -1) {
					if (session->bytes_in_buffer >= header_length + content_length) {
						break;
					}
				}
			}
		}

		FD_ZERO(&read_fds);
		FD_SET(session->client_socket, &read_fds);
		switch (select(session->client_socket + 1, &read_fds, NULL, NULL, &select_timeout)) {
			case -1:
				if (errno != EINTR) {
					result = fe_READ_ERROR;
					keep_reading = false;
				}
				break;
			case 0:
				if (session->force_quit) {
					result = fe_FORCE_QUIT;
					keep_reading = false;
				} else if (--timer <= 0) {
					result = fe_TIMEOUT;
					keep_reading = false;
				} else {
					select_timeout.tv_sec = 1;
					select_timeout.tv_usec = 0;
				}
				break;
			default:
				if ((content_length == -1) && ((session->buffer_size - session->bytes_in_buffer) < 256)) {
					session->buffer_size += REQUEST_BUFFER_CHUNK;
					if ((new_reqbuf = (char*)realloc(session->request, session->buffer_size)) != NULL) {
						session->request = new_reqbuf;
					} else {
						result = fe_READ_ERROR;
						keep_reading = false;
						break;
					}
				}

				/* Read from socket.
				 */
#ifdef HAVE_SSL
				if (session->binding->use_ssl) {
					bytes_read = ssl_receive(session->ssl_data, session->request + session->bytes_in_buffer, 
									session->buffer_size - session->bytes_in_buffer - 1);
				} else
#endif
					bytes_read = recv(session->client_socket, session->request + session->bytes_in_buffer,
									session->buffer_size - session->bytes_in_buffer - 1, 0);

				switch (bytes_read) {
					case -1:
						if (errno != EINTR) {
							result = fe_READ_ERROR;
							keep_reading = false;
						}
						break;
					case 0:
						result = fe_CLIENT_DISCONNECTED;
						keep_reading = false;
						break;
					default:
						session->bytes_in_buffer += bytes_read;
						*(session->request + session->bytes_in_buffer) = '\0';

						if (session->bytes_in_buffer > session->binding->max_request_size) {
							keep_reading = false;
							result = fe_MAX_REQUESTSIZE;
							break;
						}
				}
		}
	} while (keep_reading);

#ifdef HAVE_COMMAND
	increment_transfer(TRANSFER_RECEIVED, session->header_length + session->content_length);
#endif

	return result;
}

/* Convert the requestbuffer to a session record.
 */
int parse_request(t_session *session, int total_bytes) {
	int retval = 200;
	char *request_end, *str_end, *conn;

	request_end = session->request + total_bytes;

	/* Request method
	 */
	session->method = str_end = session->request;
	while ((*str_end != ' ') && (str_end != request_end)) {
		str_end++;
	}
	if (str_end == request_end) {
		return 400;
	}
	*str_end = '\0';
	session->uri = ++str_end;

	/* URI
	 */
	while ((*str_end != ' ') && (str_end != request_end)) {
		str_end++;
		if (*str_end == '?') {
			if (session->vars == NULL) {
				session->vars = str_end + 1;
				*str_end = '\0';
			}
		}
	}
	if (str_end == request_end) {
		return 400;
	}

	*str_end = '\0';
	str_end++;

	/* Protocol version
	 */
	if (strlen(str_end) < 8) {
		return 400;
	} else if (memcmp(str_end, "HTTP/", 5) != 0) {
		return 400;
	}

	session->http_version = str_end;
	str_end += 7;

	/* Body and other request headerlines
	 */
	if (session->content_length > 0) {
		session->body = session->request + session->header_length;
		*(session->body - 2) = '\0';
	}
	if (str_end + 5 != session->body) {
		session->headerfields = parse_headerfields(str_end + 3);
	}

	if ((*(str_end - 1) != '.') || (*(str_end + 1) != '\r') || (*(str_end + 2) != '\n')) {
		return 400;
	} else if (*(str_end - 2) != '1') {
		return 505;
	}
	*(str_end + 1) = '\0';

	if ((conn = get_headerfield("Connection:", session->headerfields)) != NULL) {
		conn = strlower(remove_spaces(conn));
	}
	session->keep_alive = false;
	switch (*str_end) {
		case '0':
			if ((conn != NULL) && (session->kept_alive < session->binding->max_keepalive)) {
				if (strcasecmp(conn, "keep-alive") == 0) {
					session->keep_alive = true;
				}
			}
			break;
		case '1':
			if (get_headerfield("Host:", session->headerfields) == NULL) {
				retval = 400;
			} else if (session->kept_alive < session->binding->max_keepalive) {
				session->keep_alive = true;
				if (conn != NULL) {
					if (strcmp(conn, "close") == 0) {
						session->keep_alive = false;
					}
				}
			}
			break;
		default:
			retval = 505;
			break;
	}
	if (session->keep_alive) {
		session->kept_alive++;
	}

	if (strcmp(session->method, "HEAD") == 0) {
		session->head_request = true;
	}

	return retval;
}

/* Convert the request uri to a filename.
 */
int uri_to_path(t_session *session) {
	size_t length, alias_length = 0;
	char *strstart, *strend;
	t_keyvalue *alias;
	int retval;
	
	/* Requested file in userdirectory?
	 */
	if ((session->host->user_websites) && (strlen(session->uri) > 3)) {
		if (*(session->uri + 1) == '~') {
			strstart = session->uri + 1;
			if ((strend = strchr(strstart, '/')) == NULL) {
				return 301;
			} else if ((length = strend - strstart) > 1) {
				if ((session->local_user = (char*)malloc(length + 1)) == NULL) {
					return 500;
				}

				memcpy(session->local_user, strstart, length);
				*(session->local_user + length) = '\0';
				
				if (duplicate_host(session)) {
					if ((retval = get_homedir(session, session->local_user + 1)) != 200) {
						return retval;
					}
					session->host->error_handler = NULL;
				} else {
					return 500;
				}
			} else {
				// uri is '/~/'
				return 404;
			}
		}
	}

	/* Search for an alias.
	 */
	alias = session->host->alias;
	while (alias != NULL) {
		alias_length = strlen(alias->key);
		if (strncmp(session->uri, alias->key, alias_length) == 0) {
			if ((*(session->uri + alias_length) == '/') || (*(session->uri + alias_length) == '\0')) {
				break;
			}
		}
		alias = alias->next;
	}

	/* Allocate memory
	 */
	if (alias == NULL) {
		length = strlen(session->host->website_root);
	} else {
		length = strlen(alias->value);
	}
	length += strlen(session->uri) + strlen(session->host->start_file);
	if ((session->file_on_disk = (char*)malloc(length + 4)) == NULL) { // + 3 for '.gz' (gzip encoding)
		return 500;
	}

	/* Copy stuff
	 */
	if (alias == NULL) {
		length = strlen(session->host->website_root);
		memcpy(session->file_on_disk, session->host->website_root, length);
		strstart = session->uri;
		if (session->local_user != NULL) {
			strstart += strlen(session->local_user) + 1;
		}
	} else {
		length = strlen(alias->value);
		memcpy(session->file_on_disk, alias->value, length);
		strstart = session->uri + alias_length;
	}
	strcpy(session->file_on_disk + length, strstart);

	return 200;
}

t_access allow_client(t_session *session) {
	char *x_forwarded_for;
	t_ipv4 forwarded_ip;
	t_access access;

	if ((access = ip_allowed(session->ip_address, session->host->access_list)) != allow) {
		return access;
	} else if ((x_forwarded_for = get_headerfield("X-Forwarded-For:", session->headerfields)) == NULL) {
		return allow;
	} else if (parse_ip(x_forwarded_for, &forwarded_ip) == -1) {
		return allow;
	} else if (ip_allowed(forwarded_ip, session->host->access_list) == deny) {
		return deny;
	}

	return unknown;
}

/* Serve the client that connected to the webserver
 */
int serve_client(t_session *session) {
	int result, length;
	char *search, *hostname, *conffile;
	t_host *host_record;
	bool put_method;

	if ((result = fetch_request(session)) != 200) {
		return result;
	} else if ((result = parse_request(session, session->header_length + session->content_length)) != 200) {
		return result;
	}

	put_method = (strcmp(session->method, "PUT") == 0);

	session->host = session->config->first_host;
	if ((hostname = get_headerfield("Host:", session->headerfields)) != NULL) {
		if ((host_record = get_hostrecord(session->config->first_host, hostname, session->binding)) != NULL) {
			session->host = host_record;
		}
	}

	url_decode(session->uri);
	if ((session->vars != NULL) && (session->host->secure_url)) {
		remove_dangerous_chars(session->vars);
	}

	if (session->host->require_resolve_ip && (session->ip_resolved == false)) {
		/* See if the client IP-address is resolvable
		 */
		if (gethostbyaddr((unsigned char*)&(session->ip_address), 4, AF_INET) == NULL) {
			log_error(session, "can't resolve IP");
			return 403;
		}
		session->ip_resolved = true;
	}

	if (valid_uri(session->uri) == false) {
		if (put_method) {
			if (session->binding->enable_alter) {
				return 403;
			}
			return 501;
		}
		return 404;
	} else if ((result = uri_to_path(session)) != 200) {
		return result;
	}

	/* Load configfile from directories
	 */
	search = session->file_on_disk;
	while (*search != '\0') {
		if (*search == '/') {
			length = search - session->file_on_disk + 1;
			if ((conffile = (char*)malloc(length + 10)) != NULL) {
				memcpy(conffile, session->file_on_disk, length);
				memcpy(conffile + length, ".hiawatha\0", 10);
				if (duplicate_host(session)) {
					if (read_user_configfile(conffile, session->host, &(session->tempdata)) > 0) {
						log_file_error(session, conffile, "error in configuration file");
						result = 500;
					}
				} else {
					result = 500;
				}
				free(conffile);
			} else {
				result = 500;
			}
		}
		if (result == 200) {
			search++;
		} else {
			return result;
		}
	}

#ifdef HAVE_SSL
	if (session->host->require_ssl && (session->binding->use_ssl == false)) {
		session->cause_of_301 = require_ssl;
		return 301;
	}
#endif

	if (client_is_rejected_bot(session)) {
		log_error(session, "bot rejected");
		return 403;
	}

	if ((result = copy_directory_settings(session)) != 200) {
		return result;
	}

	switch (is_directory(session->file_on_disk)) {
		case error:
			return 500;
		case yes:
			session->uri_is_dir = true;
			break;
		case no:
			break;
		case no_access:
			log_error(session, fb_filesystem);
			return 403;
		case not_found:
			if (session->host->use_gz_file) {
				if ((search = get_headerfield("Accept-Encoding:", session->headerfields)) != NULL) {
					if ((strstr(search, "gzip")) != NULL) {
						session->accept_gzip = true;
					}
				}
			}
			if ((session->accept_gzip == false) && (put_method == false)) {
				return 404;
			}
	}

	switch (allow_client(session)) {
		case deny:
			log_error(session, fb_accesslist);
			return 403;
		case allow:
			break;
		case pwd:
		case unknown:
			if (http_authentication_oke(session) == false) {
				return 401;
			}
	}

	length = strlen(session->file_on_disk);
	if (*(session->file_on_disk + length - 1) == '/') {
		if (session->uri_is_dir) {
			strcpy(session->file_on_disk + length, session->host->start_file);
		} else {
			return 404;
		}
	} else if (session->uri_is_dir) {
		return 301;
	}

	return handle_request(session);
}
	
/* Request has been handled, handle the return code.
 */
void handle_request_result(t_session *session, int result) {
	if ((result > 0) && (result != 400)) {
		log_request(session, result);
	} else {
		session->keep_alive = false;
	}

	switch (result) {
		case fe_MAX_REQUESTSIZE:
			log_garbage(session);
			send_code(session, 413);
			if ((session->config->ban_on_max_request_size > 0) && ip_allowed(session->ip_address, session->config->banlist_mask)) {
				ban_ip(session->ip_address, session->config->ban_on_max_request_size, session->config->kick_on_ban);
				log_system(session, "Client banned because of sending a too large request");
			}
			break;
		case fe_TIMEOUT:
			if (session->kept_alive == 0) {
				send_code(session, 408);
				if ((session->config->ban_on_timeout > 0) && ip_allowed(session->ip_address, session->config->banlist_mask)) {
					ban_ip(session->ip_address, session->config->ban_on_timeout, session->config->kick_on_ban);
					log_system(session, "Client banned because of connection timeout");
				} else {
					log_system(session, "Timeout while waiting for request");
				}
			}
			break;
		case fe_CLIENT_DISCONNECTED:
			if (session->kept_alive == 0) {
				log_system(session, "Client disconnected");
			}
			break;
		case fe_READ_ERROR:
			if (errno != ECONNRESET) {
				log_system(session, "Error while reading request");
#ifdef DEBUG
				log_system(session, strerror(errno));
#endif
			}
			break;
		case fe_FORCE_QUIT:
			log_system(session, "Client kicked");
			break;
		case rr_CMD_INJECTION:
			log_request(session, result);
			if (ip_allowed(session->ip_address, session->config->banlist_mask)) {
				ban_ip(session->ip_address, session->config->ban_on_cmdi, session->config->kick_on_ban);
				log_system(session, "Client banned because of command injection");
			}
		case rr_SQL_INJECTION:
			log_request(session, result);
			if (ip_allowed(session->ip_address, session->config->banlist_mask)) {
				ban_ip(session->ip_address, session->config->ban_on_sqli, session->config->kick_on_ban);
				log_system(session, "Client banned because of SQL injection");
			}
			send_code(session, 400);
			break;
		case 200:
			break;
		case 204:
			if (session->data_sent == false) {
				send_header(session, 204);
				send_buffer(session, "\r\n", 2);
			}
			break;
		case 304:
		case 412:
		case 413:
			if (session->data_sent == false) {
				send_header(session, result);
				send_buffer(session, "\r\n", 2);
			}
			break;
		case 400:
			log_garbage(session);
			if (session->data_sent == false) {
				send_code(session, result);
			}
			if ((session->config->ban_on_garbage > 0) && ip_allowed(session->ip_address, session->config->banlist_mask)) {
				ban_ip(session->ip_address, session->config->ban_on_garbage, session->config->kick_on_ban);
				log_system(session, "Client banned because of sending garbage");
			}
			break;
		case 500:
			session->keep_alive = false;
		default:
			if (session->data_sent == false) {
				if ((session->host->error_handler == NULL) || (result == 301)) {
					if (send_code(session, result) == -1) {
						session->keep_alive = false;
					}
				} else {
					session->error_code = result;
					switch (handle_error(session)) {
						case -1:
							session->keep_alive = false;
						case 200:
							break;
						default:
							if (send_code(session, result) == -1) {
								session->keep_alive = false;
							}
					}
				}
			}
	}

	restore_host(session);
}

/* Handle the connection of a client.
 */
void connection_handler(t_session *session) {
	int result;

#ifdef HAVE_SSL
	if (session->binding->use_ssl) {
		session->socket_open = ssl_accept(session->client_socket, &(session->ssl_data), session->binding->ssl_context);
	} else
#endif
		session->socket_open = true;

	if (session->socket_open) {
		do {
			result = serve_client(session);
			handle_request_result(session, result);
			send_buffer(session, NULL, 0); // Flush the output-buffer

			reset_session(session);

			if (session->keep_alive && (session->config->ban_on_flooding > 0)) {
				if (client_is_flooding(session)) {
					if (ip_allowed(session->ip_address, session->config->banlist_mask)) {
						ban_ip(session->ip_address, session->config->ban_on_flooding, session->config->kick_on_ban);
						log_system(session, "Client banned because of flooding");
						session->keep_alive = false;
					}
				}
			}
		} while (session->keep_alive);

		destroy_session(session);
		close_socket(session);
	} else {
		close(session->client_socket);
	}
	log_close(session->config->first_host);

	if (session->config->reconnect_delay > 0) {
		mark_client_for_removal(session, session->config->reconnect_delay);
	} else {
		remove_client(session, true);
	}
	// Client session ends here.
}

/* Signal handlers
 */
void SEGV_handler(int sig) {
	syslog(LOG_DAEMON | LOG_ALERT, "segmentation fault!");
	exit(EXIT_FAILURE);
}

void TERM_handler(int sig) {
	received_signal = rs_QUIT_SERVER;
}

void USR1_handler(int sig) {
	received_signal = rs_UNBAN_CLIENTS;
}

void USR2_handler(int sig) {
	received_signal = rs_TOUCH_LOGFILES;
}

/* Fill a filedescriptor set with sockets.
 */
#ifndef HAVE_COMMAND
int fill_read_fds(fd_set *read_fds, t_binding *binding) {
#else
int fill_read_fds(fd_set *read_fds, t_binding *binding, t_binding *bind_command) {
	t_admin *admin;
#endif
	int highest_fd = 0;

	FD_ZERO(read_fds);
	while (binding != NULL) {
		FD_SET(binding->socket, read_fds);
		if (binding->socket > highest_fd) {
			highest_fd = binding->socket;
		}
		binding = binding->next;
	}

#ifdef HAVE_COMMAND
	while (bind_command != NULL) {
		FD_SET(bind_command->socket, read_fds);
		if (bind_command->socket > highest_fd) {
			highest_fd = bind_command->socket;
		}
		bind_command = bind_command->next;
	}
	
	admin = first_admin();
	while (admin != NULL) {
		FD_SET(admin->socket, read_fds);
		if (admin->socket > highest_fd) {
			highest_fd = admin->socket;
		}
		admin = next_admin();
	}
#endif

	return highest_fd;
}

/* Create a socketlist.
 */
int bind_sockets(t_binding *binding) {
	int one;
	struct sockaddr_in saddr;
	char *ip_address;

	while (binding != NULL) {
		if ((binding->socket = socket(AF_INET, SOCK_STREAM, 0)) == -1) {
			perror("socket()");
			return -1;
		}

		bzero(&saddr, sizeof(struct sockaddr_in));
		saddr.sin_family = AF_INET;
		saddr.sin_port = htons(binding->port);
		saddr.sin_addr.s_addr = binding->interface;

		one = 1;
		if (setsockopt(binding->socket, SOL_SOCKET, SO_REUSEADDR, (void*)&one, sizeof(int)) == -1) {
			perror("setsockopt(SOL_SOCKET, SO_REUSEADDR)");
		}
		one = 1;
		if (setsockopt(binding->socket, IPPROTO_TCP, TCP_NODELAY, (void*)&one, sizeof(int)) == -1) {
			perror("setsockopt(IPPROTO_TCP, TCP_NODELAY)");
		}

		if (bind(binding->socket, (struct sockaddr*)&saddr, sizeof(struct sockaddr)) == -1) {
			ip_address = (char*)&(binding->interface);
			printf("Error binding %hhu.%hhu.%hhu.%hhu:%d\n", ip_address[0], ip_address[1], ip_address[2], ip_address[3], binding->port);
			return -1;
		}
		binding = binding->next;
	}

	return 0;
}

/* Accept or deny an incoming connection.
 */
int accept_connection(t_binding *binding, t_config *config) {
	socklen_t          size;
	bool               kick_client;
	t_session          *session;
	struct sockaddr_in caddr;
	pthread_attr_t     child_attr;
	pthread_t          child_thread;
	int                total_conns, one;

	if ((session = (t_session*)malloc(sizeof(t_session))) == NULL) {
		return -1;
	}
	session->config = config;
	session->binding = binding;
	init_session(session);

	size = sizeof(struct sockaddr_in);
	memset((void*)&caddr, 0, size);
	if ((session->client_socket = accept(binding->socket, (struct sockaddr*)&caddr, &size)) == -1) {
		free(session);
		if (errno != EINTR) {
			return -1;
		}
		return 0;
	}

	kick_client = true;
	memcpy((char*)&(session->ip_address), (char*)&caddr.sin_addr.s_addr, 4);
	if ((total_conns = connection_allowed(session->ip_address, config->connections_per_ip, config->total_connections)) >= 0) {
		if (total_conns < (config->total_connections >> 2)) {
			one = 1;
			if (setsockopt(session->client_socket, IPPROTO_TCP, TCP_NODELAY, (void*)&one, sizeof(int)) == -1) {
				close(session->client_socket);
				free(session);
				return -1;
			}
		}

		if (add_client(session) == 0) {
			pthread_attr_init(&child_attr);
			pthread_attr_setdetachstate(&child_attr, PTHREAD_CREATE_DETACHED);
			if (pthread_create(&child_thread, &child_attr, (void*)connection_handler, (void*)session) == 0) {
				kick_client = false;
			} else {
				remove_client(session, false);
			}
		}
	} else switch (total_conns) {
		case ca_TOOMUCH_PERIP:
			if ((config->ban_on_max_per_ip > 0) && ip_allowed(session->ip_address, session->config->banlist_mask)) {
				ban_ip(session->ip_address, config->ban_on_max_per_ip, config->kick_on_ban);
			}
			log_system(session, "Maximum number of connections for IP address reached");
			break;
		case ca_TOOMUCH_TOTAL:
			log_system(session, "Maximum number of total connections reached");
			break;
		case ca_BANNED:
#ifdef DEBUG
			log_system(session, "Client kicked because of ban");
#endif
			if (config->reban_during_ban && ip_allowed(session->ip_address, session->config->banlist_mask)) {
				reban_ip(session->ip_address);
			}
#ifdef HAVE_COMMAND
			increment_counter(COUNTER_DENY);
#endif
			break;
	}

	if (kick_client) {
		close(session->client_socket);
		free(session);
	}

	return 0;
}

/* Run the Hiawatha webserver.
 */
int run_server(t_settings *settings) {
	char               *main_configfile = "httpd.conf";
	int                highest_fd;
	bool               must_quit = false;
#ifdef HAVE_COMMAND
	t_admin            *admin;
	struct sockaddr_in caddr;
	socklen_t          size;
	int                admin_socket;
#endif
#ifdef HAVE_SSL
#ifndef HAVE_DEV_URANDOM
	char               random_buffer[RANDOM_BUFFER_SIZE];
#endif
#endif
	pid_t              pid;
	t_binding          *binding;
	t_config           *config;
    fd_set             read_fds;
	struct timeval     select_timeout;

	signal(SIGPIPE, SIG_IGN);
	signal(SIGCHLD, SIG_IGN);
	signal(SIGTSTP, SIG_IGN);
	signal(SIGABRT, SIG_IGN);
	signal(SIGQUIT, SIG_IGN);
	signal(SIGSEGV, SEGV_handler);

	config = default_config();
	if (chdir(settings->config_dir) == -1) {
		perror(settings->config_dir);
		return EXIT_FAILURE;
	} else if (settings->config_check) {
		printf("Using %s\n", settings->config_dir);
	}
	if (read_main_configfile(main_configfile, config, settings->config_check) == -1) {
		return EXIT_FAILURE;
	} else if (check_configuration(config) == -1) {
		return EXIT_FAILURE;
	}

	if (read_mimetypes(config->mimetype_config, &(config->mimetype)) == -1) {
		fprintf(stderr, "Error while reading mimetype configuration.\n");
		return EXIT_FAILURE;
	}
	if (read_throttleconfig(config->throttle_config, &(config->throttle)) == -1) {
		fprintf(stderr, "Error while reading throttle configuration.\n");
		return EXIT_FAILURE;
	}

	if (settings->config_check) {
		printf("Configuration oke.\n");
		return EXIT_SUCCESS;
	}
	
	/* Bind Serverports
	 */
	if (bind_sockets(config->binding) == -1) {
		perror("bind http");
		return EXIT_FAILURE;
	}

#ifdef HAVE_SSL
#ifdef HAVE_DEV_URANDOM
	ssl_init(NULL, 0);
#else
	fill_random_buffer(config, random_buffer, RANDOM_BUFFER_SIZE);
	ssl_init(random_buffer, RANDOM_BUFFER_SIZE);
#endif
	binding = config->binding;
	while (binding != NULL) {
		if (binding->use_ssl) {
			if ((binding->ssl_context = ssl_binding(binding->server_key, binding->ca_cert, binding->verify_depth, config->dh_file, config->allowed_ciphers)) == NULL) {
				perror("bind https");
				return EXIT_FAILURE;
			}
		}
		binding = binding->next;
	}
#endif

#ifdef HAVE_COMMAND
	/* Bind CommandChannels
	 */
	if (bind_sockets(config->command_port) == -1) {
		perror("bind command");
		return EXIT_FAILURE;
	}
#endif

	touch_logfiles(config, config->server_root);
	tzset();
	clearenv();

	/* Become a daemon
	 */
	if (settings->daemon) {
		switch (pid = fork()) {
			case -1:
				perror("fork()");
				return EXIT_FAILURE;
			case 0:
				if (setsid() == -1) {
					perror("setsid()");
					return EXIT_FAILURE;
				}
				break;
			default:
				log_pid(config, pid);
				return EXIT_SUCCESS;
		}
	} else {
		log_pid(config, getpid());
	}

	/* Change server root
	 */
	if (config->server_root != NULL) {
		do {
			if (chdir(config->server_root) != -1) {
				if (chroot(config->server_root) != -1) {
					break;
				}
			}
			fprintf(stderr, "\nError while changing root to %s!\n", config->server_root);
			return EXIT_FAILURE;
		} while (false);
	} else {
		chdir("/");
	}

	/* Change userid
	 */
	if (getuid() == 0) do {
		if (setgroups(config->groups.number, config->groups.array) != -1) {
			if (setgid(config->server_gid) != -1) {
				if (setuid(config->server_uid) != -1) {
					break;
				}
			}
		}
		fprintf(stderr, "\nError while changing uid/gid!\n");
		return EXIT_FAILURE;
	} while (false);

	if (settings->daemon == false) {
		printf("Press Ctrl-C to shutdown the Hiawatha webserver.\n");
		signal(SIGINT, TERM_handler);
	} else {
		signal(SIGINT, SIG_IGN);
	}

	signal(SIGTERM, TERM_handler);
	signal(SIGUSR1, USR1_handler);
	signal(SIGUSR2, USR2_handler);

	/* Start listening for incoming connections
	 */
	binding = config->binding;
	while (binding != NULL) {
		if (listen(binding->socket, 16) == -1) {
			perror("listen(http(s))");
			return EXIT_FAILURE;
		}
		binding = binding->next;
	}
#ifdef HAVE_COMMAND
	binding = config->command_port;
	while (binding != NULL) {
		if (listen(binding->socket, 1) == -1) {
			perror("listen(commandchannel)");
			return EXIT_FAILURE;
		}
		binding = binding->next;
	}
#endif

	select_timeout.tv_sec = MAIN_LOOP_TIMER;
	select_timeout.tv_usec = 0;

	init_log_module();
	init_client_module();
	init_load_balancer(config->fcgi_server);
#ifdef HAVE_CACHE
	init_cache_module();
#endif
#ifdef HAVE_COMMAND
	init_command_module();
#endif

	log_string(config->system_logfile, "Hiawatha v"VERSION" started");

	do {
#ifdef HAVE_COMMAND
		highest_fd = fill_read_fds(&read_fds, config->binding, config->command_port);
#else
		highest_fd = fill_read_fds(&read_fds, config->binding);
#endif
		switch (select(highest_fd + 1, &read_fds, NULL, NULL, &select_timeout)) {
			case -1:
				if (errno != EINTR) {
					log_string(config->system_logfile, "Fatal error selecting connection");
					usleep(1000);
				}
				break;
			case 0:
				select_timeout.tv_sec = MAIN_LOOP_TIMER;
				select_timeout.tv_usec = 0;

				check_ban_list(config);
				check_delayed_remove_timers(config);
				check_load_balancer(config);
#ifdef HAVE_CACHE
				check_cache();
#endif
#ifdef HAVE_COMMAND
				check_adminlist();
#endif
				switch (received_signal) {
					case rs_NONE:
						break;
					case rs_QUIT_SERVER:
#ifdef DEBUG
						log_string(config->system_logfile, "Received TERM signal");
#endif
						must_quit = true;
						break;
					case rs_UNBAN_CLIENTS:
						unban_ip(0);
						received_signal = rs_NONE;
						break;
					case rs_TOUCH_LOGFILES:
						touch_logfiles(config, NULL);
						received_signal = rs_NONE;
						break;
				}
				break;
			default:
#ifdef HAVE_COMMAND
				// Connected admins
				admin = first_admin();
				while (admin != NULL) {
					if (FD_ISSET(admin->socket, &read_fds)) {
						if (handle_admin(admin, config) == cc_DISCONNECT) {
							remove_admin(admin->socket);
						}
					}
					admin = next_admin();
				}
#endif

				// HTTP(S) port
				binding = config->binding;
				while (binding != NULL) {
					if (FD_ISSET(binding->socket, &read_fds)) {
						if (accept_connection(binding, config) != 0) {
							log_string(config->system_logfile, "Fatal error accepting connection");
							usleep(1000);
							break;

						}
					}
					binding = binding->next;
				}

#ifdef HAVE_COMMAND
				// Command port
				binding = config->command_port;
				while (binding != NULL) {
					if (FD_ISSET(binding->socket, &read_fds)) {
						size = sizeof(struct sockaddr_in);
						memset((void*)&caddr, 0, size);
						if ((admin_socket = accept(binding->socket, (struct sockaddr*)&caddr, &size)) == -1) {
							if (errno != EINTR) {
								log_string(config->system_logfile, "Fatal error accepting CommandChannel connection");
								usleep(1000);
								break;
							}
						} else {
							if (add_admin(admin_socket) == -1) {
								close(admin_socket);
							}
						}
					}
					binding = binding->next;
				}
#endif
		}
	} while (must_quit == false);

	signal(SIGTERM, SIG_DFL);

	disconnect_clients(config);
#ifdef HAVE_COMMAND
	disconnect_admins();
#endif

	binding = config->binding;
	while (binding != NULL) {
		close(binding->socket);
#ifdef HAVE_SSL
		if (binding->use_ssl) {
			ssl_cleanup(binding->ssl_context);
		}
#endif
		binding = binding->next;
	}

#ifdef HAVE_COMMAND
	binding = config->command_port;
	while (binding != NULL) {
		close(binding->socket);
		binding = binding->next;
	}
#endif

	log_string(config->system_logfile, "Hiawatha v"VERSION" stopped");

	return EXIT_SUCCESS;
}

void show_help(char *hiawatha) {
	printf("Usage: %s [options]\n", hiawatha);
	printf("Options: -c <path>: path to where the configrationfiles are located.\n");
	printf("         -d: don't fork to the background.\n");
	printf("         -h: show this information and exit.\n");
	printf("         -k: check configuration and exit.\n");
	printf("         -v: show version and compile information and exit.\n");
}

/* Main and stuff...
 */
int main(int argc, char *argv[]) {
	int i = 0;
	t_settings settings;

	/* Default settings
	 */
	settings.config_dir   = CONFIG_DIR;
	settings.daemon       = true;
	settings.config_check = false;

	/* Read commandline parameters
	 */
	while (++i < argc) {
		if (strcmp(argv[i], "-c") == 0) {
			if (++i < argc) {
				settings.config_dir = argv[i];
			} else {
				fprintf(stderr, "Specify a directory.\n");
				return EXIT_FAILURE;
			}
		} else if (strcmp(argv[i], "-d") == 0) {
			settings.daemon = false;
		} else if (strcmp(argv[i], "-h") == 0) {
			show_help(argv[0]);
			return EXIT_SUCCESS;
		} else if (strcmp(argv[i], "-k") == 0) {
			settings.config_check = true;
		} else if (strcmp(argv[i], "-v") == 0) {
			printf("%s\n", version_string);
			return EXIT_SUCCESS;
		} else {
			fprintf(stderr, "Unknown option. Use '-h' for help.\n");
			return EXIT_FAILURE;
		}
	}

	/* Run Hiawatha
	 */
	return run_server(&settings);
}
