#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "libstr.h"
#include "liblist.h"

void sfree(void* ptr) {
	if (ptr != NULL) {
		free(ptr);
	}
}

/* Convert an IP address to an unsigned long.
 */
int parse_ip(char *line, unsigned long *l_ip) {
	char *ip, *c, org;
	int  i, byte;

	if (line == NULL) {
		return -1;
	}
	if (*line == '\0') {
		return -1;
	}

	ip = (char*)l_ip;
	i = 0;
	c = line - 1;
	do {
		c++;
		switch (*c) {
			case '\0':
				if (i < 3) {
					return -1;
				}
			case '.':
				if (i > 3) {
					return -1;
				}
				org = *c;
				*c = '\0';
				byte = str2int(line);
				if ((byte < 0) || (byte > 255)) {
					return -1;
				}
				*(ip + i) = (char)byte;
				line = c + 1;
				*c = org;
				i++;
				break;
		}
	} while (*c != '\0');

	return 0;
}

// --< charlist >--------------------------------------------------------------

void init_charlist(t_charlist *list) {
	list->size = 0;
	list->item = NULL;
}

int parse_charlist(char *value, t_charlist *list) {
	char *scan;
	int i;

	if (list->size == 0) {
		scan = value;
		list->size = 1;
		while (*scan != '\0') {
			if (*scan == ',') {
				*scan = '\0';
				list->size++;
			}
			scan++;
		}
		
		if ((list->item = (char**)malloc(list->size * sizeof(char*))) != NULL) {
			for (i = 0; i < list->size; i++) {
				*(list->item + i) = strdup(remove_spaces(value));
				value = value + strlen(value) + 1;
			}

			return 0;
		} else {
			return -1;
		}
	} else {
		return -1;
	}
}

bool in_charlist(char *item, t_charlist *list) {
	int i;

	if (((i = list->size) > 0) && (item != NULL)) {
		while (i-- > 0) {
			if (strcmp(*(list->item + i), item) == 0) {
				return true;
			}
		}
	}

	return false;
}

// --< accesslist >------------------------------------------------------------

/* Parse an list of access levels.
 */
t_accesslist *parse_accesslist(char *line, bool pwd_allowed, t_accesslist *list) {
	t_accesslist *new = NULL;
	char *rule, *ip, *mask;
	bool error = false;

	if (list != NULL) {
		new = list;
		while (new->next != NULL) {
			new = new->next;
		}
	}

	while (line != NULL) {
		split_string(line, &ip, &line, ',');
		if (split_string(ip, &rule, &ip, ':') == 0) {
			if (list == NULL) {
				if ((list = new = (t_accesslist*)malloc(sizeof(t_accesslist))) == NULL) {
					error = true;
					break;
				}
			} else {
				if ((new->next = (t_accesslist*)malloc(sizeof(t_accesslist))) == NULL)  {
					error = true;
					break;
				}
				new = new->next;
			}
			new->next = NULL;

			if (strcmp(rule, "allow") == 0) {
				new->access = allow;
			} else if (strcmp(rule, "deny") == 0) {
				new->access = deny;
			} else if (pwd_allowed && (strcmp(rule, "pwd") == 0)) {
				new->access = pwd;
			} else {
				list = remove_accesslist(list);
				break;
			}
			if (strcmp(ip, "all") == 0) {
				new->ip = 0;
				new->netmask = 32;
			} else {
				if (split_string(ip, &ip, &mask, '/') == 0) {
					if ((new->netmask = str2int(mask)) == -1) {
						error = true;
						break;
					} else if (new->netmask > 32) {
						error = true;
						break;
					} else {
						new->netmask = 32 - new->netmask;
					}
				} else {
					new->netmask = 0;
				}
				if (parse_ip(ip, &(new->ip)) == -1) {
					error = true;
					break;
				}
			}
		} else {
			error = true;
			break;
		}
	}

	if (error) {
		list = remove_accesslist(list);
	}

	return list;
}

/* Remove an accesslist.
 */
t_accesslist *remove_accesslist(t_accesslist *list) {
	t_accesslist *ip;

	while (list != NULL) {
		ip = list;
		list = list->next;

		free(ip);
	}
	
	return NULL;
}

/* Return the access status of an IP address.
 */
#define masked(ip, mask) ((ip << mask) >> mask)
t_access ip_allowed(unsigned long ip, t_accesslist *list) {
	while (list != NULL) {
		if (list->netmask == 32) {
			return list->access;
		} else if (masked(ip, list->netmask) == masked(list->ip, list->netmask)) {
			return list->access;
		}
		list = list->next;
	}

	return unknown;
}

// --< key/valuelist >---------------------------------------------------------

/* Parse an list of key/value combinations.
 */
t_keyvalue *parse_keyvaluelist(char *line, t_keyvalue *kvlist, char seperator) {
	char *key, *value, *rest;
	t_keyvalue *prev;

	while (line != NULL) {
		split_string(line, &line, &rest, ',');
		if (split_string(line, &key, &value, seperator) == 0) {
			prev = kvlist;
			if ((kvlist = (t_keyvalue*)malloc(sizeof(t_keyvalue))) == NULL) {
				remove_keyvaluelist(prev);
				break;
			}
			kvlist->next = prev;

			kvlist->key = strdup(key);
			kvlist->value = strdup(value);
		} else {
			remove_keyvaluelist(kvlist);
			kvlist = NULL;
			break;
		}
		line = rest;
	}

	return kvlist;
}

t_keyvalue *remove_keyvaluelist(t_keyvalue *list) {
	t_keyvalue *remove;

	while (list != NULL) {
		remove = list;
		list = list->next;

		free(remove->key);
		free(remove->value);
		free(remove);
	}

	return NULL;
}
