#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <unistd.h>
#include <string.h>
#ifdef HAVE_CRYPT_H
#include <crypt.h>
#endif
#ifdef HAVE_RPCSVC_CRYPT_H
#include <rpcsvc/crypt.h>
#endif
#include "libstr.h"
#include "libmd5.h"
#include "httpauth.h"

/* If a groupfile exist, is a user in the right group?
 *  */
static bool group_oke(t_session *session, char *user) {
	bool retval;
	FILE *gfp;
	char *pwfile, *line, *item, *rest;
	bool groupfound;
	int i;

	if (session->host->groupfile != NULL) {
		retval = false;
		pwfile = file_in_chroot(session, session->host->groupfile);
		if ((gfp = fopen_neighbour(pwfile, "r", session->file_on_disk)) != NULL) {
			if ((line = (char*)malloc(257)) != NULL) {
				*(line + 256) = '\0';
				while (fgets(line, 256, gfp) != NULL) {
					if (split_string(line, &item, &rest, ':') == 0) {
						groupfound = false;
						for (i = 0; i < session->host->required_group.size; i++) {
							if (strcmp(*(session->host->required_group.item + i), item) == 0) {
								groupfound = true;
								break;
							}
						}
						if (groupfound) {
							while (rest != NULL) {
								split_string(rest, &item, &rest, ' ');
								if (strcmp(user, item) == 0) {
									retval = true;
									break;
								}
							}
						}
					}
					if (retval == true) {
						break;
					}
				}
				free(line);
			}
			fclose(gfp);
		}
	} else {
		retval = true;
	}

	return retval;
}

/* Read a password from a file
 */
static char *get_password(t_session *session, char *username) {
	char *pwfile, line[256], *result = NULL, *sep;
	FILE *pfp;
	
	pwfile = file_in_chroot(session, session->host->passwordfile);
	if ((pfp = fopen_neighbour(pwfile, "r", session->file_on_disk)) != NULL) {
		line[255] = '\0';
		while ((sep = fgets(line, 255, pfp)) != NULL) {
			while ((*sep != ':') && (*sep != '\0')) {
				sep++;
			}
			if (*sep == ':') {
				*(sep++) = '\0';
				if (strcmp(line, username) == 0) {
					result = sep;

					while ((*sep != '\n') && (*sep != ':') && (*sep != '\0')) {
						sep++;
					}
					if (*sep != '\0') {
						*sep = '\0';
						result = strdup(result);
					} else {
						result = NULL;
					}
				}
			}
		}
		fclose(pfp);
	}

	return result;
}

/* Basic HTTP authentication.
 */
bool basic_http_authentication(t_session *session, char *auth_str) {
	bool retval = false;
	char *auth_user, *auth_passwd, *passwd, *encrypted, salt[3];

	if ((auth_user = strdup(auth_str)) == NULL) {
		return false;
	}

	if (decode_base64(auth_user)) {
		auth_passwd = auth_user;
		while ((*auth_passwd != ':') && (auth_passwd != '\0')) {
			auth_passwd++;
		}
		if (*auth_passwd == ':') {
			*(auth_passwd++) = '\0';

			if ((passwd = get_password(session, auth_user)) != NULL) {
				if (group_oke(session, auth_user)) {
					salt[0] = *passwd;
					salt[1] = *(passwd + 1);
					salt[2] = '\0';
					encrypted = crypt(auth_passwd, salt);

					/* Password match?
					 */
					if (strcmp(encrypted, passwd) == 0) {
						session->remote_user = strdup(auth_user);
						retval = true;
					}
				}
				free(passwd);
			}
		}
	}
	free(auth_user);

	return retval;
}

static char *unquoted(char *str) {
	int len;

	if (str != NULL) {
		len = strlen(str);
		if (len > 0) {
			if (*str == '\"') {
				str++;
				len--;
			}
			if (str[len - 1] == '\"') {
				str[len - 1] = '\0';
			}
		}
	}

	return str;
}

/* Digest HTTP authentication.
 */
bool digest_http_authentication(t_session *session, char *auth_str) {
	bool retval = false;
	char *key, *value, *rest, *empty = "", *passwd, A1[33], A2[33], result[33];
	char *username = empty, *realm = empty, *nonce = empty, *uri = empty, *qop = empty,
		 *nc = empty, *cnonce = empty, *algoritm = empty, *response = empty, *opaque = empty;

	key = auth_str;
	while (key != NULL) {
		split_string(key, &key, &rest, ',');
		if (split_string(key, &key, &value, '=') != -1) {
			if (strcmp(key, "username") == 0) {
				username = unquoted(value);
			} else if (strcmp(key, "realm") == 0) {
				realm = unquoted(value);
			} else if (strcmp(key, "nonce") == 0) {
				nonce = unquoted(value);
			} else if (strcmp(key, "uri") == 0) {
				uri = unquoted(value);
			} else if (strcmp(key, "qop") == 0) {
				qop = unquoted(value);
			} else if (strcmp(key, "nc") == 0) {
				nc = unquoted(value);
			} else if (strcmp(key, "cnonce") == 0) {
				cnonce = unquoted(value);
			} else if (strcmp(key, "algoritm") == 0) {
				algoritm = unquoted(value);
			} else if (strcmp(key, "response") == 0) {
				response = unquoted(value);
			} else if (strcmp(key, "opaque") == 0) {
				opaque = unquoted(value);
			}
		}
		key = rest;
	}

	if (session->vars != NULL) {
		if (strncmp(session->uri, uri, strlen(session->uri)) != 0) {
			return false;
		} else if (*(uri + strlen(session->uri)) != '?') {
			return false;
		} else if (strcmp(session->vars, uri + strlen(session->uri) + 1) != 0) {
			return false;
		}
	} else if (strcmp(session->uri, uri) != 0) {
		return false;
	}

	if ((passwd = get_password(session, username)) != NULL) {
		if (group_oke(session, username)) do {
			/* Calculate A1
			 */
			if ((value = (char*)malloc(strlen(username) + strlen(realm) + strlen(passwd) + 3)) == NULL) {
				break;
			}
			sprintf(value, "%s:%s:%s", username, realm, passwd);
			md5_hash(value, strlen(value), A1);
			free(value);

			/* Calculate A2
			 */
			if ((value = (char*)malloc(strlen(session->method) + strlen(uri) + 2)) == NULL) {
				break;
			}
			sprintf(value, "%s:%s", session->method, uri);
			md5_hash(value, strlen(value), A2);
			free(value);

			/* Calculate response
			 */
			if ((value = (char*)malloc(strlen(A1) + strlen(nonce) + strlen(A2) + 6)) == NULL) {
				break;
			}
			sprintf(value, "%s:%s:%s", A1, nonce, A2);
			md5_hash(value, strlen(value), result);
			free(value);

			/* Password match?
			 */
			retval = (strcmp(result, response) == 0);
		} while (false);
		free(passwd);
	}

	return retval;
}

/* Check if the file is protected by an .hiawatha file with passwordfile setting.
 */
bool http_authentication_oke(t_session *session) {
	char *auth_str;

	if (session->host->passwordfile == NULL) {
		return true;
	} else if ((auth_str = get_headerfield("Authorization:", session->headerfields)) != NULL) {
		if ((strncmp(auth_str, "Basic ", 6) == 0) && (session->host->auth_method == basic)) {
			return basic_http_authentication(session, auth_str + 6);
		} else if ((strncmp(auth_str, "Digest ", 7) == 0) && (session->host->auth_method == digest)) {
			return digest_http_authentication(session, auth_str + 7);
		}
	}

	return false;
}
