#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <signal.h>
#include <dirent.h>
#include <errno.h>
#include <pthread.h>
#include <grp.h>
#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif
#ifdef HAVE_NETINET_TCP_H
#include <netinet/tcp.h>
#endif
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include "mimetype.h"
#include "serverconfig.h"
#include "libfs.h"
#include "liblist.h"
#include "libstr.h"
#include "cgi.h"
#include "userconfig.h"
#include "session.h"
#include "httpauth.h"
#include "send.h"
#include "client.h"
#include "target.h"
#include "log.h"
#ifndef HAVE_SETENV
#include "envir.h"
#endif
#ifdef HAVE_COMMAND
#include "command.h"
#endif
#ifdef HAVE_SSL
#include "libssl.h"
#endif
#ifdef HAVE_PLUGIN
#include "plugin.h"
#endif
#ifdef HAVE_CACHE
#include "cache.h"
#endif

#define FILENAME_BUFFER_SIZE   256
#define REQUEST_BUFFER_CHUNK  4096
#define MAIN_LOOP_TIMER          1

#define fe_MAX_REQUESTSIZE      -2
#define fe_TIMEOUT              -3
#define fe_CLIENT_DISCONNECTED  -4
#define fe_READ_ERROR           -5
#define fe_FORCE_QUIT           -6

#define rs_NONE                  0
#define rs_QUIT_SERVER           1
#define rs_UNBAN_CLIENTS         2
#define rs_TOUCH_LOGFILES        3

volatile int received_signal = rs_NONE;

char *default_configdir      = CONFIG_DIR;
char *main_configfile        = "httpd.conf";

char *hs_conlen = "Content-Length: ";

#ifdef DEBUG

/* timer functions for debugging
 */

double t1, t2;
struct timeval tp;

void start_clock(char *mesg) {
	fprintf(stderr, "(%5d) START: %s\n", getpid(), mesg);
	gettimeofday(&tp, NULL);
	t1 = (double)tp.tv_sec+(1.e-6)*tp.tv_usec;
}

void stop_clock(void) {
	gettimeofday(&tp, NULL);
	t2 = (double)tp.tv_sec+(1.e-6)*tp.tv_usec;
	fprintf(stderr, "(%5d)  time: %f\n", getpid(), t2 - t1);
}

#endif

/* Create all logfiles with the right ownership and accessrights
 */
void touch_logfiles(t_config *config, char *basedir) {
	t_host *host;

	touch_logfile(basedir, config->system_logfile, 0640, config->server_uid, config->server_gid);
	if (config->garbage_logfile != NULL) {
		touch_logfile(basedir, config->garbage_logfile, 0640, config->server_uid, config->server_gid);
	}

	host = config->first_host;
	while (host != NULL) {
		if (host->access_fileptr != NULL) {
			fflush(host->access_fileptr);
		}
		touch_logfile(basedir, host->access_logfile, 0640, config->server_uid, config->server_gid);
		touch_logfile(basedir, host->error_logfile, 0640, config->server_uid, config->server_gid);
		host = host->next;
	}
}

/* Handle an HTTP error.
 */
int handle_error(t_session *session) {
	int retval = 200, length;
	t_cgi_handler *cgi;
	
	session->handling_error = true;
	session->mimetype = NULL;
	if (session->file_on_disk != NULL) {
		free(session->file_on_disk);
	}
	length = strlen(session->host->website_root);
	if ((session->file_on_disk = (char*)malloc(length + strlen(session->host->error_handler) + 4)) == NULL) { // + 3 for .gz (gzip encoding)
		return 500;
	}
	strcpy(session->file_on_disk, session->host->website_root);
	strcpy(session->file_on_disk + length, session->host->error_handler);

	if ((session->is_cgi_script = in_charlist(file_extension(session->file_on_disk), &(session->config->cgi_extension))) == false) {
		if ((session->fcgi_server = fcgi_server_match(session->config->fcgi_server, &(session->host->fast_cgi), session->file_on_disk)) != NULL) {
			session->is_cgi_script = true;
		} else {
			cgi = session->config->cgi_handler;
			while (cgi != NULL) {
				if (in_charlist(file_extension(session->file_on_disk), &(cgi->extension))) {
					session->is_cgi_script = true;
					session->cgi_handler = cgi->handler;
					break;
				}
				cgi = cgi->next;
			}
		}
	}

	if (session->is_cgi_script) {
		if (session->host->return_code != -1) {
			session->return_code = session->host->return_code;
		}
		retval = run_script(session);
	} else {
		retval = send_file(session);
	}

	return retval;
}

/* Handle the request from a client.
 */
int handle_request(t_session *session) {
	int retval = 200;
	t_cgi_handler *cgi;

	if ((session->is_cgi_script = in_charlist(file_extension(session->file_on_disk), &(session->config->cgi_extension))) == false) {
		if ((session->fcgi_server = fcgi_server_match(session->config->fcgi_server, &(session->host->fast_cgi), session->file_on_disk)) != NULL) {
			session->is_cgi_script = true;
		} else {
			cgi = session->config->cgi_handler;
			while (cgi != NULL) {
				if (in_charlist(file_extension(session->file_on_disk), &(cgi->extension))) {
					session->cgi_handler = cgi->handler;
					session->is_cgi_script = true;
					break;
				}
				cgi = cgi->next;
			}
		}
	}

	if (strcmp(session->method, "HEAD") == 0) {
		session->head_request = true;
	}
	if ((strcmp(session->method, "GET") == 0) || session->head_request) {
		if (session->is_cgi_script) {
			session->body = NULL;
			retval = run_script(session);
		} else {
			retval = send_file(session);
		}
		if (retval == 404) {
			if (session->host->show_index && (session->uri[strlen(session->uri) - 1] == '/')) {
				retval = show_index(session);
			}
		}
	} else if (strcmp(session->method, "POST") == 0) {
		if (session->is_cgi_script) {
			retval = run_script(session);
		} else {
			retval = 405;
		}
	} else if (strcmp(session->method, "OPTIONS") == 0) {
		retval = handle_options_request(session);
	} else if (strcmp(session->method, "TRACE") == 0) {
		if (session->binding->enable_trace) {
			retval = handle_trace_request(session);
		} else {
			retval = 501;
		}
	} else if (strcmp(session->method, "PUT") == 0) {
		retval = 501;
	} else if (strcmp(session->method, "DELETE") == 0) {
		retval = 501;
	} else if (strcmp(session->method, "CONNECT") == 0) {
		retval = 501;
	} else if (strcmp(session->method, "PROPFIND") == 0) {
		retval = 501;
	} else {
		retval = 400;
	}

	return retval;
}

/* Read the request from a client socket.
 */
int fetch_request(t_session *session) {
	char *new_reqbuf, *strstart, *strend;
	long bytes_read, header_length = -1, content_length = -1;
	int result = 200, timer;
	fd_set read_fds;
	struct timeval select_timeout;
	bool keep_reading = true;

	select_timeout.tv_sec = 1;
	select_timeout.tv_usec = 0;
	if (session->kept_alive == 0) {
		timer = session->binding->time_for_1st_request;
	} else {
		timer = session->binding->time_for_request;
	}

	do {
		/* Check if requestbuffer contains a complete request.
		 */
		if (session->request != NULL) {
			if (header_length == -1) {
				if ((strstart = strstr(session->request, "\r\n\r\n")) != NULL) {
					header_length = strstart + 4 - session->request;
					session->header_length = header_length;
				}
			}
			if (header_length != -1) {
				if (content_length == -1) {
					if ((strstart = strcasestr(session->request, hs_conlen)) != NULL) {
						strstart += 16;
						if ((strend = strstr(strstart, "\r\n")) != NULL) {
							*strend = '\0';
							if ((content_length = str2int(strstart)) == -1) {
								break;
							}
							session->content_length = content_length;
							*strend = '\r';
						}
					} else {
						break;
					}
				}
				if (content_length > -1) {
					if (session->bytes_in_buffer >= header_length + content_length) {
						break;
					}
				}
			}
		}

		FD_ZERO(&read_fds);
		FD_SET(session->client_socket, &read_fds);
		switch (select(session->client_socket + 1, &read_fds, NULL, NULL, &select_timeout)) {
			case -1:
				if (errno != EINTR) {
					result = fe_READ_ERROR;
					keep_reading = false;
				}
				break;
			case 0:
				if (session->force_quit) {
					result = fe_FORCE_QUIT;
					keep_reading = false;
				} else if (--timer <= 0) {
					result = fe_TIMEOUT;
					keep_reading = false;
				} else {
					select_timeout.tv_sec = 1;
					select_timeout.tv_usec = 0;
				}
				break;
			default:
				if ((session->buffer_size - session->bytes_in_buffer) < 256) {
					session->buffer_size += REQUEST_BUFFER_CHUNK;
					if ((new_reqbuf = (char*)realloc(session->request, session->buffer_size)) != NULL) {
						session->request = new_reqbuf;
					} else {
						result = fe_READ_ERROR;
						keep_reading = false;
						break;
					}
				}

				/* Read from socket.
				 */
#ifdef HAVE_SSL
				if (session->binding->use_ssl) {
					bytes_read = ssl_receive(session->ssl_data, session->request + session->bytes_in_buffer, 
									session->buffer_size - session->bytes_in_buffer - 1);
				} else
#endif
					bytes_read = recv(session->client_socket, session->request + session->bytes_in_buffer,
									session->buffer_size - session->bytes_in_buffer - 1, 0);

				switch (bytes_read) {
					case -1:
						if (errno != EINTR) {
							result = fe_READ_ERROR;
							keep_reading = false;
						}
						break;
					case 0:
						result = fe_CLIENT_DISCONNECTED;
						keep_reading = false;
						break;
					default:
						session->bytes_in_buffer += bytes_read;
						*(session->request + session->bytes_in_buffer) = '\0';

						if (session->bytes_in_buffer > session->binding->max_request_size) {
							keep_reading = false;
							result = fe_MAX_REQUESTSIZE;
							break;
						}
				}
		}
	} while (keep_reading);

	return result;
}

/* Convert the requestbuffer to a session record.
 */
int parse_request(t_session *session, int total_bytes) {
	int retval = 200;
	char *request_end, *str_end, *conn;

	request_end = session->request + total_bytes;

	/* Scan the first request line.
	 */
	session->method = str_end = session->request;
	while ((*str_end != ' ') && (str_end != request_end)) {
		str_end++;
	}
	if (str_end == request_end) {
		return 400;
	}

	*str_end = '\0';
	session->uri = ++str_end;

	while ((*str_end != ' ') && (str_end != request_end)) {
		str_end++;
		if (*str_end == '?') {
			if (session->vars == NULL) {
				session->vars = str_end + 1;
				*str_end = '\0';
			}
		}
	}
	if (str_end == request_end) {
		return 400;
	}

	*str_end = '\0';
	str_end++;
	if (strlen(str_end) < 8) {
		return 400;
	} else if (memcmp(str_end, "HTTP/", 5) != 0) {
		return 400;
	}

	session->http_version = str_end;
	str_end += 7;
	if ((session->body = strstr(str_end + 1, "\r\n\r\n")) != NULL) {
		*(session->body + 2) = '\0';
		session->body += 4;
	}
	if (str_end + 5 != session->body) {
		session->headerfields = parse_headerfields(str_end + 3);
	}

	if ((*(str_end - 1) != '.') || (*(str_end + 1) != '\r') || (*(str_end + 2) != '\n')) {
		return 400;
	} else if (*(str_end - 2) != '1') {
		return 505;
	}
	*(str_end + 1) = '\0';

	if ((conn = get_headerfield("Connection:", session->headerfields)) != NULL) {
		conn = strlower(remove_spaces(conn));
	}
	session->keep_alive = false;
	switch (*str_end) {
		case '0':
			if ((conn != NULL) && (session->kept_alive < session->binding->max_keepalive)) {
				if (strcasecmp(conn, "keep-alive") == 0) {
					session->keep_alive = true;
				}
			}
			break;
		case '1':
			if (get_headerfield("Host:", session->headerfields) == NULL) {
				retval = 400;
			} else if (session->kept_alive < session->binding->max_keepalive) {
				session->keep_alive = true;
				if (conn != NULL) {
					if (strcmp(conn, "close") == 0) {
						session->keep_alive = false;
					}
				}
			}
			break;
		default:
			retval = 505;
			break;
	}
	if (session->keep_alive) {
		session->kept_alive++;
	}

	return retval;
}

/* Convert the request uri to a filename.
 */
int uri_to_path(t_session *session) {
	int retval = 200, length, alias_length;
	char *strstart, *strend;
	t_keyvalue *alias;
	
	/* Requested file in userdirectory?
	 */
	if ((session->host->user_websites) && (*(session->uri + 1) == '~')) {
		strstart = session->uri + 1;
		if ((strend = strchr(strstart, '/')) == NULL) {
			return 301;
		} else if ((session->local_user = (char*)malloc(strend - strstart + 1)) == NULL) {
			return 500;
		}
		memcpy(session->local_user, strstart, strend - strstart);
		*(session->local_user + (strend - strstart)) = '\0';
		
		if (duplicate_host(session)) {
			retval = get_homedir(session, session->local_user + 1);
			session->host->error_handler = NULL;
		} else {
			retval = 500;
		}
	}

	/* Convert request to complete path + filename.
	 */
	if (retval == 200) {
		if ((session->file_on_disk = (char*)malloc(FILENAME_BUFFER_SIZE + 4)) != NULL) { // + 3 for .gz (gzip encoding)
			*(session->file_on_disk + FILENAME_BUFFER_SIZE) = '\0';

			/* Search for an alias.
			 */
			alias = session->host->alias;
			while (alias != NULL) {
				alias_length = strlen(alias->key);
				if (strncmp(session->uri, alias->key, alias_length) == 0) {
					if ((*(session->uri + alias_length) == '/') ||
						(*(session->uri + alias_length) == '\0')) {
						break;
					}
				}
				alias = alias->next;
			}
			if (alias == NULL) {
				/* No alias
				 */
				strncpy(session->file_on_disk, session->host->website_root, FILENAME_BUFFER_SIZE);
				strstart = session->uri;
				if (session->local_user != NULL) {
					strstart += strlen(session->local_user) + 1;
				}
				length = strlen(session->file_on_disk);
				strncpy(session->file_on_disk + length, strstart, FILENAME_BUFFER_SIZE - length);
			} else {
				/* Use alias
				 */
				strncpy(session->file_on_disk, alias->value, FILENAME_BUFFER_SIZE);
				length = strlen(session->file_on_disk);
				strncpy(session->file_on_disk + length, session->uri + alias_length, FILENAME_BUFFER_SIZE - length);
			}

		} else {
			retval = 500;
		}
	}

	return retval;
}

#ifdef HAVE_PLUGIN
/* Should the plugin be used?
 */
bool is_plugin_uri(t_session *session) {
	if ((session->config->plugin_uri != NULL) && (session->host->plugin_active)) {
		if (strcmp(session->uri, session->config->plugin_uri) == 0) {
			return true;
		}
	}

	return false;
}
#endif

t_access allow_client(t_session *session) {
	char *x_forwarded_for;
	unsigned long forwarded_ip;
	t_access access;

	if ((access = ip_allowed(session->ip_address, session->host->access_list)) != allow) {
		return access;
	} else if ((x_forwarded_for = get_headerfield("X-Forwarded-For:", session->headerfields)) == NULL) {
		return allow;
	} else if (parse_ip(x_forwarded_for, &forwarded_ip) == -1) {
		return allow;
	} else if (ip_allowed(forwarded_ip, session->host->access_list) == deny) {
		return deny;
	}

	return unknown;
}

/* Serve the client that connected to the webserver
 */
int serve_client(t_session *session) {
	int result, length;
	char *dir_config, *dir, *search, *hostname;
	t_host *host_record;

	if ((result = fetch_request(session)) != 200) {
		return result;
	} else if ((result = parse_request(session, session->header_length + session->content_length)) != 200) {
		return result;
	}

	parse_special_chars(session->uri, strlen(session->uri) + 1);
	if (session->vars != NULL) {
		parse_special_chars(session->vars, strlen(session->vars) + 1);
	}

	session->host = session->config->first_host;
	if ((hostname = get_headerfield("Host:", session->headerfields)) != NULL) {
		if ((host_record = get_hostrecord(session->config->first_host, hostname, session->binding)) != NULL) {
			session->host = host_record;
		}
	}

	if ((result = uri_to_path(session)) != 200) {
		return result;
	}

	/* Load configfile from directories
	 */
	if ((dir_config = strdup(session->file_on_disk)) != NULL) {
		search = dir_config;
		while (*search != '\0') {
			if (*search == '/') {
				length = search - dir_config + 1;
				if ((dir = (char*)malloc(length + 16)) != NULL) {
					memcpy(dir, dir_config, length);
					memcpy(dir + length, ".hiawatha\0", 10);
					if (duplicate_host(session)) {
						if (read_user_configfile(dir, session->host) == -1) {
							result = 500;
						}
					} else {
						result = 500;
					}
					free(dir);
				} else {
					result = 500;
				}
			}

			if (result == 200) {
				search++;
			} else {
				break;
			}
		}
		free(dir_config);

		if (result != 200) {
			return result;
		}
	} else {
		return 500;
	}

#ifdef HAVE_SSL
	if (session->host->require_ssl && (session->binding->use_ssl == false)) {
		session->cause_of_301 = require_ssl;
		return 301;
	}
#endif

	if (valid_uri(session->uri) == false) {
		return 404;
	} else if (client_is_rejected_bot(session)) {
		set_403_reason(session, fb_DENYBOT);
		return 403;
	}

#ifdef HAVE_PLUGIN
	if (is_plugin_uri(session)) {
		switch (allow_client(session)) {
			case deny:
				set_403_reason(session, fb_ACCESSLIST);
				return 403;
				break;
			case allow:
				break;
			case pwd:
			case unknown:
				if (http_authentication_oke(session) == false) {
					return 401;
				}
				break;
		}
		return handle_plugin(session);
	}
#endif

	switch (is_directory(session->file_on_disk)) {
		case error:
			return 500;
		case yes:
			session->uri_is_dir = true;
			break;
		case no:
			break;
		case no_access:
			set_403_reason(session, fb_FILESYSTEM);
			return 403;
		case not_found:
			if ((search = get_headerfield("Accept-Encoding:", session->headerfields)) != NULL) {
				if ((strstr(search, "gzip")) != NULL) {
					session->accept_gzip = true;
				}
			}
	}

	if ((result = copy_directory_settings(session)) != 200) {
		return result;
	} else switch (allow_client(session)) {
		case deny:
			set_403_reason(session, fb_ACCESSLIST);
			return 403;
			break;
		case allow:
			break;
		case pwd:
		case unknown:
			if (http_authentication_oke(session) == false) {
				return 401;
			}
			break;
	}

	if (*(session->uri + strlen(session->uri) - 1) == '/') {
		length = strlen(session->file_on_disk);
		strncpy(session->file_on_disk + length, session->host->start_file, FILENAME_BUFFER_SIZE - length);
	} else if (session->uri_is_dir) {
		return 301;
	}

	return handle_request(session);
}
	
/* Request has been handled, handle the return code.
 */
void handle_request_result(t_session *session, int result) {
	if ((result > 0) && (result != 400)) {
		log_request(session, result);
	} else {
		session->keep_alive = false;
	}

	switch (result) {
		case fe_MAX_REQUESTSIZE:
			log_garbage(session);
			send_code(session, 400);
			if ((session->config->ban_on_max_request_size > 0) && ip_allowed(session->ip_address, session->config->banlist_mask)) {
				ban_ip(session->ip_address, session->config->ban_on_max_request_size, session->config->kick_on_ban);
				log_warning(session->config->system_logfile, "Client banned because of sending a too large request", session->ip_address);
			}
			break;
		case fe_TIMEOUT:
			if (session->kept_alive == 0) {
				send_code(session, 408);
				if ((session->config->ban_on_timeout > 0) && ip_allowed(session->ip_address, session->config->banlist_mask)) {
					ban_ip(session->ip_address, session->config->ban_on_timeout, session->config->kick_on_ban);
					log_warning(session->config->system_logfile, "Client banned because of connection timeout", session->ip_address);
				} else {
					log_warning(session->config->system_logfile, "Timeout while waiting for request", session->ip_address);
				}
			}
			break;
		case fe_CLIENT_DISCONNECTED:
			if (session->kept_alive == 0) {
				log_warning(session->config->system_logfile, "Client disconnected", session->ip_address);
			}
			break;
		case fe_READ_ERROR:
			if (errno != ECONNRESET) {
				log_warning(session->config->system_logfile, "Error while reading request", session->ip_address);
#ifdef DEBUG
				log_warning(session->config->system_logfile, strerror(errno), session->ip_address);
#endif
			}
			break;
		case fe_FORCE_QUIT:
			log_warning(session->config->system_logfile, "Client kicked", session->ip_address);
			break;
		case rr_CMD_INJECTION:
			log_request(session, result);
			if (ip_allowed(session->ip_address, session->config->banlist_mask)) {
				ban_ip(session->ip_address, session->config->ban_on_cmdi, session->config->kick_on_ban);
				log_warning(session->config->system_logfile, "Client banned because of command injection", session->ip_address);
			}
		case rr_SQL_INJECTION:
			log_request(session, result);
			if (ip_allowed(session->ip_address, session->config->banlist_mask)) {
				ban_ip(session->ip_address, session->config->ban_on_sqli, session->config->kick_on_ban);
				log_warning(session->config->system_logfile, "Client banned because of SQL injection", session->ip_address);
			}
			send_code(session, 400);
			break;
		case 200:
			break;
		case 304:
		case 412:
		case 413:
			send_header(session, result);
			send_buffer(session, "\r\n", 2);
			break;
		case 400:
			log_garbage(session);
			send_code(session, result);
			if ((session->config->ban_on_garbage > 0) && ip_allowed(session->ip_address, session->config->banlist_mask)) {
				ban_ip(session->ip_address, session->config->ban_on_garbage, session->config->kick_on_ban);
				log_warning(session->config->system_logfile, "Client banned because of sending garbage", session->ip_address);
			}
			break;
		case 500:
			session->keep_alive = false;
		default:
			if ((session->host->error_handler == NULL) || (result == 301)) {
				if (send_code(session, result) == -1) {
					session->keep_alive = false;
				}
			} else {
				session->error_code = result;
				switch (handle_error(session)) {
					case -1:
						session->keep_alive = false;
					case 200:
						break;
					default:
						if (send_code(session, result) == -1) {
							session->keep_alive = false;
						}
				}
			}
	}

	restore_host(session);
}

/* Handle the connection of a client.
 */
void connection_handler(t_session *session) {
	int result;

#ifdef HAVE_SSL
	if (session->binding->use_ssl) {
		session->socket_open = ssl_accept(session->client_socket, &(session->ssl_data), session->binding->ssl_context);
	} else
#endif
		session->socket_open = true;

	if (session->socket_open) {
		do {
			result = serve_client(session);
			handle_request_result(session, result);
			send_buffer(session, NULL, 0); // Flush the output-buffer

#ifdef HAVE_COMMAND
			CC_client_lock();
#endif
			reset_session(session);
#ifdef HAVE_COMMAND
			CC_client_unlock();
#endif

			if (session->keep_alive && (session->config->ban_on_flooding > 0)) {
				if (client_is_flooding(session)) {
					if (ip_allowed(session->ip_address, session->config->banlist_mask)) {
						ban_ip(session->ip_address, session->config->ban_on_flooding, session->config->kick_on_ban);
						log_warning(session->config->system_logfile, "Client banned because of flooding", session->ip_address);
						session->keep_alive = false;
					}
				}
			}
		} while (session->keep_alive);

		close_socket(session);
	} else {
		close(session->client_socket);
	}
	log_close(session->config->first_host);
	//destroy_session(session);

#ifdef HAVE_COMMAND
	CC_client_lock();
#endif
	if (session->config->reconnect_delay > 0) {
		mark_client_for_removal(session, session->config->reconnect_delay);
	} else {
		remove_client(session, true);
	}
#ifdef HAVE_COMMAND
	CC_client_unlock();
#endif
	// Client session ends here.
}

void TERM_handler() {
	received_signal = rs_QUIT_SERVER;
}

void USR1_handler() {
	received_signal = rs_UNBAN_CLIENTS;
}

void USR2_handler() {
	received_signal = rs_TOUCH_LOGFILES;
}

/* Fill a filedescriptor set with sockets.
 */
#ifndef HAVE_COMMAND
int fill_read_fds(fd_set *read_fds, t_binding *binding) {
#else
int fill_read_fds(fd_set *read_fds, t_binding *binding, t_binding *bind_command) {
	t_admin *admin;
#endif
	int highest_fd = 0;

	FD_ZERO(read_fds);
	while (binding != NULL) {
		FD_SET(binding->socket, read_fds);
		if (binding->socket > highest_fd) {
			highest_fd = binding->socket;
		}
		binding = binding->next;
	}
#ifdef HAVE_COMMAND
	while (bind_command != NULL) {
		FD_SET(bind_command->socket, read_fds);
		if (bind_command->socket > highest_fd) {
			highest_fd = bind_command->socket;
		}
		bind_command = bind_command->next;
	}
	
	admin = first_admin();
	while (admin != NULL) {
		FD_SET(admin->socket, read_fds);
		if (admin->socket > highest_fd) {
			highest_fd = admin->socket;
		}
		admin = next_admin();
	}
#endif

	return highest_fd;
}

/* Create a socketlist.
 */
int bind_socket(t_binding *binding) {
	int one;
	struct sockaddr_in saddr;
	char *ip_address;

	while (binding != NULL) {
		if ((binding->socket = socket(AF_INET, SOCK_STREAM, 0)) == -1) {
			perror("socket()");
			return -1;
		}

		bzero(&saddr, sizeof(struct sockaddr_in));
		saddr.sin_family = AF_INET;
		saddr.sin_port = htons(binding->port);
		saddr.sin_addr.s_addr = binding->interface;

		one = 1;
		if (setsockopt(binding->socket, SOL_SOCKET, SO_REUSEADDR, (void*)&one, sizeof(int)) == -1) {
			perror("setsockopt(SOL_SOCKET, SO_REUSEADDR)");
		}
		one = 1;
		if (setsockopt(binding->socket, IPPROTO_TCP, TCP_NODELAY, (void*)&one, sizeof(int)) == -1) {
			perror("setsockopt(IPPROTO_TCP, TCP_NODELAY)");
		}

		if (bind(binding->socket, (struct sockaddr*)&saddr, sizeof(struct sockaddr)) == -1) {
			ip_address = (char*)&(binding->interface);
			printf("Error binding %hhu.%hhu.%hhu.%hhu:%d\n", ip_address[0], ip_address[1], ip_address[2], ip_address[3], binding->port);
			return -1;
		}
		binding = binding->next;
	}

	return 0;
}

/* Accept or deny an incoming connection.
 */
int accept_connection(t_binding *binding, t_config *config) {
	unsigned int       size;
	bool               kick_client;
	t_session          *session;
	struct sockaddr_in caddr;
	pthread_attr_t     child_attr;
	pthread_t          child_thread;
	int                total_conns, one;

	if ((session = (t_session*)malloc(sizeof(t_session))) == NULL) {
		return -1;
	}
	session->config = config;
	session->binding = binding;
	init_session(session);

	memset((void*)&caddr, 0, sizeof(struct sockaddr_in));
	size = sizeof(struct sockaddr_in);
	if ((session->client_socket = accept(binding->socket, (struct sockaddr*)&caddr, &size)) == -1) {
		free(session);
		if (errno != EINTR) {
			return -1;
		}
		return 0;
	}

	kick_client = true;
	memcpy((char*)&(session->ip_address), (char*)&caddr.sin_addr.s_addr, 4);
	if ((total_conns = connection_allowed(session->ip_address, config->connections_per_ip, config->total_connections)) >= 0) {
		if (total_conns < (config->total_connections >> 2)) {
			one = 1;
			if (setsockopt(session->client_socket, IPPROTO_TCP, TCP_NODELAY, (void*)&one, sizeof(int)) == -1) {
				close(session->client_socket);
				free(session);
				return -1;
			}
		}

		if (add_client(session) == 0) {
			pthread_attr_init(&child_attr);
			pthread_attr_setdetachstate(&child_attr, PTHREAD_CREATE_DETACHED);
			if (pthread_create(&child_thread, &child_attr, (void*)connection_handler, (void*)session) == 0) {
				kick_client = false;
			} else {
				remove_client(session, false);
			}
		}
	} else switch (total_conns) {
		case ca_TOOMUCH_PERIP:
			if ((config->ban_on_max_per_ip > 0) && ip_allowed(session->ip_address, session->config->banlist_mask)) {
				ban_ip(session->ip_address, config->ban_on_max_per_ip, config->kick_on_ban);
			}
			log_warning(config->system_logfile, "Maximum number of connections for IP address reached", session->ip_address);
			break;
		case ca_TOOMUCH_TOTAL:
			log_warning(config->system_logfile, "Maximum number of total connections reached", session->ip_address);
			break;
		case ca_BANNED:
#ifdef DEBUG
			log_warning(config->system_logfile, "Client kicked because of ban", session->ip_address);
#endif
			if (config->reban_during_ban && ip_allowed(session->ip_address, session->config->banlist_mask)) {
				reban_ip(session->ip_address);
			}
			break;
	}

	if (kick_client) {
		close(session->client_socket);
		free(session);
	}

	return 0;
}

#ifndef HAVE_CLEARENV
void clearenv(void) {
	extern char **environ;
	char *eq, key[256];
	int len;

	while (*environ != NULL) {
		if ((eq = strchr(*environ, '=')) != NULL) {
			if ((len = eq - *environ) < 256) {
				memcpy(key, *environ, len);
				key[len] = '\0';
				unsetenv(key);
			}
		}
		environ++;
	}
}
#endif

/* Run the Hiawatha webserver.
 */
int run_server(char *configdir, bool daemon) {
	int                quit = 0, result, highest_fd;
#ifdef HAVE_COMMAND
	t_admin            *admin;
	struct sockaddr_in caddr;
	int                size, admin_socket;
#endif
	pid_t              pid;
	t_binding          *binding;
	FILE               *fp;
	t_config           *config;
    fd_set             read_fds;
	struct timeval     select_timeout;

	signal(SIGPIPE, SIG_IGN);
	signal(SIGCHLD, SIG_IGN);

	config = default_config();
	chdir(configdir);
	if ((result = read_main_configfile(main_configfile, config)) == 0) {
		result = check_configuration(config);
	}
	if (result != 0) {
		print_configerror(main_configfile, result);
		return -1;
	}
	if (read_mimetypes(config->mimetype_config, &(config->mimetype)) == -1) {
		printf("Error while reading mimetype configuration.\n");
		return -1;
	}
	if (read_throttleconfig(config->throttle_config, &(config->throttle)) == -1) {
		//printf("Error while reading throttle configuration.\n");
		//return -1
	}

	/* Bind Serverport
	 */
	if (bind_socket(config->binding) == -1) {
		return -1;
	}

#ifdef HAVE_SSL
	sslInit();
	binding = config->binding;
	while (binding != NULL) {
		if (binding->use_ssl) {
			if ((binding->ssl_context = ssl_binding(binding->server_key, "")) == NULL) {
				printf("Error while reading keyfile %s!\n", binding->server_key);
				return -1;
			}
		}
		binding = binding->next;
	}
#endif

#ifdef HAVE_COMMAND
	if (bind_socket(config->command_port) == -1) {
		return -1;
	}
#endif

#ifdef HAVE_PLUGIN
	switch (plugin_init()) {
		case 0:
			if (config->plugin_uri == NULL) {
				config->plugin_uri = get_plugin_uri();
			}
			break;
		default:
			printf("Error while initializing plugin.\n");
			return -1;
	}
#endif

	touch_logfiles(config, config->server_root);
	tzset();
	clearenv();

	/* Become a daemon
	 */
	if (daemon) {
		switch (pid = fork()) {
			case -1:
				perror("fork()");
				return -1;
			case 0:
				if (setsid() == -1) {
					perror("setsid()");
					return -1;
				}
				break;
			default:
				if ((fp = fopen(config->pidfile, "w")) != NULL) {
					fprintf(fp, "%d\n", pid);
					fclose(fp);
				} else {
					printf("Warning: can't write PID file %s.\n", config->pidfile);
				}
				return 0;
		}
	}

	/* Security
	 */
	if (config->server_root != NULL) {
		do {
			if (chdir(config->server_root) != -1) {
				if (chroot(config->server_root) != -1) {
					break;
				}
			}
			printf("\nError while changing root to %s!\n", config->server_root);
			return -1;
		} while (false);
	} else {
		chdir("/");
	}

	if (getuid() == 0) do {
		if (setgroups(config->groups.number, config->groups.array) != -1) {
			if (setgid(config->server_gid) != -1) {
				if (setuid(config->server_uid) != -1) {
					break;
				}
			}
		}
		printf("\nError while changing uid/gid!\n");
		return -1;
	} while (false);

	signal(SIGTERM, TERM_handler);
	signal(SIGUSR1, USR1_handler);
	signal(SIGUSR2, USR2_handler);

	binding = config->binding;
	while (binding != NULL) {
		if (listen(binding->socket, 16) == -1) {
			perror("listen()");
			return -1;
		}
		binding = binding->next;
	}
#ifdef HAVE_COMMAND
	binding = config->command_port;
	while (binding != NULL) {
		if (listen(binding->socket, 1) == -1) {
			perror("listen()");
			return -1;
		}
		binding = binding->next;
	}
#endif

	select_timeout.tv_sec = MAIN_LOOP_TIMER;
	select_timeout.tv_usec = 0;

	init_log_module();
	init_client_module();
	init_load_balancer(config->fcgi_server);
#ifdef HAVE_CACHE
	init_cache_module();
#endif
#ifdef HAVE_COMMAND
	init_command_module();
#endif

	log_string(config->system_logfile, "Hiawatha v"VERSION" started");

	do {
#ifdef HAVE_COMMAND
		highest_fd = fill_read_fds(&read_fds, config->binding, config->command_port);
#else
		highest_fd = fill_read_fds(&read_fds, config->binding);
#endif
		switch (select(highest_fd + 1, &read_fds, NULL, NULL, &select_timeout)) {
			case -1:
				if (errno != EINTR) {
#ifdef DEBUG
					log_error(config->system_logfile, errno);
#endif
					log_string(config->system_logfile, "Fatal error selecting connection");
					usleep(1000);
				}
				break;
			case 0:
				select_timeout.tv_sec = MAIN_LOOP_TIMER;
				select_timeout.tv_usec = 0;

				check_banlist(config);
				if (config->reconnect_delay > 0) {
					check_delayed_remove_timers();
				}
				check_load_balancer(config);
#ifdef HAVE_CACHE
				check_cache();
#endif
#ifdef HAVE_COMMAND
				check_adminlist();
#endif
#ifdef HAVE_PLUGIN
				plugin_timer();
#endif
				switch (received_signal) {
					case rs_NONE:
						break;
					case rs_QUIT_SERVER:
#ifdef DEBUG
						log_string(config->system_logfile, "Received TERM signal");
#endif
						quit = 1;
						break;
					case rs_UNBAN_CLIENTS:
						unban_ip(0);
						received_signal = rs_NONE;
						break;
					case rs_TOUCH_LOGFILES:
						touch_logfiles(config, NULL);
						received_signal = rs_NONE;
						break;
				}
				break;
			default:
#ifdef HAVE_COMMAND
				// Connected admins
				admin = first_admin();
				while (admin != NULL) {
					if (FD_ISSET(admin->socket, &read_fds)) {
						switch (handle_admin(admin, config)) {
							case cc_DISCONNECT:
								remove_admin(admin->socket);
								break;
							case cc_SHUTDOWN:
#ifdef DEBUG
								log_string(config->system_logfile, "Shutdown by admin");
#endif
								quit = 1;
								break;
							case cc_UNBAN_CLIENTS:
								received_signal = rs_UNBAN_CLIENTS;
								break;
						}
					}
					admin = next_admin();
				}
#endif

				// HTTP(S) port
				binding = config->binding;
				while (binding != NULL) {
					if (FD_ISSET(binding->socket, &read_fds)) {
						if (accept_connection(binding, config) != 0) {
							log_string(config->system_logfile, "Fatal error accepting connection");
							usleep(1000);
							break;

						}
					}
					binding = binding->next;
				}

#ifdef HAVE_COMMAND
				// Command port
				binding = config->command_port;
				while (binding != NULL) {
					if (FD_ISSET(binding->socket, &read_fds)) {
						size = sizeof(struct sockaddr_in);
						if ((admin_socket = accept(binding->socket, (struct sockaddr*)&caddr, &size)) == -1) {
							if (errno != EINTR) {
								log_string(config->system_logfile, "Fatal error accepting CommandChannel connection");
								usleep(1000);
								break;
							}
						} else {
							if (add_admin(admin_socket) == -1) {
								close(admin_socket);
							}
						}
					}
					binding = binding->next;
				}
#endif
		}
	} while (quit == 0);

	disconnect_clients(config);
#ifdef HAVE_COMMAND
	disconnect_admins();
#endif

	binding = config->binding;
	while (binding != NULL) {
		close(binding->socket);
#ifdef HAVE_SSL
		if (binding->use_ssl) {
			ssl_cleanup(binding->ssl_context);
		}
#endif
		binding = binding->next;
	}

#ifdef HAVE_COMMAND
	binding = config->command_port;
	while (binding != NULL) {
		close(binding->socket);
		binding = binding->next;
	}
#endif

	signal(SIGTERM, SIG_DFL);
	signal(SIGUSR1, SIG_DFL);
	signal(SIGUSR2, SIG_DFL);

	log_string(config->system_logfile, "Hiawatha v"VERSION" stopped");

#ifdef HAVE_PLUGIN
	plugin_destroy();
#endif

	return 0;
}

void show_help(char *hiabin) {
	printf("Usage %s: [-c path] [-d] [-h]\n", hiabin);
	printf("\t-c path: path to where the configrationfiles are located.\n");
	printf("\t-d: don't fork to the background.\n");
	printf("\t-h: show this information.\n");
}

/* Main and stuff...
 */
int main(int argc, char *argv[]) {
	int i = 0;
	char *configdir = default_configdir;
	bool daemon = true;

	while (++i < argc) {
		if (strcmp(argv[i], "-c") == 0) {
			if (++i < argc) {
				configdir = argv[i];
			}
		} else if (strcmp(argv[i], "-d") == 0) {
			daemon = false;
		} else if (strcmp(argv[i], "-h") == 0) {
			show_help(argv[0]);
			return 0;
		} else {
			printf("Unknown option\n");
			return -1;
		}
	}

	return run_server(configdir, daemon);
}
