#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdbool.h>
#include <string.h>
#include <errno.h>
#include <pwd.h>
#include <grp.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <sys/types.h>
#include "libstr.h"
#include "libfs.h"
#include "userconfig.h"
#ifndef HAVE_SETENV
#include "envir.h"
#endif

typedef struct type_wrap {
	char cgiroot[257];
	char chroot[257];
	uid_t uid;
	gid_t gid;
	t_groups groups;
} t_wrap;

int cgi_pid;

void ALRM_handler() {
	kill(cgi_pid, 9);
}

void print_code(int code) {
	switch (code) {
		case -1:
			printf("This program can only be executed by the Hiawatha webserver.\n");
			break;
		default:
			printf("Status: %d\r\n\r\n", code);
	}

	exit(-1);
}

bool get_wrap_data(char *wrapid, char *handler, char *cgi, t_wrap *wrap_data) {
	FILE *fp;
	char line[257], *item, *rest, *pipe;
	struct passwd *pwd;
	bool wrap_oke = false, handler_oke;
	int result, len;

	handler_oke = (handler == NULL);

	wrap_data->chroot[0] = '\0';

	/* WrapID is local userid?
	 */
	if (*wrapid == '~') {
		if ((pwd = getpwnam(wrapid + 1)) == NULL) {
			return false;
		}

		wrap_data->cgiroot[256] = '\0';
		if ((wrap_data->uid = pwd->pw_uid) == 0) {
			return false;
		}
		strncpy(wrap_data->cgiroot, pwd->pw_dir, 256);
		strncat(wrap_data->cgiroot, "/public_html/", 256);
		if (strncmp(wrap_data->cgiroot, cgi, strlen(wrap_data->cgiroot)) != 0) {
			return false;
		}
		if ((wrap_data->gid = lookup_group_ids(wrap_data->uid, &(wrap_data->groups))) <= 0) {
			return false;
		}

		if (handler_oke) {
			return true;
		}

		wrap_oke = true;
	}
	
	/* Read CGI wrapper configuration
	 */
	if ((fp = fopen(CONFIG_DIR"/cgi_wrapper.conf", "r")) != NULL) {
		line[256] = '\0';

		while (fgets(line, 256, fp) != NULL) {
			if ((line[0] == '#') || (line[0] == '\n')) {
				continue;
			}
			if (split_string(line, &item, &rest, '=') == 0) {
				strlower(item);
				if (strcmp(item, "cgihandler") == 0) {
					/* CGI handler
					 */
					if (handler_oke) {
						continue;
					}
					do {
						split_string(rest, &item, &rest, ',');
						if (strcmp(handler, item) == 0) {
							handler_oke = true;
							break;
						}
					} while (rest != NULL);
				} else if (strcmp(item, "wrap") == 0) {
					/* Wrap entry
					 */
					if (wrap_oke) {
						continue;
					}

					/* Wrap Id
					 */
					if (split_string(rest, &item, &rest, ':') == -1) {
						break;
					}
					if (strcmp(item, wrapid) != 0) {
						continue;
					}

					/* Homedirectory
					 */
					if (split_string(rest, &item, &rest, ':') == -1) {
						break;
					}
					wrap_data->cgiroot[256] = '\0';
					if (*item == '/') {
						/* chroot directory
						 */
						if ((pipe = strchr(item, '|')) != NULL) {
							wrap_data->chroot[256] = '\0';
							*pipe = '\0';
							strncpy(wrap_data->chroot, item, 256);
							*pipe = '/';
						}

						len = strlen(item);
						if (strncmp(item, cgi, len) != 0) {
							break;
						}
						if ((*(item + len - 1) != '/') && (*(cgi + len) != '/')) {
							break;
						}

						if (pipe != NULL) {
							cgi += (pipe - item);
							item = pipe;
						}

						strncpy(wrap_data->cgiroot, item, 256);
					} else if (*item == '~') {
						if ((pwd = getpwnam(item + 1)) == NULL) {
							break;
						}
						strncpy(wrap_data->cgiroot, pwd->pw_dir, 256);
						strncat(wrap_data->cgiroot, "/public_html/", 256);
						if (strncmp(wrap_data->cgiroot, cgi, strlen(wrap_data->cgiroot)) != 0) {
							break;
						}
					} else {
						break;
					}

					/* User Id
					 */
					result = split_string(rest, &item, &rest, ':');
					if ((wrap_data->uid = parse_userid(item)) <= 0) {
						break;
					}

					/* Group id
					 */
					if (result == -1) {
						if ((wrap_data->gid = lookup_group_ids(wrap_data->uid, &(wrap_data->groups))) <= 0) {
							break;
						}
					} else {
						if ((wrap_data->gid = parse_groups(rest, &(wrap_data->groups))) <= 0) {
							break;
						}
					}

					wrap_oke = true;
				} else {
					/* Crap in configurationfile
					 */
					break;
				}
				if (wrap_oke && handler_oke) {
					break;
				}
			} else {
				break;
			}
		}
		fclose(fp);

		return (wrap_oke && handler_oke);
	} else {
		return false;
	}
}

int main(int argc, char *argv[]) {
	char buffer[8], *wrapid, *var;
	FILE *fp;
	int i, handle, arg_offset, time_for_cgi, len;
	t_wrap wrap_data;
	bool follow_symlinks = true, usecgi_handler;

	char *cgiwrap_id   = "CGIWRAP_ID";
	char *cgiwrap_sln  = "CGIWRAP_FOLLOWSYMLINKS";
	char *cgiwrap_time = "CGIWRAP_TIMEFORCGI";

	if (argc < 3) {
		print_code(-1);
	}

	/* Check if parent is Hiawatha
	 */
	buffer[0] = buffer[7] = '\0';
	if ((fp = fopen(PIDFILE_DIR"/hiawatha.pid", "r")) == NULL) {
		print_code(-1);
	}
	if (fgets(buffer, 7, fp) == NULL) {
		print_code(-1);
	}
	fclose(fp);

	if ((i = strlen(buffer)) == 0) {
		print_code(-1);
	} else if (buffer[i - 1] != '\n') {
		print_code(-1);
	}
	buffer[i - 1] = '\0';

	if ((i = str2int(buffer)) == -1) {
		print_code(-1);
	}
	if (getsid(0) != getsid(i)) {
		print_code(-1);
	}

	/* Read environment settings
	 */
	if ((wrapid = getenv(cgiwrap_id)) == NULL) {
		print_code(500);
	}

	if ((var = getenv(cgiwrap_sln)) == NULL) {
		print_code(500);
	}
	follow_symlinks = (strcmp(var, "true") == 0);

	if ((var = getenv(cgiwrap_time)) != NULL) {
		time_for_cgi = str2int(var);
	} else {
		time_for_cgi = 5;
	}

	/* Clear environment crap
	 */
	unsetenv(cgiwrap_id);
	unsetenv(cgiwrap_sln);
	unsetenv(cgiwrap_time);

	/* Check for bad path
	 */
	if (strstr(argv[1], "/../") != NULL) {
		print_code(500);
	}

	/* Read cgi_wrapper config
	 */
	if ((usecgi_handler = (strcmp(argv[0], "-") != 0))) {
		if (get_wrap_data(wrapid, argv[0], argv[2], &wrap_data) == false) {
			print_code(403);
		}
	} else {
		if (get_wrap_data(wrapid, NULL, argv[2], &wrap_data) == false) {
			print_code(403);
		}
	}

	if (wrap_data.chroot[0] != '\0') {
		if (chdir(wrap_data.chroot) == -1) {
			print_code(403);
		} else if (chroot(wrap_data.chroot) == -1) {
			print_code(500);
		}
		len = strlen(wrap_data.chroot);
		if (usecgi_handler == false) {
			*(argv + 1) += len;
		}
		*(argv + 2) += len;

		setenv("DOCUMENT_ROOT", wrap_data.cgiroot, 1);
		setenv("SCRIPT_FILENAME", argv[2], 1);
	}

	/* Set IDs
	 */
	if (setuid(0) == -1) {
		print_code(500);
	} else if (setgroups(wrap_data.groups.number, wrap_data.groups.array) == -1) {
		print_code(500);
	} else if (setgid(wrap_data.gid) == -1) {
		print_code(500);
	} else if (setuid(wrap_data.uid) == -1) {
		print_code(500);
	}

	/* New session
	 */
	if (setsid() == -1) {
		print_code(500);
	}

	/* Does the CGI program exist?
	 */
	if ((handle = open(argv[2], O_RDONLY)) == -1) {
		if (errno == EACCES) {
			print_code(403);
		}
		print_code(404);
	} else {
		close(handle);
	}

	/* Symlink allowed?
	 */
	if (follow_symlinks == false) {
		switch (contains_not_allowed_symlink(argv[1], wrap_data.cgiroot)) {
			case error:
				print_code(500);
			case not_found:
				print_code(404);
			case no_access:
			case yes:
				print_code(403);
			case no:
				break;
		}
	}

	/* Check file accessrights
	 */
	if (usecgi_handler == false) {
		switch (can_execute(argv[1], wrap_data.uid, wrap_data.gid, &(wrap_data.groups))) {
			case error:
				print_code(500);
			case not_found:
				print_code(404);
			case no_access:
			case no:
				print_code(403);
			case yes:
				break;
		}

		arg_offset = 1;
	} else {
		arg_offset = 0;
	}

	/* And here we go
	 */
	switch (cgi_pid = fork()) {
		case -1:
			print_code(500);
		case 0:
			execvp(argv[arg_offset], argv + 1 + arg_offset);
			print_code(500);
			exit(-1);
		default:
			signal(SIGALRM, ALRM_handler);
			alarm(time_for_cgi);
			waitpid(cgi_pid, NULL, 0);
	}

	return 0;
}
