/****************************************************
  SixXS Heartbeat Abstracted Multiplatform Functions
  by Jeroen Massar <jeroen@sixxs.net>
*****************************************************

 $Author: jeroen $
 $Id: hb.c,v 1.4 2003/10/26 17:38:30 jeroen Exp $
 $Date: 2003/10/26 17:38:30 $

****************************************************/
#define _XOPEN_SOURCE
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <stdio.h>
#include <errno.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <strings.h>
#include <ctype.h>
#include <time.h>
#include <signal.h>
#include <syslog.h>
#if defined(SOL_SOCKET) && defined(SO_BINDTODEVICE)
#include <net/if.h>
#endif

#include "../common/config.h"
#include "../common/common.h"

/**************************************
  Functions
**************************************/

/* Get a socket and determine the new IP address */
int heartbeat_socket(
	int *address_changed,
	int bStaticTunnel,
	char *sIPv4Interface,
	char **sIPv4Local,
	char *sIPv4POP,
	char *sIPv4LocalResolve)
{
	int		sockfd;
	struct sockaddr	sa;
	socklen_t	socklen;
	char		local_ipv4[NI_MAXHOST];
#if defined(SOL_SOCKET) && defined(SO_BINDTODEVICE)
	struct ifreq	interface;
#endif
	struct addrinfo	hints, *res, *ressave;

	*address_changed = 0;

	// Get ourselves a nice IPv4 socket
	sockfd = socket(AF_INET, SOCK_DGRAM, 0);
	if (sockfd < 0) return -1;

#if defined(SOL_SOCKET) && defined(SO_BINDTODEVICE)
	// We don't have to bind to a device if this
	// is a static tunnel, this allows running
	// the heartbeat client as non-root.
	if (!bStaticTunnel)
	{
		// Only allowed as root, but we need root rights anyways
		// to (re)configure the tunnel

		// Bind to the underlying IPv4 device
		memset(&interface, 0, sizeof(interface));
		strncpy(interface.ifr_ifrn.ifrn_name, sIPv4Interface, IFNAMSIZ);
		if (setsockopt(sockfd, SOL_SOCKET, SO_BINDTODEVICE,
			(char *)&interface, sizeof(interface)) == -1)
		{
			hblog(LOG_ERR, "Couldn't bind to device \"%s\"\n", sIPv4Interface);
			close(sockfd);

			// We return a -1, thus the app will keep beating
			return -1;
		}
	}
#endif

	// connect to the remote port
	// this causes us to be able to use normal write()
	// and also gives us the ability to find out the local IP
	memset(&hints, 0, sizeof(struct addrinfo));
	hints.ai_family = AF_INET;
	hints.ai_socktype = SOCK_DGRAM;

	// Get the POP IPv4 into a sockaddr
	if (getaddrinfo(sIPv4POP, PORT, &hints, &res) < 0)
	{
		hblog(LOG_ERR, "Couldn't resolve POP ip %s\n", sIPv4POP);
		close(sockfd);

		// We return a -1, thus the app will keep beating
		return -1;
	}

	ressave = res;

	while (res)
	{
		if (connect(sockfd, res->ai_addr, res->ai_addrlen) == 0) break;
		res = res->ai_next;
	}
	freeaddrinfo(ressave);
	if (res == NULL)
	{
		hblog(LOG_ERR, "Failed to connect() to remote side\n");
		close(sockfd);
		// We return a -1, thus the app will keep beating
		return -1;
	}

	// Normal operation, find out our local IPv4 address
	if (sIPv4LocalResolve == NULL)
	{
		// Figure out our local IP
		socklen = sizeof(sa);
		if (getsockname(sockfd, &sa, &socklen) == -1)
		{
			hblog(LOG_WARNING, "Couldn't get local socketaddress\n");
			close(sockfd);
			return -1;
		}

		if (getnameinfo((struct sockaddr *)&sa, sizeof(sa),
				local_ipv4, sizeof(local_ipv4),
			NULL, 0,
			NI_NUMERICHOST) != 0)
		{
			hblog(LOG_WARNING, "Couldn't get local IP\n");
			close(sockfd);
			return -1;
		}
	}
	else
	{
		// this causes us to be able to use normal write()
		// and also gives us the ability to find out the local IP
		memset(&hints, 0, sizeof(struct addrinfo));
		hints.ai_family = AF_INET;
		hints.ai_socktype = SOCK_DGRAM;
		
		// Get the POP IPv4 into a sockaddr
		if (getaddrinfo(sIPv4LocalResolve, NULL, &hints, &res) < 0)
		{
			hblog(LOG_ERR, "Couldn't resolve POP ip %s\n", sIPv4POP);
			// We return a -1, thus the app will keep beating
			return -1;
		}
		ressave = res;
		while (res)
		{
			if (getnameinfo(res->ai_addr, res->ai_addrlen,
				local_ipv4, sizeof(local_ipv4),
				NULL, 0,
				NI_NUMERICHOST) == 0)
			{
				break;
			}
			hblog(LOG_WARNING, "Couldn't get local IP\n");
			res = res->ai_next;
		}
		freeaddrinfo(ressave);
	}

	// Did the IPv4 address change?
	if (*sIPv4Local == NULL ||
		strcmp(*sIPv4Local, local_ipv4) != 0)
	{
		if (*sIPv4Local) free(*sIPv4Local);
		*sIPv4Local = strdup(local_ipv4);

		if (!bStaticTunnel)
		{
			// Run a script to change the address
			*address_changed = 1;
		}
	}

	// Return it
	return sockfd;
}

/* Send a heartbeat */
int sendhb(int sockfd, char *sIPv4Local, char *sIPv6Local, char *sPassword, int bBehindNAT)
{
	struct MD5Context	md5;
	unsigned char		*p, our_digest[20], *pn = our_digest, buf[1000];
	time_t			time_tee;
	struct tm		tm;
	int			i;

	time_tee = time(NULL);

	// Create the string to send including our password
	snprintf(buf, sizeof(buf), "HEARTBEAT TUNNEL %s %s %ld %s",
		sIPv6Local,
		(bBehindNAT == 1 ? "sender" : sIPv4Local),
		time_tee, sPassword);

	// Generate a MD5
	MD5Init(&md5);
	MD5Update(&md5, buf, strlen(buf));
	MD5Final(our_digest, &md5);

	// Overwrite it without password
	p = buf;
	p += snprintf(buf, sizeof(buf)-17, "HEARTBEAT TUNNEL %s %s %ld ",
		sIPv6Local,
		(bBehindNAT == 1 ? "sender" : sIPv4Local),
		time_tee);

	// append the digest
	for (i = 0; i < 16; i++)
	{
		snprintf(p, 3, "%02x", *pn++);
		p+=2;
	}
	*p = '\0';

	// Send the heartbeat
	write(sockfd, buf, strlen(buf));

	hblog(LOG_DEBUG, "[HB] %s\n", buf);

	return 0;
}

