/*
 *  Copyright (C) 2007  Anders Gavare.  All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions are met:
 *
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright  
 *     notice, this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 *  3. The name of the author may not be used to endorse or promote products
 *     derived from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 *  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 *  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE   
 *  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 *  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 *  OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 *  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 *  LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 *  SUCH DAMAGE.
 *
 *
 *  $Id: native_x86.c,v 1.6 2007/04/10 15:37:00 debug Exp $
 *
 *  Native code generation backend for AMD64 and i386.
 *
 *  NOTE/TODO: This is ONLY for amd64 for now, not i386!
 *
 *  NOTE/TODO: This is just a dummy so far, there is no working native
 *             code generation for GXemul's new dyntrans system.
 *
 *  Register usage:
 *
 *	rdi: Function call arg 1, always the struct cpu *.
 *	rsi: Function call arg 2, struct ic * at entry to the code block.
 *	Note: rdi and rsi need to be saved, if a function call is made.
 *	(They are not preserved.)
 *
 *	rax, rbx, rcx, rdx, r08, r09, r10, r11: Temporaries
 *		(NOTE: rbx must be saved at function entry and
 *		restored at exit. The others don't need to be saved.)
 *
 *	r12-r15: Saved temporaries. These may be used, if their original
 *		contents is saved at entry and restored at exit.
 */

/*  NOTE: This file is included from native.c.  */


#include "native_x86.h"

/*  static char *x86_reg_names[] = X86_REG_NAMES;  */


static int x86_round_robin_temp_reg(void)
{
	/*  TODO: Which registers on amd64 are temp registers?  */
	static int temp_reg = 0;
	return (temp_reg++) % 2;
}


/*  Allocate a new native_op, and add it after the previous one.  */
static void new_op(struct native_op **native_ops_p, struct native_op **first_op)
{
	struct native_op *op = malloc(sizeof(struct native_op));
	if (op == NULL) {
		fprintf(stderr, "fatal error, out of memory\n");
		exit(1);
	}

	memset(op, 0, sizeof(struct native_op));

	if (*native_ops_p != NULL) {
		(*native_ops_p)->next = op;
		op->prev = (*native_ops_p);
	}

	*native_ops_p = op;

	if (*first_op == NULL)
		*first_op = op;
}


void dump_op_list(struct native_op *list);


int native_inr_to_native_ops(struct inr *inr,
	struct native_op **native_ops_p)
{
	struct native_op *first_op = NULL;
	struct native_op *ops = NULL;
	int i, nr_of_inr_entries = inr->nr_inr_entries_used;

	*native_ops_p = NULL;

	for (i = 0; i < nr_of_inr_entries; ++i) {
		/*  Get the INR opcode:  */
		struct inr_entry *inr_entry = &inr->inr_entries[i];
		int inr_opcode = inr_entry->opcode;
		int r1;

		switch (inr_opcode) {

		case INR_OPCODE_NOP:
			break;

		case INR_OPCODE_OR_DCR32_SCR32_IS16:
			new_op(&ops, &first_op);
			r1 = x86_round_robin_temp_reg();
			ops->opcode = NATIVE_X86_OPCODE_LOAD_CR64_R64;
			ops->arg1 = inr_entry->arg2;
			ops->arg2 = r1;
			new_op(&ops, &first_op);
			ops->opcode = NATIVE_X86_OPCODE_OR_R64_I32;
			ops->arg1 = r1;
			ops->arg2 = inr_entry->arg3;
			new_op(&ops, &first_op);
			ops->opcode = NATIVE_X86_OPCODE_STORE_CR64_R64;
			ops->arg1 = inr_entry->arg1;
			ops->arg2 = r1;
			break;

		default:
			/*  TODO: Break out and return partial failure/
			    success in a better way  */
			fprintf(stderr, "Internal error in native_x86.c:"
			    " INR opcode %i not yet implemented.\n",
			    inr_opcode);
			exit(1);
		}
	}

	*native_ops_p = first_op;

	return 0;
}


#ifdef TEST_NATIVE_X86

void dump_op_list(struct native_op *list)
{
	int n = 0;
	printf("OPs:\n");
	while (list != NULL) {
		printf("nr %i: opcode = %i (0x%"PRIx64",0x%"PRIx64
		    ",0x%"PRIx64")\n",
		    n++, list->opcode, list->arg1, list->arg2, list->arg3);

		list = list->next;
	}
	printf("-----\n");
}

void dump_inr_array(struct inr *inr)
{
	int i;
	printf("INR:\n");
	for (i=0; i<inr->nr_inr_entries_used; ++i) {
		struct inr_entry *e = &inr->inr_entries[i];
		printf("nr %i: opcode = %i (0x%"PRIx64",0x%"PRIx64
		    ",0x%"PRIx64")\n",
		    i, e->opcode, e->arg1, e->arg2, e->arg3);
	}
	printf("-----\n");
}

#define	N 3

void generate_inr_array_contents(struct inr *inr, int n)
{
	int i;

	memset(inr, 0, sizeof(inr));
	inr->inr_entries = malloc(sizeof(struct inr_entry) *
	    INR_MAX_ENTRIES);

	inr->nr_inr_entries_used = n;

	for (i=0; i<n; i++) {
		struct inr_entry inr_entry;
		memset(&inr_entry, 0, sizeof(inr_entry));

		inr_entry.opcode = INR_OPCODE_OR_DCR32_SCR32_IS16;
		inr_entry.arg1 = 128;
		inr_entry.arg2 = 160;
		inr_entry.arg3 = 0x1234;

		inr->inr_entries[i] = inr_entry;
	}
}

void test_native_x86(void)
{
	struct inr inr;
	struct native_op *op_list;

	printf("TESTING x86/amd64 code generation\n");

	generate_inr_array_contents(&inr, N);

	dump_inr_array(&inr);

	native_inr_to_native_ops(&inr, &op_list);
	dump_op_list(op_list);

	printf("DONE\n");

	exit(1);
}

#endif	/*  TEST_NATIVE_X86  */
