/***************************************************************************
                          application.cpp  -  description
                             -------------------
    begin                : Mon Jan 7 2002
    copyright            : (C) 1999-2002 by Jacques Fortier & Brian Ashe
    email                : gtkpool@seul.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <vector>
#include <algorithm>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <gtk/gtk.h>
#include "application.h"
#include "sound.h"
#include "connectdialog.h"
#include "check_table_collision.h"
#include "move_balls.h"
#include "apply_friction.h"
#include "draw_ball.h"
#include "pointer_selects.h"
#include "check_pocket.h"
#include "vec2d.h"
#include "point2d.h"
#include "ball.h"
#include "indentify_ball.h"

static GList *pixmaps_directories = NULL;

Application::Application(){
		gap = 29;
		width = 722 + gap * 2; height = 362 + gap * 2;
		table.width = width - (gap * 2);
		table.height = height - (gap * 2);
		table.x = 0 + gap;
		table.y = 0 + gap;

		FRICTION = .06;
		BUMPER_DRAG = .8;
		COLLIDE_DRAG = .95;

		selected = NULL;

		black  = new GdkColor;
		red = new GdkColor;
		table_color = new GdkColor;
		message_colours[1] = new GdkColor;
		message_colours[2] = new GdkColor;
		message_colours[3] = new GdkColor;
		message_colours[4] = new GdkColor;
		message_colours[14] = new GdkColor;

		buf_pixmap = NULL;
		table_pixmaps = new (GdkPixmap *)[4];
		balls_pixmaps = new (GdkPixmap *)[NUM_BALLS];
		balls_pixmap_masks = new (GdkBitmap *)[NUM_BALLS];
		balls_big_pixmaps = new (GdkPixmap *)[NUM_BALLS];
		balls_big_pixmap_masks = new (GdkBitmap *)[NUM_BALLS];
		running = true; placing_cue = false;
		sunk_tf = collide_tf = bounce_tf = false;
		connected = false;
		connect_dialog = NULL;
		game_name = eight_ball;
		table_choice = TABLE_MA;
}

Application::~Application(){
   	delete black;
   	delete red;
   	delete table_color;
   	delete[] table_pixmaps;
   	delete[] balls_pixmaps;
   	delete[] balls_pixmap_masks;
   	delete[] balls_big_pixmaps;
   	delete[] balls_big_pixmap_masks;
}

void Application::init()
{
	visual = gdk_visual_get_system();
	colormap = gdk_colormap_get_system();

	load_ball_xpms();
	if(sound)
		init_sound();

	gdk_color_parse("ForestGreen", table_color);
	gdk_color_parse("Red", message_colours[1]);
	gdk_color_parse("Green", message_colours[2]);
	gdk_color_parse("Blue", message_colours[3]);
	gdk_color_parse("Magenta", message_colours[4]);
	gdk_color_parse("Gray", message_colours[14]);
	gdk_colormap_alloc_color(colormap, table_color, FALSE, TRUE);
	gdk_colormap_alloc_color(colormap, message_colours[1], FALSE, TRUE);
	gdk_colormap_alloc_color(colormap, message_colours[2], FALSE, TRUE);
	gdk_colormap_alloc_color(colormap, message_colours[3], FALSE, TRUE);
	gdk_colormap_alloc_color(colormap, message_colours[4], FALSE, TRUE);
	gdk_colormap_alloc_color(colormap, message_colours[14], FALSE, TRUE);
	gdk_color_black(colormap, black);
	gdk_color_parse("Red", red);
	gdk_colormap_alloc_color(colormap, red, FALSE, TRUE);
	message_colours[0] = black;

	// setup adjustments -- these are used in options dialog
	// to control physics
	friction_adjustment = gtk_adjustment_new(0.06, 0, 1, 0.01, 0.1, 0);
	collide_drag_adjustment = gtk_adjustment_new(0.95, 0, 1, 0.01, 0.05, 0);
	bumper_drag_adjustment = gtk_adjustment_new(0.8, 0.0, 1.5, 0.01, 0.1, 0);

	resetTable();
}

void Application::destroy()
{
	if(buf_pixmap)
		gdk_pixmap_unref(buf_pixmap);
	for(int i = 0; i < NUM_BALLS; i++)
	{
		gdk_pixmap_unref(balls_pixmaps[i]);
		gdk_bitmap_unref(balls_pixmap_masks[i]);
	}

	gdk_colormap_free_colors(colormap, table_color, 1);

	gdk_colormap_free_colors(colormap, message_colours[1], 1);
	gdk_colormap_free_colors(colormap, message_colours[2], 1);
	gdk_colormap_free_colors(colormap, message_colours[3], 1);
	gdk_colormap_free_colors(colormap, message_colours[4], 1);
	gdk_colormap_free_colors(colormap, message_colours[14], 1);
}

void Application::updateBalls () {
	/* Every time this function is called, it moves time forward by one
animation frame.  This means that every ball will be moved by its
vel.dx and vel.dy values by the time the function returns.  However, to
process collisions the frame is broken up.  First the balls are
analyzed to see if any collisions will happen in this tick.  The
first collision is selected, and its time noted.  All the balls are
advanced to that time, and the collision is processed.  This is
reiterated until there are no more collisions left in the frame
*/
	double	md, ei, time;
	int cnt = balls.size();
	sunk_tf = collide_tf = bounce_tf = false;
	md = ei = time = 1;
	do {
		Ball *cb1 = NULL, *cb2 = NULL, *bnc = NULL;
		check_table_collision c(ei, time, bnc, table);
		// Check for collisions between balls and the edge of the table
		check_table_collision c2 = for_each(balls.begin(), balls.end(), c);
		bnc = c2.bnc;
		// Check for ball to ball collisions
		for (int ii = 1; ii < cnt; ii++) {
			for (int jj = 0; jj < ii; jj++) {
				double coll_time = balls[ii].pathIntercept(balls[jj]);
				if (coll_time < md  && coll_time >= 0  &&  coll_time < time) {
					md = coll_time;
					cb1 = &balls[ii];
					cb2 = &balls[jj];
				}
			}
		}
		// Determine closest event
		double dt = md < time ? md : time;
		double dt2 = dt < ei ? dt : ei;
		// Move all ball's forward to this point in time
		for_each(balls.begin(), balls.end(), move_balls(dt2));
		// Process any collisions
		if (ei <= dt &&  bnc != NULL) // if this event is a bumper collision
		{
			check_pocket cp(table, ei);
			if(cp(*bnc))           // if the ball is going into a pocket
			{
				int r = BALL_SIZE / 2 + 1;
				update_dirty(int(bnc->x - r), int(bnc->x + r),
				             int(bnc->y - r), int(bnc->y + r));

				sunk_balls |= (1 << cp.ball_num);

				// Put balls in the rack when sunk
				if (cp.ball_num != CUE)
				{
					if (game_name == eight_ball)
					{
						if (cp.ball_num > 0 && cp.ball_num < 9)
							sunk_balls1.push_back(bnc->ball_num);
						if (cp.ball_num > 7 && cp.ball_num < 16)
							sunk_balls2.push_back(bnc->ball_num);
					} else
					{
						sunk_balls1.push_back(bnc->ball_num);
					}
					
					last_sunk = cp.ball_num;
				}

				update_rack();
				balls.erase(bnc); // get rid of it
				if(cp.cue)
					new_cue_ball();
				sunk_tf = true;
			}
			else             // otherwise bounce it off the bumper
			{
				bnc->bounce(ei, bumper_drag());
				bounce_tf = true;
			}
		}
		else if (md <= time  &&  cb1 != NULL)
		{
			cb1->collide(cb2, collide_drag());
			collide_tf = true;
		}
		time -= dt2; // wind the clock forward
		ei = md = time;
	} while (time > 0);
	// Apply friction deceleration to all balls
	for_each(balls.begin(), balls.end(), apply_friction(friction()));
	// redraw aimer if it selects a moving ball
	if((hit_moving && selected != NULL) &&
	   (selected->vel.dx != 0 || selected->vel.dy != 0))
	{
		update_dirty(int(shoot.x), int(selected->x), int(shoot.y), int(selected->y));
	}
	
	// All balls have been sunk (restart)
	if((balls.size() == 1 && balls.back().is_cue) || balls.size() == 0)
	{
		resetTable();
	}
}

void Application::replaceBall () {
	double ox = (table.width * 0.75) + gap;
	double oy = height * 0.5;
   	
   	// Take the last one of the right rack
   	if (last_sunk > 0 && last_sunk < 9)
   		sunk_balls1.pop_back();
   	if (last_sunk > 7 && last_sunk < 16)
   		sunk_balls2.pop_back();
   	// Put it on the table at the foul spot
   	if (last_sunk > 0 && last_sunk < 16)	
		new_ball(ox, oy, last_sunk);
	
   	// Reset so you can't put too many balls on the table
   	last_sunk = 0;
	
	udl = table.x; udr = table.x + table.width;
	udt = table.y; udb = table.y + table.height;
	// Reset the racks
	update_rack();

}

void Application::ball_in_hand()
{				
	vector<Ball>::iterator cb;
	cb = find_if(balls.begin(), balls.end(), indentify_ball());
	balls.erase(cb); // remove current cue ball
	update_dirty(table.x, table.x + table.width, table.y, table.y + table.height);
	new_cue_ball();
}

void Application::resetTable () {
	double ox = (table.width * 0.75) + gap;
	double oy = height * 0.5;
	double ofx = sqrt((BALL_SIZE * BALL_SIZE) - ((BALL_SIZE / 2) * (BALL_SIZE / 2))) + .3;
	double ofy = BALL_SIZE / 2 + .3;
	
	balls.erase(balls.begin(), balls.end());
	sunk_balls1.erase(sunk_balls1.begin(), sunk_balls1.end());
	sunk_balls2.erase(sunk_balls2.begin(), sunk_balls2.end());
	if(game_name == eight_ball)
	{
		// Standard Rack up
		new_ball((table.width / 4) + gap - 1, oy, CUE, true);
		new_ball(ox, oy, PURPLE, false, false, false, true);
		new_ball(ox + ofx, oy - ofy, RED_S, false, false, false, false, true);
		new_ball(ox + ofx, oy + ofy, BLUE, false, false, false, true);
		new_ball(ox + ofx * 2, oy - ofy * 2, GREEN_S, false, false, false, false, true);
		new_ball(ox + ofx * 2, oy, EIGHT, false, true);
		new_ball(ox + ofx * 2, oy + ofy * 2, BROWN_S, false, false, false, false, true);
		new_ball(ox + ofx * 3, oy - ofy * 3, ORANGE, false, false, false, true);
		new_ball(ox + ofx * 3, oy - ofy, YELLOW_S, false, false, false, false, true);
		new_ball(ox + ofx * 3, oy + ofy, RED, false, false, false, true);
		new_ball(ox + ofx * 3, oy + ofy * 3, BLUE_S, false, false, false, false, true);
		new_ball(ox + ofx * 4, oy - ofy * 4, GREEN, false, false, false, true);
		new_ball(ox + ofx * 4, oy - ofy * 2, PURPLE_S, false, false, false, false, true);
		new_ball(ox + ofx * 4, oy, BROWN, false, false, false, true);
		new_ball(ox + ofx * 4, oy + ofy * 2, ORANGE_S, false, false, false, false, true);
		new_ball(ox + ofx * 4, oy + ofy * 4, YELLOW, false, false, false, true);
	}
	else if(game_name == nine_ball)
	{
		// Rack up for Game of 9-ball
		new_ball((table.width / 4) + gap, oy, CUE, true);
		new_ball(ox, oy, YELLOW);
		new_ball(ox + ofx, oy - ofy, BLUE);
		new_ball(ox + ofx, oy + ofy + .02, GREEN);
		new_ball(ox + ofx * 2 - .02, oy - ofy * 2 - .01, BROWN);
		new_ball(ox + ofx * 2, oy, YELLOW_S, false, false, true);
		new_ball(ox + ofx * 2, oy + ofy * 2, RED);
		new_ball(ox + ofx * 3, oy - ofy, ORANGE);
		new_ball(ox + ofx * 3, oy + ofy, EIGHT);
		new_ball(ox + ofx * 4, oy, PURPLE);
	}
	sunk_balls = 0;
	udl = table.x; udr = table.x + table.width;
	udt = table.y; udb = table.y + table.height;
	update_rack();
	
	if(placing_cue)
	{
		// this is in case the cue ball was sunk but not placed before the
		// reset, so that there can't be more than one cue ball
		gtk_signal_handler_block(GTK_OBJECT(drawing_area), qbp_hndlr);
		gtk_signal_handler_unblock(GTK_OBJECT(drawing_area),
		                           button_press_handler_id);
		gdk_window_set_cursor(drawing_area->window, (GdkCursor *)NULL);
		placing_cue = false;
	}
}

void Application::paint (GdkDrawable *pixmap, GdkGC *gc) {

	Point2D stick_start;
	Point2D stick_end;
	Vec2D stick;
	
	// Draw Table
	gdk_draw_pixmap(pixmap, gc, table_pixmaps[table_choice], 0, 0, 0, 0, width, height);

	// Place balls on table
	for_each(balls.begin(), balls.end(), draw_ball(pixmap, gc));

	// Draw Shooting Line
	if (selected != NULL) {
		stick_start.setPos(selected->x, selected->y);
		double xdist = stick_start.x - shoot.x;
		double ydist = stick_start.y - shoot.y;
		
		stick.setVec(xdist, ydist);
		double l = stick.mag() / 200;
		stick.dx /= l; stick.dy /= l;
		stick_end.x = stick_start.x - stick.dx;
		stick_end.y = stick_start.y - stick.dy;
				
		// Draw Cue Stick Line
		gdk_gc_set_foreground(gc, black);
		gdk_gc_set_line_attributes(gc, 2, GDK_LINE_SOLID, GDK_CAP_BUTT, GDK_JOIN_MITER);
		gdk_draw_line(pixmap, gc, int(stick_start.x), int(stick_start.y),
		              int(stick_end.x), int(stick_end.y));
		// Draw Power Line
		gdk_gc_set_foreground(gc, red);
		gdk_gc_set_line_attributes(gc, 2, GDK_LINE_SOLID, GDK_CAP_BUTT, GDK_JOIN_MITER);
		gdk_draw_line(pixmap, gc, int(stick_start.x), int(stick_start.y),
		              int(shoot.x), int(shoot.y));
		// Draw Aiming Line
		stick.dx = -stick.dx;
		stick.dy = -stick.dy;
		double a = stick.mag() / 800;
		stick.dx /= a; stick.dy /= a;
		double sx = stick_start.x - stick.dx;
		double sy = stick_start.y - stick.dy;

		gdk_gc_set_foreground(gc, black);
		gdk_gc_set_line_attributes(gc, 1, GDK_LINE_ON_OFF_DASH, GDK_CAP_BUTT, GDK_JOIN_MITER);
		gdk_draw_line(pixmap, gc, int(stick_start.x), int(stick_start.y),
		              int(sx), int(sy));
	}
}

void Application::mouse_down (double x, double y) {
	vector<Ball>::iterator bb;
	bb = find_if(balls.begin(), balls.end(), pointer_selects( x, y, hit_moving));
	if(bb != balls.end())
	{
		selected = bb;
		shoot.x = x; shoot.y = y;
	}
}

void Application::mouse_up (double x, double y) {
	if (selected != NULL) {
		selected->shoot(shoot);
		update_dirty(table.x, table.x + table.width, table.y, table.y + table.height);
	}
	selected = NULL;
}

void Application::mouse_drag (double x, double y) {
	if (selected != NULL) {
		Vec2D vel;
		double sx = double(selected->x), sy = double(selected->y);
		double xdist = (sx - x), ydist = (sy - y);
		vel.setVec( xdist, ydist );
		if(superhuman || (vel.mag() <= 200))
		{
			shoot.x = x; shoot.y = y;
		}
		else
		{
			// calculate a vector with the same direction but
			// a magnitude of 150, since we don't want people
			// to be able to make superhuman shots
			double k = vel.mag() / 200;
			vel.dx /= k; vel.dy /= k;
			shoot.x = selected->x - vel.dx;
			shoot.y = selected->y - vel.dy;
		}
		update_dirty(table.x, table.x + table.width, table.y, table.y + table.height);
	}
}

void Application::update_rack()
{
	if(rack_area == (GtkWidget *)NULL)
		return;
	GdkRectangle area;
	area.x = area.y = 0;
	area.width = width; area.height = 70;
	gtk_widget_draw(GTK_WIDGET(rack_area), &area);
}

void Application::update_table()
{
	if(rack_area == (GtkWidget *)NULL)
		return;
	GdkRectangle area;
	area.x = area.y = 0;
	area.width = width; area.height = height;
	gtk_widget_draw(GTK_WIDGET(drawing_area), &area);
}

void Application::new_cue_ball()
{
	GdkCursor *cursor;
	cursor = gdk_cursor_new(GDK_CIRCLE);
	gdk_window_set_cursor(drawing_area->window, cursor);
	gdk_cursor_destroy(cursor);
	gtk_signal_handler_block(GTK_OBJECT(drawing_area), button_press_handler_id);
	selected = NULL; // just in case
	gtk_signal_handler_unblock(GTK_OBJECT(drawing_area), qbp_hndlr);
//	qbp_hndlr = gtk_signal_connect(GTK_OBJECT(drawing_area), "button_press_event",
//	                               GTK_SIGNAL_FUNC(cue_ball_placed), this);
	placing_cue = true;
}

gint Application::cue_ball_placed(GtkWidget *widget,
                                           GdkEventButton *event,
								   Application *app)
{
	int x = int(event->x) ; int y = int(event->y);
	vector<Ball>::iterator bb;
	bb = find_if(app->balls.begin(), app->balls.end(), pointer_selects(x, y, FALSE));
	if (app->game_name == eight_ball || app->game_name == rotation)
	{
		if(x - BALL_SIZE / 2 > app->table.x &&
		   x + BALL_SIZE / 2 < app->table.x + int(app->table.width / 4) + 8 &&
		   y - BALL_SIZE / 2 > app->table.y &&
		   y + BALL_SIZE / 2 < app->table.y + app->table.height)
		{
			app->new_ball(x, y, CUE, true);
			update_dirty(x - BALL_SIZE / 2, x + BALL_SIZE / 2, y - BALL_SIZE / 2,
			             y + BALL_SIZE / 2);
			gtk_signal_handler_block(GTK_OBJECT(app->drawing_area), app->qbp_hndlr);
			gtk_signal_handler_unblock(GTK_OBJECT(app->drawing_area),
			                           app->button_press_handler_id);
			gdk_window_set_cursor(app->drawing_area->window, (GdkCursor *)NULL);
		}
	} else
	{
		if(x - BALL_SIZE / 2 > app->table.x &&
		   x + BALL_SIZE / 2 < app->table.x + app->table.width &&
		   y - BALL_SIZE / 2 > app->table.y &&
		   y + BALL_SIZE / 2 < app->table.y + app->table.height)
		{
			app->new_ball(x, y, CUE, true);
			update_dirty(x - BALL_SIZE / 2, x + BALL_SIZE / 2, y - BALL_SIZE / 2,
			             y + BALL_SIZE / 2);
			gtk_signal_handler_block(GTK_OBJECT(app->drawing_area), app->qbp_hndlr);
			gtk_signal_handler_unblock(GTK_OBJECT(app->drawing_area),
			                           app->button_press_handler_id);
			gdk_window_set_cursor(app->drawing_area->window, (GdkCursor *)NULL);
		}
	}
	app->placing_cue = false;

	return app->placing_cue;
}

void Application::add_pixmap_directory (const gchar *directory)
{
  pixmaps_directories = g_list_prepend (pixmaps_directories, g_strdup (directory));
}

gchar *Application::check_file_exists (const gchar *directory, const gchar *filename)
{
  gchar *full_filename;
  struct stat s;
  gint status;

  full_filename = (gchar*) g_malloc (strlen (directory) + 1
                                     + strlen (filename) + 1);
  strcpy (full_filename, directory);
  strcat (full_filename, G_DIR_SEPARATOR_S);
  strcat (full_filename, filename);

  status = stat (full_filename, &s);
  if (status == 0 && S_ISREG (s.st_mode))
    return full_filename;
  g_free (full_filename);
  return NULL;
}

GdkPixmap *Application::load_pixmap(GdkPixmap **mask, char *filename)
{
	GList *elem;
	gchar *found_filename = NULL;

	/* We first try any pixmaps directories set by the application. */
	elem = pixmaps_directories;
	while (elem)
	{
		found_filename = check_file_exists ((gchar*)elem->data, filename);
		if (found_filename)
		break;
		elem = elem->next;
	}

	GdkPixmap *pixmap = gdk_pixmap_colormap_create_from_xpm(NULL, colormap,
                            mask, NULL, found_filename);
	if(pixmap == (GdkPixmap *)NULL)
		g_error("Could not load pixmap: %s", found_filename);
	return pixmap;

}

void Application::load_ball_xpms()
{
	add_pixmap_directory("/usr/local/share/gtkpool");
	add_pixmap_directory("/usr/share/gtkpool");
	
	// Small
	balls_pixmaps[CUE] = load_pixmap(&balls_pixmap_masks[CUE], "ball_cue_sm.xpm");
	balls_pixmaps[YELLOW] = load_pixmap(&balls_pixmap_masks[YELLOW], "ball_1_sm.xpm");
	balls_pixmaps[BLUE] = load_pixmap(&balls_pixmap_masks[BLUE], "ball_2_sm.xpm");
	balls_pixmaps[RED] = load_pixmap(&balls_pixmap_masks[RED], "ball_3_sm.xpm");
	balls_pixmaps[PURPLE] = load_pixmap(&balls_pixmap_masks[PURPLE], "ball_4_sm.xpm");
	balls_pixmaps[ORANGE] = load_pixmap(&balls_pixmap_masks[ORANGE], "ball_5_sm.xpm");
	balls_pixmaps[GREEN] = load_pixmap(&balls_pixmap_masks[GREEN], "ball_6_sm.xpm");
	balls_pixmaps[BROWN] = load_pixmap(&balls_pixmap_masks[BROWN], "ball_7_sm.xpm");
	balls_pixmaps[EIGHT] = load_pixmap(&balls_pixmap_masks[EIGHT], "ball_8_sm.xpm");
	balls_pixmaps[YELLOW_S] = load_pixmap(&balls_pixmap_masks[YELLOW_S], "ball_9_sm.xpm");
	balls_pixmaps[BLUE_S] = load_pixmap(&balls_pixmap_masks[BLUE_S], "ball_10_sm.xpm");
	balls_pixmaps[RED_S] = load_pixmap(&balls_pixmap_masks[RED_S], "ball_11_sm.xpm");
	balls_pixmaps[PURPLE_S] = load_pixmap(&balls_pixmap_masks[PURPLE_S], "ball_12_sm.xpm");
	balls_pixmaps[ORANGE_S] = load_pixmap(&balls_pixmap_masks[ORANGE_S], "ball_13_sm.xpm");
	balls_pixmaps[GREEN_S] = load_pixmap(&balls_pixmap_masks[GREEN_S], "ball_14_sm.xpm");
	balls_pixmaps[BROWN_S] = load_pixmap(&balls_pixmap_masks[BROWN_S], "ball_15_sm.xpm");

	// Large
	balls_big_pixmaps[CUE] = load_pixmap(&balls_big_pixmap_masks[CUE], "ball_cue_lg.xpm");
	balls_big_pixmaps[YELLOW] = load_pixmap(&balls_big_pixmap_masks[YELLOW], "ball_1_lg.xpm");
	balls_big_pixmaps[BLUE] = load_pixmap(&balls_big_pixmap_masks[BLUE], "ball_2_lg.xpm");
	balls_big_pixmaps[RED] = load_pixmap(&balls_big_pixmap_masks[RED], "ball_3_lg.xpm");
	balls_big_pixmaps[PURPLE] = load_pixmap(&balls_big_pixmap_masks[PURPLE], "ball_4_lg.xpm");
	balls_big_pixmaps[ORANGE] = load_pixmap(&balls_big_pixmap_masks[ORANGE], "ball_5_lg.xpm");
	balls_big_pixmaps[GREEN] = load_pixmap(&balls_big_pixmap_masks[GREEN], "ball_6_lg.xpm");
	balls_big_pixmaps[BROWN] = load_pixmap(&balls_big_pixmap_masks[BROWN], "ball_7_lg.xpm");
	balls_big_pixmaps[EIGHT] = load_pixmap(&balls_big_pixmap_masks[EIGHT], "ball_8_lg.xpm");
	balls_big_pixmaps[YELLOW_S] = load_pixmap(&balls_big_pixmap_masks[YELLOW_S], "ball_9_lg.xpm");
	balls_big_pixmaps[BLUE_S] = load_pixmap(&balls_big_pixmap_masks[BLUE_S], "ball_10_lg.xpm");
	balls_big_pixmaps[RED_S] = load_pixmap(&balls_big_pixmap_masks[RED_S], "ball_11_lg.xpm");
	balls_big_pixmaps[PURPLE_S] = load_pixmap(&balls_big_pixmap_masks[PURPLE_S], "ball_12_lg.xpm");
	balls_big_pixmaps[ORANGE_S] = load_pixmap(&balls_big_pixmap_masks[ORANGE_S], "ball_13_lg.xpm");
	balls_big_pixmaps[GREEN_S] = load_pixmap(&balls_big_pixmap_masks[GREEN_S], "ball_14_lg.xpm");
	balls_big_pixmaps[BROWN_S] = load_pixmap(&balls_big_pixmap_masks[BROWN_S], "ball_15_lg.xpm");

	// Pool Table
	table_pixmaps[TABLE_MB] = load_pixmap(&table_pixmaps[TABLE_MB], "pool_table_maple-burl.xpm");
	table_pixmaps[TABLE_BL] = load_pixmap(&table_pixmaps[TABLE_BL], "pool_table_black-laquer.xpm");
	table_pixmaps[TABLE_BP] = load_pixmap(&table_pixmaps[TABLE_BP], "pool_table_black-pearl.xpm");
	table_pixmaps[TABLE_MA] = load_pixmap(&table_pixmaps[TABLE_MA], "pool_table_mahogany.xpm");
}

void Application::load_sounds()
{
	gchar *found_filename = NULL;
	GList *elem;
	char filename[200] = "click.raw";
	coll_sndc = 1024; // max size to read
	elem = pixmaps_directories;
	while (elem)
	{
		found_filename = check_file_exists ((gchar*)elem->data, filename);
		if (found_filename)
		break;
		elem = elem->next;
	}
	load_sound(found_filename, coll_snd, &coll_sndc);	

}

void Application::init_sound()
{
	try
	{
		open_sound_device();
		configure_sound_device();
		sound = true;
	}
	catch (SoundError &se)
	{
		sound = false;
		switch (se.code)
		{
			case SE_OPENFAIL:
				g_message("An error occured while opening the sound device\n\t%s", se.description);
				break;
			case SE_FRAGFAIL:
				g_message("An error occured while setting the sound card's fragment size");
				g_message("\t%s\nThe requested value was: %x", se.description,
				          se.requested);
				break;
		};
	}
	if(sound)
		load_sounds();
}

void Application::print_message(const char *message, int colour = 0)
{
	if (colour > 4)
		colour = 14;
	if (colour < 0)
		colour = 14;
	gtk_text_insert(GTK_TEXT(chat_text), NULL, message_colours[colour], NULL, "\n", -1);
	gtk_text_insert(GTK_TEXT(chat_text), NULL, message_colours[colour], NULL, message, -1);
}
