/*************************************************************************\
*    gtkpool -- a so far not-so-great pool game                           *
*    Copyright (C) 1999 Jacques Fortier                                   *
*                                                                         *
*    This program is free software; you can redistribute it and/or modify *
*    it under the terms of the GNU General Public License as published by *
*    the Free Software Foundation; either version 2 of the License, or    *
*    (at your option) any later version.                                  *
*                                                                         *
*    This program is distributed in the hope that it will be useful,      *
*    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
*    GNU General Public License for more details.                         *
*                                                                         *
*    You should have received a copy of the GNU General Public License    *
*    along with this program; if not, write to the Free Software          *
*    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
\*************************************************************************/
#include <math.h>
#include "pool.hh"


int udt, udb, udl, udr;

void update_dirty(int ludl, int ludr, int ludt, int ludb)
{
	if(ludl > ludr)
	{
		int temp = ludl;
		ludl = ludr; ludr = temp;
	}
	if(ludt > ludb)
	{
		int temp = ludt;
		ludt = ludb; ludb = temp;
	}
	if(udl == -1)
	{
		udl = ludl; udr = ludr;
		udt = ludt; udb = ludb;
	}
	else
	{
		if(ludl < udl) udl = ludl;
		if(ludr > udr) udr = ludr;
		if(ludt < udt) udt = ludt;
		if(ludb > udb) udb = ludb;
	}
}

/***************************************************************
                           Vec2D
***************************************************************/
// have to outline because it refers to Point2D members
Vec2D::Vec2D (const Point2D &p1, const Point2D &p2) {
	setVec(p2.x - p1.x, p2.y - p1.y);
}


/***************************************************************
                           Point2D
***************************************************************/
double Point2D::dist (const Point2D &loc) {
	double xSq = loc.x - x;
	double ySq = loc.y - y;
	return sqrt((xSq * xSq) + (ySq * ySq));
}

Point2D Point2D::setPos (double x, double y)
{
	this->x = x;
	this->y = y;
	return *this;
}

Point2D Point2D::addPos (const Vec2D &vel) {
	return setPos(x + vel.dx, y + vel.dy);
}


/***************************************************************
                           Ball
***************************************************************/
bool Ball::moving () {
	return vel.dx != 0  ||  vel.dy != 0;
}

void Ball::decel (double val) {
	if (val >= vel.mag())
		vel.setVec(0, 0);
	else
		vel.subVec(tvec.setVec(vel).unitVec().mulVec(val));
}

double Ball::pathIntercept (Ball b) {
	double d = radius + b.radius;
	double ddx = vel.dx - b.vel.dx;
	double ddy = vel.dy - b.vel.dy;
	double dx = x - b.x;
	double dy = y - b.y;
	double A = ddx * ddx + ddy * ddy;
	double B = 2 * (dx * ddx + dy * ddy);
	double C = dx * dx + dy * dy - d * d;
	return (-B - sqrt(B * B - 4 * A * C)) / (2 * A);
}
void Ball::collide (Ball *b, double COLLIDE_DRAG = 0.95) {
	// Calculate collision in b's reference frame
	vel.subVec(b->vel);
	double mv = vel.mag();
	Vec2D v12(*this, *b); v12.unitVec();
	Vec2D v1c(vel); v1c.unitVec();
	double cos = v1c.dotProd(v12);
	v12.mulVec(cos * mv);
	vel.subVec(v12);
	vel.addVec(b->vel);
	b->vel.addVec(v12);
	b->vel.dx *= COLLIDE_DRAG;
	b->vel.dy *= COLLIDE_DRAG;
	vel.dx *= COLLIDE_DRAG;
	vel.dy *= COLLIDE_DRAG;
}

double Ball::edgeIntercept (GdkRectangle table) {
	if (vel.dx >= 0)
		hCol = (table.width + table.x - x - radius) / vel.dx;
	else
		hCol = (table.x - x + radius) / vel.dx;
	if (vel.dy >= 0)
		vCol = (table.height + table.y - y - radius) / vel.dy;
	else
		vCol = (table.y - y + radius) / vel.dy;
	if(vel.dx == 0)
		return vCol;
	if(vel.dy == 0)
		return hCol;
	return hCol <= vCol ? hCol : vCol;
}

void Ball::bounce (double t, double BUMPER_DRAG = 0.8) {
	if (t == hCol)
		vel.dx = -vel.dx;
	if (t == vCol)
		vel.dy = -vel.dy;
	vel.dx *= BUMPER_DRAG; vel.dy *= BUMPER_DRAG;
}

void Ball::move (double t) {
	if (moving())
	{
		// move the ball, then update the dirty rectangle
		int oldx = int(x), oldy = int(y);
		int newx, newy, ludl, ludr, ludt, ludb;
		addPos(vel.copy().mulVec(t));
		newx = int(x); newy = int(y);
		if(newx != oldx || newy != oldy) // ie, ball has moved perceptibly
		{
			ludl = oldx < newx ? oldx : newx; ludl -= (BALL_SIZE / 2 + 1);
			ludr = oldx > newx ? oldx : newx; ludr += (BALL_SIZE / 2 + 1);
			ludt = oldy < newy ? oldy : newy; ludt -= (BALL_SIZE / 2 + 1);
			ludb = oldy > newy ? oldy : newy; ludb += (BALL_SIZE / 2 + 1);
			update_dirty(ludl, ludr, ludt, ludb);
		}
	}
}

void Ball::draw (GdkDrawable *pixmap, GdkGC *gc) const {
		if(picture)
		{
			gdk_gc_set_clip_mask(gc, clip_bmp);
			gdk_gc_set_clip_origin(gc, int(x - diam / 2), int(y - diam / 2));
			gdk_draw_pixmap(pixmap, gc, picture, 0, 0, int(x - diam / 2),
			                int (y - diam / 2), BALL_SIZE, BALL_SIZE);
			gdk_gc_set_clip_origin(gc, 0, 0);
			gdk_gc_set_clip_mask(gc, (GdkWindow *)NULL);
		}
		else
		{
			gdk_gc_set_foreground(gc, clr);
			gdk_gc_set_background(gc, clr);
			gdk_draw_arc(pixmap, gc, 1, int(x - radius), int(y - radius),
			             diam, diam, 0, 360 * 64);
		}
}
	
void Ball::shoot (GdkPoint ptr) {
	vel.setVec((x - ptr.x) / 10, (y - ptr.y) / 10);
}

bool Ball::touches (int mx, int my) {
	return Point2D(mx, my).dist(*this) < radius;
}
