/*************************************************************************\
*    gtkpool -- a so far not-so-great pool game                           *
*    Copyright (C) 1999 Jacques Fortier                                   *
*                                                                         *
*    This program is free software; you can redistribute it and/or modify *
*    it under the terms of the GNU General Public License as published by *
*    the Free Software Foundation; either version 2 of the License, or    *
*    (at your option) any later version.                                  *
*                                                                         *
*    This program is distributed in the hope that it will be useful,      *
*    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
*    GNU General Public License for more details.                         *
*                                                                         *
*    You should have received a copy of the GNU General Public License    *
*    along with this program; if not, write to the Free Software          *
*    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
\*************************************************************************/
#include "pool.hh"
#include "application.hh"
#include "sound.hh"
#include <vector>
#include <algorithm>

class check_table_collision
{
	public:
	double &ei, time; Ball *bnc;
	GdkRectangle table;
	check_table_collision(double &pei, double ptime, Ball *pbnc,
	                      GdkRectangle ptable)
	          : ei(pei), time(ptime), table(ptable)
	{
		bnc = pbnc;
	}
	void operator() (Ball &bb)
	{
		if(bb.vel.dx == 0 && bb.vel.dy == 0)
			return;
		double eb = bb.edgeIntercept(table);
		if(eb < ei && eb < time)
		{
			ei = eb;
			bnc = &bb;
		}
	}
};

class move_balls
{
	public:
	double dt2;
	move_balls(double pdt2)
	{
		dt2 = pdt2;
	}
	void operator() (Ball &b)
	{
		b.move(dt2);
	}
};

class apply_friction
{
	public:
	double fric;
	apply_friction(double friction)
	{
		fric = friction;
	}
	void operator()(Ball &b)
	{
		b.decel(fric);
	}
};

class draw_ball
{
	public:
	GdkDrawable *pixmap;
	GdkGC *gc;
	draw_ball(GdkDrawable *p, GdkGC *g)
	{
		pixmap = p; gc = g;
	}
	void operator() (const Ball &b)
	{
//		b.draw(pixmap, gc);
		if(b.picture)
		{
			gdk_gc_set_clip_mask(gc, b.clip_bmp);
			gdk_gc_set_clip_origin(gc, int(b.x - b.diam / 2), int(b.y - b.diam / 2));
			gdk_draw_pixmap(pixmap, gc, b.picture, 0, 0, int(b.x - b.diam / 2),
			                int (b.y - b.diam / 2), BALL_SIZE, BALL_SIZE);
			gdk_gc_set_clip_origin(gc, 0, 0);
			gdk_gc_set_clip_mask(gc, (GdkWindow *)NULL);
		}
		else
		{
			gdk_gc_set_foreground(gc, b.clr);
			gdk_gc_set_background(gc, b.clr);
			gdk_draw_arc(pixmap, gc, 1, int(b.x - b.radius), int(b.y - b.radius),
			             b.diam, b.diam, 0, 360 * 64);
		}
	}
};

class pointer_selects
{
	public:
	int x, y;
	bool hit_moving;
	pointer_selects(int px, int py, bool hm)
	{
		x = px; y = py;
		hit_moving = hm;
	}
	bool operator()(Ball &b)
	{
		return ((!b.moving() || hit_moving) && b.touches(x, y));
	}
};

class check_pocket
{
	public:
	int left, right, top, bottom, left_of_mid, right_of_mid;
	double time;
	int ball_num;
	bool cue, eight;
	
	check_pocket(GdkRectangle table, double ei)
	{
		cue = eight = false;
		left = table.x + 20;
		right = (table.width + table.x) - 20;
		top = table.y + 20;
		bottom = table.height + table.y - 20;
		left_of_mid = table.x + table.width / 2 - 15;
		right_of_mid = left_of_mid + 30;
		time = ei;
	}
	bool operator() (Ball &b)
	{
		if(b.hCol == time)
		{
			double c = b.y + b.vel.dy * b.hCol;
			if(c < top || c > bottom)
			{
				ball_num = b.ball_num;
				if(b.is_cue) cue = true;
				if(b.is_eight) eight = true;
				return true;
			}
		}
		else
		{
			double c = b.x + b.vel.dx * b.vCol;
			if(c < left || c > right || (c > left_of_mid && c < right_of_mid))
			{
				ball_num = b.ball_num;
				if(b.is_cue) cue = true;
				if(b.is_eight) eight = true;
				return true;
			}
		}
		return false;
	}
};

void Application::init()
{
	visual = gdk_visual_get_system();
	colormap = gdk_colormap_get_system();

	load_ball_xpms();
	if(sound)
		init_sound();

	gdk_color_parse("ForestGreen", table_color);
	gdk_color_parse("Red", message_colours[1]);
	gdk_color_parse("Green", message_colours[2]);
	gdk_color_parse("Blue", message_colours[3]);
	gdk_color_parse("Magenta", message_colours[4]);
	gdk_color_parse("Gray", message_colours[14]);
	gdk_colormap_alloc_color(colormap, table_color, FALSE, TRUE);
	gdk_colormap_alloc_color(colormap, message_colours[1], FALSE, TRUE);
	gdk_colormap_alloc_color(colormap, message_colours[2], FALSE, TRUE);
	gdk_colormap_alloc_color(colormap, message_colours[3], FALSE, TRUE);
	gdk_colormap_alloc_color(colormap, message_colours[4], FALSE, TRUE);
	gdk_colormap_alloc_color(colormap, message_colours[14], FALSE, TRUE);
	gdk_color_black(colormap, black);
	message_colours[0] = black;

	// setup adjustments -- these are used in options dialog
	// to control physics
	friction_adjustment = gtk_adjustment_new(0.06, 0, 1, 0.01, 0.1, 0);
	collide_drag_adjustment = gtk_adjustment_new(0.95, 0, 1, 0.01, 0.05, 0);
	bumper_drag_adjustment = gtk_adjustment_new(0.8, 0.0, 1.5, 0.01, 0.1, 0);

	resetTable();
}

void Application::destroy()
{
	if(buf_pixmap)
		gdk_pixmap_unref(buf_pixmap);
	gdk_pixmap_unref(rack_pixmap);
	gdk_pixmap_unref(rack_mask);
	for(int i = 0; i < NUM_BALLS; i++)
	{
		gdk_pixmap_unref(balls_pixmaps[i]);
		gdk_bitmap_unref(balls_pixmap_masks[i]);
	}

	gdk_colormap_free_colors(colormap, table_color, 1);

	gdk_colormap_free_colors(colormap, message_colours[1], 1);
	gdk_colormap_free_colors(colormap, message_colours[2], 1);
	gdk_colormap_free_colors(colormap, message_colours[3], 1);
	gdk_colormap_free_colors(colormap, message_colours[4], 1);
	gdk_colormap_free_colors(colormap, message_colours[14], 1);
}

void Application::updateBalls () {
	/* Every time this function is called, it moves time forward by one
animation frame.  This means that every ball will be moved by its
vel.dx and vel.dy values by the time the function returns.  However, to
process collisions the frame is broken up.  First the balls are
analyzed to see if any collisions will happen in this tick.  The
first collision is selected, and its time noted.  All the balls are
advanced to that time, and the collision is processed.  This is
reiterated until there are no more collisions left in the frame
*/
	double	md, ei, time;
	int cnt = balls.size();
	sunk_tf = collide_tf = bounce_tf = false;
	md = ei = time = 1;
	do {
		Ball *cb1 = NULL, *cb2 = NULL, *bnc = NULL;
		check_table_collision c(ei, time, bnc, table);
		// Check for collisions between balls and the edge of the table
		check_table_collision c2 = for_each(balls.begin(), balls.end(), c);
		bnc = c2.bnc;
		// Check for ball to ball collisions
		for (int ii = 1; ii < cnt; ii++) {
			for (int jj = 0; jj < ii; jj++) {
				double coll_time = balls[ii].pathIntercept(balls[jj]);
				if (coll_time < md  && coll_time >= 0  &&  coll_time < time) {
					md = coll_time;
					cb1 = &balls[ii];
					cb2 = &balls[jj];
				}
			}
		}
		// Determine closest event
		double dt = md < time ? md : time;
		double dt2 = dt < ei ? dt : ei;
		// Move all ball's forward to this point in time
		for_each(balls.begin(), balls.end(), move_balls(dt2));
		// Process any collisions
		if (ei <= dt &&  bnc != NULL) // if this event is a bumper collision
		{
			check_pocket cp(table, ei); 
			if(cp(*bnc))           // if the ball is going into a pocket
			{
				int r = BALL_SIZE / 2 + 1;
				update_dirty(int(bnc->x - r), int(bnc->x + r),
				             int(bnc->y - r), int(bnc->y + r));
				sunk_balls |= (1 << cp.ball_num);
				sunk_balls1.push_back(bnc->ball_num);
				update_rack();
				balls.erase(bnc); // get rid of it
				if(cp.cue)
					new_cue_ball();
				sunk_tf = true;
			}
			else             // otherwise bounce it off the bumper
			{
				bnc->bounce(ei, bumper_drag());
				bounce_tf = true;
			}
		}
		else if (md <= time  &&  cb1 != NULL)
		{
			cb1->collide(cb2, collide_drag());
			collide_tf = true;
		}
		time -= dt2; // wind the clock forward
		ei = md = time;
	} while (time > 0);
	// Apply friction deceleration to all balls
	for_each(balls.begin(), balls.end(), apply_friction(friction()));
	// redraw aimer if it selects a moving ball
	if((hit_moving && selected != NULL) &&
	   (selected->vel.dx != 0 || selected->vel.dy != 0))
	{
		update_dirty(shoot.x, int(selected->x), shoot.y, int(selected->y));
	}
	if((balls.size() == 1 && balls.back().is_cue) || balls.size() == 0)
	{
		resetTable();
	}
}


void Application::resetTable () {
	double ox = width * 3 / 4;
	double oy = height / 2;
	double ofx = sqrt((BALL_SIZE * BALL_SIZE) - ((BALL_SIZE / 2) * (BALL_SIZE / 2))) + .1;
	double ofy = BALL_SIZE / 2 + .1;
	double s = BALL_SIZE / 2 + 1;
	double t = BALL_SIZE - 1;
	balls.erase(balls.begin(), balls.end());
	sunk_balls1.erase(sunk_balls1.begin(), sunk_balls1.end());
	sunk_balls2.erase(sunk_balls2.begin(), sunk_balls2.end());
	if(game_name == eight_ball)
	{
		new_ball(width / 4, oy, CUE, true);
		new_ball(ox, oy, PURPLE);
		new_ball(ox + ofx, oy - ofy, RED_S);
		new_ball(ox + ofx, oy + ofy + .02, BLUE);
		new_ball(ox + ofx * 2 - .02, oy - ofy * 2 - .01, GREEN_S);
		new_ball(ox + ofx * 2, oy, EIGHT, false, true);
		new_ball(ox + ofx * 2, oy + ofy * 2, BROWN_S);
		new_ball(ox + ofx * 3 + .03, oy - ofy * 3 - .02, ORANGE);
		new_ball(ox + ofx * 3, oy - ofy, YELLOW_S);
		new_ball(ox + ofx * 3, oy + ofy, RED);
		new_ball(ox + ofx * 3 + .01, oy + ofy * 3, BLUE_S);
		new_ball(ox + ofx * 4, oy - ofy * 4 - .04, GREEN);
		new_ball(ox + ofx * 4, oy - ofy * 2, PURPLE_S);
		new_ball(ox + ofx * 4, oy, BROWN);
		new_ball(ox + ofx * 4 + .02, oy + ofy * 2, ORANGE_S);
		new_ball(ox + ofx * 4, oy + ofy * 4, YELLOW);
	}
	else if(game_name == nine_ball)
	{
		new_ball(width / 4, oy, CUE, true);
		new_ball(ox, oy, YELLOW);
		new_ball(ox + ofx, oy - ofy, BLUE);
		new_ball(ox + ofx, oy + ofy + .02, GREEN);
		new_ball(ox + ofx * 2 - .02, oy - ofy * 2 - .01, BROWN);
		new_ball(ox + ofx * 2, oy, YELLOW_S, false, true);
		new_ball(ox + ofx * 2, oy + ofy * 2, RED);
		new_ball(ox + ofx * 3, oy - ofy, ORANGE);
		new_ball(ox + ofx * 3, oy + ofy, EIGHT);
		new_ball(ox + ofx * 4, oy, PURPLE);
	}
	sunk_balls = 0;
	udl = table.x; udr = table.x + table.width;
	udt = table.y; udb = table.y + table.height;
	update_rack();
	if(placing_cue)
	{
		gtk_signal_handler_block(GTK_OBJECT(drawing_area), qbp_hndlr);
		gtk_signal_handler_unblock(GTK_OBJECT(drawing_area),
		                           button_press_handler_id);
		gdk_window_set_cursor(drawing_area->window, (GdkCursor *)NULL);
		placing_cue = false;
		// this is in case the cue ball was sunk but not placed before the
		// reset, so that htere can't be more than one cue ball
	}
}

void Application::paint (GdkDrawable *pixmap, GdkGC *gc) {
	const int d = 40;
	const int r = d / 2;
/*	gdk_gc_set_foreground(gc, table_color);
	gdk_draw_rectangle(pixmap, gc, TRUE, table.x, table.y,
	                   table.width, table.height);*/
	gdk_draw_pixmap(pixmap, gc, table_pixmap, 0, 0, 0, 0, width, height);
/*	gdk_draw_arc(pixmap, gc, 1, table.x - r,               table.y - r,                  d, d, 0, 360 * 64);
	gdk_draw_arc(pixmap, gc, 1, table.x - r + table.width, table.y - r,                  d, d, 0, 360 * 64);
	gdk_draw_arc(pixmap, gc, 1, table.x - r,               table.y - r + table.height,   d, d, 0, 360 * 64);
	gdk_draw_arc(pixmap, gc, 1, table.x - r + table.width, table.y - r + table.height,   d, d, 0, 360 * 64);

	gdk_draw_arc(pixmap, gc, 1, table.x - 15 + table.width/2, table.y - 15,                30, 30, 0, 360 * 64);
	gdk_draw_arc(pixmap, gc, 1, table.x - 15 + table.width/2, table.y - 15 + table.height, 30, 30, 0, 360 * 64);*/
	for_each(balls.begin(), balls.end(), draw_ball(pixmap, gc));
	if (selected != NULL) {
		gdk_gc_set_foreground(gc, black);
		gdk_draw_line(pixmap, gc, int(selected->x), int(selected->y),
		              shoot.x, shoot.y);
	}
}

void Application::mouse_down (int x, int y) {
	vector<Ball>::iterator bb;
	bb = find_if(balls.begin(), balls.end(), pointer_selects(x, y, hit_moving));
	if(bb != balls.end())
	{
		selected = bb;
		shoot.x = x; shoot.y = y;
	}
}

void Application::mouse_up (int x, int y) {
	if (selected != NULL) {
		int ludl, ludr, ludt, ludb;
		selected->shoot(shoot);
		ludl = shoot.x - 3 <? int(selected->x);
		ludr = shoot.x + 3 >? int(selected->x);
		ludt = shoot.y - 3 <? int(selected->y);
		ludb = shoot.y + 3 >? int(selected->y);
		update_dirty(ludl, ludr, ludt, ludb);
	}
	selected = NULL;
}

void Application::mouse_drag (int x, int y) {
	if (selected != NULL) {
		Vec2D vel;
		int ludl, ludr, ludt, ludb;
		int oldx = shoot.x, oldy = shoot.y;
		vel.setVec((selected->x - x), (selected->y - y));
		if(superhuman || (vel.mag() <= 150))
		{
			shoot.x = x; shoot.y = y;
		}
		else
		{
			// calculate a vector with the same direction but
			// a magnitude of 150, since we don't want people
			// to be able to make superhuman shots
			double k = vel.mag() / 150;
			vel.dx /= k; vel.dy /= k;
			shoot.x = short(selected->x - vel.dx);
			shoot.y = short(selected->y - vel.dy);
		}
		ludl = int(shoot.x) <? int(selected->x);
		ludr = int(shoot.x) >? int(selected->x);
		ludt = int(shoot.y) <? int(selected->y);
		ludb = int(shoot.y) >? int(selected->y);
		ludl = ludl <? oldx;
		ludr = ludr >? oldx;
		ludt = ludt <? oldy;
		ludb = ludb >? oldy;
		ludl--; ludr++; ludt--; ludb++;
		update_dirty(ludl, ludr, ludt, ludb);
	}
}

void Application::update_rack()
{
	if(rack_area == (GtkWidget *)NULL)
		return;
	GdkRectangle area;
	area.x = area.y = 0;
	area.width = width; area.height = 70;
	gtk_widget_draw(GTK_WIDGET(rack_area), &area);
}

void Application::new_cue_ball()
{
	GdkCursor *cursor;
	cursor = gdk_cursor_new(GDK_CIRCLE);
	gdk_window_set_cursor(drawing_area->window, cursor);
	gdk_cursor_destroy(cursor);
	gtk_signal_handler_block(GTK_OBJECT(drawing_area), button_press_handler_id);
	selected = NULL; // just in case
	gtk_signal_handler_unblock(GTK_OBJECT(drawing_area), qbp_hndlr);
//	qbp_hndlr = gtk_signal_connect(GTK_OBJECT(drawing_area), "button_press_event",
//	                               GTK_SIGNAL_FUNC(cue_ball_placed), this);
	placing_cue = true;				    
}

gint Application::cue_ball_placed(GtkWidget *widget,
                                           GdkEventButton *event,
								   Application *app)
{
	int x = int(event->x) ; int y = int(event->y);
	if(x - BALL_SIZE / 2 > app->table.x &&
	   x + BALL_SIZE / 2 < app->table.x + app->table.width &&
	   y - BALL_SIZE / 2 > app->table.y &&
	   y + BALL_SIZE / 2 < app->table.y + app->table.height)
	{
		app->new_ball(x, y, CUE, true);
		update_dirty(x - BALL_SIZE / 2, x + BALL_SIZE / 2, y - BALL_SIZE / 2,
		             y + BALL_SIZE / 2);
		gtk_signal_handler_block(GTK_OBJECT(app->drawing_area), app->qbp_hndlr);
		gtk_signal_handler_unblock(GTK_OBJECT(app->drawing_area),
		                           app->button_press_handler_id);
		gdk_window_set_cursor(app->drawing_area->window, (GdkCursor *)NULL);
	}
	app->placing_cue = false;
}

GdkPixmap *Application::load_pixmap(GdkPixmap **mask, char *file, char *path)
{
	char filename[200] = "";
	if(path != (char *)NULL)
		strcpy(filename, path);
	strcat(filename, file);
	
	GdkPixmap *pixmap = gdk_pixmap_colormap_create_from_xpm(NULL, colormap,
                            mask, NULL, filename);
	if(pixmap == (GdkPixmap *)NULL)
		g_error("Could not load pixmap: %s", filename);
	return pixmap;
}

void Application::load_ball_xpms()
{
//	char filename[200];
	char path[100] = POOLDATA_PATH;
//	int pathlen = strlen(POOLDATA_PATH);
//	strcpy(filename, POOLDATA_PATH);
	rack_pixmap = load_pixmap(&rack_mask, "balls-big.xpm", path);
	balls_pixmaps[RED] = load_pixmap(&balls_pixmap_masks[RED],
	                                 "ball-solid-red.xpm", path);
	balls_pixmaps[GREEN] = load_pixmap(&balls_pixmap_masks[GREEN],
	                                 "ball-solid-green.xpm", path);
	balls_pixmaps[ORANGE] = load_pixmap(&balls_pixmap_masks[ORANGE],
	                                 "ball-solid-orange.xpm", path);
	balls_pixmaps[YELLOW] = load_pixmap(&balls_pixmap_masks[YELLOW],
	                                 "ball-solid-yellow.xpm", path);
	balls_pixmaps[PURPLE] = load_pixmap(&balls_pixmap_masks[PURPLE],
	                                 "ball-solid-purple.xpm", path);
	balls_pixmaps[BROWN] = load_pixmap(&balls_pixmap_masks[BROWN],
	                                 "ball-solid-brown.xpm", path);
	balls_pixmaps[BLUE] = load_pixmap(&balls_pixmap_masks[BLUE],
	                                 "ball-solid-blue.xpm", path);
	balls_pixmaps[EIGHT] = load_pixmap(&balls_pixmap_masks[EIGHT],
	                                 "ball-eight.xpm", path);

	balls_pixmaps[RED_S] = load_pixmap(&balls_pixmap_masks[RED_S],
	                                 "ball-striped-red.xpm", path);
	balls_pixmaps[GREEN_S] = load_pixmap(&balls_pixmap_masks[GREEN_S],
	                                 "ball-striped-green.xpm", path);
	balls_pixmaps[ORANGE_S] = load_pixmap(&balls_pixmap_masks[ORANGE_S],
	                                     "ball-striped-orange.xpm", path);
	balls_pixmaps[YELLOW_S] = load_pixmap(&balls_pixmap_masks[YELLOW_S],
	                                     "ball-striped-yellow.xpm", path);
	balls_pixmaps[PURPLE_S] = load_pixmap(&balls_pixmap_masks[PURPLE_S],
	                                     "ball-striped-purple.xpm", path);
	balls_pixmaps[BROWN_S] = load_pixmap(&balls_pixmap_masks[BROWN_S],
	                                     "ball-striped-brown.xpm", path);
	balls_pixmaps[BLUE_S] = load_pixmap(&balls_pixmap_masks[BLUE_S],
	                                     "ball-striped-blue.xpm", path);
	balls_pixmaps[SHADOW] = load_pixmap(&balls_pixmap_masks[SHADOW],
	                                     "ball-shadow.xpm", path);
	balls_pixmaps[CUE] = load_pixmap(&balls_pixmap_masks[CUE],
	                                     "ball-cue.xpm", path);
	
	balls_big_pixmaps[RED] = load_pixmap(&balls_big_pixmap_masks[RED],
	                                 "ball-solid-red-big.xpm", path);
	balls_big_pixmaps[GREEN] = load_pixmap(&balls_big_pixmap_masks[GREEN],
	                                 "ball-solid-green-big.xpm", path);
	balls_big_pixmaps[ORANGE] = load_pixmap(&balls_big_pixmap_masks[ORANGE],
	                                 "ball-solid-orange-big.xpm", path);
	balls_big_pixmaps[YELLOW] = load_pixmap(&balls_big_pixmap_masks[YELLOW],
	                                 "ball-solid-yellow-big.xpm", path);
	balls_big_pixmaps[PURPLE] = load_pixmap(&balls_big_pixmap_masks[PURPLE],
	                                 "ball-solid-purple-big.xpm", path);
	balls_big_pixmaps[BROWN] = load_pixmap(&balls_big_pixmap_masks[BROWN],
	                                 "ball-solid-brown-big.xpm", path);
	balls_big_pixmaps[BLUE] = load_pixmap(&balls_big_pixmap_masks[BLUE],
	                                 "ball-solid-blue-big.xpm", path);
	balls_big_pixmaps[EIGHT] = load_pixmap(&balls_big_pixmap_masks[EIGHT],
	                                 "ball-eight-big.xpm", path);

	balls_big_pixmaps[RED_S] = load_pixmap(&balls_big_pixmap_masks[RED_S],
	                                 "ball-striped-red-big.xpm", path);
	balls_big_pixmaps[GREEN_S] = load_pixmap(&balls_big_pixmap_masks[GREEN_S],
	                                 "ball-striped-green-big.xpm", path);
	balls_big_pixmaps[ORANGE_S] = load_pixmap(&balls_big_pixmap_masks[ORANGE_S],
	                                     "ball-striped-orange-big.xpm", path);
	balls_big_pixmaps[YELLOW_S] = load_pixmap(&balls_big_pixmap_masks[YELLOW_S],
	                                     "ball-striped-yellow-big.xpm", path);
	balls_big_pixmaps[PURPLE_S] = load_pixmap(&balls_big_pixmap_masks[PURPLE_S],
	                                     "ball-striped-purple-big.xpm", path);
	balls_big_pixmaps[BROWN_S] = load_pixmap(&balls_big_pixmap_masks[BROWN_S],
	                                     "ball-striped-brown-big.xpm", path);
	balls_big_pixmaps[BLUE_S] = load_pixmap(&balls_big_pixmap_masks[BLUE_S],
	                                     "ball-striped-blue-big.xpm", path);
/*	balls_big_pixmaps[SHADOW] = load_pixmap(&balls_big_pixmap_masks[SHADOW],
	                                     "ball-shadow-big.xpm", path);*/
	balls_big_pixmaps[CUE] = load_pixmap(&balls_big_pixmap_masks[CUE],
	                                     "ball-cue-big.xpm", path);
	
	table_pixmap = load_pixmap(NULL, "table.xpm", path);
}

void Application::load_sounds()
{
	char filename[200];
	strcpy(filename, POOLDATA_PATH);
	strcat(filename, "click.raw");
	coll_sndc = 1024; // max size to read
	load_sound(filename, coll_snd, &coll_sndc);
}

void Application::init_sound()
{
	try
	{
		open_sound_device();
		configure_sound_device();
		sound = true;
	}
	catch (SoundError &se)
	{
		switch (se.code)
		{
			case SE_OPENFAIL:
				g_message("An error occured while opening the sound device\n\t%s", se.description);
				break;
			case SE_FRAGFAIL:
				g_message("An error occured while setting the sound card's fragment size");
				g_message("\t%s\nThe requested value was: %x", se.description,
				          se.requested);
				break;
		};
	}
	if(sound)
		load_sounds();
}

void Application::print_message(const char *message, int colour = 0)
{
	if (colour > 4)
		colour = 14;
	if (colour < 0)
		colour = 14;
	gtk_text_insert(GTK_TEXT(chat_text), NULL, message_colours[colour], NULL, "\n", -1);
	gtk_text_insert(GTK_TEXT(chat_text), NULL, message_colours[colour], NULL, message, -1);
}
