/* GtkBalls
 * Copyright (C) 1998-1999 Eugene V. Morozov
 * Modifyed in 2001 by drF_ckoff
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
#include <config.h>
#include <gtk/gtk.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>
#include <ctype.h>
#include <fcntl.h>
#include <errno.h>

#include "gtkballs.h"
#include "gtkutils.h"
#include "interface.h"
#include "scoreboard.h"
#include "preferences.h"
#include "gfx.h"

static GtkWidget *dialog 	= NULL;
static gint 	  saved_score 	= 0;
static gchar 	 *user_name 	= NULL;
static gchar 	  last_player_name[15];

gchar *selected_save_load=NULL;

struct point { gint x; gint y; };

struct gtkb_animarray {
	gint color, x, y, phase, time;
};

struct gtkb_rules_dialog {
  	GtkWidget *xrange, *yrange, *crange, *nrange, *drange;
};

struct gtkb_rules_dialog RD;

gint parse_save_game(gchar *sgame, GtkbGameRules *rules, gint *score, gint **board, gint **next) {
  	struct stat buf;
        gchar  *sdata, *psdata;
        FILE   *f;
        gint   i, val;

  	if(stat(sgame,&buf) != 0 || !S_ISREG(buf.st_mode) ||
	   buf.st_size < 20 ||
           (f = fopen(sgame, "r")) == NULL) {
                return 0;
        }

        sdata=g_malloc(buf.st_size);

        if(fread(sdata, 1, buf.st_size, f) != buf.st_size) {
                g_free(sdata);
                return 0;
        }

        fclose(f);

        if(sscanf(sdata, "%02d%02d%02d%02d%02d", &rules->xsize, &rules->ysize, &rules->colors, &rules->next, &rules->destroy) != 5
           || sscanf(sdata + 10, "%010d", score) !=1
           || rules->xsize < 0 || rules->xsize > 99
           || rules->ysize < 0 || rules->ysize > 99
           || rules->colors < 2 || rules->colors > 99
           || rules->next < 2 || rules->next > 99
           || rules->destroy < 3 || rules->destroy > 99
	   || score < 0
	   || buf.st_size != 20 + rules->xsize * rules->ysize * 2 + rules->next * 2) {
                g_free(sdata);
                return 0;
        }

        *board = g_malloc(rules->xsize * rules->ysize * sizeof(gint));
        *next = g_malloc(rules->next * sizeof(gint));

        psdata = sdata + 20;
        for(i=0; i<rules->xsize * rules->ysize; i++, psdata += 2) {
                if(sscanf(psdata, "%02d", &val) != 1 || val < 0 || val > rules->colors) {
                        g_free(*next);
                        g_free(*board);
                	g_free(sdata);
                	return 0;
                }
                (*board)[i] = val;
        }

        for(i=0; i<rules->next; i++, psdata += 2) {
                if(sscanf(psdata, "%02d", &val) != 1 || val < 0 || val > rules->colors) {
                        g_free(*next);
                        g_free(*board);
                	g_free(sdata);
                	return 0;
                }
                (*next)[i] = val;
        }

        g_free(sdata);

        return 1;
}

/* check that save game name is in form YYYY-MM-DD-HHMMSS.sav and have correct content
   return string "DD.MM.YYYY HH:MM:SS (score)" on success, NULL on failure */
gchar *is_valid_save_game(gchar *name, gchar *path) {
        guint  i,y,m,d,h,min,s;
        gint score, *board, *next;
        gchar  *sgame;
        GtkbGameRules rules;

        if((i=sscanf(name, "%04u-%02u-%02u-%02u%02u%02u",&y,&m,&d,&h,&min,&s))!=6 ||
           !m || m > 12 || !d || d > 31 || h > 23 || min > 59 || s > 61 ||
	   (strcmp(name+strlen(name)-4, ".sav")!=0)) {
                return NULL;
        }

        sgame=g_strconcat(path, G_DIR_SEPARATOR_S, name, NULL);

	i = parse_save_game(sgame, &rules, &score, &board, &next);
        g_free(sgame);

	if(!i) return NULL;

        g_free(board);
        g_free(next);

        return g_strdup_printf("%02d.%02d.%04d %02d:%02d:%02d (%d [%dx%d %d %d %d])",
		d, m, y, h, min, s, score, rules.xsize, rules.ysize, rules.colors, rules.next, rules.destroy);
}

/* return list of "data" strung, "file name" pairs... */
gint get_saved_games(gchar ***gamelist) {
  	gchar  *homedir, *datapath;
  	struct stat buf;
  	struct dirent *dir_entry;
  	DIR    *directory;
        gchar  **games=NULL,*game;
        gint   num=0;

  	if(!(homedir=getenv("HOME"))) {
                return -1;
        }

        datapath=g_strconcat(homedir, G_DIR_SEPARATOR_S, ".gtkballs", NULL);
  	if(stat(datapath,&buf) != 0) { /* no ~/.gtkballs */
                if(mkdir(datapath, 0700) != 0) { /* and cannot create it... */
        		g_free(datapath);
                        return -1;
                }
        } else if(!S_ISDIR(buf.st_mode)) { /* ~/.gtkballs is not a directory */
        	g_free(datapath);
                return -1;
        }
        if((directory=opendir(datapath))) {
        	while((dir_entry=readdir(directory))) {
                        if((game=is_valid_save_game(dir_entry->d_name, datapath))!=NULL) {
                                num++;
                                games=g_realloc(games, sizeof(gchar *)*num*2);
                                games[(num-1)*2]=game;
                                games[(num-1)*2+1]=g_strdup_printf("%s/%s", datapath, dir_entry->d_name);
                        }
        	}
        	closedir(directory);
        }
        g_free(datapath);
        *gamelist=games;
        return num;
}

void save_load_game_row_selected(GtkWidget *clist, gint row) {
        selected_save_load=gtk_clist_get_row_data(GTK_CLIST(clist), row);
}

void do_save_game(GtkWidget *widget, gpointer data) {
/* maybe i should do (in feature) some endianess conversion here... */
/* and handle different int sizes on different architectures? =) */
        FILE *f;
        gchar *fname, *errormsg;
        time_t nowtime;
        gchar  ftime[]="0000-00-00-000000";
        gint i;

        nowtime=time(NULL);
        if(selected_save_load) {
                /* TODO: alert stupid user about erasing file... */
                unlink(selected_save_load);
        }
        strftime(ftime, sizeof(ftime), "%Y-%m-%d-%H%M%S", localtime(&nowtime));
        fname=g_strconcat(getenv("HOME"), G_DIR_SEPARATOR_S, ".gtkballs", G_DIR_SEPARATOR_S, ftime, ".sav", NULL);
        if((f = fopen(fname, "w")) != NULL) {
                chmod(fname, 0600);
                /* TODO: check for errors ! */
                fprintf(f, "%02d%02d%02d%02d%02d", Rules.xsize, Rules.ysize, Rules.colors, Rules.next, Rules.destroy);
                fprintf(f, "%010d", Score);
                for(i=0; i<Rules.xsize*Rules.ysize; i++) {
                	fprintf(f, "%02d", Board[i]);
                }
                for(i=0; i<Rules.next; i++) {
                	fprintf(f, "%02d", NextColors[i]);
                }
                fclose(f);
        } else {
                errormsg=g_strdup_printf(_("Cannot save game to:\n%s\n%s"), fname, strerror(errno));
                simple_message_box(errormsg, _("OK"));
                g_free(errormsg);
        }

        g_free(fname);
  	if(data) {
      		gtk_widget_destroy(GTK_WIDGET(data));
  	}
}

void reinit_board(gint *newboard, gint *newnext, gint score, gint oldnext) {
        g_free(Board);
        g_free(BoardUndo);
        g_free(NextColors);
        g_free(NextColorsUndo);

        BoardUndo=g_malloc0(sizeof(gint)*Rules.xsize*Rules.ysize);
        NextColorsUndo=g_malloc(sizeof(gint)*Rules.next);

        if(newboard) {
		Board=newboard;
        } else {
                Board=g_malloc0(sizeof(gint)*Rules.xsize*Rules.ysize);
        }

        if(newnext) {
        	NextColors=newnext;
        } else {
        	NextColors=g_malloc0(sizeof(gint)*Rules.next);
        }

  	if(!read_score(Scoreboard, &FullScoreboard, &FullScoreboardNumber)) {
      		simple_message_box(_("Unable to read score.\n"), _("OK"));
    	}

        Score=score;
        Hi_score=score>Scoreboard[0].score ? score : Scoreboard[0].score;
        remake_board(oldnext, newnext ? 1 : 0);
        if(!newboard) {
        	new_game();
        } else {
        	set_hi_score_label();
        	set_user_score_label();
        	menu_set_sensitive(Menu_items_factory, "/Edit/Undo", FALSE);
  		if(DrawingArea != NULL) {
                	draw_next_balls();
        		draw_board(DrawingArea, Pixmap);
      			gtk_widget_draw(DrawingArea, NULL);
    		}
        }
}

void do_load_game(GtkWidget *widget, gpointer data) {
        gint score, *board, *next, oldnext;
        gchar *errormsg;
        GtkbGameRules rules;

        if(!selected_save_load) {
                simple_message_box(_("No game selected for load.\n"), _("OK"));
  		if(data) {
      			gtk_widget_destroy(GTK_WIDGET(data));
	  	}
                return;
        }

	if(!parse_save_game(selected_save_load, &rules, &score, &board, &next)) {
        	errormsg=g_strdup_printf(_("Cannot load game from:\n%s\n"), selected_save_load);
        	simple_message_box(errormsg, _("OK"));
                g_free(errormsg);
		return;
        }

        if(rules.colors > gtkbTheme->numballs) {
        	errormsg=g_strdup_printf(_("Not enough balls(%d) in current theme.\nWe need %d balls.\nLoad another theme and try again."), gtkbTheme->numballs, rules.colors);
        	simple_message_box(errormsg, _("OK"));
                g_free(errormsg);
		return;
        }

        if(AnimationInProgress) {
      		gtk_timeout_remove(TimerTag);
        	AnimationInProgress=FALSE;
        }

        oldnext=Rules.next;
        memcpy(&Rules, &rules, sizeof(Rules));
	reinit_board(board, next, score, oldnext);

  	if(data) {
      		gtk_widget_destroy(GTK_WIDGET(data));
  	}
}

void save_load_game_list_destroy(gpointer data) {
        if(data) g_free(data);
}

void save_load_game_dialog(GtkWidget *widget, gpointer data, gboolean is_save) {
  	GtkWidget *window, *swindow;
  	GtkWidget *vbox, *button_box;
  	GtkWidget *ok_button, *cancel_button;
  	GtkWidget *clist;
        GtkStyle *style;

        gint      i, num, width=0, w;
        gchar     *titles=_("Date (score)"), *empty=_("Empty");
        gchar     **gamelist;

        num=get_saved_games(&gamelist);
        if(!is_save && !num) {
                simple_message_box(_("No saved games found.\n"), _("OK"));
                return;
        }
  	window=gtk_widget_new(gtk_window_get_type(),
		       	      "GtkObject::user_data", NULL,
			      "GtkWindow::allow_grow", TRUE,
			      "GtkWindow::allow_shrink", TRUE,
			      "GtkContainer::border_width", 5, NULL);
        if(is_save) {
		gtk_window_set_title(GTK_WINDOW(window), _("Save game"));
        	gtk_window_set_wmclass(GTK_WINDOW(window), "GtkBalls_Save", "GtkBalls");
        } else {
		gtk_window_set_title(GTK_WINDOW(window), _("Load game"));
        	gtk_window_set_wmclass(GTK_WINDOW(window), "GtkBalls_Load", "GtkBalls");
        }

  	gtk_signal_connect(GTK_OBJECT(window), "delete_event", GTK_SIGNAL_FUNC(delete_event), NULL);
  	gtk_signal_connect(GTK_OBJECT(window), "destroy", GTK_SIGNAL_FUNC(destroy), window);
  	gtk_signal_connect(GTK_OBJECT(window), "key_press_event", GTK_SIGNAL_FUNC(key_pressed), NULL);

  	vbox=gtk_vbox_new(FALSE, 0);
  	button_box=gtk_hbox_new(TRUE, 0);
  	gtk_container_border_width(GTK_CONTAINER(vbox), 1);

        swindow=gtk_scrolled_window_new(NULL, NULL);
        gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(swindow), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  	gtk_box_pack_start(GTK_BOX(vbox), swindow, FALSE, FALSE, 4);

        clist=gtk_clist_new_with_titles(1, &titles);
        gtk_clist_set_selection_mode(GTK_CLIST(clist), GTK_SELECTION_BROWSE);
        GTK_WIDGET_UNSET_FLAGS(GTK_CLIST(clist)->column[0].button, GTK_CAN_FOCUS);

	style = gtk_widget_get_style(clist);

        for(i=0;i<num;i++) {
                if((w = gdk_string_width(style->font, gamelist[i*2])) > width) {
                        width = w;
                }
        	gtk_clist_append(GTK_CLIST(clist), &gamelist[i*2]);
                g_free(gamelist[i*2]);
                gtk_clist_set_row_data_full(GTK_CLIST(clist), i, g_strdup(gamelist[i*2+1]), save_load_game_list_destroy);
                g_free(gamelist[i*2+1]);
        }
        gtk_clist_set_column_width(GTK_CLIST(clist), 0, width);
        g_free(gamelist);

  	gtk_signal_connect(GTK_OBJECT(clist), "select_row", GTK_SIGNAL_FUNC(save_load_game_row_selected), NULL);

        if(is_save) {
        	gtk_clist_append(GTK_CLIST(clist), &empty);
                gtk_clist_set_row_data(GTK_CLIST(clist), num, NULL);
        	gtk_clist_select_row(GTK_CLIST(clist), num, 0);
        } else {
        	gtk_clist_select_row(GTK_CLIST(clist), 0, 0);
        }
  	gtk_container_add(GTK_CONTAINER(swindow), clist);


        if(is_save) {
  		ok_button=gtk_button_new_with_label(_("Save game"));
  		gtk_signal_connect(GTK_OBJECT(ok_button), "clicked", GTK_SIGNAL_FUNC(do_save_game), window);
        } else {
  		ok_button=gtk_button_new_with_label(_("Load game"));
  		gtk_signal_connect(GTK_OBJECT(ok_button), "clicked", GTK_SIGNAL_FUNC(do_load_game), window);
        }
        GTK_WIDGET_SET_FLAGS(ok_button, GTK_CAN_DEFAULT | GTK_CAN_FOCUS);
  	gtk_box_pack_start(GTK_BOX(button_box), ok_button, TRUE, TRUE, 0);

  	cancel_button=gtk_button_new_with_label(_("Cancel"));
        GTK_WIDGET_SET_FLAGS(cancel_button, GTK_CAN_DEFAULT | GTK_CAN_FOCUS);
  	gtk_signal_connect(GTK_OBJECT(cancel_button), "clicked", GTK_SIGNAL_FUNC(destroy), window);
  	gtk_box_pack_start(GTK_BOX(button_box), cancel_button, TRUE, TRUE, 0);

  	gtk_box_pack_start(GTK_BOX(vbox), button_box, FALSE, FALSE, 4);

  	gtk_container_add(GTK_CONTAINER(window), vbox);

  	gtk_widget_show_all(window);
  	gtk_grab_add(window);
        gtk_widget_grab_default(ok_button);
}

void save_game_cb(GtkWidget *widget, gpointer data) {
        save_load_game_dialog(widget, data, TRUE);
}

void load_game_cb(GtkWidget *widget, gpointer data) {
        save_load_game_dialog(widget, data, FALSE);
}

void show_hide_next_balls(gboolean show) {
        gint i;

  	if(show) {
        	for(i=0;i<Rules.next;i++) {
      			gtk_widget_show(SmallBalls[i]);
                }
    	} else {
        	for(i=0;i<Rules.next;i++) {
      			gtk_widget_hide(SmallBalls[i]);
                }
    	}
}

void init_names_scores_and_dates (GtkWidget **names, GtkWidget **scores, GtkWidget **dates, int pos) {
  	gint i;
  	gchar *str;

  	if(!read_score(Scoreboard, &FullScoreboard, &FullScoreboardNumber)) {
      		simple_message_box(_("Unable to read score.\n"), _("OK"));
    	}

  	for (i=0;i<10;i++) {
      		str=g_strdup_printf(" %s ", Scoreboard[i].name);
      		names[i]=gtk_label_new(str);
                g_free(str);
      		if(!Scoreboard[i].score) {
                        scores[i]=gtk_label_new("");
      		} else {
			str=g_strdup_printf(" %d ", Scoreboard[i].score);
                        scores[i]=gtk_label_new(str);
                	g_free(str);
                }
      		str=g_strdup_printf(" %s ", Scoreboard[i].date);
      		dates[i]=gtk_label_new(str);
                g_free(str);
    	}
}

void show_hall_of_fame(GtkWidget *widget, gpointer data) {
  	GtkWidget *hall_of_fame;
  	GtkWidget *frame;
  	GtkWidget *table;
  	GtkWidget *vbox,*button_box;
  	GtkWidget *names[10],*scores[10],*dates[10];
  	GtkWidget *close_button;
  	gint i;

  	hall_of_fame=gtk_widget_new(gtk_window_get_type(),
				    "GtkObject::user_data", NULL,
				    "GtkWindow::title", _("Hall of Fame"),
				    "GtkWindow::allow_grow", FALSE,
				    "GtkWindow::allow_shrink", FALSE,
				    "GtkContainer::border_width", 5, NULL);
        gtk_window_set_wmclass(GTK_WINDOW(hall_of_fame), "GtkBalls_Scores", "GtkBalls");

  	gtk_signal_connect(GTK_OBJECT(hall_of_fame), "delete_event", GTK_SIGNAL_FUNC(delete_event), NULL);
  	gtk_signal_connect(GTK_OBJECT(hall_of_fame), "destroy", GTK_SIGNAL_FUNC(destroy), hall_of_fame);
  	gtk_signal_connect(GTK_OBJECT(hall_of_fame), "key_press_event", GTK_SIGNAL_FUNC(key_pressed), NULL);

  	frame=gtk_frame_new(_(" Hall of Fame "));
  	vbox=gtk_vbox_new(FALSE, 0);
  	button_box=gtk_hbox_new(TRUE, 0);
  	gtk_container_border_width(GTK_CONTAINER(vbox), 1);
  	gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 0);

  	table=gtk_table_new(9, 3, FALSE);
  	init_names_scores_and_dates(names, scores, dates, (data!=NULL) ? (gint)data : -1);

  	for(i=0;i<10;i++) {
      		gtk_table_attach_defaults(GTK_TABLE(table), names[i],  0, 1, i, i+1);
      		gtk_table_attach_defaults(GTK_TABLE(table), scores[i], 1, 2, i, i+1);
      		gtk_table_attach_defaults(GTK_TABLE(table), dates[i],  2, 3, i, i+1);
    	}

  	gtk_container_add(GTK_CONTAINER(frame), table);

  	close_button=gtk_button_new_with_label(_("Close"));
        GTK_WIDGET_SET_FLAGS(close_button, GTK_CAN_DEFAULT | GTK_CAN_FOCUS);
  	gtk_signal_connect(GTK_OBJECT(close_button), "clicked", GTK_SIGNAL_FUNC(destroy), hall_of_fame);
  	gtk_box_pack_start(GTK_BOX(button_box), close_button, TRUE, TRUE, 0);

  	gtk_box_pack_start(GTK_BOX(vbox), button_box, FALSE, FALSE, 4);

  	gtk_container_add(GTK_CONTAINER(hall_of_fame), vbox);

  	gtk_widget_show_all(hall_of_fame);
  	gtk_grab_add(hall_of_fame);
        gtk_widget_grab_default(close_button);
}

/* shows simple message box */
void simple_message_box(gchar *message, gchar *label) {
  	GtkWidget *dialog;
  	GtkWidget *prompt_label;
  	GtkWidget *button;

  	dialog=gtk_dialog_new();
        gtk_window_set_wmclass(GTK_WINDOW(dialog), "GtkBalls_Dialog", "GtkBalls");
  	gtk_container_border_width(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), 10);
  	gtk_signal_connect(GTK_OBJECT(dialog), "key_press_event", GTK_SIGNAL_FUNC(key_pressed), NULL);

  	prompt_label=gtk_label_new(message);
  	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), prompt_label, TRUE, TRUE, 5);

  	button=gtk_button_new_with_label(label);
  	gtk_signal_connect(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(destroy), dialog);
  	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area), button, TRUE, TRUE, 0);
  	GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
  	gtk_widget_grab_default(button);

  	gtk_widget_show_all(dialog);
  	gtk_grab_add(dialog);
}

/* returns board coordinate of the window position */
gint calculate_board_coord_x(gint window_coord) {
        if(gtkbTheme) {
  		return (window_coord-1)/gtkbTheme->emptycell.xsize;
        }
        return 0;
}

gint calculate_board_coord_y(gint window_coord) {
        if(gtkbTheme) {
  		return (window_coord-1)/gtkbTheme->emptycell.ysize;
        }
        return 0;
}

void update_rectangle(GtkWidget *widget, gint x, gint y, gint w, gint h) {
  	GdkRectangle r;

  	r.x = x;
  	r.y = y;
  	r.width = w;
  	r.height = h;
  	gtk_widget_draw(widget, &r);
}

void set_hi_score_label(void) {
  	gchar *str;

  	str=g_strdup_printf(_("Hi-score: %i"), Hi_score);
  	gtk_label_set(GTK_LABEL(Hi_score_label), str);
        g_free(str);
}

void set_user_score_label(void) {
  	gchar *str;

  	if(Score>Hi_score) {
      		Hi_score=Score;
      		set_hi_score_label();
    	}

  	str=g_strdup_printf(_("Your score: %i"), Score);
  	gtk_label_set(GTK_LABEL(User_score_label), str);
        g_free(str);
}

void add_point_if_first(struct point *balls_to_delete, gint *number_of_balls_to_delete, gint x, gint y) {
  	gint i;

  	for(i=0;i<*number_of_balls_to_delete;i++) {
    		if((balls_to_delete[i].x==x) && (balls_to_delete[i].y==y)) {
      			return;
                }
        }
  	balls_to_delete[*number_of_balls_to_delete].x=x;
  	balls_to_delete[*number_of_balls_to_delete].y=y;
  	(*number_of_balls_to_delete)++;
}

void clear_line(gint x, gint y, gint length, gint direction, struct point *balls_to_delete, gint *number_of_balls_to_delete) {
  	gint i,x1,y1;

  	switch(direction) {
    		case 0:
      			for(i=y-length;i<y;i++) {
				add_point_if_first(balls_to_delete, number_of_balls_to_delete, x, i);
                        }
      			break;
    		case 1:
      			for(i=x-length;i<x;i++) {
				add_point_if_first(balls_to_delete, number_of_balls_to_delete, i, y);
                        }
      			break;
    		case 2:
      			/* save length */
      			i=length;
      			for(x1=x-1,y1=y-1;length>0;x1--,y1--,length--) {
				add_point_if_first(balls_to_delete, number_of_balls_to_delete, x1, y1);
                        }
      			/* restore length */
      			length=i;
      			break;
    		case 3:
      			i=length;
      			for(x1=x-1,y1=y+1;length>0;x1--,y1++,length--) {
				add_point_if_first(balls_to_delete, number_of_balls_to_delete, x1, y1);
                        }
      			length=i;
      			break;
    		case 4:
      			i=length;
      			for(x1=x-1,y1=y-1;length>0;x1--,y1--,length--) {
				add_point_if_first(balls_to_delete, number_of_balls_to_delete, x1, y1);
                        }
      			length=i;
      			break;
    		case 5:
      			i=length;
      			for(x1=x-1,y1=y+1;length>0;x1--,y1++,length--) {
				add_point_if_first(balls_to_delete, number_of_balls_to_delete, x1, y1);
                        }
      			length=i;
      			break;
    	}
}

int animsort(const void *a, const void *b) {
        if(((const struct gtkb_animarray *)a)->time == ((const struct gtkb_animarray *)b)->time) return 0;
        if(((const struct gtkb_animarray *)a)->time > ((const struct gtkb_animarray *)b)->time) return 1;
	return -1;
}

gint destroy_lines(GtkWidget *widget, GdkPixmap *pixmap, gint count_score) {
  	gint x,y,length;
  	gint x1,y1;
  	gint retval=FALSE;
  	gint number_of_balls_to_delete=0;
  	struct point *balls_to_delete;
  	gint i,j;

  	balls_to_delete=g_malloc(Rules.xsize*Rules.ysize*sizeof(struct point));

  	/* test vertical rows */
  	for(x=0;x<Rules.xsize;x++) {
      		for(length=1,y=1;y<Rules.ysize-1;y++) {
			if(Board[x+y*Rules.xsize] && (Board[x+y*Rules.xsize]==Board[x+(y-1)*Rules.xsize])) {
	    			while((Board[x+y*Rules.xsize]==Board[x+(y-1)*Rules.xsize]) && y<Rules.ysize) {
					length++;
					y++;
	      			}
	    			if(length>=Rules.destroy) {
					clear_line(x, y, length, 0, balls_to_delete, &number_of_balls_to_delete);
					retval=TRUE;
	      			}
	    			length=1;
	  		}
                }
    	}

  	/* test horizontal rows */
  	for(y=0;y<Rules.ysize;y++) {
      		for(length=1,x=1;x<Rules.xsize-1;x++) {
			if((Board[x+y*Rules.xsize] && (Board[x+y*Rules.xsize]==Board[x-1+y*Rules.xsize]))) {
	    			while((Board[x+y*Rules.xsize]==Board[x-1+y*Rules.xsize]) && x<Rules.xsize) {
					length++;
					x++;
	      			}
	    			if(length>=Rules.destroy) {
					clear_line(x, y, length, 1, balls_to_delete, &number_of_balls_to_delete);
					retval=TRUE;
	      			}
	    			length = 1;
	  		}
                }
    	}

  	/* test diagonal rows from the left side to the right */
  	for(y=0;y<Rules.ysize;y++) {
      		/* from the top to the bottom */
      		for(length=1,x1=1,y1=y+1;y1<Rules.ysize-1 && x1<Rules.xsize-1;x1++,y1++) {
			if(Board[x1+y1*Rules.xsize] && (Board[x1+y1*Rules.xsize]==Board[x1-1+(y1-1)*Rules.xsize])) {
	    			while((Board[x1+y1*Rules.xsize]==Board[x1-1+(y1-1)*Rules.xsize]) && x1<Rules.xsize && y1<Rules.ysize) {
					length++;
					x1++;
					y1++;
	      			}
	    			if(length>=Rules.destroy) {
					clear_line(x1, y1, length, 2, balls_to_delete, &number_of_balls_to_delete);
					retval=TRUE;
	      			}
	    			length = 1;
	  		}
                }

      		/* from bottom to top */
      		for (length=1,x1=1,y1=y-1;y1>=0 && x1<Rules.xsize;x1++,y1--) {
			if (Board[x1+y1*Rules.xsize] && (Board[x1+y1*Rules.xsize]==Board[x1-1+(y1+1)*Rules.xsize])) {
	    			while((Board[x1+y1*Rules.xsize]==Board[x1-1+(y1+1)*Rules.xsize]) && x1<Rules.xsize && y1>=0) {
					length++;
					x1++;
					y1--;
	      			}
	    			if(length>=Rules.destroy) {
					clear_line(x1, y1, length, 3, balls_to_delete, &number_of_balls_to_delete);
					retval=TRUE;
	      			}
	    			length = 1;
	  		}
                }
    	}

  	/* test diagonal rows from the top to the bottom */
  	for(x=0;x<Rules.xsize;x++) {
      		/* from the top to the bottom */
      		for(length=1,x1=x+1,y1=1;y1<Rules.ysize-1 && x1<Rules.xsize-1;x1++,y1++) {
			if(Board[x1+y1*Rules.xsize] && (Board[x1+y1*Rules.xsize]==Board[x1-1+(y1-1)*Rules.xsize])) {
	    			while((Board[x1+y1*Rules.xsize]==Board[x1-1+(y1-1)*Rules.xsize]) && x1<Rules.xsize && y1<Rules.ysize) {
					length++;
					x1++;
					y1++;
	      			}
	    			if(length>=Rules.destroy) {
					clear_line(x1, y1, length, 4, balls_to_delete, &number_of_balls_to_delete);
					retval=TRUE;
	      			}
	    			length = 1;
	  		}
                }

      		/* from bottom to top */
      		for(length=1,x1=x+1,y1=Rules.ysize-2;y1>=0 && x1<Rules.xsize;x1++,y1--) {
			if(Board[x1+y1*Rules.xsize] && (Board[x1+y1*Rules.xsize] == Board[x1-1+(y1+1)*Rules.xsize])) {
	    			while((Board[x1+y1*Rules.xsize]==Board[x1-1+(y1+1)*Rules.xsize]) && x1<Rules.xsize && y1>=0) {
					length++;
					x1++;
					y1--;
	      			}
	    			if(length>=Rules.destroy) {
					clear_line(x1, y1, length, 5, balls_to_delete, &number_of_balls_to_delete);
					retval=TRUE;
	      			}
	    			length = 1;
	  		}
                }
    	}

  	if(retval) {
                if(Show_destroy) {
                	gint animcadres=0,animpos=0,animtime;
        	        struct gtkb_animarray *animarray;

	                for(i=0;i<number_of_balls_to_delete;i++) {
                        	animcadres+=gtkbTheme->balls[Board[balls_to_delete[i].x+balls_to_delete[i].y*Rules.xsize]-1].destroyphases+1;
                	}
        	        animarray=g_new0(struct gtkb_animarray, animcadres);
	                for(i=0;i<number_of_balls_to_delete;i++) {
        	                gint color=Board[balls_to_delete[i].x+balls_to_delete[i].y*Rules.xsize];

	                        for(j=0,animtime=0;j<gtkbTheme->balls[color-1].destroyphases;j++) {
                                	animarray[animpos].color=color;
                        	        animarray[animpos].x=balls_to_delete[i].x;
                	                animarray[animpos].y=balls_to_delete[i].y;
        	                        animarray[animpos].phase=j;
	                                animarray[animpos].time=animtime;
                                	animtime+=gtkbTheme->balls[color-1].destroydelays[j];
                        	        animpos++;
                	        }
        	                animarray[animpos].color=0;
	                        animarray[animpos].phase=0;
                        	animarray[animpos].x=balls_to_delete[i].x;
                	        animarray[animpos].y=balls_to_delete[i].y;
        	                animarray[animpos].time=animtime;
	                        animtime+=gtkbTheme->balls[color-1].destroydelays[j];
                        	animpos++;
                	}

        	        qsort(animarray, animcadres, sizeof(struct gtkb_animarray), animsort);

	                for(animtime=0,i=0;i<animcadres;) {
                	        gint isav=i;
        	                my_usleep((animarray[i].time-animtime)*1000);
	                	for(;animtime==animarray[i].time && i<animcadres;i++) {
                        		draw_ball(widget, animarray[i].color, Pixmap, animarray[i].x, animarray[i].y, 0, animarray[i].phase+1);
                		}
                	        gdk_flush();
        	                animtime=animarray[isav].time;
	                }
                	g_free(animarray);
                }

      		for(i=0;i<number_of_balls_to_delete;i++) {
			Board[balls_to_delete[i].x+balls_to_delete[i].y*Rules.xsize]=0;
                }

      		draw_board(widget, pixmap);
      		gtk_widget_draw(widget, NULL);

      		if(count_score) {
	  		Score=Score+Rules.destroy*2+2*(number_of_balls_to_delete-Rules.destroy)*(number_of_balls_to_delete-Rules.destroy);
	  		if (!Show_next_colors) {
	    			Score++;
                        }
	  		set_user_score_label();
		}
    	}

  	g_free(balls_to_delete);
  	return retval;
}

void read_entry(GtkWidget *widget, gpointer data) {
  	struct score_board current_entry;
  	time_t current_time;
  	struct tm *timeptr;
  	gint pos;
  
  	user_name=gtk_entry_get_text(GTK_ENTRY(data));
  	if(!strlen(user_name)) {
    		strcpy(current_entry.name, _("Anonymous"));
  	} else {
    		strcpy(current_entry.name, user_name);
        }
  	strcpy(last_player_name, user_name);
  	current_entry.score=saved_score;
  	current_time=time(NULL);
  	timeptr=localtime(&current_time);
  	if(!timeptr) {
      		simple_message_box(_("Unable to determine current date.\n"), _("OK"));
      		current_entry.date[0]='\0';
    	} else {
      		/* I use _() for format string because some people might want
	 	   format of the date more traditional for their locale */
      		if(!strftime(current_entry.date, 30, _("%a %b %d %H:%M %Y"), timeptr)) {
	  		simple_message_box(_("Unable to determine current date.\n"), _("OK"));
	  		current_entry.date[0] = '\0';
		}
    	}
  
  	if(!read_score(Scoreboard, &FullScoreboard, &FullScoreboardNumber)) {
      		simple_message_box (_("Unable to read score.\n"), _("OK"));
    	}
  	pos=insert_entry_in_score_board(Scoreboard, current_entry);
  	if(!write_score(Scoreboard, FullScoreboard, FullScoreboardNumber)) {
      		simple_message_box(_("Unable to save score.\n"), _("OK"));
    	}
  	gtk_grab_remove(dialog);
  	gtk_widget_destroy(dialog);
  	/* show scores to let user see if (s)he's on top ;) */
  	show_hall_of_fame(NULL, (gpointer)pos);
}

gchar *input_name_dialog(void) {
  	GtkWidget *prompt_label;
  	GtkWidget *name;
  	GtkWidget *button;

  	/* we have to save score, because they will be set to 0 in new_game() */
  	saved_score=Score;

  	dialog=gtk_dialog_new();
        gtk_window_set_wmclass(GTK_WINDOW(dialog), "GtkBalls_Inputname", "GtkBalls");
  	gtk_signal_connect(GTK_OBJECT(dialog), "key_press_event", GTK_SIGNAL_FUNC(key_pressed), NULL);

  	gtk_container_border_width(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), 2);

  	prompt_label=gtk_label_new(_("Enter your name"));
  	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), prompt_label, TRUE, TRUE, 0);

  	name=gtk_entry_new_with_max_length(14);
  	gtk_signal_connect(GTK_OBJECT(name), "activate", GTK_SIGNAL_FUNC(read_entry), name);
  	/* restore the last player's name */
  	if(user_name) {
    		gtk_entry_set_text(GTK_ENTRY(name), last_player_name);
    		gtk_entry_select_region(GTK_ENTRY(name), 0, -1);
  	}
  	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), name, TRUE, TRUE, 5);

  	button=gtk_button_new_with_label(_("OK"));
  	gtk_signal_connect(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(read_entry), name);
  	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area), button, TRUE, TRUE, 0);

        gtk_widget_grab_focus(name);

  	GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
  	gtk_widget_grab_default(button);

  	gtk_widget_show_all(dialog);

  	return NULL;
}

void rules_ok(GtkWidget *widget, gpointer data) {
        gint oldnext = Rules.next;

        Rules.xsize = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(RD.xrange));
        Rules.ysize = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(RD.yrange));
        Rules.colors = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(RD.crange));
        Rules.next = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(RD.nrange));
        Rules.destroy = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(RD.drange));

	reinit_board(NULL, NULL, 0, oldnext);

  	if(data) {
      		gtk_widget_destroy(GTK_WIDGET(data));
  	}
}

void rules_cancel(GtkWidget *widget, gpointer data) {
  	if(data) {
      		gtk_widget_destroy(GTK_WIDGET(data));
  	}
}

void rules_classic(GtkWidget *widget, gpointer data) {
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(RD.xrange), ClassicRules.xsize);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(RD.yrange), ClassicRules.ysize);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(RD.crange), ClassicRules.colors);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(RD.nrange), ClassicRules.next);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(RD.drange), ClassicRules.destroy);
}

void rules_dialog(void) {
  	GtkWidget *dialog;
  	GtkWidget *frame;
  	GtkWidget *big_vbox, *vbox, *buttons_box;
  	GtkWidget *separator;
  	GtkWidget *ok_button, *cancel_button, *classic_button;

  	dialog=gtk_window_new(GTK_WINDOW_TOPLEVEL);
  	gtk_window_set_title(GTK_WINDOW(dialog), _("Game rules"));
        gtk_window_set_wmclass(GTK_WINDOW(dialog), "GtkBalls_Rules", "GtkBalls");
  	gtk_signal_connect(GTK_OBJECT(dialog), "key_press_event", GTK_SIGNAL_FUNC(key_pressed), NULL);
  	gtk_container_border_width(GTK_CONTAINER(dialog), 5);

  	big_vbox=gtk_vbox_new(FALSE, 0);
  	gtk_container_add(GTK_CONTAINER(dialog), big_vbox);

  	frame=gtk_frame_new(_("Game rules"));
  	gtk_box_pack_start(GTK_BOX(big_vbox), frame, FALSE, FALSE, 0);

	vbox=gtk_vbox_new(FALSE, 0);
  	gtk_container_add(GTK_CONTAINER(frame), vbox);

	RD.xrange=ut_spin_button_new(_("Board width"), 4, 99, Rules.xsize, vbox);
	RD.yrange=ut_spin_button_new(_("Board height"), 4, 99, Rules.ysize, vbox);
	RD.crange=ut_spin_button_new(_("Number of different objects"), 3, gtkbTheme->numballs, Rules.colors, vbox);
	RD.nrange=ut_spin_button_new(_("Number of 'next' objects"), 2, 99, Rules.next, vbox);
	RD.drange=ut_spin_button_new(_("How many balls at line eliminate it"), 3, 99, Rules.destroy, vbox);

	classic_button=ut_button_new(_("Classic rules"), rules_classic, GTK_OBJECT(dialog), vbox);

  	separator=gtk_hseparator_new();
  	gtk_box_pack_start(GTK_BOX(big_vbox), separator, FALSE, FALSE, 5);

  	buttons_box=gtk_hbutton_box_new();
        gtk_button_box_set_layout(GTK_BUTTON_BOX(buttons_box), GTK_BUTTONBOX_SPREAD);
  	gtk_box_pack_start(GTK_BOX(big_vbox), buttons_box, TRUE, TRUE, 0);

	ok_button=ut_button_new(_("OK"), GTK_SIGNAL_FUNC(rules_ok), GTK_OBJECT(dialog), buttons_box);
	cancel_button=ut_button_new(_("Cancel"), GTK_SIGNAL_FUNC(rules_cancel), GTK_OBJECT(dialog), buttons_box);

  	gtk_widget_show_all(dialog);
        gtk_widget_grab_default(ok_button);
  	gtk_grab_add(dialog);
}
