/* GtkamlSAXParser.vala
 * 
 * Copyright (C) 2008 Vlad Grecescu
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with main.c; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 *
 * Author:
 *        Vlad Grecescu (b100dian@gmail.com)
 */

#include "GtkamlSAXParser.h"
#include <gee/hashmap.h>
#include <gee/map.h>
#include <glib/gstdio.h>
#include <vala/valareport.h>
#include <gee/arraylist.h>
#include <gee/collection.h>
#include <config.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <vala/valausingdirective.h>
#include <vala/valaunresolvedsymbol.h>
#include <vala/valasymbol.h>
#include <vala/valanamespace.h>
#include <vala/valascope.h>
#include "GtkamlStateStack.h"
#include "GtkamlState.h"
#include "GtkamlAttribute.h"
#include "GtkamlXmlAttribute.h"
#include "GtkamlXmlNamespace.h"
#include "GtkamlSimpleAttribute.h"
#include "GtkamlReferenceClassDefinition.h"
#include "GtkamlComplexAttribute.h"




static char* string_substring (const char* self, glong offset, glong len);
struct _GtkamlSAXParserPrivate {
	ValaCodeContext* _context;
	ValaSourceFile* _source_file;
	GtkamlStateStack* _states;
	GeeMap* generated_identifiers_counter;
	GeeMap* _prefixes_namespaces;
	GtkamlRootClassDefinition* _root_class_definition;
};

#define GTKAML_SAX_PARSER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GTKAML_TYPE_SAX_PARSER, GtkamlSAXParserPrivate))
enum  {
	GTKAML_SAX_PARSER_DUMMY_PROPERTY,
	GTKAML_SAX_PARSER_CONTEXT,
	GTKAML_SAX_PARSER_SOURCE_FILE
};
static GtkamlRootClassDefinition* gtkaml_sax_parser_real_parse (GtkamlSAXParser* self);
static void gtkaml_sax_parser_parse_attribute_content_as_text (GtkamlSAXParser* self, GtkamlState* state, const char* content);
static char* gtkaml_sax_parser_prefix_to_namespace (GtkamlSAXParser* self, const char* prefix);
static ValaClass* gtkaml_sax_parser_lookup_class (GtkamlSAXParser* self, const char* xmlNamespace, const char* name);
static char* gtkaml_sax_parser_strip_attribute_hyphens (GtkamlSAXParser* self, const char* attrname);
static GeeList* gtkaml_sax_parser_parse_attributes (GtkamlSAXParser* self, char** attributes, gint nb_attributes);
static GeeList* gtkaml_sax_parser_parse_namespaces (GtkamlSAXParser* self, char** namespaces, gint nb_namespaces);
static void gtkaml_sax_parser_set_context (GtkamlSAXParser* self, ValaCodeContext* value);
static void gtkaml_sax_parser_set_source_file (GtkamlSAXParser* self, ValaSourceFile* value);
static GtkamlStateStack* gtkaml_sax_parser_get_states (GtkamlSAXParser* self);
static void gtkaml_sax_parser_set_states (GtkamlSAXParser* self, GtkamlStateStack* value);
static GeeMap* gtkaml_sax_parser_get_prefixes_namespaces (GtkamlSAXParser* self);
static void gtkaml_sax_parser_set_prefixes_namespaces (GtkamlSAXParser* self, GeeMap* value);
static GtkamlRootClassDefinition* gtkaml_sax_parser_get_root_class_definition (GtkamlSAXParser* self);
static void gtkaml_sax_parser_set_root_class_definition (GtkamlSAXParser* self, GtkamlRootClassDefinition* value);
static GObject * gtkaml_sax_parser_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties);
static gpointer gtkaml_sax_parser_parent_class = NULL;
static void gtkaml_sax_parser_finalize (GObject* obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static gint _vala_array_length (gpointer array);
static int _vala_strcmp0 (const char * str1, const char * str2);



static char* string_substring (const char* self, glong offset, glong len) {
	glong string_length;
	const char* start;
	g_return_val_if_fail (self != NULL, NULL);
	string_length = g_utf8_strlen (self, -1);
	if (offset < 0) {
		offset = string_length + offset;
		g_return_val_if_fail (offset >= 0, NULL);
	} else {
		g_return_val_if_fail (offset <= string_length, NULL);
	}
	if (len < 0) {
		len = string_length - offset;
	}
	g_return_val_if_fail ((offset + len) <= string_length, NULL);
	start = g_utf8_offset_to_pointer (self, offset);
	return g_strndup (start, ((gchar*) g_utf8_offset_to_pointer (start, len)) - ((gchar*) start));
}


GtkamlSAXParser* gtkaml_sax_parser_construct (GType object_type, ValaCodeContext* context, ValaSourceFile* source_file) {
	GtkamlSAXParser * self;
	g_return_val_if_fail (context != NULL, NULL);
	g_return_val_if_fail (source_file != NULL, NULL);
	self = g_object_newv (object_type, 0, NULL);
	gtkaml_sax_parser_set_context (self, context);
	gtkaml_sax_parser_set_source_file (self, source_file);
	return self;
}


GtkamlSAXParser* gtkaml_sax_parser_new (ValaCodeContext* context, ValaSourceFile* source_file) {
	return gtkaml_sax_parser_construct (GTKAML_TYPE_SAX_PARSER, context, source_file);
}


static GtkamlRootClassDefinition* gtkaml_sax_parser_real_parse (GtkamlSAXParser* self) {
	GError * inner_error;
	char* contents;
	gulong length;
	GtkamlState* initial_state;
	GtkamlRootClassDefinition* _tmp3;
	GtkamlRootClassDefinition* _tmp4;
	g_return_val_if_fail (self != NULL, NULL);
	inner_error = NULL;
	contents = NULL;
	length = 0UL;
	{
		char* _tmp2;
		gboolean _tmp1;
		char* _tmp0;
		_tmp2 = NULL;
		_tmp0 = NULL;
		_tmp1 = g_file_get_contents (vala_source_file_get_filename (self->priv->_source_file), &_tmp0, &length, &inner_error);
		contents = (_tmp2 = _tmp0, contents = (g_free (contents), NULL), _tmp2);
		_tmp1;
		if (inner_error != NULL) {
			if (inner_error->domain == G_FILE_ERROR) {
				goto __catch2_g_file_error;
			}
			goto __finally2;
		}
	}
	goto __finally2;
	__catch2_g_file_error:
	{
		GError * e;
		e = inner_error;
		inner_error = NULL;
		{
			vala_report_error (NULL, e->message);
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
		}
	}
	__finally2:
	if (inner_error != NULL) {
		contents = (g_free (contents), NULL);
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
		g_clear_error (&inner_error);
		return NULL;
	}
	initial_state = gtkaml_state_new (GTKAML_STATE_ID_SAX_PARSER_INITIAL_STATE, NULL, NULL, NULL);
	gtkaml_state_stack_push (self->priv->_states, initial_state);
	gtkaml_sax_parser_start_parsing (self, contents, length);
	_tmp3 = NULL;
	_tmp4 = NULL;
	return (_tmp4 = (_tmp3 = self->priv->_root_class_definition, (_tmp3 == NULL) ? NULL : g_object_ref (_tmp3)), contents = (g_free (contents), NULL), (initial_state == NULL) ? NULL : (initial_state = (g_object_unref (initial_state), NULL)), _tmp4);
}


GtkamlRootClassDefinition* gtkaml_sax_parser_parse (GtkamlSAXParser* self) {
	return GTKAML_SAX_PARSER_GET_CLASS (self)->parse (self);
}


void gtkaml_sax_parser_start_element (GtkamlSAXParser* self, const char* localname, const char* prefix, const char* URI, gint nb_namespaces, char** namespaces, gint nb_attributes, gint nb_defaulted, char** attributes) {
	GeeList* attrs;
	GtkamlState* state;
	ValaSourceReference* source_reference;
	g_return_if_fail (self != NULL);
	g_return_if_fail (localname != NULL);
	g_return_if_fail (URI != NULL);
	attrs = gtkaml_sax_parser_parse_attributes (self, attributes, nb_attributes);
	state = gtkaml_state_stack_peek (self->priv->_states, 0);
	source_reference = gtkaml_sax_parser_create_source_reference (self);
	switch (gtkaml_state_get_state_id (state)) {
		case GTKAML_STATE_ID_SAX_PARSER_INITIAL_STATE:
		{
			{
				GeeList* nss;
				char* _tmp5;
				ValaClass* _tmp6;
				ValaClass* clazz;
				GtkamlRootClassDefinition* _tmp8;
				GtkamlState* _tmp9;
				/*Frist Tag! - that means, add "using" directives first*/
				nss = gtkaml_sax_parser_parse_namespaces (self, namespaces, nb_namespaces);
				{
					GeeIterator* _ns_it;
					_ns_it = gee_iterable_iterator ((GeeIterable*) nss);
					while (gee_iterator_next (_ns_it)) {
						XmlNamespace* ns;
						gboolean _tmp0;
						ns = (XmlNamespace*) gee_iterator_get (_ns_it);
						_tmp0 = FALSE;
						if (NULL == ns->prefix) {
							_tmp0 = TRUE;
						} else {
							gboolean _tmp1;
							_tmp1 = FALSE;
							if (NULL != ns->prefix) {
								_tmp1 = _vala_strcmp0 (ns->prefix, self->gtkaml_prefix) != 0;
							} else {
								_tmp1 = FALSE;
							}
							_tmp0 = _tmp1;
						}
						if (_tmp0) {
							char** _tmp2;
							gint uri_definition_size;
							gint uri_definition_length1;
							char** uri_definition;
							ValaUnresolvedSymbol* _tmp3;
							ValaUsingDirective* _tmp4;
							ValaUsingDirective* namespace_reference;
							_tmp2 = NULL;
							uri_definition = (_tmp2 = g_strsplit_set (ns->URI, ":", 0), uri_definition_length1 = -1, uri_definition_size = uri_definition_length1, _tmp2);
							_tmp3 = NULL;
							_tmp4 = NULL;
							namespace_reference = (_tmp4 = vala_using_directive_new ((ValaSymbol*) (_tmp3 = vala_unresolved_symbol_new (NULL, "GLib", NULL)), source_reference), (_tmp3 == NULL) ? NULL : (_tmp3 = (vala_code_node_unref (_tmp3), NULL)), _tmp4);
							vala_source_file_add_using_directive (self->priv->_source_file, namespace_reference);
							if (NULL == ns->prefix) {
								/*stderr.printf ("adding '%s':'%s' to prefixes_namespaces map\n", "", uri_definition[0]);*/
								gee_map_set (self->priv->_prefixes_namespaces, "", uri_definition[0]);
							} else {
								/*stderr.printf ("adding '%s':'%s' to prefixes_namespaces map\n", ns.prefix, uri_definition[0]);*/
								gee_map_set (self->priv->_prefixes_namespaces, ns->prefix, uri_definition[0]);
							}
							uri_definition = (_vala_array_free (uri_definition, uri_definition_length1, (GDestroyNotify) g_free), NULL);
							(namespace_reference == NULL) ? NULL : (namespace_reference = (vala_code_node_unref (namespace_reference), NULL));
						}
						(ns == NULL) ? NULL : (ns = (g_object_unref (ns), NULL));
					}
					(_ns_it == NULL) ? NULL : (_ns_it = (gee_collection_object_unref (_ns_it), NULL));
				}
				_tmp5 = NULL;
				_tmp6 = NULL;
				clazz = (_tmp6 = gtkaml_sax_parser_lookup_class (self, _tmp5 = gtkaml_sax_parser_prefix_to_namespace (self, prefix), localname), _tmp5 = (g_free (_tmp5), NULL), _tmp6);
				if (clazz == NULL) {
					char* _tmp7;
					_tmp7 = NULL;
					vala_report_error (source_reference, _tmp7 = g_strdup_printf ("%s not a class", localname));
					_tmp7 = (g_free (_tmp7), NULL);
					gtkaml_sax_parser_stop_parsing (self);
					(nss == NULL) ? NULL : (nss = (gee_collection_object_unref (nss), NULL));
					(clazz == NULL) ? NULL : (clazz = (vala_code_node_unref (clazz), NULL));
					(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
					(state == NULL) ? NULL : (state = (g_object_unref (state), NULL));
					(source_reference == NULL) ? NULL : (source_reference = (vala_source_reference_unref (source_reference), NULL));
					return;
				}
				_tmp8 = NULL;
				gtkaml_sax_parser_set_root_class_definition (self, _tmp8 = gtkaml_sax_parser_get_root_definition (self, clazz, attrs, prefix));
				(_tmp8 == NULL) ? NULL : (_tmp8 = (g_object_unref (_tmp8), NULL));
				_tmp9 = NULL;
				gtkaml_state_stack_push (self->priv->_states, _tmp9 = gtkaml_state_new (GTKAML_STATE_ID_SAX_PARSER_CONTAINER_STATE, (GtkamlClassDefinition*) self->priv->_root_class_definition, NULL, NULL));
				(_tmp9 == NULL) ? NULL : (_tmp9 = (g_object_unref (_tmp9), NULL));
				(nss == NULL) ? NULL : (nss = (gee_collection_object_unref (nss), NULL));
				(clazz == NULL) ? NULL : (clazz = (vala_code_node_unref (clazz), NULL));
				break;
			}
		}
		case GTKAML_STATE_ID_SAX_PARSER_CONTAINER_STATE:
		{
			{
				char* _tmp10;
				ValaClass* _tmp11;
				ValaClass* clazz;
				char* fqan;
				_tmp10 = NULL;
				_tmp11 = NULL;
				clazz = (_tmp11 = gtkaml_sax_parser_lookup_class (self, _tmp10 = gtkaml_sax_parser_prefix_to_namespace (self, prefix), localname), _tmp10 = (g_free (_tmp10), NULL), _tmp11);
				fqan = NULL;
				if (clazz != NULL) {
					GtkamlClassDefinition* class_definition;
					GtkamlState* _tmp12;
					/*this is a member/container child object*/
					class_definition = gtkaml_sax_parser_get_child_for_container (self, clazz, gtkaml_state_get_class_definition (state), attrs, prefix);
					_tmp12 = NULL;
					gtkaml_state_stack_push (self->priv->_states, _tmp12 = gtkaml_state_new (GTKAML_STATE_ID_SAX_PARSER_CONTAINER_STATE, class_definition, NULL, NULL));
					(_tmp12 == NULL) ? NULL : (_tmp12 = (g_object_unref (_tmp12), NULL));
					(class_definition == NULL) ? NULL : (class_definition = (g_object_unref (class_definition), NULL));
				} else {
					GtkamlClassDefinition* _tmp13;
					GtkamlClassDefinition* attribute_parent_class_definition;
					GtkamlState* _tmp19;
					/*no class with this name found, assume it's an attribute*/
					_tmp13 = NULL;
					attribute_parent_class_definition = (_tmp13 = gtkaml_state_get_class_definition (state), (_tmp13 == NULL) ? NULL : g_object_ref (_tmp13));
					if (prefix != NULL) {
						char* _tmp15;
						char* _tmp14;
						_tmp15 = NULL;
						_tmp14 = NULL;
						fqan = (_tmp15 = g_strconcat (_tmp14 = g_strconcat (prefix, ".", NULL), localname, NULL), fqan = (g_free (fqan), NULL), _tmp15);
						_tmp14 = (g_free (_tmp14), NULL);
					} else {
						char* _tmp17;
						const char* _tmp16;
						_tmp17 = NULL;
						_tmp16 = NULL;
						fqan = (_tmp17 = (_tmp16 = localname, (_tmp16 == NULL) ? NULL : g_strdup (_tmp16)), fqan = (g_free (fqan), NULL), _tmp17);
					}
					if (gee_collection_get_size ((GeeCollection*) attrs) > 0) {
						char* _tmp18;
						/*an attribute cannot have.. attributes*/
						_tmp18 = NULL;
						vala_report_error (source_reference, _tmp18 = g_strdup_printf ("No class %s found.", fqan));
						_tmp18 = (g_free (_tmp18), NULL);
					}
					_tmp19 = NULL;
					gtkaml_state_stack_push (self->priv->_states, _tmp19 = gtkaml_state_new (GTKAML_STATE_ID_SAX_PARSER_ATTRIBUTE_STATE, attribute_parent_class_definition, NULL, fqan));
					(_tmp19 == NULL) ? NULL : (_tmp19 = (g_object_unref (_tmp19), NULL));
					(attribute_parent_class_definition == NULL) ? NULL : (attribute_parent_class_definition = (g_object_unref (attribute_parent_class_definition), NULL));
				}
				(clazz == NULL) ? NULL : (clazz = (vala_code_node_unref (clazz), NULL));
				fqan = (g_free (fqan), NULL);
				break;
			}
		}
		case GTKAML_STATE_ID_SAX_PARSER_ATTRIBUTE_STATE:
		{
			{
				char* _tmp21;
				ValaClass* _tmp22;
				ValaClass* clazz;
				GtkamlClassDefinition* attribute_value_definition;
				char* _tmp25;
				GtkamlComplexAttribute* _tmp26;
				GtkamlComplexAttribute* attr;
				GtkamlState* _tmp27;
				/*a tag found within an attribute state switches us to container_state*/
				if (gtkaml_state_get_attribute (state) != NULL) {
					char* _tmp20;
					/*this was created by non-discardable text nodes*/
					_tmp20 = NULL;
					vala_report_error (source_reference, _tmp20 = g_strdup_printf ("Incorrect attribute definition for %s", gtkaml_state_get_attribute_name (state)));
					_tmp20 = (g_free (_tmp20), NULL);
					gtkaml_sax_parser_stop_parsing (self);
					(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
					(state == NULL) ? NULL : (state = (g_object_unref (state), NULL));
					(source_reference == NULL) ? NULL : (source_reference = (vala_source_reference_unref (source_reference), NULL));
					return;
				}
				_tmp21 = NULL;
				_tmp22 = NULL;
				clazz = (_tmp22 = gtkaml_sax_parser_lookup_class (self, _tmp21 = gtkaml_sax_parser_prefix_to_namespace (self, prefix), localname), _tmp21 = (g_free (_tmp21), NULL), _tmp22);
				attribute_value_definition = NULL;
				if (clazz != NULL) {
					GtkamlClassDefinition* _tmp23;
					/*this is a member/container child object*/
					_tmp23 = NULL;
					attribute_value_definition = (_tmp23 = gtkaml_sax_parser_get_child_for_container (self, clazz, NULL, attrs, prefix), (attribute_value_definition == NULL) ? NULL : (attribute_value_definition = (g_object_unref (attribute_value_definition), NULL)), _tmp23);
				} else {
					char* _tmp24;
					_tmp24 = NULL;
					vala_report_error (source_reference, _tmp24 = g_strdup_printf ("No class '%s' found", localname));
					_tmp24 = (g_free (_tmp24), NULL);
					gtkaml_sax_parser_stop_parsing (self);
					(clazz == NULL) ? NULL : (clazz = (vala_code_node_unref (clazz), NULL));
					(attribute_value_definition == NULL) ? NULL : (attribute_value_definition = (g_object_unref (attribute_value_definition), NULL));
					(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
					(state == NULL) ? NULL : (state = (g_object_unref (state), NULL));
					(source_reference == NULL) ? NULL : (source_reference = (vala_source_reference_unref (source_reference), NULL));
					return;
				}
				_tmp25 = NULL;
				_tmp26 = NULL;
				attr = (_tmp26 = gtkaml_complex_attribute_new (_tmp25 = gtkaml_sax_parser_strip_attribute_hyphens (self, gtkaml_state_get_attribute_name (state)), attribute_value_definition), _tmp25 = (g_free (_tmp25), NULL), _tmp26);
				gtkaml_state_set_attribute (state, (GtkamlAttribute*) attr);
				/*add the attribute into the parent container*/
				gtkaml_class_definition_add_attribute (gtkaml_state_get_class_definition (state), (GtkamlAttribute*) attr);
				_tmp27 = NULL;
				gtkaml_state_stack_push (self->priv->_states, _tmp27 = gtkaml_state_new (GTKAML_STATE_ID_SAX_PARSER_CONTAINER_STATE, attribute_value_definition, NULL, NULL));
				(_tmp27 == NULL) ? NULL : (_tmp27 = (g_object_unref (_tmp27), NULL));
				(clazz == NULL) ? NULL : (clazz = (vala_code_node_unref (clazz), NULL));
				(attribute_value_definition == NULL) ? NULL : (attribute_value_definition = (g_object_unref (attribute_value_definition), NULL));
				(attr == NULL) ? NULL : (attr = (g_object_unref (attr), NULL));
				break;
			}
		}
		default:
		{
			vala_report_error (source_reference, "Invalid Gtkaml SAX Parser state");
			gtkaml_sax_parser_stop_parsing (self);
			(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
			(state == NULL) ? NULL : (state = (g_object_unref (state), NULL));
			(source_reference == NULL) ? NULL : (source_reference = (vala_source_reference_unref (source_reference), NULL));
			return;
		}
	}
	(attrs == NULL) ? NULL : (attrs = (gee_collection_object_unref (attrs), NULL));
	(state == NULL) ? NULL : (state = (g_object_unref (state), NULL));
	(source_reference == NULL) ? NULL : (source_reference = (vala_source_reference_unref (source_reference), NULL));
}


void gtkaml_sax_parser_characters (GtkamlSAXParser* self, const char* data, gint len) {
	GtkamlState* state;
	char* value;
	const char* _tmp0;
	char* stripped_value;
	g_return_if_fail (self != NULL);
	g_return_if_fail (data != NULL);
	state = gtkaml_state_stack_peek (self->priv->_states, 0);
	value = g_strndup (data, (gsize) len);
	_tmp0 = NULL;
	stripped_value = (_tmp0 = value, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0));
	g_strstrip (stripped_value);
	if (_vala_strcmp0 (stripped_value, "") != 0) {
		gtkaml_sax_parser_parse_attribute_content_as_text (self, state, value);
	}
	(state == NULL) ? NULL : (state = (g_object_unref (state), NULL));
	value = (g_free (value), NULL);
	stripped_value = (g_free (stripped_value), NULL);
}


void gtkaml_sax_parser_end_element (GtkamlSAXParser* self, const char* localname, const char* prefix, const char* URI) {
	GtkamlState* last_state;
	gboolean _tmp0;
	gboolean _tmp1;
	gboolean _tmp2;
	gboolean _tmp3;
	g_return_if_fail (self != NULL);
	g_return_if_fail (localname != NULL);
	g_return_if_fail (URI != NULL);
	last_state = gtkaml_state_stack_pop (self->priv->_states);
	_tmp0 = FALSE;
	_tmp1 = FALSE;
	_tmp2 = FALSE;
	_tmp3 = FALSE;
	if (last_state != NULL) {
		_tmp3 = gtkaml_state_get_state_id (last_state) == GTKAML_STATE_ID_SAX_PARSER_ATTRIBUTE_STATE;
	} else {
		_tmp3 = FALSE;
	}
	if (_tmp3) {
		_tmp2 = gtkaml_state_get_attribute (last_state) == NULL;
	} else {
		_tmp2 = FALSE;
	}
	if (_tmp2) {
		char* _tmp4;
		_tmp4 = NULL;
		_tmp1 = _vala_strcmp0 (gtkaml_state_get_attribute_name (last_state), _tmp4 = g_strconcat (self->gtkaml_prefix, ".preconstruct", NULL)) != 0;
		_tmp4 = (g_free (_tmp4), NULL);
	} else {
		_tmp1 = FALSE;
	}
	if (_tmp1) {
		char* _tmp5;
		_tmp5 = NULL;
		_tmp0 = _vala_strcmp0 (gtkaml_state_get_attribute_name (last_state), _tmp5 = g_strconcat (self->gtkaml_prefix, ".construct", NULL)) != 0;
		_tmp5 = (g_free (_tmp5), NULL);
	} else {
		_tmp0 = FALSE;
	}
	/*check if we were in 'attribute' state but no sub-tags or no text/cdata blocks were encountered*/
	if (_tmp0) {
		char* _tmp8;
		char* _tmp7;
		ValaSourceReference* _tmp6;
		_tmp8 = NULL;
		_tmp7 = NULL;
		_tmp6 = NULL;
		vala_report_error (_tmp6 = gtkaml_sax_parser_create_source_reference (self), _tmp8 = g_strdup_printf ("%s is not well defined or is not an attribute of %s", gtkaml_state_get_attribute_name (last_state), _tmp7 = gtkaml_class_definition_get_base_full_name (gtkaml_state_get_class_definition (last_state))));
		_tmp8 = (g_free (_tmp8), NULL);
		_tmp7 = (g_free (_tmp7), NULL);
		(_tmp6 == NULL) ? NULL : (_tmp6 = (vala_source_reference_unref (_tmp6), NULL));
		gtkaml_sax_parser_stop_parsing (self);
	}
	(last_state == NULL) ? NULL : (last_state = (g_object_unref (last_state), NULL));
}


void gtkaml_sax_parser_cdata_block (GtkamlSAXParser* self, const char* cdata, gint len) {
	GtkamlState* state;
	g_return_if_fail (self != NULL);
	g_return_if_fail (cdata != NULL);
	state = gtkaml_state_stack_peek (self->priv->_states, 0);
	if (gtkaml_state_get_state_id (state) != GTKAML_STATE_ID_SAX_PARSER_INITIAL_STATE) {
		GtkamlState* previous_state;
		previous_state = gtkaml_state_stack_peek (self->priv->_states, 1);
		if (gtkaml_state_get_state_id (previous_state) == GTKAML_STATE_ID_SAX_PARSER_INITIAL_STATE) {
			GtkamlRootClassDefinition* _tmp1;
			GtkamlClassDefinition* _tmp0;
			GtkamlRootClassDefinition* root_class;
			char* _tmp2;
			_tmp1 = NULL;
			_tmp0 = NULL;
			root_class = (_tmp1 = (_tmp0 = gtkaml_state_get_class_definition (state), GTKAML_IS_ROOT_CLASS_DEFINITION (_tmp0) ? ((GtkamlRootClassDefinition*) _tmp0) : NULL), (_tmp1 == NULL) ? NULL : g_object_ref (_tmp1));
			if (gtkaml_root_class_definition_get_original_first_code_line (root_class) < 0) {
				gtkaml_root_class_definition_set_original_first_code_line (root_class, gtkaml_sax_parser_line_number (self));
			}
			_tmp2 = NULL;
			gee_collection_add ((GeeCollection*) gtkaml_root_class_definition_get_code (root_class), _tmp2 = g_strndup (cdata, (gsize) len));
			_tmp2 = (g_free (_tmp2), NULL);
			(root_class == NULL) ? NULL : (root_class = (g_object_unref (root_class), NULL));
		} else {
			char* _tmp3;
			_tmp3 = NULL;
			gtkaml_sax_parser_parse_attribute_content_as_text (self, state, _tmp3 = g_strndup (cdata, (gsize) len));
			_tmp3 = (g_free (_tmp3), NULL);
		}
		(previous_state == NULL) ? NULL : (previous_state = (g_object_unref (previous_state), NULL));
	}
	(state == NULL) ? NULL : (state = (g_object_unref (state), NULL));
}


static void gtkaml_sax_parser_parse_attribute_content_as_text (GtkamlSAXParser* self, GtkamlState* state, const char* content) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (state != NULL);
	g_return_if_fail (content != NULL);
	if (gtkaml_state_get_state_id (state) == GTKAML_STATE_ID_SAX_PARSER_ATTRIBUTE_STATE) {
		char* _tmp0;
		gboolean _tmp1;
		_tmp0 = NULL;
		if ((_tmp1 = _vala_strcmp0 (gtkaml_state_get_attribute_name (state), _tmp0 = g_strconcat (self->gtkaml_prefix, ".preconstruct", NULL)) == 0, _tmp0 = (g_free (_tmp0), NULL), _tmp1)) {
			if (gtkaml_class_definition_get_preconstruct_code (gtkaml_state_get_class_definition (state)) != NULL) {
				char* _tmp3;
				ValaSourceReference* _tmp2;
				_tmp3 = NULL;
				_tmp2 = NULL;
				vala_report_error (_tmp2 = gtkaml_sax_parser_create_source_reference (self), _tmp3 = g_strdup_printf ("A preconstruct attribute already exists for %s", gtkaml_class_definition_get_identifier (gtkaml_state_get_class_definition (state))));
				_tmp3 = (g_free (_tmp3), NULL);
				(_tmp2 == NULL) ? NULL : (_tmp2 = (vala_source_reference_unref (_tmp2), NULL));
				gtkaml_sax_parser_stop_parsing (self);
				return;
			}
			gtkaml_class_definition_set_preconstruct_code (gtkaml_state_get_class_definition (state), content);
		} else {
			char* _tmp4;
			gboolean _tmp5;
			_tmp4 = NULL;
			if ((_tmp5 = _vala_strcmp0 (gtkaml_state_get_attribute_name (state), _tmp4 = g_strconcat (self->gtkaml_prefix, ".construct", NULL)) == 0, _tmp4 = (g_free (_tmp4), NULL), _tmp5)) {
				if (gtkaml_class_definition_get_construct_code (gtkaml_state_get_class_definition (state)) != NULL) {
					char* _tmp7;
					ValaSourceReference* _tmp6;
					_tmp7 = NULL;
					_tmp6 = NULL;
					vala_report_error (_tmp6 = gtkaml_sax_parser_create_source_reference (self), _tmp7 = g_strdup_printf ("A construct attribute already exists for %s", gtkaml_class_definition_get_identifier (gtkaml_state_get_class_definition (state))));
					_tmp7 = (g_free (_tmp7), NULL);
					(_tmp6 == NULL) ? NULL : (_tmp6 = (vala_source_reference_unref (_tmp6), NULL));
					gtkaml_sax_parser_stop_parsing (self);
					return;
				}
				gtkaml_class_definition_set_construct_code (gtkaml_state_get_class_definition (state), content);
			} else {
				if (gtkaml_state_get_attribute (state) == NULL) {
					GtkamlSimpleAttribute* _tmp9;
					char* _tmp8;
					_tmp9 = NULL;
					_tmp8 = NULL;
					gtkaml_state_set_attribute (state, (GtkamlAttribute*) (_tmp9 = gtkaml_simple_attribute_new (_tmp8 = gtkaml_sax_parser_strip_attribute_hyphens (self, gtkaml_state_get_attribute_name (state)), content)));
					(_tmp9 == NULL) ? NULL : (_tmp9 = (g_object_unref (_tmp9), NULL));
					_tmp8 = (g_free (_tmp8), NULL);
					gtkaml_class_definition_add_attribute (gtkaml_state_get_class_definition (state), gtkaml_state_get_attribute (state));
				} else {
					if (GTKAML_IS_SIMPLE_ATTRIBUTE (gtkaml_state_get_attribute (state))) {
						char* _tmp12;
						char* _tmp11;
						GtkamlAttribute* _tmp10;
						_tmp12 = NULL;
						_tmp11 = NULL;
						_tmp10 = NULL;
						gtkaml_simple_attribute_set_value ((_tmp10 = gtkaml_state_get_attribute (state), GTKAML_IS_SIMPLE_ATTRIBUTE (_tmp10) ? ((GtkamlSimpleAttribute*) _tmp10) : NULL), _tmp12 = g_strconcat (gtkaml_simple_attribute_get_value ((_tmp10 = gtkaml_state_get_attribute (state), GTKAML_IS_SIMPLE_ATTRIBUTE (_tmp10) ? ((GtkamlSimpleAttribute*) _tmp10) : NULL)), _tmp11 = g_strconcat ("\n", content, NULL), NULL));
						_tmp12 = (g_free (_tmp12), NULL);
						_tmp11 = (g_free (_tmp11), NULL);
					} else {
						char* _tmp14;
						ValaSourceReference* _tmp13;
						_tmp14 = NULL;
						_tmp13 = NULL;
						vala_report_error (_tmp13 = gtkaml_sax_parser_create_source_reference (self), _tmp14 = g_strdup_printf ("Cannot mix a complex attribute definition with simple values like this: attribute %s", gtkaml_attribute_get_name (gtkaml_state_get_attribute (state))));
						_tmp14 = (g_free (_tmp14), NULL);
						(_tmp13 == NULL) ? NULL : (_tmp13 = (vala_source_reference_unref (_tmp13), NULL));
						gtkaml_sax_parser_stop_parsing (self);
						return;
					}
				}
			}
		}
	} else {
		ValaSourceReference* _tmp15;
		_tmp15 = NULL;
		vala_report_error (_tmp15 = gtkaml_sax_parser_create_source_reference (self), "Invalid non-whitespace text found");
		(_tmp15 == NULL) ? NULL : (_tmp15 = (vala_source_reference_unref (_tmp15), NULL));
		gtkaml_sax_parser_stop_parsing (self);
		return;
	}
}


static char* gtkaml_sax_parser_prefix_to_namespace (GtkamlSAXParser* self, const char* prefix) {
	g_return_val_if_fail (self != NULL, NULL);
	if (prefix == NULL) {
		return (char*) gee_map_get (self->priv->_prefixes_namespaces, "");
	}
	return (char*) gee_map_get (self->priv->_prefixes_namespaces, prefix);
}


ValaSourceReference* gtkaml_sax_parser_create_source_reference (GtkamlSAXParser* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return vala_source_reference_new (self->priv->_source_file, gtkaml_sax_parser_line_number (self), gtkaml_sax_parser_column_number (self), gtkaml_sax_parser_line_number (self), gtkaml_sax_parser_column_number (self));
}


static ValaClass* gtkaml_sax_parser_lookup_class (GtkamlSAXParser* self, const char* xmlNamespace, const char* name) {
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (name != NULL, NULL);
	{
		GeeList* _tmp0;
		GeeIterator* _tmp1;
		GeeIterator* _ns_it;
		_tmp0 = NULL;
		_tmp1 = NULL;
		_ns_it = (_tmp1 = gee_iterable_iterator ((GeeIterable*) (_tmp0 = vala_namespace_get_namespaces (vala_code_context_get_root (self->priv->_context)))), (_tmp0 == NULL) ? NULL : (_tmp0 = (gee_collection_object_unref (_tmp0), NULL)), _tmp1);
		while (gee_iterator_next (_ns_it)) {
			ValaNamespace* ns;
			ns = (ValaNamespace*) gee_iterator_get (_ns_it);
			if (_vala_strcmp0 (xmlNamespace, vala_symbol_get_name ((ValaSymbol*) ns)) == 0) {
				ValaSymbol* s;
				s = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) ns), name);
				if (VALA_IS_CLASS (s)) {
					ValaClass* _tmp3;
					ValaSymbol* _tmp2;
					ValaClass* _tmp4;
					_tmp3 = NULL;
					_tmp2 = NULL;
					_tmp4 = NULL;
					return (_tmp4 = (_tmp3 = (_tmp2 = s, VALA_IS_CLASS (_tmp2) ? ((ValaClass*) _tmp2) : NULL), (_tmp3 == NULL) ? NULL : vala_code_node_ref (_tmp3)), (s == NULL) ? NULL : (s = (vala_code_node_unref (s), NULL)), (ns == NULL) ? NULL : (ns = (vala_code_node_unref (ns), NULL)), (_ns_it == NULL) ? NULL : (_ns_it = (gee_collection_object_unref (_ns_it), NULL)), _tmp4);
				}
				(s == NULL) ? NULL : (s = (vala_code_node_unref (s), NULL));
			}
			(ns == NULL) ? NULL : (ns = (vala_code_node_unref (ns), NULL));
		}
		(_ns_it == NULL) ? NULL : (_ns_it = (gee_collection_object_unref (_ns_it), NULL));
	}
	return NULL;
}


static char* gtkaml_sax_parser_strip_attribute_hyphens (GtkamlSAXParser* self, const char* attrname) {
	char** _tmp1;
	gint tokens_size;
	gint tokens_length1;
	char** _tmp0;
	char** tokens;
	char* _tmp2;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (attrname != NULL, NULL);
	/*see TDWTF, "The Hard Way"*/
	_tmp1 = NULL;
	_tmp0 = NULL;
	tokens = (_tmp1 = _tmp0 = g_strsplit (attrname, "-", 0), tokens_length1 = _vala_array_length (_tmp0), tokens_size = tokens_length1, _tmp1);
	_tmp2 = NULL;
	return (_tmp2 = g_strjoinv ("_", tokens), tokens = (_vala_array_free (tokens, tokens_length1, (GDestroyNotify) g_free), NULL), _tmp2);
}


GtkamlRootClassDefinition* gtkaml_sax_parser_get_root_definition (GtkamlSAXParser* self, ValaClass* clazz, GeeList* attrs, const char* prefix) {
	char* _tmp1;
	ValaSourceReference* _tmp0;
	GtkamlRootClassDefinition* _tmp2;
	GtkamlRootClassDefinition* root_class_definition;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (clazz != NULL, NULL);
	g_return_val_if_fail (attrs != NULL, NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp2 = NULL;
	root_class_definition = (_tmp2 = gtkaml_root_class_definition_new (_tmp0 = gtkaml_sax_parser_create_source_reference (self), "this", _tmp1 = gtkaml_sax_parser_prefix_to_namespace (self, prefix), clazz, DEFINITION_SCOPE_MAIN_CLASS, NULL), _tmp1 = (g_free (_tmp1), NULL), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_source_reference_unref (_tmp0), NULL)), _tmp2);
	gtkaml_root_class_definition_set_prefixes_namespaces (root_class_definition, self->priv->_prefixes_namespaces);
	{
		GeeIterator* _attr_it;
		_attr_it = gee_iterable_iterator ((GeeIterable*) attrs);
		while (gee_iterator_next (_attr_it)) {
			GtkamlXmlAttribute* attr;
			attr = (GtkamlXmlAttribute*) gee_iterator_get (_attr_it);
			if (attr->prefix != NULL) {
				if (_vala_strcmp0 (attr->prefix, self->gtkaml_prefix) == 0) {
					GQuark _tmp14;
					char* _tmp13;
					_tmp13 = NULL;
					static GQuark _tmp14_label0 = 0;
					static GQuark _tmp14_label1 = 0;
					static GQuark _tmp14_label2 = 0;
					static GQuark _tmp14_label3 = 0;
					static GQuark _tmp14_label4 = 0;
					static GQuark _tmp14_label5 = 0;
					static GQuark _tmp14_label6 = 0;
					_tmp13 = attr->localname;
					_tmp14 = (NULL == _tmp13) ? 0 : g_quark_from_string (_tmp13);
					if ((_tmp14 == ((0 != _tmp14_label0) ? _tmp14_label0 : (_tmp14_label0 = g_quark_from_static_string ("public")))) || (_tmp14 == ((0 != _tmp14_label1) ? _tmp14_label1 : (_tmp14_label1 = g_quark_from_static_string ("name")))))
					do {
						if (gtkaml_root_class_definition_get_target_name (root_class_definition) != NULL) {
							char* _tmp4;
							ValaSourceReference* _tmp3;
							_tmp4 = NULL;
							_tmp3 = NULL;
							vala_report_error (_tmp3 = gtkaml_sax_parser_create_source_reference (self), _tmp4 = g_strdup_printf ("A name for the class already exists ('%s')", gtkaml_root_class_definition_get_target_name (root_class_definition)));
							_tmp4 = (g_free (_tmp4), NULL);
							(_tmp3 == NULL) ? NULL : (_tmp3 = (vala_source_reference_unref (_tmp3), NULL));
							gtkaml_sax_parser_stop_parsing (self);
						}
						gtkaml_root_class_definition_set_target_name (root_class_definition, attr->value);
						break;
					} while (0); else if (_tmp14 == ((0 != _tmp14_label2) ? _tmp14_label2 : (_tmp14_label2 = g_quark_from_static_string ("namespace"))))
					do {
						gtkaml_root_class_definition_set_target_namespace (root_class_definition, attr->value);
						break;
					} while (0); else if (_tmp14 == ((0 != _tmp14_label3) ? _tmp14_label3 : (_tmp14_label3 = g_quark_from_static_string ("private"))))
					do {
						ValaSourceReference* _tmp5;
						_tmp5 = NULL;
						vala_report_error (_tmp5 = gtkaml_sax_parser_create_source_reference (self), "'private' not allowed on root tag.");
						(_tmp5 == NULL) ? NULL : (_tmp5 = (vala_source_reference_unref (_tmp5), NULL));
						gtkaml_sax_parser_stop_parsing (self);
						break;
					} while (0); else if (_tmp14 == ((0 != _tmp14_label4) ? _tmp14_label4 : (_tmp14_label4 = g_quark_from_static_string ("construct"))))
					do {
						if (gtkaml_class_definition_get_construct_code ((GtkamlClassDefinition*) root_class_definition) != NULL) {
							ValaSourceReference* _tmp6;
							_tmp6 = NULL;
							vala_report_error (_tmp6 = gtkaml_sax_parser_create_source_reference (self), "A construct attribute already exists for the root class");
							(_tmp6 == NULL) ? NULL : (_tmp6 = (vala_source_reference_unref (_tmp6), NULL));
							gtkaml_sax_parser_stop_parsing (self);
						}
						gtkaml_class_definition_set_construct_code ((GtkamlClassDefinition*) root_class_definition, attr->value);
						break;
					} while (0); else if (_tmp14 == ((0 != _tmp14_label5) ? _tmp14_label5 : (_tmp14_label5 = g_quark_from_static_string ("preconstruct"))))
					do {
						if (gtkaml_class_definition_get_preconstruct_code ((GtkamlClassDefinition*) root_class_definition) != NULL) {
							ValaSourceReference* _tmp7;
							_tmp7 = NULL;
							vala_report_error (_tmp7 = gtkaml_sax_parser_create_source_reference (self), "A preconstruct attribute already exists for the root class");
							(_tmp7 == NULL) ? NULL : (_tmp7 = (vala_source_reference_unref (_tmp7), NULL));
							gtkaml_sax_parser_stop_parsing (self);
						}
						gtkaml_class_definition_set_preconstruct_code ((GtkamlClassDefinition*) root_class_definition, attr->value);
						break;
					} while (0); else if (_tmp14 == ((0 != _tmp14_label6) ? _tmp14_label6 : (_tmp14_label6 = g_quark_from_static_string ("implements"))))
					do {
						char** _tmp9;
						gint implementsv_size;
						gint implementsv_length1;
						char** _tmp8;
						char** implementsv;
						char* _tmp10;
						_tmp9 = NULL;
						_tmp8 = NULL;
						implementsv = (_tmp9 = _tmp8 = g_strsplit (attr->value, ",", 0), implementsv_length1 = _vala_array_length (_tmp8), implementsv_size = implementsv_length1, _tmp9);
						{
							gint i;
							i = 0;
							for (; implementsv[i] != NULL; i++) {
								g_strstrip (implementsv[i]);
							}
						}
						_tmp10 = NULL;
						gtkaml_root_class_definition_set_implements (root_class_definition, _tmp10 = g_strjoinv (", ", implementsv));
						_tmp10 = (g_free (_tmp10), NULL);
						implementsv = (_vala_array_free (implementsv, implementsv_length1, (GDestroyNotify) g_free), NULL);
						break;
					} while (0); else
					do {
						char* _tmp12;
						ValaSourceReference* _tmp11;
						_tmp12 = NULL;
						_tmp11 = NULL;
						vala_report_warning (_tmp11 = gtkaml_sax_parser_create_source_reference (self), _tmp12 = g_strdup_printf ("Unknown gtkaml attribute '%s'.", attr->localname));
						_tmp12 = (g_free (_tmp12), NULL);
						(_tmp11 == NULL) ? NULL : (_tmp11 = (vala_source_reference_unref (_tmp11), NULL));
						break;
					} while (0);
				} else {
					char* _tmp16;
					ValaSourceReference* _tmp15;
					_tmp16 = NULL;
					_tmp15 = NULL;
					vala_report_error (_tmp15 = gtkaml_sax_parser_create_source_reference (self), _tmp16 = g_strdup_printf ("'%s' is the only allowed prefix for attributes. Other attributes must be left unprefixed", self->gtkaml_prefix));
					_tmp16 = (g_free (_tmp16), NULL);
					(_tmp15 == NULL) ? NULL : (_tmp15 = (vala_source_reference_unref (_tmp15), NULL));
					gtkaml_sax_parser_stop_parsing (self);
				}
			} else {
				char* _tmp17;
				GtkamlSimpleAttribute* _tmp18;
				GtkamlSimpleAttribute* simple_attribute;
				_tmp17 = NULL;
				_tmp18 = NULL;
				simple_attribute = (_tmp18 = gtkaml_simple_attribute_new (_tmp17 = gtkaml_sax_parser_strip_attribute_hyphens (self, attr->localname), attr->value), _tmp17 = (g_free (_tmp17), NULL), _tmp18);
				gtkaml_class_definition_add_attribute ((GtkamlClassDefinition*) root_class_definition, (GtkamlAttribute*) simple_attribute);
				(simple_attribute == NULL) ? NULL : (simple_attribute = (g_object_unref (simple_attribute), NULL));
			}
			(attr == NULL) ? NULL : (attr = (g_object_unref (attr), NULL));
		}
		(_attr_it == NULL) ? NULL : (_attr_it = (gee_collection_object_unref (_attr_it), NULL));
	}
	if (gtkaml_root_class_definition_get_target_name (root_class_definition) == NULL) {
		char* _tmp20;
		ValaSourceReference* _tmp19;
		_tmp20 = NULL;
		_tmp19 = NULL;
		vala_report_error (_tmp19 = gtkaml_sax_parser_create_source_reference (self), _tmp20 = g_strdup_printf ("No class name specified: use %s:name for this", self->gtkaml_prefix));
		_tmp20 = (g_free (_tmp20), NULL);
		(_tmp19 == NULL) ? NULL : (_tmp19 = (vala_source_reference_unref (_tmp19), NULL));
	}
	return root_class_definition;
}


GtkamlClassDefinition* gtkaml_sax_parser_get_child_for_container (GtkamlSAXParser* self, ValaClass* clazz, GtkamlClassDefinition* container_definition, GeeList* attrs, const char* prefix) {
	char* identifier;
	DefinitionScope identifier_scope;
	char* reference;
	char* construct_code;
	char* preconstruct_code;
	GtkamlClassDefinition* _tmp0;
	GtkamlClassDefinition* parent_container;
	gboolean _tmp19;
	GtkamlClassDefinition* class_definition;
	GtkamlClassDefinition* _tmp39;
	g_return_val_if_fail (self != NULL, NULL);
	g_return_val_if_fail (clazz != NULL, NULL);
	g_return_val_if_fail (attrs != NULL, NULL);
	identifier = NULL;
	identifier_scope = DEFINITION_SCOPE_CONSTRUCTOR;
	reference = NULL;
	construct_code = NULL;
	preconstruct_code = NULL;
	_tmp0 = NULL;
	parent_container = (_tmp0 = container_definition, (_tmp0 == NULL) ? NULL : g_object_ref (_tmp0));
	{
		GeeIterator* _attr_it;
		_attr_it = gee_iterable_iterator ((GeeIterable*) attrs);
		while (gee_iterator_next (_attr_it)) {
			GtkamlXmlAttribute* attr;
			gboolean _tmp1;
			attr = (GtkamlXmlAttribute*) gee_iterator_get (_attr_it);
			_tmp1 = FALSE;
			if (attr->prefix != NULL) {
				_tmp1 = _vala_strcmp0 (attr->prefix, self->gtkaml_prefix) == 0;
			} else {
				_tmp1 = FALSE;
			}
			if (_tmp1) {
				gboolean _tmp2;
				_tmp2 = FALSE;
				if (_vala_strcmp0 (attr->localname, "public") == 0) {
					_tmp2 = TRUE;
				} else {
					_tmp2 = _vala_strcmp0 (attr->localname, "private") == 0;
				}
				if (_tmp2) {
					char* _tmp6;
					const char* _tmp5;
					if (identifier != NULL) {
						char* _tmp4;
						ValaSourceReference* _tmp3;
						_tmp4 = NULL;
						_tmp3 = NULL;
						vala_report_error (_tmp3 = gtkaml_sax_parser_create_source_reference (self), _tmp4 = g_strdup_printf ("Cannot have multiple identifier names:%s", attr->localname));
						_tmp4 = (g_free (_tmp4), NULL);
						(_tmp3 == NULL) ? NULL : (_tmp3 = (vala_source_reference_unref (_tmp3), NULL));
						gtkaml_sax_parser_stop_parsing (self);
						(attr == NULL) ? NULL : (attr = (g_object_unref (attr), NULL));
						break;
					}
					_tmp6 = NULL;
					_tmp5 = NULL;
					identifier = (_tmp6 = (_tmp5 = attr->value, (_tmp5 == NULL) ? NULL : g_strdup (_tmp5)), identifier = (g_free (identifier), NULL), _tmp6);
					if (_vala_strcmp0 (attr->localname, "public") == 0) {
						identifier_scope = DEFINITION_SCOPE_PUBLIC;
					} else {
						identifier_scope = DEFINITION_SCOPE_PRIVATE;
					}
				} else {
					if (_vala_strcmp0 (attr->localname, "existing") == 0) {
						char* _tmp8;
						const char* _tmp7;
						_tmp8 = NULL;
						_tmp7 = NULL;
						reference = (_tmp8 = (_tmp7 = attr->value, (_tmp7 == NULL) ? NULL : g_strdup (_tmp7)), reference = (g_free (reference), NULL), _tmp8);
					} else {
						if (_vala_strcmp0 (attr->localname, "construct") == 0) {
							char* _tmp10;
							const char* _tmp9;
							_tmp10 = NULL;
							_tmp9 = NULL;
							construct_code = (_tmp10 = (_tmp9 = attr->value, (_tmp9 == NULL) ? NULL : g_strdup (_tmp9)), construct_code = (g_free (construct_code), NULL), _tmp10);
						} else {
							if (_vala_strcmp0 (attr->localname, "preconstruct") == 0) {
								char* _tmp12;
								const char* _tmp11;
								_tmp12 = NULL;
								_tmp11 = NULL;
								preconstruct_code = (_tmp12 = (_tmp11 = attr->value, (_tmp11 == NULL) ? NULL : g_strdup (_tmp11)), preconstruct_code = (g_free (preconstruct_code), NULL), _tmp12);
							} else {
								if (_vala_strcmp0 (attr->localname, "standalone") == 0) {
									if (_vala_strcmp0 (attr->value, "true") == 0) {
										GtkamlClassDefinition* _tmp13;
										_tmp13 = NULL;
										parent_container = (_tmp13 = NULL, (parent_container == NULL) ? NULL : (parent_container = (g_object_unref (parent_container), NULL)), _tmp13);
									} else {
										ValaSourceReference* _tmp14;
										_tmp14 = NULL;
										vala_report_error (_tmp14 = gtkaml_sax_parser_create_source_reference (self), "Invalid 'standalone' value");
										(_tmp14 == NULL) ? NULL : (_tmp14 = (vala_source_reference_unref (_tmp14), NULL));
										gtkaml_sax_parser_stop_parsing (self);
									}
								} else {
									char* _tmp16;
									ValaSourceReference* _tmp15;
									_tmp16 = NULL;
									_tmp15 = NULL;
									vala_report_error (_tmp15 = gtkaml_sax_parser_create_source_reference (self), _tmp16 = g_strdup_printf ("Unkown gtkaml attribute '%s'", attr->localname));
									_tmp16 = (g_free (_tmp16), NULL);
									(_tmp15 == NULL) ? NULL : (_tmp15 = (vala_source_reference_unref (_tmp15), NULL));
									gtkaml_sax_parser_stop_parsing (self);
								}
							}
						}
					}
				}
			} else {
				if (attr->prefix != NULL) {
					char* _tmp18;
					ValaSourceReference* _tmp17;
					_tmp18 = NULL;
					_tmp17 = NULL;
					vala_report_error (_tmp17 = gtkaml_sax_parser_create_source_reference (self), _tmp18 = g_strdup_printf ("%s is the only allowed prefix for attributes. Other attributes must be left unprefixed", self->gtkaml_prefix));
					_tmp18 = (g_free (_tmp18), NULL);
					(_tmp17 == NULL) ? NULL : (_tmp17 = (vala_source_reference_unref (_tmp17), NULL));
					gtkaml_sax_parser_stop_parsing (self);
				}
			}
			(attr == NULL) ? NULL : (attr = (g_object_unref (attr), NULL));
		}
		(_attr_it == NULL) ? NULL : (_attr_it = (gee_collection_object_unref (_attr_it), NULL));
	}
	_tmp19 = FALSE;
	if (identifier != NULL) {
		_tmp19 = reference != NULL;
	} else {
		_tmp19 = FALSE;
	}
	if (_tmp19) {
		ValaSourceReference* _tmp20;
		_tmp20 = NULL;
		vala_report_error (_tmp20 = gtkaml_sax_parser_create_source_reference (self), "Cannot specify both existing and a new identifier name");
		(_tmp20 == NULL) ? NULL : (_tmp20 = (vala_source_reference_unref (_tmp20), NULL));
		gtkaml_sax_parser_stop_parsing (self);
	}
	class_definition = NULL;
	if (reference == NULL) {
		gint counter;
		GtkamlClassDefinition* _tmp26;
		char* _tmp25;
		ValaSourceReference* _tmp24;
		counter = 0;
		if (identifier == NULL) {
			char* _tmp21;
			char* _tmp22;
			char* _tmp23;
			/*generate a name for the identifier*/
			_tmp21 = NULL;
			identifier = (_tmp21 = g_utf8_strdown (vala_symbol_get_name ((ValaSymbol*) clazz), g_utf8_strlen (vala_symbol_get_name ((ValaSymbol*) clazz), -1)), identifier = (g_free (identifier), NULL), _tmp21);
			if (gee_map_contains (self->priv->generated_identifiers_counter, identifier)) {
				counter = GPOINTER_TO_INT (gee_map_get (self->priv->generated_identifiers_counter, identifier));
			}
			_tmp22 = NULL;
			identifier = (_tmp22 = g_strdup_printf ("_%s%d", identifier, counter), identifier = (g_free (identifier), NULL), _tmp22);
			counter++;
			_tmp23 = NULL;
			gee_map_set (self->priv->generated_identifiers_counter, _tmp23 = g_utf8_strdown (vala_symbol_get_name ((ValaSymbol*) clazz), g_utf8_strlen (vala_symbol_get_name ((ValaSymbol*) clazz), -1)), GINT_TO_POINTER (counter));
			_tmp23 = (g_free (_tmp23), NULL);
		}
		_tmp26 = NULL;
		_tmp25 = NULL;
		_tmp24 = NULL;
		class_definition = (_tmp26 = gtkaml_class_definition_new (_tmp24 = gtkaml_sax_parser_create_source_reference (self), identifier, _tmp25 = gtkaml_sax_parser_prefix_to_namespace (self, prefix), clazz, identifier_scope, parent_container), (class_definition == NULL) ? NULL : (class_definition = (g_object_unref (class_definition), NULL)), _tmp26);
		_tmp25 = (g_free (_tmp25), NULL);
		(_tmp24 == NULL) ? NULL : (_tmp24 = (vala_source_reference_unref (_tmp24), NULL));
		gtkaml_class_definition_set_construct_code (class_definition, construct_code);
		gtkaml_class_definition_set_preconstruct_code (class_definition, preconstruct_code);
	} else {
		gboolean _tmp27;
		GtkamlClassDefinition* _tmp31;
		char* _tmp30;
		ValaSourceReference* _tmp29;
		const char* _tmp32;
		char* reference_stripped;
		_tmp27 = FALSE;
		if (construct_code != NULL) {
			_tmp27 = TRUE;
		} else {
			_tmp27 = preconstruct_code != NULL;
		}
		if (_tmp27) {
			ValaSourceReference* _tmp28;
			_tmp28 = NULL;
			vala_report_error (_tmp28 = gtkaml_sax_parser_create_source_reference (self), "Cannot specify 'construct' or 'preconstruct' code for references");
			(_tmp28 == NULL) ? NULL : (_tmp28 = (vala_source_reference_unref (_tmp28), NULL));
			gtkaml_sax_parser_stop_parsing (self);
		}
		_tmp31 = NULL;
		_tmp30 = NULL;
		_tmp29 = NULL;
		class_definition = (_tmp31 = (GtkamlClassDefinition*) gtkaml_reference_class_definition_new (_tmp29 = gtkaml_sax_parser_create_source_reference (self), reference, _tmp30 = gtkaml_sax_parser_prefix_to_namespace (self, prefix), clazz, parent_container), (class_definition == NULL) ? NULL : (class_definition = (g_object_unref (class_definition), NULL)), _tmp31);
		_tmp30 = (g_free (_tmp30), NULL);
		(_tmp29 == NULL) ? NULL : (_tmp29 = (vala_source_reference_unref (_tmp29), NULL));
		/* now post-process the reference FIXME put this in code generator or something*/
		_tmp32 = NULL;
		reference_stripped = (_tmp32 = reference, (_tmp32 == NULL) ? NULL : g_strdup (_tmp32));
		g_strstrip (reference_stripped);
		if (g_str_has_prefix (reference_stripped, "{")) {
			if (g_str_has_suffix (reference_stripped, "}")) {
				char* _tmp33;
				_tmp33 = NULL;
				gtkaml_class_definition_set_identifier (class_definition, _tmp33 = string_substring (reference_stripped, (glong) 1, g_utf8_strlen (reference_stripped, -1) - 2));
				_tmp33 = (g_free (_tmp33), NULL);
			} else {
				ValaSourceReference* _tmp34;
				_tmp34 = NULL;
				vala_report_error (_tmp34 = gtkaml_sax_parser_create_source_reference (self), "'existing' attribute not properly ended");
				(_tmp34 == NULL) ? NULL : (_tmp34 = (vala_source_reference_unref (_tmp34), NULL));
			}
		} else {
			char* _tmp36;
			char* _tmp35;
			_tmp36 = NULL;
			_tmp35 = NULL;
			gtkaml_class_definition_set_identifier (class_definition, _tmp36 = g_strdup_printf ("(%s as %s)", reference, _tmp35 = gtkaml_class_definition_get_base_full_name (class_definition)));
			_tmp36 = (g_free (_tmp36), NULL);
			_tmp35 = (g_free (_tmp35), NULL);
		}
		reference_stripped = (g_free (reference_stripped), NULL);
	}
	if (container_definition != NULL) {
		gtkaml_class_definition_add_child (container_definition, class_definition);
	}
	{
		GeeIterator* _attr_it;
		_attr_it = gee_iterable_iterator ((GeeIterable*) attrs);
		while (gee_iterator_next (_attr_it)) {
			GtkamlXmlAttribute* attr;
			attr = (GtkamlXmlAttribute*) gee_iterator_get (_attr_it);
			if (attr->prefix == NULL) {
				char* _tmp37;
				GtkamlSimpleAttribute* _tmp38;
				GtkamlSimpleAttribute* simple_attribute;
				_tmp37 = NULL;
				_tmp38 = NULL;
				simple_attribute = (_tmp38 = gtkaml_simple_attribute_new (_tmp37 = gtkaml_sax_parser_strip_attribute_hyphens (self, attr->localname), attr->value), _tmp37 = (g_free (_tmp37), NULL), _tmp38);
				gtkaml_class_definition_add_attribute (class_definition, (GtkamlAttribute*) simple_attribute);
				(simple_attribute == NULL) ? NULL : (simple_attribute = (g_object_unref (simple_attribute), NULL));
			}
			(attr == NULL) ? NULL : (attr = (g_object_unref (attr), NULL));
		}
		(_attr_it == NULL) ? NULL : (_attr_it = (gee_collection_object_unref (_attr_it), NULL));
	}
	_tmp39 = NULL;
	return (_tmp39 = class_definition, identifier = (g_free (identifier), NULL), reference = (g_free (reference), NULL), construct_code = (g_free (construct_code), NULL), preconstruct_code = (g_free (preconstruct_code), NULL), (parent_container == NULL) ? NULL : (parent_container = (g_object_unref (parent_container), NULL)), _tmp39);
}


static GeeList* gtkaml_sax_parser_parse_attributes (GtkamlSAXParser* self, char** attributes, gint nb_attributes) {
	gint walker;
	char* end;
	GeeArrayList* attribute_list;
	GeeList* _tmp11;
	g_return_val_if_fail (self != NULL, NULL);
	walker = 0;
	end = NULL;
	attribute_list = gee_array_list_new (GTKAML_TYPE_XML_ATTRIBUTE, (GBoxedCopyFunc) g_object_ref, g_object_unref, g_direct_equal);
	{
		gint i;
		i = 0;
		for (; i < nb_attributes; i++) {
			GtkamlXmlAttribute* attr;
			char* _tmp1;
			const char* _tmp0;
			char* _tmp3;
			const char* _tmp2;
			char* _tmp5;
			const char* _tmp4;
			char* _tmp7;
			const char* _tmp6;
			char* _tmp9;
			const char* _tmp8;
			char* _tmp10;
			attr = gtkaml_xml_attribute_new ();
			_tmp1 = NULL;
			_tmp0 = NULL;
			attr->localname = (_tmp1 = (_tmp0 = attributes[walker], (_tmp0 == NULL) ? NULL : g_strdup (_tmp0)), attr->localname = (g_free (attr->localname), NULL), _tmp1);
			_tmp3 = NULL;
			_tmp2 = NULL;
			attr->prefix = (_tmp3 = (_tmp2 = attributes[walker + 1], (_tmp2 == NULL) ? NULL : g_strdup (_tmp2)), attr->prefix = (g_free (attr->prefix), NULL), _tmp3);
			_tmp5 = NULL;
			_tmp4 = NULL;
			attr->URI = (_tmp5 = (_tmp4 = attributes[walker + 2], (_tmp4 == NULL) ? NULL : g_strdup (_tmp4)), attr->URI = (g_free (attr->URI), NULL), _tmp5);
			_tmp7 = NULL;
			_tmp6 = NULL;
			attr->value = (_tmp7 = (_tmp6 = attributes[walker + 3], (_tmp6 == NULL) ? NULL : g_strdup (_tmp6)), attr->value = (g_free (attr->value), NULL), _tmp7);
			_tmp9 = NULL;
			_tmp8 = NULL;
			end = (_tmp9 = (_tmp8 = attributes[walker + 4], (_tmp8 == NULL) ? NULL : g_strdup (_tmp8)), end = (g_free (end), NULL), _tmp9);
			_tmp10 = NULL;
			attr->value = (_tmp10 = g_strndup (attr->value, (gsize) (g_utf8_strlen (attr->value, -1) - g_utf8_strlen (end, -1))), attr->value = (g_free (attr->value), NULL), _tmp10);
			gee_collection_add ((GeeCollection*) attribute_list, attr);
			walker = walker + 5;
			(attr == NULL) ? NULL : (attr = (g_object_unref (attr), NULL));
		}
	}
	_tmp11 = NULL;
	return (_tmp11 = (GeeList*) attribute_list, end = (g_free (end), NULL), _tmp11);
}


static GeeList* gtkaml_sax_parser_parse_namespaces (GtkamlSAXParser* self, char** namespaces, gint nb_namespaces) {
	gint walker;
	GeeArrayList* namespace_list;
	g_return_val_if_fail (self != NULL, NULL);
	walker = 0;
	namespace_list = gee_array_list_new (TYPE_XML_NAMESPACE, (GBoxedCopyFunc) g_object_ref, g_object_unref, g_direct_equal);
	{
		gint i;
		i = 0;
		for (; i < nb_namespaces; i++) {
			XmlNamespace* ns;
			char* _tmp1;
			const char* _tmp0;
			char* _tmp3;
			const char* _tmp2;
			gboolean _tmp4;
			ns = xml_namespace_new ();
			_tmp1 = NULL;
			_tmp0 = NULL;
			ns->prefix = (_tmp1 = (_tmp0 = namespaces[walker], (_tmp0 == NULL) ? NULL : g_strdup (_tmp0)), ns->prefix = (g_free (ns->prefix), NULL), _tmp1);
			_tmp3 = NULL;
			_tmp2 = NULL;
			ns->URI = (_tmp3 = (_tmp2 = namespaces[walker + 1], (_tmp2 == NULL) ? NULL : g_strdup (_tmp2)), ns->URI = (g_free (ns->URI), NULL), _tmp3);
			_tmp4 = FALSE;
			if (ns->URI != NULL) {
				_tmp4 = g_str_has_prefix (ns->URI, "http://gtkaml.org/");
			} else {
				_tmp4 = FALSE;
			}
			if (_tmp4) {
				if (ns->prefix != NULL) {
					char* _tmp6;
					const char* _tmp5;
					char* version;
					_tmp6 = NULL;
					_tmp5 = NULL;
					self->gtkaml_prefix = (_tmp6 = (_tmp5 = ns->prefix, (_tmp5 == NULL) ? NULL : g_strdup (_tmp5)), self->gtkaml_prefix = (g_free (self->gtkaml_prefix), NULL), _tmp6);
					version = string_substring (ns->URI, g_utf8_strlen ("http://gtkaml.org/", -1), g_utf8_strlen (ns->URI, -1) - g_utf8_strlen ("http://gtkaml.org/", -1));
					if (_vala_strcmp0 (version, PACKAGE_VERSION) > 0) {
						char* _tmp8;
						ValaSourceReference* _tmp7;
						_tmp8 = NULL;
						_tmp7 = NULL;
						vala_report_warning (_tmp7 = gtkaml_sax_parser_create_source_reference (self), _tmp8 = g_strdup_printf ("Source file version (%s) newer than gtkaml compiler version (%s)", version, PACKAGE_VERSION));
						_tmp8 = (g_free (_tmp8), NULL);
						(_tmp7 == NULL) ? NULL : (_tmp7 = (vala_source_reference_unref (_tmp7), NULL));
					}
					version = (g_free (version), NULL);
				} else {
					ValaSourceReference* _tmp9;
					_tmp9 = NULL;
					vala_report_error (_tmp9 = gtkaml_sax_parser_create_source_reference (self), "You cannot use the gtkaml namespace as default namespace");
					(_tmp9 == NULL) ? NULL : (_tmp9 = (vala_source_reference_unref (_tmp9), NULL));
				}
			}
			gee_collection_add ((GeeCollection*) namespace_list, ns);
			walker = walker + 2;
			(ns == NULL) ? NULL : (ns = (g_object_unref (ns), NULL));
		}
	}
	return (GeeList*) namespace_list;
}


ValaCodeContext* gtkaml_sax_parser_get_context (GtkamlSAXParser* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_context;
}


static void gtkaml_sax_parser_set_context (GtkamlSAXParser* self, ValaCodeContext* value) {
	ValaCodeContext* _tmp2;
	ValaCodeContext* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_context = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_context_ref (_tmp1)), (self->priv->_context == NULL) ? NULL : (self->priv->_context = (vala_code_context_unref (self->priv->_context), NULL)), _tmp2);
	g_object_notify ((GObject *) self, "context");
}


ValaSourceFile* gtkaml_sax_parser_get_source_file (GtkamlSAXParser* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_source_file;
}


static void gtkaml_sax_parser_set_source_file (GtkamlSAXParser* self, ValaSourceFile* value) {
	g_return_if_fail (self != NULL);
	self->priv->_source_file = value;
	g_object_notify ((GObject *) self, "source-file");
}


static GtkamlStateStack* gtkaml_sax_parser_get_states (GtkamlSAXParser* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_states;
}


static void gtkaml_sax_parser_set_states (GtkamlSAXParser* self, GtkamlStateStack* value) {
	GtkamlStateStack* _tmp2;
	GtkamlStateStack* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_states = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : g_object_ref (_tmp1)), (self->priv->_states == NULL) ? NULL : (self->priv->_states = (g_object_unref (self->priv->_states), NULL)), _tmp2);
}


static GeeMap* gtkaml_sax_parser_get_prefixes_namespaces (GtkamlSAXParser* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_prefixes_namespaces;
}


static void gtkaml_sax_parser_set_prefixes_namespaces (GtkamlSAXParser* self, GeeMap* value) {
	GeeMap* _tmp2;
	GeeMap* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_prefixes_namespaces = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : gee_collection_object_ref (_tmp1)), (self->priv->_prefixes_namespaces == NULL) ? NULL : (self->priv->_prefixes_namespaces = (gee_collection_object_unref (self->priv->_prefixes_namespaces), NULL)), _tmp2);
}


static GtkamlRootClassDefinition* gtkaml_sax_parser_get_root_class_definition (GtkamlSAXParser* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_root_class_definition;
}


static void gtkaml_sax_parser_set_root_class_definition (GtkamlSAXParser* self, GtkamlRootClassDefinition* value) {
	GtkamlRootClassDefinition* _tmp2;
	GtkamlRootClassDefinition* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_root_class_definition = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : g_object_ref (_tmp1)), (self->priv->_root_class_definition == NULL) ? NULL : (self->priv->_root_class_definition = (g_object_unref (self->priv->_root_class_definition), NULL)), _tmp2);
}


static GObject * gtkaml_sax_parser_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties) {
	GObject * obj;
	GtkamlSAXParserClass * klass;
	GObjectClass * parent_class;
	GtkamlSAXParser * self;
	klass = GTKAML_SAX_PARSER_CLASS (g_type_class_peek (GTKAML_TYPE_SAX_PARSER));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	obj = parent_class->constructor (type, n_construct_properties, construct_properties);
	self = GTKAML_SAX_PARSER (obj);
	{
		GtkamlStateStack* _tmp0;
		GeeHashMap* _tmp1;
		_tmp0 = NULL;
		gtkaml_sax_parser_set_states (self, _tmp0 = gtkaml_state_stack_new ());
		(_tmp0 == NULL) ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL));
		_tmp1 = NULL;
		gtkaml_sax_parser_set_prefixes_namespaces (self, (GeeMap*) (_tmp1 = gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_str_hash, g_str_equal, g_str_equal)));
		(_tmp1 == NULL) ? NULL : (_tmp1 = (gee_collection_object_unref (_tmp1), NULL));
		gtkaml_sax_parser_set_root_class_definition (self, NULL);
	}
	return obj;
}


static void gtkaml_sax_parser_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	GtkamlSAXParser * self;
	gpointer boxed;
	self = GTKAML_SAX_PARSER (object);
	switch (property_id) {
		case GTKAML_SAX_PARSER_CONTEXT:
		vala_value_set_code_context (value, gtkaml_sax_parser_get_context (self));
		break;
		case GTKAML_SAX_PARSER_SOURCE_FILE:
		vala_value_set_source_file (value, gtkaml_sax_parser_get_source_file (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void gtkaml_sax_parser_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	GtkamlSAXParser * self;
	self = GTKAML_SAX_PARSER (object);
	switch (property_id) {
		case GTKAML_SAX_PARSER_CONTEXT:
		gtkaml_sax_parser_set_context (self, vala_value_get_code_context (value));
		break;
		case GTKAML_SAX_PARSER_SOURCE_FILE:
		gtkaml_sax_parser_set_source_file (self, vala_value_get_source_file (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void gtkaml_sax_parser_class_init (GtkamlSAXParserClass * klass) {
	gtkaml_sax_parser_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (GtkamlSAXParserPrivate));
	G_OBJECT_CLASS (klass)->get_property = gtkaml_sax_parser_get_property;
	G_OBJECT_CLASS (klass)->set_property = gtkaml_sax_parser_set_property;
	G_OBJECT_CLASS (klass)->constructor = gtkaml_sax_parser_constructor;
	G_OBJECT_CLASS (klass)->finalize = gtkaml_sax_parser_finalize;
	GTKAML_SAX_PARSER_CLASS (klass)->parse = gtkaml_sax_parser_real_parse;
	g_object_class_install_property (G_OBJECT_CLASS (klass), GTKAML_SAX_PARSER_CONTEXT, vala_param_spec_code_context ("context", "context", "context", VALA_TYPE_CODE_CONTEXT, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GTKAML_SAX_PARSER_SOURCE_FILE, vala_param_spec_source_file ("source-file", "source-file", "source-file", VALA_TYPE_SOURCE_FILE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void gtkaml_sax_parser_instance_init (GtkamlSAXParser * self) {
	self->priv = GTKAML_SAX_PARSER_GET_PRIVATE (self);
	self->priv->generated_identifiers_counter = (GeeMap*) gee_hash_map_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, G_TYPE_INT, NULL, NULL, g_str_hash, g_str_equal, g_direct_equal);
	self->gtkaml_prefix = g_strdup ("gtkaml");
}


static void gtkaml_sax_parser_finalize (GObject* obj) {
	GtkamlSAXParser * self;
	self = GTKAML_SAX_PARSER (obj);
	(self->priv->_context == NULL) ? NULL : (self->priv->_context = (vala_code_context_unref (self->priv->_context), NULL));
	(self->priv->_states == NULL) ? NULL : (self->priv->_states = (g_object_unref (self->priv->_states), NULL));
	(self->priv->generated_identifiers_counter == NULL) ? NULL : (self->priv->generated_identifiers_counter = (gee_collection_object_unref (self->priv->generated_identifiers_counter), NULL));
	(self->priv->_prefixes_namespaces == NULL) ? NULL : (self->priv->_prefixes_namespaces = (gee_collection_object_unref (self->priv->_prefixes_namespaces), NULL));
	(self->priv->_root_class_definition == NULL) ? NULL : (self->priv->_root_class_definition = (g_object_unref (self->priv->_root_class_definition), NULL));
	self->gtkaml_prefix = (g_free (self->gtkaml_prefix), NULL);
	G_OBJECT_CLASS (gtkaml_sax_parser_parent_class)->finalize (obj);
}


GType gtkaml_sax_parser_get_type (void) {
	static GType gtkaml_sax_parser_type_id = 0;
	if (gtkaml_sax_parser_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (GtkamlSAXParserClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gtkaml_sax_parser_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GtkamlSAXParser), 0, (GInstanceInitFunc) gtkaml_sax_parser_instance_init, NULL };
		gtkaml_sax_parser_type_id = g_type_register_static (G_TYPE_OBJECT, "GtkamlSAXParser", &g_define_type_info, 0);
	}
	return gtkaml_sax_parser_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
	g_free (array);
}


static gint _vala_array_length (gpointer array) {
	int length;
	length = 0;
	if (array) {
		while (((gpointer*) array)[length]) {
			length++;
		}
	}
	return length;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




