/*
 * $Id: utf8_tables.h,v 1.10 2005/10/29 23:30:20 cbiere Exp $
 *
 * Copyright (c) 2004-2005, Christian Biere
 *
 *----------------------------------------------------------------------
 * This file is part of gtk-gnutella.
 *
 *  gtk-gnutella is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  gtk-gnutella is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with gtk-gnutella; if not, write to the Free Software
 *  Foundation, Inc.:
 *      59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *----------------------------------------------------------------------
 */

/**
 * @ingroup lib
 * @file
 *
 * Unicode Lookup Tables.
 *
 * @author Christian Biere
 * @date 2004-2005
 */

#ifndef _utf8_tables_h_
#define _utf8_tables_h_

/*
 *	Current Unicode version: 4.1.0
 */

/*
 * This is the lookup table for normalization/decomposition.
 *
 * The table is extracted from UnicodeData.txt - the
 * sixth column is relevant.
 *
 * The complete specifications regarding Unicode normalization and
 * decomposition are available here:
 *
 * 		http://www.unicode.org/reports/tr15/
 *
 * NB:	The entries are only zero-terminated if the replacement string is
 * 		shorter than UTF32_NFKD_REPLACE_MAXLEN characters! The first
 *		element of each entry is the decomposable UTF-32 character, the
 *		next elements form the UTF-32 string.
 */

#define UTF32_NFKD_REPLACE_MAXLEN 18
#define UTF32_F_MASK 0xff000000U
#define UTF32_F_NFKD 0x80000000U	/**< Set for compatibility compositions */

#define UTF32_NFD(x)  (x)
#define UTF32_NFKD(x) (UTF32_F_NFKD | (x))

#define UNI_GC_LU UNI_GC_LETTER_UPPERCASE
#define UNI_GC_LL UNI_GC_LETTER_LOWERCASE
#define UNI_GC_LT UNI_GC_LETTER_TITLECASE
#define UNI_GC_LM UNI_GC_LETTER_MODIFIER
#define UNI_GC_LO UNI_GC_LETTER_OTHER

#define UNI_GC_MN UNI_GC_MARK_NONSPACING 
#define UNI_GC_MC UNI_GC_MARK_SPACING_COMBINE
#define UNI_GC_ME UNI_GC_MARK_ENCLOSING

#define UNI_GC_ND UNI_GC_NUMBER_DECIMAL
#define UNI_GC_NL UNI_GC_NUMBER_LETTER
#define UNI_GC_NO UNI_GC_NUMBER_OTHER

#define UNI_GC_PC UNI_GC_PUNCT_CONNECTOR
#define UNI_GC_PD UNI_GC_PUNCT_DASH
#define UNI_GC_PS UNI_GC_PUNCT_OPEN
#define UNI_GC_PE UNI_GC_PUNCT_CLOSE
#define UNI_GC_PI UNI_GC_PUNCT_INIT_QUOTE
#define UNI_GC_PF UNI_GC_PUNCT_FINAL_QUOTE
#define UNI_GC_PO UNI_GC_PUNCT_OTHER

#define UNI_GC_SM UNI_GC_SYMBOL_MATH
#define UNI_GC_SC UNI_GC_SYMBOL_CURRENCY
#define UNI_GC_SK UNI_GC_SYMBOL_MODIFIER
#define UNI_GC_SO UNI_GC_SYMBOL_OTHER

#define UNI_GC_ZS UNI_GC_SEPARATOR_SPACE
#define UNI_GC_ZL UNI_GC_SEPARATOR_LINE
#define UNI_GC_ZP UNI_GC_SEPARATOR_PARAGRAPH

#define UNI_GC_CC UNI_GC_OTHER_CONTROL
#define UNI_GC_CF UNI_GC_OTHER_FORMAT
#define UNI_GC_CS UNI_GC_OTHER_SURROGATE
#define UNI_GC_CO UNI_GC_OTHER_PRIVATE_USE
#define UNI_GC_CN UNI_GC_OTHER_NOT_ASSIGNED

typedef enum {
	UNI_GC_LETTER_UPPERCASE = 0,
	UNI_GC_LETTER_LOWERCASE,
	UNI_GC_LETTER_TITLECASE,
	UNI_GC_LETTER_MODIFIER,
	UNI_GC_LETTER_OTHER,

	UNI_GC_MARK_NONSPACING,
	UNI_GC_MARK_SPACING_COMBINE,
	UNI_GC_MARK_ENCLOSING,

	UNI_GC_NUMBER_DECIMAL,
	UNI_GC_NUMBER_LETTER,
	UNI_GC_NUMBER_OTHER,

	UNI_GC_PUNCT_CONNECTOR,
	UNI_GC_PUNCT_DASH,
	UNI_GC_PUNCT_OPEN,
	UNI_GC_PUNCT_CLOSE,
	UNI_GC_PUNCT_INIT_QUOTE,
	UNI_GC_PUNCT_FINAL_QUOTE,
	UNI_GC_PUNCT_OTHER,

	UNI_GC_SYMBOL_MATH,
	UNI_GC_SYMBOL_CURRENCY,
	UNI_GC_SYMBOL_MODIFIER,
	UNI_GC_SYMBOL_OTHER,

	UNI_GC_SEPARATOR_SPACE,
	UNI_GC_SEPARATOR_LINE,
	UNI_GC_SEPARATOR_PARAGRAPH,

	UNI_GC_OTHER_CONTROL,
	UNI_GC_OTHER_FORMAT,
	UNI_GC_OTHER_SURROGATE,
	UNI_GC_OTHER_PRIVATE_USE,
	UNI_GC_OTHER_NOT_ASSIGNED
} uni_gc_t;

/*
 * This represents an excerpt of Table 3-6 from the Unicode standard 4.1.0
 * chapter 3.10 ``Well-Formed UTF-8 Byte Sequences''.
 *
 * ASCII characters and the lowest 64 invalid bytes 0x80..0xBF are excluded.
 */
static const struct {
	const guchar start;
	const guchar end;
} utf8_2nd_byte_tab[64] = {
	{ 0x00, 0x00 }, /* 0xC0  Invalid */
	{ 0x00, 0x00 }, /* 0xC1  Invalid */
	{ 0x80, 0xBF }, /* 0xC2  2 bytes */
	{ 0x80, 0xBF }, /* 0xC3  2 bytes */
	{ 0x80, 0xBF }, /* 0xC4  2 bytes */
	{ 0x80, 0xBF }, /* 0xC5  2 bytes */
	{ 0x80, 0xBF }, /* 0xC6  2 bytes */
	{ 0x80, 0xBF }, /* 0xC7  2 bytes */
	{ 0x80, 0xBF }, /* 0xC8  2 bytes */
	{ 0x80, 0xBF }, /* 0xC9  2 bytes */
	{ 0x80, 0xBF }, /* 0xCA  2 bytes */
	{ 0x80, 0xBF }, /* 0xCB  2 bytes */
	{ 0x80, 0xBF }, /* 0xCC  2 bytes */
	{ 0x80, 0xBF }, /* 0xCD  2 bytes */
	{ 0x80, 0xBF }, /* 0xCE  2 bytes */
	{ 0x80, 0xBF }, /* 0xCF  2 bytes */
	{ 0x80, 0xBF }, /* 0xD0  2 bytes */
	{ 0x80, 0xBF }, /* 0xD1  2 bytes */
	{ 0x80, 0xBF }, /* 0xD2  2 bytes */
	{ 0x80, 0xBF }, /* 0xD3  2 bytes */
	{ 0x80, 0xBF }, /* 0xD4  2 bytes */
	{ 0x80, 0xBF }, /* 0xD5  2 bytes */
	{ 0x80, 0xBF }, /* 0xD6  2 bytes */
	{ 0x80, 0xBF }, /* 0xD7  2 bytes */
	{ 0x80, 0xBF }, /* 0xD8  2 bytes */
	{ 0x80, 0xBF }, /* 0xD9  2 bytes */
	{ 0x80, 0xBF }, /* 0xDA  2 bytes */
	{ 0x80, 0xBF }, /* 0xDB  2 bytes */
	{ 0x80, 0xBF }, /* 0xDC  2 bytes */
	{ 0x80, 0xBF }, /* 0xDD  2 bytes */
	{ 0x80, 0xBF }, /* 0xDE  2 bytes */
	{ 0x80, 0xBF }, /* 0xDF  2 bytes */
	{ 0xA0, 0xBF }, /* 0xE0  3 bytes */
	{ 0x80, 0xBF }, /* 0xE1  3 bytes */
	{ 0x80, 0xBF }, /* 0xE2  3 bytes */
	{ 0x80, 0xBF }, /* 0xE3  3 bytes */
	{ 0x80, 0xBF }, /* 0xE4  3 bytes */
	{ 0x80, 0xBF }, /* 0xE5  3 bytes */
	{ 0x80, 0xBF }, /* 0xE6  3 bytes */
	{ 0x80, 0xBF }, /* 0xE7  3 bytes */
	{ 0x80, 0xBF }, /* 0xE8  3 bytes */
	{ 0x80, 0xBF }, /* 0xE9  3 bytes */
	{ 0x80, 0xBF }, /* 0xEA  3 bytes */
	{ 0x80, 0xBF }, /* 0xEB  3 bytes */
	{ 0x80, 0xBF }, /* 0xEC  3 bytes */
	{ 0x80, 0x9F }, /* 0xED  3 bytes */
	{ 0x80, 0xBF }, /* 0xEE  3 bytes */
	{ 0x80, 0xBF }, /* 0xEF  3 bytes */
	{ 0x90, 0xBF }, /* 0xF0  4 bytes */
	{ 0x80, 0xBF }, /* 0xF1  4 bytes */
	{ 0x80, 0xBF }, /* 0xF2  4 bytes */
	{ 0x80, 0xBF }, /* 0xF3  4 bytes */
	{ 0x80, 0x8F }, /* 0xF4  4 bytes */
	{ 0x00, 0x00 }, /* 0xF5  Invalid */
	{ 0x00, 0x00 }, /* 0xF6  Invalid */
	{ 0x00, 0x00 }, /* 0xF7  Invalid */
	{ 0x00, 0x00 }, /* 0xF8  Invalid */
	{ 0x00, 0x00 }, /* 0xF9  Invalid */
	{ 0x00, 0x00 }, /* 0xFA  Invalid */
	{ 0x00, 0x00 }, /* 0xFB  Invalid */
	{ 0x00, 0x00 }, /* 0xFC  Invalid */
	{ 0x00, 0x00 }, /* 0xFD  Invalid */
	{ 0x00, 0x00 }, /* 0xFE  Invalid */
	{ 0x00, 0x00 }, /* 0xFF  Invalid */
};

/*
 * This table is extracted from UnicodeData.txt using the following
 * AWK script:
 *
	awk 'BEGIN { FS=";"}
		$6 != "" {
			if (match($6, "<.*>")) {
				s="NFKD"
				sub("<.*> ", "", $6);
			} else {
				s="NFD"
			};
			sub(".*", "UTF32_" s "(0x&)", $1);
			gsub(" ", ", 0x", $6);
			printf("\t{ %s, { 0x%s } },\n", $1, $6)}
	' UnicodeData.txt
 */

/**
 * The upper 8 bit of c are reserved for flags. The character value is, thus,
 * only the lower 24 bits.
 */
static const struct utf32_nfkd {
    const guint32 c;
    const guint32 d[UTF32_NFKD_REPLACE_MAXLEN];
} utf32_nfkd_lut[] = {
	{ UTF32_NFKD(0x00A0), { 0x0020 } },
	{ UTF32_NFKD(0x00A8), { 0x0020, 0x0308 } },
	{ UTF32_NFKD(0x00AA), { 0x0061 } },
	{ UTF32_NFKD(0x00AF), { 0x0020, 0x0304 } },
	{ UTF32_NFKD(0x00B2), { 0x0032 } },
	{ UTF32_NFKD(0x00B3), { 0x0033 } },
	{ UTF32_NFKD(0x00B4), { 0x0020, 0x0301 } },
	{ UTF32_NFKD(0x00B5), { 0x03BC } },
	{ UTF32_NFKD(0x00B8), { 0x0020, 0x0327 } },
	{ UTF32_NFKD(0x00B9), { 0x0031 } },
	{ UTF32_NFKD(0x00BA), { 0x006F } },
	{ UTF32_NFKD(0x00BC), { 0x0031, 0x2044, 0x0034 } },
	{ UTF32_NFKD(0x00BD), { 0x0031, 0x2044, 0x0032 } },
	{ UTF32_NFKD(0x00BE), { 0x0033, 0x2044, 0x0034 } },
	{ UTF32_NFD(0x00C0), { 0x0041, 0x0300 } },
	{ UTF32_NFD(0x00C1), { 0x0041, 0x0301 } },
	{ UTF32_NFD(0x00C2), { 0x0041, 0x0302 } },
	{ UTF32_NFD(0x00C3), { 0x0041, 0x0303 } },
	{ UTF32_NFD(0x00C4), { 0x0041, 0x0308 } },
	{ UTF32_NFD(0x00C5), { 0x0041, 0x030A } },
	{ UTF32_NFD(0x00C7), { 0x0043, 0x0327 } },
	{ UTF32_NFD(0x00C8), { 0x0045, 0x0300 } },
	{ UTF32_NFD(0x00C9), { 0x0045, 0x0301 } },
	{ UTF32_NFD(0x00CA), { 0x0045, 0x0302 } },
	{ UTF32_NFD(0x00CB), { 0x0045, 0x0308 } },
	{ UTF32_NFD(0x00CC), { 0x0049, 0x0300 } },
	{ UTF32_NFD(0x00CD), { 0x0049, 0x0301 } },
	{ UTF32_NFD(0x00CE), { 0x0049, 0x0302 } },
	{ UTF32_NFD(0x00CF), { 0x0049, 0x0308 } },
	{ UTF32_NFD(0x00D1), { 0x004E, 0x0303 } },
	{ UTF32_NFD(0x00D2), { 0x004F, 0x0300 } },
	{ UTF32_NFD(0x00D3), { 0x004F, 0x0301 } },
	{ UTF32_NFD(0x00D4), { 0x004F, 0x0302 } },
	{ UTF32_NFD(0x00D5), { 0x004F, 0x0303 } },
	{ UTF32_NFD(0x00D6), { 0x004F, 0x0308 } },
	{ UTF32_NFD(0x00D9), { 0x0055, 0x0300 } },
	{ UTF32_NFD(0x00DA), { 0x0055, 0x0301 } },
	{ UTF32_NFD(0x00DB), { 0x0055, 0x0302 } },
	{ UTF32_NFD(0x00DC), { 0x0055, 0x0308 } },
	{ UTF32_NFD(0x00DD), { 0x0059, 0x0301 } },
	{ UTF32_NFD(0x00E0), { 0x0061, 0x0300 } },
	{ UTF32_NFD(0x00E1), { 0x0061, 0x0301 } },
	{ UTF32_NFD(0x00E2), { 0x0061, 0x0302 } },
	{ UTF32_NFD(0x00E3), { 0x0061, 0x0303 } },
	{ UTF32_NFD(0x00E4), { 0x0061, 0x0308 } },
	{ UTF32_NFD(0x00E5), { 0x0061, 0x030A } },
	{ UTF32_NFD(0x00E7), { 0x0063, 0x0327 } },
	{ UTF32_NFD(0x00E8), { 0x0065, 0x0300 } },
	{ UTF32_NFD(0x00E9), { 0x0065, 0x0301 } },
	{ UTF32_NFD(0x00EA), { 0x0065, 0x0302 } },
	{ UTF32_NFD(0x00EB), { 0x0065, 0x0308 } },
	{ UTF32_NFD(0x00EC), { 0x0069, 0x0300 } },
	{ UTF32_NFD(0x00ED), { 0x0069, 0x0301 } },
	{ UTF32_NFD(0x00EE), { 0x0069, 0x0302 } },
	{ UTF32_NFD(0x00EF), { 0x0069, 0x0308 } },
	{ UTF32_NFD(0x00F1), { 0x006E, 0x0303 } },
	{ UTF32_NFD(0x00F2), { 0x006F, 0x0300 } },
	{ UTF32_NFD(0x00F3), { 0x006F, 0x0301 } },
	{ UTF32_NFD(0x00F4), { 0x006F, 0x0302 } },
	{ UTF32_NFD(0x00F5), { 0x006F, 0x0303 } },
	{ UTF32_NFD(0x00F6), { 0x006F, 0x0308 } },
	{ UTF32_NFD(0x00F9), { 0x0075, 0x0300 } },
	{ UTF32_NFD(0x00FA), { 0x0075, 0x0301 } },
	{ UTF32_NFD(0x00FB), { 0x0075, 0x0302 } },
	{ UTF32_NFD(0x00FC), { 0x0075, 0x0308 } },
	{ UTF32_NFD(0x00FD), { 0x0079, 0x0301 } },
	{ UTF32_NFD(0x00FF), { 0x0079, 0x0308 } },
	{ UTF32_NFD(0x0100), { 0x0041, 0x0304 } },
	{ UTF32_NFD(0x0101), { 0x0061, 0x0304 } },
	{ UTF32_NFD(0x0102), { 0x0041, 0x0306 } },
	{ UTF32_NFD(0x0103), { 0x0061, 0x0306 } },
	{ UTF32_NFD(0x0104), { 0x0041, 0x0328 } },
	{ UTF32_NFD(0x0105), { 0x0061, 0x0328 } },
	{ UTF32_NFD(0x0106), { 0x0043, 0x0301 } },
	{ UTF32_NFD(0x0107), { 0x0063, 0x0301 } },
	{ UTF32_NFD(0x0108), { 0x0043, 0x0302 } },
	{ UTF32_NFD(0x0109), { 0x0063, 0x0302 } },
	{ UTF32_NFD(0x010A), { 0x0043, 0x0307 } },
	{ UTF32_NFD(0x010B), { 0x0063, 0x0307 } },
	{ UTF32_NFD(0x010C), { 0x0043, 0x030C } },
	{ UTF32_NFD(0x010D), { 0x0063, 0x030C } },
	{ UTF32_NFD(0x010E), { 0x0044, 0x030C } },
	{ UTF32_NFD(0x010F), { 0x0064, 0x030C } },
	{ UTF32_NFD(0x0112), { 0x0045, 0x0304 } },
	{ UTF32_NFD(0x0113), { 0x0065, 0x0304 } },
	{ UTF32_NFD(0x0114), { 0x0045, 0x0306 } },
	{ UTF32_NFD(0x0115), { 0x0065, 0x0306 } },
	{ UTF32_NFD(0x0116), { 0x0045, 0x0307 } },
	{ UTF32_NFD(0x0117), { 0x0065, 0x0307 } },
	{ UTF32_NFD(0x0118), { 0x0045, 0x0328 } },
	{ UTF32_NFD(0x0119), { 0x0065, 0x0328 } },
	{ UTF32_NFD(0x011A), { 0x0045, 0x030C } },
	{ UTF32_NFD(0x011B), { 0x0065, 0x030C } },
	{ UTF32_NFD(0x011C), { 0x0047, 0x0302 } },
	{ UTF32_NFD(0x011D), { 0x0067, 0x0302 } },
	{ UTF32_NFD(0x011E), { 0x0047, 0x0306 } },
	{ UTF32_NFD(0x011F), { 0x0067, 0x0306 } },
	{ UTF32_NFD(0x0120), { 0x0047, 0x0307 } },
	{ UTF32_NFD(0x0121), { 0x0067, 0x0307 } },
	{ UTF32_NFD(0x0122), { 0x0047, 0x0327 } },
	{ UTF32_NFD(0x0123), { 0x0067, 0x0327 } },
	{ UTF32_NFD(0x0124), { 0x0048, 0x0302 } },
	{ UTF32_NFD(0x0125), { 0x0068, 0x0302 } },
	{ UTF32_NFD(0x0128), { 0x0049, 0x0303 } },
	{ UTF32_NFD(0x0129), { 0x0069, 0x0303 } },
	{ UTF32_NFD(0x012A), { 0x0049, 0x0304 } },
	{ UTF32_NFD(0x012B), { 0x0069, 0x0304 } },
	{ UTF32_NFD(0x012C), { 0x0049, 0x0306 } },
	{ UTF32_NFD(0x012D), { 0x0069, 0x0306 } },
	{ UTF32_NFD(0x012E), { 0x0049, 0x0328 } },
	{ UTF32_NFD(0x012F), { 0x0069, 0x0328 } },
	{ UTF32_NFD(0x0130), { 0x0049, 0x0307 } },
	{ UTF32_NFKD(0x0132), { 0x0049, 0x004A } },
	{ UTF32_NFKD(0x0133), { 0x0069, 0x006A } },
	{ UTF32_NFD(0x0134), { 0x004A, 0x0302 } },
	{ UTF32_NFD(0x0135), { 0x006A, 0x0302 } },
	{ UTF32_NFD(0x0136), { 0x004B, 0x0327 } },
	{ UTF32_NFD(0x0137), { 0x006B, 0x0327 } },
	{ UTF32_NFD(0x0139), { 0x004C, 0x0301 } },
	{ UTF32_NFD(0x013A), { 0x006C, 0x0301 } },
	{ UTF32_NFD(0x013B), { 0x004C, 0x0327 } },
	{ UTF32_NFD(0x013C), { 0x006C, 0x0327 } },
	{ UTF32_NFD(0x013D), { 0x004C, 0x030C } },
	{ UTF32_NFD(0x013E), { 0x006C, 0x030C } },
	{ UTF32_NFKD(0x013F), { 0x004C, 0x00B7 } },
	{ UTF32_NFKD(0x0140), { 0x006C, 0x00B7 } },
	{ UTF32_NFD(0x0143), { 0x004E, 0x0301 } },
	{ UTF32_NFD(0x0144), { 0x006E, 0x0301 } },
	{ UTF32_NFD(0x0145), { 0x004E, 0x0327 } },
	{ UTF32_NFD(0x0146), { 0x006E, 0x0327 } },
	{ UTF32_NFD(0x0147), { 0x004E, 0x030C } },
	{ UTF32_NFD(0x0148), { 0x006E, 0x030C } },
	{ UTF32_NFKD(0x0149), { 0x02BC, 0x006E } },
	{ UTF32_NFD(0x014C), { 0x004F, 0x0304 } },
	{ UTF32_NFD(0x014D), { 0x006F, 0x0304 } },
	{ UTF32_NFD(0x014E), { 0x004F, 0x0306 } },
	{ UTF32_NFD(0x014F), { 0x006F, 0x0306 } },
	{ UTF32_NFD(0x0150), { 0x004F, 0x030B } },
	{ UTF32_NFD(0x0151), { 0x006F, 0x030B } },
	{ UTF32_NFD(0x0154), { 0x0052, 0x0301 } },
	{ UTF32_NFD(0x0155), { 0x0072, 0x0301 } },
	{ UTF32_NFD(0x0156), { 0x0052, 0x0327 } },
	{ UTF32_NFD(0x0157), { 0x0072, 0x0327 } },
	{ UTF32_NFD(0x0158), { 0x0052, 0x030C } },
	{ UTF32_NFD(0x0159), { 0x0072, 0x030C } },
	{ UTF32_NFD(0x015A), { 0x0053, 0x0301 } },
	{ UTF32_NFD(0x015B), { 0x0073, 0x0301 } },
	{ UTF32_NFD(0x015C), { 0x0053, 0x0302 } },
	{ UTF32_NFD(0x015D), { 0x0073, 0x0302 } },
	{ UTF32_NFD(0x015E), { 0x0053, 0x0327 } },
	{ UTF32_NFD(0x015F), { 0x0073, 0x0327 } },
	{ UTF32_NFD(0x0160), { 0x0053, 0x030C } },
	{ UTF32_NFD(0x0161), { 0x0073, 0x030C } },
	{ UTF32_NFD(0x0162), { 0x0054, 0x0327 } },
	{ UTF32_NFD(0x0163), { 0x0074, 0x0327 } },
	{ UTF32_NFD(0x0164), { 0x0054, 0x030C } },
	{ UTF32_NFD(0x0165), { 0x0074, 0x030C } },
	{ UTF32_NFD(0x0168), { 0x0055, 0x0303 } },
	{ UTF32_NFD(0x0169), { 0x0075, 0x0303 } },
	{ UTF32_NFD(0x016A), { 0x0055, 0x0304 } },
	{ UTF32_NFD(0x016B), { 0x0075, 0x0304 } },
	{ UTF32_NFD(0x016C), { 0x0055, 0x0306 } },
	{ UTF32_NFD(0x016D), { 0x0075, 0x0306 } },
	{ UTF32_NFD(0x016E), { 0x0055, 0x030A } },
	{ UTF32_NFD(0x016F), { 0x0075, 0x030A } },
	{ UTF32_NFD(0x0170), { 0x0055, 0x030B } },
	{ UTF32_NFD(0x0171), { 0x0075, 0x030B } },
	{ UTF32_NFD(0x0172), { 0x0055, 0x0328 } },
	{ UTF32_NFD(0x0173), { 0x0075, 0x0328 } },
	{ UTF32_NFD(0x0174), { 0x0057, 0x0302 } },
	{ UTF32_NFD(0x0175), { 0x0077, 0x0302 } },
	{ UTF32_NFD(0x0176), { 0x0059, 0x0302 } },
	{ UTF32_NFD(0x0177), { 0x0079, 0x0302 } },
	{ UTF32_NFD(0x0178), { 0x0059, 0x0308 } },
	{ UTF32_NFD(0x0179), { 0x005A, 0x0301 } },
	{ UTF32_NFD(0x017A), { 0x007A, 0x0301 } },
	{ UTF32_NFD(0x017B), { 0x005A, 0x0307 } },
	{ UTF32_NFD(0x017C), { 0x007A, 0x0307 } },
	{ UTF32_NFD(0x017D), { 0x005A, 0x030C } },
	{ UTF32_NFD(0x017E), { 0x007A, 0x030C } },
	{ UTF32_NFKD(0x017F), { 0x0073 } },
	{ UTF32_NFD(0x01A0), { 0x004F, 0x031B } },
	{ UTF32_NFD(0x01A1), { 0x006F, 0x031B } },
	{ UTF32_NFD(0x01AF), { 0x0055, 0x031B } },
	{ UTF32_NFD(0x01B0), { 0x0075, 0x031B } },
	{ UTF32_NFKD(0x01C4), { 0x0044, 0x017D } },
	{ UTF32_NFKD(0x01C5), { 0x0044, 0x017E } },
	{ UTF32_NFKD(0x01C6), { 0x0064, 0x017E } },
	{ UTF32_NFKD(0x01C7), { 0x004C, 0x004A } },
	{ UTF32_NFKD(0x01C8), { 0x004C, 0x006A } },
	{ UTF32_NFKD(0x01C9), { 0x006C, 0x006A } },
	{ UTF32_NFKD(0x01CA), { 0x004E, 0x004A } },
	{ UTF32_NFKD(0x01CB), { 0x004E, 0x006A } },
	{ UTF32_NFKD(0x01CC), { 0x006E, 0x006A } },
	{ UTF32_NFD(0x01CD), { 0x0041, 0x030C } },
	{ UTF32_NFD(0x01CE), { 0x0061, 0x030C } },
	{ UTF32_NFD(0x01CF), { 0x0049, 0x030C } },
	{ UTF32_NFD(0x01D0), { 0x0069, 0x030C } },
	{ UTF32_NFD(0x01D1), { 0x004F, 0x030C } },
	{ UTF32_NFD(0x01D2), { 0x006F, 0x030C } },
	{ UTF32_NFD(0x01D3), { 0x0055, 0x030C } },
	{ UTF32_NFD(0x01D4), { 0x0075, 0x030C } },
	{ UTF32_NFD(0x01D5), { 0x00DC, 0x0304 } },
	{ UTF32_NFD(0x01D6), { 0x00FC, 0x0304 } },
	{ UTF32_NFD(0x01D7), { 0x00DC, 0x0301 } },
	{ UTF32_NFD(0x01D8), { 0x00FC, 0x0301 } },
	{ UTF32_NFD(0x01D9), { 0x00DC, 0x030C } },
	{ UTF32_NFD(0x01DA), { 0x00FC, 0x030C } },
	{ UTF32_NFD(0x01DB), { 0x00DC, 0x0300 } },
	{ UTF32_NFD(0x01DC), { 0x00FC, 0x0300 } },
	{ UTF32_NFD(0x01DE), { 0x00C4, 0x0304 } },
	{ UTF32_NFD(0x01DF), { 0x00E4, 0x0304 } },
	{ UTF32_NFD(0x01E0), { 0x0226, 0x0304 } },
	{ UTF32_NFD(0x01E1), { 0x0227, 0x0304 } },
	{ UTF32_NFD(0x01E2), { 0x00C6, 0x0304 } },
	{ UTF32_NFD(0x01E3), { 0x00E6, 0x0304 } },
	{ UTF32_NFD(0x01E6), { 0x0047, 0x030C } },
	{ UTF32_NFD(0x01E7), { 0x0067, 0x030C } },
	{ UTF32_NFD(0x01E8), { 0x004B, 0x030C } },
	{ UTF32_NFD(0x01E9), { 0x006B, 0x030C } },
	{ UTF32_NFD(0x01EA), { 0x004F, 0x0328 } },
	{ UTF32_NFD(0x01EB), { 0x006F, 0x0328 } },
	{ UTF32_NFD(0x01EC), { 0x01EA, 0x0304 } },
	{ UTF32_NFD(0x01ED), { 0x01EB, 0x0304 } },
	{ UTF32_NFD(0x01EE), { 0x01B7, 0x030C } },
	{ UTF32_NFD(0x01EF), { 0x0292, 0x030C } },
	{ UTF32_NFD(0x01F0), { 0x006A, 0x030C } },
	{ UTF32_NFKD(0x01F1), { 0x0044, 0x005A } },
	{ UTF32_NFKD(0x01F2), { 0x0044, 0x007A } },
	{ UTF32_NFKD(0x01F3), { 0x0064, 0x007A } },
	{ UTF32_NFD(0x01F4), { 0x0047, 0x0301 } },
	{ UTF32_NFD(0x01F5), { 0x0067, 0x0301 } },
	{ UTF32_NFD(0x01F8), { 0x004E, 0x0300 } },
	{ UTF32_NFD(0x01F9), { 0x006E, 0x0300 } },
	{ UTF32_NFD(0x01FA), { 0x00C5, 0x0301 } },
	{ UTF32_NFD(0x01FB), { 0x00E5, 0x0301 } },
	{ UTF32_NFD(0x01FC), { 0x00C6, 0x0301 } },
	{ UTF32_NFD(0x01FD), { 0x00E6, 0x0301 } },
	{ UTF32_NFD(0x01FE), { 0x00D8, 0x0301 } },
	{ UTF32_NFD(0x01FF), { 0x00F8, 0x0301 } },
	{ UTF32_NFD(0x0200), { 0x0041, 0x030F } },
	{ UTF32_NFD(0x0201), { 0x0061, 0x030F } },
	{ UTF32_NFD(0x0202), { 0x0041, 0x0311 } },
	{ UTF32_NFD(0x0203), { 0x0061, 0x0311 } },
	{ UTF32_NFD(0x0204), { 0x0045, 0x030F } },
	{ UTF32_NFD(0x0205), { 0x0065, 0x030F } },
	{ UTF32_NFD(0x0206), { 0x0045, 0x0311 } },
	{ UTF32_NFD(0x0207), { 0x0065, 0x0311 } },
	{ UTF32_NFD(0x0208), { 0x0049, 0x030F } },
	{ UTF32_NFD(0x0209), { 0x0069, 0x030F } },
	{ UTF32_NFD(0x020A), { 0x0049, 0x0311 } },
	{ UTF32_NFD(0x020B), { 0x0069, 0x0311 } },
	{ UTF32_NFD(0x020C), { 0x004F, 0x030F } },
	{ UTF32_NFD(0x020D), { 0x006F, 0x030F } },
	{ UTF32_NFD(0x020E), { 0x004F, 0x0311 } },
	{ UTF32_NFD(0x020F), { 0x006F, 0x0311 } },
	{ UTF32_NFD(0x0210), { 0x0052, 0x030F } },
	{ UTF32_NFD(0x0211), { 0x0072, 0x030F } },
	{ UTF32_NFD(0x0212), { 0x0052, 0x0311 } },
	{ UTF32_NFD(0x0213), { 0x0072, 0x0311 } },
	{ UTF32_NFD(0x0214), { 0x0055, 0x030F } },
	{ UTF32_NFD(0x0215), { 0x0075, 0x030F } },
	{ UTF32_NFD(0x0216), { 0x0055, 0x0311 } },
	{ UTF32_NFD(0x0217), { 0x0075, 0x0311 } },
	{ UTF32_NFD(0x0218), { 0x0053, 0x0326 } },
	{ UTF32_NFD(0x0219), { 0x0073, 0x0326 } },
	{ UTF32_NFD(0x021A), { 0x0054, 0x0326 } },
	{ UTF32_NFD(0x021B), { 0x0074, 0x0326 } },
	{ UTF32_NFD(0x021E), { 0x0048, 0x030C } },
	{ UTF32_NFD(0x021F), { 0x0068, 0x030C } },
	{ UTF32_NFD(0x0226), { 0x0041, 0x0307 } },
	{ UTF32_NFD(0x0227), { 0x0061, 0x0307 } },
	{ UTF32_NFD(0x0228), { 0x0045, 0x0327 } },
	{ UTF32_NFD(0x0229), { 0x0065, 0x0327 } },
	{ UTF32_NFD(0x022A), { 0x00D6, 0x0304 } },
	{ UTF32_NFD(0x022B), { 0x00F6, 0x0304 } },
	{ UTF32_NFD(0x022C), { 0x00D5, 0x0304 } },
	{ UTF32_NFD(0x022D), { 0x00F5, 0x0304 } },
	{ UTF32_NFD(0x022E), { 0x004F, 0x0307 } },
	{ UTF32_NFD(0x022F), { 0x006F, 0x0307 } },
	{ UTF32_NFD(0x0230), { 0x022E, 0x0304 } },
	{ UTF32_NFD(0x0231), { 0x022F, 0x0304 } },
	{ UTF32_NFD(0x0232), { 0x0059, 0x0304 } },
	{ UTF32_NFD(0x0233), { 0x0079, 0x0304 } },
	{ UTF32_NFKD(0x02B0), { 0x0068 } },
	{ UTF32_NFKD(0x02B1), { 0x0266 } },
	{ UTF32_NFKD(0x02B2), { 0x006A } },
	{ UTF32_NFKD(0x02B3), { 0x0072 } },
	{ UTF32_NFKD(0x02B4), { 0x0279 } },
	{ UTF32_NFKD(0x02B5), { 0x027B } },
	{ UTF32_NFKD(0x02B6), { 0x0281 } },
	{ UTF32_NFKD(0x02B7), { 0x0077 } },
	{ UTF32_NFKD(0x02B8), { 0x0079 } },
	{ UTF32_NFKD(0x02D8), { 0x0020, 0x0306 } },
	{ UTF32_NFKD(0x02D9), { 0x0020, 0x0307 } },
	{ UTF32_NFKD(0x02DA), { 0x0020, 0x030A } },
	{ UTF32_NFKD(0x02DB), { 0x0020, 0x0328 } },
	{ UTF32_NFKD(0x02DC), { 0x0020, 0x0303 } },
	{ UTF32_NFKD(0x02DD), { 0x0020, 0x030B } },
	{ UTF32_NFKD(0x02E0), { 0x0263 } },
	{ UTF32_NFKD(0x02E1), { 0x006C } },
	{ UTF32_NFKD(0x02E2), { 0x0073 } },
	{ UTF32_NFKD(0x02E3), { 0x0078 } },
	{ UTF32_NFKD(0x02E4), { 0x0295 } },
	{ UTF32_NFD(0x0340), { 0x0300 } },
	{ UTF32_NFD(0x0341), { 0x0301 } },
	{ UTF32_NFD(0x0343), { 0x0313 } },
	{ UTF32_NFD(0x0344), { 0x0308, 0x0301 } },
	{ UTF32_NFD(0x0374), { 0x02B9 } },
	{ UTF32_NFKD(0x037A), { 0x0020, 0x0345 } },
	{ UTF32_NFD(0x037E), { 0x003B } },
	{ UTF32_NFKD(0x0384), { 0x0020, 0x0301 } },
	{ UTF32_NFD(0x0385), { 0x00A8, 0x0301 } },
	{ UTF32_NFD(0x0386), { 0x0391, 0x0301 } },
	{ UTF32_NFD(0x0387), { 0x00B7 } },
	{ UTF32_NFD(0x0388), { 0x0395, 0x0301 } },
	{ UTF32_NFD(0x0389), { 0x0397, 0x0301 } },
	{ UTF32_NFD(0x038A), { 0x0399, 0x0301 } },
	{ UTF32_NFD(0x038C), { 0x039F, 0x0301 } },
	{ UTF32_NFD(0x038E), { 0x03A5, 0x0301 } },
	{ UTF32_NFD(0x038F), { 0x03A9, 0x0301 } },
	{ UTF32_NFD(0x0390), { 0x03CA, 0x0301 } },
	{ UTF32_NFD(0x03AA), { 0x0399, 0x0308 } },
	{ UTF32_NFD(0x03AB), { 0x03A5, 0x0308 } },
	{ UTF32_NFD(0x03AC), { 0x03B1, 0x0301 } },
	{ UTF32_NFD(0x03AD), { 0x03B5, 0x0301 } },
	{ UTF32_NFD(0x03AE), { 0x03B7, 0x0301 } },
	{ UTF32_NFD(0x03AF), { 0x03B9, 0x0301 } },
	{ UTF32_NFD(0x03B0), { 0x03CB, 0x0301 } },
	{ UTF32_NFD(0x03CA), { 0x03B9, 0x0308 } },
	{ UTF32_NFD(0x03CB), { 0x03C5, 0x0308 } },
	{ UTF32_NFD(0x03CC), { 0x03BF, 0x0301 } },
	{ UTF32_NFD(0x03CD), { 0x03C5, 0x0301 } },
	{ UTF32_NFD(0x03CE), { 0x03C9, 0x0301 } },
	{ UTF32_NFKD(0x03D0), { 0x03B2 } },
	{ UTF32_NFKD(0x03D1), { 0x03B8 } },
	{ UTF32_NFKD(0x03D2), { 0x03A5 } },
	{ UTF32_NFD(0x03D3), { 0x03D2, 0x0301 } },
	{ UTF32_NFD(0x03D4), { 0x03D2, 0x0308 } },
	{ UTF32_NFKD(0x03D5), { 0x03C6 } },
	{ UTF32_NFKD(0x03D6), { 0x03C0 } },
	{ UTF32_NFKD(0x03F0), { 0x03BA } },
	{ UTF32_NFKD(0x03F1), { 0x03C1 } },
	{ UTF32_NFKD(0x03F2), { 0x03C2 } },
	{ UTF32_NFKD(0x03F4), { 0x0398 } },
	{ UTF32_NFKD(0x03F5), { 0x03B5 } },
	{ UTF32_NFKD(0x03F9), { 0x03A3 } },
	{ UTF32_NFD(0x0400), { 0x0415, 0x0300 } },
	{ UTF32_NFD(0x0401), { 0x0415, 0x0308 } },
	{ UTF32_NFD(0x0403), { 0x0413, 0x0301 } },
	{ UTF32_NFD(0x0407), { 0x0406, 0x0308 } },
	{ UTF32_NFD(0x040C), { 0x041A, 0x0301 } },
	{ UTF32_NFD(0x040D), { 0x0418, 0x0300 } },
	{ UTF32_NFD(0x040E), { 0x0423, 0x0306 } },
	{ UTF32_NFD(0x0419), { 0x0418, 0x0306 } },
	{ UTF32_NFD(0x0439), { 0x0438, 0x0306 } },
	{ UTF32_NFD(0x0450), { 0x0435, 0x0300 } },
	{ UTF32_NFD(0x0451), { 0x0435, 0x0308 } },
	{ UTF32_NFD(0x0453), { 0x0433, 0x0301 } },
	{ UTF32_NFD(0x0457), { 0x0456, 0x0308 } },
	{ UTF32_NFD(0x045C), { 0x043A, 0x0301 } },
	{ UTF32_NFD(0x045D), { 0x0438, 0x0300 } },
	{ UTF32_NFD(0x045E), { 0x0443, 0x0306 } },
	{ UTF32_NFD(0x0476), { 0x0474, 0x030F } },
	{ UTF32_NFD(0x0477), { 0x0475, 0x030F } },
	{ UTF32_NFD(0x04C1), { 0x0416, 0x0306 } },
	{ UTF32_NFD(0x04C2), { 0x0436, 0x0306 } },
	{ UTF32_NFD(0x04D0), { 0x0410, 0x0306 } },
	{ UTF32_NFD(0x04D1), { 0x0430, 0x0306 } },
	{ UTF32_NFD(0x04D2), { 0x0410, 0x0308 } },
	{ UTF32_NFD(0x04D3), { 0x0430, 0x0308 } },
	{ UTF32_NFD(0x04D6), { 0x0415, 0x0306 } },
	{ UTF32_NFD(0x04D7), { 0x0435, 0x0306 } },
	{ UTF32_NFD(0x04DA), { 0x04D8, 0x0308 } },
	{ UTF32_NFD(0x04DB), { 0x04D9, 0x0308 } },
	{ UTF32_NFD(0x04DC), { 0x0416, 0x0308 } },
	{ UTF32_NFD(0x04DD), { 0x0436, 0x0308 } },
	{ UTF32_NFD(0x04DE), { 0x0417, 0x0308 } },
	{ UTF32_NFD(0x04DF), { 0x0437, 0x0308 } },
	{ UTF32_NFD(0x04E2), { 0x0418, 0x0304 } },
	{ UTF32_NFD(0x04E3), { 0x0438, 0x0304 } },
	{ UTF32_NFD(0x04E4), { 0x0418, 0x0308 } },
	{ UTF32_NFD(0x04E5), { 0x0438, 0x0308 } },
	{ UTF32_NFD(0x04E6), { 0x041E, 0x0308 } },
	{ UTF32_NFD(0x04E7), { 0x043E, 0x0308 } },
	{ UTF32_NFD(0x04EA), { 0x04E8, 0x0308 } },
	{ UTF32_NFD(0x04EB), { 0x04E9, 0x0308 } },
	{ UTF32_NFD(0x04EC), { 0x042D, 0x0308 } },
	{ UTF32_NFD(0x04ED), { 0x044D, 0x0308 } },
	{ UTF32_NFD(0x04EE), { 0x0423, 0x0304 } },
	{ UTF32_NFD(0x04EF), { 0x0443, 0x0304 } },
	{ UTF32_NFD(0x04F0), { 0x0423, 0x0308 } },
	{ UTF32_NFD(0x04F1), { 0x0443, 0x0308 } },
	{ UTF32_NFD(0x04F2), { 0x0423, 0x030B } },
	{ UTF32_NFD(0x04F3), { 0x0443, 0x030B } },
	{ UTF32_NFD(0x04F4), { 0x0427, 0x0308 } },
	{ UTF32_NFD(0x04F5), { 0x0447, 0x0308 } },
	{ UTF32_NFD(0x04F8), { 0x042B, 0x0308 } },
	{ UTF32_NFD(0x04F9), { 0x044B, 0x0308 } },
	{ UTF32_NFKD(0x0587), { 0x0565, 0x0582 } },
	{ UTF32_NFD(0x0622), { 0x0627, 0x0653 } },
	{ UTF32_NFD(0x0623), { 0x0627, 0x0654 } },
	{ UTF32_NFD(0x0624), { 0x0648, 0x0654 } },
	{ UTF32_NFD(0x0625), { 0x0627, 0x0655 } },
	{ UTF32_NFD(0x0626), { 0x064A, 0x0654 } },
	{ UTF32_NFKD(0x0675), { 0x0627, 0x0674 } },
	{ UTF32_NFKD(0x0676), { 0x0648, 0x0674 } },
	{ UTF32_NFKD(0x0677), { 0x06C7, 0x0674 } },
	{ UTF32_NFKD(0x0678), { 0x064A, 0x0674 } },
	{ UTF32_NFD(0x06C0), { 0x06D5, 0x0654 } },
	{ UTF32_NFD(0x06C2), { 0x06C1, 0x0654 } },
	{ UTF32_NFD(0x06D3), { 0x06D2, 0x0654 } },
	{ UTF32_NFD(0x0929), { 0x0928, 0x093C } },
	{ UTF32_NFD(0x0931), { 0x0930, 0x093C } },
	{ UTF32_NFD(0x0934), { 0x0933, 0x093C } },
	{ UTF32_NFD(0x0958), { 0x0915, 0x093C } },
	{ UTF32_NFD(0x0959), { 0x0916, 0x093C } },
	{ UTF32_NFD(0x095A), { 0x0917, 0x093C } },
	{ UTF32_NFD(0x095B), { 0x091C, 0x093C } },
	{ UTF32_NFD(0x095C), { 0x0921, 0x093C } },
	{ UTF32_NFD(0x095D), { 0x0922, 0x093C } },
	{ UTF32_NFD(0x095E), { 0x092B, 0x093C } },
	{ UTF32_NFD(0x095F), { 0x092F, 0x093C } },
	{ UTF32_NFD(0x09CB), { 0x09C7, 0x09BE } },
	{ UTF32_NFD(0x09CC), { 0x09C7, 0x09D7 } },
	{ UTF32_NFD(0x09DC), { 0x09A1, 0x09BC } },
	{ UTF32_NFD(0x09DD), { 0x09A2, 0x09BC } },
	{ UTF32_NFD(0x09DF), { 0x09AF, 0x09BC } },
	{ UTF32_NFD(0x0A33), { 0x0A32, 0x0A3C } },
	{ UTF32_NFD(0x0A36), { 0x0A38, 0x0A3C } },
	{ UTF32_NFD(0x0A59), { 0x0A16, 0x0A3C } },
	{ UTF32_NFD(0x0A5A), { 0x0A17, 0x0A3C } },
	{ UTF32_NFD(0x0A5B), { 0x0A1C, 0x0A3C } },
	{ UTF32_NFD(0x0A5E), { 0x0A2B, 0x0A3C } },
	{ UTF32_NFD(0x0B48), { 0x0B47, 0x0B56 } },
	{ UTF32_NFD(0x0B4B), { 0x0B47, 0x0B3E } },
	{ UTF32_NFD(0x0B4C), { 0x0B47, 0x0B57 } },
	{ UTF32_NFD(0x0B5C), { 0x0B21, 0x0B3C } },
	{ UTF32_NFD(0x0B5D), { 0x0B22, 0x0B3C } },
	{ UTF32_NFD(0x0B94), { 0x0B92, 0x0BD7 } },
	{ UTF32_NFD(0x0BCA), { 0x0BC6, 0x0BBE } },
	{ UTF32_NFD(0x0BCB), { 0x0BC7, 0x0BBE } },
	{ UTF32_NFD(0x0BCC), { 0x0BC6, 0x0BD7 } },
	{ UTF32_NFD(0x0C48), { 0x0C46, 0x0C56 } },
	{ UTF32_NFD(0x0CC0), { 0x0CBF, 0x0CD5 } },
	{ UTF32_NFD(0x0CC7), { 0x0CC6, 0x0CD5 } },
	{ UTF32_NFD(0x0CC8), { 0x0CC6, 0x0CD6 } },
	{ UTF32_NFD(0x0CCA), { 0x0CC6, 0x0CC2 } },
	{ UTF32_NFD(0x0CCB), { 0x0CCA, 0x0CD5 } },
	{ UTF32_NFD(0x0D4A), { 0x0D46, 0x0D3E } },
	{ UTF32_NFD(0x0D4B), { 0x0D47, 0x0D3E } },
	{ UTF32_NFD(0x0D4C), { 0x0D46, 0x0D57 } },
	{ UTF32_NFD(0x0DDA), { 0x0DD9, 0x0DCA } },
	{ UTF32_NFD(0x0DDC), { 0x0DD9, 0x0DCF } },
	{ UTF32_NFD(0x0DDD), { 0x0DDC, 0x0DCA } },
	{ UTF32_NFD(0x0DDE), { 0x0DD9, 0x0DDF } },
	{ UTF32_NFKD(0x0E33), { 0x0E4D, 0x0E32 } },
	{ UTF32_NFKD(0x0EB3), { 0x0ECD, 0x0EB2 } },
	{ UTF32_NFKD(0x0EDC), { 0x0EAB, 0x0E99 } },
	{ UTF32_NFKD(0x0EDD), { 0x0EAB, 0x0EA1 } },
	{ UTF32_NFKD(0x0F0C), { 0x0F0B } },
	{ UTF32_NFD(0x0F43), { 0x0F42, 0x0FB7 } },
	{ UTF32_NFD(0x0F4D), { 0x0F4C, 0x0FB7 } },
	{ UTF32_NFD(0x0F52), { 0x0F51, 0x0FB7 } },
	{ UTF32_NFD(0x0F57), { 0x0F56, 0x0FB7 } },
	{ UTF32_NFD(0x0F5C), { 0x0F5B, 0x0FB7 } },
	{ UTF32_NFD(0x0F69), { 0x0F40, 0x0FB5 } },
	{ UTF32_NFD(0x0F73), { 0x0F71, 0x0F72 } },
	{ UTF32_NFD(0x0F75), { 0x0F71, 0x0F74 } },
	{ UTF32_NFD(0x0F76), { 0x0FB2, 0x0F80 } },
	{ UTF32_NFKD(0x0F77), { 0x0FB2, 0x0F81 } },
	{ UTF32_NFD(0x0F78), { 0x0FB3, 0x0F80 } },
	{ UTF32_NFKD(0x0F79), { 0x0FB3, 0x0F81 } },
	{ UTF32_NFD(0x0F81), { 0x0F71, 0x0F80 } },
	{ UTF32_NFD(0x0F93), { 0x0F92, 0x0FB7 } },
	{ UTF32_NFD(0x0F9D), { 0x0F9C, 0x0FB7 } },
	{ UTF32_NFD(0x0FA2), { 0x0FA1, 0x0FB7 } },
	{ UTF32_NFD(0x0FA7), { 0x0FA6, 0x0FB7 } },
	{ UTF32_NFD(0x0FAC), { 0x0FAB, 0x0FB7 } },
	{ UTF32_NFD(0x0FB9), { 0x0F90, 0x0FB5 } },
	{ UTF32_NFD(0x1026), { 0x1025, 0x102E } },
	{ UTF32_NFKD(0x10FC), { 0x10DC } },
	{ UTF32_NFKD(0x1D2C), { 0x0041 } },
	{ UTF32_NFKD(0x1D2D), { 0x00C6 } },
	{ UTF32_NFKD(0x1D2E), { 0x0042 } },
	{ UTF32_NFKD(0x1D30), { 0x0044 } },
	{ UTF32_NFKD(0x1D31), { 0x0045 } },
	{ UTF32_NFKD(0x1D32), { 0x018E } },
	{ UTF32_NFKD(0x1D33), { 0x0047 } },
	{ UTF32_NFKD(0x1D34), { 0x0048 } },
	{ UTF32_NFKD(0x1D35), { 0x0049 } },
	{ UTF32_NFKD(0x1D36), { 0x004A } },
	{ UTF32_NFKD(0x1D37), { 0x004B } },
	{ UTF32_NFKD(0x1D38), { 0x004C } },
	{ UTF32_NFKD(0x1D39), { 0x004D } },
	{ UTF32_NFKD(0x1D3A), { 0x004E } },
	{ UTF32_NFKD(0x1D3C), { 0x004F } },
	{ UTF32_NFKD(0x1D3D), { 0x0222 } },
	{ UTF32_NFKD(0x1D3E), { 0x0050 } },
	{ UTF32_NFKD(0x1D3F), { 0x0052 } },
	{ UTF32_NFKD(0x1D40), { 0x0054 } },
	{ UTF32_NFKD(0x1D41), { 0x0055 } },
	{ UTF32_NFKD(0x1D42), { 0x0057 } },
	{ UTF32_NFKD(0x1D43), { 0x0061 } },
	{ UTF32_NFKD(0x1D44), { 0x0250 } },
	{ UTF32_NFKD(0x1D45), { 0x0251 } },
	{ UTF32_NFKD(0x1D46), { 0x1D02 } },
	{ UTF32_NFKD(0x1D47), { 0x0062 } },
	{ UTF32_NFKD(0x1D48), { 0x0064 } },
	{ UTF32_NFKD(0x1D49), { 0x0065 } },
	{ UTF32_NFKD(0x1D4A), { 0x0259 } },
	{ UTF32_NFKD(0x1D4B), { 0x025B } },
	{ UTF32_NFKD(0x1D4C), { 0x025C } },
	{ UTF32_NFKD(0x1D4D), { 0x0067 } },
	{ UTF32_NFKD(0x1D4F), { 0x006B } },
	{ UTF32_NFKD(0x1D50), { 0x006D } },
	{ UTF32_NFKD(0x1D51), { 0x014B } },
	{ UTF32_NFKD(0x1D52), { 0x006F } },
	{ UTF32_NFKD(0x1D53), { 0x0254 } },
	{ UTF32_NFKD(0x1D54), { 0x1D16 } },
	{ UTF32_NFKD(0x1D55), { 0x1D17 } },
	{ UTF32_NFKD(0x1D56), { 0x0070 } },
	{ UTF32_NFKD(0x1D57), { 0x0074 } },
	{ UTF32_NFKD(0x1D58), { 0x0075 } },
	{ UTF32_NFKD(0x1D59), { 0x1D1D } },
	{ UTF32_NFKD(0x1D5A), { 0x026F } },
	{ UTF32_NFKD(0x1D5B), { 0x0076 } },
	{ UTF32_NFKD(0x1D5C), { 0x1D25 } },
	{ UTF32_NFKD(0x1D5D), { 0x03B2 } },
	{ UTF32_NFKD(0x1D5E), { 0x03B3 } },
	{ UTF32_NFKD(0x1D5F), { 0x03B4 } },
	{ UTF32_NFKD(0x1D60), { 0x03C6 } },
	{ UTF32_NFKD(0x1D61), { 0x03C7 } },
	{ UTF32_NFKD(0x1D62), { 0x0069 } },
	{ UTF32_NFKD(0x1D63), { 0x0072 } },
	{ UTF32_NFKD(0x1D64), { 0x0075 } },
	{ UTF32_NFKD(0x1D65), { 0x0076 } },
	{ UTF32_NFKD(0x1D66), { 0x03B2 } },
	{ UTF32_NFKD(0x1D67), { 0x03B3 } },
	{ UTF32_NFKD(0x1D68), { 0x03C1 } },
	{ UTF32_NFKD(0x1D69), { 0x03C6 } },
	{ UTF32_NFKD(0x1D6A), { 0x03C7 } },
	{ UTF32_NFKD(0x1D78), { 0x043D } },
	{ UTF32_NFKD(0x1D9B), { 0x0252 } },
	{ UTF32_NFKD(0x1D9C), { 0x0063 } },
	{ UTF32_NFKD(0x1D9D), { 0x0255 } },
	{ UTF32_NFKD(0x1D9E), { 0x00F0 } },
	{ UTF32_NFKD(0x1D9F), { 0x025C } },
	{ UTF32_NFKD(0x1DA0), { 0x0066 } },
	{ UTF32_NFKD(0x1DA1), { 0x025F } },
	{ UTF32_NFKD(0x1DA2), { 0x0261 } },
	{ UTF32_NFKD(0x1DA3), { 0x0265 } },
	{ UTF32_NFKD(0x1DA4), { 0x0268 } },
	{ UTF32_NFKD(0x1DA5), { 0x0269 } },
	{ UTF32_NFKD(0x1DA6), { 0x026A } },
	{ UTF32_NFKD(0x1DA7), { 0x1D7B } },
	{ UTF32_NFKD(0x1DA8), { 0x029D } },
	{ UTF32_NFKD(0x1DA9), { 0x026D } },
	{ UTF32_NFKD(0x1DAA), { 0x1D85 } },
	{ UTF32_NFKD(0x1DAB), { 0x029F } },
	{ UTF32_NFKD(0x1DAC), { 0x0271 } },
	{ UTF32_NFKD(0x1DAD), { 0x0270 } },
	{ UTF32_NFKD(0x1DAE), { 0x0272 } },
	{ UTF32_NFKD(0x1DAF), { 0x0273 } },
	{ UTF32_NFKD(0x1DB0), { 0x0274 } },
	{ UTF32_NFKD(0x1DB1), { 0x0275 } },
	{ UTF32_NFKD(0x1DB2), { 0x0278 } },
	{ UTF32_NFKD(0x1DB3), { 0x0282 } },
	{ UTF32_NFKD(0x1DB4), { 0x0283 } },
	{ UTF32_NFKD(0x1DB5), { 0x01AB } },
	{ UTF32_NFKD(0x1DB6), { 0x0289 } },
	{ UTF32_NFKD(0x1DB7), { 0x028A } },
	{ UTF32_NFKD(0x1DB8), { 0x1D1C } },
	{ UTF32_NFKD(0x1DB9), { 0x028B } },
	{ UTF32_NFKD(0x1DBA), { 0x028C } },
	{ UTF32_NFKD(0x1DBB), { 0x007A } },
	{ UTF32_NFKD(0x1DBC), { 0x0290 } },
	{ UTF32_NFKD(0x1DBD), { 0x0291 } },
	{ UTF32_NFKD(0x1DBE), { 0x0292 } },
	{ UTF32_NFKD(0x1DBF), { 0x03B8 } },
	{ UTF32_NFD(0x1E00), { 0x0041, 0x0325 } },
	{ UTF32_NFD(0x1E01), { 0x0061, 0x0325 } },
	{ UTF32_NFD(0x1E02), { 0x0042, 0x0307 } },
	{ UTF32_NFD(0x1E03), { 0x0062, 0x0307 } },
	{ UTF32_NFD(0x1E04), { 0x0042, 0x0323 } },
	{ UTF32_NFD(0x1E05), { 0x0062, 0x0323 } },
	{ UTF32_NFD(0x1E06), { 0x0042, 0x0331 } },
	{ UTF32_NFD(0x1E07), { 0x0062, 0x0331 } },
	{ UTF32_NFD(0x1E08), { 0x00C7, 0x0301 } },
	{ UTF32_NFD(0x1E09), { 0x00E7, 0x0301 } },
	{ UTF32_NFD(0x1E0A), { 0x0044, 0x0307 } },
	{ UTF32_NFD(0x1E0B), { 0x0064, 0x0307 } },
	{ UTF32_NFD(0x1E0C), { 0x0044, 0x0323 } },
	{ UTF32_NFD(0x1E0D), { 0x0064, 0x0323 } },
	{ UTF32_NFD(0x1E0E), { 0x0044, 0x0331 } },
	{ UTF32_NFD(0x1E0F), { 0x0064, 0x0331 } },
	{ UTF32_NFD(0x1E10), { 0x0044, 0x0327 } },
	{ UTF32_NFD(0x1E11), { 0x0064, 0x0327 } },
	{ UTF32_NFD(0x1E12), { 0x0044, 0x032D } },
	{ UTF32_NFD(0x1E13), { 0x0064, 0x032D } },
	{ UTF32_NFD(0x1E14), { 0x0112, 0x0300 } },
	{ UTF32_NFD(0x1E15), { 0x0113, 0x0300 } },
	{ UTF32_NFD(0x1E16), { 0x0112, 0x0301 } },
	{ UTF32_NFD(0x1E17), { 0x0113, 0x0301 } },
	{ UTF32_NFD(0x1E18), { 0x0045, 0x032D } },
	{ UTF32_NFD(0x1E19), { 0x0065, 0x032D } },
	{ UTF32_NFD(0x1E1A), { 0x0045, 0x0330 } },
	{ UTF32_NFD(0x1E1B), { 0x0065, 0x0330 } },
	{ UTF32_NFD(0x1E1C), { 0x0228, 0x0306 } },
	{ UTF32_NFD(0x1E1D), { 0x0229, 0x0306 } },
	{ UTF32_NFD(0x1E1E), { 0x0046, 0x0307 } },
	{ UTF32_NFD(0x1E1F), { 0x0066, 0x0307 } },
	{ UTF32_NFD(0x1E20), { 0x0047, 0x0304 } },
	{ UTF32_NFD(0x1E21), { 0x0067, 0x0304 } },
	{ UTF32_NFD(0x1E22), { 0x0048, 0x0307 } },
	{ UTF32_NFD(0x1E23), { 0x0068, 0x0307 } },
	{ UTF32_NFD(0x1E24), { 0x0048, 0x0323 } },
	{ UTF32_NFD(0x1E25), { 0x0068, 0x0323 } },
	{ UTF32_NFD(0x1E26), { 0x0048, 0x0308 } },
	{ UTF32_NFD(0x1E27), { 0x0068, 0x0308 } },
	{ UTF32_NFD(0x1E28), { 0x0048, 0x0327 } },
	{ UTF32_NFD(0x1E29), { 0x0068, 0x0327 } },
	{ UTF32_NFD(0x1E2A), { 0x0048, 0x032E } },
	{ UTF32_NFD(0x1E2B), { 0x0068, 0x032E } },
	{ UTF32_NFD(0x1E2C), { 0x0049, 0x0330 } },
	{ UTF32_NFD(0x1E2D), { 0x0069, 0x0330 } },
	{ UTF32_NFD(0x1E2E), { 0x00CF, 0x0301 } },
	{ UTF32_NFD(0x1E2F), { 0x00EF, 0x0301 } },
	{ UTF32_NFD(0x1E30), { 0x004B, 0x0301 } },
	{ UTF32_NFD(0x1E31), { 0x006B, 0x0301 } },
	{ UTF32_NFD(0x1E32), { 0x004B, 0x0323 } },
	{ UTF32_NFD(0x1E33), { 0x006B, 0x0323 } },
	{ UTF32_NFD(0x1E34), { 0x004B, 0x0331 } },
	{ UTF32_NFD(0x1E35), { 0x006B, 0x0331 } },
	{ UTF32_NFD(0x1E36), { 0x004C, 0x0323 } },
	{ UTF32_NFD(0x1E37), { 0x006C, 0x0323 } },
	{ UTF32_NFD(0x1E38), { 0x1E36, 0x0304 } },
	{ UTF32_NFD(0x1E39), { 0x1E37, 0x0304 } },
	{ UTF32_NFD(0x1E3A), { 0x004C, 0x0331 } },
	{ UTF32_NFD(0x1E3B), { 0x006C, 0x0331 } },
	{ UTF32_NFD(0x1E3C), { 0x004C, 0x032D } },
	{ UTF32_NFD(0x1E3D), { 0x006C, 0x032D } },
	{ UTF32_NFD(0x1E3E), { 0x004D, 0x0301 } },
	{ UTF32_NFD(0x1E3F), { 0x006D, 0x0301 } },
	{ UTF32_NFD(0x1E40), { 0x004D, 0x0307 } },
	{ UTF32_NFD(0x1E41), { 0x006D, 0x0307 } },
	{ UTF32_NFD(0x1E42), { 0x004D, 0x0323 } },
	{ UTF32_NFD(0x1E43), { 0x006D, 0x0323 } },
	{ UTF32_NFD(0x1E44), { 0x004E, 0x0307 } },
	{ UTF32_NFD(0x1E45), { 0x006E, 0x0307 } },
	{ UTF32_NFD(0x1E46), { 0x004E, 0x0323 } },
	{ UTF32_NFD(0x1E47), { 0x006E, 0x0323 } },
	{ UTF32_NFD(0x1E48), { 0x004E, 0x0331 } },
	{ UTF32_NFD(0x1E49), { 0x006E, 0x0331 } },
	{ UTF32_NFD(0x1E4A), { 0x004E, 0x032D } },
	{ UTF32_NFD(0x1E4B), { 0x006E, 0x032D } },
	{ UTF32_NFD(0x1E4C), { 0x00D5, 0x0301 } },
	{ UTF32_NFD(0x1E4D), { 0x00F5, 0x0301 } },
	{ UTF32_NFD(0x1E4E), { 0x00D5, 0x0308 } },
	{ UTF32_NFD(0x1E4F), { 0x00F5, 0x0308 } },
	{ UTF32_NFD(0x1E50), { 0x014C, 0x0300 } },
	{ UTF32_NFD(0x1E51), { 0x014D, 0x0300 } },
	{ UTF32_NFD(0x1E52), { 0x014C, 0x0301 } },
	{ UTF32_NFD(0x1E53), { 0x014D, 0x0301 } },
	{ UTF32_NFD(0x1E54), { 0x0050, 0x0301 } },
	{ UTF32_NFD(0x1E55), { 0x0070, 0x0301 } },
	{ UTF32_NFD(0x1E56), { 0x0050, 0x0307 } },
	{ UTF32_NFD(0x1E57), { 0x0070, 0x0307 } },
	{ UTF32_NFD(0x1E58), { 0x0052, 0x0307 } },
	{ UTF32_NFD(0x1E59), { 0x0072, 0x0307 } },
	{ UTF32_NFD(0x1E5A), { 0x0052, 0x0323 } },
	{ UTF32_NFD(0x1E5B), { 0x0072, 0x0323 } },
	{ UTF32_NFD(0x1E5C), { 0x1E5A, 0x0304 } },
	{ UTF32_NFD(0x1E5D), { 0x1E5B, 0x0304 } },
	{ UTF32_NFD(0x1E5E), { 0x0052, 0x0331 } },
	{ UTF32_NFD(0x1E5F), { 0x0072, 0x0331 } },
	{ UTF32_NFD(0x1E60), { 0x0053, 0x0307 } },
	{ UTF32_NFD(0x1E61), { 0x0073, 0x0307 } },
	{ UTF32_NFD(0x1E62), { 0x0053, 0x0323 } },
	{ UTF32_NFD(0x1E63), { 0x0073, 0x0323 } },
	{ UTF32_NFD(0x1E64), { 0x015A, 0x0307 } },
	{ UTF32_NFD(0x1E65), { 0x015B, 0x0307 } },
	{ UTF32_NFD(0x1E66), { 0x0160, 0x0307 } },
	{ UTF32_NFD(0x1E67), { 0x0161, 0x0307 } },
	{ UTF32_NFD(0x1E68), { 0x1E62, 0x0307 } },
	{ UTF32_NFD(0x1E69), { 0x1E63, 0x0307 } },
	{ UTF32_NFD(0x1E6A), { 0x0054, 0x0307 } },
	{ UTF32_NFD(0x1E6B), { 0x0074, 0x0307 } },
	{ UTF32_NFD(0x1E6C), { 0x0054, 0x0323 } },
	{ UTF32_NFD(0x1E6D), { 0x0074, 0x0323 } },
	{ UTF32_NFD(0x1E6E), { 0x0054, 0x0331 } },
	{ UTF32_NFD(0x1E6F), { 0x0074, 0x0331 } },
	{ UTF32_NFD(0x1E70), { 0x0054, 0x032D } },
	{ UTF32_NFD(0x1E71), { 0x0074, 0x032D } },
	{ UTF32_NFD(0x1E72), { 0x0055, 0x0324 } },
	{ UTF32_NFD(0x1E73), { 0x0075, 0x0324 } },
	{ UTF32_NFD(0x1E74), { 0x0055, 0x0330 } },
	{ UTF32_NFD(0x1E75), { 0x0075, 0x0330 } },
	{ UTF32_NFD(0x1E76), { 0x0055, 0x032D } },
	{ UTF32_NFD(0x1E77), { 0x0075, 0x032D } },
	{ UTF32_NFD(0x1E78), { 0x0168, 0x0301 } },
	{ UTF32_NFD(0x1E79), { 0x0169, 0x0301 } },
	{ UTF32_NFD(0x1E7A), { 0x016A, 0x0308 } },
	{ UTF32_NFD(0x1E7B), { 0x016B, 0x0308 } },
	{ UTF32_NFD(0x1E7C), { 0x0056, 0x0303 } },
	{ UTF32_NFD(0x1E7D), { 0x0076, 0x0303 } },
	{ UTF32_NFD(0x1E7E), { 0x0056, 0x0323 } },
	{ UTF32_NFD(0x1E7F), { 0x0076, 0x0323 } },
	{ UTF32_NFD(0x1E80), { 0x0057, 0x0300 } },
	{ UTF32_NFD(0x1E81), { 0x0077, 0x0300 } },
	{ UTF32_NFD(0x1E82), { 0x0057, 0x0301 } },
	{ UTF32_NFD(0x1E83), { 0x0077, 0x0301 } },
	{ UTF32_NFD(0x1E84), { 0x0057, 0x0308 } },
	{ UTF32_NFD(0x1E85), { 0x0077, 0x0308 } },
	{ UTF32_NFD(0x1E86), { 0x0057, 0x0307 } },
	{ UTF32_NFD(0x1E87), { 0x0077, 0x0307 } },
	{ UTF32_NFD(0x1E88), { 0x0057, 0x0323 } },
	{ UTF32_NFD(0x1E89), { 0x0077, 0x0323 } },
	{ UTF32_NFD(0x1E8A), { 0x0058, 0x0307 } },
	{ UTF32_NFD(0x1E8B), { 0x0078, 0x0307 } },
	{ UTF32_NFD(0x1E8C), { 0x0058, 0x0308 } },
	{ UTF32_NFD(0x1E8D), { 0x0078, 0x0308 } },
	{ UTF32_NFD(0x1E8E), { 0x0059, 0x0307 } },
	{ UTF32_NFD(0x1E8F), { 0x0079, 0x0307 } },
	{ UTF32_NFD(0x1E90), { 0x005A, 0x0302 } },
	{ UTF32_NFD(0x1E91), { 0x007A, 0x0302 } },
	{ UTF32_NFD(0x1E92), { 0x005A, 0x0323 } },
	{ UTF32_NFD(0x1E93), { 0x007A, 0x0323 } },
	{ UTF32_NFD(0x1E94), { 0x005A, 0x0331 } },
	{ UTF32_NFD(0x1E95), { 0x007A, 0x0331 } },
	{ UTF32_NFD(0x1E96), { 0x0068, 0x0331 } },
	{ UTF32_NFD(0x1E97), { 0x0074, 0x0308 } },
	{ UTF32_NFD(0x1E98), { 0x0077, 0x030A } },
	{ UTF32_NFD(0x1E99), { 0x0079, 0x030A } },
	{ UTF32_NFKD(0x1E9A), { 0x0061, 0x02BE } },
	{ UTF32_NFD(0x1E9B), { 0x017F, 0x0307 } },
	{ UTF32_NFD(0x1EA0), { 0x0041, 0x0323 } },
	{ UTF32_NFD(0x1EA1), { 0x0061, 0x0323 } },
	{ UTF32_NFD(0x1EA2), { 0x0041, 0x0309 } },
	{ UTF32_NFD(0x1EA3), { 0x0061, 0x0309 } },
	{ UTF32_NFD(0x1EA4), { 0x00C2, 0x0301 } },
	{ UTF32_NFD(0x1EA5), { 0x00E2, 0x0301 } },
	{ UTF32_NFD(0x1EA6), { 0x00C2, 0x0300 } },
	{ UTF32_NFD(0x1EA7), { 0x00E2, 0x0300 } },
	{ UTF32_NFD(0x1EA8), { 0x00C2, 0x0309 } },
	{ UTF32_NFD(0x1EA9), { 0x00E2, 0x0309 } },
	{ UTF32_NFD(0x1EAA), { 0x00C2, 0x0303 } },
	{ UTF32_NFD(0x1EAB), { 0x00E2, 0x0303 } },
	{ UTF32_NFD(0x1EAC), { 0x1EA0, 0x0302 } },
	{ UTF32_NFD(0x1EAD), { 0x1EA1, 0x0302 } },
	{ UTF32_NFD(0x1EAE), { 0x0102, 0x0301 } },
	{ UTF32_NFD(0x1EAF), { 0x0103, 0x0301 } },
	{ UTF32_NFD(0x1EB0), { 0x0102, 0x0300 } },
	{ UTF32_NFD(0x1EB1), { 0x0103, 0x0300 } },
	{ UTF32_NFD(0x1EB2), { 0x0102, 0x0309 } },
	{ UTF32_NFD(0x1EB3), { 0x0103, 0x0309 } },
	{ UTF32_NFD(0x1EB4), { 0x0102, 0x0303 } },
	{ UTF32_NFD(0x1EB5), { 0x0103, 0x0303 } },
	{ UTF32_NFD(0x1EB6), { 0x1EA0, 0x0306 } },
	{ UTF32_NFD(0x1EB7), { 0x1EA1, 0x0306 } },
	{ UTF32_NFD(0x1EB8), { 0x0045, 0x0323 } },
	{ UTF32_NFD(0x1EB9), { 0x0065, 0x0323 } },
	{ UTF32_NFD(0x1EBA), { 0x0045, 0x0309 } },
	{ UTF32_NFD(0x1EBB), { 0x0065, 0x0309 } },
	{ UTF32_NFD(0x1EBC), { 0x0045, 0x0303 } },
	{ UTF32_NFD(0x1EBD), { 0x0065, 0x0303 } },
	{ UTF32_NFD(0x1EBE), { 0x00CA, 0x0301 } },
	{ UTF32_NFD(0x1EBF), { 0x00EA, 0x0301 } },
	{ UTF32_NFD(0x1EC0), { 0x00CA, 0x0300 } },
	{ UTF32_NFD(0x1EC1), { 0x00EA, 0x0300 } },
	{ UTF32_NFD(0x1EC2), { 0x00CA, 0x0309 } },
	{ UTF32_NFD(0x1EC3), { 0x00EA, 0x0309 } },
	{ UTF32_NFD(0x1EC4), { 0x00CA, 0x0303 } },
	{ UTF32_NFD(0x1EC5), { 0x00EA, 0x0303 } },
	{ UTF32_NFD(0x1EC6), { 0x1EB8, 0x0302 } },
	{ UTF32_NFD(0x1EC7), { 0x1EB9, 0x0302 } },
	{ UTF32_NFD(0x1EC8), { 0x0049, 0x0309 } },
	{ UTF32_NFD(0x1EC9), { 0x0069, 0x0309 } },
	{ UTF32_NFD(0x1ECA), { 0x0049, 0x0323 } },
	{ UTF32_NFD(0x1ECB), { 0x0069, 0x0323 } },
	{ UTF32_NFD(0x1ECC), { 0x004F, 0x0323 } },
	{ UTF32_NFD(0x1ECD), { 0x006F, 0x0323 } },
	{ UTF32_NFD(0x1ECE), { 0x004F, 0x0309 } },
	{ UTF32_NFD(0x1ECF), { 0x006F, 0x0309 } },
	{ UTF32_NFD(0x1ED0), { 0x00D4, 0x0301 } },
	{ UTF32_NFD(0x1ED1), { 0x00F4, 0x0301 } },
	{ UTF32_NFD(0x1ED2), { 0x00D4, 0x0300 } },
	{ UTF32_NFD(0x1ED3), { 0x00F4, 0x0300 } },
	{ UTF32_NFD(0x1ED4), { 0x00D4, 0x0309 } },
	{ UTF32_NFD(0x1ED5), { 0x00F4, 0x0309 } },
	{ UTF32_NFD(0x1ED6), { 0x00D4, 0x0303 } },
	{ UTF32_NFD(0x1ED7), { 0x00F4, 0x0303 } },
	{ UTF32_NFD(0x1ED8), { 0x1ECC, 0x0302 } },
	{ UTF32_NFD(0x1ED9), { 0x1ECD, 0x0302 } },
	{ UTF32_NFD(0x1EDA), { 0x01A0, 0x0301 } },
	{ UTF32_NFD(0x1EDB), { 0x01A1, 0x0301 } },
	{ UTF32_NFD(0x1EDC), { 0x01A0, 0x0300 } },
	{ UTF32_NFD(0x1EDD), { 0x01A1, 0x0300 } },
	{ UTF32_NFD(0x1EDE), { 0x01A0, 0x0309 } },
	{ UTF32_NFD(0x1EDF), { 0x01A1, 0x0309 } },
	{ UTF32_NFD(0x1EE0), { 0x01A0, 0x0303 } },
	{ UTF32_NFD(0x1EE1), { 0x01A1, 0x0303 } },
	{ UTF32_NFD(0x1EE2), { 0x01A0, 0x0323 } },
	{ UTF32_NFD(0x1EE3), { 0x01A1, 0x0323 } },
	{ UTF32_NFD(0x1EE4), { 0x0055, 0x0323 } },
	{ UTF32_NFD(0x1EE5), { 0x0075, 0x0323 } },
	{ UTF32_NFD(0x1EE6), { 0x0055, 0x0309 } },
	{ UTF32_NFD(0x1EE7), { 0x0075, 0x0309 } },
	{ UTF32_NFD(0x1EE8), { 0x01AF, 0x0301 } },
	{ UTF32_NFD(0x1EE9), { 0x01B0, 0x0301 } },
	{ UTF32_NFD(0x1EEA), { 0x01AF, 0x0300 } },
	{ UTF32_NFD(0x1EEB), { 0x01B0, 0x0300 } },
	{ UTF32_NFD(0x1EEC), { 0x01AF, 0x0309 } },
	{ UTF32_NFD(0x1EED), { 0x01B0, 0x0309 } },
	{ UTF32_NFD(0x1EEE), { 0x01AF, 0x0303 } },
	{ UTF32_NFD(0x1EEF), { 0x01B0, 0x0303 } },
	{ UTF32_NFD(0x1EF0), { 0x01AF, 0x0323 } },
	{ UTF32_NFD(0x1EF1), { 0x01B0, 0x0323 } },
	{ UTF32_NFD(0x1EF2), { 0x0059, 0x0300 } },
	{ UTF32_NFD(0x1EF3), { 0x0079, 0x0300 } },
	{ UTF32_NFD(0x1EF4), { 0x0059, 0x0323 } },
	{ UTF32_NFD(0x1EF5), { 0x0079, 0x0323 } },
	{ UTF32_NFD(0x1EF6), { 0x0059, 0x0309 } },
	{ UTF32_NFD(0x1EF7), { 0x0079, 0x0309 } },
	{ UTF32_NFD(0x1EF8), { 0x0059, 0x0303 } },
	{ UTF32_NFD(0x1EF9), { 0x0079, 0x0303 } },
	{ UTF32_NFD(0x1F00), { 0x03B1, 0x0313 } },
	{ UTF32_NFD(0x1F01), { 0x03B1, 0x0314 } },
	{ UTF32_NFD(0x1F02), { 0x1F00, 0x0300 } },
	{ UTF32_NFD(0x1F03), { 0x1F01, 0x0300 } },
	{ UTF32_NFD(0x1F04), { 0x1F00, 0x0301 } },
	{ UTF32_NFD(0x1F05), { 0x1F01, 0x0301 } },
	{ UTF32_NFD(0x1F06), { 0x1F00, 0x0342 } },
	{ UTF32_NFD(0x1F07), { 0x1F01, 0x0342 } },
	{ UTF32_NFD(0x1F08), { 0x0391, 0x0313 } },
	{ UTF32_NFD(0x1F09), { 0x0391, 0x0314 } },
	{ UTF32_NFD(0x1F0A), { 0x1F08, 0x0300 } },
	{ UTF32_NFD(0x1F0B), { 0x1F09, 0x0300 } },
	{ UTF32_NFD(0x1F0C), { 0x1F08, 0x0301 } },
	{ UTF32_NFD(0x1F0D), { 0x1F09, 0x0301 } },
	{ UTF32_NFD(0x1F0E), { 0x1F08, 0x0342 } },
	{ UTF32_NFD(0x1F0F), { 0x1F09, 0x0342 } },
	{ UTF32_NFD(0x1F10), { 0x03B5, 0x0313 } },
	{ UTF32_NFD(0x1F11), { 0x03B5, 0x0314 } },
	{ UTF32_NFD(0x1F12), { 0x1F10, 0x0300 } },
	{ UTF32_NFD(0x1F13), { 0x1F11, 0x0300 } },
	{ UTF32_NFD(0x1F14), { 0x1F10, 0x0301 } },
	{ UTF32_NFD(0x1F15), { 0x1F11, 0x0301 } },
	{ UTF32_NFD(0x1F18), { 0x0395, 0x0313 } },
	{ UTF32_NFD(0x1F19), { 0x0395, 0x0314 } },
	{ UTF32_NFD(0x1F1A), { 0x1F18, 0x0300 } },
	{ UTF32_NFD(0x1F1B), { 0x1F19, 0x0300 } },
	{ UTF32_NFD(0x1F1C), { 0x1F18, 0x0301 } },
	{ UTF32_NFD(0x1F1D), { 0x1F19, 0x0301 } },
	{ UTF32_NFD(0x1F20), { 0x03B7, 0x0313 } },
	{ UTF32_NFD(0x1F21), { 0x03B7, 0x0314 } },
	{ UTF32_NFD(0x1F22), { 0x1F20, 0x0300 } },
	{ UTF32_NFD(0x1F23), { 0x1F21, 0x0300 } },
	{ UTF32_NFD(0x1F24), { 0x1F20, 0x0301 } },
	{ UTF32_NFD(0x1F25), { 0x1F21, 0x0301 } },
	{ UTF32_NFD(0x1F26), { 0x1F20, 0x0342 } },
	{ UTF32_NFD(0x1F27), { 0x1F21, 0x0342 } },
	{ UTF32_NFD(0x1F28), { 0x0397, 0x0313 } },
	{ UTF32_NFD(0x1F29), { 0x0397, 0x0314 } },
	{ UTF32_NFD(0x1F2A), { 0x1F28, 0x0300 } },
	{ UTF32_NFD(0x1F2B), { 0x1F29, 0x0300 } },
	{ UTF32_NFD(0x1F2C), { 0x1F28, 0x0301 } },
	{ UTF32_NFD(0x1F2D), { 0x1F29, 0x0301 } },
	{ UTF32_NFD(0x1F2E), { 0x1F28, 0x0342 } },
	{ UTF32_NFD(0x1F2F), { 0x1F29, 0x0342 } },
	{ UTF32_NFD(0x1F30), { 0x03B9, 0x0313 } },
	{ UTF32_NFD(0x1F31), { 0x03B9, 0x0314 } },
	{ UTF32_NFD(0x1F32), { 0x1F30, 0x0300 } },
	{ UTF32_NFD(0x1F33), { 0x1F31, 0x0300 } },
	{ UTF32_NFD(0x1F34), { 0x1F30, 0x0301 } },
	{ UTF32_NFD(0x1F35), { 0x1F31, 0x0301 } },
	{ UTF32_NFD(0x1F36), { 0x1F30, 0x0342 } },
	{ UTF32_NFD(0x1F37), { 0x1F31, 0x0342 } },
	{ UTF32_NFD(0x1F38), { 0x0399, 0x0313 } },
	{ UTF32_NFD(0x1F39), { 0x0399, 0x0314 } },
	{ UTF32_NFD(0x1F3A), { 0x1F38, 0x0300 } },
	{ UTF32_NFD(0x1F3B), { 0x1F39, 0x0300 } },
	{ UTF32_NFD(0x1F3C), { 0x1F38, 0x0301 } },
	{ UTF32_NFD(0x1F3D), { 0x1F39, 0x0301 } },
	{ UTF32_NFD(0x1F3E), { 0x1F38, 0x0342 } },
	{ UTF32_NFD(0x1F3F), { 0x1F39, 0x0342 } },
	{ UTF32_NFD(0x1F40), { 0x03BF, 0x0313 } },
	{ UTF32_NFD(0x1F41), { 0x03BF, 0x0314 } },
	{ UTF32_NFD(0x1F42), { 0x1F40, 0x0300 } },
	{ UTF32_NFD(0x1F43), { 0x1F41, 0x0300 } },
	{ UTF32_NFD(0x1F44), { 0x1F40, 0x0301 } },
	{ UTF32_NFD(0x1F45), { 0x1F41, 0x0301 } },
	{ UTF32_NFD(0x1F48), { 0x039F, 0x0313 } },
	{ UTF32_NFD(0x1F49), { 0x039F, 0x0314 } },
	{ UTF32_NFD(0x1F4A), { 0x1F48, 0x0300 } },
	{ UTF32_NFD(0x1F4B), { 0x1F49, 0x0300 } },
	{ UTF32_NFD(0x1F4C), { 0x1F48, 0x0301 } },
	{ UTF32_NFD(0x1F4D), { 0x1F49, 0x0301 } },
	{ UTF32_NFD(0x1F50), { 0x03C5, 0x0313 } },
	{ UTF32_NFD(0x1F51), { 0x03C5, 0x0314 } },
	{ UTF32_NFD(0x1F52), { 0x1F50, 0x0300 } },
	{ UTF32_NFD(0x1F53), { 0x1F51, 0x0300 } },
	{ UTF32_NFD(0x1F54), { 0x1F50, 0x0301 } },
	{ UTF32_NFD(0x1F55), { 0x1F51, 0x0301 } },
	{ UTF32_NFD(0x1F56), { 0x1F50, 0x0342 } },
	{ UTF32_NFD(0x1F57), { 0x1F51, 0x0342 } },
	{ UTF32_NFD(0x1F59), { 0x03A5, 0x0314 } },
	{ UTF32_NFD(0x1F5B), { 0x1F59, 0x0300 } },
	{ UTF32_NFD(0x1F5D), { 0x1F59, 0x0301 } },
	{ UTF32_NFD(0x1F5F), { 0x1F59, 0x0342 } },
	{ UTF32_NFD(0x1F60), { 0x03C9, 0x0313 } },
	{ UTF32_NFD(0x1F61), { 0x03C9, 0x0314 } },
	{ UTF32_NFD(0x1F62), { 0x1F60, 0x0300 } },
	{ UTF32_NFD(0x1F63), { 0x1F61, 0x0300 } },
	{ UTF32_NFD(0x1F64), { 0x1F60, 0x0301 } },
	{ UTF32_NFD(0x1F65), { 0x1F61, 0x0301 } },
	{ UTF32_NFD(0x1F66), { 0x1F60, 0x0342 } },
	{ UTF32_NFD(0x1F67), { 0x1F61, 0x0342 } },
	{ UTF32_NFD(0x1F68), { 0x03A9, 0x0313 } },
	{ UTF32_NFD(0x1F69), { 0x03A9, 0x0314 } },
	{ UTF32_NFD(0x1F6A), { 0x1F68, 0x0300 } },
	{ UTF32_NFD(0x1F6B), { 0x1F69, 0x0300 } },
	{ UTF32_NFD(0x1F6C), { 0x1F68, 0x0301 } },
	{ UTF32_NFD(0x1F6D), { 0x1F69, 0x0301 } },
	{ UTF32_NFD(0x1F6E), { 0x1F68, 0x0342 } },
	{ UTF32_NFD(0x1F6F), { 0x1F69, 0x0342 } },
	{ UTF32_NFD(0x1F70), { 0x03B1, 0x0300 } },
	{ UTF32_NFD(0x1F71), { 0x03AC } },
	{ UTF32_NFD(0x1F72), { 0x03B5, 0x0300 } },
	{ UTF32_NFD(0x1F73), { 0x03AD } },
	{ UTF32_NFD(0x1F74), { 0x03B7, 0x0300 } },
	{ UTF32_NFD(0x1F75), { 0x03AE } },
	{ UTF32_NFD(0x1F76), { 0x03B9, 0x0300 } },
	{ UTF32_NFD(0x1F77), { 0x03AF } },
	{ UTF32_NFD(0x1F78), { 0x03BF, 0x0300 } },
	{ UTF32_NFD(0x1F79), { 0x03CC } },
	{ UTF32_NFD(0x1F7A), { 0x03C5, 0x0300 } },
	{ UTF32_NFD(0x1F7B), { 0x03CD } },
	{ UTF32_NFD(0x1F7C), { 0x03C9, 0x0300 } },
	{ UTF32_NFD(0x1F7D), { 0x03CE } },
	{ UTF32_NFD(0x1F80), { 0x1F00, 0x0345 } },
	{ UTF32_NFD(0x1F81), { 0x1F01, 0x0345 } },
	{ UTF32_NFD(0x1F82), { 0x1F02, 0x0345 } },
	{ UTF32_NFD(0x1F83), { 0x1F03, 0x0345 } },
	{ UTF32_NFD(0x1F84), { 0x1F04, 0x0345 } },
	{ UTF32_NFD(0x1F85), { 0x1F05, 0x0345 } },
	{ UTF32_NFD(0x1F86), { 0x1F06, 0x0345 } },
	{ UTF32_NFD(0x1F87), { 0x1F07, 0x0345 } },
	{ UTF32_NFD(0x1F88), { 0x1F08, 0x0345 } },
	{ UTF32_NFD(0x1F89), { 0x1F09, 0x0345 } },
	{ UTF32_NFD(0x1F8A), { 0x1F0A, 0x0345 } },
	{ UTF32_NFD(0x1F8B), { 0x1F0B, 0x0345 } },
	{ UTF32_NFD(0x1F8C), { 0x1F0C, 0x0345 } },
	{ UTF32_NFD(0x1F8D), { 0x1F0D, 0x0345 } },
	{ UTF32_NFD(0x1F8E), { 0x1F0E, 0x0345 } },
	{ UTF32_NFD(0x1F8F), { 0x1F0F, 0x0345 } },
	{ UTF32_NFD(0x1F90), { 0x1F20, 0x0345 } },
	{ UTF32_NFD(0x1F91), { 0x1F21, 0x0345 } },
	{ UTF32_NFD(0x1F92), { 0x1F22, 0x0345 } },
	{ UTF32_NFD(0x1F93), { 0x1F23, 0x0345 } },
	{ UTF32_NFD(0x1F94), { 0x1F24, 0x0345 } },
	{ UTF32_NFD(0x1F95), { 0x1F25, 0x0345 } },
	{ UTF32_NFD(0x1F96), { 0x1F26, 0x0345 } },
	{ UTF32_NFD(0x1F97), { 0x1F27, 0x0345 } },
	{ UTF32_NFD(0x1F98), { 0x1F28, 0x0345 } },
	{ UTF32_NFD(0x1F99), { 0x1F29, 0x0345 } },
	{ UTF32_NFD(0x1F9A), { 0x1F2A, 0x0345 } },
	{ UTF32_NFD(0x1F9B), { 0x1F2B, 0x0345 } },
	{ UTF32_NFD(0x1F9C), { 0x1F2C, 0x0345 } },
	{ UTF32_NFD(0x1F9D), { 0x1F2D, 0x0345 } },
	{ UTF32_NFD(0x1F9E), { 0x1F2E, 0x0345 } },
	{ UTF32_NFD(0x1F9F), { 0x1F2F, 0x0345 } },
	{ UTF32_NFD(0x1FA0), { 0x1F60, 0x0345 } },
	{ UTF32_NFD(0x1FA1), { 0x1F61, 0x0345 } },
	{ UTF32_NFD(0x1FA2), { 0x1F62, 0x0345 } },
	{ UTF32_NFD(0x1FA3), { 0x1F63, 0x0345 } },
	{ UTF32_NFD(0x1FA4), { 0x1F64, 0x0345 } },
	{ UTF32_NFD(0x1FA5), { 0x1F65, 0x0345 } },
	{ UTF32_NFD(0x1FA6), { 0x1F66, 0x0345 } },
	{ UTF32_NFD(0x1FA7), { 0x1F67, 0x0345 } },
	{ UTF32_NFD(0x1FA8), { 0x1F68, 0x0345 } },
	{ UTF32_NFD(0x1FA9), { 0x1F69, 0x0345 } },
	{ UTF32_NFD(0x1FAA), { 0x1F6A, 0x0345 } },
	{ UTF32_NFD(0x1FAB), { 0x1F6B, 0x0345 } },
	{ UTF32_NFD(0x1FAC), { 0x1F6C, 0x0345 } },
	{ UTF32_NFD(0x1FAD), { 0x1F6D, 0x0345 } },
	{ UTF32_NFD(0x1FAE), { 0x1F6E, 0x0345 } },
	{ UTF32_NFD(0x1FAF), { 0x1F6F, 0x0345 } },
	{ UTF32_NFD(0x1FB0), { 0x03B1, 0x0306 } },
	{ UTF32_NFD(0x1FB1), { 0x03B1, 0x0304 } },
	{ UTF32_NFD(0x1FB2), { 0x1F70, 0x0345 } },
	{ UTF32_NFD(0x1FB3), { 0x03B1, 0x0345 } },
	{ UTF32_NFD(0x1FB4), { 0x03AC, 0x0345 } },
	{ UTF32_NFD(0x1FB6), { 0x03B1, 0x0342 } },
	{ UTF32_NFD(0x1FB7), { 0x1FB6, 0x0345 } },
	{ UTF32_NFD(0x1FB8), { 0x0391, 0x0306 } },
	{ UTF32_NFD(0x1FB9), { 0x0391, 0x0304 } },
	{ UTF32_NFD(0x1FBA), { 0x0391, 0x0300 } },
	{ UTF32_NFD(0x1FBB), { 0x0386 } },
	{ UTF32_NFD(0x1FBC), { 0x0391, 0x0345 } },
	{ UTF32_NFKD(0x1FBD), { 0x0020, 0x0313 } },
	{ UTF32_NFD(0x1FBE), { 0x03B9 } },
	{ UTF32_NFKD(0x1FBF), { 0x0020, 0x0313 } },
	{ UTF32_NFKD(0x1FC0), { 0x0020, 0x0342 } },
	{ UTF32_NFD(0x1FC1), { 0x00A8, 0x0342 } },
	{ UTF32_NFD(0x1FC2), { 0x1F74, 0x0345 } },
	{ UTF32_NFD(0x1FC3), { 0x03B7, 0x0345 } },
	{ UTF32_NFD(0x1FC4), { 0x03AE, 0x0345 } },
	{ UTF32_NFD(0x1FC6), { 0x03B7, 0x0342 } },
	{ UTF32_NFD(0x1FC7), { 0x1FC6, 0x0345 } },
	{ UTF32_NFD(0x1FC8), { 0x0395, 0x0300 } },
	{ UTF32_NFD(0x1FC9), { 0x0388 } },
	{ UTF32_NFD(0x1FCA), { 0x0397, 0x0300 } },
	{ UTF32_NFD(0x1FCB), { 0x0389 } },
	{ UTF32_NFD(0x1FCC), { 0x0397, 0x0345 } },
	{ UTF32_NFD(0x1FCD), { 0x1FBF, 0x0300 } },
	{ UTF32_NFD(0x1FCE), { 0x1FBF, 0x0301 } },
	{ UTF32_NFD(0x1FCF), { 0x1FBF, 0x0342 } },
	{ UTF32_NFD(0x1FD0), { 0x03B9, 0x0306 } },
	{ UTF32_NFD(0x1FD1), { 0x03B9, 0x0304 } },
	{ UTF32_NFD(0x1FD2), { 0x03CA, 0x0300 } },
	{ UTF32_NFD(0x1FD3), { 0x0390 } },
	{ UTF32_NFD(0x1FD6), { 0x03B9, 0x0342 } },
	{ UTF32_NFD(0x1FD7), { 0x03CA, 0x0342 } },
	{ UTF32_NFD(0x1FD8), { 0x0399, 0x0306 } },
	{ UTF32_NFD(0x1FD9), { 0x0399, 0x0304 } },
	{ UTF32_NFD(0x1FDA), { 0x0399, 0x0300 } },
	{ UTF32_NFD(0x1FDB), { 0x038A } },
	{ UTF32_NFD(0x1FDD), { 0x1FFE, 0x0300 } },
	{ UTF32_NFD(0x1FDE), { 0x1FFE, 0x0301 } },
	{ UTF32_NFD(0x1FDF), { 0x1FFE, 0x0342 } },
	{ UTF32_NFD(0x1FE0), { 0x03C5, 0x0306 } },
	{ UTF32_NFD(0x1FE1), { 0x03C5, 0x0304 } },
	{ UTF32_NFD(0x1FE2), { 0x03CB, 0x0300 } },
	{ UTF32_NFD(0x1FE3), { 0x03B0 } },
	{ UTF32_NFD(0x1FE4), { 0x03C1, 0x0313 } },
	{ UTF32_NFD(0x1FE5), { 0x03C1, 0x0314 } },
	{ UTF32_NFD(0x1FE6), { 0x03C5, 0x0342 } },
	{ UTF32_NFD(0x1FE7), { 0x03CB, 0x0342 } },
	{ UTF32_NFD(0x1FE8), { 0x03A5, 0x0306 } },
	{ UTF32_NFD(0x1FE9), { 0x03A5, 0x0304 } },
	{ UTF32_NFD(0x1FEA), { 0x03A5, 0x0300 } },
	{ UTF32_NFD(0x1FEB), { 0x038E } },
	{ UTF32_NFD(0x1FEC), { 0x03A1, 0x0314 } },
	{ UTF32_NFD(0x1FED), { 0x00A8, 0x0300 } },
	{ UTF32_NFD(0x1FEE), { 0x0385 } },
	{ UTF32_NFD(0x1FEF), { 0x0060 } },
	{ UTF32_NFD(0x1FF2), { 0x1F7C, 0x0345 } },
	{ UTF32_NFD(0x1FF3), { 0x03C9, 0x0345 } },
	{ UTF32_NFD(0x1FF4), { 0x03CE, 0x0345 } },
	{ UTF32_NFD(0x1FF6), { 0x03C9, 0x0342 } },
	{ UTF32_NFD(0x1FF7), { 0x1FF6, 0x0345 } },
	{ UTF32_NFD(0x1FF8), { 0x039F, 0x0300 } },
	{ UTF32_NFD(0x1FF9), { 0x038C } },
	{ UTF32_NFD(0x1FFA), { 0x03A9, 0x0300 } },
	{ UTF32_NFD(0x1FFB), { 0x038F } },
	{ UTF32_NFD(0x1FFC), { 0x03A9, 0x0345 } },
	{ UTF32_NFD(0x1FFD), { 0x00B4 } },
	{ UTF32_NFKD(0x1FFE), { 0x0020, 0x0314 } },
	{ UTF32_NFD(0x2000), { 0x2002 } },
	{ UTF32_NFD(0x2001), { 0x2003 } },
	{ UTF32_NFKD(0x2002), { 0x0020 } },
	{ UTF32_NFKD(0x2003), { 0x0020 } },
	{ UTF32_NFKD(0x2004), { 0x0020 } },
	{ UTF32_NFKD(0x2005), { 0x0020 } },
	{ UTF32_NFKD(0x2006), { 0x0020 } },
	{ UTF32_NFKD(0x2007), { 0x0020 } },
	{ UTF32_NFKD(0x2008), { 0x0020 } },
	{ UTF32_NFKD(0x2009), { 0x0020 } },
	{ UTF32_NFKD(0x200A), { 0x0020 } },
	{ UTF32_NFKD(0x2011), { 0x2010 } },
	{ UTF32_NFKD(0x2017), { 0x0020, 0x0333 } },
	{ UTF32_NFKD(0x2024), { 0x002E } },
	{ UTF32_NFKD(0x2025), { 0x002E, 0x002E } },
	{ UTF32_NFKD(0x2026), { 0x002E, 0x002E, 0x002E } },
	{ UTF32_NFKD(0x202F), { 0x0020 } },
	{ UTF32_NFKD(0x2033), { 0x2032, 0x2032 } },
	{ UTF32_NFKD(0x2034), { 0x2032, 0x2032, 0x2032 } },
	{ UTF32_NFKD(0x2036), { 0x2035, 0x2035 } },
	{ UTF32_NFKD(0x2037), { 0x2035, 0x2035, 0x2035 } },
	{ UTF32_NFKD(0x203C), { 0x0021, 0x0021 } },
	{ UTF32_NFKD(0x203E), { 0x0020, 0x0305 } },
	{ UTF32_NFKD(0x2047), { 0x003F, 0x003F } },
	{ UTF32_NFKD(0x2048), { 0x003F, 0x0021 } },
	{ UTF32_NFKD(0x2049), { 0x0021, 0x003F } },
	{ UTF32_NFKD(0x2057), { 0x2032, 0x2032, 0x2032, 0x2032 } },
	{ UTF32_NFKD(0x205F), { 0x0020 } },
	{ UTF32_NFKD(0x2070), { 0x0030 } },
	{ UTF32_NFKD(0x2071), { 0x0069 } },
	{ UTF32_NFKD(0x2074), { 0x0034 } },
	{ UTF32_NFKD(0x2075), { 0x0035 } },
	{ UTF32_NFKD(0x2076), { 0x0036 } },
	{ UTF32_NFKD(0x2077), { 0x0037 } },
	{ UTF32_NFKD(0x2078), { 0x0038 } },
	{ UTF32_NFKD(0x2079), { 0x0039 } },
	{ UTF32_NFKD(0x207A), { 0x002B } },
	{ UTF32_NFKD(0x207B), { 0x2212 } },
	{ UTF32_NFKD(0x207C), { 0x003D } },
	{ UTF32_NFKD(0x207D), { 0x0028 } },
	{ UTF32_NFKD(0x207E), { 0x0029 } },
	{ UTF32_NFKD(0x207F), { 0x006E } },
	{ UTF32_NFKD(0x2080), { 0x0030 } },
	{ UTF32_NFKD(0x2081), { 0x0031 } },
	{ UTF32_NFKD(0x2082), { 0x0032 } },
	{ UTF32_NFKD(0x2083), { 0x0033 } },
	{ UTF32_NFKD(0x2084), { 0x0034 } },
	{ UTF32_NFKD(0x2085), { 0x0035 } },
	{ UTF32_NFKD(0x2086), { 0x0036 } },
	{ UTF32_NFKD(0x2087), { 0x0037 } },
	{ UTF32_NFKD(0x2088), { 0x0038 } },
	{ UTF32_NFKD(0x2089), { 0x0039 } },
	{ UTF32_NFKD(0x208A), { 0x002B } },
	{ UTF32_NFKD(0x208B), { 0x2212 } },
	{ UTF32_NFKD(0x208C), { 0x003D } },
	{ UTF32_NFKD(0x208D), { 0x0028 } },
	{ UTF32_NFKD(0x208E), { 0x0029 } },
	{ UTF32_NFKD(0x2090), { 0x0061 } },
	{ UTF32_NFKD(0x2091), { 0x0065 } },
	{ UTF32_NFKD(0x2092), { 0x006F } },
	{ UTF32_NFKD(0x2093), { 0x0078 } },
	{ UTF32_NFKD(0x2094), { 0x0259 } },
	{ UTF32_NFKD(0x20A8), { 0x0052, 0x0073 } },
	{ UTF32_NFKD(0x2100), { 0x0061, 0x002F, 0x0063 } },
	{ UTF32_NFKD(0x2101), { 0x0061, 0x002F, 0x0073 } },
	{ UTF32_NFKD(0x2102), { 0x0043 } },
	{ UTF32_NFKD(0x2103), { 0x00B0, 0x0043 } },
	{ UTF32_NFKD(0x2105), { 0x0063, 0x002F, 0x006F } },
	{ UTF32_NFKD(0x2106), { 0x0063, 0x002F, 0x0075 } },
	{ UTF32_NFKD(0x2107), { 0x0190 } },
	{ UTF32_NFKD(0x2109), { 0x00B0, 0x0046 } },
	{ UTF32_NFKD(0x210A), { 0x0067 } },
	{ UTF32_NFKD(0x210B), { 0x0048 } },
	{ UTF32_NFKD(0x210C), { 0x0048 } },
	{ UTF32_NFKD(0x210D), { 0x0048 } },
	{ UTF32_NFKD(0x210E), { 0x0068 } },
	{ UTF32_NFKD(0x210F), { 0x0127 } },
	{ UTF32_NFKD(0x2110), { 0x0049 } },
	{ UTF32_NFKD(0x2111), { 0x0049 } },
	{ UTF32_NFKD(0x2112), { 0x004C } },
	{ UTF32_NFKD(0x2113), { 0x006C } },
	{ UTF32_NFKD(0x2115), { 0x004E } },
	{ UTF32_NFKD(0x2116), { 0x004E, 0x006F } },
	{ UTF32_NFKD(0x2119), { 0x0050 } },
	{ UTF32_NFKD(0x211A), { 0x0051 } },
	{ UTF32_NFKD(0x211B), { 0x0052 } },
	{ UTF32_NFKD(0x211C), { 0x0052 } },
	{ UTF32_NFKD(0x211D), { 0x0052 } },
	{ UTF32_NFKD(0x2120), { 0x0053, 0x004D } },
	{ UTF32_NFKD(0x2121), { 0x0054, 0x0045, 0x004C } },
	{ UTF32_NFKD(0x2122), { 0x0054, 0x004D } },
	{ UTF32_NFKD(0x2124), { 0x005A } },
	{ UTF32_NFD(0x2126), { 0x03A9 } },
	{ UTF32_NFKD(0x2128), { 0x005A } },
	{ UTF32_NFD(0x212A), { 0x004B } },
	{ UTF32_NFD(0x212B), { 0x00C5 } },
	{ UTF32_NFKD(0x212C), { 0x0042 } },
	{ UTF32_NFKD(0x212D), { 0x0043 } },
	{ UTF32_NFKD(0x212F), { 0x0065 } },
	{ UTF32_NFKD(0x2130), { 0x0045 } },
	{ UTF32_NFKD(0x2131), { 0x0046 } },
	{ UTF32_NFKD(0x2133), { 0x004D } },
	{ UTF32_NFKD(0x2134), { 0x006F } },
	{ UTF32_NFKD(0x2135), { 0x05D0 } },
	{ UTF32_NFKD(0x2136), { 0x05D1 } },
	{ UTF32_NFKD(0x2137), { 0x05D2 } },
	{ UTF32_NFKD(0x2138), { 0x05D3 } },
	{ UTF32_NFKD(0x2139), { 0x0069 } },
	{ UTF32_NFKD(0x213B), { 0x0046, 0x0041, 0x0058 } },
	{ UTF32_NFKD(0x213C), { 0x03C0 } },
	{ UTF32_NFKD(0x213D), { 0x03B3 } },
	{ UTF32_NFKD(0x213E), { 0x0393 } },
	{ UTF32_NFKD(0x213F), { 0x03A0 } },
	{ UTF32_NFKD(0x2140), { 0x2211 } },
	{ UTF32_NFKD(0x2145), { 0x0044 } },
	{ UTF32_NFKD(0x2146), { 0x0064 } },
	{ UTF32_NFKD(0x2147), { 0x0065 } },
	{ UTF32_NFKD(0x2148), { 0x0069 } },
	{ UTF32_NFKD(0x2149), { 0x006A } },
	{ UTF32_NFKD(0x2153), { 0x0031, 0x2044, 0x0033 } },
	{ UTF32_NFKD(0x2154), { 0x0032, 0x2044, 0x0033 } },
	{ UTF32_NFKD(0x2155), { 0x0031, 0x2044, 0x0035 } },
	{ UTF32_NFKD(0x2156), { 0x0032, 0x2044, 0x0035 } },
	{ UTF32_NFKD(0x2157), { 0x0033, 0x2044, 0x0035 } },
	{ UTF32_NFKD(0x2158), { 0x0034, 0x2044, 0x0035 } },
	{ UTF32_NFKD(0x2159), { 0x0031, 0x2044, 0x0036 } },
	{ UTF32_NFKD(0x215A), { 0x0035, 0x2044, 0x0036 } },
	{ UTF32_NFKD(0x215B), { 0x0031, 0x2044, 0x0038 } },
	{ UTF32_NFKD(0x215C), { 0x0033, 0x2044, 0x0038 } },
	{ UTF32_NFKD(0x215D), { 0x0035, 0x2044, 0x0038 } },
	{ UTF32_NFKD(0x215E), { 0x0037, 0x2044, 0x0038 } },
	{ UTF32_NFKD(0x215F), { 0x0031, 0x2044 } },
	{ UTF32_NFKD(0x2160), { 0x0049 } },
	{ UTF32_NFKD(0x2161), { 0x0049, 0x0049 } },
	{ UTF32_NFKD(0x2162), { 0x0049, 0x0049, 0x0049 } },
	{ UTF32_NFKD(0x2163), { 0x0049, 0x0056 } },
	{ UTF32_NFKD(0x2164), { 0x0056 } },
	{ UTF32_NFKD(0x2165), { 0x0056, 0x0049 } },
	{ UTF32_NFKD(0x2166), { 0x0056, 0x0049, 0x0049 } },
	{ UTF32_NFKD(0x2167), { 0x0056, 0x0049, 0x0049, 0x0049 } },
	{ UTF32_NFKD(0x2168), { 0x0049, 0x0058 } },
	{ UTF32_NFKD(0x2169), { 0x0058 } },
	{ UTF32_NFKD(0x216A), { 0x0058, 0x0049 } },
	{ UTF32_NFKD(0x216B), { 0x0058, 0x0049, 0x0049 } },
	{ UTF32_NFKD(0x216C), { 0x004C } },
	{ UTF32_NFKD(0x216D), { 0x0043 } },
	{ UTF32_NFKD(0x216E), { 0x0044 } },
	{ UTF32_NFKD(0x216F), { 0x004D } },
	{ UTF32_NFKD(0x2170), { 0x0069 } },
	{ UTF32_NFKD(0x2171), { 0x0069, 0x0069 } },
	{ UTF32_NFKD(0x2172), { 0x0069, 0x0069, 0x0069 } },
	{ UTF32_NFKD(0x2173), { 0x0069, 0x0076 } },
	{ UTF32_NFKD(0x2174), { 0x0076 } },
	{ UTF32_NFKD(0x2175), { 0x0076, 0x0069 } },
	{ UTF32_NFKD(0x2176), { 0x0076, 0x0069, 0x0069 } },
	{ UTF32_NFKD(0x2177), { 0x0076, 0x0069, 0x0069, 0x0069 } },
	{ UTF32_NFKD(0x2178), { 0x0069, 0x0078 } },
	{ UTF32_NFKD(0x2179), { 0x0078 } },
	{ UTF32_NFKD(0x217A), { 0x0078, 0x0069 } },
	{ UTF32_NFKD(0x217B), { 0x0078, 0x0069, 0x0069 } },
	{ UTF32_NFKD(0x217C), { 0x006C } },
	{ UTF32_NFKD(0x217D), { 0x0063 } },
	{ UTF32_NFKD(0x217E), { 0x0064 } },
	{ UTF32_NFKD(0x217F), { 0x006D } },
	{ UTF32_NFD(0x219A), { 0x2190, 0x0338 } },
	{ UTF32_NFD(0x219B), { 0x2192, 0x0338 } },
	{ UTF32_NFD(0x21AE), { 0x2194, 0x0338 } },
	{ UTF32_NFD(0x21CD), { 0x21D0, 0x0338 } },
	{ UTF32_NFD(0x21CE), { 0x21D4, 0x0338 } },
	{ UTF32_NFD(0x21CF), { 0x21D2, 0x0338 } },
	{ UTF32_NFD(0x2204), { 0x2203, 0x0338 } },
	{ UTF32_NFD(0x2209), { 0x2208, 0x0338 } },
	{ UTF32_NFD(0x220C), { 0x220B, 0x0338 } },
	{ UTF32_NFD(0x2224), { 0x2223, 0x0338 } },
	{ UTF32_NFD(0x2226), { 0x2225, 0x0338 } },
	{ UTF32_NFKD(0x222C), { 0x222B, 0x222B } },
	{ UTF32_NFKD(0x222D), { 0x222B, 0x222B, 0x222B } },
	{ UTF32_NFKD(0x222F), { 0x222E, 0x222E } },
	{ UTF32_NFKD(0x2230), { 0x222E, 0x222E, 0x222E } },
	{ UTF32_NFD(0x2241), { 0x223C, 0x0338 } },
	{ UTF32_NFD(0x2244), { 0x2243, 0x0338 } },
	{ UTF32_NFD(0x2247), { 0x2245, 0x0338 } },
	{ UTF32_NFD(0x2249), { 0x2248, 0x0338 } },
	{ UTF32_NFD(0x2260), { 0x003D, 0x0338 } },
	{ UTF32_NFD(0x2262), { 0x2261, 0x0338 } },
	{ UTF32_NFD(0x226D), { 0x224D, 0x0338 } },
	{ UTF32_NFD(0x226E), { 0x003C, 0x0338 } },
	{ UTF32_NFD(0x226F), { 0x003E, 0x0338 } },
	{ UTF32_NFD(0x2270), { 0x2264, 0x0338 } },
	{ UTF32_NFD(0x2271), { 0x2265, 0x0338 } },
	{ UTF32_NFD(0x2274), { 0x2272, 0x0338 } },
	{ UTF32_NFD(0x2275), { 0x2273, 0x0338 } },
	{ UTF32_NFD(0x2278), { 0x2276, 0x0338 } },
	{ UTF32_NFD(0x2279), { 0x2277, 0x0338 } },
	{ UTF32_NFD(0x2280), { 0x227A, 0x0338 } },
	{ UTF32_NFD(0x2281), { 0x227B, 0x0338 } },
	{ UTF32_NFD(0x2284), { 0x2282, 0x0338 } },
	{ UTF32_NFD(0x2285), { 0x2283, 0x0338 } },
	{ UTF32_NFD(0x2288), { 0x2286, 0x0338 } },
	{ UTF32_NFD(0x2289), { 0x2287, 0x0338 } },
	{ UTF32_NFD(0x22AC), { 0x22A2, 0x0338 } },
	{ UTF32_NFD(0x22AD), { 0x22A8, 0x0338 } },
	{ UTF32_NFD(0x22AE), { 0x22A9, 0x0338 } },
	{ UTF32_NFD(0x22AF), { 0x22AB, 0x0338 } },
	{ UTF32_NFD(0x22E0), { 0x227C, 0x0338 } },
	{ UTF32_NFD(0x22E1), { 0x227D, 0x0338 } },
	{ UTF32_NFD(0x22E2), { 0x2291, 0x0338 } },
	{ UTF32_NFD(0x22E3), { 0x2292, 0x0338 } },
	{ UTF32_NFD(0x22EA), { 0x22B2, 0x0338 } },
	{ UTF32_NFD(0x22EB), { 0x22B3, 0x0338 } },
	{ UTF32_NFD(0x22EC), { 0x22B4, 0x0338 } },
	{ UTF32_NFD(0x22ED), { 0x22B5, 0x0338 } },
	{ UTF32_NFD(0x2329), { 0x3008 } },
	{ UTF32_NFD(0x232A), { 0x3009 } },
	{ UTF32_NFKD(0x2460), { 0x0031 } },
	{ UTF32_NFKD(0x2461), { 0x0032 } },
	{ UTF32_NFKD(0x2462), { 0x0033 } },
	{ UTF32_NFKD(0x2463), { 0x0034 } },
	{ UTF32_NFKD(0x2464), { 0x0035 } },
	{ UTF32_NFKD(0x2465), { 0x0036 } },
	{ UTF32_NFKD(0x2466), { 0x0037 } },
	{ UTF32_NFKD(0x2467), { 0x0038 } },
	{ UTF32_NFKD(0x2468), { 0x0039 } },
	{ UTF32_NFKD(0x2469), { 0x0031, 0x0030 } },
	{ UTF32_NFKD(0x246A), { 0x0031, 0x0031 } },
	{ UTF32_NFKD(0x246B), { 0x0031, 0x0032 } },
	{ UTF32_NFKD(0x246C), { 0x0031, 0x0033 } },
	{ UTF32_NFKD(0x246D), { 0x0031, 0x0034 } },
	{ UTF32_NFKD(0x246E), { 0x0031, 0x0035 } },
	{ UTF32_NFKD(0x246F), { 0x0031, 0x0036 } },
	{ UTF32_NFKD(0x2470), { 0x0031, 0x0037 } },
	{ UTF32_NFKD(0x2471), { 0x0031, 0x0038 } },
	{ UTF32_NFKD(0x2472), { 0x0031, 0x0039 } },
	{ UTF32_NFKD(0x2473), { 0x0032, 0x0030 } },
	{ UTF32_NFKD(0x2474), { 0x0028, 0x0031, 0x0029 } },
	{ UTF32_NFKD(0x2475), { 0x0028, 0x0032, 0x0029 } },
	{ UTF32_NFKD(0x2476), { 0x0028, 0x0033, 0x0029 } },
	{ UTF32_NFKD(0x2477), { 0x0028, 0x0034, 0x0029 } },
	{ UTF32_NFKD(0x2478), { 0x0028, 0x0035, 0x0029 } },
	{ UTF32_NFKD(0x2479), { 0x0028, 0x0036, 0x0029 } },
	{ UTF32_NFKD(0x247A), { 0x0028, 0x0037, 0x0029 } },
	{ UTF32_NFKD(0x247B), { 0x0028, 0x0038, 0x0029 } },
	{ UTF32_NFKD(0x247C), { 0x0028, 0x0039, 0x0029 } },
	{ UTF32_NFKD(0x247D), { 0x0028, 0x0031, 0x0030, 0x0029 } },
	{ UTF32_NFKD(0x247E), { 0x0028, 0x0031, 0x0031, 0x0029 } },
	{ UTF32_NFKD(0x247F), { 0x0028, 0x0031, 0x0032, 0x0029 } },
	{ UTF32_NFKD(0x2480), { 0x0028, 0x0031, 0x0033, 0x0029 } },
	{ UTF32_NFKD(0x2481), { 0x0028, 0x0031, 0x0034, 0x0029 } },
	{ UTF32_NFKD(0x2482), { 0x0028, 0x0031, 0x0035, 0x0029 } },
	{ UTF32_NFKD(0x2483), { 0x0028, 0x0031, 0x0036, 0x0029 } },
	{ UTF32_NFKD(0x2484), { 0x0028, 0x0031, 0x0037, 0x0029 } },
	{ UTF32_NFKD(0x2485), { 0x0028, 0x0031, 0x0038, 0x0029 } },
	{ UTF32_NFKD(0x2486), { 0x0028, 0x0031, 0x0039, 0x0029 } },
	{ UTF32_NFKD(0x2487), { 0x0028, 0x0032, 0x0030, 0x0029 } },
	{ UTF32_NFKD(0x2488), { 0x0031, 0x002E } },
	{ UTF32_NFKD(0x2489), { 0x0032, 0x002E } },
	{ UTF32_NFKD(0x248A), { 0x0033, 0x002E } },
	{ UTF32_NFKD(0x248B), { 0x0034, 0x002E } },
	{ UTF32_NFKD(0x248C), { 0x0035, 0x002E } },
	{ UTF32_NFKD(0x248D), { 0x0036, 0x002E } },
	{ UTF32_NFKD(0x248E), { 0x0037, 0x002E } },
	{ UTF32_NFKD(0x248F), { 0x0038, 0x002E } },
	{ UTF32_NFKD(0x2490), { 0x0039, 0x002E } },
	{ UTF32_NFKD(0x2491), { 0x0031, 0x0030, 0x002E } },
	{ UTF32_NFKD(0x2492), { 0x0031, 0x0031, 0x002E } },
	{ UTF32_NFKD(0x2493), { 0x0031, 0x0032, 0x002E } },
	{ UTF32_NFKD(0x2494), { 0x0031, 0x0033, 0x002E } },
	{ UTF32_NFKD(0x2495), { 0x0031, 0x0034, 0x002E } },
	{ UTF32_NFKD(0x2496), { 0x0031, 0x0035, 0x002E } },
	{ UTF32_NFKD(0x2497), { 0x0031, 0x0036, 0x002E } },
	{ UTF32_NFKD(0x2498), { 0x0031, 0x0037, 0x002E } },
	{ UTF32_NFKD(0x2499), { 0x0031, 0x0038, 0x002E } },
	{ UTF32_NFKD(0x249A), { 0x0031, 0x0039, 0x002E } },
	{ UTF32_NFKD(0x249B), { 0x0032, 0x0030, 0x002E } },
	{ UTF32_NFKD(0x249C), { 0x0028, 0x0061, 0x0029 } },
	{ UTF32_NFKD(0x249D), { 0x0028, 0x0062, 0x0029 } },
	{ UTF32_NFKD(0x249E), { 0x0028, 0x0063, 0x0029 } },
	{ UTF32_NFKD(0x249F), { 0x0028, 0x0064, 0x0029 } },
	{ UTF32_NFKD(0x24A0), { 0x0028, 0x0065, 0x0029 } },
	{ UTF32_NFKD(0x24A1), { 0x0028, 0x0066, 0x0029 } },
	{ UTF32_NFKD(0x24A2), { 0x0028, 0x0067, 0x0029 } },
	{ UTF32_NFKD(0x24A3), { 0x0028, 0x0068, 0x0029 } },
	{ UTF32_NFKD(0x24A4), { 0x0028, 0x0069, 0x0029 } },
	{ UTF32_NFKD(0x24A5), { 0x0028, 0x006A, 0x0029 } },
	{ UTF32_NFKD(0x24A6), { 0x0028, 0x006B, 0x0029 } },
	{ UTF32_NFKD(0x24A7), { 0x0028, 0x006C, 0x0029 } },
	{ UTF32_NFKD(0x24A8), { 0x0028, 0x006D, 0x0029 } },
	{ UTF32_NFKD(0x24A9), { 0x0028, 0x006E, 0x0029 } },
	{ UTF32_NFKD(0x24AA), { 0x0028, 0x006F, 0x0029 } },
	{ UTF32_NFKD(0x24AB), { 0x0028, 0x0070, 0x0029 } },
	{ UTF32_NFKD(0x24AC), { 0x0028, 0x0071, 0x0029 } },
	{ UTF32_NFKD(0x24AD), { 0x0028, 0x0072, 0x0029 } },
	{ UTF32_NFKD(0x24AE), { 0x0028, 0x0073, 0x0029 } },
	{ UTF32_NFKD(0x24AF), { 0x0028, 0x0074, 0x0029 } },
	{ UTF32_NFKD(0x24B0), { 0x0028, 0x0075, 0x0029 } },
	{ UTF32_NFKD(0x24B1), { 0x0028, 0x0076, 0x0029 } },
	{ UTF32_NFKD(0x24B2), { 0x0028, 0x0077, 0x0029 } },
	{ UTF32_NFKD(0x24B3), { 0x0028, 0x0078, 0x0029 } },
	{ UTF32_NFKD(0x24B4), { 0x0028, 0x0079, 0x0029 } },
	{ UTF32_NFKD(0x24B5), { 0x0028, 0x007A, 0x0029 } },
	{ UTF32_NFKD(0x24B6), { 0x0041 } },
	{ UTF32_NFKD(0x24B7), { 0x0042 } },
	{ UTF32_NFKD(0x24B8), { 0x0043 } },
	{ UTF32_NFKD(0x24B9), { 0x0044 } },
	{ UTF32_NFKD(0x24BA), { 0x0045 } },
	{ UTF32_NFKD(0x24BB), { 0x0046 } },
	{ UTF32_NFKD(0x24BC), { 0x0047 } },
	{ UTF32_NFKD(0x24BD), { 0x0048 } },
	{ UTF32_NFKD(0x24BE), { 0x0049 } },
	{ UTF32_NFKD(0x24BF), { 0x004A } },
	{ UTF32_NFKD(0x24C0), { 0x004B } },
	{ UTF32_NFKD(0x24C1), { 0x004C } },
	{ UTF32_NFKD(0x24C2), { 0x004D } },
	{ UTF32_NFKD(0x24C3), { 0x004E } },
	{ UTF32_NFKD(0x24C4), { 0x004F } },
	{ UTF32_NFKD(0x24C5), { 0x0050 } },
	{ UTF32_NFKD(0x24C6), { 0x0051 } },
	{ UTF32_NFKD(0x24C7), { 0x0052 } },
	{ UTF32_NFKD(0x24C8), { 0x0053 } },
	{ UTF32_NFKD(0x24C9), { 0x0054 } },
	{ UTF32_NFKD(0x24CA), { 0x0055 } },
	{ UTF32_NFKD(0x24CB), { 0x0056 } },
	{ UTF32_NFKD(0x24CC), { 0x0057 } },
	{ UTF32_NFKD(0x24CD), { 0x0058 } },
	{ UTF32_NFKD(0x24CE), { 0x0059 } },
	{ UTF32_NFKD(0x24CF), { 0x005A } },
	{ UTF32_NFKD(0x24D0), { 0x0061 } },
	{ UTF32_NFKD(0x24D1), { 0x0062 } },
	{ UTF32_NFKD(0x24D2), { 0x0063 } },
	{ UTF32_NFKD(0x24D3), { 0x0064 } },
	{ UTF32_NFKD(0x24D4), { 0x0065 } },
	{ UTF32_NFKD(0x24D5), { 0x0066 } },
	{ UTF32_NFKD(0x24D6), { 0x0067 } },
	{ UTF32_NFKD(0x24D7), { 0x0068 } },
	{ UTF32_NFKD(0x24D8), { 0x0069 } },
	{ UTF32_NFKD(0x24D9), { 0x006A } },
	{ UTF32_NFKD(0x24DA), { 0x006B } },
	{ UTF32_NFKD(0x24DB), { 0x006C } },
	{ UTF32_NFKD(0x24DC), { 0x006D } },
	{ UTF32_NFKD(0x24DD), { 0x006E } },
	{ UTF32_NFKD(0x24DE), { 0x006F } },
	{ UTF32_NFKD(0x24DF), { 0x0070 } },
	{ UTF32_NFKD(0x24E0), { 0x0071 } },
	{ UTF32_NFKD(0x24E1), { 0x0072 } },
	{ UTF32_NFKD(0x24E2), { 0x0073 } },
	{ UTF32_NFKD(0x24E3), { 0x0074 } },
	{ UTF32_NFKD(0x24E4), { 0x0075 } },
	{ UTF32_NFKD(0x24E5), { 0x0076 } },
	{ UTF32_NFKD(0x24E6), { 0x0077 } },
	{ UTF32_NFKD(0x24E7), { 0x0078 } },
	{ UTF32_NFKD(0x24E8), { 0x0079 } },
	{ UTF32_NFKD(0x24E9), { 0x007A } },
	{ UTF32_NFKD(0x24EA), { 0x0030 } },
	{ UTF32_NFKD(0x2A0C), { 0x222B, 0x222B, 0x222B, 0x222B } },
	{ UTF32_NFKD(0x2A74), { 0x003A, 0x003A, 0x003D } },
	{ UTF32_NFKD(0x2A75), { 0x003D, 0x003D } },
	{ UTF32_NFKD(0x2A76), { 0x003D, 0x003D, 0x003D } },
	{ UTF32_NFD(0x2ADC), { 0x2ADD, 0x0338 } },
	{ UTF32_NFKD(0x2D6F), { 0x2D61 } },
	{ UTF32_NFKD(0x2E9F), { 0x6BCD } },
	{ UTF32_NFKD(0x2EF3), { 0x9F9F } },
	{ UTF32_NFKD(0x2F00), { 0x4E00 } },
	{ UTF32_NFKD(0x2F01), { 0x4E28 } },
	{ UTF32_NFKD(0x2F02), { 0x4E36 } },
	{ UTF32_NFKD(0x2F03), { 0x4E3F } },
	{ UTF32_NFKD(0x2F04), { 0x4E59 } },
	{ UTF32_NFKD(0x2F05), { 0x4E85 } },
	{ UTF32_NFKD(0x2F06), { 0x4E8C } },
	{ UTF32_NFKD(0x2F07), { 0x4EA0 } },
	{ UTF32_NFKD(0x2F08), { 0x4EBA } },
	{ UTF32_NFKD(0x2F09), { 0x513F } },
	{ UTF32_NFKD(0x2F0A), { 0x5165 } },
	{ UTF32_NFKD(0x2F0B), { 0x516B } },
	{ UTF32_NFKD(0x2F0C), { 0x5182 } },
	{ UTF32_NFKD(0x2F0D), { 0x5196 } },
	{ UTF32_NFKD(0x2F0E), { 0x51AB } },
	{ UTF32_NFKD(0x2F0F), { 0x51E0 } },
	{ UTF32_NFKD(0x2F10), { 0x51F5 } },
	{ UTF32_NFKD(0x2F11), { 0x5200 } },
	{ UTF32_NFKD(0x2F12), { 0x529B } },
	{ UTF32_NFKD(0x2F13), { 0x52F9 } },
	{ UTF32_NFKD(0x2F14), { 0x5315 } },
	{ UTF32_NFKD(0x2F15), { 0x531A } },
	{ UTF32_NFKD(0x2F16), { 0x5338 } },
	{ UTF32_NFKD(0x2F17), { 0x5341 } },
	{ UTF32_NFKD(0x2F18), { 0x535C } },
	{ UTF32_NFKD(0x2F19), { 0x5369 } },
	{ UTF32_NFKD(0x2F1A), { 0x5382 } },
	{ UTF32_NFKD(0x2F1B), { 0x53B6 } },
	{ UTF32_NFKD(0x2F1C), { 0x53C8 } },
	{ UTF32_NFKD(0x2F1D), { 0x53E3 } },
	{ UTF32_NFKD(0x2F1E), { 0x56D7 } },
	{ UTF32_NFKD(0x2F1F), { 0x571F } },
	{ UTF32_NFKD(0x2F20), { 0x58EB } },
	{ UTF32_NFKD(0x2F21), { 0x5902 } },
	{ UTF32_NFKD(0x2F22), { 0x590A } },
	{ UTF32_NFKD(0x2F23), { 0x5915 } },
	{ UTF32_NFKD(0x2F24), { 0x5927 } },
	{ UTF32_NFKD(0x2F25), { 0x5973 } },
	{ UTF32_NFKD(0x2F26), { 0x5B50 } },
	{ UTF32_NFKD(0x2F27), { 0x5B80 } },
	{ UTF32_NFKD(0x2F28), { 0x5BF8 } },
	{ UTF32_NFKD(0x2F29), { 0x5C0F } },
	{ UTF32_NFKD(0x2F2A), { 0x5C22 } },
	{ UTF32_NFKD(0x2F2B), { 0x5C38 } },
	{ UTF32_NFKD(0x2F2C), { 0x5C6E } },
	{ UTF32_NFKD(0x2F2D), { 0x5C71 } },
	{ UTF32_NFKD(0x2F2E), { 0x5DDB } },
	{ UTF32_NFKD(0x2F2F), { 0x5DE5 } },
	{ UTF32_NFKD(0x2F30), { 0x5DF1 } },
	{ UTF32_NFKD(0x2F31), { 0x5DFE } },
	{ UTF32_NFKD(0x2F32), { 0x5E72 } },
	{ UTF32_NFKD(0x2F33), { 0x5E7A } },
	{ UTF32_NFKD(0x2F34), { 0x5E7F } },
	{ UTF32_NFKD(0x2F35), { 0x5EF4 } },
	{ UTF32_NFKD(0x2F36), { 0x5EFE } },
	{ UTF32_NFKD(0x2F37), { 0x5F0B } },
	{ UTF32_NFKD(0x2F38), { 0x5F13 } },
	{ UTF32_NFKD(0x2F39), { 0x5F50 } },
	{ UTF32_NFKD(0x2F3A), { 0x5F61 } },
	{ UTF32_NFKD(0x2F3B), { 0x5F73 } },
	{ UTF32_NFKD(0x2F3C), { 0x5FC3 } },
	{ UTF32_NFKD(0x2F3D), { 0x6208 } },
	{ UTF32_NFKD(0x2F3E), { 0x6236 } },
	{ UTF32_NFKD(0x2F3F), { 0x624B } },
	{ UTF32_NFKD(0x2F40), { 0x652F } },
	{ UTF32_NFKD(0x2F41), { 0x6534 } },
	{ UTF32_NFKD(0x2F42), { 0x6587 } },
	{ UTF32_NFKD(0x2F43), { 0x6597 } },
	{ UTF32_NFKD(0x2F44), { 0x65A4 } },
	{ UTF32_NFKD(0x2F45), { 0x65B9 } },
	{ UTF32_NFKD(0x2F46), { 0x65E0 } },
	{ UTF32_NFKD(0x2F47), { 0x65E5 } },
	{ UTF32_NFKD(0x2F48), { 0x66F0 } },
	{ UTF32_NFKD(0x2F49), { 0x6708 } },
	{ UTF32_NFKD(0x2F4A), { 0x6728 } },
	{ UTF32_NFKD(0x2F4B), { 0x6B20 } },
	{ UTF32_NFKD(0x2F4C), { 0x6B62 } },
	{ UTF32_NFKD(0x2F4D), { 0x6B79 } },
	{ UTF32_NFKD(0x2F4E), { 0x6BB3 } },
	{ UTF32_NFKD(0x2F4F), { 0x6BCB } },
	{ UTF32_NFKD(0x2F50), { 0x6BD4 } },
	{ UTF32_NFKD(0x2F51), { 0x6BDB } },
	{ UTF32_NFKD(0x2F52), { 0x6C0F } },
	{ UTF32_NFKD(0x2F53), { 0x6C14 } },
	{ UTF32_NFKD(0x2F54), { 0x6C34 } },
	{ UTF32_NFKD(0x2F55), { 0x706B } },
	{ UTF32_NFKD(0x2F56), { 0x722A } },
	{ UTF32_NFKD(0x2F57), { 0x7236 } },
	{ UTF32_NFKD(0x2F58), { 0x723B } },
	{ UTF32_NFKD(0x2F59), { 0x723F } },
	{ UTF32_NFKD(0x2F5A), { 0x7247 } },
	{ UTF32_NFKD(0x2F5B), { 0x7259 } },
	{ UTF32_NFKD(0x2F5C), { 0x725B } },
	{ UTF32_NFKD(0x2F5D), { 0x72AC } },
	{ UTF32_NFKD(0x2F5E), { 0x7384 } },
	{ UTF32_NFKD(0x2F5F), { 0x7389 } },
	{ UTF32_NFKD(0x2F60), { 0x74DC } },
	{ UTF32_NFKD(0x2F61), { 0x74E6 } },
	{ UTF32_NFKD(0x2F62), { 0x7518 } },
	{ UTF32_NFKD(0x2F63), { 0x751F } },
	{ UTF32_NFKD(0x2F64), { 0x7528 } },
	{ UTF32_NFKD(0x2F65), { 0x7530 } },
	{ UTF32_NFKD(0x2F66), { 0x758B } },
	{ UTF32_NFKD(0x2F67), { 0x7592 } },
	{ UTF32_NFKD(0x2F68), { 0x7676 } },
	{ UTF32_NFKD(0x2F69), { 0x767D } },
	{ UTF32_NFKD(0x2F6A), { 0x76AE } },
	{ UTF32_NFKD(0x2F6B), { 0x76BF } },
	{ UTF32_NFKD(0x2F6C), { 0x76EE } },
	{ UTF32_NFKD(0x2F6D), { 0x77DB } },
	{ UTF32_NFKD(0x2F6E), { 0x77E2 } },
	{ UTF32_NFKD(0x2F6F), { 0x77F3 } },
	{ UTF32_NFKD(0x2F70), { 0x793A } },
	{ UTF32_NFKD(0x2F71), { 0x79B8 } },
	{ UTF32_NFKD(0x2F72), { 0x79BE } },
	{ UTF32_NFKD(0x2F73), { 0x7A74 } },
	{ UTF32_NFKD(0x2F74), { 0x7ACB } },
	{ UTF32_NFKD(0x2F75), { 0x7AF9 } },
	{ UTF32_NFKD(0x2F76), { 0x7C73 } },
	{ UTF32_NFKD(0x2F77), { 0x7CF8 } },
	{ UTF32_NFKD(0x2F78), { 0x7F36 } },
	{ UTF32_NFKD(0x2F79), { 0x7F51 } },
	{ UTF32_NFKD(0x2F7A), { 0x7F8A } },
	{ UTF32_NFKD(0x2F7B), { 0x7FBD } },
	{ UTF32_NFKD(0x2F7C), { 0x8001 } },
	{ UTF32_NFKD(0x2F7D), { 0x800C } },
	{ UTF32_NFKD(0x2F7E), { 0x8012 } },
	{ UTF32_NFKD(0x2F7F), { 0x8033 } },
	{ UTF32_NFKD(0x2F80), { 0x807F } },
	{ UTF32_NFKD(0x2F81), { 0x8089 } },
	{ UTF32_NFKD(0x2F82), { 0x81E3 } },
	{ UTF32_NFKD(0x2F83), { 0x81EA } },
	{ UTF32_NFKD(0x2F84), { 0x81F3 } },
	{ UTF32_NFKD(0x2F85), { 0x81FC } },
	{ UTF32_NFKD(0x2F86), { 0x820C } },
	{ UTF32_NFKD(0x2F87), { 0x821B } },
	{ UTF32_NFKD(0x2F88), { 0x821F } },
	{ UTF32_NFKD(0x2F89), { 0x826E } },
	{ UTF32_NFKD(0x2F8A), { 0x8272 } },
	{ UTF32_NFKD(0x2F8B), { 0x8278 } },
	{ UTF32_NFKD(0x2F8C), { 0x864D } },
	{ UTF32_NFKD(0x2F8D), { 0x866B } },
	{ UTF32_NFKD(0x2F8E), { 0x8840 } },
	{ UTF32_NFKD(0x2F8F), { 0x884C } },
	{ UTF32_NFKD(0x2F90), { 0x8863 } },
	{ UTF32_NFKD(0x2F91), { 0x897E } },
	{ UTF32_NFKD(0x2F92), { 0x898B } },
	{ UTF32_NFKD(0x2F93), { 0x89D2 } },
	{ UTF32_NFKD(0x2F94), { 0x8A00 } },
	{ UTF32_NFKD(0x2F95), { 0x8C37 } },
	{ UTF32_NFKD(0x2F96), { 0x8C46 } },
	{ UTF32_NFKD(0x2F97), { 0x8C55 } },
	{ UTF32_NFKD(0x2F98), { 0x8C78 } },
	{ UTF32_NFKD(0x2F99), { 0x8C9D } },
	{ UTF32_NFKD(0x2F9A), { 0x8D64 } },
	{ UTF32_NFKD(0x2F9B), { 0x8D70 } },
	{ UTF32_NFKD(0x2F9C), { 0x8DB3 } },
	{ UTF32_NFKD(0x2F9D), { 0x8EAB } },
	{ UTF32_NFKD(0x2F9E), { 0x8ECA } },
	{ UTF32_NFKD(0x2F9F), { 0x8F9B } },
	{ UTF32_NFKD(0x2FA0), { 0x8FB0 } },
	{ UTF32_NFKD(0x2FA1), { 0x8FB5 } },
	{ UTF32_NFKD(0x2FA2), { 0x9091 } },
	{ UTF32_NFKD(0x2FA3), { 0x9149 } },
	{ UTF32_NFKD(0x2FA4), { 0x91C6 } },
	{ UTF32_NFKD(0x2FA5), { 0x91CC } },
	{ UTF32_NFKD(0x2FA6), { 0x91D1 } },
	{ UTF32_NFKD(0x2FA7), { 0x9577 } },
	{ UTF32_NFKD(0x2FA8), { 0x9580 } },
	{ UTF32_NFKD(0x2FA9), { 0x961C } },
	{ UTF32_NFKD(0x2FAA), { 0x96B6 } },
	{ UTF32_NFKD(0x2FAB), { 0x96B9 } },
	{ UTF32_NFKD(0x2FAC), { 0x96E8 } },
	{ UTF32_NFKD(0x2FAD), { 0x9751 } },
	{ UTF32_NFKD(0x2FAE), { 0x975E } },
	{ UTF32_NFKD(0x2FAF), { 0x9762 } },
	{ UTF32_NFKD(0x2FB0), { 0x9769 } },
	{ UTF32_NFKD(0x2FB1), { 0x97CB } },
	{ UTF32_NFKD(0x2FB2), { 0x97ED } },
	{ UTF32_NFKD(0x2FB3), { 0x97F3 } },
	{ UTF32_NFKD(0x2FB4), { 0x9801 } },
	{ UTF32_NFKD(0x2FB5), { 0x98A8 } },
	{ UTF32_NFKD(0x2FB6), { 0x98DB } },
	{ UTF32_NFKD(0x2FB7), { 0x98DF } },
	{ UTF32_NFKD(0x2FB8), { 0x9996 } },
	{ UTF32_NFKD(0x2FB9), { 0x9999 } },
	{ UTF32_NFKD(0x2FBA), { 0x99AC } },
	{ UTF32_NFKD(0x2FBB), { 0x9AA8 } },
	{ UTF32_NFKD(0x2FBC), { 0x9AD8 } },
	{ UTF32_NFKD(0x2FBD), { 0x9ADF } },
	{ UTF32_NFKD(0x2FBE), { 0x9B25 } },
	{ UTF32_NFKD(0x2FBF), { 0x9B2F } },
	{ UTF32_NFKD(0x2FC0), { 0x9B32 } },
	{ UTF32_NFKD(0x2FC1), { 0x9B3C } },
	{ UTF32_NFKD(0x2FC2), { 0x9B5A } },
	{ UTF32_NFKD(0x2FC3), { 0x9CE5 } },
	{ UTF32_NFKD(0x2FC4), { 0x9E75 } },
	{ UTF32_NFKD(0x2FC5), { 0x9E7F } },
	{ UTF32_NFKD(0x2FC6), { 0x9EA5 } },
	{ UTF32_NFKD(0x2FC7), { 0x9EBB } },
	{ UTF32_NFKD(0x2FC8), { 0x9EC3 } },
	{ UTF32_NFKD(0x2FC9), { 0x9ECD } },
	{ UTF32_NFKD(0x2FCA), { 0x9ED1 } },
	{ UTF32_NFKD(0x2FCB), { 0x9EF9 } },
	{ UTF32_NFKD(0x2FCC), { 0x9EFD } },
	{ UTF32_NFKD(0x2FCD), { 0x9F0E } },
	{ UTF32_NFKD(0x2FCE), { 0x9F13 } },
	{ UTF32_NFKD(0x2FCF), { 0x9F20 } },
	{ UTF32_NFKD(0x2FD0), { 0x9F3B } },
	{ UTF32_NFKD(0x2FD1), { 0x9F4A } },
	{ UTF32_NFKD(0x2FD2), { 0x9F52 } },
	{ UTF32_NFKD(0x2FD3), { 0x9F8D } },
	{ UTF32_NFKD(0x2FD4), { 0x9F9C } },
	{ UTF32_NFKD(0x2FD5), { 0x9FA0 } },
	{ UTF32_NFKD(0x3000), { 0x0020 } },
	{ UTF32_NFKD(0x3036), { 0x3012 } },
	{ UTF32_NFKD(0x3038), { 0x5341 } },
	{ UTF32_NFKD(0x3039), { 0x5344 } },
	{ UTF32_NFKD(0x303A), { 0x5345 } },
	{ UTF32_NFD(0x304C), { 0x304B, 0x3099 } },
	{ UTF32_NFD(0x304E), { 0x304D, 0x3099 } },
	{ UTF32_NFD(0x3050), { 0x304F, 0x3099 } },
	{ UTF32_NFD(0x3052), { 0x3051, 0x3099 } },
	{ UTF32_NFD(0x3054), { 0x3053, 0x3099 } },
	{ UTF32_NFD(0x3056), { 0x3055, 0x3099 } },
	{ UTF32_NFD(0x3058), { 0x3057, 0x3099 } },
	{ UTF32_NFD(0x305A), { 0x3059, 0x3099 } },
	{ UTF32_NFD(0x305C), { 0x305B, 0x3099 } },
	{ UTF32_NFD(0x305E), { 0x305D, 0x3099 } },
	{ UTF32_NFD(0x3060), { 0x305F, 0x3099 } },
	{ UTF32_NFD(0x3062), { 0x3061, 0x3099 } },
	{ UTF32_NFD(0x3065), { 0x3064, 0x3099 } },
	{ UTF32_NFD(0x3067), { 0x3066, 0x3099 } },
	{ UTF32_NFD(0x3069), { 0x3068, 0x3099 } },
	{ UTF32_NFD(0x3070), { 0x306F, 0x3099 } },
	{ UTF32_NFD(0x3071), { 0x306F, 0x309A } },
	{ UTF32_NFD(0x3073), { 0x3072, 0x3099 } },
	{ UTF32_NFD(0x3074), { 0x3072, 0x309A } },
	{ UTF32_NFD(0x3076), { 0x3075, 0x3099 } },
	{ UTF32_NFD(0x3077), { 0x3075, 0x309A } },
	{ UTF32_NFD(0x3079), { 0x3078, 0x3099 } },
	{ UTF32_NFD(0x307A), { 0x3078, 0x309A } },
	{ UTF32_NFD(0x307C), { 0x307B, 0x3099 } },
	{ UTF32_NFD(0x307D), { 0x307B, 0x309A } },
	{ UTF32_NFD(0x3094), { 0x3046, 0x3099 } },
	{ UTF32_NFKD(0x309B), { 0x0020, 0x3099 } },
	{ UTF32_NFKD(0x309C), { 0x0020, 0x309A } },
	{ UTF32_NFD(0x309E), { 0x309D, 0x3099 } },
	{ UTF32_NFKD(0x309F), { 0x3088, 0x308A } },
	{ UTF32_NFD(0x30AC), { 0x30AB, 0x3099 } },
	{ UTF32_NFD(0x30AE), { 0x30AD, 0x3099 } },
	{ UTF32_NFD(0x30B0), { 0x30AF, 0x3099 } },
	{ UTF32_NFD(0x30B2), { 0x30B1, 0x3099 } },
	{ UTF32_NFD(0x30B4), { 0x30B3, 0x3099 } },
	{ UTF32_NFD(0x30B6), { 0x30B5, 0x3099 } },
	{ UTF32_NFD(0x30B8), { 0x30B7, 0x3099 } },
	{ UTF32_NFD(0x30BA), { 0x30B9, 0x3099 } },
	{ UTF32_NFD(0x30BC), { 0x30BB, 0x3099 } },
	{ UTF32_NFD(0x30BE), { 0x30BD, 0x3099 } },
	{ UTF32_NFD(0x30C0), { 0x30BF, 0x3099 } },
	{ UTF32_NFD(0x30C2), { 0x30C1, 0x3099 } },
	{ UTF32_NFD(0x30C5), { 0x30C4, 0x3099 } },
	{ UTF32_NFD(0x30C7), { 0x30C6, 0x3099 } },
	{ UTF32_NFD(0x30C9), { 0x30C8, 0x3099 } },
	{ UTF32_NFD(0x30D0), { 0x30CF, 0x3099 } },
	{ UTF32_NFD(0x30D1), { 0x30CF, 0x309A } },
	{ UTF32_NFD(0x30D3), { 0x30D2, 0x3099 } },
	{ UTF32_NFD(0x30D4), { 0x30D2, 0x309A } },
	{ UTF32_NFD(0x30D6), { 0x30D5, 0x3099 } },
	{ UTF32_NFD(0x30D7), { 0x30D5, 0x309A } },
	{ UTF32_NFD(0x30D9), { 0x30D8, 0x3099 } },
	{ UTF32_NFD(0x30DA), { 0x30D8, 0x309A } },
	{ UTF32_NFD(0x30DC), { 0x30DB, 0x3099 } },
	{ UTF32_NFD(0x30DD), { 0x30DB, 0x309A } },
	{ UTF32_NFD(0x30F4), { 0x30A6, 0x3099 } },
	{ UTF32_NFD(0x30F7), { 0x30EF, 0x3099 } },
	{ UTF32_NFD(0x30F8), { 0x30F0, 0x3099 } },
	{ UTF32_NFD(0x30F9), { 0x30F1, 0x3099 } },
	{ UTF32_NFD(0x30FA), { 0x30F2, 0x3099 } },
	{ UTF32_NFD(0x30FE), { 0x30FD, 0x3099 } },
	{ UTF32_NFKD(0x30FF), { 0x30B3, 0x30C8 } },
	{ UTF32_NFKD(0x3131), { 0x1100 } },
	{ UTF32_NFKD(0x3132), { 0x1101 } },
	{ UTF32_NFKD(0x3133), { 0x11AA } },
	{ UTF32_NFKD(0x3134), { 0x1102 } },
	{ UTF32_NFKD(0x3135), { 0x11AC } },
	{ UTF32_NFKD(0x3136), { 0x11AD } },
	{ UTF32_NFKD(0x3137), { 0x1103 } },
	{ UTF32_NFKD(0x3138), { 0x1104 } },
	{ UTF32_NFKD(0x3139), { 0x1105 } },
	{ UTF32_NFKD(0x313A), { 0x11B0 } },
	{ UTF32_NFKD(0x313B), { 0x11B1 } },
	{ UTF32_NFKD(0x313C), { 0x11B2 } },
	{ UTF32_NFKD(0x313D), { 0x11B3 } },
	{ UTF32_NFKD(0x313E), { 0x11B4 } },
	{ UTF32_NFKD(0x313F), { 0x11B5 } },
	{ UTF32_NFKD(0x3140), { 0x111A } },
	{ UTF32_NFKD(0x3141), { 0x1106 } },
	{ UTF32_NFKD(0x3142), { 0x1107 } },
	{ UTF32_NFKD(0x3143), { 0x1108 } },
	{ UTF32_NFKD(0x3144), { 0x1121 } },
	{ UTF32_NFKD(0x3145), { 0x1109 } },
	{ UTF32_NFKD(0x3146), { 0x110A } },
	{ UTF32_NFKD(0x3147), { 0x110B } },
	{ UTF32_NFKD(0x3148), { 0x110C } },
	{ UTF32_NFKD(0x3149), { 0x110D } },
	{ UTF32_NFKD(0x314A), { 0x110E } },
	{ UTF32_NFKD(0x314B), { 0x110F } },
	{ UTF32_NFKD(0x314C), { 0x1110 } },
	{ UTF32_NFKD(0x314D), { 0x1111 } },
	{ UTF32_NFKD(0x314E), { 0x1112 } },
	{ UTF32_NFKD(0x314F), { 0x1161 } },
	{ UTF32_NFKD(0x3150), { 0x1162 } },
	{ UTF32_NFKD(0x3151), { 0x1163 } },
	{ UTF32_NFKD(0x3152), { 0x1164 } },
	{ UTF32_NFKD(0x3153), { 0x1165 } },
	{ UTF32_NFKD(0x3154), { 0x1166 } },
	{ UTF32_NFKD(0x3155), { 0x1167 } },
	{ UTF32_NFKD(0x3156), { 0x1168 } },
	{ UTF32_NFKD(0x3157), { 0x1169 } },
	{ UTF32_NFKD(0x3158), { 0x116A } },
	{ UTF32_NFKD(0x3159), { 0x116B } },
	{ UTF32_NFKD(0x315A), { 0x116C } },
	{ UTF32_NFKD(0x315B), { 0x116D } },
	{ UTF32_NFKD(0x315C), { 0x116E } },
	{ UTF32_NFKD(0x315D), { 0x116F } },
	{ UTF32_NFKD(0x315E), { 0x1170 } },
	{ UTF32_NFKD(0x315F), { 0x1171 } },
	{ UTF32_NFKD(0x3160), { 0x1172 } },
	{ UTF32_NFKD(0x3161), { 0x1173 } },
	{ UTF32_NFKD(0x3162), { 0x1174 } },
	{ UTF32_NFKD(0x3163), { 0x1175 } },
	{ UTF32_NFKD(0x3164), { 0x1160 } },
	{ UTF32_NFKD(0x3165), { 0x1114 } },
	{ UTF32_NFKD(0x3166), { 0x1115 } },
	{ UTF32_NFKD(0x3167), { 0x11C7 } },
	{ UTF32_NFKD(0x3168), { 0x11C8 } },
	{ UTF32_NFKD(0x3169), { 0x11CC } },
	{ UTF32_NFKD(0x316A), { 0x11CE } },
	{ UTF32_NFKD(0x316B), { 0x11D3 } },
	{ UTF32_NFKD(0x316C), { 0x11D7 } },
	{ UTF32_NFKD(0x316D), { 0x11D9 } },
	{ UTF32_NFKD(0x316E), { 0x111C } },
	{ UTF32_NFKD(0x316F), { 0x11DD } },
	{ UTF32_NFKD(0x3170), { 0x11DF } },
	{ UTF32_NFKD(0x3171), { 0x111D } },
	{ UTF32_NFKD(0x3172), { 0x111E } },
	{ UTF32_NFKD(0x3173), { 0x1120 } },
	{ UTF32_NFKD(0x3174), { 0x1122 } },
	{ UTF32_NFKD(0x3175), { 0x1123 } },
	{ UTF32_NFKD(0x3176), { 0x1127 } },
	{ UTF32_NFKD(0x3177), { 0x1129 } },
	{ UTF32_NFKD(0x3178), { 0x112B } },
	{ UTF32_NFKD(0x3179), { 0x112C } },
	{ UTF32_NFKD(0x317A), { 0x112D } },
	{ UTF32_NFKD(0x317B), { 0x112E } },
	{ UTF32_NFKD(0x317C), { 0x112F } },
	{ UTF32_NFKD(0x317D), { 0x1132 } },
	{ UTF32_NFKD(0x317E), { 0x1136 } },
	{ UTF32_NFKD(0x317F), { 0x1140 } },
	{ UTF32_NFKD(0x3180), { 0x1147 } },
	{ UTF32_NFKD(0x3181), { 0x114C } },
	{ UTF32_NFKD(0x3182), { 0x11F1 } },
	{ UTF32_NFKD(0x3183), { 0x11F2 } },
	{ UTF32_NFKD(0x3184), { 0x1157 } },
	{ UTF32_NFKD(0x3185), { 0x1158 } },
	{ UTF32_NFKD(0x3186), { 0x1159 } },
	{ UTF32_NFKD(0x3187), { 0x1184 } },
	{ UTF32_NFKD(0x3188), { 0x1185 } },
	{ UTF32_NFKD(0x3189), { 0x1188 } },
	{ UTF32_NFKD(0x318A), { 0x1191 } },
	{ UTF32_NFKD(0x318B), { 0x1192 } },
	{ UTF32_NFKD(0x318C), { 0x1194 } },
	{ UTF32_NFKD(0x318D), { 0x119E } },
	{ UTF32_NFKD(0x318E), { 0x11A1 } },
	{ UTF32_NFKD(0x3192), { 0x4E00 } },
	{ UTF32_NFKD(0x3193), { 0x4E8C } },
	{ UTF32_NFKD(0x3194), { 0x4E09 } },
	{ UTF32_NFKD(0x3195), { 0x56DB } },
	{ UTF32_NFKD(0x3196), { 0x4E0A } },
	{ UTF32_NFKD(0x3197), { 0x4E2D } },
	{ UTF32_NFKD(0x3198), { 0x4E0B } },
	{ UTF32_NFKD(0x3199), { 0x7532 } },
	{ UTF32_NFKD(0x319A), { 0x4E59 } },
	{ UTF32_NFKD(0x319B), { 0x4E19 } },
	{ UTF32_NFKD(0x319C), { 0x4E01 } },
	{ UTF32_NFKD(0x319D), { 0x5929 } },
	{ UTF32_NFKD(0x319E), { 0x5730 } },
	{ UTF32_NFKD(0x319F), { 0x4EBA } },
	{ UTF32_NFKD(0x3200), { 0x0028, 0x1100, 0x0029 } },
	{ UTF32_NFKD(0x3201), { 0x0028, 0x1102, 0x0029 } },
	{ UTF32_NFKD(0x3202), { 0x0028, 0x1103, 0x0029 } },
	{ UTF32_NFKD(0x3203), { 0x0028, 0x1105, 0x0029 } },
	{ UTF32_NFKD(0x3204), { 0x0028, 0x1106, 0x0029 } },
	{ UTF32_NFKD(0x3205), { 0x0028, 0x1107, 0x0029 } },
	{ UTF32_NFKD(0x3206), { 0x0028, 0x1109, 0x0029 } },
	{ UTF32_NFKD(0x3207), { 0x0028, 0x110B, 0x0029 } },
	{ UTF32_NFKD(0x3208), { 0x0028, 0x110C, 0x0029 } },
	{ UTF32_NFKD(0x3209), { 0x0028, 0x110E, 0x0029 } },
	{ UTF32_NFKD(0x320A), { 0x0028, 0x110F, 0x0029 } },
	{ UTF32_NFKD(0x320B), { 0x0028, 0x1110, 0x0029 } },
	{ UTF32_NFKD(0x320C), { 0x0028, 0x1111, 0x0029 } },
	{ UTF32_NFKD(0x320D), { 0x0028, 0x1112, 0x0029 } },
	{ UTF32_NFKD(0x320E), { 0x0028, 0x1100, 0x1161, 0x0029 } },
	{ UTF32_NFKD(0x320F), { 0x0028, 0x1102, 0x1161, 0x0029 } },
	{ UTF32_NFKD(0x3210), { 0x0028, 0x1103, 0x1161, 0x0029 } },
	{ UTF32_NFKD(0x3211), { 0x0028, 0x1105, 0x1161, 0x0029 } },
	{ UTF32_NFKD(0x3212), { 0x0028, 0x1106, 0x1161, 0x0029 } },
	{ UTF32_NFKD(0x3213), { 0x0028, 0x1107, 0x1161, 0x0029 } },
	{ UTF32_NFKD(0x3214), { 0x0028, 0x1109, 0x1161, 0x0029 } },
	{ UTF32_NFKD(0x3215), { 0x0028, 0x110B, 0x1161, 0x0029 } },
	{ UTF32_NFKD(0x3216), { 0x0028, 0x110C, 0x1161, 0x0029 } },
	{ UTF32_NFKD(0x3217), { 0x0028, 0x110E, 0x1161, 0x0029 } },
	{ UTF32_NFKD(0x3218), { 0x0028, 0x110F, 0x1161, 0x0029 } },
	{ UTF32_NFKD(0x3219), { 0x0028, 0x1110, 0x1161, 0x0029 } },
	{ UTF32_NFKD(0x321A), { 0x0028, 0x1111, 0x1161, 0x0029 } },
	{ UTF32_NFKD(0x321B), { 0x0028, 0x1112, 0x1161, 0x0029 } },
	{ UTF32_NFKD(0x321C), { 0x0028, 0x110C, 0x116E, 0x0029 } },
	{ UTF32_NFKD(0x321D), { 0x0028, 0x110B, 0x1169, 0x110C, 0x1165, 0x11AB, 0x0029 } },
	{ UTF32_NFKD(0x321E), { 0x0028, 0x110B, 0x1169, 0x1112, 0x116E, 0x0029 } },
	{ UTF32_NFKD(0x3220), { 0x0028, 0x4E00, 0x0029 } },
	{ UTF32_NFKD(0x3221), { 0x0028, 0x4E8C, 0x0029 } },
	{ UTF32_NFKD(0x3222), { 0x0028, 0x4E09, 0x0029 } },
	{ UTF32_NFKD(0x3223), { 0x0028, 0x56DB, 0x0029 } },
	{ UTF32_NFKD(0x3224), { 0x0028, 0x4E94, 0x0029 } },
	{ UTF32_NFKD(0x3225), { 0x0028, 0x516D, 0x0029 } },
	{ UTF32_NFKD(0x3226), { 0x0028, 0x4E03, 0x0029 } },
	{ UTF32_NFKD(0x3227), { 0x0028, 0x516B, 0x0029 } },
	{ UTF32_NFKD(0x3228), { 0x0028, 0x4E5D, 0x0029 } },
	{ UTF32_NFKD(0x3229), { 0x0028, 0x5341, 0x0029 } },
	{ UTF32_NFKD(0x322A), { 0x0028, 0x6708, 0x0029 } },
	{ UTF32_NFKD(0x322B), { 0x0028, 0x706B, 0x0029 } },
	{ UTF32_NFKD(0x322C), { 0x0028, 0x6C34, 0x0029 } },
	{ UTF32_NFKD(0x322D), { 0x0028, 0x6728, 0x0029 } },
	{ UTF32_NFKD(0x322E), { 0x0028, 0x91D1, 0x0029 } },
	{ UTF32_NFKD(0x322F), { 0x0028, 0x571F, 0x0029 } },
	{ UTF32_NFKD(0x3230), { 0x0028, 0x65E5, 0x0029 } },
	{ UTF32_NFKD(0x3231), { 0x0028, 0x682A, 0x0029 } },
	{ UTF32_NFKD(0x3232), { 0x0028, 0x6709, 0x0029 } },
	{ UTF32_NFKD(0x3233), { 0x0028, 0x793E, 0x0029 } },
	{ UTF32_NFKD(0x3234), { 0x0028, 0x540D, 0x0029 } },
	{ UTF32_NFKD(0x3235), { 0x0028, 0x7279, 0x0029 } },
	{ UTF32_NFKD(0x3236), { 0x0028, 0x8CA1, 0x0029 } },
	{ UTF32_NFKD(0x3237), { 0x0028, 0x795D, 0x0029 } },
	{ UTF32_NFKD(0x3238), { 0x0028, 0x52B4, 0x0029 } },
	{ UTF32_NFKD(0x3239), { 0x0028, 0x4EE3, 0x0029 } },
	{ UTF32_NFKD(0x323A), { 0x0028, 0x547C, 0x0029 } },
	{ UTF32_NFKD(0x323B), { 0x0028, 0x5B66, 0x0029 } },
	{ UTF32_NFKD(0x323C), { 0x0028, 0x76E3, 0x0029 } },
	{ UTF32_NFKD(0x323D), { 0x0028, 0x4F01, 0x0029 } },
	{ UTF32_NFKD(0x323E), { 0x0028, 0x8CC7, 0x0029 } },
	{ UTF32_NFKD(0x323F), { 0x0028, 0x5354, 0x0029 } },
	{ UTF32_NFKD(0x3240), { 0x0028, 0x796D, 0x0029 } },
	{ UTF32_NFKD(0x3241), { 0x0028, 0x4F11, 0x0029 } },
	{ UTF32_NFKD(0x3242), { 0x0028, 0x81EA, 0x0029 } },
	{ UTF32_NFKD(0x3243), { 0x0028, 0x81F3, 0x0029 } },
	{ UTF32_NFKD(0x3250), { 0x0050, 0x0054, 0x0045 } },
	{ UTF32_NFKD(0x3251), { 0x0032, 0x0031 } },
	{ UTF32_NFKD(0x3252), { 0x0032, 0x0032 } },
	{ UTF32_NFKD(0x3253), { 0x0032, 0x0033 } },
	{ UTF32_NFKD(0x3254), { 0x0032, 0x0034 } },
	{ UTF32_NFKD(0x3255), { 0x0032, 0x0035 } },
	{ UTF32_NFKD(0x3256), { 0x0032, 0x0036 } },
	{ UTF32_NFKD(0x3257), { 0x0032, 0x0037 } },
	{ UTF32_NFKD(0x3258), { 0x0032, 0x0038 } },
	{ UTF32_NFKD(0x3259), { 0x0032, 0x0039 } },
	{ UTF32_NFKD(0x325A), { 0x0033, 0x0030 } },
	{ UTF32_NFKD(0x325B), { 0x0033, 0x0031 } },
	{ UTF32_NFKD(0x325C), { 0x0033, 0x0032 } },
	{ UTF32_NFKD(0x325D), { 0x0033, 0x0033 } },
	{ UTF32_NFKD(0x325E), { 0x0033, 0x0034 } },
	{ UTF32_NFKD(0x325F), { 0x0033, 0x0035 } },
	{ UTF32_NFKD(0x3260), { 0x1100 } },
	{ UTF32_NFKD(0x3261), { 0x1102 } },
	{ UTF32_NFKD(0x3262), { 0x1103 } },
	{ UTF32_NFKD(0x3263), { 0x1105 } },
	{ UTF32_NFKD(0x3264), { 0x1106 } },
	{ UTF32_NFKD(0x3265), { 0x1107 } },
	{ UTF32_NFKD(0x3266), { 0x1109 } },
	{ UTF32_NFKD(0x3267), { 0x110B } },
	{ UTF32_NFKD(0x3268), { 0x110C } },
	{ UTF32_NFKD(0x3269), { 0x110E } },
	{ UTF32_NFKD(0x326A), { 0x110F } },
	{ UTF32_NFKD(0x326B), { 0x1110 } },
	{ UTF32_NFKD(0x326C), { 0x1111 } },
	{ UTF32_NFKD(0x326D), { 0x1112 } },
	{ UTF32_NFKD(0x326E), { 0x1100, 0x1161 } },
	{ UTF32_NFKD(0x326F), { 0x1102, 0x1161 } },
	{ UTF32_NFKD(0x3270), { 0x1103, 0x1161 } },
	{ UTF32_NFKD(0x3271), { 0x1105, 0x1161 } },
	{ UTF32_NFKD(0x3272), { 0x1106, 0x1161 } },
	{ UTF32_NFKD(0x3273), { 0x1107, 0x1161 } },
	{ UTF32_NFKD(0x3274), { 0x1109, 0x1161 } },
	{ UTF32_NFKD(0x3275), { 0x110B, 0x1161 } },
	{ UTF32_NFKD(0x3276), { 0x110C, 0x1161 } },
	{ UTF32_NFKD(0x3277), { 0x110E, 0x1161 } },
	{ UTF32_NFKD(0x3278), { 0x110F, 0x1161 } },
	{ UTF32_NFKD(0x3279), { 0x1110, 0x1161 } },
	{ UTF32_NFKD(0x327A), { 0x1111, 0x1161 } },
	{ UTF32_NFKD(0x327B), { 0x1112, 0x1161 } },
	{ UTF32_NFKD(0x327C), { 0x110E, 0x1161, 0x11B7, 0x1100, 0x1169 } },
	{ UTF32_NFKD(0x327D), { 0x110C, 0x116E, 0x110B, 0x1174 } },
	{ UTF32_NFKD(0x327E), { 0x110B, 0x116E } },
	{ UTF32_NFKD(0x3280), { 0x4E00 } },
	{ UTF32_NFKD(0x3281), { 0x4E8C } },
	{ UTF32_NFKD(0x3282), { 0x4E09 } },
	{ UTF32_NFKD(0x3283), { 0x56DB } },
	{ UTF32_NFKD(0x3284), { 0x4E94 } },
	{ UTF32_NFKD(0x3285), { 0x516D } },
	{ UTF32_NFKD(0x3286), { 0x4E03 } },
	{ UTF32_NFKD(0x3287), { 0x516B } },
	{ UTF32_NFKD(0x3288), { 0x4E5D } },
	{ UTF32_NFKD(0x3289), { 0x5341 } },
	{ UTF32_NFKD(0x328A), { 0x6708 } },
	{ UTF32_NFKD(0x328B), { 0x706B } },
	{ UTF32_NFKD(0x328C), { 0x6C34 } },
	{ UTF32_NFKD(0x328D), { 0x6728 } },
	{ UTF32_NFKD(0x328E), { 0x91D1 } },
	{ UTF32_NFKD(0x328F), { 0x571F } },
	{ UTF32_NFKD(0x3290), { 0x65E5 } },
	{ UTF32_NFKD(0x3291), { 0x682A } },
	{ UTF32_NFKD(0x3292), { 0x6709 } },
	{ UTF32_NFKD(0x3293), { 0x793E } },
	{ UTF32_NFKD(0x3294), { 0x540D } },
	{ UTF32_NFKD(0x3295), { 0x7279 } },
	{ UTF32_NFKD(0x3296), { 0x8CA1 } },
	{ UTF32_NFKD(0x3297), { 0x795D } },
	{ UTF32_NFKD(0x3298), { 0x52B4 } },
	{ UTF32_NFKD(0x3299), { 0x79D8 } },
	{ UTF32_NFKD(0x329A), { 0x7537 } },
	{ UTF32_NFKD(0x329B), { 0x5973 } },
	{ UTF32_NFKD(0x329C), { 0x9069 } },
	{ UTF32_NFKD(0x329D), { 0x512A } },
	{ UTF32_NFKD(0x329E), { 0x5370 } },
	{ UTF32_NFKD(0x329F), { 0x6CE8 } },
	{ UTF32_NFKD(0x32A0), { 0x9805 } },
	{ UTF32_NFKD(0x32A1), { 0x4F11 } },
	{ UTF32_NFKD(0x32A2), { 0x5199 } },
	{ UTF32_NFKD(0x32A3), { 0x6B63 } },
	{ UTF32_NFKD(0x32A4), { 0x4E0A } },
	{ UTF32_NFKD(0x32A5), { 0x4E2D } },
	{ UTF32_NFKD(0x32A6), { 0x4E0B } },
	{ UTF32_NFKD(0x32A7), { 0x5DE6 } },
	{ UTF32_NFKD(0x32A8), { 0x53F3 } },
	{ UTF32_NFKD(0x32A9), { 0x533B } },
	{ UTF32_NFKD(0x32AA), { 0x5B97 } },
	{ UTF32_NFKD(0x32AB), { 0x5B66 } },
	{ UTF32_NFKD(0x32AC), { 0x76E3 } },
	{ UTF32_NFKD(0x32AD), { 0x4F01 } },
	{ UTF32_NFKD(0x32AE), { 0x8CC7 } },
	{ UTF32_NFKD(0x32AF), { 0x5354 } },
	{ UTF32_NFKD(0x32B0), { 0x591C } },
	{ UTF32_NFKD(0x32B1), { 0x0033, 0x0036 } },
	{ UTF32_NFKD(0x32B2), { 0x0033, 0x0037 } },
	{ UTF32_NFKD(0x32B3), { 0x0033, 0x0038 } },
	{ UTF32_NFKD(0x32B4), { 0x0033, 0x0039 } },
	{ UTF32_NFKD(0x32B5), { 0x0034, 0x0030 } },
	{ UTF32_NFKD(0x32B6), { 0x0034, 0x0031 } },
	{ UTF32_NFKD(0x32B7), { 0x0034, 0x0032 } },
	{ UTF32_NFKD(0x32B8), { 0x0034, 0x0033 } },
	{ UTF32_NFKD(0x32B9), { 0x0034, 0x0034 } },
	{ UTF32_NFKD(0x32BA), { 0x0034, 0x0035 } },
	{ UTF32_NFKD(0x32BB), { 0x0034, 0x0036 } },
	{ UTF32_NFKD(0x32BC), { 0x0034, 0x0037 } },
	{ UTF32_NFKD(0x32BD), { 0x0034, 0x0038 } },
	{ UTF32_NFKD(0x32BE), { 0x0034, 0x0039 } },
	{ UTF32_NFKD(0x32BF), { 0x0035, 0x0030 } },
	{ UTF32_NFKD(0x32C0), { 0x0031, 0x6708 } },
	{ UTF32_NFKD(0x32C1), { 0x0032, 0x6708 } },
	{ UTF32_NFKD(0x32C2), { 0x0033, 0x6708 } },
	{ UTF32_NFKD(0x32C3), { 0x0034, 0x6708 } },
	{ UTF32_NFKD(0x32C4), { 0x0035, 0x6708 } },
	{ UTF32_NFKD(0x32C5), { 0x0036, 0x6708 } },
	{ UTF32_NFKD(0x32C6), { 0x0037, 0x6708 } },
	{ UTF32_NFKD(0x32C7), { 0x0038, 0x6708 } },
	{ UTF32_NFKD(0x32C8), { 0x0039, 0x6708 } },
	{ UTF32_NFKD(0x32C9), { 0x0031, 0x0030, 0x6708 } },
	{ UTF32_NFKD(0x32CA), { 0x0031, 0x0031, 0x6708 } },
	{ UTF32_NFKD(0x32CB), { 0x0031, 0x0032, 0x6708 } },
	{ UTF32_NFKD(0x32CC), { 0x0048, 0x0067 } },
	{ UTF32_NFKD(0x32CD), { 0x0065, 0x0072, 0x0067 } },
	{ UTF32_NFKD(0x32CE), { 0x0065, 0x0056 } },
	{ UTF32_NFKD(0x32CF), { 0x004C, 0x0054, 0x0044 } },
	{ UTF32_NFKD(0x32D0), { 0x30A2 } },
	{ UTF32_NFKD(0x32D1), { 0x30A4 } },
	{ UTF32_NFKD(0x32D2), { 0x30A6 } },
	{ UTF32_NFKD(0x32D3), { 0x30A8 } },
	{ UTF32_NFKD(0x32D4), { 0x30AA } },
	{ UTF32_NFKD(0x32D5), { 0x30AB } },
	{ UTF32_NFKD(0x32D6), { 0x30AD } },
	{ UTF32_NFKD(0x32D7), { 0x30AF } },
	{ UTF32_NFKD(0x32D8), { 0x30B1 } },
	{ UTF32_NFKD(0x32D9), { 0x30B3 } },
	{ UTF32_NFKD(0x32DA), { 0x30B5 } },
	{ UTF32_NFKD(0x32DB), { 0x30B7 } },
	{ UTF32_NFKD(0x32DC), { 0x30B9 } },
	{ UTF32_NFKD(0x32DD), { 0x30BB } },
	{ UTF32_NFKD(0x32DE), { 0x30BD } },
	{ UTF32_NFKD(0x32DF), { 0x30BF } },
	{ UTF32_NFKD(0x32E0), { 0x30C1 } },
	{ UTF32_NFKD(0x32E1), { 0x30C4 } },
	{ UTF32_NFKD(0x32E2), { 0x30C6 } },
	{ UTF32_NFKD(0x32E3), { 0x30C8 } },
	{ UTF32_NFKD(0x32E4), { 0x30CA } },
	{ UTF32_NFKD(0x32E5), { 0x30CB } },
	{ UTF32_NFKD(0x32E6), { 0x30CC } },
	{ UTF32_NFKD(0x32E7), { 0x30CD } },
	{ UTF32_NFKD(0x32E8), { 0x30CE } },
	{ UTF32_NFKD(0x32E9), { 0x30CF } },
	{ UTF32_NFKD(0x32EA), { 0x30D2 } },
	{ UTF32_NFKD(0x32EB), { 0x30D5 } },
	{ UTF32_NFKD(0x32EC), { 0x30D8 } },
	{ UTF32_NFKD(0x32ED), { 0x30DB } },
	{ UTF32_NFKD(0x32EE), { 0x30DE } },
	{ UTF32_NFKD(0x32EF), { 0x30DF } },
	{ UTF32_NFKD(0x32F0), { 0x30E0 } },
	{ UTF32_NFKD(0x32F1), { 0x30E1 } },
	{ UTF32_NFKD(0x32F2), { 0x30E2 } },
	{ UTF32_NFKD(0x32F3), { 0x30E4 } },
	{ UTF32_NFKD(0x32F4), { 0x30E6 } },
	{ UTF32_NFKD(0x32F5), { 0x30E8 } },
	{ UTF32_NFKD(0x32F6), { 0x30E9 } },
	{ UTF32_NFKD(0x32F7), { 0x30EA } },
	{ UTF32_NFKD(0x32F8), { 0x30EB } },
	{ UTF32_NFKD(0x32F9), { 0x30EC } },
	{ UTF32_NFKD(0x32FA), { 0x30ED } },
	{ UTF32_NFKD(0x32FB), { 0x30EF } },
	{ UTF32_NFKD(0x32FC), { 0x30F0 } },
	{ UTF32_NFKD(0x32FD), { 0x30F1 } },
	{ UTF32_NFKD(0x32FE), { 0x30F2 } },
	{ UTF32_NFKD(0x3300), { 0x30A2, 0x30D1, 0x30FC, 0x30C8 } },
	{ UTF32_NFKD(0x3301), { 0x30A2, 0x30EB, 0x30D5, 0x30A1 } },
	{ UTF32_NFKD(0x3302), { 0x30A2, 0x30F3, 0x30DA, 0x30A2 } },
	{ UTF32_NFKD(0x3303), { 0x30A2, 0x30FC, 0x30EB } },
	{ UTF32_NFKD(0x3304), { 0x30A4, 0x30CB, 0x30F3, 0x30B0 } },
	{ UTF32_NFKD(0x3305), { 0x30A4, 0x30F3, 0x30C1 } },
	{ UTF32_NFKD(0x3306), { 0x30A6, 0x30A9, 0x30F3 } },
	{ UTF32_NFKD(0x3307), { 0x30A8, 0x30B9, 0x30AF, 0x30FC, 0x30C9 } },
	{ UTF32_NFKD(0x3308), { 0x30A8, 0x30FC, 0x30AB, 0x30FC } },
	{ UTF32_NFKD(0x3309), { 0x30AA, 0x30F3, 0x30B9 } },
	{ UTF32_NFKD(0x330A), { 0x30AA, 0x30FC, 0x30E0 } },
	{ UTF32_NFKD(0x330B), { 0x30AB, 0x30A4, 0x30EA } },
	{ UTF32_NFKD(0x330C), { 0x30AB, 0x30E9, 0x30C3, 0x30C8 } },
	{ UTF32_NFKD(0x330D), { 0x30AB, 0x30ED, 0x30EA, 0x30FC } },
	{ UTF32_NFKD(0x330E), { 0x30AC, 0x30ED, 0x30F3 } },
	{ UTF32_NFKD(0x330F), { 0x30AC, 0x30F3, 0x30DE } },
	{ UTF32_NFKD(0x3310), { 0x30AE, 0x30AC } },
	{ UTF32_NFKD(0x3311), { 0x30AE, 0x30CB, 0x30FC } },
	{ UTF32_NFKD(0x3312), { 0x30AD, 0x30E5, 0x30EA, 0x30FC } },
	{ UTF32_NFKD(0x3313), { 0x30AE, 0x30EB, 0x30C0, 0x30FC } },
	{ UTF32_NFKD(0x3314), { 0x30AD, 0x30ED } },
	{ UTF32_NFKD(0x3315), { 0x30AD, 0x30ED, 0x30B0, 0x30E9, 0x30E0 } },
	{ UTF32_NFKD(0x3316), { 0x30AD, 0x30ED, 0x30E1, 0x30FC, 0x30C8, 0x30EB } },
	{ UTF32_NFKD(0x3317), { 0x30AD, 0x30ED, 0x30EF, 0x30C3, 0x30C8 } },
	{ UTF32_NFKD(0x3318), { 0x30B0, 0x30E9, 0x30E0 } },
	{ UTF32_NFKD(0x3319), { 0x30B0, 0x30E9, 0x30E0, 0x30C8, 0x30F3 } },
	{ UTF32_NFKD(0x331A), { 0x30AF, 0x30EB, 0x30BC, 0x30A4, 0x30ED } },
	{ UTF32_NFKD(0x331B), { 0x30AF, 0x30ED, 0x30FC, 0x30CD } },
	{ UTF32_NFKD(0x331C), { 0x30B1, 0x30FC, 0x30B9 } },
	{ UTF32_NFKD(0x331D), { 0x30B3, 0x30EB, 0x30CA } },
	{ UTF32_NFKD(0x331E), { 0x30B3, 0x30FC, 0x30DD } },
	{ UTF32_NFKD(0x331F), { 0x30B5, 0x30A4, 0x30AF, 0x30EB } },
	{ UTF32_NFKD(0x3320), { 0x30B5, 0x30F3, 0x30C1, 0x30FC, 0x30E0 } },
	{ UTF32_NFKD(0x3321), { 0x30B7, 0x30EA, 0x30F3, 0x30B0 } },
	{ UTF32_NFKD(0x3322), { 0x30BB, 0x30F3, 0x30C1 } },
	{ UTF32_NFKD(0x3323), { 0x30BB, 0x30F3, 0x30C8 } },
	{ UTF32_NFKD(0x3324), { 0x30C0, 0x30FC, 0x30B9 } },
	{ UTF32_NFKD(0x3325), { 0x30C7, 0x30B7 } },
	{ UTF32_NFKD(0x3326), { 0x30C9, 0x30EB } },
	{ UTF32_NFKD(0x3327), { 0x30C8, 0x30F3 } },
	{ UTF32_NFKD(0x3328), { 0x30CA, 0x30CE } },
	{ UTF32_NFKD(0x3329), { 0x30CE, 0x30C3, 0x30C8 } },
	{ UTF32_NFKD(0x332A), { 0x30CF, 0x30A4, 0x30C4 } },
	{ UTF32_NFKD(0x332B), { 0x30D1, 0x30FC, 0x30BB, 0x30F3, 0x30C8 } },
	{ UTF32_NFKD(0x332C), { 0x30D1, 0x30FC, 0x30C4 } },
	{ UTF32_NFKD(0x332D), { 0x30D0, 0x30FC, 0x30EC, 0x30EB } },
	{ UTF32_NFKD(0x332E), { 0x30D4, 0x30A2, 0x30B9, 0x30C8, 0x30EB } },
	{ UTF32_NFKD(0x332F), { 0x30D4, 0x30AF, 0x30EB } },
	{ UTF32_NFKD(0x3330), { 0x30D4, 0x30B3 } },
	{ UTF32_NFKD(0x3331), { 0x30D3, 0x30EB } },
	{ UTF32_NFKD(0x3332), { 0x30D5, 0x30A1, 0x30E9, 0x30C3, 0x30C9 } },
	{ UTF32_NFKD(0x3333), { 0x30D5, 0x30A3, 0x30FC, 0x30C8 } },
	{ UTF32_NFKD(0x3334), { 0x30D6, 0x30C3, 0x30B7, 0x30A7, 0x30EB } },
	{ UTF32_NFKD(0x3335), { 0x30D5, 0x30E9, 0x30F3 } },
	{ UTF32_NFKD(0x3336), { 0x30D8, 0x30AF, 0x30BF, 0x30FC, 0x30EB } },
	{ UTF32_NFKD(0x3337), { 0x30DA, 0x30BD } },
	{ UTF32_NFKD(0x3338), { 0x30DA, 0x30CB, 0x30D2 } },
	{ UTF32_NFKD(0x3339), { 0x30D8, 0x30EB, 0x30C4 } },
	{ UTF32_NFKD(0x333A), { 0x30DA, 0x30F3, 0x30B9 } },
	{ UTF32_NFKD(0x333B), { 0x30DA, 0x30FC, 0x30B8 } },
	{ UTF32_NFKD(0x333C), { 0x30D9, 0x30FC, 0x30BF } },
	{ UTF32_NFKD(0x333D), { 0x30DD, 0x30A4, 0x30F3, 0x30C8 } },
	{ UTF32_NFKD(0x333E), { 0x30DC, 0x30EB, 0x30C8 } },
	{ UTF32_NFKD(0x333F), { 0x30DB, 0x30F3 } },
	{ UTF32_NFKD(0x3340), { 0x30DD, 0x30F3, 0x30C9 } },
	{ UTF32_NFKD(0x3341), { 0x30DB, 0x30FC, 0x30EB } },
	{ UTF32_NFKD(0x3342), { 0x30DB, 0x30FC, 0x30F3 } },
	{ UTF32_NFKD(0x3343), { 0x30DE, 0x30A4, 0x30AF, 0x30ED } },
	{ UTF32_NFKD(0x3344), { 0x30DE, 0x30A4, 0x30EB } },
	{ UTF32_NFKD(0x3345), { 0x30DE, 0x30C3, 0x30CF } },
	{ UTF32_NFKD(0x3346), { 0x30DE, 0x30EB, 0x30AF } },
	{ UTF32_NFKD(0x3347), { 0x30DE, 0x30F3, 0x30B7, 0x30E7, 0x30F3 } },
	{ UTF32_NFKD(0x3348), { 0x30DF, 0x30AF, 0x30ED, 0x30F3 } },
	{ UTF32_NFKD(0x3349), { 0x30DF, 0x30EA } },
	{ UTF32_NFKD(0x334A), { 0x30DF, 0x30EA, 0x30D0, 0x30FC, 0x30EB } },
	{ UTF32_NFKD(0x334B), { 0x30E1, 0x30AC } },
	{ UTF32_NFKD(0x334C), { 0x30E1, 0x30AC, 0x30C8, 0x30F3 } },
	{ UTF32_NFKD(0x334D), { 0x30E1, 0x30FC, 0x30C8, 0x30EB } },
	{ UTF32_NFKD(0x334E), { 0x30E4, 0x30FC, 0x30C9 } },
	{ UTF32_NFKD(0x334F), { 0x30E4, 0x30FC, 0x30EB } },
	{ UTF32_NFKD(0x3350), { 0x30E6, 0x30A2, 0x30F3 } },
	{ UTF32_NFKD(0x3351), { 0x30EA, 0x30C3, 0x30C8, 0x30EB } },
	{ UTF32_NFKD(0x3352), { 0x30EA, 0x30E9 } },
	{ UTF32_NFKD(0x3353), { 0x30EB, 0x30D4, 0x30FC } },
	{ UTF32_NFKD(0x3354), { 0x30EB, 0x30FC, 0x30D6, 0x30EB } },
	{ UTF32_NFKD(0x3355), { 0x30EC, 0x30E0 } },
	{ UTF32_NFKD(0x3356), { 0x30EC, 0x30F3, 0x30C8, 0x30B2, 0x30F3 } },
	{ UTF32_NFKD(0x3357), { 0x30EF, 0x30C3, 0x30C8 } },
	{ UTF32_NFKD(0x3358), { 0x0030, 0x70B9 } },
	{ UTF32_NFKD(0x3359), { 0x0031, 0x70B9 } },
	{ UTF32_NFKD(0x335A), { 0x0032, 0x70B9 } },
	{ UTF32_NFKD(0x335B), { 0x0033, 0x70B9 } },
	{ UTF32_NFKD(0x335C), { 0x0034, 0x70B9 } },
	{ UTF32_NFKD(0x335D), { 0x0035, 0x70B9 } },
	{ UTF32_NFKD(0x335E), { 0x0036, 0x70B9 } },
	{ UTF32_NFKD(0x335F), { 0x0037, 0x70B9 } },
	{ UTF32_NFKD(0x3360), { 0x0038, 0x70B9 } },
	{ UTF32_NFKD(0x3361), { 0x0039, 0x70B9 } },
	{ UTF32_NFKD(0x3362), { 0x0031, 0x0030, 0x70B9 } },
	{ UTF32_NFKD(0x3363), { 0x0031, 0x0031, 0x70B9 } },
	{ UTF32_NFKD(0x3364), { 0x0031, 0x0032, 0x70B9 } },
	{ UTF32_NFKD(0x3365), { 0x0031, 0x0033, 0x70B9 } },
	{ UTF32_NFKD(0x3366), { 0x0031, 0x0034, 0x70B9 } },
	{ UTF32_NFKD(0x3367), { 0x0031, 0x0035, 0x70B9 } },
	{ UTF32_NFKD(0x3368), { 0x0031, 0x0036, 0x70B9 } },
	{ UTF32_NFKD(0x3369), { 0x0031, 0x0037, 0x70B9 } },
	{ UTF32_NFKD(0x336A), { 0x0031, 0x0038, 0x70B9 } },
	{ UTF32_NFKD(0x336B), { 0x0031, 0x0039, 0x70B9 } },
	{ UTF32_NFKD(0x336C), { 0x0032, 0x0030, 0x70B9 } },
	{ UTF32_NFKD(0x336D), { 0x0032, 0x0031, 0x70B9 } },
	{ UTF32_NFKD(0x336E), { 0x0032, 0x0032, 0x70B9 } },
	{ UTF32_NFKD(0x336F), { 0x0032, 0x0033, 0x70B9 } },
	{ UTF32_NFKD(0x3370), { 0x0032, 0x0034, 0x70B9 } },
	{ UTF32_NFKD(0x3371), { 0x0068, 0x0050, 0x0061 } },
	{ UTF32_NFKD(0x3372), { 0x0064, 0x0061 } },
	{ UTF32_NFKD(0x3373), { 0x0041, 0x0055 } },
	{ UTF32_NFKD(0x3374), { 0x0062, 0x0061, 0x0072 } },
	{ UTF32_NFKD(0x3375), { 0x006F, 0x0056 } },
	{ UTF32_NFKD(0x3376), { 0x0070, 0x0063 } },
	{ UTF32_NFKD(0x3377), { 0x0064, 0x006D } },
	{ UTF32_NFKD(0x3378), { 0x0064, 0x006D, 0x00B2 } },
	{ UTF32_NFKD(0x3379), { 0x0064, 0x006D, 0x00B3 } },
	{ UTF32_NFKD(0x337A), { 0x0049, 0x0055 } },
	{ UTF32_NFKD(0x337B), { 0x5E73, 0x6210 } },
	{ UTF32_NFKD(0x337C), { 0x662D, 0x548C } },
	{ UTF32_NFKD(0x337D), { 0x5927, 0x6B63 } },
	{ UTF32_NFKD(0x337E), { 0x660E, 0x6CBB } },
	{ UTF32_NFKD(0x337F), { 0x682A, 0x5F0F, 0x4F1A, 0x793E } },
	{ UTF32_NFKD(0x3380), { 0x0070, 0x0041 } },
	{ UTF32_NFKD(0x3381), { 0x006E, 0x0041 } },
	{ UTF32_NFKD(0x3382), { 0x03BC, 0x0041 } },
	{ UTF32_NFKD(0x3383), { 0x006D, 0x0041 } },
	{ UTF32_NFKD(0x3384), { 0x006B, 0x0041 } },
	{ UTF32_NFKD(0x3385), { 0x004B, 0x0042 } },
	{ UTF32_NFKD(0x3386), { 0x004D, 0x0042 } },
	{ UTF32_NFKD(0x3387), { 0x0047, 0x0042 } },
	{ UTF32_NFKD(0x3388), { 0x0063, 0x0061, 0x006C } },
	{ UTF32_NFKD(0x3389), { 0x006B, 0x0063, 0x0061, 0x006C } },
	{ UTF32_NFKD(0x338A), { 0x0070, 0x0046 } },
	{ UTF32_NFKD(0x338B), { 0x006E, 0x0046 } },
	{ UTF32_NFKD(0x338C), { 0x03BC, 0x0046 } },
	{ UTF32_NFKD(0x338D), { 0x03BC, 0x0067 } },
	{ UTF32_NFKD(0x338E), { 0x006D, 0x0067 } },
	{ UTF32_NFKD(0x338F), { 0x006B, 0x0067 } },
	{ UTF32_NFKD(0x3390), { 0x0048, 0x007A } },
	{ UTF32_NFKD(0x3391), { 0x006B, 0x0048, 0x007A } },
	{ UTF32_NFKD(0x3392), { 0x004D, 0x0048, 0x007A } },
	{ UTF32_NFKD(0x3393), { 0x0047, 0x0048, 0x007A } },
	{ UTF32_NFKD(0x3394), { 0x0054, 0x0048, 0x007A } },
	{ UTF32_NFKD(0x3395), { 0x03BC, 0x2113 } },
	{ UTF32_NFKD(0x3396), { 0x006D, 0x2113 } },
	{ UTF32_NFKD(0x3397), { 0x0064, 0x2113 } },
	{ UTF32_NFKD(0x3398), { 0x006B, 0x2113 } },
	{ UTF32_NFKD(0x3399), { 0x0066, 0x006D } },
	{ UTF32_NFKD(0x339A), { 0x006E, 0x006D } },
	{ UTF32_NFKD(0x339B), { 0x03BC, 0x006D } },
	{ UTF32_NFKD(0x339C), { 0x006D, 0x006D } },
	{ UTF32_NFKD(0x339D), { 0x0063, 0x006D } },
	{ UTF32_NFKD(0x339E), { 0x006B, 0x006D } },
	{ UTF32_NFKD(0x339F), { 0x006D, 0x006D, 0x00B2 } },
	{ UTF32_NFKD(0x33A0), { 0x0063, 0x006D, 0x00B2 } },
	{ UTF32_NFKD(0x33A1), { 0x006D, 0x00B2 } },
	{ UTF32_NFKD(0x33A2), { 0x006B, 0x006D, 0x00B2 } },
	{ UTF32_NFKD(0x33A3), { 0x006D, 0x006D, 0x00B3 } },
	{ UTF32_NFKD(0x33A4), { 0x0063, 0x006D, 0x00B3 } },
	{ UTF32_NFKD(0x33A5), { 0x006D, 0x00B3 } },
	{ UTF32_NFKD(0x33A6), { 0x006B, 0x006D, 0x00B3 } },
	{ UTF32_NFKD(0x33A7), { 0x006D, 0x2215, 0x0073 } },
	{ UTF32_NFKD(0x33A8), { 0x006D, 0x2215, 0x0073, 0x00B2 } },
	{ UTF32_NFKD(0x33A9), { 0x0050, 0x0061 } },
	{ UTF32_NFKD(0x33AA), { 0x006B, 0x0050, 0x0061 } },
	{ UTF32_NFKD(0x33AB), { 0x004D, 0x0050, 0x0061 } },
	{ UTF32_NFKD(0x33AC), { 0x0047, 0x0050, 0x0061 } },
	{ UTF32_NFKD(0x33AD), { 0x0072, 0x0061, 0x0064 } },
	{ UTF32_NFKD(0x33AE), { 0x0072, 0x0061, 0x0064, 0x2215, 0x0073 } },
	{ UTF32_NFKD(0x33AF), { 0x0072, 0x0061, 0x0064, 0x2215, 0x0073, 0x00B2 } },
	{ UTF32_NFKD(0x33B0), { 0x0070, 0x0073 } },
	{ UTF32_NFKD(0x33B1), { 0x006E, 0x0073 } },
	{ UTF32_NFKD(0x33B2), { 0x03BC, 0x0073 } },
	{ UTF32_NFKD(0x33B3), { 0x006D, 0x0073 } },
	{ UTF32_NFKD(0x33B4), { 0x0070, 0x0056 } },
	{ UTF32_NFKD(0x33B5), { 0x006E, 0x0056 } },
	{ UTF32_NFKD(0x33B6), { 0x03BC, 0x0056 } },
	{ UTF32_NFKD(0x33B7), { 0x006D, 0x0056 } },
	{ UTF32_NFKD(0x33B8), { 0x006B, 0x0056 } },
	{ UTF32_NFKD(0x33B9), { 0x004D, 0x0056 } },
	{ UTF32_NFKD(0x33BA), { 0x0070, 0x0057 } },
	{ UTF32_NFKD(0x33BB), { 0x006E, 0x0057 } },
	{ UTF32_NFKD(0x33BC), { 0x03BC, 0x0057 } },
	{ UTF32_NFKD(0x33BD), { 0x006D, 0x0057 } },
	{ UTF32_NFKD(0x33BE), { 0x006B, 0x0057 } },
	{ UTF32_NFKD(0x33BF), { 0x004D, 0x0057 } },
	{ UTF32_NFKD(0x33C0), { 0x006B, 0x03A9 } },
	{ UTF32_NFKD(0x33C1), { 0x004D, 0x03A9 } },
	{ UTF32_NFKD(0x33C2), { 0x0061, 0x002E, 0x006D, 0x002E } },
	{ UTF32_NFKD(0x33C3), { 0x0042, 0x0071 } },
	{ UTF32_NFKD(0x33C4), { 0x0063, 0x0063 } },
	{ UTF32_NFKD(0x33C5), { 0x0063, 0x0064 } },
	{ UTF32_NFKD(0x33C6), { 0x0043, 0x2215, 0x006B, 0x0067 } },
	{ UTF32_NFKD(0x33C7), { 0x0043, 0x006F, 0x002E } },
	{ UTF32_NFKD(0x33C8), { 0x0064, 0x0042 } },
	{ UTF32_NFKD(0x33C9), { 0x0047, 0x0079 } },
	{ UTF32_NFKD(0x33CA), { 0x0068, 0x0061 } },
	{ UTF32_NFKD(0x33CB), { 0x0048, 0x0050 } },
	{ UTF32_NFKD(0x33CC), { 0x0069, 0x006E } },
	{ UTF32_NFKD(0x33CD), { 0x004B, 0x004B } },
	{ UTF32_NFKD(0x33CE), { 0x004B, 0x004D } },
	{ UTF32_NFKD(0x33CF), { 0x006B, 0x0074 } },
	{ UTF32_NFKD(0x33D0), { 0x006C, 0x006D } },
	{ UTF32_NFKD(0x33D1), { 0x006C, 0x006E } },
	{ UTF32_NFKD(0x33D2), { 0x006C, 0x006F, 0x0067 } },
	{ UTF32_NFKD(0x33D3), { 0x006C, 0x0078 } },
	{ UTF32_NFKD(0x33D4), { 0x006D, 0x0062 } },
	{ UTF32_NFKD(0x33D5), { 0x006D, 0x0069, 0x006C } },
	{ UTF32_NFKD(0x33D6), { 0x006D, 0x006F, 0x006C } },
	{ UTF32_NFKD(0x33D7), { 0x0050, 0x0048 } },
	{ UTF32_NFKD(0x33D8), { 0x0070, 0x002E, 0x006D, 0x002E } },
	{ UTF32_NFKD(0x33D9), { 0x0050, 0x0050, 0x004D } },
	{ UTF32_NFKD(0x33DA), { 0x0050, 0x0052 } },
	{ UTF32_NFKD(0x33DB), { 0x0073, 0x0072 } },
	{ UTF32_NFKD(0x33DC), { 0x0053, 0x0076 } },
	{ UTF32_NFKD(0x33DD), { 0x0057, 0x0062 } },
	{ UTF32_NFKD(0x33DE), { 0x0056, 0x2215, 0x006D } },
	{ UTF32_NFKD(0x33DF), { 0x0041, 0x2215, 0x006D } },
	{ UTF32_NFKD(0x33E0), { 0x0031, 0x65E5 } },
	{ UTF32_NFKD(0x33E1), { 0x0032, 0x65E5 } },
	{ UTF32_NFKD(0x33E2), { 0x0033, 0x65E5 } },
	{ UTF32_NFKD(0x33E3), { 0x0034, 0x65E5 } },
	{ UTF32_NFKD(0x33E4), { 0x0035, 0x65E5 } },
	{ UTF32_NFKD(0x33E5), { 0x0036, 0x65E5 } },
	{ UTF32_NFKD(0x33E6), { 0x0037, 0x65E5 } },
	{ UTF32_NFKD(0x33E7), { 0x0038, 0x65E5 } },
	{ UTF32_NFKD(0x33E8), { 0x0039, 0x65E5 } },
	{ UTF32_NFKD(0x33E9), { 0x0031, 0x0030, 0x65E5 } },
	{ UTF32_NFKD(0x33EA), { 0x0031, 0x0031, 0x65E5 } },
	{ UTF32_NFKD(0x33EB), { 0x0031, 0x0032, 0x65E5 } },
	{ UTF32_NFKD(0x33EC), { 0x0031, 0x0033, 0x65E5 } },
	{ UTF32_NFKD(0x33ED), { 0x0031, 0x0034, 0x65E5 } },
	{ UTF32_NFKD(0x33EE), { 0x0031, 0x0035, 0x65E5 } },
	{ UTF32_NFKD(0x33EF), { 0x0031, 0x0036, 0x65E5 } },
	{ UTF32_NFKD(0x33F0), { 0x0031, 0x0037, 0x65E5 } },
	{ UTF32_NFKD(0x33F1), { 0x0031, 0x0038, 0x65E5 } },
	{ UTF32_NFKD(0x33F2), { 0x0031, 0x0039, 0x65E5 } },
	{ UTF32_NFKD(0x33F3), { 0x0032, 0x0030, 0x65E5 } },
	{ UTF32_NFKD(0x33F4), { 0x0032, 0x0031, 0x65E5 } },
	{ UTF32_NFKD(0x33F5), { 0x0032, 0x0032, 0x65E5 } },
	{ UTF32_NFKD(0x33F6), { 0x0032, 0x0033, 0x65E5 } },
	{ UTF32_NFKD(0x33F7), { 0x0032, 0x0034, 0x65E5 } },
	{ UTF32_NFKD(0x33F8), { 0x0032, 0x0035, 0x65E5 } },
	{ UTF32_NFKD(0x33F9), { 0x0032, 0x0036, 0x65E5 } },
	{ UTF32_NFKD(0x33FA), { 0x0032, 0x0037, 0x65E5 } },
	{ UTF32_NFKD(0x33FB), { 0x0032, 0x0038, 0x65E5 } },
	{ UTF32_NFKD(0x33FC), { 0x0032, 0x0039, 0x65E5 } },
	{ UTF32_NFKD(0x33FD), { 0x0033, 0x0030, 0x65E5 } },
	{ UTF32_NFKD(0x33FE), { 0x0033, 0x0031, 0x65E5 } },
	{ UTF32_NFKD(0x33FF), { 0x0067, 0x0061, 0x006C } },
	{ UTF32_NFD(0xF900), { 0x8C48 } },
	{ UTF32_NFD(0xF901), { 0x66F4 } },
	{ UTF32_NFD(0xF902), { 0x8ECA } },
	{ UTF32_NFD(0xF903), { 0x8CC8 } },
	{ UTF32_NFD(0xF904), { 0x6ED1 } },
	{ UTF32_NFD(0xF905), { 0x4E32 } },
	{ UTF32_NFD(0xF906), { 0x53E5 } },
	{ UTF32_NFD(0xF907), { 0x9F9C } },
	{ UTF32_NFD(0xF908), { 0x9F9C } },
	{ UTF32_NFD(0xF909), { 0x5951 } },
	{ UTF32_NFD(0xF90A), { 0x91D1 } },
	{ UTF32_NFD(0xF90B), { 0x5587 } },
	{ UTF32_NFD(0xF90C), { 0x5948 } },
	{ UTF32_NFD(0xF90D), { 0x61F6 } },
	{ UTF32_NFD(0xF90E), { 0x7669 } },
	{ UTF32_NFD(0xF90F), { 0x7F85 } },
	{ UTF32_NFD(0xF910), { 0x863F } },
	{ UTF32_NFD(0xF911), { 0x87BA } },
	{ UTF32_NFD(0xF912), { 0x88F8 } },
	{ UTF32_NFD(0xF913), { 0x908F } },
	{ UTF32_NFD(0xF914), { 0x6A02 } },
	{ UTF32_NFD(0xF915), { 0x6D1B } },
	{ UTF32_NFD(0xF916), { 0x70D9 } },
	{ UTF32_NFD(0xF917), { 0x73DE } },
	{ UTF32_NFD(0xF918), { 0x843D } },
	{ UTF32_NFD(0xF919), { 0x916A } },
	{ UTF32_NFD(0xF91A), { 0x99F1 } },
	{ UTF32_NFD(0xF91B), { 0x4E82 } },
	{ UTF32_NFD(0xF91C), { 0x5375 } },
	{ UTF32_NFD(0xF91D), { 0x6B04 } },
	{ UTF32_NFD(0xF91E), { 0x721B } },
	{ UTF32_NFD(0xF91F), { 0x862D } },
	{ UTF32_NFD(0xF920), { 0x9E1E } },
	{ UTF32_NFD(0xF921), { 0x5D50 } },
	{ UTF32_NFD(0xF922), { 0x6FEB } },
	{ UTF32_NFD(0xF923), { 0x85CD } },
	{ UTF32_NFD(0xF924), { 0x8964 } },
	{ UTF32_NFD(0xF925), { 0x62C9 } },
	{ UTF32_NFD(0xF926), { 0x81D8 } },
	{ UTF32_NFD(0xF927), { 0x881F } },
	{ UTF32_NFD(0xF928), { 0x5ECA } },
	{ UTF32_NFD(0xF929), { 0x6717 } },
	{ UTF32_NFD(0xF92A), { 0x6D6A } },
	{ UTF32_NFD(0xF92B), { 0x72FC } },
	{ UTF32_NFD(0xF92C), { 0x90CE } },
	{ UTF32_NFD(0xF92D), { 0x4F86 } },
	{ UTF32_NFD(0xF92E), { 0x51B7 } },
	{ UTF32_NFD(0xF92F), { 0x52DE } },
	{ UTF32_NFD(0xF930), { 0x64C4 } },
	{ UTF32_NFD(0xF931), { 0x6AD3 } },
	{ UTF32_NFD(0xF932), { 0x7210 } },
	{ UTF32_NFD(0xF933), { 0x76E7 } },
	{ UTF32_NFD(0xF934), { 0x8001 } },
	{ UTF32_NFD(0xF935), { 0x8606 } },
	{ UTF32_NFD(0xF936), { 0x865C } },
	{ UTF32_NFD(0xF937), { 0x8DEF } },
	{ UTF32_NFD(0xF938), { 0x9732 } },
	{ UTF32_NFD(0xF939), { 0x9B6F } },
	{ UTF32_NFD(0xF93A), { 0x9DFA } },
	{ UTF32_NFD(0xF93B), { 0x788C } },
	{ UTF32_NFD(0xF93C), { 0x797F } },
	{ UTF32_NFD(0xF93D), { 0x7DA0 } },
	{ UTF32_NFD(0xF93E), { 0x83C9 } },
	{ UTF32_NFD(0xF93F), { 0x9304 } },
	{ UTF32_NFD(0xF940), { 0x9E7F } },
	{ UTF32_NFD(0xF941), { 0x8AD6 } },
	{ UTF32_NFD(0xF942), { 0x58DF } },
	{ UTF32_NFD(0xF943), { 0x5F04 } },
	{ UTF32_NFD(0xF944), { 0x7C60 } },
	{ UTF32_NFD(0xF945), { 0x807E } },
	{ UTF32_NFD(0xF946), { 0x7262 } },
	{ UTF32_NFD(0xF947), { 0x78CA } },
	{ UTF32_NFD(0xF948), { 0x8CC2 } },
	{ UTF32_NFD(0xF949), { 0x96F7 } },
	{ UTF32_NFD(0xF94A), { 0x58D8 } },
	{ UTF32_NFD(0xF94B), { 0x5C62 } },
	{ UTF32_NFD(0xF94C), { 0x6A13 } },
	{ UTF32_NFD(0xF94D), { 0x6DDA } },
	{ UTF32_NFD(0xF94E), { 0x6F0F } },
	{ UTF32_NFD(0xF94F), { 0x7D2F } },
	{ UTF32_NFD(0xF950), { 0x7E37 } },
	{ UTF32_NFD(0xF951), { 0x964B } },
	{ UTF32_NFD(0xF952), { 0x52D2 } },
	{ UTF32_NFD(0xF953), { 0x808B } },
	{ UTF32_NFD(0xF954), { 0x51DC } },
	{ UTF32_NFD(0xF955), { 0x51CC } },
	{ UTF32_NFD(0xF956), { 0x7A1C } },
	{ UTF32_NFD(0xF957), { 0x7DBE } },
	{ UTF32_NFD(0xF958), { 0x83F1 } },
	{ UTF32_NFD(0xF959), { 0x9675 } },
	{ UTF32_NFD(0xF95A), { 0x8B80 } },
	{ UTF32_NFD(0xF95B), { 0x62CF } },
	{ UTF32_NFD(0xF95C), { 0x6A02 } },
	{ UTF32_NFD(0xF95D), { 0x8AFE } },
	{ UTF32_NFD(0xF95E), { 0x4E39 } },
	{ UTF32_NFD(0xF95F), { 0x5BE7 } },
	{ UTF32_NFD(0xF960), { 0x6012 } },
	{ UTF32_NFD(0xF961), { 0x7387 } },
	{ UTF32_NFD(0xF962), { 0x7570 } },
	{ UTF32_NFD(0xF963), { 0x5317 } },
	{ UTF32_NFD(0xF964), { 0x78FB } },
	{ UTF32_NFD(0xF965), { 0x4FBF } },
	{ UTF32_NFD(0xF966), { 0x5FA9 } },
	{ UTF32_NFD(0xF967), { 0x4E0D } },
	{ UTF32_NFD(0xF968), { 0x6CCC } },
	{ UTF32_NFD(0xF969), { 0x6578 } },
	{ UTF32_NFD(0xF96A), { 0x7D22 } },
	{ UTF32_NFD(0xF96B), { 0x53C3 } },
	{ UTF32_NFD(0xF96C), { 0x585E } },
	{ UTF32_NFD(0xF96D), { 0x7701 } },
	{ UTF32_NFD(0xF96E), { 0x8449 } },
	{ UTF32_NFD(0xF96F), { 0x8AAA } },
	{ UTF32_NFD(0xF970), { 0x6BBA } },
	{ UTF32_NFD(0xF971), { 0x8FB0 } },
	{ UTF32_NFD(0xF972), { 0x6C88 } },
	{ UTF32_NFD(0xF973), { 0x62FE } },
	{ UTF32_NFD(0xF974), { 0x82E5 } },
	{ UTF32_NFD(0xF975), { 0x63A0 } },
	{ UTF32_NFD(0xF976), { 0x7565 } },
	{ UTF32_NFD(0xF977), { 0x4EAE } },
	{ UTF32_NFD(0xF978), { 0x5169 } },
	{ UTF32_NFD(0xF979), { 0x51C9 } },
	{ UTF32_NFD(0xF97A), { 0x6881 } },
	{ UTF32_NFD(0xF97B), { 0x7CE7 } },
	{ UTF32_NFD(0xF97C), { 0x826F } },
	{ UTF32_NFD(0xF97D), { 0x8AD2 } },
	{ UTF32_NFD(0xF97E), { 0x91CF } },
	{ UTF32_NFD(0xF97F), { 0x52F5 } },
	{ UTF32_NFD(0xF980), { 0x5442 } },
	{ UTF32_NFD(0xF981), { 0x5973 } },
	{ UTF32_NFD(0xF982), { 0x5EEC } },
	{ UTF32_NFD(0xF983), { 0x65C5 } },
	{ UTF32_NFD(0xF984), { 0x6FFE } },
	{ UTF32_NFD(0xF985), { 0x792A } },
	{ UTF32_NFD(0xF986), { 0x95AD } },
	{ UTF32_NFD(0xF987), { 0x9A6A } },
	{ UTF32_NFD(0xF988), { 0x9E97 } },
	{ UTF32_NFD(0xF989), { 0x9ECE } },
	{ UTF32_NFD(0xF98A), { 0x529B } },
	{ UTF32_NFD(0xF98B), { 0x66C6 } },
	{ UTF32_NFD(0xF98C), { 0x6B77 } },
	{ UTF32_NFD(0xF98D), { 0x8F62 } },
	{ UTF32_NFD(0xF98E), { 0x5E74 } },
	{ UTF32_NFD(0xF98F), { 0x6190 } },
	{ UTF32_NFD(0xF990), { 0x6200 } },
	{ UTF32_NFD(0xF991), { 0x649A } },
	{ UTF32_NFD(0xF992), { 0x6F23 } },
	{ UTF32_NFD(0xF993), { 0x7149 } },
	{ UTF32_NFD(0xF994), { 0x7489 } },
	{ UTF32_NFD(0xF995), { 0x79CA } },
	{ UTF32_NFD(0xF996), { 0x7DF4 } },
	{ UTF32_NFD(0xF997), { 0x806F } },
	{ UTF32_NFD(0xF998), { 0x8F26 } },
	{ UTF32_NFD(0xF999), { 0x84EE } },
	{ UTF32_NFD(0xF99A), { 0x9023 } },
	{ UTF32_NFD(0xF99B), { 0x934A } },
	{ UTF32_NFD(0xF99C), { 0x5217 } },
	{ UTF32_NFD(0xF99D), { 0x52A3 } },
	{ UTF32_NFD(0xF99E), { 0x54BD } },
	{ UTF32_NFD(0xF99F), { 0x70C8 } },
	{ UTF32_NFD(0xF9A0), { 0x88C2 } },
	{ UTF32_NFD(0xF9A1), { 0x8AAA } },
	{ UTF32_NFD(0xF9A2), { 0x5EC9 } },
	{ UTF32_NFD(0xF9A3), { 0x5FF5 } },
	{ UTF32_NFD(0xF9A4), { 0x637B } },
	{ UTF32_NFD(0xF9A5), { 0x6BAE } },
	{ UTF32_NFD(0xF9A6), { 0x7C3E } },
	{ UTF32_NFD(0xF9A7), { 0x7375 } },
	{ UTF32_NFD(0xF9A8), { 0x4EE4 } },
	{ UTF32_NFD(0xF9A9), { 0x56F9 } },
	{ UTF32_NFD(0xF9AA), { 0x5BE7 } },
	{ UTF32_NFD(0xF9AB), { 0x5DBA } },
	{ UTF32_NFD(0xF9AC), { 0x601C } },
	{ UTF32_NFD(0xF9AD), { 0x73B2 } },
	{ UTF32_NFD(0xF9AE), { 0x7469 } },
	{ UTF32_NFD(0xF9AF), { 0x7F9A } },
	{ UTF32_NFD(0xF9B0), { 0x8046 } },
	{ UTF32_NFD(0xF9B1), { 0x9234 } },
	{ UTF32_NFD(0xF9B2), { 0x96F6 } },
	{ UTF32_NFD(0xF9B3), { 0x9748 } },
	{ UTF32_NFD(0xF9B4), { 0x9818 } },
	{ UTF32_NFD(0xF9B5), { 0x4F8B } },
	{ UTF32_NFD(0xF9B6), { 0x79AE } },
	{ UTF32_NFD(0xF9B7), { 0x91B4 } },
	{ UTF32_NFD(0xF9B8), { 0x96B8 } },
	{ UTF32_NFD(0xF9B9), { 0x60E1 } },
	{ UTF32_NFD(0xF9BA), { 0x4E86 } },
	{ UTF32_NFD(0xF9BB), { 0x50DA } },
	{ UTF32_NFD(0xF9BC), { 0x5BEE } },
	{ UTF32_NFD(0xF9BD), { 0x5C3F } },
	{ UTF32_NFD(0xF9BE), { 0x6599 } },
	{ UTF32_NFD(0xF9BF), { 0x6A02 } },
	{ UTF32_NFD(0xF9C0), { 0x71CE } },
	{ UTF32_NFD(0xF9C1), { 0x7642 } },
	{ UTF32_NFD(0xF9C2), { 0x84FC } },
	{ UTF32_NFD(0xF9C3), { 0x907C } },
	{ UTF32_NFD(0xF9C4), { 0x9F8D } },
	{ UTF32_NFD(0xF9C5), { 0x6688 } },
	{ UTF32_NFD(0xF9C6), { 0x962E } },
	{ UTF32_NFD(0xF9C7), { 0x5289 } },
	{ UTF32_NFD(0xF9C8), { 0x677B } },
	{ UTF32_NFD(0xF9C9), { 0x67F3 } },
	{ UTF32_NFD(0xF9CA), { 0x6D41 } },
	{ UTF32_NFD(0xF9CB), { 0x6E9C } },
	{ UTF32_NFD(0xF9CC), { 0x7409 } },
	{ UTF32_NFD(0xF9CD), { 0x7559 } },
	{ UTF32_NFD(0xF9CE), { 0x786B } },
	{ UTF32_NFD(0xF9CF), { 0x7D10 } },
	{ UTF32_NFD(0xF9D0), { 0x985E } },
	{ UTF32_NFD(0xF9D1), { 0x516D } },
	{ UTF32_NFD(0xF9D2), { 0x622E } },
	{ UTF32_NFD(0xF9D3), { 0x9678 } },
	{ UTF32_NFD(0xF9D4), { 0x502B } },
	{ UTF32_NFD(0xF9D5), { 0x5D19 } },
	{ UTF32_NFD(0xF9D6), { 0x6DEA } },
	{ UTF32_NFD(0xF9D7), { 0x8F2A } },
	{ UTF32_NFD(0xF9D8), { 0x5F8B } },
	{ UTF32_NFD(0xF9D9), { 0x6144 } },
	{ UTF32_NFD(0xF9DA), { 0x6817 } },
	{ UTF32_NFD(0xF9DB), { 0x7387 } },
	{ UTF32_NFD(0xF9DC), { 0x9686 } },
	{ UTF32_NFD(0xF9DD), { 0x5229 } },
	{ UTF32_NFD(0xF9DE), { 0x540F } },
	{ UTF32_NFD(0xF9DF), { 0x5C65 } },
	{ UTF32_NFD(0xF9E0), { 0x6613 } },
	{ UTF32_NFD(0xF9E1), { 0x674E } },
	{ UTF32_NFD(0xF9E2), { 0x68A8 } },
	{ UTF32_NFD(0xF9E3), { 0x6CE5 } },
	{ UTF32_NFD(0xF9E4), { 0x7406 } },
	{ UTF32_NFD(0xF9E5), { 0x75E2 } },
	{ UTF32_NFD(0xF9E6), { 0x7F79 } },
	{ UTF32_NFD(0xF9E7), { 0x88CF } },
	{ UTF32_NFD(0xF9E8), { 0x88E1 } },
	{ UTF32_NFD(0xF9E9), { 0x91CC } },
	{ UTF32_NFD(0xF9EA), { 0x96E2 } },
	{ UTF32_NFD(0xF9EB), { 0x533F } },
	{ UTF32_NFD(0xF9EC), { 0x6EBA } },
	{ UTF32_NFD(0xF9ED), { 0x541D } },
	{ UTF32_NFD(0xF9EE), { 0x71D0 } },
	{ UTF32_NFD(0xF9EF), { 0x7498 } },
	{ UTF32_NFD(0xF9F0), { 0x85FA } },
	{ UTF32_NFD(0xF9F1), { 0x96A3 } },
	{ UTF32_NFD(0xF9F2), { 0x9C57 } },
	{ UTF32_NFD(0xF9F3), { 0x9E9F } },
	{ UTF32_NFD(0xF9F4), { 0x6797 } },
	{ UTF32_NFD(0xF9F5), { 0x6DCB } },
	{ UTF32_NFD(0xF9F6), { 0x81E8 } },
	{ UTF32_NFD(0xF9F7), { 0x7ACB } },
	{ UTF32_NFD(0xF9F8), { 0x7B20 } },
	{ UTF32_NFD(0xF9F9), { 0x7C92 } },
	{ UTF32_NFD(0xF9FA), { 0x72C0 } },
	{ UTF32_NFD(0xF9FB), { 0x7099 } },
	{ UTF32_NFD(0xF9FC), { 0x8B58 } },
	{ UTF32_NFD(0xF9FD), { 0x4EC0 } },
	{ UTF32_NFD(0xF9FE), { 0x8336 } },
	{ UTF32_NFD(0xF9FF), { 0x523A } },
	{ UTF32_NFD(0xFA00), { 0x5207 } },
	{ UTF32_NFD(0xFA01), { 0x5EA6 } },
	{ UTF32_NFD(0xFA02), { 0x62D3 } },
	{ UTF32_NFD(0xFA03), { 0x7CD6 } },
	{ UTF32_NFD(0xFA04), { 0x5B85 } },
	{ UTF32_NFD(0xFA05), { 0x6D1E } },
	{ UTF32_NFD(0xFA06), { 0x66B4 } },
	{ UTF32_NFD(0xFA07), { 0x8F3B } },
	{ UTF32_NFD(0xFA08), { 0x884C } },
	{ UTF32_NFD(0xFA09), { 0x964D } },
	{ UTF32_NFD(0xFA0A), { 0x898B } },
	{ UTF32_NFD(0xFA0B), { 0x5ED3 } },
	{ UTF32_NFD(0xFA0C), { 0x5140 } },
	{ UTF32_NFD(0xFA0D), { 0x55C0 } },
	{ UTF32_NFD(0xFA10), { 0x585A } },
	{ UTF32_NFD(0xFA12), { 0x6674 } },
	{ UTF32_NFD(0xFA15), { 0x51DE } },
	{ UTF32_NFD(0xFA16), { 0x732A } },
	{ UTF32_NFD(0xFA17), { 0x76CA } },
	{ UTF32_NFD(0xFA18), { 0x793C } },
	{ UTF32_NFD(0xFA19), { 0x795E } },
	{ UTF32_NFD(0xFA1A), { 0x7965 } },
	{ UTF32_NFD(0xFA1B), { 0x798F } },
	{ UTF32_NFD(0xFA1C), { 0x9756 } },
	{ UTF32_NFD(0xFA1D), { 0x7CBE } },
	{ UTF32_NFD(0xFA1E), { 0x7FBD } },
	{ UTF32_NFD(0xFA20), { 0x8612 } },
	{ UTF32_NFD(0xFA22), { 0x8AF8 } },
	{ UTF32_NFD(0xFA25), { 0x9038 } },
	{ UTF32_NFD(0xFA26), { 0x90FD } },
	{ UTF32_NFD(0xFA2A), { 0x98EF } },
	{ UTF32_NFD(0xFA2B), { 0x98FC } },
	{ UTF32_NFD(0xFA2C), { 0x9928 } },
	{ UTF32_NFD(0xFA2D), { 0x9DB4 } },
	{ UTF32_NFD(0xFA30), { 0x4FAE } },
	{ UTF32_NFD(0xFA31), { 0x50E7 } },
	{ UTF32_NFD(0xFA32), { 0x514D } },
	{ UTF32_NFD(0xFA33), { 0x52C9 } },
	{ UTF32_NFD(0xFA34), { 0x52E4 } },
	{ UTF32_NFD(0xFA35), { 0x5351 } },
	{ UTF32_NFD(0xFA36), { 0x559D } },
	{ UTF32_NFD(0xFA37), { 0x5606 } },
	{ UTF32_NFD(0xFA38), { 0x5668 } },
	{ UTF32_NFD(0xFA39), { 0x5840 } },
	{ UTF32_NFD(0xFA3A), { 0x58A8 } },
	{ UTF32_NFD(0xFA3B), { 0x5C64 } },
	{ UTF32_NFD(0xFA3C), { 0x5C6E } },
	{ UTF32_NFD(0xFA3D), { 0x6094 } },
	{ UTF32_NFD(0xFA3E), { 0x6168 } },
	{ UTF32_NFD(0xFA3F), { 0x618E } },
	{ UTF32_NFD(0xFA40), { 0x61F2 } },
	{ UTF32_NFD(0xFA41), { 0x654F } },
	{ UTF32_NFD(0xFA42), { 0x65E2 } },
	{ UTF32_NFD(0xFA43), { 0x6691 } },
	{ UTF32_NFD(0xFA44), { 0x6885 } },
	{ UTF32_NFD(0xFA45), { 0x6D77 } },
	{ UTF32_NFD(0xFA46), { 0x6E1A } },
	{ UTF32_NFD(0xFA47), { 0x6F22 } },
	{ UTF32_NFD(0xFA48), { 0x716E } },
	{ UTF32_NFD(0xFA49), { 0x722B } },
	{ UTF32_NFD(0xFA4A), { 0x7422 } },
	{ UTF32_NFD(0xFA4B), { 0x7891 } },
	{ UTF32_NFD(0xFA4C), { 0x793E } },
	{ UTF32_NFD(0xFA4D), { 0x7949 } },
	{ UTF32_NFD(0xFA4E), { 0x7948 } },
	{ UTF32_NFD(0xFA4F), { 0x7950 } },
	{ UTF32_NFD(0xFA50), { 0x7956 } },
	{ UTF32_NFD(0xFA51), { 0x795D } },
	{ UTF32_NFD(0xFA52), { 0x798D } },
	{ UTF32_NFD(0xFA53), { 0x798E } },
	{ UTF32_NFD(0xFA54), { 0x7A40 } },
	{ UTF32_NFD(0xFA55), { 0x7A81 } },
	{ UTF32_NFD(0xFA56), { 0x7BC0 } },
	{ UTF32_NFD(0xFA57), { 0x7DF4 } },
	{ UTF32_NFD(0xFA58), { 0x7E09 } },
	{ UTF32_NFD(0xFA59), { 0x7E41 } },
	{ UTF32_NFD(0xFA5A), { 0x7F72 } },
	{ UTF32_NFD(0xFA5B), { 0x8005 } },
	{ UTF32_NFD(0xFA5C), { 0x81ED } },
	{ UTF32_NFD(0xFA5D), { 0x8279 } },
	{ UTF32_NFD(0xFA5E), { 0x8279 } },
	{ UTF32_NFD(0xFA5F), { 0x8457 } },
	{ UTF32_NFD(0xFA60), { 0x8910 } },
	{ UTF32_NFD(0xFA61), { 0x8996 } },
	{ UTF32_NFD(0xFA62), { 0x8B01 } },
	{ UTF32_NFD(0xFA63), { 0x8B39 } },
	{ UTF32_NFD(0xFA64), { 0x8CD3 } },
	{ UTF32_NFD(0xFA65), { 0x8D08 } },
	{ UTF32_NFD(0xFA66), { 0x8FB6 } },
	{ UTF32_NFD(0xFA67), { 0x9038 } },
	{ UTF32_NFD(0xFA68), { 0x96E3 } },
	{ UTF32_NFD(0xFA69), { 0x97FF } },
	{ UTF32_NFD(0xFA6A), { 0x983B } },
	{ UTF32_NFD(0xFA70), { 0x4E26 } },
	{ UTF32_NFD(0xFA71), { 0x51B5 } },
	{ UTF32_NFD(0xFA72), { 0x5168 } },
	{ UTF32_NFD(0xFA73), { 0x4F80 } },
	{ UTF32_NFD(0xFA74), { 0x5145 } },
	{ UTF32_NFD(0xFA75), { 0x5180 } },
	{ UTF32_NFD(0xFA76), { 0x52C7 } },
	{ UTF32_NFD(0xFA77), { 0x52FA } },
	{ UTF32_NFD(0xFA78), { 0x559D } },
	{ UTF32_NFD(0xFA79), { 0x5555 } },
	{ UTF32_NFD(0xFA7A), { 0x5599 } },
	{ UTF32_NFD(0xFA7B), { 0x55E2 } },
	{ UTF32_NFD(0xFA7C), { 0x585A } },
	{ UTF32_NFD(0xFA7D), { 0x58B3 } },
	{ UTF32_NFD(0xFA7E), { 0x5944 } },
	{ UTF32_NFD(0xFA7F), { 0x5954 } },
	{ UTF32_NFD(0xFA80), { 0x5A62 } },
	{ UTF32_NFD(0xFA81), { 0x5B28 } },
	{ UTF32_NFD(0xFA82), { 0x5ED2 } },
	{ UTF32_NFD(0xFA83), { 0x5ED9 } },
	{ UTF32_NFD(0xFA84), { 0x5F69 } },
	{ UTF32_NFD(0xFA85), { 0x5FAD } },
	{ UTF32_NFD(0xFA86), { 0x60D8 } },
	{ UTF32_NFD(0xFA87), { 0x614E } },
	{ UTF32_NFD(0xFA88), { 0x6108 } },
	{ UTF32_NFD(0xFA89), { 0x618E } },
	{ UTF32_NFD(0xFA8A), { 0x6160 } },
	{ UTF32_NFD(0xFA8B), { 0x61F2 } },
	{ UTF32_NFD(0xFA8C), { 0x6234 } },
	{ UTF32_NFD(0xFA8D), { 0x63C4 } },
	{ UTF32_NFD(0xFA8E), { 0x641C } },
	{ UTF32_NFD(0xFA8F), { 0x6452 } },
	{ UTF32_NFD(0xFA90), { 0x6556 } },
	{ UTF32_NFD(0xFA91), { 0x6674 } },
	{ UTF32_NFD(0xFA92), { 0x6717 } },
	{ UTF32_NFD(0xFA93), { 0x671B } },
	{ UTF32_NFD(0xFA94), { 0x6756 } },
	{ UTF32_NFD(0xFA95), { 0x6B79 } },
	{ UTF32_NFD(0xFA96), { 0x6BBA } },
	{ UTF32_NFD(0xFA97), { 0x6D41 } },
	{ UTF32_NFD(0xFA98), { 0x6EDB } },
	{ UTF32_NFD(0xFA99), { 0x6ECB } },
	{ UTF32_NFD(0xFA9A), { 0x6F22 } },
	{ UTF32_NFD(0xFA9B), { 0x701E } },
	{ UTF32_NFD(0xFA9C), { 0x716E } },
	{ UTF32_NFD(0xFA9D), { 0x77A7 } },
	{ UTF32_NFD(0xFA9E), { 0x7235 } },
	{ UTF32_NFD(0xFA9F), { 0x72AF } },
	{ UTF32_NFD(0xFAA0), { 0x732A } },
	{ UTF32_NFD(0xFAA1), { 0x7471 } },
	{ UTF32_NFD(0xFAA2), { 0x7506 } },
	{ UTF32_NFD(0xFAA3), { 0x753B } },
	{ UTF32_NFD(0xFAA4), { 0x761D } },
	{ UTF32_NFD(0xFAA5), { 0x761F } },
	{ UTF32_NFD(0xFAA6), { 0x76CA } },
	{ UTF32_NFD(0xFAA7), { 0x76DB } },
	{ UTF32_NFD(0xFAA8), { 0x76F4 } },
	{ UTF32_NFD(0xFAA9), { 0x774A } },
	{ UTF32_NFD(0xFAAA), { 0x7740 } },
	{ UTF32_NFD(0xFAAB), { 0x78CC } },
	{ UTF32_NFD(0xFAAC), { 0x7AB1 } },
	{ UTF32_NFD(0xFAAD), { 0x7BC0 } },
	{ UTF32_NFD(0xFAAE), { 0x7C7B } },
	{ UTF32_NFD(0xFAAF), { 0x7D5B } },
	{ UTF32_NFD(0xFAB0), { 0x7DF4 } },
	{ UTF32_NFD(0xFAB1), { 0x7F3E } },
	{ UTF32_NFD(0xFAB2), { 0x8005 } },
	{ UTF32_NFD(0xFAB3), { 0x8352 } },
	{ UTF32_NFD(0xFAB4), { 0x83EF } },
	{ UTF32_NFD(0xFAB5), { 0x8779 } },
	{ UTF32_NFD(0xFAB6), { 0x8941 } },
	{ UTF32_NFD(0xFAB7), { 0x8986 } },
	{ UTF32_NFD(0xFAB8), { 0x8996 } },
	{ UTF32_NFD(0xFAB9), { 0x8ABF } },
	{ UTF32_NFD(0xFABA), { 0x8AF8 } },
	{ UTF32_NFD(0xFABB), { 0x8ACB } },
	{ UTF32_NFD(0xFABC), { 0x8B01 } },
	{ UTF32_NFD(0xFABD), { 0x8AFE } },
	{ UTF32_NFD(0xFABE), { 0x8AED } },
	{ UTF32_NFD(0xFABF), { 0x8B39 } },
	{ UTF32_NFD(0xFAC0), { 0x8B8A } },
	{ UTF32_NFD(0xFAC1), { 0x8D08 } },
	{ UTF32_NFD(0xFAC2), { 0x8F38 } },
	{ UTF32_NFD(0xFAC3), { 0x9072 } },
	{ UTF32_NFD(0xFAC4), { 0x9199 } },
	{ UTF32_NFD(0xFAC5), { 0x9276 } },
	{ UTF32_NFD(0xFAC6), { 0x967C } },
	{ UTF32_NFD(0xFAC7), { 0x96E3 } },
	{ UTF32_NFD(0xFAC8), { 0x9756 } },
	{ UTF32_NFD(0xFAC9), { 0x97DB } },
	{ UTF32_NFD(0xFACA), { 0x97FF } },
	{ UTF32_NFD(0xFACB), { 0x980B } },
	{ UTF32_NFD(0xFACC), { 0x983B } },
	{ UTF32_NFD(0xFACD), { 0x9B12 } },
	{ UTF32_NFD(0xFACE), { 0x9F9C } },
	{ UTF32_NFD(0xFACF), { 0x2284A } },
	{ UTF32_NFD(0xFAD0), { 0x22844 } },
	{ UTF32_NFD(0xFAD1), { 0x233D5 } },
	{ UTF32_NFD(0xFAD2), { 0x3B9D } },
	{ UTF32_NFD(0xFAD3), { 0x4018 } },
	{ UTF32_NFD(0xFAD4), { 0x4039 } },
	{ UTF32_NFD(0xFAD5), { 0x25249 } },
	{ UTF32_NFD(0xFAD6), { 0x25CD0 } },
	{ UTF32_NFD(0xFAD7), { 0x27ED3 } },
	{ UTF32_NFD(0xFAD8), { 0x9F43 } },
	{ UTF32_NFD(0xFAD9), { 0x9F8E } },
	{ UTF32_NFKD(0xFB00), { 0x0066, 0x0066 } },
	{ UTF32_NFKD(0xFB01), { 0x0066, 0x0069 } },
	{ UTF32_NFKD(0xFB02), { 0x0066, 0x006C } },
	{ UTF32_NFKD(0xFB03), { 0x0066, 0x0066, 0x0069 } },
	{ UTF32_NFKD(0xFB04), { 0x0066, 0x0066, 0x006C } },
	{ UTF32_NFKD(0xFB05), { 0x017F, 0x0074 } },
	{ UTF32_NFKD(0xFB06), { 0x0073, 0x0074 } },
	{ UTF32_NFKD(0xFB13), { 0x0574, 0x0576 } },
	{ UTF32_NFKD(0xFB14), { 0x0574, 0x0565 } },
	{ UTF32_NFKD(0xFB15), { 0x0574, 0x056B } },
	{ UTF32_NFKD(0xFB16), { 0x057E, 0x0576 } },
	{ UTF32_NFKD(0xFB17), { 0x0574, 0x056D } },
	{ UTF32_NFD(0xFB1D), { 0x05D9, 0x05B4 } },
	{ UTF32_NFD(0xFB1F), { 0x05F2, 0x05B7 } },
	{ UTF32_NFKD(0xFB20), { 0x05E2 } },
	{ UTF32_NFKD(0xFB21), { 0x05D0 } },
	{ UTF32_NFKD(0xFB22), { 0x05D3 } },
	{ UTF32_NFKD(0xFB23), { 0x05D4 } },
	{ UTF32_NFKD(0xFB24), { 0x05DB } },
	{ UTF32_NFKD(0xFB25), { 0x05DC } },
	{ UTF32_NFKD(0xFB26), { 0x05DD } },
	{ UTF32_NFKD(0xFB27), { 0x05E8 } },
	{ UTF32_NFKD(0xFB28), { 0x05EA } },
	{ UTF32_NFKD(0xFB29), { 0x002B } },
	{ UTF32_NFD(0xFB2A), { 0x05E9, 0x05C1 } },
	{ UTF32_NFD(0xFB2B), { 0x05E9, 0x05C2 } },
	{ UTF32_NFD(0xFB2C), { 0xFB49, 0x05C1 } },
	{ UTF32_NFD(0xFB2D), { 0xFB49, 0x05C2 } },
	{ UTF32_NFD(0xFB2E), { 0x05D0, 0x05B7 } },
	{ UTF32_NFD(0xFB2F), { 0x05D0, 0x05B8 } },
	{ UTF32_NFD(0xFB30), { 0x05D0, 0x05BC } },
	{ UTF32_NFD(0xFB31), { 0x05D1, 0x05BC } },
	{ UTF32_NFD(0xFB32), { 0x05D2, 0x05BC } },
	{ UTF32_NFD(0xFB33), { 0x05D3, 0x05BC } },
	{ UTF32_NFD(0xFB34), { 0x05D4, 0x05BC } },
	{ UTF32_NFD(0xFB35), { 0x05D5, 0x05BC } },
	{ UTF32_NFD(0xFB36), { 0x05D6, 0x05BC } },
	{ UTF32_NFD(0xFB38), { 0x05D8, 0x05BC } },
	{ UTF32_NFD(0xFB39), { 0x05D9, 0x05BC } },
	{ UTF32_NFD(0xFB3A), { 0x05DA, 0x05BC } },
	{ UTF32_NFD(0xFB3B), { 0x05DB, 0x05BC } },
	{ UTF32_NFD(0xFB3C), { 0x05DC, 0x05BC } },
	{ UTF32_NFD(0xFB3E), { 0x05DE, 0x05BC } },
	{ UTF32_NFD(0xFB40), { 0x05E0, 0x05BC } },
	{ UTF32_NFD(0xFB41), { 0x05E1, 0x05BC } },
	{ UTF32_NFD(0xFB43), { 0x05E3, 0x05BC } },
	{ UTF32_NFD(0xFB44), { 0x05E4, 0x05BC } },
	{ UTF32_NFD(0xFB46), { 0x05E6, 0x05BC } },
	{ UTF32_NFD(0xFB47), { 0x05E7, 0x05BC } },
	{ UTF32_NFD(0xFB48), { 0x05E8, 0x05BC } },
	{ UTF32_NFD(0xFB49), { 0x05E9, 0x05BC } },
	{ UTF32_NFD(0xFB4A), { 0x05EA, 0x05BC } },
	{ UTF32_NFD(0xFB4B), { 0x05D5, 0x05B9 } },
	{ UTF32_NFD(0xFB4C), { 0x05D1, 0x05BF } },
	{ UTF32_NFD(0xFB4D), { 0x05DB, 0x05BF } },
	{ UTF32_NFD(0xFB4E), { 0x05E4, 0x05BF } },
	{ UTF32_NFKD(0xFB4F), { 0x05D0, 0x05DC } },
	{ UTF32_NFKD(0xFB50), { 0x0671 } },
	{ UTF32_NFKD(0xFB51), { 0x0671 } },
	{ UTF32_NFKD(0xFB52), { 0x067B } },
	{ UTF32_NFKD(0xFB53), { 0x067B } },
	{ UTF32_NFKD(0xFB54), { 0x067B } },
	{ UTF32_NFKD(0xFB55), { 0x067B } },
	{ UTF32_NFKD(0xFB56), { 0x067E } },
	{ UTF32_NFKD(0xFB57), { 0x067E } },
	{ UTF32_NFKD(0xFB58), { 0x067E } },
	{ UTF32_NFKD(0xFB59), { 0x067E } },
	{ UTF32_NFKD(0xFB5A), { 0x0680 } },
	{ UTF32_NFKD(0xFB5B), { 0x0680 } },
	{ UTF32_NFKD(0xFB5C), { 0x0680 } },
	{ UTF32_NFKD(0xFB5D), { 0x0680 } },
	{ UTF32_NFKD(0xFB5E), { 0x067A } },
	{ UTF32_NFKD(0xFB5F), { 0x067A } },
	{ UTF32_NFKD(0xFB60), { 0x067A } },
	{ UTF32_NFKD(0xFB61), { 0x067A } },
	{ UTF32_NFKD(0xFB62), { 0x067F } },
	{ UTF32_NFKD(0xFB63), { 0x067F } },
	{ UTF32_NFKD(0xFB64), { 0x067F } },
	{ UTF32_NFKD(0xFB65), { 0x067F } },
	{ UTF32_NFKD(0xFB66), { 0x0679 } },
	{ UTF32_NFKD(0xFB67), { 0x0679 } },
	{ UTF32_NFKD(0xFB68), { 0x0679 } },
	{ UTF32_NFKD(0xFB69), { 0x0679 } },
	{ UTF32_NFKD(0xFB6A), { 0x06A4 } },
	{ UTF32_NFKD(0xFB6B), { 0x06A4 } },
	{ UTF32_NFKD(0xFB6C), { 0x06A4 } },
	{ UTF32_NFKD(0xFB6D), { 0x06A4 } },
	{ UTF32_NFKD(0xFB6E), { 0x06A6 } },
	{ UTF32_NFKD(0xFB6F), { 0x06A6 } },
	{ UTF32_NFKD(0xFB70), { 0x06A6 } },
	{ UTF32_NFKD(0xFB71), { 0x06A6 } },
	{ UTF32_NFKD(0xFB72), { 0x0684 } },
	{ UTF32_NFKD(0xFB73), { 0x0684 } },
	{ UTF32_NFKD(0xFB74), { 0x0684 } },
	{ UTF32_NFKD(0xFB75), { 0x0684 } },
	{ UTF32_NFKD(0xFB76), { 0x0683 } },
	{ UTF32_NFKD(0xFB77), { 0x0683 } },
	{ UTF32_NFKD(0xFB78), { 0x0683 } },
	{ UTF32_NFKD(0xFB79), { 0x0683 } },
	{ UTF32_NFKD(0xFB7A), { 0x0686 } },
	{ UTF32_NFKD(0xFB7B), { 0x0686 } },
	{ UTF32_NFKD(0xFB7C), { 0x0686 } },
	{ UTF32_NFKD(0xFB7D), { 0x0686 } },
	{ UTF32_NFKD(0xFB7E), { 0x0687 } },
	{ UTF32_NFKD(0xFB7F), { 0x0687 } },
	{ UTF32_NFKD(0xFB80), { 0x0687 } },
	{ UTF32_NFKD(0xFB81), { 0x0687 } },
	{ UTF32_NFKD(0xFB82), { 0x068D } },
	{ UTF32_NFKD(0xFB83), { 0x068D } },
	{ UTF32_NFKD(0xFB84), { 0x068C } },
	{ UTF32_NFKD(0xFB85), { 0x068C } },
	{ UTF32_NFKD(0xFB86), { 0x068E } },
	{ UTF32_NFKD(0xFB87), { 0x068E } },
	{ UTF32_NFKD(0xFB88), { 0x0688 } },
	{ UTF32_NFKD(0xFB89), { 0x0688 } },
	{ UTF32_NFKD(0xFB8A), { 0x0698 } },
	{ UTF32_NFKD(0xFB8B), { 0x0698 } },
	{ UTF32_NFKD(0xFB8C), { 0x0691 } },
	{ UTF32_NFKD(0xFB8D), { 0x0691 } },
	{ UTF32_NFKD(0xFB8E), { 0x06A9 } },
	{ UTF32_NFKD(0xFB8F), { 0x06A9 } },
	{ UTF32_NFKD(0xFB90), { 0x06A9 } },
	{ UTF32_NFKD(0xFB91), { 0x06A9 } },
	{ UTF32_NFKD(0xFB92), { 0x06AF } },
	{ UTF32_NFKD(0xFB93), { 0x06AF } },
	{ UTF32_NFKD(0xFB94), { 0x06AF } },
	{ UTF32_NFKD(0xFB95), { 0x06AF } },
	{ UTF32_NFKD(0xFB96), { 0x06B3 } },
	{ UTF32_NFKD(0xFB97), { 0x06B3 } },
	{ UTF32_NFKD(0xFB98), { 0x06B3 } },
	{ UTF32_NFKD(0xFB99), { 0x06B3 } },
	{ UTF32_NFKD(0xFB9A), { 0x06B1 } },
	{ UTF32_NFKD(0xFB9B), { 0x06B1 } },
	{ UTF32_NFKD(0xFB9C), { 0x06B1 } },
	{ UTF32_NFKD(0xFB9D), { 0x06B1 } },
	{ UTF32_NFKD(0xFB9E), { 0x06BA } },
	{ UTF32_NFKD(0xFB9F), { 0x06BA } },
	{ UTF32_NFKD(0xFBA0), { 0x06BB } },
	{ UTF32_NFKD(0xFBA1), { 0x06BB } },
	{ UTF32_NFKD(0xFBA2), { 0x06BB } },
	{ UTF32_NFKD(0xFBA3), { 0x06BB } },
	{ UTF32_NFKD(0xFBA4), { 0x06C0 } },
	{ UTF32_NFKD(0xFBA5), { 0x06C0 } },
	{ UTF32_NFKD(0xFBA6), { 0x06C1 } },
	{ UTF32_NFKD(0xFBA7), { 0x06C1 } },
	{ UTF32_NFKD(0xFBA8), { 0x06C1 } },
	{ UTF32_NFKD(0xFBA9), { 0x06C1 } },
	{ UTF32_NFKD(0xFBAA), { 0x06BE } },
	{ UTF32_NFKD(0xFBAB), { 0x06BE } },
	{ UTF32_NFKD(0xFBAC), { 0x06BE } },
	{ UTF32_NFKD(0xFBAD), { 0x06BE } },
	{ UTF32_NFKD(0xFBAE), { 0x06D2 } },
	{ UTF32_NFKD(0xFBAF), { 0x06D2 } },
	{ UTF32_NFKD(0xFBB0), { 0x06D3 } },
	{ UTF32_NFKD(0xFBB1), { 0x06D3 } },
	{ UTF32_NFKD(0xFBD3), { 0x06AD } },
	{ UTF32_NFKD(0xFBD4), { 0x06AD } },
	{ UTF32_NFKD(0xFBD5), { 0x06AD } },
	{ UTF32_NFKD(0xFBD6), { 0x06AD } },
	{ UTF32_NFKD(0xFBD7), { 0x06C7 } },
	{ UTF32_NFKD(0xFBD8), { 0x06C7 } },
	{ UTF32_NFKD(0xFBD9), { 0x06C6 } },
	{ UTF32_NFKD(0xFBDA), { 0x06C6 } },
	{ UTF32_NFKD(0xFBDB), { 0x06C8 } },
	{ UTF32_NFKD(0xFBDC), { 0x06C8 } },
	{ UTF32_NFKD(0xFBDD), { 0x0677 } },
	{ UTF32_NFKD(0xFBDE), { 0x06CB } },
	{ UTF32_NFKD(0xFBDF), { 0x06CB } },
	{ UTF32_NFKD(0xFBE0), { 0x06C5 } },
	{ UTF32_NFKD(0xFBE1), { 0x06C5 } },
	{ UTF32_NFKD(0xFBE2), { 0x06C9 } },
	{ UTF32_NFKD(0xFBE3), { 0x06C9 } },
	{ UTF32_NFKD(0xFBE4), { 0x06D0 } },
	{ UTF32_NFKD(0xFBE5), { 0x06D0 } },
	{ UTF32_NFKD(0xFBE6), { 0x06D0 } },
	{ UTF32_NFKD(0xFBE7), { 0x06D0 } },
	{ UTF32_NFKD(0xFBE8), { 0x0649 } },
	{ UTF32_NFKD(0xFBE9), { 0x0649 } },
	{ UTF32_NFKD(0xFBEA), { 0x0626, 0x0627 } },
	{ UTF32_NFKD(0xFBEB), { 0x0626, 0x0627 } },
	{ UTF32_NFKD(0xFBEC), { 0x0626, 0x06D5 } },
	{ UTF32_NFKD(0xFBED), { 0x0626, 0x06D5 } },
	{ UTF32_NFKD(0xFBEE), { 0x0626, 0x0648 } },
	{ UTF32_NFKD(0xFBEF), { 0x0626, 0x0648 } },
	{ UTF32_NFKD(0xFBF0), { 0x0626, 0x06C7 } },
	{ UTF32_NFKD(0xFBF1), { 0x0626, 0x06C7 } },
	{ UTF32_NFKD(0xFBF2), { 0x0626, 0x06C6 } },
	{ UTF32_NFKD(0xFBF3), { 0x0626, 0x06C6 } },
	{ UTF32_NFKD(0xFBF4), { 0x0626, 0x06C8 } },
	{ UTF32_NFKD(0xFBF5), { 0x0626, 0x06C8 } },
	{ UTF32_NFKD(0xFBF6), { 0x0626, 0x06D0 } },
	{ UTF32_NFKD(0xFBF7), { 0x0626, 0x06D0 } },
	{ UTF32_NFKD(0xFBF8), { 0x0626, 0x06D0 } },
	{ UTF32_NFKD(0xFBF9), { 0x0626, 0x0649 } },
	{ UTF32_NFKD(0xFBFA), { 0x0626, 0x0649 } },
	{ UTF32_NFKD(0xFBFB), { 0x0626, 0x0649 } },
	{ UTF32_NFKD(0xFBFC), { 0x06CC } },
	{ UTF32_NFKD(0xFBFD), { 0x06CC } },
	{ UTF32_NFKD(0xFBFE), { 0x06CC } },
	{ UTF32_NFKD(0xFBFF), { 0x06CC } },
	{ UTF32_NFKD(0xFC00), { 0x0626, 0x062C } },
	{ UTF32_NFKD(0xFC01), { 0x0626, 0x062D } },
	{ UTF32_NFKD(0xFC02), { 0x0626, 0x0645 } },
	{ UTF32_NFKD(0xFC03), { 0x0626, 0x0649 } },
	{ UTF32_NFKD(0xFC04), { 0x0626, 0x064A } },
	{ UTF32_NFKD(0xFC05), { 0x0628, 0x062C } },
	{ UTF32_NFKD(0xFC06), { 0x0628, 0x062D } },
	{ UTF32_NFKD(0xFC07), { 0x0628, 0x062E } },
	{ UTF32_NFKD(0xFC08), { 0x0628, 0x0645 } },
	{ UTF32_NFKD(0xFC09), { 0x0628, 0x0649 } },
	{ UTF32_NFKD(0xFC0A), { 0x0628, 0x064A } },
	{ UTF32_NFKD(0xFC0B), { 0x062A, 0x062C } },
	{ UTF32_NFKD(0xFC0C), { 0x062A, 0x062D } },
	{ UTF32_NFKD(0xFC0D), { 0x062A, 0x062E } },
	{ UTF32_NFKD(0xFC0E), { 0x062A, 0x0645 } },
	{ UTF32_NFKD(0xFC0F), { 0x062A, 0x0649 } },
	{ UTF32_NFKD(0xFC10), { 0x062A, 0x064A } },
	{ UTF32_NFKD(0xFC11), { 0x062B, 0x062C } },
	{ UTF32_NFKD(0xFC12), { 0x062B, 0x0645 } },
	{ UTF32_NFKD(0xFC13), { 0x062B, 0x0649 } },
	{ UTF32_NFKD(0xFC14), { 0x062B, 0x064A } },
	{ UTF32_NFKD(0xFC15), { 0x062C, 0x062D } },
	{ UTF32_NFKD(0xFC16), { 0x062C, 0x0645 } },
	{ UTF32_NFKD(0xFC17), { 0x062D, 0x062C } },
	{ UTF32_NFKD(0xFC18), { 0x062D, 0x0645 } },
	{ UTF32_NFKD(0xFC19), { 0x062E, 0x062C } },
	{ UTF32_NFKD(0xFC1A), { 0x062E, 0x062D } },
	{ UTF32_NFKD(0xFC1B), { 0x062E, 0x0645 } },
	{ UTF32_NFKD(0xFC1C), { 0x0633, 0x062C } },
	{ UTF32_NFKD(0xFC1D), { 0x0633, 0x062D } },
	{ UTF32_NFKD(0xFC1E), { 0x0633, 0x062E } },
	{ UTF32_NFKD(0xFC1F), { 0x0633, 0x0645 } },
	{ UTF32_NFKD(0xFC20), { 0x0635, 0x062D } },
	{ UTF32_NFKD(0xFC21), { 0x0635, 0x0645 } },
	{ UTF32_NFKD(0xFC22), { 0x0636, 0x062C } },
	{ UTF32_NFKD(0xFC23), { 0x0636, 0x062D } },
	{ UTF32_NFKD(0xFC24), { 0x0636, 0x062E } },
	{ UTF32_NFKD(0xFC25), { 0x0636, 0x0645 } },
	{ UTF32_NFKD(0xFC26), { 0x0637, 0x062D } },
	{ UTF32_NFKD(0xFC27), { 0x0637, 0x0645 } },
	{ UTF32_NFKD(0xFC28), { 0x0638, 0x0645 } },
	{ UTF32_NFKD(0xFC29), { 0x0639, 0x062C } },
	{ UTF32_NFKD(0xFC2A), { 0x0639, 0x0645 } },
	{ UTF32_NFKD(0xFC2B), { 0x063A, 0x062C } },
	{ UTF32_NFKD(0xFC2C), { 0x063A, 0x0645 } },
	{ UTF32_NFKD(0xFC2D), { 0x0641, 0x062C } },
	{ UTF32_NFKD(0xFC2E), { 0x0641, 0x062D } },
	{ UTF32_NFKD(0xFC2F), { 0x0641, 0x062E } },
	{ UTF32_NFKD(0xFC30), { 0x0641, 0x0645 } },
	{ UTF32_NFKD(0xFC31), { 0x0641, 0x0649 } },
	{ UTF32_NFKD(0xFC32), { 0x0641, 0x064A } },
	{ UTF32_NFKD(0xFC33), { 0x0642, 0x062D } },
	{ UTF32_NFKD(0xFC34), { 0x0642, 0x0645 } },
	{ UTF32_NFKD(0xFC35), { 0x0642, 0x0649 } },
	{ UTF32_NFKD(0xFC36), { 0x0642, 0x064A } },
	{ UTF32_NFKD(0xFC37), { 0x0643, 0x0627 } },
	{ UTF32_NFKD(0xFC38), { 0x0643, 0x062C } },
	{ UTF32_NFKD(0xFC39), { 0x0643, 0x062D } },
	{ UTF32_NFKD(0xFC3A), { 0x0643, 0x062E } },
	{ UTF32_NFKD(0xFC3B), { 0x0643, 0x0644 } },
	{ UTF32_NFKD(0xFC3C), { 0x0643, 0x0645 } },
	{ UTF32_NFKD(0xFC3D), { 0x0643, 0x0649 } },
	{ UTF32_NFKD(0xFC3E), { 0x0643, 0x064A } },
	{ UTF32_NFKD(0xFC3F), { 0x0644, 0x062C } },
	{ UTF32_NFKD(0xFC40), { 0x0644, 0x062D } },
	{ UTF32_NFKD(0xFC41), { 0x0644, 0x062E } },
	{ UTF32_NFKD(0xFC42), { 0x0644, 0x0645 } },
	{ UTF32_NFKD(0xFC43), { 0x0644, 0x0649 } },
	{ UTF32_NFKD(0xFC44), { 0x0644, 0x064A } },
	{ UTF32_NFKD(0xFC45), { 0x0645, 0x062C } },
	{ UTF32_NFKD(0xFC46), { 0x0645, 0x062D } },
	{ UTF32_NFKD(0xFC47), { 0x0645, 0x062E } },
	{ UTF32_NFKD(0xFC48), { 0x0645, 0x0645 } },
	{ UTF32_NFKD(0xFC49), { 0x0645, 0x0649 } },
	{ UTF32_NFKD(0xFC4A), { 0x0645, 0x064A } },
	{ UTF32_NFKD(0xFC4B), { 0x0646, 0x062C } },
	{ UTF32_NFKD(0xFC4C), { 0x0646, 0x062D } },
	{ UTF32_NFKD(0xFC4D), { 0x0646, 0x062E } },
	{ UTF32_NFKD(0xFC4E), { 0x0646, 0x0645 } },
	{ UTF32_NFKD(0xFC4F), { 0x0646, 0x0649 } },
	{ UTF32_NFKD(0xFC50), { 0x0646, 0x064A } },
	{ UTF32_NFKD(0xFC51), { 0x0647, 0x062C } },
	{ UTF32_NFKD(0xFC52), { 0x0647, 0x0645 } },
	{ UTF32_NFKD(0xFC53), { 0x0647, 0x0649 } },
	{ UTF32_NFKD(0xFC54), { 0x0647, 0x064A } },
	{ UTF32_NFKD(0xFC55), { 0x064A, 0x062C } },
	{ UTF32_NFKD(0xFC56), { 0x064A, 0x062D } },
	{ UTF32_NFKD(0xFC57), { 0x064A, 0x062E } },
	{ UTF32_NFKD(0xFC58), { 0x064A, 0x0645 } },
	{ UTF32_NFKD(0xFC59), { 0x064A, 0x0649 } },
	{ UTF32_NFKD(0xFC5A), { 0x064A, 0x064A } },
	{ UTF32_NFKD(0xFC5B), { 0x0630, 0x0670 } },
	{ UTF32_NFKD(0xFC5C), { 0x0631, 0x0670 } },
	{ UTF32_NFKD(0xFC5D), { 0x0649, 0x0670 } },
	{ UTF32_NFKD(0xFC5E), { 0x0020, 0x064C, 0x0651 } },
	{ UTF32_NFKD(0xFC5F), { 0x0020, 0x064D, 0x0651 } },
	{ UTF32_NFKD(0xFC60), { 0x0020, 0x064E, 0x0651 } },
	{ UTF32_NFKD(0xFC61), { 0x0020, 0x064F, 0x0651 } },
	{ UTF32_NFKD(0xFC62), { 0x0020, 0x0650, 0x0651 } },
	{ UTF32_NFKD(0xFC63), { 0x0020, 0x0651, 0x0670 } },
	{ UTF32_NFKD(0xFC64), { 0x0626, 0x0631 } },
	{ UTF32_NFKD(0xFC65), { 0x0626, 0x0632 } },
	{ UTF32_NFKD(0xFC66), { 0x0626, 0x0645 } },
	{ UTF32_NFKD(0xFC67), { 0x0626, 0x0646 } },
	{ UTF32_NFKD(0xFC68), { 0x0626, 0x0649 } },
	{ UTF32_NFKD(0xFC69), { 0x0626, 0x064A } },
	{ UTF32_NFKD(0xFC6A), { 0x0628, 0x0631 } },
	{ UTF32_NFKD(0xFC6B), { 0x0628, 0x0632 } },
	{ UTF32_NFKD(0xFC6C), { 0x0628, 0x0645 } },
	{ UTF32_NFKD(0xFC6D), { 0x0628, 0x0646 } },
	{ UTF32_NFKD(0xFC6E), { 0x0628, 0x0649 } },
	{ UTF32_NFKD(0xFC6F), { 0x0628, 0x064A } },
	{ UTF32_NFKD(0xFC70), { 0x062A, 0x0631 } },
	{ UTF32_NFKD(0xFC71), { 0x062A, 0x0632 } },
	{ UTF32_NFKD(0xFC72), { 0x062A, 0x0645 } },
	{ UTF32_NFKD(0xFC73), { 0x062A, 0x0646 } },
	{ UTF32_NFKD(0xFC74), { 0x062A, 0x0649 } },
	{ UTF32_NFKD(0xFC75), { 0x062A, 0x064A } },
	{ UTF32_NFKD(0xFC76), { 0x062B, 0x0631 } },
	{ UTF32_NFKD(0xFC77), { 0x062B, 0x0632 } },
	{ UTF32_NFKD(0xFC78), { 0x062B, 0x0645 } },
	{ UTF32_NFKD(0xFC79), { 0x062B, 0x0646 } },
	{ UTF32_NFKD(0xFC7A), { 0x062B, 0x0649 } },
	{ UTF32_NFKD(0xFC7B), { 0x062B, 0x064A } },
	{ UTF32_NFKD(0xFC7C), { 0x0641, 0x0649 } },
	{ UTF32_NFKD(0xFC7D), { 0x0641, 0x064A } },
	{ UTF32_NFKD(0xFC7E), { 0x0642, 0x0649 } },
	{ UTF32_NFKD(0xFC7F), { 0x0642, 0x064A } },
	{ UTF32_NFKD(0xFC80), { 0x0643, 0x0627 } },
	{ UTF32_NFKD(0xFC81), { 0x0643, 0x0644 } },
	{ UTF32_NFKD(0xFC82), { 0x0643, 0x0645 } },
	{ UTF32_NFKD(0xFC83), { 0x0643, 0x0649 } },
	{ UTF32_NFKD(0xFC84), { 0x0643, 0x064A } },
	{ UTF32_NFKD(0xFC85), { 0x0644, 0x0645 } },
	{ UTF32_NFKD(0xFC86), { 0x0644, 0x0649 } },
	{ UTF32_NFKD(0xFC87), { 0x0644, 0x064A } },
	{ UTF32_NFKD(0xFC88), { 0x0645, 0x0627 } },
	{ UTF32_NFKD(0xFC89), { 0x0645, 0x0645 } },
	{ UTF32_NFKD(0xFC8A), { 0x0646, 0x0631 } },
	{ UTF32_NFKD(0xFC8B), { 0x0646, 0x0632 } },
	{ UTF32_NFKD(0xFC8C), { 0x0646, 0x0645 } },
	{ UTF32_NFKD(0xFC8D), { 0x0646, 0x0646 } },
	{ UTF32_NFKD(0xFC8E), { 0x0646, 0x0649 } },
	{ UTF32_NFKD(0xFC8F), { 0x0646, 0x064A } },
	{ UTF32_NFKD(0xFC90), { 0x0649, 0x0670 } },
	{ UTF32_NFKD(0xFC91), { 0x064A, 0x0631 } },
	{ UTF32_NFKD(0xFC92), { 0x064A, 0x0632 } },
	{ UTF32_NFKD(0xFC93), { 0x064A, 0x0645 } },
	{ UTF32_NFKD(0xFC94), { 0x064A, 0x0646 } },
	{ UTF32_NFKD(0xFC95), { 0x064A, 0x0649 } },
	{ UTF32_NFKD(0xFC96), { 0x064A, 0x064A } },
	{ UTF32_NFKD(0xFC97), { 0x0626, 0x062C } },
	{ UTF32_NFKD(0xFC98), { 0x0626, 0x062D } },
	{ UTF32_NFKD(0xFC99), { 0x0626, 0x062E } },
	{ UTF32_NFKD(0xFC9A), { 0x0626, 0x0645 } },
	{ UTF32_NFKD(0xFC9B), { 0x0626, 0x0647 } },
	{ UTF32_NFKD(0xFC9C), { 0x0628, 0x062C } },
	{ UTF32_NFKD(0xFC9D), { 0x0628, 0x062D } },
	{ UTF32_NFKD(0xFC9E), { 0x0628, 0x062E } },
	{ UTF32_NFKD(0xFC9F), { 0x0628, 0x0645 } },
	{ UTF32_NFKD(0xFCA0), { 0x0628, 0x0647 } },
	{ UTF32_NFKD(0xFCA1), { 0x062A, 0x062C } },
	{ UTF32_NFKD(0xFCA2), { 0x062A, 0x062D } },
	{ UTF32_NFKD(0xFCA3), { 0x062A, 0x062E } },
	{ UTF32_NFKD(0xFCA4), { 0x062A, 0x0645 } },
	{ UTF32_NFKD(0xFCA5), { 0x062A, 0x0647 } },
	{ UTF32_NFKD(0xFCA6), { 0x062B, 0x0645 } },
	{ UTF32_NFKD(0xFCA7), { 0x062C, 0x062D } },
	{ UTF32_NFKD(0xFCA8), { 0x062C, 0x0645 } },
	{ UTF32_NFKD(0xFCA9), { 0x062D, 0x062C } },
	{ UTF32_NFKD(0xFCAA), { 0x062D, 0x0645 } },
	{ UTF32_NFKD(0xFCAB), { 0x062E, 0x062C } },
	{ UTF32_NFKD(0xFCAC), { 0x062E, 0x0645 } },
	{ UTF32_NFKD(0xFCAD), { 0x0633, 0x062C } },
	{ UTF32_NFKD(0xFCAE), { 0x0633, 0x062D } },
	{ UTF32_NFKD(0xFCAF), { 0x0633, 0x062E } },
	{ UTF32_NFKD(0xFCB0), { 0x0633, 0x0645 } },
	{ UTF32_NFKD(0xFCB1), { 0x0635, 0x062D } },
	{ UTF32_NFKD(0xFCB2), { 0x0635, 0x062E } },
	{ UTF32_NFKD(0xFCB3), { 0x0635, 0x0645 } },
	{ UTF32_NFKD(0xFCB4), { 0x0636, 0x062C } },
	{ UTF32_NFKD(0xFCB5), { 0x0636, 0x062D } },
	{ UTF32_NFKD(0xFCB6), { 0x0636, 0x062E } },
	{ UTF32_NFKD(0xFCB7), { 0x0636, 0x0645 } },
	{ UTF32_NFKD(0xFCB8), { 0x0637, 0x062D } },
	{ UTF32_NFKD(0xFCB9), { 0x0638, 0x0645 } },
	{ UTF32_NFKD(0xFCBA), { 0x0639, 0x062C } },
	{ UTF32_NFKD(0xFCBB), { 0x0639, 0x0645 } },
	{ UTF32_NFKD(0xFCBC), { 0x063A, 0x062C } },
	{ UTF32_NFKD(0xFCBD), { 0x063A, 0x0645 } },
	{ UTF32_NFKD(0xFCBE), { 0x0641, 0x062C } },
	{ UTF32_NFKD(0xFCBF), { 0x0641, 0x062D } },
	{ UTF32_NFKD(0xFCC0), { 0x0641, 0x062E } },
	{ UTF32_NFKD(0xFCC1), { 0x0641, 0x0645 } },
	{ UTF32_NFKD(0xFCC2), { 0x0642, 0x062D } },
	{ UTF32_NFKD(0xFCC3), { 0x0642, 0x0645 } },
	{ UTF32_NFKD(0xFCC4), { 0x0643, 0x062C } },
	{ UTF32_NFKD(0xFCC5), { 0x0643, 0x062D } },
	{ UTF32_NFKD(0xFCC6), { 0x0643, 0x062E } },
	{ UTF32_NFKD(0xFCC7), { 0x0643, 0x0644 } },
	{ UTF32_NFKD(0xFCC8), { 0x0643, 0x0645 } },
	{ UTF32_NFKD(0xFCC9), { 0x0644, 0x062C } },
	{ UTF32_NFKD(0xFCCA), { 0x0644, 0x062D } },
	{ UTF32_NFKD(0xFCCB), { 0x0644, 0x062E } },
	{ UTF32_NFKD(0xFCCC), { 0x0644, 0x0645 } },
	{ UTF32_NFKD(0xFCCD), { 0x0644, 0x0647 } },
	{ UTF32_NFKD(0xFCCE), { 0x0645, 0x062C } },
	{ UTF32_NFKD(0xFCCF), { 0x0645, 0x062D } },
	{ UTF32_NFKD(0xFCD0), { 0x0645, 0x062E } },
	{ UTF32_NFKD(0xFCD1), { 0x0645, 0x0645 } },
	{ UTF32_NFKD(0xFCD2), { 0x0646, 0x062C } },
	{ UTF32_NFKD(0xFCD3), { 0x0646, 0x062D } },
	{ UTF32_NFKD(0xFCD4), { 0x0646, 0x062E } },
	{ UTF32_NFKD(0xFCD5), { 0x0646, 0x0645 } },
	{ UTF32_NFKD(0xFCD6), { 0x0646, 0x0647 } },
	{ UTF32_NFKD(0xFCD7), { 0x0647, 0x062C } },
	{ UTF32_NFKD(0xFCD8), { 0x0647, 0x0645 } },
	{ UTF32_NFKD(0xFCD9), { 0x0647, 0x0670 } },
	{ UTF32_NFKD(0xFCDA), { 0x064A, 0x062C } },
	{ UTF32_NFKD(0xFCDB), { 0x064A, 0x062D } },
	{ UTF32_NFKD(0xFCDC), { 0x064A, 0x062E } },
	{ UTF32_NFKD(0xFCDD), { 0x064A, 0x0645 } },
	{ UTF32_NFKD(0xFCDE), { 0x064A, 0x0647 } },
	{ UTF32_NFKD(0xFCDF), { 0x0626, 0x0645 } },
	{ UTF32_NFKD(0xFCE0), { 0x0626, 0x0647 } },
	{ UTF32_NFKD(0xFCE1), { 0x0628, 0x0645 } },
	{ UTF32_NFKD(0xFCE2), { 0x0628, 0x0647 } },
	{ UTF32_NFKD(0xFCE3), { 0x062A, 0x0645 } },
	{ UTF32_NFKD(0xFCE4), { 0x062A, 0x0647 } },
	{ UTF32_NFKD(0xFCE5), { 0x062B, 0x0645 } },
	{ UTF32_NFKD(0xFCE6), { 0x062B, 0x0647 } },
	{ UTF32_NFKD(0xFCE7), { 0x0633, 0x0645 } },
	{ UTF32_NFKD(0xFCE8), { 0x0633, 0x0647 } },
	{ UTF32_NFKD(0xFCE9), { 0x0634, 0x0645 } },
	{ UTF32_NFKD(0xFCEA), { 0x0634, 0x0647 } },
	{ UTF32_NFKD(0xFCEB), { 0x0643, 0x0644 } },
	{ UTF32_NFKD(0xFCEC), { 0x0643, 0x0645 } },
	{ UTF32_NFKD(0xFCED), { 0x0644, 0x0645 } },
	{ UTF32_NFKD(0xFCEE), { 0x0646, 0x0645 } },
	{ UTF32_NFKD(0xFCEF), { 0x0646, 0x0647 } },
	{ UTF32_NFKD(0xFCF0), { 0x064A, 0x0645 } },
	{ UTF32_NFKD(0xFCF1), { 0x064A, 0x0647 } },
	{ UTF32_NFKD(0xFCF2), { 0x0640, 0x064E, 0x0651 } },
	{ UTF32_NFKD(0xFCF3), { 0x0640, 0x064F, 0x0651 } },
	{ UTF32_NFKD(0xFCF4), { 0x0640, 0x0650, 0x0651 } },
	{ UTF32_NFKD(0xFCF5), { 0x0637, 0x0649 } },
	{ UTF32_NFKD(0xFCF6), { 0x0637, 0x064A } },
	{ UTF32_NFKD(0xFCF7), { 0x0639, 0x0649 } },
	{ UTF32_NFKD(0xFCF8), { 0x0639, 0x064A } },
	{ UTF32_NFKD(0xFCF9), { 0x063A, 0x0649 } },
	{ UTF32_NFKD(0xFCFA), { 0x063A, 0x064A } },
	{ UTF32_NFKD(0xFCFB), { 0x0633, 0x0649 } },
	{ UTF32_NFKD(0xFCFC), { 0x0633, 0x064A } },
	{ UTF32_NFKD(0xFCFD), { 0x0634, 0x0649 } },
	{ UTF32_NFKD(0xFCFE), { 0x0634, 0x064A } },
	{ UTF32_NFKD(0xFCFF), { 0x062D, 0x0649 } },
	{ UTF32_NFKD(0xFD00), { 0x062D, 0x064A } },
	{ UTF32_NFKD(0xFD01), { 0x062C, 0x0649 } },
	{ UTF32_NFKD(0xFD02), { 0x062C, 0x064A } },
	{ UTF32_NFKD(0xFD03), { 0x062E, 0x0649 } },
	{ UTF32_NFKD(0xFD04), { 0x062E, 0x064A } },
	{ UTF32_NFKD(0xFD05), { 0x0635, 0x0649 } },
	{ UTF32_NFKD(0xFD06), { 0x0635, 0x064A } },
	{ UTF32_NFKD(0xFD07), { 0x0636, 0x0649 } },
	{ UTF32_NFKD(0xFD08), { 0x0636, 0x064A } },
	{ UTF32_NFKD(0xFD09), { 0x0634, 0x062C } },
	{ UTF32_NFKD(0xFD0A), { 0x0634, 0x062D } },
	{ UTF32_NFKD(0xFD0B), { 0x0634, 0x062E } },
	{ UTF32_NFKD(0xFD0C), { 0x0634, 0x0645 } },
	{ UTF32_NFKD(0xFD0D), { 0x0634, 0x0631 } },
	{ UTF32_NFKD(0xFD0E), { 0x0633, 0x0631 } },
	{ UTF32_NFKD(0xFD0F), { 0x0635, 0x0631 } },
	{ UTF32_NFKD(0xFD10), { 0x0636, 0x0631 } },
	{ UTF32_NFKD(0xFD11), { 0x0637, 0x0649 } },
	{ UTF32_NFKD(0xFD12), { 0x0637, 0x064A } },
	{ UTF32_NFKD(0xFD13), { 0x0639, 0x0649 } },
	{ UTF32_NFKD(0xFD14), { 0x0639, 0x064A } },
	{ UTF32_NFKD(0xFD15), { 0x063A, 0x0649 } },
	{ UTF32_NFKD(0xFD16), { 0x063A, 0x064A } },
	{ UTF32_NFKD(0xFD17), { 0x0633, 0x0649 } },
	{ UTF32_NFKD(0xFD18), { 0x0633, 0x064A } },
	{ UTF32_NFKD(0xFD19), { 0x0634, 0x0649 } },
	{ UTF32_NFKD(0xFD1A), { 0x0634, 0x064A } },
	{ UTF32_NFKD(0xFD1B), { 0x062D, 0x0649 } },
	{ UTF32_NFKD(0xFD1C), { 0x062D, 0x064A } },
	{ UTF32_NFKD(0xFD1D), { 0x062C, 0x0649 } },
	{ UTF32_NFKD(0xFD1E), { 0x062C, 0x064A } },
	{ UTF32_NFKD(0xFD1F), { 0x062E, 0x0649 } },
	{ UTF32_NFKD(0xFD20), { 0x062E, 0x064A } },
	{ UTF32_NFKD(0xFD21), { 0x0635, 0x0649 } },
	{ UTF32_NFKD(0xFD22), { 0x0635, 0x064A } },
	{ UTF32_NFKD(0xFD23), { 0x0636, 0x0649 } },
	{ UTF32_NFKD(0xFD24), { 0x0636, 0x064A } },
	{ UTF32_NFKD(0xFD25), { 0x0634, 0x062C } },
	{ UTF32_NFKD(0xFD26), { 0x0634, 0x062D } },
	{ UTF32_NFKD(0xFD27), { 0x0634, 0x062E } },
	{ UTF32_NFKD(0xFD28), { 0x0634, 0x0645 } },
	{ UTF32_NFKD(0xFD29), { 0x0634, 0x0631 } },
	{ UTF32_NFKD(0xFD2A), { 0x0633, 0x0631 } },
	{ UTF32_NFKD(0xFD2B), { 0x0635, 0x0631 } },
	{ UTF32_NFKD(0xFD2C), { 0x0636, 0x0631 } },
	{ UTF32_NFKD(0xFD2D), { 0x0634, 0x062C } },
	{ UTF32_NFKD(0xFD2E), { 0x0634, 0x062D } },
	{ UTF32_NFKD(0xFD2F), { 0x0634, 0x062E } },
	{ UTF32_NFKD(0xFD30), { 0x0634, 0x0645 } },
	{ UTF32_NFKD(0xFD31), { 0x0633, 0x0647 } },
	{ UTF32_NFKD(0xFD32), { 0x0634, 0x0647 } },
	{ UTF32_NFKD(0xFD33), { 0x0637, 0x0645 } },
	{ UTF32_NFKD(0xFD34), { 0x0633, 0x062C } },
	{ UTF32_NFKD(0xFD35), { 0x0633, 0x062D } },
	{ UTF32_NFKD(0xFD36), { 0x0633, 0x062E } },
	{ UTF32_NFKD(0xFD37), { 0x0634, 0x062C } },
	{ UTF32_NFKD(0xFD38), { 0x0634, 0x062D } },
	{ UTF32_NFKD(0xFD39), { 0x0634, 0x062E } },
	{ UTF32_NFKD(0xFD3A), { 0x0637, 0x0645 } },
	{ UTF32_NFKD(0xFD3B), { 0x0638, 0x0645 } },
	{ UTF32_NFKD(0xFD3C), { 0x0627, 0x064B } },
	{ UTF32_NFKD(0xFD3D), { 0x0627, 0x064B } },
	{ UTF32_NFKD(0xFD50), { 0x062A, 0x062C, 0x0645 } },
	{ UTF32_NFKD(0xFD51), { 0x062A, 0x062D, 0x062C } },
	{ UTF32_NFKD(0xFD52), { 0x062A, 0x062D, 0x062C } },
	{ UTF32_NFKD(0xFD53), { 0x062A, 0x062D, 0x0645 } },
	{ UTF32_NFKD(0xFD54), { 0x062A, 0x062E, 0x0645 } },
	{ UTF32_NFKD(0xFD55), { 0x062A, 0x0645, 0x062C } },
	{ UTF32_NFKD(0xFD56), { 0x062A, 0x0645, 0x062D } },
	{ UTF32_NFKD(0xFD57), { 0x062A, 0x0645, 0x062E } },
	{ UTF32_NFKD(0xFD58), { 0x062C, 0x0645, 0x062D } },
	{ UTF32_NFKD(0xFD59), { 0x062C, 0x0645, 0x062D } },
	{ UTF32_NFKD(0xFD5A), { 0x062D, 0x0645, 0x064A } },
	{ UTF32_NFKD(0xFD5B), { 0x062D, 0x0645, 0x0649 } },
	{ UTF32_NFKD(0xFD5C), { 0x0633, 0x062D, 0x062C } },
	{ UTF32_NFKD(0xFD5D), { 0x0633, 0x062C, 0x062D } },
	{ UTF32_NFKD(0xFD5E), { 0x0633, 0x062C, 0x0649 } },
	{ UTF32_NFKD(0xFD5F), { 0x0633, 0x0645, 0x062D } },
	{ UTF32_NFKD(0xFD60), { 0x0633, 0x0645, 0x062D } },
	{ UTF32_NFKD(0xFD61), { 0x0633, 0x0645, 0x062C } },
	{ UTF32_NFKD(0xFD62), { 0x0633, 0x0645, 0x0645 } },
	{ UTF32_NFKD(0xFD63), { 0x0633, 0x0645, 0x0645 } },
	{ UTF32_NFKD(0xFD64), { 0x0635, 0x062D, 0x062D } },
	{ UTF32_NFKD(0xFD65), { 0x0635, 0x062D, 0x062D } },
	{ UTF32_NFKD(0xFD66), { 0x0635, 0x0645, 0x0645 } },
	{ UTF32_NFKD(0xFD67), { 0x0634, 0x062D, 0x0645 } },
	{ UTF32_NFKD(0xFD68), { 0x0634, 0x062D, 0x0645 } },
	{ UTF32_NFKD(0xFD69), { 0x0634, 0x062C, 0x064A } },
	{ UTF32_NFKD(0xFD6A), { 0x0634, 0x0645, 0x062E } },
	{ UTF32_NFKD(0xFD6B), { 0x0634, 0x0645, 0x062E } },
	{ UTF32_NFKD(0xFD6C), { 0x0634, 0x0645, 0x0645 } },
	{ UTF32_NFKD(0xFD6D), { 0x0634, 0x0645, 0x0645 } },
	{ UTF32_NFKD(0xFD6E), { 0x0636, 0x062D, 0x0649 } },
	{ UTF32_NFKD(0xFD6F), { 0x0636, 0x062E, 0x0645 } },
	{ UTF32_NFKD(0xFD70), { 0x0636, 0x062E, 0x0645 } },
	{ UTF32_NFKD(0xFD71), { 0x0637, 0x0645, 0x062D } },
	{ UTF32_NFKD(0xFD72), { 0x0637, 0x0645, 0x062D } },
	{ UTF32_NFKD(0xFD73), { 0x0637, 0x0645, 0x0645 } },
	{ UTF32_NFKD(0xFD74), { 0x0637, 0x0645, 0x064A } },
	{ UTF32_NFKD(0xFD75), { 0x0639, 0x062C, 0x0645 } },
	{ UTF32_NFKD(0xFD76), { 0x0639, 0x0645, 0x0645 } },
	{ UTF32_NFKD(0xFD77), { 0x0639, 0x0645, 0x0645 } },
	{ UTF32_NFKD(0xFD78), { 0x0639, 0x0645, 0x0649 } },
	{ UTF32_NFKD(0xFD79), { 0x063A, 0x0645, 0x0645 } },
	{ UTF32_NFKD(0xFD7A), { 0x063A, 0x0645, 0x064A } },
	{ UTF32_NFKD(0xFD7B), { 0x063A, 0x0645, 0x0649 } },
	{ UTF32_NFKD(0xFD7C), { 0x0641, 0x062E, 0x0645 } },
	{ UTF32_NFKD(0xFD7D), { 0x0641, 0x062E, 0x0645 } },
	{ UTF32_NFKD(0xFD7E), { 0x0642, 0x0645, 0x062D } },
	{ UTF32_NFKD(0xFD7F), { 0x0642, 0x0645, 0x0645 } },
	{ UTF32_NFKD(0xFD80), { 0x0644, 0x062D, 0x0645 } },
	{ UTF32_NFKD(0xFD81), { 0x0644, 0x062D, 0x064A } },
	{ UTF32_NFKD(0xFD82), { 0x0644, 0x062D, 0x0649 } },
	{ UTF32_NFKD(0xFD83), { 0x0644, 0x062C, 0x062C } },
	{ UTF32_NFKD(0xFD84), { 0x0644, 0x062C, 0x062C } },
	{ UTF32_NFKD(0xFD85), { 0x0644, 0x062E, 0x0645 } },
	{ UTF32_NFKD(0xFD86), { 0x0644, 0x062E, 0x0645 } },
	{ UTF32_NFKD(0xFD87), { 0x0644, 0x0645, 0x062D } },
	{ UTF32_NFKD(0xFD88), { 0x0644, 0x0645, 0x062D } },
	{ UTF32_NFKD(0xFD89), { 0x0645, 0x062D, 0x062C } },
	{ UTF32_NFKD(0xFD8A), { 0x0645, 0x062D, 0x0645 } },
	{ UTF32_NFKD(0xFD8B), { 0x0645, 0x062D, 0x064A } },
	{ UTF32_NFKD(0xFD8C), { 0x0645, 0x062C, 0x062D } },
	{ UTF32_NFKD(0xFD8D), { 0x0645, 0x062C, 0x0645 } },
	{ UTF32_NFKD(0xFD8E), { 0x0645, 0x062E, 0x062C } },
	{ UTF32_NFKD(0xFD8F), { 0x0645, 0x062E, 0x0645 } },
	{ UTF32_NFKD(0xFD92), { 0x0645, 0x062C, 0x062E } },
	{ UTF32_NFKD(0xFD93), { 0x0647, 0x0645, 0x062C } },
	{ UTF32_NFKD(0xFD94), { 0x0647, 0x0645, 0x0645 } },
	{ UTF32_NFKD(0xFD95), { 0x0646, 0x062D, 0x0645 } },
	{ UTF32_NFKD(0xFD96), { 0x0646, 0x062D, 0x0649 } },
	{ UTF32_NFKD(0xFD97), { 0x0646, 0x062C, 0x0645 } },
	{ UTF32_NFKD(0xFD98), { 0x0646, 0x062C, 0x0645 } },
	{ UTF32_NFKD(0xFD99), { 0x0646, 0x062C, 0x0649 } },
	{ UTF32_NFKD(0xFD9A), { 0x0646, 0x0645, 0x064A } },
	{ UTF32_NFKD(0xFD9B), { 0x0646, 0x0645, 0x0649 } },
	{ UTF32_NFKD(0xFD9C), { 0x064A, 0x0645, 0x0645 } },
	{ UTF32_NFKD(0xFD9D), { 0x064A, 0x0645, 0x0645 } },
	{ UTF32_NFKD(0xFD9E), { 0x0628, 0x062E, 0x064A } },
	{ UTF32_NFKD(0xFD9F), { 0x062A, 0x062C, 0x064A } },
	{ UTF32_NFKD(0xFDA0), { 0x062A, 0x062C, 0x0649 } },
	{ UTF32_NFKD(0xFDA1), { 0x062A, 0x062E, 0x064A } },
	{ UTF32_NFKD(0xFDA2), { 0x062A, 0x062E, 0x0649 } },
	{ UTF32_NFKD(0xFDA3), { 0x062A, 0x0645, 0x064A } },
	{ UTF32_NFKD(0xFDA4), { 0x062A, 0x0645, 0x0649 } },
	{ UTF32_NFKD(0xFDA5), { 0x062C, 0x0645, 0x064A } },
	{ UTF32_NFKD(0xFDA6), { 0x062C, 0x062D, 0x0649 } },
	{ UTF32_NFKD(0xFDA7), { 0x062C, 0x0645, 0x0649 } },
	{ UTF32_NFKD(0xFDA8), { 0x0633, 0x062E, 0x0649 } },
	{ UTF32_NFKD(0xFDA9), { 0x0635, 0x062D, 0x064A } },
	{ UTF32_NFKD(0xFDAA), { 0x0634, 0x062D, 0x064A } },
	{ UTF32_NFKD(0xFDAB), { 0x0636, 0x062D, 0x064A } },
	{ UTF32_NFKD(0xFDAC), { 0x0644, 0x062C, 0x064A } },
	{ UTF32_NFKD(0xFDAD), { 0x0644, 0x0645, 0x064A } },
	{ UTF32_NFKD(0xFDAE), { 0x064A, 0x062D, 0x064A } },
	{ UTF32_NFKD(0xFDAF), { 0x064A, 0x062C, 0x064A } },
	{ UTF32_NFKD(0xFDB0), { 0x064A, 0x0645, 0x064A } },
	{ UTF32_NFKD(0xFDB1), { 0x0645, 0x0645, 0x064A } },
	{ UTF32_NFKD(0xFDB2), { 0x0642, 0x0645, 0x064A } },
	{ UTF32_NFKD(0xFDB3), { 0x0646, 0x062D, 0x064A } },
	{ UTF32_NFKD(0xFDB4), { 0x0642, 0x0645, 0x062D } },
	{ UTF32_NFKD(0xFDB5), { 0x0644, 0x062D, 0x0645 } },
	{ UTF32_NFKD(0xFDB6), { 0x0639, 0x0645, 0x064A } },
	{ UTF32_NFKD(0xFDB7), { 0x0643, 0x0645, 0x064A } },
	{ UTF32_NFKD(0xFDB8), { 0x0646, 0x062C, 0x062D } },
	{ UTF32_NFKD(0xFDB9), { 0x0645, 0x062E, 0x064A } },
	{ UTF32_NFKD(0xFDBA), { 0x0644, 0x062C, 0x0645 } },
	{ UTF32_NFKD(0xFDBB), { 0x0643, 0x0645, 0x0645 } },
	{ UTF32_NFKD(0xFDBC), { 0x0644, 0x062C, 0x0645 } },
	{ UTF32_NFKD(0xFDBD), { 0x0646, 0x062C, 0x062D } },
	{ UTF32_NFKD(0xFDBE), { 0x062C, 0x062D, 0x064A } },
	{ UTF32_NFKD(0xFDBF), { 0x062D, 0x062C, 0x064A } },
	{ UTF32_NFKD(0xFDC0), { 0x0645, 0x062C, 0x064A } },
	{ UTF32_NFKD(0xFDC1), { 0x0641, 0x0645, 0x064A } },
	{ UTF32_NFKD(0xFDC2), { 0x0628, 0x062D, 0x064A } },
	{ UTF32_NFKD(0xFDC3), { 0x0643, 0x0645, 0x0645 } },
	{ UTF32_NFKD(0xFDC4), { 0x0639, 0x062C, 0x0645 } },
	{ UTF32_NFKD(0xFDC5), { 0x0635, 0x0645, 0x0645 } },
	{ UTF32_NFKD(0xFDC6), { 0x0633, 0x062E, 0x064A } },
	{ UTF32_NFKD(0xFDC7), { 0x0646, 0x062C, 0x064A } },
	{ UTF32_NFKD(0xFDF0), { 0x0635, 0x0644, 0x06D2 } },
	{ UTF32_NFKD(0xFDF1), { 0x0642, 0x0644, 0x06D2 } },
	{ UTF32_NFKD(0xFDF2), { 0x0627, 0x0644, 0x0644, 0x0647 } },
	{ UTF32_NFKD(0xFDF3), { 0x0627, 0x0643, 0x0628, 0x0631 } },
	{ UTF32_NFKD(0xFDF4), { 0x0645, 0x062D, 0x0645, 0x062F } },
	{ UTF32_NFKD(0xFDF5), { 0x0635, 0x0644, 0x0639, 0x0645 } },
	{ UTF32_NFKD(0xFDF6), { 0x0631, 0x0633, 0x0648, 0x0644 } },
	{ UTF32_NFKD(0xFDF7), { 0x0639, 0x0644, 0x064A, 0x0647 } },
	{ UTF32_NFKD(0xFDF8), { 0x0648, 0x0633, 0x0644, 0x0645 } },
	{ UTF32_NFKD(0xFDF9), { 0x0635, 0x0644, 0x0649 } },
	{ UTF32_NFKD(0xFDFA), { 0x0635, 0x0644, 0x0649, 0x0020, 0x0627, 0x0644, 0x0644, 0x0647, 0x0020, 0x0639, 0x0644, 0x064A, 0x0647, 0x0020, 0x0648, 0x0633, 0x0644, 0x0645 } },
	{ UTF32_NFKD(0xFDFB), { 0x062C, 0x0644, 0x0020, 0x062C, 0x0644, 0x0627, 0x0644, 0x0647 } },
	{ UTF32_NFKD(0xFDFC), { 0x0631, 0x06CC, 0x0627, 0x0644 } },
	{ UTF32_NFKD(0xFE10), { 0x002C } },
	{ UTF32_NFKD(0xFE11), { 0x3001 } },
	{ UTF32_NFKD(0xFE12), { 0x3002 } },
	{ UTF32_NFKD(0xFE13), { 0x003A } },
	{ UTF32_NFKD(0xFE14), { 0x003B } },
	{ UTF32_NFKD(0xFE15), { 0x0021 } },
	{ UTF32_NFKD(0xFE16), { 0x003F } },
	{ UTF32_NFKD(0xFE17), { 0x3016 } },
	{ UTF32_NFKD(0xFE18), { 0x3017 } },
	{ UTF32_NFKD(0xFE19), { 0x2026 } },
	{ UTF32_NFKD(0xFE30), { 0x2025 } },
	{ UTF32_NFKD(0xFE31), { 0x2014 } },
	{ UTF32_NFKD(0xFE32), { 0x2013 } },
	{ UTF32_NFKD(0xFE33), { 0x005F } },
	{ UTF32_NFKD(0xFE34), { 0x005F } },
	{ UTF32_NFKD(0xFE35), { 0x0028 } },
	{ UTF32_NFKD(0xFE36), { 0x0029 } },
	{ UTF32_NFKD(0xFE37), { 0x007B } },
	{ UTF32_NFKD(0xFE38), { 0x007D } },
	{ UTF32_NFKD(0xFE39), { 0x3014 } },
	{ UTF32_NFKD(0xFE3A), { 0x3015 } },
	{ UTF32_NFKD(0xFE3B), { 0x3010 } },
	{ UTF32_NFKD(0xFE3C), { 0x3011 } },
	{ UTF32_NFKD(0xFE3D), { 0x300A } },
	{ UTF32_NFKD(0xFE3E), { 0x300B } },
	{ UTF32_NFKD(0xFE3F), { 0x3008 } },
	{ UTF32_NFKD(0xFE40), { 0x3009 } },
	{ UTF32_NFKD(0xFE41), { 0x300C } },
	{ UTF32_NFKD(0xFE42), { 0x300D } },
	{ UTF32_NFKD(0xFE43), { 0x300E } },
	{ UTF32_NFKD(0xFE44), { 0x300F } },
	{ UTF32_NFKD(0xFE47), { 0x005B } },
	{ UTF32_NFKD(0xFE48), { 0x005D } },
	{ UTF32_NFKD(0xFE49), { 0x203E } },
	{ UTF32_NFKD(0xFE4A), { 0x203E } },
	{ UTF32_NFKD(0xFE4B), { 0x203E } },
	{ UTF32_NFKD(0xFE4C), { 0x203E } },
	{ UTF32_NFKD(0xFE4D), { 0x005F } },
	{ UTF32_NFKD(0xFE4E), { 0x005F } },
	{ UTF32_NFKD(0xFE4F), { 0x005F } },
	{ UTF32_NFKD(0xFE50), { 0x002C } },
	{ UTF32_NFKD(0xFE51), { 0x3001 } },
	{ UTF32_NFKD(0xFE52), { 0x002E } },
	{ UTF32_NFKD(0xFE54), { 0x003B } },
	{ UTF32_NFKD(0xFE55), { 0x003A } },
	{ UTF32_NFKD(0xFE56), { 0x003F } },
	{ UTF32_NFKD(0xFE57), { 0x0021 } },
	{ UTF32_NFKD(0xFE58), { 0x2014 } },
	{ UTF32_NFKD(0xFE59), { 0x0028 } },
	{ UTF32_NFKD(0xFE5A), { 0x0029 } },
	{ UTF32_NFKD(0xFE5B), { 0x007B } },
	{ UTF32_NFKD(0xFE5C), { 0x007D } },
	{ UTF32_NFKD(0xFE5D), { 0x3014 } },
	{ UTF32_NFKD(0xFE5E), { 0x3015 } },
	{ UTF32_NFKD(0xFE5F), { 0x0023 } },
	{ UTF32_NFKD(0xFE60), { 0x0026 } },
	{ UTF32_NFKD(0xFE61), { 0x002A } },
	{ UTF32_NFKD(0xFE62), { 0x002B } },
	{ UTF32_NFKD(0xFE63), { 0x002D } },
	{ UTF32_NFKD(0xFE64), { 0x003C } },
	{ UTF32_NFKD(0xFE65), { 0x003E } },
	{ UTF32_NFKD(0xFE66), { 0x003D } },
	{ UTF32_NFKD(0xFE68), { 0x005C } },
	{ UTF32_NFKD(0xFE69), { 0x0024 } },
	{ UTF32_NFKD(0xFE6A), { 0x0025 } },
	{ UTF32_NFKD(0xFE6B), { 0x0040 } },
	{ UTF32_NFKD(0xFE70), { 0x0020, 0x064B } },
	{ UTF32_NFKD(0xFE71), { 0x0640, 0x064B } },
	{ UTF32_NFKD(0xFE72), { 0x0020, 0x064C } },
	{ UTF32_NFKD(0xFE74), { 0x0020, 0x064D } },
	{ UTF32_NFKD(0xFE76), { 0x0020, 0x064E } },
	{ UTF32_NFKD(0xFE77), { 0x0640, 0x064E } },
	{ UTF32_NFKD(0xFE78), { 0x0020, 0x064F } },
	{ UTF32_NFKD(0xFE79), { 0x0640, 0x064F } },
	{ UTF32_NFKD(0xFE7A), { 0x0020, 0x0650 } },
	{ UTF32_NFKD(0xFE7B), { 0x0640, 0x0650 } },
	{ UTF32_NFKD(0xFE7C), { 0x0020, 0x0651 } },
	{ UTF32_NFKD(0xFE7D), { 0x0640, 0x0651 } },
	{ UTF32_NFKD(0xFE7E), { 0x0020, 0x0652 } },
	{ UTF32_NFKD(0xFE7F), { 0x0640, 0x0652 } },
	{ UTF32_NFKD(0xFE80), { 0x0621 } },
	{ UTF32_NFKD(0xFE81), { 0x0622 } },
	{ UTF32_NFKD(0xFE82), { 0x0622 } },
	{ UTF32_NFKD(0xFE83), { 0x0623 } },
	{ UTF32_NFKD(0xFE84), { 0x0623 } },
	{ UTF32_NFKD(0xFE85), { 0x0624 } },
	{ UTF32_NFKD(0xFE86), { 0x0624 } },
	{ UTF32_NFKD(0xFE87), { 0x0625 } },
	{ UTF32_NFKD(0xFE88), { 0x0625 } },
	{ UTF32_NFKD(0xFE89), { 0x0626 } },
	{ UTF32_NFKD(0xFE8A), { 0x0626 } },
	{ UTF32_NFKD(0xFE8B), { 0x0626 } },
	{ UTF32_NFKD(0xFE8C), { 0x0626 } },
	{ UTF32_NFKD(0xFE8D), { 0x0627 } },
	{ UTF32_NFKD(0xFE8E), { 0x0627 } },
	{ UTF32_NFKD(0xFE8F), { 0x0628 } },
	{ UTF32_NFKD(0xFE90), { 0x0628 } },
	{ UTF32_NFKD(0xFE91), { 0x0628 } },
	{ UTF32_NFKD(0xFE92), { 0x0628 } },
	{ UTF32_NFKD(0xFE93), { 0x0629 } },
	{ UTF32_NFKD(0xFE94), { 0x0629 } },
	{ UTF32_NFKD(0xFE95), { 0x062A } },
	{ UTF32_NFKD(0xFE96), { 0x062A } },
	{ UTF32_NFKD(0xFE97), { 0x062A } },
	{ UTF32_NFKD(0xFE98), { 0x062A } },
	{ UTF32_NFKD(0xFE99), { 0x062B } },
	{ UTF32_NFKD(0xFE9A), { 0x062B } },
	{ UTF32_NFKD(0xFE9B), { 0x062B } },
	{ UTF32_NFKD(0xFE9C), { 0x062B } },
	{ UTF32_NFKD(0xFE9D), { 0x062C } },
	{ UTF32_NFKD(0xFE9E), { 0x062C } },
	{ UTF32_NFKD(0xFE9F), { 0x062C } },
	{ UTF32_NFKD(0xFEA0), { 0x062C } },
	{ UTF32_NFKD(0xFEA1), { 0x062D } },
	{ UTF32_NFKD(0xFEA2), { 0x062D } },
	{ UTF32_NFKD(0xFEA3), { 0x062D } },
	{ UTF32_NFKD(0xFEA4), { 0x062D } },
	{ UTF32_NFKD(0xFEA5), { 0x062E } },
	{ UTF32_NFKD(0xFEA6), { 0x062E } },
	{ UTF32_NFKD(0xFEA7), { 0x062E } },
	{ UTF32_NFKD(0xFEA8), { 0x062E } },
	{ UTF32_NFKD(0xFEA9), { 0x062F } },
	{ UTF32_NFKD(0xFEAA), { 0x062F } },
	{ UTF32_NFKD(0xFEAB), { 0x0630 } },
	{ UTF32_NFKD(0xFEAC), { 0x0630 } },
	{ UTF32_NFKD(0xFEAD), { 0x0631 } },
	{ UTF32_NFKD(0xFEAE), { 0x0631 } },
	{ UTF32_NFKD(0xFEAF), { 0x0632 } },
	{ UTF32_NFKD(0xFEB0), { 0x0632 } },
	{ UTF32_NFKD(0xFEB1), { 0x0633 } },
	{ UTF32_NFKD(0xFEB2), { 0x0633 } },
	{ UTF32_NFKD(0xFEB3), { 0x0633 } },
	{ UTF32_NFKD(0xFEB4), { 0x0633 } },
	{ UTF32_NFKD(0xFEB5), { 0x0634 } },
	{ UTF32_NFKD(0xFEB6), { 0x0634 } },
	{ UTF32_NFKD(0xFEB7), { 0x0634 } },
	{ UTF32_NFKD(0xFEB8), { 0x0634 } },
	{ UTF32_NFKD(0xFEB9), { 0x0635 } },
	{ UTF32_NFKD(0xFEBA), { 0x0635 } },
	{ UTF32_NFKD(0xFEBB), { 0x0635 } },
	{ UTF32_NFKD(0xFEBC), { 0x0635 } },
	{ UTF32_NFKD(0xFEBD), { 0x0636 } },
	{ UTF32_NFKD(0xFEBE), { 0x0636 } },
	{ UTF32_NFKD(0xFEBF), { 0x0636 } },
	{ UTF32_NFKD(0xFEC0), { 0x0636 } },
	{ UTF32_NFKD(0xFEC1), { 0x0637 } },
	{ UTF32_NFKD(0xFEC2), { 0x0637 } },
	{ UTF32_NFKD(0xFEC3), { 0x0637 } },
	{ UTF32_NFKD(0xFEC4), { 0x0637 } },
	{ UTF32_NFKD(0xFEC5), { 0x0638 } },
	{ UTF32_NFKD(0xFEC6), { 0x0638 } },
	{ UTF32_NFKD(0xFEC7), { 0x0638 } },
	{ UTF32_NFKD(0xFEC8), { 0x0638 } },
	{ UTF32_NFKD(0xFEC9), { 0x0639 } },
	{ UTF32_NFKD(0xFECA), { 0x0639 } },
	{ UTF32_NFKD(0xFECB), { 0x0639 } },
	{ UTF32_NFKD(0xFECC), { 0x0639 } },
	{ UTF32_NFKD(0xFECD), { 0x063A } },
	{ UTF32_NFKD(0xFECE), { 0x063A } },
	{ UTF32_NFKD(0xFECF), { 0x063A } },
	{ UTF32_NFKD(0xFED0), { 0x063A } },
	{ UTF32_NFKD(0xFED1), { 0x0641 } },
	{ UTF32_NFKD(0xFED2), { 0x0641 } },
	{ UTF32_NFKD(0xFED3), { 0x0641 } },
	{ UTF32_NFKD(0xFED4), { 0x0641 } },
	{ UTF32_NFKD(0xFED5), { 0x0642 } },
	{ UTF32_NFKD(0xFED6), { 0x0642 } },
	{ UTF32_NFKD(0xFED7), { 0x0642 } },
	{ UTF32_NFKD(0xFED8), { 0x0642 } },
	{ UTF32_NFKD(0xFED9), { 0x0643 } },
	{ UTF32_NFKD(0xFEDA), { 0x0643 } },
	{ UTF32_NFKD(0xFEDB), { 0x0643 } },
	{ UTF32_NFKD(0xFEDC), { 0x0643 } },
	{ UTF32_NFKD(0xFEDD), { 0x0644 } },
	{ UTF32_NFKD(0xFEDE), { 0x0644 } },
	{ UTF32_NFKD(0xFEDF), { 0x0644 } },
	{ UTF32_NFKD(0xFEE0), { 0x0644 } },
	{ UTF32_NFKD(0xFEE1), { 0x0645 } },
	{ UTF32_NFKD(0xFEE2), { 0x0645 } },
	{ UTF32_NFKD(0xFEE3), { 0x0645 } },
	{ UTF32_NFKD(0xFEE4), { 0x0645 } },
	{ UTF32_NFKD(0xFEE5), { 0x0646 } },
	{ UTF32_NFKD(0xFEE6), { 0x0646 } },
	{ UTF32_NFKD(0xFEE7), { 0x0646 } },
	{ UTF32_NFKD(0xFEE8), { 0x0646 } },
	{ UTF32_NFKD(0xFEE9), { 0x0647 } },
	{ UTF32_NFKD(0xFEEA), { 0x0647 } },
	{ UTF32_NFKD(0xFEEB), { 0x0647 } },
	{ UTF32_NFKD(0xFEEC), { 0x0647 } },
	{ UTF32_NFKD(0xFEED), { 0x0648 } },
	{ UTF32_NFKD(0xFEEE), { 0x0648 } },
	{ UTF32_NFKD(0xFEEF), { 0x0649 } },
	{ UTF32_NFKD(0xFEF0), { 0x0649 } },
	{ UTF32_NFKD(0xFEF1), { 0x064A } },
	{ UTF32_NFKD(0xFEF2), { 0x064A } },
	{ UTF32_NFKD(0xFEF3), { 0x064A } },
	{ UTF32_NFKD(0xFEF4), { 0x064A } },
	{ UTF32_NFKD(0xFEF5), { 0x0644, 0x0622 } },
	{ UTF32_NFKD(0xFEF6), { 0x0644, 0x0622 } },
	{ UTF32_NFKD(0xFEF7), { 0x0644, 0x0623 } },
	{ UTF32_NFKD(0xFEF8), { 0x0644, 0x0623 } },
	{ UTF32_NFKD(0xFEF9), { 0x0644, 0x0625 } },
	{ UTF32_NFKD(0xFEFA), { 0x0644, 0x0625 } },
	{ UTF32_NFKD(0xFEFB), { 0x0644, 0x0627 } },
	{ UTF32_NFKD(0xFEFC), { 0x0644, 0x0627 } },
	{ UTF32_NFKD(0xFF01), { 0x0021 } },
	{ UTF32_NFKD(0xFF02), { 0x0022 } },
	{ UTF32_NFKD(0xFF03), { 0x0023 } },
	{ UTF32_NFKD(0xFF04), { 0x0024 } },
	{ UTF32_NFKD(0xFF05), { 0x0025 } },
	{ UTF32_NFKD(0xFF06), { 0x0026 } },
	{ UTF32_NFKD(0xFF07), { 0x0027 } },
	{ UTF32_NFKD(0xFF08), { 0x0028 } },
	{ UTF32_NFKD(0xFF09), { 0x0029 } },
	{ UTF32_NFKD(0xFF0A), { 0x002A } },
	{ UTF32_NFKD(0xFF0B), { 0x002B } },
	{ UTF32_NFKD(0xFF0C), { 0x002C } },
	{ UTF32_NFKD(0xFF0D), { 0x002D } },
	{ UTF32_NFKD(0xFF0E), { 0x002E } },
	{ UTF32_NFKD(0xFF0F), { 0x002F } },
	{ UTF32_NFKD(0xFF10), { 0x0030 } },
	{ UTF32_NFKD(0xFF11), { 0x0031 } },
	{ UTF32_NFKD(0xFF12), { 0x0032 } },
	{ UTF32_NFKD(0xFF13), { 0x0033 } },
	{ UTF32_NFKD(0xFF14), { 0x0034 } },
	{ UTF32_NFKD(0xFF15), { 0x0035 } },
	{ UTF32_NFKD(0xFF16), { 0x0036 } },
	{ UTF32_NFKD(0xFF17), { 0x0037 } },
	{ UTF32_NFKD(0xFF18), { 0x0038 } },
	{ UTF32_NFKD(0xFF19), { 0x0039 } },
	{ UTF32_NFKD(0xFF1A), { 0x003A } },
	{ UTF32_NFKD(0xFF1B), { 0x003B } },
	{ UTF32_NFKD(0xFF1C), { 0x003C } },
	{ UTF32_NFKD(0xFF1D), { 0x003D } },
	{ UTF32_NFKD(0xFF1E), { 0x003E } },
	{ UTF32_NFKD(0xFF1F), { 0x003F } },
	{ UTF32_NFKD(0xFF20), { 0x0040 } },
	{ UTF32_NFKD(0xFF21), { 0x0041 } },
	{ UTF32_NFKD(0xFF22), { 0x0042 } },
	{ UTF32_NFKD(0xFF23), { 0x0043 } },
	{ UTF32_NFKD(0xFF24), { 0x0044 } },
	{ UTF32_NFKD(0xFF25), { 0x0045 } },
	{ UTF32_NFKD(0xFF26), { 0x0046 } },
	{ UTF32_NFKD(0xFF27), { 0x0047 } },
	{ UTF32_NFKD(0xFF28), { 0x0048 } },
	{ UTF32_NFKD(0xFF29), { 0x0049 } },
	{ UTF32_NFKD(0xFF2A), { 0x004A } },
	{ UTF32_NFKD(0xFF2B), { 0x004B } },
	{ UTF32_NFKD(0xFF2C), { 0x004C } },
	{ UTF32_NFKD(0xFF2D), { 0x004D } },
	{ UTF32_NFKD(0xFF2E), { 0x004E } },
	{ UTF32_NFKD(0xFF2F), { 0x004F } },
	{ UTF32_NFKD(0xFF30), { 0x0050 } },
	{ UTF32_NFKD(0xFF31), { 0x0051 } },
	{ UTF32_NFKD(0xFF32), { 0x0052 } },
	{ UTF32_NFKD(0xFF33), { 0x0053 } },
	{ UTF32_NFKD(0xFF34), { 0x0054 } },
	{ UTF32_NFKD(0xFF35), { 0x0055 } },
	{ UTF32_NFKD(0xFF36), { 0x0056 } },
	{ UTF32_NFKD(0xFF37), { 0x0057 } },
	{ UTF32_NFKD(0xFF38), { 0x0058 } },
	{ UTF32_NFKD(0xFF39), { 0x0059 } },
	{ UTF32_NFKD(0xFF3A), { 0x005A } },
	{ UTF32_NFKD(0xFF3B), { 0x005B } },
	{ UTF32_NFKD(0xFF3C), { 0x005C } },
	{ UTF32_NFKD(0xFF3D), { 0x005D } },
	{ UTF32_NFKD(0xFF3E), { 0x005E } },
	{ UTF32_NFKD(0xFF3F), { 0x005F } },
	{ UTF32_NFKD(0xFF40), { 0x0060 } },
	{ UTF32_NFKD(0xFF41), { 0x0061 } },
	{ UTF32_NFKD(0xFF42), { 0x0062 } },
	{ UTF32_NFKD(0xFF43), { 0x0063 } },
	{ UTF32_NFKD(0xFF44), { 0x0064 } },
	{ UTF32_NFKD(0xFF45), { 0x0065 } },
	{ UTF32_NFKD(0xFF46), { 0x0066 } },
	{ UTF32_NFKD(0xFF47), { 0x0067 } },
	{ UTF32_NFKD(0xFF48), { 0x0068 } },
	{ UTF32_NFKD(0xFF49), { 0x0069 } },
	{ UTF32_NFKD(0xFF4A), { 0x006A } },
	{ UTF32_NFKD(0xFF4B), { 0x006B } },
	{ UTF32_NFKD(0xFF4C), { 0x006C } },
	{ UTF32_NFKD(0xFF4D), { 0x006D } },
	{ UTF32_NFKD(0xFF4E), { 0x006E } },
	{ UTF32_NFKD(0xFF4F), { 0x006F } },
	{ UTF32_NFKD(0xFF50), { 0x0070 } },
	{ UTF32_NFKD(0xFF51), { 0x0071 } },
	{ UTF32_NFKD(0xFF52), { 0x0072 } },
	{ UTF32_NFKD(0xFF53), { 0x0073 } },
	{ UTF32_NFKD(0xFF54), { 0x0074 } },
	{ UTF32_NFKD(0xFF55), { 0x0075 } },
	{ UTF32_NFKD(0xFF56), { 0x0076 } },
	{ UTF32_NFKD(0xFF57), { 0x0077 } },
	{ UTF32_NFKD(0xFF58), { 0x0078 } },
	{ UTF32_NFKD(0xFF59), { 0x0079 } },
	{ UTF32_NFKD(0xFF5A), { 0x007A } },
	{ UTF32_NFKD(0xFF5B), { 0x007B } },
	{ UTF32_NFKD(0xFF5C), { 0x007C } },
	{ UTF32_NFKD(0xFF5D), { 0x007D } },
	{ UTF32_NFKD(0xFF5E), { 0x007E } },
	{ UTF32_NFKD(0xFF5F), { 0x2985 } },
	{ UTF32_NFKD(0xFF60), { 0x2986 } },
	{ UTF32_NFKD(0xFF61), { 0x3002 } },
	{ UTF32_NFKD(0xFF62), { 0x300C } },
	{ UTF32_NFKD(0xFF63), { 0x300D } },
	{ UTF32_NFKD(0xFF64), { 0x3001 } },
	{ UTF32_NFKD(0xFF65), { 0x30FB } },
	{ UTF32_NFKD(0xFF66), { 0x30F2 } },
	{ UTF32_NFKD(0xFF67), { 0x30A1 } },
	{ UTF32_NFKD(0xFF68), { 0x30A3 } },
	{ UTF32_NFKD(0xFF69), { 0x30A5 } },
	{ UTF32_NFKD(0xFF6A), { 0x30A7 } },
	{ UTF32_NFKD(0xFF6B), { 0x30A9 } },
	{ UTF32_NFKD(0xFF6C), { 0x30E3 } },
	{ UTF32_NFKD(0xFF6D), { 0x30E5 } },
	{ UTF32_NFKD(0xFF6E), { 0x30E7 } },
	{ UTF32_NFKD(0xFF6F), { 0x30C3 } },
	{ UTF32_NFKD(0xFF70), { 0x30FC } },
	{ UTF32_NFKD(0xFF71), { 0x30A2 } },
	{ UTF32_NFKD(0xFF72), { 0x30A4 } },
	{ UTF32_NFKD(0xFF73), { 0x30A6 } },
	{ UTF32_NFKD(0xFF74), { 0x30A8 } },
	{ UTF32_NFKD(0xFF75), { 0x30AA } },
	{ UTF32_NFKD(0xFF76), { 0x30AB } },
	{ UTF32_NFKD(0xFF77), { 0x30AD } },
	{ UTF32_NFKD(0xFF78), { 0x30AF } },
	{ UTF32_NFKD(0xFF79), { 0x30B1 } },
	{ UTF32_NFKD(0xFF7A), { 0x30B3 } },
	{ UTF32_NFKD(0xFF7B), { 0x30B5 } },
	{ UTF32_NFKD(0xFF7C), { 0x30B7 } },
	{ UTF32_NFKD(0xFF7D), { 0x30B9 } },
	{ UTF32_NFKD(0xFF7E), { 0x30BB } },
	{ UTF32_NFKD(0xFF7F), { 0x30BD } },
	{ UTF32_NFKD(0xFF80), { 0x30BF } },
	{ UTF32_NFKD(0xFF81), { 0x30C1 } },
	{ UTF32_NFKD(0xFF82), { 0x30C4 } },
	{ UTF32_NFKD(0xFF83), { 0x30C6 } },
	{ UTF32_NFKD(0xFF84), { 0x30C8 } },
	{ UTF32_NFKD(0xFF85), { 0x30CA } },
	{ UTF32_NFKD(0xFF86), { 0x30CB } },
	{ UTF32_NFKD(0xFF87), { 0x30CC } },
	{ UTF32_NFKD(0xFF88), { 0x30CD } },
	{ UTF32_NFKD(0xFF89), { 0x30CE } },
	{ UTF32_NFKD(0xFF8A), { 0x30CF } },
	{ UTF32_NFKD(0xFF8B), { 0x30D2 } },
	{ UTF32_NFKD(0xFF8C), { 0x30D5 } },
	{ UTF32_NFKD(0xFF8D), { 0x30D8 } },
	{ UTF32_NFKD(0xFF8E), { 0x30DB } },
	{ UTF32_NFKD(0xFF8F), { 0x30DE } },
	{ UTF32_NFKD(0xFF90), { 0x30DF } },
	{ UTF32_NFKD(0xFF91), { 0x30E0 } },
	{ UTF32_NFKD(0xFF92), { 0x30E1 } },
	{ UTF32_NFKD(0xFF93), { 0x30E2 } },
	{ UTF32_NFKD(0xFF94), { 0x30E4 } },
	{ UTF32_NFKD(0xFF95), { 0x30E6 } },
	{ UTF32_NFKD(0xFF96), { 0x30E8 } },
	{ UTF32_NFKD(0xFF97), { 0x30E9 } },
	{ UTF32_NFKD(0xFF98), { 0x30EA } },
	{ UTF32_NFKD(0xFF99), { 0x30EB } },
	{ UTF32_NFKD(0xFF9A), { 0x30EC } },
	{ UTF32_NFKD(0xFF9B), { 0x30ED } },
	{ UTF32_NFKD(0xFF9C), { 0x30EF } },
	{ UTF32_NFKD(0xFF9D), { 0x30F3 } },
	{ UTF32_NFKD(0xFF9E), { 0x3099 } },
	{ UTF32_NFKD(0xFF9F), { 0x309A } },
	{ UTF32_NFKD(0xFFA0), { 0x3164 } },
	{ UTF32_NFKD(0xFFA1), { 0x3131 } },
	{ UTF32_NFKD(0xFFA2), { 0x3132 } },
	{ UTF32_NFKD(0xFFA3), { 0x3133 } },
	{ UTF32_NFKD(0xFFA4), { 0x3134 } },
	{ UTF32_NFKD(0xFFA5), { 0x3135 } },
	{ UTF32_NFKD(0xFFA6), { 0x3136 } },
	{ UTF32_NFKD(0xFFA7), { 0x3137 } },
	{ UTF32_NFKD(0xFFA8), { 0x3138 } },
	{ UTF32_NFKD(0xFFA9), { 0x3139 } },
	{ UTF32_NFKD(0xFFAA), { 0x313A } },
	{ UTF32_NFKD(0xFFAB), { 0x313B } },
	{ UTF32_NFKD(0xFFAC), { 0x313C } },
	{ UTF32_NFKD(0xFFAD), { 0x313D } },
	{ UTF32_NFKD(0xFFAE), { 0x313E } },
	{ UTF32_NFKD(0xFFAF), { 0x313F } },
	{ UTF32_NFKD(0xFFB0), { 0x3140 } },
	{ UTF32_NFKD(0xFFB1), { 0x3141 } },
	{ UTF32_NFKD(0xFFB2), { 0x3142 } },
	{ UTF32_NFKD(0xFFB3), { 0x3143 } },
	{ UTF32_NFKD(0xFFB4), { 0x3144 } },
	{ UTF32_NFKD(0xFFB5), { 0x3145 } },
	{ UTF32_NFKD(0xFFB6), { 0x3146 } },
	{ UTF32_NFKD(0xFFB7), { 0x3147 } },
	{ UTF32_NFKD(0xFFB8), { 0x3148 } },
	{ UTF32_NFKD(0xFFB9), { 0x3149 } },
	{ UTF32_NFKD(0xFFBA), { 0x314A } },
	{ UTF32_NFKD(0xFFBB), { 0x314B } },
	{ UTF32_NFKD(0xFFBC), { 0x314C } },
	{ UTF32_NFKD(0xFFBD), { 0x314D } },
	{ UTF32_NFKD(0xFFBE), { 0x314E } },
	{ UTF32_NFKD(0xFFC2), { 0x314F } },
	{ UTF32_NFKD(0xFFC3), { 0x3150 } },
	{ UTF32_NFKD(0xFFC4), { 0x3151 } },
	{ UTF32_NFKD(0xFFC5), { 0x3152 } },
	{ UTF32_NFKD(0xFFC6), { 0x3153 } },
	{ UTF32_NFKD(0xFFC7), { 0x3154 } },
	{ UTF32_NFKD(0xFFCA), { 0x3155 } },
	{ UTF32_NFKD(0xFFCB), { 0x3156 } },
	{ UTF32_NFKD(0xFFCC), { 0x3157 } },
	{ UTF32_NFKD(0xFFCD), { 0x3158 } },
	{ UTF32_NFKD(0xFFCE), { 0x3159 } },
	{ UTF32_NFKD(0xFFCF), { 0x315A } },
	{ UTF32_NFKD(0xFFD2), { 0x315B } },
	{ UTF32_NFKD(0xFFD3), { 0x315C } },
	{ UTF32_NFKD(0xFFD4), { 0x315D } },
	{ UTF32_NFKD(0xFFD5), { 0x315E } },
	{ UTF32_NFKD(0xFFD6), { 0x315F } },
	{ UTF32_NFKD(0xFFD7), { 0x3160 } },
	{ UTF32_NFKD(0xFFDA), { 0x3161 } },
	{ UTF32_NFKD(0xFFDB), { 0x3162 } },
	{ UTF32_NFKD(0xFFDC), { 0x3163 } },
	{ UTF32_NFKD(0xFFE0), { 0x00A2 } },
	{ UTF32_NFKD(0xFFE1), { 0x00A3 } },
	{ UTF32_NFKD(0xFFE2), { 0x00AC } },
	{ UTF32_NFKD(0xFFE3), { 0x00AF } },
	{ UTF32_NFKD(0xFFE4), { 0x00A6 } },
	{ UTF32_NFKD(0xFFE5), { 0x00A5 } },
	{ UTF32_NFKD(0xFFE6), { 0x20A9 } },
	{ UTF32_NFKD(0xFFE8), { 0x2502 } },
	{ UTF32_NFKD(0xFFE9), { 0x2190 } },
	{ UTF32_NFKD(0xFFEA), { 0x2191 } },
	{ UTF32_NFKD(0xFFEB), { 0x2192 } },
	{ UTF32_NFKD(0xFFEC), { 0x2193 } },
	{ UTF32_NFKD(0xFFED), { 0x25A0 } },
	{ UTF32_NFKD(0xFFEE), { 0x25CB } },
	{ UTF32_NFD(0x1D15E), { 0x1D157, 0x1D165 } },
	{ UTF32_NFD(0x1D15F), { 0x1D158, 0x1D165 } },
	{ UTF32_NFD(0x1D160), { 0x1D15F, 0x1D16E } },
	{ UTF32_NFD(0x1D161), { 0x1D15F, 0x1D16F } },
	{ UTF32_NFD(0x1D162), { 0x1D15F, 0x1D170 } },
	{ UTF32_NFD(0x1D163), { 0x1D15F, 0x1D171 } },
	{ UTF32_NFD(0x1D164), { 0x1D15F, 0x1D172 } },
	{ UTF32_NFD(0x1D1BB), { 0x1D1B9, 0x1D165 } },
	{ UTF32_NFD(0x1D1BC), { 0x1D1BA, 0x1D165 } },
	{ UTF32_NFD(0x1D1BD), { 0x1D1BB, 0x1D16E } },
	{ UTF32_NFD(0x1D1BE), { 0x1D1BC, 0x1D16E } },
	{ UTF32_NFD(0x1D1BF), { 0x1D1BB, 0x1D16F } },
	{ UTF32_NFD(0x1D1C0), { 0x1D1BC, 0x1D16F } },
	{ UTF32_NFKD(0x1D400), { 0x0041 } },
	{ UTF32_NFKD(0x1D401), { 0x0042 } },
	{ UTF32_NFKD(0x1D402), { 0x0043 } },
	{ UTF32_NFKD(0x1D403), { 0x0044 } },
	{ UTF32_NFKD(0x1D404), { 0x0045 } },
	{ UTF32_NFKD(0x1D405), { 0x0046 } },
	{ UTF32_NFKD(0x1D406), { 0x0047 } },
	{ UTF32_NFKD(0x1D407), { 0x0048 } },
	{ UTF32_NFKD(0x1D408), { 0x0049 } },
	{ UTF32_NFKD(0x1D409), { 0x004A } },
	{ UTF32_NFKD(0x1D40A), { 0x004B } },
	{ UTF32_NFKD(0x1D40B), { 0x004C } },
	{ UTF32_NFKD(0x1D40C), { 0x004D } },
	{ UTF32_NFKD(0x1D40D), { 0x004E } },
	{ UTF32_NFKD(0x1D40E), { 0x004F } },
	{ UTF32_NFKD(0x1D40F), { 0x0050 } },
	{ UTF32_NFKD(0x1D410), { 0x0051 } },
	{ UTF32_NFKD(0x1D411), { 0x0052 } },
	{ UTF32_NFKD(0x1D412), { 0x0053 } },
	{ UTF32_NFKD(0x1D413), { 0x0054 } },
	{ UTF32_NFKD(0x1D414), { 0x0055 } },
	{ UTF32_NFKD(0x1D415), { 0x0056 } },
	{ UTF32_NFKD(0x1D416), { 0x0057 } },
	{ UTF32_NFKD(0x1D417), { 0x0058 } },
	{ UTF32_NFKD(0x1D418), { 0x0059 } },
	{ UTF32_NFKD(0x1D419), { 0x005A } },
	{ UTF32_NFKD(0x1D41A), { 0x0061 } },
	{ UTF32_NFKD(0x1D41B), { 0x0062 } },
	{ UTF32_NFKD(0x1D41C), { 0x0063 } },
	{ UTF32_NFKD(0x1D41D), { 0x0064 } },
	{ UTF32_NFKD(0x1D41E), { 0x0065 } },
	{ UTF32_NFKD(0x1D41F), { 0x0066 } },
	{ UTF32_NFKD(0x1D420), { 0x0067 } },
	{ UTF32_NFKD(0x1D421), { 0x0068 } },
	{ UTF32_NFKD(0x1D422), { 0x0069 } },
	{ UTF32_NFKD(0x1D423), { 0x006A } },
	{ UTF32_NFKD(0x1D424), { 0x006B } },
	{ UTF32_NFKD(0x1D425), { 0x006C } },
	{ UTF32_NFKD(0x1D426), { 0x006D } },
	{ UTF32_NFKD(0x1D427), { 0x006E } },
	{ UTF32_NFKD(0x1D428), { 0x006F } },
	{ UTF32_NFKD(0x1D429), { 0x0070 } },
	{ UTF32_NFKD(0x1D42A), { 0x0071 } },
	{ UTF32_NFKD(0x1D42B), { 0x0072 } },
	{ UTF32_NFKD(0x1D42C), { 0x0073 } },
	{ UTF32_NFKD(0x1D42D), { 0x0074 } },
	{ UTF32_NFKD(0x1D42E), { 0x0075 } },
	{ UTF32_NFKD(0x1D42F), { 0x0076 } },
	{ UTF32_NFKD(0x1D430), { 0x0077 } },
	{ UTF32_NFKD(0x1D431), { 0x0078 } },
	{ UTF32_NFKD(0x1D432), { 0x0079 } },
	{ UTF32_NFKD(0x1D433), { 0x007A } },
	{ UTF32_NFKD(0x1D434), { 0x0041 } },
	{ UTF32_NFKD(0x1D435), { 0x0042 } },
	{ UTF32_NFKD(0x1D436), { 0x0043 } },
	{ UTF32_NFKD(0x1D437), { 0x0044 } },
	{ UTF32_NFKD(0x1D438), { 0x0045 } },
	{ UTF32_NFKD(0x1D439), { 0x0046 } },
	{ UTF32_NFKD(0x1D43A), { 0x0047 } },
	{ UTF32_NFKD(0x1D43B), { 0x0048 } },
	{ UTF32_NFKD(0x1D43C), { 0x0049 } },
	{ UTF32_NFKD(0x1D43D), { 0x004A } },
	{ UTF32_NFKD(0x1D43E), { 0x004B } },
	{ UTF32_NFKD(0x1D43F), { 0x004C } },
	{ UTF32_NFKD(0x1D440), { 0x004D } },
	{ UTF32_NFKD(0x1D441), { 0x004E } },
	{ UTF32_NFKD(0x1D442), { 0x004F } },
	{ UTF32_NFKD(0x1D443), { 0x0050 } },
	{ UTF32_NFKD(0x1D444), { 0x0051 } },
	{ UTF32_NFKD(0x1D445), { 0x0052 } },
	{ UTF32_NFKD(0x1D446), { 0x0053 } },
	{ UTF32_NFKD(0x1D447), { 0x0054 } },
	{ UTF32_NFKD(0x1D448), { 0x0055 } },
	{ UTF32_NFKD(0x1D449), { 0x0056 } },
	{ UTF32_NFKD(0x1D44A), { 0x0057 } },
	{ UTF32_NFKD(0x1D44B), { 0x0058 } },
	{ UTF32_NFKD(0x1D44C), { 0x0059 } },
	{ UTF32_NFKD(0x1D44D), { 0x005A } },
	{ UTF32_NFKD(0x1D44E), { 0x0061 } },
	{ UTF32_NFKD(0x1D44F), { 0x0062 } },
	{ UTF32_NFKD(0x1D450), { 0x0063 } },
	{ UTF32_NFKD(0x1D451), { 0x0064 } },
	{ UTF32_NFKD(0x1D452), { 0x0065 } },
	{ UTF32_NFKD(0x1D453), { 0x0066 } },
	{ UTF32_NFKD(0x1D454), { 0x0067 } },
	{ UTF32_NFKD(0x1D456), { 0x0069 } },
	{ UTF32_NFKD(0x1D457), { 0x006A } },
	{ UTF32_NFKD(0x1D458), { 0x006B } },
	{ UTF32_NFKD(0x1D459), { 0x006C } },
	{ UTF32_NFKD(0x1D45A), { 0x006D } },
	{ UTF32_NFKD(0x1D45B), { 0x006E } },
	{ UTF32_NFKD(0x1D45C), { 0x006F } },
	{ UTF32_NFKD(0x1D45D), { 0x0070 } },
	{ UTF32_NFKD(0x1D45E), { 0x0071 } },
	{ UTF32_NFKD(0x1D45F), { 0x0072 } },
	{ UTF32_NFKD(0x1D460), { 0x0073 } },
	{ UTF32_NFKD(0x1D461), { 0x0074 } },
	{ UTF32_NFKD(0x1D462), { 0x0075 } },
	{ UTF32_NFKD(0x1D463), { 0x0076 } },
	{ UTF32_NFKD(0x1D464), { 0x0077 } },
	{ UTF32_NFKD(0x1D465), { 0x0078 } },
	{ UTF32_NFKD(0x1D466), { 0x0079 } },
	{ UTF32_NFKD(0x1D467), { 0x007A } },
	{ UTF32_NFKD(0x1D468), { 0x0041 } },
	{ UTF32_NFKD(0x1D469), { 0x0042 } },
	{ UTF32_NFKD(0x1D46A), { 0x0043 } },
	{ UTF32_NFKD(0x1D46B), { 0x0044 } },
	{ UTF32_NFKD(0x1D46C), { 0x0045 } },
	{ UTF32_NFKD(0x1D46D), { 0x0046 } },
	{ UTF32_NFKD(0x1D46E), { 0x0047 } },
	{ UTF32_NFKD(0x1D46F), { 0x0048 } },
	{ UTF32_NFKD(0x1D470), { 0x0049 } },
	{ UTF32_NFKD(0x1D471), { 0x004A } },
	{ UTF32_NFKD(0x1D472), { 0x004B } },
	{ UTF32_NFKD(0x1D473), { 0x004C } },
	{ UTF32_NFKD(0x1D474), { 0x004D } },
	{ UTF32_NFKD(0x1D475), { 0x004E } },
	{ UTF32_NFKD(0x1D476), { 0x004F } },
	{ UTF32_NFKD(0x1D477), { 0x0050 } },
	{ UTF32_NFKD(0x1D478), { 0x0051 } },
	{ UTF32_NFKD(0x1D479), { 0x0052 } },
	{ UTF32_NFKD(0x1D47A), { 0x0053 } },
	{ UTF32_NFKD(0x1D47B), { 0x0054 } },
	{ UTF32_NFKD(0x1D47C), { 0x0055 } },
	{ UTF32_NFKD(0x1D47D), { 0x0056 } },
	{ UTF32_NFKD(0x1D47E), { 0x0057 } },
	{ UTF32_NFKD(0x1D47F), { 0x0058 } },
	{ UTF32_NFKD(0x1D480), { 0x0059 } },
	{ UTF32_NFKD(0x1D481), { 0x005A } },
	{ UTF32_NFKD(0x1D482), { 0x0061 } },
	{ UTF32_NFKD(0x1D483), { 0x0062 } },
	{ UTF32_NFKD(0x1D484), { 0x0063 } },
	{ UTF32_NFKD(0x1D485), { 0x0064 } },
	{ UTF32_NFKD(0x1D486), { 0x0065 } },
	{ UTF32_NFKD(0x1D487), { 0x0066 } },
	{ UTF32_NFKD(0x1D488), { 0x0067 } },
	{ UTF32_NFKD(0x1D489), { 0x0068 } },
	{ UTF32_NFKD(0x1D48A), { 0x0069 } },
	{ UTF32_NFKD(0x1D48B), { 0x006A } },
	{ UTF32_NFKD(0x1D48C), { 0x006B } },
	{ UTF32_NFKD(0x1D48D), { 0x006C } },
	{ UTF32_NFKD(0x1D48E), { 0x006D } },
	{ UTF32_NFKD(0x1D48F), { 0x006E } },
	{ UTF32_NFKD(0x1D490), { 0x006F } },
	{ UTF32_NFKD(0x1D491), { 0x0070 } },
	{ UTF32_NFKD(0x1D492), { 0x0071 } },
	{ UTF32_NFKD(0x1D493), { 0x0072 } },
	{ UTF32_NFKD(0x1D494), { 0x0073 } },
	{ UTF32_NFKD(0x1D495), { 0x0074 } },
	{ UTF32_NFKD(0x1D496), { 0x0075 } },
	{ UTF32_NFKD(0x1D497), { 0x0076 } },
	{ UTF32_NFKD(0x1D498), { 0x0077 } },
	{ UTF32_NFKD(0x1D499), { 0x0078 } },
	{ UTF32_NFKD(0x1D49A), { 0x0079 } },
	{ UTF32_NFKD(0x1D49B), { 0x007A } },
	{ UTF32_NFKD(0x1D49C), { 0x0041 } },
	{ UTF32_NFKD(0x1D49E), { 0x0043 } },
	{ UTF32_NFKD(0x1D49F), { 0x0044 } },
	{ UTF32_NFKD(0x1D4A2), { 0x0047 } },
	{ UTF32_NFKD(0x1D4A5), { 0x004A } },
	{ UTF32_NFKD(0x1D4A6), { 0x004B } },
	{ UTF32_NFKD(0x1D4A9), { 0x004E } },
	{ UTF32_NFKD(0x1D4AA), { 0x004F } },
	{ UTF32_NFKD(0x1D4AB), { 0x0050 } },
	{ UTF32_NFKD(0x1D4AC), { 0x0051 } },
	{ UTF32_NFKD(0x1D4AE), { 0x0053 } },
	{ UTF32_NFKD(0x1D4AF), { 0x0054 } },
	{ UTF32_NFKD(0x1D4B0), { 0x0055 } },
	{ UTF32_NFKD(0x1D4B1), { 0x0056 } },
	{ UTF32_NFKD(0x1D4B2), { 0x0057 } },
	{ UTF32_NFKD(0x1D4B3), { 0x0058 } },
	{ UTF32_NFKD(0x1D4B4), { 0x0059 } },
	{ UTF32_NFKD(0x1D4B5), { 0x005A } },
	{ UTF32_NFKD(0x1D4B6), { 0x0061 } },
	{ UTF32_NFKD(0x1D4B7), { 0x0062 } },
	{ UTF32_NFKD(0x1D4B8), { 0x0063 } },
	{ UTF32_NFKD(0x1D4B9), { 0x0064 } },
	{ UTF32_NFKD(0x1D4BB), { 0x0066 } },
	{ UTF32_NFKD(0x1D4BD), { 0x0068 } },
	{ UTF32_NFKD(0x1D4BE), { 0x0069 } },
	{ UTF32_NFKD(0x1D4BF), { 0x006A } },
	{ UTF32_NFKD(0x1D4C0), { 0x006B } },
	{ UTF32_NFKD(0x1D4C1), { 0x006C } },
	{ UTF32_NFKD(0x1D4C2), { 0x006D } },
	{ UTF32_NFKD(0x1D4C3), { 0x006E } },
	{ UTF32_NFKD(0x1D4C5), { 0x0070 } },
	{ UTF32_NFKD(0x1D4C6), { 0x0071 } },
	{ UTF32_NFKD(0x1D4C7), { 0x0072 } },
	{ UTF32_NFKD(0x1D4C8), { 0x0073 } },
	{ UTF32_NFKD(0x1D4C9), { 0x0074 } },
	{ UTF32_NFKD(0x1D4CA), { 0x0075 } },
	{ UTF32_NFKD(0x1D4CB), { 0x0076 } },
	{ UTF32_NFKD(0x1D4CC), { 0x0077 } },
	{ UTF32_NFKD(0x1D4CD), { 0x0078 } },
	{ UTF32_NFKD(0x1D4CE), { 0x0079 } },
	{ UTF32_NFKD(0x1D4CF), { 0x007A } },
	{ UTF32_NFKD(0x1D4D0), { 0x0041 } },
	{ UTF32_NFKD(0x1D4D1), { 0x0042 } },
	{ UTF32_NFKD(0x1D4D2), { 0x0043 } },
	{ UTF32_NFKD(0x1D4D3), { 0x0044 } },
	{ UTF32_NFKD(0x1D4D4), { 0x0045 } },
	{ UTF32_NFKD(0x1D4D5), { 0x0046 } },
	{ UTF32_NFKD(0x1D4D6), { 0x0047 } },
	{ UTF32_NFKD(0x1D4D7), { 0x0048 } },
	{ UTF32_NFKD(0x1D4D8), { 0x0049 } },
	{ UTF32_NFKD(0x1D4D9), { 0x004A } },
	{ UTF32_NFKD(0x1D4DA), { 0x004B } },
	{ UTF32_NFKD(0x1D4DB), { 0x004C } },
	{ UTF32_NFKD(0x1D4DC), { 0x004D } },
	{ UTF32_NFKD(0x1D4DD), { 0x004E } },
	{ UTF32_NFKD(0x1D4DE), { 0x004F } },
	{ UTF32_NFKD(0x1D4DF), { 0x0050 } },
	{ UTF32_NFKD(0x1D4E0), { 0x0051 } },
	{ UTF32_NFKD(0x1D4E1), { 0x0052 } },
	{ UTF32_NFKD(0x1D4E2), { 0x0053 } },
	{ UTF32_NFKD(0x1D4E3), { 0x0054 } },
	{ UTF32_NFKD(0x1D4E4), { 0x0055 } },
	{ UTF32_NFKD(0x1D4E5), { 0x0056 } },
	{ UTF32_NFKD(0x1D4E6), { 0x0057 } },
	{ UTF32_NFKD(0x1D4E7), { 0x0058 } },
	{ UTF32_NFKD(0x1D4E8), { 0x0059 } },
	{ UTF32_NFKD(0x1D4E9), { 0x005A } },
	{ UTF32_NFKD(0x1D4EA), { 0x0061 } },
	{ UTF32_NFKD(0x1D4EB), { 0x0062 } },
	{ UTF32_NFKD(0x1D4EC), { 0x0063 } },
	{ UTF32_NFKD(0x1D4ED), { 0x0064 } },
	{ UTF32_NFKD(0x1D4EE), { 0x0065 } },
	{ UTF32_NFKD(0x1D4EF), { 0x0066 } },
	{ UTF32_NFKD(0x1D4F0), { 0x0067 } },
	{ UTF32_NFKD(0x1D4F1), { 0x0068 } },
	{ UTF32_NFKD(0x1D4F2), { 0x0069 } },
	{ UTF32_NFKD(0x1D4F3), { 0x006A } },
	{ UTF32_NFKD(0x1D4F4), { 0x006B } },
	{ UTF32_NFKD(0x1D4F5), { 0x006C } },
	{ UTF32_NFKD(0x1D4F6), { 0x006D } },
	{ UTF32_NFKD(0x1D4F7), { 0x006E } },
	{ UTF32_NFKD(0x1D4F8), { 0x006F } },
	{ UTF32_NFKD(0x1D4F9), { 0x0070 } },
	{ UTF32_NFKD(0x1D4FA), { 0x0071 } },
	{ UTF32_NFKD(0x1D4FB), { 0x0072 } },
	{ UTF32_NFKD(0x1D4FC), { 0x0073 } },
	{ UTF32_NFKD(0x1D4FD), { 0x0074 } },
	{ UTF32_NFKD(0x1D4FE), { 0x0075 } },
	{ UTF32_NFKD(0x1D4FF), { 0x0076 } },
	{ UTF32_NFKD(0x1D500), { 0x0077 } },
	{ UTF32_NFKD(0x1D501), { 0x0078 } },
	{ UTF32_NFKD(0x1D502), { 0x0079 } },
	{ UTF32_NFKD(0x1D503), { 0x007A } },
	{ UTF32_NFKD(0x1D504), { 0x0041 } },
	{ UTF32_NFKD(0x1D505), { 0x0042 } },
	{ UTF32_NFKD(0x1D507), { 0x0044 } },
	{ UTF32_NFKD(0x1D508), { 0x0045 } },
	{ UTF32_NFKD(0x1D509), { 0x0046 } },
	{ UTF32_NFKD(0x1D50A), { 0x0047 } },
	{ UTF32_NFKD(0x1D50D), { 0x004A } },
	{ UTF32_NFKD(0x1D50E), { 0x004B } },
	{ UTF32_NFKD(0x1D50F), { 0x004C } },
	{ UTF32_NFKD(0x1D510), { 0x004D } },
	{ UTF32_NFKD(0x1D511), { 0x004E } },
	{ UTF32_NFKD(0x1D512), { 0x004F } },
	{ UTF32_NFKD(0x1D513), { 0x0050 } },
	{ UTF32_NFKD(0x1D514), { 0x0051 } },
	{ UTF32_NFKD(0x1D516), { 0x0053 } },
	{ UTF32_NFKD(0x1D517), { 0x0054 } },
	{ UTF32_NFKD(0x1D518), { 0x0055 } },
	{ UTF32_NFKD(0x1D519), { 0x0056 } },
	{ UTF32_NFKD(0x1D51A), { 0x0057 } },
	{ UTF32_NFKD(0x1D51B), { 0x0058 } },
	{ UTF32_NFKD(0x1D51C), { 0x0059 } },
	{ UTF32_NFKD(0x1D51E), { 0x0061 } },
	{ UTF32_NFKD(0x1D51F), { 0x0062 } },
	{ UTF32_NFKD(0x1D520), { 0x0063 } },
	{ UTF32_NFKD(0x1D521), { 0x0064 } },
	{ UTF32_NFKD(0x1D522), { 0x0065 } },
	{ UTF32_NFKD(0x1D523), { 0x0066 } },
	{ UTF32_NFKD(0x1D524), { 0x0067 } },
	{ UTF32_NFKD(0x1D525), { 0x0068 } },
	{ UTF32_NFKD(0x1D526), { 0x0069 } },
	{ UTF32_NFKD(0x1D527), { 0x006A } },
	{ UTF32_NFKD(0x1D528), { 0x006B } },
	{ UTF32_NFKD(0x1D529), { 0x006C } },
	{ UTF32_NFKD(0x1D52A), { 0x006D } },
	{ UTF32_NFKD(0x1D52B), { 0x006E } },
	{ UTF32_NFKD(0x1D52C), { 0x006F } },
	{ UTF32_NFKD(0x1D52D), { 0x0070 } },
	{ UTF32_NFKD(0x1D52E), { 0x0071 } },
	{ UTF32_NFKD(0x1D52F), { 0x0072 } },
	{ UTF32_NFKD(0x1D530), { 0x0073 } },
	{ UTF32_NFKD(0x1D531), { 0x0074 } },
	{ UTF32_NFKD(0x1D532), { 0x0075 } },
	{ UTF32_NFKD(0x1D533), { 0x0076 } },
	{ UTF32_NFKD(0x1D534), { 0x0077 } },
	{ UTF32_NFKD(0x1D535), { 0x0078 } },
	{ UTF32_NFKD(0x1D536), { 0x0079 } },
	{ UTF32_NFKD(0x1D537), { 0x007A } },
	{ UTF32_NFKD(0x1D538), { 0x0041 } },
	{ UTF32_NFKD(0x1D539), { 0x0042 } },
	{ UTF32_NFKD(0x1D53B), { 0x0044 } },
	{ UTF32_NFKD(0x1D53C), { 0x0045 } },
	{ UTF32_NFKD(0x1D53D), { 0x0046 } },
	{ UTF32_NFKD(0x1D53E), { 0x0047 } },
	{ UTF32_NFKD(0x1D540), { 0x0049 } },
	{ UTF32_NFKD(0x1D541), { 0x004A } },
	{ UTF32_NFKD(0x1D542), { 0x004B } },
	{ UTF32_NFKD(0x1D543), { 0x004C } },
	{ UTF32_NFKD(0x1D544), { 0x004D } },
	{ UTF32_NFKD(0x1D546), { 0x004F } },
	{ UTF32_NFKD(0x1D54A), { 0x0053 } },
	{ UTF32_NFKD(0x1D54B), { 0x0054 } },
	{ UTF32_NFKD(0x1D54C), { 0x0055 } },
	{ UTF32_NFKD(0x1D54D), { 0x0056 } },
	{ UTF32_NFKD(0x1D54E), { 0x0057 } },
	{ UTF32_NFKD(0x1D54F), { 0x0058 } },
	{ UTF32_NFKD(0x1D550), { 0x0059 } },
	{ UTF32_NFKD(0x1D552), { 0x0061 } },
	{ UTF32_NFKD(0x1D553), { 0x0062 } },
	{ UTF32_NFKD(0x1D554), { 0x0063 } },
	{ UTF32_NFKD(0x1D555), { 0x0064 } },
	{ UTF32_NFKD(0x1D556), { 0x0065 } },
	{ UTF32_NFKD(0x1D557), { 0x0066 } },
	{ UTF32_NFKD(0x1D558), { 0x0067 } },
	{ UTF32_NFKD(0x1D559), { 0x0068 } },
	{ UTF32_NFKD(0x1D55A), { 0x0069 } },
	{ UTF32_NFKD(0x1D55B), { 0x006A } },
	{ UTF32_NFKD(0x1D55C), { 0x006B } },
	{ UTF32_NFKD(0x1D55D), { 0x006C } },
	{ UTF32_NFKD(0x1D55E), { 0x006D } },
	{ UTF32_NFKD(0x1D55F), { 0x006E } },
	{ UTF32_NFKD(0x1D560), { 0x006F } },
	{ UTF32_NFKD(0x1D561), { 0x0070 } },
	{ UTF32_NFKD(0x1D562), { 0x0071 } },
	{ UTF32_NFKD(0x1D563), { 0x0072 } },
	{ UTF32_NFKD(0x1D564), { 0x0073 } },
	{ UTF32_NFKD(0x1D565), { 0x0074 } },
	{ UTF32_NFKD(0x1D566), { 0x0075 } },
	{ UTF32_NFKD(0x1D567), { 0x0076 } },
	{ UTF32_NFKD(0x1D568), { 0x0077 } },
	{ UTF32_NFKD(0x1D569), { 0x0078 } },
	{ UTF32_NFKD(0x1D56A), { 0x0079 } },
	{ UTF32_NFKD(0x1D56B), { 0x007A } },
	{ UTF32_NFKD(0x1D56C), { 0x0041 } },
	{ UTF32_NFKD(0x1D56D), { 0x0042 } },
	{ UTF32_NFKD(0x1D56E), { 0x0043 } },
	{ UTF32_NFKD(0x1D56F), { 0x0044 } },
	{ UTF32_NFKD(0x1D570), { 0x0045 } },
	{ UTF32_NFKD(0x1D571), { 0x0046 } },
	{ UTF32_NFKD(0x1D572), { 0x0047 } },
	{ UTF32_NFKD(0x1D573), { 0x0048 } },
	{ UTF32_NFKD(0x1D574), { 0x0049 } },
	{ UTF32_NFKD(0x1D575), { 0x004A } },
	{ UTF32_NFKD(0x1D576), { 0x004B } },
	{ UTF32_NFKD(0x1D577), { 0x004C } },
	{ UTF32_NFKD(0x1D578), { 0x004D } },
	{ UTF32_NFKD(0x1D579), { 0x004E } },
	{ UTF32_NFKD(0x1D57A), { 0x004F } },
	{ UTF32_NFKD(0x1D57B), { 0x0050 } },
	{ UTF32_NFKD(0x1D57C), { 0x0051 } },
	{ UTF32_NFKD(0x1D57D), { 0x0052 } },
	{ UTF32_NFKD(0x1D57E), { 0x0053 } },
	{ UTF32_NFKD(0x1D57F), { 0x0054 } },
	{ UTF32_NFKD(0x1D580), { 0x0055 } },
	{ UTF32_NFKD(0x1D581), { 0x0056 } },
	{ UTF32_NFKD(0x1D582), { 0x0057 } },
	{ UTF32_NFKD(0x1D583), { 0x0058 } },
	{ UTF32_NFKD(0x1D584), { 0x0059 } },
	{ UTF32_NFKD(0x1D585), { 0x005A } },
	{ UTF32_NFKD(0x1D586), { 0x0061 } },
	{ UTF32_NFKD(0x1D587), { 0x0062 } },
	{ UTF32_NFKD(0x1D588), { 0x0063 } },
	{ UTF32_NFKD(0x1D589), { 0x0064 } },
	{ UTF32_NFKD(0x1D58A), { 0x0065 } },
	{ UTF32_NFKD(0x1D58B), { 0x0066 } },
	{ UTF32_NFKD(0x1D58C), { 0x0067 } },
	{ UTF32_NFKD(0x1D58D), { 0x0068 } },
	{ UTF32_NFKD(0x1D58E), { 0x0069 } },
	{ UTF32_NFKD(0x1D58F), { 0x006A } },
	{ UTF32_NFKD(0x1D590), { 0x006B } },
	{ UTF32_NFKD(0x1D591), { 0x006C } },
	{ UTF32_NFKD(0x1D592), { 0x006D } },
	{ UTF32_NFKD(0x1D593), { 0x006E } },
	{ UTF32_NFKD(0x1D594), { 0x006F } },
	{ UTF32_NFKD(0x1D595), { 0x0070 } },
	{ UTF32_NFKD(0x1D596), { 0x0071 } },
	{ UTF32_NFKD(0x1D597), { 0x0072 } },
	{ UTF32_NFKD(0x1D598), { 0x0073 } },
	{ UTF32_NFKD(0x1D599), { 0x0074 } },
	{ UTF32_NFKD(0x1D59A), { 0x0075 } },
	{ UTF32_NFKD(0x1D59B), { 0x0076 } },
	{ UTF32_NFKD(0x1D59C), { 0x0077 } },
	{ UTF32_NFKD(0x1D59D), { 0x0078 } },
	{ UTF32_NFKD(0x1D59E), { 0x0079 } },
	{ UTF32_NFKD(0x1D59F), { 0x007A } },
	{ UTF32_NFKD(0x1D5A0), { 0x0041 } },
	{ UTF32_NFKD(0x1D5A1), { 0x0042 } },
	{ UTF32_NFKD(0x1D5A2), { 0x0043 } },
	{ UTF32_NFKD(0x1D5A3), { 0x0044 } },
	{ UTF32_NFKD(0x1D5A4), { 0x0045 } },
	{ UTF32_NFKD(0x1D5A5), { 0x0046 } },
	{ UTF32_NFKD(0x1D5A6), { 0x0047 } },
	{ UTF32_NFKD(0x1D5A7), { 0x0048 } },
	{ UTF32_NFKD(0x1D5A8), { 0x0049 } },
	{ UTF32_NFKD(0x1D5A9), { 0x004A } },
	{ UTF32_NFKD(0x1D5AA), { 0x004B } },
	{ UTF32_NFKD(0x1D5AB), { 0x004C } },
	{ UTF32_NFKD(0x1D5AC), { 0x004D } },
	{ UTF32_NFKD(0x1D5AD), { 0x004E } },
	{ UTF32_NFKD(0x1D5AE), { 0x004F } },
	{ UTF32_NFKD(0x1D5AF), { 0x0050 } },
	{ UTF32_NFKD(0x1D5B0), { 0x0051 } },
	{ UTF32_NFKD(0x1D5B1), { 0x0052 } },
	{ UTF32_NFKD(0x1D5B2), { 0x0053 } },
	{ UTF32_NFKD(0x1D5B3), { 0x0054 } },
	{ UTF32_NFKD(0x1D5B4), { 0x0055 } },
	{ UTF32_NFKD(0x1D5B5), { 0x0056 } },
	{ UTF32_NFKD(0x1D5B6), { 0x0057 } },
	{ UTF32_NFKD(0x1D5B7), { 0x0058 } },
	{ UTF32_NFKD(0x1D5B8), { 0x0059 } },
	{ UTF32_NFKD(0x1D5B9), { 0x005A } },
	{ UTF32_NFKD(0x1D5BA), { 0x0061 } },
	{ UTF32_NFKD(0x1D5BB), { 0x0062 } },
	{ UTF32_NFKD(0x1D5BC), { 0x0063 } },
	{ UTF32_NFKD(0x1D5BD), { 0x0064 } },
	{ UTF32_NFKD(0x1D5BE), { 0x0065 } },
	{ UTF32_NFKD(0x1D5BF), { 0x0066 } },
	{ UTF32_NFKD(0x1D5C0), { 0x0067 } },
	{ UTF32_NFKD(0x1D5C1), { 0x0068 } },
	{ UTF32_NFKD(0x1D5C2), { 0x0069 } },
	{ UTF32_NFKD(0x1D5C3), { 0x006A } },
	{ UTF32_NFKD(0x1D5C4), { 0x006B } },
	{ UTF32_NFKD(0x1D5C5), { 0x006C } },
	{ UTF32_NFKD(0x1D5C6), { 0x006D } },
	{ UTF32_NFKD(0x1D5C7), { 0x006E } },
	{ UTF32_NFKD(0x1D5C8), { 0x006F } },
	{ UTF32_NFKD(0x1D5C9), { 0x0070 } },
	{ UTF32_NFKD(0x1D5CA), { 0x0071 } },
	{ UTF32_NFKD(0x1D5CB), { 0x0072 } },
	{ UTF32_NFKD(0x1D5CC), { 0x0073 } },
	{ UTF32_NFKD(0x1D5CD), { 0x0074 } },
	{ UTF32_NFKD(0x1D5CE), { 0x0075 } },
	{ UTF32_NFKD(0x1D5CF), { 0x0076 } },
	{ UTF32_NFKD(0x1D5D0), { 0x0077 } },
	{ UTF32_NFKD(0x1D5D1), { 0x0078 } },
	{ UTF32_NFKD(0x1D5D2), { 0x0079 } },
	{ UTF32_NFKD(0x1D5D3), { 0x007A } },
	{ UTF32_NFKD(0x1D5D4), { 0x0041 } },
	{ UTF32_NFKD(0x1D5D5), { 0x0042 } },
	{ UTF32_NFKD(0x1D5D6), { 0x0043 } },
	{ UTF32_NFKD(0x1D5D7), { 0x0044 } },
	{ UTF32_NFKD(0x1D5D8), { 0x0045 } },
	{ UTF32_NFKD(0x1D5D9), { 0x0046 } },
	{ UTF32_NFKD(0x1D5DA), { 0x0047 } },
	{ UTF32_NFKD(0x1D5DB), { 0x0048 } },
	{ UTF32_NFKD(0x1D5DC), { 0x0049 } },
	{ UTF32_NFKD(0x1D5DD), { 0x004A } },
	{ UTF32_NFKD(0x1D5DE), { 0x004B } },
	{ UTF32_NFKD(0x1D5DF), { 0x004C } },
	{ UTF32_NFKD(0x1D5E0), { 0x004D } },
	{ UTF32_NFKD(0x1D5E1), { 0x004E } },
	{ UTF32_NFKD(0x1D5E2), { 0x004F } },
	{ UTF32_NFKD(0x1D5E3), { 0x0050 } },
	{ UTF32_NFKD(0x1D5E4), { 0x0051 } },
	{ UTF32_NFKD(0x1D5E5), { 0x0052 } },
	{ UTF32_NFKD(0x1D5E6), { 0x0053 } },
	{ UTF32_NFKD(0x1D5E7), { 0x0054 } },
	{ UTF32_NFKD(0x1D5E8), { 0x0055 } },
	{ UTF32_NFKD(0x1D5E9), { 0x0056 } },
	{ UTF32_NFKD(0x1D5EA), { 0x0057 } },
	{ UTF32_NFKD(0x1D5EB), { 0x0058 } },
	{ UTF32_NFKD(0x1D5EC), { 0x0059 } },
	{ UTF32_NFKD(0x1D5ED), { 0x005A } },
	{ UTF32_NFKD(0x1D5EE), { 0x0061 } },
	{ UTF32_NFKD(0x1D5EF), { 0x0062 } },
	{ UTF32_NFKD(0x1D5F0), { 0x0063 } },
	{ UTF32_NFKD(0x1D5F1), { 0x0064 } },
	{ UTF32_NFKD(0x1D5F2), { 0x0065 } },
	{ UTF32_NFKD(0x1D5F3), { 0x0066 } },
	{ UTF32_NFKD(0x1D5F4), { 0x0067 } },
	{ UTF32_NFKD(0x1D5F5), { 0x0068 } },
	{ UTF32_NFKD(0x1D5F6), { 0x0069 } },
	{ UTF32_NFKD(0x1D5F7), { 0x006A } },
	{ UTF32_NFKD(0x1D5F8), { 0x006B } },
	{ UTF32_NFKD(0x1D5F9), { 0x006C } },
	{ UTF32_NFKD(0x1D5FA), { 0x006D } },
	{ UTF32_NFKD(0x1D5FB), { 0x006E } },
	{ UTF32_NFKD(0x1D5FC), { 0x006F } },
	{ UTF32_NFKD(0x1D5FD), { 0x0070 } },
	{ UTF32_NFKD(0x1D5FE), { 0x0071 } },
	{ UTF32_NFKD(0x1D5FF), { 0x0072 } },
	{ UTF32_NFKD(0x1D600), { 0x0073 } },
	{ UTF32_NFKD(0x1D601), { 0x0074 } },
	{ UTF32_NFKD(0x1D602), { 0x0075 } },
	{ UTF32_NFKD(0x1D603), { 0x0076 } },
	{ UTF32_NFKD(0x1D604), { 0x0077 } },
	{ UTF32_NFKD(0x1D605), { 0x0078 } },
	{ UTF32_NFKD(0x1D606), { 0x0079 } },
	{ UTF32_NFKD(0x1D607), { 0x007A } },
	{ UTF32_NFKD(0x1D608), { 0x0041 } },
	{ UTF32_NFKD(0x1D609), { 0x0042 } },
	{ UTF32_NFKD(0x1D60A), { 0x0043 } },
	{ UTF32_NFKD(0x1D60B), { 0x0044 } },
	{ UTF32_NFKD(0x1D60C), { 0x0045 } },
	{ UTF32_NFKD(0x1D60D), { 0x0046 } },
	{ UTF32_NFKD(0x1D60E), { 0x0047 } },
	{ UTF32_NFKD(0x1D60F), { 0x0048 } },
	{ UTF32_NFKD(0x1D610), { 0x0049 } },
	{ UTF32_NFKD(0x1D611), { 0x004A } },
	{ UTF32_NFKD(0x1D612), { 0x004B } },
	{ UTF32_NFKD(0x1D613), { 0x004C } },
	{ UTF32_NFKD(0x1D614), { 0x004D } },
	{ UTF32_NFKD(0x1D615), { 0x004E } },
	{ UTF32_NFKD(0x1D616), { 0x004F } },
	{ UTF32_NFKD(0x1D617), { 0x0050 } },
	{ UTF32_NFKD(0x1D618), { 0x0051 } },
	{ UTF32_NFKD(0x1D619), { 0x0052 } },
	{ UTF32_NFKD(0x1D61A), { 0x0053 } },
	{ UTF32_NFKD(0x1D61B), { 0x0054 } },
	{ UTF32_NFKD(0x1D61C), { 0x0055 } },
	{ UTF32_NFKD(0x1D61D), { 0x0056 } },
	{ UTF32_NFKD(0x1D61E), { 0x0057 } },
	{ UTF32_NFKD(0x1D61F), { 0x0058 } },
	{ UTF32_NFKD(0x1D620), { 0x0059 } },
	{ UTF32_NFKD(0x1D621), { 0x005A } },
	{ UTF32_NFKD(0x1D622), { 0x0061 } },
	{ UTF32_NFKD(0x1D623), { 0x0062 } },
	{ UTF32_NFKD(0x1D624), { 0x0063 } },
	{ UTF32_NFKD(0x1D625), { 0x0064 } },
	{ UTF32_NFKD(0x1D626), { 0x0065 } },
	{ UTF32_NFKD(0x1D627), { 0x0066 } },
	{ UTF32_NFKD(0x1D628), { 0x0067 } },
	{ UTF32_NFKD(0x1D629), { 0x0068 } },
	{ UTF32_NFKD(0x1D62A), { 0x0069 } },
	{ UTF32_NFKD(0x1D62B), { 0x006A } },
	{ UTF32_NFKD(0x1D62C), { 0x006B } },
	{ UTF32_NFKD(0x1D62D), { 0x006C } },
	{ UTF32_NFKD(0x1D62E), { 0x006D } },
	{ UTF32_NFKD(0x1D62F), { 0x006E } },
	{ UTF32_NFKD(0x1D630), { 0x006F } },
	{ UTF32_NFKD(0x1D631), { 0x0070 } },
	{ UTF32_NFKD(0x1D632), { 0x0071 } },
	{ UTF32_NFKD(0x1D633), { 0x0072 } },
	{ UTF32_NFKD(0x1D634), { 0x0073 } },
	{ UTF32_NFKD(0x1D635), { 0x0074 } },
	{ UTF32_NFKD(0x1D636), { 0x0075 } },
	{ UTF32_NFKD(0x1D637), { 0x0076 } },
	{ UTF32_NFKD(0x1D638), { 0x0077 } },
	{ UTF32_NFKD(0x1D639), { 0x0078 } },
	{ UTF32_NFKD(0x1D63A), { 0x0079 } },
	{ UTF32_NFKD(0x1D63B), { 0x007A } },
	{ UTF32_NFKD(0x1D63C), { 0x0041 } },
	{ UTF32_NFKD(0x1D63D), { 0x0042 } },
	{ UTF32_NFKD(0x1D63E), { 0x0043 } },
	{ UTF32_NFKD(0x1D63F), { 0x0044 } },
	{ UTF32_NFKD(0x1D640), { 0x0045 } },
	{ UTF32_NFKD(0x1D641), { 0x0046 } },
	{ UTF32_NFKD(0x1D642), { 0x0047 } },
	{ UTF32_NFKD(0x1D643), { 0x0048 } },
	{ UTF32_NFKD(0x1D644), { 0x0049 } },
	{ UTF32_NFKD(0x1D645), { 0x004A } },
	{ UTF32_NFKD(0x1D646), { 0x004B } },
	{ UTF32_NFKD(0x1D647), { 0x004C } },
	{ UTF32_NFKD(0x1D648), { 0x004D } },
	{ UTF32_NFKD(0x1D649), { 0x004E } },
	{ UTF32_NFKD(0x1D64A), { 0x004F } },
	{ UTF32_NFKD(0x1D64B), { 0x0050 } },
	{ UTF32_NFKD(0x1D64C), { 0x0051 } },
	{ UTF32_NFKD(0x1D64D), { 0x0052 } },
	{ UTF32_NFKD(0x1D64E), { 0x0053 } },
	{ UTF32_NFKD(0x1D64F), { 0x0054 } },
	{ UTF32_NFKD(0x1D650), { 0x0055 } },
	{ UTF32_NFKD(0x1D651), { 0x0056 } },
	{ UTF32_NFKD(0x1D652), { 0x0057 } },
	{ UTF32_NFKD(0x1D653), { 0x0058 } },
	{ UTF32_NFKD(0x1D654), { 0x0059 } },
	{ UTF32_NFKD(0x1D655), { 0x005A } },
	{ UTF32_NFKD(0x1D656), { 0x0061 } },
	{ UTF32_NFKD(0x1D657), { 0x0062 } },
	{ UTF32_NFKD(0x1D658), { 0x0063 } },
	{ UTF32_NFKD(0x1D659), { 0x0064 } },
	{ UTF32_NFKD(0x1D65A), { 0x0065 } },
	{ UTF32_NFKD(0x1D65B), { 0x0066 } },
	{ UTF32_NFKD(0x1D65C), { 0x0067 } },
	{ UTF32_NFKD(0x1D65D), { 0x0068 } },
	{ UTF32_NFKD(0x1D65E), { 0x0069 } },
	{ UTF32_NFKD(0x1D65F), { 0x006A } },
	{ UTF32_NFKD(0x1D660), { 0x006B } },
	{ UTF32_NFKD(0x1D661), { 0x006C } },
	{ UTF32_NFKD(0x1D662), { 0x006D } },
	{ UTF32_NFKD(0x1D663), { 0x006E } },
	{ UTF32_NFKD(0x1D664), { 0x006F } },
	{ UTF32_NFKD(0x1D665), { 0x0070 } },
	{ UTF32_NFKD(0x1D666), { 0x0071 } },
	{ UTF32_NFKD(0x1D667), { 0x0072 } },
	{ UTF32_NFKD(0x1D668), { 0x0073 } },
	{ UTF32_NFKD(0x1D669), { 0x0074 } },
	{ UTF32_NFKD(0x1D66A), { 0x0075 } },
	{ UTF32_NFKD(0x1D66B), { 0x0076 } },
	{ UTF32_NFKD(0x1D66C), { 0x0077 } },
	{ UTF32_NFKD(0x1D66D), { 0x0078 } },
	{ UTF32_NFKD(0x1D66E), { 0x0079 } },
	{ UTF32_NFKD(0x1D66F), { 0x007A } },
	{ UTF32_NFKD(0x1D670), { 0x0041 } },
	{ UTF32_NFKD(0x1D671), { 0x0042 } },
	{ UTF32_NFKD(0x1D672), { 0x0043 } },
	{ UTF32_NFKD(0x1D673), { 0x0044 } },
	{ UTF32_NFKD(0x1D674), { 0x0045 } },
	{ UTF32_NFKD(0x1D675), { 0x0046 } },
	{ UTF32_NFKD(0x1D676), { 0x0047 } },
	{ UTF32_NFKD(0x1D677), { 0x0048 } },
	{ UTF32_NFKD(0x1D678), { 0x0049 } },
	{ UTF32_NFKD(0x1D679), { 0x004A } },
	{ UTF32_NFKD(0x1D67A), { 0x004B } },
	{ UTF32_NFKD(0x1D67B), { 0x004C } },
	{ UTF32_NFKD(0x1D67C), { 0x004D } },
	{ UTF32_NFKD(0x1D67D), { 0x004E } },
	{ UTF32_NFKD(0x1D67E), { 0x004F } },
	{ UTF32_NFKD(0x1D67F), { 0x0050 } },
	{ UTF32_NFKD(0x1D680), { 0x0051 } },
	{ UTF32_NFKD(0x1D681), { 0x0052 } },
	{ UTF32_NFKD(0x1D682), { 0x0053 } },
	{ UTF32_NFKD(0x1D683), { 0x0054 } },
	{ UTF32_NFKD(0x1D684), { 0x0055 } },
	{ UTF32_NFKD(0x1D685), { 0x0056 } },
	{ UTF32_NFKD(0x1D686), { 0x0057 } },
	{ UTF32_NFKD(0x1D687), { 0x0058 } },
	{ UTF32_NFKD(0x1D688), { 0x0059 } },
	{ UTF32_NFKD(0x1D689), { 0x005A } },
	{ UTF32_NFKD(0x1D68A), { 0x0061 } },
	{ UTF32_NFKD(0x1D68B), { 0x0062 } },
	{ UTF32_NFKD(0x1D68C), { 0x0063 } },
	{ UTF32_NFKD(0x1D68D), { 0x0064 } },
	{ UTF32_NFKD(0x1D68E), { 0x0065 } },
	{ UTF32_NFKD(0x1D68F), { 0x0066 } },
	{ UTF32_NFKD(0x1D690), { 0x0067 } },
	{ UTF32_NFKD(0x1D691), { 0x0068 } },
	{ UTF32_NFKD(0x1D692), { 0x0069 } },
	{ UTF32_NFKD(0x1D693), { 0x006A } },
	{ UTF32_NFKD(0x1D694), { 0x006B } },
	{ UTF32_NFKD(0x1D695), { 0x006C } },
	{ UTF32_NFKD(0x1D696), { 0x006D } },
	{ UTF32_NFKD(0x1D697), { 0x006E } },
	{ UTF32_NFKD(0x1D698), { 0x006F } },
	{ UTF32_NFKD(0x1D699), { 0x0070 } },
	{ UTF32_NFKD(0x1D69A), { 0x0071 } },
	{ UTF32_NFKD(0x1D69B), { 0x0072 } },
	{ UTF32_NFKD(0x1D69C), { 0x0073 } },
	{ UTF32_NFKD(0x1D69D), { 0x0074 } },
	{ UTF32_NFKD(0x1D69E), { 0x0075 } },
	{ UTF32_NFKD(0x1D69F), { 0x0076 } },
	{ UTF32_NFKD(0x1D6A0), { 0x0077 } },
	{ UTF32_NFKD(0x1D6A1), { 0x0078 } },
	{ UTF32_NFKD(0x1D6A2), { 0x0079 } },
	{ UTF32_NFKD(0x1D6A3), { 0x007A } },
	{ UTF32_NFKD(0x1D6A4), { 0x0131 } },
	{ UTF32_NFKD(0x1D6A5), { 0x0237 } },
	{ UTF32_NFKD(0x1D6A8), { 0x0391 } },
	{ UTF32_NFKD(0x1D6A9), { 0x0392 } },
	{ UTF32_NFKD(0x1D6AA), { 0x0393 } },
	{ UTF32_NFKD(0x1D6AB), { 0x0394 } },
	{ UTF32_NFKD(0x1D6AC), { 0x0395 } },
	{ UTF32_NFKD(0x1D6AD), { 0x0396 } },
	{ UTF32_NFKD(0x1D6AE), { 0x0397 } },
	{ UTF32_NFKD(0x1D6AF), { 0x0398 } },
	{ UTF32_NFKD(0x1D6B0), { 0x0399 } },
	{ UTF32_NFKD(0x1D6B1), { 0x039A } },
	{ UTF32_NFKD(0x1D6B2), { 0x039B } },
	{ UTF32_NFKD(0x1D6B3), { 0x039C } },
	{ UTF32_NFKD(0x1D6B4), { 0x039D } },
	{ UTF32_NFKD(0x1D6B5), { 0x039E } },
	{ UTF32_NFKD(0x1D6B6), { 0x039F } },
	{ UTF32_NFKD(0x1D6B7), { 0x03A0 } },
	{ UTF32_NFKD(0x1D6B8), { 0x03A1 } },
	{ UTF32_NFKD(0x1D6B9), { 0x03F4 } },
	{ UTF32_NFKD(0x1D6BA), { 0x03A3 } },
	{ UTF32_NFKD(0x1D6BB), { 0x03A4 } },
	{ UTF32_NFKD(0x1D6BC), { 0x03A5 } },
	{ UTF32_NFKD(0x1D6BD), { 0x03A6 } },
	{ UTF32_NFKD(0x1D6BE), { 0x03A7 } },
	{ UTF32_NFKD(0x1D6BF), { 0x03A8 } },
	{ UTF32_NFKD(0x1D6C0), { 0x03A9 } },
	{ UTF32_NFKD(0x1D6C1), { 0x2207 } },
	{ UTF32_NFKD(0x1D6C2), { 0x03B1 } },
	{ UTF32_NFKD(0x1D6C3), { 0x03B2 } },
	{ UTF32_NFKD(0x1D6C4), { 0x03B3 } },
	{ UTF32_NFKD(0x1D6C5), { 0x03B4 } },
	{ UTF32_NFKD(0x1D6C6), { 0x03B5 } },
	{ UTF32_NFKD(0x1D6C7), { 0x03B6 } },
	{ UTF32_NFKD(0x1D6C8), { 0x03B7 } },
	{ UTF32_NFKD(0x1D6C9), { 0x03B8 } },
	{ UTF32_NFKD(0x1D6CA), { 0x03B9 } },
	{ UTF32_NFKD(0x1D6CB), { 0x03BA } },
	{ UTF32_NFKD(0x1D6CC), { 0x03BB } },
	{ UTF32_NFKD(0x1D6CD), { 0x03BC } },
	{ UTF32_NFKD(0x1D6CE), { 0x03BD } },
	{ UTF32_NFKD(0x1D6CF), { 0x03BE } },
	{ UTF32_NFKD(0x1D6D0), { 0x03BF } },
	{ UTF32_NFKD(0x1D6D1), { 0x03C0 } },
	{ UTF32_NFKD(0x1D6D2), { 0x03C1 } },
	{ UTF32_NFKD(0x1D6D3), { 0x03C2 } },
	{ UTF32_NFKD(0x1D6D4), { 0x03C3 } },
	{ UTF32_NFKD(0x1D6D5), { 0x03C4 } },
	{ UTF32_NFKD(0x1D6D6), { 0x03C5 } },
	{ UTF32_NFKD(0x1D6D7), { 0x03C6 } },
	{ UTF32_NFKD(0x1D6D8), { 0x03C7 } },
	{ UTF32_NFKD(0x1D6D9), { 0x03C8 } },
	{ UTF32_NFKD(0x1D6DA), { 0x03C9 } },
	{ UTF32_NFKD(0x1D6DB), { 0x2202 } },
	{ UTF32_NFKD(0x1D6DC), { 0x03F5 } },
	{ UTF32_NFKD(0x1D6DD), { 0x03D1 } },
	{ UTF32_NFKD(0x1D6DE), { 0x03F0 } },
	{ UTF32_NFKD(0x1D6DF), { 0x03D5 } },
	{ UTF32_NFKD(0x1D6E0), { 0x03F1 } },
	{ UTF32_NFKD(0x1D6E1), { 0x03D6 } },
	{ UTF32_NFKD(0x1D6E2), { 0x0391 } },
	{ UTF32_NFKD(0x1D6E3), { 0x0392 } },
	{ UTF32_NFKD(0x1D6E4), { 0x0393 } },
	{ UTF32_NFKD(0x1D6E5), { 0x0394 } },
	{ UTF32_NFKD(0x1D6E6), { 0x0395 } },
	{ UTF32_NFKD(0x1D6E7), { 0x0396 } },
	{ UTF32_NFKD(0x1D6E8), { 0x0397 } },
	{ UTF32_NFKD(0x1D6E9), { 0x0398 } },
	{ UTF32_NFKD(0x1D6EA), { 0x0399 } },
	{ UTF32_NFKD(0x1D6EB), { 0x039A } },
	{ UTF32_NFKD(0x1D6EC), { 0x039B } },
	{ UTF32_NFKD(0x1D6ED), { 0x039C } },
	{ UTF32_NFKD(0x1D6EE), { 0x039D } },
	{ UTF32_NFKD(0x1D6EF), { 0x039E } },
	{ UTF32_NFKD(0x1D6F0), { 0x039F } },
	{ UTF32_NFKD(0x1D6F1), { 0x03A0 } },
	{ UTF32_NFKD(0x1D6F2), { 0x03A1 } },
	{ UTF32_NFKD(0x1D6F3), { 0x03F4 } },
	{ UTF32_NFKD(0x1D6F4), { 0x03A3 } },
	{ UTF32_NFKD(0x1D6F5), { 0x03A4 } },
	{ UTF32_NFKD(0x1D6F6), { 0x03A5 } },
	{ UTF32_NFKD(0x1D6F7), { 0x03A6 } },
	{ UTF32_NFKD(0x1D6F8), { 0x03A7 } },
	{ UTF32_NFKD(0x1D6F9), { 0x03A8 } },
	{ UTF32_NFKD(0x1D6FA), { 0x03A9 } },
	{ UTF32_NFKD(0x1D6FB), { 0x2207 } },
	{ UTF32_NFKD(0x1D6FC), { 0x03B1 } },
	{ UTF32_NFKD(0x1D6FD), { 0x03B2 } },
	{ UTF32_NFKD(0x1D6FE), { 0x03B3 } },
	{ UTF32_NFKD(0x1D6FF), { 0x03B4 } },
	{ UTF32_NFKD(0x1D700), { 0x03B5 } },
	{ UTF32_NFKD(0x1D701), { 0x03B6 } },
	{ UTF32_NFKD(0x1D702), { 0x03B7 } },
	{ UTF32_NFKD(0x1D703), { 0x03B8 } },
	{ UTF32_NFKD(0x1D704), { 0x03B9 } },
	{ UTF32_NFKD(0x1D705), { 0x03BA } },
	{ UTF32_NFKD(0x1D706), { 0x03BB } },
	{ UTF32_NFKD(0x1D707), { 0x03BC } },
	{ UTF32_NFKD(0x1D708), { 0x03BD } },
	{ UTF32_NFKD(0x1D709), { 0x03BE } },
	{ UTF32_NFKD(0x1D70A), { 0x03BF } },
	{ UTF32_NFKD(0x1D70B), { 0x03C0 } },
	{ UTF32_NFKD(0x1D70C), { 0x03C1 } },
	{ UTF32_NFKD(0x1D70D), { 0x03C2 } },
	{ UTF32_NFKD(0x1D70E), { 0x03C3 } },
	{ UTF32_NFKD(0x1D70F), { 0x03C4 } },
	{ UTF32_NFKD(0x1D710), { 0x03C5 } },
	{ UTF32_NFKD(0x1D711), { 0x03C6 } },
	{ UTF32_NFKD(0x1D712), { 0x03C7 } },
	{ UTF32_NFKD(0x1D713), { 0x03C8 } },
	{ UTF32_NFKD(0x1D714), { 0x03C9 } },
	{ UTF32_NFKD(0x1D715), { 0x2202 } },
	{ UTF32_NFKD(0x1D716), { 0x03F5 } },
	{ UTF32_NFKD(0x1D717), { 0x03D1 } },
	{ UTF32_NFKD(0x1D718), { 0x03F0 } },
	{ UTF32_NFKD(0x1D719), { 0x03D5 } },
	{ UTF32_NFKD(0x1D71A), { 0x03F1 } },
	{ UTF32_NFKD(0x1D71B), { 0x03D6 } },
	{ UTF32_NFKD(0x1D71C), { 0x0391 } },
	{ UTF32_NFKD(0x1D71D), { 0x0392 } },
	{ UTF32_NFKD(0x1D71E), { 0x0393 } },
	{ UTF32_NFKD(0x1D71F), { 0x0394 } },
	{ UTF32_NFKD(0x1D720), { 0x0395 } },
	{ UTF32_NFKD(0x1D721), { 0x0396 } },
	{ UTF32_NFKD(0x1D722), { 0x0397 } },
	{ UTF32_NFKD(0x1D723), { 0x0398 } },
	{ UTF32_NFKD(0x1D724), { 0x0399 } },
	{ UTF32_NFKD(0x1D725), { 0x039A } },
	{ UTF32_NFKD(0x1D726), { 0x039B } },
	{ UTF32_NFKD(0x1D727), { 0x039C } },
	{ UTF32_NFKD(0x1D728), { 0x039D } },
	{ UTF32_NFKD(0x1D729), { 0x039E } },
	{ UTF32_NFKD(0x1D72A), { 0x039F } },
	{ UTF32_NFKD(0x1D72B), { 0x03A0 } },
	{ UTF32_NFKD(0x1D72C), { 0x03A1 } },
	{ UTF32_NFKD(0x1D72D), { 0x03F4 } },
	{ UTF32_NFKD(0x1D72E), { 0x03A3 } },
	{ UTF32_NFKD(0x1D72F), { 0x03A4 } },
	{ UTF32_NFKD(0x1D730), { 0x03A5 } },
	{ UTF32_NFKD(0x1D731), { 0x03A6 } },
	{ UTF32_NFKD(0x1D732), { 0x03A7 } },
	{ UTF32_NFKD(0x1D733), { 0x03A8 } },
	{ UTF32_NFKD(0x1D734), { 0x03A9 } },
	{ UTF32_NFKD(0x1D735), { 0x2207 } },
	{ UTF32_NFKD(0x1D736), { 0x03B1 } },
	{ UTF32_NFKD(0x1D737), { 0x03B2 } },
	{ UTF32_NFKD(0x1D738), { 0x03B3 } },
	{ UTF32_NFKD(0x1D739), { 0x03B4 } },
	{ UTF32_NFKD(0x1D73A), { 0x03B5 } },
	{ UTF32_NFKD(0x1D73B), { 0x03B6 } },
	{ UTF32_NFKD(0x1D73C), { 0x03B7 } },
	{ UTF32_NFKD(0x1D73D), { 0x03B8 } },
	{ UTF32_NFKD(0x1D73E), { 0x03B9 } },
	{ UTF32_NFKD(0x1D73F), { 0x03BA } },
	{ UTF32_NFKD(0x1D740), { 0x03BB } },
	{ UTF32_NFKD(0x1D741), { 0x03BC } },
	{ UTF32_NFKD(0x1D742), { 0x03BD } },
	{ UTF32_NFKD(0x1D743), { 0x03BE } },
	{ UTF32_NFKD(0x1D744), { 0x03BF } },
	{ UTF32_NFKD(0x1D745), { 0x03C0 } },
	{ UTF32_NFKD(0x1D746), { 0x03C1 } },
	{ UTF32_NFKD(0x1D747), { 0x03C2 } },
	{ UTF32_NFKD(0x1D748), { 0x03C3 } },
	{ UTF32_NFKD(0x1D749), { 0x03C4 } },
	{ UTF32_NFKD(0x1D74A), { 0x03C5 } },
	{ UTF32_NFKD(0x1D74B), { 0x03C6 } },
	{ UTF32_NFKD(0x1D74C), { 0x03C7 } },
	{ UTF32_NFKD(0x1D74D), { 0x03C8 } },
	{ UTF32_NFKD(0x1D74E), { 0x03C9 } },
	{ UTF32_NFKD(0x1D74F), { 0x2202 } },
	{ UTF32_NFKD(0x1D750), { 0x03F5 } },
	{ UTF32_NFKD(0x1D751), { 0x03D1 } },
	{ UTF32_NFKD(0x1D752), { 0x03F0 } },
	{ UTF32_NFKD(0x1D753), { 0x03D5 } },
	{ UTF32_NFKD(0x1D754), { 0x03F1 } },
	{ UTF32_NFKD(0x1D755), { 0x03D6 } },
	{ UTF32_NFKD(0x1D756), { 0x0391 } },
	{ UTF32_NFKD(0x1D757), { 0x0392 } },
	{ UTF32_NFKD(0x1D758), { 0x0393 } },
	{ UTF32_NFKD(0x1D759), { 0x0394 } },
	{ UTF32_NFKD(0x1D75A), { 0x0395 } },
	{ UTF32_NFKD(0x1D75B), { 0x0396 } },
	{ UTF32_NFKD(0x1D75C), { 0x0397 } },
	{ UTF32_NFKD(0x1D75D), { 0x0398 } },
	{ UTF32_NFKD(0x1D75E), { 0x0399 } },
	{ UTF32_NFKD(0x1D75F), { 0x039A } },
	{ UTF32_NFKD(0x1D760), { 0x039B } },
	{ UTF32_NFKD(0x1D761), { 0x039C } },
	{ UTF32_NFKD(0x1D762), { 0x039D } },
	{ UTF32_NFKD(0x1D763), { 0x039E } },
	{ UTF32_NFKD(0x1D764), { 0x039F } },
	{ UTF32_NFKD(0x1D765), { 0x03A0 } },
	{ UTF32_NFKD(0x1D766), { 0x03A1 } },
	{ UTF32_NFKD(0x1D767), { 0x03F4 } },
	{ UTF32_NFKD(0x1D768), { 0x03A3 } },
	{ UTF32_NFKD(0x1D769), { 0x03A4 } },
	{ UTF32_NFKD(0x1D76A), { 0x03A5 } },
	{ UTF32_NFKD(0x1D76B), { 0x03A6 } },
	{ UTF32_NFKD(0x1D76C), { 0x03A7 } },
	{ UTF32_NFKD(0x1D76D), { 0x03A8 } },
	{ UTF32_NFKD(0x1D76E), { 0x03A9 } },
	{ UTF32_NFKD(0x1D76F), { 0x2207 } },
	{ UTF32_NFKD(0x1D770), { 0x03B1 } },
	{ UTF32_NFKD(0x1D771), { 0x03B2 } },
	{ UTF32_NFKD(0x1D772), { 0x03B3 } },
	{ UTF32_NFKD(0x1D773), { 0x03B4 } },
	{ UTF32_NFKD(0x1D774), { 0x03B5 } },
	{ UTF32_NFKD(0x1D775), { 0x03B6 } },
	{ UTF32_NFKD(0x1D776), { 0x03B7 } },
	{ UTF32_NFKD(0x1D777), { 0x03B8 } },
	{ UTF32_NFKD(0x1D778), { 0x03B9 } },
	{ UTF32_NFKD(0x1D779), { 0x03BA } },
	{ UTF32_NFKD(0x1D77A), { 0x03BB } },
	{ UTF32_NFKD(0x1D77B), { 0x03BC } },
	{ UTF32_NFKD(0x1D77C), { 0x03BD } },
	{ UTF32_NFKD(0x1D77D), { 0x03BE } },
	{ UTF32_NFKD(0x1D77E), { 0x03BF } },
	{ UTF32_NFKD(0x1D77F), { 0x03C0 } },
	{ UTF32_NFKD(0x1D780), { 0x03C1 } },
	{ UTF32_NFKD(0x1D781), { 0x03C2 } },
	{ UTF32_NFKD(0x1D782), { 0x03C3 } },
	{ UTF32_NFKD(0x1D783), { 0x03C4 } },
	{ UTF32_NFKD(0x1D784), { 0x03C5 } },
	{ UTF32_NFKD(0x1D785), { 0x03C6 } },
	{ UTF32_NFKD(0x1D786), { 0x03C7 } },
	{ UTF32_NFKD(0x1D787), { 0x03C8 } },
	{ UTF32_NFKD(0x1D788), { 0x03C9 } },
	{ UTF32_NFKD(0x1D789), { 0x2202 } },
	{ UTF32_NFKD(0x1D78A), { 0x03F5 } },
	{ UTF32_NFKD(0x1D78B), { 0x03D1 } },
	{ UTF32_NFKD(0x1D78C), { 0x03F0 } },
	{ UTF32_NFKD(0x1D78D), { 0x03D5 } },
	{ UTF32_NFKD(0x1D78E), { 0x03F1 } },
	{ UTF32_NFKD(0x1D78F), { 0x03D6 } },
	{ UTF32_NFKD(0x1D790), { 0x0391 } },
	{ UTF32_NFKD(0x1D791), { 0x0392 } },
	{ UTF32_NFKD(0x1D792), { 0x0393 } },
	{ UTF32_NFKD(0x1D793), { 0x0394 } },
	{ UTF32_NFKD(0x1D794), { 0x0395 } },
	{ UTF32_NFKD(0x1D795), { 0x0396 } },
	{ UTF32_NFKD(0x1D796), { 0x0397 } },
	{ UTF32_NFKD(0x1D797), { 0x0398 } },
	{ UTF32_NFKD(0x1D798), { 0x0399 } },
	{ UTF32_NFKD(0x1D799), { 0x039A } },
	{ UTF32_NFKD(0x1D79A), { 0x039B } },
	{ UTF32_NFKD(0x1D79B), { 0x039C } },
	{ UTF32_NFKD(0x1D79C), { 0x039D } },
	{ UTF32_NFKD(0x1D79D), { 0x039E } },
	{ UTF32_NFKD(0x1D79E), { 0x039F } },
	{ UTF32_NFKD(0x1D79F), { 0x03A0 } },
	{ UTF32_NFKD(0x1D7A0), { 0x03A1 } },
	{ UTF32_NFKD(0x1D7A1), { 0x03F4 } },
	{ UTF32_NFKD(0x1D7A2), { 0x03A3 } },
	{ UTF32_NFKD(0x1D7A3), { 0x03A4 } },
	{ UTF32_NFKD(0x1D7A4), { 0x03A5 } },
	{ UTF32_NFKD(0x1D7A5), { 0x03A6 } },
	{ UTF32_NFKD(0x1D7A6), { 0x03A7 } },
	{ UTF32_NFKD(0x1D7A7), { 0x03A8 } },
	{ UTF32_NFKD(0x1D7A8), { 0x03A9 } },
	{ UTF32_NFKD(0x1D7A9), { 0x2207 } },
	{ UTF32_NFKD(0x1D7AA), { 0x03B1 } },
	{ UTF32_NFKD(0x1D7AB), { 0x03B2 } },
	{ UTF32_NFKD(0x1D7AC), { 0x03B3 } },
	{ UTF32_NFKD(0x1D7AD), { 0x03B4 } },
	{ UTF32_NFKD(0x1D7AE), { 0x03B5 } },
	{ UTF32_NFKD(0x1D7AF), { 0x03B6 } },
	{ UTF32_NFKD(0x1D7B0), { 0x03B7 } },
	{ UTF32_NFKD(0x1D7B1), { 0x03B8 } },
	{ UTF32_NFKD(0x1D7B2), { 0x03B9 } },
	{ UTF32_NFKD(0x1D7B3), { 0x03BA } },
	{ UTF32_NFKD(0x1D7B4), { 0x03BB } },
	{ UTF32_NFKD(0x1D7B5), { 0x03BC } },
	{ UTF32_NFKD(0x1D7B6), { 0x03BD } },
	{ UTF32_NFKD(0x1D7B7), { 0x03BE } },
	{ UTF32_NFKD(0x1D7B8), { 0x03BF } },
	{ UTF32_NFKD(0x1D7B9), { 0x03C0 } },
	{ UTF32_NFKD(0x1D7BA), { 0x03C1 } },
	{ UTF32_NFKD(0x1D7BB), { 0x03C2 } },
	{ UTF32_NFKD(0x1D7BC), { 0x03C3 } },
	{ UTF32_NFKD(0x1D7BD), { 0x03C4 } },
	{ UTF32_NFKD(0x1D7BE), { 0x03C5 } },
	{ UTF32_NFKD(0x1D7BF), { 0x03C6 } },
	{ UTF32_NFKD(0x1D7C0), { 0x03C7 } },
	{ UTF32_NFKD(0x1D7C1), { 0x03C8 } },
	{ UTF32_NFKD(0x1D7C2), { 0x03C9 } },
	{ UTF32_NFKD(0x1D7C3), { 0x2202 } },
	{ UTF32_NFKD(0x1D7C4), { 0x03F5 } },
	{ UTF32_NFKD(0x1D7C5), { 0x03D1 } },
	{ UTF32_NFKD(0x1D7C6), { 0x03F0 } },
	{ UTF32_NFKD(0x1D7C7), { 0x03D5 } },
	{ UTF32_NFKD(0x1D7C8), { 0x03F1 } },
	{ UTF32_NFKD(0x1D7C9), { 0x03D6 } },
	{ UTF32_NFKD(0x1D7CE), { 0x0030 } },
	{ UTF32_NFKD(0x1D7CF), { 0x0031 } },
	{ UTF32_NFKD(0x1D7D0), { 0x0032 } },
	{ UTF32_NFKD(0x1D7D1), { 0x0033 } },
	{ UTF32_NFKD(0x1D7D2), { 0x0034 } },
	{ UTF32_NFKD(0x1D7D3), { 0x0035 } },
	{ UTF32_NFKD(0x1D7D4), { 0x0036 } },
	{ UTF32_NFKD(0x1D7D5), { 0x0037 } },
	{ UTF32_NFKD(0x1D7D6), { 0x0038 } },
	{ UTF32_NFKD(0x1D7D7), { 0x0039 } },
	{ UTF32_NFKD(0x1D7D8), { 0x0030 } },
	{ UTF32_NFKD(0x1D7D9), { 0x0031 } },
	{ UTF32_NFKD(0x1D7DA), { 0x0032 } },
	{ UTF32_NFKD(0x1D7DB), { 0x0033 } },
	{ UTF32_NFKD(0x1D7DC), { 0x0034 } },
	{ UTF32_NFKD(0x1D7DD), { 0x0035 } },
	{ UTF32_NFKD(0x1D7DE), { 0x0036 } },
	{ UTF32_NFKD(0x1D7DF), { 0x0037 } },
	{ UTF32_NFKD(0x1D7E0), { 0x0038 } },
	{ UTF32_NFKD(0x1D7E1), { 0x0039 } },
	{ UTF32_NFKD(0x1D7E2), { 0x0030 } },
	{ UTF32_NFKD(0x1D7E3), { 0x0031 } },
	{ UTF32_NFKD(0x1D7E4), { 0x0032 } },
	{ UTF32_NFKD(0x1D7E5), { 0x0033 } },
	{ UTF32_NFKD(0x1D7E6), { 0x0034 } },
	{ UTF32_NFKD(0x1D7E7), { 0x0035 } },
	{ UTF32_NFKD(0x1D7E8), { 0x0036 } },
	{ UTF32_NFKD(0x1D7E9), { 0x0037 } },
	{ UTF32_NFKD(0x1D7EA), { 0x0038 } },
	{ UTF32_NFKD(0x1D7EB), { 0x0039 } },
	{ UTF32_NFKD(0x1D7EC), { 0x0030 } },
	{ UTF32_NFKD(0x1D7ED), { 0x0031 } },
	{ UTF32_NFKD(0x1D7EE), { 0x0032 } },
	{ UTF32_NFKD(0x1D7EF), { 0x0033 } },
	{ UTF32_NFKD(0x1D7F0), { 0x0034 } },
	{ UTF32_NFKD(0x1D7F1), { 0x0035 } },
	{ UTF32_NFKD(0x1D7F2), { 0x0036 } },
	{ UTF32_NFKD(0x1D7F3), { 0x0037 } },
	{ UTF32_NFKD(0x1D7F4), { 0x0038 } },
	{ UTF32_NFKD(0x1D7F5), { 0x0039 } },
	{ UTF32_NFKD(0x1D7F6), { 0x0030 } },
	{ UTF32_NFKD(0x1D7F7), { 0x0031 } },
	{ UTF32_NFKD(0x1D7F8), { 0x0032 } },
	{ UTF32_NFKD(0x1D7F9), { 0x0033 } },
	{ UTF32_NFKD(0x1D7FA), { 0x0034 } },
	{ UTF32_NFKD(0x1D7FB), { 0x0035 } },
	{ UTF32_NFKD(0x1D7FC), { 0x0036 } },
	{ UTF32_NFKD(0x1D7FD), { 0x0037 } },
	{ UTF32_NFKD(0x1D7FE), { 0x0038 } },
	{ UTF32_NFKD(0x1D7FF), { 0x0039 } },
	{ UTF32_NFD(0x2F800), { 0x4E3D } },
	{ UTF32_NFD(0x2F801), { 0x4E38 } },
	{ UTF32_NFD(0x2F802), { 0x4E41 } },
	{ UTF32_NFD(0x2F803), { 0x20122 } },
	{ UTF32_NFD(0x2F804), { 0x4F60 } },
	{ UTF32_NFD(0x2F805), { 0x4FAE } },
	{ UTF32_NFD(0x2F806), { 0x4FBB } },
	{ UTF32_NFD(0x2F807), { 0x5002 } },
	{ UTF32_NFD(0x2F808), { 0x507A } },
	{ UTF32_NFD(0x2F809), { 0x5099 } },
	{ UTF32_NFD(0x2F80A), { 0x50E7 } },
	{ UTF32_NFD(0x2F80B), { 0x50CF } },
	{ UTF32_NFD(0x2F80C), { 0x349E } },
	{ UTF32_NFD(0x2F80D), { 0x2063A } },
	{ UTF32_NFD(0x2F80E), { 0x514D } },
	{ UTF32_NFD(0x2F80F), { 0x5154 } },
	{ UTF32_NFD(0x2F810), { 0x5164 } },
	{ UTF32_NFD(0x2F811), { 0x5177 } },
	{ UTF32_NFD(0x2F812), { 0x2051C } },
	{ UTF32_NFD(0x2F813), { 0x34B9 } },
	{ UTF32_NFD(0x2F814), { 0x5167 } },
	{ UTF32_NFD(0x2F815), { 0x518D } },
	{ UTF32_NFD(0x2F816), { 0x2054B } },
	{ UTF32_NFD(0x2F817), { 0x5197 } },
	{ UTF32_NFD(0x2F818), { 0x51A4 } },
	{ UTF32_NFD(0x2F819), { 0x4ECC } },
	{ UTF32_NFD(0x2F81A), { 0x51AC } },
	{ UTF32_NFD(0x2F81B), { 0x51B5 } },
	{ UTF32_NFD(0x2F81C), { 0x291DF } },
	{ UTF32_NFD(0x2F81D), { 0x51F5 } },
	{ UTF32_NFD(0x2F81E), { 0x5203 } },
	{ UTF32_NFD(0x2F81F), { 0x34DF } },
	{ UTF32_NFD(0x2F820), { 0x523B } },
	{ UTF32_NFD(0x2F821), { 0x5246 } },
	{ UTF32_NFD(0x2F822), { 0x5272 } },
	{ UTF32_NFD(0x2F823), { 0x5277 } },
	{ UTF32_NFD(0x2F824), { 0x3515 } },
	{ UTF32_NFD(0x2F825), { 0x52C7 } },
	{ UTF32_NFD(0x2F826), { 0x52C9 } },
	{ UTF32_NFD(0x2F827), { 0x52E4 } },
	{ UTF32_NFD(0x2F828), { 0x52FA } },
	{ UTF32_NFD(0x2F829), { 0x5305 } },
	{ UTF32_NFD(0x2F82A), { 0x5306 } },
	{ UTF32_NFD(0x2F82B), { 0x5317 } },
	{ UTF32_NFD(0x2F82C), { 0x5349 } },
	{ UTF32_NFD(0x2F82D), { 0x5351 } },
	{ UTF32_NFD(0x2F82E), { 0x535A } },
	{ UTF32_NFD(0x2F82F), { 0x5373 } },
	{ UTF32_NFD(0x2F830), { 0x537D } },
	{ UTF32_NFD(0x2F831), { 0x537F } },
	{ UTF32_NFD(0x2F832), { 0x537F } },
	{ UTF32_NFD(0x2F833), { 0x537F } },
	{ UTF32_NFD(0x2F834), { 0x20A2C } },
	{ UTF32_NFD(0x2F835), { 0x7070 } },
	{ UTF32_NFD(0x2F836), { 0x53CA } },
	{ UTF32_NFD(0x2F837), { 0x53DF } },
	{ UTF32_NFD(0x2F838), { 0x20B63 } },
	{ UTF32_NFD(0x2F839), { 0x53EB } },
	{ UTF32_NFD(0x2F83A), { 0x53F1 } },
	{ UTF32_NFD(0x2F83B), { 0x5406 } },
	{ UTF32_NFD(0x2F83C), { 0x549E } },
	{ UTF32_NFD(0x2F83D), { 0x5438 } },
	{ UTF32_NFD(0x2F83E), { 0x5448 } },
	{ UTF32_NFD(0x2F83F), { 0x5468 } },
	{ UTF32_NFD(0x2F840), { 0x54A2 } },
	{ UTF32_NFD(0x2F841), { 0x54F6 } },
	{ UTF32_NFD(0x2F842), { 0x5510 } },
	{ UTF32_NFD(0x2F843), { 0x5553 } },
	{ UTF32_NFD(0x2F844), { 0x5563 } },
	{ UTF32_NFD(0x2F845), { 0x5584 } },
	{ UTF32_NFD(0x2F846), { 0x5584 } },
	{ UTF32_NFD(0x2F847), { 0x5599 } },
	{ UTF32_NFD(0x2F848), { 0x55AB } },
	{ UTF32_NFD(0x2F849), { 0x55B3 } },
	{ UTF32_NFD(0x2F84A), { 0x55C2 } },
	{ UTF32_NFD(0x2F84B), { 0x5716 } },
	{ UTF32_NFD(0x2F84C), { 0x5606 } },
	{ UTF32_NFD(0x2F84D), { 0x5717 } },
	{ UTF32_NFD(0x2F84E), { 0x5651 } },
	{ UTF32_NFD(0x2F84F), { 0x5674 } },
	{ UTF32_NFD(0x2F850), { 0x5207 } },
	{ UTF32_NFD(0x2F851), { 0x58EE } },
	{ UTF32_NFD(0x2F852), { 0x57CE } },
	{ UTF32_NFD(0x2F853), { 0x57F4 } },
	{ UTF32_NFD(0x2F854), { 0x580D } },
	{ UTF32_NFD(0x2F855), { 0x578B } },
	{ UTF32_NFD(0x2F856), { 0x5832 } },
	{ UTF32_NFD(0x2F857), { 0x5831 } },
	{ UTF32_NFD(0x2F858), { 0x58AC } },
	{ UTF32_NFD(0x2F859), { 0x214E4 } },
	{ UTF32_NFD(0x2F85A), { 0x58F2 } },
	{ UTF32_NFD(0x2F85B), { 0x58F7 } },
	{ UTF32_NFD(0x2F85C), { 0x5906 } },
	{ UTF32_NFD(0x2F85D), { 0x591A } },
	{ UTF32_NFD(0x2F85E), { 0x5922 } },
	{ UTF32_NFD(0x2F85F), { 0x5962 } },
	{ UTF32_NFD(0x2F860), { 0x216A8 } },
	{ UTF32_NFD(0x2F861), { 0x216EA } },
	{ UTF32_NFD(0x2F862), { 0x59EC } },
	{ UTF32_NFD(0x2F863), { 0x5A1B } },
	{ UTF32_NFD(0x2F864), { 0x5A27 } },
	{ UTF32_NFD(0x2F865), { 0x59D8 } },
	{ UTF32_NFD(0x2F866), { 0x5A66 } },
	{ UTF32_NFD(0x2F867), { 0x36EE } },
	{ UTF32_NFD(0x2F868), { 0x36FC } },
	{ UTF32_NFD(0x2F869), { 0x5B08 } },
	{ UTF32_NFD(0x2F86A), { 0x5B3E } },
	{ UTF32_NFD(0x2F86B), { 0x5B3E } },
	{ UTF32_NFD(0x2F86C), { 0x219C8 } },
	{ UTF32_NFD(0x2F86D), { 0x5BC3 } },
	{ UTF32_NFD(0x2F86E), { 0x5BD8 } },
	{ UTF32_NFD(0x2F86F), { 0x5BE7 } },
	{ UTF32_NFD(0x2F870), { 0x5BF3 } },
	{ UTF32_NFD(0x2F871), { 0x21B18 } },
	{ UTF32_NFD(0x2F872), { 0x5BFF } },
	{ UTF32_NFD(0x2F873), { 0x5C06 } },
	{ UTF32_NFD(0x2F874), { 0x5F53 } },
	{ UTF32_NFD(0x2F875), { 0x5C22 } },
	{ UTF32_NFD(0x2F876), { 0x3781 } },
	{ UTF32_NFD(0x2F877), { 0x5C60 } },
	{ UTF32_NFD(0x2F878), { 0x5C6E } },
	{ UTF32_NFD(0x2F879), { 0x5CC0 } },
	{ UTF32_NFD(0x2F87A), { 0x5C8D } },
	{ UTF32_NFD(0x2F87B), { 0x21DE4 } },
	{ UTF32_NFD(0x2F87C), { 0x5D43 } },
	{ UTF32_NFD(0x2F87D), { 0x21DE6 } },
	{ UTF32_NFD(0x2F87E), { 0x5D6E } },
	{ UTF32_NFD(0x2F87F), { 0x5D6B } },
	{ UTF32_NFD(0x2F880), { 0x5D7C } },
	{ UTF32_NFD(0x2F881), { 0x5DE1 } },
	{ UTF32_NFD(0x2F882), { 0x5DE2 } },
	{ UTF32_NFD(0x2F883), { 0x382F } },
	{ UTF32_NFD(0x2F884), { 0x5DFD } },
	{ UTF32_NFD(0x2F885), { 0x5E28 } },
	{ UTF32_NFD(0x2F886), { 0x5E3D } },
	{ UTF32_NFD(0x2F887), { 0x5E69 } },
	{ UTF32_NFD(0x2F888), { 0x3862 } },
	{ UTF32_NFD(0x2F889), { 0x22183 } },
	{ UTF32_NFD(0x2F88A), { 0x387C } },
	{ UTF32_NFD(0x2F88B), { 0x5EB0 } },
	{ UTF32_NFD(0x2F88C), { 0x5EB3 } },
	{ UTF32_NFD(0x2F88D), { 0x5EB6 } },
	{ UTF32_NFD(0x2F88E), { 0x5ECA } },
	{ UTF32_NFD(0x2F88F), { 0x2A392 } },
	{ UTF32_NFD(0x2F890), { 0x5EFE } },
	{ UTF32_NFD(0x2F891), { 0x22331 } },
	{ UTF32_NFD(0x2F892), { 0x22331 } },
	{ UTF32_NFD(0x2F893), { 0x8201 } },
	{ UTF32_NFD(0x2F894), { 0x5F22 } },
	{ UTF32_NFD(0x2F895), { 0x5F22 } },
	{ UTF32_NFD(0x2F896), { 0x38C7 } },
	{ UTF32_NFD(0x2F897), { 0x232B8 } },
	{ UTF32_NFD(0x2F898), { 0x261DA } },
	{ UTF32_NFD(0x2F899), { 0x5F62 } },
	{ UTF32_NFD(0x2F89A), { 0x5F6B } },
	{ UTF32_NFD(0x2F89B), { 0x38E3 } },
	{ UTF32_NFD(0x2F89C), { 0x5F9A } },
	{ UTF32_NFD(0x2F89D), { 0x5FCD } },
	{ UTF32_NFD(0x2F89E), { 0x5FD7 } },
	{ UTF32_NFD(0x2F89F), { 0x5FF9 } },
	{ UTF32_NFD(0x2F8A0), { 0x6081 } },
	{ UTF32_NFD(0x2F8A1), { 0x393A } },
	{ UTF32_NFD(0x2F8A2), { 0x391C } },
	{ UTF32_NFD(0x2F8A3), { 0x6094 } },
	{ UTF32_NFD(0x2F8A4), { 0x226D4 } },
	{ UTF32_NFD(0x2F8A5), { 0x60C7 } },
	{ UTF32_NFD(0x2F8A6), { 0x6148 } },
	{ UTF32_NFD(0x2F8A7), { 0x614C } },
	{ UTF32_NFD(0x2F8A8), { 0x614E } },
	{ UTF32_NFD(0x2F8A9), { 0x614C } },
	{ UTF32_NFD(0x2F8AA), { 0x617A } },
	{ UTF32_NFD(0x2F8AB), { 0x618E } },
	{ UTF32_NFD(0x2F8AC), { 0x61B2 } },
	{ UTF32_NFD(0x2F8AD), { 0x61A4 } },
	{ UTF32_NFD(0x2F8AE), { 0x61AF } },
	{ UTF32_NFD(0x2F8AF), { 0x61DE } },
	{ UTF32_NFD(0x2F8B0), { 0x61F2 } },
	{ UTF32_NFD(0x2F8B1), { 0x61F6 } },
	{ UTF32_NFD(0x2F8B2), { 0x6210 } },
	{ UTF32_NFD(0x2F8B3), { 0x621B } },
	{ UTF32_NFD(0x2F8B4), { 0x625D } },
	{ UTF32_NFD(0x2F8B5), { 0x62B1 } },
	{ UTF32_NFD(0x2F8B6), { 0x62D4 } },
	{ UTF32_NFD(0x2F8B7), { 0x6350 } },
	{ UTF32_NFD(0x2F8B8), { 0x22B0C } },
	{ UTF32_NFD(0x2F8B9), { 0x633D } },
	{ UTF32_NFD(0x2F8BA), { 0x62FC } },
	{ UTF32_NFD(0x2F8BB), { 0x6368 } },
	{ UTF32_NFD(0x2F8BC), { 0x6383 } },
	{ UTF32_NFD(0x2F8BD), { 0x63E4 } },
	{ UTF32_NFD(0x2F8BE), { 0x22BF1 } },
	{ UTF32_NFD(0x2F8BF), { 0x6422 } },
	{ UTF32_NFD(0x2F8C0), { 0x63C5 } },
	{ UTF32_NFD(0x2F8C1), { 0x63A9 } },
	{ UTF32_NFD(0x2F8C2), { 0x3A2E } },
	{ UTF32_NFD(0x2F8C3), { 0x6469 } },
	{ UTF32_NFD(0x2F8C4), { 0x647E } },
	{ UTF32_NFD(0x2F8C5), { 0x649D } },
	{ UTF32_NFD(0x2F8C6), { 0x6477 } },
	{ UTF32_NFD(0x2F8C7), { 0x3A6C } },
	{ UTF32_NFD(0x2F8C8), { 0x654F } },
	{ UTF32_NFD(0x2F8C9), { 0x656C } },
	{ UTF32_NFD(0x2F8CA), { 0x2300A } },
	{ UTF32_NFD(0x2F8CB), { 0x65E3 } },
	{ UTF32_NFD(0x2F8CC), { 0x66F8 } },
	{ UTF32_NFD(0x2F8CD), { 0x6649 } },
	{ UTF32_NFD(0x2F8CE), { 0x3B19 } },
	{ UTF32_NFD(0x2F8CF), { 0x6691 } },
	{ UTF32_NFD(0x2F8D0), { 0x3B08 } },
	{ UTF32_NFD(0x2F8D1), { 0x3AE4 } },
	{ UTF32_NFD(0x2F8D2), { 0x5192 } },
	{ UTF32_NFD(0x2F8D3), { 0x5195 } },
	{ UTF32_NFD(0x2F8D4), { 0x6700 } },
	{ UTF32_NFD(0x2F8D5), { 0x669C } },
	{ UTF32_NFD(0x2F8D6), { 0x80AD } },
	{ UTF32_NFD(0x2F8D7), { 0x43D9 } },
	{ UTF32_NFD(0x2F8D8), { 0x6717 } },
	{ UTF32_NFD(0x2F8D9), { 0x671B } },
	{ UTF32_NFD(0x2F8DA), { 0x6721 } },
	{ UTF32_NFD(0x2F8DB), { 0x675E } },
	{ UTF32_NFD(0x2F8DC), { 0x6753 } },
	{ UTF32_NFD(0x2F8DD), { 0x233C3 } },
	{ UTF32_NFD(0x2F8DE), { 0x3B49 } },
	{ UTF32_NFD(0x2F8DF), { 0x67FA } },
	{ UTF32_NFD(0x2F8E0), { 0x6785 } },
	{ UTF32_NFD(0x2F8E1), { 0x6852 } },
	{ UTF32_NFD(0x2F8E2), { 0x6885 } },
	{ UTF32_NFD(0x2F8E3), { 0x2346D } },
	{ UTF32_NFD(0x2F8E4), { 0x688E } },
	{ UTF32_NFD(0x2F8E5), { 0x681F } },
	{ UTF32_NFD(0x2F8E6), { 0x6914 } },
	{ UTF32_NFD(0x2F8E7), { 0x3B9D } },
	{ UTF32_NFD(0x2F8E8), { 0x6942 } },
	{ UTF32_NFD(0x2F8E9), { 0x69A3 } },
	{ UTF32_NFD(0x2F8EA), { 0x69EA } },
	{ UTF32_NFD(0x2F8EB), { 0x6AA8 } },
	{ UTF32_NFD(0x2F8EC), { 0x236A3 } },
	{ UTF32_NFD(0x2F8ED), { 0x6ADB } },
	{ UTF32_NFD(0x2F8EE), { 0x3C18 } },
	{ UTF32_NFD(0x2F8EF), { 0x6B21 } },
	{ UTF32_NFD(0x2F8F0), { 0x238A7 } },
	{ UTF32_NFD(0x2F8F1), { 0x6B54 } },
	{ UTF32_NFD(0x2F8F2), { 0x3C4E } },
	{ UTF32_NFD(0x2F8F3), { 0x6B72 } },
	{ UTF32_NFD(0x2F8F4), { 0x6B9F } },
	{ UTF32_NFD(0x2F8F5), { 0x6BBA } },
	{ UTF32_NFD(0x2F8F6), { 0x6BBB } },
	{ UTF32_NFD(0x2F8F7), { 0x23A8D } },
	{ UTF32_NFD(0x2F8F8), { 0x21D0B } },
	{ UTF32_NFD(0x2F8F9), { 0x23AFA } },
	{ UTF32_NFD(0x2F8FA), { 0x6C4E } },
	{ UTF32_NFD(0x2F8FB), { 0x23CBC } },
	{ UTF32_NFD(0x2F8FC), { 0x6CBF } },
	{ UTF32_NFD(0x2F8FD), { 0x6CCD } },
	{ UTF32_NFD(0x2F8FE), { 0x6C67 } },
	{ UTF32_NFD(0x2F8FF), { 0x6D16 } },
	{ UTF32_NFD(0x2F900), { 0x6D3E } },
	{ UTF32_NFD(0x2F901), { 0x6D77 } },
	{ UTF32_NFD(0x2F902), { 0x6D41 } },
	{ UTF32_NFD(0x2F903), { 0x6D69 } },
	{ UTF32_NFD(0x2F904), { 0x6D78 } },
	{ UTF32_NFD(0x2F905), { 0x6D85 } },
	{ UTF32_NFD(0x2F906), { 0x23D1E } },
	{ UTF32_NFD(0x2F907), { 0x6D34 } },
	{ UTF32_NFD(0x2F908), { 0x6E2F } },
	{ UTF32_NFD(0x2F909), { 0x6E6E } },
	{ UTF32_NFD(0x2F90A), { 0x3D33 } },
	{ UTF32_NFD(0x2F90B), { 0x6ECB } },
	{ UTF32_NFD(0x2F90C), { 0x6EC7 } },
	{ UTF32_NFD(0x2F90D), { 0x23ED1 } },
	{ UTF32_NFD(0x2F90E), { 0x6DF9 } },
	{ UTF32_NFD(0x2F90F), { 0x6F6E } },
	{ UTF32_NFD(0x2F910), { 0x23F5E } },
	{ UTF32_NFD(0x2F911), { 0x23F8E } },
	{ UTF32_NFD(0x2F912), { 0x6FC6 } },
	{ UTF32_NFD(0x2F913), { 0x7039 } },
	{ UTF32_NFD(0x2F914), { 0x701E } },
	{ UTF32_NFD(0x2F915), { 0x701B } },
	{ UTF32_NFD(0x2F916), { 0x3D96 } },
	{ UTF32_NFD(0x2F917), { 0x704A } },
	{ UTF32_NFD(0x2F918), { 0x707D } },
	{ UTF32_NFD(0x2F919), { 0x7077 } },
	{ UTF32_NFD(0x2F91A), { 0x70AD } },
	{ UTF32_NFD(0x2F91B), { 0x20525 } },
	{ UTF32_NFD(0x2F91C), { 0x7145 } },
	{ UTF32_NFD(0x2F91D), { 0x24263 } },
	{ UTF32_NFD(0x2F91E), { 0x719C } },
	{ UTF32_NFD(0x2F91F), { 0x243AB } },
	{ UTF32_NFD(0x2F920), { 0x7228 } },
	{ UTF32_NFD(0x2F921), { 0x7235 } },
	{ UTF32_NFD(0x2F922), { 0x7250 } },
	{ UTF32_NFD(0x2F923), { 0x24608 } },
	{ UTF32_NFD(0x2F924), { 0x7280 } },
	{ UTF32_NFD(0x2F925), { 0x7295 } },
	{ UTF32_NFD(0x2F926), { 0x24735 } },
	{ UTF32_NFD(0x2F927), { 0x24814 } },
	{ UTF32_NFD(0x2F928), { 0x737A } },
	{ UTF32_NFD(0x2F929), { 0x738B } },
	{ UTF32_NFD(0x2F92A), { 0x3EAC } },
	{ UTF32_NFD(0x2F92B), { 0x73A5 } },
	{ UTF32_NFD(0x2F92C), { 0x3EB8 } },
	{ UTF32_NFD(0x2F92D), { 0x3EB8 } },
	{ UTF32_NFD(0x2F92E), { 0x7447 } },
	{ UTF32_NFD(0x2F92F), { 0x745C } },
	{ UTF32_NFD(0x2F930), { 0x7471 } },
	{ UTF32_NFD(0x2F931), { 0x7485 } },
	{ UTF32_NFD(0x2F932), { 0x74CA } },
	{ UTF32_NFD(0x2F933), { 0x3F1B } },
	{ UTF32_NFD(0x2F934), { 0x7524 } },
	{ UTF32_NFD(0x2F935), { 0x24C36 } },
	{ UTF32_NFD(0x2F936), { 0x753E } },
	{ UTF32_NFD(0x2F937), { 0x24C92 } },
	{ UTF32_NFD(0x2F938), { 0x7570 } },
	{ UTF32_NFD(0x2F939), { 0x2219F } },
	{ UTF32_NFD(0x2F93A), { 0x7610 } },
	{ UTF32_NFD(0x2F93B), { 0x24FA1 } },
	{ UTF32_NFD(0x2F93C), { 0x24FB8 } },
	{ UTF32_NFD(0x2F93D), { 0x25044 } },
	{ UTF32_NFD(0x2F93E), { 0x3FFC } },
	{ UTF32_NFD(0x2F93F), { 0x4008 } },
	{ UTF32_NFD(0x2F940), { 0x76F4 } },
	{ UTF32_NFD(0x2F941), { 0x250F3 } },
	{ UTF32_NFD(0x2F942), { 0x250F2 } },
	{ UTF32_NFD(0x2F943), { 0x25119 } },
	{ UTF32_NFD(0x2F944), { 0x25133 } },
	{ UTF32_NFD(0x2F945), { 0x771E } },
	{ UTF32_NFD(0x2F946), { 0x771F } },
	{ UTF32_NFD(0x2F947), { 0x771F } },
	{ UTF32_NFD(0x2F948), { 0x774A } },
	{ UTF32_NFD(0x2F949), { 0x4039 } },
	{ UTF32_NFD(0x2F94A), { 0x778B } },
	{ UTF32_NFD(0x2F94B), { 0x4046 } },
	{ UTF32_NFD(0x2F94C), { 0x4096 } },
	{ UTF32_NFD(0x2F94D), { 0x2541D } },
	{ UTF32_NFD(0x2F94E), { 0x784E } },
	{ UTF32_NFD(0x2F94F), { 0x788C } },
	{ UTF32_NFD(0x2F950), { 0x78CC } },
	{ UTF32_NFD(0x2F951), { 0x40E3 } },
	{ UTF32_NFD(0x2F952), { 0x25626 } },
	{ UTF32_NFD(0x2F953), { 0x7956 } },
	{ UTF32_NFD(0x2F954), { 0x2569A } },
	{ UTF32_NFD(0x2F955), { 0x256C5 } },
	{ UTF32_NFD(0x2F956), { 0x798F } },
	{ UTF32_NFD(0x2F957), { 0x79EB } },
	{ UTF32_NFD(0x2F958), { 0x412F } },
	{ UTF32_NFD(0x2F959), { 0x7A40 } },
	{ UTF32_NFD(0x2F95A), { 0x7A4A } },
	{ UTF32_NFD(0x2F95B), { 0x7A4F } },
	{ UTF32_NFD(0x2F95C), { 0x2597C } },
	{ UTF32_NFD(0x2F95D), { 0x25AA7 } },
	{ UTF32_NFD(0x2F95E), { 0x25AA7 } },
	{ UTF32_NFD(0x2F95F), { 0x7AEE } },
	{ UTF32_NFD(0x2F960), { 0x4202 } },
	{ UTF32_NFD(0x2F961), { 0x25BAB } },
	{ UTF32_NFD(0x2F962), { 0x7BC6 } },
	{ UTF32_NFD(0x2F963), { 0x7BC9 } },
	{ UTF32_NFD(0x2F964), { 0x4227 } },
	{ UTF32_NFD(0x2F965), { 0x25C80 } },
	{ UTF32_NFD(0x2F966), { 0x7CD2 } },
	{ UTF32_NFD(0x2F967), { 0x42A0 } },
	{ UTF32_NFD(0x2F968), { 0x7CE8 } },
	{ UTF32_NFD(0x2F969), { 0x7CE3 } },
	{ UTF32_NFD(0x2F96A), { 0x7D00 } },
	{ UTF32_NFD(0x2F96B), { 0x25F86 } },
	{ UTF32_NFD(0x2F96C), { 0x7D63 } },
	{ UTF32_NFD(0x2F96D), { 0x4301 } },
	{ UTF32_NFD(0x2F96E), { 0x7DC7 } },
	{ UTF32_NFD(0x2F96F), { 0x7E02 } },
	{ UTF32_NFD(0x2F970), { 0x7E45 } },
	{ UTF32_NFD(0x2F971), { 0x4334 } },
	{ UTF32_NFD(0x2F972), { 0x26228 } },
	{ UTF32_NFD(0x2F973), { 0x26247 } },
	{ UTF32_NFD(0x2F974), { 0x4359 } },
	{ UTF32_NFD(0x2F975), { 0x262D9 } },
	{ UTF32_NFD(0x2F976), { 0x7F7A } },
	{ UTF32_NFD(0x2F977), { 0x2633E } },
	{ UTF32_NFD(0x2F978), { 0x7F95 } },
	{ UTF32_NFD(0x2F979), { 0x7FFA } },
	{ UTF32_NFD(0x2F97A), { 0x8005 } },
	{ UTF32_NFD(0x2F97B), { 0x264DA } },
	{ UTF32_NFD(0x2F97C), { 0x26523 } },
	{ UTF32_NFD(0x2F97D), { 0x8060 } },
	{ UTF32_NFD(0x2F97E), { 0x265A8 } },
	{ UTF32_NFD(0x2F97F), { 0x8070 } },
	{ UTF32_NFD(0x2F980), { 0x2335F } },
	{ UTF32_NFD(0x2F981), { 0x43D5 } },
	{ UTF32_NFD(0x2F982), { 0x80B2 } },
	{ UTF32_NFD(0x2F983), { 0x8103 } },
	{ UTF32_NFD(0x2F984), { 0x440B } },
	{ UTF32_NFD(0x2F985), { 0x813E } },
	{ UTF32_NFD(0x2F986), { 0x5AB5 } },
	{ UTF32_NFD(0x2F987), { 0x267A7 } },
	{ UTF32_NFD(0x2F988), { 0x267B5 } },
	{ UTF32_NFD(0x2F989), { 0x23393 } },
	{ UTF32_NFD(0x2F98A), { 0x2339C } },
	{ UTF32_NFD(0x2F98B), { 0x8201 } },
	{ UTF32_NFD(0x2F98C), { 0x8204 } },
	{ UTF32_NFD(0x2F98D), { 0x8F9E } },
	{ UTF32_NFD(0x2F98E), { 0x446B } },
	{ UTF32_NFD(0x2F98F), { 0x8291 } },
	{ UTF32_NFD(0x2F990), { 0x828B } },
	{ UTF32_NFD(0x2F991), { 0x829D } },
	{ UTF32_NFD(0x2F992), { 0x52B3 } },
	{ UTF32_NFD(0x2F993), { 0x82B1 } },
	{ UTF32_NFD(0x2F994), { 0x82B3 } },
	{ UTF32_NFD(0x2F995), { 0x82BD } },
	{ UTF32_NFD(0x2F996), { 0x82E6 } },
	{ UTF32_NFD(0x2F997), { 0x26B3C } },
	{ UTF32_NFD(0x2F998), { 0x82E5 } },
	{ UTF32_NFD(0x2F999), { 0x831D } },
	{ UTF32_NFD(0x2F99A), { 0x8363 } },
	{ UTF32_NFD(0x2F99B), { 0x83AD } },
	{ UTF32_NFD(0x2F99C), { 0x8323 } },
	{ UTF32_NFD(0x2F99D), { 0x83BD } },
	{ UTF32_NFD(0x2F99E), { 0x83E7 } },
	{ UTF32_NFD(0x2F99F), { 0x8457 } },
	{ UTF32_NFD(0x2F9A0), { 0x8353 } },
	{ UTF32_NFD(0x2F9A1), { 0x83CA } },
	{ UTF32_NFD(0x2F9A2), { 0x83CC } },
	{ UTF32_NFD(0x2F9A3), { 0x83DC } },
	{ UTF32_NFD(0x2F9A4), { 0x26C36 } },
	{ UTF32_NFD(0x2F9A5), { 0x26D6B } },
	{ UTF32_NFD(0x2F9A6), { 0x26CD5 } },
	{ UTF32_NFD(0x2F9A7), { 0x452B } },
	{ UTF32_NFD(0x2F9A8), { 0x84F1 } },
	{ UTF32_NFD(0x2F9A9), { 0x84F3 } },
	{ UTF32_NFD(0x2F9AA), { 0x8516 } },
	{ UTF32_NFD(0x2F9AB), { 0x273CA } },
	{ UTF32_NFD(0x2F9AC), { 0x8564 } },
	{ UTF32_NFD(0x2F9AD), { 0x26F2C } },
	{ UTF32_NFD(0x2F9AE), { 0x455D } },
	{ UTF32_NFD(0x2F9AF), { 0x4561 } },
	{ UTF32_NFD(0x2F9B0), { 0x26FB1 } },
	{ UTF32_NFD(0x2F9B1), { 0x270D2 } },
	{ UTF32_NFD(0x2F9B2), { 0x456B } },
	{ UTF32_NFD(0x2F9B3), { 0x8650 } },
	{ UTF32_NFD(0x2F9B4), { 0x865C } },
	{ UTF32_NFD(0x2F9B5), { 0x8667 } },
	{ UTF32_NFD(0x2F9B6), { 0x8669 } },
	{ UTF32_NFD(0x2F9B7), { 0x86A9 } },
	{ UTF32_NFD(0x2F9B8), { 0x8688 } },
	{ UTF32_NFD(0x2F9B9), { 0x870E } },
	{ UTF32_NFD(0x2F9BA), { 0x86E2 } },
	{ UTF32_NFD(0x2F9BB), { 0x8779 } },
	{ UTF32_NFD(0x2F9BC), { 0x8728 } },
	{ UTF32_NFD(0x2F9BD), { 0x876B } },
	{ UTF32_NFD(0x2F9BE), { 0x8786 } },
	{ UTF32_NFD(0x2F9BF), { 0x45D7 } },
	{ UTF32_NFD(0x2F9C0), { 0x87E1 } },
	{ UTF32_NFD(0x2F9C1), { 0x8801 } },
	{ UTF32_NFD(0x2F9C2), { 0x45F9 } },
	{ UTF32_NFD(0x2F9C3), { 0x8860 } },
	{ UTF32_NFD(0x2F9C4), { 0x8863 } },
	{ UTF32_NFD(0x2F9C5), { 0x27667 } },
	{ UTF32_NFD(0x2F9C6), { 0x88D7 } },
	{ UTF32_NFD(0x2F9C7), { 0x88DE } },
	{ UTF32_NFD(0x2F9C8), { 0x4635 } },
	{ UTF32_NFD(0x2F9C9), { 0x88FA } },
	{ UTF32_NFD(0x2F9CA), { 0x34BB } },
	{ UTF32_NFD(0x2F9CB), { 0x278AE } },
	{ UTF32_NFD(0x2F9CC), { 0x27966 } },
	{ UTF32_NFD(0x2F9CD), { 0x46BE } },
	{ UTF32_NFD(0x2F9CE), { 0x46C7 } },
	{ UTF32_NFD(0x2F9CF), { 0x8AA0 } },
	{ UTF32_NFD(0x2F9D0), { 0x8AED } },
	{ UTF32_NFD(0x2F9D1), { 0x8B8A } },
	{ UTF32_NFD(0x2F9D2), { 0x8C55 } },
	{ UTF32_NFD(0x2F9D3), { 0x27CA8 } },
	{ UTF32_NFD(0x2F9D4), { 0x8CAB } },
	{ UTF32_NFD(0x2F9D5), { 0x8CC1 } },
	{ UTF32_NFD(0x2F9D6), { 0x8D1B } },
	{ UTF32_NFD(0x2F9D7), { 0x8D77 } },
	{ UTF32_NFD(0x2F9D8), { 0x27F2F } },
	{ UTF32_NFD(0x2F9D9), { 0x20804 } },
	{ UTF32_NFD(0x2F9DA), { 0x8DCB } },
	{ UTF32_NFD(0x2F9DB), { 0x8DBC } },
	{ UTF32_NFD(0x2F9DC), { 0x8DF0 } },
	{ UTF32_NFD(0x2F9DD), { 0x208DE } },
	{ UTF32_NFD(0x2F9DE), { 0x8ED4 } },
	{ UTF32_NFD(0x2F9DF), { 0x8F38 } },
	{ UTF32_NFD(0x2F9E0), { 0x285D2 } },
	{ UTF32_NFD(0x2F9E1), { 0x285ED } },
	{ UTF32_NFD(0x2F9E2), { 0x9094 } },
	{ UTF32_NFD(0x2F9E3), { 0x90F1 } },
	{ UTF32_NFD(0x2F9E4), { 0x9111 } },
	{ UTF32_NFD(0x2F9E5), { 0x2872E } },
	{ UTF32_NFD(0x2F9E6), { 0x911B } },
	{ UTF32_NFD(0x2F9E7), { 0x9238 } },
	{ UTF32_NFD(0x2F9E8), { 0x92D7 } },
	{ UTF32_NFD(0x2F9E9), { 0x92D8 } },
	{ UTF32_NFD(0x2F9EA), { 0x927C } },
	{ UTF32_NFD(0x2F9EB), { 0x93F9 } },
	{ UTF32_NFD(0x2F9EC), { 0x9415 } },
	{ UTF32_NFD(0x2F9ED), { 0x28BFA } },
	{ UTF32_NFD(0x2F9EE), { 0x958B } },
	{ UTF32_NFD(0x2F9EF), { 0x4995 } },
	{ UTF32_NFD(0x2F9F0), { 0x95B7 } },
	{ UTF32_NFD(0x2F9F1), { 0x28D77 } },
	{ UTF32_NFD(0x2F9F2), { 0x49E6 } },
	{ UTF32_NFD(0x2F9F3), { 0x96C3 } },
	{ UTF32_NFD(0x2F9F4), { 0x5DB2 } },
	{ UTF32_NFD(0x2F9F5), { 0x9723 } },
	{ UTF32_NFD(0x2F9F6), { 0x29145 } },
	{ UTF32_NFD(0x2F9F7), { 0x2921A } },
	{ UTF32_NFD(0x2F9F8), { 0x4A6E } },
	{ UTF32_NFD(0x2F9F9), { 0x4A76 } },
	{ UTF32_NFD(0x2F9FA), { 0x97E0 } },
	{ UTF32_NFD(0x2F9FB), { 0x2940A } },
	{ UTF32_NFD(0x2F9FC), { 0x4AB2 } },
	{ UTF32_NFD(0x2F9FD), { 0x29496 } },
	{ UTF32_NFD(0x2F9FE), { 0x980B } },
	{ UTF32_NFD(0x2F9FF), { 0x980B } },
	{ UTF32_NFD(0x2FA00), { 0x9829 } },
	{ UTF32_NFD(0x2FA01), { 0x295B6 } },
	{ UTF32_NFD(0x2FA02), { 0x98E2 } },
	{ UTF32_NFD(0x2FA03), { 0x4B33 } },
	{ UTF32_NFD(0x2FA04), { 0x9929 } },
	{ UTF32_NFD(0x2FA05), { 0x99A7 } },
	{ UTF32_NFD(0x2FA06), { 0x99C2 } },
	{ UTF32_NFD(0x2FA07), { 0x99FE } },
	{ UTF32_NFD(0x2FA08), { 0x4BCE } },
	{ UTF32_NFD(0x2FA09), { 0x29B30 } },
	{ UTF32_NFD(0x2FA0A), { 0x9B12 } },
	{ UTF32_NFD(0x2FA0B), { 0x9C40 } },
	{ UTF32_NFD(0x2FA0C), { 0x9CFD } },
	{ UTF32_NFD(0x2FA0D), { 0x4CCE } },
	{ UTF32_NFD(0x2FA0E), { 0x4CED } },
	{ UTF32_NFD(0x2FA0F), { 0x9D67 } },
	{ UTF32_NFD(0x2FA10), { 0x2A0CE } },
	{ UTF32_NFD(0x2FA11), { 0x4CF8 } },
	{ UTF32_NFD(0x2FA12), { 0x2A105 } },
	{ UTF32_NFD(0x2FA13), { 0x2A20E } },
	{ UTF32_NFD(0x2FA14), { 0x2A291 } },
	{ UTF32_NFD(0x2FA15), { 0x9EBB } },
	{ UTF32_NFD(0x2FA16), { 0x4D56 } },
	{ UTF32_NFD(0x2FA17), { 0x9EF9 } },
	{ UTF32_NFD(0x2FA18), { 0x9EFE } },
	{ UTF32_NFD(0x2FA19), { 0x9F05 } },
	{ UTF32_NFD(0x2FA1A), { 0x9F0F } },
	{ UTF32_NFD(0x2FA1B), { 0x9F16 } },
	{ UTF32_NFD(0x2FA1C), { 0x9F3B } },
	{ UTF32_NFD(0x2FA1D), { 0x2A600 } },
};

/*
 * The table is extracted from UnicodeData.txt - the 13th column
 * is relevant.
 *
 	awk 'BEGIN { FS=";"}
		$13 != "" {
			printf("\t{ 0x%s, 0x%s },\n", $1, $13)
		}' UnicodeData.txt
 *
 * NB: To reduce the table size 0x10400-0x1044F are excluded to allow 16-bit
 *     integers. ASCII characters are excluded as well. Both are handled
 *     explicitely in utf32_uppercase().
 */
static const struct {
	guint16 lower;
	guint16 upper;
} utf32_uppercase_lut[] = {
	{ 0x00B5, 0x039C },
	{ 0x00E0, 0x00C0 },
	{ 0x00E1, 0x00C1 },
	{ 0x00E2, 0x00C2 },
	{ 0x00E3, 0x00C3 },
	{ 0x00E4, 0x00C4 },
	{ 0x00E5, 0x00C5 },
	{ 0x00E6, 0x00C6 },
	{ 0x00E7, 0x00C7 },
	{ 0x00E8, 0x00C8 },
	{ 0x00E9, 0x00C9 },
	{ 0x00EA, 0x00CA },
	{ 0x00EB, 0x00CB },
	{ 0x00EC, 0x00CC },
	{ 0x00ED, 0x00CD },
	{ 0x00EE, 0x00CE },
	{ 0x00EF, 0x00CF },
	{ 0x00F0, 0x00D0 },
	{ 0x00F1, 0x00D1 },
	{ 0x00F2, 0x00D2 },
	{ 0x00F3, 0x00D3 },
	{ 0x00F4, 0x00D4 },
	{ 0x00F5, 0x00D5 },
	{ 0x00F6, 0x00D6 },
	{ 0x00F8, 0x00D8 },
	{ 0x00F9, 0x00D9 },
	{ 0x00FA, 0x00DA },
	{ 0x00FB, 0x00DB },
	{ 0x00FC, 0x00DC },
	{ 0x00FD, 0x00DD },
	{ 0x00FE, 0x00DE },
	{ 0x00FF, 0x0178 },
	{ 0x0101, 0x0100 },
	{ 0x0103, 0x0102 },
	{ 0x0105, 0x0104 },
	{ 0x0107, 0x0106 },
	{ 0x0109, 0x0108 },
	{ 0x010B, 0x010A },
	{ 0x010D, 0x010C },
	{ 0x010F, 0x010E },
	{ 0x0111, 0x0110 },
	{ 0x0113, 0x0112 },
	{ 0x0115, 0x0114 },
	{ 0x0117, 0x0116 },
	{ 0x0119, 0x0118 },
	{ 0x011B, 0x011A },
	{ 0x011D, 0x011C },
	{ 0x011F, 0x011E },
	{ 0x0121, 0x0120 },
	{ 0x0123, 0x0122 },
	{ 0x0125, 0x0124 },
	{ 0x0127, 0x0126 },
	{ 0x0129, 0x0128 },
	{ 0x012B, 0x012A },
	{ 0x012D, 0x012C },
	{ 0x012F, 0x012E },
	{ 0x0131, 0x0049 },
	{ 0x0133, 0x0132 },
	{ 0x0135, 0x0134 },
	{ 0x0137, 0x0136 },
	{ 0x013A, 0x0139 },
	{ 0x013C, 0x013B },
	{ 0x013E, 0x013D },
	{ 0x0140, 0x013F },
	{ 0x0142, 0x0141 },
	{ 0x0144, 0x0143 },
	{ 0x0146, 0x0145 },
	{ 0x0148, 0x0147 },
	{ 0x014B, 0x014A },
	{ 0x014D, 0x014C },
	{ 0x014F, 0x014E },
	{ 0x0151, 0x0150 },
	{ 0x0153, 0x0152 },
	{ 0x0155, 0x0154 },
	{ 0x0157, 0x0156 },
	{ 0x0159, 0x0158 },
	{ 0x015B, 0x015A },
	{ 0x015D, 0x015C },
	{ 0x015F, 0x015E },
	{ 0x0161, 0x0160 },
	{ 0x0163, 0x0162 },
	{ 0x0165, 0x0164 },
	{ 0x0167, 0x0166 },
	{ 0x0169, 0x0168 },
	{ 0x016B, 0x016A },
	{ 0x016D, 0x016C },
	{ 0x016F, 0x016E },
	{ 0x0171, 0x0170 },
	{ 0x0173, 0x0172 },
	{ 0x0175, 0x0174 },
	{ 0x0177, 0x0176 },
	{ 0x017A, 0x0179 },
	{ 0x017C, 0x017B },
	{ 0x017E, 0x017D },
	{ 0x017F, 0x0053 },
	{ 0x0183, 0x0182 },
	{ 0x0185, 0x0184 },
	{ 0x0188, 0x0187 },
	{ 0x018C, 0x018B },
	{ 0x0192, 0x0191 },
	{ 0x0195, 0x01F6 },
	{ 0x0199, 0x0198 },
	{ 0x019A, 0x023D },
	{ 0x019E, 0x0220 },
	{ 0x01A1, 0x01A0 },
	{ 0x01A3, 0x01A2 },
	{ 0x01A5, 0x01A4 },
	{ 0x01A8, 0x01A7 },
	{ 0x01AD, 0x01AC },
	{ 0x01B0, 0x01AF },
	{ 0x01B4, 0x01B3 },
	{ 0x01B6, 0x01B5 },
	{ 0x01B9, 0x01B8 },
	{ 0x01BD, 0x01BC },
	{ 0x01BF, 0x01F7 },
	{ 0x01C5, 0x01C4 },
	{ 0x01C6, 0x01C4 },
	{ 0x01C8, 0x01C7 },
	{ 0x01C9, 0x01C7 },
	{ 0x01CB, 0x01CA },
	{ 0x01CC, 0x01CA },
	{ 0x01CE, 0x01CD },
	{ 0x01D0, 0x01CF },
	{ 0x01D2, 0x01D1 },
	{ 0x01D4, 0x01D3 },
	{ 0x01D6, 0x01D5 },
	{ 0x01D8, 0x01D7 },
	{ 0x01DA, 0x01D9 },
	{ 0x01DC, 0x01DB },
	{ 0x01DD, 0x018E },
	{ 0x01DF, 0x01DE },
	{ 0x01E1, 0x01E0 },
	{ 0x01E3, 0x01E2 },
	{ 0x01E5, 0x01E4 },
	{ 0x01E7, 0x01E6 },
	{ 0x01E9, 0x01E8 },
	{ 0x01EB, 0x01EA },
	{ 0x01ED, 0x01EC },
	{ 0x01EF, 0x01EE },
	{ 0x01F2, 0x01F1 },
	{ 0x01F3, 0x01F1 },
	{ 0x01F5, 0x01F4 },
	{ 0x01F9, 0x01F8 },
	{ 0x01FB, 0x01FA },
	{ 0x01FD, 0x01FC },
	{ 0x01FF, 0x01FE },
	{ 0x0201, 0x0200 },
	{ 0x0203, 0x0202 },
	{ 0x0205, 0x0204 },
	{ 0x0207, 0x0206 },
	{ 0x0209, 0x0208 },
	{ 0x020B, 0x020A },
	{ 0x020D, 0x020C },
	{ 0x020F, 0x020E },
	{ 0x0211, 0x0210 },
	{ 0x0213, 0x0212 },
	{ 0x0215, 0x0214 },
	{ 0x0217, 0x0216 },
	{ 0x0219, 0x0218 },
	{ 0x021B, 0x021A },
	{ 0x021D, 0x021C },
	{ 0x021F, 0x021E },
	{ 0x0223, 0x0222 },
	{ 0x0225, 0x0224 },
	{ 0x0227, 0x0226 },
	{ 0x0229, 0x0228 },
	{ 0x022B, 0x022A },
	{ 0x022D, 0x022C },
	{ 0x022F, 0x022E },
	{ 0x0231, 0x0230 },
	{ 0x0233, 0x0232 },
	{ 0x023C, 0x023B },
	{ 0x0253, 0x0181 },
	{ 0x0254, 0x0186 },
	{ 0x0256, 0x0189 },
	{ 0x0257, 0x018A },
	{ 0x0259, 0x018F },
	{ 0x025B, 0x0190 },
	{ 0x0260, 0x0193 },
	{ 0x0263, 0x0194 },
	{ 0x0268, 0x0197 },
	{ 0x0269, 0x0196 },
	{ 0x026F, 0x019C },
	{ 0x0272, 0x019D },
	{ 0x0275, 0x019F },
	{ 0x0280, 0x01A6 },
	{ 0x0283, 0x01A9 },
	{ 0x0288, 0x01AE },
	{ 0x028A, 0x01B1 },
	{ 0x028B, 0x01B2 },
	{ 0x0292, 0x01B7 },
	{ 0x0294, 0x0241 },
	{ 0x0345, 0x0399 },
	{ 0x03AC, 0x0386 },
	{ 0x03AD, 0x0388 },
	{ 0x03AE, 0x0389 },
	{ 0x03AF, 0x038A },
	{ 0x03B1, 0x0391 },
	{ 0x03B2, 0x0392 },
	{ 0x03B3, 0x0393 },
	{ 0x03B4, 0x0394 },
	{ 0x03B5, 0x0395 },
	{ 0x03B6, 0x0396 },
	{ 0x03B7, 0x0397 },
	{ 0x03B8, 0x0398 },
	{ 0x03B9, 0x0399 },
	{ 0x03BA, 0x039A },
	{ 0x03BB, 0x039B },
	{ 0x03BC, 0x039C },
	{ 0x03BD, 0x039D },
	{ 0x03BE, 0x039E },
	{ 0x03BF, 0x039F },
	{ 0x03C0, 0x03A0 },
	{ 0x03C1, 0x03A1 },
	{ 0x03C2, 0x03A3 },
	{ 0x03C3, 0x03A3 },
	{ 0x03C4, 0x03A4 },
	{ 0x03C5, 0x03A5 },
	{ 0x03C6, 0x03A6 },
	{ 0x03C7, 0x03A7 },
	{ 0x03C8, 0x03A8 },
	{ 0x03C9, 0x03A9 },
	{ 0x03CA, 0x03AA },
	{ 0x03CB, 0x03AB },
	{ 0x03CC, 0x038C },
	{ 0x03CD, 0x038E },
	{ 0x03CE, 0x038F },
	{ 0x03D0, 0x0392 },
	{ 0x03D1, 0x0398 },
	{ 0x03D5, 0x03A6 },
	{ 0x03D6, 0x03A0 },
	{ 0x03D9, 0x03D8 },
	{ 0x03DB, 0x03DA },
	{ 0x03DD, 0x03DC },
	{ 0x03DF, 0x03DE },
	{ 0x03E1, 0x03E0 },
	{ 0x03E3, 0x03E2 },
	{ 0x03E5, 0x03E4 },
	{ 0x03E7, 0x03E6 },
	{ 0x03E9, 0x03E8 },
	{ 0x03EB, 0x03EA },
	{ 0x03ED, 0x03EC },
	{ 0x03EF, 0x03EE },
	{ 0x03F0, 0x039A },
	{ 0x03F1, 0x03A1 },
	{ 0x03F2, 0x03F9 },
	{ 0x03F5, 0x0395 },
	{ 0x03F8, 0x03F7 },
	{ 0x03FB, 0x03FA },
	{ 0x0430, 0x0410 },
	{ 0x0431, 0x0411 },
	{ 0x0432, 0x0412 },
	{ 0x0433, 0x0413 },
	{ 0x0434, 0x0414 },
	{ 0x0435, 0x0415 },
	{ 0x0436, 0x0416 },
	{ 0x0437, 0x0417 },
	{ 0x0438, 0x0418 },
	{ 0x0439, 0x0419 },
	{ 0x043A, 0x041A },
	{ 0x043B, 0x041B },
	{ 0x043C, 0x041C },
	{ 0x043D, 0x041D },
	{ 0x043E, 0x041E },
	{ 0x043F, 0x041F },
	{ 0x0440, 0x0420 },
	{ 0x0441, 0x0421 },
	{ 0x0442, 0x0422 },
	{ 0x0443, 0x0423 },
	{ 0x0444, 0x0424 },
	{ 0x0445, 0x0425 },
	{ 0x0446, 0x0426 },
	{ 0x0447, 0x0427 },
	{ 0x0448, 0x0428 },
	{ 0x0449, 0x0429 },
	{ 0x044A, 0x042A },
	{ 0x044B, 0x042B },
	{ 0x044C, 0x042C },
	{ 0x044D, 0x042D },
	{ 0x044E, 0x042E },
	{ 0x044F, 0x042F },
	{ 0x0450, 0x0400 },
	{ 0x0451, 0x0401 },
	{ 0x0452, 0x0402 },
	{ 0x0453, 0x0403 },
	{ 0x0454, 0x0404 },
	{ 0x0455, 0x0405 },
	{ 0x0456, 0x0406 },
	{ 0x0457, 0x0407 },
	{ 0x0458, 0x0408 },
	{ 0x0459, 0x0409 },
	{ 0x045A, 0x040A },
	{ 0x045B, 0x040B },
	{ 0x045C, 0x040C },
	{ 0x045D, 0x040D },
	{ 0x045E, 0x040E },
	{ 0x045F, 0x040F },
	{ 0x0461, 0x0460 },
	{ 0x0463, 0x0462 },
	{ 0x0465, 0x0464 },
	{ 0x0467, 0x0466 },
	{ 0x0469, 0x0468 },
	{ 0x046B, 0x046A },
	{ 0x046D, 0x046C },
	{ 0x046F, 0x046E },
	{ 0x0471, 0x0470 },
	{ 0x0473, 0x0472 },
	{ 0x0475, 0x0474 },
	{ 0x0477, 0x0476 },
	{ 0x0479, 0x0478 },
	{ 0x047B, 0x047A },
	{ 0x047D, 0x047C },
	{ 0x047F, 0x047E },
	{ 0x0481, 0x0480 },
	{ 0x048B, 0x048A },
	{ 0x048D, 0x048C },
	{ 0x048F, 0x048E },
	{ 0x0491, 0x0490 },
	{ 0x0493, 0x0492 },
	{ 0x0495, 0x0494 },
	{ 0x0497, 0x0496 },
	{ 0x0499, 0x0498 },
	{ 0x049B, 0x049A },
	{ 0x049D, 0x049C },
	{ 0x049F, 0x049E },
	{ 0x04A1, 0x04A0 },
	{ 0x04A3, 0x04A2 },
	{ 0x04A5, 0x04A4 },
	{ 0x04A7, 0x04A6 },
	{ 0x04A9, 0x04A8 },
	{ 0x04AB, 0x04AA },
	{ 0x04AD, 0x04AC },
	{ 0x04AF, 0x04AE },
	{ 0x04B1, 0x04B0 },
	{ 0x04B3, 0x04B2 },
	{ 0x04B5, 0x04B4 },
	{ 0x04B7, 0x04B6 },
	{ 0x04B9, 0x04B8 },
	{ 0x04BB, 0x04BA },
	{ 0x04BD, 0x04BC },
	{ 0x04BF, 0x04BE },
	{ 0x04C2, 0x04C1 },
	{ 0x04C4, 0x04C3 },
	{ 0x04C6, 0x04C5 },
	{ 0x04C8, 0x04C7 },
	{ 0x04CA, 0x04C9 },
	{ 0x04CC, 0x04CB },
	{ 0x04CE, 0x04CD },
	{ 0x04D1, 0x04D0 },
	{ 0x04D3, 0x04D2 },
	{ 0x04D5, 0x04D4 },
	{ 0x04D7, 0x04D6 },
	{ 0x04D9, 0x04D8 },
	{ 0x04DB, 0x04DA },
	{ 0x04DD, 0x04DC },
	{ 0x04DF, 0x04DE },
	{ 0x04E1, 0x04E0 },
	{ 0x04E3, 0x04E2 },
	{ 0x04E5, 0x04E4 },
	{ 0x04E7, 0x04E6 },
	{ 0x04E9, 0x04E8 },
	{ 0x04EB, 0x04EA },
	{ 0x04ED, 0x04EC },
	{ 0x04EF, 0x04EE },
	{ 0x04F1, 0x04F0 },
	{ 0x04F3, 0x04F2 },
	{ 0x04F5, 0x04F4 },
	{ 0x04F7, 0x04F6 },
	{ 0x04F9, 0x04F8 },
	{ 0x0501, 0x0500 },
	{ 0x0503, 0x0502 },
	{ 0x0505, 0x0504 },
	{ 0x0507, 0x0506 },
	{ 0x0509, 0x0508 },
	{ 0x050B, 0x050A },
	{ 0x050D, 0x050C },
	{ 0x050F, 0x050E },
	{ 0x0561, 0x0531 },
	{ 0x0562, 0x0532 },
	{ 0x0563, 0x0533 },
	{ 0x0564, 0x0534 },
	{ 0x0565, 0x0535 },
	{ 0x0566, 0x0536 },
	{ 0x0567, 0x0537 },
	{ 0x0568, 0x0538 },
	{ 0x0569, 0x0539 },
	{ 0x056A, 0x053A },
	{ 0x056B, 0x053B },
	{ 0x056C, 0x053C },
	{ 0x056D, 0x053D },
	{ 0x056E, 0x053E },
	{ 0x056F, 0x053F },
	{ 0x0570, 0x0540 },
	{ 0x0571, 0x0541 },
	{ 0x0572, 0x0542 },
	{ 0x0573, 0x0543 },
	{ 0x0574, 0x0544 },
	{ 0x0575, 0x0545 },
	{ 0x0576, 0x0546 },
	{ 0x0577, 0x0547 },
	{ 0x0578, 0x0548 },
	{ 0x0579, 0x0549 },
	{ 0x057A, 0x054A },
	{ 0x057B, 0x054B },
	{ 0x057C, 0x054C },
	{ 0x057D, 0x054D },
	{ 0x057E, 0x054E },
	{ 0x057F, 0x054F },
	{ 0x0580, 0x0550 },
	{ 0x0581, 0x0551 },
	{ 0x0582, 0x0552 },
	{ 0x0583, 0x0553 },
	{ 0x0584, 0x0554 },
	{ 0x0585, 0x0555 },
	{ 0x0586, 0x0556 },
	{ 0x1E01, 0x1E00 },
	{ 0x1E03, 0x1E02 },
	{ 0x1E05, 0x1E04 },
	{ 0x1E07, 0x1E06 },
	{ 0x1E09, 0x1E08 },
	{ 0x1E0B, 0x1E0A },
	{ 0x1E0D, 0x1E0C },
	{ 0x1E0F, 0x1E0E },
	{ 0x1E11, 0x1E10 },
	{ 0x1E13, 0x1E12 },
	{ 0x1E15, 0x1E14 },
	{ 0x1E17, 0x1E16 },
	{ 0x1E19, 0x1E18 },
	{ 0x1E1B, 0x1E1A },
	{ 0x1E1D, 0x1E1C },
	{ 0x1E1F, 0x1E1E },
	{ 0x1E21, 0x1E20 },
	{ 0x1E23, 0x1E22 },
	{ 0x1E25, 0x1E24 },
	{ 0x1E27, 0x1E26 },
	{ 0x1E29, 0x1E28 },
	{ 0x1E2B, 0x1E2A },
	{ 0x1E2D, 0x1E2C },
	{ 0x1E2F, 0x1E2E },
	{ 0x1E31, 0x1E30 },
	{ 0x1E33, 0x1E32 },
	{ 0x1E35, 0x1E34 },
	{ 0x1E37, 0x1E36 },
	{ 0x1E39, 0x1E38 },
	{ 0x1E3B, 0x1E3A },
	{ 0x1E3D, 0x1E3C },
	{ 0x1E3F, 0x1E3E },
	{ 0x1E41, 0x1E40 },
	{ 0x1E43, 0x1E42 },
	{ 0x1E45, 0x1E44 },
	{ 0x1E47, 0x1E46 },
	{ 0x1E49, 0x1E48 },
	{ 0x1E4B, 0x1E4A },
	{ 0x1E4D, 0x1E4C },
	{ 0x1E4F, 0x1E4E },
	{ 0x1E51, 0x1E50 },
	{ 0x1E53, 0x1E52 },
	{ 0x1E55, 0x1E54 },
	{ 0x1E57, 0x1E56 },
	{ 0x1E59, 0x1E58 },
	{ 0x1E5B, 0x1E5A },
	{ 0x1E5D, 0x1E5C },
	{ 0x1E5F, 0x1E5E },
	{ 0x1E61, 0x1E60 },
	{ 0x1E63, 0x1E62 },
	{ 0x1E65, 0x1E64 },
	{ 0x1E67, 0x1E66 },
	{ 0x1E69, 0x1E68 },
	{ 0x1E6B, 0x1E6A },
	{ 0x1E6D, 0x1E6C },
	{ 0x1E6F, 0x1E6E },
	{ 0x1E71, 0x1E70 },
	{ 0x1E73, 0x1E72 },
	{ 0x1E75, 0x1E74 },
	{ 0x1E77, 0x1E76 },
	{ 0x1E79, 0x1E78 },
	{ 0x1E7B, 0x1E7A },
	{ 0x1E7D, 0x1E7C },
	{ 0x1E7F, 0x1E7E },
	{ 0x1E81, 0x1E80 },
	{ 0x1E83, 0x1E82 },
	{ 0x1E85, 0x1E84 },
	{ 0x1E87, 0x1E86 },
	{ 0x1E89, 0x1E88 },
	{ 0x1E8B, 0x1E8A },
	{ 0x1E8D, 0x1E8C },
	{ 0x1E8F, 0x1E8E },
	{ 0x1E91, 0x1E90 },
	{ 0x1E93, 0x1E92 },
	{ 0x1E95, 0x1E94 },
	{ 0x1E9B, 0x1E60 },
	{ 0x1EA1, 0x1EA0 },
	{ 0x1EA3, 0x1EA2 },
	{ 0x1EA5, 0x1EA4 },
	{ 0x1EA7, 0x1EA6 },
	{ 0x1EA9, 0x1EA8 },
	{ 0x1EAB, 0x1EAA },
	{ 0x1EAD, 0x1EAC },
	{ 0x1EAF, 0x1EAE },
	{ 0x1EB1, 0x1EB0 },
	{ 0x1EB3, 0x1EB2 },
	{ 0x1EB5, 0x1EB4 },
	{ 0x1EB7, 0x1EB6 },
	{ 0x1EB9, 0x1EB8 },
	{ 0x1EBB, 0x1EBA },
	{ 0x1EBD, 0x1EBC },
	{ 0x1EBF, 0x1EBE },
	{ 0x1EC1, 0x1EC0 },
	{ 0x1EC3, 0x1EC2 },
	{ 0x1EC5, 0x1EC4 },
	{ 0x1EC7, 0x1EC6 },
	{ 0x1EC9, 0x1EC8 },
	{ 0x1ECB, 0x1ECA },
	{ 0x1ECD, 0x1ECC },
	{ 0x1ECF, 0x1ECE },
	{ 0x1ED1, 0x1ED0 },
	{ 0x1ED3, 0x1ED2 },
	{ 0x1ED5, 0x1ED4 },
	{ 0x1ED7, 0x1ED6 },
	{ 0x1ED9, 0x1ED8 },
	{ 0x1EDB, 0x1EDA },
	{ 0x1EDD, 0x1EDC },
	{ 0x1EDF, 0x1EDE },
	{ 0x1EE1, 0x1EE0 },
	{ 0x1EE3, 0x1EE2 },
	{ 0x1EE5, 0x1EE4 },
	{ 0x1EE7, 0x1EE6 },
	{ 0x1EE9, 0x1EE8 },
	{ 0x1EEB, 0x1EEA },
	{ 0x1EED, 0x1EEC },
	{ 0x1EEF, 0x1EEE },
	{ 0x1EF1, 0x1EF0 },
	{ 0x1EF3, 0x1EF2 },
	{ 0x1EF5, 0x1EF4 },
	{ 0x1EF7, 0x1EF6 },
	{ 0x1EF9, 0x1EF8 },
	{ 0x1F00, 0x1F08 },
	{ 0x1F01, 0x1F09 },
	{ 0x1F02, 0x1F0A },
	{ 0x1F03, 0x1F0B },
	{ 0x1F04, 0x1F0C },
	{ 0x1F05, 0x1F0D },
	{ 0x1F06, 0x1F0E },
	{ 0x1F07, 0x1F0F },
	{ 0x1F10, 0x1F18 },
	{ 0x1F11, 0x1F19 },
	{ 0x1F12, 0x1F1A },
	{ 0x1F13, 0x1F1B },
	{ 0x1F14, 0x1F1C },
	{ 0x1F15, 0x1F1D },
	{ 0x1F20, 0x1F28 },
	{ 0x1F21, 0x1F29 },
	{ 0x1F22, 0x1F2A },
	{ 0x1F23, 0x1F2B },
	{ 0x1F24, 0x1F2C },
	{ 0x1F25, 0x1F2D },
	{ 0x1F26, 0x1F2E },
	{ 0x1F27, 0x1F2F },
	{ 0x1F30, 0x1F38 },
	{ 0x1F31, 0x1F39 },
	{ 0x1F32, 0x1F3A },
	{ 0x1F33, 0x1F3B },
	{ 0x1F34, 0x1F3C },
	{ 0x1F35, 0x1F3D },
	{ 0x1F36, 0x1F3E },
	{ 0x1F37, 0x1F3F },
	{ 0x1F40, 0x1F48 },
	{ 0x1F41, 0x1F49 },
	{ 0x1F42, 0x1F4A },
	{ 0x1F43, 0x1F4B },
	{ 0x1F44, 0x1F4C },
	{ 0x1F45, 0x1F4D },
	{ 0x1F51, 0x1F59 },
	{ 0x1F53, 0x1F5B },
	{ 0x1F55, 0x1F5D },
	{ 0x1F57, 0x1F5F },
	{ 0x1F60, 0x1F68 },
	{ 0x1F61, 0x1F69 },
	{ 0x1F62, 0x1F6A },
	{ 0x1F63, 0x1F6B },
	{ 0x1F64, 0x1F6C },
	{ 0x1F65, 0x1F6D },
	{ 0x1F66, 0x1F6E },
	{ 0x1F67, 0x1F6F },
	{ 0x1F70, 0x1FBA },
	{ 0x1F71, 0x1FBB },
	{ 0x1F72, 0x1FC8 },
	{ 0x1F73, 0x1FC9 },
	{ 0x1F74, 0x1FCA },
	{ 0x1F75, 0x1FCB },
	{ 0x1F76, 0x1FDA },
	{ 0x1F77, 0x1FDB },
	{ 0x1F78, 0x1FF8 },
	{ 0x1F79, 0x1FF9 },
	{ 0x1F7A, 0x1FEA },
	{ 0x1F7B, 0x1FEB },
	{ 0x1F7C, 0x1FFA },
	{ 0x1F7D, 0x1FFB },
	{ 0x1F80, 0x1F88 },
	{ 0x1F81, 0x1F89 },
	{ 0x1F82, 0x1F8A },
	{ 0x1F83, 0x1F8B },
	{ 0x1F84, 0x1F8C },
	{ 0x1F85, 0x1F8D },
	{ 0x1F86, 0x1F8E },
	{ 0x1F87, 0x1F8F },
	{ 0x1F90, 0x1F98 },
	{ 0x1F91, 0x1F99 },
	{ 0x1F92, 0x1F9A },
	{ 0x1F93, 0x1F9B },
	{ 0x1F94, 0x1F9C },
	{ 0x1F95, 0x1F9D },
	{ 0x1F96, 0x1F9E },
	{ 0x1F97, 0x1F9F },
	{ 0x1FA0, 0x1FA8 },
	{ 0x1FA1, 0x1FA9 },
	{ 0x1FA2, 0x1FAA },
	{ 0x1FA3, 0x1FAB },
	{ 0x1FA4, 0x1FAC },
	{ 0x1FA5, 0x1FAD },
	{ 0x1FA6, 0x1FAE },
	{ 0x1FA7, 0x1FAF },
	{ 0x1FB0, 0x1FB8 },
	{ 0x1FB1, 0x1FB9 },
	{ 0x1FB3, 0x1FBC },
	{ 0x1FBE, 0x0399 },
	{ 0x1FC3, 0x1FCC },
	{ 0x1FD0, 0x1FD8 },
	{ 0x1FD1, 0x1FD9 },
	{ 0x1FE0, 0x1FE8 },
	{ 0x1FE1, 0x1FE9 },
	{ 0x1FE5, 0x1FEC },
	{ 0x1FF3, 0x1FFC },
	{ 0x2170, 0x2160 },
	{ 0x2171, 0x2161 },
	{ 0x2172, 0x2162 },
	{ 0x2173, 0x2163 },
	{ 0x2174, 0x2164 },
	{ 0x2175, 0x2165 },
	{ 0x2176, 0x2166 },
	{ 0x2177, 0x2167 },
	{ 0x2178, 0x2168 },
	{ 0x2179, 0x2169 },
	{ 0x217A, 0x216A },
	{ 0x217B, 0x216B },
	{ 0x217C, 0x216C },
	{ 0x217D, 0x216D },
	{ 0x217E, 0x216E },
	{ 0x217F, 0x216F },
	{ 0x24D0, 0x24B6 },
	{ 0x24D1, 0x24B7 },
	{ 0x24D2, 0x24B8 },
	{ 0x24D3, 0x24B9 },
	{ 0x24D4, 0x24BA },
	{ 0x24D5, 0x24BB },
	{ 0x24D6, 0x24BC },
	{ 0x24D7, 0x24BD },
	{ 0x24D8, 0x24BE },
	{ 0x24D9, 0x24BF },
	{ 0x24DA, 0x24C0 },
	{ 0x24DB, 0x24C1 },
	{ 0x24DC, 0x24C2 },
	{ 0x24DD, 0x24C3 },
	{ 0x24DE, 0x24C4 },
	{ 0x24DF, 0x24C5 },
	{ 0x24E0, 0x24C6 },
	{ 0x24E1, 0x24C7 },
	{ 0x24E2, 0x24C8 },
	{ 0x24E3, 0x24C9 },
	{ 0x24E4, 0x24CA },
	{ 0x24E5, 0x24CB },
	{ 0x24E6, 0x24CC },
	{ 0x24E7, 0x24CD },
	{ 0x24E8, 0x24CE },
	{ 0x24E9, 0x24CF },
	{ 0x2C30, 0x2C00 },
	{ 0x2C31, 0x2C01 },
	{ 0x2C32, 0x2C02 },
	{ 0x2C33, 0x2C03 },
	{ 0x2C34, 0x2C04 },
	{ 0x2C35, 0x2C05 },
	{ 0x2C36, 0x2C06 },
	{ 0x2C37, 0x2C07 },
	{ 0x2C38, 0x2C08 },
	{ 0x2C39, 0x2C09 },
	{ 0x2C3A, 0x2C0A },
	{ 0x2C3B, 0x2C0B },
	{ 0x2C3C, 0x2C0C },
	{ 0x2C3D, 0x2C0D },
	{ 0x2C3E, 0x2C0E },
	{ 0x2C3F, 0x2C0F },
	{ 0x2C40, 0x2C10 },
	{ 0x2C41, 0x2C11 },
	{ 0x2C42, 0x2C12 },
	{ 0x2C43, 0x2C13 },
	{ 0x2C44, 0x2C14 },
	{ 0x2C45, 0x2C15 },
	{ 0x2C46, 0x2C16 },
	{ 0x2C47, 0x2C17 },
	{ 0x2C48, 0x2C18 },
	{ 0x2C49, 0x2C19 },
	{ 0x2C4A, 0x2C1A },
	{ 0x2C4B, 0x2C1B },
	{ 0x2C4C, 0x2C1C },
	{ 0x2C4D, 0x2C1D },
	{ 0x2C4E, 0x2C1E },
	{ 0x2C4F, 0x2C1F },
	{ 0x2C50, 0x2C20 },
	{ 0x2C51, 0x2C21 },
	{ 0x2C52, 0x2C22 },
	{ 0x2C53, 0x2C23 },
	{ 0x2C54, 0x2C24 },
	{ 0x2C55, 0x2C25 },
	{ 0x2C56, 0x2C26 },
	{ 0x2C57, 0x2C27 },
	{ 0x2C58, 0x2C28 },
	{ 0x2C59, 0x2C29 },
	{ 0x2C5A, 0x2C2A },
	{ 0x2C5B, 0x2C2B },
	{ 0x2C5C, 0x2C2C },
	{ 0x2C5D, 0x2C2D },
	{ 0x2C5E, 0x2C2E },
	{ 0x2C81, 0x2C80 },
	{ 0x2C83, 0x2C82 },
	{ 0x2C85, 0x2C84 },
	{ 0x2C87, 0x2C86 },
	{ 0x2C89, 0x2C88 },
	{ 0x2C8B, 0x2C8A },
	{ 0x2C8D, 0x2C8C },
	{ 0x2C8F, 0x2C8E },
	{ 0x2C91, 0x2C90 },
	{ 0x2C93, 0x2C92 },
	{ 0x2C95, 0x2C94 },
	{ 0x2C97, 0x2C96 },
	{ 0x2C99, 0x2C98 },
	{ 0x2C9B, 0x2C9A },
	{ 0x2C9D, 0x2C9C },
	{ 0x2C9F, 0x2C9E },
	{ 0x2CA1, 0x2CA0 },
	{ 0x2CA3, 0x2CA2 },
	{ 0x2CA5, 0x2CA4 },
	{ 0x2CA7, 0x2CA6 },
	{ 0x2CA9, 0x2CA8 },
	{ 0x2CAB, 0x2CAA },
	{ 0x2CAD, 0x2CAC },
	{ 0x2CAF, 0x2CAE },
	{ 0x2CB1, 0x2CB0 },
	{ 0x2CB3, 0x2CB2 },
	{ 0x2CB5, 0x2CB4 },
	{ 0x2CB7, 0x2CB6 },
	{ 0x2CB9, 0x2CB8 },
	{ 0x2CBB, 0x2CBA },
	{ 0x2CBD, 0x2CBC },
	{ 0x2CBF, 0x2CBE },
	{ 0x2CC1, 0x2CC0 },
	{ 0x2CC3, 0x2CC2 },
	{ 0x2CC5, 0x2CC4 },
	{ 0x2CC7, 0x2CC6 },
	{ 0x2CC9, 0x2CC8 },
	{ 0x2CCB, 0x2CCA },
	{ 0x2CCD, 0x2CCC },
	{ 0x2CCF, 0x2CCE },
	{ 0x2CD1, 0x2CD0 },
	{ 0x2CD3, 0x2CD2 },
	{ 0x2CD5, 0x2CD4 },
	{ 0x2CD7, 0x2CD6 },
	{ 0x2CD9, 0x2CD8 },
	{ 0x2CDB, 0x2CDA },
	{ 0x2CDD, 0x2CDC },
	{ 0x2CDF, 0x2CDE },
	{ 0x2CE1, 0x2CE0 },
	{ 0x2CE3, 0x2CE2 },
	{ 0x2D00, 0x10A0 },
	{ 0x2D01, 0x10A1 },
	{ 0x2D02, 0x10A2 },
	{ 0x2D03, 0x10A3 },
	{ 0x2D04, 0x10A4 },
	{ 0x2D05, 0x10A5 },
	{ 0x2D06, 0x10A6 },
	{ 0x2D07, 0x10A7 },
	{ 0x2D08, 0x10A8 },
	{ 0x2D09, 0x10A9 },
	{ 0x2D0A, 0x10AA },
	{ 0x2D0B, 0x10AB },
	{ 0x2D0C, 0x10AC },
	{ 0x2D0D, 0x10AD },
	{ 0x2D0E, 0x10AE },
	{ 0x2D0F, 0x10AF },
	{ 0x2D10, 0x10B0 },
	{ 0x2D11, 0x10B1 },
	{ 0x2D12, 0x10B2 },
	{ 0x2D13, 0x10B3 },
	{ 0x2D14, 0x10B4 },
	{ 0x2D15, 0x10B5 },
	{ 0x2D16, 0x10B6 },
	{ 0x2D17, 0x10B7 },
	{ 0x2D18, 0x10B8 },
	{ 0x2D19, 0x10B9 },
	{ 0x2D1A, 0x10BA },
	{ 0x2D1B, 0x10BB },
	{ 0x2D1C, 0x10BC },
	{ 0x2D1D, 0x10BD },
	{ 0x2D1E, 0x10BE },
	{ 0x2D1F, 0x10BF },
	{ 0x2D20, 0x10C0 },
	{ 0x2D21, 0x10C1 },
	{ 0x2D22, 0x10C2 },
	{ 0x2D23, 0x10C3 },
	{ 0x2D24, 0x10C4 },
	{ 0x2D25, 0x10C5 },
	{ 0xFF41, 0xFF21 },
	{ 0xFF42, 0xFF22 },
	{ 0xFF43, 0xFF23 },
	{ 0xFF44, 0xFF24 },
	{ 0xFF45, 0xFF25 },
	{ 0xFF46, 0xFF26 },
	{ 0xFF47, 0xFF27 },
	{ 0xFF48, 0xFF28 },
	{ 0xFF49, 0xFF29 },
	{ 0xFF4A, 0xFF2A },
	{ 0xFF4B, 0xFF2B },
	{ 0xFF4C, 0xFF2C },
	{ 0xFF4D, 0xFF2D },
	{ 0xFF4E, 0xFF2E },
	{ 0xFF4F, 0xFF2F },
	{ 0xFF50, 0xFF30 },
	{ 0xFF51, 0xFF31 },
	{ 0xFF52, 0xFF32 },
	{ 0xFF53, 0xFF33 },
	{ 0xFF54, 0xFF34 },
	{ 0xFF55, 0xFF35 },
	{ 0xFF56, 0xFF36 },
	{ 0xFF57, 0xFF37 },
	{ 0xFF58, 0xFF38 },
	{ 0xFF59, 0xFF39 },
	{ 0xFF5A, 0xFF3A },
};

/*
 * The table is extracted from UnicodeData.txt - the 14th column is
 * relevant.
 *
 	awk 'BEGIN { FS=";"}
		$14 != "" {
			printf("\t{ 0x%s, 0x%s },\n", $1, $14)
		}' UnicodeData.txt
 *
 * NB: To reduce the table size 0x10400-0x10425 are excluded to allow 16-bit
 *     integers. ASCII characters are excluded as well. Both are handled
 *     explicitely in utf32_lowercase().
 */
static const struct {
	guint16 upper;
	guint16 lower;
} utf32_lowercase_lut[] = {
	{ 0x00C0, 0x00E0 },
	{ 0x00C1, 0x00E1 },
	{ 0x00C2, 0x00E2 },
	{ 0x00C3, 0x00E3 },
	{ 0x00C4, 0x00E4 },
	{ 0x00C5, 0x00E5 },
	{ 0x00C6, 0x00E6 },
	{ 0x00C7, 0x00E7 },
	{ 0x00C8, 0x00E8 },
	{ 0x00C9, 0x00E9 },
	{ 0x00CA, 0x00EA },
	{ 0x00CB, 0x00EB },
	{ 0x00CC, 0x00EC },
	{ 0x00CD, 0x00ED },
	{ 0x00CE, 0x00EE },
	{ 0x00CF, 0x00EF },
	{ 0x00D0, 0x00F0 },
	{ 0x00D1, 0x00F1 },
	{ 0x00D2, 0x00F2 },
	{ 0x00D3, 0x00F3 },
	{ 0x00D4, 0x00F4 },
	{ 0x00D5, 0x00F5 },
	{ 0x00D6, 0x00F6 },
	{ 0x00D8, 0x00F8 },
	{ 0x00D9, 0x00F9 },
	{ 0x00DA, 0x00FA },
	{ 0x00DB, 0x00FB },
	{ 0x00DC, 0x00FC },
	{ 0x00DD, 0x00FD },
	{ 0x00DE, 0x00FE },
	{ 0x0100, 0x0101 },
	{ 0x0102, 0x0103 },
	{ 0x0104, 0x0105 },
	{ 0x0106, 0x0107 },
	{ 0x0108, 0x0109 },
	{ 0x010A, 0x010B },
	{ 0x010C, 0x010D },
	{ 0x010E, 0x010F },
	{ 0x0110, 0x0111 },
	{ 0x0112, 0x0113 },
	{ 0x0114, 0x0115 },
	{ 0x0116, 0x0117 },
	{ 0x0118, 0x0119 },
	{ 0x011A, 0x011B },
	{ 0x011C, 0x011D },
	{ 0x011E, 0x011F },
	{ 0x0120, 0x0121 },
	{ 0x0122, 0x0123 },
	{ 0x0124, 0x0125 },
	{ 0x0126, 0x0127 },
	{ 0x0128, 0x0129 },
	{ 0x012A, 0x012B },
	{ 0x012C, 0x012D },
	{ 0x012E, 0x012F },
	{ 0x0130, 0x0069 },
	{ 0x0132, 0x0133 },
	{ 0x0134, 0x0135 },
	{ 0x0136, 0x0137 },
	{ 0x0139, 0x013A },
	{ 0x013B, 0x013C },
	{ 0x013D, 0x013E },
	{ 0x013F, 0x0140 },
	{ 0x0141, 0x0142 },
	{ 0x0143, 0x0144 },
	{ 0x0145, 0x0146 },
	{ 0x0147, 0x0148 },
	{ 0x014A, 0x014B },
	{ 0x014C, 0x014D },
	{ 0x014E, 0x014F },
	{ 0x0150, 0x0151 },
	{ 0x0152, 0x0153 },
	{ 0x0154, 0x0155 },
	{ 0x0156, 0x0157 },
	{ 0x0158, 0x0159 },
	{ 0x015A, 0x015B },
	{ 0x015C, 0x015D },
	{ 0x015E, 0x015F },
	{ 0x0160, 0x0161 },
	{ 0x0162, 0x0163 },
	{ 0x0164, 0x0165 },
	{ 0x0166, 0x0167 },
	{ 0x0168, 0x0169 },
	{ 0x016A, 0x016B },
	{ 0x016C, 0x016D },
	{ 0x016E, 0x016F },
	{ 0x0170, 0x0171 },
	{ 0x0172, 0x0173 },
	{ 0x0174, 0x0175 },
	{ 0x0176, 0x0177 },
	{ 0x0178, 0x00FF },
	{ 0x0179, 0x017A },
	{ 0x017B, 0x017C },
	{ 0x017D, 0x017E },
	{ 0x0181, 0x0253 },
	{ 0x0182, 0x0183 },
	{ 0x0184, 0x0185 },
	{ 0x0186, 0x0254 },
	{ 0x0187, 0x0188 },
	{ 0x0189, 0x0256 },
	{ 0x018A, 0x0257 },
	{ 0x018B, 0x018C },
	{ 0x018E, 0x01DD },
	{ 0x018F, 0x0259 },
	{ 0x0190, 0x025B },
	{ 0x0191, 0x0192 },
	{ 0x0193, 0x0260 },
	{ 0x0194, 0x0263 },
	{ 0x0196, 0x0269 },
	{ 0x0197, 0x0268 },
	{ 0x0198, 0x0199 },
	{ 0x019C, 0x026F },
	{ 0x019D, 0x0272 },
	{ 0x019F, 0x0275 },
	{ 0x01A0, 0x01A1 },
	{ 0x01A2, 0x01A3 },
	{ 0x01A4, 0x01A5 },
	{ 0x01A6, 0x0280 },
	{ 0x01A7, 0x01A8 },
	{ 0x01A9, 0x0283 },
	{ 0x01AC, 0x01AD },
	{ 0x01AE, 0x0288 },
	{ 0x01AF, 0x01B0 },
	{ 0x01B1, 0x028A },
	{ 0x01B2, 0x028B },
	{ 0x01B3, 0x01B4 },
	{ 0x01B5, 0x01B6 },
	{ 0x01B7, 0x0292 },
	{ 0x01B8, 0x01B9 },
	{ 0x01BC, 0x01BD },
	{ 0x01C4, 0x01C6 },
	{ 0x01C5, 0x01C6 },
	{ 0x01C7, 0x01C9 },
	{ 0x01C8, 0x01C9 },
	{ 0x01CA, 0x01CC },
	{ 0x01CB, 0x01CC },
	{ 0x01CD, 0x01CE },
	{ 0x01CF, 0x01D0 },
	{ 0x01D1, 0x01D2 },
	{ 0x01D3, 0x01D4 },
	{ 0x01D5, 0x01D6 },
	{ 0x01D7, 0x01D8 },
	{ 0x01D9, 0x01DA },
	{ 0x01DB, 0x01DC },
	{ 0x01DE, 0x01DF },
	{ 0x01E0, 0x01E1 },
	{ 0x01E2, 0x01E3 },
	{ 0x01E4, 0x01E5 },
	{ 0x01E6, 0x01E7 },
	{ 0x01E8, 0x01E9 },
	{ 0x01EA, 0x01EB },
	{ 0x01EC, 0x01ED },
	{ 0x01EE, 0x01EF },
	{ 0x01F1, 0x01F3 },
	{ 0x01F2, 0x01F3 },
	{ 0x01F4, 0x01F5 },
	{ 0x01F6, 0x0195 },
	{ 0x01F7, 0x01BF },
	{ 0x01F8, 0x01F9 },
	{ 0x01FA, 0x01FB },
	{ 0x01FC, 0x01FD },
	{ 0x01FE, 0x01FF },
	{ 0x0200, 0x0201 },
	{ 0x0202, 0x0203 },
	{ 0x0204, 0x0205 },
	{ 0x0206, 0x0207 },
	{ 0x0208, 0x0209 },
	{ 0x020A, 0x020B },
	{ 0x020C, 0x020D },
	{ 0x020E, 0x020F },
	{ 0x0210, 0x0211 },
	{ 0x0212, 0x0213 },
	{ 0x0214, 0x0215 },
	{ 0x0216, 0x0217 },
	{ 0x0218, 0x0219 },
	{ 0x021A, 0x021B },
	{ 0x021C, 0x021D },
	{ 0x021E, 0x021F },
	{ 0x0220, 0x019E },
	{ 0x0222, 0x0223 },
	{ 0x0224, 0x0225 },
	{ 0x0226, 0x0227 },
	{ 0x0228, 0x0229 },
	{ 0x022A, 0x022B },
	{ 0x022C, 0x022D },
	{ 0x022E, 0x022F },
	{ 0x0230, 0x0231 },
	{ 0x0232, 0x0233 },
	{ 0x023B, 0x023C },
	{ 0x023D, 0x019A },
	{ 0x0241, 0x0294 },
	{ 0x0386, 0x03AC },
	{ 0x0388, 0x03AD },
	{ 0x0389, 0x03AE },
	{ 0x038A, 0x03AF },
	{ 0x038C, 0x03CC },
	{ 0x038E, 0x03CD },
	{ 0x038F, 0x03CE },
	{ 0x0391, 0x03B1 },
	{ 0x0392, 0x03B2 },
	{ 0x0393, 0x03B3 },
	{ 0x0394, 0x03B4 },
	{ 0x0395, 0x03B5 },
	{ 0x0396, 0x03B6 },
	{ 0x0397, 0x03B7 },
	{ 0x0398, 0x03B8 },
	{ 0x0399, 0x03B9 },
	{ 0x039A, 0x03BA },
	{ 0x039B, 0x03BB },
	{ 0x039C, 0x03BC },
	{ 0x039D, 0x03BD },
	{ 0x039E, 0x03BE },
	{ 0x039F, 0x03BF },
	{ 0x03A0, 0x03C0 },
	{ 0x03A1, 0x03C1 },
	{ 0x03A3, 0x03C3 },
	{ 0x03A4, 0x03C4 },
	{ 0x03A5, 0x03C5 },
	{ 0x03A6, 0x03C6 },
	{ 0x03A7, 0x03C7 },
	{ 0x03A8, 0x03C8 },
	{ 0x03A9, 0x03C9 },
	{ 0x03AA, 0x03CA },
	{ 0x03AB, 0x03CB },
	{ 0x03D8, 0x03D9 },
	{ 0x03DA, 0x03DB },
	{ 0x03DC, 0x03DD },
	{ 0x03DE, 0x03DF },
	{ 0x03E0, 0x03E1 },
	{ 0x03E2, 0x03E3 },
	{ 0x03E4, 0x03E5 },
	{ 0x03E6, 0x03E7 },
	{ 0x03E8, 0x03E9 },
	{ 0x03EA, 0x03EB },
	{ 0x03EC, 0x03ED },
	{ 0x03EE, 0x03EF },
	{ 0x03F4, 0x03B8 },
	{ 0x03F7, 0x03F8 },
	{ 0x03F9, 0x03F2 },
	{ 0x03FA, 0x03FB },
	{ 0x0400, 0x0450 },
	{ 0x0401, 0x0451 },
	{ 0x0402, 0x0452 },
	{ 0x0403, 0x0453 },
	{ 0x0404, 0x0454 },
	{ 0x0405, 0x0455 },
	{ 0x0406, 0x0456 },
	{ 0x0407, 0x0457 },
	{ 0x0408, 0x0458 },
	{ 0x0409, 0x0459 },
	{ 0x040A, 0x045A },
	{ 0x040B, 0x045B },
	{ 0x040C, 0x045C },
	{ 0x040D, 0x045D },
	{ 0x040E, 0x045E },
	{ 0x040F, 0x045F },
	{ 0x0410, 0x0430 },
	{ 0x0411, 0x0431 },
	{ 0x0412, 0x0432 },
	{ 0x0413, 0x0433 },
	{ 0x0414, 0x0434 },
	{ 0x0415, 0x0435 },
	{ 0x0416, 0x0436 },
	{ 0x0417, 0x0437 },
	{ 0x0418, 0x0438 },
	{ 0x0419, 0x0439 },
	{ 0x041A, 0x043A },
	{ 0x041B, 0x043B },
	{ 0x041C, 0x043C },
	{ 0x041D, 0x043D },
	{ 0x041E, 0x043E },
	{ 0x041F, 0x043F },
	{ 0x0420, 0x0440 },
	{ 0x0421, 0x0441 },
	{ 0x0422, 0x0442 },
	{ 0x0423, 0x0443 },
	{ 0x0424, 0x0444 },
	{ 0x0425, 0x0445 },
	{ 0x0426, 0x0446 },
	{ 0x0427, 0x0447 },
	{ 0x0428, 0x0448 },
	{ 0x0429, 0x0449 },
	{ 0x042A, 0x044A },
	{ 0x042B, 0x044B },
	{ 0x042C, 0x044C },
	{ 0x042D, 0x044D },
	{ 0x042E, 0x044E },
	{ 0x042F, 0x044F },
	{ 0x0460, 0x0461 },
	{ 0x0462, 0x0463 },
	{ 0x0464, 0x0465 },
	{ 0x0466, 0x0467 },
	{ 0x0468, 0x0469 },
	{ 0x046A, 0x046B },
	{ 0x046C, 0x046D },
	{ 0x046E, 0x046F },
	{ 0x0470, 0x0471 },
	{ 0x0472, 0x0473 },
	{ 0x0474, 0x0475 },
	{ 0x0476, 0x0477 },
	{ 0x0478, 0x0479 },
	{ 0x047A, 0x047B },
	{ 0x047C, 0x047D },
	{ 0x047E, 0x047F },
	{ 0x0480, 0x0481 },
	{ 0x048A, 0x048B },
	{ 0x048C, 0x048D },
	{ 0x048E, 0x048F },
	{ 0x0490, 0x0491 },
	{ 0x0492, 0x0493 },
	{ 0x0494, 0x0495 },
	{ 0x0496, 0x0497 },
	{ 0x0498, 0x0499 },
	{ 0x049A, 0x049B },
	{ 0x049C, 0x049D },
	{ 0x049E, 0x049F },
	{ 0x04A0, 0x04A1 },
	{ 0x04A2, 0x04A3 },
	{ 0x04A4, 0x04A5 },
	{ 0x04A6, 0x04A7 },
	{ 0x04A8, 0x04A9 },
	{ 0x04AA, 0x04AB },
	{ 0x04AC, 0x04AD },
	{ 0x04AE, 0x04AF },
	{ 0x04B0, 0x04B1 },
	{ 0x04B2, 0x04B3 },
	{ 0x04B4, 0x04B5 },
	{ 0x04B6, 0x04B7 },
	{ 0x04B8, 0x04B9 },
	{ 0x04BA, 0x04BB },
	{ 0x04BC, 0x04BD },
	{ 0x04BE, 0x04BF },
	{ 0x04C1, 0x04C2 },
	{ 0x04C3, 0x04C4 },
	{ 0x04C5, 0x04C6 },
	{ 0x04C7, 0x04C8 },
	{ 0x04C9, 0x04CA },
	{ 0x04CB, 0x04CC },
	{ 0x04CD, 0x04CE },
	{ 0x04D0, 0x04D1 },
	{ 0x04D2, 0x04D3 },
	{ 0x04D4, 0x04D5 },
	{ 0x04D6, 0x04D7 },
	{ 0x04D8, 0x04D9 },
	{ 0x04DA, 0x04DB },
	{ 0x04DC, 0x04DD },
	{ 0x04DE, 0x04DF },
	{ 0x04E0, 0x04E1 },
	{ 0x04E2, 0x04E3 },
	{ 0x04E4, 0x04E5 },
	{ 0x04E6, 0x04E7 },
	{ 0x04E8, 0x04E9 },
	{ 0x04EA, 0x04EB },
	{ 0x04EC, 0x04ED },
	{ 0x04EE, 0x04EF },
	{ 0x04F0, 0x04F1 },
	{ 0x04F2, 0x04F3 },
	{ 0x04F4, 0x04F5 },
	{ 0x04F6, 0x04F7 },
	{ 0x04F8, 0x04F9 },
	{ 0x0500, 0x0501 },
	{ 0x0502, 0x0503 },
	{ 0x0504, 0x0505 },
	{ 0x0506, 0x0507 },
	{ 0x0508, 0x0509 },
	{ 0x050A, 0x050B },
	{ 0x050C, 0x050D },
	{ 0x050E, 0x050F },
	{ 0x0531, 0x0561 },
	{ 0x0532, 0x0562 },
	{ 0x0533, 0x0563 },
	{ 0x0534, 0x0564 },
	{ 0x0535, 0x0565 },
	{ 0x0536, 0x0566 },
	{ 0x0537, 0x0567 },
	{ 0x0538, 0x0568 },
	{ 0x0539, 0x0569 },
	{ 0x053A, 0x056A },
	{ 0x053B, 0x056B },
	{ 0x053C, 0x056C },
	{ 0x053D, 0x056D },
	{ 0x053E, 0x056E },
	{ 0x053F, 0x056F },
	{ 0x0540, 0x0570 },
	{ 0x0541, 0x0571 },
	{ 0x0542, 0x0572 },
	{ 0x0543, 0x0573 },
	{ 0x0544, 0x0574 },
	{ 0x0545, 0x0575 },
	{ 0x0546, 0x0576 },
	{ 0x0547, 0x0577 },
	{ 0x0548, 0x0578 },
	{ 0x0549, 0x0579 },
	{ 0x054A, 0x057A },
	{ 0x054B, 0x057B },
	{ 0x054C, 0x057C },
	{ 0x054D, 0x057D },
	{ 0x054E, 0x057E },
	{ 0x054F, 0x057F },
	{ 0x0550, 0x0580 },
	{ 0x0551, 0x0581 },
	{ 0x0552, 0x0582 },
	{ 0x0553, 0x0583 },
	{ 0x0554, 0x0584 },
	{ 0x0555, 0x0585 },
	{ 0x0556, 0x0586 },
	{ 0x10A0, 0x2D00 },
	{ 0x10A1, 0x2D01 },
	{ 0x10A2, 0x2D02 },
	{ 0x10A3, 0x2D03 },
	{ 0x10A4, 0x2D04 },
	{ 0x10A5, 0x2D05 },
	{ 0x10A6, 0x2D06 },
	{ 0x10A7, 0x2D07 },
	{ 0x10A8, 0x2D08 },
	{ 0x10A9, 0x2D09 },
	{ 0x10AA, 0x2D0A },
	{ 0x10AB, 0x2D0B },
	{ 0x10AC, 0x2D0C },
	{ 0x10AD, 0x2D0D },
	{ 0x10AE, 0x2D0E },
	{ 0x10AF, 0x2D0F },
	{ 0x10B0, 0x2D10 },
	{ 0x10B1, 0x2D11 },
	{ 0x10B2, 0x2D12 },
	{ 0x10B3, 0x2D13 },
	{ 0x10B4, 0x2D14 },
	{ 0x10B5, 0x2D15 },
	{ 0x10B6, 0x2D16 },
	{ 0x10B7, 0x2D17 },
	{ 0x10B8, 0x2D18 },
	{ 0x10B9, 0x2D19 },
	{ 0x10BA, 0x2D1A },
	{ 0x10BB, 0x2D1B },
	{ 0x10BC, 0x2D1C },
	{ 0x10BD, 0x2D1D },
	{ 0x10BE, 0x2D1E },
	{ 0x10BF, 0x2D1F },
	{ 0x10C0, 0x2D20 },
	{ 0x10C1, 0x2D21 },
	{ 0x10C2, 0x2D22 },
	{ 0x10C3, 0x2D23 },
	{ 0x10C4, 0x2D24 },
	{ 0x10C5, 0x2D25 },
	{ 0x1E00, 0x1E01 },
	{ 0x1E02, 0x1E03 },
	{ 0x1E04, 0x1E05 },
	{ 0x1E06, 0x1E07 },
	{ 0x1E08, 0x1E09 },
	{ 0x1E0A, 0x1E0B },
	{ 0x1E0C, 0x1E0D },
	{ 0x1E0E, 0x1E0F },
	{ 0x1E10, 0x1E11 },
	{ 0x1E12, 0x1E13 },
	{ 0x1E14, 0x1E15 },
	{ 0x1E16, 0x1E17 },
	{ 0x1E18, 0x1E19 },
	{ 0x1E1A, 0x1E1B },
	{ 0x1E1C, 0x1E1D },
	{ 0x1E1E, 0x1E1F },
	{ 0x1E20, 0x1E21 },
	{ 0x1E22, 0x1E23 },
	{ 0x1E24, 0x1E25 },
	{ 0x1E26, 0x1E27 },
	{ 0x1E28, 0x1E29 },
	{ 0x1E2A, 0x1E2B },
	{ 0x1E2C, 0x1E2D },
	{ 0x1E2E, 0x1E2F },
	{ 0x1E30, 0x1E31 },
	{ 0x1E32, 0x1E33 },
	{ 0x1E34, 0x1E35 },
	{ 0x1E36, 0x1E37 },
	{ 0x1E38, 0x1E39 },
	{ 0x1E3A, 0x1E3B },
	{ 0x1E3C, 0x1E3D },
	{ 0x1E3E, 0x1E3F },
	{ 0x1E40, 0x1E41 },
	{ 0x1E42, 0x1E43 },
	{ 0x1E44, 0x1E45 },
	{ 0x1E46, 0x1E47 },
	{ 0x1E48, 0x1E49 },
	{ 0x1E4A, 0x1E4B },
	{ 0x1E4C, 0x1E4D },
	{ 0x1E4E, 0x1E4F },
	{ 0x1E50, 0x1E51 },
	{ 0x1E52, 0x1E53 },
	{ 0x1E54, 0x1E55 },
	{ 0x1E56, 0x1E57 },
	{ 0x1E58, 0x1E59 },
	{ 0x1E5A, 0x1E5B },
	{ 0x1E5C, 0x1E5D },
	{ 0x1E5E, 0x1E5F },
	{ 0x1E60, 0x1E61 },
	{ 0x1E62, 0x1E63 },
	{ 0x1E64, 0x1E65 },
	{ 0x1E66, 0x1E67 },
	{ 0x1E68, 0x1E69 },
	{ 0x1E6A, 0x1E6B },
	{ 0x1E6C, 0x1E6D },
	{ 0x1E6E, 0x1E6F },
	{ 0x1E70, 0x1E71 },
	{ 0x1E72, 0x1E73 },
	{ 0x1E74, 0x1E75 },
	{ 0x1E76, 0x1E77 },
	{ 0x1E78, 0x1E79 },
	{ 0x1E7A, 0x1E7B },
	{ 0x1E7C, 0x1E7D },
	{ 0x1E7E, 0x1E7F },
	{ 0x1E80, 0x1E81 },
	{ 0x1E82, 0x1E83 },
	{ 0x1E84, 0x1E85 },
	{ 0x1E86, 0x1E87 },
	{ 0x1E88, 0x1E89 },
	{ 0x1E8A, 0x1E8B },
	{ 0x1E8C, 0x1E8D },
	{ 0x1E8E, 0x1E8F },
	{ 0x1E90, 0x1E91 },
	{ 0x1E92, 0x1E93 },
	{ 0x1E94, 0x1E95 },
	{ 0x1EA0, 0x1EA1 },
	{ 0x1EA2, 0x1EA3 },
	{ 0x1EA4, 0x1EA5 },
	{ 0x1EA6, 0x1EA7 },
	{ 0x1EA8, 0x1EA9 },
	{ 0x1EAA, 0x1EAB },
	{ 0x1EAC, 0x1EAD },
	{ 0x1EAE, 0x1EAF },
	{ 0x1EB0, 0x1EB1 },
	{ 0x1EB2, 0x1EB3 },
	{ 0x1EB4, 0x1EB5 },
	{ 0x1EB6, 0x1EB7 },
	{ 0x1EB8, 0x1EB9 },
	{ 0x1EBA, 0x1EBB },
	{ 0x1EBC, 0x1EBD },
	{ 0x1EBE, 0x1EBF },
	{ 0x1EC0, 0x1EC1 },
	{ 0x1EC2, 0x1EC3 },
	{ 0x1EC4, 0x1EC5 },
	{ 0x1EC6, 0x1EC7 },
	{ 0x1EC8, 0x1EC9 },
	{ 0x1ECA, 0x1ECB },
	{ 0x1ECC, 0x1ECD },
	{ 0x1ECE, 0x1ECF },
	{ 0x1ED0, 0x1ED1 },
	{ 0x1ED2, 0x1ED3 },
	{ 0x1ED4, 0x1ED5 },
	{ 0x1ED6, 0x1ED7 },
	{ 0x1ED8, 0x1ED9 },
	{ 0x1EDA, 0x1EDB },
	{ 0x1EDC, 0x1EDD },
	{ 0x1EDE, 0x1EDF },
	{ 0x1EE0, 0x1EE1 },
	{ 0x1EE2, 0x1EE3 },
	{ 0x1EE4, 0x1EE5 },
	{ 0x1EE6, 0x1EE7 },
	{ 0x1EE8, 0x1EE9 },
	{ 0x1EEA, 0x1EEB },
	{ 0x1EEC, 0x1EED },
	{ 0x1EEE, 0x1EEF },
	{ 0x1EF0, 0x1EF1 },
	{ 0x1EF2, 0x1EF3 },
	{ 0x1EF4, 0x1EF5 },
	{ 0x1EF6, 0x1EF7 },
	{ 0x1EF8, 0x1EF9 },
	{ 0x1F08, 0x1F00 },
	{ 0x1F09, 0x1F01 },
	{ 0x1F0A, 0x1F02 },
	{ 0x1F0B, 0x1F03 },
	{ 0x1F0C, 0x1F04 },
	{ 0x1F0D, 0x1F05 },
	{ 0x1F0E, 0x1F06 },
	{ 0x1F0F, 0x1F07 },
	{ 0x1F18, 0x1F10 },
	{ 0x1F19, 0x1F11 },
	{ 0x1F1A, 0x1F12 },
	{ 0x1F1B, 0x1F13 },
	{ 0x1F1C, 0x1F14 },
	{ 0x1F1D, 0x1F15 },
	{ 0x1F28, 0x1F20 },
	{ 0x1F29, 0x1F21 },
	{ 0x1F2A, 0x1F22 },
	{ 0x1F2B, 0x1F23 },
	{ 0x1F2C, 0x1F24 },
	{ 0x1F2D, 0x1F25 },
	{ 0x1F2E, 0x1F26 },
	{ 0x1F2F, 0x1F27 },
	{ 0x1F38, 0x1F30 },
	{ 0x1F39, 0x1F31 },
	{ 0x1F3A, 0x1F32 },
	{ 0x1F3B, 0x1F33 },
	{ 0x1F3C, 0x1F34 },
	{ 0x1F3D, 0x1F35 },
	{ 0x1F3E, 0x1F36 },
	{ 0x1F3F, 0x1F37 },
	{ 0x1F48, 0x1F40 },
	{ 0x1F49, 0x1F41 },
	{ 0x1F4A, 0x1F42 },
	{ 0x1F4B, 0x1F43 },
	{ 0x1F4C, 0x1F44 },
	{ 0x1F4D, 0x1F45 },
	{ 0x1F59, 0x1F51 },
	{ 0x1F5B, 0x1F53 },
	{ 0x1F5D, 0x1F55 },
	{ 0x1F5F, 0x1F57 },
	{ 0x1F68, 0x1F60 },
	{ 0x1F69, 0x1F61 },
	{ 0x1F6A, 0x1F62 },
	{ 0x1F6B, 0x1F63 },
	{ 0x1F6C, 0x1F64 },
	{ 0x1F6D, 0x1F65 },
	{ 0x1F6E, 0x1F66 },
	{ 0x1F6F, 0x1F67 },
	{ 0x1F88, 0x1F80 },
	{ 0x1F89, 0x1F81 },
	{ 0x1F8A, 0x1F82 },
	{ 0x1F8B, 0x1F83 },
	{ 0x1F8C, 0x1F84 },
	{ 0x1F8D, 0x1F85 },
	{ 0x1F8E, 0x1F86 },
	{ 0x1F8F, 0x1F87 },
	{ 0x1F98, 0x1F90 },
	{ 0x1F99, 0x1F91 },
	{ 0x1F9A, 0x1F92 },
	{ 0x1F9B, 0x1F93 },
	{ 0x1F9C, 0x1F94 },
	{ 0x1F9D, 0x1F95 },
	{ 0x1F9E, 0x1F96 },
	{ 0x1F9F, 0x1F97 },
	{ 0x1FA8, 0x1FA0 },
	{ 0x1FA9, 0x1FA1 },
	{ 0x1FAA, 0x1FA2 },
	{ 0x1FAB, 0x1FA3 },
	{ 0x1FAC, 0x1FA4 },
	{ 0x1FAD, 0x1FA5 },
	{ 0x1FAE, 0x1FA6 },
	{ 0x1FAF, 0x1FA7 },
	{ 0x1FB8, 0x1FB0 },
	{ 0x1FB9, 0x1FB1 },
	{ 0x1FBA, 0x1F70 },
	{ 0x1FBB, 0x1F71 },
	{ 0x1FBC, 0x1FB3 },
	{ 0x1FC8, 0x1F72 },
	{ 0x1FC9, 0x1F73 },
	{ 0x1FCA, 0x1F74 },
	{ 0x1FCB, 0x1F75 },
	{ 0x1FCC, 0x1FC3 },
	{ 0x1FD8, 0x1FD0 },
	{ 0x1FD9, 0x1FD1 },
	{ 0x1FDA, 0x1F76 },
	{ 0x1FDB, 0x1F77 },
	{ 0x1FE8, 0x1FE0 },
	{ 0x1FE9, 0x1FE1 },
	{ 0x1FEA, 0x1F7A },
	{ 0x1FEB, 0x1F7B },
	{ 0x1FEC, 0x1FE5 },
	{ 0x1FF8, 0x1F78 },
	{ 0x1FF9, 0x1F79 },
	{ 0x1FFA, 0x1F7C },
	{ 0x1FFB, 0x1F7D },
	{ 0x1FFC, 0x1FF3 },
	{ 0x2126, 0x03C9 },
	{ 0x212A, 0x006B },
	{ 0x212B, 0x00E5 },
	{ 0x2160, 0x2170 },
	{ 0x2161, 0x2171 },
	{ 0x2162, 0x2172 },
	{ 0x2163, 0x2173 },
	{ 0x2164, 0x2174 },
	{ 0x2165, 0x2175 },
	{ 0x2166, 0x2176 },
	{ 0x2167, 0x2177 },
	{ 0x2168, 0x2178 },
	{ 0x2169, 0x2179 },
	{ 0x216A, 0x217A },
	{ 0x216B, 0x217B },
	{ 0x216C, 0x217C },
	{ 0x216D, 0x217D },
	{ 0x216E, 0x217E },
	{ 0x216F, 0x217F },
	{ 0x24B6, 0x24D0 },
	{ 0x24B7, 0x24D1 },
	{ 0x24B8, 0x24D2 },
	{ 0x24B9, 0x24D3 },
	{ 0x24BA, 0x24D4 },
	{ 0x24BB, 0x24D5 },
	{ 0x24BC, 0x24D6 },
	{ 0x24BD, 0x24D7 },
	{ 0x24BE, 0x24D8 },
	{ 0x24BF, 0x24D9 },
	{ 0x24C0, 0x24DA },
	{ 0x24C1, 0x24DB },
	{ 0x24C2, 0x24DC },
	{ 0x24C3, 0x24DD },
	{ 0x24C4, 0x24DE },
	{ 0x24C5, 0x24DF },
	{ 0x24C6, 0x24E0 },
	{ 0x24C7, 0x24E1 },
	{ 0x24C8, 0x24E2 },
	{ 0x24C9, 0x24E3 },
	{ 0x24CA, 0x24E4 },
	{ 0x24CB, 0x24E5 },
	{ 0x24CC, 0x24E6 },
	{ 0x24CD, 0x24E7 },
	{ 0x24CE, 0x24E8 },
	{ 0x24CF, 0x24E9 },
	{ 0x2C00, 0x2C30 },
	{ 0x2C01, 0x2C31 },
	{ 0x2C02, 0x2C32 },
	{ 0x2C03, 0x2C33 },
	{ 0x2C04, 0x2C34 },
	{ 0x2C05, 0x2C35 },
	{ 0x2C06, 0x2C36 },
	{ 0x2C07, 0x2C37 },
	{ 0x2C08, 0x2C38 },
	{ 0x2C09, 0x2C39 },
	{ 0x2C0A, 0x2C3A },
	{ 0x2C0B, 0x2C3B },
	{ 0x2C0C, 0x2C3C },
	{ 0x2C0D, 0x2C3D },
	{ 0x2C0E, 0x2C3E },
	{ 0x2C0F, 0x2C3F },
	{ 0x2C10, 0x2C40 },
	{ 0x2C11, 0x2C41 },
	{ 0x2C12, 0x2C42 },
	{ 0x2C13, 0x2C43 },
	{ 0x2C14, 0x2C44 },
	{ 0x2C15, 0x2C45 },
	{ 0x2C16, 0x2C46 },
	{ 0x2C17, 0x2C47 },
	{ 0x2C18, 0x2C48 },
	{ 0x2C19, 0x2C49 },
	{ 0x2C1A, 0x2C4A },
	{ 0x2C1B, 0x2C4B },
	{ 0x2C1C, 0x2C4C },
	{ 0x2C1D, 0x2C4D },
	{ 0x2C1E, 0x2C4E },
	{ 0x2C1F, 0x2C4F },
	{ 0x2C20, 0x2C50 },
	{ 0x2C21, 0x2C51 },
	{ 0x2C22, 0x2C52 },
	{ 0x2C23, 0x2C53 },
	{ 0x2C24, 0x2C54 },
	{ 0x2C25, 0x2C55 },
	{ 0x2C26, 0x2C56 },
	{ 0x2C27, 0x2C57 },
	{ 0x2C28, 0x2C58 },
	{ 0x2C29, 0x2C59 },
	{ 0x2C2A, 0x2C5A },
	{ 0x2C2B, 0x2C5B },
	{ 0x2C2C, 0x2C5C },
	{ 0x2C2D, 0x2C5D },
	{ 0x2C2E, 0x2C5E },
	{ 0x2C80, 0x2C81 },
	{ 0x2C82, 0x2C83 },
	{ 0x2C84, 0x2C85 },
	{ 0x2C86, 0x2C87 },
	{ 0x2C88, 0x2C89 },
	{ 0x2C8A, 0x2C8B },
	{ 0x2C8C, 0x2C8D },
	{ 0x2C8E, 0x2C8F },
	{ 0x2C90, 0x2C91 },
	{ 0x2C92, 0x2C93 },
	{ 0x2C94, 0x2C95 },
	{ 0x2C96, 0x2C97 },
	{ 0x2C98, 0x2C99 },
	{ 0x2C9A, 0x2C9B },
	{ 0x2C9C, 0x2C9D },
	{ 0x2C9E, 0x2C9F },
	{ 0x2CA0, 0x2CA1 },
	{ 0x2CA2, 0x2CA3 },
	{ 0x2CA4, 0x2CA5 },
	{ 0x2CA6, 0x2CA7 },
	{ 0x2CA8, 0x2CA9 },
	{ 0x2CAA, 0x2CAB },
	{ 0x2CAC, 0x2CAD },
	{ 0x2CAE, 0x2CAF },
	{ 0x2CB0, 0x2CB1 },
	{ 0x2CB2, 0x2CB3 },
	{ 0x2CB4, 0x2CB5 },
	{ 0x2CB6, 0x2CB7 },
	{ 0x2CB8, 0x2CB9 },
	{ 0x2CBA, 0x2CBB },
	{ 0x2CBC, 0x2CBD },
	{ 0x2CBE, 0x2CBF },
	{ 0x2CC0, 0x2CC1 },
	{ 0x2CC2, 0x2CC3 },
	{ 0x2CC4, 0x2CC5 },
	{ 0x2CC6, 0x2CC7 },
	{ 0x2CC8, 0x2CC9 },
	{ 0x2CCA, 0x2CCB },
	{ 0x2CCC, 0x2CCD },
	{ 0x2CCE, 0x2CCF },
	{ 0x2CD0, 0x2CD1 },
	{ 0x2CD2, 0x2CD3 },
	{ 0x2CD4, 0x2CD5 },
	{ 0x2CD6, 0x2CD7 },
	{ 0x2CD8, 0x2CD9 },
	{ 0x2CDA, 0x2CDB },
	{ 0x2CDC, 0x2CDD },
	{ 0x2CDE, 0x2CDF },
	{ 0x2CE0, 0x2CE1 },
	{ 0x2CE2, 0x2CE3 },
	{ 0xFF21, 0xFF41 },
	{ 0xFF22, 0xFF42 },
	{ 0xFF23, 0xFF43 },
	{ 0xFF24, 0xFF44 },
	{ 0xFF25, 0xFF45 },
	{ 0xFF26, 0xFF46 },
	{ 0xFF27, 0xFF47 },
	{ 0xFF28, 0xFF48 },
	{ 0xFF29, 0xFF49 },
	{ 0xFF2A, 0xFF4A },
	{ 0xFF2B, 0xFF4B },
	{ 0xFF2C, 0xFF4C },
	{ 0xFF2D, 0xFF4D },
	{ 0xFF2E, 0xFF4E },
	{ 0xFF2F, 0xFF4F },
	{ 0xFF30, 0xFF50 },
	{ 0xFF31, 0xFF51 },
	{ 0xFF32, 0xFF52 },
	{ 0xFF33, 0xFF53 },
	{ 0xFF34, 0xFF54 },
	{ 0xFF35, 0xFF55 },
	{ 0xFF36, 0xFF56 },
	{ 0xFF37, 0xFF57 },
	{ 0xFF38, 0xFF58 },
	{ 0xFF39, 0xFF59 },
	{ 0xFF3A, 0xFF5A },
};

/*
 * The table is extracted from UnicodeData.txt - the 4th column is
 * relevant.
 *
	awk 'BEGIN { FS=";"}
		$4 != "0" {
			printf("\t{ 0x%s, %s },\n", $1, $4)
		}' UnicodeData.txt
 *
 * Lookup table to determine the combining class of a character. The
 * default class is 0, which means the character is a ``starter''.
 */
static const struct utf32_comb_class {
	guint32 uc;
	guint8 cc;
} utf32_comb_class_lut[] = {
	{ 0x0300, 230 },
	{ 0x0301, 230 },
	{ 0x0302, 230 },
	{ 0x0303, 230 },
	{ 0x0304, 230 },
	{ 0x0305, 230 },
	{ 0x0306, 230 },
	{ 0x0307, 230 },
	{ 0x0308, 230 },
	{ 0x0309, 230 },
	{ 0x030A, 230 },
	{ 0x030B, 230 },
	{ 0x030C, 230 },
	{ 0x030D, 230 },
	{ 0x030E, 230 },
	{ 0x030F, 230 },
	{ 0x0310, 230 },
	{ 0x0311, 230 },
	{ 0x0312, 230 },
	{ 0x0313, 230 },
	{ 0x0314, 230 },
	{ 0x0315, 232 },
	{ 0x0316, 220 },
	{ 0x0317, 220 },
	{ 0x0318, 220 },
	{ 0x0319, 220 },
	{ 0x031A, 232 },
	{ 0x031B, 216 },
	{ 0x031C, 220 },
	{ 0x031D, 220 },
	{ 0x031E, 220 },
	{ 0x031F, 220 },
	{ 0x0320, 220 },
	{ 0x0321, 202 },
	{ 0x0322, 202 },
	{ 0x0323, 220 },
	{ 0x0324, 220 },
	{ 0x0325, 220 },
	{ 0x0326, 220 },
	{ 0x0327, 202 },
	{ 0x0328, 202 },
	{ 0x0329, 220 },
	{ 0x032A, 220 },
	{ 0x032B, 220 },
	{ 0x032C, 220 },
	{ 0x032D, 220 },
	{ 0x032E, 220 },
	{ 0x032F, 220 },
	{ 0x0330, 220 },
	{ 0x0331, 220 },
	{ 0x0332, 220 },
	{ 0x0333, 220 },
	{ 0x0334, 1 },
	{ 0x0335, 1 },
	{ 0x0336, 1 },
	{ 0x0337, 1 },
	{ 0x0338, 1 },
	{ 0x0339, 220 },
	{ 0x033A, 220 },
	{ 0x033B, 220 },
	{ 0x033C, 220 },
	{ 0x033D, 230 },
	{ 0x033E, 230 },
	{ 0x033F, 230 },
	{ 0x0340, 230 },
	{ 0x0341, 230 },
	{ 0x0342, 230 },
	{ 0x0343, 230 },
	{ 0x0344, 230 },
	{ 0x0345, 240 },
	{ 0x0346, 230 },
	{ 0x0347, 220 },
	{ 0x0348, 220 },
	{ 0x0349, 220 },
	{ 0x034A, 230 },
	{ 0x034B, 230 },
	{ 0x034C, 230 },
	{ 0x034D, 220 },
	{ 0x034E, 220 },
	{ 0x0350, 230 },
	{ 0x0351, 230 },
	{ 0x0352, 230 },
	{ 0x0353, 220 },
	{ 0x0354, 220 },
	{ 0x0355, 220 },
	{ 0x0356, 220 },
	{ 0x0357, 230 },
	{ 0x0358, 232 },
	{ 0x0359, 220 },
	{ 0x035A, 220 },
	{ 0x035B, 230 },
	{ 0x035C, 233 },
	{ 0x035D, 234 },
	{ 0x035E, 234 },
	{ 0x035F, 233 },
	{ 0x0360, 234 },
	{ 0x0361, 234 },
	{ 0x0362, 233 },
	{ 0x0363, 230 },
	{ 0x0364, 230 },
	{ 0x0365, 230 },
	{ 0x0366, 230 },
	{ 0x0367, 230 },
	{ 0x0368, 230 },
	{ 0x0369, 230 },
	{ 0x036A, 230 },
	{ 0x036B, 230 },
	{ 0x036C, 230 },
	{ 0x036D, 230 },
	{ 0x036E, 230 },
	{ 0x036F, 230 },
	{ 0x0483, 230 },
	{ 0x0484, 230 },
	{ 0x0485, 230 },
	{ 0x0486, 230 },
	{ 0x0591, 220 },
	{ 0x0592, 230 },
	{ 0x0593, 230 },
	{ 0x0594, 230 },
	{ 0x0595, 230 },
	{ 0x0596, 220 },
	{ 0x0597, 230 },
	{ 0x0598, 230 },
	{ 0x0599, 230 },
	{ 0x059A, 222 },
	{ 0x059B, 220 },
	{ 0x059C, 230 },
	{ 0x059D, 230 },
	{ 0x059E, 230 },
	{ 0x059F, 230 },
	{ 0x05A0, 230 },
	{ 0x05A1, 230 },
	{ 0x05A2, 220 },
	{ 0x05A3, 220 },
	{ 0x05A4, 220 },
	{ 0x05A5, 220 },
	{ 0x05A6, 220 },
	{ 0x05A7, 220 },
	{ 0x05A8, 230 },
	{ 0x05A9, 230 },
	{ 0x05AA, 220 },
	{ 0x05AB, 230 },
	{ 0x05AC, 230 },
	{ 0x05AD, 222 },
	{ 0x05AE, 228 },
	{ 0x05AF, 230 },
	{ 0x05B0, 10 },
	{ 0x05B1, 11 },
	{ 0x05B2, 12 },
	{ 0x05B3, 13 },
	{ 0x05B4, 14 },
	{ 0x05B5, 15 },
	{ 0x05B6, 16 },
	{ 0x05B7, 17 },
	{ 0x05B8, 18 },
	{ 0x05B9, 19 },
	{ 0x05BB, 20 },
	{ 0x05BC, 21 },
	{ 0x05BD, 22 },
	{ 0x05BF, 23 },
	{ 0x05C1, 24 },
	{ 0x05C2, 25 },
	{ 0x05C4, 230 },
	{ 0x05C5, 220 },
	{ 0x05C7, 18 },
	{ 0x0610, 230 },
	{ 0x0611, 230 },
	{ 0x0612, 230 },
	{ 0x0613, 230 },
	{ 0x0614, 230 },
	{ 0x0615, 230 },
	{ 0x064B, 27 },
	{ 0x064C, 28 },
	{ 0x064D, 29 },
	{ 0x064E, 30 },
	{ 0x064F, 31 },
	{ 0x0650, 32 },
	{ 0x0651, 33 },
	{ 0x0652, 34 },
	{ 0x0653, 230 },
	{ 0x0654, 230 },
	{ 0x0655, 220 },
	{ 0x0656, 220 },
	{ 0x0657, 230 },
	{ 0x0658, 230 },
	{ 0x0659, 230 },
	{ 0x065A, 230 },
	{ 0x065B, 230 },
	{ 0x065C, 220 },
	{ 0x065D, 230 },
	{ 0x065E, 230 },
	{ 0x0670, 35 },
	{ 0x06D6, 230 },
	{ 0x06D7, 230 },
	{ 0x06D8, 230 },
	{ 0x06D9, 230 },
	{ 0x06DA, 230 },
	{ 0x06DB, 230 },
	{ 0x06DC, 230 },
	{ 0x06DF, 230 },
	{ 0x06E0, 230 },
	{ 0x06E1, 230 },
	{ 0x06E2, 230 },
	{ 0x06E3, 220 },
	{ 0x06E4, 230 },
	{ 0x06E7, 230 },
	{ 0x06E8, 230 },
	{ 0x06EA, 220 },
	{ 0x06EB, 230 },
	{ 0x06EC, 230 },
	{ 0x06ED, 220 },
	{ 0x0711, 36 },
	{ 0x0730, 230 },
	{ 0x0731, 220 },
	{ 0x0732, 230 },
	{ 0x0733, 230 },
	{ 0x0734, 220 },
	{ 0x0735, 230 },
	{ 0x0736, 230 },
	{ 0x0737, 220 },
	{ 0x0738, 220 },
	{ 0x0739, 220 },
	{ 0x073A, 230 },
	{ 0x073B, 220 },
	{ 0x073C, 220 },
	{ 0x073D, 230 },
	{ 0x073E, 220 },
	{ 0x073F, 230 },
	{ 0x0740, 230 },
	{ 0x0741, 230 },
	{ 0x0742, 220 },
	{ 0x0743, 230 },
	{ 0x0744, 220 },
	{ 0x0745, 230 },
	{ 0x0746, 220 },
	{ 0x0747, 230 },
	{ 0x0748, 220 },
	{ 0x0749, 230 },
	{ 0x074A, 230 },
	{ 0x093C, 7 },
	{ 0x094D, 9 },
	{ 0x0951, 230 },
	{ 0x0952, 220 },
	{ 0x0953, 230 },
	{ 0x0954, 230 },
	{ 0x09BC, 7 },
	{ 0x09CD, 9 },
	{ 0x0A3C, 7 },
	{ 0x0A4D, 9 },
	{ 0x0ABC, 7 },
	{ 0x0ACD, 9 },
	{ 0x0B3C, 7 },
	{ 0x0B4D, 9 },
	{ 0x0BCD, 9 },
	{ 0x0C4D, 9 },
	{ 0x0C55, 84 },
	{ 0x0C56, 91 },
	{ 0x0CBC, 7 },
	{ 0x0CCD, 9 },
	{ 0x0D4D, 9 },
	{ 0x0DCA, 9 },
	{ 0x0E38, 103 },
	{ 0x0E39, 103 },
	{ 0x0E3A, 9 },
	{ 0x0E48, 107 },
	{ 0x0E49, 107 },
	{ 0x0E4A, 107 },
	{ 0x0E4B, 107 },
	{ 0x0EB8, 118 },
	{ 0x0EB9, 118 },
	{ 0x0EC8, 122 },
	{ 0x0EC9, 122 },
	{ 0x0ECA, 122 },
	{ 0x0ECB, 122 },
	{ 0x0F18, 220 },
	{ 0x0F19, 220 },
	{ 0x0F35, 220 },
	{ 0x0F37, 220 },
	{ 0x0F39, 216 },
	{ 0x0F71, 129 },
	{ 0x0F72, 130 },
	{ 0x0F74, 132 },
	{ 0x0F7A, 130 },
	{ 0x0F7B, 130 },
	{ 0x0F7C, 130 },
	{ 0x0F7D, 130 },
	{ 0x0F80, 130 },
	{ 0x0F82, 230 },
	{ 0x0F83, 230 },
	{ 0x0F84, 9 },
	{ 0x0F86, 230 },
	{ 0x0F87, 230 },
	{ 0x0FC6, 220 },
	{ 0x1037, 7 },
	{ 0x1039, 9 },
	{ 0x135F, 230 },
	{ 0x1714, 9 },
	{ 0x1734, 9 },
	{ 0x17D2, 9 },
	{ 0x17DD, 230 },
	{ 0x18A9, 228 },
	{ 0x1939, 222 },
	{ 0x193A, 230 },
	{ 0x193B, 220 },
	{ 0x1A17, 230 },
	{ 0x1A18, 220 },
	{ 0x1DC0, 230 },
	{ 0x1DC1, 230 },
	{ 0x1DC2, 220 },
	{ 0x1DC3, 230 },
	{ 0x20D0, 230 },
	{ 0x20D1, 230 },
	{ 0x20D2, 1 },
	{ 0x20D3, 1 },
	{ 0x20D4, 230 },
	{ 0x20D5, 230 },
	{ 0x20D6, 230 },
	{ 0x20D7, 230 },
	{ 0x20D8, 1 },
	{ 0x20D9, 1 },
	{ 0x20DA, 1 },
	{ 0x20DB, 230 },
	{ 0x20DC, 230 },
	{ 0x20E1, 230 },
	{ 0x20E5, 1 },
	{ 0x20E6, 1 },
	{ 0x20E7, 230 },
	{ 0x20E8, 220 },
	{ 0x20E9, 230 },
	{ 0x20EA, 1 },
	{ 0x20EB, 1 },
	{ 0x302A, 218 },
	{ 0x302B, 228 },
	{ 0x302C, 232 },
	{ 0x302D, 222 },
	{ 0x302E, 224 },
	{ 0x302F, 224 },
	{ 0x3099, 8 },
	{ 0x309A, 8 },
	{ 0xA806, 9 },
	{ 0xFB1E, 26 },
	{ 0xFE20, 230 },
	{ 0xFE21, 230 },
	{ 0xFE22, 230 },
	{ 0xFE23, 230 },
	{ 0x10A0D, 220 },
	{ 0x10A0F, 230 },
	{ 0x10A38, 230 },
	{ 0x10A39, 1 },
	{ 0x10A3A, 220 },
	{ 0x10A3F, 9 },
	{ 0x1D165, 216 },
	{ 0x1D166, 216 },
	{ 0x1D167, 1 },
	{ 0x1D168, 1 },
	{ 0x1D169, 1 },
	{ 0x1D16D, 226 },
	{ 0x1D16E, 216 },
	{ 0x1D16F, 216 },
	{ 0x1D170, 216 },
	{ 0x1D171, 216 },
	{ 0x1D172, 216 },
	{ 0x1D17B, 220 },
	{ 0x1D17C, 220 },
	{ 0x1D17D, 220 },
	{ 0x1D17E, 220 },
	{ 0x1D17F, 220 },
	{ 0x1D180, 220 },
	{ 0x1D181, 220 },
	{ 0x1D182, 220 },
	{ 0x1D185, 230 },
	{ 0x1D186, 230 },
	{ 0x1D187, 230 },
	{ 0x1D188, 230 },
	{ 0x1D189, 230 },
	{ 0x1D18A, 220 },
	{ 0x1D18B, 220 },
	{ 0x1D1AA, 230 },
	{ 0x1D1AB, 230 },
	{ 0x1D1AC, 230 },
	{ 0x1D1AD, 230 },
	{ 0x1D242, 230 },
	{ 0x1D243, 230 },
	{ 0x1D244, 230 },
};

/**
 * These are special exlusions which cannot be derived from
 * UnicodeData.txt but are listed in CompositionExclusions.txt.
 * Note that the entries in that file are not completely sorted.
 */
static const guint32 utf32_composition_exclusions[] = {
	0x0958,
	0x0959,
	0x095A,
	0x095B,
	0x095C,
	0x095D,
	0x095E,
	0x095F,
	0x09DC,
	0x09DD,
	0x09DF,
	0x0A33,
	0x0A36,
	0x0A59,
	0x0A5A,
	0x0A5B,
	0x0A5E,
	0x0B5C,
	0x0B5D,
	0x0F43,
	0x0F4D,
	0x0F52,
	0x0F57,
	0x0F5C,
	0x0F69,
	0x0F76,
	0x0F78,
	0x0F93,
	0x0F9D,
	0x0FA2,
	0x0FA7,
	0x0FAC,
	0x0FB9,
	0x2ADC,
	0xFB1D,
	0xFB1F,
	0xFB2A,
	0xFB2B,
	0xFB2C,
	0xFB2D,
	0xFB2E,
	0xFB2F,
	0xFB30,
	0xFB31,
	0xFB32,
	0xFB33,
	0xFB34,
	0xFB35,
	0xFB36,
	0xFB38,
	0xFB39,
	0xFB3A,
	0xFB3B,
	0xFB3C,
	0xFB3E,
	0xFB40,
	0xFB41,
	0xFB43,
	0xFB44,
	0xFB46,
	0xFB47,
	0xFB48,
	0xFB49,
	0xFB4A,
	0xFB4B,
	0xFB4C,
	0xFB4D,
	0xFB4E,
	0x1D15E,
	0x1D15F,
	0x1D160,
	0x1D161,
	0x1D162,
	0x1D163,
	0x1D164,
	0x1D1BB,
	0x1D1BC,
	0x1D1BD,
	0x1D1BE,
	0x1D1BF,
	0x1D1C0,
};

/*
 * The table is extracted from UnicodeData.txt - the 3rd column is
 * relevant.

	gawk 'BEGIN {
		FS=";"; start=0; last=0; n=0; p_gc="";
	}
    $3 != "Lo" {
		gc = toupper($3);
		uc = strtonum("0x" $1);
		if ((gc != p_gc) || (last + 1) != uc) {
			if (n > 0) {
				printf("\t{ 0x%04X, %4d, UNI_GC_%s },\n", start, n, p_gc);
			}
			n = 1;
			start = uc;
			p_gc = gc;
		} else {
			n++;
		}
	   	last = uc;
	}
	END {
		printf("\t{ 0x%04X, %4d, UNI_GC_%s },\n", start, n, p_gc)
	}' UnicodeData.txt
 *
 * The default general category for characters not included in this
 * table is UNI_GC_CN ("Cn", "Other, Not Assigned").
 */
static const struct utf32_general_category {
	guint32 uc;		/**< The first unicode character in the array */
	guint16 len;	/**< The array length */
	guint8  gc;		/**< general category */
} utf32_general_category_lut[] = {
	{ 0x0000,   32, UNI_GC_CC },
	{ 0x0020,    1, UNI_GC_ZS },
	{ 0x0021,    3, UNI_GC_PO },
	{ 0x0024,    1, UNI_GC_SC },
	{ 0x0025,    3, UNI_GC_PO },
	{ 0x0028,    1, UNI_GC_PS },
	{ 0x0029,    1, UNI_GC_PE },
	{ 0x002A,    1, UNI_GC_PO },
	{ 0x002B,    1, UNI_GC_SM },
	{ 0x002C,    1, UNI_GC_PO },
	{ 0x002D,    1, UNI_GC_PD },
	{ 0x002E,    2, UNI_GC_PO },
	{ 0x0030,   10, UNI_GC_ND },
	{ 0x003A,    2, UNI_GC_PO },
	{ 0x003C,    3, UNI_GC_SM },
	{ 0x003F,    2, UNI_GC_PO },
	{ 0x0041,   26, UNI_GC_LU },
	{ 0x005B,    1, UNI_GC_PS },
	{ 0x005C,    1, UNI_GC_PO },
	{ 0x005D,    1, UNI_GC_PE },
	{ 0x005E,    1, UNI_GC_SK },
	{ 0x005F,    1, UNI_GC_PC },
	{ 0x0060,    1, UNI_GC_SK },
	{ 0x0061,   26, UNI_GC_LL },
	{ 0x007B,    1, UNI_GC_PS },
	{ 0x007C,    1, UNI_GC_SM },
	{ 0x007D,    1, UNI_GC_PE },
	{ 0x007E,    1, UNI_GC_SM },
	{ 0x007F,   33, UNI_GC_CC },
	{ 0x00A0,    1, UNI_GC_ZS },
	{ 0x00A1,    1, UNI_GC_PO },
	{ 0x00A2,    4, UNI_GC_SC },
	{ 0x00A6,    2, UNI_GC_SO },
	{ 0x00A8,    1, UNI_GC_SK },
	{ 0x00A9,    1, UNI_GC_SO },
	{ 0x00AA,    1, UNI_GC_LL },
	{ 0x00AB,    1, UNI_GC_PI },
	{ 0x00AC,    1, UNI_GC_SM },
	{ 0x00AD,    1, UNI_GC_CF },
	{ 0x00AE,    1, UNI_GC_SO },
	{ 0x00AF,    1, UNI_GC_SK },
	{ 0x00B0,    1, UNI_GC_SO },
	{ 0x00B1,    1, UNI_GC_SM },
	{ 0x00B2,    2, UNI_GC_NO },
	{ 0x00B4,    1, UNI_GC_SK },
	{ 0x00B5,    1, UNI_GC_LL },
	{ 0x00B6,    1, UNI_GC_SO },
	{ 0x00B7,    1, UNI_GC_PO },
	{ 0x00B8,    1, UNI_GC_SK },
	{ 0x00B9,    1, UNI_GC_NO },
	{ 0x00BA,    1, UNI_GC_LL },
	{ 0x00BB,    1, UNI_GC_PF },
	{ 0x00BC,    3, UNI_GC_NO },
	{ 0x00BF,    1, UNI_GC_PO },
	{ 0x00C0,   23, UNI_GC_LU },
	{ 0x00D7,    1, UNI_GC_SM },
	{ 0x00D8,    7, UNI_GC_LU },
	{ 0x00DF,   24, UNI_GC_LL },
	{ 0x00F7,    1, UNI_GC_SM },
	{ 0x00F8,    8, UNI_GC_LL },
	{ 0x0100,    1, UNI_GC_LU },
	{ 0x0101,    1, UNI_GC_LL },
	{ 0x0102,    1, UNI_GC_LU },
	{ 0x0103,    1, UNI_GC_LL },
	{ 0x0104,    1, UNI_GC_LU },
	{ 0x0105,    1, UNI_GC_LL },
	{ 0x0106,    1, UNI_GC_LU },
	{ 0x0107,    1, UNI_GC_LL },
	{ 0x0108,    1, UNI_GC_LU },
	{ 0x0109,    1, UNI_GC_LL },
	{ 0x010A,    1, UNI_GC_LU },
	{ 0x010B,    1, UNI_GC_LL },
	{ 0x010C,    1, UNI_GC_LU },
	{ 0x010D,    1, UNI_GC_LL },
	{ 0x010E,    1, UNI_GC_LU },
	{ 0x010F,    1, UNI_GC_LL },
	{ 0x0110,    1, UNI_GC_LU },
	{ 0x0111,    1, UNI_GC_LL },
	{ 0x0112,    1, UNI_GC_LU },
	{ 0x0113,    1, UNI_GC_LL },
	{ 0x0114,    1, UNI_GC_LU },
	{ 0x0115,    1, UNI_GC_LL },
	{ 0x0116,    1, UNI_GC_LU },
	{ 0x0117,    1, UNI_GC_LL },
	{ 0x0118,    1, UNI_GC_LU },
	{ 0x0119,    1, UNI_GC_LL },
	{ 0x011A,    1, UNI_GC_LU },
	{ 0x011B,    1, UNI_GC_LL },
	{ 0x011C,    1, UNI_GC_LU },
	{ 0x011D,    1, UNI_GC_LL },
	{ 0x011E,    1, UNI_GC_LU },
	{ 0x011F,    1, UNI_GC_LL },
	{ 0x0120,    1, UNI_GC_LU },
	{ 0x0121,    1, UNI_GC_LL },
	{ 0x0122,    1, UNI_GC_LU },
	{ 0x0123,    1, UNI_GC_LL },
	{ 0x0124,    1, UNI_GC_LU },
	{ 0x0125,    1, UNI_GC_LL },
	{ 0x0126,    1, UNI_GC_LU },
	{ 0x0127,    1, UNI_GC_LL },
	{ 0x0128,    1, UNI_GC_LU },
	{ 0x0129,    1, UNI_GC_LL },
	{ 0x012A,    1, UNI_GC_LU },
	{ 0x012B,    1, UNI_GC_LL },
	{ 0x012C,    1, UNI_GC_LU },
	{ 0x012D,    1, UNI_GC_LL },
	{ 0x012E,    1, UNI_GC_LU },
	{ 0x012F,    1, UNI_GC_LL },
	{ 0x0130,    1, UNI_GC_LU },
	{ 0x0131,    1, UNI_GC_LL },
	{ 0x0132,    1, UNI_GC_LU },
	{ 0x0133,    1, UNI_GC_LL },
	{ 0x0134,    1, UNI_GC_LU },
	{ 0x0135,    1, UNI_GC_LL },
	{ 0x0136,    1, UNI_GC_LU },
	{ 0x0137,    2, UNI_GC_LL },
	{ 0x0139,    1, UNI_GC_LU },
	{ 0x013A,    1, UNI_GC_LL },
	{ 0x013B,    1, UNI_GC_LU },
	{ 0x013C,    1, UNI_GC_LL },
	{ 0x013D,    1, UNI_GC_LU },
	{ 0x013E,    1, UNI_GC_LL },
	{ 0x013F,    1, UNI_GC_LU },
	{ 0x0140,    1, UNI_GC_LL },
	{ 0x0141,    1, UNI_GC_LU },
	{ 0x0142,    1, UNI_GC_LL },
	{ 0x0143,    1, UNI_GC_LU },
	{ 0x0144,    1, UNI_GC_LL },
	{ 0x0145,    1, UNI_GC_LU },
	{ 0x0146,    1, UNI_GC_LL },
	{ 0x0147,    1, UNI_GC_LU },
	{ 0x0148,    2, UNI_GC_LL },
	{ 0x014A,    1, UNI_GC_LU },
	{ 0x014B,    1, UNI_GC_LL },
	{ 0x014C,    1, UNI_GC_LU },
	{ 0x014D,    1, UNI_GC_LL },
	{ 0x014E,    1, UNI_GC_LU },
	{ 0x014F,    1, UNI_GC_LL },
	{ 0x0150,    1, UNI_GC_LU },
	{ 0x0151,    1, UNI_GC_LL },
	{ 0x0152,    1, UNI_GC_LU },
	{ 0x0153,    1, UNI_GC_LL },
	{ 0x0154,    1, UNI_GC_LU },
	{ 0x0155,    1, UNI_GC_LL },
	{ 0x0156,    1, UNI_GC_LU },
	{ 0x0157,    1, UNI_GC_LL },
	{ 0x0158,    1, UNI_GC_LU },
	{ 0x0159,    1, UNI_GC_LL },
	{ 0x015A,    1, UNI_GC_LU },
	{ 0x015B,    1, UNI_GC_LL },
	{ 0x015C,    1, UNI_GC_LU },
	{ 0x015D,    1, UNI_GC_LL },
	{ 0x015E,    1, UNI_GC_LU },
	{ 0x015F,    1, UNI_GC_LL },
	{ 0x0160,    1, UNI_GC_LU },
	{ 0x0161,    1, UNI_GC_LL },
	{ 0x0162,    1, UNI_GC_LU },
	{ 0x0163,    1, UNI_GC_LL },
	{ 0x0164,    1, UNI_GC_LU },
	{ 0x0165,    1, UNI_GC_LL },
	{ 0x0166,    1, UNI_GC_LU },
	{ 0x0167,    1, UNI_GC_LL },
	{ 0x0168,    1, UNI_GC_LU },
	{ 0x0169,    1, UNI_GC_LL },
	{ 0x016A,    1, UNI_GC_LU },
	{ 0x016B,    1, UNI_GC_LL },
	{ 0x016C,    1, UNI_GC_LU },
	{ 0x016D,    1, UNI_GC_LL },
	{ 0x016E,    1, UNI_GC_LU },
	{ 0x016F,    1, UNI_GC_LL },
	{ 0x0170,    1, UNI_GC_LU },
	{ 0x0171,    1, UNI_GC_LL },
	{ 0x0172,    1, UNI_GC_LU },
	{ 0x0173,    1, UNI_GC_LL },
	{ 0x0174,    1, UNI_GC_LU },
	{ 0x0175,    1, UNI_GC_LL },
	{ 0x0176,    1, UNI_GC_LU },
	{ 0x0177,    1, UNI_GC_LL },
	{ 0x0178,    2, UNI_GC_LU },
	{ 0x017A,    1, UNI_GC_LL },
	{ 0x017B,    1, UNI_GC_LU },
	{ 0x017C,    1, UNI_GC_LL },
	{ 0x017D,    1, UNI_GC_LU },
	{ 0x017E,    3, UNI_GC_LL },
	{ 0x0181,    2, UNI_GC_LU },
	{ 0x0183,    1, UNI_GC_LL },
	{ 0x0184,    1, UNI_GC_LU },
	{ 0x0185,    1, UNI_GC_LL },
	{ 0x0186,    2, UNI_GC_LU },
	{ 0x0188,    1, UNI_GC_LL },
	{ 0x0189,    3, UNI_GC_LU },
	{ 0x018C,    2, UNI_GC_LL },
	{ 0x018E,    4, UNI_GC_LU },
	{ 0x0192,    1, UNI_GC_LL },
	{ 0x0193,    2, UNI_GC_LU },
	{ 0x0195,    1, UNI_GC_LL },
	{ 0x0196,    3, UNI_GC_LU },
	{ 0x0199,    3, UNI_GC_LL },
	{ 0x019C,    2, UNI_GC_LU },
	{ 0x019E,    1, UNI_GC_LL },
	{ 0x019F,    2, UNI_GC_LU },
	{ 0x01A1,    1, UNI_GC_LL },
	{ 0x01A2,    1, UNI_GC_LU },
	{ 0x01A3,    1, UNI_GC_LL },
	{ 0x01A4,    1, UNI_GC_LU },
	{ 0x01A5,    1, UNI_GC_LL },
	{ 0x01A6,    2, UNI_GC_LU },
	{ 0x01A8,    1, UNI_GC_LL },
	{ 0x01A9,    1, UNI_GC_LU },
	{ 0x01AA,    2, UNI_GC_LL },
	{ 0x01AC,    1, UNI_GC_LU },
	{ 0x01AD,    1, UNI_GC_LL },
	{ 0x01AE,    2, UNI_GC_LU },
	{ 0x01B0,    1, UNI_GC_LL },
	{ 0x01B1,    3, UNI_GC_LU },
	{ 0x01B4,    1, UNI_GC_LL },
	{ 0x01B5,    1, UNI_GC_LU },
	{ 0x01B6,    1, UNI_GC_LL },
	{ 0x01B7,    2, UNI_GC_LU },
	{ 0x01B9,    2, UNI_GC_LL },
	{ 0x01BB,    1, UNI_GC_LO },
	{ 0x01BC,    1, UNI_GC_LU },
	{ 0x01BD,    3, UNI_GC_LL },
	{ 0x01C0,    4, UNI_GC_LO },
	{ 0x01C4,    1, UNI_GC_LU },
	{ 0x01C5,    1, UNI_GC_LT },
	{ 0x01C6,    1, UNI_GC_LL },
	{ 0x01C7,    1, UNI_GC_LU },
	{ 0x01C8,    1, UNI_GC_LT },
	{ 0x01C9,    1, UNI_GC_LL },
	{ 0x01CA,    1, UNI_GC_LU },
	{ 0x01CB,    1, UNI_GC_LT },
	{ 0x01CC,    1, UNI_GC_LL },
	{ 0x01CD,    1, UNI_GC_LU },
	{ 0x01CE,    1, UNI_GC_LL },
	{ 0x01CF,    1, UNI_GC_LU },
	{ 0x01D0,    1, UNI_GC_LL },
	{ 0x01D1,    1, UNI_GC_LU },
	{ 0x01D2,    1, UNI_GC_LL },
	{ 0x01D3,    1, UNI_GC_LU },
	{ 0x01D4,    1, UNI_GC_LL },
	{ 0x01D5,    1, UNI_GC_LU },
	{ 0x01D6,    1, UNI_GC_LL },
	{ 0x01D7,    1, UNI_GC_LU },
	{ 0x01D8,    1, UNI_GC_LL },
	{ 0x01D9,    1, UNI_GC_LU },
	{ 0x01DA,    1, UNI_GC_LL },
	{ 0x01DB,    1, UNI_GC_LU },
	{ 0x01DC,    2, UNI_GC_LL },
	{ 0x01DE,    1, UNI_GC_LU },
	{ 0x01DF,    1, UNI_GC_LL },
	{ 0x01E0,    1, UNI_GC_LU },
	{ 0x01E1,    1, UNI_GC_LL },
	{ 0x01E2,    1, UNI_GC_LU },
	{ 0x01E3,    1, UNI_GC_LL },
	{ 0x01E4,    1, UNI_GC_LU },
	{ 0x01E5,    1, UNI_GC_LL },
	{ 0x01E6,    1, UNI_GC_LU },
	{ 0x01E7,    1, UNI_GC_LL },
	{ 0x01E8,    1, UNI_GC_LU },
	{ 0x01E9,    1, UNI_GC_LL },
	{ 0x01EA,    1, UNI_GC_LU },
	{ 0x01EB,    1, UNI_GC_LL },
	{ 0x01EC,    1, UNI_GC_LU },
	{ 0x01ED,    1, UNI_GC_LL },
	{ 0x01EE,    1, UNI_GC_LU },
	{ 0x01EF,    2, UNI_GC_LL },
	{ 0x01F1,    1, UNI_GC_LU },
	{ 0x01F2,    1, UNI_GC_LT },
	{ 0x01F3,    1, UNI_GC_LL },
	{ 0x01F4,    1, UNI_GC_LU },
	{ 0x01F5,    1, UNI_GC_LL },
	{ 0x01F6,    3, UNI_GC_LU },
	{ 0x01F9,    1, UNI_GC_LL },
	{ 0x01FA,    1, UNI_GC_LU },
	{ 0x01FB,    1, UNI_GC_LL },
	{ 0x01FC,    1, UNI_GC_LU },
	{ 0x01FD,    1, UNI_GC_LL },
	{ 0x01FE,    1, UNI_GC_LU },
	{ 0x01FF,    1, UNI_GC_LL },
	{ 0x0200,    1, UNI_GC_LU },
	{ 0x0201,    1, UNI_GC_LL },
	{ 0x0202,    1, UNI_GC_LU },
	{ 0x0203,    1, UNI_GC_LL },
	{ 0x0204,    1, UNI_GC_LU },
	{ 0x0205,    1, UNI_GC_LL },
	{ 0x0206,    1, UNI_GC_LU },
	{ 0x0207,    1, UNI_GC_LL },
	{ 0x0208,    1, UNI_GC_LU },
	{ 0x0209,    1, UNI_GC_LL },
	{ 0x020A,    1, UNI_GC_LU },
	{ 0x020B,    1, UNI_GC_LL },
	{ 0x020C,    1, UNI_GC_LU },
	{ 0x020D,    1, UNI_GC_LL },
	{ 0x020E,    1, UNI_GC_LU },
	{ 0x020F,    1, UNI_GC_LL },
	{ 0x0210,    1, UNI_GC_LU },
	{ 0x0211,    1, UNI_GC_LL },
	{ 0x0212,    1, UNI_GC_LU },
	{ 0x0213,    1, UNI_GC_LL },
	{ 0x0214,    1, UNI_GC_LU },
	{ 0x0215,    1, UNI_GC_LL },
	{ 0x0216,    1, UNI_GC_LU },
	{ 0x0217,    1, UNI_GC_LL },
	{ 0x0218,    1, UNI_GC_LU },
	{ 0x0219,    1, UNI_GC_LL },
	{ 0x021A,    1, UNI_GC_LU },
	{ 0x021B,    1, UNI_GC_LL },
	{ 0x021C,    1, UNI_GC_LU },
	{ 0x021D,    1, UNI_GC_LL },
	{ 0x021E,    1, UNI_GC_LU },
	{ 0x021F,    1, UNI_GC_LL },
	{ 0x0220,    1, UNI_GC_LU },
	{ 0x0221,    1, UNI_GC_LL },
	{ 0x0222,    1, UNI_GC_LU },
	{ 0x0223,    1, UNI_GC_LL },
	{ 0x0224,    1, UNI_GC_LU },
	{ 0x0225,    1, UNI_GC_LL },
	{ 0x0226,    1, UNI_GC_LU },
	{ 0x0227,    1, UNI_GC_LL },
	{ 0x0228,    1, UNI_GC_LU },
	{ 0x0229,    1, UNI_GC_LL },
	{ 0x022A,    1, UNI_GC_LU },
	{ 0x022B,    1, UNI_GC_LL },
	{ 0x022C,    1, UNI_GC_LU },
	{ 0x022D,    1, UNI_GC_LL },
	{ 0x022E,    1, UNI_GC_LU },
	{ 0x022F,    1, UNI_GC_LL },
	{ 0x0230,    1, UNI_GC_LU },
	{ 0x0231,    1, UNI_GC_LL },
	{ 0x0232,    1, UNI_GC_LU },
	{ 0x0233,    7, UNI_GC_LL },
	{ 0x023A,    2, UNI_GC_LU },
	{ 0x023C,    1, UNI_GC_LL },
	{ 0x023D,    2, UNI_GC_LU },
	{ 0x023F,    2, UNI_GC_LL },
	{ 0x0241,    1, UNI_GC_LU },
	{ 0x0250,   96, UNI_GC_LL },
	{ 0x02B0,   18, UNI_GC_LM },
	{ 0x02C2,    4, UNI_GC_SK },
	{ 0x02C6,   12, UNI_GC_LM },
	{ 0x02D2,   14, UNI_GC_SK },
	{ 0x02E0,    5, UNI_GC_LM },
	{ 0x02E5,    9, UNI_GC_SK },
	{ 0x02EE,    1, UNI_GC_LM },
	{ 0x02EF,   17, UNI_GC_SK },
	{ 0x0300,  112, UNI_GC_MN },
	{ 0x0374,    2, UNI_GC_SK },
	{ 0x037A,    1, UNI_GC_LM },
	{ 0x037E,    1, UNI_GC_PO },
	{ 0x0384,    2, UNI_GC_SK },
	{ 0x0386,    1, UNI_GC_LU },
	{ 0x0387,    1, UNI_GC_PO },
	{ 0x0388,    3, UNI_GC_LU },
	{ 0x038C,    1, UNI_GC_LU },
	{ 0x038E,    2, UNI_GC_LU },
	{ 0x0390,    1, UNI_GC_LL },
	{ 0x0391,   17, UNI_GC_LU },
	{ 0x03A3,    9, UNI_GC_LU },
	{ 0x03AC,   35, UNI_GC_LL },
	{ 0x03D0,    2, UNI_GC_LL },
	{ 0x03D2,    3, UNI_GC_LU },
	{ 0x03D5,    3, UNI_GC_LL },
	{ 0x03D8,    1, UNI_GC_LU },
	{ 0x03D9,    1, UNI_GC_LL },
	{ 0x03DA,    1, UNI_GC_LU },
	{ 0x03DB,    1, UNI_GC_LL },
	{ 0x03DC,    1, UNI_GC_LU },
	{ 0x03DD,    1, UNI_GC_LL },
	{ 0x03DE,    1, UNI_GC_LU },
	{ 0x03DF,    1, UNI_GC_LL },
	{ 0x03E0,    1, UNI_GC_LU },
	{ 0x03E1,    1, UNI_GC_LL },
	{ 0x03E2,    1, UNI_GC_LU },
	{ 0x03E3,    1, UNI_GC_LL },
	{ 0x03E4,    1, UNI_GC_LU },
	{ 0x03E5,    1, UNI_GC_LL },
	{ 0x03E6,    1, UNI_GC_LU },
	{ 0x03E7,    1, UNI_GC_LL },
	{ 0x03E8,    1, UNI_GC_LU },
	{ 0x03E9,    1, UNI_GC_LL },
	{ 0x03EA,    1, UNI_GC_LU },
	{ 0x03EB,    1, UNI_GC_LL },
	{ 0x03EC,    1, UNI_GC_LU },
	{ 0x03ED,    1, UNI_GC_LL },
	{ 0x03EE,    1, UNI_GC_LU },
	{ 0x03EF,    5, UNI_GC_LL },
	{ 0x03F4,    1, UNI_GC_LU },
	{ 0x03F5,    1, UNI_GC_LL },
	{ 0x03F6,    1, UNI_GC_SM },
	{ 0x03F7,    1, UNI_GC_LU },
	{ 0x03F8,    1, UNI_GC_LL },
	{ 0x03F9,    2, UNI_GC_LU },
	{ 0x03FB,    2, UNI_GC_LL },
	{ 0x03FD,   51, UNI_GC_LU },
	{ 0x0430,   48, UNI_GC_LL },
	{ 0x0460,    1, UNI_GC_LU },
	{ 0x0461,    1, UNI_GC_LL },
	{ 0x0462,    1, UNI_GC_LU },
	{ 0x0463,    1, UNI_GC_LL },
	{ 0x0464,    1, UNI_GC_LU },
	{ 0x0465,    1, UNI_GC_LL },
	{ 0x0466,    1, UNI_GC_LU },
	{ 0x0467,    1, UNI_GC_LL },
	{ 0x0468,    1, UNI_GC_LU },
	{ 0x0469,    1, UNI_GC_LL },
	{ 0x046A,    1, UNI_GC_LU },
	{ 0x046B,    1, UNI_GC_LL },
	{ 0x046C,    1, UNI_GC_LU },
	{ 0x046D,    1, UNI_GC_LL },
	{ 0x046E,    1, UNI_GC_LU },
	{ 0x046F,    1, UNI_GC_LL },
	{ 0x0470,    1, UNI_GC_LU },
	{ 0x0471,    1, UNI_GC_LL },
	{ 0x0472,    1, UNI_GC_LU },
	{ 0x0473,    1, UNI_GC_LL },
	{ 0x0474,    1, UNI_GC_LU },
	{ 0x0475,    1, UNI_GC_LL },
	{ 0x0476,    1, UNI_GC_LU },
	{ 0x0477,    1, UNI_GC_LL },
	{ 0x0478,    1, UNI_GC_LU },
	{ 0x0479,    1, UNI_GC_LL },
	{ 0x047A,    1, UNI_GC_LU },
	{ 0x047B,    1, UNI_GC_LL },
	{ 0x047C,    1, UNI_GC_LU },
	{ 0x047D,    1, UNI_GC_LL },
	{ 0x047E,    1, UNI_GC_LU },
	{ 0x047F,    1, UNI_GC_LL },
	{ 0x0480,    1, UNI_GC_LU },
	{ 0x0481,    1, UNI_GC_LL },
	{ 0x0482,    1, UNI_GC_SO },
	{ 0x0483,    4, UNI_GC_MN },
	{ 0x0488,    2, UNI_GC_ME },
	{ 0x048A,    1, UNI_GC_LU },
	{ 0x048B,    1, UNI_GC_LL },
	{ 0x048C,    1, UNI_GC_LU },
	{ 0x048D,    1, UNI_GC_LL },
	{ 0x048E,    1, UNI_GC_LU },
	{ 0x048F,    1, UNI_GC_LL },
	{ 0x0490,    1, UNI_GC_LU },
	{ 0x0491,    1, UNI_GC_LL },
	{ 0x0492,    1, UNI_GC_LU },
	{ 0x0493,    1, UNI_GC_LL },
	{ 0x0494,    1, UNI_GC_LU },
	{ 0x0495,    1, UNI_GC_LL },
	{ 0x0496,    1, UNI_GC_LU },
	{ 0x0497,    1, UNI_GC_LL },
	{ 0x0498,    1, UNI_GC_LU },
	{ 0x0499,    1, UNI_GC_LL },
	{ 0x049A,    1, UNI_GC_LU },
	{ 0x049B,    1, UNI_GC_LL },
	{ 0x049C,    1, UNI_GC_LU },
	{ 0x049D,    1, UNI_GC_LL },
	{ 0x049E,    1, UNI_GC_LU },
	{ 0x049F,    1, UNI_GC_LL },
	{ 0x04A0,    1, UNI_GC_LU },
	{ 0x04A1,    1, UNI_GC_LL },
	{ 0x04A2,    1, UNI_GC_LU },
	{ 0x04A3,    1, UNI_GC_LL },
	{ 0x04A4,    1, UNI_GC_LU },
	{ 0x04A5,    1, UNI_GC_LL },
	{ 0x04A6,    1, UNI_GC_LU },
	{ 0x04A7,    1, UNI_GC_LL },
	{ 0x04A8,    1, UNI_GC_LU },
	{ 0x04A9,    1, UNI_GC_LL },
	{ 0x04AA,    1, UNI_GC_LU },
	{ 0x04AB,    1, UNI_GC_LL },
	{ 0x04AC,    1, UNI_GC_LU },
	{ 0x04AD,    1, UNI_GC_LL },
	{ 0x04AE,    1, UNI_GC_LU },
	{ 0x04AF,    1, UNI_GC_LL },
	{ 0x04B0,    1, UNI_GC_LU },
	{ 0x04B1,    1, UNI_GC_LL },
	{ 0x04B2,    1, UNI_GC_LU },
	{ 0x04B3,    1, UNI_GC_LL },
	{ 0x04B4,    1, UNI_GC_LU },
	{ 0x04B5,    1, UNI_GC_LL },
	{ 0x04B6,    1, UNI_GC_LU },
	{ 0x04B7,    1, UNI_GC_LL },
	{ 0x04B8,    1, UNI_GC_LU },
	{ 0x04B9,    1, UNI_GC_LL },
	{ 0x04BA,    1, UNI_GC_LU },
	{ 0x04BB,    1, UNI_GC_LL },
	{ 0x04BC,    1, UNI_GC_LU },
	{ 0x04BD,    1, UNI_GC_LL },
	{ 0x04BE,    1, UNI_GC_LU },
	{ 0x04BF,    1, UNI_GC_LL },
	{ 0x04C0,    2, UNI_GC_LU },
	{ 0x04C2,    1, UNI_GC_LL },
	{ 0x04C3,    1, UNI_GC_LU },
	{ 0x04C4,    1, UNI_GC_LL },
	{ 0x04C5,    1, UNI_GC_LU },
	{ 0x04C6,    1, UNI_GC_LL },
	{ 0x04C7,    1, UNI_GC_LU },
	{ 0x04C8,    1, UNI_GC_LL },
	{ 0x04C9,    1, UNI_GC_LU },
	{ 0x04CA,    1, UNI_GC_LL },
	{ 0x04CB,    1, UNI_GC_LU },
	{ 0x04CC,    1, UNI_GC_LL },
	{ 0x04CD,    1, UNI_GC_LU },
	{ 0x04CE,    1, UNI_GC_LL },
	{ 0x04D0,    1, UNI_GC_LU },
	{ 0x04D1,    1, UNI_GC_LL },
	{ 0x04D2,    1, UNI_GC_LU },
	{ 0x04D3,    1, UNI_GC_LL },
	{ 0x04D4,    1, UNI_GC_LU },
	{ 0x04D5,    1, UNI_GC_LL },
	{ 0x04D6,    1, UNI_GC_LU },
	{ 0x04D7,    1, UNI_GC_LL },
	{ 0x04D8,    1, UNI_GC_LU },
	{ 0x04D9,    1, UNI_GC_LL },
	{ 0x04DA,    1, UNI_GC_LU },
	{ 0x04DB,    1, UNI_GC_LL },
	{ 0x04DC,    1, UNI_GC_LU },
	{ 0x04DD,    1, UNI_GC_LL },
	{ 0x04DE,    1, UNI_GC_LU },
	{ 0x04DF,    1, UNI_GC_LL },
	{ 0x04E0,    1, UNI_GC_LU },
	{ 0x04E1,    1, UNI_GC_LL },
	{ 0x04E2,    1, UNI_GC_LU },
	{ 0x04E3,    1, UNI_GC_LL },
	{ 0x04E4,    1, UNI_GC_LU },
	{ 0x04E5,    1, UNI_GC_LL },
	{ 0x04E6,    1, UNI_GC_LU },
	{ 0x04E7,    1, UNI_GC_LL },
	{ 0x04E8,    1, UNI_GC_LU },
	{ 0x04E9,    1, UNI_GC_LL },
	{ 0x04EA,    1, UNI_GC_LU },
	{ 0x04EB,    1, UNI_GC_LL },
	{ 0x04EC,    1, UNI_GC_LU },
	{ 0x04ED,    1, UNI_GC_LL },
	{ 0x04EE,    1, UNI_GC_LU },
	{ 0x04EF,    1, UNI_GC_LL },
	{ 0x04F0,    1, UNI_GC_LU },
	{ 0x04F1,    1, UNI_GC_LL },
	{ 0x04F2,    1, UNI_GC_LU },
	{ 0x04F3,    1, UNI_GC_LL },
	{ 0x04F4,    1, UNI_GC_LU },
	{ 0x04F5,    1, UNI_GC_LL },
	{ 0x04F6,    1, UNI_GC_LU },
	{ 0x04F7,    1, UNI_GC_LL },
	{ 0x04F8,    1, UNI_GC_LU },
	{ 0x04F9,    1, UNI_GC_LL },
	{ 0x0500,    1, UNI_GC_LU },
	{ 0x0501,    1, UNI_GC_LL },
	{ 0x0502,    1, UNI_GC_LU },
	{ 0x0503,    1, UNI_GC_LL },
	{ 0x0504,    1, UNI_GC_LU },
	{ 0x0505,    1, UNI_GC_LL },
	{ 0x0506,    1, UNI_GC_LU },
	{ 0x0507,    1, UNI_GC_LL },
	{ 0x0508,    1, UNI_GC_LU },
	{ 0x0509,    1, UNI_GC_LL },
	{ 0x050A,    1, UNI_GC_LU },
	{ 0x050B,    1, UNI_GC_LL },
	{ 0x050C,    1, UNI_GC_LU },
	{ 0x050D,    1, UNI_GC_LL },
	{ 0x050E,    1, UNI_GC_LU },
	{ 0x050F,    1, UNI_GC_LL },
	{ 0x0531,   38, UNI_GC_LU },
	{ 0x0559,    1, UNI_GC_LM },
	{ 0x055A,    6, UNI_GC_PO },
	{ 0x0561,   39, UNI_GC_LL },
	{ 0x0589,    1, UNI_GC_PO },
	{ 0x058A,    1, UNI_GC_PD },
	{ 0x0591,   41, UNI_GC_MN },
	{ 0x05BB,    3, UNI_GC_MN },
	{ 0x05BE,    1, UNI_GC_PO },
	{ 0x05BF,    1, UNI_GC_MN },
	{ 0x05C0,    1, UNI_GC_PO },
	{ 0x05C1,    2, UNI_GC_MN },
	{ 0x05C3,    1, UNI_GC_PO },
	{ 0x05C4,    2, UNI_GC_MN },
	{ 0x05C6,    1, UNI_GC_PO },
	{ 0x05C7,    1, UNI_GC_MN },
	{ 0x05D0,   27, UNI_GC_LO },
	{ 0x05F0,    3, UNI_GC_LO },
	{ 0x05F3,    2, UNI_GC_PO },
	{ 0x0600,    4, UNI_GC_CF },
	{ 0x060B,    1, UNI_GC_SC },
	{ 0x060C,    2, UNI_GC_PO },
	{ 0x060E,    2, UNI_GC_SO },
	{ 0x0610,    6, UNI_GC_MN },
	{ 0x061B,    1, UNI_GC_PO },
	{ 0x061E,    2, UNI_GC_PO },
	{ 0x0621,   26, UNI_GC_LO },
	{ 0x0640,    1, UNI_GC_LM },
	{ 0x0641,   10, UNI_GC_LO },
	{ 0x064B,   20, UNI_GC_MN },
	{ 0x0660,   10, UNI_GC_ND },
	{ 0x066A,    4, UNI_GC_PO },
	{ 0x066E,    2, UNI_GC_LO },
	{ 0x0670,    1, UNI_GC_MN },
	{ 0x0671,   99, UNI_GC_LO },
	{ 0x06D4,    1, UNI_GC_PO },
	{ 0x06D5,    1, UNI_GC_LO },
	{ 0x06D6,    7, UNI_GC_MN },
	{ 0x06DD,    1, UNI_GC_CF },
	{ 0x06DE,    1, UNI_GC_ME },
	{ 0x06DF,    6, UNI_GC_MN },
	{ 0x06E5,    2, UNI_GC_LM },
	{ 0x06E7,    2, UNI_GC_MN },
	{ 0x06E9,    1, UNI_GC_SO },
	{ 0x06EA,    4, UNI_GC_MN },
	{ 0x06EE,    2, UNI_GC_LO },
	{ 0x06F0,   10, UNI_GC_ND },
	{ 0x06FA,    3, UNI_GC_LO },
	{ 0x06FD,    2, UNI_GC_SO },
	{ 0x06FF,    1, UNI_GC_LO },
	{ 0x0700,   14, UNI_GC_PO },
	{ 0x070F,    1, UNI_GC_CF },
	{ 0x0710,    1, UNI_GC_LO },
	{ 0x0711,    1, UNI_GC_MN },
	{ 0x0712,   30, UNI_GC_LO },
	{ 0x0730,   27, UNI_GC_MN },
	{ 0x074D,   33, UNI_GC_LO },
	{ 0x0780,   38, UNI_GC_LO },
	{ 0x07A6,   11, UNI_GC_MN },
	{ 0x07B1,    1, UNI_GC_LO },
	{ 0x0901,    2, UNI_GC_MN },
	{ 0x0903,    1, UNI_GC_MC },
	{ 0x0904,   54, UNI_GC_LO },
	{ 0x093C,    1, UNI_GC_MN },
	{ 0x093D,    1, UNI_GC_LO },
	{ 0x093E,    3, UNI_GC_MC },
	{ 0x0941,    8, UNI_GC_MN },
	{ 0x0949,    4, UNI_GC_MC },
	{ 0x094D,    1, UNI_GC_MN },
	{ 0x0950,    1, UNI_GC_LO },
	{ 0x0951,    4, UNI_GC_MN },
	{ 0x0958,   10, UNI_GC_LO },
	{ 0x0962,    2, UNI_GC_MN },
	{ 0x0964,    2, UNI_GC_PO },
	{ 0x0966,   10, UNI_GC_ND },
	{ 0x0970,    1, UNI_GC_PO },
	{ 0x097D,    1, UNI_GC_LO },
	{ 0x0981,    1, UNI_GC_MN },
	{ 0x0982,    2, UNI_GC_MC },
	{ 0x0985,    8, UNI_GC_LO },
	{ 0x098F,    2, UNI_GC_LO },
	{ 0x0993,   22, UNI_GC_LO },
	{ 0x09AA,    7, UNI_GC_LO },
	{ 0x09B2,    1, UNI_GC_LO },
	{ 0x09B6,    4, UNI_GC_LO },
	{ 0x09BC,    1, UNI_GC_MN },
	{ 0x09BD,    1, UNI_GC_LO },
	{ 0x09BE,    3, UNI_GC_MC },
	{ 0x09C1,    4, UNI_GC_MN },
	{ 0x09C7,    2, UNI_GC_MC },
	{ 0x09CB,    2, UNI_GC_MC },
	{ 0x09CD,    1, UNI_GC_MN },
	{ 0x09CE,    1, UNI_GC_LO },
	{ 0x09D7,    1, UNI_GC_MC },
	{ 0x09DC,    2, UNI_GC_LO },
	{ 0x09DF,    3, UNI_GC_LO },
	{ 0x09E2,    2, UNI_GC_MN },
	{ 0x09E6,   10, UNI_GC_ND },
	{ 0x09F0,    2, UNI_GC_LO },
	{ 0x09F2,    2, UNI_GC_SC },
	{ 0x09F4,    6, UNI_GC_NO },
	{ 0x09FA,    1, UNI_GC_SO },
	{ 0x0A01,    2, UNI_GC_MN },
	{ 0x0A03,    1, UNI_GC_MC },
	{ 0x0A05,    6, UNI_GC_LO },
	{ 0x0A0F,    2, UNI_GC_LO },
	{ 0x0A13,   22, UNI_GC_LO },
	{ 0x0A2A,    7, UNI_GC_LO },
	{ 0x0A32,    2, UNI_GC_LO },
	{ 0x0A35,    2, UNI_GC_LO },
	{ 0x0A38,    2, UNI_GC_LO },
	{ 0x0A3C,    1, UNI_GC_MN },
	{ 0x0A3E,    3, UNI_GC_MC },
	{ 0x0A41,    2, UNI_GC_MN },
	{ 0x0A47,    2, UNI_GC_MN },
	{ 0x0A4B,    3, UNI_GC_MN },
	{ 0x0A59,    4, UNI_GC_LO },
	{ 0x0A5E,    1, UNI_GC_LO },
	{ 0x0A66,   10, UNI_GC_ND },
	{ 0x0A70,    2, UNI_GC_MN },
	{ 0x0A72,    3, UNI_GC_LO },
	{ 0x0A81,    2, UNI_GC_MN },
	{ 0x0A83,    1, UNI_GC_MC },
	{ 0x0A85,    9, UNI_GC_LO },
	{ 0x0A8F,    3, UNI_GC_LO },
	{ 0x0A93,   22, UNI_GC_LO },
	{ 0x0AAA,    7, UNI_GC_LO },
	{ 0x0AB2,    2, UNI_GC_LO },
	{ 0x0AB5,    5, UNI_GC_LO },
	{ 0x0ABC,    1, UNI_GC_MN },
	{ 0x0ABD,    1, UNI_GC_LO },
	{ 0x0ABE,    3, UNI_GC_MC },
	{ 0x0AC1,    5, UNI_GC_MN },
	{ 0x0AC7,    2, UNI_GC_MN },
	{ 0x0AC9,    1, UNI_GC_MC },
	{ 0x0ACB,    2, UNI_GC_MC },
	{ 0x0ACD,    1, UNI_GC_MN },
	{ 0x0AD0,    1, UNI_GC_LO },
	{ 0x0AE0,    2, UNI_GC_LO },
	{ 0x0AE2,    2, UNI_GC_MN },
	{ 0x0AE6,   10, UNI_GC_ND },
	{ 0x0AF1,    1, UNI_GC_SC },
	{ 0x0B01,    1, UNI_GC_MN },
	{ 0x0B02,    2, UNI_GC_MC },
	{ 0x0B05,    8, UNI_GC_LO },
	{ 0x0B0F,    2, UNI_GC_LO },
	{ 0x0B13,   22, UNI_GC_LO },
	{ 0x0B2A,    7, UNI_GC_LO },
	{ 0x0B32,    2, UNI_GC_LO },
	{ 0x0B35,    5, UNI_GC_LO },
	{ 0x0B3C,    1, UNI_GC_MN },
	{ 0x0B3D,    1, UNI_GC_LO },
	{ 0x0B3E,    1, UNI_GC_MC },
	{ 0x0B3F,    1, UNI_GC_MN },
	{ 0x0B40,    1, UNI_GC_MC },
	{ 0x0B41,    3, UNI_GC_MN },
	{ 0x0B47,    2, UNI_GC_MC },
	{ 0x0B4B,    2, UNI_GC_MC },
	{ 0x0B4D,    1, UNI_GC_MN },
	{ 0x0B56,    1, UNI_GC_MN },
	{ 0x0B57,    1, UNI_GC_MC },
	{ 0x0B5C,    2, UNI_GC_LO },
	{ 0x0B5F,    3, UNI_GC_LO },
	{ 0x0B66,   10, UNI_GC_ND },
	{ 0x0B70,    1, UNI_GC_SO },
	{ 0x0B71,    1, UNI_GC_LO },
	{ 0x0B82,    1, UNI_GC_MN },
	{ 0x0B83,    1, UNI_GC_LO },
	{ 0x0B85,    6, UNI_GC_LO },
	{ 0x0B8E,    3, UNI_GC_LO },
	{ 0x0B92,    4, UNI_GC_LO },
	{ 0x0B99,    2, UNI_GC_LO },
	{ 0x0B9C,    1, UNI_GC_LO },
	{ 0x0B9E,    2, UNI_GC_LO },
	{ 0x0BA3,    2, UNI_GC_LO },
	{ 0x0BA8,    3, UNI_GC_LO },
	{ 0x0BAE,   12, UNI_GC_LO },
	{ 0x0BBE,    2, UNI_GC_MC },
	{ 0x0BC0,    1, UNI_GC_MN },
	{ 0x0BC1,    2, UNI_GC_MC },
	{ 0x0BC6,    3, UNI_GC_MC },
	{ 0x0BCA,    3, UNI_GC_MC },
	{ 0x0BCD,    1, UNI_GC_MN },
	{ 0x0BD7,    1, UNI_GC_MC },
	{ 0x0BE6,   10, UNI_GC_ND },
	{ 0x0BF0,    3, UNI_GC_NO },
	{ 0x0BF3,    6, UNI_GC_SO },
	{ 0x0BF9,    1, UNI_GC_SC },
	{ 0x0BFA,    1, UNI_GC_SO },
	{ 0x0C01,    3, UNI_GC_MC },
	{ 0x0C05,    8, UNI_GC_LO },
	{ 0x0C0E,    3, UNI_GC_LO },
	{ 0x0C12,   23, UNI_GC_LO },
	{ 0x0C2A,   10, UNI_GC_LO },
	{ 0x0C35,    5, UNI_GC_LO },
	{ 0x0C3E,    3, UNI_GC_MN },
	{ 0x0C41,    4, UNI_GC_MC },
	{ 0x0C46,    3, UNI_GC_MN },
	{ 0x0C4A,    4, UNI_GC_MN },
	{ 0x0C55,    2, UNI_GC_MN },
	{ 0x0C60,    2, UNI_GC_LO },
	{ 0x0C66,   10, UNI_GC_ND },
	{ 0x0C82,    2, UNI_GC_MC },
	{ 0x0C85,    8, UNI_GC_LO },
	{ 0x0C8E,    3, UNI_GC_LO },
	{ 0x0C92,   23, UNI_GC_LO },
	{ 0x0CAA,   10, UNI_GC_LO },
	{ 0x0CB5,    5, UNI_GC_LO },
	{ 0x0CBC,    1, UNI_GC_MN },
	{ 0x0CBD,    1, UNI_GC_LO },
	{ 0x0CBE,    1, UNI_GC_MC },
	{ 0x0CBF,    1, UNI_GC_MN },
	{ 0x0CC0,    5, UNI_GC_MC },
	{ 0x0CC6,    1, UNI_GC_MN },
	{ 0x0CC7,    2, UNI_GC_MC },
	{ 0x0CCA,    2, UNI_GC_MC },
	{ 0x0CCC,    2, UNI_GC_MN },
	{ 0x0CD5,    2, UNI_GC_MC },
	{ 0x0CDE,    1, UNI_GC_LO },
	{ 0x0CE0,    2, UNI_GC_LO },
	{ 0x0CE6,   10, UNI_GC_ND },
	{ 0x0D02,    2, UNI_GC_MC },
	{ 0x0D05,    8, UNI_GC_LO },
	{ 0x0D0E,    3, UNI_GC_LO },
	{ 0x0D12,   23, UNI_GC_LO },
	{ 0x0D2A,   16, UNI_GC_LO },
	{ 0x0D3E,    3, UNI_GC_MC },
	{ 0x0D41,    3, UNI_GC_MN },
	{ 0x0D46,    3, UNI_GC_MC },
	{ 0x0D4A,    3, UNI_GC_MC },
	{ 0x0D4D,    1, UNI_GC_MN },
	{ 0x0D57,    1, UNI_GC_MC },
	{ 0x0D60,    2, UNI_GC_LO },
	{ 0x0D66,   10, UNI_GC_ND },
	{ 0x0D82,    2, UNI_GC_MC },
	{ 0x0D85,   18, UNI_GC_LO },
	{ 0x0D9A,   24, UNI_GC_LO },
	{ 0x0DB3,    9, UNI_GC_LO },
	{ 0x0DBD,    1, UNI_GC_LO },
	{ 0x0DC0,    7, UNI_GC_LO },
	{ 0x0DCA,    1, UNI_GC_MN },
	{ 0x0DCF,    3, UNI_GC_MC },
	{ 0x0DD2,    3, UNI_GC_MN },
	{ 0x0DD6,    1, UNI_GC_MN },
	{ 0x0DD8,    8, UNI_GC_MC },
	{ 0x0DF2,    2, UNI_GC_MC },
	{ 0x0DF4,    1, UNI_GC_PO },
	{ 0x0E01,   48, UNI_GC_LO },
	{ 0x0E31,    1, UNI_GC_MN },
	{ 0x0E32,    2, UNI_GC_LO },
	{ 0x0E34,    7, UNI_GC_MN },
	{ 0x0E3F,    1, UNI_GC_SC },
	{ 0x0E40,    6, UNI_GC_LO },
	{ 0x0E46,    1, UNI_GC_LM },
	{ 0x0E47,    8, UNI_GC_MN },
	{ 0x0E4F,    1, UNI_GC_PO },
	{ 0x0E50,   10, UNI_GC_ND },
	{ 0x0E5A,    2, UNI_GC_PO },
	{ 0x0E81,    2, UNI_GC_LO },
	{ 0x0E84,    1, UNI_GC_LO },
	{ 0x0E87,    2, UNI_GC_LO },
	{ 0x0E8A,    1, UNI_GC_LO },
	{ 0x0E8D,    1, UNI_GC_LO },
	{ 0x0E94,    4, UNI_GC_LO },
	{ 0x0E99,    7, UNI_GC_LO },
	{ 0x0EA1,    3, UNI_GC_LO },
	{ 0x0EA5,    1, UNI_GC_LO },
	{ 0x0EA7,    1, UNI_GC_LO },
	{ 0x0EAA,    2, UNI_GC_LO },
	{ 0x0EAD,    4, UNI_GC_LO },
	{ 0x0EB1,    1, UNI_GC_MN },
	{ 0x0EB2,    2, UNI_GC_LO },
	{ 0x0EB4,    6, UNI_GC_MN },
	{ 0x0EBB,    2, UNI_GC_MN },
	{ 0x0EBD,    1, UNI_GC_LO },
	{ 0x0EC0,    5, UNI_GC_LO },
	{ 0x0EC6,    1, UNI_GC_LM },
	{ 0x0EC8,    6, UNI_GC_MN },
	{ 0x0ED0,   10, UNI_GC_ND },
	{ 0x0EDC,    2, UNI_GC_LO },
	{ 0x0F00,    1, UNI_GC_LO },
	{ 0x0F01,    3, UNI_GC_SO },
	{ 0x0F04,   15, UNI_GC_PO },
	{ 0x0F13,    5, UNI_GC_SO },
	{ 0x0F18,    2, UNI_GC_MN },
	{ 0x0F1A,    6, UNI_GC_SO },
	{ 0x0F20,   10, UNI_GC_ND },
	{ 0x0F2A,   10, UNI_GC_NO },
	{ 0x0F34,    1, UNI_GC_SO },
	{ 0x0F35,    1, UNI_GC_MN },
	{ 0x0F36,    1, UNI_GC_SO },
	{ 0x0F37,    1, UNI_GC_MN },
	{ 0x0F38,    1, UNI_GC_SO },
	{ 0x0F39,    1, UNI_GC_MN },
	{ 0x0F3A,    1, UNI_GC_PS },
	{ 0x0F3B,    1, UNI_GC_PE },
	{ 0x0F3C,    1, UNI_GC_PS },
	{ 0x0F3D,    1, UNI_GC_PE },
	{ 0x0F3E,    2, UNI_GC_MC },
	{ 0x0F40,    8, UNI_GC_LO },
	{ 0x0F49,   34, UNI_GC_LO },
	{ 0x0F71,   14, UNI_GC_MN },
	{ 0x0F7F,    1, UNI_GC_MC },
	{ 0x0F80,    5, UNI_GC_MN },
	{ 0x0F85,    1, UNI_GC_PO },
	{ 0x0F86,    2, UNI_GC_MN },
	{ 0x0F88,    4, UNI_GC_LO },
	{ 0x0F90,    8, UNI_GC_MN },
	{ 0x0F99,   36, UNI_GC_MN },
	{ 0x0FBE,    8, UNI_GC_SO },
	{ 0x0FC6,    1, UNI_GC_MN },
	{ 0x0FC7,    6, UNI_GC_SO },
	{ 0x0FCF,    1, UNI_GC_SO },
	{ 0x0FD0,    2, UNI_GC_PO },
	{ 0x1000,   34, UNI_GC_LO },
	{ 0x1023,    5, UNI_GC_LO },
	{ 0x1029,    2, UNI_GC_LO },
	{ 0x102C,    1, UNI_GC_MC },
	{ 0x102D,    4, UNI_GC_MN },
	{ 0x1031,    1, UNI_GC_MC },
	{ 0x1032,    1, UNI_GC_MN },
	{ 0x1036,    2, UNI_GC_MN },
	{ 0x1038,    1, UNI_GC_MC },
	{ 0x1039,    1, UNI_GC_MN },
	{ 0x1040,   10, UNI_GC_ND },
	{ 0x104A,    6, UNI_GC_PO },
	{ 0x1050,    6, UNI_GC_LO },
	{ 0x1056,    2, UNI_GC_MC },
	{ 0x1058,    2, UNI_GC_MN },
	{ 0x10A0,   38, UNI_GC_LU },
	{ 0x10D0,   43, UNI_GC_LO },
	{ 0x10FB,    1, UNI_GC_PO },
	{ 0x10FC,    1, UNI_GC_LM },
	{ 0x1100,   90, UNI_GC_LO },
	{ 0x115F,   68, UNI_GC_LO },
	{ 0x11A8,   82, UNI_GC_LO },
	{ 0x1200,   73, UNI_GC_LO },
	{ 0x124A,    4, UNI_GC_LO },
	{ 0x1250,    7, UNI_GC_LO },
	{ 0x1258,    1, UNI_GC_LO },
	{ 0x125A,    4, UNI_GC_LO },
	{ 0x1260,   41, UNI_GC_LO },
	{ 0x128A,    4, UNI_GC_LO },
	{ 0x1290,   33, UNI_GC_LO },
	{ 0x12B2,    4, UNI_GC_LO },
	{ 0x12B8,    7, UNI_GC_LO },
	{ 0x12C0,    1, UNI_GC_LO },
	{ 0x12C2,    4, UNI_GC_LO },
	{ 0x12C8,   15, UNI_GC_LO },
	{ 0x12D8,   57, UNI_GC_LO },
	{ 0x1312,    4, UNI_GC_LO },
	{ 0x1318,   67, UNI_GC_LO },
	{ 0x135F,    1, UNI_GC_MN },
	{ 0x1360,    1, UNI_GC_SO },
	{ 0x1361,    8, UNI_GC_PO },
	{ 0x1369,   20, UNI_GC_NO },
	{ 0x1380,   16, UNI_GC_LO },
	{ 0x1390,   10, UNI_GC_SO },
	{ 0x13A0,   85, UNI_GC_LO },
	{ 0x1401,  620, UNI_GC_LO },
	{ 0x166D,    2, UNI_GC_PO },
	{ 0x166F,    8, UNI_GC_LO },
	{ 0x1680,    1, UNI_GC_ZS },
	{ 0x1681,   26, UNI_GC_LO },
	{ 0x169B,    1, UNI_GC_PS },
	{ 0x169C,    1, UNI_GC_PE },
	{ 0x16A0,   75, UNI_GC_LO },
	{ 0x16EB,    3, UNI_GC_PO },
	{ 0x16EE,    3, UNI_GC_NL },
	{ 0x1700,   13, UNI_GC_LO },
	{ 0x170E,    4, UNI_GC_LO },
	{ 0x1712,    3, UNI_GC_MN },
	{ 0x1720,   18, UNI_GC_LO },
	{ 0x1732,    3, UNI_GC_MN },
	{ 0x1735,    2, UNI_GC_PO },
	{ 0x1740,   18, UNI_GC_LO },
	{ 0x1752,    2, UNI_GC_MN },
	{ 0x1760,   13, UNI_GC_LO },
	{ 0x176E,    3, UNI_GC_LO },
	{ 0x1772,    2, UNI_GC_MN },
	{ 0x1780,   52, UNI_GC_LO },
	{ 0x17B4,    2, UNI_GC_CF },
	{ 0x17B6,    1, UNI_GC_MC },
	{ 0x17B7,    7, UNI_GC_MN },
	{ 0x17BE,    8, UNI_GC_MC },
	{ 0x17C6,    1, UNI_GC_MN },
	{ 0x17C7,    2, UNI_GC_MC },
	{ 0x17C9,   11, UNI_GC_MN },
	{ 0x17D4,    3, UNI_GC_PO },
	{ 0x17D7,    1, UNI_GC_LM },
	{ 0x17D8,    3, UNI_GC_PO },
	{ 0x17DB,    1, UNI_GC_SC },
	{ 0x17DC,    1, UNI_GC_LO },
	{ 0x17DD,    1, UNI_GC_MN },
	{ 0x17E0,   10, UNI_GC_ND },
	{ 0x17F0,   10, UNI_GC_NO },
	{ 0x1800,    6, UNI_GC_PO },
	{ 0x1806,    1, UNI_GC_PD },
	{ 0x1807,    4, UNI_GC_PO },
	{ 0x180B,    3, UNI_GC_MN },
	{ 0x180E,    1, UNI_GC_ZS },
	{ 0x1810,   10, UNI_GC_ND },
	{ 0x1820,   35, UNI_GC_LO },
	{ 0x1843,    1, UNI_GC_LM },
	{ 0x1844,   52, UNI_GC_LO },
	{ 0x1880,   41, UNI_GC_LO },
	{ 0x18A9,    1, UNI_GC_MN },
	{ 0x1900,   29, UNI_GC_LO },
	{ 0x1920,    3, UNI_GC_MN },
	{ 0x1923,    4, UNI_GC_MC },
	{ 0x1927,    2, UNI_GC_MN },
	{ 0x1929,    3, UNI_GC_MC },
	{ 0x1930,    2, UNI_GC_MC },
	{ 0x1932,    1, UNI_GC_MN },
	{ 0x1933,    6, UNI_GC_MC },
	{ 0x1939,    3, UNI_GC_MN },
	{ 0x1940,    1, UNI_GC_SO },
	{ 0x1944,    2, UNI_GC_PO },
	{ 0x1946,   10, UNI_GC_ND },
	{ 0x1950,   30, UNI_GC_LO },
	{ 0x1970,    5, UNI_GC_LO },
	{ 0x1980,   42, UNI_GC_LO },
	{ 0x19B0,   17, UNI_GC_MC },
	{ 0x19C1,    7, UNI_GC_LO },
	{ 0x19C8,    2, UNI_GC_MC },
	{ 0x19D0,   10, UNI_GC_ND },
	{ 0x19DE,    2, UNI_GC_PO },
	{ 0x19E0,   32, UNI_GC_SO },
	{ 0x1A00,   23, UNI_GC_LO },
	{ 0x1A17,    2, UNI_GC_MN },
	{ 0x1A19,    3, UNI_GC_MC },
	{ 0x1A1E,    2, UNI_GC_PO },
	{ 0x1D00,   44, UNI_GC_LL },
	{ 0x1D2C,   54, UNI_GC_LM },
	{ 0x1D62,   22, UNI_GC_LL },
	{ 0x1D78,    1, UNI_GC_LM },
	{ 0x1D79,   34, UNI_GC_LL },
	{ 0x1D9B,   37, UNI_GC_LM },
	{ 0x1DC0,    4, UNI_GC_MN },
	{ 0x1E00,    1, UNI_GC_LU },
	{ 0x1E01,    1, UNI_GC_LL },
	{ 0x1E02,    1, UNI_GC_LU },
	{ 0x1E03,    1, UNI_GC_LL },
	{ 0x1E04,    1, UNI_GC_LU },
	{ 0x1E05,    1, UNI_GC_LL },
	{ 0x1E06,    1, UNI_GC_LU },
	{ 0x1E07,    1, UNI_GC_LL },
	{ 0x1E08,    1, UNI_GC_LU },
	{ 0x1E09,    1, UNI_GC_LL },
	{ 0x1E0A,    1, UNI_GC_LU },
	{ 0x1E0B,    1, UNI_GC_LL },
	{ 0x1E0C,    1, UNI_GC_LU },
	{ 0x1E0D,    1, UNI_GC_LL },
	{ 0x1E0E,    1, UNI_GC_LU },
	{ 0x1E0F,    1, UNI_GC_LL },
	{ 0x1E10,    1, UNI_GC_LU },
	{ 0x1E11,    1, UNI_GC_LL },
	{ 0x1E12,    1, UNI_GC_LU },
	{ 0x1E13,    1, UNI_GC_LL },
	{ 0x1E14,    1, UNI_GC_LU },
	{ 0x1E15,    1, UNI_GC_LL },
	{ 0x1E16,    1, UNI_GC_LU },
	{ 0x1E17,    1, UNI_GC_LL },
	{ 0x1E18,    1, UNI_GC_LU },
	{ 0x1E19,    1, UNI_GC_LL },
	{ 0x1E1A,    1, UNI_GC_LU },
	{ 0x1E1B,    1, UNI_GC_LL },
	{ 0x1E1C,    1, UNI_GC_LU },
	{ 0x1E1D,    1, UNI_GC_LL },
	{ 0x1E1E,    1, UNI_GC_LU },
	{ 0x1E1F,    1, UNI_GC_LL },
	{ 0x1E20,    1, UNI_GC_LU },
	{ 0x1E21,    1, UNI_GC_LL },
	{ 0x1E22,    1, UNI_GC_LU },
	{ 0x1E23,    1, UNI_GC_LL },
	{ 0x1E24,    1, UNI_GC_LU },
	{ 0x1E25,    1, UNI_GC_LL },
	{ 0x1E26,    1, UNI_GC_LU },
	{ 0x1E27,    1, UNI_GC_LL },
	{ 0x1E28,    1, UNI_GC_LU },
	{ 0x1E29,    1, UNI_GC_LL },
	{ 0x1E2A,    1, UNI_GC_LU },
	{ 0x1E2B,    1, UNI_GC_LL },
	{ 0x1E2C,    1, UNI_GC_LU },
	{ 0x1E2D,    1, UNI_GC_LL },
	{ 0x1E2E,    1, UNI_GC_LU },
	{ 0x1E2F,    1, UNI_GC_LL },
	{ 0x1E30,    1, UNI_GC_LU },
	{ 0x1E31,    1, UNI_GC_LL },
	{ 0x1E32,    1, UNI_GC_LU },
	{ 0x1E33,    1, UNI_GC_LL },
	{ 0x1E34,    1, UNI_GC_LU },
	{ 0x1E35,    1, UNI_GC_LL },
	{ 0x1E36,    1, UNI_GC_LU },
	{ 0x1E37,    1, UNI_GC_LL },
	{ 0x1E38,    1, UNI_GC_LU },
	{ 0x1E39,    1, UNI_GC_LL },
	{ 0x1E3A,    1, UNI_GC_LU },
	{ 0x1E3B,    1, UNI_GC_LL },
	{ 0x1E3C,    1, UNI_GC_LU },
	{ 0x1E3D,    1, UNI_GC_LL },
	{ 0x1E3E,    1, UNI_GC_LU },
	{ 0x1E3F,    1, UNI_GC_LL },
	{ 0x1E40,    1, UNI_GC_LU },
	{ 0x1E41,    1, UNI_GC_LL },
	{ 0x1E42,    1, UNI_GC_LU },
	{ 0x1E43,    1, UNI_GC_LL },
	{ 0x1E44,    1, UNI_GC_LU },
	{ 0x1E45,    1, UNI_GC_LL },
	{ 0x1E46,    1, UNI_GC_LU },
	{ 0x1E47,    1, UNI_GC_LL },
	{ 0x1E48,    1, UNI_GC_LU },
	{ 0x1E49,    1, UNI_GC_LL },
	{ 0x1E4A,    1, UNI_GC_LU },
	{ 0x1E4B,    1, UNI_GC_LL },
	{ 0x1E4C,    1, UNI_GC_LU },
	{ 0x1E4D,    1, UNI_GC_LL },
	{ 0x1E4E,    1, UNI_GC_LU },
	{ 0x1E4F,    1, UNI_GC_LL },
	{ 0x1E50,    1, UNI_GC_LU },
	{ 0x1E51,    1, UNI_GC_LL },
	{ 0x1E52,    1, UNI_GC_LU },
	{ 0x1E53,    1, UNI_GC_LL },
	{ 0x1E54,    1, UNI_GC_LU },
	{ 0x1E55,    1, UNI_GC_LL },
	{ 0x1E56,    1, UNI_GC_LU },
	{ 0x1E57,    1, UNI_GC_LL },
	{ 0x1E58,    1, UNI_GC_LU },
	{ 0x1E59,    1, UNI_GC_LL },
	{ 0x1E5A,    1, UNI_GC_LU },
	{ 0x1E5B,    1, UNI_GC_LL },
	{ 0x1E5C,    1, UNI_GC_LU },
	{ 0x1E5D,    1, UNI_GC_LL },
	{ 0x1E5E,    1, UNI_GC_LU },
	{ 0x1E5F,    1, UNI_GC_LL },
	{ 0x1E60,    1, UNI_GC_LU },
	{ 0x1E61,    1, UNI_GC_LL },
	{ 0x1E62,    1, UNI_GC_LU },
	{ 0x1E63,    1, UNI_GC_LL },
	{ 0x1E64,    1, UNI_GC_LU },
	{ 0x1E65,    1, UNI_GC_LL },
	{ 0x1E66,    1, UNI_GC_LU },
	{ 0x1E67,    1, UNI_GC_LL },
	{ 0x1E68,    1, UNI_GC_LU },
	{ 0x1E69,    1, UNI_GC_LL },
	{ 0x1E6A,    1, UNI_GC_LU },
	{ 0x1E6B,    1, UNI_GC_LL },
	{ 0x1E6C,    1, UNI_GC_LU },
	{ 0x1E6D,    1, UNI_GC_LL },
	{ 0x1E6E,    1, UNI_GC_LU },
	{ 0x1E6F,    1, UNI_GC_LL },
	{ 0x1E70,    1, UNI_GC_LU },
	{ 0x1E71,    1, UNI_GC_LL },
	{ 0x1E72,    1, UNI_GC_LU },
	{ 0x1E73,    1, UNI_GC_LL },
	{ 0x1E74,    1, UNI_GC_LU },
	{ 0x1E75,    1, UNI_GC_LL },
	{ 0x1E76,    1, UNI_GC_LU },
	{ 0x1E77,    1, UNI_GC_LL },
	{ 0x1E78,    1, UNI_GC_LU },
	{ 0x1E79,    1, UNI_GC_LL },
	{ 0x1E7A,    1, UNI_GC_LU },
	{ 0x1E7B,    1, UNI_GC_LL },
	{ 0x1E7C,    1, UNI_GC_LU },
	{ 0x1E7D,    1, UNI_GC_LL },
	{ 0x1E7E,    1, UNI_GC_LU },
	{ 0x1E7F,    1, UNI_GC_LL },
	{ 0x1E80,    1, UNI_GC_LU },
	{ 0x1E81,    1, UNI_GC_LL },
	{ 0x1E82,    1, UNI_GC_LU },
	{ 0x1E83,    1, UNI_GC_LL },
	{ 0x1E84,    1, UNI_GC_LU },
	{ 0x1E85,    1, UNI_GC_LL },
	{ 0x1E86,    1, UNI_GC_LU },
	{ 0x1E87,    1, UNI_GC_LL },
	{ 0x1E88,    1, UNI_GC_LU },
	{ 0x1E89,    1, UNI_GC_LL },
	{ 0x1E8A,    1, UNI_GC_LU },
	{ 0x1E8B,    1, UNI_GC_LL },
	{ 0x1E8C,    1, UNI_GC_LU },
	{ 0x1E8D,    1, UNI_GC_LL },
	{ 0x1E8E,    1, UNI_GC_LU },
	{ 0x1E8F,    1, UNI_GC_LL },
	{ 0x1E90,    1, UNI_GC_LU },
	{ 0x1E91,    1, UNI_GC_LL },
	{ 0x1E92,    1, UNI_GC_LU },
	{ 0x1E93,    1, UNI_GC_LL },
	{ 0x1E94,    1, UNI_GC_LU },
	{ 0x1E95,    7, UNI_GC_LL },
	{ 0x1EA0,    1, UNI_GC_LU },
	{ 0x1EA1,    1, UNI_GC_LL },
	{ 0x1EA2,    1, UNI_GC_LU },
	{ 0x1EA3,    1, UNI_GC_LL },
	{ 0x1EA4,    1, UNI_GC_LU },
	{ 0x1EA5,    1, UNI_GC_LL },
	{ 0x1EA6,    1, UNI_GC_LU },
	{ 0x1EA7,    1, UNI_GC_LL },
	{ 0x1EA8,    1, UNI_GC_LU },
	{ 0x1EA9,    1, UNI_GC_LL },
	{ 0x1EAA,    1, UNI_GC_LU },
	{ 0x1EAB,    1, UNI_GC_LL },
	{ 0x1EAC,    1, UNI_GC_LU },
	{ 0x1EAD,    1, UNI_GC_LL },
	{ 0x1EAE,    1, UNI_GC_LU },
	{ 0x1EAF,    1, UNI_GC_LL },
	{ 0x1EB0,    1, UNI_GC_LU },
	{ 0x1EB1,    1, UNI_GC_LL },
	{ 0x1EB2,    1, UNI_GC_LU },
	{ 0x1EB3,    1, UNI_GC_LL },
	{ 0x1EB4,    1, UNI_GC_LU },
	{ 0x1EB5,    1, UNI_GC_LL },
	{ 0x1EB6,    1, UNI_GC_LU },
	{ 0x1EB7,    1, UNI_GC_LL },
	{ 0x1EB8,    1, UNI_GC_LU },
	{ 0x1EB9,    1, UNI_GC_LL },
	{ 0x1EBA,    1, UNI_GC_LU },
	{ 0x1EBB,    1, UNI_GC_LL },
	{ 0x1EBC,    1, UNI_GC_LU },
	{ 0x1EBD,    1, UNI_GC_LL },
	{ 0x1EBE,    1, UNI_GC_LU },
	{ 0x1EBF,    1, UNI_GC_LL },
	{ 0x1EC0,    1, UNI_GC_LU },
	{ 0x1EC1,    1, UNI_GC_LL },
	{ 0x1EC2,    1, UNI_GC_LU },
	{ 0x1EC3,    1, UNI_GC_LL },
	{ 0x1EC4,    1, UNI_GC_LU },
	{ 0x1EC5,    1, UNI_GC_LL },
	{ 0x1EC6,    1, UNI_GC_LU },
	{ 0x1EC7,    1, UNI_GC_LL },
	{ 0x1EC8,    1, UNI_GC_LU },
	{ 0x1EC9,    1, UNI_GC_LL },
	{ 0x1ECA,    1, UNI_GC_LU },
	{ 0x1ECB,    1, UNI_GC_LL },
	{ 0x1ECC,    1, UNI_GC_LU },
	{ 0x1ECD,    1, UNI_GC_LL },
	{ 0x1ECE,    1, UNI_GC_LU },
	{ 0x1ECF,    1, UNI_GC_LL },
	{ 0x1ED0,    1, UNI_GC_LU },
	{ 0x1ED1,    1, UNI_GC_LL },
	{ 0x1ED2,    1, UNI_GC_LU },
	{ 0x1ED3,    1, UNI_GC_LL },
	{ 0x1ED4,    1, UNI_GC_LU },
	{ 0x1ED5,    1, UNI_GC_LL },
	{ 0x1ED6,    1, UNI_GC_LU },
	{ 0x1ED7,    1, UNI_GC_LL },
	{ 0x1ED8,    1, UNI_GC_LU },
	{ 0x1ED9,    1, UNI_GC_LL },
	{ 0x1EDA,    1, UNI_GC_LU },
	{ 0x1EDB,    1, UNI_GC_LL },
	{ 0x1EDC,    1, UNI_GC_LU },
	{ 0x1EDD,    1, UNI_GC_LL },
	{ 0x1EDE,    1, UNI_GC_LU },
	{ 0x1EDF,    1, UNI_GC_LL },
	{ 0x1EE0,    1, UNI_GC_LU },
	{ 0x1EE1,    1, UNI_GC_LL },
	{ 0x1EE2,    1, UNI_GC_LU },
	{ 0x1EE3,    1, UNI_GC_LL },
	{ 0x1EE4,    1, UNI_GC_LU },
	{ 0x1EE5,    1, UNI_GC_LL },
	{ 0x1EE6,    1, UNI_GC_LU },
	{ 0x1EE7,    1, UNI_GC_LL },
	{ 0x1EE8,    1, UNI_GC_LU },
	{ 0x1EE9,    1, UNI_GC_LL },
	{ 0x1EEA,    1, UNI_GC_LU },
	{ 0x1EEB,    1, UNI_GC_LL },
	{ 0x1EEC,    1, UNI_GC_LU },
	{ 0x1EED,    1, UNI_GC_LL },
	{ 0x1EEE,    1, UNI_GC_LU },
	{ 0x1EEF,    1, UNI_GC_LL },
	{ 0x1EF0,    1, UNI_GC_LU },
	{ 0x1EF1,    1, UNI_GC_LL },
	{ 0x1EF2,    1, UNI_GC_LU },
	{ 0x1EF3,    1, UNI_GC_LL },
	{ 0x1EF4,    1, UNI_GC_LU },
	{ 0x1EF5,    1, UNI_GC_LL },
	{ 0x1EF6,    1, UNI_GC_LU },
	{ 0x1EF7,    1, UNI_GC_LL },
	{ 0x1EF8,    1, UNI_GC_LU },
	{ 0x1EF9,    1, UNI_GC_LL },
	{ 0x1F00,    8, UNI_GC_LL },
	{ 0x1F08,    8, UNI_GC_LU },
	{ 0x1F10,    6, UNI_GC_LL },
	{ 0x1F18,    6, UNI_GC_LU },
	{ 0x1F20,    8, UNI_GC_LL },
	{ 0x1F28,    8, UNI_GC_LU },
	{ 0x1F30,    8, UNI_GC_LL },
	{ 0x1F38,    8, UNI_GC_LU },
	{ 0x1F40,    6, UNI_GC_LL },
	{ 0x1F48,    6, UNI_GC_LU },
	{ 0x1F50,    8, UNI_GC_LL },
	{ 0x1F59,    1, UNI_GC_LU },
	{ 0x1F5B,    1, UNI_GC_LU },
	{ 0x1F5D,    1, UNI_GC_LU },
	{ 0x1F5F,    1, UNI_GC_LU },
	{ 0x1F60,    8, UNI_GC_LL },
	{ 0x1F68,    8, UNI_GC_LU },
	{ 0x1F70,   14, UNI_GC_LL },
	{ 0x1F80,    8, UNI_GC_LL },
	{ 0x1F88,    8, UNI_GC_LT },
	{ 0x1F90,    8, UNI_GC_LL },
	{ 0x1F98,    8, UNI_GC_LT },
	{ 0x1FA0,    8, UNI_GC_LL },
	{ 0x1FA8,    8, UNI_GC_LT },
	{ 0x1FB0,    5, UNI_GC_LL },
	{ 0x1FB6,    2, UNI_GC_LL },
	{ 0x1FB8,    4, UNI_GC_LU },
	{ 0x1FBC,    1, UNI_GC_LT },
	{ 0x1FBD,    1, UNI_GC_SK },
	{ 0x1FBE,    1, UNI_GC_LL },
	{ 0x1FBF,    3, UNI_GC_SK },
	{ 0x1FC2,    3, UNI_GC_LL },
	{ 0x1FC6,    2, UNI_GC_LL },
	{ 0x1FC8,    4, UNI_GC_LU },
	{ 0x1FCC,    1, UNI_GC_LT },
	{ 0x1FCD,    3, UNI_GC_SK },
	{ 0x1FD0,    4, UNI_GC_LL },
	{ 0x1FD6,    2, UNI_GC_LL },
	{ 0x1FD8,    4, UNI_GC_LU },
	{ 0x1FDD,    3, UNI_GC_SK },
	{ 0x1FE0,    8, UNI_GC_LL },
	{ 0x1FE8,    5, UNI_GC_LU },
	{ 0x1FED,    3, UNI_GC_SK },
	{ 0x1FF2,    3, UNI_GC_LL },
	{ 0x1FF6,    2, UNI_GC_LL },
	{ 0x1FF8,    4, UNI_GC_LU },
	{ 0x1FFC,    1, UNI_GC_LT },
	{ 0x1FFD,    2, UNI_GC_SK },
	{ 0x2000,   11, UNI_GC_ZS },
	{ 0x200B,    5, UNI_GC_CF },
	{ 0x2010,    6, UNI_GC_PD },
	{ 0x2016,    2, UNI_GC_PO },
	{ 0x2018,    1, UNI_GC_PI },
	{ 0x2019,    1, UNI_GC_PF },
	{ 0x201A,    1, UNI_GC_PS },
	{ 0x201B,    2, UNI_GC_PI },
	{ 0x201D,    1, UNI_GC_PF },
	{ 0x201E,    1, UNI_GC_PS },
	{ 0x201F,    1, UNI_GC_PI },
	{ 0x2020,    8, UNI_GC_PO },
	{ 0x2028,    1, UNI_GC_ZL },
	{ 0x2029,    1, UNI_GC_ZP },
	{ 0x202A,    5, UNI_GC_CF },
	{ 0x202F,    1, UNI_GC_ZS },
	{ 0x2030,    9, UNI_GC_PO },
	{ 0x2039,    1, UNI_GC_PI },
	{ 0x203A,    1, UNI_GC_PF },
	{ 0x203B,    4, UNI_GC_PO },
	{ 0x203F,    2, UNI_GC_PC },
	{ 0x2041,    3, UNI_GC_PO },
	{ 0x2044,    1, UNI_GC_SM },
	{ 0x2045,    1, UNI_GC_PS },
	{ 0x2046,    1, UNI_GC_PE },
	{ 0x2047,   11, UNI_GC_PO },
	{ 0x2052,    1, UNI_GC_SM },
	{ 0x2053,    1, UNI_GC_PO },
	{ 0x2054,    1, UNI_GC_PC },
	{ 0x2055,   10, UNI_GC_PO },
	{ 0x205F,    1, UNI_GC_ZS },
	{ 0x2060,    4, UNI_GC_CF },
	{ 0x206A,    6, UNI_GC_CF },
	{ 0x2070,    1, UNI_GC_NO },
	{ 0x2071,    1, UNI_GC_LL },
	{ 0x2074,    6, UNI_GC_NO },
	{ 0x207A,    3, UNI_GC_SM },
	{ 0x207D,    1, UNI_GC_PS },
	{ 0x207E,    1, UNI_GC_PE },
	{ 0x207F,    1, UNI_GC_LL },
	{ 0x2080,   10, UNI_GC_NO },
	{ 0x208A,    3, UNI_GC_SM },
	{ 0x208D,    1, UNI_GC_PS },
	{ 0x208E,    1, UNI_GC_PE },
	{ 0x2090,    5, UNI_GC_LM },
	{ 0x20A0,   22, UNI_GC_SC },
	{ 0x20D0,   13, UNI_GC_MN },
	{ 0x20DD,    4, UNI_GC_ME },
	{ 0x20E1,    1, UNI_GC_MN },
	{ 0x20E2,    3, UNI_GC_ME },
	{ 0x20E5,    7, UNI_GC_MN },
	{ 0x2100,    2, UNI_GC_SO },
	{ 0x2102,    1, UNI_GC_LU },
	{ 0x2103,    4, UNI_GC_SO },
	{ 0x2107,    1, UNI_GC_LU },
	{ 0x2108,    2, UNI_GC_SO },
	{ 0x210A,    1, UNI_GC_LL },
	{ 0x210B,    3, UNI_GC_LU },
	{ 0x210E,    2, UNI_GC_LL },
	{ 0x2110,    3, UNI_GC_LU },
	{ 0x2113,    1, UNI_GC_LL },
	{ 0x2114,    1, UNI_GC_SO },
	{ 0x2115,    1, UNI_GC_LU },
	{ 0x2116,    3, UNI_GC_SO },
	{ 0x2119,    5, UNI_GC_LU },
	{ 0x211E,    6, UNI_GC_SO },
	{ 0x2124,    1, UNI_GC_LU },
	{ 0x2125,    1, UNI_GC_SO },
	{ 0x2126,    1, UNI_GC_LU },
	{ 0x2127,    1, UNI_GC_SO },
	{ 0x2128,    1, UNI_GC_LU },
	{ 0x2129,    1, UNI_GC_SO },
	{ 0x212A,    4, UNI_GC_LU },
	{ 0x212E,    1, UNI_GC_SO },
	{ 0x212F,    1, UNI_GC_LL },
	{ 0x2130,    2, UNI_GC_LU },
	{ 0x2132,    1, UNI_GC_SO },
	{ 0x2133,    1, UNI_GC_LU },
	{ 0x2134,    1, UNI_GC_LL },
	{ 0x2135,    4, UNI_GC_LO },
	{ 0x2139,    1, UNI_GC_LL },
	{ 0x213A,    2, UNI_GC_SO },
	{ 0x213C,    2, UNI_GC_LL },
	{ 0x213E,    2, UNI_GC_LU },
	{ 0x2140,    5, UNI_GC_SM },
	{ 0x2145,    1, UNI_GC_LU },
	{ 0x2146,    4, UNI_GC_LL },
	{ 0x214A,    1, UNI_GC_SO },
	{ 0x214B,    1, UNI_GC_SM },
	{ 0x214C,    1, UNI_GC_SO },
	{ 0x2153,   13, UNI_GC_NO },
	{ 0x2160,   36, UNI_GC_NL },
	{ 0x2190,    5, UNI_GC_SM },
	{ 0x2195,    5, UNI_GC_SO },
	{ 0x219A,    2, UNI_GC_SM },
	{ 0x219C,    4, UNI_GC_SO },
	{ 0x21A0,    1, UNI_GC_SM },
	{ 0x21A1,    2, UNI_GC_SO },
	{ 0x21A3,    1, UNI_GC_SM },
	{ 0x21A4,    2, UNI_GC_SO },
	{ 0x21A6,    1, UNI_GC_SM },
	{ 0x21A7,    7, UNI_GC_SO },
	{ 0x21AE,    1, UNI_GC_SM },
	{ 0x21AF,   31, UNI_GC_SO },
	{ 0x21CE,    2, UNI_GC_SM },
	{ 0x21D0,    2, UNI_GC_SO },
	{ 0x21D2,    1, UNI_GC_SM },
	{ 0x21D3,    1, UNI_GC_SO },
	{ 0x21D4,    1, UNI_GC_SM },
	{ 0x21D5,   31, UNI_GC_SO },
	{ 0x21F4,  268, UNI_GC_SM },
	{ 0x2300,    8, UNI_GC_SO },
	{ 0x2308,    4, UNI_GC_SM },
	{ 0x230C,   20, UNI_GC_SO },
	{ 0x2320,    2, UNI_GC_SM },
	{ 0x2322,    7, UNI_GC_SO },
	{ 0x2329,    1, UNI_GC_PS },
	{ 0x232A,    1, UNI_GC_PE },
	{ 0x232B,   81, UNI_GC_SO },
	{ 0x237C,    1, UNI_GC_SM },
	{ 0x237D,   30, UNI_GC_SO },
	{ 0x239B,   25, UNI_GC_SM },
	{ 0x23B4,    1, UNI_GC_PS },
	{ 0x23B5,    1, UNI_GC_PE },
	{ 0x23B6,    1, UNI_GC_PO },
	{ 0x23B7,   37, UNI_GC_SO },
	{ 0x2400,   39, UNI_GC_SO },
	{ 0x2440,   11, UNI_GC_SO },
	{ 0x2460,   60, UNI_GC_NO },
	{ 0x249C,   78, UNI_GC_SO },
	{ 0x24EA,   22, UNI_GC_NO },
	{ 0x2500,  183, UNI_GC_SO },
	{ 0x25B7,    1, UNI_GC_SM },
	{ 0x25B8,    9, UNI_GC_SO },
	{ 0x25C1,    1, UNI_GC_SM },
	{ 0x25C2,   54, UNI_GC_SO },
	{ 0x25F8,    8, UNI_GC_SM },
	{ 0x2600,  111, UNI_GC_SO },
	{ 0x266F,    1, UNI_GC_SM },
	{ 0x2670,   45, UNI_GC_SO },
	{ 0x26A0,   18, UNI_GC_SO },
	{ 0x2701,    4, UNI_GC_SO },
	{ 0x2706,    4, UNI_GC_SO },
	{ 0x270C,   28, UNI_GC_SO },
	{ 0x2729,   35, UNI_GC_SO },
	{ 0x274D,    1, UNI_GC_SO },
	{ 0x274F,    4, UNI_GC_SO },
	{ 0x2756,    1, UNI_GC_SO },
	{ 0x2758,    7, UNI_GC_SO },
	{ 0x2761,    7, UNI_GC_SO },
	{ 0x2768,    1, UNI_GC_PS },
	{ 0x2769,    1, UNI_GC_PE },
	{ 0x276A,    1, UNI_GC_PS },
	{ 0x276B,    1, UNI_GC_PE },
	{ 0x276C,    1, UNI_GC_PS },
	{ 0x276D,    1, UNI_GC_PE },
	{ 0x276E,    1, UNI_GC_PS },
	{ 0x276F,    1, UNI_GC_PE },
	{ 0x2770,    1, UNI_GC_PS },
	{ 0x2771,    1, UNI_GC_PE },
	{ 0x2772,    1, UNI_GC_PS },
	{ 0x2773,    1, UNI_GC_PE },
	{ 0x2774,    1, UNI_GC_PS },
	{ 0x2775,    1, UNI_GC_PE },
	{ 0x2776,   30, UNI_GC_NO },
	{ 0x2794,    1, UNI_GC_SO },
	{ 0x2798,   24, UNI_GC_SO },
	{ 0x27B1,   14, UNI_GC_SO },
	{ 0x27C0,    5, UNI_GC_SM },
	{ 0x27C5,    1, UNI_GC_PS },
	{ 0x27C6,    1, UNI_GC_PE },
	{ 0x27D0,   22, UNI_GC_SM },
	{ 0x27E6,    1, UNI_GC_PS },
	{ 0x27E7,    1, UNI_GC_PE },
	{ 0x27E8,    1, UNI_GC_PS },
	{ 0x27E9,    1, UNI_GC_PE },
	{ 0x27EA,    1, UNI_GC_PS },
	{ 0x27EB,    1, UNI_GC_PE },
	{ 0x27F0,   16, UNI_GC_SM },
	{ 0x2800,  256, UNI_GC_SO },
	{ 0x2900,  131, UNI_GC_SM },
	{ 0x2983,    1, UNI_GC_PS },
	{ 0x2984,    1, UNI_GC_PE },
	{ 0x2985,    1, UNI_GC_PS },
	{ 0x2986,    1, UNI_GC_PE },
	{ 0x2987,    1, UNI_GC_PS },
	{ 0x2988,    1, UNI_GC_PE },
	{ 0x2989,    1, UNI_GC_PS },
	{ 0x298A,    1, UNI_GC_PE },
	{ 0x298B,    1, UNI_GC_PS },
	{ 0x298C,    1, UNI_GC_PE },
	{ 0x298D,    1, UNI_GC_PS },
	{ 0x298E,    1, UNI_GC_PE },
	{ 0x298F,    1, UNI_GC_PS },
	{ 0x2990,    1, UNI_GC_PE },
	{ 0x2991,    1, UNI_GC_PS },
	{ 0x2992,    1, UNI_GC_PE },
	{ 0x2993,    1, UNI_GC_PS },
	{ 0x2994,    1, UNI_GC_PE },
	{ 0x2995,    1, UNI_GC_PS },
	{ 0x2996,    1, UNI_GC_PE },
	{ 0x2997,    1, UNI_GC_PS },
	{ 0x2998,    1, UNI_GC_PE },
	{ 0x2999,   63, UNI_GC_SM },
	{ 0x29D8,    1, UNI_GC_PS },
	{ 0x29D9,    1, UNI_GC_PE },
	{ 0x29DA,    1, UNI_GC_PS },
	{ 0x29DB,    1, UNI_GC_PE },
	{ 0x29DC,   32, UNI_GC_SM },
	{ 0x29FC,    1, UNI_GC_PS },
	{ 0x29FD,    1, UNI_GC_PE },
	{ 0x29FE,  258, UNI_GC_SM },
	{ 0x2B00,   20, UNI_GC_SO },
	{ 0x2C00,   47, UNI_GC_LU },
	{ 0x2C30,   47, UNI_GC_LL },
	{ 0x2C80,    1, UNI_GC_LU },
	{ 0x2C81,    1, UNI_GC_LL },
	{ 0x2C82,    1, UNI_GC_LU },
	{ 0x2C83,    1, UNI_GC_LL },
	{ 0x2C84,    1, UNI_GC_LU },
	{ 0x2C85,    1, UNI_GC_LL },
	{ 0x2C86,    1, UNI_GC_LU },
	{ 0x2C87,    1, UNI_GC_LL },
	{ 0x2C88,    1, UNI_GC_LU },
	{ 0x2C89,    1, UNI_GC_LL },
	{ 0x2C8A,    1, UNI_GC_LU },
	{ 0x2C8B,    1, UNI_GC_LL },
	{ 0x2C8C,    1, UNI_GC_LU },
	{ 0x2C8D,    1, UNI_GC_LL },
	{ 0x2C8E,    1, UNI_GC_LU },
	{ 0x2C8F,    1, UNI_GC_LL },
	{ 0x2C90,    1, UNI_GC_LU },
	{ 0x2C91,    1, UNI_GC_LL },
	{ 0x2C92,    1, UNI_GC_LU },
	{ 0x2C93,    1, UNI_GC_LL },
	{ 0x2C94,    1, UNI_GC_LU },
	{ 0x2C95,    1, UNI_GC_LL },
	{ 0x2C96,    1, UNI_GC_LU },
	{ 0x2C97,    1, UNI_GC_LL },
	{ 0x2C98,    1, UNI_GC_LU },
	{ 0x2C99,    1, UNI_GC_LL },
	{ 0x2C9A,    1, UNI_GC_LU },
	{ 0x2C9B,    1, UNI_GC_LL },
	{ 0x2C9C,    1, UNI_GC_LU },
	{ 0x2C9D,    1, UNI_GC_LL },
	{ 0x2C9E,    1, UNI_GC_LU },
	{ 0x2C9F,    1, UNI_GC_LL },
	{ 0x2CA0,    1, UNI_GC_LU },
	{ 0x2CA1,    1, UNI_GC_LL },
	{ 0x2CA2,    1, UNI_GC_LU },
	{ 0x2CA3,    1, UNI_GC_LL },
	{ 0x2CA4,    1, UNI_GC_LU },
	{ 0x2CA5,    1, UNI_GC_LL },
	{ 0x2CA6,    1, UNI_GC_LU },
	{ 0x2CA7,    1, UNI_GC_LL },
	{ 0x2CA8,    1, UNI_GC_LU },
	{ 0x2CA9,    1, UNI_GC_LL },
	{ 0x2CAA,    1, UNI_GC_LU },
	{ 0x2CAB,    1, UNI_GC_LL },
	{ 0x2CAC,    1, UNI_GC_LU },
	{ 0x2CAD,    1, UNI_GC_LL },
	{ 0x2CAE,    1, UNI_GC_LU },
	{ 0x2CAF,    1, UNI_GC_LL },
	{ 0x2CB0,    1, UNI_GC_LU },
	{ 0x2CB1,    1, UNI_GC_LL },
	{ 0x2CB2,    1, UNI_GC_LU },
	{ 0x2CB3,    1, UNI_GC_LL },
	{ 0x2CB4,    1, UNI_GC_LU },
	{ 0x2CB5,    1, UNI_GC_LL },
	{ 0x2CB6,    1, UNI_GC_LU },
	{ 0x2CB7,    1, UNI_GC_LL },
	{ 0x2CB8,    1, UNI_GC_LU },
	{ 0x2CB9,    1, UNI_GC_LL },
	{ 0x2CBA,    1, UNI_GC_LU },
	{ 0x2CBB,    1, UNI_GC_LL },
	{ 0x2CBC,    1, UNI_GC_LU },
	{ 0x2CBD,    1, UNI_GC_LL },
	{ 0x2CBE,    1, UNI_GC_LU },
	{ 0x2CBF,    1, UNI_GC_LL },
	{ 0x2CC0,    1, UNI_GC_LU },
	{ 0x2CC1,    1, UNI_GC_LL },
	{ 0x2CC2,    1, UNI_GC_LU },
	{ 0x2CC3,    1, UNI_GC_LL },
	{ 0x2CC4,    1, UNI_GC_LU },
	{ 0x2CC5,    1, UNI_GC_LL },
	{ 0x2CC6,    1, UNI_GC_LU },
	{ 0x2CC7,    1, UNI_GC_LL },
	{ 0x2CC8,    1, UNI_GC_LU },
	{ 0x2CC9,    1, UNI_GC_LL },
	{ 0x2CCA,    1, UNI_GC_LU },
	{ 0x2CCB,    1, UNI_GC_LL },
	{ 0x2CCC,    1, UNI_GC_LU },
	{ 0x2CCD,    1, UNI_GC_LL },
	{ 0x2CCE,    1, UNI_GC_LU },
	{ 0x2CCF,    1, UNI_GC_LL },
	{ 0x2CD0,    1, UNI_GC_LU },
	{ 0x2CD1,    1, UNI_GC_LL },
	{ 0x2CD2,    1, UNI_GC_LU },
	{ 0x2CD3,    1, UNI_GC_LL },
	{ 0x2CD4,    1, UNI_GC_LU },
	{ 0x2CD5,    1, UNI_GC_LL },
	{ 0x2CD6,    1, UNI_GC_LU },
	{ 0x2CD7,    1, UNI_GC_LL },
	{ 0x2CD8,    1, UNI_GC_LU },
	{ 0x2CD9,    1, UNI_GC_LL },
	{ 0x2CDA,    1, UNI_GC_LU },
	{ 0x2CDB,    1, UNI_GC_LL },
	{ 0x2CDC,    1, UNI_GC_LU },
	{ 0x2CDD,    1, UNI_GC_LL },
	{ 0x2CDE,    1, UNI_GC_LU },
	{ 0x2CDF,    1, UNI_GC_LL },
	{ 0x2CE0,    1, UNI_GC_LU },
	{ 0x2CE1,    1, UNI_GC_LL },
	{ 0x2CE2,    1, UNI_GC_LU },
	{ 0x2CE3,    2, UNI_GC_LL },
	{ 0x2CE5,    6, UNI_GC_SO },
	{ 0x2CF9,    4, UNI_GC_PO },
	{ 0x2CFD,    1, UNI_GC_NO },
	{ 0x2CFE,    2, UNI_GC_PO },
	{ 0x2D00,   38, UNI_GC_LL },
	{ 0x2D30,   54, UNI_GC_LO },
	{ 0x2D6F,    1, UNI_GC_LM },
	{ 0x2D80,   23, UNI_GC_LO },
	{ 0x2DA0,    7, UNI_GC_LO },
	{ 0x2DA8,    7, UNI_GC_LO },
	{ 0x2DB0,    7, UNI_GC_LO },
	{ 0x2DB8,    7, UNI_GC_LO },
	{ 0x2DC0,    7, UNI_GC_LO },
	{ 0x2DC8,    7, UNI_GC_LO },
	{ 0x2DD0,    7, UNI_GC_LO },
	{ 0x2DD8,    7, UNI_GC_LO },
	{ 0x2E00,    2, UNI_GC_PO },
	{ 0x2E02,    1, UNI_GC_PI },
	{ 0x2E03,    1, UNI_GC_PF },
	{ 0x2E04,    1, UNI_GC_PI },
	{ 0x2E05,    1, UNI_GC_PF },
	{ 0x2E06,    3, UNI_GC_PO },
	{ 0x2E09,    1, UNI_GC_PI },
	{ 0x2E0A,    1, UNI_GC_PF },
	{ 0x2E0B,    1, UNI_GC_PO },
	{ 0x2E0C,    1, UNI_GC_PI },
	{ 0x2E0D,    1, UNI_GC_PF },
	{ 0x2E0E,    9, UNI_GC_PO },
	{ 0x2E17,    1, UNI_GC_PD },
	{ 0x2E1C,    1, UNI_GC_PI },
	{ 0x2E1D,    1, UNI_GC_PF },
	{ 0x2E80,   26, UNI_GC_SO },
	{ 0x2E9B,   89, UNI_GC_SO },
	{ 0x2F00,  214, UNI_GC_SO },
	{ 0x2FF0,   12, UNI_GC_SO },
	{ 0x3000,    1, UNI_GC_ZS },
	{ 0x3001,    3, UNI_GC_PO },
	{ 0x3004,    1, UNI_GC_SO },
	{ 0x3005,    1, UNI_GC_LM },
	{ 0x3006,    1, UNI_GC_LO },
	{ 0x3007,    1, UNI_GC_NL },
	{ 0x3008,    1, UNI_GC_PS },
	{ 0x3009,    1, UNI_GC_PE },
	{ 0x300A,    1, UNI_GC_PS },
	{ 0x300B,    1, UNI_GC_PE },
	{ 0x300C,    1, UNI_GC_PS },
	{ 0x300D,    1, UNI_GC_PE },
	{ 0x300E,    1, UNI_GC_PS },
	{ 0x300F,    1, UNI_GC_PE },
	{ 0x3010,    1, UNI_GC_PS },
	{ 0x3011,    1, UNI_GC_PE },
	{ 0x3012,    2, UNI_GC_SO },
	{ 0x3014,    1, UNI_GC_PS },
	{ 0x3015,    1, UNI_GC_PE },
	{ 0x3016,    1, UNI_GC_PS },
	{ 0x3017,    1, UNI_GC_PE },
	{ 0x3018,    1, UNI_GC_PS },
	{ 0x3019,    1, UNI_GC_PE },
	{ 0x301A,    1, UNI_GC_PS },
	{ 0x301B,    1, UNI_GC_PE },
	{ 0x301C,    1, UNI_GC_PD },
	{ 0x301D,    1, UNI_GC_PS },
	{ 0x301E,    2, UNI_GC_PE },
	{ 0x3020,    1, UNI_GC_SO },
	{ 0x3021,    9, UNI_GC_NL },
	{ 0x302A,    6, UNI_GC_MN },
	{ 0x3030,    1, UNI_GC_PD },
	{ 0x3031,    5, UNI_GC_LM },
	{ 0x3036,    2, UNI_GC_SO },
	{ 0x3038,    3, UNI_GC_NL },
	{ 0x303B,    1, UNI_GC_LM },
	{ 0x303C,    1, UNI_GC_LO },
	{ 0x303D,    1, UNI_GC_PO },
	{ 0x303E,    2, UNI_GC_SO },
	{ 0x3041,   86, UNI_GC_LO },
	{ 0x3099,    2, UNI_GC_MN },
	{ 0x309B,    2, UNI_GC_SK },
	{ 0x309D,    2, UNI_GC_LM },
	{ 0x309F,    1, UNI_GC_LO },
	{ 0x30A0,    1, UNI_GC_PD },
	{ 0x30A1,   90, UNI_GC_LO },
	{ 0x30FB,    1, UNI_GC_PO },
	{ 0x30FC,    3, UNI_GC_LM },
	{ 0x30FF,    1, UNI_GC_LO },
	{ 0x3105,   40, UNI_GC_LO },
	{ 0x3131,   94, UNI_GC_LO },
	{ 0x3190,    2, UNI_GC_SO },
	{ 0x3192,    4, UNI_GC_NO },
	{ 0x3196,   10, UNI_GC_SO },
	{ 0x31A0,   24, UNI_GC_LO },
	{ 0x31C0,   16, UNI_GC_SO },
	{ 0x31F0,   16, UNI_GC_LO },
	{ 0x3200,   31, UNI_GC_SO },
	{ 0x3220,   10, UNI_GC_NO },
	{ 0x322A,   26, UNI_GC_SO },
	{ 0x3250,    1, UNI_GC_SO },
	{ 0x3251,   15, UNI_GC_NO },
	{ 0x3260,   32, UNI_GC_SO },
	{ 0x3280,   10, UNI_GC_NO },
	{ 0x328A,   39, UNI_GC_SO },
	{ 0x32B1,   15, UNI_GC_NO },
	{ 0x32C0,   63, UNI_GC_SO },
	{ 0x3300,  256, UNI_GC_SO },
	{ 0x3400,    1, UNI_GC_LO },
	{ 0x4DB5,    1, UNI_GC_LO },
	{ 0x4DC0,   64, UNI_GC_SO },
	{ 0x4E00,    1, UNI_GC_LO },
	{ 0x9FBB,    1, UNI_GC_LO },
	{ 0xA000,   21, UNI_GC_LO },
	{ 0xA015,    1, UNI_GC_LM },
	{ 0xA016, 1143, UNI_GC_LO },
	{ 0xA490,   55, UNI_GC_SO },
	{ 0xA700,   23, UNI_GC_SK },
	{ 0xA800,    2, UNI_GC_LO },
	{ 0xA802,    1, UNI_GC_MC },
	{ 0xA803,    3, UNI_GC_LO },
	{ 0xA806,    1, UNI_GC_MN },
	{ 0xA807,    4, UNI_GC_LO },
	{ 0xA80B,    1, UNI_GC_MN },
	{ 0xA80C,   23, UNI_GC_LO },
	{ 0xA823,    2, UNI_GC_MC },
	{ 0xA825,    2, UNI_GC_MN },
	{ 0xA827,    1, UNI_GC_MC },
	{ 0xA828,    4, UNI_GC_SO },
	{ 0xAC00,    1, UNI_GC_LO },
	{ 0xD7A3,    1, UNI_GC_LO },
	{ 0xD800,    1, UNI_GC_CS },
	{ 0xDB7F,    2, UNI_GC_CS },
	{ 0xDBFF,    2, UNI_GC_CS },
	{ 0xDFFF,    1, UNI_GC_CS },
	{ 0xE000,    1, UNI_GC_CO },
	{ 0xF8FF,    1, UNI_GC_CO },
	{ 0xF900,  302, UNI_GC_LO },
	{ 0xFA30,   59, UNI_GC_LO },
	{ 0xFA70,  106, UNI_GC_LO },
	{ 0xFB00,    7, UNI_GC_LL },
	{ 0xFB13,    5, UNI_GC_LL },
	{ 0xFB1D,    1, UNI_GC_LO },
	{ 0xFB1E,    1, UNI_GC_MN },
	{ 0xFB1F,   10, UNI_GC_LO },
	{ 0xFB29,    1, UNI_GC_SM },
	{ 0xFB2A,   13, UNI_GC_LO },
	{ 0xFB38,    5, UNI_GC_LO },
	{ 0xFB3E,    1, UNI_GC_LO },
	{ 0xFB40,    2, UNI_GC_LO },
	{ 0xFB43,    2, UNI_GC_LO },
	{ 0xFB46,  108, UNI_GC_LO },
	{ 0xFBD3,  363, UNI_GC_LO },
	{ 0xFD3E,    1, UNI_GC_PS },
	{ 0xFD3F,    1, UNI_GC_PE },
	{ 0xFD50,   64, UNI_GC_LO },
	{ 0xFD92,   54, UNI_GC_LO },
	{ 0xFDF0,   12, UNI_GC_LO },
	{ 0xFDFC,    1, UNI_GC_SC },
	{ 0xFDFD,    1, UNI_GC_SO },
	{ 0xFE00,   16, UNI_GC_MN },
	{ 0xFE10,    7, UNI_GC_PO },
	{ 0xFE17,    1, UNI_GC_PS },
	{ 0xFE18,    1, UNI_GC_PE },
	{ 0xFE19,    1, UNI_GC_PO },
	{ 0xFE20,    4, UNI_GC_MN },
	{ 0xFE30,    1, UNI_GC_PO },
	{ 0xFE31,    2, UNI_GC_PD },
	{ 0xFE33,    2, UNI_GC_PC },
	{ 0xFE35,    1, UNI_GC_PS },
	{ 0xFE36,    1, UNI_GC_PE },
	{ 0xFE37,    1, UNI_GC_PS },
	{ 0xFE38,    1, UNI_GC_PE },
	{ 0xFE39,    1, UNI_GC_PS },
	{ 0xFE3A,    1, UNI_GC_PE },
	{ 0xFE3B,    1, UNI_GC_PS },
	{ 0xFE3C,    1, UNI_GC_PE },
	{ 0xFE3D,    1, UNI_GC_PS },
	{ 0xFE3E,    1, UNI_GC_PE },
	{ 0xFE3F,    1, UNI_GC_PS },
	{ 0xFE40,    1, UNI_GC_PE },
	{ 0xFE41,    1, UNI_GC_PS },
	{ 0xFE42,    1, UNI_GC_PE },
	{ 0xFE43,    1, UNI_GC_PS },
	{ 0xFE44,    1, UNI_GC_PE },
	{ 0xFE45,    2, UNI_GC_PO },
	{ 0xFE47,    1, UNI_GC_PS },
	{ 0xFE48,    1, UNI_GC_PE },
	{ 0xFE49,    4, UNI_GC_PO },
	{ 0xFE4D,    3, UNI_GC_PC },
	{ 0xFE50,    3, UNI_GC_PO },
	{ 0xFE54,    4, UNI_GC_PO },
	{ 0xFE58,    1, UNI_GC_PD },
	{ 0xFE59,    1, UNI_GC_PS },
	{ 0xFE5A,    1, UNI_GC_PE },
	{ 0xFE5B,    1, UNI_GC_PS },
	{ 0xFE5C,    1, UNI_GC_PE },
	{ 0xFE5D,    1, UNI_GC_PS },
	{ 0xFE5E,    1, UNI_GC_PE },
	{ 0xFE5F,    3, UNI_GC_PO },
	{ 0xFE62,    1, UNI_GC_SM },
	{ 0xFE63,    1, UNI_GC_PD },
	{ 0xFE64,    3, UNI_GC_SM },
	{ 0xFE68,    1, UNI_GC_PO },
	{ 0xFE69,    1, UNI_GC_SC },
	{ 0xFE6A,    2, UNI_GC_PO },
	{ 0xFE70,    5, UNI_GC_LO },
	{ 0xFE76,  135, UNI_GC_LO },
	{ 0xFEFF,    1, UNI_GC_CF },
	{ 0xFF01,    3, UNI_GC_PO },
	{ 0xFF04,    1, UNI_GC_SC },
	{ 0xFF05,    3, UNI_GC_PO },
	{ 0xFF08,    1, UNI_GC_PS },
	{ 0xFF09,    1, UNI_GC_PE },
	{ 0xFF0A,    1, UNI_GC_PO },
	{ 0xFF0B,    1, UNI_GC_SM },
	{ 0xFF0C,    1, UNI_GC_PO },
	{ 0xFF0D,    1, UNI_GC_PD },
	{ 0xFF0E,    2, UNI_GC_PO },
	{ 0xFF10,   10, UNI_GC_ND },
	{ 0xFF1A,    2, UNI_GC_PO },
	{ 0xFF1C,    3, UNI_GC_SM },
	{ 0xFF1F,    2, UNI_GC_PO },
	{ 0xFF21,   26, UNI_GC_LU },
	{ 0xFF3B,    1, UNI_GC_PS },
	{ 0xFF3C,    1, UNI_GC_PO },
	{ 0xFF3D,    1, UNI_GC_PE },
	{ 0xFF3E,    1, UNI_GC_SK },
	{ 0xFF3F,    1, UNI_GC_PC },
	{ 0xFF40,    1, UNI_GC_SK },
	{ 0xFF41,   26, UNI_GC_LL },
	{ 0xFF5B,    1, UNI_GC_PS },
	{ 0xFF5C,    1, UNI_GC_SM },
	{ 0xFF5D,    1, UNI_GC_PE },
	{ 0xFF5E,    1, UNI_GC_SM },
	{ 0xFF5F,    1, UNI_GC_PS },
	{ 0xFF60,    1, UNI_GC_PE },
	{ 0xFF61,    1, UNI_GC_PO },
	{ 0xFF62,    1, UNI_GC_PS },
	{ 0xFF63,    1, UNI_GC_PE },
	{ 0xFF64,    2, UNI_GC_PO },
	{ 0xFF66,   10, UNI_GC_LO },
	{ 0xFF70,    1, UNI_GC_LM },
	{ 0xFF71,   45, UNI_GC_LO },
	{ 0xFF9E,    2, UNI_GC_LM },
	{ 0xFFA0,   31, UNI_GC_LO },
	{ 0xFFC2,    6, UNI_GC_LO },
	{ 0xFFCA,    6, UNI_GC_LO },
	{ 0xFFD2,    6, UNI_GC_LO },
	{ 0xFFDA,    3, UNI_GC_LO },
	{ 0xFFE0,    2, UNI_GC_SC },
	{ 0xFFE2,    1, UNI_GC_SM },
	{ 0xFFE3,    1, UNI_GC_SK },
	{ 0xFFE4,    1, UNI_GC_SO },
	{ 0xFFE5,    2, UNI_GC_SC },
	{ 0xFFE8,    1, UNI_GC_SO },
	{ 0xFFE9,    4, UNI_GC_SM },
	{ 0xFFED,    2, UNI_GC_SO },
	{ 0xFFF9,    3, UNI_GC_CF },
	{ 0xFFFC,    2, UNI_GC_SO },
	{ 0x10000,   12, UNI_GC_LO },
	{ 0x1000D,   26, UNI_GC_LO },
	{ 0x10028,   19, UNI_GC_LO },
	{ 0x1003C,    2, UNI_GC_LO },
	{ 0x1003F,   15, UNI_GC_LO },
	{ 0x10050,   14, UNI_GC_LO },
	{ 0x10080,  123, UNI_GC_LO },
	{ 0x10100,    2, UNI_GC_PO },
	{ 0x10102,    1, UNI_GC_SO },
	{ 0x10107,   45, UNI_GC_NO },
	{ 0x10137,    9, UNI_GC_SO },
	{ 0x10140,   53, UNI_GC_NL },
	{ 0x10175,    4, UNI_GC_NO },
	{ 0x10179,   17, UNI_GC_SO },
	{ 0x1018A,    1, UNI_GC_NO },
	{ 0x10300,   31, UNI_GC_LO },
	{ 0x10320,    4, UNI_GC_NO },
	{ 0x10330,   26, UNI_GC_LO },
	{ 0x1034A,    1, UNI_GC_NL },
	{ 0x10380,   30, UNI_GC_LO },
	{ 0x1039F,    1, UNI_GC_PO },
	{ 0x103A0,   36, UNI_GC_LO },
	{ 0x103C8,    8, UNI_GC_LO },
	{ 0x103D0,    1, UNI_GC_SO },
	{ 0x103D1,    5, UNI_GC_NL },
	{ 0x10400,   40, UNI_GC_LU },
	{ 0x10428,   40, UNI_GC_LL },
	{ 0x10450,   78, UNI_GC_LO },
	{ 0x104A0,   10, UNI_GC_ND },
	{ 0x10800,    6, UNI_GC_LO },
	{ 0x10808,    1, UNI_GC_LO },
	{ 0x1080A,   44, UNI_GC_LO },
	{ 0x10837,    2, UNI_GC_LO },
	{ 0x1083C,    1, UNI_GC_LO },
	{ 0x1083F,    1, UNI_GC_LO },
	{ 0x10A00,    1, UNI_GC_LO },
	{ 0x10A01,    3, UNI_GC_MN },
	{ 0x10A05,    2, UNI_GC_MN },
	{ 0x10A0C,    4, UNI_GC_MN },
	{ 0x10A10,    4, UNI_GC_LO },
	{ 0x10A15,    3, UNI_GC_LO },
	{ 0x10A19,   27, UNI_GC_LO },
	{ 0x10A38,    3, UNI_GC_MN },
	{ 0x10A3F,    1, UNI_GC_MN },
	{ 0x10A40,    8, UNI_GC_NO },
	{ 0x10A50,    9, UNI_GC_PO },
	{ 0x1D000,  246, UNI_GC_SO },
	{ 0x1D100,   39, UNI_GC_SO },
	{ 0x1D12A,   59, UNI_GC_SO },
	{ 0x1D165,    2, UNI_GC_MC },
	{ 0x1D167,    3, UNI_GC_MN },
	{ 0x1D16A,    3, UNI_GC_SO },
	{ 0x1D16D,    6, UNI_GC_MC },
	{ 0x1D173,    8, UNI_GC_CF },
	{ 0x1D17B,    8, UNI_GC_MN },
	{ 0x1D183,    2, UNI_GC_SO },
	{ 0x1D185,    7, UNI_GC_MN },
	{ 0x1D18C,   30, UNI_GC_SO },
	{ 0x1D1AA,    4, UNI_GC_MN },
	{ 0x1D1AE,   48, UNI_GC_SO },
	{ 0x1D200,   66, UNI_GC_SO },
	{ 0x1D242,    3, UNI_GC_MN },
	{ 0x1D245,    1, UNI_GC_SO },
	{ 0x1D300,   87, UNI_GC_SO },
	{ 0x1D400,   26, UNI_GC_LU },
	{ 0x1D41A,   26, UNI_GC_LL },
	{ 0x1D434,   26, UNI_GC_LU },
	{ 0x1D44E,    7, UNI_GC_LL },
	{ 0x1D456,   18, UNI_GC_LL },
	{ 0x1D468,   26, UNI_GC_LU },
	{ 0x1D482,   26, UNI_GC_LL },
	{ 0x1D49C,    1, UNI_GC_LU },
	{ 0x1D49E,    2, UNI_GC_LU },
	{ 0x1D4A2,    1, UNI_GC_LU },
	{ 0x1D4A5,    2, UNI_GC_LU },
	{ 0x1D4A9,    4, UNI_GC_LU },
	{ 0x1D4AE,    8, UNI_GC_LU },
	{ 0x1D4B6,    4, UNI_GC_LL },
	{ 0x1D4BB,    1, UNI_GC_LL },
	{ 0x1D4BD,    7, UNI_GC_LL },
	{ 0x1D4C5,   11, UNI_GC_LL },
	{ 0x1D4D0,   26, UNI_GC_LU },
	{ 0x1D4EA,   26, UNI_GC_LL },
	{ 0x1D504,    2, UNI_GC_LU },
	{ 0x1D507,    4, UNI_GC_LU },
	{ 0x1D50D,    8, UNI_GC_LU },
	{ 0x1D516,    7, UNI_GC_LU },
	{ 0x1D51E,   26, UNI_GC_LL },
	{ 0x1D538,    2, UNI_GC_LU },
	{ 0x1D53B,    4, UNI_GC_LU },
	{ 0x1D540,    5, UNI_GC_LU },
	{ 0x1D546,    1, UNI_GC_LU },
	{ 0x1D54A,    7, UNI_GC_LU },
	{ 0x1D552,   26, UNI_GC_LL },
	{ 0x1D56C,   26, UNI_GC_LU },
	{ 0x1D586,   26, UNI_GC_LL },
	{ 0x1D5A0,   26, UNI_GC_LU },
	{ 0x1D5BA,   26, UNI_GC_LL },
	{ 0x1D5D4,   26, UNI_GC_LU },
	{ 0x1D5EE,   26, UNI_GC_LL },
	{ 0x1D608,   26, UNI_GC_LU },
	{ 0x1D622,   26, UNI_GC_LL },
	{ 0x1D63C,   26, UNI_GC_LU },
	{ 0x1D656,   26, UNI_GC_LL },
	{ 0x1D670,   26, UNI_GC_LU },
	{ 0x1D68A,   28, UNI_GC_LL },
	{ 0x1D6A8,   25, UNI_GC_LU },
	{ 0x1D6C1,    1, UNI_GC_SM },
	{ 0x1D6C2,   25, UNI_GC_LL },
	{ 0x1D6DB,    1, UNI_GC_SM },
	{ 0x1D6DC,    6, UNI_GC_LL },
	{ 0x1D6E2,   25, UNI_GC_LU },
	{ 0x1D6FB,    1, UNI_GC_SM },
	{ 0x1D6FC,   25, UNI_GC_LL },
	{ 0x1D715,    1, UNI_GC_SM },
	{ 0x1D716,    6, UNI_GC_LL },
	{ 0x1D71C,   25, UNI_GC_LU },
	{ 0x1D735,    1, UNI_GC_SM },
	{ 0x1D736,   25, UNI_GC_LL },
	{ 0x1D74F,    1, UNI_GC_SM },
	{ 0x1D750,    6, UNI_GC_LL },
	{ 0x1D756,   25, UNI_GC_LU },
	{ 0x1D76F,    1, UNI_GC_SM },
	{ 0x1D770,   25, UNI_GC_LL },
	{ 0x1D789,    1, UNI_GC_SM },
	{ 0x1D78A,    6, UNI_GC_LL },
	{ 0x1D790,   25, UNI_GC_LU },
	{ 0x1D7A9,    1, UNI_GC_SM },
	{ 0x1D7AA,   25, UNI_GC_LL },
	{ 0x1D7C3,    1, UNI_GC_SM },
	{ 0x1D7C4,    6, UNI_GC_LL },
	{ 0x1D7CE,   50, UNI_GC_ND },
	{ 0x20000,    1, UNI_GC_LO },
	{ 0x2A6D6,    1, UNI_GC_LO },
	{ 0x2F800,  542, UNI_GC_LO },
	{ 0xE0001,    1, UNI_GC_CF },
	{ 0xE0020,   96, UNI_GC_CF },
	{ 0xE0100,  240, UNI_GC_MN },
	{ 0xF0000,    1, UNI_GC_CO },
	{ 0xFFFFD,    1, UNI_GC_CO },
	{ 0x100000,   1, UNI_GC_CO },
	{ 0x10FFFD,   1, UNI_GC_CO },
};

/*
 * The table is extracted from Blocks.txt using the following AWK script:
 *

	awk '/^[0-9A-F]/{
		sub(";.*$", "", $1);
		start = $1;
		sub("[.][.].*$", "", start);
		end = $1;
		sub("^[^.]*[.][.]", "", end);
		printf("\t{ 0x%s, 0x%s },\n", start, end)
	}' Blocks.txt
 *
 */
static const struct {
	guint32 start;
	guint32 end;
} utf32_block_id_lut[] = {
	{ 0x0000, 0x007F },
	{ 0x0080, 0x00FF },
	{ 0x0100, 0x017F },
	{ 0x0180, 0x024F },
	{ 0x0250, 0x02AF },
	{ 0x02B0, 0x02FF },
	{ 0x0300, 0x036F },
	{ 0x0370, 0x03FF },
	{ 0x0400, 0x04FF },
	{ 0x0500, 0x052F },
	{ 0x0530, 0x058F },
	{ 0x0590, 0x05FF },
	{ 0x0600, 0x06FF },
	{ 0x0700, 0x074F },
	{ 0x0750, 0x077F },
	{ 0x0780, 0x07BF },
	{ 0x0900, 0x097F },
	{ 0x0980, 0x09FF },
	{ 0x0A00, 0x0A7F },
	{ 0x0A80, 0x0AFF },
	{ 0x0B00, 0x0B7F },
	{ 0x0B80, 0x0BFF },
	{ 0x0C00, 0x0C7F },
	{ 0x0C80, 0x0CFF },
	{ 0x0D00, 0x0D7F },
	{ 0x0D80, 0x0DFF },
	{ 0x0E00, 0x0E7F },
	{ 0x0E80, 0x0EFF },
	{ 0x0F00, 0x0FFF },
	{ 0x1000, 0x109F },
	{ 0x10A0, 0x10FF },
	{ 0x1100, 0x11FF },
	{ 0x1200, 0x137F },
	{ 0x1380, 0x139F },
	{ 0x13A0, 0x13FF },
	{ 0x1400, 0x167F },
	{ 0x1680, 0x169F },
	{ 0x16A0, 0x16FF },
	{ 0x1700, 0x171F },
	{ 0x1720, 0x173F },
	{ 0x1740, 0x175F },
	{ 0x1760, 0x177F },
	{ 0x1780, 0x17FF },
	{ 0x1800, 0x18AF },
	{ 0x1900, 0x194F },
	{ 0x1950, 0x197F },
	{ 0x1980, 0x19DF },
	{ 0x19E0, 0x19FF },
	{ 0x1A00, 0x1A1F },
	{ 0x1D00, 0x1D7F },
	{ 0x1D80, 0x1DBF },
	{ 0x1DC0, 0x1DFF },
	{ 0x1E00, 0x1EFF },
	{ 0x1F00, 0x1FFF },
	{ 0x2000, 0x206F },
	{ 0x2070, 0x209F },
	{ 0x20A0, 0x20CF },
	{ 0x20D0, 0x20FF },
	{ 0x2100, 0x214F },
	{ 0x2150, 0x218F },
	{ 0x2190, 0x21FF },
	{ 0x2200, 0x22FF },
	{ 0x2300, 0x23FF },
	{ 0x2400, 0x243F },
	{ 0x2440, 0x245F },
	{ 0x2460, 0x24FF },
	{ 0x2500, 0x257F },
	{ 0x2580, 0x259F },
	{ 0x25A0, 0x25FF },
	{ 0x2600, 0x26FF },
	{ 0x2700, 0x27BF },
	{ 0x27C0, 0x27EF },
	{ 0x27F0, 0x27FF },
	{ 0x2800, 0x28FF },
	{ 0x2900, 0x297F },
	{ 0x2980, 0x29FF },
	{ 0x2A00, 0x2AFF },
	{ 0x2B00, 0x2BFF },
	{ 0x2C00, 0x2C5F },
	{ 0x2C80, 0x2CFF },
	{ 0x2D00, 0x2D2F },
	{ 0x2D30, 0x2D7F },
	{ 0x2D80, 0x2DDF },
	{ 0x2E00, 0x2E7F },
	{ 0x2E80, 0x2EFF },
	{ 0x2F00, 0x2FDF },
	{ 0x2FF0, 0x2FFF },
	{ 0x3000, 0x303F },
	{ 0x3040, 0x309F },
	{ 0x30A0, 0x30FF },
	{ 0x3100, 0x312F },
	{ 0x3130, 0x318F },
	{ 0x3190, 0x319F },
	{ 0x31A0, 0x31BF },
	{ 0x31C0, 0x31EF },
	{ 0x31F0, 0x31FF },
	{ 0x3200, 0x32FF },
	{ 0x3300, 0x33FF },
	{ 0x3400, 0x4DBF },
	{ 0x4DC0, 0x4DFF },
	{ 0x4E00, 0x9FFF },
	{ 0xA000, 0xA48F },
	{ 0xA490, 0xA4CF },
	{ 0xA700, 0xA71F },
	{ 0xA800, 0xA82F },
	{ 0xAC00, 0xD7AF },
	{ 0xD800, 0xDB7F },
	{ 0xDB80, 0xDBFF },
	{ 0xDC00, 0xDFFF },
	{ 0xE000, 0xF8FF },
	{ 0xF900, 0xFAFF },
	{ 0xFB00, 0xFB4F },
	{ 0xFB50, 0xFDFF },
	{ 0xFE00, 0xFE0F },
	{ 0xFE10, 0xFE1F },
	{ 0xFE20, 0xFE2F },
	{ 0xFE30, 0xFE4F },
	{ 0xFE50, 0xFE6F },
	{ 0xFE70, 0xFEFF },
	{ 0xFF00, 0xFFEF },
	{ 0xFFF0, 0xFFFF },
	{ 0x10000, 0x1007F },
	{ 0x10080, 0x100FF },
	{ 0x10100, 0x1013F },
	{ 0x10140, 0x1018F },
	{ 0x10300, 0x1032F },
	{ 0x10330, 0x1034F },
	{ 0x10380, 0x1039F },
	{ 0x103A0, 0x103DF },
	{ 0x10400, 0x1044F },
	{ 0x10450, 0x1047F },
	{ 0x10480, 0x104AF },
	{ 0x10800, 0x1083F },
	{ 0x10A00, 0x10A5F },
	{ 0x1D000, 0x1D0FF },
	{ 0x1D100, 0x1D1FF },
	{ 0x1D200, 0x1D24F },
	{ 0x1D300, 0x1D35F },
	{ 0x1D400, 0x1D7FF },
	{ 0x20000, 0x2A6DF },
	{ 0x2F800, 0x2FA1F },
	{ 0xE0000, 0xE007F },
	{ 0xE0100, 0xE01EF },
	{ 0xF0000, 0xFFFFF },
	{ 0x100000, 0x10FFFF },
};

/**
 * This table is huge! It can be generated with the following AWK script:

	awk 'BEGIN { FS=";" }
		/^[0-9A-Z]/ {
			sub("#.*", "");
			printf("{ { ");
			for (i = 1; i < 6; i++) {
				gsub(" ", ", 0x", $i);
				sub("$", ",", $i);
				printf("{ 0x%s }%s", $i, i < 5 ? ", " : " } },\n" );
			}
		}
	' NormalizationTest.txt 
*/

#if defined(TEST_NORMALIZATION_TEST_TXT)
static const struct {
	guint32 c[5][20];
} normalization_test_txt[] = {
#include "NormalizationTest.inc"
};
#endif /* TEST_NORMALIZATION_TEST_TXT */

/*
 * This is extracted from Part1 of NormalizationTest.txt

	awk '
		/^@Part/ {
			part1 = $1 == "@Part1";
		}
		/^[A-Z0-9]/ {
			if (!part1)
				next;
			gsub(" ", "\n", $1);
			print $1 }
	' NormalizationTest.txt | \
	gawk '{ 
		uc = strtonum("0x" $1);
		printf("0x%06X\n", uc);
	}' | sort | uniq | \
	gawk '{
		uc = strtonum($1);
		if (last + 1 != uc) {
			if (start != 0)
			printf("{ 0x%04X, %d },\n", start, n + 1);
			start = uc;
			n = 0;
		} else {
			n++;
		}
		last = uc;
		
	}
	END {
		if (n)
			printf("{ 0x%04X, %d },\n", start, n + 1);
	}'

 */
static const struct {
	guint32 uc;		/**< The first unicode character in the array */
	guint16 len;	/**< The array length */
} utf32_normalization_specials[] = {
	{ 0x00A0, 1 },
	{ 0x00A8, 1 },
	{ 0x00AA, 1 },
	{ 0x00AF, 1 },
	{ 0x00B2, 4 },
	{ 0x00B8, 3 },
	{ 0x00BC, 3 },
	{ 0x00C0, 6 },
	{ 0x00C7, 9 },
	{ 0x00D1, 6 },
	{ 0x00D9, 5 },
	{ 0x00E0, 6 },
	{ 0x00E7, 9 },
	{ 0x00F1, 6 },
	{ 0x00F9, 5 },
	{ 0x00FF, 17 },
	{ 0x0112, 20 },
	{ 0x0128, 9 },
	{ 0x0132, 6 },
	{ 0x0139, 8 },
	{ 0x0143, 7 },
	{ 0x014C, 6 },
	{ 0x0154, 18 },
	{ 0x0168, 24 },
	{ 0x01A0, 2 },
	{ 0x01AF, 2 },
	{ 0x01C4, 25 },
	{ 0x01DE, 6 },
	{ 0x01E6, 16 },
	{ 0x01F8, 36 },
	{ 0x021E, 2 },
	{ 0x0226, 14 },
	{ 0x02B0, 9 },
	{ 0x02D8, 6 },
	{ 0x02E0, 5 },
	{ 0x0340, 2 },
	{ 0x0343, 2 },
	{ 0x0374, 1 },
	{ 0x037A, 1 },
	{ 0x037E, 1 },
	{ 0x0384, 7 },
	{ 0x038C, 1 },
	{ 0x038E, 3 },
	{ 0x03AA, 7 },
	{ 0x03CA, 5 },
	{ 0x03D0, 7 },
	{ 0x03F0, 3 },
	{ 0x03F4, 2 },
	{ 0x03F9, 1 },
	{ 0x0400, 2 },
	{ 0x0403, 1 },
	{ 0x0407, 1 },
	{ 0x040C, 3 },
	{ 0x0419, 1 },
	{ 0x0439, 1 },
	{ 0x0450, 2 },
	{ 0x0453, 1 },
	{ 0x0457, 1 },
	{ 0x045C, 3 },
	{ 0x0476, 2 },
	{ 0x04C1, 2 },
	{ 0x04D0, 4 },
	{ 0x04D6, 2 },
	{ 0x04DA, 6 },
	{ 0x04E2, 6 },
	{ 0x04EA, 12 },
	{ 0x04F8, 2 },
	{ 0x0587, 1 },
	{ 0x0622, 5 },
	{ 0x0675, 4 },
	{ 0x06C0, 1 },
	{ 0x06C2, 1 },
	{ 0x06D3, 1 },
	{ 0x0929, 1 },
	{ 0x0931, 1 },
	{ 0x0934, 1 },
	{ 0x0958, 8 },
	{ 0x09CB, 2 },
	{ 0x09DC, 2 },
	{ 0x09DF, 1 },
	{ 0x0A33, 1 },
	{ 0x0A36, 1 },
	{ 0x0A59, 3 },
	{ 0x0A5E, 1 },
	{ 0x0B48, 1 },
	{ 0x0B4B, 2 },
	{ 0x0B5C, 2 },
	{ 0x0B94, 1 },
	{ 0x0BCA, 3 },
	{ 0x0C48, 1 },
	{ 0x0CC0, 1 },
	{ 0x0CC7, 2 },
	{ 0x0CCA, 2 },
	{ 0x0D4A, 3 },
	{ 0x0DDA, 1 },
	{ 0x0DDC, 3 },
	{ 0x0E33, 1 },
	{ 0x0EB3, 1 },
	{ 0x0EDC, 2 },
	{ 0x0F0C, 1 },
	{ 0x0F43, 1 },
	{ 0x0F4D, 1 },
	{ 0x0F52, 1 },
	{ 0x0F57, 1 },
	{ 0x0F5C, 1 },
	{ 0x0F69, 1 },
	{ 0x0F73, 1 },
	{ 0x0F75, 5 },
	{ 0x0F81, 1 },
	{ 0x0F93, 1 },
	{ 0x0F9D, 1 },
	{ 0x0FA2, 1 },
	{ 0x0FA7, 1 },
	{ 0x0FAC, 1 },
	{ 0x0FB9, 1 },
	{ 0x1026, 1 },
	{ 0x10FC, 1 },
	{ 0x1D2C, 3 },
	{ 0x1D30, 11 },
	{ 0x1D3C, 18 },
	{ 0x1D4F, 28 },
	{ 0x1D78, 1 },
	{ 0x1D9B, 37 },
	{ 0x1E00, 156 },
	{ 0x1EA0, 90 },
	{ 0x1F00, 22 },
	{ 0x1F18, 6 },
	{ 0x1F20, 38 },
	{ 0x1F48, 6 },
	{ 0x1F50, 8 },
	{ 0x1F59, 1 },
	{ 0x1F5B, 1 },
	{ 0x1F5D, 1 },
	{ 0x1F5F, 31 },
	{ 0x1F80, 53 },
	{ 0x1FB6, 15 },
	{ 0x1FC6, 14 },
	{ 0x1FD6, 6 },
	{ 0x1FDD, 19 },
	{ 0x1FF2, 3 },
	{ 0x1FF6, 9 },
	{ 0x2000, 11 },
	{ 0x2011, 1 },
	{ 0x2017, 1 },
	{ 0x2024, 3 },
	{ 0x202F, 1 },
	{ 0x2033, 2 },
	{ 0x2036, 2 },
	{ 0x203C, 1 },
	{ 0x203E, 1 },
	{ 0x2047, 3 },
	{ 0x2057, 1 },
	{ 0x205F, 1 },
	{ 0x2070, 2 },
	{ 0x2074, 27 },
	{ 0x2090, 5 },
	{ 0x20A8, 1 },
	{ 0x2100, 4 },
	{ 0x2105, 3 },
	{ 0x2109, 11 },
	{ 0x2115, 2 },
	{ 0x2119, 5 },
	{ 0x2120, 3 },
	{ 0x2124, 1 },
	{ 0x2126, 1 },
	{ 0x2128, 1 },
	{ 0x212A, 4 },
	{ 0x212F, 3 },
	{ 0x2133, 7 },
	{ 0x213B, 6 },
	{ 0x2145, 5 },
	{ 0x2153, 45 },
	{ 0x219A, 2 },
	{ 0x21AE, 1 },
	{ 0x21CD, 3 },
	{ 0x2204, 1 },
	{ 0x2209, 1 },
	{ 0x220C, 1 },
	{ 0x2224, 1 },
	{ 0x2226, 1 },
	{ 0x222C, 2 },
	{ 0x222F, 2 },
	{ 0x2241, 1 },
	{ 0x2244, 1 },
	{ 0x2247, 1 },
	{ 0x2249, 1 },
	{ 0x2260, 1 },
	{ 0x2262, 1 },
	{ 0x226D, 5 },
	{ 0x2274, 2 },
	{ 0x2278, 2 },
	{ 0x2280, 2 },
	{ 0x2284, 2 },
	{ 0x2288, 2 },
	{ 0x22AC, 4 },
	{ 0x22E0, 4 },
	{ 0x22EA, 4 },
	{ 0x2329, 2 },
	{ 0x2460, 139 },
	{ 0x2A0C, 1 },
	{ 0x2A74, 3 },
	{ 0x2ADC, 1 },
	{ 0x2D6F, 1 },
	{ 0x2E9F, 1 },
	{ 0x2EF3, 1 },
	{ 0x2F00, 214 },
	{ 0x3000, 1 },
	{ 0x3036, 1 },
	{ 0x3038, 3 },
	{ 0x304C, 1 },
	{ 0x304E, 1 },
	{ 0x3050, 1 },
	{ 0x3052, 1 },
	{ 0x3054, 1 },
	{ 0x3056, 1 },
	{ 0x3058, 1 },
	{ 0x305A, 1 },
	{ 0x305C, 1 },
	{ 0x305E, 1 },
	{ 0x3060, 1 },
	{ 0x3062, 1 },
	{ 0x3065, 1 },
	{ 0x3067, 1 },
	{ 0x3069, 1 },
	{ 0x3070, 2 },
	{ 0x3073, 2 },
	{ 0x3076, 2 },
	{ 0x3079, 2 },
	{ 0x307C, 2 },
	{ 0x3094, 1 },
	{ 0x309B, 2 },
	{ 0x309E, 2 },
	{ 0x30AC, 1 },
	{ 0x30AE, 1 },
	{ 0x30B0, 1 },
	{ 0x30B2, 1 },
	{ 0x30B4, 1 },
	{ 0x30B6, 1 },
	{ 0x30B8, 1 },
	{ 0x30BA, 1 },
	{ 0x30BC, 1 },
	{ 0x30BE, 1 },
	{ 0x30C0, 1 },
	{ 0x30C2, 1 },
	{ 0x30C5, 1 },
	{ 0x30C7, 1 },
	{ 0x30C9, 1 },
	{ 0x30D0, 2 },
	{ 0x30D3, 2 },
	{ 0x30D6, 2 },
	{ 0x30D9, 2 },
	{ 0x30DC, 2 },
	{ 0x30F4, 1 },
	{ 0x30F7, 4 },
	{ 0x30FE, 2 },
	{ 0x3131, 94 },
	{ 0x3192, 14 },
	{ 0x3200, 31 },
	{ 0x3220, 36 },
	{ 0x3250, 47 },
	{ 0x3280, 127 },
	{ 0x3300, 256 },
	{ 0xAC00, 11172 },
	{ 0xF900, 270 },
	{ 0xFA10, 1 },
	{ 0xFA12, 1 },
	{ 0xFA15, 10 },
	{ 0xFA20, 1 },
	{ 0xFA22, 1 },
	{ 0xFA25, 2 },
	{ 0xFA2A, 4 },
	{ 0xFA30, 59 },
	{ 0xFA70, 106 },
	{ 0xFB00, 7 },
	{ 0xFB13, 5 },
	{ 0xFB1D, 1 },
	{ 0xFB1F, 24 },
	{ 0xFB38, 5 },
	{ 0xFB3E, 1 },
	{ 0xFB40, 2 },
	{ 0xFB43, 2 },
	{ 0xFB46, 108 },
	{ 0xFBD3, 363 },
	{ 0xFD50, 64 },
	{ 0xFD92, 54 },
	{ 0xFDF0, 13 },
	{ 0xFE10, 10 },
	{ 0xFE30, 21 },
	{ 0xFE47, 12 },
	{ 0xFE54, 19 },
	{ 0xFE68, 4 },
	{ 0xFE70, 3 },
	{ 0xFE74, 1 },
	{ 0xFE76, 135 },
	{ 0xFF01, 190 },
	{ 0xFFC2, 6 },
	{ 0xFFCA, 6 },
	{ 0xFFD2, 6 },
	{ 0xFFDA, 3 },
	{ 0xFFE0, 7 },
	{ 0xFFE8, 7 },
	{ 0x1D15E, 7 },
	{ 0x1D1BB, 6 },
	{ 0x1D400, 85 },
	{ 0x1D456, 71 },
	{ 0x1D49E, 2 },
	{ 0x1D4A2, 1 },
	{ 0x1D4A5, 2 },
	{ 0x1D4A9, 4 },
	{ 0x1D4AE, 12 },
	{ 0x1D4BB, 1 },
	{ 0x1D4BD, 7 },
	{ 0x1D4C5, 65 },
	{ 0x1D507, 4 },
	{ 0x1D50D, 8 },
	{ 0x1D516, 7 },
	{ 0x1D51E, 28 },
	{ 0x1D53B, 4 },
	{ 0x1D540, 5 },
	{ 0x1D546, 1 },
	{ 0x1D54A, 7 },
	{ 0x1D552, 340 },
	{ 0x1D6A8, 290 },
	{ 0x1D7CE, 50 },
	{ 0x2F800, 542 },
};

#endif /* _utf8_tables_h_ */
/* vi: set ts=4 sw=4 cindent: */
