/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  GThumb
 *
 *  Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <libgnomeui/gnome-propertybox.h>
#include <libgnomeui/gnome-pixmap.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include <gtk/gtk.h>
#include <libgnome/libgnome.h>
#include <libgnomeui/gnome-dialog.h>
#include <libgnomeui/gnome-dialog-util.h>
#include <glade/glade.h>
#include "thumb-cache.h"
#include "comments.h"
#include "file-utils.h"
#include "main.h"
#include "typedefs.h"
#include "window.h"

#include "icons/layout1.xpm"
#include "icons/layout2.xpm"
#include "icons/layout3.xpm"
#include "icons/layout4.xpm"

static gint thumb_size[] = {48, 64, 85, 95, 112, 128, 164, 200, 256};
static gint thumb_sizes = sizeof (thumb_size) / sizeof (gint);

typedef struct {
	GThumbWindow    *window;

	GladeXML  *gui;
	GtkWidget *dialog;

	GtkWidget *radio_current_location;
	GtkWidget *radio_last_location;
	GtkWidget *radio_use_startup;
	GtkWidget *startup_dir_entry;
	GtkWidget *file_entry;
	GtkWidget *toggle_confirm_file_del;
	GtkWidget *toggle_confirm_catalog_del;
	GtkWidget *btn_set_to_current;
	GtkWidget *radio_layout1;
	GtkWidget *radio_layout2;
	GtkWidget *radio_layout3;
	GtkWidget *radio_layout4;
	GtkWidget *toggle_location_on_main_bar;

	GtkWidget *radio_from_extension;
	GtkWidget *radio_from_content;
	GtkWidget *toggle_show_hidden;
	GtkWidget *toggle_show_comments;
	GtkWidget *btn_delete_comments;
	GtkWidget *toggle_show_thumbs;
	GtkWidget *toggle_save_thumbs;
	GtkWidget *btn_delete_thumbs;
	GtkWidget *opt_thumbs_quality;
	GtkWidget *opt_thumbs_size;
	GtkWidget *spin_history_length;

	GtkWidget *opt_zoom_quality;
	GtkWidget *opt_zoom_change;
	GtkWidget *opt_transparency;
	GtkWidget *opt_check_type;
	GtkWidget *opt_check_size;

	GtkWidget *opt_ss_direction;
	GtkWidget *spin_ss_delay;
	GtkWidget *toggle_ss_wrap_around;
	GtkWidget *toggle_ss_fullscreen;
} DialogData;


static gint     original_layout_type;
static guint    original_zoom_quality;
static guint    original_zoom_change;
TranspType      original_transp_type;
CheckType       original_check_type;
CheckSize       original_check_size;
static gboolean original_show_dot_files;
static gboolean original_show_comments;
static gboolean original_show_thumbs;
static guint    original_thumbs_size;


/* called when the main dialog is closed. */
static void
destroy_cb (GtkWidget *widget, 
	    DialogData *data)
{
        gtk_object_unref (GTK_OBJECT (data->gui));
	g_free (data);
}


static guint
opt_menu_get_active_idx (GtkWidget *opt_menu)
{
	GtkWidget *item;
	guint idx;
	GList *scan;
	GtkWidget *menu;

	menu = gtk_option_menu_get_menu (GTK_OPTION_MENU (opt_menu));
	item = gtk_menu_get_active (GTK_MENU (menu));

	idx = 0;
	scan = GTK_MENU_SHELL (menu)->children;
	while (scan && (item != scan->data)) {
		idx++;
		scan = scan->next;
	}

	return idx;
}


static void
save_original_values (GThumbWindow *window)
{
	/* Save original values. */
	original_layout_type = preferences.layout_type;
	original_zoom_quality = preferences.zoom_quality;
	original_zoom_change = preferences.zoom_change;
	original_transp_type = image_viewer_get_transp_type (IMAGE_VIEWER (window->viewer));
	original_check_type = preferences.check_type;
	original_check_size = preferences.check_size;
	original_show_dot_files	= preferences.show_dot_files;
	original_show_comments = preferences.show_comments;
	original_show_thumbs = window->file_list->enable_thumbs;
	original_thumbs_size = window->file_list->thumb_size;
}


/* called when the "apply" button is clicked. */
static void
apply_cb (GtkWidget *widget, 
	  DialogData *data)
{
	/* Startup dir. */
	preferences.go_to_last_location = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->radio_last_location));
	preferences.use_startup_location = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->radio_use_startup));
	if (preferences.use_startup_location) {
		gchar *temp;
		if (preferences.startup_location)
			g_free (preferences.startup_location);
		
		temp = remove_ending_separator (gtk_entry_get_text (GTK_ENTRY (data->startup_dir_entry)));
		preferences.startup_location = g_strconcat ("file://",
							    temp,
							    NULL);
		g_free (temp);
	}

	/* Confirmations. */
	preferences.confirm_file_del = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->toggle_confirm_file_del));
	preferences.confirm_catalog_del = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->toggle_confirm_catalog_del));

	/* Filter options. */
	preferences.show_dot_files = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->toggle_show_hidden));
	preferences.show_comments = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->toggle_show_comments));
	preferences.fast_file_type = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->radio_from_extension));

	/* Thumbnails options. */
	preferences.enable_thumbnails = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->toggle_show_thumbs));
	preferences.enable_thumb_caching = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->toggle_save_thumbs));
	preferences.thumb_quality = opt_menu_get_active_idx (data->opt_thumbs_quality);
	preferences.thumb_size = thumb_size[opt_menu_get_active_idx (data->opt_thumbs_size)];

	/* History options. */
	preferences.history_max_length = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (data->spin_history_length));

	/* Viewer options. */
	preferences.zoom_quality = opt_menu_get_active_idx (data->opt_zoom_quality);
	preferences.zoom_change = opt_menu_get_active_idx (data->opt_zoom_change);
	preferences.transp_type = opt_menu_get_active_idx (data->opt_transparency);
	image_viewer_set_transp_type (IMAGE_VIEWER (data->window->viewer),
				      preferences.transp_type);
	preferences.check_type = opt_menu_get_active_idx (data->opt_check_type);
	switch (opt_menu_get_active_idx (data->opt_check_size)) {
	case 0: preferences.check_size = CHECK_SIZE_SMALL; break;
	case 1: preferences.check_size = CHECK_SIZE_MEDIUM; break;
	case 2: preferences.check_size = CHECK_SIZE_LARGE; break;
	}

	/* Slide Show. */
	preferences.ss_wrap_around = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->toggle_ss_wrap_around));
	preferences.ss_fullscreen = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->toggle_ss_fullscreen));
	preferences.ss_delay = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (data->spin_ss_delay));
	preferences.ss_direction = opt_menu_get_active_idx (data->opt_ss_direction);

	if ((original_zoom_quality != preferences.zoom_quality)
	    || (original_zoom_change != preferences.zoom_change)
	    || (original_transp_type != preferences.transp_type)
	    || (original_check_type != preferences.check_type)
	    || (original_check_size != preferences.check_size))
		all_windows_update_viewer_options ();

	if ((original_show_dot_files != preferences.show_dot_files)
	    || (original_show_comments != preferences.show_comments)
	    || (original_show_thumbs != preferences.enable_thumbnails)
	    || (original_thumbs_size != preferences.thumb_size)) 
		all_windows_update_browser_options ();

	/* Layout. */
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->radio_layout1)))
		preferences.layout_type = 0;
	else if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->radio_layout2)))
		preferences.layout_type = 1;
	else if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->radio_layout3)))
		preferences.layout_type = 2;
	else
		preferences.layout_type = 3;

	preferences.location_on_main_bar = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->toggle_location_on_main_bar));

	if (original_layout_type != preferences.layout_type)
		window_notify_update_layout (data->window);

	save_original_values (data->window);
	gnome_property_box_changed (GNOME_PROPERTY_BOX (data->dialog));
}


/* called when the "ok" button is clicked. */
static void
ok_cb (GtkWidget *widget, 
       DialogData *data)
{
	apply_cb (widget, data);
	gtk_widget_destroy (data->dialog);
}


/* called when the "use startup" is clicked. */
static void
use_startup_toggled_cb (GtkWidget *widget, 
			DialogData *data)
{
	gtk_widget_set_sensitive (data->file_entry,
				  GTK_TOGGLE_BUTTON (widget)->active);
	gtk_widget_set_sensitive (data->btn_set_to_current,
				  GTK_TOGGLE_BUTTON (widget)->active);
}


/* called when the "delete comments" button is called. */
static void
del_comments_cb (GtkWidget *widget, 
		 gpointer data)
{
	comments_remove_old_comments_async (NULL, TRUE, TRUE);
}


/* called when the "delete thumbnails" button is called. */
static void
del_thumbs_cb (GtkWidget *widget, 
	       gpointer data)
{
	cache_remove_old_previews_async (NULL, TRUE, TRUE);
}


/* called when the "set to current" button is clicked. */
static void
set_to_current_cb (GtkWidget *widget, 
		   DialogData *data)
{
	if (data->window->dir_list->path != NULL)
		gtk_entry_set_text (GTK_ENTRY (data->startup_dir_entry),
				    data->window->dir_list->path);
}


/* get the option menu index from the size value. */
static gint
get_idx_from_size (gint size)
{
	int i;

	for (i = 0; i < thumb_sizes; i++) 
		if (size == thumb_size[i])
			return i;
	return -1;
}


/* create the main dialog. */
void
dlg_preferences (GThumbWindow *window)
{
	DialogData *data;
	GtkWidget *btn_ok;
	GtkWidget *btn_cancel;
	GtkWidget *btn_apply;
	gint i = 0;

	data = g_new (DialogData, 1);

	data->window = window;

	data->gui = glade_xml_new (GTHUMB_GLADEDIR "/" GLADE_FILE , NULL);
        if (!data->gui) {
                g_warning ("Could not find " GLADE_FILE "\n");
		g_free (data);
                return;
        }

	save_original_values (data->window);

	/* Get the widgets. */

	data->dialog = glade_xml_get_widget (data->gui, "preferences_dialog");

        data->radio_current_location = glade_xml_get_widget (data->gui, "radio_current_location");
        data->radio_last_location = glade_xml_get_widget (data->gui, "radio_last_location");
        data->radio_use_startup = glade_xml_get_widget (data->gui, "radio_use_startup");
        data->startup_dir_entry = glade_xml_get_widget (data->gui, "startup_dir_entry");
        data->file_entry = glade_xml_get_widget (data->gui, "file_entry");
	data->btn_set_to_current = glade_xml_get_widget (data->gui, "btn_set_to_current");
        data->toggle_confirm_file_del = glade_xml_get_widget (data->gui, "toggle_confirm_file_del");
        data->toggle_confirm_catalog_del = glade_xml_get_widget (data->gui, "toggle_confirm_catalog_del");
	data->radio_layout1 = glade_xml_get_widget (data->gui, "radio_layout1");
	data->radio_layout2 = glade_xml_get_widget (data->gui, "radio_layout2");
	data->radio_layout3 = glade_xml_get_widget (data->gui, "radio_layout3");
	data->radio_layout4 = glade_xml_get_widget (data->gui, "radio_layout4");
	data->toggle_location_on_main_bar = glade_xml_get_widget (data->gui, "toggle_location_on_main_bar");


	data->radio_from_extension = glade_xml_get_widget (data->gui, "radio_from_extension");
	data->radio_from_content = glade_xml_get_widget (data->gui, "radio_from_content");
        data->toggle_show_hidden = glade_xml_get_widget (data->gui, "toggle_show_hidden");
        data->toggle_show_comments = glade_xml_get_widget (data->gui, "toggle_show_comments");
        data->btn_delete_comments = glade_xml_get_widget (data->gui, "btn_delete_comments");
        data->toggle_show_thumbs = glade_xml_get_widget (data->gui, "toggle_show_thumbs");
        data->toggle_save_thumbs = glade_xml_get_widget (data->gui, "toggle_save_thumbs");
        data->btn_delete_thumbs = glade_xml_get_widget (data->gui, "btn_delete_thumbs");
        data->opt_thumbs_quality = glade_xml_get_widget (data->gui, "opt_thumbs_quality");
        data->opt_thumbs_size = glade_xml_get_widget (data->gui, "opt_thumbs_size");
        data->spin_history_length = glade_xml_get_widget (data->gui, "spin_history_length");

        data->opt_zoom_quality = glade_xml_get_widget (data->gui, "opt_zoom_quality");
        data->opt_zoom_change = glade_xml_get_widget (data->gui, "opt_zoom_change");
	data->opt_transparency = glade_xml_get_widget (data->gui, "opt_transparency");
	data->opt_check_type = glade_xml_get_widget (data->gui, "opt_check_type");
	data->opt_check_size = glade_xml_get_widget (data->gui, "opt_check_size");

        data->opt_ss_direction = glade_xml_get_widget (data->gui, "opt_ss_direction");
        data->spin_ss_delay = glade_xml_get_widget (data->gui, "spin_ss_delay");
        data->toggle_ss_wrap_around = glade_xml_get_widget (data->gui, "toggle_ss_wrap_around");
        data->toggle_ss_fullscreen = glade_xml_get_widget (data->gui, "toggle_ss_fullscreen");

        btn_ok = GNOME_PROPERTY_BOX (data->dialog)->ok_button;
        btn_cancel = GNOME_PROPERTY_BOX (data->dialog)->cancel_button;
        btn_apply = GNOME_PROPERTY_BOX (data->dialog)->apply_button;

	/* Set widgets data. */
	
	/* * general */
	if (preferences.use_startup_location)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->radio_use_startup), TRUE);
	else if (preferences.go_to_last_location)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->radio_last_location), TRUE);
	else 
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->radio_current_location), TRUE);

	if (!preferences.use_startup_location) {
		gtk_widget_set_sensitive (data->file_entry, FALSE);
		gtk_widget_set_sensitive (data->btn_set_to_current, FALSE);
	}
	if (preferences.startup_location
	    && pref_util_location_is_file (preferences.startup_location))
		gtk_entry_set_text (GTK_ENTRY (data->startup_dir_entry),
				    pref_util_get_file_location (preferences.startup_location));

	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->toggle_confirm_file_del), preferences.confirm_file_del);

	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->toggle_confirm_catalog_del), preferences.confirm_catalog_del);
	gtk_container_add (GTK_CONTAINER (data->radio_layout1),
			   gnome_pixmap_new_from_xpm_d (layout1_xpm));
	gtk_container_add (GTK_CONTAINER (data->radio_layout2),
			   gnome_pixmap_new_from_xpm_d (layout2_xpm));
	gtk_container_add (GTK_CONTAINER (data->radio_layout3),
			   gnome_pixmap_new_from_xpm_d (layout3_xpm));
	gtk_container_add (GTK_CONTAINER (data->radio_layout4),
			   gnome_pixmap_new_from_xpm_d (layout4_xpm));

	/* ** layout */
	if (preferences.layout_type == 0)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->radio_layout1), TRUE);
	else if (preferences.layout_type == 1)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->radio_layout2), TRUE);
	else if (preferences.layout_type == 2)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->radio_layout3), TRUE);
	else
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->radio_layout4), TRUE);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->toggle_location_on_main_bar), preferences.location_on_main_bar);

	/* * browser */
	if (preferences.fast_file_type)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->radio_from_extension), TRUE);
	else
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->radio_from_content), TRUE);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->toggle_show_hidden), preferences.show_dot_files);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->toggle_show_comments), preferences.show_comments);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->toggle_show_thumbs), window->file_list->enable_thumbs);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->toggle_save_thumbs), preferences.enable_thumb_caching);
	gtk_option_menu_set_history (GTK_OPTION_MENU (data->opt_thumbs_quality), preferences.thumb_quality);
	gtk_option_menu_set_history (GTK_OPTION_MENU (data->opt_thumbs_size), get_idx_from_size (preferences.thumb_size));
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (data->spin_history_length),
				   (gfloat) preferences.history_max_length);

	/* * viewer */
	gtk_option_menu_set_history (GTK_OPTION_MENU (data->opt_zoom_quality), preferences.zoom_quality);
	gtk_option_menu_set_history (GTK_OPTION_MENU (data->opt_zoom_change), preferences.zoom_change);
	gtk_option_menu_set_history (GTK_OPTION_MENU (data->opt_transparency), image_viewer_get_transp_type (IMAGE_VIEWER (window->viewer)));
	gtk_option_menu_set_history (GTK_OPTION_MENU (data->opt_check_type), preferences.check_type);
	switch (preferences.check_size) {
	case CHECK_SIZE_SMALL:   i = 0; break;
	case CHECK_SIZE_MEDIUM:  i = 1; break;
	case CHECK_SIZE_LARGE:   i = 2; break;
	}
	gtk_option_menu_set_history (GTK_OPTION_MENU (data->opt_check_size), i);

	/* * slide show */
	gtk_option_menu_set_history (GTK_OPTION_MENU (data->opt_ss_direction), preferences.ss_direction);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (data->spin_ss_delay),
				   (gfloat) preferences.ss_delay);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->toggle_ss_wrap_around), preferences.ss_wrap_around);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->toggle_ss_fullscreen), preferences.ss_fullscreen);

	/* Set the signals handlers. */

	gtk_signal_connect (GTK_OBJECT (data->dialog), "destroy",
			    (GtkSignalFunc) destroy_cb,
			    data);

	gtk_signal_connect (GTK_OBJECT (btn_ok), "clicked",
                            GTK_SIGNAL_FUNC (ok_cb),
                            data);
	gtk_signal_connect_after (GTK_OBJECT (btn_apply), "clicked",
				  GTK_SIGNAL_FUNC (apply_cb),
				  data);
	gtk_signal_connect_object (GTK_OBJECT (btn_cancel), "clicked",
				   GTK_SIGNAL_FUNC (gtk_widget_destroy),
				   GTK_OBJECT (data->dialog));

	gtk_signal_connect (GTK_OBJECT (data->radio_use_startup), "toggled",
                            GTK_SIGNAL_FUNC (use_startup_toggled_cb),
                            data);
	gtk_signal_connect (GTK_OBJECT (data->btn_delete_comments), "clicked",
			    GTK_SIGNAL_FUNC (del_comments_cb),
			    NULL);
	gtk_signal_connect (GTK_OBJECT (data->btn_delete_thumbs), "clicked",
			    GTK_SIGNAL_FUNC (del_thumbs_cb),
			    NULL);
	gtk_signal_connect (GTK_OBJECT (data->btn_set_to_current), "clicked",
			    GTK_SIGNAL_FUNC (set_to_current_cb),
			    data);


	/* run dialog. */
	gnome_property_box_changed (GNOME_PROPERTY_BOX (data->dialog));
	gtk_window_set_transient_for (GTK_WINDOW (data->dialog), GTK_WINDOW (window->app));
	gtk_window_set_modal (GTK_WINDOW (data->dialog), TRUE);
	gtk_widget_show_all (data->dialog);
}
