/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  GThumb
 *
 *  Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <gtk/gtk.h>
#include <gnome.h>
#include <glade/glade.h>
#include <libgnomevfs/gnome-vfs-mime.h>
#include <libgnomevfs/gnome-vfs-mime-handlers.h>
#include "file-utils.h"
#include "image-list.h"
#include "image-list-utils.h"
#include "typedefs.h"
#include "main.h"
#include "window.h"


typedef struct {
	GThumbWindow    *window;
	GladeXML  *gui;

	GtkWidget *dialog;
	GtkWidget *app_list_clist;
	GtkWidget *recent_clist;
	GtkWidget *app_entry;
	GtkWidget *del_button;

	GList     *app_list;
	GList     *file_list;
} DialogData;


/* called when the main dialog is closed. */
static void
open_with__destroy_cb (GtkWidget *widget, 
		       DialogData *data)
{
        gtk_object_unref (GTK_OBJECT (data->gui));
	if (data->file_list)
		path_list_free (data->file_list);
	if (data->app_list)
		gnome_vfs_mime_application_list_free (data->app_list);
	g_free (data);
}


static void
exec_command (gchar *application, GList *file_list)
{
        GString * command;
        GList *   scan;

        command = g_string_new ("");
        g_string_append (command, application);
        for (scan = file_list; scan; scan = scan->next) {
                gchar *filename = scan->data;
                gchar *e_filename;

                g_string_append_c (command, ' ');
                e_filename = shell_escape (filename);
                g_string_append (command, e_filename);

                g_free (e_filename);
        }

        gnome_execute_shell (g_get_home_dir (), command->str); 
        g_string_free (command, TRUE);
}


static void
open_cb (GtkWidget *widget,
	 gpointer callback_data)
{
	DialogData *data = callback_data;
	gchar *application;
	GList *scan;
	gboolean present = FALSE;

	application = gtk_entry_get_text (GTK_ENTRY (data->app_entry));

	/* add the command to the editors list if not already present. */

	for (scan = data->app_list; scan; scan = scan->next) {
		GnomeVFSMimeApplication *app = scan->data;
		if (strcmp (app->command, application) == 0) {
			present = TRUE;
			break;
		}
	}

	for (scan = preferences.editors; scan; scan = scan->next) {
		gchar *command = scan->data;
		if (strcmp (command, application) == 0) {
			present = TRUE;
			break;
		}
	}

	if (! present) {
		preferences.editors_n++;
		preferences.editors = g_list_prepend (preferences.editors,
						      g_strdup (application));
	}

	/* exec the application. */

	exec_command (application, data->file_list);
	gtk_widget_destroy (data->dialog);
}


static void 
app_selected_cb (GtkWidget *widget, 
		 gint row, 
		 gint col,
		 GdkEvent *event,
		 gpointer callback_data)
{
	DialogData *data = callback_data;
	GnomeVFSMimeApplication *app;
	
	app = gtk_clist_get_row_data (GTK_CLIST (data->app_list_clist), row);
	gtk_entry_set_text (GTK_ENTRY (data->app_entry), app->command);
}


static void 
recent_selected_cb (GtkWidget *widget, 
		    gint row, 
		    gint col,
		    GdkEvent *event,
		    gpointer callback_data)
{
	DialogData *data = callback_data;
	gchar *editor;

	editor = gtk_clist_get_row_data (GTK_CLIST (data->recent_clist), row);
	gtk_entry_set_text (GTK_ENTRY (data->app_entry), editor);
}


static void
delete_recent_cb (GtkWidget *widget,
		  gpointer callback_data)
{
	DialogData *data = callback_data;
	GList *selection;
	gint row;
	gchar *editor;

	selection = GTK_CLIST (data->recent_clist)->selection;
	if (selection == NULL)
		return;

	row = GPOINTER_TO_INT (selection->data);
	editor = gtk_clist_get_row_data (GTK_CLIST (data->recent_clist), row);

	gtk_clist_remove (GTK_CLIST (data->recent_clist), row);

	preferences.editors = g_list_remove (preferences.editors, editor);
	g_free (editor);
}


/* create the "open with" dialog. */
void
open_with_cb (GtkWidget *widget, 
	      void *callback_data)
{
	GThumbWindow *window = callback_data;
	DialogData *data;
	ImageList *ilist;
	GList *scan;
	gchar *col_data[2];
	GnomeVFSMimeApplication *app;
	gint row;
	GList *app_names = NULL;
	GtkWidget *ok_btn, *cancel_btn;

	data = g_new (DialogData, 1);

	ilist = IMAGE_LIST (window->file_list->ilist);
	data->file_list = ilist_utils_get_file_list_selection (ilist);
	if (data->file_list == NULL) {
		/* no selection. */
		
		if ((window->image_path == NULL) 
		    || ! path_is_file (window->image_path)) {
			/* no image in the viewer. */
			g_free (data);
			return;
		}

		data->file_list = g_list_prepend (NULL, window->image_path);
	}
	
	data->window = window;
	data->gui = glade_xml_new (GTHUMB_GLADEDIR "/" GLADE_FILE , NULL);
        if (! data->gui) {
                g_warning ("Could not find " GLADE_FILE "\n");
                return;
        }

	/* Get the widgets. */

	data->dialog = glade_xml_get_widget (data->gui, "open_with_dialog");
	data->app_list_clist = glade_xml_get_widget (data->gui, "app_list_clist");
	data->recent_clist = glade_xml_get_widget (data->gui, "recent_clist");
	ok_btn = glade_xml_get_widget (data->gui, "open_with_ok_btn");
	cancel_btn = glade_xml_get_widget (data->gui, "open_with_cancel_btn");
	data->app_entry = glade_xml_get_widget (data->gui, "open_with_app_entry");
	data->del_button = glade_xml_get_widget (data->gui, "delete_button");

	/* Set the signals handlers. */

	gtk_signal_connect (GTK_OBJECT (data->dialog), 
			    "destroy",
			    (GtkSignalFunc) open_with__destroy_cb,
			    data);
	gtk_signal_connect (GTK_OBJECT (data->app_list_clist), 
			    "select_row",
			    (GtkSignalFunc) app_selected_cb, 
			    data);
	gtk_signal_connect (GTK_OBJECT (data->recent_clist), 
			    "select_row",
			    (GtkSignalFunc) recent_selected_cb, 
			    data);
	gtk_signal_connect_object (GTK_OBJECT (cancel_btn), 
				   "clicked",
				   GTK_SIGNAL_FUNC (gtk_widget_destroy),
				   GTK_OBJECT (data->dialog));
	gtk_signal_connect (GTK_OBJECT (ok_btn), 
			    "clicked",
			    GTK_SIGNAL_FUNC (open_cb),
			    data);
	gtk_signal_connect_object (GTK_OBJECT (cancel_btn), 
				   "clicked",
				   GTK_SIGNAL_FUNC (gtk_widget_destroy),
				   GTK_OBJECT (data->dialog));
	gtk_signal_connect (GTK_OBJECT (data->del_button), 
			    "clicked",
			    GTK_SIGNAL_FUNC (delete_recent_cb),
			    data);

	/* Set data. */

	/* * registered editors list. */

	data->app_list = NULL;
	for (scan = data->file_list; scan; scan = scan->next) {
		const char *result;
		const char *name = scan->data;

		if (preferences.fast_file_type)
			result = gnome_vfs_mime_type_from_name_or_default (name, NULL);
		else 
			result = gnome_mime_type_from_magic (name);

		data->app_list = g_list_concat (data->app_list, gnome_vfs_mime_get_all_applications (result));
	}

	gtk_clist_column_titles_passive (GTK_CLIST (data->app_list_clist));
	gtk_clist_set_sort_column (GTK_CLIST (data->app_list_clist), 1);

	col_data[1] = NULL;
	for (scan = data->app_list; scan; scan = scan->next) {
		gboolean found;

		app = scan->data;

		found = FALSE;
		if (app_names != NULL) {
			GList *p;
			for (p = app_names; p; p = p->next)
				if (strcmp ((char*)p->data, app->command) == 0)
					found = TRUE;
		}

		if (found)
			continue;

		app_names = g_list_prepend (app_names, app->command);

		col_data[0] = app->name;
		row = gtk_clist_append (GTK_CLIST (data->app_list_clist), 
					col_data);
		gtk_clist_set_row_data (GTK_CLIST (data->app_list_clist), row,
					app);
	}
	if (app_names)
		g_list_free (app_names);

	gtk_clist_sort (GTK_CLIST (data->app_list_clist));


	/* * recent editors list. */

	gtk_clist_column_titles_passive (GTK_CLIST (data->recent_clist));
	gtk_clist_set_sort_column (GTK_CLIST (data->recent_clist), 1);

	col_data[1] = NULL;
	for (scan = preferences.editors; scan; scan = scan->next) {
		gchar *editor = scan->data;

		col_data[0] = editor;
		row = gtk_clist_append (GTK_CLIST (data->recent_clist), 
					col_data);
		gtk_clist_set_row_data (GTK_CLIST (data->recent_clist), 
					row,
					editor);
	}
	gtk_clist_sort (GTK_CLIST (data->recent_clist));

	/* Run dialog. */
	gtk_window_set_transient_for (GTK_WINDOW (data->dialog), 
				      GTK_WINDOW (window->app));
	gtk_window_set_modal (GTK_WINDOW (data->dialog), TRUE);
	gtk_widget_show_all (data->dialog);
}
