/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  GThumb
 *
 *  Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#include <string.h>
#include <libgnome/libgnome.h>
#include <libgnomeui/gnome-preferences.h>
#include "typedefs.h"
#include "bookmarks.h"
#include "catalog-png-exporter.h"
#include "file-utils.h"
#include "image-viewer.h"
#include "gthumb-init.h"
#include "preferences.h"


#define THUMB_MAX_SIZE_DEFAULT      85

#define DEFAULT_FONT "-adobe-helvetica-medium-r-normal--*-100-*-*-*-*-*-*"

void 
preferences_init ()
{
	preferences.starting_location = NULL;
	preferences.startup_location = NULL;
	preferences.bookmarks = bookmarks_new (RC_BOOKMARKS_FILE);
	preferences.history = bookmarks_new (RC_HISTORY_FILE);

	preferences.wallpaper = gnome_config_get_string ("/Background/Default/wallpaper");
	preferences.wallpapers_dir = gnome_config_get_string ("/Background/Default/wallpapers_dir");
	preferences.wallpaperAlign = gnome_config_get_int ("/Background/Default/wallpaperAlign");

	preferences.editors_n = 0;
	preferences.editors = NULL;

	png_exporter_pref.text_font = NULL;
}


void preferences_release ()
{
	if (preferences.starting_location)
		g_free (preferences.starting_location);
	if (preferences.startup_location)
		g_free (preferences.startup_location);
	if (preferences.bookmarks)
		bookmarks_free (preferences.bookmarks);
	if (preferences.history)
		bookmarks_free (preferences.history);

	if (preferences.wallpaper)
		g_free (preferences.wallpaper);
	if (preferences.wallpapers_dir)
		g_free (preferences.wallpapers_dir);

	g_list_foreach (preferences.editors, (GFunc) g_free, NULL);
	g_list_free (preferences.editors);

	if (png_exporter_pref.text_font != NULL)
		g_free (png_exporter_pref.text_font);
}


void 
preferences_set_default_values ()
{
	gchar *temp;

	temp = g_get_current_dir ();
	preferences.starting_location = g_strconcat ("file://",
						     temp,
						     NULL);
	g_free (temp);

	/* Startup dir. */
	preferences.go_to_last_location = FALSE;
	preferences.use_startup_location = FALSE;
	preferences.startup_location = NULL;

	/* Confirmations. */
	preferences.confirm_file_del = TRUE;
	preferences.confirm_catalog_del = TRUE;

	/* Filter options. */
	preferences.show_dot_files = FALSE;
	preferences.show_comments = TRUE;
	preferences.fast_file_type = TRUE;

	/* Thumbnails options. */
	preferences.enable_thumbnails = TRUE;
	preferences.enable_thumb_caching = TRUE;
	preferences.thumb_size = THUMB_MAX_SIZE_DEFAULT;
	preferences.thumb_quality = ZOOM_QUALITY_HIGH;

	/* History options. */
	preferences.history_max_length = HISTORY_LIST_MAX_LENGTH;

	/* Viewer options. */
	preferences.zoom_quality = ZOOM_QUALITY_HIGH;
	preferences.zoom_change = ZOOM_CHANGE_KEEP_PREV;
	preferences.transp_type = TRANSP_TYPE_CHECKED;
	preferences.check_type = CHECK_TYPE_MIDTONE;
	preferences.check_size = CHECK_SIZE_LARGE;

	/* Slide Show. */
	preferences.ss_direction = DIRECTION_FORWARD;
	preferences.ss_delay = 4;
	preferences.ss_wrap_around = FALSE;
	preferences.ss_fullscreen = FALSE;

	preferences.file_list_sort = SORT_BY_NAME;
	preferences.file_list_sort_type = GTK_SORT_ASCENDING;

	/* Visualization options. */
	preferences.layout_type = 0;
	preferences.location_on_main_bar = TRUE;
	preferences.main_win_width = 680;
	preferences.main_win_height = 500;

	preferences.sidebar_width = 230;
	preferences.sidebar_content_height = 150;

	/* Load bookmark. */
	bookmarks_load_from_disk (preferences.bookmarks);
	bookmarks_load_from_disk (preferences.history);

	preferences.menu_have_tearoff = gnome_preferences_get_menus_have_tearoff ();

	png_exporter_pref.text_font = g_strdup (DEFAULT_FONT);
}


static gint
get_int_with_default (gchar *config_path, gint def)
{
	gchar *path_with_def;
	gint result;

	path_with_def = g_strdup_printf ("%s=%d", config_path, def);
	result = gnome_config_get_int (path_with_def);
	g_free (path_with_def);
					 
	return result;
}


static gint
get_bool_with_default (gchar *config_path, gboolean def)
{
	gchar *path_with_def;
	gboolean result;

	path_with_def = g_strdup_printf ("%s=%s", config_path, 
					 def ? "true" : "false");
	result = gnome_config_get_bool (path_with_def);
	g_free (path_with_def);
					 
	return result;
}


void 
preferences_load () 
{
	int i;

	/* Startup dir. */
	preferences.go_to_last_location = gnome_config_get_bool_with_default (
		"gthumb/Start Up/use last location", FALSE);
	preferences.use_startup_location = gnome_config_get_bool_with_default (
		"gthumb/Start Up/use", FALSE);
	if (preferences.use_startup_location 
	    || preferences.go_to_last_location) {
		preferences.startup_location = gnome_config_get_string (
			"gthumb/Start Up/location");

		if (preferences.starting_location) 
			g_free (preferences.starting_location);

		if (pref_util_location_is_file (preferences.startup_location)
		    && ! path_is_dir (pref_util_get_file_location (preferences.startup_location)))
			preferences.starting_location = 
				g_strconcat ("file://",
					     g_get_home_dir (),
					     NULL);
		else
			preferences.starting_location = g_strdup (preferences.startup_location);
	}

	/* Confirmations. */
	preferences.confirm_file_del = get_bool_with_default (
		"gthumb/Confirmations/file", TRUE);
	preferences.confirm_catalog_del = get_bool_with_default (
		"gthumb/Confirmations/catalog", TRUE);
	
	/* Filter options. */
	preferences.show_dot_files = get_bool_with_default (
		 "gthumb/Filter/show dot files", FALSE);
	preferences.show_comments = get_bool_with_default (
		 "gthumb/Filter/show comments", TRUE);
	preferences.fast_file_type = get_bool_with_default (
		"gthumb/Filter/fast file type", TRUE);

	/* Thumbnails options. */
	preferences.enable_thumbnails = get_bool_with_default (
		"gthumb/Thumbnails/enable", TRUE);
	preferences.enable_thumb_caching = get_bool_with_default (
		"gthumb/Thumbnails/caching", TRUE);
	preferences.thumb_quality = get_int_with_default  (
		"gthumb/Thumbnails/quality", ZOOM_QUALITY_HIGH);
	preferences.thumb_size = get_int_with_default (
		"gthumb/Thumbnails/size", THUMB_MAX_SIZE_DEFAULT);

	/* History options. */
	preferences.history_max_length = get_int_with_default (
		"gthumb/History/max length", HISTORY_LIST_MAX_LENGTH);

	/* Viewer options. */
	preferences.zoom_quality = get_int_with_default (
		"gthumb/Viewer/zoom quality", ZOOM_QUALITY_HIGH);
	preferences.zoom_change = get_int_with_default (
		"gthumb/Viewer/zoom change", ZOOM_CHANGE_KEEP_PREV); 
	preferences.transp_type = get_int_with_default (
		"gthumb/Viewer/transparenty type", TRANSP_TYPE_CHECKED);
	preferences.check_type = get_int_with_default (
		"gthumb/Viewer/check type", CHECK_TYPE_MIDTONE);
	preferences.check_size = get_int_with_default (
		"gthumb/Viewer/check size", CHECK_SIZE_LARGE); 

	/* Slide Show. */
	preferences.ss_direction = get_int_with_default (
		"gthumb/Slide Show/direction", DIRECTION_FORWARD);
	preferences.ss_delay = get_int_with_default (
		"gthumb/Slide Show/delay", 4);
	preferences.ss_wrap_around = get_bool_with_default (
		"gthumb/Slide Show/wrap around", FALSE);
	preferences.ss_fullscreen = get_bool_with_default (
		"gthumb/Slide Show/fullscreen", FALSE);

	/**/
	preferences.file_list_sort = get_int_with_default (
		"gthumb/File List/Sort", SORT_BY_NAME);
	preferences.file_list_sort_type = get_int_with_default (
		"gthumb/File List/Sort Type", GTK_SORT_ASCENDING);

	/* Visualization options. */
	preferences.location_on_main_bar = get_bool_with_default (
		"gthumb/Main Window/location on main bar", TRUE);
	preferences.layout_type = gnome_config_get_int (
		"gthumb/Main Window/layout=1");
	preferences.main_win_width = gnome_config_get_int (
		"gthumb/Main Window/width=680");
	preferences.main_win_height = gnome_config_get_int (
		"gthumb/Main Window/height=500");
	preferences.sidebar_width = gnome_config_get_int (
		"gthumb/Sidebar/width=230");
	preferences.sidebar_content_height = gnome_config_get_int (
		"gthumb/Sidebar/content height=150");

	preferences.editors_n = gnome_config_get_int (
		"gthumb/Editors/editors=0");
	for (i = 0; i < preferences.editors_n; i++) {
		gchar *prop_name;
		gchar *prop_val;

		prop_name = g_strdup_printf ("gthumb/Editors/editor%d", i);
		prop_val = gnome_config_get_string (prop_name);
		if (prop_val != NULL)
			preferences.editors = g_list_prepend (preferences.editors, prop_val);
		g_free (prop_name);
	}

	/* Exporter preferences. */

	/* * Page */
	png_exporter_pref.page_width = gnome_config_get_int (
		"gthumb/Exporter-Page/width=400");
	png_exporter_pref.page_height = gnome_config_get_int (
		"gthumb/Exporter-Page/height=400");
	png_exporter_pref.size_use_row_col = get_bool_with_default (
		"gthumb/Exporter-Page/use rows cols", TRUE);
	png_exporter_pref.page_rows = gnome_config_get_int (
		"gthumb/Exporter-Page/rows=3");
	png_exporter_pref.page_cols = gnome_config_get_int (
		"gthumb/Exporter-Page/cols=4");
	png_exporter_pref.all_pages_same_size = get_bool_with_default (
		"gthumb/Exporter-Page/same size", TRUE);

	png_exporter_pref.sort_method = get_int_with_default (
		"gthumb/Exporter-Page/Sort", SORT_BY_NAME);
	png_exporter_pref.sort_type = get_int_with_default (
		"gthumb/Exporter-Page/Sort Type", GTK_SORT_ASCENDING);

	png_exporter_pref.bg_color.red = gnome_config_get_int (
		"gthumb/Exporter-Page/bg_col.red=65535");
	png_exporter_pref.bg_color.green = gnome_config_get_int (
		"gthumb/Exporter-Page/bg_col.green=65535");
	png_exporter_pref.bg_color.blue = gnome_config_get_int (
		"gthumb/Exporter-Page/bg_col.blue=65535");

	/* * Thumbnail */
	png_exporter_pref.caption = get_int_with_default (
		"gthumb/Exporter-Thumbnail/caption", (CAPTION_FILE_NAME 
						      | CAPTION_IMAGE_DIM));
	png_exporter_pref.frame_style = get_int_with_default (
		"gthumb/Exporter-Thumbnail/frame style", FRAME_STYLE_SIMPLE);

	png_exporter_pref.frame_color.red = get_int_with_default (
		"gthumb/Exporter-Thumbnail/frame_color.red", 0);
	png_exporter_pref.frame_color.green = get_int_with_default (
		"gthumb/Exporter-Thumbnail/frame_color.green", 0);
	png_exporter_pref.frame_color.blue = get_int_with_default (
		"gthumb/Exporter-Thumbnail/frame_color.blue", 0);

	png_exporter_pref.thumb_size = get_int_with_default (
		"gthumb/Exporter-Thumbnail/thumb_size", 75);
	
	png_exporter_pref.text_color.red = get_int_with_default (
		"gthumb/Exporter-Thumbnail/text_color.red", 65535);
	png_exporter_pref.text_color.green = get_int_with_default (
		"gthumb/Exporter-Thumbnail/text_color.green", 0);
	png_exporter_pref.text_color.blue = get_int_with_default (
		"gthumb/Exporter-Thumbnail/text_color.blue", 0);

	if (png_exporter_pref.text_font != NULL)
		g_free (png_exporter_pref.text_font);
	png_exporter_pref.text_font = gnome_config_get_string (
		"gthumb/Exporter-Thumbnail/text_font="DEFAULT_FONT);

	png_exporter_pref.write_imap = get_bool_with_default (
		"gthumb/Exporter/write image map", FALSE);
}


void 
preferences_save ()
{
	GList *scan;
	int i;

	/* Startup dir. */
	gnome_config_set_bool ("gthumb/Start Up/use last location",
			       preferences.go_to_last_location);
	gnome_config_set_bool ("gthumb/Start Up/use",
			       preferences.use_startup_location);
	if (preferences.startup_location || preferences.go_to_last_location)
		gnome_config_set_string ("gthumb/Start Up/location",
					 preferences.startup_location);

	/* Confirmations. */
	gnome_config_set_bool ("gthumb/Confirmations/file",
			       preferences.confirm_file_del);
	gnome_config_set_bool ("gthumb/Confirmations/catalog",
			       preferences.confirm_catalog_del);
	
	/* Filter options. */
	gnome_config_set_bool ("gthumb/Filter/show dot files",
			       preferences.show_dot_files);
	gnome_config_set_bool ("gthumb/Filter/show comments",
			       preferences.show_comments);
	gnome_config_set_bool ("gthumb/Filter/fast file type",
			       preferences.fast_file_type);

	/* Thumbnails options. */
	gnome_config_set_bool ("gthumb/Thumbnails/enable",
			       preferences.enable_thumbnails);
	gnome_config_set_bool ("gthumb/Thumbnails/caching",
			       preferences.enable_thumb_caching);
	gnome_config_set_int  ("gthumb/Thumbnails/quality",
			       preferences.thumb_quality);
	gnome_config_set_int  ("gthumb/Thumbnails/size",
			       preferences.thumb_size);

	/* History options. */
	gnome_config_set_int  ("gthumb/History/max length",
			       preferences.history_max_length);

	/* Viewer options. */
	gnome_config_set_int  ("gthumb/Viewer/zoom quality",
			       preferences.zoom_quality);
	gnome_config_set_int  ("gthumb/Viewer/zoom change",
			       preferences.zoom_change);
	gnome_config_set_int  ("gthumb/Viewer/transparenty type",
			       preferences.transp_type);
	gnome_config_set_int  ("gthumb/Viewer/check type",
			       preferences.check_type);
	gnome_config_set_int  ("gthumb/Viewer/check size",
			       preferences.check_size);

	/* Slide Show. */
	gnome_config_set_int  ("gthumb/Slide Show/direction",
			       preferences.ss_direction);
	gnome_config_set_int  ("gthumb/Slide Show/delay",
			       preferences.ss_delay);
	gnome_config_set_bool ("gthumb/Slide Show/wrap around",
			       preferences.ss_wrap_around);
	gnome_config_set_bool ("gthumb/Slide Show/fullscreen",
			       preferences.ss_fullscreen);

	/**/
	gnome_config_set_int  ("gthumb/File List/Sort",
			       preferences.file_list_sort);
	gnome_config_set_int  ("gthumb/File List/Sort Type",
			       preferences.file_list_sort_type);

	/* Visualization options. */
	gnome_config_set_bool ("gthumb/Main Window/location on main bar",
			       preferences.location_on_main_bar);
	gnome_config_set_int  ("gthumb/Main Window/layout",
			       preferences.layout_type);
	gnome_config_set_int  ("gthumb/Main Window/width",
			       preferences.main_win_width);
	gnome_config_set_int  ("gthumb/Main Window/height",
			       preferences.main_win_height);
	gnome_config_set_int  ("gthumb/Sidebar/width",
			       preferences.sidebar_width);
	gnome_config_set_int  ("gthumb/Sidebar/content height",
			       preferences.sidebar_content_height);

	gnome_config_set_int ("gthumb/Editors/editors", preferences.editors_n);
	for (i = 0, scan = preferences.editors; scan; scan = scan->next) {
		gchar *prop_name;
		prop_name = g_strdup_printf ("gthumb/Editors/editor%d", i++);
		gnome_config_set_string (prop_name, (gchar*) scan->data);
		g_free (prop_name);
	}

	/* Exporter preferences. */

	/* * Page */
	gnome_config_set_int  ("gthumb/Exporter-Page/width",
			       png_exporter_pref.page_width);
	gnome_config_set_int  ("gthumb/Exporter-Page/height",
			       png_exporter_pref.page_height);
	gnome_config_set_bool ("gthumb/Exporter-Page/use rows cols",
			       png_exporter_pref.size_use_row_col);
	gnome_config_set_int  ("gthumb/Exporter-Page/rows",
			       png_exporter_pref.page_rows);
	gnome_config_set_int  ("gthumb/Exporter-Page/cols",
			       png_exporter_pref.page_cols);
	gnome_config_set_bool ("gthumb/Exporter-Page/same size",
			       png_exporter_pref.all_pages_same_size);

	gnome_config_set_int  ("gthumb/Exporter-Page/Sort",
			       png_exporter_pref.sort_method);
	gnome_config_set_int  ("gthumb/Exporter-Page/Sort Type",
			       png_exporter_pref.sort_type);

	gnome_config_set_int  ("gthumb/Exporter-Page/bg_col.red",
			       png_exporter_pref.bg_color.red);
	gnome_config_set_int  ("gthumb/Exporter-Page/bg_col.green",
			       png_exporter_pref.bg_color.green);
	gnome_config_set_int  ("gthumb/Exporter-Page/bg_col.blue",
			       png_exporter_pref.bg_color.blue);

	/* * Thumbnail */
	gnome_config_set_int  ("gthumb/Exporter-Thumbnail/caption",
			       png_exporter_pref.caption);
	gnome_config_set_int  ("gthumb/Exporter-Thumbnail/frame style",
			       png_exporter_pref.frame_style);

	gnome_config_set_int  ("gthumb/Exporter-Thumbnail/frame_color.red",
			       png_exporter_pref.frame_color.red);
	gnome_config_set_int  ("gthumb/Exporter-Thumbnail/frame_color.green",
			       png_exporter_pref.frame_color.green);
	gnome_config_set_int  ("gthumb/Exporter-Thumbnail/frame_color.blue",
			       png_exporter_pref.frame_color.blue);

	gnome_config_set_int  ("gthumb/Exporter-Thumbnail/thumb_size",
			       png_exporter_pref.thumb_size);

	gnome_config_set_int  ("gthumb/Exporter-Thumbnail/text_color.red",
			       png_exporter_pref.text_color.red);
	gnome_config_set_int  ("gthumb/Exporter-Thumbnail/text_color.green",
			       png_exporter_pref.text_color.green);
	gnome_config_set_int  ("gthumb/Exporter-Thumbnail/text_color.blue",
			       png_exporter_pref.text_color.blue);
	gnome_config_set_string ("gthumb/Exporter-Thumbnail/text_font",
				 png_exporter_pref.text_font);
	gnome_config_set_bool ("gthumb/Exporter/write image map",
			       png_exporter_pref.write_imap);

	gnome_config_sync ();
}


const gchar * 
pref_util_get_file_location (const gchar *location)
{
	if (strlen (location) <= FILE_PREFIX_L)
		return NULL;
	return location + FILE_PREFIX_L;
}


const gchar * 
pref_util_get_catalog_location (const gchar *location)
{
	if (strlen (location) <= CATALOG_PREFIX_L)
		return NULL;
	return location + CATALOG_PREFIX_L;
}


const gchar * 
pref_util_get_search_location (const gchar *location)
{
	if (strlen (location) <= SEARCH_PREFIX_L)
		return NULL;
	return location + SEARCH_PREFIX_L;
}


gboolean
pref_util_location_is_file (const gchar *location)
{
	if (strlen (location) <= FILE_PREFIX_L)
		return FALSE;
	return strncmp (location, FILE_PREFIX, FILE_PREFIX_L) == 0;
}


gboolean
pref_util_location_is_catalog (const gchar *location)
{
	if (strlen (location) <= CATALOG_PREFIX_L)
		return FALSE;
	return strncmp (location, CATALOG_PREFIX, CATALOG_PREFIX_L) == 0;
}


gboolean
pref_util_location_is_search (const gchar *location)
{
	if (strlen (location) <= SEARCH_PREFIX_L)
		return FALSE;
	return strncmp (location, SEARCH_PREFIX, SEARCH_PREFIX_L) == 0;
}


const gchar *
pref_util_remove_prefix (const gchar *location)
{
	if (pref_util_location_is_catalog (location))
		return pref_util_get_catalog_location (location);
	else if (pref_util_location_is_search (location))
		return pref_util_get_search_location (location);
	else if (pref_util_location_is_file (location))
		return pref_util_get_file_location (location);
	else
		return location;
}
